/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <usb.h>			/* needed for usb functions */
#include <getopt.h>
#include <assert.h>
#include <math.h>
#include "time_stuff.h"
#include "usrp0.h"

char *prog_name;

static bool test_output (usrp0_tx *utx, bool forever_p);
static bool test_input  (usrp0_rx *urx, bool forever_p, FILE *fp);

static void
set_progname (char *path)
{
  char *p = strrchr (path, '/');
  if (p != 0)
    prog_name = p+1;
  else
    prog_name = path;
}

static void
usage ()
{
  fprintf (stderr, "usage: %s [-i] [-f] [-v] [-I <interp>] [-D <decim>] [-o output_file] [-F freq]\n", prog_name);
  fprintf (stderr, "  [-i] input\n");
  fprintf (stderr, "  [-f] loop forever\n");
  fprintf (stderr, "  [-v] verbose\n");
  exit (1);
}

static void
die (const char *msg)
{
  fprintf (stderr, "die: %s: %s\n", prog_name, msg);
  exit (1);
}

int
main (int argc, char **argv)
{
  bool	input_p = false;
  bool 	verbose_p = false;
  bool	forever_p = false;
  int	ch;
  char	*output_filename = 0;
  int	which_board = 0;
  int	interp = 16;	// 31.25 MB/sec 
  int	decim  =  8;    // 31.25 MB/sec
  double	center_freq = 0;

  set_progname (argv[0]);

  while ((ch = getopt (argc, argv, "ivfo:I:D:F:")) != EOF){
    switch (ch){
    case 'i':
      input_p = true;
      break;
      
    case 'f':
      forever_p = true;
      break;

    case 'v':
      verbose_p = true;
      break;

    case 'o':
      output_filename = optarg;
      break;
      
    case 'D':
      decim = strtol (optarg, 0, 0);
      break;

    case 'I':
      interp = strtol (optarg, 0, 0);
      break;
		      
    case 'F':
      center_freq = strtod (optarg, 0);
      break;
      
    default:
      usage ();
    }
  }


  if (input_p){
    FILE *fp = 0;

    usrp0_rx *urx = usrp0_rx::make (which_board);
    if (urx == 0)
      die ("usrp0_rx::make");

    if (!urx->set_decim_rate (decim))
      die ("urx->set_decim_rate");
    
    if (!urx->set_rx_freq (0, center_freq))
      die ("urx->set_rx_freq");
    
    if (output_filename){
      fp = fopen (output_filename, "wb");
      if (fp == 0)
	perror (output_filename);
    }
      
    test_input (urx, forever_p, fp);

    if (fp)
      fclose (fp);

    delete urx;
  }

  else {

    usrp0_tx *utx = usrp0_tx::make (which_board);
    if (utx == 0)
      die ("usrp0_tx::make");
    
    if (!utx->set_interp_rate (interp))
      die ("urx->set_interp_rate");

    test_output (utx, forever_p);

    delete utx;
  }

  return 0;
}


static bool
test_output (usrp0_tx *utx, bool forever_p)
{
  static const int BUFSIZE = 16 * 1024;
  static const int N = BUFSIZE/sizeof (short);
  short 	   buf[N];
  int		   nbytes = 0;
  int		   max_bytes = 512 * (1L << 20);

  static const int    PERIOD = 65;		// any value is valid
  static const int    PATLEN = 2 * PERIOD;	
  static const double AMPL = 10000;
  short		      pattern[PATLEN];

  for (int i = 0; i < PERIOD; i++){
    pattern[2*i+0] = (short) (AMPL * cos (2*M_PI * i / PERIOD));
    pattern[2*i+1] = (short) (AMPL * sin (2*M_PI * i / PERIOD));
  }
  

  double	   start_wall_time = get_elapsed_time ();
  double	   start_cpu_time  = get_cpu_usage ();

  bool 	underrun;
  int 	nunderruns = 0;
  int 	pi = 0;

  for (nbytes = 0; forever_p || nbytes < max_bytes; nbytes += BUFSIZE){

    for (int i = 0; i < N; i++){
      buf[i] = pattern[pi];
      pi++;
      if (pi >= PATLEN)
	pi = 0;
    }

    int	ret = utx->write (buf, sizeof (buf), &underrun);
    if (ret != sizeof (buf)){
      fprintf (stderr, "test_output: error, ret = %d\n", ret);
    }

    if (underrun){
      nunderruns++;
      printf ("tx_underrun\n");
    }
  }

  utx->wait_for_completion ();

  double stop_wall_time = get_elapsed_time ();
  double stop_cpu_time  = get_cpu_usage ();

  double delta_wall = stop_wall_time - start_wall_time;
  double delta_cpu  = stop_cpu_time  - start_cpu_time;

  printf ("xfered %.3g bytes in %.3g seconds.  %.4g bytes/sec.  cpu time = %.3g\n",
	  (double) max_bytes, delta_wall, max_bytes / delta_wall, delta_cpu);

  printf ("%d underruns\n", nunderruns);

  return true;
}


static bool
test_input  (usrp0_rx *urx, bool forever_p, FILE *fp)
{
  int		   fd = -1;
  static const int BUFSIZE = 16 * 1024;
  static const int N = BUFSIZE/sizeof (short);
  short 	   buf[N];
  int		   nbytes = 0;
  int	   	   max_bytes = 512 * (1L << 20);
  // int		   max_bytes = 128 * (1L << 20);

  double	   start_wall_time = get_elapsed_time ();
  double	   start_cpu_time  = get_cpu_usage ();

  if (fp)
    fd = fileno (fp);
  
  bool overrun;
  int noverruns = 0;

  for (nbytes = 0; forever_p || nbytes < max_bytes; nbytes += BUFSIZE){

    int	ret = urx->read (buf, sizeof (buf), &overrun);
    if (ret != sizeof (buf)){
      fprintf (stderr, "test_input: error, ret = %d\n", ret);
    }

    if (overrun){
      printf ("rx_overrun\n");
      noverruns++;
    }
    
    if (fd != -1){
      if (write (fd, buf, sizeof (buf)) == -1){
	perror ("write");
	fd = -1;
      }
    }
  }

  double stop_wall_time = get_elapsed_time ();
  double stop_cpu_time  = get_cpu_usage ();

  double delta_wall = stop_wall_time - start_wall_time;
  double delta_cpu  = stop_cpu_time  - start_cpu_time;

  printf ("xfered %.3g bytes in %.3g seconds.  %.4g bytes/sec.  cpu time = %.3g\n",
	  (double) max_bytes, delta_wall, max_bytes / delta_wall, delta_cpu);
  printf ("noverruns = %d\n", noverruns);

  return true;
}
