/* Display generation from window structure and buffer text.

Copyright (C) 1985-2023 Free Software Foundation, Inc.

This file is part of GNU Emacs.

GNU Emacs is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

GNU Emacs is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs.  If not, see <https://www.gnu.org/licenses/>.  */

/* New redisplay written by Gerd Moellmann <gerd@gnu.org>.

   Redisplay.

   Emacs separates the task of updating the display from code
   modifying global state, e.g. buffer text.  This way functions
   operating on buffers don't also have to be concerned with updating
   the display.

   Updating the display is triggered by the Lisp interpreter when it
   decides it's time to do it.  This is done either automatically for
   you as part of the interpreter's command loop or as the result of
   calling Lisp functions like `sit-for'.  The C function
   `redisplay_internal' in xdisp.c is the only entry into the inner
   redisplay code.

   The following diagram shows how redisplay code is invoked.  As you
   can see, Lisp calls redisplay and vice versa.

   Under window systems like X, some portions of the redisplay code
   are also called asynchronously, due to mouse movement or expose
   events.  "Asynchronously" in this context means that any C function
   which calls maybe_quit or process_pending_signals could enter
   redisplay via expose_frame and/or note_mouse_highlight, if X events
   were recently reported to Emacs about mouse movements or frame(s)
   that were exposed.  And such redisplay could invoke the Lisp
   interpreter, e.g. via the :eval forms in mode-line-format, and as
   result the global state could change.  It is therefore very
   important that C functions which might cause such "asynchronous"
   redisplay, but cannot tolerate the results, use
   block_input/unblock_input around code fragments which assume that
   global Lisp state doesn't change.  If you don't follow this rule,
   you will encounter bugs which are very hard to explain.  One place
   that needs to take such precautions is timer_check, some of whose
   code cannot tolerate changes in timer alists while it processes
   timers.

   +--------------+   redisplay     +----------------+
   | Lisp machine |---------------->| Redisplay code |<--+
   +--------------+   (xdisp.c)     +----------------+   |
	  ^				     |		 |
	  +----------------------------------+           |
	    Block input to prevent this when             |
	    called asynchronously!			 |
							 |
		    note_mouse_highlight (asynchronous)	 |
							 |
				    X mouse events  -----+
							 |
			    expose_frame (asynchronous)	 |
							 |
				   X expose events  -----+

   What does redisplay do?  Obviously, it has to figure out somehow what
   has been changed since the last time the display has been updated,
   and to make these changes visible.  Preferably it would do that in
   a moderately intelligent way, i.e. fast.

   Changes in buffer text can be deduced from window and buffer
   structures, and from some global variables like `beg_unchanged' and
   `end_unchanged'.  The contents of the display are additionally
   recorded in a `glyph matrix', a two-dimensional matrix of glyph
   structures.  Each row in such a matrix corresponds to a line on the
   display, and each glyph in a row corresponds to a column displaying
   a character, an image, or what else.  This matrix is called the
   `current glyph matrix' or `current matrix' in redisplay
   terminology.

   For buffer parts that have been changed since the last update, a
   second glyph matrix is constructed, the so called `desired glyph
   matrix' or short `desired matrix'.  Current and desired matrix are
   then compared to find a cheap way to update the display, e.g. by
   reusing part of the display by scrolling lines.  The actual update
   of the display of each window by comparing the desired and the
   current matrix is done by `update_window', which calls functions
   which draw to the glass (those functions are specific to the type
   of the window's frame: X, w32, NS, etc.).

   Once the display of a window on the glass has been updated, its
   desired matrix is used to update the corresponding rows of the
   current matrix, and then the desired matrix is discarded.

   You will find a lot of redisplay optimizations when you start
   looking at the innards of redisplay.  The overall goal of all these
   optimizations is to make redisplay fast because it is done
   frequently.  Some of these optimizations are implemented by the
   following functions:

    . try_cursor_movement

      This function tries to update the display if the text in the
      window did not change and did not scroll, only point moved, and
      it did not move off the displayed portion of the text.

    . try_window_reusing_current_matrix

      This function reuses the current matrix of a window when text
      has not changed, but the window start changed (e.g., due to
      scrolling).

    . try_window_id

      This function attempts to redisplay a window by reusing parts of
      its existing display.  It finds and reuses the part that was not
      changed, and redraws the rest.  (The "id" part in the function's
      name stands for "insert/delete", not for "identification" or
      somesuch.)

    . try_window

      This function performs the full, unoptimized, redisplay of a
      single window assuming that its fonts were not changed and that
      the cursor will not end up in the scroll margins.  (Loading
      fonts requires re-adjustment of dimensions of glyph matrices,
      which makes this method impossible to use.)

   The optimizations are tried in sequence (some can be skipped if
   it is known that they are not applicable).  If none of the
   optimizations were successful, redisplay calls redisplay_windows,
   which performs a full redisplay of all windows.

   Note that there's one more important optimization up Emacs's
   sleeve, but it is related to actually redrawing the potentially
   changed portions of the window/frame, not to reproducing the
   desired matrices of those potentially changed portions.  Namely,
   the function update_frame and its subroutines, which you will find
   in dispnew.c, compare the desired matrices with the current
   matrices, and only redraw the portions that changed.  So it could
   happen that the functions in this file for some reason decide that
   the entire desired matrix needs to be regenerated from scratch, and
   still only parts of the Emacs display, or even nothing at all, will
   be actually delivered to the glass, because update_frame has found
   that the new and the old screen contents are similar or identical.

   Desired matrices.

   Desired matrices are always built per Emacs window.  It is
   important to know that a desired matrix is in general "sparse": it
   only has some of the glyph rows "enabled".  This is because
   redisplay tries to optimize its work, and thus only generates
   glyphs for rows that need to be updated on the screen.  Rows that
   don't need to be updated are left "disabled", and their contents
   should be ignored.

   The function `display_line' is the central function to look at if
   you are interested in how the rows of the desired matrix are
   produced.  It constructs one row in a desired matrix given an
   iterator structure containing both a buffer position and a
   description of the environment in which the text is to be
   displayed.  But this is too early, read on.

   Glyph rows.

   A glyph row is an array of `struct glyph', where each glyph element
   describes a "display element" to be shown on the screen.  More
   accurately, a glyph row can have up to 3 different arrays of
   glyphs: one each for every display margins, and one for the "text
   area", where buffer text is displayed.  The text-area glyph array
   is always present, whereas the arrays for the marginal areas are
   present (non-empty) only if the corresponding display margin is
   shown in the window.  If the glyph array for a marginal area is not
   present its beginning and end coincide, i.e. such arrays are
   actually empty (they contain no glyphs).  Frame glyph matrices, used
   on text-mode terminals (see below) never have marginal areas; they
   treat the entire frame-wide row of glyphs as a single large "text
   area".

   Iteration over buffer and strings.

   Characters and pixmaps displayed for a range of buffer text depend
   on various settings of buffers and windows, on overlays and text
   properties, on display tables, on selective display.  The good news
   is that all this hairy stuff is hidden behind a small set of
   interface functions taking an iterator structure (`struct it')
   argument.

   Iteration over things to be displayed is then simple.  It is
   started by initializing an iterator with a call to `init_iterator',
   passing it the buffer position where to start iteration.  For
   iteration over strings, pass -1 as the position to `init_iterator',
   and call `reseat_to_string' when the string is ready, to initialize
   the iterator for that string.  Thereafter, calls to
   `get_next_display_element' fill the iterator structure with
   relevant information about the next thing to display.  Calls to
   `set_iterator_to_next' move the iterator to the next thing.

   Besides this, an iterator also contains information about the
   display environment in which glyphs for display elements are to be
   produced.  It has fields for the width and height of the display,
   the information whether long lines are truncated or continued, a
   current X and Y position, the face currently in effect, and lots of
   other stuff you can better see in dispextern.h.

   The "stop position".

   Some of the fields maintained by the iterator change relatively
   infrequently.  These include the face of the characters, whether
   text is invisible, the object (buffer or display or overlay string)
   being iterated, character composition info, etc.  For any given
   buffer or string position, the sources of information that affects
   the display can be determined by calling the appropriate
   primitives, such as `Fnext_single_property_change', but both these
   calls and the processing of their return values is relatively
   expensive.  To optimize redisplay, the display engine checks these
   sources of display information only when needed, not for every
   character.  To that end, it always maintains the position of the
   next place where it must stop and re-examine all those potential
   sources.  This is called "the stop position" and is stored in the
   `stop_charpos' field of the iterator.  The stop position is updated
   by `compute_stop_pos', which is called whenever the iteration
   reaches the current stop position and processes it.  Processing a
   stop position is done by `handle_stop', which invokes a series of
   handlers, one each for every potential source of display-related
   information; see the `it_props' array for those handlers.  For
   example, one handler is `handle_face_prop', which detects changes
   in face properties, and supplies the face ID that the iterator will
   use for all the glyphs it generates up to the next stop position;
   this face ID is the result of "realizing" the face specified by the
   relevant text properties at this position (see xfaces.c).  Each
   handler called by `handle_stop' processes the sources of display
   information for which it is "responsible", and returns a value
   which tells `handle_stop' what to do next.

   Once `handle_stop' returns, the information it stores in the
   iterator fields will not be refreshed until the iteration reaches
   the next stop position, which is computed by `compute_stop_pos'
   called at the end of `handle_stop'.  `compute_stop_pos' examines
   the buffer's or string's interval tree to determine where the text
   properties change, finds the next position where overlays and
   character composition can change, and stores in `stop_charpos' the
   closest position where any of these factors should be reconsidered.

   Handling of the stop position is done as part of the code in
   `get_next_display_element'.

   Producing glyphs.

   Glyphs in a desired matrix are normally constructed in a loop
   calling `get_next_display_element' and then `PRODUCE_GLYPHS'.  The
   call to `PRODUCE_GLYPHS' will fill the iterator structure with
   pixel information about the element being displayed and at the same
   time will produce glyphs for it.  If the display element fits on
   the line being displayed, `set_iterator_to_next' is called next,
   otherwise the glyphs produced are discarded, and `display_line'
   marks this glyph row as a "continued line".  The function
   `display_line' is the workhorse of filling glyph rows in the
   desired matrix with glyphs.  In addition to producing glyphs, it
   also handles line truncation and continuation, word wrap, and
   cursor positioning (for the latter, see `set_cursor_from_row').

   Frame matrices.

   That just couldn't be all, could it?  What about terminal types not
   supporting operations on sub-windows of the screen (a.k.a. "TTY" or
   "text-mode terminals")?  To update the display on such a terminal,
   window-based glyph matrices are not well suited.  To be able to
   reuse part of the display (scrolling lines up and down), we must
   instead have a view of the whole screen.  This is what `frame
   matrices' are for.  They are a trick.

   Frames on text terminals have a glyph pool.  Windows on such a
   frame sub-allocate their glyph memory from their frame's glyph
   pool.  The frame itself is given its own glyph matrices.  By
   coincidence---or maybe something else---rows in window glyph
   matrices are slices of corresponding rows in frame matrices.  Thus
   writing to window matrices implicitly updates a frame matrix which
   provides us with the view of the whole screen that we originally
   wanted to have without having to move many bytes around.  Then
   updating all the visible windows on text-terminal frames is done by
   using the frame matrices, which allows frame-global optimization of
   what is actually written to the glass.

   Frame matrices don't have marginal areas, only a text area.  That
   is, the entire row of glyphs that spans the width of a text-mode
   frame is treated as a single large "text area" for the purposes of
   manipulating and updating a frame glyph matrix.

   To be honest, there is a little bit more done for frame matrices,
   but not much more.  If you plan to extend that code, take a look at
   dispnew.c.  The function build_frame_matrix is a good starting
   point.

   Simulating display.

   Some of Emacs commands and functions need to take display layout
   into consideration.  For example, C-n moves to the next screen
   line, but to implement that, Emacs needs to find the buffer
   position which is directly below the cursor position on display.
   This is not trivial when buffer display includes variable-size
   elements such as different fonts, tall images, etc.

   To solve this problem, the display engine implements several
   functions that can move through buffer text in the same manner as
   `display_line' and `display_string' do, but without producing any
   glyphs for the glyph matrices.  The workhorse of this is
   `move_it_in_display_line_to'.  Its code and logic are very similar
   to `display_line', but it differs in two important aspects: it
   doesn't produce glyphs for any glyph matrix, and it returns a
   status telling the caller how it ended the iteration: whether it
   reached the required position, hit the end of line, arrived at the
   window edge without exhausting the buffer's line, etc.  Since the
   glyphs are not produced, the layout information available to the
   callers of this function is what is recorded in `struct it' by the
   iteration process.

   Several higher-level functions call `move_it_in_display_line_to' to
   perform more complex tasks: `move_it_by_lines' can move N lines up
   or down from a given buffer position and `move_it_to' can move to a
   given buffer position or to a given X or Y pixel coordinate.

   These functions are called by the display engine itself as well,
   when it needs to make layout decisions before producing the glyphs.
   For example, one of the first things to decide when redisplaying a
   window is where to put the `window-start' position; if the window
   is to be recentered (the default), Emacs makes that decision by
   starting from the position of point, then moving up the number of
   lines corresponding to half the window height using
   `move_it_by_lines'.

   Bidirectional display.

   Bidirectional display adds quite some hair to this already complex
   design.  The good news are that a large portion of that hairy stuff
   is hidden in bidi.c behind only 3 interfaces.  bidi.c implements a
   reordering engine which is called by `set_iterator_to_next' and
   returns the next character to display in the visual order.  See
   commentary on bidi.c for more details.  As far as redisplay is
   concerned, the effect of calling `bidi_move_to_visually_next', the
   main interface of the reordering engine, is that the iterator gets
   magically placed on the buffer or string position that is to be
   displayed next in the visual order.  In other words, a linear
   iteration through the buffer/string is replaced with a non-linear
   one.  All the rest of the redisplay is oblivious to the bidi
   reordering.

   Well, almost oblivious---there are still complications, most of
   them due to the fact that buffer and string positions no longer
   change monotonously with glyph indices in a glyph row.  Moreover,
   for continued lines, the buffer positions may not even be
   monotonously changing with vertical positions.  Also, accounting
   for face changes, overlays, etc. becomes more complex because
   non-linear iteration could potentially skip many positions with
   such changes, and then cross them again on the way back (see
   `handle_stop_backwards')...

   One other prominent effect of bidirectional display is that some
   paragraphs of text need to be displayed starting at the right
   margin of the window---the so-called right-to-left, or R2L
   paragraphs.  R2L paragraphs are displayed with R2L glyph rows,
   which have their `reversed_p' flag set.  The bidi reordering engine
   produces characters in such rows starting from the character which
   should be the rightmost on display.  `PRODUCE_GLYPHS' then reverses
   the order, when it fills up the glyph row whose `reversed_p' flag
   is set, by prepending each new glyph to what is already there,
   instead of appending it.  When the glyph row is complete, the
   function `extend_face_to_end_of_line' fills the empty space to the
   left of the leftmost character with special glyphs, which will
   display as, well, empty.  On text terminals, these special glyphs
   are simply blank characters.  On graphics terminals, there's a
   single stretch glyph of a suitably computed width.  Both the blanks
   and the stretch glyph are given the face of the background of the
   line.  This way, the terminal-specific back-end can still draw the
   glyphs left to right, even for R2L lines.

   Bidirectional display and character compositions.

   Some scripts cannot be displayed by drawing each character
   individually, because adjacent characters change each other's shape
   on display.  For example, Arabic and Indic scripts belong to this
   category.

   Emacs display supports this by providing "character compositions",
   most of which is implemented in composite.c.  During the buffer
   scan that delivers characters to `PRODUCE_GLYPHS', if the next
   character to be delivered is a composed character, the iteration
   calls `composition_reseat_it' and `next_element_from_composition'.
   If they succeed to compose the character with one or more of the
   following characters, the whole sequence of characters that were
   composed is recorded in the `struct composition_it' object that is
   part of the buffer iterator.  The composed sequence could produce
   one or more font glyphs (called "grapheme clusters") on the screen.
   Each of these grapheme clusters is then delivered to
   `PRODUCE_GLYPHS' in the direction corresponding to the current bidi
   scan direction (recorded in the `scan_dir' member of the `struct
   bidi_it' object that is part of the iterator).  In particular, if
   the bidi iterator currently scans the buffer backwards, the
   grapheme clusters are delivered back to front.  This reorders the
   grapheme clusters as appropriate for the current bidi context.
   Note that this means that the grapheme clusters are always stored
   in the `LGSTRING' object (see composite.c) in the logical order.

   Moving an iterator in bidirectional text
   without producing glyphs.

   Note one important detail mentioned above: that the bidi reordering
   engine, driven by the iterator, produces characters in R2L rows
   starting at the character that will be the rightmost on display.
   As far as the iterator is concerned, the geometry of such rows is
   still left to right, i.e. the iterator "thinks" the first character
   is at the leftmost pixel position.  The iterator does not know that
   `PRODUCE_GLYPHS' reverses the order of the glyphs that the iterator
   delivers.  This is important when functions from the `move_it_*'
   family are used to get to certain screen position or to match
   screen coordinates with buffer coordinates: these functions use the
   iterator geometry, which is left to right even in R2L paragraphs.
   This works well with most callers of `move_it_*', because they need
   to get to a specific column, and columns are still numbered in the
   reading order, i.e. the rightmost character in a R2L paragraph is
   still column zero.  But some callers do not get well with this; a
   notable example is mouse clicks that need to find the character
   that corresponds to certain pixel coordinates.  See
   `buffer_posn_from_coords' in dispnew.c for how this is handled.  */

#include <config.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>

#include "lisp.h"
#include "atimer.h"
#include "composite.h"
#include "keyboard.h"
#include "sysstdio.h"
#include "systime.h"
#include "frame.h"
#include "window.h"
#include "termchar.h"
#include "dispextern.h"
#include "character.h"
#include "category.h"
#include "buffer.h"
#include "charset.h"
#include "indent.h"
#include "commands.h"
#include "keymap.h"
#include "disptab.h"
#include "termhooks.h"
#include "termopts.h"
#include "intervals.h"
#include "coding.h"
#include "region-cache.h"
#include "font.h"
#include "fontset.h"
#include "blockinput.h"
#include "xwidget.h"
#ifdef HAVE_WINDOW_SYSTEM
#include TERM_HEADER
#endif /* HAVE_WINDOW_SYSTEM */

#ifndef FRAME_OUTPUT_DATA
#define FRAME_OUTPUT_DATA(f) (NULL)
#endif

#define DISP_INFINITY 10000000

/* Holds the list (error).  */
static Lisp_Object list_of_error;

#ifdef HAVE_WINDOW_SYSTEM

/* Test if overflow newline into fringe.  Called with iterator IT
   at or past right window margin, and with IT->current_x set.  */

#define IT_OVERFLOW_NEWLINE_INTO_FRINGE(IT)		\
  (!NILP (Voverflow_newline_into_fringe)		\
   && FRAME_WINDOW_P ((IT)->f)				\
   && ((IT)->bidi_it.paragraph_dir == R2L		\
       ? (WINDOW_LEFT_FRINGE_WIDTH ((IT)->w) > 0)	\
       : (WINDOW_RIGHT_FRINGE_WIDTH ((IT)->w) > 0))	\
   && (IT)->current_x == (IT)->last_visible_x)

#else /* !HAVE_WINDOW_SYSTEM */
#define IT_OVERFLOW_NEWLINE_INTO_FRINGE(it) false
#endif /* HAVE_WINDOW_SYSTEM */

/* Test if the display element loaded in IT, or the underlying buffer
   or string character, is a space or a TAB character.  This is used
   to determine where word wrapping can occur.  */

#define IT_DISPLAYING_WHITESPACE(it)					\
  ((it->what == IT_CHARACTER && (it->c == ' ' || it->c == '\t'))	\
   || ((STRINGP (it->string)						\
	&& (SREF (it->string, IT_STRING_BYTEPOS (*it)) == ' '		\
	    || SREF (it->string, IT_STRING_BYTEPOS (*it)) == '\t'))	\
       || (it->s							\
	   && (it->s[IT_BYTEPOS (*it)] == ' '				\
	       || it->s[IT_BYTEPOS (*it)] == '\t'))			\
       || (IT_BYTEPOS (*it) < ZV_BYTE					\
	   && (*BYTE_POS_ADDR (IT_BYTEPOS (*it)) == ' '			\
	       || *BYTE_POS_ADDR (IT_BYTEPOS (*it)) == '\t'))))

/* These are the category sets we use.  They are defined by
   kinsoku.el and characters.el.  */
#define NOT_AT_EOL '<'
#define NOT_AT_BOL '>'
#define LINE_BREAKABLE '|'

static bool
it_char_has_category(struct it *it, int cat)
{
  int ch = 0;
  if (it->what == IT_CHARACTER)
    ch = it->c;
  else if (STRINGP (it->string))
    ch = SREF (it->string, IT_STRING_BYTEPOS (*it));
  else if (it->s)
    ch = it->s[IT_BYTEPOS (*it)];
  else if (IT_BYTEPOS (*it) < ZV_BYTE)
    ch = *BYTE_POS_ADDR (IT_BYTEPOS (*it));

  if (ch == 0)
    return false;
  else
    return CHAR_HAS_CATEGORY (ch, cat);
}

/* Return true if the current character allows wrapping before it.   */
static bool
char_can_wrap_before (struct it *it)
{
  if (!word_wrap_by_category)
    return !IT_DISPLAYING_WHITESPACE (it);

  /* For CJK (LTR) text in RTL paragraph, EOL and BOL are flipped.
     Because in RTL paragraph, each glyph is prepended to the last
     one, effectively drawing right to left.  */
  int not_at_bol;
  if (it->glyph_row && it->glyph_row->reversed_p)
    not_at_bol = NOT_AT_EOL;
  else
    not_at_bol = NOT_AT_BOL;
  /* You cannot wrap before a space or tab because that way you'll
     have space and tab at the beginning of next line.  */
  return (!IT_DISPLAYING_WHITESPACE (it)
	  /* Can be at BOL.  */
	  && !it_char_has_category (it, not_at_bol));
}

/* Return true if the current character allows wrapping after it.   */
static bool
char_can_wrap_after (struct it *it)
{
  if (!word_wrap_by_category)
    return IT_DISPLAYING_WHITESPACE (it);

  /* For CJK (LTR) text in RTL paragraph, EOL and BOL are flipped.
     Because in RTL paragraph, each glyph is prepended to the last
     one, effectively drawing right to left.  */
  int not_at_eol;
  if (it->glyph_row && it->glyph_row->reversed_p)
    not_at_eol = NOT_AT_BOL;
  else
    not_at_eol = NOT_AT_EOL;

  return (IT_DISPLAYING_WHITESPACE (it)
	  /* Can break after && can be at EOL.  */
	  || (it_char_has_category (it, LINE_BREAKABLE)
	      && !it_char_has_category (it, not_at_eol)));
}

#undef IT_DISPLAYING_WHITESPACE
#undef NOT_AT_EOL
#undef NOT_AT_BOL
#undef LINE_BREAKABLE

/* If all the conditions needed to print the fill column indicator are
   met, return the (nonnegative) column number, else return a negative
   value.  */
static int
fill_column_indicator_column (struct it *it, int char_width)
{
  if (display_fill_column_indicator
      && !it->w->pseudo_window_p
      && it->continuation_lines_width == 0
      && CHARACTERP (Vdisplay_fill_column_indicator_character))
    {
      Lisp_Object col = (EQ (Vdisplay_fill_column_indicator_column, Qt)
			 ? BVAR (current_buffer, fill_column)
			 : Vdisplay_fill_column_indicator_column);

      /* The stretch width needs to consider the latter
	 added glyph in append_space_for_newline.  */
      if (RANGED_FIXNUMP (0, col, INT_MAX))
	{
          int icol = XFIXNUM (col);
	  if (!INT_MULTIPLY_WRAPV (char_width, icol, &icol)
	      && !INT_ADD_WRAPV (it->lnum_pixel_width, icol, &icol))
	    return icol;
	}
    }
  return -1;
}

/* True means print newline to stdout before next mini-buffer message.  */

bool noninteractive_need_newline;

/* True means print newline to message log before next message.  */

static bool message_log_need_newline;

/* Three markers that message_dolog uses.
   It could allocate them itself, but that causes trouble
   in handling memory-full errors.  */
static Lisp_Object message_dolog_marker1;
static Lisp_Object message_dolog_marker2;
static Lisp_Object message_dolog_marker3;

/* The buffer position of the first character appearing entirely or
   partially on the line of the selected window which contains the
   cursor; <= 0 if not known.  Set by set_cursor_from_row, used for
   redisplay optimization in redisplay_internal.  */

static struct text_pos this_line_start_pos;

/* Number of characters past the end of the line above, including the
   terminating newline.  */

static struct text_pos this_line_end_pos;

/* The vertical positions and the height of this line.  */

static int this_line_vpos;
static int this_line_y;
static int this_line_pixel_height;

/* X position at which this display line starts.  Usually zero;
   negative if first character is partially visible.  */

static int this_line_start_x;

/* The smallest character position seen by move_it_* functions as they
   move across display lines.  Used to set MATRIX_ROW_START_CHARPOS of
   hscrolled lines, see display_line.  */

static struct text_pos this_line_min_pos;

/* Buffer that this_line_.* variables are referring to.  */

static struct buffer *this_line_buffer;

/* True if an overlay arrow has been displayed in this window.  */

static bool overlay_arrow_seen;

/* Vector containing glyphs for an ellipsis `...'.  */

static Lisp_Object default_invis_vector[3];

/* This is the window where the echo area message was displayed.  It
   is always a mini-buffer window, but it may not be the same window
   currently active as a mini-buffer.  */

Lisp_Object echo_area_window;

/* Stack of messages, which are pushed by push_message and popped and
   displayed by restore_message.  */

static Lisp_Object Vmessage_stack;

/* True means multibyte characters were enabled when the echo area
   message was specified.  */

static bool message_enable_multibyte;

/* At each redisplay cycle, we should refresh everything there is to refresh.
   To do that efficiently, we use many optimizations that try to make sure we
   don't waste too much time updating things that haven't changed.
   The coarsest such optimization is that, in the most common cases, we only
   look at the selected-window.

   To know whether other windows should be considered for redisplay, we use the
   variable windows_or_buffers_changed: as long as it is 0, it means that we
   have not noticed anything that should require updating anything else than
   the selected-window.  If it is set to REDISPLAY_SOME, it means that since
   last redisplay, some changes have been made which could impact other
   windows.  To know which ones need redisplay, every buffer, window, and frame
   has a `redisplay' bit, which (if true) means that this object needs to be
   redisplayed.  If windows_or_buffers_changed is 0, we know there's no point
   looking for those `redisplay' bits (actually, there might be some such bits
   set, but then only on objects which aren't displayed anyway).

   OTOH if it's non-zero we will have to loop through all windows and then
   check the `redisplay' bit of the corresponding window, frame, and buffer, in
   order to decide whether that window needs attention or not.  Note that we
   can't just look at the frame's redisplay bit to decide that the whole frame
   can be skipped, since even if the frame's redisplay bit is unset, some of
   its windows's redisplay bits may be set.

   Mostly for historical reasons, windows_or_buffers_changed can also take
   other non-zero values.  In that case, the precise value doesn't matter (it
   encodes the cause of the setting but is only used for debugging purposes),
   and what it means is that we shouldn't pay attention to any `redisplay' bits
   and we should simply try and redisplay every window out there.  */

int windows_or_buffers_changed;

/* Nonzero if we should redraw the mode lines on the next redisplay.
   Similarly to `windows_or_buffers_changed', if it has value REDISPLAY_SOME,
   then only redisplay the mode lines in those buffers/windows/frames where the
   `redisplay' bit has been set.
   For any other value, redisplay all mode lines (the number used is then only
   used to track down the cause for this full-redisplay).

   Since the frame title uses the same %-constructs as the mode line
   (except %c, %C, and %l), if this variable is non-zero, we also consider
   redisplaying the title of each frame, see gui_consider_frame_title.

   The `redisplay' bits are the same as those used for
   windows_or_buffers_changed, and setting windows_or_buffers_changed also
   causes recomputation of the mode lines of all those windows.  IOW this
   variable only has an effect if windows_or_buffers_changed is zero, in which
   case we should only need to redisplay the mode-line of those objects with
   a `redisplay' bit set but not the window's text content (tho we may still
   need to refresh the text content of the selected-window).  */

int update_mode_lines;

/* True after display_mode_line if %l was used and it displayed a
   line number.  */

static bool line_number_displayed;

/* Current, index 0, and last displayed echo area message.  Either
   buffers from echo_buffers, or nil to indicate no message.  */

Lisp_Object echo_area_buffer[2];

/* The buffers referenced from echo_area_buffer.  */

static Lisp_Object echo_buffer[2];

/* A vector saved used in with_area_buffer to reduce consing.  */

static Lisp_Object Vwith_echo_area_save_vector;

/* True means display_echo_area should display the last echo area
   message again.  Set by redisplay_preserve_echo_area.  */

static bool display_last_displayed_message_p;

/* True if echo area is being used by print; false if being used by
   message.  */

static bool message_buf_print;

/* Set to true in clear_message to make redisplay_internal aware
   of an emptied echo area.  */

static bool message_cleared_p;

/* A scratch glyph row with contents used for generating truncation
   glyphs and overlay-arrow glyphs.  */

#define MAX_SCRATCH_GLYPHS 100
static struct glyph_row scratch_glyph_row;
static struct glyph scratch_glyphs[MAX_SCRATCH_GLYPHS];

/* Ascent and height of the last line processed by move_it_to.  */

static int last_height;

/* True if there's a help-echo in the echo area.  */

bool help_echo_showing_p;

/* The maximum distance to look ahead for text properties.  Values
   that are too small let us call compute_char_face and similar
   functions too often which is expensive.  Values that are too large
   let us call compute_char_face and alike too often because we
   might not be interested in text properties that far away.  */

#define TEXT_PROP_DISTANCE_LIMIT 100

/* SAVE_IT and RESTORE_IT are called when we save a snapshot of the
   iterator state and later restore it.  This is needed because the
   bidi iterator on bidi.c keeps a stacked cache of its states, which
   is really a singleton.  When we use scratch iterator objects to
   move around the buffer, we can cause the bidi cache to be pushed or
   popped, and therefore we need to restore the cache state when we
   return to the original iterator.  */
#define SAVE_IT(ITCOPY, ITORIG, CACHE)		\
  do {						\
    if (CACHE)					\
      bidi_unshelve_cache (CACHE, true);	\
    ITCOPY = ITORIG;				\
    CACHE = bidi_shelve_cache ();		\
  } while (false)

#define RESTORE_IT(pITORIG, pITCOPY, CACHE)	\
  do {						\
    if (pITORIG != pITCOPY)			\
      *(pITORIG) = *(pITCOPY);			\
    bidi_unshelve_cache (CACHE, false);		\
    CACHE = NULL;				\
  } while (false)

/* Functions to mark elements as needing redisplay.  */
enum { REDISPLAY_SOME = 2};	/* Arbitrary choice.  */

static bool calc_pixel_width_or_height (double *, struct it *, Lisp_Object,
					struct font *, bool, int *);

void
redisplay_other_windows (void)
{
  if (!windows_or_buffers_changed)
    windows_or_buffers_changed = REDISPLAY_SOME;
}

void
wset_redisplay (struct window *w)
{
  /* Beware: selected_window can be nil during early stages.  */
  if (!BASE_EQ (make_lisp_ptr (w, Lisp_Vectorlike), selected_window))
    redisplay_other_windows ();
  w->redisplay = true;
}

void
fset_redisplay (struct frame *f)
{
  redisplay_other_windows ();
  f->redisplay = true;
}

void
bset_redisplay (struct buffer *b)
{
  int count = buffer_window_count (b);
  if (count > 0)
    {
      /* ... it's visible in other window than selected,  */
      if (count > 1 || b != XBUFFER (XWINDOW (selected_window)->contents))
	redisplay_other_windows ();
      /* Even if we don't set windows_or_buffers_changed, do set `redisplay'
	 so that if we later set windows_or_buffers_changed, this buffer will
	 not be omitted.  */
      b->text->redisplay = true;
    }
}

void
bset_update_mode_line (struct buffer *b)
{
  if (!update_mode_lines)
    update_mode_lines = REDISPLAY_SOME;
  b->text->redisplay = true;
}

void
wset_update_mode_line (struct window *w)
{
  w->update_mode_line = true;
  /* When a window's mode line needs to be updated, the window's frame's
     title may also need to be updated, but we don't need to worry about it
     here.  Instead, `gui_consider_frame_title' is automatically called
     whenever w->update_mode_line is set for that frame's selected window.
     But for this to work reliably, we have to make sure the window
     is considered, so we have to mark it for redisplay.  */
  wset_redisplay (w);
}

DEFUN ("set-buffer-redisplay", Fset_buffer_redisplay,
       Sset_buffer_redisplay, 4, 4, 0,
       doc: /* Mark the current buffer for redisplay.
This function may be passed to `add-variable-watcher'.  */)
  (Lisp_Object symbol, Lisp_Object newval, Lisp_Object op, Lisp_Object where)
{
  bset_update_mode_line (current_buffer);
  current_buffer->prevent_redisplay_optimizations_p = true;
  return Qnil;
}

/* redisplay_trace is for displaying traces of redisplay.
   If Emacs was compiled with GLYPH_DEBUG defined, the variable
   trace_redisplay_p can be set to a non-zero value in debugging
   sessions to activate traces.  */
#ifdef GLYPH_DEBUG
extern bool trace_redisplay_p EXTERNALLY_VISIBLE;
bool trace_redisplay_p;
#else
enum { trace_redisplay_p = false };
#endif
static void ATTRIBUTE_FORMAT_PRINTF (1, 2)
redisplay_trace (char const *fmt, ...)
{
  if (trace_redisplay_p)
    {
      va_list ap;
      va_start (ap, fmt);
      vprintf (fmt, ap);
      va_end (ap);
    }
}

#ifdef DEBUG_TRACE_MOVE
extern bool trace_move EXTERNALLY_VISIBLE;
bool trace_move;
#else
enum { trace_move = false };
#endif
static void ATTRIBUTE_FORMAT_PRINTF (1, 2)
move_trace (char const *fmt, ...)
{
  if (trace_move)
    {
      va_list ap;
      va_start (ap, fmt);
      vprintf (fmt, ap);
      va_end (ap);
    }
}

/* Buffer being redisplayed -- for redisplay_window_error.  */

static struct buffer *displayed_buffer;

/* Value returned from text property handlers (see below).  */

enum prop_handled
{
  HANDLED_NORMALLY,
  HANDLED_RECOMPUTE_PROPS,
  HANDLED_OVERLAY_STRING_CONSUMED,
  HANDLED_RETURN
};

/* A description of text properties that redisplay is interested
   in.  */

struct props
{
  /* The symbol index of the name of the property.  */
  short name;

  /* A unique index for the property.  */
  enum prop_idx idx;

  /* A handler function called to set up iterator IT from the property
     at IT's current position.  Value is used to steer handle_stop.  */
  enum prop_handled (*handler) (struct it *it);
};

static enum prop_handled handle_face_prop (struct it *);
static enum prop_handled handle_invisible_prop (struct it *);
static enum prop_handled handle_display_prop (struct it *);
static enum prop_handled handle_composition_prop (struct it *);
static enum prop_handled handle_overlay_change (struct it *);
static enum prop_handled handle_fontified_prop (struct it *);

/* Properties handled by iterators.  */

static struct props it_props[] =
{
  {SYMBOL_INDEX (Qfontified),	FONTIFIED_PROP_IDX,	handle_fontified_prop},
  /* Handle `face' before `display' because some sub-properties of
     `display' need to know the face.  */
  {SYMBOL_INDEX (Qface),	FACE_PROP_IDX,		handle_face_prop},
  {SYMBOL_INDEX (Qdisplay),	DISPLAY_PROP_IDX,	handle_display_prop},
  {SYMBOL_INDEX (Qinvisible),	INVISIBLE_PROP_IDX,	handle_invisible_prop},
  {SYMBOL_INDEX (Qcomposition),	COMPOSITION_PROP_IDX, handle_composition_prop},
  {0,				0,			NULL}
};

/* Enumeration returned by some move_it_.* functions internally.  */

enum move_it_result
{
  /* Not used.  Undefined value.  */
  MOVE_UNDEFINED,

  /* Move ended at the requested buffer position or ZV.  */
  MOVE_POS_MATCH_OR_ZV,

  /* Move ended at the requested X pixel position.  */
  MOVE_X_REACHED,

  /* Move within a line ended at the end of a line that must be
     continued.  */
  MOVE_LINE_CONTINUED,

  /* Move within a line ended at the end of a line that would
     be displayed truncated.  */
  MOVE_LINE_TRUNCATED,

  /* Move within a line ended at a line end.  */
  MOVE_NEWLINE_OR_CR
};

/* This counter is used to clear the face cache every once in a while
   in redisplay_internal.  It is incremented for each redisplay.
   Every CLEAR_FACE_CACHE_COUNT full redisplays, the face cache is
   cleared.  */

#define CLEAR_FACE_CACHE_COUNT	500
static int clear_face_cache_count;

/* Similarly for the image cache.  */

#ifdef HAVE_WINDOW_SYSTEM
#define CLEAR_IMAGE_CACHE_COUNT	101
static int clear_image_cache_count;

/* Null glyph slice */
static struct glyph_slice null_glyph_slice = { 0, 0, 0, 0 };
#endif

/* True while redisplay_internal is in progress.  */

bool redisplaying_p;

/* True while some display-engine code is working on layout of some
   window.

   WARNING: Use sparingly, preferably only in top level of commands
   and important functions, because using it in nested calls might
   reset the flag when the inner call returns, behind the back of
   the callers.  */
bool display_working_on_window_p;

/* If a string, XTread_socket generates an event to display that string.
   (The display is done in read_char.)  */

Lisp_Object help_echo_string;
Lisp_Object help_echo_window;
Lisp_Object help_echo_object;
ptrdiff_t help_echo_pos;

/* Temporary variable for XTread_socket.  */

Lisp_Object previous_help_echo_string;

/* Platform-independent portion of hourglass implementation.  */

#ifdef HAVE_WINDOW_SYSTEM

/* True means an hourglass cursor is currently shown.  */
static bool hourglass_shown_p;

/* If non-null, an asynchronous timer that, when it expires, displays
   an hourglass cursor on all frames.  */
static struct atimer *hourglass_atimer;

#endif /* HAVE_WINDOW_SYSTEM */

/* Default number of seconds to wait before displaying an hourglass
   cursor.  */
#define DEFAULT_HOURGLASS_DELAY 1

#ifdef HAVE_WINDOW_SYSTEM

/* Default pixel width of `thin-space' display method.  */
#define THIN_SPACE_WIDTH 1

#endif /* HAVE_WINDOW_SYSTEM */

/* Function prototypes.  */

static void setup_for_ellipsis (struct it *, int);
static void set_iterator_to_next (struct it *, bool);
static void mark_window_display_accurate_1 (struct window *, bool);
static bool row_for_charpos_p (struct glyph_row *, ptrdiff_t);
static bool cursor_row_p (struct glyph_row *);
static int redisplay_mode_lines (Lisp_Object, bool);

static void handle_line_prefix (struct it *);

static void handle_stop_backwards (struct it *, ptrdiff_t);
static void unwind_with_echo_area_buffer (Lisp_Object);
static Lisp_Object with_echo_area_buffer_unwind_data (struct window *);
static bool current_message_1 (void *, Lisp_Object);
static bool truncate_message_1 (void *, Lisp_Object);
static void set_message (Lisp_Object);
static bool set_message_1 (void *, Lisp_Object);
static bool display_echo_area_1 (void *, Lisp_Object);
static bool resize_mini_window_1 (void *, Lisp_Object);
static void unwind_redisplay (void);
static void extend_face_to_end_of_line (struct it *);
static intmax_t message_log_check_duplicate (ptrdiff_t, ptrdiff_t);
static void push_it (struct it *, struct text_pos *);
static void iterate_out_of_display_property (struct it *);
static void pop_it (struct it *);
static void redisplay_internal (void);
static void echo_area_display (bool);
static void block_buffer_flips (void);
static void unblock_buffer_flips (void);
static void redisplay_windows (Lisp_Object);
static void redisplay_window (Lisp_Object, bool);
static Lisp_Object redisplay_window_error (Lisp_Object);
static Lisp_Object redisplay_window_0 (Lisp_Object);
static Lisp_Object redisplay_window_1 (Lisp_Object);
static bool set_cursor_from_row (struct window *, struct glyph_row *,
				 struct glyph_matrix *, ptrdiff_t, ptrdiff_t,
				 int, int);
static bool cursor_row_fully_visible_p (struct window *, bool, bool, bool);
static bool update_menu_bar (struct frame *, bool, bool);
static bool try_window_reusing_current_matrix (struct window *);
static int try_window_id (struct window *);
static void maybe_produce_line_number (struct it *);
static bool should_produce_line_number (struct it *);
static bool display_line (struct it *, int);
static int display_mode_lines (struct window *);
static int display_mode_line (struct window *, enum face_id, Lisp_Object);
static int display_mode_element (struct it *, int, int, int, Lisp_Object,
				 Lisp_Object, bool);
static int store_mode_line_string (const char *, Lisp_Object, bool, int, int,
				   Lisp_Object);
static const char *decode_mode_spec (struct window *, int, int, Lisp_Object *);
static void display_menu_bar (struct window *);
static void display_tab_bar (struct window *);
static void update_tab_bar (struct frame *, bool);
static ptrdiff_t display_count_lines (ptrdiff_t, ptrdiff_t, ptrdiff_t,
				      ptrdiff_t *);
static void pint2str (register char *, register int, register ptrdiff_t);

static int display_string (const char *, Lisp_Object, Lisp_Object,
                           ptrdiff_t, ptrdiff_t, struct it *, int, int, int,
			   int);
static void compute_line_metrics (struct it *);
static bool get_overlay_strings (struct it *, ptrdiff_t);
static bool get_overlay_strings_1 (struct it *, ptrdiff_t, bool);
static void next_overlay_string (struct it *);
static void reseat (struct it *, struct text_pos, bool);
static void reseat_1 (struct it *, struct text_pos, bool);
static bool next_element_from_display_vector (struct it *);
static bool next_element_from_string (struct it *);
static bool next_element_from_c_string (struct it *);
static bool next_element_from_buffer (struct it *);
static bool next_element_from_composition (struct it *);
static bool next_element_from_image (struct it *);
static bool next_element_from_stretch (struct it *);
static bool next_element_from_xwidget (struct it *);
static void load_overlay_strings (struct it *, ptrdiff_t);
static bool get_next_display_element (struct it *);
static enum move_it_result
       move_it_in_display_line_to (struct it *, ptrdiff_t, int,
				   enum move_operation_enum);
static void get_visually_first_element (struct it *);
static void compute_stop_pos (struct it *);
static int face_before_or_after_it_pos (struct it *, bool);
static int handle_display_spec (struct it *, Lisp_Object, Lisp_Object,
				Lisp_Object, struct text_pos *, ptrdiff_t, bool);
static int handle_single_display_spec (struct it *, Lisp_Object, Lisp_Object,
				       Lisp_Object, struct text_pos *,
				       ptrdiff_t, int, bool, bool);
static int underlying_face_id (const struct it *);

#define face_before_it_pos(IT) face_before_or_after_it_pos (IT, true)
#define face_after_it_pos(IT)  face_before_or_after_it_pos (IT, false)

#ifdef HAVE_WINDOW_SYSTEM

static void update_tool_bar (struct frame *, bool);
static void gui_draw_bottom_divider (struct window *w);
static void notice_overwritten_cursor (struct window *,
                                       enum glyph_row_area,
                                       int, int, int, int);
static int  normal_char_height (struct font *, int);
static void normal_char_ascent_descent (struct font *, int, int *, int *);

static void append_stretch_glyph (struct it *, Lisp_Object,
                                  int, int, int);

static Lisp_Object get_it_property (struct it *, Lisp_Object);
static Lisp_Object calc_line_height_property (struct it *, Lisp_Object,
					      struct font *, int, bool);
static int adjust_glyph_width_for_mouse_face (struct glyph *,
					      struct glyph_row *,
					      struct window *, struct face *,
					      struct face *);
static void get_cursor_offset_for_mouse_face (struct window *w,
					      struct glyph_row *row,
					      int *offset);
#endif /* HAVE_WINDOW_SYSTEM */

static void produce_special_glyphs (struct it *, enum display_element_type);
static void show_mouse_face (Mouse_HLInfo *, enum draw_glyphs_face);
static bool coords_in_mouse_face_p (struct window *, int, int);
static void reset_box_start_end_flags (struct it *);



/***********************************************************************
		      Window display dimensions
 ***********************************************************************/

/* Return the bottom boundary y-position for text lines in window W.
   This is the first y position at which a line cannot start.
   It is relative to the top of the window.

   This is the height of W minus the height of a mode line, if any.  */

int
window_text_bottom_y (struct window *w)
{
  int height = WINDOW_PIXEL_HEIGHT (w);

  height -= WINDOW_BOTTOM_DIVIDER_WIDTH (w);

  if (window_wants_mode_line (w))
    height -= CURRENT_MODE_LINE_HEIGHT (w);

  height -= WINDOW_SCROLL_BAR_AREA_HEIGHT (w);

  return height;
}

/* Return the pixel width of display area AREA of window W.
   ANY_AREA means return the total width of W, not including
   fringes to the left and right of the window.  */

int
window_box_width (struct window *w, enum glyph_row_area area)
{
  int width = w->pixel_width;

  if (!w->pseudo_window_p)
    {
      width -= WINDOW_SCROLL_BAR_AREA_WIDTH (w);
      width -= WINDOW_RIGHT_DIVIDER_WIDTH (w);

      if (area == TEXT_AREA)
	width -= (WINDOW_MARGINS_WIDTH (w)
		   + WINDOW_FRINGES_WIDTH (w));
      else if (area == LEFT_MARGIN_AREA)
	width = WINDOW_LEFT_MARGIN_WIDTH (w);
      else if (area == RIGHT_MARGIN_AREA)
	width = WINDOW_RIGHT_MARGIN_WIDTH (w);
    }

  /* With wide margins, fringes, etc. we might end up with a negative
     width, correct that here.  */
  return max (0, width);
}


/* Return the pixel height of the display area of window W, not
   including mode lines of W, if any.  */

int
window_box_height (struct window *w)
{
  struct frame *f = XFRAME (w->frame);
  int height = WINDOW_PIXEL_HEIGHT (w);

  eassert (height >= 0);

  height -= WINDOW_BOTTOM_DIVIDER_WIDTH (w);
  height -= WINDOW_SCROLL_BAR_AREA_HEIGHT (w);

  /* Note: the code below that determines the mode-line/header-line/tab-line
     height is essentially the same as that contained in the macro
     CURRENT_{MODE,HEADER,TAB}_LINE_HEIGHT, except that it checks whether
     the appropriate glyph row has its `mode_line_p' flag set, and if
     it doesn't, uses estimate_mode_line_height instead.  */

  if (window_wants_mode_line (w))
    {
      if (w->mode_line_height >= 0)
	height -= w->mode_line_height;
      else
	{
	  struct glyph_row *ml_row
	    = (w->current_matrix && w->current_matrix->rows
	       ? MATRIX_MODE_LINE_ROW (w->current_matrix)
	       : 0);
	  if (ml_row && ml_row->mode_line_p)
	    height -= ml_row->height;
	  else
	    height -= estimate_mode_line_height
	      (f, CURRENT_MODE_LINE_ACTIVE_FACE_ID (w));
	}
    }

  if (window_wants_tab_line (w))
    {
      if (w->tab_line_height >= 0)
	height -= w->tab_line_height;
      else
	{
	  struct glyph_row *tl_row
	    = (w->current_matrix && w->current_matrix->rows
	       ? MATRIX_TAB_LINE_ROW (w->current_matrix)
	       : 0);
	  if (tl_row && tl_row->mode_line_p)
	    height -= tl_row->height;
	  else
	    height -= estimate_mode_line_height (f, TAB_LINE_FACE_ID);
	}
    }

  if (window_wants_header_line (w))
    {
      if (w->header_line_height >= 0)
	height -= w->header_line_height;
      else
	{
	  struct glyph_row *hl_row
	    = (w->current_matrix && w->current_matrix->rows
	       ? MATRIX_HEADER_LINE_ROW (w->current_matrix)
	       : 0);
	  if (hl_row && hl_row->mode_line_p)
	    height -= hl_row->height;
	  else
	    height -= estimate_mode_line_height (f, HEADER_LINE_FACE_ID);
	}
    }

  /* With a very small font and a mode-line that's taller than
     default, we might end up with a negative height.  */
  return max (0, height);
}

/* Return the window-relative coordinate of the left edge of display
   area AREA of window W.  ANY_AREA means return the left edge of the
   whole window, to the right of the left fringe of W.  */

int
window_box_left_offset (struct window *w, enum glyph_row_area area)
{
  int x;

  if (w->pseudo_window_p)
    return 0;

  x = WINDOW_LEFT_SCROLL_BAR_AREA_WIDTH (w);

  if (area == TEXT_AREA)
    x += (WINDOW_LEFT_FRINGE_WIDTH (w)
	  + window_box_width (w, LEFT_MARGIN_AREA));
  else if (area == RIGHT_MARGIN_AREA)
    x += (WINDOW_LEFT_FRINGE_WIDTH (w)
	  + window_box_width (w, LEFT_MARGIN_AREA)
	  + window_box_width (w, TEXT_AREA)
	  + (WINDOW_HAS_FRINGES_OUTSIDE_MARGINS (w)
	     ? 0
	     : WINDOW_RIGHT_FRINGE_WIDTH (w)));
  else if (area == LEFT_MARGIN_AREA
	   && WINDOW_HAS_FRINGES_OUTSIDE_MARGINS (w))
    x += WINDOW_LEFT_FRINGE_WIDTH (w);

  /* Don't return more than the window's pixel width.  */
  return min (x, w->pixel_width);
}


/* Return the window-relative coordinate of the right edge of display
   area AREA of window W.  ANY_AREA means return the right edge of the
   whole window, to the left of the right fringe of W.  */

static int
window_box_right_offset (struct window *w, enum glyph_row_area area)
{
  /* Don't return more than the window's pixel width.  */
  return min (window_box_left_offset (w, area) + window_box_width (w, area),
	      w->pixel_width);
}

/* Return the frame-relative coordinate of the left edge of display
   area AREA of window W.  ANY_AREA means return the left edge of the
   whole window, to the right of the left fringe of W.  */

int
window_box_left (struct window *w, enum glyph_row_area area)
{
  struct frame *f = XFRAME (w->frame);
  int x;

  if (w->pseudo_window_p)
    return FRAME_INTERNAL_BORDER_WIDTH (f);

  x = (WINDOW_LEFT_EDGE_X (w)
       + window_box_left_offset (w, area));

  return x;
}


/* Return the frame-relative coordinate of the right edge of display
   area AREA of window W.  ANY_AREA means return the right edge of the
   whole window, to the left of the right fringe of W.  */

int
window_box_right (struct window *w, enum glyph_row_area area)
{
  return window_box_left (w, area) + window_box_width (w, area);
}

/* Get the bounding box of the display area AREA of window W, without
   mode lines, in frame-relative coordinates.  ANY_AREA means the
   whole window, not including the left and right fringes of
   the window.  Return in *BOX_X and *BOX_Y the frame-relative pixel
   coordinates of the upper-left corner of the box.  Return in
   *BOX_WIDTH, and *BOX_HEIGHT the pixel width and height of the box.  */

void
window_box (struct window *w, enum glyph_row_area area, int *box_x,
	    int *box_y, int *box_width, int *box_height)
{
  if (box_width)
    *box_width = window_box_width (w, area);
  if (box_height)
    *box_height = window_box_height (w);
  if (box_x)
    *box_x = window_box_left (w, area);
  if (box_y)
    {
      *box_y = WINDOW_TOP_EDGE_Y (w);
      if (window_wants_tab_line (w))
	*box_y += CURRENT_TAB_LINE_HEIGHT (w);
      if (window_wants_header_line (w))
	*box_y += CURRENT_HEADER_LINE_HEIGHT (w);
    }
}

#ifdef HAVE_WINDOW_SYSTEM

/* Get the bounding box of the display area AREA of window W, without
   mode lines and both fringes of the window.  Return in *TOP_LEFT_X
   and TOP_LEFT_Y the frame-relative pixel coordinates of the
   upper-left corner of the box.  Return in *BOTTOM_RIGHT_X, and
   *BOTTOM_RIGHT_Y the coordinates of the bottom-right corner of the
   box.  */

static void
window_box_edges (struct window *w, int *top_left_x, int *top_left_y,
		  int *bottom_right_x, int *bottom_right_y)
{
  window_box (w, ANY_AREA, top_left_x, top_left_y,
	      bottom_right_x, bottom_right_y);
  *bottom_right_x += *top_left_x;
  *bottom_right_y += *top_left_y;
}

#endif /* HAVE_WINDOW_SYSTEM */

/***********************************************************************
			      Utilities
 ***********************************************************************/

/* Return the bottom y-position of the line the iterator IT is in.
   This can modify IT's settings.  */

int
line_bottom_y (struct it *it)
{
  int line_height = it->max_ascent + it->max_descent;
  int line_top_y = it->current_y;

  if (line_height == 0)
    {
      if (last_height)
	line_height = last_height;
      else if (IT_CHARPOS (*it) < ZV)
	{
	  move_it_by_lines (it, 1);
	  line_height = (it->max_ascent || it->max_descent
			 ? it->max_ascent + it->max_descent
			 : last_height);
	}
      else
	{
	  struct glyph_row *row = it->glyph_row;

	  /* Use the default character height.  */
	  it->glyph_row = NULL;
	  it->what = IT_CHARACTER;
	  it->c = ' ';
	  it->len = 1;
	  PRODUCE_GLYPHS (it);
	  line_height = it->ascent + it->descent;
	  it->glyph_row = row;
	}
    }

  return line_top_y + line_height;
}

DEFUN ("line-pixel-height", Fline_pixel_height,
       Sline_pixel_height, 0, 0, 0,
       doc: /* Return height in pixels of text line in the selected window.

Value is the height in pixels of the line at point.  */)
  (void)
{
  struct it it;
  struct text_pos pt;
  struct window *w = XWINDOW (selected_window);
  struct buffer *old_buffer = NULL;
  Lisp_Object result;

  if (XBUFFER (w->contents) != current_buffer)
    {
      old_buffer = current_buffer;
      set_buffer_internal_1 (XBUFFER (w->contents));
    }
  SET_TEXT_POS (pt, PT, PT_BYTE);
  void *itdata = bidi_shelve_cache ();
  start_display (&it, w, pt);
  /* Start from the beginning of the screen line, to make sure we
     traverse all of its display elements, and thus capture the
     correct metrics.  */
  move_it_by_lines (&it, 0);
  it.vpos = it.current_y = 0;
  last_height = 0;
  result = make_fixnum (line_bottom_y (&it));
  if (old_buffer)
    set_buffer_internal_1 (old_buffer);

  bidi_unshelve_cache (itdata, false);
  return result;
}

/* Return the default pixel height of text lines in window W.  The
   value is the canonical height of the W frame's default font, plus
   any extra space required by the line-spacing variable or frame
   parameter.

   Implementation note: this ignores any line-spacing text properties
   put on the newline characters.  This is because those properties
   only affect the _screen_ line ending in the newline (i.e., in a
   continued line, only the last screen line will be affected), which
   means only a small number of lines in a buffer can ever use this
   feature.  Since this function is used to compute the default pixel
   equivalent of text lines in a window, we can safely ignore those
   few lines.  For the same reasons, we ignore the line-height
   properties.  */
int
default_line_pixel_height (struct window *w)
{
  struct frame *f = WINDOW_XFRAME (w);
  int height = FRAME_LINE_HEIGHT (f);

  if (!FRAME_INITIAL_P (f) && BUFFERP (w->contents))
    {
      struct buffer *b = XBUFFER (w->contents);
      Lisp_Object val = BVAR (b, extra_line_spacing);

      if (NILP (val))
	val = BVAR (&buffer_defaults, extra_line_spacing);
      if (!NILP (val))
	{
	  if (RANGED_FIXNUMP (0, val, INT_MAX))
	    height += XFIXNAT (val);
	  else if (FLOATP (val))
	    {
	      int addon = XFLOAT_DATA (val) * height + 0.5;

	      if (addon >= 0)
		height += addon;
	    }
	}
      else
	height += f->extra_line_spacing;
    }

  return height;
}

/* Subroutine of pos_visible_p below.  Extracts a display string, if
   any, from the display spec given as its argument.  */
static Lisp_Object
string_from_display_spec (Lisp_Object spec)
{
  if (VECTORP (spec))
    {
      for (ptrdiff_t i = 0; i < ASIZE (spec); i++)
	if (STRINGP (AREF (spec, i)))
	  return AREF (spec, i);
    }
  else
    {
      for (; CONSP (spec); spec = XCDR (spec))
	if (STRINGP (XCAR (spec)))
	  return XCAR (spec);
    }
  return spec;
}


/* Limit insanely large values of W->hscroll on frame F to the largest
   value that will still prevent first_visible_x and last_visible_x of
   'struct it' from overflowing an int.  */
static int
window_hscroll_limited (struct window *w, struct frame *f)
{
  ptrdiff_t window_hscroll = w->hscroll;
  int window_text_width = window_box_width (w, TEXT_AREA);
  int colwidth = FRAME_COLUMN_WIDTH (f);

  if (window_hscroll > (INT_MAX - window_text_width) / colwidth - 1)
    window_hscroll = (INT_MAX - window_text_width) / colwidth - 1;

  return window_hscroll;
}

/* Reset the box-face start and end flags in the iterator.  This is
   called after producing glyphs, such that we reset these flags only
   after producing a glyph with the flag set.  */

static void
reset_box_start_end_flags (struct it *it)
{
  /* Don't reset if we've drawn the glyph in the display margins --
     those don't count as "produced glyphs".  */
  if (it->area == TEXT_AREA
      /* Don't reset if we displayed a fringe bitmap.  */
      && !(it->what == IT_IMAGE && it->image_id < 0))
    {
      /* Don't reset if the face is not a box face: that might mean we
	 are iterating some overlay or display string, and the first
	 character to have the box face is yet to be seen, when we pop
	 the iterator stack. */
      if (it->face_box_p)
	it->start_of_box_run_p = false;
      it->end_of_box_run_p = false;
    }
}

/* Return true if position CHARPOS is visible in window W.
   CHARPOS < 0 means return info about WINDOW_END position.
   If visible, set *X and *Y to pixel coordinates of top left corner.
   Set *RTOP and *RBOT to pixel height of an invisible area of glyph at POS.
   Set *ROWH and *VPOS to row's visible height and VPOS (row number).  */

bool
pos_visible_p (struct window *w, ptrdiff_t charpos, int *x, int *y,
	       int *rtop, int *rbot, int *rowh, int *vpos)
{
  struct it it;
  void *itdata = bidi_shelve_cache ();
  struct text_pos top;
  bool visible_p = false;
  struct buffer *old_buffer = NULL;
  bool r2l = false;

  if (FRAME_INITIAL_P (XFRAME (WINDOW_FRAME (w))))
    return visible_p;

  if (XBUFFER (w->contents) != current_buffer)
    {
      old_buffer = current_buffer;
      set_buffer_internal_1 (XBUFFER (w->contents));
    }

  SET_TEXT_POS_FROM_MARKER (top, w->start);
  /* Scrolling a minibuffer window via scroll bar when the echo area
     shows long text sometimes resets the minibuffer contents behind
     our backs.  Also, someone might narrow-to-region and immediately
     call a scroll function.  */
  if (CHARPOS (top) > ZV || CHARPOS (top) < BEGV)
    SET_TEXT_POS (top, BEGV, BEGV_BYTE);

  /* If the top of the window is after CHARPOS, the latter is surely
     not visible.  */
  if (charpos >= 0 && CHARPOS (top) > charpos)
    return visible_p;

  /* Some Lisp hook could call us in the middle of redisplaying this
     very window.  If, by some bad luck, we are retrying redisplay
     because we found that the mode-line height and/or tab/header-line
     height needs to be updated, the assignment of mode_line_height
     and header_line_height below could disrupt that, due to the
     selected/nonselected window dance during mode-line display, and
     we could infloop.  Avoid that.  */
  int prev_mode_line_height = w->mode_line_height;
  int prev_header_line_height = w->header_line_height;
  int prev_tab_line_height = w->tab_line_height;
  /* Compute exact mode line heights.  */
  if (window_wants_mode_line (w))
    {
      Lisp_Object window_mode_line_format
	= window_parameter (w, Qmode_line_format);

      w->mode_line_height
	= display_mode_line (w, CURRENT_MODE_LINE_ACTIVE_FACE_ID (w),
			     NILP (window_mode_line_format)
			     ? BVAR (current_buffer, mode_line_format)
			     : window_mode_line_format);
    }

  if (window_wants_tab_line (w))
    {
      Lisp_Object window_tab_line_format
	= window_parameter (w, Qtab_line_format);

      w->tab_line_height
	= display_mode_line (w, TAB_LINE_FACE_ID,
			     NILP (window_tab_line_format)
			     ? BVAR (current_buffer, tab_line_format)
			     : window_tab_line_format);
    }

  if (window_wants_header_line (w))
    {
      Lisp_Object window_header_line_format
	= window_parameter (w, Qheader_line_format);

      w->header_line_height
	= display_mode_line (w, HEADER_LINE_FACE_ID,
			     NILP (window_header_line_format)
			     ? BVAR (current_buffer, header_line_format)
			     : window_header_line_format);
    }

  start_display (&it, w, top);
  move_it_to (&it, charpos, -1, it.last_visible_y - 1, -1,
	      (charpos >= 0 ? MOVE_TO_POS : 0) | MOVE_TO_Y);

  /* Adjust for line numbers, if CHARPOS is at or beyond first_visible_x,
     but we didn't yet produce the line-number glyphs.  */
  if (!NILP (Vdisplay_line_numbers)
      && it.current_x >= it.first_visible_x
      && IT_CHARPOS (it) == charpos
      && !it.line_number_produced_p)
    {
      /* If the pixel width of line numbers was not yet known, compute
	 it now.  This usually happens in the first display line of a
	 window.  */
      if (!it.lnum_pixel_width)
	{
	  struct it it2;
	  void *it2data = NULL;

	  SAVE_IT (it2, it, it2data);
	  move_it_by_lines (&it, 1);
	  it2.lnum_pixel_width = it.lnum_pixel_width;
	  RESTORE_IT (&it, &it2, it2data);
	}
      it.current_x += it.lnum_pixel_width;
    }

  if (charpos >= 0
      && (((!it.bidi_p || it.bidi_it.scan_dir != -1)
	   && IT_CHARPOS (it) >= charpos)
	  /* When scanning backwards under bidi iteration, move_it_to
	     stops at or _before_ CHARPOS, because it stops at or to
	     the _right_ of the character at CHARPOS.  */
	  || (it.bidi_p && it.bidi_it.scan_dir == -1
	      && IT_CHARPOS (it) <= charpos)))
    {
      /* We have reached CHARPOS, or passed it.  How the call to
	 move_it_to can overshoot: (i) If CHARPOS is on invisible text
	 or covered by a display property, move_it_to stops at the end
	 of the invisible text, to the right of CHARPOS.  (ii) If
	 CHARPOS is in a display vector, move_it_to stops on its last
	 glyph.  */
      int top_x = it.current_x;
      int top_y = it.current_y;
      int window_top_y = WINDOW_TAB_LINE_HEIGHT (w) + WINDOW_HEADER_LINE_HEIGHT (w);
      int bottom_y;
      struct it save_it;
      void *save_it_data = NULL;

      /* Calling line_bottom_y may change it.method, it.position, etc.  */
      SAVE_IT (save_it, it, save_it_data);
      last_height = 0;
      bottom_y = line_bottom_y (&it);
      if (top_y < window_top_y)
	visible_p = bottom_y > window_top_y;
      else if (top_y < it.last_visible_y)
	visible_p = true;
      if (bottom_y >= it.last_visible_y
	  && it.bidi_p && it.bidi_it.scan_dir == -1
	  && IT_CHARPOS (it) < charpos)
	{
	  /* When the last line of the window is scanned backwards
	     under bidi iteration, we could be duped into thinking
	     that we have passed CHARPOS, when in fact move_it_to
	     simply stopped short of CHARPOS because it reached
	     last_visible_y.  To see if that's what happened, we call
	     move_it_to again with a slightly larger vertical limit,
	     and see if it actually moved vertically; if it did, we
	     didn't really reach CHARPOS, which is beyond window end.  */
	  /* Why 10? because we don't know how many canonical lines
	     will the height of the next line(s) be.  So we guess.  */
	  int ten_more_lines = 10 * default_line_pixel_height (w);

	  move_it_to (&it, charpos, -1, bottom_y + ten_more_lines, -1,
		      MOVE_TO_POS | MOVE_TO_Y);
	  if (it.current_y > top_y)
	    visible_p = false;

	}
      RESTORE_IT (&it, &save_it, save_it_data);
      if (visible_p)
	{
	  if (it.method == GET_FROM_DISPLAY_VECTOR)
	    {
	      /* We stopped on the last glyph of a display vector.
		 Try and recompute.  Hack alert!  */
	      if (charpos < 2 || top.charpos >= charpos)
		top_x = it.glyph_row->x;
	      else
		{
		  struct it it2, it2_prev;
		  /* The idea is to get to the previous buffer
		     position, consume the character there, and use
		     the pixel coordinates we get after that.  But if
		     the previous buffer position is also displayed
		     from a display vector, we need to consume all of
		     the glyphs from that display vector.  */
		  start_display (&it2, w, top);
		  it2.glyph_row = NULL;
		  move_it_to (&it2, charpos - 1, -1, -1, -1, MOVE_TO_POS);
		  /* If we didn't get to CHARPOS - 1, there's some
		     replacing display property at that position, and
		     we stopped after it.  That is exactly the place
		     whose coordinates we want.  */
		  if (IT_CHARPOS (it2) != charpos - 1)
		    it2_prev = it2;
		  else
		    {
		      /* Iterate until we get out of the display
			 vector that displays the character at
			 CHARPOS - 1.  */
		      do {
			get_next_display_element (&it2);
			PRODUCE_GLYPHS (&it2);
			it2_prev = it2;
			set_iterator_to_next (&it2, true);
		      } while (it2.method == GET_FROM_DISPLAY_VECTOR
			       && IT_CHARPOS (it2) < charpos);
		    }
		  if (ITERATOR_AT_END_OF_LINE_P (&it2_prev)
		      || it2_prev.current_x > it2_prev.last_visible_x)
		    top_x = it.glyph_row->x;
		  else
		    {
		      top_x = it2_prev.current_x;
		      top_y = it2_prev.current_y;
		    }
		}
	    }
	  else if (IT_CHARPOS (it) != charpos)
	    {
	      Lisp_Object cpos = make_fixnum (charpos);
	      Lisp_Object spec = Fget_char_property (cpos, Qdisplay, Qnil);
	      Lisp_Object string = string_from_display_spec (spec);
	      struct text_pos tpos;
	      bool newline_in_string
		= (STRINGP (string)
		   && memchr (SDATA (string), '\n', SBYTES (string)));

	      SET_TEXT_POS (tpos, charpos, CHAR_TO_BYTE (charpos));
	      bool replacing_spec_p
		= (!NILP (spec)
		   && handle_display_spec (NULL, spec, Qnil, Qnil, &tpos,
					   charpos, FRAME_WINDOW_P (it.f)));
	      /* The tricky code below is needed because there's a
		 discrepancy between move_it_to and how we set cursor
		 when PT is at the beginning of a portion of text
		 covered by a display property or an overlay with a
		 display property, or the display line ends in a
		 newline from a display string.  move_it_to will stop
		 _after_ such display strings, whereas
		 set_cursor_from_row conspires with cursor_row_p to
		 place the cursor on the first glyph produced from the
		 display string.  */

	      /* We have overshoot PT because it is covered by a
		 display property that replaces the text it covers.
		 If the string includes embedded newlines, we are also
		 in the wrong display line.  Backtrack to the correct
		 line, where the display property begins.  */
	      if (replacing_spec_p)
		{
		  Lisp_Object startpos, endpos;
		  EMACS_INT start, end;
		  struct it it3;

		  /* Find the first and the last buffer positions
		     covered by the display string.  */
		  endpos =
		    Fnext_single_char_property_change (cpos, Qdisplay,
						       Qnil, Qnil);
		  startpos =
		    Fprevious_single_char_property_change (endpos, Qdisplay,
							   Qnil, Qnil);
		  start = XFIXNAT (startpos);
		  end = XFIXNAT (endpos);
		  /* Move to the last buffer position before the
		     display property.  */
		  start_display (&it3, w, top);
		  if (start > CHARPOS (top))
		    move_it_to (&it3, start - 1, -1, -1, -1, MOVE_TO_POS);
		  /* Move forward one more line if the position before
		     the display string is a newline or if it is the
		     rightmost character on a line that is
		     continued or word-wrapped.  */
		  if (it3.method == GET_FROM_BUFFER
		      && (it3.c == '\n'
			  || FETCH_BYTE (IT_BYTEPOS (it3)) == '\n'))
		    move_it_by_lines (&it3, 1);
		  else if (move_it_in_display_line_to (&it3, -1,
						       it3.current_x
						       + it3.pixel_width,
						       MOVE_TO_X)
			   == MOVE_LINE_CONTINUED)
		    {
		      move_it_by_lines (&it3, 1);
		      /* When we are under word-wrap, the #$@%!
			 move_it_by_lines moves 2 lines, so we need to
			 fix that up.  */
		      if (it3.line_wrap == WORD_WRAP)
			move_it_by_lines (&it3, -1);
		    }

		  /* Record the vertical coordinate of the display
		     line where we wound up.  */
		  top_y = it3.current_y;
		  if (it3.bidi_p)
		    {
		      /* When characters are reordered for display,
			 the character displayed to the left of the
			 display string could be _after_ the display
			 property in the logical order.  Use the
			 smallest vertical position of these two.  */
		      start_display (&it3, w, top);
		      move_it_to (&it3, end + 1, -1, -1, -1, MOVE_TO_POS);
		      if (it3.current_y < top_y)
			top_y = it3.current_y;
		    }
		  /* Move from the top of the window to the beginning
		     of the display line where the display string
		     begins.  */
		  start_display (&it3, w, top);
		  it3.glyph_row = NULL;
		  move_it_to (&it3, -1, 0, top_y, -1, MOVE_TO_X | MOVE_TO_Y);
		  /* If it3_moved stays false after the 'while' loop
		     below, that means we already were at a newline
		     before the loop (e.g., the display string begins
		     with a newline), so we don't need to return to
		     the last position before the display string,
		     because PRODUCE_GLYPHS will not produce anything
		     for a newline.  */
		  bool it3_moved = false;
		  int top_x_before_string = it3.current_x;
		  /* Finally, advance the iterator until we hit the
		     first display element whose character position is
		     at or beyond CHARPOS, or until the first newline
		     from the display string, which signals the end of
		     the display line.  */
		  while (get_next_display_element (&it3))
		    {
		      if (!EQ (it3.object, string))
			top_x_before_string = it3.current_x;
		      PRODUCE_GLYPHS (&it3);
		      if ((it3.bidi_it.scan_dir == 1
			   && IT_CHARPOS (it3) >= charpos)
			  || (it3.bidi_it.scan_dir == -1
			      && IT_CHARPOS (it3) <= charpos)
			  || ITERATOR_AT_END_OF_LINE_P (&it3))
			break;
		      it3_moved = true;
		      set_iterator_to_next (&it3, false);
		    }
		  top_x = it3.current_x - it3.pixel_width;
		  /* Account for line-number display, if IT3 still
		     didn't.  This can happen if START - 1 is the
		     first or the last character on its display line.  */
		  if (!it3.line_number_produced_p)
		    {
		      if (it3.lnum_pixel_width > 0)
			{
			  top_x += it3.lnum_pixel_width;
			  top_x_before_string += it3.lnum_pixel_width;
			}
		      else if (it.line_number_produced_p)
			{
			  top_x += it.lnum_pixel_width;
			  top_x_before_string += it3.lnum_pixel_width;
			}
		    }
		  /* Normally, we would exit the above loop because we
		     found the display element whose character
		     position is CHARPOS.  For the contingency that we
		     didn't, and stopped at the first newline from the
		     display string, reset top_x to the coordinate of
		     the rightmost glyph not from the string.  */
		  if (it3_moved
		      && newline_in_string
		      && IT_CHARPOS (it3) != charpos && EQ (it3.object, string))
		    top_x = top_x_before_string;
		}
	    }

	  *x = top_x;
	  /* The condition below is a heuristic fix for the situation
	     where move_it_to stops just after finishing the display
	     of a fringe bitmap, which resets it.ascent to zero, and
	     thus causes Y to be offset by it.max_ascent.  */
	  if (it.ascent == 0 && it.what == IT_IMAGE
	      && it.method != GET_FROM_IMAGE
	      && it.image_id < 0
	      && it.max_ascent > 0)
	    *y = max (top_y, window_top_y);
	  else
	    *y = max (top_y + max (0, it.max_ascent - it.ascent), window_top_y);
	  *rtop = max (0, window_top_y - top_y);
	  *rbot = max (0, bottom_y - it.last_visible_y);
	  *rowh = max (0, (min (bottom_y, it.last_visible_y)
			   - max (top_y, window_top_y)));
	  *vpos = it.vpos;
	  if (it.bidi_it.paragraph_dir == R2L)
	    r2l = true;
	}
    }
  else
    {
      /* Either we were asked to provide info about WINDOW_END, or
	 CHARPOS is in the partially visible glyph row at end of
	 window.  */
      struct it it2;
      void *it2data = NULL;

      SAVE_IT (it2, it, it2data);
      if (IT_CHARPOS (it) < ZV && FETCH_BYTE (IT_BYTEPOS (it)) != '\n')
	move_it_by_lines (&it, 1);
      if (charpos < IT_CHARPOS (it)
	  || (it.what == IT_EOB && charpos == IT_CHARPOS (it)))
	{
	  visible_p = true;
	  RESTORE_IT (&it2, &it2, it2data);
	  move_it_to (&it2, charpos, -1, -1, -1, MOVE_TO_POS);
	  *x = it2.current_x;
	  if (it2.ascent == 0 && it2.what == IT_IMAGE
	      && it2.method != GET_FROM_IMAGE
	      && it2.image_id < 0
	      && it2.max_ascent > 0)
	    *y = it2.current_y;
	  else
	    *y = it2.current_y + it2.max_ascent - it2.ascent;
	  *rtop = max (0, -it2.current_y);
	  *rbot = max (0, ((it2.current_y + it2.max_ascent + it2.max_descent)
			   - it.last_visible_y));
	  *rowh = max (0, (min (it2.current_y + it2.max_ascent + it2.max_descent,
				it.last_visible_y)
			   - max (max (it2.current_y,
				       WINDOW_TAB_LINE_HEIGHT (w)),
				  WINDOW_HEADER_LINE_HEIGHT (w))));
	  *vpos = it2.vpos;
	  if (it2.bidi_it.paragraph_dir == R2L)
	    r2l = true;
	}
      else
	bidi_unshelve_cache (it2data, true);
    }
  bidi_unshelve_cache (itdata, false);

  if (old_buffer)
    set_buffer_internal_1 (old_buffer);

  if (visible_p)
    {
      if (w->hscroll > 0)
	*x -=
	  window_hscroll_limited (w, WINDOW_XFRAME (w))
	  * WINDOW_FRAME_COLUMN_WIDTH (w);
      /* For lines in an R2L paragraph, we need to mirror the X pixel
         coordinate wrt the text area.  For the reasons, see the
         commentary in buffer_posn_from_coords and the explanation of
         the geometry used by the move_it_* functions at the end of
         the large commentary near the beginning of this file.  */
      if (r2l)
	*x = window_box_width (w, TEXT_AREA) - *x - 1;
    }

#if false
  /* Debugging code.  */
  if (visible_p)
    fprintf (stderr, "+pv pt=%d vs=%d --> x=%d y=%d rt=%d rb=%d rh=%d vp=%d\n",
	     charpos, w->vscroll, *x, *y, *rtop, *rbot, *rowh, *vpos);
  else
    fprintf (stderr, "-pv pt=%d vs=%d\n", charpos, w->vscroll);
#endif

  /* Restore potentially overwritten values.  */
  w->mode_line_height = prev_mode_line_height;
  w->header_line_height = prev_header_line_height;
  w->tab_line_height = prev_tab_line_height;

  return visible_p;
}


/* Return the next character from STR.  Return in *LEN the length of
   the character.  This is like string_char_and_length but never
   returns an invalid character.  If we find one, we return a `?', but
   with the length of the invalid character.  */

static int
check_char_and_length (const unsigned char *str, int *len)
{
  int c = string_char_and_length (str, len);
  if (!CHAR_VALID_P (c))
    /* We may not change the length here because other places in Emacs
       don't use this function, i.e. they silently accept invalid
       characters.  */
    c = '?';

  return c;
}



/* Given a position POS containing a valid character and byte position
   in STRING, return the position NCHARS ahead (NCHARS >= 0).  */

static struct text_pos
string_pos_nchars_ahead (struct text_pos pos, Lisp_Object string, ptrdiff_t nchars)
{
  eassert (STRINGP (string) && nchars >= 0);

  if (STRING_MULTIBYTE (string))
    {
      const unsigned char *p = SDATA (string) + BYTEPOS (pos);

      while (nchars--)
	{
	  int len = BYTES_BY_CHAR_HEAD (*p);
	  p += len;
	  CHARPOS (pos) += 1;
	  BYTEPOS (pos) += len;
	}
    }
  else
    SET_TEXT_POS (pos, CHARPOS (pos) + nchars, BYTEPOS (pos) + nchars);

  return pos;
}


/* Value is the text position, i.e. character and byte position,
   for character position CHARPOS in STRING.  */

static struct text_pos
string_pos (ptrdiff_t charpos, Lisp_Object string)
{
  struct text_pos pos;
  eassert (STRINGP (string));
  eassert (charpos >= 0);
  SET_TEXT_POS (pos, charpos, string_char_to_byte (string, charpos));
  return pos;
}


/* Value is a text position, i.e. character and byte position, for
   character position CHARPOS in C string S.  MULTIBYTE_P
   means recognize multibyte characters.  */

static struct text_pos
c_string_pos (ptrdiff_t charpos, const char *s, bool multibyte_p)
{
  struct text_pos pos;

  eassert (s != NULL);
  eassert (charpos >= 0);

  if (multibyte_p)
    {
      SET_TEXT_POS (pos, 0, 0);
      while (charpos--)
	{
	  int len = BYTES_BY_CHAR_HEAD (*s);
	  s += len;
	  CHARPOS (pos) += 1;
	  BYTEPOS (pos) += len;
	}
    }
  else
    SET_TEXT_POS (pos, charpos, charpos);

  return pos;
}


/* Value is the number of characters in C string S.  MULTIBYTE_P
   means recognize multibyte characters.  */

static ptrdiff_t
number_of_chars (const char *s, bool multibyte_p)
{
  ptrdiff_t nchars;

  if (multibyte_p)
    {
      ptrdiff_t rest = strlen (s);
      const unsigned char *p = (const unsigned char *) s;

      for (nchars = 0; rest > 0; ++nchars)
	{
	  int len = BYTES_BY_CHAR_HEAD (*p);
	  rest -= len, p += len;
	}
    }
  else
    nchars = strlen (s);

  return nchars;
}


/* Compute byte position NEWPOS->bytepos corresponding to
   NEWPOS->charpos.  POS is a known position in string STRING.
   NEWPOS->charpos must be >= POS.charpos.  */

static void
compute_string_pos (struct text_pos *newpos, struct text_pos pos, Lisp_Object string)
{
  eassert (STRINGP (string));
  eassert (CHARPOS (*newpos) >= CHARPOS (pos));

  if (STRING_MULTIBYTE (string))
    *newpos = string_pos_nchars_ahead (pos, string,
				       CHARPOS (*newpos) - CHARPOS (pos));
  else
    BYTEPOS (*newpos) = CHARPOS (*newpos);
}

/* EXPORT:
   Return an estimation of the pixel height of mode or header lines on
   frame F.  FACE_ID specifies what line's height to estimate.  */

int
estimate_mode_line_height (struct frame *f, enum face_id face_id)
{
#ifdef HAVE_WINDOW_SYSTEM
  if (FRAME_WINDOW_P (f))
    {
      int height = FONT_HEIGHT (FRAME_FONT (f));

      /* This function is called so early when Emacs starts that the face
	 cache and mode line face are not yet initialized.  */
      if (FRAME_FACE_CACHE (f))
	{
	  struct face *face = FACE_FROM_ID_OR_NULL (f, face_id);
	  if (face)
	    {
	      if (face->font)
		height = normal_char_height (face->font, -1);
	      if (face->box_horizontal_line_width > 0)
		height += 2 * face->box_horizontal_line_width;
	    }
	}

      return height;
    }
#endif

  return 1;
}

/* Given a pixel position (PIX_X, PIX_Y) on frame F, return glyph
   coordinates in (*X, *Y).  Set *BOUNDS to the rectangle that the
   glyph at X, Y occupies, if BOUNDS != 0.  If NOCLIP, do
   not force the value into range.  */

void
pixel_to_glyph_coords (struct frame *f, int pix_x, int pix_y, int *x, int *y,
		       NativeRectangle *bounds, bool noclip)
{

#ifdef HAVE_WINDOW_SYSTEM
  if (FRAME_WINDOW_P (f))
    {
      /* Arrange for the division in FRAME_PIXEL_X_TO_COL etc. to round down
	 even for negative values.  */
      if (pix_x < 0)
	pix_x -= FRAME_COLUMN_WIDTH (f) - 1;
      if (pix_y < 0)
	pix_y -= FRAME_LINE_HEIGHT (f) - 1;

      pix_x = FRAME_PIXEL_X_TO_COL (f, pix_x);
      pix_y = FRAME_PIXEL_Y_TO_LINE (f, pix_y);

      if (bounds)
	STORE_NATIVE_RECT (*bounds,
			   FRAME_COL_TO_PIXEL_X (f, pix_x),
			   FRAME_LINE_TO_PIXEL_Y (f, pix_y),
			   FRAME_COLUMN_WIDTH (f) - 1,
			   FRAME_LINE_HEIGHT (f) - 1);

      /* PXW: Should we clip pixels before converting to columns/lines?  */
      if (!noclip)
	{
	  if (pix_x < 0)
	    pix_x = 0;
	  else if (pix_x > FRAME_TOTAL_COLS (f))
	    pix_x = FRAME_TOTAL_COLS (f);

	  if (pix_y < 0)
	    pix_y = 0;
	  else if (pix_y > FRAME_TOTAL_LINES (f))
	    pix_y = FRAME_TOTAL_LINES (f);
	}
    }
#endif

  *x = pix_x;
  *y = pix_y;
}


/* Find the glyph under window-relative coordinates X/Y in window W.
   Consider only glyphs from buffer text, i.e. no glyphs from overlay
   strings.  Return in *HPOS and *VPOS the row and column number of
   the glyph found.  Return in *AREA the glyph area containing X.
   Value is a pointer to the glyph found or null if X/Y is not on
   text, or we can't tell because W's current matrix is not up to
   date.  */

struct glyph *
x_y_to_hpos_vpos (struct window *w, int x, int y, int *hpos, int *vpos,
		  int *dx, int *dy, int *area)
{
  struct glyph *glyph, *end;
  struct glyph_row *row = NULL;
  int x0, i;

  /* Find row containing Y.  Give up if some row is not enabled.  */
  for (i = 0; i < w->current_matrix->nrows; ++i)
    {
      row = MATRIX_ROW (w->current_matrix, i);
      if (!row->enabled_p)
	return NULL;
      if (y >= row->y && y < MATRIX_ROW_BOTTOM_Y (row))
	break;
    }

  *vpos = i;
  *hpos = 0;

  /* Give up if Y is not in the window.  */
  if (i == w->current_matrix->nrows)
    return NULL;

  /* Get the glyph area containing X.  */
  if (w->pseudo_window_p)
    {
      *area = TEXT_AREA;
      x0 = 0;
    }
  else
    {
      if (x < window_box_left_offset (w, TEXT_AREA))
	{
	  *area = LEFT_MARGIN_AREA;
	  x0 = window_box_left_offset (w, LEFT_MARGIN_AREA);
	}
      else if (x < window_box_right_offset (w, TEXT_AREA))
	{
	  *area = TEXT_AREA;
	  x0 = window_box_left_offset (w, TEXT_AREA) + min (row->x, 0);
	}
      else
	{
	  *area = RIGHT_MARGIN_AREA;
	  x0 = window_box_left_offset (w, RIGHT_MARGIN_AREA);
	}
    }

  /* Find glyph containing X.  */
  glyph = row->glyphs[*area];
  end = glyph + row->used[*area];
  x -= x0;
  while (glyph < end && x >= glyph->pixel_width)
    {
      x -= glyph->pixel_width;
      ++glyph;
    }

  if (glyph == end)
    return NULL;

  if (dx)
    {
      *dx = x;
      *dy = y - (row->y + row->ascent - glyph->ascent);
    }

  *hpos = glyph - row->glyphs[*area];
  return glyph;
}

/* Convert frame-relative x/y to coordinates relative to window W.
   Takes pseudo-windows into account.  */

static void
frame_to_window_pixel_xy (struct window *w, int *x, int *y)
{
  if (w->pseudo_window_p)
    {
      /* A pseudo-window is always full-width, and starts at the
	 left edge of the frame, plus a frame border.  */
      struct frame *f = XFRAME (w->frame);
      *x -= FRAME_INTERNAL_BORDER_WIDTH (f);
      *y = FRAME_TO_WINDOW_PIXEL_Y (w, *y);
    }
  else
    {
      *x -= WINDOW_LEFT_EDGE_X (w);
      *y = FRAME_TO_WINDOW_PIXEL_Y (w, *y);
    }
}

#ifdef HAVE_WINDOW_SYSTEM

/* EXPORT:
   Return in RECTS[] at most N clipping rectangles for glyph string S.
   Return the number of stored rectangles.  */

int
get_glyph_string_clip_rects (struct glyph_string *s, NativeRectangle *rects, int n)
{
  Emacs_Rectangle r;

  if (n <= 0)
    return 0;

  if (s->row->full_width_p)
    {
      /* Draw full-width.  X coordinates are relative to S->w->left_col.  */
      r.x = WINDOW_LEFT_EDGE_X (s->w);
      if (s->row->mode_line_p)
	r.width = WINDOW_PIXEL_WIDTH (s->w) - WINDOW_RIGHT_DIVIDER_WIDTH (s->w);
      else
	r.width = WINDOW_PIXEL_WIDTH (s->w);

      /* Unless displaying a mode or menu bar line, which are always
	 fully visible, clip to the visible part of the row.  */
      if (s->w->pseudo_window_p)
	r.height = s->row->visible_height;
      else
	r.height = s->height;
    }
  else
    {
      /* This is a text line that may be partially visible.  */
      r.x = window_box_left (s->w, s->area);
      r.width = window_box_width (s->w, s->area);
      r.height = s->row->visible_height;
    }

  if (s->clip_head)
    if (r.x < s->clip_head->x)
      {
	if (r.width >= s->clip_head->x - r.x)
	  r.width -= s->clip_head->x - r.x;
	else
	  r.width = 0;
	r.x = s->clip_head->x;
      }
  if (s->clip_tail)
    if (r.x + r.width > s->clip_tail->x + s->clip_tail->background_width)
      {
	if (s->clip_tail->x + s->clip_tail->background_width >= r.x)
	  r.width = s->clip_tail->x + s->clip_tail->background_width - r.x;
	else
	  r.width = 0;
      }

  /* If S draws overlapping rows, it's sufficient to use the top and
     bottom of the window for clipping because this glyph string
     intentionally draws over other lines.  */
  if (s->for_overlaps)
    {
      r.y = WINDOW_TAB_LINE_HEIGHT (s->w) + WINDOW_HEADER_LINE_HEIGHT (s->w);
      r.height = window_text_bottom_y (s->w) - r.y;

      /* Alas, the above simple strategy does not work for the
	 environments with anti-aliased text: if the same text is
	 drawn onto the same place multiple times, it gets thicker.
	 If the overlap we are processing is for the erased cursor, we
	 take the intersection with the rectangle of the cursor.  */
      if (s->for_overlaps & OVERLAPS_ERASED_CURSOR)
	{
	  Emacs_Rectangle rc, r_save = r;

	  rc.x = WINDOW_TEXT_TO_FRAME_PIXEL_X (s->w, s->w->phys_cursor.x);
	  rc.y = s->w->phys_cursor.y;
	  rc.width = s->w->phys_cursor_width;
	  rc.height = s->w->phys_cursor_height;

	  gui_intersect_rectangles (&r_save, &rc, &r);
	}
    }
  else
    {
      /* Don't use S->y for clipping because it doesn't take partially
	 visible lines into account.  For example, it can be negative for
	 partially visible lines at the top of a window.  */
      if (!s->row->full_width_p
	  && MATRIX_ROW_PARTIALLY_VISIBLE_AT_TOP_P (s->w, s->row))
	r.y = WINDOW_TAB_LINE_HEIGHT (s->w) + WINDOW_HEADER_LINE_HEIGHT (s->w);
      else
	r.y = max (0, s->row->y);
    }

  r.y = WINDOW_TO_FRAME_PIXEL_Y (s->w, r.y);

  /* If drawing the cursor, don't let glyph draw outside its
     advertised boundaries. Cleartype does this under some circumstances.  */
  if (s->hl == DRAW_CURSOR)
    {
      struct glyph *glyph = s->first_glyph;
      int height, max_y;

      if (s->x > r.x)
	{
	  if (r.width >= s->x - r.x)
	    r.width -= s->x - r.x;
	  else	/* R2L hscrolled row with cursor outside text area */
	    r.width = 0;
	  r.x = s->x;
	}
      r.width = min (r.width, glyph->pixel_width);

      /* If r.y is below window bottom, ensure that we still see a cursor.  */
      height = min (glyph->ascent + glyph->descent,
		    min (FRAME_LINE_HEIGHT (s->f), s->row->visible_height));
      max_y = window_text_bottom_y (s->w) - height;
      max_y = WINDOW_TO_FRAME_PIXEL_Y (s->w, max_y);
      if (s->ybase - glyph->ascent > max_y)
	{
	  r.y = max_y;
	  r.height = height;
	}
      else
	{
	  /* Don't draw cursor glyph taller than our actual glyph.  */
	  height = max (FRAME_LINE_HEIGHT (s->f), glyph->ascent + glyph->descent);
	  if (height < r.height)
	    {
	      max_y = r.y + r.height;
	      r.y = min (max_y, max (r.y, s->ybase + glyph->descent - height));
	      r.height = min (max_y - r.y, height);
	    }
	}
    }

  if (s->row->clip)
    {
      Emacs_Rectangle r_save = r;

      if (! gui_intersect_rectangles (&r_save, s->row->clip, &r))
	r.width = 0;
    }

  if ((s->for_overlaps & OVERLAPS_BOTH) == 0
      || ((s->for_overlaps & OVERLAPS_BOTH) == OVERLAPS_BOTH && n == 1))
    {
#ifdef CONVERT_FROM_EMACS_RECT
      CONVERT_FROM_EMACS_RECT (r, *rects);
#else
      *rects = r;
#endif
      return 1;
    }
  else
    {
      /* If we are processing overlapping and allowed to return
	 multiple clipping rectangles, we exclude the row of the glyph
	 string from the clipping rectangle.  This is to avoid drawing
	 the same text on the environment with anti-aliasing.  */
#ifdef CONVERT_FROM_EMACS_RECT
      Emacs_Rectangle rs[2];
#else
      Emacs_Rectangle *rs = rects;
#endif
      int i = 0, row_y = WINDOW_TO_FRAME_PIXEL_Y (s->w, s->row->y);

      if (s->for_overlaps & OVERLAPS_PRED)
	{
	  rs[i] = r;
	  if (r.y + r.height > row_y)
	    {
	      if (r.y < row_y)
		rs[i].height = row_y - r.y;
	      else
		rs[i].height = 0;
	    }
	  i++;
	}
      if (s->for_overlaps & OVERLAPS_SUCC)
	{
	  rs[i] = r;
	  if (r.y < row_y + s->row->visible_height)
	    {
	      if (r.y + r.height > row_y + s->row->visible_height)
		{
		  rs[i].y = row_y + s->row->visible_height;
		  rs[i].height = r.y + r.height - rs[i].y;
		}
	      else
		rs[i].height = 0;
	    }
	  i++;
	}

      n = i;
#ifdef CONVERT_FROM_EMACS_RECT
      for (i = 0; i < n; i++)
	CONVERT_FROM_EMACS_RECT (rs[i], rects[i]);
#endif
      return n;
    }
}

/* EXPORT:
   Return in *NR the clipping rectangle for glyph string S.  */

void
get_glyph_string_clip_rect (struct glyph_string *s, NativeRectangle *nr)
{
  get_glyph_string_clip_rects (s, nr, 1);
}


/* EXPORT:
   Return the position and height of the phys cursor in window W.
   Set w->phys_cursor_width to width of phys cursor.
*/

void
get_phys_cursor_geometry (struct window *w, struct glyph_row *row,
			  struct glyph *glyph, int *xp, int *yp, int *heightp)
{
  struct frame *f = XFRAME (WINDOW_FRAME (w));
  int x, y, wd, h, h0, y0, ascent;

  /* Compute the width of the rectangle to draw.  If on a stretch
     glyph, and `x-stretch-cursor' is nil, don't draw a rectangle
     as wide as the glyph, but use a canonical character width
     instead.  */
  wd = glyph->pixel_width;

  x = w->phys_cursor.x;
  if (x < 0)
    {
      wd += x;
      x = 0;
    }

  if (glyph->type == STRETCH_GLYPH
      && !x_stretch_cursor_p)
    wd = min (FRAME_COLUMN_WIDTH (f), wd);
  w->phys_cursor_width = wd;

  /* Don't let the hollow cursor glyph descend below the glyph row's
     ascent value, lest the hollow cursor looks funny.  */
  y = w->phys_cursor.y;
  ascent = row->ascent;
  /* The test for row at ZV is for when line numbers are displayed and
     point is at EOB: the cursor could then be smaller or larger than
     the default face's font.  */
  if (!row->ends_at_zv_p && row->ascent < glyph->ascent)
    {
      y -= glyph->ascent - row->ascent;
      ascent = glyph->ascent;
    }

  /* If y is below window bottom, ensure that we still see a cursor.  */
  h0 = min (FRAME_LINE_HEIGHT (f), row->visible_height);

  h = max (h0, ascent + glyph->descent);
  /* Don't let the cursor exceed the dimensions of the row, so that
     the upper/lower side of the box aren't clipped.  */
  h = min (h, row->height);
  h0 = min (h0, ascent + glyph->descent);

  y0 = WINDOW_TAB_LINE_HEIGHT (w) + WINDOW_HEADER_LINE_HEIGHT (w);
  if (y < y0)
    {
      h = max (h - (y0 - y) + 1, h0);
      y = y0 - 1;
    }
  else
    {
      y0 = window_text_bottom_y (w) - h0;
      if (y > y0)
	{
	  h += y - y0;
	  y = y0;
	}
    }

  *xp = WINDOW_TEXT_TO_FRAME_PIXEL_X (w, x);
  *yp = WINDOW_TO_FRAME_PIXEL_Y (w, y);
  *heightp = h;
}

/*
 * Remember which glyph the mouse is over.
 */

void
remember_mouse_glyph (struct frame *f, int gx, int gy, NativeRectangle *rect)
{
  Lisp_Object window;
  struct window *w;
  struct glyph_row *r, *gr, *end_row;
  enum window_part part;
  enum glyph_row_area area;
  int x, y, width, height;

  if (mouse_fine_grained_tracking)
    {
      STORE_NATIVE_RECT (*rect, gx, gy, 1, 1);
      return;
    }

  /* Try to determine frame pixel position and size of the glyph under
     frame pixel coordinates X/Y on frame F.  */

  if (window_resize_pixelwise)
    {
      width = height = 1;
      goto virtual_glyph;
    }
  else if (!f->glyphs_initialized_p
	   || (window = window_from_coordinates (f, gx, gy, &part, false, false),
	       NILP (window)))
    {
      width = FRAME_SMALLEST_CHAR_WIDTH (f);
      height = FRAME_SMALLEST_FONT_HEIGHT (f);
      goto virtual_glyph;
    }

  w = XWINDOW (window);
  width = WINDOW_FRAME_COLUMN_WIDTH (w);
  height = WINDOW_FRAME_LINE_HEIGHT (w);

  x = window_relative_x_coord (w, part, gx);
  y = gy - WINDOW_TOP_EDGE_Y (w);

  r = MATRIX_FIRST_TEXT_ROW (w->current_matrix);
  end_row = MATRIX_BOTTOM_TEXT_ROW (w->current_matrix, w);

  if (w->pseudo_window_p)
    {
      area = TEXT_AREA;
      part = ON_MODE_LINE; /* Don't adjust margin. */
      goto text_glyph;
    }

  switch (part)
    {
    case ON_LEFT_MARGIN:
      area = LEFT_MARGIN_AREA;
      goto text_glyph;

    case ON_RIGHT_MARGIN:
      area = RIGHT_MARGIN_AREA;
      goto text_glyph;

    case ON_TAB_LINE:
    case ON_HEADER_LINE:
    case ON_MODE_LINE:
      gr = (part == ON_TAB_LINE
	    ? MATRIX_TAB_LINE_ROW (w->current_matrix)
	    : (part == ON_HEADER_LINE
	       ? MATRIX_HEADER_LINE_ROW (w->current_matrix)
	       : MATRIX_MODE_LINE_ROW (w->current_matrix)));
      gy = gr->y;
      area = TEXT_AREA;
      goto text_glyph_row_found;

    case ON_TEXT:
      area = TEXT_AREA;

    text_glyph:
      gr = 0; gy = 0;
      for (; r <= end_row && r->enabled_p; ++r)
	if (r->y + r->height > y)
	  {
	    gr = r; gy = r->y;
	    break;
	  }

    text_glyph_row_found:
      if (gr && gy <= y)
	{
	  struct glyph *g = gr->glyphs[area];
	  struct glyph *end = g + gr->used[area];

	  height = gr->height;
	  for (gx = gr->x; g < end; gx += g->pixel_width, ++g)
	    if (gx + g->pixel_width > x)
	      break;

	  if (g < end)
	    {
	      if (g->type == IMAGE_GLYPH)
		{
		  /* Don't remember when mouse is over image, as
		     image may have hot-spots.  */
		  STORE_NATIVE_RECT (*rect, 0, 0, 0, 0);
		  return;
		}
	      width = g->pixel_width;
	    }
	  else
	    {
	      /* Use nominal char spacing at end of line.  */
	      x -= gx;
	      gx += (x / width) * width;
	    }

	  if (part != ON_MODE_LINE && part != ON_HEADER_LINE
	      && part != ON_TAB_LINE)
	    {
	      gx += window_box_left_offset (w, area);
	      /* Don't expand over the modeline to make sure the vertical
		 drag cursor is shown early enough.  */
	      height = min (height,
			    max (0, WINDOW_BOX_HEIGHT_NO_MODE_LINE (w) - gy));
	    }
	}
      else
	{
	  /* Use nominal line height at end of window.  */
	  gx = (x / width) * width;
	  y -= gy;
	  gy += (y / height) * height;
	  if (part != ON_MODE_LINE && part != ON_HEADER_LINE
	      && part != ON_TAB_LINE)
	    /* See comment above.  */
	    height = min (height,
			  max (0, WINDOW_BOX_HEIGHT_NO_MODE_LINE (w) - gy));
	}
      break;

    case ON_LEFT_FRINGE:
      gx = (WINDOW_HAS_FRINGES_OUTSIDE_MARGINS (w)
	    ? WINDOW_LEFT_SCROLL_BAR_AREA_WIDTH (w)
	    : window_box_right_offset (w, LEFT_MARGIN_AREA));
      width = WINDOW_LEFT_FRINGE_WIDTH (w);
      goto row_glyph;

    case ON_RIGHT_FRINGE:
      gx = (WINDOW_HAS_FRINGES_OUTSIDE_MARGINS (w)
	    ? window_box_right_offset (w, RIGHT_MARGIN_AREA)
	    : window_box_right_offset (w, TEXT_AREA));
      if (WINDOW_RIGHT_DIVIDER_WIDTH (w) == 0
	  && !WINDOW_HAS_VERTICAL_SCROLL_BAR (w)
	  && !WINDOW_RIGHTMOST_P (w))
	if (gx < WINDOW_PIXEL_WIDTH (w) - width)
	  /* Make sure the vertical border can get her own glyph to the
	     right of the one we build here.  */
	  width = WINDOW_RIGHT_FRINGE_WIDTH (w) - width;
	else
	  width = WINDOW_PIXEL_WIDTH (w) - gx;
      else
	width = WINDOW_RIGHT_FRINGE_WIDTH (w);

      goto row_glyph;

    case ON_VERTICAL_BORDER:
      gx = WINDOW_PIXEL_WIDTH (w) - width;
      goto row_glyph;

    case ON_VERTICAL_SCROLL_BAR:
      gx = (WINDOW_HAS_VERTICAL_SCROLL_BAR_ON_LEFT (w)
	    ? 0
	    : (window_box_right_offset (w, RIGHT_MARGIN_AREA)
	       + (WINDOW_HAS_FRINGES_OUTSIDE_MARGINS (w)
		  ? WINDOW_RIGHT_FRINGE_WIDTH (w)
		  : 0)));
      width = WINDOW_SCROLL_BAR_AREA_WIDTH (w);

    row_glyph:
      gr = 0, gy = 0;
      for (; r <= end_row && r->enabled_p; ++r)
	if (r->y + r->height > y)
	  {
	    gr = r; gy = r->y;
	    break;
	  }

      if (gr && gy <= y)
	height = gr->height;
      else
	{
	  /* Use nominal line height at end of window.  */
	  y -= gy;
	  gy += (y / height) * height;
	}
      break;

    case ON_RIGHT_DIVIDER:
      gx = WINDOW_PIXEL_WIDTH (w) - WINDOW_RIGHT_DIVIDER_WIDTH (w);
      width = WINDOW_RIGHT_DIVIDER_WIDTH (w);
      gy = 0;
      /* The bottom divider prevails. */
      height = WINDOW_PIXEL_HEIGHT (w) - WINDOW_BOTTOM_DIVIDER_WIDTH (w);
      goto add_edge;

    case ON_BOTTOM_DIVIDER:
      gx = 0;
      width = WINDOW_PIXEL_WIDTH (w);
      gy = WINDOW_PIXEL_HEIGHT (w) - WINDOW_BOTTOM_DIVIDER_WIDTH (w);
      height = WINDOW_BOTTOM_DIVIDER_WIDTH (w);
      goto add_edge;

    default:
      ;
    virtual_glyph:
      /* If there is no glyph under the mouse, then we divide the screen
	 into a grid of the smallest glyph in the frame, and use that
	 as our "glyph".  */

      /* Arrange for the division in FRAME_PIXEL_X_TO_COL etc. to
	 round down even for negative values.  */
      if (gx < 0)
	gx -= width - 1;
      if (gy < 0)
	gy -= height - 1;

      gx = (gx / width) * width;
      gy = (gy / height) * height;

      goto store_rect;
    }

 add_edge:
  gx += WINDOW_LEFT_EDGE_X (w);
  gy += WINDOW_TOP_EDGE_Y (w);

 store_rect:
  STORE_NATIVE_RECT (*rect, gx, gy, width, height);

  /* Visible feedback for debugging.  */
#if false && defined HAVE_X_WINDOWS
  XDrawRectangle (FRAME_X_DISPLAY (f), FRAME_X_DRAWABLE (f),
		  f->output_data.x->normal_gc,
		  gx, gy, width, height);
#endif
}


#endif /* HAVE_WINDOW_SYSTEM */

static void
adjust_window_ends (struct window *w, struct glyph_row *row, bool current)
{
  eassert (w);
  w->window_end_pos = Z - MATRIX_ROW_END_CHARPOS (row);
  w->window_end_bytepos = Z_BYTE - MATRIX_ROW_END_BYTEPOS (row);
  w->window_end_vpos
    = MATRIX_ROW_VPOS (row, current ? w->current_matrix : w->desired_matrix);
}

static bool
hscrolling_current_line_p (struct window *w)
{
  return (!w->suspend_auto_hscroll
	  && EQ (Fbuffer_local_value (Qauto_hscroll_mode, w->contents),
		 Qcurrent_line));
}

/***********************************************************************
			Lisp form evaluation
 ***********************************************************************/

/* Error handler for safe_eval and safe_call.  */

static Lisp_Object
safe_eval_handler (Lisp_Object arg, ptrdiff_t nargs, Lisp_Object *args)
{
  add_to_log ("Error during redisplay: %S signaled %S",
	      Flist (nargs, args), arg);
  return Qnil;
}

/* Call function FUNC with the rest of NARGS - 1 arguments
   following.  Return the result, or nil if something went
   wrong.  Prevent redisplay during the evaluation.  */

static Lisp_Object
safe__call (bool inhibit_quit, ptrdiff_t nargs, Lisp_Object func, va_list ap)
{
  Lisp_Object val;

  if (inhibit_eval_during_redisplay)
    val = Qnil;
  else
    {
      ptrdiff_t i;
      specpdl_ref count = SPECPDL_INDEX ();
      Lisp_Object *args;
      USE_SAFE_ALLOCA;
      SAFE_ALLOCA_LISP (args, nargs);

      args[0] = func;
      for (i = 1; i < nargs; i++)
	args[i] = va_arg (ap, Lisp_Object);

      specbind (Qinhibit_redisplay, Qt);
      if (inhibit_quit)
	specbind (Qinhibit_quit, Qt);
      /* Use Qt to ensure debugger does not run,
	 so there is no possibility of wanting to redisplay.  */
      val = internal_condition_case_n (Ffuncall, nargs, args, Qt,
				       safe_eval_handler);
      val = SAFE_FREE_UNBIND_TO (count, val);
    }

  return val;
}

Lisp_Object
safe_call (ptrdiff_t nargs, Lisp_Object func, ...)
{
  Lisp_Object retval;
  va_list ap;

  va_start (ap, func);
  retval = safe__call (false, nargs, func, ap);
  va_end (ap);
  return retval;
}

/* Call function FN with one argument ARG.
   Return the result, or nil if something went wrong.  */

Lisp_Object
safe_call1 (Lisp_Object fn, Lisp_Object arg)
{
  return safe_call (2, fn, arg);
}

static Lisp_Object
safe__call1 (bool inhibit_quit, Lisp_Object fn, ...)
{
  Lisp_Object retval;
  va_list ap;

  va_start (ap, fn);
  retval = safe__call (inhibit_quit, 2, fn, ap);
  va_end (ap);
  return retval;
}

Lisp_Object
safe_eval (Lisp_Object sexpr)
{
  return safe__call1 (false, Qeval, sexpr);
}

static Lisp_Object
safe__eval (bool inhibit_quit, Lisp_Object sexpr)
{
  return safe__call1 (inhibit_quit, Qeval, sexpr);
}

/* Call function FN with two arguments ARG1 and ARG2.
   Return the result, or nil if something went wrong.  */

Lisp_Object
safe_call2 (Lisp_Object fn, Lisp_Object arg1, Lisp_Object arg2)
{
  return safe_call (3, fn, arg1, arg2);
}



/***********************************************************************
			      Debugging
 ***********************************************************************/

/* Define CHECK_IT to perform sanity checks on iterators.
   This is for debugging.  It is too slow to do unconditionally.  */

static void
CHECK_IT (struct it *it)
{
#if false
  if (it->method == GET_FROM_STRING)
    {
      eassert (STRINGP (it->string));
      eassert (IT_STRING_CHARPOS (*it) >= 0);
    }
  else
    {
      eassert (IT_STRING_CHARPOS (*it) < 0);
      if (it->method == GET_FROM_BUFFER)
	{
	  /* Check that character and byte positions agree.  */
	  eassert (IT_CHARPOS (*it) == BYTE_TO_CHAR (IT_BYTEPOS (*it)));
	}
    }

  if (it->dpvec)
    eassert (it->current.dpvec_index >= 0);
  else
    eassert (it->current.dpvec_index < 0);
#endif
}


/* Check that the window end of window W is what we expect it
   to be---the last row in the current matrix displaying text.  */

static void
CHECK_WINDOW_END (struct window *w)
{
#if defined GLYPH_DEBUG && defined ENABLE_CHECKING
  if (!MINI_WINDOW_P (w) && w->window_end_valid)
    {
      struct glyph_row *row;
      eassert ((row = MATRIX_ROW (w->current_matrix, w->window_end_vpos),
		!row->enabled_p
		|| MATRIX_ROW_DISPLAYS_TEXT_P (row)
		|| MATRIX_ROW_VPOS (row, w->current_matrix) == 0));
    }
#endif
}

/***********************************************************************
		       Iterator initialization
 ***********************************************************************/

/* Initialize IT for displaying current_buffer in window W, starting
   at character position CHARPOS.  CHARPOS < 0 means that no buffer
   position is specified which is useful when the iterator is assigned
   a position later.  BYTEPOS is the byte position corresponding to
   CHARPOS.

   If ROW is not null, calls to produce_glyphs with IT as parameter
   will produce glyphs in that row.

   BASE_FACE_ID is the id of a base face to use.  It must be one of
   DEFAULT_FACE_ID for normal text, MODE_LINE_ACTIVE_FACE_ID,
   MODE_LINE_INACTIVE_FACE_ID, or HEADER_LINE_FACE_ID for displaying
   mode lines, or TOOL_BAR_FACE_ID for displaying the tool-bar.

   If ROW is null and BASE_FACE_ID is equal to MODE_LINE_ACTIVE_FACE_ID,
   MODE_LINE_INACTIVE_FACE_ID, or HEADER_LINE_FACE_ID, the iterator
   will be initialized to use the corresponding mode line glyph row of
   the desired matrix of W.  */

void
init_iterator (struct it *it, struct window *w,
	       ptrdiff_t charpos, ptrdiff_t bytepos,
	       struct glyph_row *row, enum face_id base_face_id)
{
  enum face_id remapped_base_face_id = base_face_id;
  int body_width = 0, body_height = 0;

  /* Some precondition checks.  */
  eassert (w != NULL && it != NULL);
  eassert (charpos < 0 || (charpos >= BUF_BEG (current_buffer)
			   && charpos <= ZV));

  /* If face attributes have been changed since the last redisplay,
     free realized faces now because they depend on face definitions
     that might have changed.  Don't free faces while there might be
     desired matrices pending which reference these faces.  */
  if (!inhibit_free_realized_faces)
    {
      if (face_change)
	{
	  face_change = false;
	  XFRAME (w->frame)->face_change = 0;
	  free_all_realized_faces (Qnil);
	}
      else if (XFRAME (w->frame)->face_change)
	{
	  XFRAME (w->frame)->face_change = 0;
	  free_all_realized_faces (w->frame);
	}
    }

  /* Perhaps remap BASE_FACE_ID to a user-specified alternative.  */
  if (! NILP (Vface_remapping_alist))
    remapped_base_face_id
      = lookup_basic_face (w, XFRAME (w->frame), base_face_id);

  /* Use one of the mode line rows of W's desired matrix if
     appropriate.  */
  if (row == NULL)
    {
      if (base_face_id == MODE_LINE_ACTIVE_FACE_ID
	  || base_face_id == MODE_LINE_INACTIVE_FACE_ID)
	row = MATRIX_MODE_LINE_ROW (w->desired_matrix);
      else if (base_face_id == TAB_LINE_FACE_ID)
	row = MATRIX_TAB_LINE_ROW (w->desired_matrix);
      else if (base_face_id == HEADER_LINE_FACE_ID)
	{
	  /* Header line row depends on whether tab line is enabled.  */
	  w->desired_matrix->tab_line_p = window_wants_tab_line (w);
	  row = MATRIX_HEADER_LINE_ROW (w->desired_matrix);
	}
    }

  /* Clear IT, and set it->object and other IT's Lisp objects to Qnil.
     Other parts of redisplay rely on that.  */
  memclear (it, sizeof *it);
  it->current.overlay_string_index = -1;
  it->current.dpvec_index = -1;
  it->base_face_id = remapped_base_face_id;
  IT_STRING_CHARPOS (*it) = IT_STRING_BYTEPOS (*it) = -1;
  it->paragraph_embedding = L2R;
  it->bidi_it.w = w;

  /* The window in which we iterate over current_buffer:  */
  XSETWINDOW (it->window, w);
  it->w = w;
  it->f = XFRAME (w->frame);

  it->cmp_it.id = -1;
  it->cmp_it.parent_it = it;

  if (max_redisplay_ticks > 0)
    update_redisplay_ticks (0, w);

  /* Extra space between lines (on window systems only).  */
  if (base_face_id == DEFAULT_FACE_ID
      && FRAME_WINDOW_P (it->f))
    {
      if (FIXNATP (BVAR (current_buffer, extra_line_spacing)))
	it->extra_line_spacing = XFIXNAT (BVAR (current_buffer, extra_line_spacing));
      else if (FLOATP (BVAR (current_buffer, extra_line_spacing)))
	it->extra_line_spacing = (XFLOAT_DATA (BVAR (current_buffer, extra_line_spacing))
				  * FRAME_LINE_HEIGHT (it->f));
      else if (it->f->extra_line_spacing > 0)
	it->extra_line_spacing = it->f->extra_line_spacing;
    }

  /* If realized faces have been removed, e.g. because of face
     attribute changes of named faces, recompute them.  When running
     in batch mode, the face cache of the initial frame is null.  If
     we happen to get called, make a dummy face cache.  */
  if (FRAME_FACE_CACHE (it->f) == NULL)
    init_frame_faces (it->f);
  if (FRAME_FACE_CACHE (it->f)->used == 0)
    recompute_basic_faces (it->f);

  it->override_ascent = -1;

  /* Are control characters displayed as `^C'?  */
  it->ctl_arrow_p = !NILP (BVAR (current_buffer, ctl_arrow));

  /* -1 means everything between a CR and the following line end
     is invisible.  >0 means lines indented more than this value are
     invisible.  */
  it->selective = (FIXNUMP (BVAR (current_buffer, selective_display))
		   ? (clip_to_bounds
		      (-1, XFIXNUM (BVAR (current_buffer, selective_display)),
		       PTRDIFF_MAX))
		   : (!NILP (BVAR (current_buffer, selective_display))
		      ? -1 : 0));
  it->selective_display_ellipsis_p
    = !NILP (BVAR (current_buffer, selective_display_ellipses));

  /* Display table to use.  */
  it->dp = window_display_table (w);

  /* Are multibyte characters enabled in current_buffer?  */
  it->multibyte_p = !NILP (BVAR (current_buffer, enable_multibyte_characters));

  it->tab_width = SANE_TAB_WIDTH (current_buffer);

  /* Are lines in the display truncated?  */
  if (TRUNCATE != 0)
    it->line_wrap = TRUNCATE;
  if (base_face_id == DEFAULT_FACE_ID
      && !it->w->hscroll
      && (WINDOW_FULL_WIDTH_P (it->w)
	  || NILP (Vtruncate_partial_width_windows)
	  || (FIXNUMP (Vtruncate_partial_width_windows)
	      /* PXW: Shall we do something about this?  */
	      && (XFIXNUM (Vtruncate_partial_width_windows)
		  <= WINDOW_TOTAL_COLS (it->w))))
      && NILP (BVAR (current_buffer, truncate_lines)))
    it->line_wrap = NILP (BVAR (current_buffer, word_wrap))
      ? WINDOW_WRAP : WORD_WRAP;

  /* Get dimensions of truncation and continuation glyphs.  These are
     displayed as fringe bitmaps under X, but we need them for such
     frames when the fringes are turned off.  The no_special_glyphs slot
     of the iterator's frame, when set, suppresses their display - by
     default for tooltip frames and when set via the 'no-special-glyphs'
     frame parameter.  */
#ifdef HAVE_WINDOW_SYSTEM
  if (!(FRAME_WINDOW_P (it->f) && it->f->no_special_glyphs))
#endif
    {
      if (it->line_wrap == TRUNCATE)
	{
	  /* We will need the truncation glyph.  */
	  eassert (it->glyph_row == NULL);
	  produce_special_glyphs (it, IT_TRUNCATION);
	  it->truncation_pixel_width = it->pixel_width;
	}
      else
	{
	  /* We will need the continuation glyph.  */
	  eassert (it->glyph_row == NULL);
	  produce_special_glyphs (it, IT_CONTINUATION);
	  it->continuation_pixel_width = it->pixel_width;
	}
    }

  /* Reset these values to zero because the produce_special_glyphs
     above has changed them.  */
  it->pixel_width = it->ascent = it->descent = 0;
  it->phys_ascent = it->phys_descent = 0;

  /* Set this after getting the dimensions of truncation and
     continuation glyphs, so that we don't produce glyphs when calling
     produce_special_glyphs, above.  */
  it->glyph_row = row;
  it->area = TEXT_AREA;

  /* Get the dimensions of the display area.  The display area
     consists of the visible window area plus a horizontally scrolled
     part to the left of the window.  All x-values are relative to the
     start of this total display area.  */
  if (base_face_id != DEFAULT_FACE_ID)
    {
      /* Mode lines, menu bar in terminal frames.  */
      it->first_visible_x = 0;
      it->last_visible_x =
	WINDOW_PIXEL_WIDTH (w) - WINDOW_RIGHT_DIVIDER_WIDTH (w);
    }
  else
    {
      /* When hscrolling only the current line, don't apply the
	 hscroll here, it will be applied by display_line when it gets
	 to laying out the line showing point.  However, if the
	 window's min_hscroll is positive, the user specified a lower
	 bound for automatic hscrolling, so they expect the
	 non-current lines to obey that hscroll amount.  */
      if (hscrolling_current_line_p (w))
	{
	  if (w->min_hscroll > 0)
	    it->first_visible_x = w->min_hscroll * FRAME_COLUMN_WIDTH (it->f);
	  else
	    it->first_visible_x = 0;
	}
      else
	it->first_visible_x =
	  window_hscroll_limited (w, it->f) * FRAME_COLUMN_WIDTH (it->f);

      body_width = window_box_width (w, TEXT_AREA);
      if (!w->pseudo_window_p && !MINI_WINDOW_P (w)
	  && body_width != w->old_body_pixel_width)
	FRAME_WINDOW_CHANGE (it->f) = true;
      it->last_visible_x = it->first_visible_x + body_width;

      /* If we truncate lines, leave room for the truncation glyph(s) at
	 the right margin.  Otherwise, leave room for the continuation
	 glyph(s).  Done only if the window has no right fringe.  */
      if (WINDOW_RIGHT_FRINGE_WIDTH (it->w) == 0)
	{
	  if (it->line_wrap == TRUNCATE)
	    it->last_visible_x -= it->truncation_pixel_width;
	  else
	    it->last_visible_x -= it->continuation_pixel_width;
	}

      it->tab_line_p = window_wants_tab_line (w);
      it->header_line_p = window_wants_header_line (w);
      body_height = WINDOW_TAB_LINE_HEIGHT (w) + WINDOW_HEADER_LINE_HEIGHT (w);
      it->current_y =  body_height + w->vscroll;
    }

  /* Leave room for a border glyph.  */
  if (!FRAME_WINDOW_P (it->f)
      && !WINDOW_RIGHTMOST_P (it->w))
    it->last_visible_x -= 1;

  it->last_visible_y = window_text_bottom_y (w);
  body_height += it->last_visible_y;
  if (!w->pseudo_window_p && !MINI_WINDOW_P (w)
      && body_height != w->old_body_pixel_height)
    FRAME_WINDOW_CHANGE (it->f) = true;

  /* For mode lines and alike, arrange for the first glyph having a
     left box line if the face specifies a box.  */
  if (base_face_id != DEFAULT_FACE_ID)
    {
      struct face *face;

      it->face_id = remapped_base_face_id;

      /* If we have a boxed mode line, make the first character appear
	 with a left box line.  */
      face = FACE_FROM_ID_OR_NULL (it->f, remapped_base_face_id);
      if (face && face->box != FACE_NO_BOX)
	{
	  int box_thickness = face->box_vertical_line_width;
	  it->face_box_p = true;
	  it->start_of_box_run_p = true;
	  /* Make sure we will have enough horizontal space to add the
	     right box line at the end.  */
	  if (box_thickness > 0)
	    it->last_visible_x -= box_thickness;
	}
    }

  /* If a buffer position was specified, set the iterator there,
     getting overlays and face properties from that position.  */
  if (charpos >= BUF_BEG (current_buffer))
    {
      it->stop_charpos = charpos;
      it->end_charpos = ZV;
      eassert (charpos == BYTE_TO_CHAR (bytepos));
      IT_CHARPOS (*it) = charpos;
      IT_BYTEPOS (*it) = bytepos;

      /* We will rely on `reseat' to set this up properly, via
	 handle_face_prop.  */
      it->face_id = it->base_face_id;

      it->start = it->current;
      /* Do we need to reorder bidirectional text?  Not if this is a
	 unibyte buffer: by definition, none of the single-byte
	 characters are strong R2L, so no reordering is needed.  And
	 bidi.c doesn't support unibyte buffers anyway.  Also, don't
	 reorder while we are loading loadup.el, since the tables of
	 character properties needed for reordering are not yet
	 available.  */
      it->bidi_p =
	!redisplay__inhibit_bidi
	&& !NILP (BVAR (current_buffer, bidi_display_reordering))
	&& it->multibyte_p;

      /* If we are to reorder bidirectional text, init the bidi
	 iterator.  */
      if (it->bidi_p)
	{
	  /*  Since we don't know at this point whether there will be
	      any R2L lines in the window, we reserve space for
	      truncation/continuation glyphs even if only the left
	      fringe is absent.  */
	  if (base_face_id == DEFAULT_FACE_ID
	      && WINDOW_LEFT_FRINGE_WIDTH (it->w) == 0
	      && WINDOW_RIGHT_FRINGE_WIDTH (it->w) != 0)
	    {
	      if (it->line_wrap == TRUNCATE)
		it->last_visible_x -= it->truncation_pixel_width;
	      else
		it->last_visible_x -= it->continuation_pixel_width;
	    }
	  /* Note the paragraph direction that this buffer wants to
	     use.  */
	  if (EQ (BVAR (current_buffer, bidi_paragraph_direction),
		  Qleft_to_right))
	    it->paragraph_embedding = L2R;
	  else if (EQ (BVAR (current_buffer, bidi_paragraph_direction),
		       Qright_to_left))
	    it->paragraph_embedding = R2L;
	  else
	    it->paragraph_embedding = NEUTRAL_DIR;
	  bidi_unshelve_cache (NULL, false);
	  bidi_init_it (charpos, IT_BYTEPOS (*it), FRAME_WINDOW_P (it->f),
			&it->bidi_it);
	}

      /* This is set only when long_line_optimizations_p is non-zero
	 for the current buffer.  */
      it->narrowed_begv = 0;

      /* Compute faces etc.  */
      reseat (it, it->current.pos, true);
    }

  CHECK_IT (it);
}

/* Compute a suitable alternate value for BEGV and ZV that may be used
   temporarily to optimize display if the buffer in window W contains
   long lines.  */

static int
get_narrowed_width (struct window *w)
{
  /* In a character-only terminal, only one font size is used, so we
     can use a smaller factor.  */
  int fact = EQ (Fterminal_live_p (Qnil), Qt) ? 2 : 3;
  int width = window_body_width (w, WINDOW_BODY_IN_CANONICAL_CHARS);
  return fact * max (1, width);
}

static int
get_narrowed_len (struct window *w)
{
  int height = window_body_height (w, WINDOW_BODY_IN_CANONICAL_CHARS);
  return get_narrowed_width (w) * max (1, height);
}

ptrdiff_t
get_narrowed_begv (struct window *w, ptrdiff_t pos)
{
  int len = get_narrowed_len (w);
  return max ((pos / len - 1) * len, BEGV);
}

ptrdiff_t
get_narrowed_zv (struct window *w, ptrdiff_t pos)
{
  int len = get_narrowed_len (w);
  return min ((pos / len + 1) * len, ZV);
}

ptrdiff_t
get_closer_narrowed_begv (struct window *w, ptrdiff_t pos)
{
  int len = get_narrowed_width (w);
  return max ((pos / len - 1) * len, BEGV);
}

ptrdiff_t
get_locked_narrowing_begv (ptrdiff_t pos)
{
  if (long_line_optimizations_region_size <= 0)
    return BEGV;
  int len = long_line_optimizations_region_size / 2;
  int begv = max (pos - len, BEGV);
  int limit = long_line_optimizations_bol_search_limit;
  while (limit > 0)
    {
      if (begv == BEGV || FETCH_BYTE (CHAR_TO_BYTE (begv) - 1) == '\n')
	return begv;
      begv--;
      limit--;
    }
  return begv;
}

ptrdiff_t
get_locked_narrowing_zv (ptrdiff_t pos)
{
  if (long_line_optimizations_region_size <= 0)
    return ZV;
  int len = long_line_optimizations_region_size / 2;
  return min (pos + len, ZV);
}

static void
unwind_narrowed_begv (Lisp_Object point_min)
{
  SET_BUF_BEGV (current_buffer, XFIXNUM (point_min));
}

/* Set DST to EXPR.  When IT indicates that BEGV should temporarily be
   updated to optimize display, evaluate EXPR with BEGV set to BV.  */

#define SET_WITH_NARROWED_BEGV(IT,DST,EXPR,BV)				\
  do {									\
    if (IT->narrowed_begv)						\
      {									\
	specpdl_ref count = SPECPDL_INDEX ();				\
	record_unwind_protect (unwind_narrowed_begv, Fpoint_min ());	\
	SET_BUF_BEGV (current_buffer, BV);				\
	DST = EXPR;							\
	unbind_to (count, Qnil);					\
      }									\
    else								\
      DST = EXPR;							\
  } while (0)

/* Initialize IT for the display of window W with window start POS.  */

void
start_display (struct it *it, struct window *w, struct text_pos pos)
{
  struct glyph_row *row;
  int first_vpos = window_wants_tab_line (w) + window_wants_header_line (w);

  row = w->desired_matrix->rows + first_vpos;
  init_iterator (it, w, CHARPOS (pos), BYTEPOS (pos), row, DEFAULT_FACE_ID);
  it->first_vpos = first_vpos;

  /* Don't reseat to previous visible line start if current start
     position is in a string or image.  */
  if (it->method == GET_FROM_BUFFER && it->line_wrap != TRUNCATE)
    {
      int first_y = it->current_y;

      /* If window start is not at a line start, skip forward to POS to
	 get the correct continuation lines width.  */
      bool start_at_line_beg_p = (CHARPOS (pos) == BEGV
				  || FETCH_BYTE (BYTEPOS (pos) - 1) == '\n');
      if (!start_at_line_beg_p)
	{
	  int new_x;

	  reseat_at_previous_visible_line_start (it);
	  move_it_to (it, CHARPOS (pos), -1, -1, -1, MOVE_TO_POS);

	  new_x = it->current_x + it->pixel_width;

	  /* If lines are continued, this line may end in the middle
	     of a multi-glyph character (e.g. a control character
	     displayed as \003, or in the middle of an overlay
	     string).  In this case move_it_to above will not have
	     taken us to the start of the continuation line but to the
	     end of the continued line.  */
	  if (it->current_x > 0
	      && it->line_wrap != TRUNCATE /* Lines are continued.  */
	      && (/* And glyph doesn't fit on the line.  */
		  new_x > it->last_visible_x
		  /* Or it fits exactly and we're on a window
		     system frame.  */
		  || (new_x == it->last_visible_x
		      && FRAME_WINDOW_P (it->f)
		      && ((it->bidi_p && it->bidi_it.paragraph_dir == R2L)
			  ? WINDOW_LEFT_FRINGE_WIDTH (it->w)
			  : WINDOW_RIGHT_FRINGE_WIDTH (it->w)))))
	    {
	      if ((it->current.dpvec_index >= 0
		   || it->current.overlay_string_index >= 0)
		  /* If we are on a newline from a display vector or
		     overlay string, then we are already at the end of
		     a screen line; no need to go to the next line in
		     that case, as this line is not really continued.
		     (If we do go to the next line, C-e will not DTRT.)  */
		  && it->c != '\n')
		{
		  set_iterator_to_next (it, true);
		  move_it_in_display_line_to (it, -1, -1, 0);
		}

	      it->continuation_lines_width += it->current_x;
	    }
	  /* If the character at POS is displayed via a display
	     vector, move_it_to above stops at the final glyph of
	     IT->dpvec.  To make the caller redisplay that character
	     again (a.k.a. start at POS), we need to reset the
	     dpvec_index to the beginning of IT->dpvec.  */
	  else if (it->current.dpvec_index >= 0)
	    it->current.dpvec_index = 0;

	  /* We're starting a new display line, not affected by the
	     height of the continued line, so clear the appropriate
	     fields in the iterator structure.  */
	  it->max_ascent = it->max_descent = 0;
	  it->max_phys_ascent = it->max_phys_descent = 0;

	  it->current_y = first_y;
	  it->vpos = 0;
	  it->current_x = it->hpos = 0;
	}
    }
}


/* Return true if POS is a position in ellipses displayed for invisible
   text.  W is the window we display, for text property lookup.  */

static bool
in_ellipses_for_invisible_text_p (struct display_pos *pos, struct window *w)
{
  Lisp_Object prop, window;
  bool ellipses_p = false;
  ptrdiff_t charpos = CHARPOS (pos->pos);

  /* If POS specifies a position in a display vector, this might
     be for an ellipsis displayed for invisible text.  We won't
     get the iterator set up for delivering that ellipsis unless
     we make sure that it gets aware of the invisible text.  */
  if (pos->dpvec_index >= 0
      && pos->overlay_string_index < 0
      && CHARPOS (pos->string_pos) < 0
      && charpos > BEGV
      && (XSETWINDOW (window, w),
	  prop = Fget_char_property (make_fixnum (charpos),
				     Qinvisible, window),
	  TEXT_PROP_MEANS_INVISIBLE (prop) == 0))
    {
      prop = Fget_char_property (make_fixnum (charpos - 1), Qinvisible,
				 window);
      ellipses_p = 2 == TEXT_PROP_MEANS_INVISIBLE (prop);
    }

  return ellipses_p;
}


/* Initialize IT for stepping through current_buffer in window W,
   starting at position POS that includes overlay string and display
   vector/ control character translation position information.  Value
   is false if there are overlay strings with newlines at POS.  */

static bool
init_from_display_pos (struct it *it, struct window *w, struct display_pos *pos)
{
  ptrdiff_t charpos = CHARPOS (pos->pos), bytepos = BYTEPOS (pos->pos);
  int i;
  bool overlay_strings_with_newlines = false;

  /* If POS specifies a position in a display vector, this might
     be for an ellipsis displayed for invisible text.  We won't
     get the iterator set up for delivering that ellipsis unless
     we make sure that it gets aware of the invisible text.  */
  if (in_ellipses_for_invisible_text_p (pos, w))
    {
      --charpos;
      bytepos = 0;
    }

  /* Keep in mind: the call to reseat in init_iterator skips invisible
     text, so we might end up at a position different from POS.  This
     is only a problem when POS is a row start after a newline and an
     overlay starts there with an after-string, and the overlay has an
     invisible property.  Since we don't skip invisible text in
     display_line and elsewhere immediately after consuming the
     newline before the row start, such a POS will not be in a string,
     but the call to init_iterator below will move us to the
     after-string.  */
  init_iterator (it, w, charpos, bytepos, NULL, DEFAULT_FACE_ID);

  /* This only scans the current chunk -- it should scan all chunks.
     However, OVERLAY_STRING_CHUNK_SIZE has been increased from 3 in 21.1
     to 16 in 22.1 to make this a lesser problem.  */
  for (i = 0; i < it->n_overlay_strings && i < OVERLAY_STRING_CHUNK_SIZE; ++i)
    {
      const char *s = SSDATA (it->overlay_strings[i]);
      const char *e = s + SBYTES (it->overlay_strings[i]);

      while (s < e && *s != '\n')
	++s;

      if (s < e)
	{
	  overlay_strings_with_newlines = true;
	  break;
	}
    }

  /* If position is within an overlay string, set up IT to the right
     overlay string.  */
  if (pos->overlay_string_index >= 0)
    {
      int relative_index;

      /* If the first overlay string happens to have a `display'
	 property for an image, the iterator will be set up for that
	 image, and we have to undo that setup first before we can
	 correct the overlay string index.  */
      if (it->method == GET_FROM_IMAGE)
	pop_it (it);

      /* We already have the first chunk of overlay strings in
	 IT->overlay_strings.  Load more until the one for
	 pos->overlay_string_index is in IT->overlay_strings.  */
      if (pos->overlay_string_index >= OVERLAY_STRING_CHUNK_SIZE)
	{
	  ptrdiff_t n = pos->overlay_string_index / OVERLAY_STRING_CHUNK_SIZE;
	  it->current.overlay_string_index = 0;
	  while (n--)
	    {
	      load_overlay_strings (it, 0);
	      it->current.overlay_string_index += OVERLAY_STRING_CHUNK_SIZE;
	    }
	}

      it->current.overlay_string_index = pos->overlay_string_index;
      relative_index = (it->current.overlay_string_index
			% OVERLAY_STRING_CHUNK_SIZE);
      it->string = it->overlay_strings[relative_index];
      eassert (STRINGP (it->string));
      it->current.string_pos = pos->string_pos;
      it->method = GET_FROM_STRING;
      it->end_charpos = SCHARS (it->string);
      /* Set up the bidi iterator for this overlay string.  */
      if (it->bidi_p)
	{
	  it->bidi_it.string.lstring = it->string;
	  it->bidi_it.string.s = NULL;
	  it->bidi_it.string.schars = SCHARS (it->string);
	  it->bidi_it.string.bufpos = it->overlay_strings_charpos;
	  it->bidi_it.string.from_disp_str = it->string_from_display_prop_p;
	  it->bidi_it.string.unibyte = !it->multibyte_p;
	  it->bidi_it.w = it->w;
	  bidi_init_it (IT_STRING_CHARPOS (*it), IT_STRING_BYTEPOS (*it),
			FRAME_WINDOW_P (it->f), &it->bidi_it);

	  /* Synchronize the state of the bidi iterator with
	     pos->string_pos.  For any string position other than
	     zero, this will be done automagically when we resume
	     iteration over the string and get_visually_first_element
	     is called.  But if string_pos is zero, and the string is
	     to be reordered for display, we need to resync manually,
	     since it could be that the iteration state recorded in
	     pos ended at string_pos of 0 moving backwards in string.  */
	  if (CHARPOS (pos->string_pos) == 0)
	    {
	      get_visually_first_element (it);
	      if (IT_STRING_CHARPOS (*it) != 0)
		do {
		  /* Paranoia.  */
		  eassert (it->bidi_it.charpos < it->bidi_it.string.schars);
		  bidi_move_to_visually_next (&it->bidi_it);
		} while (it->bidi_it.charpos != 0);
	    }
	  eassert (IT_STRING_CHARPOS (*it) == it->bidi_it.charpos
		   && IT_STRING_BYTEPOS (*it) == it->bidi_it.bytepos);
	}
    }

  if (CHARPOS (pos->string_pos) >= 0)
    {
      /* Recorded position is not in an overlay string, but in another
	 string.  This can only be a string from a `display' property.
	 IT should already be filled with that string.  */
      it->current.string_pos = pos->string_pos;
      eassert (STRINGP (it->string));
      if (it->bidi_p)
	bidi_init_it (IT_STRING_CHARPOS (*it), IT_STRING_BYTEPOS (*it),
		      FRAME_WINDOW_P (it->f), &it->bidi_it);
    }

  /* Restore position in display vector translations, control
     character translations or ellipses.  */
  if (pos->dpvec_index >= 0)
    {
      if (it->dpvec == NULL)
	get_next_display_element (it);
      eassert (it->dpvec && it->current.dpvec_index == 0);
      it->current.dpvec_index = pos->dpvec_index;
    }

  CHECK_IT (it);
  return !overlay_strings_with_newlines;
}


/* Initialize IT for stepping through current_buffer in window W
   starting at ROW->start.  */

static void
init_to_row_start (struct it *it, struct window *w, struct glyph_row *row)
{
  init_from_display_pos (it, w, &row->start);
  it->start = row->start;
  it->continuation_lines_width = row->continuation_lines_width;
  CHECK_IT (it);
}


/* Initialize IT for stepping through current_buffer in window W
   starting in the line following ROW, i.e. starting at ROW->end.
   Value is false if there are overlay strings with newlines at ROW's
   end position, or if the following row begins with bidi-reordered
   characters that could be composed.  */

static bool
init_to_row_end (struct it *it, struct window *w, struct glyph_row *row)
{
  bool success = false;

  if (init_from_display_pos (it, w, &row->end))
    {
      if (row->continued_p)
	it->continuation_lines_width
	  = row->continuation_lines_width + row->pixel_width;
      CHECK_IT (it);
      /* Initializing IT in the presence of compositions in reordered
	 rows is tricky: row->end above will generally cause us to
	 start at position that is not the first one in the logical
	 order, and we might therefore miss the composition earlier in
	 the buffer that affects how glypsh are laid out in this row.
	 So we punt instead.  Note: the test below works because
	 get_next_display_element calls get_visually_first_element,
	 which calls composition_compute_stop_pos, which populates
	 it->cmp_it.  */
      if (get_next_display_element (it)
	  && (it->bidi_it.scan_dir == -1 && it->cmp_it.id >= 0))
	success = false;
      else
	success = true;
    }

  return success;
}




/***********************************************************************
			   Text properties
 ***********************************************************************/

/* Called when IT reaches IT->stop_charpos.  Handle text property and
   overlay changes.  Set IT->stop_charpos to the next position where
   to stop.  */

static void
handle_stop (struct it *it)
{
  enum prop_handled handled;
  bool handle_overlay_change_p;
  struct props *p;

  it->dpvec = NULL;
  it->current.dpvec_index = -1;
  handle_overlay_change_p = !it->ignore_overlay_strings_at_pos_p;
  it->ellipsis_p = false;

  /* Use face of preceding text for ellipsis (if invisible) */
  if (it->selective_display_ellipsis_p)
    it->saved_face_id = it->face_id;

  /* Here's the description of the semantics of, and the logic behind,
     the various HANDLED_* statuses:

     HANDLED_NORMALLY means the handler did its job, and the loop
     should proceed to calling the next handler in order.

     HANDLED_RECOMPUTE_PROPS means the handler caused a significant
     change in the properties and overlays at current position, so the
     loop should be restarted, to re-invoke the handlers that were
     already called.  This happens when fontification-functions were
     called by handle_fontified_prop, and actually fontified
     something.  Another case where HANDLED_RECOMPUTE_PROPS is
     returned is when we discover overlay strings that need to be
     displayed right away.  The loop below will continue for as long
     as the status is HANDLED_RECOMPUTE_PROPS.

     HANDLED_RETURN means return immediately to the caller, to
     continue iteration without calling any further handlers.  This is
     used when we need to act on some property right away, for example
     when we need to display the ellipsis or a replacing display
     property, such as display string or image.

     HANDLED_OVERLAY_STRING_CONSUMED means an overlay string was just
     consumed, and the handler switched to the next overlay string.
     This signals the loop below to refrain from looking for more
     overlays before all the overlay strings of the current overlay
     are processed.

     Some of the handlers called by the loop push the iterator state
     onto the stack (see 'push_it'), and arrange for the iteration to
     continue with another object, such as an image, a display string,
     or an overlay string.  In most such cases, it->stop_charpos is
     set to the first character of the string, so that when the
     iteration resumes, this function will immediately be called
     again, to examine the properties at the beginning of the string.

     When a display or overlay string is exhausted, the iterator state
     is popped (see 'pop_it'), and iteration continues with the
     previous object.  Again, in many such cases this function is
     called again to find the next position where properties might
     change.  */

  do
    {
      handled = HANDLED_NORMALLY;

      /* Call text property handlers.  */
      for (p = it_props; p->handler; ++p)
	{
	  handled = p->handler (it);

	  if (handled == HANDLED_RECOMPUTE_PROPS)
	    break;
	  else if (handled == HANDLED_RETURN)
	    {
	      /* We still want to show before and after strings from
		 overlays even if the actual buffer text is replaced.  */
	      if (!handle_overlay_change_p
		  || it->sp > 1
		  /* Don't call get_overlay_strings_1 if we already
		     have overlay strings loaded, because doing so
		     will load them again and push the iterator state
		     onto the stack one more time, which is not
		     expected by the rest of the code that processes
		     overlay strings.  */
		  || (it->current.overlay_string_index < 0
		      && !get_overlay_strings_1 (it, 0, false)))
		{
		  if (it->ellipsis_p)
		    setup_for_ellipsis (it, 0);
		  /* When handling a display spec, we might load an
		     empty string.  In that case, discard it here.  We
		     used to discard it in handle_single_display_spec,
		     but that causes get_overlay_strings_1, above, to
		     ignore overlay strings that we must check.  */
		  if (STRINGP (it->string) && !SCHARS (it->string))
		    pop_it (it);
		  return;
		}
	      else if (STRINGP (it->string) && !SCHARS (it->string))
		pop_it (it);
	      else
		{
		  it->string_from_display_prop_p = false;
		  it->from_disp_prop_p = false;
		  handle_overlay_change_p = false;
		}
	      handled = HANDLED_RECOMPUTE_PROPS;
	      break;
	    }
	  else if (handled == HANDLED_OVERLAY_STRING_CONSUMED)
	    handle_overlay_change_p = false;
	}

      if (handled != HANDLED_RECOMPUTE_PROPS)
	{
	  /* Don't check for overlay strings below when set to deliver
	     characters from a display vector.  */
	  if (it->method == GET_FROM_DISPLAY_VECTOR)
	    handle_overlay_change_p = false;

	  /* Handle overlay changes.
	     This sets HANDLED to HANDLED_RECOMPUTE_PROPS
	     if it finds overlays.  */
	  if (handle_overlay_change_p)
	    handled = handle_overlay_change (it);
	}

      if (it->ellipsis_p)
	{
	  setup_for_ellipsis (it, 0);
	  break;
	}
    }
  while (handled == HANDLED_RECOMPUTE_PROPS);

  /* Determine where to stop next.  */
  if (handled == HANDLED_NORMALLY)
    compute_stop_pos (it);
}


/* Compute IT->stop_charpos from text property and overlay change
   information for IT's current position.  */

static void
compute_stop_pos (struct it *it)
{
  register INTERVAL iv, next_iv;
  Lisp_Object object, limit, position;
  ptrdiff_t charpos, bytepos;

  if (STRINGP (it->string))
    {
      /* Strings are usually short, so don't limit the search for
	 properties.  */
      it->stop_charpos = it->end_charpos;
      object = it->string;
      limit = Qnil;
      charpos = IT_STRING_CHARPOS (*it);
      bytepos = IT_STRING_BYTEPOS (*it);
    }
  else
    {
      ptrdiff_t pos;

      /* If end_charpos is out of range for some reason, such as a
	 misbehaving display function, rationalize it (Bug#5984).  */
      if (it->end_charpos > ZV)
	it->end_charpos = ZV;
      it->stop_charpos = it->end_charpos;

      /* If next overlay change is in front of the current stop pos
	 (which is IT->end_charpos), stop there.  Note: value of
	 next_overlay_change is point-max if no overlay change
	 follows.  */
      charpos = IT_CHARPOS (*it);
      bytepos = IT_BYTEPOS (*it);
      pos = next_overlay_change (charpos);
      if (pos < it->stop_charpos)
	it->stop_charpos = pos;
      /* If we are breaking compositions at point, stop at point.  */
      if (!NILP (BVAR (current_buffer, enable_multibyte_characters))
	  && !NILP (Vauto_composition_mode)
	  && composition_break_at_point
	  && charpos < PT && PT < it->stop_charpos)
	it->stop_charpos = PT;

      /* Set up variables for computing the stop position from text
         property changes.  */
      XSETBUFFER (object, current_buffer);
      pos = charpos + TEXT_PROP_DISTANCE_LIMIT;
      /* Make sure the above arbitrary limit position is not in the
	 middle of composable text, so we don't break compositions by
	 submitting the composable text to the shaper in separate
	 chunks.  We play safe here by assuming that only SPC, TAB,
	 FF, and NL cannot be in some composition; in particular, most
	 ASCII punctuation characters could be composed into ligatures.  */
      if (!composition_break_at_point
	  && !NILP (BVAR (current_buffer, enable_multibyte_characters))
	  && !NILP (Vauto_composition_mode))
	{
	  ptrdiff_t endpos = charpos + 10 * TEXT_PROP_DISTANCE_LIMIT;
	  bool found = false;

	  if (pos > ZV)
	    pos = ZV;
	  if (endpos > ZV)
	    endpos = ZV;
	  ptrdiff_t bpos = CHAR_TO_BYTE (pos);
	  while (pos < endpos)
	    {
	      int ch = fetch_char_advance_no_check (&pos, &bpos);
	      if (ch == ' ' || ch == '\t' || ch == '\n' || ch == '\f')
		{
		  found = true;
		  break;
		}
	    }
	  if (found)
	    pos--;
	  else if (it->stop_charpos < endpos)
	    pos = it->stop_charpos;
	  else
	    {
	      /* Give up and use the original arbitrary limit.  */
	      pos = charpos + TEXT_PROP_DISTANCE_LIMIT;
	    }
	}
      limit = make_fixnum (pos);
    }

  /* Get the interval containing IT's position.  Value is a null
     interval if there isn't such an interval.  */
  position = make_fixnum (charpos);
  iv = validate_interval_range (object, &position, &position, false);
  if (iv)
    {
      Lisp_Object values_here[LAST_PROP_IDX];
      struct props *p;

      /* Get properties here.  */
      for (p = it_props; p->handler; ++p)
	values_here[p->idx] = textget (iv->plist,
				       builtin_lisp_symbol (p->name));

      /* Look for an interval following iv that has different
	 properties.  */
      for (next_iv = next_interval (iv);
	   (next_iv
	    && (NILP (limit)
		|| XFIXNAT (limit) > next_iv->position));
	   next_iv = next_interval (next_iv))
	{
	  for (p = it_props; p->handler; ++p)
	    {
	      Lisp_Object new_value = textget (next_iv->plist,
					       builtin_lisp_symbol (p->name));
	      if (!EQ (values_here[p->idx], new_value))
		break;
	    }

	  if (p->handler)
	    break;
	}

      if (next_iv)
	{
	  if (FIXNUMP (limit)
	      && next_iv->position >= XFIXNAT (limit))
	    /* No text property change up to limit.  */
	    it->stop_charpos = min (XFIXNAT (limit), it->stop_charpos);
	  else
	    /* Text properties change in next_iv.  */
	    it->stop_charpos = min (it->stop_charpos, next_iv->position);
	}
    }

  if (it->cmp_it.id < 0)
    {
      ptrdiff_t stoppos = it->end_charpos;

      if (it->bidi_p && it->bidi_it.scan_dir < 0)
	stoppos = -1;
      composition_compute_stop_pos (&it->cmp_it, charpos, bytepos,
				    stoppos, it->string);
    }

  eassert (STRINGP (it->string)
	   || (it->stop_charpos >= BEGV
	       && it->stop_charpos >= IT_CHARPOS (*it)));
}

/* How many characters forward to search for a display property or
   display string.  Searching too far forward makes the bidi display
   sluggish, especially in small windows.  */
#define MAX_DISP_SCAN 250

/* Return the character position of a display string at or after
   position specified by POSITION.  If no display string exists at or
   after POSITION, return ZV.  A display string is either an overlay
   with `display' property whose value is a string, or a `display'
   text property whose value is a string.  STRING is data about the
   string to iterate; if STRING->lstring is nil, we are iterating a
   buffer.  FRAME_WINDOW_P is true when we are displaying a window
   on a GUI frame.  DISP_PROP is set to zero if we searched
   MAX_DISP_SCAN characters forward without finding any display
   strings, non-zero otherwise.  It is set to 2 if the display string
   uses any kind of `(space ...)' spec that will produce a stretch of
   white space in the text area.  */
ptrdiff_t
compute_display_string_pos (struct text_pos *position,
			    struct bidi_string_data *string,
			    struct window *w,
			    bool frame_window_p, int *disp_prop)
{
  /* OBJECT = nil means current buffer.  */
  Lisp_Object object, object1;
  Lisp_Object pos, spec, limpos;
  bool string_p = string && (STRINGP (string->lstring) || string->s);
  ptrdiff_t eob = string_p ? string->schars : ZV;
  ptrdiff_t begb = string_p ? 0 : BEGV;
  ptrdiff_t bufpos, charpos = CHARPOS (*position);
  ptrdiff_t lim =
    (charpos < eob - MAX_DISP_SCAN) ? charpos + MAX_DISP_SCAN : eob;
  struct text_pos tpos;
  int rv = 0;

  if (string && STRINGP (string->lstring))
    object1 = object = string->lstring;
  else if (w && !string_p)
    {
      XSETWINDOW (object, w);
      object1 = Qnil;
    }
  else
    object1 = object = Qnil;

  *disp_prop = 1;

  if (charpos >= eob
      /* We don't support display properties whose values are strings
	 that have display string properties.  */
      || string->from_disp_str
      /* C strings cannot have display properties.  */
      || (string->s && !STRINGP (object)))
    {
      *disp_prop = 0;
      return eob;
    }

  /* If the character at CHARPOS is where the display string begins,
     return CHARPOS.  */
  pos = make_fixnum (charpos);
  if (STRINGP (object))
    bufpos = string->bufpos;
  else
    bufpos = charpos;
  tpos = *position;
  if (!NILP (spec = Fget_char_property (pos, Qdisplay, object))
      && (charpos <= begb
	  || !EQ (Fget_char_property (make_fixnum (charpos - 1), Qdisplay,
				      object),
		  spec))
      && (rv = handle_display_spec (NULL, spec, object1, Qnil, &tpos, bufpos,
				    frame_window_p)))
    {
      if (rv == 2)
	*disp_prop = 2;
      return charpos;
    }

  /* Look forward for the first character with a `display' property
     that will replace the underlying text when displayed.  */
  limpos = make_fixnum (lim);
  do {
    pos = Fnext_single_char_property_change (pos, Qdisplay, object1, limpos);
    CHARPOS (tpos) = XFIXNAT (pos);
    if (CHARPOS (tpos) >= lim)
      {
	*disp_prop = 0;
	break;
      }
    if (STRINGP (object))
      BYTEPOS (tpos) = string_char_to_byte (object, CHARPOS (tpos));
    else
      BYTEPOS (tpos) = CHAR_TO_BYTE (CHARPOS (tpos));
    spec = Fget_char_property (pos, Qdisplay, object);
    if (!STRINGP (object))
      bufpos = CHARPOS (tpos);
  } while (NILP (spec)
	   || !(rv = handle_display_spec (NULL, spec, object1, Qnil, &tpos,
					  bufpos, frame_window_p)));
  if (rv == 2)
    *disp_prop = 2;

  return CHARPOS (tpos);
}

/* Return the character position of the end of the display string that
   started at CHARPOS.  If there's no display string at CHARPOS,
   return -1.  A display string is either an overlay with `display'
   property whose value is a string or a `display' text property whose
   value is a string.  */
ptrdiff_t
compute_display_string_end (ptrdiff_t charpos, struct bidi_string_data *string)
{
  /* OBJECT = nil means current buffer.  */
  Lisp_Object object =
    (string && STRINGP (string->lstring)) ? string->lstring : Qnil;
  Lisp_Object pos = make_fixnum (charpos);
  ptrdiff_t eob =
    (STRINGP (object) || (string && string->s)) ? string->schars : ZV;

  if (charpos >= eob || (string->s && !STRINGP (object)))
    return eob;

  /* It could happen that the display property or overlay was removed
     since we found it in compute_display_string_pos above.  One way
     this can happen is if JIT font-lock was called (through
     handle_fontified_prop), and jit-lock-functions remove text
     properties or overlays from the portion of buffer that includes
     CHARPOS.  Muse mode is known to do that, for example.  In this
     case, we return -1 to the caller, to signal that no display
     string is actually present at CHARPOS.  See bidi_fetch_char for
     how this is handled.

     An alternative would be to never look for display properties past
     it->stop_charpos.  But neither compute_display_string_pos nor
     bidi_fetch_char that calls it know or care where the next
     stop_charpos is.  */
  if (NILP (Fget_char_property (pos, Qdisplay, object)))
    return -1;

  /* Look forward for the first character where the `display' property
     changes.  */
  pos = Fnext_single_char_property_change (pos, Qdisplay, object, Qnil);

  return XFIXNAT (pos);
}



/***********************************************************************
			    Fontification
 ***********************************************************************/

/* Handle changes in the `fontified' property of the current buffer by
   calling hook functions from Qfontification_functions to fontify
   regions of text.  */

static enum prop_handled
handle_fontified_prop (struct it *it)
{
  Lisp_Object prop, pos;
  enum prop_handled handled = HANDLED_NORMALLY;

  if (!NILP (Vmemory_full))
    return handled;

  /* Get the value of the `fontified' property at IT's current buffer
     position.  (The `fontified' property doesn't have a special
     meaning in strings.)  If the value is nil, call functions from
     Qfontification_functions.  */
  if (!STRINGP (it->string)
      && it->s == NULL
      && !NILP (Vfontification_functions)
      && !(input_was_pending && redisplay_skip_fontification_on_input)
      && !NILP (Vrun_hooks)
      && (pos = make_fixnum (IT_CHARPOS (*it)),
	  prop = Fget_char_property (pos, Qfontified, Qnil),
	  /* Ignore the special cased nil value always present at EOB since
	     no amount of fontifying will be able to change it.  */
	  NILP (prop) && IT_CHARPOS (*it) < Z))
    {
      specpdl_ref count = SPECPDL_INDEX ();
      Lisp_Object val;
      struct buffer *obuf = current_buffer;
      ptrdiff_t begv = BEGV, zv = ZV;
      bool old_clip_changed = current_buffer->clip_changed;
      bool saved_inhibit_flag = it->f->inhibit_clear_image_cache;

      val = Vfontification_functions;
      specbind (Qfontification_functions, Qnil);

      eassert (it->end_charpos == ZV);

      if (current_buffer->long_line_optimizations_p
	  && long_line_optimizations_region_size > 0)
	{
	  ptrdiff_t begv = it->locked_narrowing_begv;
	  ptrdiff_t zv = it->locked_narrowing_zv;
	  ptrdiff_t charpos = IT_CHARPOS (*it);
	  if (charpos < begv || charpos > zv)
	    {
	      begv = get_locked_narrowing_begv (charpos);
	      zv = get_locked_narrowing_zv (charpos);
	    }
	  if (begv != BEG || zv != Z)
	    narrow_to_region_locked (make_fixnum (begv), make_fixnum (zv),
				     Qlong_line_optimizations_in_fontification_functions);
	}

      /* Don't allow Lisp that runs from 'fontification-functions'
	 clear our face and image caches behind our back.  */
      it->f->inhibit_clear_image_cache = true;

      if (!CONSP (val) || EQ (XCAR (val), Qlambda))
	safe_call1 (val, pos);
      else
	{
	  Lisp_Object fns, fn;

	  fns = Qnil;

	  for (; CONSP (val); val = XCDR (val))
	    {
	      fn = XCAR (val);

	      if (EQ (fn, Qt))
		{
		  /* A value of t indicates this hook has a local
		     binding; it means to run the global binding too.
		     In a global value, t should not occur.  If it
		     does, we must ignore it to avoid an endless
		     loop.  */
		  for (fns = Fdefault_value (Qfontification_functions);
		       CONSP (fns);
		       fns = XCDR (fns))
		    {
		      fn = XCAR (fns);
		      if (!EQ (fn, Qt))
			safe_call1 (fn, pos);
		    }
		}
	      else
		safe_call1 (fn, pos);
	    }
	}

      it->f->inhibit_clear_image_cache = saved_inhibit_flag;
      unbind_to (count, Qnil);

      /* Fontification functions routinely call `save-restriction'.
	 Normally, this tags clip_changed, which can confuse redisplay
	 (see discussion in Bug#6671).  Since we don't perform any
	 special handling of fontification changes in the case where
	 `save-restriction' isn't called, there's no point doing so in
	 this case either.  So, if the buffer's restrictions are
	 actually left unchanged, reset clip_changed.  */
      if (obuf == current_buffer)
      	{
      	  if (begv == BEGV && zv == ZV)
	    current_buffer->clip_changed = old_clip_changed;
      	}
      /* There isn't much we can reasonably do to protect against
      	 misbehaving fontification, but here's a fig leaf.  */
      else if (BUFFER_LIVE_P (obuf))
      	set_buffer_internal_1 (obuf);

      /* The fontification code may have added/removed text.
	 It could do even a lot worse, but let's at least protect against
	 the most obvious case where only the text past `pos' gets changed',
	 as is/was done in grep.el where some escapes sequences are turned
	 into face properties (bug#7876).  */
      it->end_charpos = ZV;

      /* Return HANDLED_RECOMPUTE_PROPS only if function fontified
	 something.  This avoids an endless loop if they failed to
	 fontify the text for which reason ever.  */
      if (!NILP (Fget_char_property (pos, Qfontified, Qnil)))
	handled = HANDLED_RECOMPUTE_PROPS;
    }

  return handled;
}



/***********************************************************************
				Faces
 ***********************************************************************/

static int
face_at_pos (const struct it *it, enum lface_attribute_index attr_filter)
{
  ptrdiff_t next_stop;

  if (!STRINGP (it->string))
    {
      return face_at_buffer_position (it->w,
                                      IT_CHARPOS (*it),
                                      &next_stop,
                                      (IT_CHARPOS (*it)
				       + TEXT_PROP_DISTANCE_LIMIT),
                                      false, it->base_face_id,
                                      attr_filter);
    }
  else
    {
      int base_face_id;
      ptrdiff_t bufpos;
      int i;
      Lisp_Object from_overlay
	= (it->current.overlay_string_index >= 0
	   ? it->string_overlays[it->current.overlay_string_index
	                         % OVERLAY_STRING_CHUNK_SIZE]
	   : Qnil);

      /* See if we got to this string directly or indirectly from
	 an overlay property.  That includes the before-string or
	 after-string of an overlay, strings in display properties
	 provided by an overlay, their text properties, etc.

	 FROM_OVERLAY is the overlay that brought us here, or nil if none.  */
      if (! NILP (from_overlay))
	for (i = it->sp - 1; i >= 0; i--)
	  {
	    if (it->stack[i].current.overlay_string_index >= 0)
	      from_overlay
		= it->string_overlays[it->stack[i].current.overlay_string_index
		                      % OVERLAY_STRING_CHUNK_SIZE];
	    else if (! NILP (it->stack[i].from_overlay))
	      from_overlay = it->stack[i].from_overlay;

	    if (!NILP (from_overlay))
	      break;
	  }

      if (! NILP (from_overlay))
	{
	  bufpos = IT_CHARPOS (*it);
	  /* For a string from an overlay, the base face depends
	     only on text properties and ignores overlays.  */
	  base_face_id
	    = face_for_overlay_string (it->w,
	                               IT_CHARPOS (*it),
	                               &next_stop,
	                               (IT_CHARPOS (*it)
				        + TEXT_PROP_DISTANCE_LIMIT),
	                               false,
	                               from_overlay, attr_filter);
	}
      else
	{
	  bufpos = 0;

	  /* For strings from a `display' property, use the face at
	     IT's current buffer position as the base face to merge
	     with, so that overlay strings appear in the same face as
	     surrounding text, unless they specify their own faces.
	     For strings from wrap-prefix and line-prefix properties,
	     use the default face, possibly remapped via
	     Vface_remapping_alist.  */
	  /* Note that the fact that we use the face at _buffer_
	     position means that a 'display' property on an overlay
	     string will not inherit the face of that overlay string,
	     but will instead revert to the face of buffer text
	     covered by the overlay.  This is visible, e.g., when the
	     overlay specifies a box face, but neither the buffer nor
	     the display string do.  This sounds like a design bug,
	     but Emacs always did that since v21.1, so changing that
	     might be a big deal.  */
	  base_face_id = it->string_from_prefix_prop_p
	    ? (!NILP (Vface_remapping_alist)
	       ? lookup_basic_face (it->w, it->f, DEFAULT_FACE_ID)
	       : DEFAULT_FACE_ID)
	    : underlying_face_id (it);
	}

      return face_at_string_position (it->w,
                                      it->string,
                                      IT_STRING_CHARPOS (*it),
                                      bufpos,
                                      &next_stop,
                                      base_face_id, false,
                                      attr_filter);
    } /* !STRINGP (it->string) */
}


/* Set up iterator IT from face properties at its current position.
   Called from handle_stop.  */
static enum prop_handled
handle_face_prop (struct it *it)
{
  specpdl_ref count = SPECPDL_INDEX ();
  /* Don't allow the user to quit out of face-merging code, in case
     this is called when redisplaying a non-selected window, with
     point temporarily moved to window-point.  */
  specbind (Qinhibit_quit, Qt);
  const int new_face_id = face_at_pos (it, 0);
  unbind_to (count, Qnil);


  /* Is this a start of a run of characters with box face?
     Caveat: this can be called for a freshly initialized
     iterator; face_id is -1 in this case.  We know that the new
     face will not change until limit, i.e. if the new face has a
     box, all characters up to limit will have one.  But, as
     usual, we don't know whether limit is really the end.  */
  if (new_face_id != it->face_id)
    {
      struct face *new_face = FACE_FROM_ID (it->f, new_face_id);
      /* If it->face_id is -1, old_face below will be NULL, see
	 the definition of FACE_FROM_ID_OR_NULL.  This will happen
	 if this is the initial call that gets the face.  */
      struct face *old_face = FACE_FROM_ID_OR_NULL (it->f, it->face_id);

      /* If the value of face_id of the iterator is -1, we have to
	 look in front of IT's position and see whether there is a
	 face there that's different from new_face_id.  */
      if (!STRINGP (it->string)
          && !old_face
          && IT_CHARPOS (*it) > BEG)
	{
	  const int prev_face_id = face_before_it_pos (it);

	  old_face = FACE_FROM_ID_OR_NULL (it->f, prev_face_id);
	}

      /* If the new face has a box, but the old face does not,
	 this is the start of a run of characters with box face,
	 i.e. this character has a shadow on the left side.  */
      it->face_id = new_face_id;
      /* Don't reset the start_of_box_run_p flag, only set it if
	 needed.  */
      if (!(it->start_of_box_run_p && old_face && old_face->box))
	it->start_of_box_run_p = (new_face->box != FACE_NO_BOX
				  && (old_face == NULL || !old_face->box));
      it->face_box_p = new_face->box != FACE_NO_BOX;
    }

  return HANDLED_NORMALLY;
}


/* Return the ID of the face ``underlying'' IT's current position,
   which is in a string.  If the iterator is associated with a
   buffer, return the face at IT's current buffer position.
   Otherwise, use the iterator's base_face_id.  */

static int
underlying_face_id (const struct it *it)
{
  int face_id = it->base_face_id, i;

  eassert (STRINGP (it->string));

  for (i = it->sp - 1; i >= 0; --i)
    if (NILP (it->stack[i].string))
      face_id = it->stack[i].face_id;

  return face_id;
}


/* Compute the face one character before or after the current position
   of IT, in the visual order.  BEFORE_P means get the face
   in front (to the left in L2R paragraphs, to the right in R2L
   paragraphs) of IT's screen position.  Value is the ID of the face.  */

static int
face_before_or_after_it_pos (struct it *it, bool before_p)
{
  int face_id, limit;
  ptrdiff_t next_check_charpos;
  struct it it_copy;
  void *it_copy_data = NULL;

  eassert (it->s == NULL);

  if (STRINGP (it->string))
    {
      ptrdiff_t bufpos, charpos;
      int base_face_id;

      /* No face change past the end of the string (for the case we
	 are padding with spaces).  No face change before the string
	 start.  Ignore face changes before the first visible
	 character on this display line.  */
      if (IT_STRING_CHARPOS (*it) >= SCHARS (it->string)
	  || (IT_STRING_CHARPOS (*it) == 0 && before_p)
	  || it->current_x <= it->first_visible_x)
	return it->face_id;

      if (!it->bidi_p)
	{
	  /* Set charpos to the position before or after IT's current
	     position, in the logical order, which in the non-bidi
	     case is the same as the visual order.  */
	  if (before_p)
	    charpos = IT_STRING_CHARPOS (*it) - 1;
	  else if (it->what == IT_COMPOSITION)
	    /* For composition, we must check the character after the
	       composition.  */
	    charpos = IT_STRING_CHARPOS (*it) + it->cmp_it.nchars;
	  else
	    charpos = IT_STRING_CHARPOS (*it) + 1;
	}
      else
	{
	  /* With bidi iteration, the character before the current in
	     the visual order cannot be found by simple iteration,
	     because "reverse" reordering is not supported.  Instead,
	     we need to start from the string beginning and go all the
	     way to the current string position, remembering the
	     visually-previous position.  We need to start from the
	     string beginning for the character after the current as
	     well, since the iterator state in IT may have been
	     pushed, and the bidi cache is no longer coherent with the
	     string's text.  */
	  SAVE_IT (it_copy, *it, it_copy_data);
	  IT_STRING_CHARPOS (it_copy) = 0;
	  bidi_init_it (0, 0, FRAME_WINDOW_P (it_copy.f), &it_copy.bidi_it);
	  it_copy.bidi_it.scan_dir = 0;

	  do
	    {
	      charpos = it_copy.bidi_it.charpos;
	      if (charpos >= SCHARS (it->string))
		break;
	      bidi_move_to_visually_next (&it_copy.bidi_it);
	    }
	  while (it_copy.bidi_it.charpos != IT_STRING_CHARPOS (*it));

	  if (!before_p)
	    {
	      /* Set charpos to the string position of the character
		 that comes after IT's current position in the visual
		 order.  */
	      int n = (it->what == IT_COMPOSITION ? it->cmp_it.nchars : 1);
	      /* If this is the first string character,
		 bidi_move_to_visually_next will deliver character at
		 current position without moving, so we need to enlarge N.  */
	      if (it_copy.bidi_it.first_elt)
		n++;
	      while (n--)
		bidi_move_to_visually_next (&it_copy.bidi_it);

	      charpos = it_copy.bidi_it.charpos;
	    }

	  RESTORE_IT (it, it, it_copy_data);
	}
      eassert (0 <= charpos && charpos <= SCHARS (it->string));

      if (it->current.overlay_string_index >= 0)
	bufpos = IT_CHARPOS (*it);
      else
	bufpos = 0;

      base_face_id = underlying_face_id (it);

      /* Get the face for ASCII, or unibyte.  */
      face_id = face_at_string_position (it->w, it->string, charpos,
                                         bufpos, &next_check_charpos,
                                         base_face_id, false, 0);

      /* Correct the face for charsets different from ASCII.  Do it
	 for the multibyte case only.  The face returned above is
	 suitable for unibyte text if IT->string is unibyte.  */
      if (STRING_MULTIBYTE (it->string))
	{
	  struct text_pos pos1 = string_pos (charpos, it->string);
	  const unsigned char *p = SDATA (it->string) + BYTEPOS (pos1);
	  struct face *face = FACE_FROM_ID (it->f, face_id);
	  int len, c = check_char_and_length (p, &len);
	  face_id = FACE_FOR_CHAR (it->f, face, c, charpos, it->string);
	}
    }
  else
    {
      struct text_pos pos;

      if ((IT_CHARPOS (*it) >= ZV && !before_p)
	  || (IT_CHARPOS (*it) <= BEGV && before_p))
	return it->face_id;

      limit = IT_CHARPOS (*it) + TEXT_PROP_DISTANCE_LIMIT;
      pos = it->current.pos;

      if (!it->bidi_p)
	{
	  if (before_p)
	    DEC_TEXT_POS (pos, it->multibyte_p);
	  else
	    {
	      if (it->what == IT_COMPOSITION)
		{
		  /* For composition, we must check the position after
		     the composition.  */
		  pos.charpos += it->cmp_it.nchars;
		  pos.bytepos += it->len;
		}
	      else
		INC_TEXT_POS (pos, it->multibyte_p);
	    }
	}
      else
	{
	  if (before_p)
	    {
	      int current_x;

	      /* With bidi iteration, the character before the current
		 in the visual order cannot be found by simple
		 iteration, because "reverse" reordering is not
		 supported.  Instead, we need to use the move_it_*
		 family of functions, and move to the previous
		 character starting from the beginning of the visual
		 line.  */
	      /* Ignore face changes before the first visible
		 character on this display line.  */
	      if (it->current_x <= it->first_visible_x)
		return it->face_id;
	      SAVE_IT (it_copy, *it, it_copy_data);
	      /* Implementation note: Since move_it_in_display_line
		 works in the iterator geometry, and thinks the first
		 character is always the leftmost, even in R2L lines,
		 we don't need to distinguish between the R2L and L2R
		 cases here.  */
	      current_x = it_copy.current_x;
	      move_it_vertically_backward (&it_copy, 0);
	      move_it_in_display_line (&it_copy, ZV, current_x - 1, MOVE_TO_X);
	      pos = it_copy.current.pos;
	      RESTORE_IT (it, it, it_copy_data);
	    }
	  else
	    {
	      /* Set charpos to the buffer position of the character
		 that comes after IT's current position in the visual
		 order.  */
	      int n = (it->what == IT_COMPOSITION ? it->cmp_it.nchars : 1);

	      it_copy = *it;
	      /* If this is the first display element,
		 bidi_move_to_visually_next will deliver character at
		 current position without moving, so we need to enlarge N.  */
	      if (it->bidi_it.first_elt)
		n++;
	      while (n--)
		bidi_move_to_visually_next (&it_copy.bidi_it);

	      SET_TEXT_POS (pos,
			    it_copy.bidi_it.charpos, it_copy.bidi_it.bytepos);
	    }
	}
      eassert (BEGV <= CHARPOS (pos) && CHARPOS (pos) <= ZV);

      /* Determine face for CHARSET_ASCII, or unibyte.  */
      face_id = face_at_buffer_position (it->w,
					 CHARPOS (pos),
					 &next_check_charpos,
                                         limit, false, -1, 0);

      /* Correct the face for charsets different from ASCII.  Do it
	 for the multibyte case only.  The face returned above is
	 suitable for unibyte text if current_buffer is unibyte.  */
      if (it->multibyte_p)
	{
	  int c = FETCH_MULTIBYTE_CHAR (BYTEPOS (pos));
	  struct face *face = FACE_FROM_ID (it->f, face_id);
	  face_id = FACE_FOR_CHAR (it->f, face, c, CHARPOS (pos), Qnil);
	}
    }

  return face_id;
}



/***********************************************************************
			    Invisible text
 ***********************************************************************/

/* Set up iterator IT from invisible properties at its current
   position.  Called from handle_stop.  */

static enum prop_handled
handle_invisible_prop (struct it *it)
{
  enum prop_handled handled = HANDLED_NORMALLY;
  int invis;
  Lisp_Object prop;

  if (STRINGP (it->string))
    {
      Lisp_Object end_charpos, limit;

      /* Get the value of the invisible text property at the
	 current position.  Value will be nil if there is no such
	 property.  */
      end_charpos = make_fixnum (IT_STRING_CHARPOS (*it));
      prop = Fget_text_property (end_charpos, Qinvisible, it->string);
      invis = TEXT_PROP_MEANS_INVISIBLE (prop);

      if (invis != 0 && IT_STRING_CHARPOS (*it) < it->end_charpos)
	{
	  /* Record whether we have to display an ellipsis for the
	     invisible text.  */
	  bool display_ellipsis_p = (invis == 2);
	  ptrdiff_t len, endpos;

	  handled = HANDLED_RECOMPUTE_PROPS;

	  /* Get the position at which the next visible text can be
	     found in IT->string, if any.  */
	  endpos = len = SCHARS (it->string);
	  XSETINT (limit, len);
	  do
	    {
	      end_charpos
		= Fnext_single_property_change (end_charpos, Qinvisible,
						it->string, limit);
	      /* Since LIMIT is always an integer, so should be the
		 value returned by Fnext_single_property_change.  */
	      eassert (FIXNUMP (end_charpos));
	      if (FIXNUMP (end_charpos))
		{
		  endpos = XFIXNAT (end_charpos);
		  prop = Fget_text_property (end_charpos, Qinvisible, it->string);
		  invis = TEXT_PROP_MEANS_INVISIBLE (prop);
		  if (invis == 2)
		    display_ellipsis_p = true;
		}
	      else /* Should never happen; but if it does, exit the loop.  */
		endpos = len;
	    }
	  while (invis != 0 && endpos < len);

	  if (display_ellipsis_p)
	    it->ellipsis_p = true;

	  if (endpos < len)
	    {
	      /* Text at END_CHARPOS is visible.  Move IT there.  */
	      struct text_pos old;
	      ptrdiff_t oldpos;

	      old = it->current.string_pos;
	      oldpos = CHARPOS (old);
	      if (it->bidi_p)
		{
		  if (it->bidi_it.first_elt
		      && it->bidi_it.charpos < SCHARS (it->string))
		    bidi_paragraph_init (it->paragraph_embedding,
					 &it->bidi_it, true);
		  /* Bidi-iterate out of the invisible text.  */
		  do
		    {
		      bidi_move_to_visually_next (&it->bidi_it);
		    }
		  while (oldpos <= it->bidi_it.charpos
			 && it->bidi_it.charpos < endpos
			 && it->bidi_it.charpos < it->bidi_it.string.schars);

		  IT_STRING_CHARPOS (*it) = it->bidi_it.charpos;
		  IT_STRING_BYTEPOS (*it) = it->bidi_it.bytepos;
		  if (IT_CHARPOS (*it) >= endpos)
		    it->prev_stop = endpos;
		}
	      else
		{
		  IT_STRING_CHARPOS (*it) = endpos;
		  compute_string_pos (&it->current.string_pos, old, it->string);
		}
	    }
	  else
	    {
	      /* The rest of the string is invisible.  If this is an
		 overlay string, proceed with the next overlay string
		 or whatever comes and return a character from there.  */
	      if (it->current.overlay_string_index >= 0
		  && !display_ellipsis_p)
		{
		  next_overlay_string (it);
		  /* Don't check for overlay strings when we just
		     finished processing them.  */
		  handled = HANDLED_OVERLAY_STRING_CONSUMED;
		}
	      else
		{
		  IT_STRING_CHARPOS (*it) = SCHARS (it->string);
		  IT_STRING_BYTEPOS (*it) = SBYTES (it->string);
		}
	    }
	}
    }
  else
    {
      ptrdiff_t newpos, next_stop, start_charpos, tem;
      Lisp_Object pos, overlay;

      /* First of all, is there invisible text at this position?  */
      tem = start_charpos = IT_CHARPOS (*it);
      pos = make_fixnum (tem);
      prop = get_char_property_and_overlay (pos, Qinvisible, it->window,
					    &overlay);
      invis = TEXT_PROP_MEANS_INVISIBLE (prop);

      /* If we are on invisible text, skip over it.  */
      if (invis != 0 && start_charpos < it->end_charpos)
	{
	  /* Record whether we have to display an ellipsis for the
	     invisible text.  */
	  bool display_ellipsis_p = invis == 2;

	  handled = HANDLED_RECOMPUTE_PROPS;

	  /* Loop skipping over invisible text.  The loop is left at
	     ZV or with IT on the first char being visible again.  */
	  do
	    {
	      /* Try to skip some invisible text.  Return value is the
		 position reached which can be equal to where we start
		 if there is nothing invisible there.  This skips both
		 over invisible text properties and overlays with
		 invisible property.  */
	      newpos = skip_invisible (tem, &next_stop, ZV, it->window);

	      /* If we skipped nothing at all we weren't at invisible
		 text in the first place.  If everything to the end of
		 the buffer was skipped, end the loop.  */
	      if (newpos == tem || newpos >= ZV)
		invis = 0;
	      else
		{
		  /* We skipped some characters but not necessarily
		     all there are.  Check if we ended up on visible
		     text.  Fget_char_property returns the property of
		     the char before the given position, i.e. if we
		     get invis = 0, this means that the char at
		     newpos is visible.  */
		  pos = make_fixnum (newpos);
		  prop = Fget_char_property (pos, Qinvisible, it->window);
		  invis = TEXT_PROP_MEANS_INVISIBLE (prop);
		}

	      /* If we ended up on invisible text, proceed to
		 skip starting with next_stop.  */
	      if (invis != 0)
		tem = next_stop;

              /* If there are adjacent invisible texts, don't lose the
                 second one's ellipsis.  */
              if (invis == 2)
                display_ellipsis_p = true;
	    }
	  while (invis != 0);

	  /* The position newpos is now either ZV or on visible text.  */
	  if (it->bidi_p)
	    {
	      ptrdiff_t bpos = CHAR_TO_BYTE (newpos);
	      bool on_newline
		= bpos == ZV_BYTE || FETCH_BYTE (bpos) == '\n';
	      bool after_newline
		= newpos <= BEGV || FETCH_BYTE (bpos - 1) == '\n';

	      /* If the invisible text ends on a newline or on a
		 character after a newline, we can avoid the costly,
		 character by character, bidi iteration to NEWPOS, and
		 instead simply reseat the iterator there.  That's
		 because all bidi reordering information is tossed at
		 the newline.  This is a big win for modes that hide
		 complete lines, like Outline, Org, etc.  */
	      if (on_newline || after_newline)
		{
		  struct text_pos tpos;
		  bidi_dir_t pdir = it->bidi_it.paragraph_dir;

		  SET_TEXT_POS (tpos, newpos, bpos);
		  reseat_1 (it, tpos, false);
		  /* If we reseat on a newline/ZV, we need to prep the
		     bidi iterator for advancing to the next character
		     after the newline/EOB, keeping the current paragraph
		     direction (so that PRODUCE_GLYPHS does TRT wrt
		     prepending/appending glyphs to a glyph row).  */
		  if (on_newline)
		    {
		      it->bidi_it.first_elt = false;
		      it->bidi_it.paragraph_dir = pdir;
		      it->bidi_it.ch = (bpos == ZV_BYTE) ? -1 : '\n';
		      it->bidi_it.nchars = 1;
		      it->bidi_it.ch_len = 1;
		    }
		}
	      else	/* Must use the slow method.  */
		{
		  /* With bidi iteration, the region of invisible text
		     could start and/or end in the middle of a
		     non-base embedding level.  Therefore, we need to
		     skip invisible text using the bidi iterator,
		     starting at IT's current position, until we find
		     ourselves outside of the invisible text.
		     Skipping invisible text _after_ bidi iteration
		     avoids affecting the visual order of the
		     displayed text when invisible properties are
		     added or removed.  */
		  if (it->bidi_it.first_elt && it->bidi_it.charpos < ZV)
		    {
		      /* If we were `reseat'ed to a new paragraph,
			 determine the paragraph base direction.  We
			 need to do it now because
			 next_element_from_buffer may not have a
			 chance to do it, if we are going to skip any
			 text at the beginning, which resets the
			 FIRST_ELT flag.  */
		      bidi_paragraph_init (it->paragraph_embedding,
					   &it->bidi_it, true);
		    }
		  do
		    {
		      bidi_move_to_visually_next (&it->bidi_it);
		    }
		  while (it->stop_charpos <= it->bidi_it.charpos
			 && it->bidi_it.charpos < newpos);
		  IT_CHARPOS (*it) = it->bidi_it.charpos;
		  IT_BYTEPOS (*it) = it->bidi_it.bytepos;
		  /* If we overstepped NEWPOS, record its position in
		     the iterator, so that we skip invisible text if
		     later the bidi iteration lands us in the
		     invisible region again. */
		  if (IT_CHARPOS (*it) >= newpos)
		    it->prev_stop = newpos;
		}
	    }
	  else
	    {
	      IT_CHARPOS (*it) = newpos;
	      IT_BYTEPOS (*it) = CHAR_TO_BYTE (newpos);
	    }

	  if (display_ellipsis_p)
            {
              /* Make sure that the glyphs of the ellipsis will get
                 correct `charpos' values.  If we would not update
                 it->position here, the glyphs would belong to the
                 last visible character _before_ the invisible
                 text, which confuses `set_cursor_from_row'.

                 We use the last invisible position instead of the
                 first because this way the cursor is always drawn on
                 the first "." of the ellipsis, whenever PT is inside
                 the invisible text.  Otherwise the cursor would be
                 placed _after_ the ellipsis when the point is after the
                 first invisible character.  */
	      if (!STRINGP (it->object))
		{
		  it->position.charpos = newpos - 1;
		  it->position.bytepos = CHAR_TO_BYTE (it->position.charpos);
		}
	    }

	  /* If there are before-strings at the start of invisible
	     text, and the text is invisible because of a text
	     property, arrange to show before-strings because 20.x did
	     it that way.  (If the text is invisible because of an
	     overlay property instead of a text property, this is
	     already handled in the overlay code.)  */
	  if (NILP (overlay)
	      && get_overlay_strings (it, it->stop_charpos))
	    {
	      handled = HANDLED_RECOMPUTE_PROPS;
	      if (it->sp > 0)
		{
		  it->stack[it->sp - 1].display_ellipsis_p = display_ellipsis_p;
		  /* The call to get_overlay_strings above recomputes
		     it->stop_charpos, but it only considers changes
		     in properties and overlays beyond iterator's
		     current position.  This causes us to miss changes
		     that happen exactly where the invisible property
		     ended.  So we play it safe here and force the
		     iterator to check for potential stop positions
		     immediately after the invisible text.  Note that
		     if get_overlay_strings returns true, it
		     normally also pushed the iterator stack, so we
		     need to update the stop position in the slot
		     below the current one.  */
		  it->stack[it->sp - 1].stop_charpos
		    = CHARPOS (it->stack[it->sp - 1].current.pos);
		}
	    }
	  else if (display_ellipsis_p)
            {
	      it->ellipsis_p = true;
	      /* Let the ellipsis display before
		 considering any properties of the following char.
		 Fixes jasonr@gnu.org 01 Oct 07 bug.  */
	      handled = HANDLED_RETURN;
            }
	}
    }

  return handled;
}


/* Make iterator IT return `...' next.
   Replaces LEN characters from buffer.  */

static void
setup_for_ellipsis (struct it *it, int len)
{
  /* Use the display table definition for `...'.  Invalid glyphs
     will be handled by the method returning elements from dpvec.  */
  if (it->dp && VECTORP (DISP_INVIS_VECTOR (it->dp)))
    {
      struct Lisp_Vector *v = XVECTOR (DISP_INVIS_VECTOR (it->dp));
      it->dpvec = v->contents;
      it->dpend = v->contents + v->header.size;
    }
  else
    {
      /* Default `...'.  */
      it->dpvec = default_invis_vector;
      it->dpend = default_invis_vector + 3;
    }

  it->dpvec_char_len = len;
  it->current.dpvec_index = 0;
  it->dpvec_face_id = -1;

  /* Use IT->saved_face_id for the ellipsis, so that it has the same
     face as the preceding text.  IT->saved_face_id was set in
     handle_stop to the face of the preceding character, and will be
     different from IT->face_id only if the invisible text skipped in
     handle_invisible_prop has some non-default face on its first
     character.  We thus ignore the face of the invisible text when we
     display the ellipsis.  IT's face is restored in set_iterator_to_next.  */
  if (it->saved_face_id >= 0)
    it->face_id = it->saved_face_id;

  /* If the ellipsis represents buffer text, it means we advanced in
     the buffer, so we should no longer ignore overlay strings.  */
  if (it->method == GET_FROM_BUFFER)
    it->ignore_overlay_strings_at_pos_p = false;

  it->method = GET_FROM_DISPLAY_VECTOR;
  it->ellipsis_p = true;
}


static Lisp_Object
find_display_property (Lisp_Object disp, Lisp_Object prop)
{
  if (NILP (disp))
    return Qnil;
  /* We have a vector of display specs.  */
  if (VECTORP (disp))
    {
      for (ptrdiff_t i = 0; i < ASIZE (disp); i++)
	{
	  Lisp_Object elem = AREF (disp, i);
	  if (CONSP (elem)
	      && CONSP (XCDR (elem))
	      && EQ (XCAR (elem), prop))
	    return XCAR (XCDR (elem));
	}
      return Qnil;
    }
  /* We have a list of display specs.  */
  else if (CONSP (disp)
	   && CONSP (XCAR (disp)))
    {
      while (!NILP (disp))
	{
	  Lisp_Object elem = XCAR (disp);
	  if (CONSP (elem)
	      && CONSP (XCDR (elem))
	      && EQ (XCAR (elem), prop))
	    return XCAR (XCDR (elem));

	  /* Check that we have a proper list before going to the next
	     element.  */
	  if (CONSP (XCDR (disp)))
	    disp = XCDR (disp);
	  else
	    disp = Qnil;
	}
      return Qnil;
    }
  /* A simple display spec.  */
  else if (CONSP (disp)
	   && CONSP (XCDR (disp))
	   && EQ (XCAR (disp), prop))
    return XCAR (XCDR (disp));
  else
    return Qnil;
}

static Lisp_Object
get_display_property (ptrdiff_t bufpos, Lisp_Object prop, Lisp_Object object)
{
  return find_display_property (Fget_text_property (make_fixnum (bufpos),
						    Qdisplay, object),
				prop);
}

static void
display_min_width (struct it *it, ptrdiff_t bufpos,
		   Lisp_Object object, Lisp_Object width_spec)
{
  /* We're being called at the end of the `min-width' sequence,
     probably. */
  if (!NILP (it->min_width_property)
      && !EQ (width_spec, it->min_width_property))
    {
      if (!it->glyph_row)
	return;

      /* When called from display_string (i.e., the mode line),
	 we're being called with a string as the object, and we
	 may be called with many sub-strings belonging to the same
	 :propertize run. */
      if ((bufpos == 0
	   && !EQ (it->min_width_property,
		   get_display_property (0, Qmin_width, object)))
	  /* In a buffer -- check that we're really right after the
	     sequence of characters covered by this `min-width'.  */
	  || (bufpos > BEGV
	      && EQ (it->min_width_property,
		     get_display_property (bufpos - 1, Qmin_width, object))))
	{
	  Lisp_Object w = Qnil;
	  double width;
#ifdef HAVE_WINDOW_SYSTEM
	  if (FRAME_WINDOW_P (it->f))
	    {
	      struct font *font = NULL;
	      struct face *face = FACE_FROM_ID (it->f, it->face_id);
	      font = face->font ? face->font : FRAME_FONT (it->f);
	      calc_pixel_width_or_height (&width, it,
					  XCAR (it->min_width_property),
					  font, true, NULL);
	      width -= it->current_x - it->min_width_start;
	      w = list1 (make_int (width));
	    }
	  else
#endif
	    {
	      calc_pixel_width_or_height (&width, it,
					  XCAR (it->min_width_property),
					  NULL, true, NULL);
	      width -= (it->current_x - it->min_width_start) /
		FRAME_COLUMN_WIDTH (it->f);
	      w = make_int (width);
	    }

	  /* Insert the stretch glyph.  */
	  it->object = list3 (Qspace, QCwidth, w);
	  produce_stretch_glyph (it);
	  if (it->area == TEXT_AREA)
	    it->current_x += it->pixel_width;
	  it->min_width_property = Qnil;
	}
    }

  /* We're at the start of a `min-width' sequence -- record the
     position and the property, so that we can later see if we're at
     the end.  */
  if (CONSP (width_spec))
    {
      if (bufpos == BEGV
	  /* Mode line (see above).  */
	  || (bufpos == 0
	      && !EQ (it->min_width_property,
		      get_display_property (0, Qmin_width, object)))
	  /* Buffer.  */
	  || (bufpos > BEGV
	      && !EQ (width_spec,
		      get_display_property (bufpos - 1, Qmin_width, object))))
	{
	  it->min_width_property = width_spec;
	  it->min_width_start = it->current_x;
	}
    }
}

DEFUN ("get-display-property", Fget_display_property,
       Sget_display_property, 2, 4, 0,
       doc: /* Get the value of the `display' property PROP at POSITION.
If OBJECT, this should be a buffer or string where the property is
fetched from.  If omitted, OBJECT defaults to the current buffer.

If PROPERTIES, look for value of PROP in PROPERTIES instead of the
properties at POSITION.  */)
  (Lisp_Object position, Lisp_Object prop, Lisp_Object object,
   Lisp_Object properties)
{
  if (NILP (properties))
    properties = Fget_text_property (position, Qdisplay, object);
  else
    CHECK_LIST (properties);

  return find_display_property (properties, prop);
}



/***********************************************************************
			    'display' property
 ***********************************************************************/

/* Set up iterator IT from `display' property at its current position.
   Called from handle_stop.
   We return HANDLED_RETURN if some part of the display property
   overrides the display of the buffer text itself.
   Otherwise we return HANDLED_NORMALLY.  */

static enum prop_handled
handle_display_prop (struct it *it)
{
  Lisp_Object propval, object, overlay;
  struct text_pos *position;
  ptrdiff_t bufpos;
  /* Nonzero if some property replaces the display of the text itself.  */
  int display_replaced = 0;

  if (STRINGP (it->string))
    {
      object = it->string;
      position = &it->current.string_pos;
      bufpos = CHARPOS (it->current.pos);
    }
  else
    {
      XSETWINDOW (object, it->w);
      position = &it->current.pos;
      bufpos = CHARPOS (*position);
    }

  /* Reset those iterator values set from display property values.  */
  it->slice.x = it->slice.y = it->slice.width = it->slice.height = Qnil;
  it->space_width = Qnil;
  it->font_height = Qnil;
  it->voffset = 0;

  /* We don't support recursive `display' properties, i.e. string
     values that have a string `display' property, that have a string
     `display' property etc.  */
  if (!it->string_from_display_prop_p)
    it->area = TEXT_AREA;

  propval = get_char_property_and_overlay (make_fixnum (position->charpos),
					   Qdisplay, object, &overlay);

  /* Rest of the code must have OBJECT be either a string or a buffer.  */
  if (!STRINGP (it->string))
    object = it->w->contents;

  /* Handle min-width ends. */
  if (!NILP (it->min_width_property)
      && NILP (find_display_property (propval, Qmin_width)))
    display_min_width (it, bufpos, object, Qnil);

  if (NILP (propval))
    return HANDLED_NORMALLY;
  /* Now OVERLAY is the overlay that gave us this property, or nil
     if it was a text property.  */

  display_replaced = handle_display_spec (it, propval, object, overlay,
					  position, bufpos,
					  FRAME_WINDOW_P (it->f));
  return display_replaced != 0 ? HANDLED_RETURN : HANDLED_NORMALLY;
}

/* Subroutine of handle_display_prop.  Returns non-zero if the display
   specification in SPEC is a replacing specification, i.e. it would
   replace the text covered by `display' property with something else,
   such as an image or a display string.  If SPEC includes any kind or
   `(space ...) specification, the value is 2; this is used by
   compute_display_string_pos, which see.

   See handle_single_display_spec for documentation of arguments.
   FRAME_WINDOW_P is true if the window being redisplayed is on a
   GUI frame; this argument is used only if IT is NULL, see below.

   IT can be NULL, if this is called by the bidi reordering code
   through compute_display_string_pos, which see.  In that case, this
   function only examines SPEC, but does not otherwise "handle" it, in
   the sense that it doesn't set up members of IT from the display
   spec.  */
static int
handle_display_spec (struct it *it, Lisp_Object spec, Lisp_Object object,
		     Lisp_Object overlay, struct text_pos *position,
		     ptrdiff_t bufpos, bool frame_window_p)
{
  int replacing = 0;
  bool enable_eval = true;

  /* Support (disable-eval PROP) which is used by enriched.el.  */
  if (CONSP (spec) && EQ (XCAR (spec), Qdisable_eval))
    {
      enable_eval = false;
      spec = CONSP (XCDR (spec)) ? XCAR (XCDR (spec)) : Qnil;
    }

  if (CONSP (spec)
      /* Simple specifications.  */
      && !EQ (XCAR (spec), Qimage)
#ifdef HAVE_XWIDGETS
      && !EQ (XCAR (spec), Qxwidget)
#endif
      && !EQ (XCAR (spec), Qspace)
      && !EQ (XCAR (spec), Qwhen)
      && !EQ (XCAR (spec), Qslice)
      && !EQ (XCAR (spec), Qspace_width)
      && !EQ (XCAR (spec), Qheight)
      && !EQ (XCAR (spec), Qraise)
      /* Marginal area specifications.  */
      && !(CONSP (XCAR (spec)) && EQ (XCAR (XCAR (spec)), Qmargin))
      && !EQ (XCAR (spec), Qleft_fringe)
      && !EQ (XCAR (spec), Qright_fringe)
      && !EQ (XCAR (spec), Qmin_width)
      && !NILP (XCAR (spec)))
    {
      for (; CONSP (spec); spec = XCDR (spec))
	{
	  int rv = handle_single_display_spec (it, XCAR (spec), object,
					       overlay, position, bufpos,
					       replacing, frame_window_p,
					       enable_eval);
	  if (rv != 0)
	    {
	      replacing = rv;
	      /* If some text in a string is replaced, `position' no
		 longer points to the position of `object'.  */
	      if (!it || STRINGP (object))
		break;
	    }
	}
    }
  else if (VECTORP (spec))
    {
      ptrdiff_t i;
      for (i = 0; i < ASIZE (spec); ++i)
	{
	  int rv = handle_single_display_spec (it, AREF (spec, i), object,
					       overlay, position, bufpos,
					       replacing, frame_window_p,
					       enable_eval);
	  if (rv != 0)
	    {
	      replacing = rv;
	      /* If some text in a string is replaced, `position' no
		 longer points to the position of `object'.  */
	      if (!it || STRINGP (object))
		break;
	    }
	}
    }
  else
    replacing = handle_single_display_spec (it, spec, object, overlay, position,
					    bufpos, 0, frame_window_p,
					    enable_eval);
  return replacing;
}

/* Value is the position of the end of the `display' property starting
   at START_POS in OBJECT.  */

static struct text_pos
display_prop_end (struct it *it, Lisp_Object object, struct text_pos start_pos)
{
  Lisp_Object end;
  struct text_pos end_pos;

  end = Fnext_single_char_property_change (make_fixnum (CHARPOS (start_pos)),
					   Qdisplay, object, Qnil);
  CHARPOS (end_pos) = XFIXNAT (end);
  if (STRINGP (object))
    compute_string_pos (&end_pos, start_pos, it->string);
  else
    BYTEPOS (end_pos) = CHAR_TO_BYTE (XFIXNAT (end));

  return end_pos;
}


/* Set up IT from a single `display' property specification SPEC.  OBJECT
   is the object in which the `display' property was found.  *POSITION
   is the position in OBJECT at which the `display' property was found.
   BUFPOS is the buffer position of OBJECT (different from POSITION if
   OBJECT is not a buffer).  DISPLAY_REPLACED non-zero means that we
   previously saw a display specification which already replaced text
   display with something else, for example an image; we ignore such
   properties after the first one has been processed.

   OVERLAY is the overlay this `display' property came from,
   or nil if it was a text property.

   If SPEC is a `space' or `image' specification, and in some other
   cases too, set *POSITION to the position where the `display'
   property ends.

   If IT is NULL, only examine the property specification in SPEC, but
   don't set up IT.  In that case, FRAME_WINDOW_P means SPEC
   is intended to be displayed in a window on a GUI frame.

   Enable evaluation of Lisp forms only if ENABLE_EVAL_P is true.

   Value is non-zero if something was found which replaces the display
   of buffer or string text.  */

static int
handle_single_display_spec (struct it *it, Lisp_Object spec, Lisp_Object object,
			    Lisp_Object overlay, struct text_pos *position,
			    ptrdiff_t bufpos, int display_replaced,
			    bool frame_window_p, bool enable_eval_p)
{
  Lisp_Object form;
  Lisp_Object location, value;
  struct text_pos start_pos = *position;
  void *itdata = NULL;

  /* If SPEC is a list of the form `(when FORM . VALUE)', evaluate FORM.
     If the result is non-nil, use VALUE instead of SPEC.  */
  form = Qt;
  if (CONSP (spec) && EQ (XCAR (spec), Qwhen))
    {
      spec = XCDR (spec);
      if (!CONSP (spec))
	return 0;
      form = XCAR (spec);
      spec = XCDR (spec);
    }

  if (!NILP (form) && !EQ (form, Qt) && !enable_eval_p)
    form = Qnil;
  if (!NILP (form) && !EQ (form, Qt))
    {
      specpdl_ref count = SPECPDL_INDEX ();

      /* Bind `object' to the object having the `display' property, a
	 buffer or string.  Bind `position' to the position in the
	 object where the property was found, and `buffer-position'
	 to the current position in the buffer.  */

      if (NILP (object))
	XSETBUFFER (object, current_buffer);
      specbind (Qobject, object);
      specbind (Qposition, make_fixnum (CHARPOS (*position)));
      specbind (Qbuffer_position, make_fixnum (bufpos));
      /* Save and restore the bidi cache, since FORM could be crazy
	 enough to re-enter redisplay, e.g., by calling 'message'.  */
      itdata = bidi_shelve_cache ();
      form = safe_eval (form);
      bidi_unshelve_cache (itdata, false);
      form = unbind_to (count, form);
    }

  if (NILP (form))
    return 0;

  /* Handle `(height HEIGHT)' specifications.  */
  if (CONSP (spec)
      && EQ (XCAR (spec), Qheight)
      && CONSP (XCDR (spec)))
    {
      if (it)
	{
	  if (!FRAME_WINDOW_P (it->f))
	    return 0;

	  it->font_height = XCAR (XCDR (spec));
	  if (!NILP (it->font_height))
	    {
	      int new_height = -1;

	      if (CONSP (it->font_height)
		  && (EQ (XCAR (it->font_height), Qplus)
		      || EQ (XCAR (it->font_height), Qminus))
		  && CONSP (XCDR (it->font_height))
		  && RANGED_FIXNUMP (0, XCAR (XCDR (it->font_height)), INT_MAX))
		{
		  /* `(+ N)' or `(- N)' where N is an integer.  */
		  int steps = XFIXNUM (XCAR (XCDR (it->font_height)));
		  if (EQ (XCAR (it->font_height), Qplus))
		    steps = - steps;
		  it->face_id = smaller_face (it->f, it->face_id, steps);
		}
	      else if (FUNCTIONP (it->font_height) && enable_eval_p)
		{
		  /* Call function with current height as argument.
		     Value is the new height.  */
		  struct face *face = FACE_FROM_ID (it->f, it->face_id);
		  Lisp_Object height;
		  itdata = bidi_shelve_cache ();
		  height = safe_call1 (it->font_height,
				       face->lface[LFACE_HEIGHT_INDEX]);
		  bidi_unshelve_cache (itdata, false);
		  if (NUMBERP (height))
		    new_height = XFLOATINT (height);
		}
	      else if (NUMBERP (it->font_height))
		{
		  /* Value is a multiple of the canonical char height.  */
		  struct face *f;

		  f = FACE_FROM_ID (it->f,
				    lookup_basic_face (it->w, it->f, DEFAULT_FACE_ID));
		  new_height = (XFLOATINT (it->font_height)
				* XFIXNUM (f->lface[LFACE_HEIGHT_INDEX]));
		}
	      else if (enable_eval_p)
		{
		  /* Evaluate IT->font_height with `height' bound to the
		     current specified height to get the new height.  */
		  specpdl_ref count = SPECPDL_INDEX ();
		  struct face *face = FACE_FROM_ID (it->f, it->face_id);

		  specbind (Qheight, face->lface[LFACE_HEIGHT_INDEX]);
		  itdata = bidi_shelve_cache ();
		  value = safe_eval (it->font_height);
		  bidi_unshelve_cache (itdata, false);
		  value = unbind_to (count, value);

		  if (NUMBERP (value))
		    new_height = XFLOATINT (value);
		}

	      if (new_height > 0)
		it->face_id = face_with_height (it->f, it->face_id, new_height);
	    }
	}

      return 0;
    }

  /* Handle `(space-width WIDTH)'.  */
  if (CONSP (spec)
      && EQ (XCAR (spec), Qspace_width)
      && CONSP (XCDR (spec)))
    {
      if (it)
	{
	  if (!FRAME_WINDOW_P (it->f))
	    return 0;

	  value = XCAR (XCDR (spec));
	  if (NUMBERP (value) && XFLOATINT (value) > 0)
	    it->space_width = value;
	}

      return 0;
    }

  /* Handle `(min-width (WIDTH))'.  */
  if (CONSP (spec)
      && EQ (XCAR (spec), Qmin_width)
      && CONSP (XCDR (spec))
      && CONSP (XCAR (XCDR (spec))))
    {
      if (it)
	display_min_width (it, bufpos, object, XCAR (XCDR (spec)));
      return 0;
    }

  /* Handle `(slice X Y WIDTH HEIGHT)'.  */
  if (CONSP (spec)
      && EQ (XCAR (spec), Qslice))
    {
      Lisp_Object tem;

      if (it)
	{
	  if (!FRAME_WINDOW_P (it->f))
	    return 0;

	  if (tem = XCDR (spec), CONSP (tem))
	    {
	      it->slice.x = XCAR (tem);
	      if (tem = XCDR (tem), CONSP (tem))
		{
		  it->slice.y = XCAR (tem);
		  if (tem = XCDR (tem), CONSP (tem))
		    {
		      it->slice.width = XCAR (tem);
		      if (tem = XCDR (tem), CONSP (tem))
			it->slice.height = XCAR (tem);
		    }
		}
	    }
	}

      return 0;
    }

  /* Handle `(raise FACTOR)'.  */
  if (CONSP (spec)
      && EQ (XCAR (spec), Qraise)
      && CONSP (XCDR (spec)))
    {
      if (it)
	{
	  if (!FRAME_WINDOW_P (it->f))
	    return 0;

#ifdef HAVE_WINDOW_SYSTEM
	  value = XCAR (XCDR (spec));
	  if (NUMBERP (value))
	    {
	      struct face *face = FACE_FROM_ID (it->f, it->face_id);
	      it->voffset = - (XFLOATINT (value)
			       * (normal_char_height (face->font, -1)));
	    }
#endif /* HAVE_WINDOW_SYSTEM */
	}

      return 0;
    }

  /* Don't handle the other kinds of display specifications
     inside a string that we got from a `display' property.  */
  if (it && it->string_from_display_prop_p)
    return 0;

  /* Characters having this form of property are not displayed, so
     we have to find the end of the property.  */
  if (it)
    {
      start_pos = *position;
      *position = display_prop_end (it, object, start_pos);
      /* If the display property comes from an overlay, don't consider
	 any potential stop_charpos values before the end of that
	 overlay.  Since display_prop_end will happily find another
	 'display' property coming from some other overlay or text
	 property on buffer positions before this overlay's end, we
	 need to ignore them, or else we risk displaying this
	 overlay's display string/image twice.  */
      if (!NILP (overlay))
	{
	  ptrdiff_t ovendpos = OVERLAY_END (overlay);

	  /* Some borderline-sane Lisp might call us with the current
	     buffer narrowed so that overlay-end is outside the
	     POINT_MIN..POINT_MAX region, which will then cause
	     various assertion violations and crashes down the road,
	     starting with pop_it when it will attempt to use POSITION
	     set below.  Prevent that.  */
	  ovendpos = clip_to_bounds (BEGV, ovendpos, ZV);

	  if (ovendpos > CHARPOS (*position))
	    SET_TEXT_POS (*position, ovendpos, CHAR_TO_BYTE (ovendpos));
	}
    }
  value = Qnil;

  /* Stop the scan at that end position--we assume that all
     text properties change there.  */
  if (it)
    it->stop_charpos = position->charpos;

  /* Handle `(left-fringe BITMAP [FACE])'
     and `(right-fringe BITMAP [FACE])'.  */
  if (CONSP (spec)
      && (EQ (XCAR (spec), Qleft_fringe)
	  || EQ (XCAR (spec), Qright_fringe))
      && CONSP (XCDR (spec)))
    {
      if (it)
	{
	  if (!FRAME_WINDOW_P (it->f))
	    /* If we return here, POSITION has been advanced
	       across the text with this property.  */
	    {
	      /* Synchronize the bidi iterator with POSITION.  This is
		 needed because we are not going to push the iterator
		 on behalf of this display property, so there will be
		 no pop_it call to do this synchronization for us.  */
	      if (it->bidi_p)
		{
		  it->position = *position;
		  iterate_out_of_display_property (it);
		  *position = it->position;
		}
	      return 1;
	    }
	}
      else if (!frame_window_p)
	return 1;

#ifdef HAVE_WINDOW_SYSTEM
      value = XCAR (XCDR (spec));
      int fringe_bitmap = SYMBOLP (value) ? lookup_fringe_bitmap (value) : 0;
      if (! fringe_bitmap)
	/* If we return here, POSITION has been advanced
	   across the text with this property.  */
	{
	  if (it && it->bidi_p)
	    {
	      it->position = *position;
	      iterate_out_of_display_property (it);
	      *position = it->position;
	    }
	  return 1;
	}

      if (it)
	{
	  int face_id = lookup_basic_face (it->w, it->f, DEFAULT_FACE_ID);

	  if (CONSP (XCDR (XCDR (spec))))
	    {
	      Lisp_Object face_name = XCAR (XCDR (XCDR (spec)));
	      int face_id2;
	      /* Don't allow quitting from lookup_derived_face, for when
		 we are displaying a non-selected window, and the buffer's
		 point was temporarily moved to the window-point.  */
	      specpdl_ref count1 = SPECPDL_INDEX ();
	      specbind (Qinhibit_quit, Qt);
	      face_id2 = lookup_derived_face (it->w, it->f, face_name,
					      FRINGE_FACE_ID, false);
	      unbind_to (count1, Qnil);
	      if (face_id2 >= 0)
		face_id = face_id2;
	    }

	  /* Save current settings of IT so that we can restore them
	     when we are finished with the glyph property value.  */
	  push_it (it, position);

	  it->area = TEXT_AREA;
	  it->what = IT_IMAGE;
	  it->image_id = -1; /* no image */
	  it->position = start_pos;
	  it->object = NILP (object) ? it->w->contents : object;
	  it->method = GET_FROM_IMAGE;
	  it->from_overlay = Qnil;
	  it->face_id = face_id;
	  it->from_disp_prop_p = true;

	  /* Say that we haven't consumed the characters with
	     `display' property yet.  The call to pop_it in
	     set_iterator_to_next will clean this up.  */
	  *position = start_pos;

	  if (EQ (XCAR (spec), Qleft_fringe))
	    {
	      it->left_user_fringe_bitmap = fringe_bitmap;
	      it->left_user_fringe_face_id = face_id;
	    }
	  else
	    {
	      it->right_user_fringe_bitmap = fringe_bitmap;
	      it->right_user_fringe_face_id = face_id;
	    }
	}
#endif /* HAVE_WINDOW_SYSTEM */
      return 1;
    }

  /* Prepare to handle `((margin left-margin) ...)',
     `((margin right-margin) ...)' and `((margin nil) ...)'
     prefixes for display specifications.  */
  location = Qunbound;
  if (CONSP (spec) && CONSP (XCAR (spec)))
    {
      Lisp_Object tem;

      value = XCDR (spec);
      if (CONSP (value))
	value = XCAR (value);

      tem = XCAR (spec);
      if (EQ (XCAR (tem), Qmargin)
	  && (tem = XCDR (tem),
	      tem = CONSP (tem) ? XCAR (tem) : Qnil,
	      (NILP (tem)
	       || EQ (tem, Qleft_margin)
	       || EQ (tem, Qright_margin))))
	location = tem;
    }

  if (BASE_EQ (location, Qunbound))
    {
      location = Qnil;
      value = spec;
    }

  /* After this point, VALUE is the property after any
     margin prefix has been stripped.  It must be a string,
     an image specification, or `(space ...)'.

     LOCATION specifies where to display: `left-margin',
     `right-margin' or nil.  */

  bool valid_p = (STRINGP (value)
#ifdef HAVE_WINDOW_SYSTEM
		  || ((it ? FRAME_WINDOW_P (it->f) : frame_window_p)
		      && valid_image_p (value))
#endif /* not HAVE_WINDOW_SYSTEM */
             || (CONSP (value) && EQ (XCAR (value), Qspace))
             || ((it ? FRAME_WINDOW_P (it->f) : frame_window_p)
		 && valid_xwidget_spec_p (value)));

  if (valid_p && display_replaced == 0)
    {
      int retval = 1;

      if (!it)
	{
	  /* Callers need to know whether the display spec is any kind
	     of `(space ...)' spec that is about to affect text-area
	     display.  */
	  if (CONSP (value) && EQ (XCAR (value), Qspace) && NILP (location))
	    retval = 2;
	  return retval;
	}

      /* Save current settings of IT so that we can restore them
	 when we are finished with the glyph property value.  */
      push_it (it, position);
      it->from_overlay = overlay;
      it->from_disp_prop_p = true;

      if (NILP (location))
	it->area = TEXT_AREA;
      else if (EQ (location, Qleft_margin))
	it->area = LEFT_MARGIN_AREA;
      else
	it->area = RIGHT_MARGIN_AREA;

      if (STRINGP (value))
	{
	  it->string = value;
	  it->multibyte_p = STRING_MULTIBYTE (it->string);
	  it->current.overlay_string_index = -1;
	  IT_STRING_CHARPOS (*it) = IT_STRING_BYTEPOS (*it) = 0;
	  it->end_charpos = it->string_nchars = SCHARS (it->string);
	  it->method = GET_FROM_STRING;
	  it->stop_charpos = 0;
	  it->prev_stop = 0;
	  it->base_level_stop = 0;
	  it->string_from_display_prop_p = true;
	  it->cmp_it.id = -1;
	  /* Say that we haven't consumed the characters with
	     `display' property yet.  The call to pop_it in
	     set_iterator_to_next will clean this up.  */
	  if (BUFFERP (object))
	    *position = start_pos;

	  /* Force paragraph direction to be that of the parent
	     object.  If the parent object's paragraph direction is
	     not yet determined, default to L2R.  */
	  if (it->bidi_p && it->bidi_it.paragraph_dir == R2L)
	    it->paragraph_embedding = it->bidi_it.paragraph_dir;
	  else
	    it->paragraph_embedding = L2R;

	  /* Set up the bidi iterator for this display string.  */
	  if (it->bidi_p)
	    {
	      it->bidi_it.string.lstring = it->string;
	      it->bidi_it.string.s = NULL;
	      it->bidi_it.string.schars = it->end_charpos;
	      it->bidi_it.string.bufpos = bufpos;
	      it->bidi_it.string.from_disp_str = true;
	      it->bidi_it.string.unibyte = !it->multibyte_p;
	      it->bidi_it.w = it->w;
	      bidi_init_it (0, 0, FRAME_WINDOW_P (it->f), &it->bidi_it);
	    }
	}
      else if (CONSP (value) && EQ (XCAR (value), Qspace))
	{
	  it->method = GET_FROM_STRETCH;
	  it->object = value;
	  *position = it->position = start_pos;
	  retval = 1 + (it->area == TEXT_AREA);
	}
      else if (valid_xwidget_spec_p (value))
	{
          it->what = IT_XWIDGET;
          it->method = GET_FROM_XWIDGET;
          it->position = start_pos;
	  it->object = NILP (object) ? it->w->contents : object;
	  *position = start_pos;
          it->xwidget = lookup_xwidget (value);
	}
#ifdef HAVE_WINDOW_SYSTEM
      else
	{
	  specpdl_ref count = SPECPDL_INDEX ();

	  it->what = IT_IMAGE;
	  /* Don't allow quitting from lookup_image, for when we are
	     displaying a non-selected window, and the buffer's point
	     was temporarily moved to the window-point.  */
	  specbind (Qinhibit_quit, Qt);
	  it->image_id = lookup_image (it->f, value, it->face_id);
	  unbind_to (count, Qnil);
	  it->position = start_pos;
	  it->object = NILP (object) ? it->w->contents : object;
	  it->method = GET_FROM_IMAGE;

	  /* Say that we haven't consumed the characters with
	     `display' property yet.  The call to pop_it in
	     set_iterator_to_next will clean this up.  */
	  *position = start_pos;
	}
#endif /* HAVE_WINDOW_SYSTEM */

      return retval;
    }

  /* Invalid property or property not supported.  Restore
     POSITION to what it was before.  */
  *position = start_pos;
  return 0;
}

/* Check if PROP is a display property value whose text should be
   treated as intangible.  OVERLAY is the overlay from which PROP
   came, or nil if it came from a text property.  CHARPOS and BYTEPOS
   specify the buffer position covered by PROP.  */

bool
display_prop_intangible_p (Lisp_Object prop, Lisp_Object overlay,
			   ptrdiff_t charpos, ptrdiff_t bytepos)
{
  bool frame_window_p = FRAME_WINDOW_P (XFRAME (selected_frame));
  struct text_pos position;

  SET_TEXT_POS (position, charpos, bytepos);
  return (handle_display_spec (NULL, prop, Qnil, overlay,
			       &position, charpos, frame_window_p)
	  != 0);
}


/* Return true if PROP is a display sub-property value containing STRING.

   Implementation note: this and the following function are really
   special cases of handle_display_spec and
   handle_single_display_spec, and should ideally use the same code.
   Until they do, these two pairs must be consistent and must be
   modified in sync.  */

static bool
single_display_spec_string_p (Lisp_Object prop, Lisp_Object string)
{
  if (EQ (string, prop))
    return true;

  /* Skip over `when FORM'.  */
  if (CONSP (prop) && EQ (XCAR (prop), Qwhen))
    {
      prop = XCDR (prop);
      if (!CONSP (prop))
	return false;
      /* Actually, the condition following `when' should be eval'ed,
	 like handle_single_display_spec does, and we should return
	 false if it evaluates to nil.  However, this function is
	 called only when the buffer was already displayed and some
	 glyph in the glyph matrix was found to come from a display
	 string.  Therefore, the condition was already evaluated, and
	 the result was non-nil, otherwise the display string wouldn't
	 have been displayed and we would have never been called for
	 this property.  Thus, we can skip the evaluation and assume
	 its result is non-nil.  */
      prop = XCDR (prop);
    }

  if (CONSP (prop))
    /* Skip over `margin LOCATION'.  */
    if (EQ (XCAR (prop), Qmargin))
      {
	prop = XCDR (prop);
	if (!CONSP (prop))
	  return false;

	prop = XCDR (prop);
	if (!CONSP (prop))
	  return false;
      }

  return EQ (prop, string) || (CONSP (prop) && EQ (XCAR (prop), string));
}


/* Return true if STRING appears in the `display' property PROP.  */

static bool
display_prop_string_p (Lisp_Object prop, Lisp_Object string)
{
  if (CONSP (prop)
      && !EQ (XCAR (prop), Qwhen)
      && !(CONSP (XCAR (prop)) && EQ (Qmargin, XCAR (XCAR (prop)))))
    {
      /* A list of sub-properties.  */
      while (CONSP (prop))
	{
	  if (single_display_spec_string_p (XCAR (prop), string))
	    return true;
	  prop = XCDR (prop);
	}
    }
  else if (VECTORP (prop))
    {
      /* A vector of sub-properties.  */
      ptrdiff_t i;
      for (i = 0; i < ASIZE (prop); ++i)
	if (single_display_spec_string_p (AREF (prop, i), string))
	  return true;
    }
  else
    return single_display_spec_string_p (prop, string);

  return false;
}

/* Look for STRING in overlays and text properties in the current
   buffer, between character positions FROM and TO (excluding TO).
   BACK_P means look back (in this case, TO is supposed to be
   less than FROM).
   Value is the first character position where STRING was found, or
   zero if it wasn't found before hitting TO.

   This function may only use code that doesn't eval because it is
   called asynchronously from note_mouse_highlight.  */

static ptrdiff_t
string_buffer_position_lim (Lisp_Object string,
			    ptrdiff_t from, ptrdiff_t to, bool back_p)
{
  Lisp_Object limit, prop, pos;
  bool found = false;

  pos = make_fixnum (max (from, BEGV));

  if (!back_p)	/* looking forward */
    {
      limit = make_fixnum (min (to, ZV));
      while (!found && !EQ (pos, limit))
	{
	  prop = Fget_char_property (pos, Qdisplay, Qnil);
	  if (!NILP (prop) && display_prop_string_p (prop, string))
	    found = true;
	  else
	    pos = Fnext_single_char_property_change (pos, Qdisplay, Qnil,
						     limit);
	}
    }
  else		/* looking back */
    {
      limit = make_fixnum (max (to, BEGV));
      while (!found && !EQ (pos, limit))
	{
	  prop = Fget_char_property (pos, Qdisplay, Qnil);
	  if (!NILP (prop) && display_prop_string_p (prop, string))
	    found = true;
	  else
	    pos = Fprevious_single_char_property_change (pos, Qdisplay, Qnil,
							 limit);
	}
    }

  return found ? XFIXNUM (pos) : 0;
}

/* Determine which buffer position in current buffer STRING comes from.
   AROUND_CHARPOS is an approximate position where it could come from.
   Value is the buffer position or 0 if it couldn't be determined.

   This function is necessary because we don't record buffer positions
   in glyphs generated from strings (to keep struct glyph small).
   This function may only use code that doesn't eval because it is
   called asynchronously from note_mouse_highlight.  */

static ptrdiff_t
string_buffer_position (Lisp_Object string, ptrdiff_t around_charpos)
{
  const int MAX_DISTANCE = 1000;
  ptrdiff_t forward_limit = min (around_charpos + MAX_DISTANCE, ZV);
  ptrdiff_t found = string_buffer_position_lim (string, around_charpos,
						forward_limit, false);

  if (!found)
    {
      ptrdiff_t backward_limit = max (around_charpos - MAX_DISTANCE, BEGV);
      found = string_buffer_position_lim (string, around_charpos,
					  backward_limit, true);
    }
  return found;
}



/***********************************************************************
			`composition' property
 ***********************************************************************/

/* Set up iterator IT from `composition' property at its current
   position.  Called from handle_stop.  */

static enum prop_handled
handle_composition_prop (struct it *it)
{
  Lisp_Object prop, string;
  ptrdiff_t pos, pos_byte, start, end;

  if (STRINGP (it->string))
    {
      unsigned char *s;

      pos = IT_STRING_CHARPOS (*it);
      pos_byte = IT_STRING_BYTEPOS (*it);
      string = it->string;
      s = SDATA (string) + pos_byte;
      if (STRING_MULTIBYTE (string))
	it->c = STRING_CHAR (s);
      else
	it->c = *s;
    }
  else
    {
      pos = IT_CHARPOS (*it);
      pos_byte = IT_BYTEPOS (*it);
      string = Qnil;
      it->c = FETCH_CHAR (pos_byte);
    }

  /* If there's a valid composition and point is not inside of the
     composition (in the case that the composition is from the current
     buffer), draw a glyph composed from the composition components.  */
  if (find_composition (pos, -1, &start, &end, &prop, string)
      && composition_valid_p (start, end, prop)
      && (STRINGP (it->string) || (PT <= start || PT >= end)))
    {
      if (start < pos)
	/* As we can't handle this situation (perhaps font-lock added
	   a new composition), we just return here hoping that next
	   redisplay will detect this composition much earlier.  */
	return HANDLED_NORMALLY;
      if (start != pos)
	{
	  if (STRINGP (it->string))
	    pos_byte = string_char_to_byte (it->string, start);
	  else
	    pos_byte = CHAR_TO_BYTE (start);
	}
      it->cmp_it.id = get_composition_id (start, pos_byte, end - start,
					       prop, string);

      if (it->cmp_it.id >= 0)
	{
	  it->cmp_it.ch = -1;
	  it->cmp_it.nchars = COMPOSITION_LENGTH (prop);
	  it->cmp_it.nglyphs = -1;
	}
    }

  return HANDLED_NORMALLY;
}



/***********************************************************************
			   Overlay strings
 ***********************************************************************/

/* The following structure is used to record overlay strings for
   later sorting in load_overlay_strings.  */

struct overlay_entry
{
  Lisp_Object overlay;
  Lisp_Object string;
  EMACS_INT priority;
  bool after_string_p;
};


/* Set up iterator IT from overlay strings at its current position.
   Called from handle_stop.  */

static enum prop_handled
handle_overlay_change (struct it *it)
{
  if (!STRINGP (it->string) && get_overlay_strings (it, 0))
    return HANDLED_RECOMPUTE_PROPS;
  else
    return HANDLED_NORMALLY;
}


/* Set up the next overlay string for delivery by IT, if there is an
   overlay string to deliver.  Called by set_iterator_to_next when the
   end of the current overlay string is reached.  If there are more
   overlay strings to display, IT->string and
   IT->current.overlay_string_index are set appropriately here.
   Otherwise IT->string is set to nil.  */

static void
next_overlay_string (struct it *it)
{
  ++it->current.overlay_string_index;
  if (it->current.overlay_string_index == it->n_overlay_strings)
    {
      /* No more overlay strings.  Restore IT's settings to what
	 they were before overlay strings were processed, and
	 continue to deliver from current_buffer.  */

      it->ellipsis_p = it->stack[it->sp - 1].display_ellipsis_p;
      pop_it (it);
      eassert (it->sp > 0
	       || (NILP (it->string)
		   && it->method == GET_FROM_BUFFER
		   && it->stop_charpos >= BEGV
		   && it->stop_charpos <= it->end_charpos));
      it->current.overlay_string_index = -1;
      it->n_overlay_strings = 0;
      /* If there's an empty display string on the stack, pop the
	 stack, to resync the bidi iterator with IT's position.  Such
	 empty strings are pushed onto the stack in
	 get_overlay_strings_1.  */
      if (it->sp > 0 && STRINGP (it->string) && !SCHARS (it->string))
	pop_it (it);

      /* Since we've exhausted overlay strings at this buffer
	 position, set the flag to ignore overlays until we move to
	 another position.  (The flag will be reset in
	 next_element_from_buffer.)  But don't do that if the overlay
	 strings were loaded at position other than the current one,
	 which could happen if we called pop_it above, or if the
	 overlay strings were loaded by handle_invisible_prop at the
	 beginning of invisible text.  */
      if (it->overlay_strings_charpos == IT_CHARPOS (*it))
	it->ignore_overlay_strings_at_pos_p = true;

      /* If we're at the end of the buffer, record that we have
	 processed the overlay strings there already, so that
	 next_element_from_buffer doesn't try it again.  */
      if (NILP (it->string)
	  && IT_CHARPOS (*it) >= it->end_charpos
	  && it->overlay_strings_charpos >= it->end_charpos)
	it->overlay_strings_at_end_processed_p = true;
      /* Note: we reset overlay_strings_charpos only here, to make
	 sure the just-processed overlays were indeed at EOB.
	 Otherwise, overlays on text with invisible text property,
	 which are processed with IT's position past the invisible
	 text, might fool us into thinking the overlays at EOB were
	 already processed (linum-mode can cause this, for
	 example).  */
      it->overlay_strings_charpos = -1;
    }
  else
    {
      /* There are more overlay strings to process.  If
	 IT->current.overlay_string_index has advanced to a position
	 where we must load IT->overlay_strings with more strings, do
	 it.  We must load at the IT->overlay_strings_charpos where
	 IT->n_overlay_strings was originally computed; when invisible
	 text is present, this might not be IT_CHARPOS (Bug#7016).  */
      int i = it->current.overlay_string_index % OVERLAY_STRING_CHUNK_SIZE;

      if (it->current.overlay_string_index && i == 0)
	load_overlay_strings (it, it->overlay_strings_charpos);

      /* Initialize IT to deliver display elements from the overlay
         string.  */
      it->string = it->overlay_strings[i];
      it->multibyte_p = STRING_MULTIBYTE (it->string);
      SET_TEXT_POS (it->current.string_pos, 0, 0);
      it->method = GET_FROM_STRING;
      it->stop_charpos = 0;
      it->end_charpos = SCHARS (it->string);
      if (it->cmp_it.stop_pos >= 0)
	it->cmp_it.stop_pos = 0;
      it->prev_stop = 0;
      it->base_level_stop = 0;

      /* Set up the bidi iterator for this overlay string.  */
      if (it->bidi_p)
	{
	  it->bidi_it.string.lstring = it->string;
	  it->bidi_it.string.s = NULL;
	  it->bidi_it.string.schars = SCHARS (it->string);
	  it->bidi_it.string.bufpos = it->overlay_strings_charpos;
	  it->bidi_it.string.from_disp_str = it->string_from_display_prop_p;
	  it->bidi_it.string.unibyte = !it->multibyte_p;
	  it->bidi_it.w = it->w;
	  bidi_init_it (0, 0, FRAME_WINDOW_P (it->f), &it->bidi_it);
	}
    }

  CHECK_IT (it);
}


/* Compare two overlay_entry structures E1 and E2.  Used as a
   comparison function for qsort in load_overlay_strings.  Overlay
   strings for the same position are sorted so that

   1. All after-strings come in front of before-strings, except
   when they come from the same overlay.

   2. Within after-strings, strings are sorted so that overlay strings
   from overlays with higher priorities come first.

   2. Within before-strings, strings are sorted so that overlay
   strings from overlays with higher priorities come last.

   Value is analogous to strcmp.  */


static int
compare_overlay_entries (const void *e1, const void *e2)
{
  struct overlay_entry const *entry1 = e1;
  struct overlay_entry const *entry2 = e2;
  int result;

  if (entry1->after_string_p != entry2->after_string_p)
    {
      /* Let after-strings appear in front of before-strings if
	 they come from different overlays.  */
      if (EQ (entry1->overlay, entry2->overlay))
	result = entry1->after_string_p ? 1 : -1;
      else
	result = entry1->after_string_p ? -1 : 1;
    }
  else if (entry1->priority != entry2->priority)
    {
      if (entry1->after_string_p)
	/* After-strings sorted in order of decreasing priority.  */
	result = entry2->priority < entry1->priority ? -1 : 1;
      else
	/* Before-strings sorted in order of increasing priority.  */
	result = entry1->priority < entry2->priority ? -1 : 1;
    }
  else
    result = 0;

  return result;
}


/* Load the vector IT->overlay_strings with overlay strings from IT's
   current buffer position, or from CHARPOS if that is > 0.  Set
   IT->n_overlays to the total number of overlay strings found.

   Overlay strings are processed OVERLAY_STRING_CHUNK_SIZE strings at
   a time.  On entry into load_overlay_strings,
   IT->current.overlay_string_index gives the number of overlay
   strings that have already been loaded by previous calls to this
   function.

   IT->add_overlay_start contains an additional overlay start
   position to consider for taking overlay strings from, if non-zero.
   This position comes into play when the overlay has an `invisible'
   property, and both before and after-strings.  When we've skipped to
   the end of the overlay, because of its `invisible' property, we
   nevertheless want its before-string to appear.
   IT->add_overlay_start will contain the overlay start position
   in this case.

   Overlay strings are sorted so that after-string strings come in
   front of before-string strings.  Within before and after-strings,
   strings are sorted by overlay priority.  See also function
   compare_overlay_entries.  */

static void
load_overlay_strings (struct it *it, ptrdiff_t charpos)
{
  ptrdiff_t n = 0;
  struct overlay_entry entriesbuf[20];
  ptrdiff_t size = ARRAYELTS (entriesbuf);
  struct overlay_entry *entries = entriesbuf;
  struct itree_node *node;

  USE_SAFE_ALLOCA;

  if (charpos <= 0)
    charpos = IT_CHARPOS (*it);

  /* Append the overlay string STRING of overlay OVERLAY to vector
     `entries' which has size `size' and currently contains `n'
     elements.  AFTER_P means STRING is an after-string of
     OVERLAY.  */
#define RECORD_OVERLAY_STRING(OVERLAY, STRING, AFTER_P)			\
  do									\
    {									\
      Lisp_Object priority;						\
									\
      if (n == size)							\
	{								\
	  struct overlay_entry *old = entries;				\
	  SAFE_NALLOCA (entries, 2, size);				\
	  memcpy (entries, old, size * sizeof *entries);		\
	  size *= 2;							\
	}								\
									\
      entries[n].string = (STRING);					\
      entries[n].overlay = (OVERLAY);					\
      priority = Foverlay_get ((OVERLAY), Qpriority);			\
      entries[n].priority = FIXNUMP (priority) ? XFIXNUM (priority) : 0;  \
      entries[n].after_string_p = (AFTER_P);				\
      ++n;								\
    }									\
  while (false)


  /* Process overlays.  */
  ITREE_FOREACH (node, current_buffer->overlays, charpos - 1, charpos + 1, DESCENDING)
    {
      Lisp_Object overlay = node->data;
      eassert (OVERLAYP (overlay));
      ptrdiff_t start = node->begin;
      ptrdiff_t end = node->end;

      /* Skip this overlay if it doesn't start or end at IT's current
         position.  */
      if (end != charpos && start != charpos)
        continue;

      /* Skip this overlay if it doesn't apply to IT->w.  */
      Lisp_Object window = Foverlay_get (overlay, Qwindow);
      if (WINDOWP (window) && XWINDOW (window) != it->w)
        continue;

      /* If the text ``under'' the overlay is invisible, both before-
	 and after-strings from this overlay are visible; start and
	 end position are indistinguishable.  */
      Lisp_Object invisible = Foverlay_get (overlay, Qinvisible);
      int invis = TEXT_PROP_MEANS_INVISIBLE (invisible);

      /* If overlay has a non-empty before-string, record it.  */
      Lisp_Object str;
      if ((start == charpos || (end == charpos && invis != 0))
          && (str = Foverlay_get (overlay, Qbefore_string), STRINGP (str))
          && SCHARS (str))
        RECORD_OVERLAY_STRING (overlay, str, false);

      /* If overlay has a non-empty after-string, record it.  */
      if ((end == charpos || (start == charpos && invis != 0))
          && (str = Foverlay_get (overlay, Qafter_string), STRINGP (str))
          && SCHARS (str))
        RECORD_OVERLAY_STRING (overlay, str, true);
    }

#undef RECORD_OVERLAY_STRING

  /* Sort entries.  */
  if (n > 1)
    qsort (entries, n, sizeof *entries, compare_overlay_entries);

  /* Record number of overlay strings, and where we computed it.  */
  it->n_overlay_strings = n;
  it->overlay_strings_charpos = charpos;

  /* IT->current.overlay_string_index is the number of overlay strings
     that have already been consumed by IT.  Copy some of the
     remaining overlay strings to IT->overlay_strings.  */
  ptrdiff_t j = it->current.overlay_string_index;
  for (ptrdiff_t i = 0; i < OVERLAY_STRING_CHUNK_SIZE && j < n; i++, j++)
    {
      it->overlay_strings[i] = entries[j].string;
      it->string_overlays[i] = entries[j].overlay;
    }

  CHECK_IT (it);
  SAFE_FREE ();
}


/* Get the first chunk of overlay strings at IT's current buffer
   position, or at CHARPOS if that is > 0.  Value is true if at
   least one overlay string was found.  */

static bool
get_overlay_strings_1 (struct it *it, ptrdiff_t charpos, bool compute_stop_p)
{
  /* Get the first OVERLAY_STRING_CHUNK_SIZE overlay strings to
     process.  This fills IT->overlay_strings with strings, and sets
     IT->n_overlay_strings to the total number of strings to process.
     IT->pos.overlay_string_index has to be set temporarily to zero
     because load_overlay_strings needs this; it must be set to -1
     when no overlay strings are found because a zero value would
     indicate a position in the first overlay string.  */
  it->current.overlay_string_index = 0;
  load_overlay_strings (it, charpos);

  /* If we found overlay strings, set up IT to deliver display
     elements from the first one.  Otherwise set up IT to deliver
     from current_buffer.  */
  if (it->n_overlay_strings)
    {
      /* Make sure we know settings in current_buffer, so that we can
	 restore meaningful values when we're done with the overlay
	 strings.  */
      if (compute_stop_p)
	compute_stop_pos (it);
      eassert (it->face_id >= 0);

      /* Save IT's settings.  They are restored after all overlay
	 strings have been processed.  */
      eassert (!compute_stop_p || it->sp == 0);

      /* When called from handle_stop, there might be an empty display
         string loaded.  In that case, don't bother saving it.  But
         don't use this optimization with the bidi iterator, since we
         need the corresponding pop_it call to resync the bidi
         iterator's position with IT's position, after we are done
         with the overlay strings.  (The corresponding call to pop_it
         in case of an empty display string is in
         next_overlay_string.)  */
      if (!(!it->bidi_p
	    && STRINGP (it->string) && !SCHARS (it->string)))
	push_it (it, NULL);

      /* Set up IT to deliver display elements from the first overlay
	 string.  */
      IT_STRING_CHARPOS (*it) = IT_STRING_BYTEPOS (*it) = 0;
      it->string = it->overlay_strings[0];
      it->from_overlay = Qnil;
      it->stop_charpos = 0;
      eassert (STRINGP (it->string));
      it->end_charpos = SCHARS (it->string);
      it->prev_stop = 0;
      it->base_level_stop = 0;
      it->multibyte_p = STRING_MULTIBYTE (it->string);
      it->method = GET_FROM_STRING;
      it->from_disp_prop_p = 0;
      it->cmp_it.id = -1;

      /* Force paragraph direction to be that of the parent
	 buffer.  */
      if (it->bidi_p && it->bidi_it.paragraph_dir == R2L)
	it->paragraph_embedding = it->bidi_it.paragraph_dir;
      else
	it->paragraph_embedding = L2R;

      /* Set up the bidi iterator for this overlay string.  */
      if (it->bidi_p)
	{
	  ptrdiff_t pos = (charpos > 0 ? charpos : IT_CHARPOS (*it));

	  it->bidi_it.string.lstring = it->string;
	  it->bidi_it.string.s = NULL;
	  it->bidi_it.string.schars = SCHARS (it->string);
	  it->bidi_it.string.bufpos = pos;
	  it->bidi_it.string.from_disp_str = it->string_from_display_prop_p;
	  it->bidi_it.string.unibyte = !it->multibyte_p;
	  it->bidi_it.w = it->w;
	  bidi_init_it (0, 0, FRAME_WINDOW_P (it->f), &it->bidi_it);
	}
      return true;
    }

  it->current.overlay_string_index = -1;
  return false;
}

static bool
get_overlay_strings (struct it *it, ptrdiff_t charpos)
{
  it->string = Qnil;
  it->method = GET_FROM_BUFFER;

  get_overlay_strings_1 (it, charpos, true);

  CHECK_IT (it);

  /* Value is true if we found at least one overlay string.  */
  return STRINGP (it->string);
}



/***********************************************************************
		      Saving and restoring state
 ***********************************************************************/

/* Save current settings of IT on IT->stack.  Called, for example,
   before setting up IT for an overlay string, to be able to restore
   IT's settings to what they were after the overlay string has been
   processed.  If POSITION is non-NULL, it is the position to save on
   the stack instead of IT->position.  */

static void
push_it (struct it *it, struct text_pos *position)
{
  struct iterator_stack_entry *p;

  eassert (it->sp < IT_STACK_SIZE);
  p = it->stack + it->sp;

  p->stop_charpos = it->stop_charpos;
  p->prev_stop = it->prev_stop;
  p->base_level_stop = it->base_level_stop;
  p->cmp_it = it->cmp_it;
  eassert (it->face_id >= 0);
  p->face_id = it->face_id;
  p->string = it->string;
  p->method = it->method;
  p->from_overlay = it->from_overlay;
  switch (p->method)
    {
    case GET_FROM_IMAGE:
      p->u.image.object = it->object;
      p->u.image.image_id = it->image_id;
      p->u.image.slice = it->slice;
      break;
    case GET_FROM_STRETCH:
      p->u.stretch.object = it->object;
      break;
    case GET_FROM_XWIDGET:
      p->u.xwidget.object = it->object;
      break;
    case GET_FROM_BUFFER:
    case GET_FROM_DISPLAY_VECTOR:
    case GET_FROM_STRING:
    case GET_FROM_C_STRING:
      break;
    default:
      emacs_abort ();
    }
  p->position = position ? *position : it->position;
  p->current = it->current;
  p->end_charpos = it->end_charpos;
  p->string_nchars = it->string_nchars;
  p->area = it->area;
  p->multibyte_p = it->multibyte_p;
  p->avoid_cursor_p = it->avoid_cursor_p;
  p->space_width = it->space_width;
  p->font_height = it->font_height;
  p->voffset = it->voffset;
  p->string_from_display_prop_p = it->string_from_display_prop_p;
  p->string_from_prefix_prop_p = it->string_from_prefix_prop_p;
  p->display_ellipsis_p = false;
  p->line_wrap = it->line_wrap;
  p->bidi_p = it->bidi_p;
  p->paragraph_embedding = it->paragraph_embedding;
  p->from_disp_prop_p = it->from_disp_prop_p;
  ++it->sp;

  /* Save the state of the bidi iterator as well. */
  if (it->bidi_p)
    bidi_push_it (&it->bidi_it);
}

static void
iterate_out_of_display_property (struct it *it)
{
  bool buffer_p = !STRINGP (it->string);
  ptrdiff_t eob = (buffer_p ? ZV : it->end_charpos);
  ptrdiff_t bob = (buffer_p ? BEGV : 0);

  eassert (eob >= CHARPOS (it->position) && CHARPOS (it->position) >= bob);

  /* Maybe initialize paragraph direction.  If we are at the beginning
     of a new paragraph, next_element_from_buffer may not have a
     chance to do that.  */
  if (it->bidi_it.first_elt && it->bidi_it.charpos < eob)
    bidi_paragraph_init (it->paragraph_embedding, &it->bidi_it, true);
  /* prev_stop can be zero, so check against BEGV as well.  */
  while (it->bidi_it.charpos >= bob
	 && it->prev_stop <= it->bidi_it.charpos
	 && it->bidi_it.charpos < CHARPOS (it->position)
	 && it->bidi_it.charpos < eob)
    bidi_move_to_visually_next (&it->bidi_it);
  /* Record the stop_pos we just crossed, for when we cross it
     back, maybe.  */
  if (it->bidi_it.charpos > CHARPOS (it->position))
    it->prev_stop = CHARPOS (it->position);
  /* If we ended up not where pop_it put us, resync IT's
     positional members with the bidi iterator. */
  if (it->bidi_it.charpos != CHARPOS (it->position))
    SET_TEXT_POS (it->position, it->bidi_it.charpos, it->bidi_it.bytepos);
  if (buffer_p)
    it->current.pos = it->position;
  else
    it->current.string_pos = it->position;
}

/* Restore the IT->face_box_p flag, since it could have been
   overwritten by the face of the object that we just finished
   displaying.  Also, set the IT->start_of_box_run_p flag if the
   change in faces requires that.  */
static void
restore_face_box_flags (struct it *it, int prev_face_id)
{
  struct face *face = FACE_FROM_ID_OR_NULL (it->f, it->face_id);

  if (face)
    {
      struct face *prev_face = FACE_FROM_ID_OR_NULL (it->f, prev_face_id);

      if (!(it->start_of_box_run_p && prev_face && prev_face->box))
	it->start_of_box_run_p = (face->box != FACE_NO_BOX
				  && (prev_face == NULL
				      || prev_face->box == FACE_NO_BOX));
      it->face_box_p = face->box != FACE_NO_BOX;
    }
}

/* Restore IT's settings from IT->stack.  Called, for example, when no
   more overlay strings must be processed, and we return to delivering
   display elements from a buffer, or when the end of a string from a
   `display' property is reached and we return to delivering display
   elements from an overlay string, or from a buffer.  */

static void
pop_it (struct it *it)
{
  struct iterator_stack_entry *p;
  bool from_display_prop = it->from_disp_prop_p;
  ptrdiff_t prev_pos = IT_CHARPOS (*it);
  int prev_face_id = it->face_id;

  eassert (it->sp > 0);
  --it->sp;
  p = it->stack + it->sp;
  it->stop_charpos = p->stop_charpos;
  it->prev_stop = p->prev_stop;
  it->base_level_stop = p->base_level_stop;
  it->cmp_it = p->cmp_it;
  it->face_id = p->face_id;
  it->current = p->current;
  it->position = p->position;
  it->string = p->string;
  it->from_overlay = p->from_overlay;
  if (NILP (it->string))
    SET_TEXT_POS (it->current.string_pos, -1, -1);
  it->method = p->method;
  switch (it->method)
    {
    case GET_FROM_IMAGE:
      it->image_id = p->u.image.image_id;
      it->object = p->u.image.object;
      it->slice = p->u.image.slice;
      break;
    case GET_FROM_XWIDGET:
      it->object = p->u.xwidget.object;
      break;
    case GET_FROM_STRETCH:
      it->object = p->u.stretch.object;
      break;
    case GET_FROM_BUFFER:
      {
	restore_face_box_flags (it, prev_face_id);
	it->object = it->w->contents;
      }
      break;
    case GET_FROM_STRING:
      {
	restore_face_box_flags (it, prev_face_id);
	it->object = it->string;
      }
      break;
    case GET_FROM_DISPLAY_VECTOR:
      if (it->s)
	it->method = GET_FROM_C_STRING;
      else if (STRINGP (it->string))
	it->method = GET_FROM_STRING;
      else
	{
	  it->method = GET_FROM_BUFFER;
	  it->object = it->w->contents;
	}
      break;
    case GET_FROM_C_STRING:
      break;
    default:
      emacs_abort ();
    }
  it->end_charpos = p->end_charpos;
  it->string_nchars = p->string_nchars;
  it->area = p->area;
  it->multibyte_p = p->multibyte_p;
  it->avoid_cursor_p = p->avoid_cursor_p;
  it->space_width = p->space_width;
  it->font_height = p->font_height;
  it->voffset = p->voffset;
  it->string_from_display_prop_p = p->string_from_display_prop_p;
  it->string_from_prefix_prop_p = p->string_from_prefix_prop_p;
  it->line_wrap = p->line_wrap;
  it->bidi_p = p->bidi_p;
  it->paragraph_embedding = p->paragraph_embedding;
  it->from_disp_prop_p = p->from_disp_prop_p;
  if (it->bidi_p)
    {
      bidi_pop_it (&it->bidi_it);
      /* Bidi-iterate until we get out of the portion of text, if any,
	 covered by a `display' text property or by an overlay with
	 `display' property.  (We cannot just jump there, because the
	 internal coherency of the bidi iterator state can not be
	 preserved across such jumps.)  We also must determine the
	 paragraph base direction if the overlay we just processed is
	 at the beginning of a new paragraph.  */
      if (from_display_prop
	  && (it->method == GET_FROM_BUFFER || it->method == GET_FROM_STRING))
	iterate_out_of_display_property (it);

      eassert ((BUFFERP (it->object)
		&& IT_CHARPOS (*it) == it->bidi_it.charpos
		&& IT_BYTEPOS (*it) == it->bidi_it.bytepos)
	       || (STRINGP (it->object)
		   && IT_STRING_CHARPOS (*it) == it->bidi_it.charpos
		   && IT_STRING_BYTEPOS (*it) == it->bidi_it.bytepos)
	       || (CONSP (it->object) && it->method == GET_FROM_STRETCH)
	       /* We could be in the middle of handling a list or a
		  vector of several 'display' properties, in which
		  case we should only verify the above conditions when
		  we pop the iterator stack the last time, because
		  higher stack levels cannot "iterate out of the
		  display property".  */
	       || it->sp > 0);
    }
  /* If we move the iterator over text covered by a display property
     to a new buffer position, any info about previously seen overlays
     is no longer valid.  */
  if (from_display_prop && it->sp == 0 && CHARPOS (it->position) != prev_pos)
    it->ignore_overlay_strings_at_pos_p = false;
}



/***********************************************************************
			  Moving over lines
 ***********************************************************************/

/* Set IT's current position to the previous line start.  */

static void
back_to_previous_line_start (struct it *it)
{
  ptrdiff_t cp = IT_CHARPOS (*it), bp = IT_BYTEPOS (*it);

  dec_both (&cp, &bp);
  SET_WITH_NARROWED_BEGV (it, IT_CHARPOS (*it),
			  find_newline_no_quit (cp, bp, -1, &IT_BYTEPOS (*it)),
			  get_closer_narrowed_begv (it->w, IT_CHARPOS (*it)));
}

/* Find in the current buffer the first display or overlay string
   between STARTPOS and ENDPOS that includes embedded newlines.
   Consider only overlays that apply to window W.
   Value is non-zero if such a display/overlay string is found.  */
static bool
strings_with_newlines (ptrdiff_t startpos, ptrdiff_t endpos, struct window *w)
{
  struct itree_node *node;
  /* Process overlays.  */
  ITREE_FOREACH (node, current_buffer->overlays, startpos, endpos, DESCENDING)
    {
      Lisp_Object overlay = node->data;
      eassert (OVERLAYP (overlay));

      /* Skip this overlay if it doesn't apply to our window.  */
      Lisp_Object window = Foverlay_get (overlay, Qwindow);
      if (WINDOWP (window) && XWINDOW (window) != w)
	continue;

      ptrdiff_t ostart = node->begin;
      ptrdiff_t oend = node->end;

      /* Skip overlays that don't overlap the range.  */
      if (!((startpos < oend && ostart < endpos)
	    || (ostart == oend
		&& (startpos == oend || (endpos == ZV && oend == endpos)))))
	continue;

      Lisp_Object str;
      str = Foverlay_get (overlay, Qbefore_string);
      if (STRINGP (str) && SCHARS (str)
	  && memchr (SDATA (str), '\n', SBYTES (str)))
	return true;
      str = Foverlay_get (overlay, Qafter_string);
      if (STRINGP (str) && SCHARS (str)
	  && memchr (SDATA (str), '\n', SBYTES (str)))
	return true;
    }

  /* Check for 'display' properties whose values include strings.  */
  Lisp_Object cpos = make_fixnum (startpos);
  Lisp_Object limpos = make_fixnum (endpos);

  while ((cpos = Fnext_single_property_change (cpos, Qdisplay, Qnil, limpos),
	  !(NILP (cpos) || XFIXNAT (cpos) >= endpos)))
    {
      Lisp_Object spec = Fget_char_property (cpos, Qdisplay, Qnil);
      Lisp_Object string = string_from_display_spec (spec);
      if (STRINGP (string)
	  && memchr (SDATA (string), '\n', SBYTES (string)))
	return true;
    }

  return false;
}


/* Move IT to the next line start.

   Value is true if a newline was found.  Set *SKIPPED_P to true if
   we skipped over part of the text (as opposed to moving the iterator
   continuously over the text).  Otherwise, don't change the value
   of *SKIPPED_P.

   If BIDI_IT_PREV is non-NULL, store into it the state of the bidi
   iterator on the newline, if it was found.

   Newlines may come from buffer text, overlay strings, or strings
   displayed via the `display' property.  That's the reason we can't
   simply use find_newline_no_quit.

   Note that this function may not skip over invisible text that is so
   because of text properties and immediately follows a newline.  If
   it would, function reseat_at_next_visible_line_start, when called
   from set_iterator_to_next, would effectively make invisible
   characters following a newline part of the wrong glyph row, which
   leads to wrong cursor motion.  */

static bool
forward_to_next_line_start (struct it *it, bool *skipped_p,
			    struct bidi_it *bidi_it_prev)
{
  ptrdiff_t old_selective;
  bool newline_found_p = false;
  int n;
  const int MAX_NEWLINE_DISTANCE = 500;

  /* If already on a newline, just consume it to avoid unintended
     skipping over invisible text below.  */
  if (it->what == IT_CHARACTER
      && it->c == '\n'
      && CHARPOS (it->position) == IT_CHARPOS (*it))
    {
      if (it->bidi_p && bidi_it_prev)
	*bidi_it_prev = it->bidi_it;
      set_iterator_to_next (it, false);
      it->c = 0;
      return true;
    }

  /* Don't handle selective display in the following.  It's (a)
     unnecessary because it's done by the caller, and (b) leads to an
     infinite recursion because next_element_from_ellipsis indirectly
     calls this function.  */
  old_selective = it->selective;
  it->selective = 0;

  /* Scan for a newline within MAX_NEWLINE_DISTANCE display elements
     from buffer text, or till the end of the string if iterating a
     string.  */
  for (n = 0;
       !newline_found_p && n < MAX_NEWLINE_DISTANCE;
       n += !STRINGP (it->string))
    {
      if (!get_next_display_element (it))
	return false;
      newline_found_p = it->what == IT_CHARACTER && it->c == '\n';
      if (newline_found_p && it->bidi_p && bidi_it_prev)
	*bidi_it_prev = it->bidi_it;
      set_iterator_to_next (it, false);
    }

  /* If we didn't find a newline near enough, see if we can use a
     short-cut.  */
  if (!newline_found_p)
    {
      ptrdiff_t bytepos, start = IT_CHARPOS (*it);
      ptrdiff_t limit = find_newline_no_quit (start, IT_BYTEPOS (*it),
					      1, &bytepos);
      eassert (!STRINGP (it->string));

      /* it->stop_charpos >= limit means we already know there's no
	 stop position up until the newline at LIMIT, so there's no
	 need for any further checks.  */
      bool no_strings_with_newlines = it->stop_charpos >= limit;

      if (!no_strings_with_newlines)
	{
	  if (!(current_buffer->long_line_optimizations_p
		&& it->line_wrap == TRUNCATE))
	    {
	      /* Quick-and-dirty check: if there isn't any `display'
		 property in sight, and no overlays, we're done.  */
	      Lisp_Object pos =
		Fnext_single_property_change (make_fixnum (start),
					      Qdisplay, Qnil,
					      make_fixnum (limit));
	      no_strings_with_newlines =
		(NILP (pos) || XFIXNAT (pos) == limit) /* no 'display' props */
		&& next_overlay_change (start) == ZV;  /* no overlays */
	    }
	  else
	    {
	      /* For buffers with very long and truncated lines we try
		 harder, because it's worth our while to spend some
		 time looking into the overlays and 'display' properties
		 if we can then avoid iterating through all of them.  */
	      no_strings_with_newlines =
		!strings_with_newlines (start, limit, it->w);
	    }
	}

      /* If there's no display or overlay strings with embedded
	 newlines until the position of the newline in buffer text, we
	 can just use that position.  */
      if (no_strings_with_newlines)
	{
	  if (!it->bidi_p || !bidi_it_prev)
	    {
	      /* The optimal case: just jump there.  */
	      IT_CHARPOS (*it) = limit;
	      IT_BYTEPOS (*it) = bytepos;
	    }
	  else
	    {
	      /* The less optimal case: need to bidi-walk there, but
		 this is still cheaper that the full iteration using
		 get_next_display_element and set_iterator_to_next.  */
	      struct bidi_it bprev;

	      /* Help bidi.c avoid expensive searches for display
		 properties and overlays, by telling it that there are
		 none up to `limit'.  */
	      if (it->bidi_it.disp_pos < limit)
		{
		  it->bidi_it.disp_pos = limit;
		  it->bidi_it.disp_prop = 0;
		}
	      do {
		bprev = it->bidi_it;
		bidi_move_to_visually_next (&it->bidi_it);
	      } while (it->bidi_it.charpos != limit);
	      IT_CHARPOS (*it) = limit;
	      IT_BYTEPOS (*it) = it->bidi_it.bytepos;
	      if (bidi_it_prev)
		*bidi_it_prev = bprev;
	    }
	  *skipped_p = newline_found_p = true;
	}
      else
	{
	  /* The slow case.  */
	  while (!newline_found_p)
	    {
	      if (!get_next_display_element (it))
		break;
	      newline_found_p = ITERATOR_AT_END_OF_LINE_P (it);
	      if (newline_found_p && it->bidi_p && bidi_it_prev)
		*bidi_it_prev = it->bidi_it;
	      set_iterator_to_next (it, false);
	    }
	}
    }

  it->selective = old_selective;
  return newline_found_p;
}


/* Set IT's current position to the previous visible line start.  Skip
   invisible text that is so either due to text properties or due to
   selective display.  Caution: this does not change IT->current_x and
   IT->hpos.  */

static void
back_to_previous_visible_line_start (struct it *it)
{
  while (IT_CHARPOS (*it) > BEGV)
    {
      back_to_previous_line_start (it);

      if (IT_CHARPOS (*it) <= BEGV)
	break;

      /* If selective > 0, then lines indented more than its value are
	 invisible.  */
      if (it->selective > 0
	  && indented_beyond_p (IT_CHARPOS (*it), IT_BYTEPOS (*it),
				it->selective))
	continue;

      /* Check the newline before point for invisibility.  */
      {
	Lisp_Object prop;
	prop = Fget_char_property (make_fixnum (IT_CHARPOS (*it) - 1),
				   Qinvisible, it->window);
	if (TEXT_PROP_MEANS_INVISIBLE (prop) != 0)
	  continue;
      }

      if (IT_CHARPOS (*it) <= BEGV)
	break;

      {
	struct it it2;
	void *it2data = NULL;
	ptrdiff_t pos;
	ptrdiff_t beg, end;
	Lisp_Object val, overlay;

	SAVE_IT (it2, *it, it2data);

	/* If newline is part of a composition, continue from start of composition */
	if (find_composition (IT_CHARPOS (*it), -1, &beg, &end, &val, Qnil)
	    && beg < IT_CHARPOS (*it))
	  goto replaced;

	/* If newline is replaced by a display property, find start of overlay
	   or interval and continue search from that point.  */
	pos = --IT_CHARPOS (it2);
	--IT_BYTEPOS (it2);
	it2.sp = 0;
	bidi_unshelve_cache (NULL, false);
	it2.string_from_display_prop_p = false;
	it2.from_disp_prop_p = false;
	if (handle_display_prop (&it2) == HANDLED_RETURN
	    && !NILP (val = get_char_property_and_overlay
		      (make_fixnum (pos), Qdisplay, Qnil, &overlay))
	    && (OVERLAYP (overlay)
		? (beg = OVERLAY_START (overlay))
		: get_property_and_range (pos, Qdisplay, &val, &beg, &end, Qnil)))
	  {
	    RESTORE_IT (it, it, it2data);
	    goto replaced;
	  }

	/* Newline is not replaced by anything -- so we are done.  */
	RESTORE_IT (it, it, it2data);
	break;

      replaced:
	if (beg < BEGV)
	  beg = BEGV;
	IT_CHARPOS (*it) = beg;
	IT_BYTEPOS (*it) = buf_charpos_to_bytepos (current_buffer, beg);
      }
    }

  it->continuation_lines_width = 0;

  eassert (IT_CHARPOS (*it) >= BEGV);
  eassert (it->narrowed_begv > 0 /* long-line optimizations: all bets off */
	   || IT_CHARPOS (*it) == BEGV
	   || FETCH_BYTE (IT_BYTEPOS (*it) - 1) == '\n');
  CHECK_IT (it);
}


/* Reseat iterator IT at the previous visible line start.  Skip
   invisible text that is so either due to text properties or due to
   selective display.  At the end, update IT's overlay information,
   face information etc.  */

void
reseat_at_previous_visible_line_start (struct it *it)
{
  back_to_previous_visible_line_start (it);
  reseat (it, it->current.pos, true);
  CHECK_IT (it);
}


/* Reseat iterator IT on the next visible line start in the current
   buffer.  ON_NEWLINE_P means position IT on the newline
   preceding the line start.  Skip over invisible text that is so
   because of selective display.  Compute faces, overlays etc at the
   new position.  Note that this function does not skip over text that
   is invisible because of text properties.  */

static void
reseat_at_next_visible_line_start (struct it *it, bool on_newline_p)
{
  bool skipped_p = false;
  struct bidi_it bidi_it_prev;
  bool newline_found_p
    = forward_to_next_line_start (it, &skipped_p,
				  on_newline_p ? &bidi_it_prev : NULL);

  /* Skip over lines that are invisible because they are indented
     more than the value of IT->selective.  */
  if (it->selective > 0)
    while (IT_CHARPOS (*it) < ZV
	   && indented_beyond_p (IT_CHARPOS (*it), IT_BYTEPOS (*it),
				 it->selective))
      {
	eassert (IT_BYTEPOS (*it) == BEGV
		 || FETCH_BYTE (IT_BYTEPOS (*it) - 1) == '\n');
	newline_found_p =
	  forward_to_next_line_start (it, &skipped_p,
				      on_newline_p ? &bidi_it_prev : NULL);
      }

  /* Position on the newline if that's what's requested.  */
  if (on_newline_p && newline_found_p)
    {
      if (STRINGP (it->string))
	{
	  if (IT_STRING_CHARPOS (*it) > 0)
	    {
	      if (!it->bidi_p)
		{
		  --IT_STRING_CHARPOS (*it);
		  --IT_STRING_BYTEPOS (*it);
		}
	      else
		{
		  /* We need to restore the bidi iterator to the state
		     it had on the newline, and resync the IT's
		     position with that.  */
		  it->bidi_it = bidi_it_prev;
		  IT_STRING_CHARPOS (*it) = it->bidi_it.charpos;
		  IT_STRING_BYTEPOS (*it) = it->bidi_it.bytepos;
		}
	    }
	}
      else if (IT_CHARPOS (*it) > BEGV)
	{
	  if (!it->bidi_p)
	    {
	      --IT_CHARPOS (*it);
	      --IT_BYTEPOS (*it);
	    }
	  else
	    {
	      /* We need to restore the bidi iterator to the state it
		 had on the newline and resync IT with that.  */
	      it->bidi_it = bidi_it_prev;
	      IT_CHARPOS (*it) = it->bidi_it.charpos;
	      IT_BYTEPOS (*it) = it->bidi_it.bytepos;
	    }
	  reseat (it, it->current.pos, false);
	}
    }
  else if (skipped_p)
    reseat (it, it->current.pos, false);

  CHECK_IT (it);
}



/***********************************************************************
		   Changing an iterator's position
***********************************************************************/

/* Change IT's current position to POS in current_buffer.
   If FORCE_P, always check for text properties at the new position.
   Otherwise, text properties are only looked up if POS >=
   IT->check_charpos of a property.  */

static void
reseat (struct it *it, struct text_pos pos, bool force_p)
{
  ptrdiff_t original_pos = IT_CHARPOS (*it);

  reseat_1 (it, pos, false);

  if (current_buffer->long_line_optimizations_p)
    {
      if (!it->narrowed_begv)
	{
	  it->narrowed_begv = get_narrowed_begv (it->w, window_point (it->w));
	  it->narrowed_zv = get_narrowed_zv (it->w, window_point (it->w));
	  it->locked_narrowing_begv
	    = get_locked_narrowing_begv (window_point (it->w));
	  it->locked_narrowing_zv
	    = get_locked_narrowing_zv (window_point (it->w));
	}
      else if ((pos.charpos < it->narrowed_begv || pos.charpos > it->narrowed_zv)
		&& (!redisplaying_p || it->line_wrap == TRUNCATE))
	{
	  it->narrowed_begv = get_narrowed_begv (it->w, pos.charpos);
	  it->narrowed_zv = get_narrowed_zv (it->w, pos.charpos);
	  it->locked_narrowing_begv
	    = get_locked_narrowing_begv (window_point (it->w));
	  it->locked_narrowing_zv
	    = get_locked_narrowing_zv (window_point (it->w));
	}
    }

  /* Determine where to check text properties.  Avoid doing it
     where possible because text property lookup is very expensive.  */
  if (force_p
      || CHARPOS (pos) > it->stop_charpos
      || CHARPOS (pos) < original_pos)
    {
      if (it->bidi_p)
	{
	  /* For bidi iteration, we need to prime prev_stop and
	     base_level_stop with our best estimations.  */
	  /* Implementation note: Of course, POS is not necessarily a
	     stop position, so assigning prev_pos to it is a lie; we
	     should have called compute_stop_backwards.  However, if
	     the current buffer does not include any R2L characters,
	     that call would be a waste of cycles, because the
	     iterator will never move back, and thus never cross this
	     "fake" stop position.  So we delay that backward search
	     until the time we really need it, in next_element_from_buffer.  */
	  if (CHARPOS (pos) != it->prev_stop)
	    it->prev_stop = CHARPOS (pos);
	  if (CHARPOS (pos) < it->base_level_stop)
	    it->base_level_stop = 0; /* meaning it's unknown */
	  handle_stop (it);
	}
      else
	{
	  handle_stop (it);
	  it->prev_stop = it->base_level_stop = 0;
	}

    }

  CHECK_IT (it);
}


/* Change IT's buffer position to POS.  SET_STOP_P means set
   IT->stop_pos to POS, also.  */

static void
reseat_1 (struct it *it, struct text_pos pos, bool set_stop_p)
{
  /* Don't call this function when scanning a C string.  */
  eassert (it->s == NULL);

  /* POS must be a reasonable value.  */
  eassert (CHARPOS (pos) >= BEGV && CHARPOS (pos) <= ZV);

  it->current.pos = it->position = pos;
  it->end_charpos = ZV;
  it->dpvec = NULL;
  it->current.dpvec_index = -1;
  it->current.overlay_string_index = -1;
  IT_STRING_CHARPOS (*it) = -1;
  IT_STRING_BYTEPOS (*it) = -1;
  it->string = Qnil;
  it->method = GET_FROM_BUFFER;
  it->object = it->w->contents;
  it->area = TEXT_AREA;
  it->multibyte_p = !NILP (BVAR (current_buffer, enable_multibyte_characters));
  it->sp = 0;
  it->string_from_display_prop_p = false;
  it->string_from_prefix_prop_p = false;

  it->from_disp_prop_p = false;
  it->face_before_selective_p = false;
  if (it->bidi_p)
    {
      bidi_init_it (IT_CHARPOS (*it), IT_BYTEPOS (*it), FRAME_WINDOW_P (it->f),
		    &it->bidi_it);
      bidi_unshelve_cache (NULL, false);
      it->bidi_it.paragraph_dir = NEUTRAL_DIR;
      it->bidi_it.string.s = NULL;
      it->bidi_it.string.lstring = Qnil;
      it->bidi_it.string.bufpos = 0;
      it->bidi_it.string.from_disp_str = false;
      it->bidi_it.string.unibyte = false;
      it->bidi_it.w = it->w;
    }

  if (set_stop_p)
    {
      it->stop_charpos = CHARPOS (pos);
      it->base_level_stop = CHARPOS (pos);
    }
  /* This make the information stored in it->cmp_it invalidate.  */
  it->cmp_it.id = -1;
  it->min_width_property = Qnil;
}


/* Set up IT for displaying a string, starting at CHARPOS in window W.
   If S is non-null, it is a C string to iterate over.  Otherwise,
   STRING gives a Lisp string to iterate over.

   If PRECISION > 0, don't return more then PRECISION number of
   characters from the string.

   If FIELD_WIDTH > 0, return padding spaces until FIELD_WIDTH
   characters have been returned.  FIELD_WIDTH < 0 means an infinite
   field width.

   MULTIBYTE = 0 means disable processing of multibyte characters,
   MULTIBYTE > 0 means enable it,
   MULTIBYTE < 0 means use IT->multibyte_p.

   IT must be initialized via a prior call to init_iterator before
   calling this function.  */

static void
reseat_to_string (struct it *it, const char *s, Lisp_Object string,
		  ptrdiff_t charpos, ptrdiff_t precision, int field_width,
		  int multibyte)
{
  /* No text property checks performed by default, but see below.  */
  it->stop_charpos = -1;

  /* Set iterator position and end position.  */
  memset (&it->current, 0, sizeof it->current);
  it->current.overlay_string_index = -1;
  it->current.dpvec_index = -1;
  eassert (charpos >= 0);

  /* If STRING is specified, use its multibyteness, otherwise use the
     setting of MULTIBYTE, if specified.  */
  if (multibyte >= 0)
    it->multibyte_p = multibyte > 0;

  /* Bidirectional reordering of strings is controlled by the default
     value of bidi-display-reordering.  Don't try to reorder while
     loading loadup.el, as the necessary character property tables are
     not yet available.  */
  it->bidi_p =
    !redisplay__inhibit_bidi
    && !NILP (BVAR (&buffer_defaults, bidi_display_reordering));

  if (s == NULL)
    {
      eassert (STRINGP (string));
      it->string = string;
      it->s = NULL;
      it->end_charpos = it->string_nchars = SCHARS (string);
      it->method = GET_FROM_STRING;
      it->current.string_pos = string_pos (charpos, string);

      if (it->bidi_p)
	{
	  it->bidi_it.string.lstring = string;
	  it->bidi_it.string.s = NULL;
	  it->bidi_it.string.schars = it->end_charpos;
	  it->bidi_it.string.bufpos = 0;
	  it->bidi_it.string.from_disp_str = false;
	  it->bidi_it.string.unibyte = !it->multibyte_p;
	  it->bidi_it.w = it->w;
	  bidi_init_it (charpos, IT_STRING_BYTEPOS (*it),
			FRAME_WINDOW_P (it->f), &it->bidi_it);
	}
    }
  else
    {
      it->s = (const unsigned char *) s;
      it->string = Qnil;

      /* Note that we use IT->current.pos, not it->current.string_pos,
	 for displaying C strings.  */
      IT_STRING_CHARPOS (*it) = IT_STRING_BYTEPOS (*it) = -1;
      if (it->multibyte_p)
	{
	  it->current.pos = c_string_pos (charpos, s, true);
	  it->end_charpos = it->string_nchars = number_of_chars (s, true);
	}
      else
	{
	  IT_CHARPOS (*it) = IT_BYTEPOS (*it) = charpos;
	  it->end_charpos = it->string_nchars = strlen (s);
	}

      if (it->bidi_p)
	{
	  it->bidi_it.string.lstring = Qnil;
	  it->bidi_it.string.s = (const unsigned char *) s;
	  it->bidi_it.string.schars = it->end_charpos;
	  it->bidi_it.string.bufpos = 0;
	  it->bidi_it.string.from_disp_str = false;
	  it->bidi_it.string.unibyte = !it->multibyte_p;
	  it->bidi_it.w = it->w;
	  bidi_init_it (charpos, IT_BYTEPOS (*it), FRAME_WINDOW_P (it->f),
			&it->bidi_it);
	}
      it->method = GET_FROM_C_STRING;
    }

  /* PRECISION > 0 means don't return more than PRECISION characters
     from the string.  */
  if (precision > 0 && it->end_charpos - charpos > precision)
    {
      it->end_charpos = it->string_nchars = charpos + precision;
      if (it->bidi_p)
	it->bidi_it.string.schars = it->end_charpos;
    }

  /* FIELD_WIDTH > 0 means pad with spaces until FIELD_WIDTH
     characters have been returned.  FIELD_WIDTH == 0 means don't pad,
     FIELD_WIDTH < 0 means infinite field width.  This is useful for
     padding with `-' at the end of a mode line.  */
  if (field_width < 0)
    field_width = DISP_INFINITY;
  /* Implementation note: We deliberately don't enlarge
     it->bidi_it.string.schars here to fit it->end_charpos, because
     the bidi iterator cannot produce characters out of thin air.  */
  if (field_width > it->end_charpos - charpos)
    it->end_charpos = charpos + field_width;

  /* Use the standard display table for displaying strings.  */
  if (DISP_TABLE_P (Vstandard_display_table))
    it->dp = XCHAR_TABLE (Vstandard_display_table);

  it->stop_charpos = charpos;
  it->prev_stop = charpos;
  it->base_level_stop = 0;
  if (it->bidi_p)
    {
      it->bidi_it.first_elt = true;
      it->bidi_it.paragraph_dir = NEUTRAL_DIR;
      it->bidi_it.disp_pos = -1;
    }
  if (s == NULL && it->multibyte_p)
    {
      ptrdiff_t endpos = SCHARS (it->string);
      if (endpos > it->end_charpos)
	endpos = it->end_charpos;
      composition_compute_stop_pos (&it->cmp_it, charpos, -1, endpos,
				    it->string);
    }
  CHECK_IT (it);
}



/***********************************************************************
			      Iteration
***********************************************************************/

/* Map enum it_method value to corresponding next_element_from_* function.  */

typedef bool (*next_element_function) (struct it *);

static next_element_function const get_next_element[NUM_IT_METHODS] =
{
  next_element_from_buffer,
  next_element_from_display_vector,
  next_element_from_string,
  next_element_from_c_string,
  next_element_from_image,
  next_element_from_stretch,
  next_element_from_xwidget,
};

#define GET_NEXT_DISPLAY_ELEMENT(it) (*get_next_element[(it)->method]) (it)


/* Return true iff a character at CHARPOS (and BYTEPOS) is composed
   (possibly with the following characters).

  Note: we pass -1 as the "resolved bidi level" when the iterator
  doesn't have the bidi_p flag set, because in that case we really
  don't know what is the directionality of the text, so we leave it to
  the shaping engine to figure that out.  */

#define CHAR_COMPOSED_P(IT,CHARPOS,BYTEPOS,END_CHARPOS)			\
  ((IT)->cmp_it.id >= 0							\
   || ((IT)->cmp_it.stop_pos == (CHARPOS)				\
       && composition_reseat_it (&(IT)->cmp_it, CHARPOS, BYTEPOS,	\
				 END_CHARPOS, (IT)->w,			\
				 (IT)->bidi_p				\
				 ? (IT)->bidi_it.resolved_level		\
				 : -1,					\
				 FACE_FROM_ID_OR_NULL ((IT)->f,		\
						       (IT)->face_id),	\
				 (IT)->string)))


/* Lookup the char-table Vglyphless_char_display for character C (-1
   if we want information for no-font case), and return the display
   method symbol.  By side-effect, update it->what and
   it->glyphless_method.  This function is called from
   get_next_display_element for each character element, and from
   gui_produce_glyphs when no suitable font was found.  */

Lisp_Object
lookup_glyphless_char_display (int c, struct it *it)
{
  Lisp_Object glyphless_method = Qnil;

  if (CHAR_TABLE_P (Vglyphless_char_display)
      && CHAR_TABLE_EXTRA_SLOTS (XCHAR_TABLE (Vglyphless_char_display)) >= 1)
    {
      if (c >= 0)
	glyphless_method = CHAR_TABLE_REF (Vglyphless_char_display, c);
      else
	glyphless_method = XCHAR_TABLE (Vglyphless_char_display)->extras[0];

      if (CONSP (glyphless_method))
	glyphless_method = FRAME_WINDOW_P (it->f)
	  ? XCAR (glyphless_method)
	  : XCDR (glyphless_method);
    }

 retry:
  if (NILP (glyphless_method))
    {
      if (c >= 0)
	/* The default is to display the character by a proper font.  */
	return Qnil;
      /* The default for the no-font case is to display an empty box.  */
      glyphless_method = Qempty_box;
    }
  if (EQ (glyphless_method, Qzero_width))
    {
      if (c >= 0)
	return glyphless_method;
      /* This method can't be used for the no-font case.  */
      glyphless_method = Qempty_box;
    }
  if (EQ (glyphless_method, Qthin_space))
    it->glyphless_method = GLYPHLESS_DISPLAY_THIN_SPACE;
  else if (EQ (glyphless_method, Qempty_box))
    it->glyphless_method = GLYPHLESS_DISPLAY_EMPTY_BOX;
  else if (EQ (glyphless_method, Qhex_code))
    it->glyphless_method = GLYPHLESS_DISPLAY_HEX_CODE;
  else if (STRINGP (glyphless_method))
    it->glyphless_method = GLYPHLESS_DISPLAY_ACRONYM;
  else
    {
      /* Invalid value.  We use the default method.  */
      glyphless_method = Qnil;
      goto retry;
    }
  it->what = IT_GLYPHLESS;
  return glyphless_method;
}

/* Merge escape glyph face and cache the result.  */

static struct frame *last_escape_glyph_frame = NULL;
static int last_escape_glyph_face_id = (1 << FACE_ID_BITS);
static int last_escape_glyph_merged_face_id = 0;

static int
merge_escape_glyph_face (struct it *it)
{
  int face_id;

  if (it->f == last_escape_glyph_frame
      && it->face_id == last_escape_glyph_face_id)
    face_id = last_escape_glyph_merged_face_id;
  else
    {
      /* Merge the `escape-glyph' face into the current face.  */
      face_id = merge_faces (it->w, Qescape_glyph, 0, it->face_id);
      last_escape_glyph_frame = it->f;
      last_escape_glyph_face_id = it->face_id;
      last_escape_glyph_merged_face_id = face_id;
    }
  return face_id;
}

/* Likewise for glyphless glyph face.  */

static struct frame *last_glyphless_glyph_frame = NULL;
static int last_glyphless_glyph_face_id = (1 << FACE_ID_BITS);
static int last_glyphless_glyph_merged_face_id = 0;

int
merge_glyphless_glyph_face (struct it *it)
{
  int face_id;

  if (it->f == last_glyphless_glyph_frame
      && it->face_id == last_glyphless_glyph_face_id)
    face_id = last_glyphless_glyph_merged_face_id;
  else
    {
      /* Merge the `glyphless-char' face into the current face.  */
      face_id = merge_faces (it->w, Qglyphless_char, 0, it->face_id);
      last_glyphless_glyph_frame = it->f;
      last_glyphless_glyph_face_id = it->face_id;
      last_glyphless_glyph_merged_face_id = face_id;
    }
  return face_id;
}

/* Forget the `escape-glyph' and `glyphless-char' faces.  This should
   be called before redisplaying windows, and when the frame's face
   cache is freed.  */
void
forget_escape_and_glyphless_faces (void)
{
  last_escape_glyph_frame = NULL;
  last_escape_glyph_face_id = (1 << FACE_ID_BITS);
  last_glyphless_glyph_frame = NULL;
  last_glyphless_glyph_face_id = (1 << FACE_ID_BITS);
}

/* Load IT's display element fields with information about the next
   display element from the current position of IT.  Value is false if
   end of buffer (or C string) is reached.  */

static bool
get_next_display_element (struct it *it)
{
  /* True means that we found a display element.  False means that
     we hit the end of what we iterate over.  Performance note: the
     function pointer `method' used here turns out to be faster than
     using a sequence of if-statements.  */
  bool success_p;

 get_next:
  success_p = GET_NEXT_DISPLAY_ELEMENT (it);

  if (it->what == IT_CHARACTER)
    {
      /* UAX#9, L4: "A character is depicted by a mirrored glyph if
	 and only if (a) the resolved directionality of that character
	 is R..."  */
      /* FIXME: Do we need an exception for characters from display
	 tables?  */
      if (it->bidi_p && it->bidi_it.type == STRONG_R
	  && !inhibit_bidi_mirroring)
	it->c = bidi_mirror_char (it->c);
      /* Map via display table or translate control characters.
	 IT->c, IT->len etc. have been set to the next character by
	 the function call above.  If we have a display table, and it
	 contains an entry for IT->c, translate it.  Don't do this if
	 IT->c itself comes from a display table, otherwise we could
	 end up in an infinite recursion.  (An alternative could be to
	 count the recursion depth of this function and signal an
	 error when a certain maximum depth is reached.)  Is it worth
	 it?  */
      if (success_p && it->dpvec == NULL)
	{
	  Lisp_Object dv;
	  struct charset *unibyte = CHARSET_FROM_ID (charset_unibyte);
	  bool nonascii_space_p = false;
	  bool nonascii_hyphen_p = false;
	  int c = it->c;	/* This is the character to display.  */

	  if (! it->multibyte_p && ! ASCII_CHAR_P (c))
	    {
	      eassert (SINGLE_BYTE_CHAR_P (c));
	      if (unibyte_display_via_language_environment)
		{
		  c = DECODE_CHAR (unibyte, c);
		  if (c < 0)
		    c = BYTE8_TO_CHAR (it->c);
		}
	      else
		c = BYTE8_TO_CHAR (it->c);
	    }

	  if (it->dp
	      && (dv = DISP_CHAR_VECTOR (it->dp, c),
		  VECTORP (dv)))
	    {
	      struct Lisp_Vector *v = XVECTOR (dv);

	      /* Return the first character from the display table
		 entry, if not empty.  If empty, don't display the
		 current character.  */
	      if (v->header.size)
		{
		  it->dpvec_char_len = it->len;
		  it->dpvec = v->contents;
		  it->dpend = v->contents + v->header.size;
		  it->current.dpvec_index = 0;
		  it->dpvec_face_id = -1;
		  it->saved_face_id = it->face_id;
		  it->method = GET_FROM_DISPLAY_VECTOR;
		  it->ellipsis_p = false;
		}
	      else
		{
		  set_iterator_to_next (it, false);
		}
	      goto get_next;
	    }

	  if (! NILP (lookup_glyphless_char_display (c, it)))
	    {
	      if (it->what == IT_GLYPHLESS)
		goto done;
	      /* Don't display this character.  */
	      set_iterator_to_next (it, false);
	      goto get_next;
	    }

	  /* If `nobreak-char-display' is non-nil, we display
	     non-ASCII spaces and hyphens specially.  */
	  if (! ASCII_CHAR_P (c) && ! NILP (Vnobreak_char_display))
	    {
	      if (blankp (c))
		nonascii_space_p = true;
	      else if (c == SOFT_HYPHEN || c == HYPHEN
		       || c == NON_BREAKING_HYPHEN)
		nonascii_hyphen_p = true;
	    }

	  /* Translate control characters into `\003' or `^C' form.
	     Control characters coming from a display table entry are
	     currently not translated because we use IT->dpvec to hold
	     the translation.  This could easily be changed but I
	     don't believe that it is worth doing.

	     The characters handled by `nobreak-char-display' must be
	     translated too.

	     Non-printable characters and raw-byte characters are also
	     translated to octal or hexadecimal form.  */
	  if (((c < ' ' || c == 127) /* ASCII control chars.  */
	       ? (it->area != TEXT_AREA
		  /* In mode line, treat \n, \t like other crl chars.  */
		  || (c != '\t'
		      && it->glyph_row
		      && (it->glyph_row->mode_line_p || it->avoid_cursor_p))
		  || (c != '\n' && c != '\t'))
	       : (nonascii_space_p
		  || nonascii_hyphen_p
		  || CHAR_BYTE8_P (c)
		  || ! CHAR_PRINTABLE_P (c))))
	    {
	      /* C is a control character, non-ASCII space/hyphen,
		 raw-byte, or a non-printable character which must be
		 displayed either as '\003' or as `^C' where the '\\'
		 and '^' can be defined in the display table.  Fill
		 IT->ctl_chars with glyphs for what we have to
		 display.  Then, set IT->dpvec to these glyphs.  */
	      Lisp_Object gc;
	      int ctl_len;
	      int face_id;
	      int lface_id = 0;
	      int escape_glyph;

	      /* Handle control characters with ^.  */

	      if (ASCII_CHAR_P (c) && it->ctl_arrow_p)
		{
		  int g;

		  g = '^';	     /* default glyph for Control */
		  /* Set IT->ctl_chars[0] to the glyph for `^'.  */
		  if (it->dp
		      && (gc = DISP_CTRL_GLYPH (it->dp), GLYPH_CODE_P (gc)))
		    {
		      g = GLYPH_CODE_CHAR (gc);
		      lface_id = GLYPH_CODE_FACE (gc);
		    }

		  face_id = (lface_id
			     ? merge_faces (it->w, Qt, lface_id, it->face_id)
			     : merge_escape_glyph_face (it));

		  XSETINT (it->ctl_chars[0], g);
		  XSETINT (it->ctl_chars[1], c ^ 0100);
		  ctl_len = 2;
		  goto display_control;
		}

	      /* Handle non-ascii space in the mode where it only gets
		 highlighting.  */

	      if (nonascii_space_p && EQ (Vnobreak_char_display, Qt))
		{
		  /* Merge `nobreak-space' into the current face.  */
		  face_id = merge_faces (it->w, Qnobreak_space, 0,
					 it->face_id);
		  XSETINT (it->ctl_chars[0],
			   nobreak_char_ascii_display ? ' ' : it->c);
		  ctl_len = 1;
		  goto display_control;
		}

	      /* Handle non-ascii hyphens in the mode where it only
		 gets highlighting.  */

	      if (nonascii_hyphen_p && EQ (Vnobreak_char_display, Qt))
		{
		  /* Merge `nobreak-space' into the current face.  */
		  face_id = merge_faces (it->w, Qnobreak_hyphen, 0,
					 it->face_id);
		  XSETINT (it->ctl_chars[0],
			   nobreak_char_ascii_display ? '-' : it->c);
		  ctl_len = 1;
		  goto display_control;
		}

	      /* Handle sequences that start with the "escape glyph".  */

	      /* the default escape glyph is \.  */
	      escape_glyph = '\\';

	      if (it->dp
		  && (gc = DISP_ESCAPE_GLYPH (it->dp), GLYPH_CODE_P (gc)))
		{
		  escape_glyph = GLYPH_CODE_CHAR (gc);
		  lface_id = GLYPH_CODE_FACE (gc);
		}

	      face_id = (lface_id
			 ? merge_faces (it->w, Qt, lface_id, it->face_id)
			 : merge_escape_glyph_face (it));

	      /* Draw non-ASCII space/hyphen with escape glyph: */

	      if (nonascii_space_p || nonascii_hyphen_p)
		{
		  XSETINT (it->ctl_chars[0], escape_glyph);
		  XSETINT (it->ctl_chars[1], nonascii_space_p ? ' ' : '-');
		  ctl_len = 2;
		  goto display_control;
		}

	      {
		char str[10];
		int len, i;

		if (CHAR_BYTE8_P (c))
		  /* Display \200 or \x80 instead of \17777600.  */
		  c = CHAR_TO_BYTE8 (c);
		const char *format_string = display_raw_bytes_as_hex
					    ? "x%02x"
					    : "%03o";
		len = sprintf (str, format_string, c + 0u);

		XSETINT (it->ctl_chars[0], escape_glyph);
		for (i = 0; i < len; i++)
		  XSETINT (it->ctl_chars[i + 1], str[i]);
		ctl_len = len + 1;
	      }

	    display_control:
	      /* Set up IT->dpvec and return first character from it.  */
	      it->dpvec_char_len = it->len;
	      it->dpvec = it->ctl_chars;
	      it->dpend = it->dpvec + ctl_len;
	      it->current.dpvec_index = 0;
	      it->dpvec_face_id = face_id;
	      it->saved_face_id = it->face_id;
	      it->method = GET_FROM_DISPLAY_VECTOR;
	      it->ellipsis_p = false;
	      goto get_next;
	    }
	  it->char_to_display = c;
	}
      else if (success_p)
	{
	  it->char_to_display = it->c;
	}
    }

#ifdef HAVE_WINDOW_SYSTEM
  /* Adjust face id for a multibyte character.  There are no multibyte
     character in unibyte text.  */
  if ((it->what == IT_CHARACTER || it->what == IT_COMPOSITION)
      && it->multibyte_p
      && success_p
      && FRAME_WINDOW_P (it->f))
    {
      struct face *face = FACE_FROM_ID (it->f, it->face_id);

      if (it->what == IT_COMPOSITION && it->cmp_it.ch >= 0)
	{
	  /* Automatic composition with glyph-string.   */
	  Lisp_Object gstring = composition_gstring_from_id (it->cmp_it.id);

	  it->face_id = face_for_font (it->f, LGSTRING_FONT (gstring), face);
	}
      else
	{
	  ptrdiff_t pos = (it->s ? -1
		     : STRINGP (it->string) ? IT_STRING_CHARPOS (*it)
		     : IT_CHARPOS (*it));
	  int c;

	  if (it->what == IT_CHARACTER)
	    c = it->char_to_display;
	  else
	    {
	      struct composition *cmp = composition_table[it->cmp_it.id];
	      int i;

	      c = ' ';
	      for (i = 0; i < cmp->glyph_len; i++)
		/* TAB in a composition means display glyphs with
		   padding space on the left or right.  */
		if ((c = COMPOSITION_GLYPH (cmp, i)) != '\t')
		  break;
	    }
	  it->face_id = FACE_FOR_CHAR (it->f, face, c, pos, it->string);
	}
    }
#endif	/* HAVE_WINDOW_SYSTEM */

 done:
  /* Is this character the last one of a run of characters with
     box?  If yes, set IT->end_of_box_run_p to true.  */
  if (it->face_box_p
      && it->s == NULL)
    {
      if (it->method == GET_FROM_STRING && it->sp)
	{
	  int face_id = underlying_face_id (it);
	  struct face *face = FACE_FROM_ID_OR_NULL (it->f, face_id);

	  if (face)
	    {
	      if (face->box == FACE_NO_BOX)
		{
		  /* If the box comes from face properties in a
		     display string, check faces in that string.  */
		  int string_face_id = face_after_it_pos (it);
		  if (FACE_FROM_ID (it->f, string_face_id)->box == FACE_NO_BOX)
		    it->end_of_box_run_p = true;
		}
	      /* Otherwise, the box comes from the underlying face.
		 If this is the last string character displayed, check
		 the next buffer location.  */
	      else if (((IT_STRING_CHARPOS (*it) >= SCHARS (it->string) - 1)
			/* For a composition, see if the string ends
			   at the last character included in the
			   composition.  */
			|| (it->what == IT_COMPOSITION
			    && (IT_STRING_CHARPOS (*it) + it->cmp_it.nchars
				>= SCHARS (it->string))))
		       /* n_overlay_strings is unreliable unless
			  overlay_string_index is non-negative.  */
		       && ((it->current.overlay_string_index >= 0
			    && (it->current.overlay_string_index
				== it->n_overlay_strings - 1))
			   /* A string from display property.  */
			   || it->from_disp_prop_p))
		{
		  ptrdiff_t ignore;
		  int next_face_id;
		  bool text_from_string = false;
		  /* Normally, the next buffer location is stored in
		     IT->current.pos...  */
		  struct text_pos pos = it->current.pos;

		  /* ...but for a string from a display property, the
		     next buffer position is stored in the 'position'
		     member of the iteration stack slot below the
		     current one, see handle_single_display_spec.  By
		     contrast, it->current.pos was not yet updated to
		     point to that buffer position; that will happen
		     in pop_it, after we finish displaying the current
		     string.  Note that we already checked above that
		     it->sp is positive, so subtracting one from it is
		     safe.  */
		  if (it->from_disp_prop_p)
		    {
		      int stackp = it->sp - 1;

		      /* Find the stack level with data from buffer.  */
		      while (stackp >= 0
			     && STRINGP ((it->stack + stackp)->string))
			stackp--;
		      if (stackp < 0)
			{
			  /* If no stack slot was found for iterating
			     a buffer, we are displaying text from a
			     string, most probably the mode line or
			     the header line, and that string has a
			     display string on some of its
			     characters.  */
			  text_from_string = true;
			  pos = it->stack[it->sp - 1].position;
			}
		      else
			pos = (it->stack + stackp)->position;
		    }
		  else
		    INC_TEXT_POS (pos, it->multibyte_p);

		  if (text_from_string)
		    {
		      Lisp_Object base_string = it->stack[it->sp - 1].string;

		      if (CHARPOS (pos) >= SCHARS (base_string) - 1)
			it->end_of_box_run_p = true;
		      else
			{
			  next_face_id
			    = face_at_string_position (it->w, base_string,
						       CHARPOS (pos), 0,
						       &ignore, face_id,
			                               false, 0);
			  if (FACE_FROM_ID (it->f, next_face_id)->box
			      == FACE_NO_BOX)
			    it->end_of_box_run_p = true;
			}
		    }
		  else if (CHARPOS (pos) >= ZV)
		    it->end_of_box_run_p = true;
		  else
		    {
		      next_face_id =
			face_at_buffer_position (it->w, CHARPOS (pos),
			                         &ignore,
						 CHARPOS (pos)
						 + TEXT_PROP_DISTANCE_LIMIT,
			                         false, -1, 0);
		      if (FACE_FROM_ID (it->f, next_face_id)->box
			  == FACE_NO_BOX)
			it->end_of_box_run_p = true;
		    }
		}
	    }
	}
      /* next_element_from_display_vector sets this flag according to
	 faces of the display vector glyphs, see there.  */
      else if (it->method != GET_FROM_DISPLAY_VECTOR)
	{
	  int face_id = face_after_it_pos (it);
	  if (face_id != it->face_id
	      && FACE_FROM_ID (it->f, face_id)->box == FACE_NO_BOX)
	    it->end_of_box_run_p = true;
	}
    }
  /* If we reached the end of the object we've been iterating (e.g., a
     display string or an overlay string), and there's something on
     IT->stack, proceed with what's on the stack.  It doesn't make
     sense to return false if there's unprocessed stuff on the stack,
     because otherwise that stuff will never be displayed.  */
  if (!success_p && it->sp > 0)
    {
      set_iterator_to_next (it, false);
      success_p = get_next_display_element (it);
    }

  /* Value is false if end of buffer or string reached.  */
  return success_p;
}


/* Move IT to the next display element.

   RESEAT_P means if called on a newline in buffer text,
   skip to the next visible line start.

   Functions get_next_display_element and set_iterator_to_next are
   separate because I find this arrangement easier to handle than a
   get_next_display_element function that also increments IT's
   position.  The way it is we can first look at an iterator's current
   display element, decide whether it fits on a line, and if it does,
   increment the iterator position.  The other way around we probably
   would either need a flag indicating whether the iterator has to be
   incremented the next time, or we would have to implement a
   decrement position function which would not be easy to write.  */

void
set_iterator_to_next (struct it *it, bool reseat_p)
{

  if (max_redisplay_ticks > 0)
    update_redisplay_ticks (1, it->w);

  switch (it->method)
    {
    case GET_FROM_BUFFER:
      /* The current display element of IT is a character from
	 current_buffer.  Advance in the buffer, and maybe skip over
	 invisible lines that are so because of selective display.  */
      if (ITERATOR_AT_END_OF_LINE_P (it) && reseat_p)
	reseat_at_next_visible_line_start (it, false);
      else if (it->cmp_it.id >= 0)
	{
	  /* We are currently getting glyphs from a composition.  */
	  if (! it->bidi_p)
	    {
	      IT_CHARPOS (*it) += it->cmp_it.nchars;
	      IT_BYTEPOS (*it) += it->cmp_it.nbytes;
	    }
	  else
	    {
	      int i;

	      /* Update IT's char/byte positions to point to the first
		 character of the next grapheme cluster, or to the
		 character visually after the current composition.  */
	      for (i = 0; i < it->cmp_it.nchars; i++)
		bidi_move_to_visually_next (&it->bidi_it);
	      IT_BYTEPOS (*it) = it->bidi_it.bytepos;
	      IT_CHARPOS (*it) = it->bidi_it.charpos;
	    }

	  if ((! it->bidi_p || ! it->cmp_it.reversed_p)
	      && it->cmp_it.to < it->cmp_it.nglyphs)
	    {
	      /* Composition created while scanning forward.  Proceed
		 to the next grapheme cluster.  */
	      it->cmp_it.from = it->cmp_it.to;
	    }
	  else if ((it->bidi_p && it->cmp_it.reversed_p)
		   && it->cmp_it.from > 0)
	    {
	      /* Composition created while scanning backward.  Proceed
		 to the previous grapheme cluster.  */
	      it->cmp_it.to = it->cmp_it.from;
	    }
	  else
	    {
	      /* No more grapheme clusters in this composition.
		 Find the next stop position.  */
	      ptrdiff_t stop = it->end_charpos;

	      if (it->bidi_it.scan_dir < 0)
		/* Now we are scanning backward and don't know
		   where to stop.  */
		stop = -1;
	      composition_compute_stop_pos (&it->cmp_it, IT_CHARPOS (*it),
					    IT_BYTEPOS (*it), stop, Qnil);
	    }
	}
      else
	{
	  eassert (it->len != 0);

	  if (!it->bidi_p)
	    {
	      IT_BYTEPOS (*it) += it->len;
	      IT_CHARPOS (*it) += 1;
	    }
	  else
	    {
	      int prev_scan_dir = it->bidi_it.scan_dir;
	      /* If this is a new paragraph, determine its base
		 direction (a.k.a. its base embedding level).  */
	      if (it->bidi_it.new_paragraph)
		bidi_paragraph_init (it->paragraph_embedding, &it->bidi_it,
				     false);
	      bidi_move_to_visually_next (&it->bidi_it);
	      IT_BYTEPOS (*it) = it->bidi_it.bytepos;
	      IT_CHARPOS (*it) = it->bidi_it.charpos;
	      if (prev_scan_dir != it->bidi_it.scan_dir)
		{
		  /* As the scan direction was changed, we must
		     re-compute the stop position for composition.  */
		  ptrdiff_t stop = it->end_charpos;
		  if (it->bidi_it.scan_dir < 0)
		    stop = -1;
		  composition_compute_stop_pos (&it->cmp_it, IT_CHARPOS (*it),
						IT_BYTEPOS (*it), stop, Qnil);
		}
	    }
	  eassert (IT_BYTEPOS (*it) == CHAR_TO_BYTE (IT_CHARPOS (*it)));
	}
      break;

    case GET_FROM_C_STRING:
      /* Current display element of IT is from a C string.  */
      if (!it->bidi_p
	  /* If the string position is beyond string's end, it means
	     next_element_from_c_string is padding the string with
	     blanks, in which case we bypass the bidi iterator,
	     because it cannot deal with such virtual characters.  */
	  || IT_CHARPOS (*it) >= it->bidi_it.string.schars)
	{
	  IT_BYTEPOS (*it) += it->len;
	  IT_CHARPOS (*it) += 1;
	}
      else
	{
	  bidi_move_to_visually_next (&it->bidi_it);
	  IT_BYTEPOS (*it) = it->bidi_it.bytepos;
	  IT_CHARPOS (*it) = it->bidi_it.charpos;
	}
      break;

    case GET_FROM_DISPLAY_VECTOR:
      /* Current display element of IT is from a display table entry.
	 Advance in the display table definition.  Reset it to null if
	 end reached, and continue with characters from buffers/
	 strings.  */
      ++it->current.dpvec_index;

      /* Restore face of the iterator to what they were before the
         display vector entry (these entries may contain faces).  */
      it->face_id = it->saved_face_id;

      if (it->dpvec + it->current.dpvec_index >= it->dpend)
	{
	  bool recheck_faces = it->ellipsis_p;

	  if (it->s)
	    it->method = GET_FROM_C_STRING;
	  else if (STRINGP (it->string))
	    it->method = GET_FROM_STRING;
	  else
	    {
	      it->method = GET_FROM_BUFFER;
	      it->object = it->w->contents;
	    }

	  it->dpvec = NULL;
	  it->current.dpvec_index = -1;

	  /* Skip over characters which were displayed via IT->dpvec.  */
	  if (it->dpvec_char_len < 0)
	    reseat_at_next_visible_line_start (it, true);
	  else if (it->dpvec_char_len > 0)
	    {
	      it->len = it->dpvec_char_len;
	      set_iterator_to_next (it, reseat_p);
	    }

	  /* Maybe recheck faces after display vector.  */
	  if (recheck_faces)
	    {
	      if (it->method == GET_FROM_STRING)
		it->stop_charpos = IT_STRING_CHARPOS (*it);
	      else
		it->stop_charpos = IT_CHARPOS (*it);
	    }
	}
      break;

    case GET_FROM_STRING:
      /* Current display element is a character from a Lisp string.  */
      eassert (it->s == NULL && STRINGP (it->string));
      /* Don't advance past string end.  These conditions are true
	 when set_iterator_to_next is called at the end of
	 get_next_display_element, in which case the Lisp string is
	 already exhausted, and all we want is pop the iterator
	 stack.  */
      if (it->current.overlay_string_index >= 0)
	{
	  /* This is an overlay string, so there's no padding with
	     spaces, and the number of characters in the string is
	     where the string ends.  */
	  if (IT_STRING_CHARPOS (*it) >= SCHARS (it->string))
	    goto consider_string_end;
	}
      else
	{
	  /* Not an overlay string.  There could be padding, so test
	     against it->end_charpos.  */
	  if (IT_STRING_CHARPOS (*it) >= it->end_charpos)
	    goto consider_string_end;
	}
      if (it->cmp_it.id >= 0)
	{
	  /* We are delivering display elements from a composition.
	     Update the string position past the grapheme cluster
	     we've just processed.  */
	  if (! it->bidi_p)
	    {
	      IT_STRING_CHARPOS (*it) += it->cmp_it.nchars;
	      IT_STRING_BYTEPOS (*it) += it->cmp_it.nbytes;
	    }
	  else
	    {
	      int i;

	      for (i = 0; i < it->cmp_it.nchars; i++)
		bidi_move_to_visually_next (&it->bidi_it);
	      IT_STRING_BYTEPOS (*it) = it->bidi_it.bytepos;
	      IT_STRING_CHARPOS (*it) = it->bidi_it.charpos;
	    }

	  /* Did we exhaust all the grapheme clusters of this
	     composition?  */
	  if ((! it->bidi_p || ! it->cmp_it.reversed_p)
	      && (it->cmp_it.to < it->cmp_it.nglyphs))
	    {
	      /* Not all the grapheme clusters were processed yet;
		 advance to the next cluster.  */
	      it->cmp_it.from = it->cmp_it.to;
	    }
	  else if ((it->bidi_p && it->cmp_it.reversed_p)
		   && it->cmp_it.from > 0)
	    {
	      /* Likewise: advance to the next cluster, but going in
		 the reverse direction.  */
	      it->cmp_it.to = it->cmp_it.from;
	    }
	  else
	    {
	      /* This composition was fully processed; find the next
		 candidate place for checking for composed
		 characters.  */
	      /* Always limit string searches to the string length;
		 any padding spaces are not part of the string, and
		 there cannot be any compositions in that padding.  */
	      ptrdiff_t stop = SCHARS (it->string);

	      if (it->bidi_p && it->bidi_it.scan_dir < 0)
		stop = -1;
	      else if (it->end_charpos < stop)
		{
		  /* Cf. PRECISION in reseat_to_string: we might be
		     limited in how many of the string characters we
		     need to deliver.  */
		  stop = it->end_charpos;
		}
	      composition_compute_stop_pos (&it->cmp_it,
					    IT_STRING_CHARPOS (*it),
					    IT_STRING_BYTEPOS (*it), stop,
					    it->string);
	    }
	}
      else
	{
	  if (!it->bidi_p
	      /* If the string position is beyond string's end, it
		 means next_element_from_string is padding the string
		 with blanks, in which case we bypass the bidi
		 iterator, because it cannot deal with such virtual
		 characters.  */
	      || IT_STRING_CHARPOS (*it) >= it->bidi_it.string.schars)
	    {
	      IT_STRING_BYTEPOS (*it) += it->len;
	      IT_STRING_CHARPOS (*it) += 1;
	    }
	  else
	    {
	      int prev_scan_dir = it->bidi_it.scan_dir;

	      bidi_move_to_visually_next (&it->bidi_it);
	      IT_STRING_BYTEPOS (*it) = it->bidi_it.bytepos;
	      IT_STRING_CHARPOS (*it) = it->bidi_it.charpos;
	      /* If the scan direction changes, we may need to update
		 the place where to check for composed characters.  */
	      if (prev_scan_dir != it->bidi_it.scan_dir)
		{
		  ptrdiff_t stop = SCHARS (it->string);

		  if (it->bidi_it.scan_dir < 0)
		    stop = -1;
		  else if (it->end_charpos < stop)
		    stop = it->end_charpos;

		  composition_compute_stop_pos (&it->cmp_it,
						IT_STRING_CHARPOS (*it),
						IT_STRING_BYTEPOS (*it), stop,
						it->string);
		}
	    }
	}

    consider_string_end:

      if (it->current.overlay_string_index >= 0)
	{
	  /* IT->string is an overlay string.  Advance to the
	     next, if there is one.  */
	  if (IT_STRING_CHARPOS (*it) >= SCHARS (it->string))
	    {
	      it->ellipsis_p = false;
	      next_overlay_string (it);
	      if (it->ellipsis_p)
		setup_for_ellipsis (it, 0);
	    }
	}
      else
	{
	  /* IT->string is not an overlay string.  If we reached
	     its end, and there is something on IT->stack, proceed
	     with what is on the stack.  This can be either another
	     string, this time an overlay string, or a buffer.  */
	  if (IT_STRING_CHARPOS (*it) == SCHARS (it->string)
	      && it->sp > 0)
	    {
	      pop_it (it);
	      if (it->method == GET_FROM_STRING)
		goto consider_string_end;
	    }
	}
      break;

    case GET_FROM_IMAGE:
    case GET_FROM_STRETCH:
    case GET_FROM_XWIDGET:

      /* The position etc with which we have to proceed are on
	 the stack.  The position may be at the end of a string,
         if the `display' property takes up the whole string.  */
      eassert (it->sp > 0);
      pop_it (it);
      if (it->method == GET_FROM_STRING)
	goto consider_string_end;
      break;

    default:
      /* There are no other methods defined, so this should be a bug.  */
      emacs_abort ();
    }

  eassert (it->method != GET_FROM_STRING
	   || (STRINGP (it->string)
	       && IT_STRING_CHARPOS (*it) >= 0));
}

/* Load IT's display element fields with information about the next
   display element which comes from a display table entry or from the
   result of translating a control character to one of the forms `^C'
   or `\003'.

   IT->dpvec holds the glyphs to return as characters.
   IT->saved_face_id holds the face id before the display vector--it
   is restored into IT->face_id in set_iterator_to_next.  */

static bool
next_element_from_display_vector (struct it *it)
{
  Lisp_Object gc;
  int prev_face_id = it->face_id;
  int next_face_id;

  /* Precondition.  */
  eassert (it->dpvec && it->current.dpvec_index >= 0);

  it->face_id = it->saved_face_id;

  /* KFS: This code used to check ip->dpvec[0] instead of the current element.
     That seemed totally bogus - so I changed it...  */
  if (it->dpend - it->dpvec > 0	/* empty dpvec[] is invalid */
      && (gc = it->dpvec[it->current.dpvec_index], GLYPH_CODE_P (gc)))
    {
      struct face *this_face, *prev_face, *next_face;

      it->c = GLYPH_CODE_CHAR (gc);
      it->len = CHAR_BYTES (it->c);

      /* The entry may contain a face id to use.  Such a face id is
	 the id of a Lisp face, not a realized face.  A face id of
	 zero means no face is specified.  */
      if (it->dpvec_face_id >= 0)
	it->face_id = it->dpvec_face_id;
      else
	{
	  int lface_id = GLYPH_CODE_FACE (gc);
	  if (lface_id > 0)
	    it->face_id = merge_faces (it->w, Qt, lface_id,
				       it->saved_face_id);
	}

      /* Glyphs in the display vector could have the box face, so we
	 need to set the related flags in the iterator, as
	 appropriate.  */
      this_face = FACE_FROM_ID_OR_NULL (it->f, it->face_id);
      prev_face = FACE_FROM_ID_OR_NULL (it->f, prev_face_id);

      /* Is this character the first character of a box-face run?  */
      it->start_of_box_run_p = (this_face && this_face->box != FACE_NO_BOX
				&& (!prev_face
				    || prev_face->box == FACE_NO_BOX));

      /* For the last character of the box-face run, we need to look
	 either at the next glyph from the display vector, or at the
	 face we saw before the display vector.  */
      next_face_id = it->saved_face_id;
      if (it->current.dpvec_index < it->dpend - it->dpvec - 1)
	{
	  if (it->dpvec_face_id >= 0)
	    next_face_id = it->dpvec_face_id;
	  else
	    {
              Lisp_Object gc = it->dpvec[it->current.dpvec_index + 1];
              int lface_id = GLYPH_CODE_P (gc) ? GLYPH_CODE_FACE (gc) : 0;

              if (lface_id > 0)
		next_face_id = merge_faces (it->w, Qt, lface_id,
					    it->saved_face_id);
	    }
	}
      next_face = FACE_FROM_ID_OR_NULL (it->f, next_face_id);
      if (this_face && this_face->box != FACE_NO_BOX
	  && (!next_face || next_face->box == FACE_NO_BOX))
	it->end_of_box_run_p = true;
      it->face_box_p = this_face && this_face->box != FACE_NO_BOX;
    }
  else
    /* Display table entry is invalid.  Return a space.  */
    it->c = ' ', it->len = 1;

  /* Don't change position and object of the iterator here.  They are
     still the values of the character that had this display table
     entry or was translated, and that's what we want.  */
  it->what = IT_CHARACTER;
  return true;
}

/* Get the first element of string/buffer in the visual order, after
   being reseated to a new position in a string or a buffer.  */
static void
get_visually_first_element (struct it *it)
{
  bool string_p = STRINGP (it->string) || it->s;
  ptrdiff_t eob = (string_p ? it->bidi_it.string.schars : ZV);
  ptrdiff_t bob;
  ptrdiff_t obegv = BEGV;

  SET_WITH_NARROWED_BEGV (it, bob,
			  string_p ? 0 :
			  IT_CHARPOS (*it) < BEGV ? obegv : BEGV,
			  it->narrowed_begv);

  if (STRINGP (it->string))
    {
      it->bidi_it.charpos = IT_STRING_CHARPOS (*it);
      it->bidi_it.bytepos = IT_STRING_BYTEPOS (*it);
    }
  else
    {
      it->bidi_it.charpos = IT_CHARPOS (*it);
      it->bidi_it.bytepos = IT_BYTEPOS (*it);
    }

  if (it->bidi_it.charpos == eob)
    {
      /* Nothing to do, but reset the FIRST_ELT flag, like
	 bidi_paragraph_init does, because we are not going to
	 call it.  */
      it->bidi_it.first_elt = false;
    }
  else if (it->bidi_it.charpos == bob
	   || (!string_p
	       && (FETCH_BYTE (it->bidi_it.bytepos - 1) == '\n'
		   || FETCH_BYTE (it->bidi_it.bytepos) == '\n')))
    {
      /* If we are at the beginning of a line/string, we can produce
	 the next element right away.  */
      bidi_paragraph_init (it->paragraph_embedding, &it->bidi_it, true);
      bidi_move_to_visually_next (&it->bidi_it);
    }
  else
    {
      ptrdiff_t orig_bytepos = it->bidi_it.bytepos;

      /* We need to prime the bidi iterator starting at the line's or
	 string's beginning, before we will be able to produce the
	 next element.  */
      if (string_p)
	it->bidi_it.charpos = it->bidi_it.bytepos = 0;
      else
	SET_WITH_NARROWED_BEGV (it, it->bidi_it.charpos,
				find_newline_no_quit (IT_CHARPOS (*it),
						      IT_BYTEPOS (*it), -1,
						      &it->bidi_it.bytepos),
				it->narrowed_begv);
      bidi_paragraph_init (it->paragraph_embedding, &it->bidi_it, true);
      do
	{
	  /* Now return to buffer/string position where we were asked
	     to get the next display element, and produce that.  */
	  bidi_move_to_visually_next (&it->bidi_it);
	}
      while (it->bidi_it.bytepos != orig_bytepos
	     && it->bidi_it.charpos < eob);
    }

  /*  Adjust IT's position information to where we ended up.  */
  if (STRINGP (it->string))
    {
      IT_STRING_CHARPOS (*it) = it->bidi_it.charpos;
      IT_STRING_BYTEPOS (*it) = it->bidi_it.bytepos;
    }
  else
    {
      IT_CHARPOS (*it) = it->bidi_it.charpos;
      IT_BYTEPOS (*it) = it->bidi_it.bytepos;
    }

  if (STRINGP (it->string) || !it->s)
    {
      ptrdiff_t stop, charpos, bytepos;

      if (STRINGP (it->string))
	{
	  eassert (!it->s);
	  stop = SCHARS (it->string);
	  if (stop > it->end_charpos)
	    stop = it->end_charpos;
	  charpos = IT_STRING_CHARPOS (*it);
	  bytepos = IT_STRING_BYTEPOS (*it);
	}
      else
	{
	  stop = it->end_charpos;
	  charpos = IT_CHARPOS (*it);
	  bytepos = IT_BYTEPOS (*it);
	}
      if (it->bidi_it.scan_dir < 0)
	stop = -1;
      composition_compute_stop_pos (&it->cmp_it, charpos, bytepos, stop,
				    it->string);
    }
}

/* Load IT with the next display element from Lisp string IT->string.
   IT->current.string_pos is the current position within the string.
   If IT->current.overlay_string_index >= 0, the Lisp string is an
   overlay string.  */

static bool
next_element_from_string (struct it *it)
{
  struct text_pos position;

  eassert (STRINGP (it->string));
  eassert (!it->bidi_p || EQ (it->string, it->bidi_it.string.lstring));
  eassert (IT_STRING_CHARPOS (*it) >= 0);
  position = it->current.string_pos;

  /* With bidi reordering, the character to display might not be the
     character at IT_STRING_CHARPOS.  BIDI_IT.FIRST_ELT means
     that we were reseat()ed to a new string, whose paragraph
     direction is not known.  */
  if (it->bidi_p && it->bidi_it.first_elt)
    {
      get_visually_first_element (it);
      SET_TEXT_POS (position, IT_STRING_CHARPOS (*it), IT_STRING_BYTEPOS (*it));
    }

  /* Time to check for invisible text?  */
  if (IT_STRING_CHARPOS (*it) < it->end_charpos)
    {
      if (IT_STRING_CHARPOS (*it) >= it->stop_charpos)
	{
	  if (!(!it->bidi_p
		|| BIDI_AT_BASE_LEVEL (it->bidi_it)
		|| IT_STRING_CHARPOS (*it) == it->stop_charpos))
	    {
	      /* With bidi non-linear iteration, we could find
		 ourselves far beyond the last computed stop_charpos,
		 with several other stop positions in between that we
		 missed.  Scan them all now, in buffer's logical
		 order, until we find and handle the last stop_charpos
		 that precedes our current position.  */
	      handle_stop_backwards (it, it->stop_charpos);
	      return GET_NEXT_DISPLAY_ELEMENT (it);
	    }
	  else
	    {
	      if (it->bidi_p)
		{
		  /* Take note of the stop position we just moved
		     across, for when we will move back across it.  */
		  it->prev_stop = it->stop_charpos;
		  /* If we are at base paragraph embedding level, take
		     note of the last stop position seen at this
		     level.  */
		  if (BIDI_AT_BASE_LEVEL (it->bidi_it))
		    it->base_level_stop = it->stop_charpos;
		}
	      handle_stop (it);

	      /* Since a handler may have changed IT->method, we must
		 recurse here.  */
	      return GET_NEXT_DISPLAY_ELEMENT (it);
	    }
	}
      else if (it->bidi_p
	       /* If we are before prev_stop, we may have overstepped
		  on our way backwards a stop_pos, and if so, we need
		  to handle that stop_pos.  */
	       && IT_STRING_CHARPOS (*it) < it->prev_stop
	       /* We can sometimes back up for reasons that have nothing
		  to do with bidi reordering.  E.g., compositions.  The
		  code below is only needed when we are above the base
		  embedding level, so test for that explicitly.  */
	       && !BIDI_AT_BASE_LEVEL (it->bidi_it))
	{
	  /* If we lost track of base_level_stop, we have no better
	     place for handle_stop_backwards to start from than string
	     beginning.  This happens, e.g., when we were reseated to
	     the previous screenful of text by vertical-motion.  */
	  if (it->base_level_stop <= 0
	      || IT_STRING_CHARPOS (*it) < it->base_level_stop)
	    it->base_level_stop = 0;
	  handle_stop_backwards (it, it->base_level_stop);
	  return GET_NEXT_DISPLAY_ELEMENT (it);
	}
    }

  if (it->current.overlay_string_index >= 0)
    {
      /* Get the next character from an overlay string.  In overlay
	 strings, there is no field width or padding with spaces to
	 do.  */
      if (IT_STRING_CHARPOS (*it) >= SCHARS (it->string))
	{
	  it->what = IT_EOB;
	  return false;
	}
      else if (CHAR_COMPOSED_P (it, IT_STRING_CHARPOS (*it),
				IT_STRING_BYTEPOS (*it),
				it->bidi_it.scan_dir < 0
				? -1
				: SCHARS (it->string))
	       && next_element_from_composition (it))
	{
	  return true;
	}
      else if (STRING_MULTIBYTE (it->string))
	{
	  const unsigned char *s = (SDATA (it->string)
				    + IT_STRING_BYTEPOS (*it));
	  it->c = check_char_and_length (s, &it->len);
	}
      else
	{
	  it->c = SREF (it->string, IT_STRING_BYTEPOS (*it));
	  it->len = 1;
	}
    }
  else
    {
      /* Get the next character from a Lisp string that is not an
	 overlay string.  Such strings come from the mode line, for
	 example.  We may have to pad with spaces, or truncate the
	 string.  See also next_element_from_c_string.  */
      if (IT_STRING_CHARPOS (*it) >= it->end_charpos)
	{
	  it->what = IT_EOB;
	  return false;
	}
      else if (IT_STRING_CHARPOS (*it) >= it->string_nchars)
	{
	  /* Pad with spaces.  */
	  it->c = ' ', it->len = 1;
	  CHARPOS (position) = BYTEPOS (position) = -1;
	}
      else if (CHAR_COMPOSED_P (it, IT_STRING_CHARPOS (*it),
				IT_STRING_BYTEPOS (*it),
				it->bidi_it.scan_dir < 0
				? -1
				: it->string_nchars)
	       && next_element_from_composition (it))
	{
	  return true;
	}
      else if (STRING_MULTIBYTE (it->string))
	{
	  const unsigned char *s = (SDATA (it->string)
				    + IT_STRING_BYTEPOS (*it));
	  it->c = check_char_and_length (s, &it->len);
	}
      else
	{
	  it->c = SREF (it->string, IT_STRING_BYTEPOS (*it));
	  it->len = 1;
	}
    }

  /* Record what we have and where it came from.  */
  it->what = IT_CHARACTER;
  it->object = it->string;
  it->position = position;
  return true;
}


/* Load IT with next display element from C string IT->s.
   IT->string_nchars is the maximum number of characters to return
   from the string.  IT->end_charpos may be greater than
   IT->string_nchars when this function is called, in which case we
   may have to return padding spaces.  Value is false if end of string
   reached, including padding spaces.  */

static bool
next_element_from_c_string (struct it *it)
{
  bool success_p = true;

  eassert (it->s);
  eassert (!it->bidi_p || it->s == it->bidi_it.string.s);
  it->what = IT_CHARACTER;
  BYTEPOS (it->position) = CHARPOS (it->position) = 0;
  it->object = make_fixnum (0);

  /* With bidi reordering, the character to display might not be the
     character at IT_CHARPOS.  BIDI_IT.FIRST_ELT means that
     we were reseated to a new string, whose paragraph direction is
     not known.  */
  if (it->bidi_p && it->bidi_it.first_elt)
    get_visually_first_element (it);

  /* IT's position can be greater than IT->string_nchars in case a
     field width or precision has been specified when the iterator was
     initialized.  */
  if (IT_CHARPOS (*it) >= it->end_charpos)
    {
      /* End of the game.  */
      it->what = IT_EOB;
      success_p = false;
    }
  else if (IT_CHARPOS (*it) >= it->string_nchars)
    {
      /* Pad with spaces.  */
      it->c = ' ', it->len = 1;
      BYTEPOS (it->position) = CHARPOS (it->position) = -1;
    }
  else if (it->multibyte_p)
    it->c = check_char_and_length (it->s + IT_BYTEPOS (*it), &it->len);
  else
    it->c = it->s[IT_BYTEPOS (*it)], it->len = 1;

  return success_p;
}


/* Set up IT to return characters from an ellipsis, if appropriate.
   The definition of the ellipsis glyphs may come from a display table
   entry.  This function fills IT with the first glyph from the
   ellipsis if an ellipsis is to be displayed.  */

static bool
next_element_from_ellipsis (struct it *it)
{
  if (it->selective_display_ellipsis_p)
    setup_for_ellipsis (it, it->len);
  else
    {
      /* The face at the current position may be different from the
	 face we find after the invisible text.  Remember what it
	 was in IT->saved_face_id, and signal that it's there by
	 setting face_before_selective_p.  */
      it->saved_face_id = it->face_id;
      it->method = GET_FROM_BUFFER;
      it->object = it->w->contents;
      reseat_at_next_visible_line_start (it, true);
      it->face_before_selective_p = true;
    }

  return GET_NEXT_DISPLAY_ELEMENT (it);
}


/* Deliver an image display element.  The iterator IT is already
   filled with image information (done in handle_display_prop).  Value
   is always true.  */


static bool
next_element_from_image (struct it *it)
{
  it->what = IT_IMAGE;
  return true;
}

static bool
next_element_from_xwidget (struct it *it)
{
  it->what = IT_XWIDGET;
  return true;
}


/* Fill iterator IT with next display element from a stretch glyph
   property.  IT->object is the value of the text property.  Value is
   always true.  */

static bool
next_element_from_stretch (struct it *it)
{
  it->what = IT_STRETCH;
  return true;
}

/* Scan backwards from IT's current position until we find a stop
   position, or until BEGV.  This is called when we find ourself
   before both the last known prev_stop and base_level_stop while
   reordering bidirectional text.  */

static void
compute_stop_pos_backwards (struct it *it)
{
  const int SCAN_BACK_LIMIT = 1000;
  struct text_pos pos;
  struct display_pos save_current = it->current;
  struct text_pos save_position = it->position;
  ptrdiff_t charpos = IT_CHARPOS (*it);
  ptrdiff_t where_we_are = charpos;
  ptrdiff_t save_stop_pos = it->stop_charpos;
  ptrdiff_t save_end_pos = it->end_charpos;

  eassert (NILP (it->string) && !it->s);
  eassert (it->bidi_p);
  it->bidi_p = false;
  do
    {
      it->end_charpos = min (charpos + 1, ZV);
      charpos = max (charpos - SCAN_BACK_LIMIT, BEGV);
      SET_TEXT_POS (pos, charpos, CHAR_TO_BYTE (charpos));
      reseat_1 (it, pos, false);
      compute_stop_pos (it);
      /* We must advance forward, right?  */
      if (it->stop_charpos <= charpos)
	emacs_abort ();
    }
  while (charpos > BEGV && it->stop_charpos >= it->end_charpos);

  if (it->stop_charpos <= where_we_are)
    it->prev_stop = it->stop_charpos;
  else
    it->prev_stop = BEGV;
  it->bidi_p = true;
  it->current = save_current;
  it->position = save_position;
  it->stop_charpos = save_stop_pos;
  it->end_charpos = save_end_pos;
}

/* Scan forward from CHARPOS in the current buffer/string, until we
   find a stop position > current IT's position.  Then handle the stop
   position before that.  This is called when we bump into a stop
   position while reordering bidirectional text.  CHARPOS should be
   the last previously processed stop_pos (or BEGV/0, if none were
   processed yet) whose position is less than IT's current
   position.  */

static void
handle_stop_backwards (struct it *it, ptrdiff_t charpos)
{
  bool bufp = !STRINGP (it->string);
  ptrdiff_t where_we_are = (bufp ? IT_CHARPOS (*it) : IT_STRING_CHARPOS (*it));
  struct display_pos save_current = it->current;
  struct text_pos save_position = it->position;
  struct composition_it save_cmp_it = it->cmp_it;
  struct text_pos pos1;
  ptrdiff_t next_stop;

  /* Scan in strict logical order.  */
  eassert (it->bidi_p);
  it->bidi_p = false;
  do
    {
      it->prev_stop = charpos;
      if (bufp)
	{
	  SET_TEXT_POS (pos1, charpos, CHAR_TO_BYTE (charpos));
	  reseat_1 (it, pos1, false);
	}
      else
	it->current.string_pos = string_pos (charpos, it->string);
      compute_stop_pos (it);
      /* We must advance forward, right?  */
      if (it->stop_charpos <= it->prev_stop)
	emacs_abort ();
      charpos = it->stop_charpos;
    }
  while (charpos <= where_we_are);

  it->bidi_p = true;
  it->current = save_current;
  it->position = save_position;
  it->cmp_it = save_cmp_it;
  next_stop = it->stop_charpos;
  it->stop_charpos = it->prev_stop;
  handle_stop (it);
  it->stop_charpos = next_stop;
}

/* Load IT with the next display element from current_buffer.  Value
   is false if end of buffer reached.  IT->stop_charpos is the next
   position at which to stop and check for text properties or buffer
   end.  */

static bool
next_element_from_buffer (struct it *it)
{
  bool success_p = true;

  eassert (IT_CHARPOS (*it) >= BEGV);
  eassert (NILP (it->string) && !it->s);
  eassert (!it->bidi_p
	   || (NILP (it->bidi_it.string.lstring)
	       && it->bidi_it.string.s == NULL));

  /* With bidi reordering, the character to display might not be the
     character at IT_CHARPOS.  BIDI_IT.FIRST_ELT means that
     we were reseat()ed to a new buffer position, which is potentially
     a different paragraph.  */
  if (it->bidi_p && it->bidi_it.first_elt)
    {
      get_visually_first_element (it);
      SET_TEXT_POS (it->position, IT_CHARPOS (*it), IT_BYTEPOS (*it));
    }

  if (IT_CHARPOS (*it) >= it->stop_charpos)
    {
      if (IT_CHARPOS (*it) >= it->end_charpos)
	{
	  bool overlay_strings_follow_p;

	  /* End of the game, except when overlay strings follow that
	     haven't been returned yet.  */
	  if (it->overlay_strings_at_end_processed_p)
	    overlay_strings_follow_p = false;
	  else
	    {
	      it->overlay_strings_at_end_processed_p = true;
	      overlay_strings_follow_p = get_overlay_strings (it, 0);
	    }

	  if (overlay_strings_follow_p)
	    success_p = GET_NEXT_DISPLAY_ELEMENT (it);
	  else
	    {
	      it->what = IT_EOB;
	      it->position = it->current.pos;
	      success_p = false;
	    }
	}
      else if (!(!it->bidi_p
		 || BIDI_AT_BASE_LEVEL (it->bidi_it)
		 || IT_CHARPOS (*it) == it->stop_charpos))
	{
	  /* With bidi non-linear iteration, we could find ourselves
	     far beyond the last computed stop_charpos, with several
	     other stop positions in between that we missed.  Scan
	     them all now, in buffer's logical order, until we find
	     and handle the last stop_charpos that precedes our
	     current position.  */
	  handle_stop_backwards (it, it->stop_charpos);
	  it->ignore_overlay_strings_at_pos_p = false;
	  return GET_NEXT_DISPLAY_ELEMENT (it);
	}
      else
	{
	  if (it->bidi_p)
	    {
	      /* Take note of the stop position we just moved across,
		 for when we will move back across it.  */
	      it->prev_stop = it->stop_charpos;
	      /* If we are at base paragraph embedding level, take
		 note of the last stop position seen at this
		 level.  */
	      if (BIDI_AT_BASE_LEVEL (it->bidi_it))
		it->base_level_stop = it->stop_charpos;
	    }
	  handle_stop (it);
	  it->ignore_overlay_strings_at_pos_p = false;
	  return GET_NEXT_DISPLAY_ELEMENT (it);
	}
    }
  else if (it->bidi_p
	   /* If we are before prev_stop, we may have overstepped on
	      our way backwards a stop_pos, and if so, we need to
	      handle that stop_pos.  */
	   && IT_CHARPOS (*it) < it->prev_stop
	   /* We can sometimes back up for reasons that have nothing
	      to do with bidi reordering.  E.g., compositions.  The
	      code below is only needed when we are above the base
	      embedding level, so test for that explicitly.  */
	   && !BIDI_AT_BASE_LEVEL (it->bidi_it))
    {
      if (it->base_level_stop <= 0
	  || IT_CHARPOS (*it) < it->base_level_stop)
	{
	  /* If we lost track of base_level_stop, we need to find
	     prev_stop by looking backwards.  This happens, e.g., when
	     we were reseated to the previous screenful of text by
	     vertical-motion.  */
	  it->base_level_stop = BEGV;
	  compute_stop_pos_backwards (it);
	  handle_stop_backwards (it, it->prev_stop);
	}
      else
	handle_stop_backwards (it, it->base_level_stop);
      it->ignore_overlay_strings_at_pos_p = false;
      return GET_NEXT_DISPLAY_ELEMENT (it);
    }
  else
    {
      /* No face changes, overlays etc. in sight, so just return a
	 character from current_buffer.  */
      unsigned char *p;
      ptrdiff_t stop;

      /* We moved to the next buffer position, so any info about
	 previously seen overlays is no longer valid.  */
      it->ignore_overlay_strings_at_pos_p = false;

      if (composition_break_at_point
	  && !NILP (BVAR (current_buffer, enable_multibyte_characters))
	  && !NILP (Vauto_composition_mode))
	{
	  /* Limit search for composable characters to point's position.  */
	  if (it->bidi_it.scan_dir < 0)
	    stop = (PT <= IT_CHARPOS (*it)) ? PT : -1;
	  else
	    stop = (IT_CHARPOS (*it) < PT
		    && PT < it->end_charpos) ? PT : it->end_charpos;
	}
      else
	stop = it->bidi_it.scan_dir < 0 ? -1 : it->end_charpos;
      if (CHAR_COMPOSED_P (it, IT_CHARPOS (*it), IT_BYTEPOS (*it),
			   stop)
	  && next_element_from_composition (it))
	{
	  return true;
	}

      /* Get the next character, maybe multibyte.  */
      p = BYTE_POS_ADDR (IT_BYTEPOS (*it));
      if (it->multibyte_p && !ASCII_CHAR_P (*p))
	it->c = string_char_and_length (p, &it->len);
      else
	it->c = *p, it->len = 1;

      /* Record what we have and where it came from.  */
      it->what = IT_CHARACTER;
      it->object = it->w->contents;
      it->position = it->current.pos;

      /* Normally we return the character found above, except when we
	 really want to return an ellipsis for selective display.  */
      if (it->selective)
	{
	  if (it->c == '\n')
	    {
	      /* A value of selective > 0 means hide lines indented more
		 than that number of columns.  */
	      if (it->selective > 0
		  && IT_CHARPOS (*it) + 1 < ZV
		  && indented_beyond_p (IT_CHARPOS (*it) + 1,
					IT_BYTEPOS (*it) + 1,
					it->selective))
		{
		  success_p = next_element_from_ellipsis (it);
		  it->dpvec_char_len = -1;
		}
	    }
	  else if (it->c == '\r' && it->selective == -1)
	    {
	      /* A value of selective == -1 means that everything from the
		 CR to the end of the line is invisible, with maybe an
		 ellipsis displayed for it.  */
	      success_p = next_element_from_ellipsis (it);
	      it->dpvec_char_len = -1;
	    }
	}
    }

  /* Value is false if end of buffer reached.  */
  eassert (!success_p || it->what != IT_CHARACTER || it->len > 0);
  return success_p;
}


/* Deliver a composition display element.  Unlike the other
   next_element_from_XXX, this function is not registered in the array
   get_next_element[].  It is called from next_element_from_buffer and
   next_element_from_string when necessary.  */

static bool
next_element_from_composition (struct it *it)
{
  it->what = IT_COMPOSITION;
  it->len = it->cmp_it.nbytes;
  if (STRINGP (it->string))
    {
      if (it->c < 0)
	{
	  IT_STRING_CHARPOS (*it) += it->cmp_it.nchars;
	  IT_STRING_BYTEPOS (*it) += it->cmp_it.nbytes;
	  return false;
	}
      it->position = it->current.string_pos;
      it->object = it->string;
      it->c = composition_update_it (&it->cmp_it, IT_STRING_CHARPOS (*it),
				     IT_STRING_BYTEPOS (*it), it->string);
    }
  else
    {
      if (it->c < 0)
	{
	  IT_CHARPOS (*it) += it->cmp_it.nchars;
	  IT_BYTEPOS (*it) += it->cmp_it.nbytes;
	  if (it->bidi_p)
	    {
	      if (it->bidi_it.new_paragraph)
		bidi_paragraph_init (it->paragraph_embedding, &it->bidi_it,
				     false);
	      /* Resync the bidi iterator with IT's new position.
		 FIXME: this doesn't support bidirectional text.  */
	      while (it->bidi_it.charpos < IT_CHARPOS (*it))
		bidi_move_to_visually_next (&it->bidi_it);
	    }
	  return false;
	}
      it->position = it->current.pos;
      it->object = it->w->contents;
      it->c = composition_update_it (&it->cmp_it, IT_CHARPOS (*it),
				     IT_BYTEPOS (*it), Qnil);
    }
  return true;
}



/***********************************************************************
	     Moving an iterator without producing glyphs
 ***********************************************************************/

/* Check if iterator is at a position corresponding to a valid buffer
   position after some move_it_ call.  */

#define IT_POS_VALID_AFTER_MOVE_P(it)			\
  ((it)->method != GET_FROM_STRING || IT_STRING_CHARPOS (*it) == 0)


/* Move iterator IT to a specified buffer or X position within one
   line on the display without producing glyphs.

   OP should be a bit mask including some or all of these bits:
    MOVE_TO_X: Stop upon reaching x-position TO_X.
    MOVE_TO_POS: Stop upon reaching buffer or string position TO_CHARPOS.
   Regardless of OP's value, stop upon reaching the end of the display line.

   TO_X is normally a value 0 <= TO_X <= IT->last_visible_x.
   This means, in particular, that TO_X includes window's horizontal
   scroll amount.

   The return value has several possible values that
   say what condition caused the scan to stop:

   MOVE_POS_MATCH_OR_ZV
     - when TO_POS or ZV was reached.

   MOVE_X_REACHED
     -when TO_X was reached before TO_POS or ZV were reached.

   MOVE_LINE_CONTINUED
     - when we reached the end of the display area and the line must
     be continued.

   MOVE_LINE_TRUNCATED
     - when we reached the end of the display area and the line is
     truncated.

   MOVE_NEWLINE_OR_CR
     - when we stopped at a line end, i.e. a newline or a CR and selective
     display is on.  */

static enum move_it_result
move_it_in_display_line_to (struct it *it,
			    ptrdiff_t to_charpos, int to_x,
			    enum move_operation_enum op)
{
  enum move_it_result result = MOVE_UNDEFINED;
  struct glyph_row *saved_glyph_row;
  struct it wrap_it, atpos_it, atx_it, ppos_it;
  void *wrap_data = NULL, *atpos_data = NULL, *atx_data = NULL;
  void *ppos_data = NULL;
  bool may_wrap = false;
  enum it_method prev_method = it->method;
  ptrdiff_t closest_pos UNINIT;
  ptrdiff_t prev_pos = IT_CHARPOS (*it);
  bool saw_smaller_pos = prev_pos < to_charpos;
  bool line_number_pending = false;

  /* Don't produce glyphs in produce_glyphs.  */
  saved_glyph_row = it->glyph_row;
  it->glyph_row = NULL;

  /* Use wrap_it to save a copy of IT wherever a word wrap could
     occur.  Use atpos_it to save a copy of IT at the desired buffer
     position, if found, so that we can scan ahead and check if the
     word later overshoots the window edge.  Use atx_it similarly, for
     pixel positions.  */
  wrap_it.sp = -1;
  atpos_it.sp = -1;
  atx_it.sp = -1;

  /* Use ppos_it under bidi reordering to save a copy of IT for the
     initial position.  We restore that position in IT when we have
     scanned the entire display line without finding a match for
     TO_CHARPOS and all the character positions are greater than
     TO_CHARPOS.  We then restart the scan from the initial position,
     and stop at CLOSEST_POS, which is a position > TO_CHARPOS that is
     the closest to TO_CHARPOS.  */
  if (it->bidi_p)
    {
      if ((op & MOVE_TO_POS) && IT_CHARPOS (*it) >= to_charpos)
	{
	  SAVE_IT (ppos_it, *it, ppos_data);
	  closest_pos = IT_CHARPOS (*it);
	}
      else
	closest_pos = ZV;
    }

#define BUFFER_POS_REACHED_P()					\
  ((op & MOVE_TO_POS) != 0					\
   && BUFFERP (it->object)					\
   && (IT_CHARPOS (*it) == to_charpos				\
       || ((!it->bidi_p						\
	    || BIDI_AT_BASE_LEVEL (it->bidi_it))		\
	   && IT_CHARPOS (*it) > to_charpos)			\
       || (it->what == IT_COMPOSITION				\
	   && ((IT_CHARPOS (*it) > to_charpos			\
		&& to_charpos >= it->cmp_it.charpos)		\
	       || (IT_CHARPOS (*it) < to_charpos		\
		   && to_charpos <= it->cmp_it.charpos))))	\
   && (it->method == GET_FROM_BUFFER				\
       || (it->method == GET_FROM_DISPLAY_VECTOR		\
	   && it->dpvec + it->current.dpvec_index + 1 >= it->dpend)))

  if (it->hpos == 0)
    {
      /* If line numbers are being displayed, produce a line number.
	 But don't do that if we are to reach first_visible_x, because
	 line numbers are not relevant to stuff that is not visible on
	 display.  */
      if (!((op && MOVE_TO_X) && to_x == it->first_visible_x)
	  && should_produce_line_number (it))
	{
	  if (it->current_x == it->first_visible_x)
	    maybe_produce_line_number (it);
	  else
	    line_number_pending = true;
	}
      /* If there's a line-/wrap-prefix, handle it, if we didn't already.  */
      if (it->area == TEXT_AREA && !it->string_from_prefix_prop_p)
	handle_line_prefix (it);
    }

  if (IT_CHARPOS (*it) < CHARPOS (this_line_min_pos))
    SET_TEXT_POS (this_line_min_pos, IT_CHARPOS (*it), IT_BYTEPOS (*it));

  while (true)
    {
      int x, i, ascent = 0, descent = 0;

/* Utility macro to reset an iterator with x, ascent, and descent.  */
#define IT_RESET_X_ASCENT_DESCENT(IT)			\
  ((IT)->current_x = x, (IT)->max_ascent = ascent,	\
   (IT)->max_descent = descent)

      /* Stop if we move beyond TO_CHARPOS (after an image or a
	 display string or stretch glyph).  */
      if ((op & MOVE_TO_POS) != 0
	  && BUFFERP (it->object)
	  && it->method == GET_FROM_BUFFER
	  && (((!it->bidi_p
		/* When the iterator is at base embedding level, we
		   are guaranteed that characters are delivered for
		   display in strictly increasing order of their
		   buffer positions.  */
		|| BIDI_AT_BASE_LEVEL (it->bidi_it))
	       && IT_CHARPOS (*it) > to_charpos)
	      || (it->bidi_p
		  && (prev_method == GET_FROM_IMAGE
		      || prev_method == GET_FROM_STRETCH
		      || prev_method == GET_FROM_STRING)
		  /* Passed TO_CHARPOS from left to right.  */
		  && ((prev_pos < to_charpos
		       && IT_CHARPOS (*it) >= to_charpos)
		      /* Passed TO_CHARPOS from right to left.  */
		      || (prev_pos > to_charpos
			  && IT_CHARPOS (*it) <= to_charpos)))))
	{
	  if (it->line_wrap != WORD_WRAP || wrap_it.sp < 0)
	    {
	      result = MOVE_POS_MATCH_OR_ZV;
	      break;
	    }
	  else if (it->line_wrap == WORD_WRAP && atpos_it.sp < 0)
	    /* If wrap_it is valid, the current position might be in a
	       word that is wrapped.  So, save the iterator in
	       atpos_it and continue to see if wrapping happens.  */
	    SAVE_IT (atpos_it, *it, atpos_data);
	}

      /* Stop when ZV reached.
         We used to stop here when TO_CHARPOS reached as well, but that is
         too soon if this glyph does not fit on this line.  So we handle it
         explicitly below.  */
      if (!get_next_display_element (it))
	{
	  result = MOVE_POS_MATCH_OR_ZV;
	  break;
	}

      if (it->line_wrap == TRUNCATE)
	{
	  /* If it->pixel_width is zero, the last PRODUCE_GLYPHS call
	     produced something that doesn't consume any screen estate
	     in the text area, so we don't want to exit the loop at
	     TO_CHARPOS, before we produce the glyph for that buffer
	     position.  This happens, e.g., when there's an overlay at
	     TO_CHARPOS that draws a fringe bitmap.  */
	  if (BUFFER_POS_REACHED_P ()
	      && (it->pixel_width > 0
		  || IT_CHARPOS (*it) > to_charpos
		  || it->area != TEXT_AREA))
	    {
	      result = MOVE_POS_MATCH_OR_ZV;
	      break;
	    }
	}
      else
	{
	  if (it->line_wrap == WORD_WRAP && it->area == TEXT_AREA)
	    {
              bool next_may_wrap = may_wrap;
              /* Can we wrap after this character?  */
              if (char_can_wrap_after (it))
		next_may_wrap = true;
              else
                next_may_wrap = false;
	      /* Can we wrap here? */
	      if (may_wrap && char_can_wrap_before (it))
		{
		  /* We have reached a glyph that follows one or more
		     whitespace characters or characters that allow
		     wrapping after them.  If this character allows
		     wrapping before it, save this position as a
		     wrapping point.  */
		  if (atpos_it.sp >= 0)
		    {
		      RESTORE_IT (it, &atpos_it, atpos_data);
		      result = MOVE_POS_MATCH_OR_ZV;
		      goto done;
		    }
		  if (atx_it.sp >= 0)
		    {
		      RESTORE_IT (it, &atx_it, atx_data);
		      result = MOVE_X_REACHED;
		      goto done;
		    }
		  /* Otherwise, we can wrap here.  */
		  SAVE_IT (wrap_it, *it, wrap_data);
		}
              /* Update may_wrap for the next iteration.  */
              may_wrap = next_may_wrap;
	    }
	}

      /* Remember the line height for the current line, in case
	 the next element doesn't fit on the line.  */
      ascent = it->max_ascent;
      descent = it->max_descent;

      /* The call to produce_glyphs will get the metrics of the
	 display element IT is loaded with.  Record the x-position
	 before this display element, in case it doesn't fit on the
	 line.  */
      x = it->current_x;

      PRODUCE_GLYPHS (it);

      if (it->area != TEXT_AREA)
	{
	  prev_method = it->method;
	  if (it->method == GET_FROM_BUFFER)
	    prev_pos = IT_CHARPOS (*it);
	  set_iterator_to_next (it, true);
	  if (IT_CHARPOS (*it) < CHARPOS (this_line_min_pos))
	    SET_TEXT_POS (this_line_min_pos,
			  IT_CHARPOS (*it), IT_BYTEPOS (*it));
	  if (it->bidi_p
	      && (op & MOVE_TO_POS)
	      && IT_CHARPOS (*it) > to_charpos
	      && IT_CHARPOS (*it) < closest_pos)
	    closest_pos = IT_CHARPOS (*it);
	  continue;
	}

      /* The number of glyphs we get back in IT->nglyphs will normally
	 be 1 except when IT->c is (i) a TAB, or (ii) a multi-glyph
	 character on a terminal frame, or (iii) a line end.  For the
	 second case, IT->nglyphs - 1 padding glyphs will be present.
	 (On X frames, there is only one glyph produced for a
	 composite character.)

	 The behavior implemented below means, for continuation lines,
	 that as many spaces of a TAB as fit on the current line are
	 displayed there.  For terminal frames, as many glyphs of a
	 multi-glyph character are displayed in the current line, too.
	 This is what the old redisplay code did, and we keep it that
	 way.  Under X, the whole shape of a complex character must
	 fit on the line or it will be completely displayed in the
	 next line.

	 Note that both for tabs and padding glyphs, all glyphs have
	 the same width.  */
      if (it->nglyphs)
	{
	  /* More than one glyph or glyph doesn't fit on line.  All
	     glyphs have the same width.  */
	  int single_glyph_width = it->pixel_width / it->nglyphs;
	  int new_x;
	  int x_before_this_char = x;
	  int hpos_before_this_char = it->hpos;

	  for (i = 0; i < it->nglyphs; ++i, x = new_x)
	    {
	      new_x = x + single_glyph_width;

	      /* We want to leave anything reaching TO_X to the caller.  */
	      if ((op & MOVE_TO_X) && new_x > to_x)
		{
		  if (BUFFER_POS_REACHED_P ())
		    {
		      if (it->line_wrap != WORD_WRAP || wrap_it.sp < 0)
			goto buffer_pos_reached;
		      if (atpos_it.sp < 0)
			{
			  SAVE_IT (atpos_it, *it, atpos_data);
			  IT_RESET_X_ASCENT_DESCENT (&atpos_it);
			}
		    }
		  else
		    {
		      if (it->line_wrap != WORD_WRAP || wrap_it.sp < 0)
			{
			  it->current_x = x;
			  result = MOVE_X_REACHED;
			  break;
			}
		      if (atx_it.sp < 0)
			{
			  SAVE_IT (atx_it, *it, atx_data);
			  IT_RESET_X_ASCENT_DESCENT (&atx_it);
			}
		    }
		}

	      if (/* Lines are continued.  */
		  it->line_wrap != TRUNCATE
		  && (/* And glyph doesn't fit on the line.  */
		      new_x > it->last_visible_x
		      /* Or it fits exactly and we're on a window
			 system frame.  */
		      || (new_x == it->last_visible_x
			  && FRAME_WINDOW_P (it->f)
			  && ((it->bidi_p && it->bidi_it.paragraph_dir == R2L)
			      ? WINDOW_LEFT_FRINGE_WIDTH (it->w)
			      : WINDOW_RIGHT_FRINGE_WIDTH (it->w)))))
		{
		  bool moved_forward = false;

		  if (/* IT->hpos == 0 means the very first glyph
			 doesn't fit on the line, e.g. a wide image.  */
		      it->hpos == 0
		      || (new_x == it->last_visible_x
			  && FRAME_WINDOW_P (it->f)))
		    {
		      ++it->hpos;
		      it->current_x = new_x;

		      /* The character's last glyph just barely fits
			 in this row.  */
		      if (i == it->nglyphs - 1)
			{
			  /* If this is the destination position,
			     return a position *before* it in this row,
			     now that we know it fits in this row.  */
			  if (BUFFER_POS_REACHED_P ())
			    {
			      bool can_wrap = true;

			      /* If the previous character says we can
				 wrap after it, but the current
				 character says we can't wrap before
				 it, then we can't wrap here.  */
			      if (it->line_wrap == WORD_WRAP
				  && wrap_it.sp >= 0
				  && may_wrap
				  && IT_OVERFLOW_NEWLINE_INTO_FRINGE (it))
				{
				  struct it tem_it;
				  void *tem_data = NULL;

				  SAVE_IT (tem_it, *it, tem_data);
				  set_iterator_to_next (it, true);
				  if (get_next_display_element (it)
				      && !char_can_wrap_before (it))
				    can_wrap = false;
				  RESTORE_IT (it, &tem_it, tem_data);
				}
			      if (it->line_wrap != WORD_WRAP
				  || wrap_it.sp < 0
				  /* If we've just found whitespace
				     where we can wrap, effectively
				     ignore the previous wrap point --
				     it is no longer relevant, but we
				     won't have an opportunity to
				     update it, since we've reached
				     the edge of this screen line.  */
				  || (may_wrap && can_wrap
				      && IT_OVERFLOW_NEWLINE_INTO_FRINGE (it)))
				{
				  it->hpos = hpos_before_this_char;
				  it->current_x = x_before_this_char;
				  result = MOVE_POS_MATCH_OR_ZV;
				  break;
				}
			      if (it->line_wrap == WORD_WRAP
				  && atpos_it.sp < 0)
				{
				  SAVE_IT (atpos_it, *it, atpos_data);
				  atpos_it.current_x = x_before_this_char;
				  atpos_it.hpos = hpos_before_this_char;
				}
			    }

			  prev_method = it->method;
			  if (it->method == GET_FROM_BUFFER)
			    prev_pos = IT_CHARPOS (*it);
			  set_iterator_to_next (it, true);
			  if (IT_CHARPOS (*it) < CHARPOS (this_line_min_pos))
			    SET_TEXT_POS (this_line_min_pos,
					  IT_CHARPOS (*it), IT_BYTEPOS (*it));
			  /* On graphical terminals, newlines may
			     "overflow" into the fringe if
			     overflow-newline-into-fringe is non-nil.
			     On text terminals, and on graphical
			     terminals with no right margin, newlines
			     may overflow into the last glyph on the
			     display line.*/
			  if (!FRAME_WINDOW_P (it->f)
			      || ((it->bidi_p
				   && it->bidi_it.paragraph_dir == R2L)
				  ? WINDOW_LEFT_FRINGE_WIDTH (it->w)
				  : WINDOW_RIGHT_FRINGE_WIDTH (it->w)) == 0
			      || IT_OVERFLOW_NEWLINE_INTO_FRINGE (it))
			    {
			      if (!get_next_display_element (it))
				{
				  result = MOVE_POS_MATCH_OR_ZV;
				  break;
				}
			      moved_forward = true;
			      if (BUFFER_POS_REACHED_P ())
				{
				  if (ITERATOR_AT_END_OF_LINE_P (it))
				    result = MOVE_POS_MATCH_OR_ZV;
				  else
				    result = MOVE_LINE_CONTINUED;
				  break;
				}
			      if (ITERATOR_AT_END_OF_LINE_P (it)
				  && (it->line_wrap != WORD_WRAP
				      || wrap_it.sp < 0
				      || IT_OVERFLOW_NEWLINE_INTO_FRINGE (it)))
				{
				  result = MOVE_NEWLINE_OR_CR;
				  break;
				}
			    }
			}
		    }
		  else
		    IT_RESET_X_ASCENT_DESCENT (it);

		  /* If the screen line ends with whitespace (or
		     wrap-able character), and we are under word-wrap,
		     don't use wrap_it: it is no longer relevant, but
		     we won't have an opportunity to update it, since
		     we are done with this screen line.  */
		  if (may_wrap && IT_OVERFLOW_NEWLINE_INTO_FRINGE (it)
		      /* If the character after the one which set the
			 may_wrap flag says we can't wrap before it,
			 we can't wrap here.  Therefore, wrap_it
			 (previously found wrap-point) _is_ relevant
			 in that case.  */
		      && (!moved_forward || char_can_wrap_before (it)))
		    {
		      /* If we've found TO_X, go back there, as we now
			 know the last word fits on this screen line.  */
		      if ((op & MOVE_TO_X) && new_x == it->last_visible_x
			  && atx_it.sp >= 0)
			{
			  RESTORE_IT (it, &atx_it, atx_data);
			  atpos_it.sp = -1;
			  atx_it.sp = -1;
			  result = MOVE_X_REACHED;
			  break;
			}
		    }
		  else if (wrap_it.sp >= 0)
		    {
		      RESTORE_IT (it, &wrap_it, wrap_data);
		      atpos_it.sp = -1;
		      atx_it.sp = -1;
		    }

		  move_trace ("move_it_in: continued at %td\n",
			      IT_CHARPOS (*it));
		  result = MOVE_LINE_CONTINUED;
		  break;
		}

	      if (BUFFER_POS_REACHED_P ())
		{
		  if (it->line_wrap != WORD_WRAP || wrap_it.sp < 0)
		    goto buffer_pos_reached;
		  if (it->line_wrap == WORD_WRAP && atpos_it.sp < 0)
		    {
		      SAVE_IT (atpos_it, *it, atpos_data);
		      IT_RESET_X_ASCENT_DESCENT (&atpos_it);
		    }
		}

	      if (new_x > it->first_visible_x)
		{
		  /* If we have reached the visible portion of the
		     screen line, produce the line number if needed.  */
		  if (line_number_pending)
		    {
		      line_number_pending = false;
		      it->current_x = it->first_visible_x;
		      maybe_produce_line_number (it);
		      it->current_x += new_x - it->first_visible_x;
		    }
		  /* Glyph is visible.  Increment number of glyphs that
		     would be displayed.  */
		  ++it->hpos;
		}
	    }

	  if (result != MOVE_UNDEFINED)
	    break;
	}
      else if (BUFFER_POS_REACHED_P ())
	{
	buffer_pos_reached:
	  IT_RESET_X_ASCENT_DESCENT (it);
	  result = MOVE_POS_MATCH_OR_ZV;
	  break;
	}
      else if ((op & MOVE_TO_X) && it->current_x >= to_x)
	{
	  /* Stop when TO_X specified and reached.  This check is
	     necessary here because of lines consisting of a line end,
	     only.  The line end will not produce any glyphs and we
	     would never get MOVE_X_REACHED.  */
	  eassert (it->nglyphs == 0);
	  result = MOVE_X_REACHED;
	  break;
	}

      /* Is this a line end?  If yes, we're done.  */
      if (ITERATOR_AT_END_OF_LINE_P (it))
	{
	  /* If we are past TO_CHARPOS, but never saw any character
	     positions smaller than TO_CHARPOS, return
	     MOVE_POS_MATCH_OR_ZV, like the unidirectional display
	     did.  */
	  if (it->bidi_p && (op & MOVE_TO_POS) != 0)
	    {
	      if (!saw_smaller_pos && IT_CHARPOS (*it) > to_charpos)
		{
		  if (closest_pos < ZV)
		    {
		      RESTORE_IT (it, &ppos_it, ppos_data);
		      /* Don't recurse if closest_pos is equal to
			 to_charpos, since we have just tried that.  */
		      if (closest_pos != to_charpos)
			move_it_in_display_line_to (it, closest_pos, -1,
						    MOVE_TO_POS);
		      result = MOVE_POS_MATCH_OR_ZV;
		    }
		  else
		    goto buffer_pos_reached;
		}
	      else if (it->line_wrap == WORD_WRAP && atpos_it.sp >= 0
		       && IT_CHARPOS (*it) > to_charpos)
		goto buffer_pos_reached;
	      else
		result = MOVE_NEWLINE_OR_CR;
	    }
	  else
	    result = MOVE_NEWLINE_OR_CR;
	  /* If lines are truncated, and the line we moved across is
	     completely hscrolled out of view, reset the line metrics
	     to those of the newline we've just processed, so that
	     glyphs not on display don't affect the line's height.  */
	  if (it->line_wrap == TRUNCATE
	      && it->current_x <= it->first_visible_x
	      && result == MOVE_NEWLINE_OR_CR
	      && it->char_to_display == '\n')
	    {
	      it->max_ascent = it->ascent;
	      it->max_descent = it->descent;
	    }
	  /* If we've processed the newline, make sure this flag is
	     reset, as it must only be set when the newline itself is
	     processed.  */
	  if (result == MOVE_NEWLINE_OR_CR)
	    it->constrain_row_ascent_descent_p = false;
	  break;
	}

      prev_method = it->method;
      if (it->method == GET_FROM_BUFFER)
	prev_pos = IT_CHARPOS (*it);

      /* Detect overly-wide wrap-prefixes made of (space ...) display
	 properties.  When such a wrap prefix reaches past the right
	 margin of the window, we need to avoid the call to
	 set_iterator_to_next below, so that it->line_wrap is left at
	 its TRUNCATE value wisely set by handle_line_prefix.
	 Otherwise, set_iterator_to_next will pop the iterator stack,
	 restore it->line_wrap, and we might miss the opportunity to
	 exit the loop and return.  */
      bool overwide_wrap_prefix =
	CONSP (it->object) && EQ (XCAR (it->object), Qspace)
	&& it->sp > 0 && it->method == GET_FROM_STRETCH
	&& it->current_x >= it->last_visible_x
	&& it->continuation_lines_width > 0
	&& it->line_wrap == TRUNCATE && it->stack[0].line_wrap != TRUNCATE;
      /* The current display element has been consumed.  Advance
	 to the next.  */
      if (!overwide_wrap_prefix)
	set_iterator_to_next (it, true);
      if (IT_CHARPOS (*it) < CHARPOS (this_line_min_pos))
	SET_TEXT_POS (this_line_min_pos, IT_CHARPOS (*it), IT_BYTEPOS (*it));
      if (IT_CHARPOS (*it) < to_charpos)
	saw_smaller_pos = true;
      if (it->bidi_p
	  && (op & MOVE_TO_POS)
	  && IT_CHARPOS (*it) >= to_charpos
	  && IT_CHARPOS (*it) < closest_pos)
	closest_pos = IT_CHARPOS (*it);

      /* Stop if lines are truncated and IT's current x-position is
	 past the right edge of the window now.  */
      if (it->line_wrap == TRUNCATE
	  && it->current_x >= it->last_visible_x)
	{
	  if (!FRAME_WINDOW_P (it->f)
	      || ((it->bidi_p && it->bidi_it.paragraph_dir == R2L)
		  ? WINDOW_LEFT_FRINGE_WIDTH (it->w)
		  : WINDOW_RIGHT_FRINGE_WIDTH (it->w)) == 0
	      || IT_OVERFLOW_NEWLINE_INTO_FRINGE (it))
	    {
	      bool at_eob_p = false;

	      if ((at_eob_p = !get_next_display_element (it))
		  || BUFFER_POS_REACHED_P ()
		  /* If we are past TO_CHARPOS, but never saw any
		     character positions smaller than TO_CHARPOS,
		     return MOVE_POS_MATCH_OR_ZV, like the
		     unidirectional display did.  */
		  || (it->bidi_p && (op & MOVE_TO_POS) != 0
		      && !saw_smaller_pos
		      && IT_CHARPOS (*it) > to_charpos))
		{
		  if (it->bidi_p
		      && !BUFFER_POS_REACHED_P ()
		      && !at_eob_p && closest_pos < ZV)
		    {
		      RESTORE_IT (it, &ppos_it, ppos_data);
		      if (closest_pos != to_charpos)
			move_it_in_display_line_to (it, closest_pos, -1,
						    MOVE_TO_POS);
		    }
		  result = MOVE_POS_MATCH_OR_ZV;
		  break;
		}
	      if (ITERATOR_AT_END_OF_LINE_P (it))
		{
		  result = MOVE_NEWLINE_OR_CR;
		  break;
		}
	    }
	  else if (it->bidi_p && (op & MOVE_TO_POS) != 0
		   && !saw_smaller_pos
		   && IT_CHARPOS (*it) > to_charpos)
	    {
	      if (closest_pos < ZV)
		{
		  RESTORE_IT (it, &ppos_it, ppos_data);
		  if (closest_pos != to_charpos)
		    move_it_in_display_line_to (it, closest_pos, -1,
						MOVE_TO_POS);
		}
	      result = MOVE_POS_MATCH_OR_ZV;
	      break;
	    }
	  result = MOVE_LINE_TRUNCATED;
	  break;
	}
#undef IT_RESET_X_ASCENT_DESCENT
    }

#undef BUFFER_POS_REACHED_P

  /* If we scanned beyond TO_POS, restore the saved iterator either to
     the wrap point (if found), or to atpos/atx location.  We decide which
     data to use to restore the saved iterator state by their X coordinates,
     since buffer positions might increase non-monotonically with screen
     coordinates due to bidi reordering.  */
  if (result == MOVE_LINE_CONTINUED
      && it->line_wrap == WORD_WRAP
      && wrap_it.sp >= 0
      && ((atpos_it.sp >= 0 && wrap_it.current_x < atpos_it.current_x)
	  || (atx_it.sp >= 0 && wrap_it.current_x < atx_it.current_x)))
    RESTORE_IT (it, &wrap_it, wrap_data);
  else if (atpos_it.sp >= 0)
    RESTORE_IT (it, &atpos_it, atpos_data);
  else if (atx_it.sp >= 0)
    RESTORE_IT (it, &atx_it, atx_data);

 done:

  if (atpos_data)
    bidi_unshelve_cache (atpos_data, true);
  if (atx_data)
    bidi_unshelve_cache (atx_data, true);
  if (wrap_data)
    bidi_unshelve_cache (wrap_data, true);
  if (ppos_data)
    bidi_unshelve_cache (ppos_data, true);

  /* Restore the iterator settings altered at the beginning of this
     function.  */
  it->glyph_row = saved_glyph_row;
  return result;
}

/* For external use.  */
void
move_it_in_display_line (struct it *it,
			 ptrdiff_t to_charpos, int to_x,
			 enum move_operation_enum op)
{
  if (it->line_wrap == WORD_WRAP
      && (op & MOVE_TO_X))
    {
      struct it save_it;
      void *save_data = NULL;
      int skip;

      SAVE_IT (save_it, *it, save_data);
      skip = move_it_in_display_line_to (it, to_charpos, to_x, op);
      /* When word-wrap is on, TO_X may lie past the end
	 of a wrapped line.  Then it->current is the
	 character on the next line, so backtrack to the
	 space before the wrap point.  */
      if (skip == MOVE_LINE_CONTINUED)
	{
	  int prev_x = max (it->current_x - 1, 0);
	  RESTORE_IT (it, &save_it, save_data);
	  move_it_in_display_line_to
	    (it, -1, prev_x, MOVE_TO_X);
	}
      else
	bidi_unshelve_cache (save_data, true);
    }
  else
    move_it_in_display_line_to (it, to_charpos, to_x, op);
}


/* Move IT forward until it satisfies one or more of the criteria in
   TO_CHARPOS, TO_X, TO_Y, and TO_VPOS.

   OP is a bit-mask that specifies where to stop, and in particular,
   which of those four position arguments makes a difference.  See the
   description of enum move_operation_enum.

   If TO_CHARPOS is in invisible text, e.g. a truncated part of a
   screen line, this function will set IT to the next position that is
   displayed to the right of TO_CHARPOS on the screen.

   Return the maximum pixel length of any line scanned but never more
   than it.last_visible_x.  */

int
move_it_to (struct it *it, ptrdiff_t to_charpos, int to_x, int to_y, int to_vpos, int op)
{
  enum move_it_result skip, skip2 = MOVE_X_REACHED;
  int line_height, line_start_x = 0, reached = 0;
  int max_current_x = 0;
  void *backup_data = NULL;
  ptrdiff_t orig_charpos = -1;
  enum it_method orig_method = NUM_IT_METHODS;

  for (;;)
    {
      orig_charpos = IT_CHARPOS (*it);
      orig_method = it->method;
      if (op & MOVE_TO_VPOS)
	{
	  /* If no TO_CHARPOS and no TO_X specified, stop at the
	     start of the line TO_VPOS.  */
	  if ((op & (MOVE_TO_X | MOVE_TO_POS)) == 0)
	    {
	      if (it->vpos == to_vpos)
		{
		  reached = 1;
		  break;
		}
	      else
		skip = move_it_in_display_line_to (it, -1, -1, 0);
	    }
	  else
	    {
	      /* TO_VPOS >= 0 means stop at TO_X in the line at
		 TO_VPOS, or at TO_POS, whichever comes first.  */
	      if (it->vpos == to_vpos)
		{
		  reached = 2;
		  break;
		}

	      skip = move_it_in_display_line_to (it, to_charpos, to_x, op);

	      if (skip == MOVE_POS_MATCH_OR_ZV || it->vpos == to_vpos)
		{
		  reached = 3;
		  break;
		}
	      else if (skip == MOVE_X_REACHED && it->vpos != to_vpos)
		{
		  /* We have reached TO_X but not in the line we want.  */
		  skip = move_it_in_display_line_to (it, to_charpos,
						     -1, MOVE_TO_POS);
		  if (skip == MOVE_POS_MATCH_OR_ZV)
		    {
		      reached = 4;
		      break;
		    }
		}
	    }
	}
      else if (op & MOVE_TO_Y)
	{
	  struct it it_backup;

	  if (it->line_wrap == WORD_WRAP)
	    SAVE_IT (it_backup, *it, backup_data);

	  /* TO_Y specified means stop at TO_X in the line containing
	     TO_Y---or at TO_CHARPOS if this is reached first.  The
	     problem is that we can't really tell whether the line
	     contains TO_Y before we have completely scanned it, and
	     this may skip past TO_X.  What we do is to first scan to
	     TO_X.

	     If TO_X is not specified, use a TO_X of zero.  The reason
	     is to make the outcome of this function more predictable.
	     If we didn't use TO_X == 0, we would stop at the end of
	     the line which is probably not what a caller would expect
	     to happen.  */
	  skip = move_it_in_display_line_to
	    (it, to_charpos, ((op & MOVE_TO_X) ? to_x : 0),
	     (MOVE_TO_X | (op & MOVE_TO_POS)));

	  /* If TO_CHARPOS is reached or ZV, we don't have to do more.  */
	  if (skip == MOVE_POS_MATCH_OR_ZV)
	    reached = 5;
	  else if (skip == MOVE_X_REACHED)
	    {
	      /* If TO_X was reached, we want to know whether TO_Y is
		 in the line.  We know this is the case if the already
		 scanned glyphs make the line tall enough.  Otherwise,
		 we must check by scanning the rest of the line.  */
	      line_height = it->max_ascent + it->max_descent;
	      if (to_y >= it->current_y
		  && to_y < it->current_y + line_height)
		{
		  reached = 6;
		  break;
		}
	      SAVE_IT (it_backup, *it, backup_data);
	      move_trace ("move_it: from %td\n", IT_CHARPOS (*it));
	      skip2 = move_it_in_display_line_to (it, to_charpos, -1,
						  op & MOVE_TO_POS);
	      move_trace ("move_it: to %td\n", IT_CHARPOS (*it));
	      line_height = it->max_ascent + it->max_descent;
	      move_trace ("move_it: line_height = %d\n", line_height);

	      if (to_y >= it->current_y
		  && to_y < it->current_y + line_height)
		{
		  /* If TO_Y is in this line and TO_X was reached
		     above, we scanned too far.  We have to restore
		     IT's settings to the ones before skipping.  But
		     keep the more accurate values of max_ascent and
		     max_descent we've found while skipping the rest
		     of the line, for the sake of callers, such as
		     pos_visible_p, that need to know the line
		     height.  */
		  int max_ascent = it->max_ascent;
		  int max_descent = it->max_descent;

		  RESTORE_IT (it, &it_backup, backup_data);
		  it->max_ascent = max_ascent;
		  it->max_descent = max_descent;
		  reached = 6;
		}
	      else
		{
		  skip = skip2;
		  if (skip == MOVE_POS_MATCH_OR_ZV)
		    {
		      reached = 7;
		      /* If the last move_it_in_display_line_to call
			 took us away from TO_CHARPOS, back up to the
			 previous position, as it is a better
			 approximation of TO_CHARPOS.  (Note that we
			 could have both positions after TO_CHARPOS or
			 both positions before it, due to bidi
			 reordering.)  */
		      if (to_charpos > 0
			  && IT_CHARPOS (*it) != to_charpos
			  && ((IT_CHARPOS (it_backup) > to_charpos)
			      == (IT_CHARPOS (*it) > to_charpos)))
			{
			  int max_ascent = it->max_ascent;
			  int max_descent = it->max_descent;

			  RESTORE_IT (it, &it_backup, backup_data);
			  it->max_ascent = max_ascent;
			  it->max_descent = max_descent;
			}
		    }
		}
	    }
	  else
	    {
	      /* Check whether TO_Y is in this line.  */
	      line_height = it->max_ascent + it->max_descent;
	      move_trace ("move_it: line_height = %d\n", line_height);

	      if (to_y >= it->current_y
		  && to_y < it->current_y + line_height)
		{
		  if (to_y > it->current_y)
		    max_current_x = max (it->current_x, max_current_x);

		  /* When word-wrap is on, TO_X may lie past the end
		     of a wrapped line.  Then it->current is the
		     character on the next line, so backtrack to the
		     space before the wrap point.  */
		  if (skip == MOVE_LINE_CONTINUED
		      && it->line_wrap == WORD_WRAP)
		    {
		      int prev_x = max (it->current_x - 1, 0);
		      RESTORE_IT (it, &it_backup, backup_data);
		      skip = move_it_in_display_line_to
			(it, -1, prev_x, MOVE_TO_X);
		    }

		  reached = 6;
		}
	    }

	  if (reached)
	    {
	      max_current_x = max (it->current_x, max_current_x);
	      break;
	    }
	}
      else if (BUFFERP (it->object)
	       && (it->method == GET_FROM_BUFFER
		   || it->method == GET_FROM_STRETCH)
	       && IT_CHARPOS (*it) >= to_charpos
	       /* Under bidi iteration, a call to set_iterator_to_next
		  can scan far beyond to_charpos if the initial
		  portion of the next line needs to be reordered.  In
		  that case, give move_it_in_display_line_to another
		  chance below.  */
	       && !(it->bidi_p
		    && it->bidi_it.scan_dir == -1))
	skip = MOVE_POS_MATCH_OR_ZV;
      else
	skip = move_it_in_display_line_to (it, to_charpos, -1, MOVE_TO_POS);

      switch (skip)
	{
	case MOVE_POS_MATCH_OR_ZV:
	  max_current_x = max (it->current_x, max_current_x);
	  reached = 8;
	  goto out;

	case MOVE_NEWLINE_OR_CR:
	  max_current_x = max (it->current_x, max_current_x);
	  if (!IT_OVERFLOW_NEWLINE_INTO_FRINGE (it))
	    it->override_ascent = -1;
	  set_iterator_to_next (it, true);
	  it->continuation_lines_width = 0;
	  break;

	case MOVE_LINE_TRUNCATED:
	  max_current_x = it->last_visible_x;
	  it->continuation_lines_width = 0;
	  reseat_at_next_visible_line_start (it, false);
	  if ((op & MOVE_TO_POS) != 0
	      && (IT_CHARPOS (*it) > to_charpos
		  || (IT_CHARPOS (*it) == to_charpos
		      /* Consider TO_CHARPOS as REACHED if we are at
			 EOB that ends in something other than a newline.  */
		      && to_charpos == ZV
		      && (ZV_BYTE <= 1 || FETCH_BYTE (ZV_BYTE - 1) != '\n')
		      /* But if we have a display or an overlay string
			 at EOB, keep going until we exhaust all the
			 characters of the string(s).  */
		      && (it->sp == 0
			  || (STRINGP (it->string)
			      && (it->current.overlay_string_index < 0
				  || (it->current.overlay_string_index >= 0
				      && it->current.overlay_string_index
				         >= it->n_overlay_strings - 1))
			      && IT_STRING_CHARPOS (*it) >= it->end_charpos)))))
	    {
	      reached = 9;
	      goto out;
	    }
	  break;

	case MOVE_LINE_CONTINUED:
	  max_current_x = it->last_visible_x;
	  /* For continued lines ending in a tab, some of the glyphs
	     associated with the tab are displayed on the current
	     line.  Since it->current_x does not include these glyphs,
	     we use it->last_visible_x instead.  */
	  if (it->c == '\t')
	    {
	      it->continuation_lines_width += it->last_visible_x;
	      /* When moving by vpos, ensure that the iterator really
		 advances to the next line (bug#847, bug#969).  Fixme:
		 do we need to do this in other circumstances?  */
	      if (it->current_x != it->last_visible_x
		  && (op & MOVE_TO_VPOS)
	      	  && !(op & (MOVE_TO_X | MOVE_TO_POS)))
		{
		  line_start_x = it->current_x + it->pixel_width
		    - it->last_visible_x;
		  if (FRAME_WINDOW_P (it->f))
		    {
		      struct face *face = FACE_FROM_ID (it->f, it->face_id);
		      struct font *face_font = face->font;

		      /* When display_line produces a continued line
			 that ends in a TAB, it skips a tab stop that
			 is closer than the font's space character
			 width (see gui_produce_glyphs where it produces
			 the stretch glyph which represents a TAB).
			 We need to reproduce the same logic here.  */
		      eassert (face_font);
		      if (face_font)
			{
			  if (line_start_x < face_font->space_width)
			    line_start_x
			      += it->tab_width * face_font->space_width;
			}
		    }
		  set_iterator_to_next (it, false);
		}
	    }
	  else
	    {
	      /* Make sure we do advance, otherwise we might infloop.
		 This could happen when the first display element is
		 wider than the window, or if we have a wrap-prefix
		 that doesn't leave enough space after it to display
		 even a single character.  We only do this for moving
		 through buffer text, as with display/overlay strings
		 we'd need to also compare it->object's, and this is
		 unlikely to happen in that case anyway.  */
	      if (IT_CHARPOS (*it) == orig_charpos
		  && it->method == orig_method
		  && orig_method == GET_FROM_BUFFER)
		set_iterator_to_next (it, false);
	      it->continuation_lines_width += it->current_x;
	    }
	  break;

	default:
	  emacs_abort ();
	}

      /* Reset/increment for the next run.  */
      it->current_x = line_start_x;
      line_start_x = 0;
      it->hpos = 0;
      it->line_number_produced_p = false;
      it->current_y += it->max_ascent + it->max_descent;
      ++it->vpos;
      last_height = it->max_ascent + it->max_descent;
      it->max_ascent = it->max_descent = 0;
    }

 out:

  /* On text terminals, we may stop at the end of a line in the middle
     of a multi-character glyph.  If the glyph itself is continued,
     i.e. it is actually displayed on the next line, don't treat this
     stopping point as valid; move to the next line instead (unless
     that brings us offscreen).  */
  if (!FRAME_WINDOW_P (it->f)
      && op & MOVE_TO_POS
      && IT_CHARPOS (*it) == to_charpos
      && it->what == IT_CHARACTER
      && it->nglyphs > 1
      && it->line_wrap == WINDOW_WRAP
      && it->current_x == it->last_visible_x - 1
      && it->c != '\n'
      && it->c != '\t'
      && it->w->window_end_valid
      && it->vpos < it->w->window_end_vpos)
    {
      it->continuation_lines_width += it->current_x;
      it->current_x = it->hpos = it->max_ascent = it->max_descent = 0;
      it->current_y += it->max_ascent + it->max_descent;
      ++it->vpos;
      last_height = it->max_ascent + it->max_descent;
    }

  if (backup_data)
    bidi_unshelve_cache (backup_data, true);

  move_trace ("move_it_to: reached %d\n", reached);

  return max_current_x;
}


/* Move iterator IT backward by a specified y-distance DY, DY >= 0.

   If DY > 0, move IT backward that many pixels.
   DY = 0 means move IT backward to the preceding line start or to BEGV.
   This function may move over less or more than DY pixels if
   IT->current_y - DY ends up in the middle of a line; in this case
   IT->current_y will be set to the top of the line either before or
   after the exact pixel coordinate.  */

void
move_it_vertically_backward (struct it *it, int dy)
{
  int nlines, h;
  struct it it2, it3;
  void *it2data = NULL, *it3data = NULL;
  ptrdiff_t start_pos;
  int nchars_per_row
    = (it->last_visible_x - it->first_visible_x) / FRAME_COLUMN_WIDTH (it->f);
  ptrdiff_t pos_limit;

 move_further_back:
  eassert (dy >= 0);

  start_pos = IT_CHARPOS (*it);

  /* Estimate how many newlines we must move back.  */
  nlines = max (1, dy / default_line_pixel_height (it->w));
  if (it->line_wrap == TRUNCATE || nchars_per_row == 0)
    pos_limit = BEGV;
  else
    pos_limit = max (start_pos - nlines * nchars_per_row, BEGV);

  /* Set the iterator's position that many lines back.  But don't go
     back more than NLINES full screen lines -- this wins a day with
     buffers which have very long lines.  */
  while (nlines-- && IT_CHARPOS (*it) > pos_limit)
    back_to_previous_visible_line_start (it);

  /* Reseat the iterator here.  When moving backward, we don't want
     reseat to skip forward over invisible text, set up the iterator
     to deliver from overlay strings at the new position etc.  So,
     use reseat_1 here.  */
  reseat_1 (it, it->current.pos, true);

  /* We are now surely at a line start.  */
  it->current_x = it->hpos = 0;	/* FIXME: this is incorrect when bidi
				   reordering is in effect.  */
  it->continuation_lines_width = 0;

  /* Move forward and see what y-distance we moved.  First move to the
     start of the next line so that we get its height.  We need this
     height to be able to tell whether we reached the specified
     y-distance.  */
  SAVE_IT (it2, *it, it2data);
  it2.max_ascent = it2.max_descent = 0;
  do
    {
      move_it_to (&it2, start_pos, -1, -1, it2.vpos + 1,
		  MOVE_TO_POS | MOVE_TO_VPOS);
    }
  while (!(IT_POS_VALID_AFTER_MOVE_P (&it2)
	   /* If we are in a display string which starts at START_POS,
	      and that display string includes a newline, and we are
	      right after that newline (i.e. at the beginning of a
	      display line), exit the loop, because otherwise we will
	      infloop, since move_it_to will see that it is already at
	      START_POS and will not move.  */
	   || (it2.method == GET_FROM_STRING
	       && IT_CHARPOS (it2) == start_pos
	       && SREF (it2.string, IT_STRING_BYTEPOS (it2) - 1) == '\n')));
  eassert (IT_CHARPOS (*it) >= BEGV);
  SAVE_IT (it3, it2, it3data);

  move_it_to (&it2, start_pos, -1, -1, -1, MOVE_TO_POS);
  eassert (IT_CHARPOS (*it) >= BEGV);
  /* H is the actual vertical distance from the position in *IT
     and the starting position.  */
  h = it2.current_y - it->current_y;
  /* NLINES is the distance in number of lines.  */
  nlines = it2.vpos - it->vpos;

  /* Correct IT's y and vpos position
     so that they are relative to the starting point.  */
  it->vpos -= nlines;
  it->current_y -= h;

  if (dy == 0)
    {
      /* DY == 0 means move to the start of the screen line.  The
	 value of nlines is > 0 if continuation lines were involved,
	 or if the original IT position was at start of a line.  */
      RESTORE_IT (it, it, it2data);
      if (nlines > 0)
	move_it_by_lines (it, nlines);
      /* The above code moves us to some position NLINES down,
	 usually to its first glyph (leftmost in an L2R line), but
	 that's not necessarily the start of the line, under bidi
	 reordering.  We want to get to the character position
	 that is immediately after the newline of the previous
	 line.  */
      if (it->bidi_p
	  && !it->continuation_lines_width
	  && !STRINGP (it->string)
	  && IT_CHARPOS (*it) > BEGV
	  && FETCH_BYTE (IT_BYTEPOS (*it) - 1) != '\n')
	{
	  ptrdiff_t cp = IT_CHARPOS (*it), bp = IT_BYTEPOS (*it);

	  dec_both (&cp, &bp);
	  SET_WITH_NARROWED_BEGV (it, cp,
				  find_newline_no_quit (cp, bp, -1, NULL),
				  get_closer_narrowed_begv (it->w, IT_CHARPOS (*it)));
	  move_it_to (it, cp, -1, -1, -1, MOVE_TO_POS);
	}
      bidi_unshelve_cache (it3data, true);
    }
  else
    {
      /* The y-position we try to reach, relative to *IT.
	 Note that H has been subtracted in front of the if-statement.  */
      int target_y = it->current_y + h - dy;
      int y0 = it3.current_y;
      int y1;
      int line_height;

      RESTORE_IT (&it3, &it3, it3data);
      y1 = line_bottom_y (&it3);
      line_height = y1 - y0;
      RESTORE_IT (it, it, it2data);
      /* If we did not reach target_y, try to move further backward if
	 we can.  If we moved too far backward, try to move forward.  */
      if (target_y < it->current_y
	  /* This is heuristic.  In a window that's 3 lines high, with
	     a line height of 13 pixels each, recentering with point
	     on the bottom line will try to move -39/2 = 19 pixels
	     backward.  Try to avoid moving into the first line.  */
	  && (it->current_y - target_y
	      > min (window_box_height (it->w), line_height * 2 / 3))
	  && IT_CHARPOS (*it) > BEGV)
	{
	  move_trace ("  not far enough -> move_vert %d\n",
		      target_y - it->current_y);
	  dy = it->current_y - target_y;
	  goto move_further_back;
	}
      else if (target_y >= it->current_y + line_height
	       && IT_CHARPOS (*it) < ZV)
	{
	  /* Should move forward by at least one line, maybe more.

	     Note: Calling move_it_by_lines can be expensive on
	     terminal frames, where compute_motion is used (via
	     vmotion) to do the job, when there are very long lines
	     and truncate-lines is nil.  That's the reason for
	     treating terminal frames specially here.  */

	  if (!FRAME_WINDOW_P (it->f))
	    move_it_vertically (it, target_y - it->current_y);
	  else
	    {
	      struct text_pos last_pos;
	      int last_y, last_vpos;
	      do
		{
		  last_pos = it->current.pos;
		  last_y = it->current_y;
		  last_vpos = it->vpos;
		  move_it_by_lines (it, 1);
		}
	      while (target_y > it->current_y && IT_CHARPOS (*it) < ZV);
	      if (it->current_y > target_y)
		{
		  reseat (it, last_pos, true);
		  it->current_y = last_y;
		  it->vpos = last_vpos;
		}
	    }
	}
    }
}


/* Move IT by a specified amount of pixel lines DY.  DY negative means
   move backwards.  DY = 0 means move to start of screen line.  At the
   end, IT will be on the start of a screen line.  */

void
move_it_vertically (struct it *it, int dy)
{
  if (dy <= 0)
    move_it_vertically_backward (it, -dy);
  else
    {
      move_trace ("move_it_v: from %td, %d\n", IT_CHARPOS (*it), dy);
      move_it_to (it, ZV, -1, it->current_y + dy, -1,
		  MOVE_TO_POS | MOVE_TO_Y);
      move_trace ("move_it_v: to %td\n", IT_CHARPOS (*it));

      /* If buffer ends in ZV without a newline, move to the start of
	 the line to satisfy the post-condition.  */
      if (IT_CHARPOS (*it) == ZV
	  && ZV > BEGV
	  && FETCH_BYTE (IT_BYTEPOS (*it) - 1) != '\n')
	move_it_by_lines (it, 0);
    }
}


/* Move iterator IT past the end of the text line it is in.  */

void
move_it_past_eol (struct it *it)
{
  enum move_it_result rc;

  rc = move_it_in_display_line_to (it, Z, 0, MOVE_TO_POS);
  if (rc == MOVE_NEWLINE_OR_CR)
    set_iterator_to_next (it, false);
}


/* Move IT by a specified number DVPOS of screen lines down.  DVPOS
   negative means move up.  DVPOS == 0 means move to the start of the
   screen line.

   Optimization idea: If we would know that IT->f doesn't use
   a face with proportional font, we could be faster for
   truncate-lines nil.  */

void
move_it_by_lines (struct it *it, ptrdiff_t dvpos)
{

  /* The commented-out optimization uses vmotion on terminals.  This
     gives bad results, because elements like it->what, on which
     callers such as pos_visible_p rely, aren't updated.  */
  /* struct position pos;
    if (!FRAME_WINDOW_P (it->f))
    {
      struct text_pos textpos;

      pos = *vmotion (IT_CHARPOS (*it), dvpos, it->w);
      SET_TEXT_POS (textpos, pos.bufpos, pos.bytepos);
      reseat (it, textpos, true);
      it->vpos += pos.vpos;
      it->current_y += pos.vpos;
    }
    else */

  if (dvpos == 0)
    {
      /* DVPOS == 0 means move to the start of the screen line.  */
      move_it_vertically_backward (it, 0);
      /* Let next call to line_bottom_y calculate real line height.  */
      last_height = 0;
    }
  else if (dvpos > 0)
    {
      move_it_to (it, -1, -1, -1, it->vpos + dvpos, MOVE_TO_VPOS);
      if (!IT_POS_VALID_AFTER_MOVE_P (it))
	{
	  /* Only move to the next buffer position if we ended up in a
	     string from display property, not in an overlay string
	     (before-string or after-string).  That is because the
	     latter don't conceal the underlying buffer position, so
	     we can ask to move the iterator to the exact position we
	     are interested in.  Note that, even if we are already at
	     IT_CHARPOS (*it), the call below is not a no-op, as it
	     will detect that we are at the end of the string, pop the
	     iterator, and compute it->current_x and it->hpos
	     correctly.  */
	  move_it_to (it, IT_CHARPOS (*it) + it->string_from_display_prop_p,
		      -1, -1, -1, MOVE_TO_POS);
	}
    }
  else
    {
      struct it it2;
      void *it2data = NULL;
      ptrdiff_t start_charpos, orig_charpos, i;
      int nchars_per_row
	= (it->last_visible_x - it->first_visible_x) / FRAME_COLUMN_WIDTH (it->f);
      bool hit_pos_limit = false;
      ptrdiff_t pos_limit;

      /* Start at the beginning of the screen line containing IT's
	 position.  This may actually move vertically backwards,
         in case of overlays, so adjust dvpos accordingly.  */
      dvpos += it->vpos;
      orig_charpos = IT_CHARPOS (*it);
      move_it_vertically_backward (it, 0);
      dvpos -= it->vpos;

      /* Go back -DVPOS buffer lines, but no farther than -DVPOS full
	 screen lines, and reseat the iterator there.  */
      start_charpos = IT_CHARPOS (*it);
      if (it->line_wrap == TRUNCATE || nchars_per_row == 0)
	pos_limit = BEGV;
      else
	pos_limit = max (start_charpos + dvpos * nchars_per_row, BEGV);

      for (i = -dvpos; i > 0 && IT_CHARPOS (*it) > pos_limit; --i)
	back_to_previous_visible_line_start (it);
      if (i > 0 && IT_CHARPOS (*it) <= pos_limit)
	hit_pos_limit = true;
      reseat (it, it->current.pos, true);

      /* Move further back if we end up in a string or an image.  */
      while (!IT_POS_VALID_AFTER_MOVE_P (it))
	{
	  /* First try to move to start of display line.  */
	  dvpos += it->vpos;
	  move_it_vertically_backward (it, 0);
	  dvpos -= it->vpos;
	  if (IT_POS_VALID_AFTER_MOVE_P (it))
	    break;
	  /* If start of line is still in string or image,
	     move further back.  */
	  back_to_previous_visible_line_start (it);
	  reseat (it, it->current.pos, true);
	  dvpos--;
	}

      it->current_x = it->hpos = 0;

      /* Above call may have moved too far if continuation lines
	 are involved.  Scan forward and see if it did.  */
      SAVE_IT (it2, *it, it2data);
      it2.vpos = it2.current_y = 0;
      move_it_to (&it2, start_charpos, -1, -1, -1, MOVE_TO_POS);
      it->vpos -= it2.vpos;
      it->current_y -= it2.current_y;
      it->current_x = it->hpos = 0;

      /* If we moved too far back, move IT some lines forward.  */
      if (it2.vpos > -dvpos)
	{
	  int delta = it2.vpos + dvpos;

	  RESTORE_IT (&it2, &it2, it2data);
	  SAVE_IT (it2, *it, it2data);
	  move_it_to (it, -1, -1, -1, it->vpos + delta, MOVE_TO_VPOS);
	  /* Move back again if we got too far ahead,
	     or didn't move at all.  */
	  if (it->vpos - it2.vpos > delta || IT_CHARPOS (*it) == orig_charpos)
	    RESTORE_IT (it, &it2, it2data);
	  else
	    bidi_unshelve_cache (it2data, true);
	}
      else if (hit_pos_limit && pos_limit > BEGV
	       && dvpos < 0 && it2.vpos < -dvpos)
	{
	  /* If we hit the limit, but still didn't make it far enough
	     back, that means there's a display string with a newline
	     covering a large chunk of text, and that caused
	     back_to_previous_visible_line_start try to go too far.
	     Punish those who commit such atrocities by going back
	     until we've reached DVPOS, after lifting the limit, which
	     could make it slow for very long lines.  "If it hurts,
	     don't do that!"  */
	  dvpos += it2.vpos;
	  RESTORE_IT (it, it, it2data);
	  for (i = -dvpos; i > 0; --i)
	    {
	      back_to_previous_visible_line_start (it);
	      it->vpos--;
	    }
	  reseat_1 (it, it->current.pos, true);
	}
      else
	RESTORE_IT (it, it, it2data);
    }
}

int
partial_line_height (struct it *it_origin)
{
  /* In a buffer with very long and truncated lines, we ignore the
     possibly-partial height of the last line in the window: it is too
     expensive to compute that (since in most cases that involves
     going all the way to ZV), and the effect of ignoring it is
     relatively minor.  */
  if (XBUFFER (it_origin->w->contents)->long_line_optimizations_p
      && it_origin->line_wrap == TRUNCATE)
    return 0;

  int partial_height;
  void *it_data = NULL;
  struct it it;
  SAVE_IT (it, *it_origin, it_data);
  move_it_to (&it, ZV, -1, it.last_visible_y, -1,
              MOVE_TO_POS | MOVE_TO_Y);
  if (it.what == IT_EOB)
    {
      int vis_height = it.last_visible_y - it.current_y;
      int height = it.ascent + it.descent;
      partial_height = (vis_height < height) ? vis_height : 0;
    }
  else
    {
      int last_line_y = it.current_y;
      move_it_by_lines (&it, 1);
      partial_height = (it.current_y > it.last_visible_y)
        ? it.last_visible_y - last_line_y : 0;
    }
  RESTORE_IT (&it, &it, it_data);
  return partial_height;
}

/* Approximate move_it_in_display_line_to for very long and truncated
   display lines, when moving horizontally.  This is used when the
   buffer's long_line_optimizations_p flag is set.  It ignores various
   complications, like different font sizes, invisible text, display
   and overlay strings, and, to some degree, bidirectional text.  So
   caveat emptor!

   Starting from IT's position, reseat IT after skipping NCHARS
   characters or to the next newline/ZV, whichever comes first.  Return
   what move_it_in_display_line_to would have returned in this case.  */

static enum move_it_result
fast_move_it_horizontally (struct it *it, ptrdiff_t nchars)
{
  ptrdiff_t nl_bytepos;
  ptrdiff_t nl_pos = find_newline_no_quit (IT_CHARPOS (*it), IT_BYTEPOS (*it),
					   1, &nl_bytepos);
  struct text_pos new_pos;
  enum move_it_result move_result;

  if (nl_pos - IT_CHARPOS (*it) > nchars)
    {
      SET_TEXT_POS (new_pos,
		    IT_CHARPOS (*it) + nchars,
		    CHAR_TO_BYTE (IT_CHARPOS (*it) + nchars));
      move_result = MOVE_X_REACHED;
    }
  else
    {
      if (nl_bytepos < ZV_BYTE
	  || (nl_bytepos > BEGV_BYTE
	      && FETCH_BYTE (nl_bytepos - 1) == '\n'))
	{
	  nl_pos--;
	  nl_bytepos--;
	  move_result = MOVE_NEWLINE_OR_CR;
	}
      else
	move_result = MOVE_POS_MATCH_OR_ZV;
      SET_TEXT_POS (new_pos, nl_pos, nl_bytepos);
    }
  reseat (it, new_pos, false);
  return move_result;
}

/* Return true if IT points into the middle of a display vector.  */

bool
in_display_vector_p (struct it *it)
{
  return (it->method == GET_FROM_DISPLAY_VECTOR
	  && it->current.dpvec_index > 0
	  && it->dpvec + it->current.dpvec_index != it->dpend);
}

/* This is like Fwindow_text_pixel_size but assumes that WINDOW's buffer
   is the current buffer.  Fbuffer_text_pixel_size calls it after it has
   set WINDOW's buffer to the buffer specified by its BUFFER_OR_NAME
   argument.  */
static Lisp_Object
window_text_pixel_size (Lisp_Object window, Lisp_Object from, Lisp_Object to,
			Lisp_Object x_limit, Lisp_Object y_limit,
			Lisp_Object mode_lines, Lisp_Object ignore_line_at_end)
{
  struct window *w = decode_live_window (window);
  struct it it;
  ptrdiff_t start, end, bpos;
  struct text_pos startp;
  void *itdata = NULL;
  int c, max_x = 0, max_y = 0, x = 0, y = 0, vertical_offset = 0, doff = 0;

  if (NILP (from))
    {
      start = BEGV;
      bpos = BEGV_BYTE;
    }
  else if (EQ (from, Qt))
    {
      start = BEGV;
      bpos = BEGV_BYTE;
      while (bpos < ZV_BYTE)
	{
	  c = FETCH_BYTE (bpos);
	  if (!(c == ' ' || c == '\t' || c == '\n' || c == '\r'))
	    break;
	  inc_both (&start, &bpos);
	}
      while (bpos > BEGV_BYTE)
	{
	  dec_both (&start, &bpos);
	  c = FETCH_BYTE (bpos);
	  if (!(c == ' ' || c == '\t'))
	    break;
	}
    }
  else if (CONSP (from))
    {
      start = clip_to_bounds (BEGV, fix_position (XCAR (from)), ZV);
      bpos = CHAR_TO_BYTE (start);
      CHECK_FIXNUM (XCDR (from));
      vertical_offset = XFIXNUM (XCDR (from));
    }
  else
    {
      start = clip_to_bounds (BEGV, fix_position (from), ZV);
      bpos = CHAR_TO_BYTE (start);
    }

  SET_TEXT_POS (startp, start, bpos);

  if (NILP (to))
    end = ZV;
  else if (EQ (to, Qt))
    {
      end = ZV;
      bpos = ZV_BYTE;
      while (bpos > BEGV_BYTE)
	{
	  dec_both (&end, &bpos);
	  c = FETCH_BYTE (bpos);
	  if (!(c == ' ' || c == '\t' || c == '\n' || c == '\r'))
            {
	      inc_both (&end, &bpos);
	      break;
            }
	}
      while (bpos < ZV_BYTE)
	{
	  c = fetch_char_advance (&end, &bpos);
	  if (!(c == ' ' || c == '\t'))
	    break;
	}
    }
  else
    end = clip_to_bounds (start, fix_position (to), ZV);

  if (RANGED_FIXNUMP (0, x_limit, INT_MAX))
    max_x = XFIXNUM (x_limit);
  else if (!NILP (x_limit))
    max_x = INT_MAX;

  if (NILP (y_limit))
    max_y = INT_MAX;
  else if (RANGED_FIXNUMP (0, y_limit, INT_MAX))
    max_y = XFIXNUM (y_limit);

  itdata = bidi_shelve_cache ();

  start_display (&it, w, startp);

  int start_y = it.current_y;

  /* It makes no sense to measure dimensions of region of text that
     crosses the point where bidi reordering changes scan direction.
     By using unidirectional movement here we at least support the use
     case of measuring regions of text that have a uniformly R2L
     directionality, and regions that begin and end in text of the
     same directionality.  */
  it.bidi_p = false;

  int start_x;
  if (vertical_offset != 0)
    {
      int last_y;
      it.current_y = 0;

      move_it_by_lines (&it, 0);

      /* `move_it_vertically_backward' is called by move_it_vertically
         to move by a negative value (upwards), but it is not always
         guaranteed to leave the iterator at or above the position
         given by the offset, which this loop ensures.  */
      if (vertical_offset < 0)
	{
	  while (it.current_y > vertical_offset)
	    {
	      last_y = it.current_y;
	      move_it_vertically_backward (&it,
					   (abs (vertical_offset)
					    + it.current_y));

	      if (it.current_y == last_y)
		break;
	    }
	}
      else
	{
	  move_it_vertically (&it, vertical_offset);
	}

      it.current_y = (WINDOW_TAB_LINE_HEIGHT (w)
		      + WINDOW_HEADER_LINE_HEIGHT (w));
      start = clip_to_bounds (BEGV, IT_CHARPOS (it), ZV);
      start_y = it.current_y;
      start_x = it.current_x;
    }
  else
    {
      /* Start at the beginning of the line containing FROM.  Otherwise
	 IT.current_x will be incorrectly set to zero at some arbitrary
	 non-zero X coordinate.  */
      reseat_at_previous_visible_line_start (&it);
      it.current_x = it.hpos = 0;
      if (IT_CHARPOS (it) != start)
	{
	  void *it1data = NULL;
	  struct it it1;

	  SAVE_IT (it1, it, it1data);
	  move_it_to (&it, start, -1, -1, -1, MOVE_TO_POS);
	  /* We could have a display property at START, in which case
	     asking move_it_to to stop at START will overshoot and
	     stop at position after START.  So we try again, stopping
	     before START, and account for the width of the last
	     buffer position manually.  */
	  if (IT_CHARPOS (it) > start && start > BEGV)
	    {
	      ptrdiff_t it1pos = IT_CHARPOS (it1);
	      int it1_x = it1.current_x;

	      RESTORE_IT (&it, &it1, it1data);
	      /* If START - 1 is the beginning of screen line,
		 move_it_to will not move, so we need to use a
		 lower-level move_it_in_display_line subroutine, and
		 tell it to move just 1 pixel, so it stops at the next
		 display element.  */
	      if (start - 1 > it1pos)
		move_it_to (&it, start - 1, -1, -1, -1, MOVE_TO_POS);
	      else
		move_it_in_display_line (&it, start, it1_x + 1,
					 MOVE_TO_POS | MOVE_TO_X);
	      move_it_to (&it, start - 1, -1, -1, -1, MOVE_TO_POS);
	      start_x = it.current_x;
	      /* If we didn't change our buffer position, the pixel
		 width of what's here was not yet accounted for; do it
		 manually.  */
	      if (IT_CHARPOS (it) == start - 1)
		start_x += it.pixel_width;
	    }
	  else
	    {
	      start_x = it.current_x;
	      bidi_unshelve_cache (it1data, true);
	    }
	}
      else
	start_x = it.current_x;
    }

  /* Now move to TO.  */
  int move_op = MOVE_TO_POS | MOVE_TO_Y;
  int to_x = -1;
  it.current_y = start_y;
  /* If FROM is on a newline, pretend that we start at the beginning
     of the next line, because the newline takes no place on display.  */
  if (FETCH_BYTE (start) == '\n')
    it.current_x = 0;
  if (!NILP (x_limit))
    {
      it.last_visible_x = max_x;
      /* Actually, we never want move_it_to stop at to_x.  But to make
	 sure that move_it_in_display_line_to always moves far enough,
	 we set to_x to INT_MAX and specify MOVE_TO_X.  */
      move_op |= MOVE_TO_X;
      to_x = INT_MAX;
    }

  void *it2data = NULL;
  struct it it2;
  SAVE_IT (it2, it, it2data);

  x = move_it_to (&it, end, to_x, max_y, -1, move_op);

  /* We could have a display property at END, in which case asking
     move_it_to to stop at END will overshoot and stop at position
     after END.  So we try again, stopping before END, and account for
     the width of the last buffer position manually.  */
  if (IT_CHARPOS (it) > end)
    {
      end--;
      RESTORE_IT (&it, &it2, it2data);
      x = move_it_to (&it, end, to_x, max_y, -1, move_op);
      /* Add the width of the thing at TO, but only if we didn't
	 overshoot it; if we did, it is already accounted for.  Also,
	 account for the height of the thing at TO.  */
      if (IT_CHARPOS (it) == end)
	{
	  x += it.pixel_width;

	  /* DTRT if ignore_line_at_end is t.  */
	  if (!NILP (ignore_line_at_end))
	    doff = (max (it.max_ascent, it.ascent)
		    + max (it.max_descent, it.descent));
	  else
	    {
	      it.max_ascent = max (it.max_ascent, it.ascent);
	      it.max_descent = max (it.max_descent, it.descent);
	    }
	}
    }
  else
    bidi_unshelve_cache (it2data, true);

  if (!NILP (x_limit))
    {
      /* Don't return more than X-LIMIT.  */
      if (x > max_x)
        x = max_x;
    }

  /* If text spans more than one screen line, we don't need to adjust
     the x-span for start_x, since the second and subsequent lines
     will begin at zero X coordinate.  */
  if (it.current_y > start_y)
    start_x = 0;

  /* Subtract height of header-line and tab-line which was counted
     automatically by start_display.  */
  if (!NILP (ignore_line_at_end))
    y = (it.current_y + doff
	 - WINDOW_TAB_LINE_HEIGHT (w)
	 - WINDOW_HEADER_LINE_HEIGHT (w));
  else
    y = (it.current_y + it.max_ascent + it.max_descent + doff
	 - WINDOW_TAB_LINE_HEIGHT (w) - WINDOW_HEADER_LINE_HEIGHT (w));

  /* Don't return more than Y-LIMIT.  */
  if (y > max_y)
    y = max_y;

  if ((EQ (mode_lines, Qtab_line) || EQ (mode_lines, Qt))
      && window_wants_tab_line (w))
    /* Add height of tab-line as requested.  */
    {
      Lisp_Object window_tab_line_format
	= window_parameter (w, Qtab_line_format);

      y = y + display_mode_line (w, TAB_LINE_FACE_ID,
				 NILP (window_tab_line_format)
				 ? BVAR (current_buffer, tab_line_format)
				 : window_tab_line_format);
    }

  if ((EQ (mode_lines, Qheader_line) || EQ (mode_lines, Qt))
      && window_wants_header_line (w))
    {
      Lisp_Object window_header_line_format
	= window_parameter (w, Qheader_line_format);

      y = y + display_mode_line (w, HEADER_LINE_FACE_ID,
				 NILP (window_header_line_format)
				 ? BVAR (current_buffer, header_line_format)
				 : window_header_line_format);
    }

  if ((EQ (mode_lines, Qmode_line) || EQ (mode_lines, Qt))
      && window_wants_mode_line (w))
    {
      Lisp_Object window_mode_line_format
	= window_parameter (w, Qmode_line_format);

      y = y + display_mode_line (w, CURRENT_MODE_LINE_ACTIVE_FACE_ID (w),
				 NILP (window_mode_line_format)
				 ? BVAR (current_buffer, mode_line_format)
				 : window_mode_line_format);
    }

  bidi_unshelve_cache (itdata, false);

  return (!vertical_offset
	  ? Fcons (make_fixnum (x - start_x), make_fixnum (y))
	  : list3i (x - start_x, y, start));
}

DEFUN ("window-text-pixel-size", Fwindow_text_pixel_size, Swindow_text_pixel_size, 0, 7, 0,
       doc: /* Return the size of the text of WINDOW's buffer in pixels.
WINDOW must be a live window and defaults to the selected one.  The
return value is a cons of the maximum pixel-width of any text line and
the pixel-height of all the text lines in the accessible portion of
buffer text.

If FROM is a cons cell, the return value includes, in addition to the
dimensions, also a third element that provides the buffer position
from which measuring of the text dimensions was actually started.

This function exists to allow Lisp programs to adjust the dimensions
of WINDOW to the buffer text it needs to display.

The optional argument FROM, if non-nil, specifies the first text
position to consider, and defaults to the minimum accessible position
of the buffer.  If FROM is a cons, its car specifies a buffer
position, and its cdr specifies the vertical offset in pixels from
that position to the first screen line to be measured.  If FROM is t,
it stands for the minimum accessible position that starts a non-empty
line.  TO, if non-nil, specifies the last text position and defaults
to the maximum accessible position of the buffer.  If TO is t, it
stands for the maximum accessible position that ends a non-empty line.

The optional argument X-LIMIT, if non-nil, specifies the maximum X
coordinate beyond which the text should be ignored.  It is therefore
also the maximum width that the function can return.  X-LIMIT nil or
omitted means to use the pixel-width of WINDOW's body.  This default
means text of truncated lines wider than the window will be ignored;
specify a non-nil value for X-LIMIT if lines are truncated and you need
to account for the truncated text.

Use nil for X-LIMIT if you want to know how high WINDOW should become in
order to fit all of its buffer's text with the width of WINDOW
unaltered.  Use the maximum width WINDOW may assume if you intend to
change WINDOW's width.  Use t for the maximum possible value.  Since
calculating the width of long lines can take some time, it's always a
good idea to make this argument as small as possible; in particular, if
the buffer contains long lines that shall be truncated anyway.

The optional argument Y-LIMIT, if non-nil, specifies the maximum Y
coordinate beyond which the text is to be ignored; it is therefore
also the maximum height that the function can return (excluding the
height of the mode- or header-line, if any).  Y-LIMIT nil or omitted
means consider all of the accessible portion of buffer text up to the
position specified by TO.  Since calculating the text height of a
large buffer can take some time, it makes sense to specify this
argument if the size of the buffer is large or unknown.

Optional argument MODE-LINES nil or omitted means do not include the
height of the mode-, tab- or header-line of WINDOW in the return value.
If it is the symbol `mode-line', `tab-line' or `header-line', include
only the height of that line, if present, in the return value.  If t,
include the height of any of these, if present, in the return value.

IGNORE-LINE-AT-END, if non-nil, means to not add the height of the
screen line that includes TO to the returned height of the text.  */)
  (Lisp_Object window, Lisp_Object from, Lisp_Object to, Lisp_Object x_limit,
   Lisp_Object y_limit, Lisp_Object mode_lines, Lisp_Object ignore_line_at_end)
{
  struct window *w = decode_live_window (window);
  struct buffer *b = XBUFFER (w->contents);
  struct buffer *old_b = NULL;
  Lisp_Object value;

  if (b != current_buffer)
    {
      old_b = current_buffer;
      set_buffer_internal_1 (b);
    }

  value = window_text_pixel_size (window, from, to, x_limit, y_limit, mode_lines,
				  ignore_line_at_end);

  if (old_b)
    set_buffer_internal_1 (old_b);

  return value;
}

DEFUN ("buffer-text-pixel-size", Fbuffer_text_pixel_size, Sbuffer_text_pixel_size, 0, 4, 0,
       doc: /* Return size of whole text of BUFFER-OR-NAME in WINDOW.
BUFFER-OR-NAME must specify a live buffer or the name of a live buffer
and defaults to the current buffer.  WINDOW must be a live window and
defaults to the selected one.  The return value is a cons of the maximum
pixel-width of any text line and the pixel-height of all the text lines
of the buffer specified by BUFFER-OR-NAME.

The optional arguments X-LIMIT and Y-LIMIT have the same meaning as with
`window-text-pixel-size'.

Do not use this function if the buffer specified by BUFFER-OR-NAME is
already displayed in WINDOW.  `window-text-pixel-size' is cheaper in
that case because it does not have to temporarily show that buffer in
WINDOW.  */)
  (Lisp_Object buffer_or_name, Lisp_Object window, Lisp_Object x_limit,
   Lisp_Object y_limit)
{
  struct window *w = decode_live_window (window);
  struct buffer *b = (NILP (buffer_or_name)
		      ? current_buffer
		      : XBUFFER (Fget_buffer (buffer_or_name)));
  Lisp_Object buffer, value;
  specpdl_ref count = SPECPDL_INDEX ();

  XSETBUFFER (buffer, b);

  /* The unwind form of with_echo_area_buffer is what we need here to
     make WINDOW temporarily show our buffer.  */
  /* FIXME: Can we move this into the `if (!EQ (buffer, w->contents))`?  */
  record_unwind_protect (unwind_with_echo_area_buffer,
			 with_echo_area_buffer_unwind_data (w));

  set_buffer_internal_1 (b);

  if (!EQ (buffer, w->contents))
    {
      wset_buffer (w, buffer);
      set_marker_both (w->pointm, buffer, BEG, BEG_BYTE);
      set_marker_both (w->old_pointm, buffer, BEG, BEG_BYTE);
    }

  value = window_text_pixel_size (window, Qnil, Qnil, x_limit, y_limit, Qnil,
				  Qnil);

  unbind_to (count, Qnil);

  return value;
}


DEFUN ("display--line-is-continued-p", Fdisplay__line_is_continued_p,
       Sdisplay__line_is_continued_p, 0, 0, 0,
       doc: /* Return non-nil if the current screen line is continued on display.  */)
  (void)
{
  struct buffer *oldb = current_buffer;
  struct window *w = XWINDOW (selected_window);
  enum move_it_result rc = MOVE_POS_MATCH_OR_ZV;

  set_buffer_internal_1 (XBUFFER (w->contents));

  if (PT < ZV)
    {
      struct text_pos startpos;
      struct it it;
      void *itdata;
      /* Use a marker, since vertical-motion enters redisplay, which can
	 trigger fontifications, which in turn could modify buffer text.  */
      Lisp_Object opoint = Fpoint_marker ();

      /* Make sure to start from the beginning of the current screen
	 line, so that move_it_in_display_line_to counts pixels correctly.  */
      Fvertical_motion (make_fixnum (0), selected_window, Qnil);
      SET_TEXT_POS (startpos, PT, PT_BYTE);
      itdata = bidi_shelve_cache ();
      start_display (&it, w, startpos);
      /* If lines are truncated, no line is continued.  */
      if (it.line_wrap != TRUNCATE)
	{
	  it.glyph_row = NULL;
	  rc = move_it_in_display_line_to (&it, ZV, -1, MOVE_TO_POS);
	}
      SET_PT_BOTH (marker_position (opoint), marker_byte_position (opoint));
      bidi_unshelve_cache (itdata, false);
    }
  set_buffer_internal_1 (oldb);

  return rc == MOVE_LINE_CONTINUED ? Qt : Qnil;
}


/***********************************************************************
			       Messages
 ***********************************************************************/

/* Return the number of arguments the format string FORMAT needs.  */

static ptrdiff_t
format_nargs (char const *format)
{
  ptrdiff_t nargs = 0;
  for (char const *p = format; (p = strchr (p, '%')); p++)
    if (p[1] == '%')
      p++;
    else
      nargs++;
  return nargs;
}

/* Add a message with format string FORMAT and formatted arguments
   to *Messages*.  */

void
add_to_log (const char *format, ...)
{
  va_list ap;
  va_start (ap, format);
  vadd_to_log (format, ap);
  va_end (ap);
}

void
vadd_to_log (char const *format, va_list ap)
{
  ptrdiff_t form_nargs = format_nargs (format);
  ptrdiff_t nargs = 1 + form_nargs;
  Lisp_Object args[10];
  eassert (nargs <= ARRAYELTS (args));
  AUTO_STRING (args0, format);
  args[0] = args0;
  for (ptrdiff_t i = 1; i <= nargs; i++)
    args[i] = va_arg (ap, Lisp_Object);
  Lisp_Object msg = Qnil;
  msg = Fformat_message (nargs, args);

  ptrdiff_t len = SBYTES (msg) + 1;
  USE_SAFE_ALLOCA;
  char *buffer = SAFE_ALLOCA (len);
  memcpy (buffer, SDATA (msg), len);

  message_dolog (buffer, len - 1, true, STRING_MULTIBYTE (msg));
  SAFE_FREE ();
}


/* Output a newline in the *Messages* buffer if "needs" one.  */

void
message_log_maybe_newline (void)
{
  if (message_log_need_newline)
    message_dolog ("", 0, true, false);
}


/* Add a string M of length NBYTES to the message log, optionally
   terminated with a newline when NLFLAG is true.  MULTIBYTE, if
   true, means interpret the contents of M as multibyte.  This
   function calls low-level routines in order to bypass text property
   hooks, etc. which might not be safe to run.

   This may GC (insert may run before/after change hooks),
   so the buffer M must NOT point to a Lisp string.  */

void
message_dolog (const char *m, ptrdiff_t nbytes, bool nlflag, bool multibyte)
{
  const unsigned char *msg = (const unsigned char *) m;

  if (!NILP (Vmemory_full))
    return;

  if (!NILP (Vmessage_log_max))
    {
      struct buffer *oldbuf;
      Lisp_Object oldpoint, oldbegv, oldzv;
      int old_windows_or_buffers_changed = windows_or_buffers_changed;
      ptrdiff_t point_at_end = 0;
      ptrdiff_t zv_at_end = 0;
      Lisp_Object old_deactivate_mark;

      old_deactivate_mark = Vdeactivate_mark;
      oldbuf = current_buffer;

      /* Sanity check, in case the variable has been set to something
	 invalid.  */
      if (! STRINGP (Vmessages_buffer_name))
	Vmessages_buffer_name = build_string ("*Messages*");
      /* Ensure the Messages buffer exists, and switch to it.
         If we created it, set the major-mode.  */
      bool newbuffer = NILP (Fget_buffer (Vmessages_buffer_name));
      Fset_buffer (Fget_buffer_create (Vmessages_buffer_name, Qnil));
      if (newbuffer
	  && !NILP (Ffboundp (intern ("messages-buffer-mode"))))
	call0 (intern ("messages-buffer-mode"));

      bset_undo_list (current_buffer, Qt);
      bset_cache_long_scans (current_buffer, Qnil);

      oldpoint = message_dolog_marker1;
      set_marker_restricted_both (oldpoint, Qnil, PT, PT_BYTE);
      oldbegv = message_dolog_marker2;
      set_marker_restricted_both (oldbegv, Qnil, BEGV, BEGV_BYTE);
      oldzv = message_dolog_marker3;
      set_marker_restricted_both (oldzv, Qnil, ZV, ZV_BYTE);

      if (PT == Z)
	point_at_end = 1;
      if (ZV == Z)
	zv_at_end = 1;

      BEGV = BEG;
      BEGV_BYTE = BEG_BYTE;
      ZV = Z;
      ZV_BYTE = Z_BYTE;
      TEMP_SET_PT_BOTH (Z, Z_BYTE);

      /* Insert the string--maybe converting multibyte to single byte
	 or vice versa, so that all the text fits the buffer.  */
      if (multibyte
	  && NILP (BVAR (current_buffer, enable_multibyte_characters)))
	{
	  /* Convert a multibyte string to single-byte
	     for the *Message* buffer.  */
	  for (ptrdiff_t i = 0; i < nbytes; )
	    {
	      int char_bytes, c = check_char_and_length (msg + i, &char_bytes);
	      char work = CHAR_TO_BYTE8 (c);
	      insert_1_both (&work, 1, 1, true, false, false);
	      i += char_bytes;
	    }
	}
      else if (! multibyte
	       && ! NILP (BVAR (current_buffer, enable_multibyte_characters)))
	{
	  /* Convert a single-byte string to multibyte
	     for the *Message* buffer.  */
	  for (ptrdiff_t i = 0; i < nbytes; i++)
	    {
	      int c = make_char_multibyte (msg[i]);
	      unsigned char str[MAX_MULTIBYTE_LENGTH];
	      int char_bytes = CHAR_STRING (c, str);
	      insert_1_both ((char *) str, 1, char_bytes, true, false, false);
	    }
	}
      else if (nbytes)
	insert_1_both (m, chars_in_text (msg, nbytes), nbytes,
		       true, false, false);

      if (nlflag)
	{
	  ptrdiff_t this_bol, this_bol_byte, prev_bol, prev_bol_byte;
	  intmax_t dups;

          /* Since we call del_range_both passing false for PREPARE,
             we aren't prepared to run modification hooks (we could
             end up calling modification hooks from another buffer and
             only with AFTER=t, Bug#21824).  */
          specpdl_ref count = SPECPDL_INDEX ();
          specbind (Qinhibit_modification_hooks, Qt);

	  insert_1_both ("\n", 1, 1, true, false, false);

	  scan_newline (Z, Z_BYTE, BEG, BEG_BYTE, -2, false);
	  this_bol = PT;
	  this_bol_byte = PT_BYTE;

	  /* See if this line duplicates the previous one.
	     If so, combine duplicates.  */
	  if (this_bol > BEG)
	    {
	      scan_newline (PT, PT_BYTE, BEG, BEG_BYTE, -2, false);
	      prev_bol = PT;
	      prev_bol_byte = PT_BYTE;

	      dups = message_log_check_duplicate (prev_bol_byte,
                                                  this_bol_byte);
	      if (dups)
		{
		  del_range_both (prev_bol, prev_bol_byte,
				  this_bol, this_bol_byte, false);
		  if (dups > 1)
		    {
		      char dupstr[sizeof " [ times]"
				  + INT_STRLEN_BOUND (dups)];

		      /* If you change this format, don't forget to also
			 change message_log_check_duplicate.  */
		      int duplen = sprintf (dupstr, " [%"PRIdMAX" times]",
					    dups);
		      TEMP_SET_PT_BOTH (Z - 1, Z_BYTE - 1);
		      insert_1_both (dupstr, duplen, duplen,
				     true, false, true);
		    }
		}
	    }

	  /* If we have more than the desired maximum number of lines
	     in the *Messages* buffer now, delete the oldest ones.
	     This is safe because we don't have undo in this buffer.  */

	  if (FIXNATP (Vmessage_log_max))
	    {
	      scan_newline (Z, Z_BYTE, BEG, BEG_BYTE,
			    -XFIXNAT (Vmessage_log_max) - 1, false);
	      del_range_both (BEG, BEG_BYTE, PT, PT_BYTE, false);
	    }

          unbind_to (count, Qnil);
	}
      BEGV = marker_position (oldbegv);
      BEGV_BYTE = marker_byte_position (oldbegv);

      if (zv_at_end)
	{
	  ZV = Z;
	  ZV_BYTE = Z_BYTE;
	}
      else
	{
	  ZV = marker_position (oldzv);
	  ZV_BYTE = marker_byte_position (oldzv);
	}

      if (point_at_end)
	TEMP_SET_PT_BOTH (Z, Z_BYTE);
      else
	/* We can't do Fgoto_char (oldpoint) because it will run some
           Lisp code.  */
	TEMP_SET_PT_BOTH (marker_position (oldpoint),
			  marker_byte_position (oldpoint));

      unchain_marker (XMARKER (oldpoint));
      unchain_marker (XMARKER (oldbegv));
      unchain_marker (XMARKER (oldzv));

      /* We called insert_1_both above with its 5th argument (PREPARE)
	 false, which prevents insert_1_both from calling
	 prepare_to_modify_buffer, which in turns prevents us from
	 incrementing windows_or_buffers_changed even if *Messages* is
	 shown in some window.  So we must manually set
	 windows_or_buffers_changed here to make up for that.  */
      windows_or_buffers_changed = old_windows_or_buffers_changed;
      bset_redisplay (current_buffer);

      set_buffer_internal (oldbuf);

      message_log_need_newline = !nlflag;
      Vdeactivate_mark = old_deactivate_mark;
    }
}


/* We are at the end of the buffer after just having inserted a newline.
   (Note: We depend on the fact we won't be crossing the gap.)
   Check to see if the most recent message looks a lot like the previous one.
   Return 0 if different, 1 if the new one should just replace it, or a
   value N > 1 if we should also append " [N times]".  */

static intmax_t
message_log_check_duplicate (ptrdiff_t prev_bol_byte, ptrdiff_t this_bol_byte)
{
  ptrdiff_t i;
  ptrdiff_t len = Z_BYTE - 1 - this_bol_byte;
  bool seen_dots = false;
  unsigned char *p1 = BUF_BYTE_ADDRESS (current_buffer, prev_bol_byte);
  unsigned char *p2 = BUF_BYTE_ADDRESS (current_buffer, this_bol_byte);

  for (i = 0; i < len; i++)
    {
      if (i >= 3 && p1[i - 3] == '.' && p1[i - 2] == '.' && p1[i - 1] == '.')
	seen_dots = true;
      if (p1[i] != p2[i])
	return seen_dots;
    }
  p1 += len;
  if (*p1 == '\n')
    return 2;
  if (*p1++ == ' ' && *p1++ == '[')
    {
      char *pend;
      intmax_t n = strtoimax ((char *) p1, &pend, 10);
      if (0 < n && n < INTMAX_MAX && strncmp (pend, " times]\n", 8) == 0)
	return n + 1;
    }
  return 0;
}


/* Display an echo area message M with a specified length of NBYTES
   bytes.  The string may include null characters.  If M is not a
   string, clear out any existing message, and let the mini-buffer
   text show through.

   This function cancels echoing.  */

void
message3 (Lisp_Object m)
{
  clear_message (true, true);
  cancel_echoing ();

  /* First flush out any partial line written with print.  */
  message_log_maybe_newline ();
  if (STRINGP (m))
    {
      ptrdiff_t nbytes = SBYTES (m);
      bool multibyte = STRING_MULTIBYTE (m);
      char *buffer;
      USE_SAFE_ALLOCA;
      SAFE_ALLOCA_STRING (buffer, m);
      message_dolog (buffer, nbytes, true, multibyte);
      SAFE_FREE ();
    }
  if (! inhibit_message)
    message3_nolog (m);
}

/* Log the message M to stderr.  Log an empty line if M is not a string.  */

static void
message_to_stderr (Lisp_Object m)
{
  if (noninteractive_need_newline)
    {
      noninteractive_need_newline = false;
      errputc ('\n');
    }
  if (STRINGP (m))
    {
      Lisp_Object coding_system = Vlocale_coding_system;
      Lisp_Object s;

      if (!NILP (Vcoding_system_for_write))
	coding_system = Vcoding_system_for_write;
      if (!NILP (coding_system))
	s = code_convert_string_norecord (m, coding_system, true);
      else
	s = m;

      errwrite (SDATA (s), SBYTES (s));
    }
  if (STRINGP (m) || !cursor_in_echo_area)
    errputc ('\n');
}

/* The non-logging version of message3.
   This does not cancel echoing, because it is used for echoing.
   Perhaps we need to make a separate function for echoing
   and make this cancel echoing.  */

void
message3_nolog (Lisp_Object m)
{
  struct frame *sf = SELECTED_FRAME ();

  if (FRAME_INITIAL_P (sf))
    message_to_stderr (m);
  /* Error messages get reported properly by cmd_error, so this must be just an
     informative message; if the frame hasn't really been initialized yet, just
     toss it.  */
  else if (INTERACTIVE && sf->glyphs_initialized_p)
    {
      /* Get the frame containing the mini-buffer
	 that the selected frame is using.  */
      Lisp_Object mini_window = FRAME_MINIBUF_WINDOW (sf);
      Lisp_Object frame = XWINDOW (mini_window)->frame;
      struct frame *f = XFRAME (frame);

      if (FRAME_VISIBLE_P (sf) && !FRAME_VISIBLE_P (f))
	Fmake_frame_visible (frame);

      if (STRINGP (m) && SCHARS (m) > 0)
	{
	  set_message (m);
	  if (minibuffer_auto_raise)
	    Fraise_frame (frame);
	  /* Assume we are not echoing.
	     (If we are, echo_now will override this.)  */
	  echo_message_buffer = Qnil;
	}
      else
	clear_message (true, true);

      do_pending_window_change (false);
      echo_area_display (true);
      do_pending_window_change (false);
      if (FRAME_TERMINAL (f)->frame_up_to_date_hook)
	(*FRAME_TERMINAL (f)->frame_up_to_date_hook) (f);
    }
}


/* Display a null-terminated echo area message M.  If M is 0, clear
   out any existing message, and let the mini-buffer text show through.

   The buffer M must continue to exist until after the echo area gets
   cleared or some other message gets displayed there.  Do not pass
   text that is stored in a Lisp string.  Do not pass text in a buffer
   that was alloca'd.  */

void
message1 (const char *m)
{
  message3 (m ? build_unibyte_string (m) : Qnil);
}


/* The non-logging counterpart of message1.  */

void
message1_nolog (const char *m)
{
  message3_nolog (m ? build_unibyte_string (m) : Qnil);
}

/* Display a message M which contains a single %s
   which gets replaced with STRING.  */

void
message_with_string (const char *m, Lisp_Object string, bool log)
{
  CHECK_STRING (string);

  bool need_message;
  if (noninteractive)
    need_message = !!m;
  else if (!INTERACTIVE)
    need_message = false;
  else
    {
      /* The frame whose minibuffer we're going to display the message on.
	 It may be larger than the selected frame, so we need
	 to use its buffer, not the selected frame's buffer.  */
      Lisp_Object mini_window;
      struct frame *f, *sf = SELECTED_FRAME ();

      /* Get the frame containing the minibuffer
	 that the selected frame is using.  */
      mini_window = FRAME_MINIBUF_WINDOW (sf);
      f = XFRAME (WINDOW_FRAME (XWINDOW (mini_window)));

      /* Error messages get reported properly by cmd_error, so this must be
	 just an informative message; if the frame hasn't really been
	 initialized yet, just toss it.  */
      need_message = f->glyphs_initialized_p;
    }

  if (need_message)
    {
      AUTO_STRING (fmt, m);
      Lisp_Object msg = CALLN (Fformat_message, fmt, string);

      if (noninteractive)
	message_to_stderr (msg);
      else
	{
	  if (log)
	    message3 (msg);
	  else
	    message3_nolog (msg);

	  /* Print should start at the beginning of the message
	     buffer next time.  */
	  message_buf_print = false;
	}
    }
}


/* Dump an informative message to the minibuf.  If M is 0, clear out
   any existing message, and let the mini-buffer text show through.

   The message must be safe ASCII (because when Emacs is
   non-interactive the message is sent straight to stderr without
   encoding first) and the format must not contain ` or ' (because
   this function does not account for `text-quoting-style').  If your
   message and format do not fit into this category, convert your
   arguments to Lisp objects and use Fmessage instead.  */

static void ATTRIBUTE_FORMAT_PRINTF (1, 0)
vmessage (const char *m, va_list ap)
{
  if (noninteractive)
    {
      if (m)
	{
	  if (noninteractive_need_newline)
	    putc ('\n', stderr);
	  noninteractive_need_newline = false;
	  vfprintf (stderr, m, ap);
	  if (!cursor_in_echo_area)
	    putc ('\n', stderr);
	  fflush (stderr);
	}
    }
  else if (INTERACTIVE)
    {
      /* The frame whose mini-buffer we're going to display the message
	 on.  It may be larger than the selected frame, so we need to
	 use its buffer, not the selected frame's buffer.  */
      Lisp_Object mini_window;
      struct frame *f, *sf = SELECTED_FRAME ();

      /* Get the frame containing the mini-buffer
	 that the selected frame is using.  */
      mini_window = FRAME_MINIBUF_WINDOW (sf);
      f = XFRAME (WINDOW_FRAME (XWINDOW (mini_window)));

      /* Error messages get reported properly by cmd_error, so this must be
	 just an informative message; if the frame hasn't really been
	 initialized yet, just toss it.  */
      if (f->glyphs_initialized_p)
	{
	  if (m)
	    {
	      ptrdiff_t len;
	      ptrdiff_t maxsize = FRAME_MESSAGE_BUF_SIZE (f);
	      USE_SAFE_ALLOCA;
	      char *message_buf = SAFE_ALLOCA (maxsize + 1);

	      len = doprnt (message_buf, maxsize, m, 0, ap);

	      message3 (make_string (message_buf, len));
	      SAFE_FREE ();
	    }
	  else
	    message1 (0);

	  /* Print should start at the beginning of the message
	     buffer next time.  */
	  message_buf_print = false;
	}
    }
}

/* See vmessage for restrictions on the text of the message.  */
void
message (const char *m, ...)
{
  va_list ap;
  va_start (ap, m);
  vmessage (m, ap);
  va_end (ap);
}


/* Display the current message in the current mini-buffer.  This is
   only called from error handlers in process.c, and is not time
   critical.  */

void
update_echo_area (void)
{
  if (!NILP (echo_area_buffer[0]))
    {
      Lisp_Object string;
      string = Fcurrent_message ();
      message3 (string);
    }
}


/* Make sure echo area buffers in `echo_buffers' are live.
   If they aren't, make new ones.  */

static void
ensure_echo_area_buffers (void)
{
  for (int i = 0; i < 2; i++)
    if (!BUFFERP (echo_buffer[i])
	|| !BUFFER_LIVE_P (XBUFFER (echo_buffer[i])))
      {
	Lisp_Object old_buffer = echo_buffer[i];
	static char const name_fmt[] = " *Echo Area %d*";
	char name[sizeof name_fmt + INT_STRLEN_BOUND (int)];
	AUTO_STRING_WITH_LEN (lname, name, sprintf (name, name_fmt, i));
	echo_buffer[i] = Fget_buffer_create (lname, Qnil);
	bset_truncate_lines (XBUFFER (echo_buffer[i]), Qnil);
	/* to force word wrap in echo area -
	   it was decided to postpone this*/
	/* XBUFFER (echo_buffer[i])->word_wrap = Qt; */

	for (int j = 0; j < 2; j++)
	  if (EQ (old_buffer, echo_area_buffer[j]))
	    echo_area_buffer[j] = echo_buffer[i];
      }
}


/* Call FN with args A1..A2 with either the current or last displayed
   echo_area_buffer as current buffer.

   WHICH zero means use the current message buffer
   echo_area_buffer[0].  If that is nil, choose a suitable buffer
   from echo_buffer[] and clear it.

   WHICH > 0 means use echo_area_buffer[1].  If that is nil, choose a
   suitable buffer from echo_buffer[] and clear it.

   If WHICH < 0, set echo_area_buffer[1] to echo_area_buffer[0], so
   that the current message becomes the last displayed one, choose a
   suitable buffer for echo_area_buffer[0], and clear it.

   Value is what FN returns.  */

static bool
with_echo_area_buffer (struct window *w, int which,
		       bool (*fn) (void *, Lisp_Object),
		       void *a1, Lisp_Object a2)
{
  Lisp_Object buffer;
  bool this_one, the_other, clear_buffer_p, rc;
  specpdl_ref count = SPECPDL_INDEX ();

  /* If buffers aren't live, make new ones.  */
  ensure_echo_area_buffers ();

  clear_buffer_p = false;

  if (which == 0)
    this_one = false, the_other = true;
  else if (which > 0)
    this_one = true, the_other = false;
  else
    {
      this_one = false, the_other = true;
      clear_buffer_p = true;

      /* We need a fresh one in case the current echo buffer equals
	 the one containing the last displayed echo area message.  */
      if (!NILP (echo_area_buffer[this_one])
	  && EQ (echo_area_buffer[this_one], echo_area_buffer[the_other]))
	echo_area_buffer[this_one] = Qnil;
    }

  /* Choose a suitable buffer from echo_buffer[] if we don't
     have one.  */
  if (NILP (echo_area_buffer[this_one]))
    {
      echo_area_buffer[this_one]
	= (EQ (echo_area_buffer[the_other], echo_buffer[this_one])
	   ? echo_buffer[the_other]
	   : echo_buffer[this_one]);
      clear_buffer_p = true;
    }

  buffer = echo_area_buffer[this_one];

  /* Don't get confused by reusing the buffer used for echoing
     for a different purpose.  */
  if (echo_kboard == NULL && EQ (buffer, echo_message_buffer))
    cancel_echoing ();

  record_unwind_protect (unwind_with_echo_area_buffer,
			 with_echo_area_buffer_unwind_data (w));

  /* Make the echo area buffer current.  Note that for display
     purposes, it is not necessary that the displayed window's buffer
     == current_buffer, except for text property lookup.  So, let's
     only set that buffer temporarily here without doing a full
     Fset_window_buffer.  We must also change w->pointm, though,
     because otherwise an assertions in unshow_buffer fails, and Emacs
     aborts.  */
  set_buffer_internal_1 (XBUFFER (buffer));
  if (w)
    {
      wset_buffer (w, buffer);
      set_marker_both (w->pointm, buffer, BEG, BEG_BYTE);
      set_marker_both (w->old_pointm, buffer, BEG, BEG_BYTE);
    }

  bset_undo_list (current_buffer, Qt);
  bset_read_only (current_buffer, Qnil);
  specbind (Qinhibit_read_only, Qt);
  specbind (Qinhibit_modification_hooks, Qt);

  if (clear_buffer_p && Z > BEG)
    del_range (BEG, Z);

  eassert (BEGV >= BEG);
  eassert (ZV <= Z && ZV >= BEGV);

  rc = fn (a1, a2);

  eassert (BEGV >= BEG);
  eassert (ZV <= Z && ZV >= BEGV);

  unbind_to (count, Qnil);
  return rc;
}


/* Save state that should be preserved around the call to the function
   FN called in with_echo_area_buffer.  */

static Lisp_Object
with_echo_area_buffer_unwind_data (struct window *w)
{
  int i = 0;
  Lisp_Object vector, tmp;

  /* Reduce consing by keeping one vector in
     Vwith_echo_area_save_vector.  */
  vector = Vwith_echo_area_save_vector;
  Vwith_echo_area_save_vector = Qnil;

  if (NILP (vector))
    vector = make_nil_vector (11);

  XSETBUFFER (tmp, current_buffer); ASET (vector, i, tmp); ++i;
  ASET (vector, i, Vdeactivate_mark); ++i;
  ASET (vector, i, make_fixnum (windows_or_buffers_changed)); ++i;

  if (w)
    {
      XSETWINDOW (tmp, w); ASET (vector, i, tmp); ++i;
      ASET (vector, i, w->contents); ++i;
      ASET (vector, i, make_fixnum (marker_position (w->pointm))); ++i;
      ASET (vector, i, make_fixnum (marker_byte_position (w->pointm))); ++i;
      ASET (vector, i, make_fixnum (marker_position (w->old_pointm))); ++i;
      ASET (vector, i, make_fixnum (marker_byte_position (w->old_pointm))); ++i;
      ASET (vector, i, make_fixnum (marker_position (w->start))); ++i;
      ASET (vector, i, make_fixnum (marker_byte_position (w->start))); ++i;
    }
  else
    {
      int end = i + 8;
      for (; i < end; ++i)
	ASET (vector, i, Qnil);
    }

  eassert (i == ASIZE (vector));
  return vector;
}


/* Restore global state from VECTOR which was created by
   with_echo_area_buffer_unwind_data.  */

static void
unwind_with_echo_area_buffer (Lisp_Object vector)
{
  set_buffer_internal_1 (XBUFFER (AREF (vector, 0)));
  Vdeactivate_mark = AREF (vector, 1);
  windows_or_buffers_changed = XFIXNAT (AREF (vector, 2));

  if (WINDOWP (AREF (vector, 3)))
    {
      struct window *w;
      Lisp_Object buffer;

      w = XWINDOW (AREF (vector, 3));
      buffer = AREF (vector, 4);

      wset_buffer (w, buffer);
      set_marker_restricted_both (w->pointm, buffer,
				  XFIXNAT (AREF (vector, 5)),
				  XFIXNAT (AREF (vector, 6)));
      set_marker_restricted_both (w->old_pointm, buffer,
				  XFIXNAT (AREF (vector, 7)),
				  XFIXNAT (AREF (vector, 8)));
      set_marker_restricted_both (w->start, buffer,
				  XFIXNAT (AREF (vector, 9)),
				  XFIXNAT (AREF (vector, 10)));
    }

  Vwith_echo_area_save_vector = vector;
}


/* Set up the echo area for use by print functions.  MULTIBYTE_P
   means we will print multibyte.  */

void
setup_echo_area_for_printing (bool multibyte_p)
{
  /* If we can't find an echo area any more, exit.  */
  if (! FRAME_LIVE_P (XFRAME (selected_frame)))
    Fkill_emacs (Qnil, Qnil);

  ensure_echo_area_buffers ();

  if (!message_buf_print)
    {
      /* A message has been output since the last time we printed.
	 Choose a fresh echo area buffer.  */
      if (EQ (echo_area_buffer[1], echo_buffer[0]))
	echo_area_buffer[0] = echo_buffer[1];
      else
	echo_area_buffer[0] = echo_buffer[0];

      /* Switch to that buffer and clear it.  */
      set_buffer_internal (XBUFFER (echo_area_buffer[0]));
      bset_truncate_lines (current_buffer, Qnil);

      if (Z > BEG)
	{
	  specpdl_ref count = SPECPDL_INDEX ();
	  specbind (Qinhibit_read_only, Qt);
	  /* Note that undo recording is always disabled.  */
	  del_range (BEG, Z);
	  unbind_to (count, Qnil);
	}
      TEMP_SET_PT_BOTH (BEG, BEG_BYTE);

      /* Set up the buffer for the multibyteness we need.  We always
	 set it to be multibyte, except when
	 unibyte-display-via-language-environment is non-nil and the
	 buffer from which we are called is unibyte, because in that
	 case unibyte characters should not be displayed as octal
	 escapes.  */
      if (unibyte_display_via_language_environment
	  && !multibyte_p
	  && !NILP (BVAR (current_buffer, enable_multibyte_characters)))
	Fset_buffer_multibyte (Qnil);
      else if (NILP (BVAR (current_buffer, enable_multibyte_characters)))
	Fset_buffer_multibyte (Qt);

      /* Raise the frame containing the echo area.  */
      if (minibuffer_auto_raise)
	{
	  struct frame *sf = SELECTED_FRAME ();
	  Lisp_Object mini_window;
	  mini_window = FRAME_MINIBUF_WINDOW (sf);
	  Fraise_frame  (WINDOW_FRAME (XWINDOW (mini_window)));
	}

      message_log_maybe_newline ();
      message_buf_print = true;
    }
  else
    {
      if (NILP (echo_area_buffer[0]))
	{
	  if (EQ (echo_area_buffer[1], echo_buffer[0]))
	    echo_area_buffer[0] = echo_buffer[1];
	  else
	    echo_area_buffer[0] = echo_buffer[0];
	}

      if (current_buffer != XBUFFER (echo_area_buffer[0]))
	{
	  /* Someone switched buffers between print requests.  */
	  set_buffer_internal (XBUFFER (echo_area_buffer[0]));
	  bset_truncate_lines (current_buffer, Qnil);
	}
    }
}


/* Display an echo area message in window W.  Value is true if W's
   height is changed.  If display_last_displayed_message_p,
   display the message that was last displayed, otherwise
   display the current message.  */

static bool
display_echo_area (struct window *w)
{
  bool no_message_p, window_height_changed_p;

  /* Temporarily disable garbage collections while displaying the echo
     area.  This is done because a GC can print a message itself.
     That message would modify the echo area buffer's contents while a
     redisplay of the buffer is going on, and seriously confuse
     redisplay.  */
  specpdl_ref count = inhibit_garbage_collection ();

  /* If there is no message, we must call display_echo_area_1
     nevertheless because it resizes the window.  But we will have to
     reset the echo_area_buffer in question to nil at the end because
     with_echo_area_buffer will set it to an empty buffer.  */
  bool i = display_last_displayed_message_p;
  /* According to the C99, C11 and C++11 standards, the integral value
     of a "bool" is always 0 or 1, so this array access is safe here,
     if oddly typed. */
  no_message_p = NILP (echo_area_buffer[i]);

  window_height_changed_p
    = with_echo_area_buffer (w, display_last_displayed_message_p,
			     display_echo_area_1, w, Qnil);

  if (no_message_p)
    echo_area_buffer[i] = Qnil;

  unbind_to (count, Qnil);
  return window_height_changed_p;
}


/* Helper for display_echo_area.  Display the current buffer which
   contains the current echo area message in window W, a mini-window,
   a pointer to which is passed in A1.  A2 is currently not used.
   Change the height of W so that all of the message is displayed.
   Value is true if height of W was changed.  */

static bool
display_echo_area_1 (void *a1, Lisp_Object a2)
{
  struct window *w = a1;
  Lisp_Object window;
  struct text_pos start;

  /* We are about to enter redisplay without going through
     redisplay_internal, so we need to forget these faces by hand
     here.  */
  forget_escape_and_glyphless_faces ();

  /* Do this before displaying, so that we have a large enough glyph
     matrix for the display.  If we can't get enough space for the
     whole text, display the last N lines.  That works by setting w->start.  */
  bool window_height_changed_p = resize_mini_window (w, false);

  /* Use the starting position chosen by resize_mini_window.  */
  SET_TEXT_POS_FROM_MARKER (start, w->start);

  /* Display.  */
  clear_glyph_matrix (w->desired_matrix);
  XSETWINDOW (window, w);
  void *itdata = bidi_shelve_cache ();
  try_window (window, start, 0);
  bidi_unshelve_cache (itdata, false);

  return window_height_changed_p;
}


/* Resize the echo area window to exactly the size needed for the
   currently displayed message, if there is one.  If a mini-buffer
   is active, don't shrink it.  */

void
resize_echo_area_exactly (void)
{
  if (BUFFERP (echo_area_buffer[0])
      && WINDOWP (echo_area_window))
    {
      struct window *w = XWINDOW (echo_area_window);
      Lisp_Object resize_exactly = (minibuf_level == 0 ? Qt : Qnil);
      bool resized_p = with_echo_area_buffer (w, 0, resize_mini_window_1,
					      w, resize_exactly);
      if (resized_p)
	{
	  windows_or_buffers_changed = 42;
	  update_mode_lines = 30;
	  redisplay_internal ();
	}
    }
}


/* Callback function for with_echo_area_buffer, when used from
   resize_echo_area_exactly.  A1 contains a pointer to the window to
   resize, EXACTLY non-nil means resize the mini-window exactly to the
   size of the text displayed.  Value is what resize_mini_window
   returns.  */

static bool
resize_mini_window_1 (void *a1, Lisp_Object exactly)
{
  return resize_mini_window (a1, !NILP (exactly));
}


/* Resize mini-window W to fit the size of its contents.  EXACT_P
   means size the window exactly to the size needed.  Otherwise, it's
   only enlarged until W's buffer is empty.

   Set W->start to the right place to begin display.  If the whole
   contents fit, start at the beginning.  Otherwise, start so as
   to make the end of the contents appear.  This is particularly
   important for y-or-n-p, but seems desirable generally.

   Value is true if the window height has been changed.  */

bool
resize_mini_window (struct window *w, bool exact_p)
{
  struct frame *f = XFRAME (w->frame);
  int old_height = WINDOW_BOX_TEXT_HEIGHT (w);

  eassert (MINI_WINDOW_P (w));

  /* Don't resize windows while redisplaying a window; it would
     confuse redisplay functions when the size of the window they are
     displaying changes from under them.  Such a resizing can happen,
     for instance, when which-func prints a long message while
     we are running fontification-functions.  We're running these
     functions with safe_call which binds inhibit-redisplay to t.  */
  if (!NILP (Vinhibit_redisplay))
    return false;

  /* By default, start display at the beginning.  */
  if (redisplay_adhoc_scroll_in_resize_mini_windows)
    set_marker_both (w->start, w->contents,
		     BUF_BEGV (XBUFFER (w->contents)),
		     BUF_BEGV_BYTE (XBUFFER (w->contents)));

  /* Nil means don't try to resize.  */
  if ((NILP (Vresize_mini_windows)
       && (NILP (resize_mini_frames) || !FRAME_MINIBUF_ONLY_P (f)))
      || (FRAME_X_P (f) && FRAME_OUTPUT_DATA (f) == NULL))
    return false;

  if (FRAME_MINIBUF_ONLY_P (f))
    {
      if (!NILP (resize_mini_frames))
	safe_call1 (Qwindow__resize_mini_frame, WINDOW_FRAME (w));
    }
  else
    {
      struct it it;
      int unit = FRAME_LINE_HEIGHT (f);
      int height, max_height;
      struct text_pos start;
      struct buffer *old_current_buffer = NULL;
      int windows_height = FRAME_INNER_HEIGHT (f);

      if (current_buffer != XBUFFER (w->contents))
	{
	  old_current_buffer = current_buffer;
	  set_buffer_internal (XBUFFER (w->contents));
	}

      init_iterator (&it, w, BEGV, BEGV_BYTE, NULL, DEFAULT_FACE_ID);

      /* Compute the max. number of lines specified by the user.  */
      if (FLOATP (Vmax_mini_window_height))
	max_height = XFLOAT_DATA (Vmax_mini_window_height) * windows_height;
      else if (FIXNUMP (Vmax_mini_window_height))
	max_height = XFIXNUM (Vmax_mini_window_height) * unit;
      else
	max_height = windows_height / 4;

      /* Correct that max. height if it's bogus.  */
      max_height = clip_to_bounds (unit, max_height, windows_height);

      /* Find out the height of the text in the window.  */
      last_height = 0;
      move_it_to (&it, ZV, -1, -1, -1, MOVE_TO_POS);
      /* If move_it_to moved to the next visible line after EOB,
	 account for the height of the last full line.  */
      if (it.max_ascent == 0 && it.max_descent == 0)
	{
	  height = it.current_y;
	  /* Don't add the last line's height if lines are truncated
	     and the text doesn't end in a newline.
	     FIXME: if the text ends in a newline from a display
	     property or an overlay string, they lose: the mini-window
	     might not show the last empty line.  */
	  if (!(it.line_wrap == TRUNCATE
		&& it.current_x <= it.first_visible_x
		&& ZV_BYTE > 1
		&& FETCH_BYTE (ZV_BYTE - 1) != '\n'))
	    height += last_height;
	}
      else
	height = it.current_y + it.max_ascent + it.max_descent;
      height -= min (it.extra_line_spacing, it.max_extra_line_spacing);

      /* Compute a suitable window start.  */
      if (height > max_height)
	{
	  height = (max_height / unit) * unit;
	  if (redisplay_adhoc_scroll_in_resize_mini_windows)
	    {
	      init_iterator (&it, w, ZV, ZV_BYTE, NULL, DEFAULT_FACE_ID);
	      move_it_vertically_backward (&it, height - unit);
              /* The following move is usually a no-op when the stuff
                 displayed in the mini-window comes entirely from buffer
                 text, but it is needed when some of it comes from overlay
                 strings, especially when there's an after-string at ZV.
                 This happens with some completion packages, like
                 icomplete, ido-vertical, etc.  With those packages, if we
                 don't force w->start to be at the beginning of a screen
                 line, important parts of the stuff in the mini-window,
                 such as user prompt, will be hidden from view.  */
              move_it_by_lines (&it, 0);
              start = it.current.pos;
              /* Prevent redisplay_window from recentering, and thus from
                 overriding the window-start point we computed here.  */
              w->start_at_line_beg = false;
              SET_MARKER_FROM_TEXT_POS (w->start, start);
            }
	}
      else
	{
	  SET_TEXT_POS (start, BEGV, BEGV_BYTE);
          SET_MARKER_FROM_TEXT_POS (w->start, start);
        }

      if (EQ (Vresize_mini_windows, Qgrow_only))
	{
	  /* Let it grow only, until we display an empty message, in which
	     case the window shrinks again.  */
	  if (height > old_height)
	    grow_mini_window (w, height - old_height);
	  else if (height < old_height && (exact_p || BEGV == ZV))
	    shrink_mini_window (w);
	}
      else if (height != old_height)
	/* Always resize to exact size needed.  */
	grow_mini_window (w, height - old_height);

      if (old_current_buffer)
	set_buffer_internal (old_current_buffer);
    }

  return WINDOW_BOX_TEXT_HEIGHT (w) != old_height;
}


/* Value is the current message, a string, or nil if there is no
   current message.  */

Lisp_Object
current_message (void)
{
  Lisp_Object msg;

  if (!BUFFERP (echo_area_buffer[0]))
    msg = Qnil;
  else
    {
      with_echo_area_buffer (0, 0, current_message_1, &msg, Qnil);
      if (NILP (msg))
	echo_area_buffer[0] = Qnil;
    }

  return msg;
}


static bool
current_message_1 (void *a1, Lisp_Object a2)
{
  Lisp_Object *msg = a1;

  if (Z > BEG)
    *msg = make_buffer_string (BEG, Z, true);
  else
    *msg = Qnil;
  return false;
}


/* Push the current message on Vmessage_stack for later restoration
   by restore_message.  Value is true if the current message isn't
   empty.  This is a relatively infrequent operation, so it's not
   worth optimizing.  */

bool
push_message (void)
{
  Lisp_Object msg = current_message ();
  Vmessage_stack = Fcons (msg, Vmessage_stack);
  return STRINGP (msg);
}


/* Restore message display from the top of Vmessage_stack.  */

void
restore_message (void)
{
  eassert (CONSP (Vmessage_stack));
  message3_nolog (XCAR (Vmessage_stack));
}


/* Handler for unwind-protect calling pop_message.  */

void
pop_message_unwind (void)
{
  /* Pop the top-most entry off Vmessage_stack.  */
  eassert (CONSP (Vmessage_stack));
  Vmessage_stack = XCDR (Vmessage_stack);
}


/* Check that Vmessage_stack is nil.  Called from emacs.c when Emacs
   exits.  If the stack is not empty, we have a missing
   pop_message_unwind somewhere.  */

void
check_message_stack (void)
{
  if (!NILP (Vmessage_stack))
    emacs_abort ();
}

void
clear_message_stack (void)
{
  Vmessage_stack = Qnil;
}

/* Truncate to NCHARS what will be displayed in the echo area the next
   time we display it---but don't redisplay it now.  */

void
truncate_echo_area (ptrdiff_t nchars)
{
  if (nchars == 0)
    echo_area_buffer[0] = Qnil;
  else if (!noninteractive
	   && INTERACTIVE
	   && !NILP (echo_area_buffer[0]))
    {
      struct frame *sf = SELECTED_FRAME ();
      /* Error messages get reported properly by cmd_error, so this must be
	 just an informative message; if the frame hasn't really been
	 initialized yet, just toss it.  */
      if (sf->glyphs_initialized_p)
	with_echo_area_buffer (0, 0, truncate_message_1,
			       (void *) (intptr_t) nchars, Qnil);
    }
}


/* Helper function for truncate_echo_area.  Truncate the current
   message to at most NCHARS characters.  */

static bool
truncate_message_1 (void *a1, Lisp_Object a2)
{
  intptr_t nchars = (intptr_t) a1;
  if (BEG + nchars < Z)
    del_range (BEG + nchars, Z);
  if (Z == BEG)
    echo_area_buffer[0] = Qnil;
  return false;
}

/* Set the current message to STRING.  */

static void
set_message (Lisp_Object string)
{
  Lisp_Object message = Qnil;

  eassert (STRINGP (string));

  if (FUNCTIONP (Vset_message_function))
    {
      specpdl_ref count = SPECPDL_INDEX ();
      specbind (Qinhibit_quit, Qt);
      message = safe_call1 (Vset_message_function, string);
      unbind_to (count, Qnil);

      if (STRINGP (message))
        {
          string = message;
          message = Qnil;
        }
    }

  if (NILP (message))
    {
      message_enable_multibyte = STRING_MULTIBYTE (string);

      with_echo_area_buffer (0, -1, set_message_1, 0, string);
      message_buf_print = false;
      help_echo_showing_p = false;
    }

  if (STRINGP (Vdebug_on_message)
      && STRINGP (string)
      && fast_string_match (Vdebug_on_message, string) >= 0)
    call_debugger (list2 (Qerror, string));
}


/* Helper function for set_message.  First argument is ignored and second
   argument has the same meaning as for set_message.
   This function is called with the echo area buffer being current.  */

static bool
set_message_1 (void *a1, Lisp_Object string)
{
  eassert (STRINGP (string));

  /* Change multibyteness of the echo buffer appropriately.  We always
     set it to be multibyte, except when
     unibyte-display-via-language-environment is non-nil and the
     string to display is unibyte, because in that case unibyte
     characters should not be displayed as octal escapes.  */
  if (!message_enable_multibyte
      && unibyte_display_via_language_environment
      && !NILP (BVAR (current_buffer, enable_multibyte_characters)))
    Fset_buffer_multibyte (Qnil);
  else if (NILP (BVAR (current_buffer, enable_multibyte_characters)))
    Fset_buffer_multibyte (Qt);

  bset_truncate_lines (current_buffer, message_truncate_lines ? Qt : Qnil);
  if (!NILP (BVAR (current_buffer, bidi_display_reordering)))
    bset_bidi_paragraph_direction (current_buffer, Qleft_to_right);

  /* Insert new message at BEG.  */
  TEMP_SET_PT_BOTH (BEG, BEG_BYTE);

  /* This function takes care of single/multibyte conversion.
     We just have to ensure that the echo area buffer has the right
     setting of enable_multibyte_characters.  */
  insert_from_string (string, 0, 0, SCHARS (string), SBYTES (string), true);

  return false;
}


/* Clear messages.  CURRENT_P means clear the current message.
   LAST_DISPLAYED_P means clear the message last displayed.  */

void
clear_message (bool current_p, bool last_displayed_p)
{
  Lisp_Object preserve = Qnil;

  if (current_p)
    {
      if (FUNCTIONP (Vclear_message_function))
        {
          specpdl_ref count = SPECPDL_INDEX ();
          specbind (Qinhibit_quit, Qt);
          preserve = safe_call (1, Vclear_message_function);
          unbind_to (count, Qnil);
        }

      if (!EQ (preserve, Qdont_clear_message))
        {
          echo_area_buffer[0] = Qnil;
          message_cleared_p = true;
        }
    }

  if (last_displayed_p)
    echo_area_buffer[1] = Qnil;

  message_buf_print = false;
}

/* Clear garbaged frames.

   This function is used where the old redisplay called
   redraw_garbaged_frames which in turn called redraw_frame which in
   turn called clear_frame.  The call to clear_frame was a source of
   flickering.  I believe a clear_frame is not necessary.  It should
   suffice in the new redisplay to invalidate all current matrices,
   and ensure a complete redisplay of all windows.  */

static void
clear_garbaged_frames (void)
{
  if (frame_garbaged)
    {
      Lisp_Object tail, frame;
      struct frame *sf = SELECTED_FRAME ();

      FOR_EACH_FRAME (tail, frame)
	{
	  struct frame *f = XFRAME (frame);

	  if (FRAME_VISIBLE_P (f) && FRAME_GARBAGED_P (f))
	    {
	      if (f->resized_p
		  /* It makes no sense to redraw a non-selected TTY
		     frame, since that will actually clear the
		     selected frame, and might leave the selected
		     frame with corrupted display, if it happens not
		     to be marked garbaged.  */
		  && !(f != sf && (FRAME_TERMCAP_P (f) || FRAME_MSDOS_P (f))))
		redraw_frame (f);
	      else
		clear_current_matrices (f);

#ifdef HAVE_WINDOW_SYSTEM
              if (FRAME_WINDOW_P (f)
                  && FRAME_RIF (f)->clear_under_internal_border)
                FRAME_RIF (f)->clear_under_internal_border (f);
#endif
	      fset_redisplay (f);
	      f->garbaged = false;
	      f->resized_p = false;
            }
	}

      frame_garbaged = false;
    }
}


/* Redisplay the echo area of the selected frame.  If UPDATE_FRAME_P, update
   selected_frame.  */

static void
echo_area_display (bool update_frame_p)
{
  Lisp_Object mini_window;
  struct window *w;
  struct frame *f;
  bool window_height_changed_p = false;
  struct frame *sf = SELECTED_FRAME ();

  mini_window = FRAME_MINIBUF_WINDOW (sf);
  if (NILP (mini_window))
    return;

  w = XWINDOW (mini_window);
  f = XFRAME (WINDOW_FRAME (w));

  /* Don't display if frame is invisible or not yet initialized.  */
  if (!FRAME_VISIBLE_P (f) || !f->glyphs_initialized_p)
    return;

#ifdef HAVE_WINDOW_SYSTEM
  /* When Emacs starts, selected_frame may be the initial terminal
     frame.  If we let this through, a message would be displayed on
     the terminal.  */
  if (FRAME_INITIAL_P (XFRAME (selected_frame)))
    return;
#endif /* HAVE_WINDOW_SYSTEM */

  /* Redraw garbaged frames.  */
  clear_garbaged_frames ();

  if (!NILP (echo_area_buffer[0]) || minibuf_level == 0)
    {
      echo_area_window = mini_window;
      window_height_changed_p = display_echo_area (w);
      w->must_be_updated_p = true;

      /* Update the display, unless called from redisplay_internal.
	 Also don't update the screen during redisplay itself.  The
	 update will happen at the end of redisplay, and an update
	 here could cause confusion.  */
      if (update_frame_p && !redisplaying_p)
	{
	  int n = 0;

	  /* If the display update has been interrupted by pending
	     input, update mode lines in the frame.  Due to the
	     pending input, it might have been that redisplay hasn't
	     been called, so that mode lines above the echo area are
	     garbaged.  This looks odd, so we prevent it here.  */
	  if (!display_completed)
	    {
	      n = redisplay_mode_lines (FRAME_ROOT_WINDOW (f), false);

#ifdef HAVE_WINDOW_SYSTEM
              if (FRAME_WINDOW_P (f)
                  && FRAME_RIF (f)->clear_under_internal_border)
                FRAME_RIF (f)->clear_under_internal_border (f);
#endif
	    }

	  if (window_height_changed_p
	      /* Don't do this if Emacs is shutting down.  Redisplay
	         needs to run hooks.  */
	      && !NILP (Vrun_hooks))
	    {
	      /* Must update other windows.  Likewise as in other
		 cases, don't let this update be interrupted by
		 pending input.  */
	      specpdl_ref count = SPECPDL_INDEX ();
	      specbind (Qredisplay_dont_pause, Qt);
	      fset_redisplay (f);
	      redisplay_internal ();
	      unbind_to (count, Qnil);
	    }
	  else if (FRAME_WINDOW_P (f) && n == 0)
	    {
	      /* Window configuration is the same as before.
		 Can do with a display update of the echo area,
		 unless we displayed some mode lines.  */
	      update_single_window (w);
	      flush_frame (f);
	    }
	  else
	    update_frame (f, true, true);

	  /* If cursor is in the echo area, make sure that the next
	     redisplay displays the minibuffer, so that the cursor will
	     be replaced with what the minibuffer wants.  */
	  if (cursor_in_echo_area)
	    wset_redisplay (XWINDOW (mini_window));
	}
    }
  else if (!EQ (mini_window, selected_window))
    wset_redisplay (XWINDOW (mini_window));

  /* Last displayed message is now the current message.  */
  echo_area_buffer[1] = echo_area_buffer[0];
  /* Inform read_char that we're not echoing.  */
  echo_message_buffer = Qnil;

  /* Prevent redisplay optimization in redisplay_internal by resetting
     this_line_start_pos.  This is done because the mini-buffer now
     displays the message instead of its buffer text.  */
  if (EQ (mini_window, selected_window))
    CHARPOS (this_line_start_pos) = 0;

  if (window_height_changed_p)
    {
      fset_redisplay (f);

      /* If window configuration was changed, frames may have been
	 marked garbaged.  Clear them or we will experience
	 surprises wrt scrolling.
	 FIXME: How/why/when?  */
      clear_garbaged_frames ();
    }
}

/* True if W's buffer was changed but not saved.  */

static bool
window_buffer_changed (struct window *w)
{
  struct buffer *b = XBUFFER (w->contents);

  eassert (BUFFER_LIVE_P (b));

  return (BUF_SAVE_MODIFF (b) < BUF_MODIFF (b)) != w->last_had_star;
}

/* True if W has %c or %C in its mode line and mode line should be updated.  */

static bool
mode_line_update_needed (struct window *w)
{
  return (w->column_number_displayed != -1
	  && !(PT == w->last_point && !window_outdated (w))
	  && (w->column_number_displayed != current_column ()));
}

/* True if window start of W is frozen and may not be changed during
   redisplay.  */

static bool
window_frozen_p (struct window *w)
{
  if (FRAME_WINDOWS_FROZEN (XFRAME (WINDOW_FRAME (w))))
    {
      Lisp_Object window;

      XSETWINDOW (window, w);
      if (MINI_WINDOW_P (w))
	return false;
      else if (EQ (window, selected_window))
	return false;
      else if (MINI_WINDOW_P (XWINDOW (selected_window))
	       && EQ (window, Vminibuf_scroll_window))
	/* This special window can't be frozen too.  */
	return false;
      else
	return true;
    }
  return false;
}

/***********************************************************************
		     Mode Lines and Frame Titles
 ***********************************************************************/

/* A buffer for constructing non-propertized mode-line strings and
   frame titles in it; allocated from the heap in init_xdisp and
   resized as needed in store_mode_line_noprop_char.  */

static char *mode_line_noprop_buf;

/* The buffer's end, and a current output position in it.  */

static char *mode_line_noprop_buf_end;
static char *mode_line_noprop_ptr;

#define MODE_LINE_NOPROP_LEN(start) \
  ((mode_line_noprop_ptr - mode_line_noprop_buf) - start)

static enum {
  MODE_LINE_DISPLAY = 0,
  MODE_LINE_TITLE,
  MODE_LINE_NOPROP,
  MODE_LINE_STRING
} mode_line_target;

/* Alist that caches the results of :propertize.
   Each element is (PROPERTIZED-STRING . PROPERTY-LIST).  */
static Lisp_Object mode_line_proptrans_alist;

/* List of strings making up the mode-line.  */
static Lisp_Object mode_line_string_list;

/* Base face property when building propertized mode line string.  */
static Lisp_Object mode_line_string_face;
static Lisp_Object mode_line_string_face_prop;


/* Unwind data for mode line strings */

static Lisp_Object Vmode_line_unwind_vector;

static Lisp_Object
format_mode_line_unwind_data (struct frame *target_frame,
			      struct buffer *obuf,
			      Lisp_Object owin,
			      bool save_proptrans)
{
  Lisp_Object vector, tmp;

  /* Reduce consing by keeping one vector in
     Vwith_echo_area_save_vector.  */
  vector = Vmode_line_unwind_vector;
  Vmode_line_unwind_vector = Qnil;

  if (NILP (vector))
    vector = make_nil_vector (12);

  ASET (vector, 0, make_fixnum (mode_line_target));
  ASET (vector, 1, make_fixnum (MODE_LINE_NOPROP_LEN (0)));
  ASET (vector, 2, mode_line_string_list);
  ASET (vector, 3, save_proptrans ? mode_line_proptrans_alist : Qt);
  ASET (vector, 4, mode_line_string_face);
  ASET (vector, 5, mode_line_string_face_prop);

  if (obuf)
    XSETBUFFER (tmp, obuf);
  else
    tmp = Qnil;
  ASET (vector, 6, tmp);
  ASET (vector, 7, owin);
  if (target_frame)
    {
      Lisp_Object buffer = XWINDOW (target_frame->selected_window)->contents;
      struct buffer *b = XBUFFER (buffer);
      struct buffer *cb = current_buffer;

      /* Similarly to `with-selected-window', if the operation selects
	 a window on another frame, we must restore that frame's
	 selected window, and (for a tty) the top-frame.  */
      ASET (vector, 8, target_frame->selected_window);
      if (FRAME_TERMCAP_P (target_frame))
	ASET (vector, 9, FRAME_TTY (target_frame)->top_frame);

      /* If we select a window on another frame, make sure that that
	 selection does not leave its buffer's point modified when
	 unwinding (Bug#32777).  */
      ASET (vector, 10, buffer);
      current_buffer = b;
      ASET (vector, 11, build_marker (current_buffer, PT, PT_BYTE));
      current_buffer = cb;
    }

  return vector;
}

static void
unwind_format_mode_line (Lisp_Object vector)
{
  Lisp_Object old_window = AREF (vector, 7);
  Lisp_Object target_frame_window = AREF (vector, 8);
  Lisp_Object old_top_frame = AREF (vector, 9);

  mode_line_target = XFIXNUM (AREF (vector, 0));
  mode_line_noprop_ptr = mode_line_noprop_buf + XFIXNUM (AREF (vector, 1));
  mode_line_string_list = AREF (vector, 2);
  if (! EQ (AREF (vector, 3), Qt))
    mode_line_proptrans_alist = AREF (vector, 3);
  mode_line_string_face = AREF (vector, 4);
  mode_line_string_face_prop = AREF (vector, 5);

  /* Select window before buffer, since it may change the buffer.  */
  if (WINDOW_LIVE_P (old_window))
    {
      /* If the operation that we are unwinding had selected a window
	 on a different frame, reset its frame-selected-window.  For a
	 text terminal, reset its top-frame if necessary.  */
      if (WINDOW_LIVE_P (target_frame_window))
	{
	  Lisp_Object frame
	    = WINDOW_FRAME (XWINDOW (target_frame_window));

	  if (!EQ (frame, WINDOW_FRAME (XWINDOW (old_window))))
	    Fselect_window (target_frame_window, Qt);

	  if (!NILP (old_top_frame) && !EQ (old_top_frame, frame)
	      /* This could've been destroyed during the formatting,
		 possibly because the terminal was deleted.  */
	      && FRAME_LIVE_P (XFRAME (old_top_frame)))
	    Fselect_frame (old_top_frame, Qt);
	}

      Fselect_window (old_window, Qt);

      /* Restore point of target_frame_window's buffer (Bug#32777).
	 But do this only after old_window has been reselected to
	 avoid that the window point of target_frame_window moves.  */
      if (WINDOW_LIVE_P (target_frame_window))
	{
	  Lisp_Object buffer = AREF (vector, 10);

	  if (BUFFER_LIVE_P (XBUFFER (buffer)))
	    {
	      struct buffer *cb = current_buffer;

	      current_buffer = XBUFFER (buffer);
	      set_point_from_marker (AREF (vector, 11));
	      ASET (vector, 11, Qnil);
	      current_buffer = cb;
	    }
	}
    }

  if (!NILP (AREF (vector, 6)))
    {
      set_buffer_internal_1 (XBUFFER (AREF (vector, 6)));
      ASET (vector, 6, Qnil);
    }

  Vmode_line_unwind_vector = vector;
}


/* Store a single character C for the frame title in mode_line_noprop_buf.
   Re-allocate mode_line_noprop_buf if necessary.  */

static void
store_mode_line_noprop_char (char c)
{
  /* If output position has reached the end of the allocated buffer,
     increase the buffer's size.  */
  if (mode_line_noprop_ptr == mode_line_noprop_buf_end)
    {
      ptrdiff_t len = MODE_LINE_NOPROP_LEN (0);
      ptrdiff_t size = len;
      mode_line_noprop_buf =
	xpalloc (mode_line_noprop_buf, &size, 1, STRING_BYTES_BOUND, 1);
      mode_line_noprop_buf_end = mode_line_noprop_buf + size;
      mode_line_noprop_ptr = mode_line_noprop_buf + len;
    }

  *mode_line_noprop_ptr++ = c;
}


/* Store part of a frame title in mode_line_noprop_buf, beginning at
   mode_line_noprop_ptr.  STRING is the string to store.  Do not copy
   characters that yield more columns than PRECISION; PRECISION <= 0
   means copy the whole string.  Pad with spaces until FIELD_WIDTH
   number of characters have been copied; FIELD_WIDTH <= 0 means don't
   pad.  Called from display_mode_element when it is used to build a
   frame title.  */

static int
store_mode_line_noprop (const char *string, int field_width, int precision)
{
  const unsigned char *str = (const unsigned char *) string;
  int n = 0;
  ptrdiff_t dummy, nbytes;

  /* Copy at most PRECISION chars from STR.  */
  nbytes = strlen (string);
  n += c_string_width (str, nbytes, precision, &dummy, &nbytes);
  while (nbytes--)
    store_mode_line_noprop_char (*str++);

  /* Fill up with spaces until FIELD_WIDTH reached.  */
  while (field_width > 0
	 && n < field_width)
    {
      store_mode_line_noprop_char (' ');
      ++n;
    }

  return n;
}

/***********************************************************************
			     Frame Titles
 ***********************************************************************/

#ifdef HAVE_WINDOW_SYSTEM

/* Set the title of FRAME, if it has changed.  The title format is
   Vicon_title_format if FRAME is iconified, otherwise it is
   frame_title_format.  */

void
gui_consider_frame_title (Lisp_Object frame)
{
  struct frame *f = XFRAME (frame);
  Lisp_Object format_data;

  if ((FRAME_WINDOW_P (f)
       || FRAME_MINIBUF_ONLY_P (f)
       || f->explicit_name)
      && !FRAME_TOOLTIP_P (f))
    {
      /* Do we have more than one visible frame on this X display?  */
      Lisp_Object tail, other_frame, fmt;
      ptrdiff_t title_start;
      char *title;
      ptrdiff_t len;
      struct it it;
      specpdl_ref count = SPECPDL_INDEX ();

      FOR_EACH_FRAME (tail, other_frame)
	{
	  struct frame *tf = XFRAME (other_frame);

	  if (tf != f
	      && FRAME_KBOARD (tf) == FRAME_KBOARD (f)
	      && !FRAME_MINIBUF_ONLY_P (tf)
	      && !FRAME_PARENT_FRAME (tf)
	      && !FRAME_TOOLTIP_P (tf)
	      && (FRAME_VISIBLE_P (tf) || FRAME_ICONIFIED_P (tf)))
	    break;
	}

      /* Set global variable indicating that multiple frames exist.  */
      multiple_frames = CONSP (tail);

      /* select-frame calls resize_mini_window, which could resize the
	 mini-window and by that undo the effect of this redisplay
	 cycle wrt minibuffer and echo-area display.  Binding
	 inhibit-redisplay to t makes the call to resize_mini_window a
	 no-op, thus avoiding the adverse side effects.  */

      /* The following was moved before the record_unwind_protect form
	 below to inhibit redisplay also when restoring the selected
	 window/frame: This avoids that resize_mini_window sizes back
	 the minibuffer window of a temporarily selected frame.  See
	 Bug#34317.  */
      specbind (Qinhibit_redisplay, Qt);

      /* Switch to the buffer of selected window of the frame.  Set up
	 mode_line_target so that display_mode_element will output
	 into mode_line_noprop_buf; then display the title.  Save the
	 original frame and selected window, and possibly the topmost
	 frame of the tty (for tty frames) into a vector; it will be
	 restored later.  */

      format_data = format_mode_line_unwind_data (f, current_buffer,
						  selected_window,
						  false);
      record_unwind_protect (unwind_format_mode_line, format_data);

      Fselect_window (f->selected_window, Qt);
      set_buffer_internal_1 (XBUFFER (XWINDOW (f->selected_window)->contents));
      fmt = (FRAME_ICONIFIED_P (f) && !EQ (Vicon_title_format, Qt)
	     ? Vicon_title_format : Vframe_title_format);

      mode_line_target = MODE_LINE_TITLE;
      title_start = MODE_LINE_NOPROP_LEN (0);
      init_iterator (&it, XWINDOW (f->selected_window), -1, -1,
		     NULL, DEFAULT_FACE_ID);
      display_mode_element (&it, 0, -1, -1, fmt, Qnil, false);
      len = MODE_LINE_NOPROP_LEN (title_start);
      title = mode_line_noprop_buf + title_start;
      /* Make sure that any raw bytes in the title are properly
         represented by their multibyte sequences.  */
      ptrdiff_t nchars = 0;
      len = str_as_multibyte ((unsigned char *) title,
			      mode_line_noprop_buf_end - title,
			      len, &nchars);
      unbind_to (count, Qnil);

      /* Set the title only if it's changed.  This avoids consing in
	 the common case where it hasn't.  (If it turns out that we've
	 already wasted too much time by walking through the list with
	 display_mode_element, then we might need to optimize at a
	 higher level than this.)  */
      if ((! STRINGP (f->name)
           || SBYTES (f->name) != len
           || memcmp (title, SDATA (f->name), len) != 0)
          && FRAME_TERMINAL (f)->implicit_set_name_hook)
        {
          Lisp_Object title_string = make_multibyte_string (title, nchars, len);
          FRAME_TERMINAL (f)->implicit_set_name_hook (f, title_string, Qnil);
        }
    }
}

#endif /* not HAVE_WINDOW_SYSTEM */


/***********************************************************************
			      Menu Bars
 ***********************************************************************/

/* True if we will not redisplay all visible windows.  */
#define REDISPLAY_SOME_P()				\
  ((windows_or_buffers_changed == 0			\
    || windows_or_buffers_changed == REDISPLAY_SOME)	\
   && (update_mode_lines == 0				\
       || update_mode_lines == REDISPLAY_SOME))

static bool
needs_no_redisplay (struct window *w)
{
  struct buffer *buffer = XBUFFER (w->contents);
  struct frame *f = XFRAME (w->frame);
  return (REDISPLAY_SOME_P ()
          && !w->redisplay
          && !w->update_mode_line
          && !f->face_change
          && !f->redisplay
          && !buffer->text->redisplay
          && window_point (w) == w->last_point);
}

/* Prepare for redisplay by updating menu-bar item lists when
   appropriate.  This can call eval.  */

static void
prepare_menu_bars (void)
{
  bool all_windows = windows_or_buffers_changed || update_mode_lines;
  bool some_windows = REDISPLAY_SOME_P ();

  if (FUNCTIONP (Vpre_redisplay_function))
    {
      Lisp_Object windows = all_windows ? Qt : Qnil;
      if (all_windows && some_windows)
	{
	  Lisp_Object ws = window_list ();
	  for (windows = Qnil; CONSP (ws); ws = XCDR (ws))
	    {
	      Lisp_Object this = XCAR (ws);
	      struct window *w = XWINDOW (this);
	      /* Cf. conditions for redisplaying a window at the
		 beginning of redisplay_window.  */
	      if (!needs_no_redisplay (w))
		windows = Fcons (this, windows);
	    }
	}
      safe__call1 (true, Vpre_redisplay_function, windows);
    }

  /* Update all frame titles based on their buffer names, etc.  We do
     this before the menu bars so that the buffer-menu will show the
     up-to-date frame titles.  */
#ifdef HAVE_WINDOW_SYSTEM
  if (all_windows)
    {
      Lisp_Object tail, frame;

      FOR_EACH_FRAME (tail, frame)
	{
	  struct frame *f = XFRAME (frame);
	  struct window *w = XWINDOW (FRAME_SELECTED_WINDOW (f));
	  if (some_windows
	      && !f->redisplay
	      && !w->redisplay
	      && !XBUFFER (w->contents)->text->redisplay)
	    continue;

	  if (!FRAME_TOOLTIP_P (f)
	      && (FRAME_ICONIFIED_P (f)
		  || FRAME_VISIBLE_P (f) == 1
		  /* Exclude TTY frames that are obscured because they
		     are not the top frame on their console.  This is
		     because gui_consider_frame_title actually switches
		     to the frame, which for TTY frames means it is
		     marked as garbaged, and will be completely
		     redrawn on the next redisplay cycle.  This causes
		     TTY frames to be completely redrawn, when there
		     are more than one of them, even though nothing
		     should be changed on display.  */
		  || (FRAME_VISIBLE_P (f) == 2 && FRAME_WINDOW_P (f))))
	    gui_consider_frame_title (frame);
	}
    }
#endif /* HAVE_WINDOW_SYSTEM */

  /* Update the menu bar item lists, if appropriate.  This has to be
     done before any actual redisplay or generation of display lines.  */

  if (all_windows)
    {
      Lisp_Object tail, frame;
      specpdl_ref count = SPECPDL_INDEX ();
      /* True means that update_menu_bar has run its hooks
	 so any further calls to update_menu_bar shouldn't do so again.  */
      bool menu_bar_hooks_run = false;

      record_unwind_save_match_data ();

      FOR_EACH_FRAME (tail, frame)
	{
	  struct frame *f = XFRAME (frame);
	  struct window *w = XWINDOW (FRAME_SELECTED_WINDOW (f));

	  /* Ignore tooltip frame.  */
	  if (FRAME_TOOLTIP_P (f))
	    continue;

	  if (some_windows
	      && !f->redisplay
	      && !w->redisplay
	      && !XBUFFER (w->contents)->text->redisplay)
	    continue;

	  if (!FRAME_PARENT_FRAME (f))
	    menu_bar_hooks_run = update_menu_bar (f, false, menu_bar_hooks_run);

	  update_tab_bar (f, false);
#ifdef HAVE_WINDOW_SYSTEM
	  update_tool_bar (f, false);
#endif
	}

      unbind_to (count, Qnil);
    }
  else
    {
      struct frame *sf = SELECTED_FRAME ();

      if (!FRAME_PARENT_FRAME (sf))
	update_menu_bar (sf, true, false);

      update_tab_bar (sf, true);
#ifdef HAVE_WINDOW_SYSTEM
      update_tool_bar (sf, true);
#endif
    }
}


/* Update the menu bar item list for frame F.  This has to be done
   before we start to fill in any display lines, because it can call
   eval.

   If SAVE_MATCH_DATA, we must save and restore it here.

   If HOOKS_RUN, a previous call to update_menu_bar
   already ran the menu bar hooks for this redisplay, so there
   is no need to run them again.  The return value is the
   updated value of this flag, to pass to the next call.  */

static bool
update_menu_bar (struct frame *f, bool save_match_data, bool hooks_run)
{
  Lisp_Object window;
  struct window *w;

  /* If called recursively during a menu update, do nothing.  This can
     happen when, for instance, an activate-menubar-hook causes a
     redisplay.  */
  if (inhibit_menubar_update)
    return hooks_run;

  window = FRAME_SELECTED_WINDOW (f);
  w = XWINDOW (window);

  if (FRAME_WINDOW_P (f)
      ?
#ifdef HAVE_EXT_MENU_BAR
      FRAME_EXTERNAL_MENU_BAR (f)
#else
      FRAME_MENU_BAR_LINES (f) > 0
#endif
      : FRAME_MENU_BAR_LINES (f) > 0)
    {
      /* If the user has switched buffers or windows, we need to
	 recompute to reflect the new bindings.  But we'll
	 recompute when update_mode_lines is set too; that means
	 that people can use force-mode-line-update to request
	 that the menu bar be recomputed.  The adverse effect on
	 the rest of the redisplay algorithm is about the same as
	 windows_or_buffers_changed anyway.  */
      if (windows_or_buffers_changed
	  /* This used to test w->update_mode_line, but we believe
	     there is no need to recompute the menu in that case.  */
	  || update_mode_lines
	  || window_buffer_changed (w))
	{
	  struct buffer *prev = current_buffer;
	  specpdl_ref count = SPECPDL_INDEX ();

	  specbind (Qinhibit_menubar_update, Qt);

	  set_buffer_internal_1 (XBUFFER (w->contents));
	  if (save_match_data)
	    record_unwind_save_match_data ();
	  if (NILP (Voverriding_local_map_menu_flag))
	    {
	      specbind (Qoverriding_terminal_local_map, Qnil);
	      specbind (Qoverriding_local_map, Qnil);
	    }

	  if (!hooks_run)
	    {
	      /* Run the Lucid hook.  */
	      safe_run_hooks (Qactivate_menubar_hook);

	      /* If it has changed current-menubar from previous value,
		 really recompute the menu-bar from the value.  */
	      safe_run_hooks (Qmenu_bar_update_hook);

	      hooks_run = true;
	    }

	  XSETFRAME (Vmenu_updating_frame, f);
	  fset_menu_bar_items (f, menu_bar_items (FRAME_MENU_BAR_ITEMS (f)));

	  /* Redisplay the menu bar in case we changed it.  */
#ifdef HAVE_EXT_MENU_BAR
	  if (FRAME_WINDOW_P (f))
            {
#if defined (HAVE_NS)
              /* All frames on Mac OS share the same menubar.  So only
                 the selected frame should be allowed to set it.  */
              if (f == SELECTED_FRAME ())
#endif
		set_frame_menubar (f, false);
	    }
	  else
	    /* On a terminal screen, the menu bar is an ordinary screen
	       line, and this makes it get updated.  */
	    w->update_mode_line = true;
#else /* ! (HAVE_EXT_MENU_BAR) */
	  /* In the non-toolkit version, the menu bar is an ordinary screen
	     line, and this makes it get updated.  */
	  w->update_mode_line = true;
#endif /* HAVE_EXT_MENU_BAR */

	  unbind_to (count, Qnil);
	  set_buffer_internal_1 (prev);
	}
    }

  return hooks_run;
}



/***********************************************************************
			       Tab-bars
 ***********************************************************************/

/* Restore WINDOW as the selected window and its frame as the selected
   frame.  If WINDOW is dead but the selected frame is live, make the
   latter's selected window the selected window.  If both, WINDOW and
   the selected frame, are dead, assign selected frame and window from
   some arbitrary live frame.  Abort if no such frame can be found.  */
static void
restore_selected_window (Lisp_Object window)
{
  if (WINDOW_LIVE_P (window))
    /* If WINDOW is live, make it the selected window and its frame's
       selected window and set the selected frame to its frame.  */
    {
      selected_window = window;
      selected_frame = XWINDOW (window)->frame;
      FRAME_SELECTED_WINDOW (XFRAME (selected_frame)) = window;
    }
  else if (FRAMEP (selected_frame) && FRAME_LIVE_P (XFRAME (selected_frame)))
    /* If WINDOW is dead but the selected frame is still live, make the
       latter's selected window the selected one.  */
    selected_window = FRAME_SELECTED_WINDOW (XFRAME (selected_frame));
  else
    /* If WINDOW and the selected frame are dead, choose some live,
       non-child and non-tooltip frame as the new selected frame and
       make its selected window the selected window.  */
    {
      Lisp_Object tail;
      Lisp_Object frame UNINIT;

      FOR_EACH_FRAME (tail, frame)
	{
	  struct frame *f = XFRAME (frame);

	  if (!FRAME_PARENT_FRAME (f) && !FRAME_TOOLTIP_P (f))
	    {
	      selected_frame = frame;
	      selected_window = FRAME_SELECTED_WINDOW (f);

	      return;
	    }
	}

      /* Abort if we cannot find a live frame.  */
      emacs_abort ();
    }
}

/* Restore WINDOW, if live, as its frame's selected window.  */
static void
restore_frame_selected_window (Lisp_Object window)
{
  if (WINDOW_LIVE_P (window))
    /* If WINDOW is live, make it its frame's selected window.  If that
       frame is the selected frame, make WINDOW the selected window as
       well.  */
    {
      Lisp_Object frame = XWINDOW (window)->frame;

      FRAME_SELECTED_WINDOW (XFRAME (frame)) = window;
      if (EQ (frame, selected_frame))
	selected_window = window;
    }
}

/* Update the tab-bar item list for frame F.  This has to be done
   before we start to fill in any display lines.  Called from
   prepare_menu_bars.  If SAVE_MATCH_DATA, we must save
   and restore it here.  */

static void
update_tab_bar (struct frame *f, bool save_match_data)
{
  bool do_update = false;

#ifdef HAVE_WINDOW_SYSTEM
  if (FRAME_WINDOW_P (f) && WINDOWP (f->tab_bar_window)) {
    if (WINDOW_TOTAL_LINES (XWINDOW (f->tab_bar_window)) > 0)
      do_update = true;
  }
  else
#endif
  if (FRAME_TAB_BAR_LINES (f) > 0)
    do_update = true;

  if (do_update)
    {
      Lisp_Object window;
      struct window *w;

      window = FRAME_SELECTED_WINDOW (f);
      w = XWINDOW (window);

      /* If the user has switched buffers or windows, we need to
	 recompute to reflect the new bindings.  But we'll
	 recompute when update_mode_lines is set too; that means
	 that people can use force-mode-line-update to request
	 that the menu bar be recomputed.  The adverse effect on
	 the rest of the redisplay algorithm is about the same as
	 windows_or_buffers_changed anyway.  */
      if (windows_or_buffers_changed
	  || w->update_mode_line
	  || update_mode_lines
	  || window_buffer_changed (w))
	{
	  struct buffer *prev = current_buffer;
	  specpdl_ref count = SPECPDL_INDEX ();
	  Lisp_Object new_tab_bar;
          int new_n_tab_bar;

	  /* Set current_buffer to the buffer of the selected
	     window of the frame, so that we get the right local
	     keymaps.  */
	  set_buffer_internal_1 (XBUFFER (w->contents));

	  /* Save match data, if we must.  */
	  if (save_match_data)
	    record_unwind_save_match_data ();

	  /* Make sure that we don't accidentally use bogus keymaps.  */
	  if (NILP (Voverriding_local_map_menu_flag))
	    {
	      specbind (Qoverriding_terminal_local_map, Qnil);
	      specbind (Qoverriding_local_map, Qnil);
	    }

	  /* We must temporarily set the selected frame to this frame
	     before calling tab_bar_items, because the calculation of
	     the tab-bar keymap uses the selected frame (see
	     `tab-bar-make-keymap' in tab-bar.el).  */
	  eassert (EQ (selected_window,
		       /* Since we only explicitly preserve selected_frame,
			  check that selected_window would be redundant.  */
		       XFRAME (selected_frame)->selected_window));
#ifdef HAVE_WINDOW_SYSTEM
	  Lisp_Object frame;
	  record_unwind_protect (restore_selected_window, selected_window);
	  XSETFRAME (frame, f);
	  selected_frame = frame;
	  selected_window = FRAME_SELECTED_WINDOW (f);
#endif

	  /* Build desired tab-bar items from keymaps.  */
          new_tab_bar
	    = tab_bar_items (Fcopy_sequence (f->tab_bar_items),
			      &new_n_tab_bar);

	  /* Redisplay the tab-bar if we changed it.  */
	  if (new_n_tab_bar != f->n_tab_bar_items
	      || NILP (Fequal (new_tab_bar, f->tab_bar_items)))
            {
              /* Redisplay that happens asynchronously due to an expose event
                 may access f->tab_bar_items.  Make sure we update both
                 variables within BLOCK_INPUT so no such event interrupts.  */
              block_input ();
              fset_tab_bar_items (f, new_tab_bar);
              f->n_tab_bar_items = new_n_tab_bar;
              w->update_mode_line = true;
              unblock_input ();
            }

	  unbind_to (count, Qnil);
	  set_buffer_internal_1 (prev);
	}
    }
}

/* Redisplay the tab bar in the frame for window W.

   The tab bar of X frames that don't have X toolkit support is
   displayed in a special window W->frame->tab_bar_window.

   The tab bar of terminal frames is treated specially as far as
   glyph matrices are concerned.  Tab bar lines are not part of
   windows, so the update is done directly on the frame matrix rows
   for the tab bar.  */

static void
display_tab_bar (struct window *w)
{
  struct frame *f = XFRAME (WINDOW_FRAME (w));
  struct it it;
  Lisp_Object items;
  int i;

  /* Don't do all this for graphical frames.  */
#ifdef HAVE_NTGUI
  if (FRAME_W32_P (f))
    return;
#endif
#if defined (USE_X_TOOLKIT) || defined (USE_GTK)
  if (FRAME_X_P (f))
    return;
#endif

#ifdef HAVE_NS
  if (FRAME_NS_P (f))
    return;
#endif /* HAVE_NS */

#if defined (USE_X_TOOLKIT) || defined (USE_GTK)
  eassert (!FRAME_WINDOW_P (f));
  init_iterator (&it, w, -1, -1, f->desired_matrix->rows
                 + (FRAME_MENU_BAR_LINES (f) > 0 ? 1 : 0),
                 TAB_BAR_FACE_ID);
  it.first_visible_x = 0;
  it.last_visible_x = FRAME_PIXEL_WIDTH (f);
#elif defined (HAVE_X_WINDOWS) /* X without toolkit.  */
  if (FRAME_WINDOW_P (f))
    {
      /* Tab bar lines are displayed in the desired matrix of the
	 dummy window tab_bar_window.  */
      struct window *tab_w;
      tab_w = XWINDOW (f->tab_bar_window);
      init_iterator (&it, tab_w, -1, -1, tab_w->desired_matrix->rows,
		     TAB_BAR_FACE_ID);
      it.first_visible_x = 0;
      it.last_visible_x = FRAME_PIXEL_WIDTH (f);
    }
  else
#endif /* not USE_X_TOOLKIT and not USE_GTK */
    {
      /* This is a TTY frame, i.e. character hpos/vpos are used as
	 pixel x/y.  */
      init_iterator (&it, w, -1, -1, f->desired_matrix->rows
                     + (FRAME_MENU_BAR_LINES (f) > 0 ? 1 : 0),
		     TAB_BAR_FACE_ID);
      it.first_visible_x = 0;
      it.last_visible_x = FRAME_COLS (f);
    }

  /* FIXME: This should be controlled by a user option.  See the
     comments in redisplay_tool_bar and display_mode_line about
     this.  */
  it.paragraph_embedding = L2R;

  /* Clear all rows of the tab bar.  */
  for (i = 0; i < FRAME_TAB_BAR_LINES (f); ++i)
    {
      struct glyph_row *row = it.glyph_row + i;
      clear_glyph_row (row);
      row->enabled_p = true;
      row->full_width_p = true;
      row->reversed_p = false;
    }

  /* Display all items of the tab bar.  */
  items = it.f->tab_bar_items;
  int j;
  for (i = 0, j = 0; i < it.f->n_tab_bar_items; ++i, j += TAB_BAR_ITEM_NSLOTS)
    {
      Lisp_Object string = AREF (items, j + TAB_BAR_ITEM_CAPTION);

      /* Stop at nil string.  */
      if (NILP (string))
	break;

      if (it.current_x < it.last_visible_x)
	display_string (NULL, string, Qnil, 0, 0, &it,
			SCHARS (string), 0, 0, STRING_MULTIBYTE (string));
    }

  /* Fill out the line with spaces.  */
  if (it.current_x < it.last_visible_x)
    display_string ("", Qnil, Qnil, 0, 0, &it, -1, 0, 0, -1);

  /* Compute the total height of the lines.  */
  compute_line_metrics (&it);
}

#ifdef HAVE_WINDOW_SYSTEM

/* Set F->desired_tab_bar_string to a Lisp string representing frame
   F's desired tab-bar contents.  F->tab_bar_items must have
   been set up previously by calling prepare_menu_bars.  */

static void
build_desired_tab_bar_string (struct frame *f)
{
  int i;
  Lisp_Object caption;

  caption = Qnil;

  /* Prepare F->desired_tab_bar_string.  Make a new string.  */
  fset_desired_tab_bar_string (f, build_string (""));

  /* Put a `display' property on the string for the captions to display,
     put a `menu_item' property on tab-bar items with a value that
     is the index of the item in F's tab-bar item vector.  */
  for (i = 0; i < f->n_tab_bar_items; ++i)
    {
#define PROP(IDX) \
  AREF (f->tab_bar_items, i * TAB_BAR_ITEM_NSLOTS + (IDX))

      caption = Fcopy_sequence (PROP (TAB_BAR_ITEM_CAPTION));

      /* Put a `display' text property on the string for the caption to
	 display.  Put a `menu-item' property on the string that gives
	 the start of this item's properties in the tab-bar items
	 vector.  */
      AUTO_LIST2 (props, Qmenu_item, make_fixnum (i * TAB_BAR_ITEM_NSLOTS));

      Fadd_text_properties (make_fixnum (0), make_fixnum (SCHARS (caption)),
			    props, caption);

      f->desired_tab_bar_string =
	concat2 (f->desired_tab_bar_string, caption);

#undef PROP
    }
}


/* Display one line of the tab-bar of frame IT->f.

   HEIGHT specifies the desired height of the tab-bar line.
   If the actual height of the glyph row is less than HEIGHT, the
   row's height is increased to HEIGHT, and the icons are centered
   vertically in the new height.

   If HEIGHT is -1, we are counting needed tab-bar lines, so don't
   count a final empty row in case the tab-bar width exactly matches
   the window width.
*/

static void
display_tab_bar_line (struct it *it, int height)
{
  struct glyph_row *row = it->glyph_row;
  int max_x = it->last_visible_x;
  struct glyph *last;

  /* Don't extend on a previously drawn tab bar items (Bug#16058).  */
  clear_glyph_row (row);
  row->enabled_p = true;
  row->y = it->current_y;

  /* Note that this isn't made use of if the face hasn't a box,
     so there's no need to check the face here.  */
  it->start_of_box_run_p = true;

  bool enough = false;
  while (it->current_x < max_x)
    {
      int x, n_glyphs_before, i, nglyphs;
      struct it it_before;

      /* Get the next display element.  */
      if (!get_next_display_element (it))
	{
	  /* Don't count empty row if we are counting needed tab-bar lines.  */
	  if (height < 0 && !it->hpos)
	    return;
	  break;
	}

      /* Produce glyphs.  */
      n_glyphs_before = row->used[TEXT_AREA];
      it_before = *it;

      PRODUCE_GLYPHS (it);

      nglyphs = row->used[TEXT_AREA] - n_glyphs_before;
      i = 0;
      x = it_before.current_x;
      while (i < nglyphs)
	{
	  struct glyph *glyph = row->glyphs[TEXT_AREA] + n_glyphs_before + i;

	  if (x + glyph->pixel_width > max_x)
	    {
	      /* Glyph doesn't fit on line.  Backtrack.  */
	      row->used[TEXT_AREA] = n_glyphs_before;
	      *it = it_before;
	      /* If this is the only glyph on this line, it will never fit on the
		 tab-bar, so skip it.  But ensure there is at least one glyph,
		 so we don't accidentally disable the tab-bar.  */
	      if (n_glyphs_before == 0
		  && (it->vpos > 0 || IT_STRING_CHARPOS (*it) < it->end_charpos-1))
		break;
	      goto out;
	    }

	  ++it->hpos;
	  x += glyph->pixel_width;
	  ++i;
	}

      enough = ITERATOR_AT_END_OF_LINE_P (it);
      set_iterator_to_next (it, true);

      /* Stop at line end.  */
      if (enough)
	break;
    }

 out:;

  row->displays_text_p = row->used[TEXT_AREA] != 0;

  /* Use default face for the border below the tab bar.

     FIXME: When auto-resize-tab-bars is grow-only, there is
     no additional border below the possibly empty tab-bar lines.
     So to make the extra empty lines look "normal", we have to
     use the tab-bar face for the border too.  */
  if (!MATRIX_ROW_DISPLAYS_TEXT_P (row)
      && !EQ (Vauto_resize_tab_bars, Qgrow_only))
    it->face_id = DEFAULT_FACE_ID;

  extend_face_to_end_of_line (it);
  last = row->glyphs[TEXT_AREA] + row->used[TEXT_AREA] - 1;
  last->right_box_line_p = true;
  if (last == row->glyphs[TEXT_AREA])
    last->left_box_line_p = true;

  /* Make line the desired height and center it vertically.  */
  if ((height -= it->max_ascent + it->max_descent) > 0)
    {
      /* Don't add more than one line height.  */
      height %= FRAME_LINE_HEIGHT (it->f);
      it->max_ascent += height / 2;
      it->max_descent += (height + 1) / 2;
    }

  compute_line_metrics (it);

  /* If line is empty, make it occupy the rest of the tab-bar.  */
  if (!MATRIX_ROW_DISPLAYS_TEXT_P (row))
    {
      row->height = row->phys_height = it->last_visible_y - row->y;
      row->visible_height = row->height;
      row->ascent = row->phys_ascent = 0;
      row->extra_line_spacing = 0;
    }

  row->full_width_p = true;
  row->continued_p = false;
  row->truncated_on_left_p = false;
  row->truncated_on_right_p = false;

  it->current_x = it->hpos = 0;
  it->current_y += row->height;
  ++it->vpos;
  ++it->glyph_row;
}


/* Value is the number of pixels needed to make all tab-bar items of
   frame F visible.  The actual number of glyph rows needed is
   returned in *N_ROWS if non-NULL.  */
static int
tab_bar_height (struct frame *f, int *n_rows, bool pixelwise)
{
  struct window *w = XWINDOW (f->tab_bar_window);
  struct it it;
  /* tab_bar_height is called from redisplay_tab_bar after building
     the desired matrix, so use (unused) mode-line row as temporary row to
     avoid destroying the first tab-bar row.  */
  struct glyph_row *temp_row = MATRIX_MODE_LINE_ROW (w->desired_matrix);

  /* Initialize an iterator for iteration over
     F->desired_tab_bar_string in the tab-bar window of frame F.  */
  init_iterator (&it, w, -1, -1, temp_row, TAB_BAR_FACE_ID);
  temp_row->reversed_p = false;
  it.first_visible_x = 0;
  it.last_visible_x = WINDOW_PIXEL_WIDTH (w);
  reseat_to_string (&it, NULL, f->desired_tab_bar_string,
                    0, 0, 0, STRING_MULTIBYTE (f->desired_tab_bar_string));
  it.paragraph_embedding = L2R;

  clear_glyph_row (temp_row);
  while (!ITERATOR_AT_END_P (&it))
    {
      it.glyph_row = temp_row;
      display_tab_bar_line (&it, -1);
    }
  clear_glyph_row (temp_row);

  /* f->n_tab_bar_rows == 0 means "unknown"; -1 means no tab-bar.  */
  if (n_rows)
    *n_rows = it.vpos > 0 ? it.vpos : -1;

  if (pixelwise)
    return it.current_y;
  else
    return (it.current_y + FRAME_LINE_HEIGHT (f) - 1) / FRAME_LINE_HEIGHT (f);
}

DEFUN ("tab-bar-height", Ftab_bar_height, Stab_bar_height,
       0, 2, 0,
       doc: /* Return the number of lines occupied by the tab bar of FRAME.
If FRAME is nil or omitted, use the selected frame.  Optional argument
PIXELWISE non-nil means return the height of the tab bar in pixels.  */)
  (Lisp_Object frame, Lisp_Object pixelwise)
{
  int height = 0;

  struct frame *f = decode_any_frame (frame);

  if (WINDOWP (f->tab_bar_window)
      && WINDOW_PIXEL_HEIGHT (XWINDOW (f->tab_bar_window)) > 0)
    {
      update_tab_bar (f, true);
      if (f->n_tab_bar_items)
	{
	  build_desired_tab_bar_string (f);
	  height = tab_bar_height (f, NULL, !NILP (pixelwise));
	}
    }

  return make_fixnum (height);
}


/* Display the tab-bar of frame F.  Value is true if tab-bar's
   height should be changed.  */
static bool
redisplay_tab_bar (struct frame *f)
{
  struct window *w;
  struct it it;
  struct glyph_row *row;

  f->tab_bar_redisplayed = true;

  /* If frame hasn't a tab-bar window or if it is zero-height, don't
     do anything.  This means you must start with tab-bar-lines
     non-zero to get the auto-sizing effect.  Or in other words, you
     can turn off tab-bars by specifying tab-bar-lines zero.  */
  if (!WINDOWP (f->tab_bar_window)
      || (w = XWINDOW (f->tab_bar_window),
          WINDOW_TOTAL_LINES (w) == 0))
    {
      /* Even if we do not display a tab bar initially, still pretend
	 that we have resized it.  This avoids that a later activation
	 of the tab bar resizes the frame, despite of the fact that the
	 setting of 'frame-inhibit-implied-resize' should inhibit it
	 (Bug#52986).  */
      f->tab_bar_resized = true;

      return false;
    }

  /* Build a string that represents the contents of the tab-bar.  */
  build_desired_tab_bar_string (f);

  int new_nrows;
  int new_height = tab_bar_height (f, &new_nrows, true);

  if (f->n_tab_bar_rows == 0)
    {
      f->n_tab_bar_rows = new_nrows;
      if (new_height != WINDOW_PIXEL_HEIGHT (w))
	frame_default_tab_bar_height = new_height;
    }

  /* If new_height or new_nrows indicate that we need to enlarge or
     shrink the tab-bar window, we can return right away.  */
  if (new_nrows > f->n_tab_bar_rows
      || (EQ (Vauto_resize_tab_bars, Qgrow_only)
	  && !f->minimize_tab_bar_window_p
	  && new_height > WINDOW_PIXEL_HEIGHT (w))
      || (! EQ (Vauto_resize_tab_bars, Qgrow_only)
	  && new_height < WINDOW_PIXEL_HEIGHT (w)))
    {
      if (FRAME_TERMINAL (f)->change_tab_bar_height_hook)
	FRAME_TERMINAL (f)->change_tab_bar_height_hook (f, new_height);
      if (new_nrows != f->n_tab_bar_rows)
	f->n_tab_bar_rows = new_nrows;
      clear_glyph_matrix (w->desired_matrix);
      f->fonts_changed = true;
      return true;
    }

  /* Set up an iterator for the tab-bar window.  */
  init_iterator (&it, w, -1, -1, w->desired_matrix->rows, TAB_BAR_FACE_ID);
  it.first_visible_x = 0;
  it.last_visible_x = WINDOW_PIXEL_WIDTH (w);
  row = it.glyph_row;
  row->reversed_p = false;
  reseat_to_string (&it, NULL, f->desired_tab_bar_string, 0, 0, 0,
                    STRING_MULTIBYTE (f->desired_tab_bar_string));
  /* FIXME: This should be controlled by a user option.  But it
     doesn't make sense to have an R2L tab bar if the menu bar cannot
     be drawn also R2L, and making the menu bar R2L is tricky due
     tabkit-specific code that implements it.  If an R2L tab bar is
     ever supported, display_tab_bar_line should also be augmented to
     call unproduce_glyphs like display_line and display_string
     do.  */
  it.paragraph_embedding = L2R;

  /* Display as many lines as needed to display all tab-bar items.  */

  if (f->n_tab_bar_rows > 0)
    {
      int border, rows, height, extra;

      if (TYPE_RANGED_FIXNUMP (int, Vtab_bar_border))
	border = XFIXNUM (Vtab_bar_border);
      else if (EQ (Vtab_bar_border, Qinternal_border_width))
	border = FRAME_INTERNAL_BORDER_WIDTH (f);
      else if (EQ (Vtab_bar_border, Qborder_width))
	border = f->border_width;
      else
	border = 0;
      if (border < 0)
	border = 0;

      rows = f->n_tab_bar_rows;
      height = max (1, (it.last_visible_y - border) / rows);
      extra = it.last_visible_y - border - height * rows;

      while (it.current_y < it.last_visible_y)
	{
	  int h = 0;
	  if (extra > 0 && rows-- > 0)
	    {
	      h = (extra + rows - 1) / rows;
	      extra -= h;
	    }
	  display_tab_bar_line (&it, height + h);
	}
    }
  else
    {
      while (it.current_y < it.last_visible_y)
	display_tab_bar_line (&it, 0);
    }

  /* It doesn't make much sense to try scrolling in the tab-bar
     window, so don't do it.  */
  w->desired_matrix->no_scrolling_p = true;
  w->must_be_updated_p = true;

  if (!NILP (Vauto_resize_tab_bars))
    {
      bool change_height_p = false;

      /* If we couldn't display everything, change the tab-bar's
	 height if there is room for more.  */
      if (IT_STRING_CHARPOS (it) < it.end_charpos)
	change_height_p = true;

      /* We subtract 1 because display_tab_bar_line advances the
	 glyph_row pointer before returning to its caller.  We want to
	 examine the last glyph row produced by
	 display_tab_bar_line.  */
      row = it.glyph_row - 1;

      /* If there are blank lines at the end, except for a partially
	 visible blank line at the end that is smaller than
	 FRAME_LINE_HEIGHT, change the tab-bar's height.  */
      if (!MATRIX_ROW_DISPLAYS_TEXT_P (row)
	  && row->height >= FRAME_LINE_HEIGHT (f))
	change_height_p = true;

      /* If row displays tab-bar items, but is partially visible,
	 change the tab-bar's height.  */
      if (MATRIX_ROW_DISPLAYS_TEXT_P (row)
	  && MATRIX_ROW_BOTTOM_Y (row) > it.last_visible_y)
	change_height_p = true;

      /* Resize windows as needed by changing the `tab-bar-lines'
	 frame parameter.  */
      if (change_height_p)
	{
	  int nrows;
	  int new_height = tab_bar_height (f, &nrows, true);

	  change_height_p = ((EQ (Vauto_resize_tab_bars, Qgrow_only)
			      && !f->minimize_tab_bar_window_p)
			     ? (new_height > WINDOW_PIXEL_HEIGHT (w))
			     : (new_height != WINDOW_PIXEL_HEIGHT (w)));
	  f->minimize_tab_bar_window_p = false;

	  if (change_height_p)
	    {
              if (FRAME_TERMINAL (f)->change_tab_bar_height_hook)
                FRAME_TERMINAL (f)->change_tab_bar_height_hook (f, new_height);
	      frame_default_tab_bar_height = new_height;
	      clear_glyph_matrix (w->desired_matrix);
	      f->n_tab_bar_rows = nrows;
	      f->fonts_changed = true;

	      return true;
	    }
	}
    }

  f->minimize_tab_bar_window_p = false;
  return false;
}

/* Get information about the tab-bar item which is displayed in GLYPH
   on frame F.  Return in *PROP_IDX the index where tab-bar item
   properties start in F->tab_bar_items.  Return in CLOSE_P an
   indication whether the click was on the close-tab icon of the tab.
   Value is false if GLYPH doesn't display a tab-bar item.  */

static bool
tab_bar_item_info (struct frame *f, struct glyph *glyph,
		   int *prop_idx, bool *close_p)
{
  Lisp_Object prop;
  ptrdiff_t charpos;

  /* This function can be called asynchronously, which means we must
     exclude any possibility that Fget_text_property signals an
     error.  */
  charpos = min (SCHARS (f->current_tab_bar_string), glyph->charpos);
  charpos = max (0, charpos);

  /* Get the text property `menu-item' at pos. The value of that
     property is the start index of this item's properties in
     F->tab_bar_items.  */
  prop = Fget_text_property (make_fixnum (charpos),
			     Qmenu_item, f->current_tab_bar_string);
  if (! FIXNUMP (prop))
    return false;
  *prop_idx = XFIXNUM (prop);

  *close_p = !NILP (Fget_text_property (make_fixnum (charpos),
                                        Qclose_tab,
                                        f->current_tab_bar_string));

  return true;
}


/* Get information about the tab-bar item at position X/Y on frame F.
   Return in *GLYPH a pointer to the glyph of the tab-bar item in
   the current matrix of the tab-bar window of F, or NULL if not
   on a tab-bar item.  Return in *PROP_IDX the index of the tab-bar
   item in F->tab_bar_items.  Value is

   -1	if X/Y is not on a tab-bar item
   0	if X/Y is on the same item that was highlighted before.
   1	otherwise.  */

static int
get_tab_bar_item (struct frame *f, int x, int y, struct glyph **glyph,
		   int *hpos, int *vpos, int *prop_idx, bool *close_p)
{
  struct window *w = XWINDOW (f->tab_bar_window);
  int area;

  /* Find the glyph under X/Y.  */
  *glyph = x_y_to_hpos_vpos (w, x, y, hpos, vpos, 0, 0, &area);
  if (*glyph == NULL)
    return -1;

  /* Get the start of this tab-bar item's properties in
     f->tab_bar_items.  */
  if (!tab_bar_item_info (f, *glyph, prop_idx, close_p))
    return -1;

  return *prop_idx == f->last_tab_bar_item ? 0 : 1;
}


/* EXPORT:
   Handle mouse button event on the tab-bar of frame F, at
   frame-relative coordinates X/Y.  DOWN_P is true for a button press,
   false for button release.  MODIFIERS is event modifiers for button
   release.  */

Lisp_Object
handle_tab_bar_click (struct frame *f, int x, int y, bool down_p,
		      int modifiers)
{
  Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (f);
  struct window *w = XWINDOW (f->tab_bar_window);
  int hpos, vpos, prop_idx;
  bool close_p;
  struct glyph *glyph;
  Lisp_Object enabled_p;
  int ts;

  frame_to_window_pixel_xy (w, &x, &y);
  ts = get_tab_bar_item (f, x, y, &glyph, &hpos, &vpos, &prop_idx, &close_p);
  if (ts == -1)
    return Fcons (Qtab_bar, Qnil);

  /* If item is disabled, do nothing.  */
  enabled_p = AREF (f->tab_bar_items, prop_idx + TAB_BAR_ITEM_ENABLED_P);
  if (NILP (enabled_p))
    return Qnil;

  if (down_p)
    {
      /* Show the clicked button in pressed state.  */
      if (!NILP (Vmouse_highlight))
	show_mouse_face (hlinfo, DRAW_IMAGE_SUNKEN);
      f->last_tab_bar_item = prop_idx; /* record the pressed tab */
    }
  else
    {
      /* Show item in released state.  */
      if (!NILP (Vmouse_highlight))
	show_mouse_face (hlinfo, DRAW_IMAGE_RAISED);
      f->last_tab_bar_item = -1;
    }

  Lisp_Object caption =
    Fcopy_sequence (AREF (f->tab_bar_items, prop_idx + TAB_BAR_ITEM_CAPTION));

  AUTO_LIST2 (props, Qmenu_item,
	      list3 (AREF (f->tab_bar_items, prop_idx + TAB_BAR_ITEM_KEY),
		     AREF (f->tab_bar_items, prop_idx + TAB_BAR_ITEM_BINDING),
		     close_p ? Qt : Qnil));

  Fadd_text_properties (make_fixnum (0), make_fixnum (SCHARS (caption)),
			props, caption);

  return Fcons (Qtab_bar, Fcons (caption, make_fixnum (0)));
}


/* Possibly highlight a tab-bar item on frame F when mouse moves to
   tab-bar window-relative coordinates X/Y.  Called from
   note_mouse_highlight.  */

static void
note_tab_bar_highlight (struct frame *f, int x, int y)
{
  Lisp_Object window = f->tab_bar_window;
  struct window *w = XWINDOW (window);
  Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (f);
  int hpos, vpos;
  struct glyph *glyph;
  struct glyph_row *row;
  int i;
  Lisp_Object enabled_p;
  int prop_idx;
  bool close_p;
  enum draw_glyphs_face draw = DRAW_IMAGE_RAISED;
  int rc;

  /* Function note_mouse_highlight is called with negative X/Y
     values when mouse moves outside of the frame.  */
  if (x <= 0 || y <= 0)
    {
      clear_mouse_face (hlinfo);
      return;
    }

  rc = get_tab_bar_item (f, x, y, &glyph, &hpos, &vpos, &prop_idx, &close_p);
  if (rc < 0)
    {
      /* Not on tab-bar item.  */
      clear_mouse_face (hlinfo);
      return;
    }
  else if (rc == 0)
    /* On same tab-bar item as before.  */
    goto set_help_echo;

  clear_mouse_face (hlinfo);

  bool mouse_down_p = false;
  /* Mouse is down, but on different tab-bar item?  Or alternatively,
     the mouse might've been pressed somewhere we don't know about,
     and then have moved onto the tab bar.  In this case,
     last_tab_bar_item is -1, so we DTRT and behave like other
     programs by displaying the item as sunken. */
  Display_Info *dpyinfo = FRAME_DISPLAY_INFO (f);
  mouse_down_p = (gui_mouse_grabbed (dpyinfo)
		  && f == dpyinfo->last_mouse_frame);

  if (mouse_down_p && f->last_tab_bar_item != prop_idx
      && f->last_tab_bar_item != -1)
    return;
  draw = mouse_down_p ? DRAW_IMAGE_SUNKEN : DRAW_IMAGE_RAISED;

  /* If tab-bar item is not enabled, don't highlight it.  */
  enabled_p = AREF (f->tab_bar_items, prop_idx + TAB_BAR_ITEM_ENABLED_P);
  if (!NILP (enabled_p) && !NILP (Vmouse_highlight))
    {
      /* Compute the x-position of the glyph.  In front and past the
	 image is a space.  We include this in the highlighted area.  */
      row = MATRIX_ROW (w->current_matrix, vpos);
      for (i = x = 0; i < hpos; ++i)
	x += row->glyphs[TEXT_AREA][i].pixel_width;

      /* Record this as the current active region.  */
      hlinfo->mouse_face_beg_col = hpos;
      hlinfo->mouse_face_beg_row = vpos;
      hlinfo->mouse_face_beg_x = x;
      hlinfo->mouse_face_past_end = false;

      hlinfo->mouse_face_end_col = hpos + 1;
      hlinfo->mouse_face_end_row = vpos;
      hlinfo->mouse_face_end_x = x + glyph->pixel_width;
      hlinfo->mouse_face_window = window;
      hlinfo->mouse_face_face_id = TAB_BAR_FACE_ID;

      /* Display it as active.  */
      show_mouse_face (hlinfo, draw);
    }

 set_help_echo:

  /* Set help_echo_string to a help string to display for this tab-bar item.
     XTread_socket does the rest.  */
  help_echo_object = help_echo_window = Qnil;
  help_echo_pos = -1;
  help_echo_string = AREF (f->tab_bar_items, prop_idx + TAB_BAR_ITEM_HELP);
  if (NILP (help_echo_string))
    help_echo_string = AREF (f->tab_bar_items, prop_idx + TAB_BAR_ITEM_CAPTION);
}

#endif /* HAVE_WINDOW_SYSTEM */

/* Find the tab-bar item at X coordinate and return its information.  */
static Lisp_Object
tty_get_tab_bar_item (struct frame *f, int x, int *prop_idx, bool *close_p)
{
  ptrdiff_t clen = 0;

  for (int i = 0; i < f->n_tab_bar_items; i++)
    {
      Lisp_Object caption = AREF (f->tab_bar_items, (i * TAB_BAR_ITEM_NSLOTS
						     + TAB_BAR_ITEM_CAPTION));
      if (NILP (caption))
	return Qnil;
      clen += SCHARS (caption);
      if (x < clen)
	{
	  *prop_idx = i;
	  *close_p = !NILP (Fget_text_property (make_fixnum (SCHARS (caption)
							     - (clen - x)),
						Qclose_tab,
						caption));
	  return caption;
	}
    }
  return Qnil;
}

/* Handle a mouse click at X/Y on the tab bar of TTY frame F.  If the
   click was on the tab bar and was handled, populate the EVENT
   structure, store it in keyboard queue, and return true; otherwise
   return false.  MODIFIERS are event modifiers for generating the tab
   release event.  */
Lisp_Object
tty_handle_tab_bar_click (struct frame *f, int x, int y, bool down_p,
			  struct input_event *event)
{
  /* Did they click on the tab bar?  */
  if (y < FRAME_MENU_BAR_LINES (f)
      || y >= FRAME_MENU_BAR_LINES (f) + FRAME_TAB_BAR_LINES (f))
    return Qnil;

  /* Find the tab-bar item where the X,Y coordinates belong.  */
  int prop_idx;
  bool close_p;
  Lisp_Object caption = tty_get_tab_bar_item (f, x, &prop_idx, &close_p);

  if (NILP (caption))
    return Qnil;

  if (NILP (AREF (f->tab_bar_items,
		  prop_idx * TAB_BAR_ITEM_NSLOTS + TAB_BAR_ITEM_ENABLED_P)))
    return Qnil;

  if (down_p)
    f->last_tab_bar_item = prop_idx;
  else
    f->last_tab_bar_item = -1;

  caption = Fcopy_sequence (caption);

  AUTO_LIST2 (props, Qmenu_item,
	      list3 (AREF (f->tab_bar_items, prop_idx * TAB_BAR_ITEM_NSLOTS
			   + TAB_BAR_ITEM_KEY),
		     AREF (f->tab_bar_items, prop_idx * TAB_BAR_ITEM_NSLOTS
			   + TAB_BAR_ITEM_BINDING),
		     close_p ? Qt : Qnil));

  Fadd_text_properties (make_fixnum (0), make_fixnum (SCHARS (caption)),
			props, caption);

  return Fcons (Qtab_bar, Fcons (caption, make_fixnum (0)));
}



/***********************************************************************
			       Tool-bars
 ***********************************************************************/

#ifdef HAVE_WINDOW_SYSTEM

/* Update the tool-bar item list for frame F.  This has to be done
   before we start to fill in any display lines.  Called from
   prepare_menu_bars.  If SAVE_MATCH_DATA, we must save
   and restore it here.  */

static void
update_tool_bar (struct frame *f, bool save_match_data)
{
#ifdef HAVE_EXT_TOOL_BAR
  bool do_update = FRAME_EXTERNAL_TOOL_BAR (f);
#else
  bool do_update = (WINDOWP (f->tool_bar_window)
		    && WINDOW_TOTAL_LINES (XWINDOW (f->tool_bar_window)) > 0);
#endif

  if (do_update)
    {
      Lisp_Object window;
      struct window *w;

      window = FRAME_SELECTED_WINDOW (f);
      w = XWINDOW (window);

      /* If the user has switched buffers or windows, we need to
	 recompute to reflect the new bindings.  But we'll
	 recompute when update_mode_lines is set too; that means
	 that people can use force-mode-line-update to request
	 that the menu bar be recomputed.  The adverse effect on
	 the rest of the redisplay algorithm is about the same as
	 windows_or_buffers_changed anyway.  */
      if (windows_or_buffers_changed
	  || w->update_mode_line
	  || update_mode_lines
	  || window_buffer_changed (w))
	{
	  struct buffer *prev = current_buffer;
	  specpdl_ref count = SPECPDL_INDEX ();
	  Lisp_Object frame, new_tool_bar;
          int new_n_tool_bar;

	  /* Set current_buffer to the buffer of the selected
	     window of the frame, so that we get the right local
	     keymaps.  */
	  set_buffer_internal_1 (XBUFFER (w->contents));

	  /* Save match data, if we must.  */
	  if (save_match_data)
	    record_unwind_save_match_data ();

	  /* Make sure that we don't accidentally use bogus keymaps.  */
	  if (NILP (Voverriding_local_map_menu_flag))
	    {
	      specbind (Qoverriding_terminal_local_map, Qnil);
	      specbind (Qoverriding_local_map, Qnil);
	    }

	  /* We must temporarily set the selected frame to this frame
	     before calling tool_bar_items, because the calculation of
	     the tool-bar keymap uses the selected frame (see
	     `tool-bar-make-keymap' in tool-bar.el).  */
	  eassert (EQ (selected_window,
		       /* Since we only explicitly preserve selected_frame,
			  check that selected_window would be redundant.  */
		       XFRAME (selected_frame)->selected_window));
	  record_unwind_protect (restore_selected_window, selected_window);
	  XSETFRAME (frame, f);
	  selected_frame = frame;
	  selected_window = FRAME_SELECTED_WINDOW (f);

	  /* Build desired tool-bar items from keymaps.  */
          new_tool_bar
	    = tool_bar_items (Fcopy_sequence (f->tool_bar_items),
			      &new_n_tool_bar);

	  /* Redisplay the tool-bar if we changed it.  */
	  if (new_n_tool_bar != f->n_tool_bar_items
	      || NILP (Fequal (new_tool_bar, f->tool_bar_items)))
            {
              /* Redisplay that happens asynchronously due to an expose event
                 may access f->tool_bar_items.  Make sure we update both
                 variables within BLOCK_INPUT so no such event interrupts.  */
              block_input ();
              fset_tool_bar_items (f, new_tool_bar);
              f->n_tool_bar_items = new_n_tool_bar;
              w->update_mode_line = true;
              unblock_input ();
            }

	  unbind_to (count, Qnil);
	  set_buffer_internal_1 (prev);
	}
    }
}

#ifndef HAVE_EXT_TOOL_BAR

/* Set F->desired_tool_bar_string to a Lisp string representing frame
   F's desired tool-bar contents.  F->tool_bar_items must have
   been set up previously by calling prepare_menu_bars.  */

static void
build_desired_tool_bar_string (struct frame *f)
{
  int i, size, size_needed;
  Lisp_Object image, plist;

  image = plist = Qnil;

  /* Prepare F->desired_tool_bar_string.  If we can reuse it, do so.
     Otherwise, make a new string.  */

  /* The size of the string we might be able to reuse.  */
  size = (STRINGP (f->desired_tool_bar_string)
	  ? SCHARS (f->desired_tool_bar_string)
	  : 0);

  /* We need one space in the string for each image.  */
  size_needed = f->n_tool_bar_items;

  /* Reuse f->desired_tool_bar_string, if possible.  */
  if (size < size_needed || NILP (f->desired_tool_bar_string))
    fset_desired_tool_bar_string
      (f, Fmake_string (make_fixnum (size_needed), make_fixnum (' '), Qnil));
  else
    {
      AUTO_LIST4 (props, Qdisplay, Qnil, Qmenu_item, Qnil);
      Fremove_text_properties (make_fixnum (0), make_fixnum (size),
			       props, f->desired_tool_bar_string);
    }

  /* Put a `display' property on the string for the images to display,
     put a `menu_item' property on tool-bar items with a value that
     is the index of the item in F's tool-bar item vector.  */
  for (i = 0; i < f->n_tool_bar_items; ++i)
    {
#define PROP(IDX) \
  AREF (f->tool_bar_items, i * TOOL_BAR_ITEM_NSLOTS + (IDX))

      bool enabled_p = !NILP (PROP (TOOL_BAR_ITEM_ENABLED_P));
      bool selected_p = !NILP (PROP (TOOL_BAR_ITEM_SELECTED_P));
      int hmargin, vmargin, relief, idx, end;

      /* If image is a vector, choose the image according to the
	 button state.  */
      image = PROP (TOOL_BAR_ITEM_IMAGES);
      if (VECTORP (image))
	{
	  if (enabled_p)
	    idx = (selected_p
		   ? TOOL_BAR_IMAGE_ENABLED_SELECTED
		   : TOOL_BAR_IMAGE_ENABLED_DESELECTED);
	  else
	    idx = (selected_p
		   ? TOOL_BAR_IMAGE_DISABLED_SELECTED
		   : TOOL_BAR_IMAGE_DISABLED_DESELECTED);

	  eassert (ASIZE (image) >= idx);
	  image = AREF (image, idx);
	}
      else
	idx = -1;

      /* Ignore invalid image specifications.  */
      if (!valid_image_p (image))
	continue;

      /* Display the tool-bar button pressed, or depressed.  */
      plist = Fcopy_sequence (XCDR (image));

      /* Compute margin and relief to draw.  */
      relief = (tool_bar_button_relief >= 0
		? min (tool_bar_button_relief,
		       min (INT_MAX, MOST_POSITIVE_FIXNUM))
		: DEFAULT_TOOL_BAR_BUTTON_RELIEF);
      hmargin = vmargin = relief;

      if (RANGED_FIXNUMP (1, Vtool_bar_button_margin,
			   INT_MAX - max (hmargin, vmargin)))
	{
	  hmargin += XFIXNAT (Vtool_bar_button_margin);
	  vmargin += XFIXNAT (Vtool_bar_button_margin);
	}
      else if (CONSP (Vtool_bar_button_margin))
	{
	  if (RANGED_FIXNUMP (1, XCAR (Vtool_bar_button_margin),
			       INT_MAX - hmargin))
	    hmargin += XFIXNAT (XCAR (Vtool_bar_button_margin));

	  if (RANGED_FIXNUMP (1, XCDR (Vtool_bar_button_margin),
			       INT_MAX - vmargin))
	    vmargin += XFIXNAT (XCDR (Vtool_bar_button_margin));
	}

      if (auto_raise_tool_bar_buttons_p)
	{
	  /* Add a `:relief' property to the image spec if the item is
	     selected.  */
	  if (selected_p)
	    {
	      plist = plist_put (plist, QCrelief, make_fixnum (-relief));
	      hmargin -= relief;
	      vmargin -= relief;
	    }
	}
      else
	{
	  /* If image is selected, display it pressed, i.e. with a
	     negative relief.  If it's not selected, display it with a
	     raised relief.  */
	  plist = plist_put (plist, QCrelief,
			     (selected_p
			      ? make_fixnum (-relief)
			      : make_fixnum (relief)));
	  hmargin -= relief;
	  vmargin -= relief;
	}

      /* Put a margin around the image.  */
      if (hmargin || vmargin)
	{
	  if (hmargin == vmargin)
	    plist = plist_put (plist, QCmargin, make_fixnum (hmargin));
	  else
	    plist = plist_put (plist, QCmargin,
			       Fcons (make_fixnum (hmargin),
				      make_fixnum (vmargin)));
	}

      /* If button is not enabled, and we don't have special images
	 for the disabled state, make the image appear disabled by
	 applying an appropriate algorithm to it.  */
      if (!enabled_p && idx < 0)
	plist = plist_put (plist, QCconversion, Qdisabled);

      /* Put a `display' text property on the string for the image to
	 display.  Put a `menu-item' property on the string that gives
	 the start of this item's properties in the tool-bar items
	 vector.  */
      image = Fcons (Qimage, plist);
      AUTO_LIST4 (props, Qdisplay, image, Qmenu_item,
		  make_fixnum (i * TOOL_BAR_ITEM_NSLOTS));

      /* Let the last image hide all remaining spaces in the tool bar
         string.  The string can be longer than needed when we reuse a
         previous string.  */
      if (i + 1 == f->n_tool_bar_items)
	end = SCHARS (f->desired_tool_bar_string);
      else
	end = i + 1;
      Fadd_text_properties (make_fixnum (i), make_fixnum (end),
			    props, f->desired_tool_bar_string);
#undef PROP
    }
}


/* Display one line of the tool-bar of frame IT->f.

   HEIGHT specifies the desired height of the tool-bar line.
   If the actual height of the glyph row is less than HEIGHT, the
   row's height is increased to HEIGHT, and the icons are centered
   vertically in the new height.

   If HEIGHT is -1, we are counting needed tool-bar lines, so don't
   count a final empty row in case the tool-bar width exactly matches
   the window width.
*/

static void
display_tool_bar_line (struct it *it, int height)
{
  struct glyph_row *row = it->glyph_row;
  int max_x = it->last_visible_x;
  struct glyph *last;

  /* Don't extend on a previously drawn tool bar items (Bug#16058).  */
  clear_glyph_row (row);
  row->enabled_p = true;
  row->y = it->current_y;

  /* Note that this isn't made use of if the face hasn't a box,
     so there's no need to check the face here.  */
  it->start_of_box_run_p = true;

  while (it->current_x < max_x)
    {
      int x, n_glyphs_before, i, nglyphs;
      struct it it_before;

      /* Get the next display element.  */
      if (!get_next_display_element (it))
	{
	  /* Don't count empty row if we are counting needed tool-bar lines.  */
	  if (height < 0 && !it->hpos)
	    return;
	  break;
	}

      /* Produce glyphs.  */
      n_glyphs_before = row->used[TEXT_AREA];
      it_before = *it;

      PRODUCE_GLYPHS (it);

      nglyphs = row->used[TEXT_AREA] - n_glyphs_before;
      i = 0;
      x = it_before.current_x;
      while (i < nglyphs)
	{
	  struct glyph *glyph = row->glyphs[TEXT_AREA] + n_glyphs_before + i;

	  if (x + glyph->pixel_width > max_x)
	    {
	      /* Glyph doesn't fit on line.  Backtrack.  */
	      row->used[TEXT_AREA] = n_glyphs_before;
	      *it = it_before;
	      /* If this is the only glyph on this line, it will never fit on the
		 tool-bar, so skip it.  But ensure there is at least one glyph,
		 so we don't accidentally disable the tool-bar.  */
	      if (n_glyphs_before == 0
		  && (it->vpos > 0 || IT_STRING_CHARPOS (*it) < it->end_charpos-1))
		break;
	      goto out;
	    }

	  ++it->hpos;
	  x += glyph->pixel_width;
	  ++i;
	}

      /* Stop at line end.  */
      if (ITERATOR_AT_END_OF_LINE_P (it))
	break;

      set_iterator_to_next (it, true);
    }

 out:;

  row->displays_text_p = row->used[TEXT_AREA] != 0;

  /* Use default face for the border below the tool bar.

     FIXME: When auto-resize-tool-bars is grow-only, there is
     no additional border below the possibly empty tool-bar lines.
     So to make the extra empty lines look "normal", we have to
     use the tool-bar face for the border too.  */
  if (!MATRIX_ROW_DISPLAYS_TEXT_P (row)
      && !EQ (Vauto_resize_tool_bars, Qgrow_only))
    it->face_id = DEFAULT_FACE_ID;

  extend_face_to_end_of_line (it);
  last = row->glyphs[TEXT_AREA] + row->used[TEXT_AREA] - 1;
  last->right_box_line_p = true;
  if (last == row->glyphs[TEXT_AREA])
    last->left_box_line_p = true;

  /* Make line the desired height and center it vertically.  */
  if ((height -= it->max_ascent + it->max_descent) > 0)
    {
      /* Don't add more than one line height.  */
      height %= FRAME_LINE_HEIGHT (it->f);
      it->max_ascent += height / 2;
      it->max_descent += (height + 1) / 2;
    }

  compute_line_metrics (it);

  /* If line is empty, make it occupy the rest of the tool-bar.  */
  if (!MATRIX_ROW_DISPLAYS_TEXT_P (row))
    {
      row->height = row->phys_height = it->last_visible_y - row->y;
      row->visible_height = row->height;
      row->ascent = row->phys_ascent = 0;
      row->extra_line_spacing = 0;
    }

  row->full_width_p = true;
  row->continued_p = false;
  row->truncated_on_left_p = false;
  row->truncated_on_right_p = false;

  it->current_x = it->hpos = 0;
  it->current_y += row->height;
  ++it->vpos;
  ++it->glyph_row;
}


/* Value is the number of pixels needed to make all tool-bar items of
   frame F visible.  The actual number of glyph rows needed is
   returned in *N_ROWS if non-NULL.  */
static int
tool_bar_height (struct frame *f, int *n_rows, bool pixelwise)
{
  struct window *w = XWINDOW (f->tool_bar_window);
  struct it it;
  /* tool_bar_height is called from redisplay_tool_bar after building
     the desired matrix, so use (unused) mode-line row as temporary row to
     avoid destroying the first tool-bar row.  */
  struct glyph_row *temp_row = MATRIX_MODE_LINE_ROW (w->desired_matrix);

  /* Initialize an iterator for iteration over
     F->desired_tool_bar_string in the tool-bar window of frame F.  */
  init_iterator (&it, w, -1, -1, temp_row, TOOL_BAR_FACE_ID);
  temp_row->reversed_p = false;
  it.first_visible_x = 0;
  it.last_visible_x = WINDOW_PIXEL_WIDTH (w);
  reseat_to_string (&it, NULL, f->desired_tool_bar_string,
		    0, 0, 0, STRING_MULTIBYTE (f->desired_tool_bar_string));
  it.paragraph_embedding = L2R;

  while (!ITERATOR_AT_END_P (&it))
    {
      clear_glyph_row (temp_row);
      it.glyph_row = temp_row;
      display_tool_bar_line (&it, -1);
    }
  clear_glyph_row (temp_row);

  /* f->n_tool_bar_rows == 0 means "unknown"; -1 means no tool-bar.  */
  if (n_rows)
    *n_rows = it.vpos > 0 ? it.vpos : -1;

  if (pixelwise)
    return it.current_y;
  else
    return (it.current_y + FRAME_LINE_HEIGHT (f) - 1) / FRAME_LINE_HEIGHT (f);
}

#endif /* ! (HAVE_EXT_TOOL_BAR) */

DEFUN ("tool-bar-height", Ftool_bar_height, Stool_bar_height,
       0, 2, 0,
       doc: /* Return the number of lines occupied by the tool bar of FRAME.
If FRAME is nil or omitted, use the selected frame.  Optional argument
PIXELWISE non-nil means return the height of the tool bar in pixels.  */)
  (Lisp_Object frame, Lisp_Object pixelwise)
{
  int height = 0;

#ifndef HAVE_EXT_TOOL_BAR
  struct frame *f = decode_any_frame (frame);

  if (WINDOWP (f->tool_bar_window)
      && WINDOW_PIXEL_HEIGHT (XWINDOW (f->tool_bar_window)) > 0)
    {
      update_tool_bar (f, true);
      if (f->n_tool_bar_items)
	{
	  build_desired_tool_bar_string (f);
	  height = tool_bar_height (f, NULL, !NILP (pixelwise));
	}
    }
#endif

  return make_fixnum (height);
}

#ifndef HAVE_EXT_TOOL_BAR

/* Display the internal tool-bar of frame F.  Value is true if
   tool-bar's height should be changed.  */
static bool
redisplay_tool_bar (struct frame *f)
{
  struct window *w;
  struct it it;
  struct glyph_row *row;

  f->tool_bar_redisplayed = true;

  /* If frame hasn't a tool-bar window or if it is zero-height, don't
     do anything.  This means you must start with tool-bar-lines
     non-zero to get the auto-sizing effect.  Or in other words, you
     can turn off tool-bars by specifying tool-bar-lines zero.  */
  if (!WINDOWP (f->tool_bar_window)
      || (w = XWINDOW (f->tool_bar_window),
          WINDOW_TOTAL_LINES (w) == 0))
    {
      /* Even if we do not display a tool bar initially, still pretend
	 that we have resized it already.  This avoids that a later
	 activation of the tool bar resizes the frame, despite of the
	 fact that a setting of 'frame-inhibit-implied-resize' should
	 inhibit it (Bug#52986).  */
      f->tool_bar_resized = true;

      return false;
    }

  /* Set up an iterator for the tool-bar window.  */
  init_iterator (&it, w, -1, -1, w->desired_matrix->rows, TOOL_BAR_FACE_ID);
  it.first_visible_x = 0;
  it.last_visible_x = WINDOW_PIXEL_WIDTH (w);
  row = it.glyph_row;
  row->reversed_p = false;

  /* Build a string that represents the contents of the tool-bar.  */
  build_desired_tool_bar_string (f);
  reseat_to_string (&it, NULL, f->desired_tool_bar_string,
		    0, 0, 0, STRING_MULTIBYTE (f->desired_tool_bar_string));
  /* FIXME: This should be controlled by a user option.  But it
     doesn't make sense to have an R2L tool bar if the menu bar cannot
     be drawn also R2L, and making the menu bar R2L is tricky due
     toolkit-specific code that implements it.  If an R2L tool bar is
     ever supported, display_tool_bar_line should also be augmented to
     call unproduce_glyphs like display_line and display_string
     do.  */
  it.paragraph_embedding = L2R;

  if (f->n_tool_bar_rows == 0)
    {
      int new_height = tool_bar_height (f, &f->n_tool_bar_rows, true);

      if (new_height != WINDOW_PIXEL_HEIGHT (w))
	{
          if (FRAME_TERMINAL (f)->change_tool_bar_height_hook)
            FRAME_TERMINAL (f)->change_tool_bar_height_hook (f, new_height);
	  frame_default_tool_bar_height = new_height;
	  /* Always do that now.  */
	  clear_glyph_matrix (w->desired_matrix);
	  f->fonts_changed = true;
	  return true;
	}
    }

  /* Display as many lines as needed to display all tool-bar items.  */

  if (f->n_tool_bar_rows > 0)
    {
      int border, rows, height, extra;

      if (TYPE_RANGED_FIXNUMP (int, Vtool_bar_border))
	border = XFIXNUM (Vtool_bar_border);
      else if (EQ (Vtool_bar_border, Qinternal_border_width))
	border = FRAME_INTERNAL_BORDER_WIDTH (f);
      else if (EQ (Vtool_bar_border, Qborder_width))
	border = f->border_width;
      else
	border = 0;
      if (border < 0)
	border = 0;

      rows = f->n_tool_bar_rows;
      height = max (1, (it.last_visible_y - border) / rows);
      extra = it.last_visible_y - border - height * rows;

      while (it.current_y < it.last_visible_y)
	{
	  int h = 0;
	  if (extra > 0 && rows-- > 0)
	    {
	      h = (extra + rows - 1) / rows;
	      extra -= h;
	    }
	  display_tool_bar_line (&it, height + h);
	}
    }
  else
    {
      while (it.current_y < it.last_visible_y)
	display_tool_bar_line (&it, 0);
    }

  /* It doesn't make much sense to try scrolling in the tool-bar
     window, so don't do it.  */
  w->desired_matrix->no_scrolling_p = true;
  w->must_be_updated_p = true;

  if (!NILP (Vauto_resize_tool_bars))
    {
      bool change_height_p = false;

      /* If we couldn't display everything, change the tool-bar's
	 height if there is room for more.  */
      if (IT_STRING_CHARPOS (it) < it.end_charpos)
	change_height_p = true;

      /* We subtract 1 because display_tool_bar_line advances the
	 glyph_row pointer before returning to its caller.  We want to
	 examine the last glyph row produced by
	 display_tool_bar_line.  */
      row = it.glyph_row - 1;

      /* If there are blank lines at the end, except for a partially
	 visible blank line at the end that is smaller than
	 FRAME_LINE_HEIGHT, change the tool-bar's height.  */
      if (!MATRIX_ROW_DISPLAYS_TEXT_P (row)
	  && row->height >= FRAME_LINE_HEIGHT (f))
	change_height_p = true;

      /* If row displays tool-bar items, but is partially visible,
	 change the tool-bar's height.  */
      if (MATRIX_ROW_DISPLAYS_TEXT_P (row)
	  && MATRIX_ROW_BOTTOM_Y (row) > it.last_visible_y)
	change_height_p = true;

      /* Resize windows as needed by changing the `tool-bar-lines'
	 frame parameter.  */
      if (change_height_p)
	{
	  int nrows;
	  int new_height = tool_bar_height (f, &nrows, true);

	  change_height_p = ((EQ (Vauto_resize_tool_bars, Qgrow_only)
			      && !f->minimize_tool_bar_window_p)
			     ? (new_height > WINDOW_PIXEL_HEIGHT (w))
			     : (new_height != WINDOW_PIXEL_HEIGHT (w)));
	  f->minimize_tool_bar_window_p = false;

	  if (change_height_p)
	    {
              if (FRAME_TERMINAL (f)->change_tool_bar_height_hook)
                FRAME_TERMINAL (f)->change_tool_bar_height_hook (f, new_height);
	      frame_default_tool_bar_height = new_height;
	      clear_glyph_matrix (w->desired_matrix);
	      f->n_tool_bar_rows = nrows;
	      f->fonts_changed = true;

	      return true;
	    }
	}
    }

  f->minimize_tool_bar_window_p = false;

  return false;
}

/* Get information about the tool-bar item which is displayed in GLYPH
   on frame F.  Return in *PROP_IDX the index where tool-bar item
   properties start in F->tool_bar_items.  Value is false if
   GLYPH doesn't display a tool-bar item.  */

static bool
tool_bar_item_info (struct frame *f, struct glyph *glyph, int *prop_idx)
{
  Lisp_Object prop;
  ptrdiff_t charpos;

  /* This function can be called asynchronously, which means we must
     exclude any possibility that Fget_text_property signals an
     error.  */
  charpos = min (SCHARS (f->current_tool_bar_string), glyph->charpos);
  charpos = max (0, charpos);

  /* Get the text property `menu-item' at pos. The value of that
     property is the start index of this item's properties in
     F->tool_bar_items.  */
  prop = Fget_text_property (make_fixnum (charpos),
			     Qmenu_item, f->current_tool_bar_string);
  if (! FIXNUMP (prop))
    return false;
  *prop_idx = XFIXNUM (prop);
  return true;
}


/* Get information about the tool-bar item at position X/Y on frame F.
   Return in *GLYPH a pointer to the glyph of the tool-bar item in
   the current matrix of the tool-bar window of F, or NULL if not
   on a tool-bar item.  Return in *PROP_IDX the index of the tool-bar
   item in F->tool_bar_items.  Value is

   -1	if X/Y is not on a tool-bar item
   0	if X/Y is on the same item that was highlighted before.
   1	otherwise.  */

static int
get_tool_bar_item (struct frame *f, int x, int y, struct glyph **glyph,
		   int *hpos, int *vpos, int *prop_idx)
{
  Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (f);
  struct window *w = XWINDOW (f->tool_bar_window);
  int area;

  /* Find the glyph under X/Y.  */
  *glyph = x_y_to_hpos_vpos (w, x, y, hpos, vpos, 0, 0, &area);
  if (*glyph == NULL)
    return -1;

  /* Get the start of this tool-bar item's properties in
     f->tool_bar_items.  */
  if (!tool_bar_item_info (f, *glyph, prop_idx))
    return -1;

  /* Is mouse on the highlighted item?  */
  if (EQ (f->tool_bar_window, hlinfo->mouse_face_window)
      && *vpos >= hlinfo->mouse_face_beg_row
      && *vpos <= hlinfo->mouse_face_end_row
      && (*vpos > hlinfo->mouse_face_beg_row
	  || *hpos >= hlinfo->mouse_face_beg_col)
      && (*vpos < hlinfo->mouse_face_end_row
	  || *hpos < hlinfo->mouse_face_end_col
	  || hlinfo->mouse_face_past_end))
    return 0;

  return 1;
}


/* EXPORT:
   Handle mouse button event on the tool-bar of frame F, at
   frame-relative coordinates X/Y.  DOWN_P is true for a button press,
   false for button release.  MODIFIERS is event modifiers for button
   release.  DEVICE is the device the click came from, or Qt.  */

void
handle_tool_bar_click_with_device (struct frame *f, int x, int y, bool down_p,
				   int modifiers, Lisp_Object device)
{
  Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (f);
  struct window *w = XWINDOW (f->tool_bar_window);
  int hpos, vpos, prop_idx;
  struct glyph *glyph;
  Lisp_Object enabled_p;
  int ts;

  /* If not on the highlighted tool-bar item, and mouse-highlight is
     non-nil, return.  This is so we generate the tool-bar button
     click only when the mouse button is released on the same item as
     where it was pressed.  However, when mouse-highlight is disabled,
     generate the click when the button is released regardless of the
     highlight, since tool-bar items are not highlighted in that
     case.  */
  frame_to_window_pixel_xy (w, &x, &y);
  ts = get_tool_bar_item (f, x, y, &glyph, &hpos, &vpos, &prop_idx);
  if (ts == -1
      || (ts != 0 && !NILP (Vmouse_highlight)))
    return;

  /* When mouse-highlight is off, generate the click for the item
     where the button was pressed, disregarding where it was
     released.  */
  if (NILP (Vmouse_highlight) && !down_p)
    prop_idx = f->last_tool_bar_item;

  /* If item is disabled, do nothing.  */
  enabled_p = AREF (f->tool_bar_items, prop_idx + TOOL_BAR_ITEM_ENABLED_P);
  if (NILP (enabled_p))
    return;

  if (down_p)
    {
      /* Show item in pressed state.  */
      if (!NILP (Vmouse_highlight))
	show_mouse_face (hlinfo, DRAW_IMAGE_SUNKEN);
      f->last_tool_bar_item = prop_idx;
    }
  else
    {
      Lisp_Object key, frame;
      struct input_event event;
      EVENT_INIT (event);

      /* Show item in released state.  */
      if (!NILP (Vmouse_highlight))
	show_mouse_face (hlinfo, DRAW_IMAGE_RAISED);

      key = AREF (f->tool_bar_items, prop_idx + TOOL_BAR_ITEM_KEY);

      XSETFRAME (frame, f);
      event.kind = TOOL_BAR_EVENT;
      event.frame_or_window = frame;
      event.arg = key;
      event.modifiers = modifiers;
      event.device = device;
      kbd_buffer_store_event (&event);
      f->last_tool_bar_item = -1;
    }
}

void
handle_tool_bar_click (struct frame *f, int x, int y, bool down_p,
		       int modifiers)
{
  handle_tool_bar_click_with_device (f, x, y, down_p, modifiers, Qt);
}

/* Possibly highlight a tool-bar item on frame F when mouse moves to
   tool-bar window-relative coordinates X/Y.  Called from
   note_mouse_highlight.  */

static void
note_tool_bar_highlight (struct frame *f, int x, int y)
{
  Lisp_Object window = f->tool_bar_window;
  struct window *w = XWINDOW (window);
  Display_Info *dpyinfo = FRAME_DISPLAY_INFO (f);
  Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (f);
  int hpos, vpos;
  struct glyph *glyph;
  struct glyph_row *row;
  int i;
  Lisp_Object enabled_p;
  int prop_idx;
  enum draw_glyphs_face draw = DRAW_IMAGE_RAISED;
  bool mouse_down_p;
  int rc;

  /* Function note_mouse_highlight is called with negative X/Y
     values when mouse moves outside of the frame.  */
  if (x <= 0 || y <= 0)
    {
      clear_mouse_face (hlinfo);
      return;
    }

  rc = get_tool_bar_item (f, x, y, &glyph, &hpos, &vpos, &prop_idx);
  if (rc < 0)
    {
      /* Not on tool-bar item.  */
      clear_mouse_face (hlinfo);
      return;
    }
  else if (rc == 0)
    /* On same tool-bar item as before.  */
    goto set_help_echo;

  clear_mouse_face (hlinfo);

  /* Mouse is down, but on different tool-bar item?  */
  mouse_down_p = (gui_mouse_grabbed (dpyinfo)
		  && f == dpyinfo->last_mouse_frame);

  if (mouse_down_p && f->last_tool_bar_item != prop_idx)
    return;

  draw = mouse_down_p ? DRAW_IMAGE_SUNKEN : DRAW_IMAGE_RAISED;

  /* If tool-bar item is not enabled, don't highlight it.  */
  enabled_p = AREF (f->tool_bar_items, prop_idx + TOOL_BAR_ITEM_ENABLED_P);
  if (!NILP (enabled_p) && !NILP (Vmouse_highlight))
    {
      /* Compute the x-position of the glyph.  In front and past the
	 image is a space.  We include this in the highlighted area.  */
      row = MATRIX_ROW (w->current_matrix, vpos);
      for (i = x = 0; i < hpos; ++i)
	x += row->glyphs[TEXT_AREA][i].pixel_width;

      /* Record this as the current active region.  */
      hlinfo->mouse_face_beg_col = hpos;
      hlinfo->mouse_face_beg_row = vpos;
      hlinfo->mouse_face_beg_x = x;
      hlinfo->mouse_face_past_end = false;

      hlinfo->mouse_face_end_col = hpos + 1;
      hlinfo->mouse_face_end_row = vpos;
      hlinfo->mouse_face_end_x = x + glyph->pixel_width;
      hlinfo->mouse_face_window = window;
      hlinfo->mouse_face_face_id = TOOL_BAR_FACE_ID;

      /* Display it as active.  */
      show_mouse_face (hlinfo, draw);
    }

 set_help_echo:

  /* Set help_echo_string to a help string to display for this tool-bar item.
     XTread_socket does the rest.  */
  help_echo_object = help_echo_window = Qnil;
  help_echo_pos = -1;
  help_echo_string = AREF (f->tool_bar_items, prop_idx + TOOL_BAR_ITEM_HELP);
  if (NILP (help_echo_string))
    help_echo_string = AREF (f->tool_bar_items, prop_idx + TOOL_BAR_ITEM_CAPTION);
}

#endif /* ! (HAVE_EXT_TOOL_BAR) */

#endif /* HAVE_WINDOW_SYSTEM */



/************************************************************************
			 Horizontal scrolling
 ************************************************************************/

/* For all leaf windows in the window tree rooted at WINDOW, set their
   hscroll value so that PT is (i) visible in the window, and (ii) so
   that it is not within a certain margin at the window's left and
   right border.  Value is true if any window's hscroll has been
   changed.  */

static bool
hscroll_window_tree (Lisp_Object window)
{
  bool hscrolled_p = false;
  bool hscroll_relative_p = FLOATP (Vhscroll_step);
  int hscroll_step_abs = 0;
  double hscroll_step_rel = 0;

  if (hscroll_relative_p)
    {
      hscroll_step_rel = XFLOAT_DATA (Vhscroll_step);
      if (hscroll_step_rel < 0)
	{
	  hscroll_relative_p = false;
	  hscroll_step_abs = 0;
	}
    }
  else if (TYPE_RANGED_FIXNUMP (int, Vhscroll_step))
    {
      hscroll_step_abs = XFIXNUM (Vhscroll_step);
      if (hscroll_step_abs < 0)
	hscroll_step_abs = 0;
    }
  else
    hscroll_step_abs = 0;

  while (WINDOWP (window))
    {
      struct window *w = XWINDOW (window);

      if (WINDOWP (w->contents))
	hscrolled_p |= hscroll_window_tree (w->contents);
      else if (w->cursor.vpos >= 0
	       /* Don't allow hscroll in mini-windows that display
		  echo-area messages.  This is because desired_matrix
		  of such windows was prepared while momentarily
		  switched to an echo-area buffer, which is different
		  from w->contents, and we simply cannot hscroll such
		  windows safely.  */
	       && !(w == XWINDOW (echo_area_window)
		    && !NILP (echo_area_buffer[0])))
	{
	  int h_margin;
	  int text_area_width;
	  struct glyph_row *cursor_row;
	  struct glyph_row *bottom_row;

	  bottom_row = MATRIX_BOTTOM_TEXT_ROW (w->desired_matrix, w);
	  if (w->cursor.vpos < bottom_row - w->desired_matrix->rows)
	    cursor_row = MATRIX_ROW (w->desired_matrix, w->cursor.vpos);
	  else
	    cursor_row = bottom_row - 1;

	  if (!cursor_row->enabled_p)
	    {
	      bottom_row = MATRIX_BOTTOM_TEXT_ROW (w->current_matrix, w);
	      if (w->cursor.vpos < bottom_row - w->current_matrix->rows)
		cursor_row = MATRIX_ROW (w->current_matrix, w->cursor.vpos);
	      else
		cursor_row = bottom_row - 1;
	    }
	  bool row_r2l_p = cursor_row->reversed_p;
	  bool hscl = hscrolling_current_line_p (w);
	  int x_offset = 0;
	  /* When line numbers are displayed, we need to account for
	     the horizontal space they consume.  */
	  if (!NILP (Vdisplay_line_numbers))
	    {
	      struct glyph *g;
	      if (!row_r2l_p)
		{
		  for (g = cursor_row->glyphs[TEXT_AREA];
		       g < cursor_row->glyphs[TEXT_AREA]
			 + cursor_row->used[TEXT_AREA];
		       g++)
		    {
		      if (!(NILP (g->object) && g->charpos < 0))
			break;
		      x_offset += g->pixel_width;
		    }
		}
	      else
		{
		  for (g = cursor_row->glyphs[TEXT_AREA]
			 + cursor_row->used[TEXT_AREA];
		       g > cursor_row->glyphs[TEXT_AREA];
		       g--)
		    {
		      if (!(NILP ((g - 1)->object) && (g - 1)->charpos < 0))
			break;
		      x_offset += (g - 1)->pixel_width;
		    }
		}
	    }
	  if (cursor_row->truncated_on_left_p)
	    {
	      /* On TTY frames, don't count the left truncation glyph.  */
	      struct frame *f = XFRAME (WINDOW_FRAME (w));
	      x_offset -= (FRAME_TERMCAP_P (f) || FRAME_MSDOS_P (f));
	    }

	  text_area_width = window_box_width (w, TEXT_AREA);

	  /* Scroll when cursor is inside this scroll margin.  */
	  h_margin = (clip_to_bounds (0, hscroll_margin, 1000000)
		      * WINDOW_FRAME_COLUMN_WIDTH (w));

	  /* If the position of this window's point has explicitly
	     changed, no more suspend auto hscrolling.  */
	  if (w->suspend_auto_hscroll
	      && NILP (Fequal (Fwindow_point (window),
			       Fwindow_old_point (window))))
	    {
	      w->suspend_auto_hscroll = false;
	      /* When hscrolling just the current line, and the rest
		 of lines were temporarily hscrolled, but no longer
		 are, force thorough redisplay of this window, to show
		 the effect of disabling hscroll suspension immediately.  */
	      if (w->min_hscroll == 0 && w->hscroll > 0
		  && EQ (Fbuffer_local_value (Qauto_hscroll_mode, w->contents),
			 Qcurrent_line))
		SET_FRAME_GARBAGED (XFRAME (w->frame));
	    }

	  /* Remember window point.  */
	  Fset_marker (w->old_pointm,
		       ((w == XWINDOW (selected_window))
			? make_fixnum (BUF_PT (XBUFFER (w->contents)))
			: Fmarker_position (w->pointm)),
		       w->contents);

	  if (!NILP (Fbuffer_local_value (Qauto_hscroll_mode, w->contents))
	      && !w->suspend_auto_hscroll
	      /* In some pathological cases, like restoring a window
		 configuration into a frame that is much smaller than
		 the one from which the configuration was saved, we
		 get glyph rows whose start and end have zero buffer
		 positions, which we cannot handle below.  Just skip
		 such windows.  */
	      && (CHARPOS (cursor_row->start.pos)
		  >= BUF_BEG (XBUFFER (w->contents)))
	      /* For left-to-right rows, hscroll when cursor is either
		 (i) inside the right hscroll margin, or (ii) if it is
		 inside the left margin and the window is already
		 hscrolled.  */
	      && ((!row_r2l_p
		   && ((w->hscroll && w->cursor.x <= h_margin + x_offset)
		       || (cursor_row->enabled_p
			   && cursor_row->truncated_on_right_p
			   && (w->cursor.x >= text_area_width - h_margin))))
		  /* For right-to-left rows, the logic is similar,
		     except that rules for scrolling to left and right
		     are reversed.  E.g., if cursor.x <= h_margin, we
		     need to hscroll "to the right" unconditionally,
		     and that will scroll the screen to the left so as
		     to reveal the next portion of the row.  */
		  || (row_r2l_p
		      && ((cursor_row->enabled_p
			   /* FIXME: It is confusing to set the
			      truncated_on_right_p flag when R2L rows
			      are actually truncated on the left.  */
			   && cursor_row->truncated_on_right_p
			   && w->cursor.x <= h_margin)
			  || (w->hscroll
			      && (w->cursor.x >= (text_area_width - h_margin
						  - x_offset)))))
		  /* This last condition is needed when moving
		     vertically from an hscrolled line to a short line
		     that doesn't need to be hscrolled.  If we omit
		     this condition, the line from which we move will
		     remain hscrolled.  */
		  || (hscl
		      && w->hscroll != w->min_hscroll
		      && !cursor_row->truncated_on_left_p)))
	    {
	      struct it it;
	      ptrdiff_t hscroll;
	      struct buffer *saved_current_buffer;
	      ptrdiff_t pt;
	      int wanted_x;

	      /* Find point in a display of infinite width.  */
	      saved_current_buffer = current_buffer;
	      current_buffer = XBUFFER (w->contents);

	      if (w == XWINDOW (selected_window))
		pt = PT;
	      else
		pt = clip_to_bounds (BEGV, marker_position (w->pointm), ZV);

	      /* Move iterator to pt starting at cursor_row->start in
		 a line with infinite width.  */
	      init_to_row_start (&it, w, cursor_row);
	      if (hscl)
		it.first_visible_x = window_hscroll_limited (w, it.f)
				     * FRAME_COLUMN_WIDTH (it.f);
	      it.last_visible_x = DISP_INFINITY;

	      ptrdiff_t nchars = pt - IT_CHARPOS (it);
	      if (current_buffer->long_line_optimizations_p
		  && nchars > large_hscroll_threshold)
		{
		  /* Special optimization for very long and truncated
		     lines which need to be hscrolled far to the left:
		     jump directly to the (approximate) first position
		     that is visible, instead of slowly walking there.  */
		  fast_move_it_horizontally (&it, nchars);
		  it.current_x += nchars * FRAME_COLUMN_WIDTH (it.f);
		}
	      else
		move_it_in_display_line_to (&it, pt, -1, MOVE_TO_POS);
	      /* If the line ends in an overlay string with a newline,
		 we might infloop, because displaying the window will
		 want to put the cursor after the overlay, i.e. at X
		 coordinate of zero on the next screen line.  So we
		 use the buffer position prior to the overlay string
		 instead.  */
	      if (it.method == GET_FROM_STRING && pt > 1)
		{
		  init_to_row_start (&it, w, cursor_row);
		  if (hscl)
		    it.first_visible_x = (window_hscroll_limited (w, it.f)
					  * FRAME_COLUMN_WIDTH (it.f));
		  if (current_buffer->long_line_optimizations_p
		      && nchars > large_hscroll_threshold)
		    {
		      fast_move_it_horizontally (&it, nchars - 1);
		      it.current_x += (nchars - 1) * FRAME_COLUMN_WIDTH (it.f);
		    }
		  else
		    move_it_in_display_line_to (&it, pt - 1, -1, MOVE_TO_POS);
		}
	      current_buffer = saved_current_buffer;

	      /* Position cursor in window.  */
	      if (!hscroll_relative_p && hscroll_step_abs == 0)
		hscroll = max (0, (it.current_x
				   - (ITERATOR_AT_END_OF_LINE_P (&it)
				      ? (text_area_width - 4 * FRAME_COLUMN_WIDTH (it.f))
				      : (text_area_width / 2))))
		    	  / FRAME_COLUMN_WIDTH (it.f);
	      else if ((!row_r2l_p
			&& w->cursor.x >= text_area_width - h_margin)
		       || (row_r2l_p && w->cursor.x <= h_margin))
		{
		  if (hscroll_relative_p)
		    wanted_x = text_area_width * (1 - hscroll_step_rel)
		      	       - h_margin;
		  else
		    wanted_x = text_area_width
		      	       - hscroll_step_abs * FRAME_COLUMN_WIDTH (it.f)
		      	       - h_margin;
		  hscroll
		    = max (0, it.current_x - wanted_x) / FRAME_COLUMN_WIDTH (it.f);
		}
	      else
		{
		  if (hscroll_relative_p)
		    wanted_x =
		      text_area_width * hscroll_step_rel + h_margin + x_offset;
		  else
		    wanted_x =
		      hscroll_step_abs * FRAME_COLUMN_WIDTH (it.f)
		      + h_margin + x_offset;
		  hscroll
		    = max (0, it.current_x - wanted_x) / FRAME_COLUMN_WIDTH (it.f);
		}
	      hscroll = max (hscroll, w->min_hscroll);

	      /* Don't prevent redisplay optimizations if hscroll
		 hasn't changed, as it will unnecessarily slow down
		 redisplay.  */
	      if (w->hscroll != hscroll
		  /* When hscrolling only the current line, we need to
		     report hscroll even if its value is equal to the
		     previous one, because the new line might need a
		     different value.  */
		  || (hscl && w->last_cursor_vpos != w->cursor.vpos))
		{
		  struct buffer *b = XBUFFER (w->contents);
		  b->prevent_redisplay_optimizations_p = true;
		  w->hscroll = hscroll;
		  hscrolled_p = true;
		}
	    }
	}

      window = w->next;
    }

  /* Value is true if hscroll of any leaf window has been changed.  */
  return hscrolled_p;
}


/* Set hscroll so that cursor is visible and not inside horizontal
   scroll margins for all windows in the tree rooted at WINDOW.  See
   also hscroll_window_tree above.  Value is true if any window's
   hscroll has been changed.  If it has, desired matrices on the frame
   of WINDOW are cleared.  */

static bool
hscroll_windows (Lisp_Object window)
{
  bool hscrolled_p = hscroll_window_tree (window);
  if (hscrolled_p)
    clear_desired_matrices (XFRAME (WINDOW_FRAME (XWINDOW (window))));
  return hscrolled_p;
}



/************************************************************************
				Redisplay
 ************************************************************************/

/* Variables holding some state of redisplay if GLYPH_DEBUG is defined.
   This is sometimes handy to have in a debugger session.  */

#ifdef GLYPH_DEBUG

/* First and last unchanged row for try_window_id.  */

static int debug_first_unchanged_at_end_vpos;
static int debug_last_unchanged_at_beg_vpos;

/* Delta vpos and y.  */

static int debug_dvpos, debug_dy;

/* Delta in characters and bytes for try_window_id.  */

static ptrdiff_t debug_delta, debug_delta_bytes;

/* Values of window_end_pos and window_end_vpos at the end of
   try_window_id.  */

static ptrdiff_t debug_end_vpos;

/* Append a string to W->desired_matrix->method.  FMT is a printf
   format string.  If trace_redisplay_p is true also printf the
   resulting string to stderr.  */

static void debug_method_add (struct window *, char const *, ...)
  ATTRIBUTE_FORMAT_PRINTF (2, 3);

static void
debug_method_add (struct window *w, char const *fmt, ...)
{
  void *ptr = w;
  char *method = w->desired_matrix->method;
  int len = strlen (method);
  int size = sizeof w->desired_matrix->method;
  int remaining = size - len - 1;
  va_list ap;

  if (len && remaining)
    {
      method[len] = '|';
      --remaining, ++len;
    }

  va_start (ap, fmt);
  vsnprintf (method + len, remaining + 1, fmt, ap);
  va_end (ap);

  if (trace_redisplay_p)
    fprintf (stderr, "%p (%s): %s\n",
	     ptr,
	     ((BUFFERP (w->contents)
	       && STRINGP (BVAR (XBUFFER (w->contents), name)))
	      ? SSDATA (BVAR (XBUFFER (w->contents), name))
	      : "no buffer"),
	     method + len);
}

#endif /* GLYPH_DEBUG */


/* Value is true if all changes in window W, which displays
   current_buffer, are in the text between START and END.  START is a
   buffer position, END is given as a distance from Z.  Used in
   redisplay_internal for display optimization.  */

static bool
text_outside_line_unchanged_p (struct window *w,
			       ptrdiff_t start, ptrdiff_t end)
{
  bool unchanged_p = true;

  /* If text or overlays have changed, see where.  */
  if (window_outdated (w))
    {
      /* Gap in the line?  */
      if (GPT < start || Z - GPT < end)
	unchanged_p = false;

      /* Changes start in front of the line, or end after it?  */
      if (unchanged_p
	  && (BEG_UNCHANGED < start - 1
	      || END_UNCHANGED < end))
	unchanged_p = false;

      /* If selective display, can't optimize if changes start at the
	 beginning of the line.  */
      if (unchanged_p
	  && FIXNUMP (BVAR (current_buffer, selective_display))
	  && XFIXNUM (BVAR (current_buffer, selective_display)) > 0
	  && (BEG_UNCHANGED < start || GPT <= start))
	unchanged_p = false;

      /* If there are overlays at the start or end of the line, these
	 may have overlay strings with newlines in them.  A change at
	 START, for instance, may actually concern the display of such
	 overlay strings as well, and they are displayed on different
	 lines.  So, quickly rule out this case.  (For the future, it
	 might be desirable to implement something more telling than
	 just BEG/END_UNCHANGED.)  */
      if (unchanged_p)
	{
	  if (BEG + BEG_UNCHANGED == start
	      && overlay_touches_p (start))
	    unchanged_p = false;
	  if (END_UNCHANGED == end
	      && overlay_touches_p (Z - end))
	    unchanged_p = false;
	}

      /* Under bidi reordering, adding or deleting a character in the
	 beginning of a paragraph, before the first strong directional
	 character, can change the base direction of the paragraph (unless
	 the buffer specifies a fixed paragraph direction), which will
	 require redisplaying the whole paragraph.  It might be worthwhile
	 to find the paragraph limits and widen the range of redisplayed
	 lines to that, but for now just give up this optimization.  */
      if (!NILP (BVAR (XBUFFER (w->contents), bidi_display_reordering))
	  && NILP (BVAR (XBUFFER (w->contents), bidi_paragraph_direction)))
	unchanged_p = false;
    }

  return unchanged_p;
}


/* Do a frame update, taking possible shortcuts into account.  This is
   the main external entry point for redisplay.

   If the last redisplay displayed an echo area message and that message
   is no longer requested, we clear the echo area or bring back the
   mini-buffer if that is in use.  */

void
redisplay (void)
{
  redisplay_internal ();
}


static Lisp_Object
overlay_arrow_string_or_property (Lisp_Object var)
{
  Lisp_Object val;

  if (val = Fget (var, Qoverlay_arrow_string), STRINGP (val))
    return val;

  return Voverlay_arrow_string;
}

/* Return true if there are any overlay-arrows in current_buffer.  */
static bool
overlay_arrow_in_current_buffer_p (void)
{
  Lisp_Object vlist;

  for (vlist = Voverlay_arrow_variable_list;
       CONSP (vlist);
       vlist = XCDR (vlist))
    {
      Lisp_Object var = XCAR (vlist);
      Lisp_Object val;

      if (!SYMBOLP (var))
	continue;
      val = find_symbol_value (var);
      if (MARKERP (val)
	  && current_buffer == XMARKER (val)->buffer)
	return true;
    }
  return false;
}


/* Return true if any overlay_arrows have moved or overlay-arrow-string
   has changed.
   If SET_REDISPLAY is true, additionally, set the `redisplay' bit in those
   buffers that are affected.  */

static bool
overlay_arrows_changed_p (bool set_redisplay)
{
  Lisp_Object vlist;
  bool changed = false;

  for (vlist = Voverlay_arrow_variable_list;
       CONSP (vlist);
       vlist = XCDR (vlist))
    {
      Lisp_Object var = XCAR (vlist);
      Lisp_Object val, pstr;

      if (!SYMBOLP (var))
	continue;
      val = find_symbol_value (var);
      if (!MARKERP (val))
	continue;
      if (! EQ (Fmarker_position (val),
                /* FIXME: Don't we have a problem, using such a global
                 * "last-position" if the variable is buffer-local?  */
		Fget (var, Qlast_arrow_position))
	  || ! (pstr = overlay_arrow_string_or_property (var),
		EQ (pstr, Fget (var, Qlast_arrow_string))))
	{
	  struct buffer *buf = XMARKER (val)->buffer;

	  if (set_redisplay)
            {
              if (buf)
	        bset_redisplay (buf);
              changed = true;
            }
	  else
	    return true;
	}
    }
  return changed;
}

/* Mark overlay arrows to be updated on next redisplay.  */

static void
update_overlay_arrows (int up_to_date)
{
  Lisp_Object vlist;

  for (vlist = Voverlay_arrow_variable_list;
       CONSP (vlist);
       vlist = XCDR (vlist))
    {
      Lisp_Object var = XCAR (vlist);

      if (!SYMBOLP (var))
	continue;

      if (up_to_date > 0)
	{
	  Lisp_Object val = find_symbol_value (var);
          if (!MARKERP (val))
	    continue;
	  Fput (var, Qlast_arrow_position, Fmarker_position (val));
	  Fput (var, Qlast_arrow_string,
		overlay_arrow_string_or_property (var));
	}
      else if (up_to_date < 0
	       || !NILP (Fget (var, Qlast_arrow_position)))
	{
	  Fput (var, Qlast_arrow_position, Qt);
	  Fput (var, Qlast_arrow_string, Qt);
	}
    }
}


/* Return overlay arrow string to display at row.
   Return integer (bitmap number) for arrow bitmap in left fringe.
   Return nil if no overlay arrow.  */

static Lisp_Object
overlay_arrow_at_row (struct it *it, struct glyph_row *row)
{
  Lisp_Object vlist;

  for (vlist = Voverlay_arrow_variable_list;
       CONSP (vlist);
       vlist = XCDR (vlist))
    {
      Lisp_Object var = XCAR (vlist);
      Lisp_Object val;

      if (!SYMBOLP (var))
	continue;

      val = find_symbol_value (var);

      if (MARKERP (val)
	  && current_buffer == XMARKER (val)->buffer
	  && (MATRIX_ROW_START_CHARPOS (row) == marker_position (val)))
	{
	  if (FRAME_WINDOW_P (it->f)
	      /* FIXME: if ROW->reversed_p is set, this should test
		 the right fringe, not the left one.  */
	      && WINDOW_LEFT_FRINGE_WIDTH (it->w) > 0)
	    {
#ifdef HAVE_WINDOW_SYSTEM
	      if (val = Fget (var, Qoverlay_arrow_bitmap), SYMBOLP (val))
		{
		  int fringe_bitmap = lookup_fringe_bitmap (val);
		  if (fringe_bitmap != 0)
		    return make_fixnum (fringe_bitmap);
		}
#endif
	      return make_fixnum (-1); /* Use default arrow bitmap.  */
	    }
	  return overlay_arrow_string_or_property (var);
	}
    }

  return Qnil;
}

/* Return true if point moved out of or into a composition.  Otherwise
   return false.  PREV_BUF and PREV_PT are the last point buffer and
   position.  BUF and PT are the current point buffer and position.  */

static bool
check_point_in_composition (struct buffer *prev_buf, ptrdiff_t prev_pt,
			    struct buffer *buf, ptrdiff_t pt)
{
  ptrdiff_t start, end;
  Lisp_Object prop;
  Lisp_Object buffer;

  XSETBUFFER (buffer, buf);
  /* Check a composition at the last point if point moved within the
     same buffer.  */
  if (prev_buf == buf)
    {
      if (prev_pt == pt)
	/* Point didn't move.  */
	return false;

      if (prev_pt > BUF_BEGV (buf) && prev_pt < BUF_ZV (buf)
	  && find_composition (prev_pt, -1, &start, &end, &prop, buffer)
	  && composition_valid_p (start, end, prop)
	  && start < prev_pt && end > prev_pt)
	/* The last point was within the composition.  Return true iff
            point moved out of the composition.  */
	return (pt <= start || pt >= end);
    }

  /* Check a composition at the current point.  */
  return (pt > BUF_BEGV (buf) && pt < BUF_ZV (buf)
	  && find_composition (pt, -1, &start, &end, &prop, buffer)
	  && composition_valid_p (start, end, prop)
	  && start < pt && end > pt);
}

/* Reconsider the clip changes of buffer which is displayed in W.  */

static void
reconsider_clip_changes (struct window *w)
{
  struct buffer *b = XBUFFER (w->contents);

  if (b->clip_changed
      && w->window_end_valid
      && w->current_matrix->buffer == b
      && w->current_matrix->zv == BUF_ZV (b)
      && w->current_matrix->begv == BUF_BEGV (b))
    b->clip_changed = false;

  /* If display wasn't paused, and W is not a tool bar window, see if
     point has been moved into or out of a composition.  In that case,
     set b->clip_changed to force updating the screen.  If
     b->clip_changed has already been set, skip this check.  */
  if (!b->clip_changed && w->window_end_valid)
    {
      ptrdiff_t pt = (w == XWINDOW (selected_window)
		      ? PT : marker_position (w->pointm));

      if ((w->current_matrix->buffer != b || pt != w->last_point)
	  && check_point_in_composition (w->current_matrix->buffer,
					 w->last_point, b, pt))
	b->clip_changed = true;
    }
}

static void
propagate_buffer_redisplay (void)
{ /* Resetting b->text->redisplay is problematic!
     We can't just reset it in the case that some window that displays
     it has not been redisplayed; and such a window can stay
     unredisplayed for a long time if it's currently invisible.
     But we do want to reset it at the end of redisplay otherwise
     its displayed windows will keep being redisplayed over and over
     again.
     So we copy all b->text->redisplay flags up to their windows here,
     such that mark_window_display_accurate can safely reset
     b->text->redisplay.  */
  Lisp_Object ws = window_list ();
  for (; CONSP (ws); ws = XCDR (ws))
    {
      struct window *thisw = XWINDOW (XCAR (ws));
      struct buffer *thisb = XBUFFER (thisw->contents);
      if (thisb->text->redisplay)
	thisw->redisplay = true;
    }
}

#define STOP_POLLING					\
do { if (! polling_stopped_here) stop_polling ();	\
       polling_stopped_here = true; } while (false)

#define RESUME_POLLING					\
do { if (polling_stopped_here) start_polling ();	\
       polling_stopped_here = false; } while (false)

/* Perhaps in the future avoid recentering windows if it
   is not necessary; currently that causes some problems.  */

static void
redisplay_internal (void)
{
  struct window *w = XWINDOW (selected_window);
  struct window *sw;
  struct frame *fr;
  bool pending;
  bool must_finish = false, match_p;
  struct text_pos tlbufpos, tlendpos;
  int number_of_visible_frames;
  struct frame *sf;
  bool polling_stopped_here = false;
  Lisp_Object tail, frame;

  /* Set a limit to the number of retries we perform due to horizontal
     scrolling, this avoids getting stuck in an uninterruptible
     infinite loop (Bug #24633).  */
  enum { MAX_HSCROLL_RETRIES = 16 };
  int hscroll_retries = 0;

  /* Limit the number of retries for when frame(s) become garbaged as
     result of redisplaying them.  Some packages set various redisplay
     hooks, such as window-scroll-functions, to run Lisp that always
     calls APIs which cause the frame's garbaged flag to become set,
     so we loop indefinitely.  */
  enum {MAX_GARBAGED_FRAME_RETRIES = 2 };
  int garbaged_frame_retries = 0;

  /* True means redisplay has to consider all windows on all
     frames.  False, only selected_window is considered.  */
  bool consider_all_windows_p;

  /* True means redisplay has to redisplay the miniwindow.  */
  bool update_miniwindow_p = false;

  redisplay_trace ("redisplay_internal %d\n", redisplaying_p);

  /* No redisplay if running in batch mode or frame is not yet fully
     initialized, or redisplay is explicitly turned off by setting
     Vinhibit_redisplay.  */
  if ((FRAME_INITIAL_P (SELECTED_FRAME ())
       && redisplay_skip_initial_frame)
      || !NILP (Vinhibit_redisplay))
    return;

  /* Don't examine these until after testing Vinhibit_redisplay.
     When Emacs is shutting down, perhaps because its connection to
     X has dropped, we should not look at them at all.  */
  fr = XFRAME (w->frame);
  sf = SELECTED_FRAME ();

  if (!fr->glyphs_initialized_p)
    return;

#if defined (USE_X_TOOLKIT) || defined (USE_GTK) || defined (HAVE_NS)
  if (popup_activated ())
    return;
#endif

#if defined (HAVE_HAIKU)
  if (popup_activated_p)
    return;
#endif

  /* I don't think this happens but let's be paranoid.  */
  if (redisplaying_p)
    return;

  /* Record a function that clears redisplaying_p
     when we leave this function.  */
  specpdl_ref count = SPECPDL_INDEX ();
  record_unwind_protect_void (unwind_redisplay);
  redisplaying_p = true;
  block_buffer_flips ();
  specbind (Qinhibit_free_realized_faces, Qnil);

  /* Record this function, so it appears on the profiler's backtraces.  */
  record_in_backtrace (Qredisplay_internal_xC_functionx, 0, 0);

  FOR_EACH_FRAME (tail, frame)
    XFRAME (frame)->already_hscrolled_p = false;

  reset_outermost_narrowings ();

 retry:
  /* Remember the currently selected window.  */
  sw = w;

  pending = false;
  forget_escape_and_glyphless_faces ();

  inhibit_free_realized_faces = false;

  /* If face_change, init_iterator will free all realized faces, which
     includes the faces referenced from current matrices.  So, we
     can't reuse current matrices in this case.  */
  if (face_change)
    windows_or_buffers_changed = 47;

  if ((FRAME_TERMCAP_P (sf) || FRAME_MSDOS_P (sf))
      && FRAME_TTY (sf)->previous_frame != sf)
    {
      /* Since frames on a single ASCII terminal share the same
	 display area, displaying a different frame means redisplay
	 the whole thing.  */
      SET_FRAME_GARBAGED (sf);
#ifndef DOS_NT
      set_tty_color_mode (FRAME_TTY (sf), sf);
#endif
      FRAME_TTY (sf)->previous_frame = sf;
    }

  /* Set the visible flags for all frames.  Do this before checking for
     resized or garbaged frames; they want to know if their frames are
     visible.  See the comment in frame.h for FRAME_SAMPLE_VISIBILITY.  */
  number_of_visible_frames = 0;

  FOR_EACH_FRAME (tail, frame)
    {
      struct frame *f = XFRAME (frame);

      if (FRAME_VISIBLE_P (f))
	{
	  ++number_of_visible_frames;
	  /* Adjust matrices for visible frames only.  */
	  if (f->fonts_changed)
	    {
	      adjust_frame_glyphs (f);
	      /* Disable all redisplay optimizations for this frame.
		 This is because adjust_frame_glyphs resets the
		 enabled_p flag for all glyph rows of all windows, so
		 many optimizations will fail anyway, and some might
		 fail to test that flag and do bogus things as
		 result.  */
	      SET_FRAME_GARBAGED (f);
	      f->fonts_changed = false;
	    }
	  /* If cursor type has been changed on the frame
	     other than selected, consider all frames.  */
	  if (f != sf && f->cursor_type_changed)
	    fset_redisplay (f);
	}
      clear_desired_matrices (f);
    }

  /* Notice any pending interrupt request to change frame size.  */
  do_pending_window_change (true);

  /* Clear frames marked as garbaged.  */
  clear_garbaged_frames ();

  /* Build menubar and tool-bar items.  */
  if (NILP (Vmemory_full))
    prepare_menu_bars ();

  /* do_pending_window_change could change the selected_window due to
     frame resizing which makes the selected window too small.
     prepare_menu_bars may call lisp hooks and hence also change the
     selected_window.  */
  if (WINDOWP (selected_window) && (w = XWINDOW (selected_window)) != sw)
    sw = w;

  reconsider_clip_changes (w);

  /* In most cases selected window displays current buffer.  */
  match_p = XBUFFER (w->contents) == current_buffer;
  if (match_p)
    {
      /* Detect case that we need to write or remove a star in the mode line.  */
      if ((SAVE_MODIFF < MODIFF) != w->last_had_star)
	w->update_mode_line = true;

      if (mode_line_update_needed (w))
	w->update_mode_line = true;

      /* If reconsider_clip_changes above decided that the narrowing
	 in the current buffer changed, make sure all other windows
	 showing that buffer will be redisplayed.  */
      if (current_buffer->clip_changed)
	bset_update_mode_line (current_buffer);
    }

  /* Normally the message* functions will have already displayed and
     updated the echo area, but the frame may have been trashed, or
     the update may have been preempted, so display the echo area
     again here.  Checking message_cleared_p captures the case that
     the echo area should be cleared.  */
  if ((!NILP (echo_area_buffer[0]) && !display_last_displayed_message_p)
      || (!NILP (echo_area_buffer[1]) && display_last_displayed_message_p)
      || (message_cleared_p
	  && minibuf_level == 0
	  /* If the mini-window is currently selected, this means the
	     echo-area doesn't show through.  */
	  && !MINI_WINDOW_P (XWINDOW (selected_window))))
    {
      echo_area_display (false);

      if (message_cleared_p)
	update_miniwindow_p = true;

      must_finish = true;

      /* If we don't display the current message, don't clear the
	 message_cleared_p flag, because, if we did, we wouldn't clear
	 the echo area in the next redisplay which doesn't preserve
	 the echo area.  */
      if (!display_last_displayed_message_p)
	message_cleared_p = false;
    }
  else if (EQ (selected_window, minibuf_window)
	   && (current_buffer->clip_changed || window_outdated (w))
	   && resize_mini_window (w, false))
    {
      /* Resized active mini-window to fit the size of what it is
         showing if its contents might have changed.  */
      must_finish = true;

      /* If window configuration was changed, frames may have been
	 marked garbaged.  Clear them or we will experience
	 surprises wrt scrolling.  */
      clear_garbaged_frames ();
    }

  if (!NILP (Vrun_hooks))
    run_window_change_functions ();

  if (windows_or_buffers_changed && !update_mode_lines)
    /* Code that sets windows_or_buffers_changed doesn't distinguish whether
       only the windows's contents needs to be refreshed, or whether the
       mode-lines also need a refresh.  */
    update_mode_lines = (windows_or_buffers_changed == REDISPLAY_SOME
  			 ? REDISPLAY_SOME : 32);

  /* If specs for an arrow have changed, do thorough redisplay
     to ensure we remove any arrow that should no longer exist.  */
  /* Apparently, this is the only case where we update other windows,
     without updating other mode-lines.  */
  overlay_arrows_changed_p (true);

  consider_all_windows_p = (update_mode_lines
			    || windows_or_buffers_changed);

#define AINC(a,i)							\
  {									\
    Lisp_Object entry = Fgethash (make_fixnum (i), a, make_fixnum (0));	\
    if (FIXNUMP (entry))						\
      Fputhash (make_fixnum (i), make_fixnum (1 + XFIXNUM (entry)), a);	\
  }

  AINC (Vredisplay__all_windows_cause, windows_or_buffers_changed);
  AINC (Vredisplay__mode_lines_cause, update_mode_lines);

  /* Optimize the case that only the line containing the cursor in the
     selected window has changed.  Variables starting with this_ are
     set in display_line and record information about the line
     containing the cursor.  */
  tlbufpos = this_line_start_pos;
  tlendpos = this_line_end_pos;
  if (!consider_all_windows_p
      && CHARPOS (tlbufpos) > 0
      && !w->update_mode_line
      && !current_buffer->clip_changed
      && !current_buffer->prevent_redisplay_optimizations_p
      && FRAME_VISIBLE_P (XFRAME (w->frame))
      && !FRAME_OBSCURED_P (XFRAME (w->frame))
      && !XFRAME (w->frame)->cursor_type_changed
      && !XFRAME (w->frame)->face_change
      /* Make sure recorded data applies to current buffer, etc.  */
      && this_line_buffer == current_buffer
      && match_p
      && !w->force_start
      && !w->optional_new_start
      /* Point must be on the line that we have info recorded about.  */
      && PT >= CHARPOS (tlbufpos)
      && PT <= Z - CHARPOS (tlendpos)
      /* FIXME: The following condition is only needed when
	 significant parts of the buffer are hidden (e.g., under
	 hs-minor-mode), but there doesn't seem to be a simple way of
	 detecting that, so we always disable the one-line redisplay
	 optimizations whenever display-line-numbers-mode is turned on
	 in the buffer.  */
      && (NILP (Vdisplay_line_numbers)
	  || EQ (Vdisplay_line_numbers, Qvisual))
      /* All text outside that line, including its final newline,
	 must be unchanged.  */
      && text_outside_line_unchanged_p (w, CHARPOS (tlbufpos),
					CHARPOS (tlendpos)))
    {
      if (CHARPOS (tlbufpos) > BEGV
	  && FETCH_BYTE (BYTEPOS (tlbufpos) - 1) != '\n'
	  && (CHARPOS (tlbufpos) == ZV
	      || FETCH_BYTE (BYTEPOS (tlbufpos)) == '\n'))
	/* Former continuation line has disappeared by becoming empty.  */
	goto cancel;
      else if (window_outdated (w) || MINI_WINDOW_P (w))
	{
	  /* We have to handle the case of continuation around a
	     wide-column character (see the comment in indent.c around
	     line 1340).

	     For instance, in the following case:

	     --------  Insert  --------
	     K_A_N_\\   `a'    K_A_N_a\		`X_' are wide-column chars.
	     J_I_       ==>    J_I_		`^^' are cursors.
	     ^^                ^^
	     --------          --------

	     As we have to redraw the line above, we cannot use this
	     optimization.  */

	  struct it it;
	  int line_height_before = this_line_pixel_height;

	  /* Note that start_display will handle the case that the
	     line starting at tlbufpos is a continuation line.  */
	  start_display (&it, w, tlbufpos);

	  /* Implementation note: It this still necessary?  */
	  if (it.current_x != this_line_start_x)
	    goto cancel;

	  /* Give up on this optimization if the line starts with a
	     string with display property that draws on the fringes,
	     as that might interfere with line-prefix display.  */
	  if (it.sp > 1
	      && it.method == GET_FROM_IMAGE && it.image_id == -1)
	    goto cancel;
	  redisplay_trace ("trying display optimization 1\n");
	  w->cursor.vpos = -1;
	  overlay_arrow_seen = false;
	  it.vpos = this_line_vpos;
	  it.current_y = this_line_y;
	  it.glyph_row = MATRIX_ROW (w->desired_matrix, this_line_vpos);
	  display_line (&it, -1);

	  /* If line contains point, is not continued,
             and ends at same distance from eob as before, we win.  */
	  if (w->cursor.vpos >= 0
              /* Line is not continued, otherwise this_line_start_pos
                 would have been set to 0 in display_line.  */
	      && CHARPOS (this_line_start_pos)
	      /* Line ends as before.  */
	      && CHARPOS (this_line_end_pos) == CHARPOS (tlendpos)
              /* Line has same height as before.  Otherwise other lines
                 would have to be shifted up or down.  */
	      && this_line_pixel_height == line_height_before
	      /* Cannot use this optimization if hscrolling current
		 line and this line is the current one, because
		 display_line above is not informed about the
		 current-line's vpos, and cannot DTRT in that case.  */
	      && !hscrolling_current_line_p (w))
	    {
 	      /* If this is not the window's last line, we must adjust
 		 the charstarts of the lines below.  */
 	      if (it.current_y < it.last_visible_y)
  		{
 		  struct glyph_row *row
 		    = MATRIX_ROW (w->current_matrix, this_line_vpos + 1);
  		  ptrdiff_t delta, delta_bytes;

		  /* We used to distinguish between two cases here,
		     conditioned by Z - CHARPOS (tlendpos) == ZV, for
		     when the line ends in a newline or the end of the
		     buffer's accessible portion.  But both cases did
		     the same, so they were collapsed.  */
		  delta = (Z
			   - CHARPOS (tlendpos)
			   - MATRIX_ROW_START_CHARPOS (row));
		  delta_bytes = (Z_BYTE
				 - BYTEPOS (tlendpos)
				 - MATRIX_ROW_START_BYTEPOS (row));

  		  increment_matrix_positions (w->current_matrix,
					      this_line_vpos + 1,
					      w->current_matrix->nrows,
					      delta, delta_bytes);
		}

	      /* If this row displays text now but previously didn't,
		 or vice versa, w->window_end_vpos may have to be
		 adjusted.  */
	      if (MATRIX_ROW_DISPLAYS_TEXT_P (it.glyph_row - 1))
		{
		  if (w->window_end_vpos < this_line_vpos)
		    w->window_end_vpos = this_line_vpos;
		}
	      else if (w->window_end_vpos == this_line_vpos
		       && this_line_vpos > 0)
		w->window_end_vpos = this_line_vpos - 1;
	      w->window_end_valid = false;

	      /* Update hint: No need to try to scroll in update_window.  */
	      w->desired_matrix->no_scrolling_p = true;

#ifdef GLYPH_DEBUG
	      *w->desired_matrix->method = 0;
	      debug_method_add (w, "optimization 1");
#endif
#ifdef HAVE_WINDOW_SYSTEM
	      update_window_fringes (w, false);
#endif
	      goto update;
	    }
	  else
	    goto cancel;
	}
      else if (/* Cursor position hasn't changed.  */
	       PT == w->last_point
	       /* Make sure the cursor was last displayed
		  in this window.  Otherwise we have to reposition it.  */

	       /* PXW: Must be converted to pixels, probably.  */
	       && 0 <= w->cursor.vpos
	       && w->cursor.vpos < WINDOW_TOTAL_LINES (w))
	{
	  if (!must_finish)
	    {
	      do_pending_window_change (true);
	      /* If selected_window changed, redisplay again.  */
	      if (WINDOWP (selected_window)
		  && (w = XWINDOW (selected_window)) != sw)
		goto retry;

		/* We used to always goto end_of_redisplay here, but this
		 isn't enough if we have a blinking cursor.  */
	      if (w->cursor_off_p == w->last_cursor_off_p)
		goto end_of_redisplay;
	    }
	  goto update;
	}
      /* If highlighting the region, or if the cursor is in the echo area,
	 then we can't just move the cursor.  */
      else if (NILP (Vshow_trailing_whitespace)
	       && !cursor_in_echo_area
	       && !composition_break_at_point)
	{
	  struct it it;
	  struct glyph_row *row;

	  /* Skip from tlbufpos to PT and see where it is.  Note that
	     PT may be in invisible text.  If so, we will end at the
	     next visible position.  */
	  init_iterator (&it, w, CHARPOS (tlbufpos), BYTEPOS (tlbufpos),
			 NULL, DEFAULT_FACE_ID);
	  it.current_x = this_line_start_x;
	  it.current_y = this_line_y;
	  it.vpos = this_line_vpos;

	  if (current_buffer->long_line_optimizations_p
	      && it.line_wrap == TRUNCATE
	      && PT - CHARPOS (tlbufpos) > large_hscroll_threshold)
	    {
	      /* When lines are very long and truncated, jumping to
		 the next visible line is much faster than slowly
		 iterating there.  */
	      reseat_at_next_visible_line_start (&it, false);
	      if (IT_CHARPOS (it) <= PT) /* point moved off this line */
		it.vpos = this_line_vpos + 1;
	    }
	  else
	    {
	      /* The call to move_it_to stops in front of PT, but
		 moves over before-strings.  */
	      move_it_to (&it, PT, -1, -1, -1, MOVE_TO_POS);
	    }

	  if (it.vpos == this_line_vpos
	      && (row = MATRIX_ROW (w->current_matrix, this_line_vpos),
		  row->enabled_p))
	    {
	      eassert (this_line_vpos == it.vpos);
	      eassert (this_line_y == it.current_y);
	      set_cursor_from_row (w, row, w->current_matrix, 0, 0, 0, 0);
	      if (cursor_row_fully_visible_p (w, false, true, false))
		{
#ifdef GLYPH_DEBUG
		  *w->desired_matrix->method = 0;
		  debug_method_add (w, "optimization 3");
#endif
		  goto update;
		}
	      else
		goto cancel;
	    }
	  else
	    goto cancel;
	}

    cancel:
      /* Text changed drastically or point moved off of line.  */
      SET_MATRIX_ROW_ENABLED_P (w->desired_matrix, this_line_vpos, false);
    }

  CHARPOS (this_line_start_pos) = 0;
  ++clear_face_cache_count;
#ifdef HAVE_WINDOW_SYSTEM
  ++clear_image_cache_count;
#endif

  /* Build desired matrices, and update the display.  If
     consider_all_windows_p, do it for all windows on all frames that
     require redisplay, as specified by their 'redisplay' flag.
     Otherwise do it for selected_window, only.  */

  if (consider_all_windows_p)
    {
      FOR_EACH_FRAME (tail, frame)
	XFRAME (frame)->updated_p = false;

      propagate_buffer_redisplay ();

      FOR_EACH_FRAME (tail, frame)
	{
	  struct frame *f = XFRAME (frame);

	  /* We don't have to do anything for unselected terminal
	     frames.  */
	  if ((FRAME_TERMCAP_P (f) || FRAME_MSDOS_P (f))
	      && !EQ (FRAME_TTY (f)->top_frame, frame))
	    continue;

	retry_frame:
	  if (FRAME_WINDOW_P (f) || FRAME_TERMCAP_P (f) || f == sf)
	    {
	      bool gcscrollbars
		/* Only GC scrollbars when we redisplay the whole frame.  */
		= f->redisplay || !REDISPLAY_SOME_P ();
	      bool f_redisplay_flag = f->redisplay;

	      /* The X error handler may have deleted that frame
		 before we went back to retry_frame.  This must come
		 before any accesses to f->terminal.  */
	      if (!FRAME_LIVE_P (f))
		continue;

	      /* Mark all the scroll bars to be removed; we'll redeem
		 the ones we want when we redisplay their windows.  */
	      if (gcscrollbars && FRAME_TERMINAL (f)->condemn_scroll_bars_hook)
		FRAME_TERMINAL (f)->condemn_scroll_bars_hook (f);

	      if (FRAME_VISIBLE_P (f) && !FRAME_OBSCURED_P (f))
		{
		  /* Don't allow freeing images and faces for this
		     frame as long as the frame's update wasn't
		     completed.  This prevents crashes when some Lisp
		     that runs from the various hooks or font-lock
		     decides to clear the frame's image cache and face
		     cache, when the images and faces in those caches
		     are referenced by the desired matrix.  */
		  f->inhibit_clear_image_cache = true;
		  redisplay_windows (FRAME_ROOT_WINDOW (f));
		}
	      /* Remember that the invisible frames need to be redisplayed next
		 time they're visible.  */
	      else if (!REDISPLAY_SOME_P ())
		f->redisplay = true;

	      /* The X error handler may have deleted that frame.  */
	      if (!FRAME_LIVE_P (f))
		continue;

	      /* Any scroll bars which redisplay_windows should have
		 nuked should now go away.  */
	      if (gcscrollbars && FRAME_TERMINAL (f)->judge_scroll_bars_hook)
		FRAME_TERMINAL (f)->judge_scroll_bars_hook (f);

	      if (FRAME_VISIBLE_P (f) && !FRAME_OBSCURED_P (f))
		{
		  /* If fonts changed on visible frame, display again.  */
		  if (f->fonts_changed)
		    {
		      adjust_frame_glyphs (f);
		      /* Disable all redisplay optimizations for this
			 frame.  For the reasons, see the comment near
			 the previous call to adjust_frame_glyphs above.  */
		      SET_FRAME_GARBAGED (f);
		      f->fonts_changed = false;
		      goto retry_frame;
		    }

		  /* See if we have to hscroll.  */
		  if (!f->already_hscrolled_p)
		    {
		      f->already_hscrolled_p = true;
                      if (hscroll_retries <= MAX_HSCROLL_RETRIES
                          && hscroll_windows (f->root_window))
                        {
                          hscroll_retries++;
                          goto retry_frame;
                        }
		    }

		  /* If the frame's redisplay flag was not set before
		     we went about redisplaying its windows, but it is
		     set now, that means we employed some redisplay
		     optimizations inside redisplay_windows, and
		     bypassed producing some screen lines.  But if
		     f->redisplay is now set, it might mean the old
		     faces are no longer valid (e.g., if redisplaying
		     some window called some Lisp which defined a new
		     face or redefined an existing face), so trying to
		     use them in update_frame will segfault.
		     Therefore, we must redisplay this frame.  */
		  if (!f_redisplay_flag && f->redisplay)
                    goto retry_frame;
                  /* In some case (e.g., window resize), we notice
                     only during window updating that the window
                     content changed unpredictably (e.g., a GTK
                     scrollbar moved, or some Lisp hook that winds up
                     calling adjust_frame_glyphs) and that our
                     previous estimation of the frame content was
                     garbage.  We have to start over.  These cases
                     should be rare, so going all the way back to the
                     top of redisplay should be good enough.  */
                  if (FRAME_GARBAGED_P (f)
		      && garbaged_frame_retries++ < MAX_GARBAGED_FRAME_RETRIES)
                    goto retry;

#ifdef HAVE_WINDOW_SYSTEM
                  if (FRAME_WINDOW_P (f)
                      && FRAME_RIF (f)->clear_under_internal_border)
                    FRAME_RIF (f)->clear_under_internal_border (f);
#endif
		  /* Prevent various kinds of signals during display
		     update.  stdio is not robust about handling
		     signals, which can cause an apparent I/O error.  */
		  if (interrupt_input)
		    unrequest_sigio ();
		  STOP_POLLING;

		  pending |= update_frame (f, false, false);
		  /* On some platforms (at least MS-Windows), the
		     scroll_run_hook called from scrolling_window
		     called from update_frame could set the frame's
		     garbaged flag, in which case we need to redisplay
		     the frame.  Don't do that on TTY frames, since we
		     need to keep the garbaged flag in that case when
		     the frame has been resized.  */
                  if (FRAME_GARBAGED_P (f))
		    {
		      fset_redisplay (f);
		      f->garbaged = false;
		      goto retry_frame;
		    }
		  f->cursor_type_changed = false;
		  f->updated_p = true;
		  f->inhibit_clear_image_cache = false;
		}
	    }
	}

      eassert (EQ (XFRAME (selected_frame)->selected_window, selected_window));

      if (!pending)
	{
	  /* Do the mark_window_display_accurate after all windows have
	     been redisplayed because this call resets flags in buffers
	     which are needed for proper redisplay.  */
	  FOR_EACH_FRAME (tail, frame)
	    {
	      struct frame *f = XFRAME (frame);
              if (f->updated_p)
                {
		  f->redisplay = false;
		  f->garbaged = false;
                  mark_window_display_accurate (f->root_window, true);
                  if (FRAME_TERMINAL (f)->frame_up_to_date_hook)
                    FRAME_TERMINAL (f)->frame_up_to_date_hook (f);
                }
	    }
	}
    }
  else if (FRAME_VISIBLE_P (sf) && !FRAME_OBSCURED_P (sf))
    {
      sf->inhibit_clear_image_cache = true;
      displayed_buffer = XBUFFER (XWINDOW (selected_window)->contents);
      /* Use list_of_error, not Qerror, so that
	 we catch only errors and don't run the debugger.  */
      internal_condition_case_1 (redisplay_window_1, selected_window,
				 list_of_error,
				 redisplay_window_error);
      if (update_miniwindow_p)
	{
	  Lisp_Object mini_window = FRAME_MINIBUF_WINDOW (sf);

	  displayed_buffer = XBUFFER (XWINDOW (mini_window)->contents);
	  internal_condition_case_1 (redisplay_window_1, mini_window,
				     list_of_error,
				     redisplay_window_error);
	}

      /* Compare desired and current matrices, perform output.  */

    update:
      /* If fonts changed, display again.  Likewise if redisplay_window_1
	 above caused some change (e.g., a change in faces) that requires
	 considering the entire frame again.  */
      if (sf->fonts_changed || sf->redisplay)
	{
	  if (sf->redisplay)
	    {
	      /* Set this to force a more thorough redisplay.
		 Otherwise, we might immediately loop back to the
		 above "else-if" clause (since all the conditions that
		 led here might still be true), and we will then
		 infloop, because the selected-frame's redisplay flag
		 is not (and cannot be) reset.  */
	      windows_or_buffers_changed = 50;
	    }
	  goto retry;
	}

      /* Prevent freeing of realized faces, since desired matrices are
	 pending that reference the faces we computed and cached.  */
      inhibit_free_realized_faces = true;

      /* Prevent various kinds of signals during display update.
	 stdio is not robust about handling signals,
	 which can cause an apparent I/O error.  */
      if (interrupt_input)
	unrequest_sigio ();
      STOP_POLLING;

      if (FRAME_VISIBLE_P (sf) && !FRAME_OBSCURED_P (sf))
	{
          if (hscroll_retries <= MAX_HSCROLL_RETRIES
              && hscroll_windows (selected_window))
            {
              hscroll_retries++;
              goto retry;
            }

	  XWINDOW (selected_window)->must_be_updated_p = true;
	  pending = update_frame (sf, false, false);
	  sf->cursor_type_changed = false;
	  sf->inhibit_clear_image_cache = false;
	}

      /* We may have called echo_area_display at the top of this
	 function.  If the echo area is on another frame, that may
	 have put text on a frame other than the selected one, so the
	 above call to update_frame would not have caught it.  Catch
	 it here.  */
      Lisp_Object mini_window = FRAME_MINIBUF_WINDOW (sf);
      struct frame *mini_frame = XFRAME (WINDOW_FRAME (XWINDOW (mini_window)));

      if (mini_frame != sf && FRAME_WINDOW_P (mini_frame))
	{
	  XWINDOW (mini_window)->must_be_updated_p = true;
	  pending |= update_frame (mini_frame, false, false);
	  mini_frame->cursor_type_changed = false;
          if (!pending && hscroll_retries <= MAX_HSCROLL_RETRIES
              && hscroll_windows (mini_window))
            {
              hscroll_retries++;
              goto retry;
            }
	}
    }

  /* If display was paused because of pending input, make sure we do a
     thorough update the next time.  */
  if (pending)
    {
      /* Prevent the optimization at the beginning of
	 redisplay_internal that tries a single-line update of the
	 line containing the cursor in the selected window.  */
      CHARPOS (this_line_start_pos) = 0;

      /* Let the overlay arrow be updated the next time.  */
      update_overlay_arrows (0);

      /* If we pause after scrolling, some rows in the current
	 matrices of some windows are not valid.  */
      if (!WINDOW_FULL_WIDTH_P (w)
	  && !FRAME_WINDOW_P (XFRAME (w->frame)))
	update_mode_lines = 36;
    }
  else
    {
      if (!consider_all_windows_p)
	{
	  /* This has already been done above if
	     consider_all_windows_p is set.  */
	  if (XBUFFER (w->contents)->text->redisplay
	      && buffer_window_count (XBUFFER (w->contents)) > 1)
	    /* This can happen if b->text->redisplay was set during
	       jit-lock.  */
	    propagate_buffer_redisplay ();
	  mark_window_display_accurate_1 (w, true);

	  /* Say overlay arrows are up to date.  */
	  update_overlay_arrows (1);

	  if (FRAME_TERMINAL (sf)->frame_up_to_date_hook != 0)
	    FRAME_TERMINAL (sf)->frame_up_to_date_hook (sf);
	}

      update_mode_lines = 0;
      windows_or_buffers_changed = 0;
    }

  /* Start SIGIO interrupts coming again.  Having them off during the
     code above makes it less likely one will discard output, but not
     impossible, since there might be stuff in the system buffer here.
     But it is much hairier to try to do anything about that.  */
  if (interrupt_input)
    request_sigio ();
  RESUME_POLLING;

  /* If a frame has become visible which was not before, redisplay
     again, so that we display it.  Expose events for such a frame
     (which it gets when becoming visible) don't call the parts of
     redisplay constructing glyphs, so simply exposing a frame won't
     display anything in this case.  So, we have to display these
     frames here explicitly.  */
  if (!pending)
    {
      int new_count = 0;

      FOR_EACH_FRAME (tail, frame)
	{
	  if (XFRAME (frame)->visible)
	    new_count++;
	}

      if (new_count != number_of_visible_frames)
	windows_or_buffers_changed = 52;
    }

  /* Change frame size now if a change is pending.  */
  do_pending_window_change (true);

  /* If we just did a pending size change, or have additional
     visible frames, or selected_window changed, redisplay again.  */
  if ((windows_or_buffers_changed && !pending)
      || (WINDOWP (selected_window)
	  && (w = XWINDOW (selected_window)) != sw))
    goto retry;

  /* Clear the face and image caches.

     We used to do this only if consider_all_windows_p.  But the cache
     needs to be cleared if a timer creates images in the current
     buffer (e.g. the test case in Bug#6230).  */

  if (clear_face_cache_count > CLEAR_FACE_CACHE_COUNT)
    {
      clear_face_cache (false);
      clear_face_cache_count = 0;
    }

#ifdef HAVE_WINDOW_SYSTEM
  if (clear_image_cache_count > CLEAR_IMAGE_CACHE_COUNT)
    {
      clear_image_caches (Qnil);
      clear_image_cache_count = 0;
    }
#endif /* HAVE_WINDOW_SYSTEM */

 end_of_redisplay:
#ifdef HAVE_NS
  ns_set_doc_edited ();
#endif
  if (interrupt_input && interrupts_deferred)
    request_sigio ();

  /* We're done with this redisplay cycle, so reset the tick count in
     preparation for the next redisplay cycle.  */
  if (max_redisplay_ticks > 0)
    update_redisplay_ticks (0, NULL);

  unbind_to (count, Qnil);
  RESUME_POLLING;
}

static void
unwind_redisplay_preserve_echo_area (void)
{
  unblock_buffer_flips ();
}

/* Redisplay, but leave alone any recent echo area message unless
   another message has been requested in its place.

   This is useful in situations where you need to redisplay but no
   user action has occurred, making it inappropriate for the message
   area to be cleared.  See tracking_off and
   wait_reading_process_output for examples of these situations.

   FROM_WHERE is an integer saying from where this function was
   called.  This is useful for debugging.  */

void
redisplay_preserve_echo_area (int from_where)
{
  redisplay_trace ("redisplay_preserve_echo_area (%d)\n", from_where);

  block_input ();
  specpdl_ref count = SPECPDL_INDEX ();
  record_unwind_protect_void (unwind_redisplay_preserve_echo_area);
  block_buffer_flips ();
  unblock_input ();

  if (!NILP (echo_area_buffer[1]))
    {
      /* We have a previously displayed message, but no current
	 message.  Redisplay the previous message.  */
      display_last_displayed_message_p = true;
      redisplay_internal ();
      display_last_displayed_message_p = false;
    }
  else
    redisplay_internal ();

  flush_frame (SELECTED_FRAME ());
  unbind_to (count, Qnil);
}


/* Function registered with record_unwind_protect in redisplay_internal.  */

static void
unwind_redisplay (void)
{
  redisplaying_p = false;
  unblock_buffer_flips ();
}

/* Function registered with record_unwind_protect before calling
   start_display outside of redisplay_internal.  */
void
unwind_display_working_on_window (void)
{
  display_working_on_window_p = false;
}

/* Mark the display of leaf window W as accurate or inaccurate.
   If ACCURATE_P, mark display of W as accurate.
   If !ACCURATE_P, arrange for W to be redisplayed the next
   time redisplay_internal is called.  */

static void
mark_window_display_accurate_1 (struct window *w, bool accurate_p)
{
  struct buffer *b = XBUFFER (w->contents);

  w->last_modified = accurate_p ? BUF_MODIFF (b) : 0;
  w->last_overlay_modified = accurate_p ? BUF_OVERLAY_MODIFF (b) : 0;
  w->last_had_star = BUF_MODIFF (b) > BUF_SAVE_MODIFF (b);

  if (accurate_p)
    {
      b->clip_changed = false;
      b->prevent_redisplay_optimizations_p = false;
      eassert (buffer_window_count (b) > 0);
      /* Resetting b->text->redisplay is problematic!
	 In order to make it safer to do it here, redisplay_internal must
	 have copied all b->text->redisplay to their respective windows.  */
      b->text->redisplay = false;

      BUF_UNCHANGED_MODIFIED (b) = BUF_MODIFF (b);
      BUF_OVERLAY_UNCHANGED_MODIFIED (b) = BUF_OVERLAY_MODIFF (b);
      BUF_BEG_UNCHANGED (b) = BUF_GPT (b) - BUF_BEG (b);
      BUF_END_UNCHANGED (b) = BUF_Z (b) - BUF_GPT (b);

      w->current_matrix->buffer = b;
      w->current_matrix->begv = BUF_BEGV (b);
      w->current_matrix->zv = BUF_ZV (b);
      w->current_matrix->header_line_p = window_wants_header_line (w);
      w->current_matrix->tab_line_p = window_wants_tab_line (w);

      w->last_cursor_vpos = w->cursor.vpos;
      w->last_cursor_off_p = w->cursor_off_p;

      if (w == XWINDOW (selected_window))
	w->last_point = BUF_PT (b);
      else
	w->last_point = marker_position (w->pointm);

      w->window_end_valid = true;
      w->update_mode_line = false;
      w->preserve_vscroll_p = false;
    }

  w->redisplay = !accurate_p;
}


/* Mark the display of windows in the window tree rooted at WINDOW as
   accurate or inaccurate.  If ACCURATE_P, mark display of
   windows as accurate.  If !ACCURATE_P, arrange for windows to
   be redisplayed the next time redisplay_internal is called.  */

void
mark_window_display_accurate (Lisp_Object window, bool accurate_p)
{
  struct window *w;

  for (; !NILP (window); window = w->next)
    {
      w = XWINDOW (window);
      if (WINDOWP (w->contents))
	mark_window_display_accurate (w->contents, accurate_p);
      else
	mark_window_display_accurate_1 (w, accurate_p);
    }

  if (accurate_p)
    update_overlay_arrows (1);
  else
    /* Force a thorough redisplay the next time by setting
       last_arrow_position and last_arrow_string to t, which is
       unequal to any useful value of Voverlay_arrow_...  */
    update_overlay_arrows (-1);
}


/* Return value in display table DP (Lisp_Char_Table *) for character
   C.  Since a display table doesn't have any parent, we don't have to
   follow parent.  Do not call this function directly but use the
   macro DISP_CHAR_VECTOR.  */

Lisp_Object
disp_char_vector (struct Lisp_Char_Table *dp, int c)
{
  Lisp_Object val;

  if (ASCII_CHAR_P (c))
    {
      val = dp->ascii;
      if (SUB_CHAR_TABLE_P (val))
	val = XSUB_CHAR_TABLE (val)->contents[c];
    }
  else
    {
      Lisp_Object table;

      XSETCHAR_TABLE (table, dp);
      val = char_table_ref (table, c);
    }
  if (NILP (val))
    val = dp->defalt;
  return val;
}

static int buffer_flip_blocked_depth;

static void
block_buffer_flips (void)
{
  eassert (buffer_flip_blocked_depth >= 0);
  buffer_flip_blocked_depth++;
}

static void
unblock_buffer_flips (void)
{
  eassert (buffer_flip_blocked_depth > 0);
  if (--buffer_flip_blocked_depth == 0)
    {
      Lisp_Object tail, frame;
      block_input ();
      FOR_EACH_FRAME (tail, frame)
        {
          struct frame *f = XFRAME (frame);
          if (FRAME_TERMINAL (f)->buffer_flipping_unblocked_hook)
            (*FRAME_TERMINAL (f)->buffer_flipping_unblocked_hook) (f);
        }
      unblock_input ();
    }
}

bool
buffer_flipping_blocked_p (void)
{
  return buffer_flip_blocked_depth > 0;
}


/***********************************************************************
			   Window Redisplay
 ***********************************************************************/

/* Redisplay all leaf windows in the window tree rooted at WINDOW.  */

static void
redisplay_windows (Lisp_Object window)
{
  while (!NILP (window))
    {
      struct window *w = XWINDOW (window);

      if (WINDOWP (w->contents))
	redisplay_windows (w->contents);
      else if (BUFFERP (w->contents))
	{
	  displayed_buffer = XBUFFER (w->contents);
	  /* Use list_of_error, not Qerror, so that
	     we catch only errors and don't run the debugger.  */
	  internal_condition_case_1 (redisplay_window_0, window,
				     list_of_error,
				     redisplay_window_error);
	}

      window = w->next;
    }
}

static Lisp_Object
redisplay_window_error (Lisp_Object error_data)
{
  displayed_buffer->display_error_modiff = BUF_MODIFF (displayed_buffer);

  /* When in redisplay, the error is captured and not shown.  Arrange
     for it to be shown later.  */
  if (max_redisplay_ticks > 0
      && CONSP (error_data)
      && EQ (XCAR (error_data), Qerror)
      && STRINGP (XCAR (XCDR (error_data))))
    Vdelayed_warnings_list = Fcons (list2 (XCAR (error_data),
					   XCAR (XCDR (error_data))),
				    Vdelayed_warnings_list);
  return Qnil;
}

static Lisp_Object
redisplay_window_0 (Lisp_Object window)
{
  if (displayed_buffer->display_error_modiff < BUF_MODIFF (displayed_buffer))
    redisplay_window (window, false);
  return Qnil;
}

static Lisp_Object
redisplay_window_1 (Lisp_Object window)
{
  if (displayed_buffer->display_error_modiff < BUF_MODIFF (displayed_buffer))
    redisplay_window (window, true);
  return Qnil;
}


/***********************************************************************
		      Aborting runaway redisplay
 ***********************************************************************/

/* Update the redisplay-tick count for window W, and signal an error
   if the tick count is above some threshold, indicating that
   redisplay of the window takes "too long".

   TICKS is the amount of ticks to add to the W's current count; zero
   means to initialize the tick count to zero.

   W can be NULL if TICKS is zero: that means unconditionally
   re-initialize the current tick count to zero.

   W can also be NULL if the caller doesn't know which window is being
   processed by the display code.  In that case, if TICKS is non-zero,
   we assume it's the last window that shows the current buffer.  */
void
update_redisplay_ticks (int ticks, struct window *w)
{
  /* This keeps track of the window on which redisplay is working.  */
  static struct window *cwindow;
  static EMACS_INT window_ticks;

  /* We only initialize the count if this is a different window or
     NULL.  Otherwise, this is a call from init_iterator for the same
     window we tracked before, and we should keep the count.  */
  if (!ticks && w != cwindow)
    {
      cwindow = w;
      window_ticks = 0;
    }
  /* Some callers can be run in contexts unrelated to display code, so
     don't abort them and don't update the tick count in those cases.  */
  if ((!w && !redisplaying_p && !display_working_on_window_p)
      /* We never disable redisplay of a mini-window, since that is
	 absolutely essential for communicating with Emacs.  */
      || (w && MINI_WINDOW_P (w)))
    return;

  if (ticks > 0)
    window_ticks += ticks;
  if (max_redisplay_ticks > 0 && window_ticks > max_redisplay_ticks)
    {
      /* In addition to a buffer, this could be a window (for non-leaf
	 windows, not expected here) or nil (for pseudo-windows like
	 the one used for the native tool bar).  */
      Lisp_Object contents = w ? w->contents : Qnil;
      char *bufname =
	NILP (contents)
	? SSDATA (BVAR (current_buffer, name))
	: (BUFFERP (contents)
	   ? SSDATA (BVAR (XBUFFER (contents), name))
	   : (char *) "<unknown>");

      windows_or_buffers_changed = 177;
      /* scrolling_window depends too much on the glyph matrices being
	 correct, and we cannot guarantee that if we abort the
	 redisplay of this window.  */
      if (w && w->desired_matrix)
	w->desired_matrix->no_scrolling_p = true;
      error ("Window showing buffer %s takes too long to redisplay", bufname);
    }
}



/* Set cursor position of W.  PT is assumed to be displayed in ROW.
   DELTA and DELTA_BYTES are the numbers of characters and bytes by
   which positions recorded in ROW differ from current buffer
   positions.

   Return true iff cursor is on this row.  */

static bool
set_cursor_from_row (struct window *w, struct glyph_row *row,
		     struct glyph_matrix *matrix,
		     ptrdiff_t delta, ptrdiff_t delta_bytes,
		     int dy, int dvpos)
{
  struct glyph *glyph = row->glyphs[TEXT_AREA];
  struct glyph *end = glyph + row->used[TEXT_AREA];
  struct glyph *cursor = NULL;
  /* The last known character position in row.  */
  ptrdiff_t last_pos = MATRIX_ROW_START_CHARPOS (row) + delta;
  int x = row->x;
  ptrdiff_t pt_old = PT - delta;
  ptrdiff_t pos_before = MATRIX_ROW_START_CHARPOS (row) + delta;
  ptrdiff_t pos_after = MATRIX_ROW_END_CHARPOS (row) + delta;
  struct glyph *glyph_before = glyph - 1, *glyph_after = end;
  /* A glyph beyond the edge of TEXT_AREA which we should never
     touch.  */
  struct glyph *glyphs_end = end;
  /* True means we've found a match for cursor position, but that
     glyph has the avoid_cursor_p flag set.  */
  bool match_with_avoid_cursor = false;
  /* True means we've seen at least one glyph that came from a
     display string.  */
  bool string_seen = false;
  /* Largest and smallest buffer positions seen so far during scan of
     glyph row.  */
  ptrdiff_t bpos_max = pos_before;
  ptrdiff_t bpos_min = pos_after;
  /* Last buffer position covered by an overlay string with an integer
     `cursor' property.  */
  ptrdiff_t bpos_covered = 0;
  /* True means the display string on which to display the cursor
     comes from a text property, not from an overlay.  */
  bool string_from_text_prop = false;

  /* Don't even try doing anything if called for a mode-line or
     header-line or tab-line row, since the rest of the code isn't
     prepared to deal with such calamities.  */
  eassert (!row->mode_line_p);
  if (row->mode_line_p)
    return false;

  /* Skip over glyphs not having an object at the start and the end of
     the row.  These are special glyphs like truncation marks on
     terminal frames.  */
  if (MATRIX_ROW_DISPLAYS_TEXT_P (row))
    {
      if (!row->reversed_p)
	{
	  while (glyph < end
		 && NILP (glyph->object)
		 && glyph->charpos < 0)
	    {
	      x += glyph->pixel_width;
	      ++glyph;
	    }
	  while (end > glyph
		 && NILP ((end - 1)->object)
		 /* CHARPOS is zero for blanks and stretch glyphs
		    inserted by extend_face_to_end_of_line.  */
		 && (end - 1)->charpos <= 0)
	    --end;
	  glyph_before = glyph - 1;
	  glyph_after = end;
	}
      else
	{
	  struct glyph *g;

	  /* If the glyph row is reversed, we need to process it from back
	     to front, so swap the edge pointers.  */
	  glyphs_end = end = glyph - 1;
	  glyph += row->used[TEXT_AREA] - 1;

	  while (glyph > end + 1
		 && NILP (glyph->object)
		 && glyph->charpos < 0)
	    --glyph;
	  if (NILP (glyph->object) && glyph->charpos < 0)
	    --glyph;
	  /* By default, in reversed rows we put the cursor on the
	     rightmost (first in the reading order) glyph.  */
	  for (x = 0, g = end + 1; g < glyph; g++)
	    x += g->pixel_width;
	  while (end < glyph
		 && NILP ((end + 1)->object)
		 && (end + 1)->charpos <= 0)
	    ++end;
	  glyph_before = glyph + 1;
	  glyph_after = end;
	}
    }
  else if (row->reversed_p)
    {
      /* In R2L rows that don't display text, put the cursor on the
	 rightmost glyph.  Case in point: an empty last line that is
	 part of an R2L paragraph.  */
      cursor = end - 1;
      /* Avoid placing the cursor on the last glyph of the row, where
	 on terminal frames we hold the vertical border between
	 adjacent windows.  */
      if (!FRAME_WINDOW_P (WINDOW_XFRAME (w))
	  && !WINDOW_RIGHTMOST_P (w)
	  && cursor == row->glyphs[LAST_AREA] - 1)
	cursor--;
      x = -1;	/* will be computed below, at label compute_x */
    }

  /* Step 1: Try to find the glyph whose character position
     corresponds to point.  If that's not possible, find 2 glyphs
     whose character positions are the closest to point, one before
     point, the other after it.  */
  if (!row->reversed_p)
    while (/* not marched to end of glyph row */
	   glyph < end
	   /* glyph was not inserted by redisplay for internal purposes */
	   && !NILP (glyph->object))
      {
	if (BUFFERP (glyph->object))
	  {
	    ptrdiff_t dpos = glyph->charpos - pt_old;

	    if (glyph->charpos > bpos_max)
	      bpos_max = glyph->charpos;
	    if (glyph->charpos < bpos_min)
	      bpos_min = glyph->charpos;
	    if (!glyph->avoid_cursor_p)
	      {
		/* If we hit point, we've found the glyph on which to
		   display the cursor.  */
		if (dpos == 0)
		  {
		    match_with_avoid_cursor = false;
		    break;
		  }
		/* See if we've found a better approximation to
		   POS_BEFORE or to POS_AFTER.  */
		if (0 > dpos && dpos > pos_before - pt_old)
		  {
		    pos_before = glyph->charpos;
		    glyph_before = glyph;
		  }
		else if (0 < dpos && dpos < pos_after - pt_old)
		  {
		    pos_after = glyph->charpos;
		    glyph_after = glyph;
		  }
	      }
	    else if (dpos == 0)
	      match_with_avoid_cursor = true;
	  }
	else if (STRINGP (glyph->object))
	  {
	    Lisp_Object chprop;
	    ptrdiff_t glyph_pos = glyph->charpos;

	    chprop = Fget_char_property (make_fixnum (glyph_pos), Qcursor,
					 glyph->object);
	    if (!NILP (chprop))
	      {
		/* If the string came from a `display' text property,
		   look up the buffer position of that property and
		   use that position to update bpos_max, as if we
		   actually saw such a position in one of the row's
		   glyphs.  This helps with supporting integer values
		   of `cursor' property on the display string in
		   situations where most or all of the row's buffer
		   text is completely covered by display properties,
		   so that no glyph with valid buffer positions is
		   ever seen in the row.  */
		ptrdiff_t prop_pos =
		  string_buffer_position_lim (glyph->object, pos_before,
					      pos_after, false);

		if (prop_pos >= pos_before)
		  bpos_max = prop_pos;
	      }
	    if (FIXNUMP (chprop))
	      {
		bpos_covered = bpos_max + XFIXNUM (chprop);
		/* If the `cursor' property covers buffer positions up
		   to and including point, we should display cursor on
		   this glyph.  Note that, if a `cursor' property on one
		   of the string's characters has an integer value, we
		   will break out of the loop below _before_ we get to
		   the position match above.  IOW, integer values of
		   the `cursor' property override the "exact match for
		   point" strategy of positioning the cursor.  */
		/* Implementation note: bpos_max == pt_old when, e.g.,
		   we are in an empty line, where bpos_max is set to
		   MATRIX_ROW_START_CHARPOS, see above.  */
		if (bpos_max <= pt_old && bpos_covered >= pt_old)
		  {
		    cursor = glyph;
		    break;
		  }
	      }

	    string_seen = true;
	  }
	x += glyph->pixel_width;
	++glyph;
      }
  else if (glyph > end)	/* row is reversed */
    while (!NILP (glyph->object))
      {
	if (BUFFERP (glyph->object))
	  {
	    ptrdiff_t dpos = glyph->charpos - pt_old;

	    if (glyph->charpos > bpos_max)
	      bpos_max = glyph->charpos;
	    if (glyph->charpos < bpos_min)
	      bpos_min = glyph->charpos;
	    if (!glyph->avoid_cursor_p)
	      {
		if (dpos == 0)
		  {
		    match_with_avoid_cursor = false;
		    break;
		  }
		if (0 > dpos && dpos > pos_before - pt_old)
		  {
		    pos_before = glyph->charpos;
		    glyph_before = glyph;
		  }
		else if (0 < dpos && dpos < pos_after - pt_old)
		  {
		    pos_after = glyph->charpos;
		    glyph_after = glyph;
		  }
	      }
	    else if (dpos == 0)
	      match_with_avoid_cursor = true;
	  }
	else if (STRINGP (glyph->object))
	  {
	    Lisp_Object chprop;
	    ptrdiff_t glyph_pos = glyph->charpos;

	    chprop = Fget_char_property (make_fixnum (glyph_pos), Qcursor,
					 glyph->object);
	    if (!NILP (chprop))
	      {
		ptrdiff_t prop_pos =
		  string_buffer_position_lim (glyph->object, pos_before,
					      pos_after, false);

		if (prop_pos >= pos_before)
		  bpos_max = prop_pos;
	      }
	    if (FIXNUMP (chprop))
	      {
		bpos_covered = bpos_max + XFIXNUM (chprop);
		/* If the `cursor' property covers buffer positions up
		   to and including point, we should display cursor on
		   this glyph.  */
		if (bpos_max <= pt_old && bpos_covered >= pt_old)
		  {
		    cursor = glyph;
		    break;
		  }
	      }
	    string_seen = true;
	  }
	--glyph;
	if (glyph == glyphs_end) /* don't dereference outside TEXT_AREA */
	  {
	    x--;		/* can't use any pixel_width */
	    break;
	  }
	x -= glyph->pixel_width;
    }

  /* Step 2: If we didn't find an exact match for point, we need to
     look for a proper place to put the cursor among glyphs between
     GLYPH_BEFORE and GLYPH_AFTER.  */
  if (!((row->reversed_p ? glyph > glyphs_end : glyph < glyphs_end)
	&& BUFFERP (glyph->object) && glyph->charpos == pt_old)
      && !(bpos_max <= pt_old && pt_old <= bpos_covered))
    {
      /* An empty line has a single glyph whose OBJECT is nil and
	 whose CHARPOS is the position of a newline on that line.
	 Note that on a TTY, there are more glyphs after that, which
	 were produced by extend_face_to_end_of_line, but their
	 CHARPOS is zero or negative.  */
      bool empty_line_p =
	((row->reversed_p ? glyph > glyphs_end : glyph < glyphs_end)
	 && NILP (glyph->object) && glyph->charpos > 0
	 /* On a TTY, continued and truncated rows also have a glyph at
	    their end whose OBJECT is nil and whose CHARPOS is
	    positive (the continuation and truncation glyphs), but such
	    rows are obviously not "empty".  */
	 && !(row->continued_p || row->truncated_on_right_p));

      if (row->ends_in_ellipsis_p && pos_after == last_pos)
	{
	  ptrdiff_t ellipsis_pos;

	  /* Scan back over the ellipsis glyphs.  */
	  if (!row->reversed_p)
	    {
	      ellipsis_pos = (glyph - 1)->charpos;
	      while (glyph > row->glyphs[TEXT_AREA]
		     && (glyph - 1)->charpos == ellipsis_pos)
		glyph--, x -= glyph->pixel_width;
	      /* That loop always goes one position too far, including
		 the glyph before the ellipsis.  So scan forward over
		 that one.  */
	      x += glyph->pixel_width;
	      glyph++;
	    }
	  else	/* row is reversed */
	    {
	      ellipsis_pos = (glyph + 1)->charpos;
	      while (glyph < row->glyphs[TEXT_AREA] + row->used[TEXT_AREA] - 1
		     && (glyph + 1)->charpos == ellipsis_pos)
		glyph++, x += glyph->pixel_width;
	      x -= glyph->pixel_width;
	      glyph--;
	    }
	}
      else if (match_with_avoid_cursor)
	{
	  cursor = glyph_after;
	  x = -1;
	}
      else if (string_seen)
	{
	  int incr = row->reversed_p ? -1 : +1;

	  /* Need to find the glyph that came out of a string which is
	     present at point.  That glyph is somewhere between
	     GLYPH_BEFORE and GLYPH_AFTER, and it came from a string
	     positioned between POS_BEFORE and POS_AFTER in the
	     buffer.  */
	  struct glyph *start, *stop;
	  ptrdiff_t pos = pos_before;

	  x = -1;

	  /* If the row ends in a newline from a display string,
	     reordering could have moved the glyphs belonging to the
	     string out of the [GLYPH_BEFORE..GLYPH_AFTER] range.  So
	     in this case we extend the search to the last glyph in
	     the row that was not inserted by redisplay.  */
	  if (row->ends_in_newline_from_string_p)
	    {
	      glyph_after = end;
	      pos_after = MATRIX_ROW_END_CHARPOS (row) + delta;
	    }

	  /* GLYPH_BEFORE and GLYPH_AFTER are the glyphs that
	     correspond to POS_BEFORE and POS_AFTER, respectively.  We
	     need START and STOP in the order that corresponds to the
	     row's direction as given by its reversed_p flag.  If the
	     directionality of characters between POS_BEFORE and
	     POS_AFTER is the opposite of the row's base direction,
	     these characters will have been reordered for display,
	     and we need to reverse START and STOP.  */
	  if (!row->reversed_p)
	    {
	      start = min (glyph_before, glyph_after);
	      stop = max (glyph_before, glyph_after);
	    }
	  else
	    {
	      start = max (glyph_before, glyph_after);
	      stop = min (glyph_before, glyph_after);
	    }
	  for (glyph = start + incr;
	       row->reversed_p ? glyph > stop : glyph < stop; )
	    {

	      /* Any glyphs that come from the buffer are here because
		 of bidi reordering.  Skip them, and only pay
		 attention to glyphs that came from some string.  */
	      if (STRINGP (glyph->object))
		{
		  Lisp_Object str;
		  ptrdiff_t tem;
		  /* If the display property covers the newline, we
		     need to search for it one position farther.  */
		  ptrdiff_t lim = pos_after
		    + (pos_after == MATRIX_ROW_END_CHARPOS (row) + delta);

		  string_from_text_prop = false;
		  str = glyph->object;
		  tem = string_buffer_position_lim (str, pos, lim, false);
		  if (tem == 0	/* from overlay */
		      || pos <= tem)
		    {
		      /* If the string from which this glyph came is
			 found in the buffer at point, or at position
			 that is closer to point than pos_after, then
			 we've found the glyph we've been looking for.
			 If it comes from an overlay (tem == 0), and
			 it has the `cursor' property on one of its
			 glyphs, record that glyph as a candidate for
			 displaying the cursor.  (As in the
			 unidirectional version, we will display the
			 cursor on the last candidate we find.)  */
		      if (tem == 0
			  || tem == pt_old
			  || (tem - pt_old > 0 && tem < pos_after))
			{
			  /* The glyphs from this string could have
			     been reordered.  Find the one with the
			     smallest string position.  Or there could
			     be a character in the string with the
			     `cursor' property, which means display
			     cursor on that character's glyph.  */
			  ptrdiff_t strpos = glyph->charpos;

			  if (tem)
			    {
			      cursor = glyph;
			      string_from_text_prop = true;
			    }
			  for ( ;
			       (row->reversed_p ? glyph > stop : glyph < stop)
				 && EQ (glyph->object, str);
			       glyph += incr)
			    {
			      Lisp_Object cprop;
			      ptrdiff_t gpos = glyph->charpos;

			      cprop = Fget_char_property (make_fixnum (gpos),
							  Qcursor,
							  glyph->object);
			      if (!NILP (cprop))
				{
				  cursor = glyph;
				  break;
				}
			      if (tem && glyph->charpos < strpos)
				{
				  strpos = glyph->charpos;
				  cursor = glyph;
				}
			    }

			  if (tem == pt_old
			      || (tem - pt_old > 0 && tem < pos_after))
			    goto compute_x;
			}
		      if (tem)
			pos = tem + 1; /* don't find previous instances */
		    }
		  /* This string is not what we want; skip all of the
		     glyphs that came from it.  */
		  while ((row->reversed_p ? glyph > stop : glyph < stop)
			 && EQ (glyph->object, str))
		    glyph += incr;
		}
	      else
		glyph += incr;
	    }

	  /* If we reached the end of the line, and END was from a string,
	     the cursor is not on this line.  */
	  if (cursor == NULL
	      && (row->reversed_p ? glyph <= end : glyph >= end)
	      && (row->reversed_p ? end > glyphs_end : end < glyphs_end)
	      && STRINGP (end->object)
	      && row->continued_p)
	    return false;
	}
      /* A truncated row may not include PT among its character positions.
	 Setting the cursor inside the scroll margin will trigger
	 recalculation of hscroll in hscroll_window_tree.  But if a
	 display string covers point, defer to the string-handling
	 code below to figure this out.  */
      else if (row->truncated_on_left_p && pt_old < bpos_min)
	{
	  cursor = glyph_before;
	  x = -1;
	}
      else if ((row->truncated_on_right_p && pt_old > bpos_max)
	       /* Zero-width characters produce no glyphs.  */
	       || (!empty_line_p
		   && (row->reversed_p
		       ? glyph_after > glyphs_end
		       : glyph_after < glyphs_end)))
	{
	  cursor = glyph_after;
	  x = -1;
	}
    }

 compute_x:
  if (cursor != NULL)
    glyph = cursor;
  else if (glyph == glyphs_end
	   && pos_before == pos_after
	   && STRINGP ((row->reversed_p
			? row->glyphs[TEXT_AREA] + row->used[TEXT_AREA] - 1
			: row->glyphs[TEXT_AREA])->object))
    {
      /* If all the glyphs of this row came from strings, put the
	 cursor on the first glyph of the row.  This avoids having the
	 cursor outside of the text area in this very rare and hard
	 use case.  */
      glyph =
	row->reversed_p
	? row->glyphs[TEXT_AREA] + row->used[TEXT_AREA] - 1
	: row->glyphs[TEXT_AREA];
    }
  if (x < 0)
    {
      struct glyph *g;

      /* Need to compute x that corresponds to GLYPH.  */
      for (g = row->glyphs[TEXT_AREA], x = row->x; g < glyph; g++)
	{
	  if (g >= row->glyphs[TEXT_AREA] + row->used[TEXT_AREA])
	    emacs_abort ();
	  x += g->pixel_width;
	}
    }

  /* ROW could be part of a continued line, which, under bidi
     reordering, might have other rows whose start and end charpos
     occlude point.  Only set w->cursor if we found a better
     approximation to the cursor position than we have from previously
     examined candidate rows belonging to the same continued line.  */
  if (/* We already have a candidate row.  */
      w->cursor.vpos >= 0
      /* That candidate is not the row we are processing.  */
      && MATRIX_ROW (matrix, w->cursor.vpos) != row
      /* Make sure cursor.vpos specifies a row whose start and end
	 charpos occlude point, and it is valid candidate for being a
	 cursor-row.  This is because some callers of this function
	 leave cursor.vpos at the row where the cursor was displayed
	 during the last redisplay cycle.  */
      && MATRIX_ROW_START_CHARPOS (MATRIX_ROW (matrix, w->cursor.vpos)) <= pt_old
      && pt_old <= MATRIX_ROW_END_CHARPOS (MATRIX_ROW (matrix, w->cursor.vpos))
      && cursor_row_p (MATRIX_ROW (matrix, w->cursor.vpos)))
    {
      struct glyph *g1
	= MATRIX_ROW_GLYPH_START (matrix, w->cursor.vpos) + w->cursor.hpos;

      /* Don't consider glyphs that are outside TEXT_AREA.  */
      if (!(row->reversed_p ? glyph > glyphs_end : glyph < glyphs_end))
	return false;
      /* Keep the candidate whose buffer position is the closest to
	 point or has the `cursor' property.  */
      if (/* Previous candidate is a glyph in TEXT_AREA of that row.  */
	  w->cursor.hpos >= 0
	  && w->cursor.hpos < MATRIX_ROW_USED (matrix, w->cursor.vpos)
	  && ((BUFFERP (g1->object)
	       && (g1->charpos == pt_old /* An exact match always wins.  */
		   || (BUFFERP (glyph->object)
		       && eabs (g1->charpos - pt_old)
		       < eabs (glyph->charpos - pt_old))))
	      /* Previous candidate is a glyph from a string that has
		 a non-nil `cursor' property.  */
	      || (STRINGP (g1->object)
		  && (!NILP (Fget_char_property (make_fixnum (g1->charpos),
						Qcursor, g1->object))
		      /* Previous candidate is from the same display
			 string as this one, and the display string
			 came from a text property.  */
		      || (EQ (g1->object, glyph->object)
			  && string_from_text_prop)
		      /* this candidate is from newline and its
			 position is not an exact match */
		      || (NILP (glyph->object)
			  && glyph->charpos != pt_old)))))
	return false;
      /* If this candidate gives an exact match, use that.  */
      if (!((BUFFERP (glyph->object) && glyph->charpos == pt_old)
	    /* If this candidate is a glyph created for the
	       terminating newline of a line, and point is on that
	       newline, it wins because it's an exact match.  */
	    || (!row->continued_p
		&& NILP (glyph->object)
		&& glyph->charpos == 0
		&& pt_old == MATRIX_ROW_END_CHARPOS (row) - 1))
	  /* Otherwise, keep the candidate that comes from a row
	     spanning less buffer positions.  This may win when one or
	     both candidate positions are on glyphs that came from
	     display strings, for which we cannot compare buffer
	     positions.  */
	  && MATRIX_ROW_END_CHARPOS (MATRIX_ROW (matrix, w->cursor.vpos))
	     - MATRIX_ROW_START_CHARPOS (MATRIX_ROW (matrix, w->cursor.vpos))
	     < MATRIX_ROW_END_CHARPOS (row) - MATRIX_ROW_START_CHARPOS (row))
	return false;
    }
  w->cursor.hpos = glyph - row->glyphs[TEXT_AREA];
  w->cursor.x = x;
  w->cursor.vpos = MATRIX_ROW_VPOS (row, matrix) + dvpos;
  w->cursor.y = row->y + dy;

  if (w == XWINDOW (selected_window))
    {
      if (!row->continued_p
	  && !MATRIX_ROW_CONTINUATION_LINE_P (row)
	  && row->x == 0)
	{
	  this_line_buffer = XBUFFER (w->contents);

	  CHARPOS (this_line_start_pos)
	    = MATRIX_ROW_START_CHARPOS (row) + delta;
	  BYTEPOS (this_line_start_pos)
	    = MATRIX_ROW_START_BYTEPOS (row) + delta_bytes;

	  CHARPOS (this_line_end_pos)
	    = Z - (MATRIX_ROW_END_CHARPOS (row) + delta);
	  BYTEPOS (this_line_end_pos)
	    = Z_BYTE - (MATRIX_ROW_END_BYTEPOS (row) + delta_bytes);

	  this_line_y = w->cursor.y;
	  this_line_pixel_height = row->height;
	  this_line_vpos = w->cursor.vpos;
	  this_line_start_x = row->x;
	}
      else
	CHARPOS (this_line_start_pos) = 0;
    }

  return true;
}


/* Run window scroll functions, if any, for WINDOW with new window
   start STARTP.  Sets the window start of WINDOW to that position.

   We assume that the window's buffer is really current.  */

static struct text_pos
run_window_scroll_functions (Lisp_Object window, struct text_pos startp)
{
  struct window *w = XWINDOW (window);
  SET_MARKER_FROM_TEXT_POS (w->start, startp);

  eassert (current_buffer == XBUFFER (w->contents));

  if (!NILP (Vwindow_scroll_functions))
    {
      specpdl_ref count = SPECPDL_INDEX ();
      specbind (Qinhibit_quit, Qt);
      safe_run_hooks_2
	(Qwindow_scroll_functions, window, make_fixnum (CHARPOS (startp)));
      unbind_to (count, Qnil);
      SET_TEXT_POS_FROM_MARKER (startp, w->start);
      /* In case the hook functions switch buffers.  */
      set_buffer_internal (XBUFFER (w->contents));
    }

  return startp;
}


/* Make sure the line containing the cursor is fully visible.
   A value of true means there is nothing to be done.
   (Either the line is fully visible, or it cannot be made so,
   or we cannot tell.)

   If FORCE_P, return false even if partial visible cursor row
   is higher than window.

   If CURRENT_MATRIX_P, use the information from the
   window's current glyph matrix; otherwise use the desired glyph
   matrix.

   If JUST_TEST_USER_PREFERENCE_P, just test what the value of
   make-cursor-row-fully-visible requires, don't test the actual
   cursor position.  The assumption is that in that case the caller
   performs the necessary testing of the cursor position.

   A value of false means the caller should do scrolling
   as if point had gone off the screen.  */

static bool
cursor_row_fully_visible_p (struct window *w, bool force_p,
			    bool current_matrix_p,
			    bool just_test_user_preference_p)
{
  struct glyph_matrix *matrix;
  struct glyph_row *row;
  int window_height;
  Lisp_Object mclfv_p =
    buffer_local_value (Qmake_cursor_line_fully_visible, w->contents);

  /* If no local binding, use the global value.  */
  if (BASE_EQ (mclfv_p, Qunbound))
    mclfv_p = Vmake_cursor_line_fully_visible;
  /* Follow mode sets the variable to a Lisp function in buffers that
     are under Follow mode.  */
  if (FUNCTIONP (mclfv_p))
    {
      Lisp_Object window;
      XSETWINDOW (window, w);
      /* Implementation note: if the function we call here signals an
	 error, we will NOT scroll when the cursor is partially-visible.  */
      Lisp_Object val = safe_call1 (mclfv_p, window);
      if (NILP (val))
	return true;
      else if (just_test_user_preference_p)
	return false;
    }
  else if (NILP (mclfv_p))
    return true;
  else if (just_test_user_preference_p)
    return false;

  /* It's not always possible to find the cursor, e.g, when a window
     is full of overlay strings.  Don't do anything in that case.  */
  if (w->cursor.vpos < 0)
    return true;

  matrix = current_matrix_p ? w->current_matrix : w->desired_matrix;
  row = MATRIX_ROW (matrix, w->cursor.vpos);

  /* If the cursor row is not partially visible, there's nothing to do.  */
  if (!MATRIX_ROW_PARTIALLY_VISIBLE_P (w, row))
    return true;

  /* If the row the cursor is in is taller than the window's height,
     it's not clear what to do, so do nothing.  */
  window_height = window_box_height (w);
  if (row->height >= window_height)
    {
      if (!force_p || MINI_WINDOW_P (w)
	  || w->vscroll || w->cursor.vpos == 0)
	return true;
    }
  return false;
}


/* Try scrolling PT into view in window WINDOW.  JUST_THIS_ONE_P
   means only WINDOW is redisplayed in redisplay_internal.
   TEMP_SCROLL_STEP has the same meaning as emacs_scroll_step, and is used
   in redisplay_window to bring a partially visible line into view in
   the case that only the cursor has moved.

   LAST_LINE_MISFIT should be true if we're scrolling because the
   last screen line's vertical height extends past the end of the screen.

   Value is

   1	if scrolling succeeded

   0	if scrolling didn't find point.

   -1	if new fonts have been loaded so that we must interrupt
   redisplay, adjust glyph matrices, and try again.  */

enum
{
  SCROLLING_SUCCESS = 1,
  SCROLLING_FAILED = 0,
  SCROLLING_NEED_LARGER_MATRICES = -1
};

/* If scroll-conservatively is more than this, never recenter.

   If you change this, don't forget to update the doc string of
   `scroll-conservatively' and the Emacs manual.  */
#define SCROLL_LIMIT 100

static int
try_scrolling (Lisp_Object window, bool just_this_one_p,
	       intmax_t arg_scroll_conservatively, intmax_t scroll_step,
	       bool temp_scroll_step, bool last_line_misfit)
{
  struct window *w = XWINDOW (window);
  struct text_pos pos, startp;
  struct it it;
  int this_scroll_margin, scroll_max, rc, height;
  int dy = 0, amount_to_scroll = 0;
  bool scroll_down_p = false;
  int extra_scroll_margin_lines = last_line_misfit;
  Lisp_Object aggressive;
  /* We will never try scrolling more than this number of lines.  */
  int scroll_limit = SCROLL_LIMIT;
  int frame_line_height = default_line_pixel_height (w);

#ifdef GLYPH_DEBUG
  debug_method_add (w, "try_scrolling");
#endif

  SET_TEXT_POS_FROM_MARKER (startp, w->start);

  this_scroll_margin = window_scroll_margin (w, MARGIN_IN_PIXELS);

  /* Force arg_scroll_conservatively to have a reasonable value, to
     avoid scrolling too far away with slow move_it_* functions.  Note
     that the user can supply scroll-conservatively equal to
     `most-positive-fixnum', which can be larger than INT_MAX.  */
  if (arg_scroll_conservatively > scroll_limit)
    {
      arg_scroll_conservatively = scroll_limit + 1;
      scroll_max = scroll_limit * frame_line_height;
    }
  else if (0 < scroll_step || 0 < arg_scroll_conservatively || temp_scroll_step)
    /* Compute how much we should try to scroll maximally to bring
       point into view.  */
    {
      intmax_t scroll_lines_max
	= max (scroll_step, max (arg_scroll_conservatively, temp_scroll_step));
      int scroll_lines = clip_to_bounds (0, scroll_lines_max, 1000000);
      scroll_max = scroll_lines * frame_line_height;
    }
  else if (NUMBERP (BVAR (current_buffer, scroll_down_aggressively))
	   || NUMBERP (BVAR (current_buffer, scroll_up_aggressively)))
    /* We're trying to scroll because of aggressive scrolling but no
       scroll_step is set.  Choose an arbitrary one.  */
    scroll_max = 10 * frame_line_height;
  else
    scroll_max = 0;

 too_near_end:

  /* Decide whether to scroll down.  */
  if (PT > CHARPOS (startp))
    {
      int scroll_margin_y;

      /* Compute the pixel ypos of the scroll margin, then move IT to
	 either that ypos or PT, whichever comes first.  */
      start_display (&it, w, startp);
      scroll_margin_y = it.last_visible_y - partial_line_height (&it)
        - this_scroll_margin
	- frame_line_height * extra_scroll_margin_lines;
      move_it_to (&it, PT, -1, scroll_margin_y - 1, -1,
		  (MOVE_TO_POS | MOVE_TO_Y));

      if (PT > CHARPOS (it.current.pos))
	{
	  int y0 = line_bottom_y (&it);
	  /* Compute how many pixels below window bottom to stop searching
	     for PT.  This avoids costly search for PT that is far away if
	     the user limited scrolling by a small number of lines, but
	     always finds PT if scroll_conservatively is set to a large
	     number, such as most-positive-fixnum.  */
	  int slack = max (scroll_max, 10 * frame_line_height);
	  int y_to_move = it.last_visible_y + slack;

	  /* Compute the distance from the scroll margin to PT or to
	     the scroll limit, whichever comes first.  This should
	     include the height of the cursor line, to make that line
	     fully visible.  */
	  move_it_to (&it, PT, -1, y_to_move,
	  	      -1, MOVE_TO_POS | MOVE_TO_Y);
	  dy = line_bottom_y (&it) - y0;

	  if (dy > scroll_max)
	    return SCROLLING_FAILED;

	  if (dy > 0)
	    scroll_down_p = true;
	}
      else if (PT == IT_CHARPOS (it)
	       && IT_CHARPOS (it) < ZV
	       && it.method == GET_FROM_STRING
	       && arg_scroll_conservatively > scroll_limit
	       && it.current_x == 0)
	{
	  enum move_it_result skip;
	  int y1 = it.current_y;
	  int vpos;

	  /* A before-string that includes newlines and is displayed
	     on the last visible screen line could fail us under
	     scroll-conservatively > 100, because we will be unable to
	     position the cursor on that last visible line.  Try to
	     recover by finding the first screen line that has some
	     glyphs coming from the buffer text.  */
	  do {
	    skip = move_it_in_display_line_to (&it, ZV, -1, MOVE_TO_POS);
	    if (skip != MOVE_NEWLINE_OR_CR
		|| IT_CHARPOS (it) != PT
		|| it.method == GET_FROM_BUFFER)
	      break;
	    vpos = it.vpos;
	    move_it_to (&it, -1, -1, -1, vpos + 1, MOVE_TO_VPOS);
	  } while (it.vpos > vpos);

	  dy = it.current_y - y1;

	  if (dy > scroll_max)
	    return SCROLLING_FAILED;

	  if (dy > 0)
	    scroll_down_p = true;
	}
    }

  if (scroll_down_p)
    {
      /* Point is in or below the bottom scroll margin, so move the
	 window start down.  If scrolling conservatively, move it just
	 enough down to make point visible.  If scroll_step is set,
	 move it down by scroll_step.  */
      if (arg_scroll_conservatively)
	amount_to_scroll
	  = min (max (dy, frame_line_height),
		 frame_line_height * arg_scroll_conservatively);
      else if (scroll_step || temp_scroll_step)
	amount_to_scroll = scroll_max;
      else
	{
	  aggressive = BVAR (current_buffer, scroll_up_aggressively);
	  height = WINDOW_BOX_TEXT_HEIGHT (w);
	  if (NUMBERP (aggressive))
	    {
	      double float_amount = XFLOATINT (aggressive) * height;
	      int aggressive_scroll = float_amount;
	      if (aggressive_scroll == 0 && float_amount > 0)
		aggressive_scroll = 1;
	      /* Don't let point enter the scroll margin near top of
		 the window.  This could happen if the value of
		 scroll_up_aggressively is too large and there are
		 non-zero margins, because scroll_up_aggressively
		 means put point that fraction of window height
		 _from_the_bottom_margin_.  */
	      if (aggressive_scroll + 2 * this_scroll_margin > height)
		aggressive_scroll = height - 2 * this_scroll_margin;
	      amount_to_scroll = dy + aggressive_scroll;
	    }
	}

      if (amount_to_scroll <= 0)
	return SCROLLING_FAILED;

      start_display (&it, w, startp);
      if (arg_scroll_conservatively <= scroll_limit)
	move_it_vertically (&it, amount_to_scroll);
      else
	{
	  /* Extra precision for users who set scroll-conservatively
	     to a large number: make sure the amount we scroll
	     the window start is never less than amount_to_scroll,
	     which was computed as distance from window bottom to
	     point.  This matters when lines at window top and lines
	     below window bottom have different height.  */
	  struct it it1;
	  void *it1data = NULL;
	  /* We use a temporary it1 because line_bottom_y can modify
	     its argument, if it moves one line down; see there.  */
	  int start_y;

	  SAVE_IT (it1, it, it1data);
	  start_y = line_bottom_y (&it1);
	  do {
	    RESTORE_IT (&it, &it, it1data);
	    move_it_by_lines (&it, 1);
	    SAVE_IT (it1, it, it1data);
	  } while (IT_CHARPOS (it) < ZV
		   && line_bottom_y (&it1) - start_y < amount_to_scroll);
	  bidi_unshelve_cache (it1data, true);
	}

      /* If STARTP is unchanged, move it down another screen line.  */
      if (IT_CHARPOS (it) == CHARPOS (startp))
	move_it_by_lines (&it, 1);
      startp = it.current.pos;
    }
  else
    {
      struct text_pos scroll_margin_pos = startp;
      int y_offset = 0;

      /* See if point is inside the scroll margin at the top of the
         window.  */
      if (this_scroll_margin)
	{
	  int y_start;

	  start_display (&it, w, startp);
	  y_start = it.current_y;
	  move_it_vertically (&it, this_scroll_margin);
	  scroll_margin_pos = it.current.pos;
	  /* If we didn't move enough before hitting ZV, request
	     additional amount of scroll, to move point out of the
	     scroll margin.  */
	  if (IT_CHARPOS (it) == ZV
	      && it.current_y - y_start < this_scroll_margin)
	    y_offset = this_scroll_margin - (it.current_y - y_start);
	}

      if (PT < CHARPOS (scroll_margin_pos))
	{
	  /* Point is in the scroll margin at the top of the window or
	     above what is displayed in the window.  */
	  int y0, y_to_move;

	  /* Compute the vertical distance from PT to the scroll
	     margin position.  Move as far as scroll_max allows, or
	     one screenful, or 10 screen lines, whichever is largest.
	     Give up if distance is greater than scroll_max or if we
	     didn't reach the scroll margin position.  */
	  SET_TEXT_POS (pos, PT, PT_BYTE);
	  start_display (&it, w, pos);
	  y0 = it.current_y;
	  y_to_move = max (it.last_visible_y,
			   max (scroll_max, 10 * frame_line_height));
	  move_it_to (&it, CHARPOS (scroll_margin_pos), 0,
		      y_to_move, -1,
		      MOVE_TO_POS | MOVE_TO_X | MOVE_TO_Y);
	  dy = it.current_y - y0;
	  if (dy > scroll_max
	      || IT_CHARPOS (it) < CHARPOS (scroll_margin_pos))
	    return SCROLLING_FAILED;

	  /* Additional scroll for when ZV was too close to point.  */
	  dy += y_offset;

	  /* Compute new window start.  */
	  start_display (&it, w, startp);

	  if (arg_scroll_conservatively)
	    amount_to_scroll
	      = min (max (dy, frame_line_height),
		     frame_line_height * arg_scroll_conservatively);
	  else if (scroll_step || temp_scroll_step)
	    amount_to_scroll = scroll_max;
	  else
	    {
	      aggressive = BVAR (current_buffer, scroll_down_aggressively);
	      height = WINDOW_BOX_TEXT_HEIGHT (w);
	      if (NUMBERP (aggressive))
		{
		  double float_amount = XFLOATINT (aggressive) * height;
		  int aggressive_scroll = float_amount;
		  if (aggressive_scroll == 0 && float_amount > 0)
		    aggressive_scroll = 1;
		  /* Don't let point enter the scroll margin near
		     bottom of the window, if the value of
		     scroll_down_aggressively happens to be too
		     large.  */
		  if (aggressive_scroll + 2 * this_scroll_margin > height)
		    aggressive_scroll = height - 2 * this_scroll_margin;
		  amount_to_scroll = dy + aggressive_scroll;
		}
	    }

	  if (amount_to_scroll <= 0)
	    return SCROLLING_FAILED;

	  move_it_vertically_backward (&it, amount_to_scroll);
	  startp = it.current.pos;
	}
    }

  /* Run window scroll functions.  */
  startp = run_window_scroll_functions (window, startp);

  /* Display the window.  Give up if new fonts are loaded, or if point
     doesn't appear.  */
  if (!try_window (window, startp, 0))
    rc = SCROLLING_NEED_LARGER_MATRICES;
  else if (w->cursor.vpos < 0)
    {
      clear_glyph_matrix (w->desired_matrix);
      rc = SCROLLING_FAILED;
    }
  else
    {
      /* Maybe forget recorded base line for line number display.  */
      if (!just_this_one_p
	  || current_buffer->clip_changed
	  || BEG_UNCHANGED < CHARPOS (startp))
	w->base_line_number = 0;

      /* If cursor ends up on a partially visible line,
	 treat that as being off the bottom of the screen.  */
      if (! cursor_row_fully_visible_p (w, extra_scroll_margin_lines <= 1,
					false, false)
	  /* It's possible that the cursor is on the first line of the
	     buffer, which is partially obscured due to a vscroll
	     (Bug#7537).  In that case, avoid looping forever. */
	  && extra_scroll_margin_lines < w->desired_matrix->nrows - 1)
	{
	  clear_glyph_matrix (w->desired_matrix);
	  ++extra_scroll_margin_lines;
	  goto too_near_end;
	}
      rc = SCROLLING_SUCCESS;
    }

  return rc;
}


/* Compute a suitable window start for window W if display of W starts
   on a continuation line.  Value is true if a new window start
   was computed.

   The new window start will be computed, based on W's width, starting
   from the start of the continued line.  It is the start of the
   screen line with the minimum distance from the old start W->start,
   which is still before point (otherwise point will definitely not
   be visible in the window).  */

static bool
compute_window_start_on_continuation_line (struct window *w)
{
  struct text_pos pos, start_pos, pos_before_pt;
  bool window_start_changed_p = false;

  SET_TEXT_POS_FROM_MARKER (start_pos, w->start);

  /* If window start is on a continuation line...  Window start may be
     < BEGV in case there's invisible text at the start of the
     buffer (M-x rmail, for example).  */
  if (CHARPOS (start_pos) > BEGV
      && FETCH_BYTE (BYTEPOS (start_pos) - 1) != '\n')
    {
      struct it it;
      struct glyph_row *row;

      /* Handle the case that the window start is out of range.  */
      if (CHARPOS (start_pos) < BEGV)
	SET_TEXT_POS (start_pos, BEGV, BEGV_BYTE);
      else if (CHARPOS (start_pos) > ZV)
	SET_TEXT_POS (start_pos, ZV, ZV_BYTE);

      /* Find the start of the continued line.  This should be fast
	 because find_newline is fast (newline cache).  */
      row = w->desired_matrix->rows + window_wants_tab_line (w)
				    + window_wants_header_line (w);
      init_iterator (&it, w, CHARPOS (start_pos), BYTEPOS (start_pos),
		     row, DEFAULT_FACE_ID);
      reseat_at_previous_visible_line_start (&it);

      /* Give up (by not using the code in the block below) and say it
         takes too much time to compute a new window start, if the
         line start is "too far" away from the window start.  Also,
         give up if the line start is after point, as in that case
         point will not be visible with any window start we
         compute.  */
      if (IT_CHARPOS (it) <= PT
	  && (CHARPOS (start_pos) - IT_CHARPOS (it)
	      /* PXW: Do we need upper bounds here?  */
	      < WINDOW_TOTAL_LINES (w) * WINDOW_TOTAL_COLS (w)))
	{
	  int min_distance, distance;

	  /* Move forward by display lines to find the new window
	     start.  If window width was enlarged, the new start can
	     be expected to be > the old start.  If window width was
	     decreased, the new window start will be < the old start.
	     So, we're looking for the display line start with the
	     minimum distance from the old window start.  */
	  pos_before_pt = pos = it.current.pos;
	  min_distance = DISP_INFINITY;
	  while ((distance = eabs (CHARPOS (start_pos) - IT_CHARPOS (it))),
		 distance < min_distance)
	    {
	      min_distance = distance;
	      if (CHARPOS (pos) <= PT)
		pos_before_pt = pos;
	      pos = it.current.pos;
	      if (it.line_wrap == WORD_WRAP)
		{
		  /* Under WORD_WRAP, move_it_by_lines is likely to
		     overshoot and stop not at the first, but the
		     second character from the left margin.  So in
		     that case, we need a more tight control on the X
		     coordinate of the iterator than move_it_by_lines
		     promises in its contract.  The method is to first
		     go to the last (rightmost) visible character of a
		     line, then move to the leftmost character on the
		     next line in a separate call.  */
		  move_it_to (&it, ZV, it.last_visible_x, it.current_y, -1,
			      MOVE_TO_POS | MOVE_TO_X | MOVE_TO_Y);
		  move_it_to (&it, ZV, 0,
			      it.current_y + it.max_ascent + it.max_descent, -1,
			      MOVE_TO_POS | MOVE_TO_X | MOVE_TO_Y);
		}
	      else
		move_it_by_lines (&it, 1);
	    }

	  /* It makes very little sense to make the new window start
	     after point, as point won't be visible.  If that's what
	     the loop above finds, fall back on the candidate before
	     or at point that is closest to the old window start.  */
	  if (CHARPOS (pos) > PT)
	    pos = pos_before_pt;

	  /* Set the window start there.  */
	  SET_MARKER_FROM_TEXT_POS (w->start, pos);
	  window_start_changed_p = true;
	}
    }

  return window_start_changed_p;
}


/* Try cursor movement in case text has not changed in window WINDOW,
   with window start STARTP.  Value is

   CURSOR_MOVEMENT_SUCCESS if successful

   CURSOR_MOVEMENT_CANNOT_BE_USED if this method cannot be used

   CURSOR_MOVEMENT_MUST_SCROLL if we know we have to scroll the
   display.  *SCROLL_STEP is set to true, under certain circumstances, if
   we want to scroll as if scroll-step were set to 1.  See the code.

   CURSOR_MOVEMENT_NEED_LARGER_MATRICES if we need larger matrices, in
   which case we have to abort this redisplay, and adjust matrices
   first.  */

enum
{
  CURSOR_MOVEMENT_SUCCESS,
  CURSOR_MOVEMENT_CANNOT_BE_USED,
  CURSOR_MOVEMENT_MUST_SCROLL,
  CURSOR_MOVEMENT_NEED_LARGER_MATRICES
};

static int
try_cursor_movement (Lisp_Object window, struct text_pos startp,
		     bool *scroll_step)
{
  struct window *w = XWINDOW (window);
  struct frame *f = XFRAME (w->frame);
  int rc = CURSOR_MOVEMENT_CANNOT_BE_USED;

#ifdef GLYPH_DEBUG
  if (inhibit_try_cursor_movement)
    return rc;
#endif

  /* Previously, there was a check for Lisp integer in the
     if-statement below. Now, this field is converted to
     ptrdiff_t, thus zero means invalid position in a buffer.  */
  eassert (w->last_point > 0);
  /* Likewise there was a check whether window_end_vpos is nil or larger
     than the window.  Now window_end_vpos is int and so never nil, but
     let's leave eassert to check whether it fits in the window.  */
  eassert (!w->window_end_valid
	   || w->window_end_vpos < w->current_matrix->nrows);

  /* Handle case where text has not changed, only point, and it has
     not moved off the frame.  */
  if (/* Point may be in this window.  */
      PT >= CHARPOS (startp)
      /* Selective display hasn't changed.  */
      && !current_buffer->clip_changed
      /* Function force-mode-line-update is used to force a thorough
	 redisplay.  It sets either windows_or_buffers_changed or
	 update_mode_lines.  So don't take a shortcut here for these
	 cases.  */
      && !update_mode_lines
      && !windows_or_buffers_changed
      && !f->cursor_type_changed
      && NILP (Vshow_trailing_whitespace)
      /* When display-line-numbers is in relative mode, moving point
	 requires to redraw the entire window.  */
      && !EQ (Vdisplay_line_numbers, Qrelative)
      && !EQ (Vdisplay_line_numbers, Qvisual)
      /* When the current line number should be displayed in a
	 distinct face, moving point cannot be handled in optimized
	 way as below.  */
      && !(!NILP (Vdisplay_line_numbers)
	   && NILP (Finternal_lisp_face_equal_p (Qline_number,
						 Qline_number_current_line,
						 w->frame)))
      /* This code is not used for mini-buffer for the sake of the case
	 of redisplaying to replace an echo area message; since in
	 that case the mini-buffer contents per se are usually
	 unchanged.  This code is of no real use in the mini-buffer
	 since the handling of this_line_start_pos, etc., in redisplay
	 handles the same cases.  */
      && !EQ (window, minibuf_window)
      /* When overlay arrow is shown in current buffer, point movement
	 is no longer "simple", as it typically causes the overlay
	 arrow to move as well.  */
      && !overlay_arrow_in_current_buffer_p ())
    {
      int this_scroll_margin, top_scroll_margin;
      struct glyph_row *row = NULL;

#ifdef GLYPH_DEBUG
      debug_method_add (w, "cursor movement");
#endif

      this_scroll_margin = window_scroll_margin (w, MARGIN_IN_PIXELS);

      top_scroll_margin = this_scroll_margin;
      if (window_wants_tab_line (w))
	top_scroll_margin += CURRENT_TAB_LINE_HEIGHT (w);
      if (window_wants_header_line (w))
	top_scroll_margin += CURRENT_HEADER_LINE_HEIGHT (w);

      /* Start with the row the cursor was displayed during the last
	 not paused redisplay.  Give up if that row is not valid.  */
      if (w->last_cursor_vpos < 0
	  || w->last_cursor_vpos >= w->current_matrix->nrows)
	rc = CURSOR_MOVEMENT_MUST_SCROLL;
      else
	{
	  row = MATRIX_ROW (w->current_matrix, w->last_cursor_vpos);
	  /* Skip the tab-line and header-line rows, if any.  */
	  if (row->tab_line_p)
	    ++row;
	  if (row->mode_line_p)
	    ++row;
	  if (!row->enabled_p)
	    rc = CURSOR_MOVEMENT_MUST_SCROLL;
	}

      if (rc == CURSOR_MOVEMENT_CANNOT_BE_USED)
	{
	  bool scroll_p = false, must_scroll = false;
	  int last_y = window_text_bottom_y (w) - this_scroll_margin;

	  if (PT > w->last_point)
	    {
	      /* Point has moved forward.  */
	      while (MATRIX_ROW_END_CHARPOS (row) < PT
		     && MATRIX_ROW_BOTTOM_Y (row) < last_y)
		{
		  eassert (row->enabled_p);
		  ++row;
		}

	      /* If the end position of a row equals the start
		 position of the next row, and PT is at that position,
		 we would rather display cursor in the next line.  */
	      while (MATRIX_ROW_BOTTOM_Y (row) < last_y
		     && MATRIX_ROW_END_CHARPOS (row) == PT
		     && row < MATRIX_MODE_LINE_ROW (w->current_matrix)
		     && MATRIX_ROW_START_CHARPOS (row+1) >= PT
		     && !cursor_row_p (row))
		++row;

	      /* If within the scroll margin, scroll.  Note that
		 MATRIX_ROW_BOTTOM_Y gives the pixel position at which
		 the next line would be drawn, and that
		 this_scroll_margin can be zero.  */
	      if (MATRIX_ROW_BOTTOM_Y (row) > last_y
		  || PT > MATRIX_ROW_END_CHARPOS (row)
		  /* Line is completely visible last line in window
		     and PT is to be set in the next line.  */
		  || (MATRIX_ROW_BOTTOM_Y (row) == last_y
		      && PT == MATRIX_ROW_END_CHARPOS (row)
		      && !row->ends_at_zv_p
		      && !MATRIX_ROW_ENDS_IN_MIDDLE_OF_CHAR_P (row)))
		scroll_p = true;
	    }
	  else if (PT < w->last_point)
	    {
	      /* Cursor has to be moved backward.  Note that PT >=
		 CHARPOS (startp) because of the outer if-statement.  */
	      struct glyph_row *row0 = row;

	      while (!row->mode_line_p
		     && (MATRIX_ROW_START_CHARPOS (row) > PT
			 || (MATRIX_ROW_START_CHARPOS (row) == PT
			     && (MATRIX_ROW_STARTS_IN_MIDDLE_OF_CHAR_P (row)
				 || (/* STARTS_IN_MIDDLE_OF_STRING_P (row) */
				     row > w->current_matrix->rows
				     && (row-1)->ends_in_newline_from_string_p))))
		     && (row->y > top_scroll_margin
			 || CHARPOS (startp) == BEGV))
		{
		  eassert (row->enabled_p);
		  --row;
		}

	      /* With bidi-reordered rows we can have buffer positions
		 _decrease_ when going down by rows.  If we haven't
		 found our row in the loop above, give it another try
		 now going in the other direction from the original row.  */
	      if (!(MATRIX_ROW_START_CHARPOS (row) <= PT
		    && PT <= MATRIX_ROW_END_CHARPOS (row))
		  && row0->continued_p)
		{
		  row = row0;
		  while (MATRIX_ROW_START_CHARPOS (row) > PT
			 && MATRIX_ROW_BOTTOM_Y (row) < last_y)
		    {
		      eassert (row->enabled_p);
		      ++row;
		    }
		}

	      /* Consider the following case: Window starts at BEGV,
		 there is invisible, intangible text at BEGV, so that
		 display starts at some point START > BEGV.  It can
		 happen that we are called with PT somewhere between
		 BEGV and START.  Try to handle that case.  */
	      if (row < w->current_matrix->rows
		  || row->mode_line_p)
		{
		  row = w->current_matrix->rows;
		  /* Skip the tab-line and header-line rows, if any.  */
		  if (row->tab_line_p)
		    ++row;
		  if (row->mode_line_p)
		    ++row;
		}

	      /* Due to newlines in overlay strings, we may have to
		 skip forward over overlay strings.  */
	      while (MATRIX_ROW_BOTTOM_Y (row) < last_y
		     && MATRIX_ROW_END_CHARPOS (row) == PT
		     && !cursor_row_p (row))
		++row;

	      /* If within the scroll margin, either the top one or
		 the bottom one, scroll.  */
	      if ((row->y < top_scroll_margin
		   && CHARPOS (startp) != BEGV)
		  || MATRIX_ROW_BOTTOM_Y (row) > last_y
		  || PT > MATRIX_ROW_END_CHARPOS (row)
		  || (MATRIX_ROW_BOTTOM_Y (row) == last_y
		      && PT == MATRIX_ROW_END_CHARPOS (row)
		      && !row->ends_at_zv_p
		      && !MATRIX_ROW_ENDS_IN_MIDDLE_OF_CHAR_P (row)))
		scroll_p = true;
	    }
	  else
	    {
	      /* Cursor did not move.  So don't scroll even if cursor line
		 is partially visible, as it was so before.  */
		 rc = CURSOR_MOVEMENT_SUCCESS;
	    }

	  if ((PT < MATRIX_ROW_START_CHARPOS (row)
	       && (row == MATRIX_FIRST_TEXT_ROW (w->current_matrix)
		   /* Don't give up if point is inside invisible text
		      at the beginning of its glyph row.  */
		   || (MATRIX_ROW_END_CHARPOS (row-1)
		       == MATRIX_ROW_START_CHARPOS (row))))
	      || PT > MATRIX_ROW_END_CHARPOS (row))
	    {
	      /* if PT is not in the glyph row, give up.  */
	      rc = CURSOR_MOVEMENT_MUST_SCROLL;
	      must_scroll = true;
	    }
	  else if (rc != CURSOR_MOVEMENT_SUCCESS
		   && !NILP (BVAR (XBUFFER (w->contents), bidi_display_reordering)))
	    {
	      struct glyph_row *row1;

	      /* If rows are bidi-reordered and point moved, back up
		 until we find a row that does not belong to a
		 continuation line.  This is because we must consider
		 all rows of a continued line as candidates for the
		 new cursor positioning, since row start and end
		 positions change non-linearly with vertical position
		 in such rows.  */
	      /* FIXME: Revisit this when glyph ``spilling'' in
		 continuation lines' rows is implemented for
		 bidi-reordered rows.  */
	      for (row1 = MATRIX_FIRST_TEXT_ROW (w->current_matrix);
		   MATRIX_ROW_CONTINUATION_LINE_P (row);
		   --row)
		{
		  /* If we hit the beginning of the displayed portion
		     without finding the first row of a continued
		     line, give up.  */
		  if (row <= row1)
		    {
		      rc = CURSOR_MOVEMENT_MUST_SCROLL;
		      break;
		    }
		  eassert (row->enabled_p);
		}
	    }
	  if (must_scroll)
	    ;
	  else if (rc != CURSOR_MOVEMENT_SUCCESS
	      && MATRIX_ROW_PARTIALLY_VISIBLE_P (w, row)
	      /* Make sure this isn't a header line nor a tab-line by
		 any chance, since then MATRIX_ROW_PARTIALLY_VISIBLE_P
		 might yield true.  */
	      && !row->mode_line_p
	      && !cursor_row_fully_visible_p (w, true, true, true))
	    {
	      if (PT == MATRIX_ROW_END_CHARPOS (row)
		  && !row->ends_at_zv_p
		  && !MATRIX_ROW_ENDS_IN_MIDDLE_OF_CHAR_P (row))
		rc = CURSOR_MOVEMENT_MUST_SCROLL;
	      else if (row->height > window_box_height (w))
		{
		  /* If we end up in a partially visible line, let's
		     make it fully visible, except when it's taller
		     than the window, in which case we can't do much
		     about it.  */
		  *scroll_step = true;
		  rc = CURSOR_MOVEMENT_MUST_SCROLL;
		}
	      else
		{
		  set_cursor_from_row (w, row, w->current_matrix, 0, 0, 0, 0);
		  if (!cursor_row_fully_visible_p (w, false, true, false))
		    rc = CURSOR_MOVEMENT_MUST_SCROLL;
		  else
		    rc = CURSOR_MOVEMENT_SUCCESS;
		}
	    }
	  else if (scroll_p)
	    rc = CURSOR_MOVEMENT_MUST_SCROLL;
	  else if (rc != CURSOR_MOVEMENT_SUCCESS
		   && !NILP (BVAR (XBUFFER (w->contents), bidi_display_reordering)))
	    {
	      /* With bidi-reordered rows, there could be more than
		 one candidate row whose start and end positions
		 occlude point.  We need to let set_cursor_from_row
		 find the best candidate.  */
	      /* FIXME: Revisit this when glyph ``spilling'' in
		 continuation lines' rows is implemented for
		 bidi-reordered rows.  */
	      bool rv = false;
	      bool pt_invis = false;
	      Lisp_Object val = get_char_property_and_overlay (make_fixnum (PT),
							       Qinvisible,
							       Qnil, NULL);

	      if (TEXT_PROP_MEANS_INVISIBLE (val) != 0)
		pt_invis = true;

	      do
		{
		  bool at_zv_p = false, exact_match_p = false;

		  /* If point is in invisible text, we cannot assume
		     it must be after row's start position, since the
		     row could have invisible text at its beginning
		     where point is located.  */
		  if ((pt_invis || MATRIX_ROW_START_CHARPOS (row) <= PT)
		      && PT <= MATRIX_ROW_END_CHARPOS (row)
		      && cursor_row_p (row))
		    rv |= set_cursor_from_row (w, row, w->current_matrix,
					       0, 0, 0, 0);
		  /* As soon as we've found the exact match for point,
		     or the first suitable row whose ends_at_zv_p flag
		     is set, we are done.  */
		  if (rv)
		    {
		      at_zv_p = MATRIX_ROW (w->current_matrix,
					    w->cursor.vpos)->ends_at_zv_p;
		      if (!at_zv_p
			  && w->cursor.hpos >= 0
			  && w->cursor.hpos < MATRIX_ROW_USED (w->current_matrix,
							       w->cursor.vpos))
			{
			  struct glyph_row *candidate =
			    MATRIX_ROW (w->current_matrix, w->cursor.vpos);
			  struct glyph *g =
			    candidate->glyphs[TEXT_AREA] + w->cursor.hpos;
			  ptrdiff_t endpos = MATRIX_ROW_END_CHARPOS (candidate);

			  exact_match_p =
			    (BUFFERP (g->object) && g->charpos == PT)
			    || (NILP (g->object)
				&& (g->charpos == PT
				    || (g->charpos == 0 && endpos - 1 == PT)));
			  /* Perhaps the point position is inside
			     invisible text?  In that case, we trust
			     'set_cursor_from_row' to do its job and
			     find the best position for the cursor.  */
			  if (!exact_match_p && pt_invis)
			    exact_match_p = true;
			}
		      if (at_zv_p || exact_match_p)
			{
			  rc = CURSOR_MOVEMENT_SUCCESS;
			  break;
			}
		    }
		  if (MATRIX_ROW_BOTTOM_Y (row) == last_y)
		    break;
		  ++row;
		}
	      while (((MATRIX_ROW_CONTINUATION_LINE_P (row)
		       || row->continued_p)
		      && MATRIX_ROW_BOTTOM_Y (row) <= last_y)
		     || (MATRIX_ROW_START_CHARPOS (row) == PT
			 && MATRIX_ROW_BOTTOM_Y (row) < last_y));
	      /* If we didn't find any candidate rows, or exited the
		 loop before all the candidates were examined, signal
		 to the caller that this method failed.  */
	      if (rc != CURSOR_MOVEMENT_SUCCESS
		  && !(rv
		       && !MATRIX_ROW_CONTINUATION_LINE_P (row)
		       && !row->continued_p))
		rc = CURSOR_MOVEMENT_MUST_SCROLL;
	      else if (rv)
		rc = CURSOR_MOVEMENT_SUCCESS;
	    }
	  else
	    {
	      do
		{
		  if (set_cursor_from_row (w, row, w->current_matrix, 0, 0, 0, 0))
		    {
		      rc = CURSOR_MOVEMENT_SUCCESS;
		      break;
		    }
		  ++row;
		}
	      while (MATRIX_ROW_BOTTOM_Y (row) < last_y
		     && MATRIX_ROW_START_CHARPOS (row) == PT
		     && cursor_row_p (row));
	    }
	}
    }

  return rc;
}


void
set_vertical_scroll_bar (struct window *w)
{
  ptrdiff_t start, end, whole;

  /* Calculate the start and end positions for the current window.
     At some point, it would be nice to choose between scrollbars
     which reflect the whole buffer size, with special markers
     indicating narrowing, and scrollbars which reflect only the
     visible region.

     Note that mini-buffers sometimes aren't displaying any text.  */
  if (!MINI_WINDOW_P (w)
      || (w == XWINDOW (minibuf_window)
	  && NILP (echo_area_buffer[0])))
    {
      struct buffer *buf = XBUFFER (w->contents);

      whole = BUF_ZV (buf) - BUF_BEGV (buf);
      start = marker_position (w->start) - BUF_BEGV (buf);
      end = BUF_Z (buf) - w->window_end_pos - BUF_BEGV (buf);

      /* If w->window_end_pos cannot be trusted, recompute it "the
	 hard way".  But don't bother to be too accurate when
	 long-line shortcuts are in effect.  */
      if (!w->window_end_valid && !buf->long_line_optimizations_p)
	{
	  struct it it;
	  struct text_pos start_pos;
	  struct buffer *obuf = current_buffer;
	  /* When we display the scroll bar of a mini-window,
	     current_buffer is not guaranteed to be the mini-window's
	     buffer, see the beginning of redisplay_window.  */
	  set_buffer_internal_1 (XBUFFER (w->contents));
	  SET_TEXT_POS_FROM_MARKER (start_pos, w->start);
	  start_display (&it, w, start_pos);
	  move_it_to (&it, -1, it.last_visible_x, window_box_height (w), -1,
		      MOVE_TO_X | MOVE_TO_Y);
	  end -= (BUF_Z (buf) - IT_CHARPOS (it)) - w->window_end_pos;
	  set_buffer_internal_1 (obuf);
	}

      if (end < start)
	end = start;
      if (whole < (end - start))
	whole = end - start;
    }
  else
    start = end = whole = 0;

  /* Indicate what this scroll bar ought to be displaying now.  */
  if (FRAME_TERMINAL (XFRAME (w->frame))->set_vertical_scroll_bar_hook)
    (*FRAME_TERMINAL (XFRAME (w->frame))->set_vertical_scroll_bar_hook)
      (w, end - start, whole, start);
}


void
set_horizontal_scroll_bar (struct window *w)
{
  int start, end, whole, portion;

  if (!MINI_WINDOW_P (w) || EQ (w->horizontal_scroll_bar_type, Qbottom))
    {
      struct buffer *b = XBUFFER (w->contents);
      struct buffer *old_buffer = NULL;
      struct it it;
      struct text_pos startp;

      if (b != current_buffer)
	{
	  old_buffer = current_buffer;
	  set_buffer_internal (b);
	}

      SET_TEXT_POS_FROM_MARKER (startp, w->start);
      start_display (&it, w, startp);
      it.last_visible_x = INT_MAX;
      whole = move_it_to (&it, -1, INT_MAX, window_box_height (w), -1,
			  MOVE_TO_X | MOVE_TO_Y);
      /* whole = move_it_to (&it, w->window_end_pos, INT_MAX,
			  window_box_height (w), -1,
			  MOVE_TO_POS | MOVE_TO_X | MOVE_TO_Y); */

      start = w->hscroll * FRAME_COLUMN_WIDTH (WINDOW_XFRAME (w));
      end = start + window_box_width (w, TEXT_AREA);
      portion = end - start;
      /* After enlarging a horizontally scrolled window such that it
	 gets at least as wide as the text it contains, make sure that
	 the thumb doesn't fill the entire scroll bar so we can still
	 drag it back to see the entire text.  */
      whole = max (whole, end);

      if (it.bidi_p)
	{
	  Lisp_Object pdir;

	  pdir = Fcurrent_bidi_paragraph_direction (Qnil);
	  if (EQ (pdir, Qright_to_left))
	    {
	      start = whole - end;
	      end = start + portion;
	    }
	}

      if (old_buffer)
	set_buffer_internal (old_buffer);
    }
  else
    start = end = whole = portion = 0;

  w->hscroll_whole = whole;

  /* Indicate what this scroll bar ought to be displaying now.  */
  if (FRAME_TERMINAL (XFRAME (w->frame))->set_horizontal_scroll_bar_hook)
    (*FRAME_TERMINAL (XFRAME (w->frame))->set_horizontal_scroll_bar_hook)
      (w, portion, whole, start);
}

/* Subroutine of redisplay_window, to determine whether a window-start
   point STARTP of WINDOW should be rejected.  */
static bool
window_start_acceptable_p (Lisp_Object window, ptrdiff_t startp)
{
  if (!make_window_start_visible)
    return true;

  struct window *w = XWINDOW (window);
  struct frame *f = XFRAME (w->frame);
  Lisp_Object startpos = make_fixnum (startp);
  Lisp_Object invprop, disp_spec;
  struct text_pos ignored;

  /* Is STARTP in invisible text?  */
  if ((invprop = Fget_char_property (startpos, Qinvisible, window)),
      TEXT_PROP_MEANS_INVISIBLE (invprop) != 0)
    return false;

  /* Is STARTP covered by a replacing 'display' property?  */
  if (!NILP (disp_spec = Fget_char_property (startpos, Qdisplay, window))
      && handle_display_spec (NULL, disp_spec, Qnil, Qnil, &ignored, startp,
			      FRAME_WINDOW_P (f)) > 0)
    return false;

  return true;
}

DEFUN ("long-line-optimizations-p", Flong_line_optimizations_p, Slong_line_optimizations_p,
       0, 0, 0,
       doc: /* Return non-nil if long-line optimizations are in effect in current buffer.
See `long-line-threshold' and `large-hscroll-threshold' for what these
optimizations mean and when they are in effect.  */)
  (void)
{
  return current_buffer->long_line_optimizations_p ? Qt : Qnil;
}

/* Redisplay leaf window WINDOW.  JUST_THIS_ONE_P means only
   selected_window is redisplayed.

   We can return without actually redisplaying the window if fonts has been
   changed on window's frame.  In that case, redisplay_internal will retry.

   As one of the important parts of redisplaying a window, we need to
   decide whether the previous window-start position (stored in the
   window's w->start marker position) is still valid, and if it isn't,
   recompute it.  Some details about that:

    . The previous window-start could be in a continuation line, in
      which case we need to recompute it when the window width
      changes.  See compute_window_start_on_continuation_line and its
      call below.

    . The text that changed since last redisplay could include the
      previous window-start position.  In that case, we try to salvage
      what we can from the current glyph matrix by calling
      try_scrolling, which see.

    . Some Emacs command could force us to use a specific window-start
      position by setting the window's force_start flag, or gently
      propose doing that by setting the window's optional_new_start
      flag.  In these cases, we try using the specified start point if
      that succeeds (i.e. the window desired matrix is successfully
      recomputed, and point location is within the window).  In case
      of optional_new_start, we first check if the specified start
      position is feasible, i.e. if it will allow point to be
      displayed in the window.  If using the specified start point
      fails, e.g., if new fonts are needed to be loaded, we abort the
      redisplay cycle and leave it up to the next cycle to figure out
      things.

    . Note that the window's force_start flag is sometimes set by
      redisplay itself, when it decides that the previous window start
      point is fine and should be kept.  Search for "goto force_start"
      below to see the details.  Like the values of window-start
      specified outside of redisplay, these internally-deduced values
      are tested for feasibility, and ignored if found to be
      unfeasible.

    . Note that the function try_window, used to completely redisplay
      a window, accepts the window's start point as its argument.
      This is used several times in the redisplay code to control
      where the window start will be, according to user options such
      as scroll-conservatively, and also to ensure the screen line
      showing point will be fully (as opposed to partially) visible on
      display.  */

static void
redisplay_window (Lisp_Object window, bool just_this_one_p)
{
  struct window *w = XWINDOW (window);
  struct frame *f = XFRAME (w->frame);
  struct buffer *buffer = XBUFFER (w->contents);
  struct buffer *old = current_buffer;
  struct text_pos lpoint, opoint, startp;
  bool update_mode_line;
  int tem;
  struct it it;
  /* Record it now because it's overwritten.  */
  bool current_matrix_up_to_date_p = false;
  bool used_current_matrix_p = false;
  /* This is less strict than current_matrix_up_to_date_p.
     It indicates that the buffer contents and narrowing are unchanged.  */
  bool buffer_unchanged_p = false;
  bool temp_scroll_step = false;
  specpdl_ref count = SPECPDL_INDEX ();
  int rc;
  int centering_position = -1;
  bool last_line_misfit = false;
  ptrdiff_t beg_unchanged, end_unchanged;
  int frame_line_height, margin;
  bool use_desired_matrix;
  void *itdata = NULL;

  SET_TEXT_POS (lpoint, PT, PT_BYTE);
  opoint = lpoint;

#ifdef GLYPH_DEBUG
  *w->desired_matrix->method = 0;
#endif

  if (!just_this_one_p && needs_no_redisplay (w))
    return;

  /* Make sure that both W's markers are valid.  */
  eassert (XMARKER (w->start)->buffer == buffer);
  eassert (XMARKER (w->pointm)->buffer == buffer);

  reconsider_clip_changes (w);
  frame_line_height = default_line_pixel_height (w);
  margin = window_scroll_margin (w, MARGIN_IN_LINES);


  /* Has the mode line to be updated?  */
  update_mode_line = (w->update_mode_line
		      || update_mode_lines
		      || buffer->clip_changed
		      || buffer->prevent_redisplay_optimizations_p);

  if (!just_this_one_p)
    /* If `just_this_one_p' is set, we apparently set must_be_updated_p more
       cleverly elsewhere.  */
    w->must_be_updated_p = true;

  if (MINI_WINDOW_P (w))
    {
      if (w == XWINDOW (echo_area_window)
	  && !NILP (echo_area_buffer[0]))
	{
	  if (update_mode_line)
	    /* We may have to update a tty frame's menu bar or a
	       tool-bar.  Example `M-x C-h C-h C-g'.  */
	    goto finish_menu_bars;
	  else
	    /* We've already displayed the echo area glyphs in this window.  */
	    goto finish_scroll_bars;
	}
      else if ((w != XWINDOW (minibuf_window)
		|| minibuf_level == 0)
	       /* When buffer is nonempty, redisplay window normally.  */
	       && BUF_Z (XBUFFER (w->contents)) == BUF_BEG (XBUFFER (w->contents))
	       /* Quail displays non-mini buffers in minibuffer window.
		  In that case, redisplay the window normally.  */
	       && !NILP (Fmemq (w->contents, Vminibuffer_list)))
	{
	  /* W is a mini-buffer window, but it's not active, so clear
	     it.  */
	  int yb = window_text_bottom_y (w);
	  struct glyph_row *row;
	  int y;

	  for (y = 0, row = w->desired_matrix->rows;
	       y < yb;
	       y += row->height, ++row)
	    blank_row (w, row, y);
	  goto finish_scroll_bars;
	}
      else if (minibuf_level >= 1)
	{
	  /* We could have a message produced by set-minibuffer-message
	     displayed in the mini-window as an overlay, so resize the
	     mini-window if needed.  */
	  resize_mini_window (w, false);
	}

      clear_glyph_matrix (w->desired_matrix);
    }

  /* Otherwise set up data on this window; select its buffer and point
     value.  */
  /* Really select the buffer, for the sake of buffer-local
     variables.  */
  set_buffer_internal_1 (XBUFFER (w->contents));

  current_matrix_up_to_date_p
    = (w->window_end_valid
       && !current_buffer->clip_changed
       && !current_buffer->prevent_redisplay_optimizations_p
       && !window_outdated (w)
       && !composition_break_at_point
       && !hscrolling_current_line_p (w));

  beg_unchanged = BEG_UNCHANGED;
  end_unchanged = END_UNCHANGED;

  SET_TEXT_POS (opoint, PT, PT_BYTE);

  specbind (Qinhibit_point_motion_hooks, Qt);

  buffer_unchanged_p
    = (w->window_end_valid
       && !current_buffer->clip_changed
       && !window_outdated (w));

  /* When windows_or_buffers_changed is non-zero, we can't rely
     on the window end being valid, so set it to zero there.  */
  if (windows_or_buffers_changed)
    {
      /* If window starts on a continuation line, maybe adjust the
	 window start in case the window's width changed.  */
      if (XMARKER (w->start)->buffer == current_buffer)
	compute_window_start_on_continuation_line (w);

      w->window_end_valid = false;
      /* If so, we also can't rely on current matrix
	 and should not fool try_cursor_movement below.  */
      current_matrix_up_to_date_p = false;
    }

  /* Some sanity checks.  */
  CHECK_WINDOW_END (w);
  if (Z == Z_BYTE && CHARPOS (opoint) != BYTEPOS (opoint))
    emacs_abort ();
  if (BYTEPOS (opoint) < CHARPOS (opoint))
    emacs_abort ();

  if (mode_line_update_needed (w))
    update_mode_line = true;

  /* Point refers normally to the selected window.  For any other
     window, set up appropriate value.  */
  if (!EQ (window, selected_window))
    {
      ptrdiff_t new_pt = marker_position (w->pointm);
      ptrdiff_t new_pt_byte = marker_byte_position (w->pointm);

      if (new_pt < BEGV)
	{
	  new_pt = BEGV;
	  new_pt_byte = BEGV_BYTE;
	  set_marker_both (w->pointm, Qnil, BEGV, BEGV_BYTE);
	}
      else if (new_pt > (ZV - 1))
	{
	  new_pt = ZV;
	  new_pt_byte = ZV_BYTE;
	  set_marker_both (w->pointm, Qnil, ZV, ZV_BYTE);
	}

      /* We don't use SET_PT so that the point-motion hooks don't run.  */
      TEMP_SET_PT_BOTH (new_pt, new_pt_byte);
    }

  /* If any of the character widths specified in the display table
     have changed, invalidate the width run cache.  It's true that
     this may be a bit late to catch such changes, but the rest of
     redisplay goes (non-fatally) haywire when the display table is
     changed, so why should we worry about doing any better?  */
  if (current_buffer->width_run_cache
      || (current_buffer->base_buffer
	  && current_buffer->base_buffer->width_run_cache))
    {
      struct Lisp_Char_Table *disptab = buffer_display_table ();

      if (! disptab_matches_widthtab
	  (disptab, XVECTOR (BVAR (current_buffer, width_table))))
        {
	  struct buffer *buf = current_buffer;

	  if (buf->base_buffer)
	    buf = buf->base_buffer;
          invalidate_region_cache (buf, buf->width_run_cache, BEG, Z);
          recompute_width_table (current_buffer, disptab);
        }
    }

  /* Check whether the buffer to be displayed contains long lines.  */
  if (!NILP (Vlong_line_threshold)
      && !current_buffer->long_line_optimizations_p
      && (CHARS_MODIFF - UNCHANGED_MODIFIED > 8
	  || current_buffer->clip_changed))
    {
      ptrdiff_t cur, next, found, max = 0, threshold;
      threshold = XFIXNUM (Vlong_line_threshold);
      for (cur = BEGV; cur < ZV; cur = next)
	{
	  next = find_newline1 (cur, CHAR_TO_BYTE (cur), 0, -1, 1,
				&found, NULL, true);
	  if (next - cur > max) max = next - cur;
	  if (!found || max > threshold) break;
	}
      if (max > threshold)
	current_buffer->long_line_optimizations_p = true;
    }

  /* If window-start is screwed up, choose a new one.  */
  if (XMARKER (w->start)->buffer != current_buffer)
    goto recenter;

  SET_TEXT_POS_FROM_MARKER (startp, w->start);

  /* If someone specified a new starting point but did not insist,
     check whether it can be used.  */
  if ((w->optional_new_start || window_frozen_p (w))
      && CHARPOS (startp) >= BEGV
      && CHARPOS (startp) <= ZV)
    {
      ptrdiff_t it_charpos;

      w->optional_new_start = false;
      if (!w->force_start)
	{
	  start_display (&it, w, startp);
	  move_it_to (&it, PT, 0, it.last_visible_y, -1,
		      MOVE_TO_POS | MOVE_TO_X | MOVE_TO_Y);
	  /* Record IT's position now, since line_bottom_y might
	     change that.  */
	  it_charpos = IT_CHARPOS (it);
	  /* Make sure we set the force_start flag only if the cursor
	     row will be fully visible.  Otherwise, the code under
	     force_start label below will try to move point back into
	     view, which is not what the code which sets
	     optional_new_start wants.  */
	  if (it.current_y == 0 || line_bottom_y (&it) < it.last_visible_y)
	    {
	      if (it_charpos == PT)
		w->force_start = true;
	      /* IT may overshoot PT if text at PT is invisible.  */
	      else if (it_charpos > PT && CHARPOS (startp) <= PT)
		w->force_start = true;
#ifdef GLYPH_DEBUG
	      if (w->force_start)
		{
		  if (window_frozen_p (w))
		    debug_method_add (w, "set force_start from frozen window start");
		  else
		    debug_method_add (w, "set force_start from optional_new_start");
		}
#endif
	    }
	}
    }

 force_start:

  /* Handle case where place to start displaying has been specified,
     unless the specified location is outside the accessible range.  */
  if (w->force_start)
    {
      /* We set this later on if we have to adjust point.  */
      int new_vpos = -1;

      w->force_start = false;

      /* The vscroll should be preserved in this case, since
	 `pixel-scroll-precision-mode' must continue working normally
	 when a mini-window is resized.  (bug#55312) */
      if (!w->preserve_vscroll_p || !window_frozen_p (w))
	w->vscroll = 0;

      w->preserve_vscroll_p = false;
      w->window_end_valid = false;

      /* Forget any recorded base line for line number display.  */
      if (!buffer_unchanged_p)
	w->base_line_number = 0;

      /* Redisplay the mode line.  Select the buffer properly for that.
	 Also, run the hook window-scroll-functions
	 because we have scrolled.  */
      /* Note, we do this after clearing force_start because
	 if there's an error, it is better to forget about force_start
	 than to get into an infinite loop calling the hook functions
	 and having them get more errors.  */
      if (!update_mode_line
	  || ! NILP (Vwindow_scroll_functions))
	{
	  update_mode_line = true;
	  w->update_mode_line = true;
	  startp = run_window_scroll_functions (window, startp);
	}

      if (CHARPOS (startp) < BEGV)
	SET_TEXT_POS (startp, BEGV, BEGV_BYTE);
      else if (CHARPOS (startp) > ZV)
	SET_TEXT_POS (startp, ZV, ZV_BYTE);

      /* Reject the specified start location if it is invisible, and
	 the buffer wants it always visible.  */
      if (!window_start_acceptable_p (window, CHARPOS (startp)))
	goto ignore_start;

      /* Redisplay, then check if cursor has been set during the
	 redisplay.  Give up if new fonts were loaded.  */
      /* We used to issue a CHECK_MARGINS argument to try_window here,
	 but this causes scrolling to fail when point begins inside
	 the scroll margin (bug#148) -- cyd  */
      clear_glyph_matrix (w->desired_matrix);
      if (!try_window (window, startp, 0))
	{
	  w->force_start = true;
	  clear_glyph_matrix (w->desired_matrix);
	  goto need_larger_matrices;
	}

      if (w->cursor.vpos < 0)
	{
	  /* If point does not appear, try to move point so it does
	     appear.  The desired matrix has been built above, so we
	     can use it here.  First see if point is in invisible
	     text, and if so, move it to the first visible buffer
	     position past that.  */
	  struct glyph_row *r = NULL;
	  Lisp_Object invprop =
	    get_char_property_and_overlay (make_fixnum (PT), Qinvisible,
					   Qnil, NULL);

	  if (TEXT_PROP_MEANS_INVISIBLE (invprop) != 0)
	    {
	      ptrdiff_t alt_pt;
	      Lisp_Object invprop_end =
		Fnext_single_char_property_change (make_fixnum (PT), Qinvisible,
						   Qnil, Qnil);

	      if (FIXNATP (invprop_end))
		alt_pt = XFIXNAT (invprop_end);
	      else
		alt_pt = ZV;
	      r = row_containing_pos (w, alt_pt, w->desired_matrix->rows,
				      NULL, 0);
	    }
	  if (r)
	    new_vpos = MATRIX_ROW_BOTTOM_Y (r);
	  else	/* Give up and just move to the middle of the window.  */
	    new_vpos = window_box_height (w) / 2;
	}

      if (!cursor_row_fully_visible_p (w, false, false, false))
	{
	  /* Point does appear, but on a line partly visible at end of window.
	     Move it back to a fully-visible line.  */
	  new_vpos = window_box_height (w);
	  /* But if window_box_height suggests a Y coordinate that is
	     not less than we already have, that line will clearly not
	     be fully visible, so give up and scroll the display.
	     This can happen when the default face uses a font whose
	     dimensions are different from the frame's default
	     font.  */
	  if (new_vpos >= w->cursor.y)
	    {
	      w->cursor.vpos = -1;
	      clear_glyph_matrix (w->desired_matrix);
	      goto try_to_scroll;
	    }
	}
      else if (w->cursor.vpos >= 0)
	{
	  /* Some people insist on not letting point enter the scroll
	     margin, even though this part handles windows that didn't
	     scroll at all.  */
          int pixel_margin = margin * frame_line_height;
	  bool tab_line = window_wants_tab_line (w);
	  bool header_line = window_wants_header_line (w);

	  /* Note: We add an extra FRAME_LINE_HEIGHT, because the loop
	     below, which finds the row to move point to, advances by
	     the Y coordinate of the _next_ row, see the definition of
	     MATRIX_ROW_BOTTOM_Y.  */
	  if (w->cursor.vpos < margin + tab_line + header_line)
	    {
	      w->cursor.vpos = -1;
	      clear_glyph_matrix (w->desired_matrix);
	      goto try_to_scroll;
	    }
	  else
	    {
	      int window_height = window_box_height (w);

	      if (tab_line)
		window_height += CURRENT_TAB_LINE_HEIGHT (w);
	      if (header_line)
		window_height += CURRENT_HEADER_LINE_HEIGHT (w);
	      if (w->cursor.y >= window_height - pixel_margin)
		{
		  w->cursor.vpos = -1;
		  clear_glyph_matrix (w->desired_matrix);
		  goto try_to_scroll;
		}
	    }
	}

      /* If we need to move point for either of the above reasons,
	 now actually do it.  */
      if (new_vpos >= 0)
	{
	  struct glyph_row *row;

	  row = MATRIX_FIRST_TEXT_ROW (w->desired_matrix);
	  while (MATRIX_ROW_BOTTOM_Y (row) < new_vpos
		 && !row->ends_at_zv_p)
	    ++row;

	  TEMP_SET_PT_BOTH (MATRIX_ROW_START_CHARPOS (row),
			    MATRIX_ROW_START_BYTEPOS (row));

	  if (w != XWINDOW (selected_window))
	    set_marker_both (w->pointm, Qnil, PT, PT_BYTE);
	  else if (current_buffer == old)
	    SET_TEXT_POS (lpoint, PT, PT_BYTE);

	  set_cursor_from_row (w, row, w->desired_matrix, 0, 0, 0, 0);

	  /* Re-run pre-redisplay-function so it can update the region
	     according to the new position of point.  */
	  /* Other than the cursor, w's redisplay is done so we can set its
	     redisplay to false.  Also the buffer's redisplay can be set to
	     false, since propagate_buffer_redisplay should have already
	     propagated its info to `w' anyway.  */
	  w->redisplay = false;
	  XBUFFER (w->contents)->text->redisplay = false;
	  safe__call1 (true, Vpre_redisplay_function, Fcons (window, Qnil));

	  if (w->redisplay || XBUFFER (w->contents)->text->redisplay
	      || ((EQ (Vdisplay_line_numbers, Qrelative)
		   || EQ (Vdisplay_line_numbers, Qvisual))
		  && row != MATRIX_FIRST_TEXT_ROW (w->desired_matrix)))
	    {
	      /* Either pre-redisplay-function made changes (e.g. move
		 the region), or we moved point in a window that is
		 under display-line-numbers = relative mode.  We need
		 another round of redisplay.  */
	      clear_glyph_matrix (w->desired_matrix);
	      if (!try_window (window, startp, 0))
		goto need_larger_matrices;
	    }
	}
      if (w->cursor.vpos < 0
	  || !cursor_row_fully_visible_p (w, false, false, false))
	{
	  clear_glyph_matrix (w->desired_matrix);
	  goto try_to_scroll;
	}

#ifdef GLYPH_DEBUG
      debug_method_add (w, "forced window start");
#endif
      goto done;
    }

 ignore_start:

  /* Handle case where text has not changed, only point, and it has
     not moved off the frame, and we are not retrying after hscroll.
     (current_matrix_up_to_date_p is true when retrying.)  */
  if (current_matrix_up_to_date_p
      && (rc = try_cursor_movement (window, startp, &temp_scroll_step),
	  rc != CURSOR_MOVEMENT_CANNOT_BE_USED))
    {
      switch (rc)
	{
	case CURSOR_MOVEMENT_SUCCESS:
	  used_current_matrix_p = true;
	  goto done;

	case CURSOR_MOVEMENT_MUST_SCROLL:
	  goto try_to_scroll;

	default:
	  emacs_abort ();
	}
    }
  /* If current starting point was originally the beginning of a line
     but no longer is, or if the starting point is invisible but the
     buffer wants it always visible, find a new starting point.  */
  else if (w->start_at_line_beg
	   && ((CHARPOS (startp) > BEGV
		&& FETCH_BYTE (BYTEPOS (startp) - 1) != '\n')
	       || (CHARPOS (startp) >= BEGV
		   && CHARPOS (startp) <= ZV
		   && !window_start_acceptable_p (window, CHARPOS (startp)))))
    {
#ifdef GLYPH_DEBUG
      debug_method_add (w, "recenter 1");
#endif
      goto recenter;
    }

  /* Try scrolling with try_window_id.  Value is > 0 if update has
     been done, it is -1 if we know that the same window start will
     not work.  It is 0 if unsuccessful for some other reason.  */
  else if ((tem = try_window_id (w)) != 0)
    {
#ifdef GLYPH_DEBUG
      debug_method_add (w, "try_window_id %d", tem);
#endif

      if (f->fonts_changed)
	goto need_larger_matrices;
      if (tem > 0)
	goto done;

      /* Otherwise try_window_id has returned -1 which means that we
	 don't want the alternative below this comment to execute.  */
    }
  else if (CHARPOS (startp) >= BEGV
	   && CHARPOS (startp) <= ZV
	   && PT >= CHARPOS (startp)
	   && (CHARPOS (startp) < ZV
	       /* Avoid starting at end of buffer.  */
	       || CHARPOS (startp) == BEGV
	       || !window_outdated (w)))
    {
      int d1, d2, d5, d6;
      int rtop, rbot;

      /* If first window line is a continuation line, and window start
	 is inside the modified region, but the first change is before
	 current window start, we must select a new window start.

	 However, if this is the result of a down-mouse event (e.g. by
	 extending the mouse-drag-overlay), we don't want to select a
	 new window start, since that would change the position under
	 the mouse, resulting in an unwanted mouse-movement rather
	 than a simple mouse-click.  */
      if (!w->start_at_line_beg
	  && NILP (track_mouse)
      	  && CHARPOS (startp) > BEGV
	  && CHARPOS (startp) > BEG + beg_unchanged
	  && CHARPOS (startp) <= Z - end_unchanged
	  /* Even if w->start_at_line_beg is nil, a new window may
	     start at a line_beg, since that's how set_buffer_window
	     sets it.  So, we need to check the return value of
	     compute_window_start_on_continuation_line.  (See also
	     bug#197).  */
	  && XMARKER (w->start)->buffer == current_buffer
	  && compute_window_start_on_continuation_line (w)
	  /* It doesn't make sense to force the window start like we
	     do at label force_start if it is already known that point
	     will not be fully visible in the resulting window, because
	     doing so will move point from its correct position
	     instead of scrolling the window to bring point into view.
	     See bug#9324.  */
	  && pos_visible_p (w, PT, &d1, &d2, &rtop, &rbot, &d5, &d6)
	  /* A very tall row could need more than the window height,
	     in which case we accept that it is partially visible.  */
	  && (rtop != 0) == (rbot != 0))
	{
	  w->force_start = true;
	  SET_TEXT_POS_FROM_MARKER (startp, w->start);
#ifdef GLYPH_DEBUG
	  debug_method_add (w, "recomputed window start in continuation line");
#endif
	  goto force_start;
      	}

      /* Don't use the same window-start if it is invisible or covered
	 by a replacing 'display' property and the buffer requested
	 the window-start to be always visible.  */
      if (!window_start_acceptable_p (window, CHARPOS (startp)))
	{
#ifdef GLYPH_DEBUG
	  debug_method_add (w, "recenter 2");
#endif
	  goto recenter;
	}

#ifdef GLYPH_DEBUG
      debug_method_add (w, "same window start");
#endif

      /* Try to redisplay starting at same place as before.
         If point has not moved off frame, accept the results.  */
      if (!current_matrix_up_to_date_p
	  /* Don't use try_window_reusing_current_matrix in this case
	     because a window scroll function can have changed the
	     buffer.  */
	  || !NILP (Vwindow_scroll_functions)
	  || MINI_WINDOW_P (w)
	  || !(used_current_matrix_p
	       = try_window_reusing_current_matrix (w)))
	{
	  IF_DEBUG (debug_method_add (w, "1"));
	  clear_glyph_matrix (w->desired_matrix);
	  if (try_window (window, startp, TRY_WINDOW_CHECK_MARGINS) < 0)
	    /* -1 means we need to scroll.
	       0 means we need new matrices, but fonts_changed
	       is set in that case, so we will detect it below.  */
	    goto try_to_scroll;
	}

      if (f->fonts_changed)
	goto need_larger_matrices;

      if (w->cursor.vpos >= 0)
	{
	  if (!just_this_one_p
	      || current_buffer->clip_changed
	      || BEG_UNCHANGED < CHARPOS (startp))
	    /* Forget any recorded base line for line number display.  */
	    w->base_line_number = 0;

	  if (!cursor_row_fully_visible_p (w, true, false, false))
	    {
	      clear_glyph_matrix (w->desired_matrix);
	      last_line_misfit = true;
	    }
	    /* Drop through and scroll.  */
	  else
	    goto done;
	}
      else
	clear_glyph_matrix (w->desired_matrix);
    }

 try_to_scroll:

  /* Redisplay the mode line.  Select the buffer properly for that.  */
  if (!update_mode_line)
    {
      update_mode_line = true;
      w->update_mode_line = true;
    }

  /* Try to scroll by specified few lines.  */
  if ((0 < scroll_conservatively
       /* FIXME: the option is supposed to affect minibuffers, but we
	  test MINI_WINDOW_P, which can also catch uses of
	  mini-windows for displaying the echo area.  Do we need to
	  distinguish these two use cases?  */
       || (scroll_minibuffer_conservatively && MINI_WINDOW_P (w))
       || 0 < emacs_scroll_step
       || temp_scroll_step
       || NUMBERP (BVAR (current_buffer, scroll_up_aggressively))
       || NUMBERP (BVAR (current_buffer, scroll_down_aggressively)))
      && CHARPOS (startp) >= BEGV
      && CHARPOS (startp) <= ZV)
    {
      /* The function returns -1 if new fonts were loaded, 1 if
	 successful, 0 if not successful.  */
      int ss = try_scrolling (window, just_this_one_p,
			      ((scroll_minibuffer_conservatively
			        && MINI_WINDOW_P (w))
			       ? SCROLL_LIMIT + 1
			       : scroll_conservatively),
			      emacs_scroll_step,
			      temp_scroll_step, last_line_misfit);
      switch (ss)
	{
	case SCROLLING_SUCCESS:
	  goto done;

	case SCROLLING_NEED_LARGER_MATRICES:
	  goto need_larger_matrices;

	case SCROLLING_FAILED:
	  break;

	default:
	  emacs_abort ();
	}
    }

  /* Finally, just choose a place to start which positions point
     according to user preferences.  */

 recenter:

#ifdef GLYPH_DEBUG
  debug_method_add (w, "recenter");
#endif

  /* Forget any previously recorded base line for line number display.  */
  if (!buffer_unchanged_p)
    w->base_line_number = 0;

  /* Determine the window start relative to point.  */
  init_iterator (&it, w, PT, PT_BYTE, NULL, DEFAULT_FACE_ID);
  it.current_y = it.last_visible_y;
  if (centering_position < 0)
    {
      ptrdiff_t margin_pos = CHARPOS (startp);
      Lisp_Object aggressive;
      bool scrolling_up;

      /* If there is a scroll margin at the top of the window, find
	 its character position.  */
      if (margin
	  /* Cannot call start_display if startp is not in the
	     accessible region of the buffer.  This can happen when we
	     have just switched to a different buffer and/or changed
	     its restriction.  In that case, startp is initialized to
	     the character position 1 (BEGV) because we did not yet
	     have chance to display the buffer even once.  */
	  && BEGV <= CHARPOS (startp) && CHARPOS (startp) <= ZV)
	{
	  struct it it1;
	  void *it1data = NULL;

	  SAVE_IT (it1, it, it1data);
	  start_display (&it1, w, startp);
	  move_it_vertically (&it1, margin * frame_line_height);
	  margin_pos = IT_CHARPOS (it1);
	  RESTORE_IT (&it, &it, it1data);
	}
      scrolling_up = PT > margin_pos;
      aggressive =
	scrolling_up
	? BVAR (current_buffer, scroll_up_aggressively)
	: BVAR (current_buffer, scroll_down_aggressively);

      if (!MINI_WINDOW_P (w)
	  && (scroll_conservatively > SCROLL_LIMIT || NUMBERP (aggressive)))
	{
	  int pt_offset = 0;

	  /* Setting scroll-conservatively overrides
	     scroll-*-aggressively.  */
	  if (!scroll_conservatively && NUMBERP (aggressive))
	    {
	      double float_amount = XFLOATINT (aggressive);

	      pt_offset = float_amount * WINDOW_BOX_TEXT_HEIGHT (w);
	      if (pt_offset == 0 && float_amount > 0)
		pt_offset = 1;
	      if (pt_offset && margin > 0)
		margin -= 1;
	    }
	  /* Compute how much to move the window start backward from
	     point so that point will be displayed where the user
	     wants it.  */
	  if (scrolling_up)
	    {
	      centering_position = it.last_visible_y;
	      if (pt_offset)
		centering_position -= pt_offset;
	      centering_position -=
		(frame_line_height * (1 + margin + last_line_misfit)
		 + WINDOW_TAB_LINE_HEIGHT (w) + WINDOW_HEADER_LINE_HEIGHT (w));
	      /* Don't let point enter the scroll margin near top of
		 the window.  */
	      if (centering_position < margin * frame_line_height)
		centering_position = margin * frame_line_height;
	    }
	  else
	    centering_position = margin * frame_line_height + pt_offset;
	}
      else
	/* Set the window start half the height of the window backward
	   from point.  */
	centering_position = window_box_height (w) / 2;
    }
  if (current_buffer->long_line_optimizations_p
      && it.line_wrap == TRUNCATE)
    {
      /* For very long and truncated lines, go back using a simplified
	 method, which ignored any inaccuracies due to line-height
	 differences, display properties/overlays, etc.  */
      int nlines = centering_position / frame_line_height;

      while (nlines-- && IT_CHARPOS (it) > BEGV)
	back_to_previous_visible_line_start (&it);
      reseat_1 (&it, it.current.pos, true);
    }
  else
    move_it_vertically_backward (&it, centering_position);

  eassert (IT_CHARPOS (it) >= BEGV);

  /* The function move_it_vertically_backward may move over more
     than the specified y-distance.  If it->w is small, e.g. a
     mini-buffer window, we may end up in front of the window's
     display area.  Start displaying at the start of the line
     containing PT in this case.  */
  if (it.current_y <= 0)
    {
      init_iterator (&it, w, PT, PT_BYTE, NULL, DEFAULT_FACE_ID);
      move_it_vertically_backward (&it, 0);
      it.current_y = 0;
    }

  it.current_x = it.hpos = 0;

  /* Set the window start position here explicitly, to avoid an
     infinite loop in case the functions in window-scroll-functions
     get errors.  */
  set_marker_both (w->start, Qnil, IT_CHARPOS (it), IT_BYTEPOS (it));

  /* Run scroll hooks.  */
  startp = run_window_scroll_functions (window, it.current.pos);

  /* We invoke try_window and try_window_reusing_current_matrix below,
     and they manipulate the bidi cache.  Save and restore the cache
     state of our iterator, so we could continue using it after that.  */
  itdata = bidi_shelve_cache ();

  /* Redisplay the window.  */
  use_desired_matrix = false;
  if (!current_matrix_up_to_date_p
      || windows_or_buffers_changed
      || f->cursor_type_changed
      /* Don't use try_window_reusing_current_matrix in this case
	 because it can have changed the buffer.  */
      || !NILP (Vwindow_scroll_functions)
      || !just_this_one_p
      || MINI_WINDOW_P (w)
      || !(used_current_matrix_p
	   = try_window_reusing_current_matrix (w)))
    use_desired_matrix = (try_window (window, startp, 0) == 1);

  bidi_unshelve_cache (itdata, false);

  /* If new fonts have been loaded (due to fontsets), give up.  We
     have to start a new redisplay since we need to re-adjust glyph
     matrices.  */
  if (f->fonts_changed)
    goto need_larger_matrices;

  /* If cursor did not appear assume that the middle of the window is
     in the first line of the window.  Do it again with the next line.
     (Imagine a window of height 100, displaying two lines of height
     60.  Moving back 50 from it->last_visible_y will end in the first
     line.)  */
  if (w->cursor.vpos < 0)
    {
      if (w->window_end_valid && PT >= Z - w->window_end_pos)
	{
	  clear_glyph_matrix (w->desired_matrix);
	  move_it_by_lines (&it, 1);
	  try_window (window, it.current.pos, 0);
	}
      else if (PT < IT_CHARPOS (it))
	{
	  clear_glyph_matrix (w->desired_matrix);
	  move_it_by_lines (&it, -1);
	  try_window (window, it.current.pos, 0);
	}
      else if (scroll_conservatively > SCROLL_LIMIT
	       && (it.method == GET_FROM_STRING
		   || overlay_touches_p (IT_CHARPOS (it)))
	       && IT_CHARPOS (it) < ZV)
	{
	  /* If the window starts with a before-string that spans more
	     than one screen line, using that position to display the
	     window might fail to bring point into the view, because
	     start_display will always start by displaying the string,
	     whereas the code above determines where to set w->start
	     by the buffer position of the place where it takes screen
	     coordinates.  Try to recover by finding the next screen
	     line that displays buffer text.  */
	  ptrdiff_t pos0 = IT_CHARPOS (it);

	  clear_glyph_matrix (w->desired_matrix);
	  do {
	    move_it_by_lines (&it, 1);
	  } while (IT_CHARPOS (it) == pos0);
	  try_window (window, it.current.pos, 0);
	}
      else
	{
	  /* Not much we can do about it.  */
	}
    }

  /* Consider the following case: Window starts at BEGV, there is
     invisible, intangible text at BEGV, so that display starts at
     some point START > BEGV.  It can happen that we are called with
     PT somewhere between BEGV and START.  Try to handle that case,
     and similar ones.  */
  if (w->cursor.vpos < 0)
    {
      /* Prefer the desired matrix to the current matrix, if possible,
	 in the fallback calculations below.  This is because using
	 the current matrix might completely goof, e.g. if its first
	 row is after point.  */
      struct glyph_matrix *matrix =
	use_desired_matrix ? w->desired_matrix : w->current_matrix;
      /* First, try locating the proper glyph row for PT.  */
      struct glyph_row *row =
	row_containing_pos (w, PT, matrix->rows, NULL, 0);

      /* Sometimes point is at the beginning of invisible text that is
	 before the 1st character displayed in the row.  In that case,
	 row_containing_pos fails to find the row, because no glyphs
	 with appropriate buffer positions are present in the row.
	 Therefore, we next try to find the row which shows the 1st
	 position after the invisible text.  */
      if (!row)
	{
	  Lisp_Object val =
	    get_char_property_and_overlay (make_fixnum (PT), Qinvisible,
					   Qnil, NULL);

	  if (TEXT_PROP_MEANS_INVISIBLE (val) != 0)
	    {
	      ptrdiff_t alt_pos;
	      Lisp_Object invis_end =
		Fnext_single_char_property_change (make_fixnum (PT), Qinvisible,
						   Qnil, Qnil);

	      if (FIXNATP (invis_end))
		alt_pos = XFIXNAT (invis_end);
	      else
		alt_pos = ZV;
	      row = row_containing_pos (w, alt_pos, matrix->rows, NULL, 0);
	    }
	}
      /* Finally, fall back on the first row of the window after the
	 tab-line and header line (if any).  This is slightly better
	 than not displaying the cursor at all.  */
      if (!row)
	{
	  row = matrix->rows;
	  /* Skip the tab-line and header-line rows, if any.  */
	  if (row->tab_line_p)
	    ++row;
	  if (row->mode_line_p)
	    ++row;
	}
      set_cursor_from_row (w, row, matrix, 0, 0, 0, 0);
    }

  if (!cursor_row_fully_visible_p (w, false, false, false))
    {
      /* If vscroll is enabled, disable it and try again.  */
      if (w->vscroll)
	{
	  w->vscroll = 0;
	  clear_glyph_matrix (w->desired_matrix);
	  goto recenter;
	}

      /* Users who set scroll-conservatively to a large number want
	 point just above/below the scroll margin.  If we ended up
	 with point's row partially visible, move the window start to
	 make that row fully visible and out of the margin.  */
      if (scroll_conservatively > SCROLL_LIMIT)
	{
	  int window_total_lines
	    = WINDOW_TOTAL_LINES (w) * FRAME_LINE_HEIGHT (f) / frame_line_height;
	  bool move_down = w->cursor.vpos >= window_total_lines / 2;

	  move_it_by_lines (&it, move_down ? margin + 1 : -(margin + 1));
	  clear_glyph_matrix (w->desired_matrix);
	  if (1 == try_window (window, it.current.pos,
			       TRY_WINDOW_CHECK_MARGINS))
	    goto done;
	}

      /* If centering point failed to make the whole line visible,
	 put point at the top instead.  That has to make the whole line
	 visible, if it can be done.  */
      if (centering_position == 0)
	goto done;

      clear_glyph_matrix (w->desired_matrix);
      centering_position = 0;
      goto recenter;
    }

 done:

  SET_TEXT_POS_FROM_MARKER (startp, w->start);
  w->start_at_line_beg = (CHARPOS (startp) == BEGV
			  || FETCH_BYTE (BYTEPOS (startp) - 1) == '\n');

  /* Display the mode line, header line, and tab-line, if we must.  */
  if ((update_mode_line
       /* If window not full width, must redo its mode line
	  if (a) the window to its side is being redone and
	  (b) we do a frame-based redisplay.  This is a consequence
	  of how inverted lines are drawn in frame-based redisplay.  */
       || (!just_this_one_p
	   && !FRAME_WINDOW_P (f)
	   && !WINDOW_FULL_WIDTH_P (w))
       /* Line number to display.  */
       || w->base_line_pos > 0
       /* Column number is displayed and different from the one displayed.  */
       || (w->column_number_displayed != -1
	   && (w->column_number_displayed != current_column ())))
      /* This means that the window has a mode line.  */
      && (window_wants_mode_line (w)
	  || window_wants_header_line (w)
	  || window_wants_tab_line (w)))
    {
      specpdl_ref count1 = SPECPDL_INDEX ();

      specbind (Qinhibit_quit, Qt);
      display_mode_lines (w);
      unbind_to (count1, Qnil);

      /* If mode line height has changed, arrange for a thorough
	 immediate redisplay using the correct mode line height.  */
      if (window_wants_mode_line (w)
	  && CURRENT_MODE_LINE_HEIGHT (w) != DESIRED_MODE_LINE_HEIGHT (w))
	{
	  f->fonts_changed = true;
	  w->mode_line_height = -1;
	  MATRIX_MODE_LINE_ROW (w->current_matrix)->height
	    = DESIRED_MODE_LINE_HEIGHT (w);
	}

      /* If tab line height has changed, arrange for a thorough
	 immediate redisplay using the correct tab line height.  */
      if (window_wants_tab_line (w)
	  && CURRENT_TAB_LINE_HEIGHT (w) != DESIRED_TAB_LINE_HEIGHT (w))
	{
	  f->fonts_changed = true;
	  w->tab_line_height = -1;
	  MATRIX_TAB_LINE_ROW (w->current_matrix)->height
	    = DESIRED_TAB_LINE_HEIGHT (w);
	}

      /* If header line height has changed, arrange for a thorough
	 immediate redisplay using the correct header line height.  */
      if (window_wants_header_line (w)
	  && CURRENT_HEADER_LINE_HEIGHT (w) != DESIRED_HEADER_LINE_HEIGHT (w))
	{
	  f->fonts_changed = true;
	  w->header_line_height = -1;
	  MATRIX_HEADER_LINE_ROW (w->current_matrix)->height
	    = DESIRED_HEADER_LINE_HEIGHT (w);
	}

      if (f->fonts_changed)
	goto need_larger_matrices;
    }

  if (!line_number_displayed && w->base_line_pos != -1)
    {
      w->base_line_pos = 0;
      w->base_line_number = 0;
    }

 finish_menu_bars:

  /* When we reach a frame's selected window, redo the frame's menu
     bar, tool bar, tab-bar, and the frame's title.  */
  if (update_mode_line
      && EQ (FRAME_SELECTED_WINDOW (f), window))
    {
      bool redisplay_menu_p;

      if (FRAME_WINDOW_P (f))
	{
#ifdef HAVE_EXT_MENU_BAR
	  redisplay_menu_p = FRAME_EXTERNAL_MENU_BAR (f);
#else
	  redisplay_menu_p = FRAME_MENU_BAR_LINES (f) > 0;
#endif
	}
      else
        redisplay_menu_p = FRAME_MENU_BAR_LINES (f) > 0;

      if (redisplay_menu_p)
        display_menu_bar (w);

#ifdef HAVE_WINDOW_SYSTEM
      if (FRAME_WINDOW_P (f))
        {
	  if (WINDOWP (f->tab_bar_window)
	      && (FRAME_TAB_BAR_LINES (f) > 0
		  || !NILP (Vauto_resize_tab_bars))
	      && redisplay_tab_bar (f))
	    ignore_mouse_drag_p = true;

#ifdef HAVE_EXT_TOOL_BAR
	  if (FRAME_EXTERNAL_TOOL_BAR (f))
	    update_frame_tool_bar (f);
#else
	  if (WINDOWP (f->tool_bar_window)
	      && (FRAME_TOOL_BAR_LINES (f) > 0
		  || !NILP (Vauto_resize_tool_bars))
	      && redisplay_tool_bar (f))
	    ignore_mouse_drag_p = true;
#endif
        }
      else
        {
          if ((FRAME_TAB_BAR_LINES (f) > 0))
            display_tab_bar (w);
        }

      gui_consider_frame_title (w->frame);
#else
      if ((FRAME_TAB_BAR_LINES (f) > 0))
        display_tab_bar (w);
#endif
    }

#ifdef HAVE_WINDOW_SYSTEM
  if (FRAME_WINDOW_P (f)
      && update_window_fringes (w, (just_this_one_p
				    || (!used_current_matrix_p && !overlay_arrow_seen)
				    || w->pseudo_window_p)))
    {
      update_begin (f);
      block_input ();
      if (draw_window_fringes (w, true))
	{
	  if (WINDOW_RIGHT_DIVIDER_WIDTH (w))
	    gui_draw_right_divider (w);
	  else
	    gui_draw_vertical_border (w);
	}
      unblock_input ();
      update_end (f);
    }

  if (WINDOW_BOTTOM_DIVIDER_WIDTH (w))
    gui_draw_bottom_divider (w);
#endif /* HAVE_WINDOW_SYSTEM */

  /* We go to this label, with fonts_changed set, if it is
     necessary to try again using larger glyph matrices.
     We have to redeem the scroll bar even in this case,
     because the loop in redisplay_internal expects that.  */
 need_larger_matrices:
  ;
 finish_scroll_bars:

   if (WINDOW_HAS_VERTICAL_SCROLL_BAR (w) || WINDOW_HAS_HORIZONTAL_SCROLL_BAR (w))
    {
      if (WINDOW_HAS_VERTICAL_SCROLL_BAR (w))
	/* Set the thumb's position and size.  */
	set_vertical_scroll_bar (w);

      if (WINDOW_HAS_HORIZONTAL_SCROLL_BAR (w))
	/* Set the thumb's position and size.  */
	set_horizontal_scroll_bar (w);

      /* Note that we actually used the scroll bar attached to this
	 window, so it shouldn't be deleted at the end of redisplay.  */
      if (FRAME_TERMINAL (f)->redeem_scroll_bar_hook)
        (*FRAME_TERMINAL (f)->redeem_scroll_bar_hook) (w);
    }

  /* Restore current_buffer and value of point in it.  The window
     update may have changed the buffer, so first make sure `opoint'
     is still valid (Bug#6177).  */
  if (CHARPOS (opoint) < BEGV)
    TEMP_SET_PT_BOTH (BEGV, BEGV_BYTE);
  else if (CHARPOS (opoint) > ZV)
    TEMP_SET_PT_BOTH (Z, Z_BYTE);
  else
    TEMP_SET_PT_BOTH (CHARPOS (opoint), BYTEPOS (opoint));

  set_buffer_internal_1 (old);
  /* Avoid an abort in TEMP_SET_PT_BOTH if the buffer has become
     shorter.  This can be caused by log truncation in *Messages*.  */
  if (CHARPOS (lpoint) <= ZV)
    TEMP_SET_PT_BOTH (CHARPOS (lpoint), BYTEPOS (lpoint));

  unbind_to (count, Qnil);
}


/* Build the complete desired matrix of WINDOW with a window start
   buffer position POS.

   Value is 1 if successful.  It is zero if fonts were loaded during
   redisplay or the dimensions of the desired matrix were found
   insufficient, which makes re-adjusting glyph matrices necessary.
   Value is -1 if point would appear in the scroll margins.  (We check
   the former only if TRY_WINDOW_IGNORE_FONTS_CHANGE is unset in
   FLAGS, and the latter only if TRY_WINDOW_CHECK_MARGINS is set in
   FLAGS.)

   Note that 'x-show-tip' invokes this function in a special way, and
   in that case the return value of zero doesn't necessarily mean the
   glyph matrices need to be re-adjusted, if the entire text of the
   tooltip was processed and has its glyphs in the matrix's glyph
   rows, i.e. if the dimensions of the matrix were found insufficient
   while producing empty glyph rows beyond ZV.  */

int
try_window (Lisp_Object window, struct text_pos pos, int flags)
{
  struct window *w = XWINDOW (window);
  struct it it;
  struct glyph_row *last_text_row = NULL;
  struct frame *f = XFRAME (w->frame);
  int cursor_vpos = w->cursor.vpos;

  /* Make POS the new window start.  */
  set_marker_both (w->start, Qnil, CHARPOS (pos), BYTEPOS (pos));

  /* Mark cursor position as unknown.  No overlay arrow seen.  */
  w->cursor.vpos = -1;
  overlay_arrow_seen = false;

  /* Initialize iterator and info to start at POS.  */
  start_display (&it, w, pos);
  it.glyph_row->reversed_p = false;

  /* Display all lines of W.  */
  while (it.current_y < it.last_visible_y)
    {
      int last_row_scale = it.w->nrows_scale_factor;
      int last_col_scale = it.w->ncols_scale_factor;
      if (display_line (&it, cursor_vpos))
	last_text_row = it.glyph_row - 1;
      if (f->fonts_changed
	  && !((flags & TRY_WINDOW_IGNORE_FONTS_CHANGE)
	       /* If the matrix dimensions are insufficient, we _must_
		  fail and let dispnew.c reallocate the matrix.  */
	       && last_row_scale == it.w->nrows_scale_factor
	       && last_col_scale == it.w->ncols_scale_factor))
	return 0;
    }

  /* Save the character position of 'it' before we call
     'start_display' again.  */
  ptrdiff_t it_charpos = IT_CHARPOS (it);

  /* Don't let the cursor end in the scroll margins.  However, when
     the window is vscrolled, we leave it to vscroll to handle the
     margins, see window_scroll_pixel_based.  */
  if ((flags & TRY_WINDOW_CHECK_MARGINS)
      && w->vscroll == 0
      && !MINI_WINDOW_P (w))
    {
      int top_scroll_margin = window_scroll_margin (w, MARGIN_IN_PIXELS);
      int bot_scroll_margin = top_scroll_margin;
      if (window_wants_header_line (w))
	top_scroll_margin += CURRENT_HEADER_LINE_HEIGHT (w);
      start_display (&it, w, pos);

      if ((w->cursor.y >= 0
	   && w->cursor.y < top_scroll_margin
	   && CHARPOS (pos) > BEGV)
	  /* rms: considering make_cursor_line_fully_visible_p here
	     seems to give wrong results.  We don't want to recenter
	     when the last line is partly visible, we want to allow
	     that case to be handled in the usual way.  */
	  || w->cursor.y > (it.last_visible_y - partial_line_height (&it)
			    - bot_scroll_margin - 1))
	{
	  w->cursor.vpos = -1;
	  clear_glyph_matrix (w->desired_matrix);
	  return -1;
	}
    }

  /* If bottom moved off end of frame, change mode line percentage.  */
  if (w->window_end_pos <= 0 && Z != it_charpos)
    w->update_mode_line = true;

  /* Set window_end_pos to the offset of the last character displayed
     on the window from the end of current_buffer.  Set
     window_end_vpos to its row number.  */
  if (last_text_row)
    {
      eassert (MATRIX_ROW_DISPLAYS_TEXT_P (last_text_row));
      adjust_window_ends (w, last_text_row, false);
      eassert
	(MATRIX_ROW_DISPLAYS_TEXT_P (MATRIX_ROW (w->desired_matrix,
						 w->window_end_vpos)));
    }
  else
    {
      w->window_end_bytepos = Z_BYTE - ZV_BYTE;
      w->window_end_pos = Z - ZV;
      w->window_end_vpos = 0;
    }

  /* But that is not valid info until redisplay finishes.  */
  w->window_end_valid = false;
  return 1;
}



/************************************************************************
    Window redisplay reusing current matrix when buffer has not changed
 ************************************************************************/

/* Try redisplay of window W showing an unchanged buffer with a
   different window start than the last time it was displayed by
   reusing its current matrix.  Value is true if successful.
   W->start is the new window start.  */

static bool
try_window_reusing_current_matrix (struct window *w)
{
  struct frame *f = XFRAME (w->frame);
  struct glyph_row *bottom_row;
  struct it it;
  struct run run;
  struct text_pos start, new_start;
  int nrows_scrolled, i;
  struct glyph_row *last_text_row;
  struct glyph_row *last_reused_text_row;
  struct glyph_row *start_row;
  int start_vpos, min_y, max_y;

#ifdef GLYPH_DEBUG
  if (inhibit_try_window_reusing)
    return false;
#endif

  if (/* This function doesn't handle terminal frames.  */
      !FRAME_WINDOW_P (f)
      /* Don't try to reuse the display if windows have been split
	 or such.  */
      || windows_or_buffers_changed
      || f->cursor_type_changed
      /* This function cannot handle buffers where the overlay arrow
	 is shown on the fringes, because if the arrow position
	 changes, we cannot just reuse the current matrix.  */
      || overlay_arrow_in_current_buffer_p ())
    return false;

  /* Can't do this if showing trailing whitespace.  */
  if (!NILP (Vshow_trailing_whitespace))
    return false;

  /* If top-line visibility has changed, give up.  */
  if (window_wants_tab_line (w)
      != MATRIX_TAB_LINE_ROW (w->current_matrix)->mode_line_p)
    return false;

  /* If top-line visibility has changed, give up.  */
  if (window_wants_header_line (w)
      != MATRIX_HEADER_LINE_ROW (w->current_matrix)->mode_line_p)
    return false;

  /* Give up if old or new display is scrolled vertically.  We could
     make this function handle this, but right now it doesn't.  */
  start_row = MATRIX_FIRST_TEXT_ROW (w->current_matrix);
  if (w->vscroll || MATRIX_ROW_PARTIALLY_VISIBLE_P (w, start_row))
    return false;

  /* Clear the desired matrix for the display below.  */
  clear_glyph_matrix (w->desired_matrix);

  /* Give up if line numbers are being displayed, because reusing the
     current matrix might use the wrong width for line-number
     display.  */
  if (!NILP (Vdisplay_line_numbers))
    return false;

  /* Can't scroll the display of w32 GUI frames when position of point
     is indicated by the system caret, because scrolling the display
     will then "copy" the pixels used by the caret.  */
#ifdef HAVE_NTGUI
  if (w32_use_visible_system_caret)
    return false;
#endif

  /* The variable new_start now holds the new window start.  The old
     start `start' can be determined from the current matrix.  */
  SET_TEXT_POS_FROM_MARKER (new_start, w->start);
  start = start_row->minpos;
  start_vpos = MATRIX_ROW_VPOS (start_row, w->current_matrix);

  if (CHARPOS (new_start) <= CHARPOS (start))
    {
      /* Don't use this method if the display starts with an ellipsis
	 displayed for invisible text.  It's not easy to handle that case
	 below, and it's certainly not worth the effort since this is
	 not a frequent case.  */
      if (in_ellipses_for_invisible_text_p (&start_row->start, w))
	return false;

      IF_DEBUG (debug_method_add (w, "twu1"));

      /* Display up to a row that can be reused.  The variable
	 last_text_row is set to the last row displayed that displays
	 text.  Note that it.vpos == 0 if or if not there is a
         header-line; it's not the same as the MATRIX_ROW_VPOS!  */
      start_display (&it, w, new_start);
      w->cursor.vpos = -1;
      last_text_row = last_reused_text_row = NULL;

      while (it.current_y < it.last_visible_y && !f->fonts_changed)
	{
	  /* If we have reached into the characters in the START row,
	     that means the line boundaries have changed.  So we
	     can't start copying with the row START.  Maybe it will
	     work to start copying with the following row.  */
	  while (IT_CHARPOS (it) > CHARPOS (start))
	    {
	      /* Advance to the next row as the "start".  */
	      start_row++;
	      start = start_row->minpos;
	      /* If there are no more rows to try, or just one, give up.  */
	      if (start_row == MATRIX_MODE_LINE_ROW (w->current_matrix) - 1
		  || w->vscroll || MATRIX_ROW_PARTIALLY_VISIBLE_P (w, start_row)
		  || CHARPOS (start) == ZV)
		{
		  clear_glyph_matrix (w->desired_matrix);
		  return false;
		}

	      start_vpos = MATRIX_ROW_VPOS (start_row, w->current_matrix);
	    }
	  /* If we have reached alignment, we can copy the rest of the
	     rows.  */
	  if (IT_CHARPOS (it) == CHARPOS (start)
	      /* Don't accept "alignment" inside a display vector,
		 since start_row could have started in the middle of
		 that same display vector (thus their character
		 positions match), and we have no way of telling if
		 that is the case.  */
	      && it.current.dpvec_index < 0)
	    break;

	  it.glyph_row->reversed_p = false;
	  if (display_line (&it, -1))
	    last_text_row = it.glyph_row - 1;

	}

      /* A value of current_y < last_visible_y means that we stopped
	 at the previous window start, which in turn means that we
	 have at least one reusable row.  */
      if (it.current_y < it.last_visible_y)
	{
	  struct glyph_row *row;

	  /* IT.vpos always starts from 0; it counts text lines.  */
	  nrows_scrolled = it.vpos - (start_row - MATRIX_FIRST_TEXT_ROW (w->current_matrix));

	  /* Find PT if not already found in the lines displayed.  */
	  if (w->cursor.vpos < 0)
	    {
	      int dy = it.current_y - start_row->y;

	      row = MATRIX_FIRST_TEXT_ROW (w->current_matrix);
	      row = row_containing_pos (w, PT, row, NULL, dy);
	      if (row)
		set_cursor_from_row (w, row, w->current_matrix, 0, 0,
				     dy, nrows_scrolled);
	      else
		{
		  clear_glyph_matrix (w->desired_matrix);
		  return false;
		}
	    }

	  /* Scroll the display.  Do it before the current matrix is
	     changed.  The problem here is that update has not yet
	     run, i.e. part of the current matrix is not up to date.
	     scroll_run_hook will clear the cursor, and use the
	     current matrix to get the height of the row the cursor is
	     in.  */
	  run.current_y = start_row->y;
	  run.desired_y = it.current_y;
	  run.height = it.last_visible_y - it.current_y;

	  if (run.height > 0 && run.current_y != run.desired_y)
	    {
#ifdef HAVE_WINDOW_SYSTEM
	      update_begin (f);
	      gui_update_window_begin (w);
	      FRAME_RIF (f)->clear_window_mouse_face (w);
	      FRAME_RIF (f)->scroll_run_hook (w, &run);
	      gui_update_window_end (w, false, false);
	      update_end (f);
#endif
	    }

	  /* Shift current matrix down by nrows_scrolled lines.  */
	  bottom_row = MATRIX_BOTTOM_TEXT_ROW (w->current_matrix, w);
	  rotate_matrix (w->current_matrix,
			 start_vpos,
			 MATRIX_ROW_VPOS (bottom_row, w->current_matrix),
			 nrows_scrolled);

	  /* Disable lines that must be updated.  */
	  for (i = 0; i < nrows_scrolled; ++i)
	    (start_row + i)->enabled_p = false;

	  /* Re-compute Y positions.  */
	  min_y = WINDOW_TAB_LINE_HEIGHT (w) + WINDOW_HEADER_LINE_HEIGHT (w);
	  max_y = it.last_visible_y;
	  for (row = start_row + nrows_scrolled;
	       row < bottom_row;
	       ++row)
	    {
	      row->y = it.current_y;
	      row->visible_height = row->height;

	      if (row->y < min_y)
		row->visible_height -= min_y - row->y;
	      if (row->y + row->height > max_y)
		row->visible_height -= row->y + row->height - max_y;
	      if (row->fringe_bitmap_periodic_p)
		row->redraw_fringe_bitmaps_p = true;

	      it.current_y += row->height;

	      if (MATRIX_ROW_DISPLAYS_TEXT_P (row))
		last_reused_text_row = row;
	      if (MATRIX_ROW_BOTTOM_Y (row) >= it.last_visible_y)
		break;
	    }

	  /* Disable lines in the current matrix which are now
	     below the window.  */
	  for (++row; row < bottom_row; ++row)
	    row->enabled_p = row->mode_line_p = row->tab_line_p = false;
	}

      /* Update window_end_pos etc.; last_reused_text_row is the last
	 reused row from the current matrix containing text, if any.
	 The value of last_text_row is the last displayed line
	 containing text.  */
      if (last_reused_text_row)
	adjust_window_ends (w, last_reused_text_row, true);
      else if (last_text_row)
	adjust_window_ends (w, last_text_row, false);
      else
	{
	  /* This window must be completely empty.  */
	  w->window_end_bytepos = Z_BYTE - ZV_BYTE;
	  w->window_end_pos = Z - ZV;
	  w->window_end_vpos = 0;
	}
      w->window_end_valid = false;

      /* Update hint: don't try scrolling again in update_window.  */
      w->desired_matrix->no_scrolling_p = true;

#ifdef GLYPH_DEBUG
      debug_method_add (w, "try_window_reusing_current_matrix 1");
#endif
      return true;
    }
  else if (CHARPOS (new_start) > CHARPOS (start))
    {
      struct glyph_row *pt_row, *row;
      struct glyph_row *first_reusable_row;
      struct glyph_row *first_row_to_display;
      int dy;
      int yb = window_text_bottom_y (w);

      /* Find the row starting at new_start, if there is one.  Don't
	 reuse a partially visible line at the end.  */
      first_reusable_row = start_row;
      while (first_reusable_row->enabled_p
	     && MATRIX_ROW_BOTTOM_Y (first_reusable_row) < yb
	     && (MATRIX_ROW_START_CHARPOS (first_reusable_row)
		 < CHARPOS (new_start)))
	++first_reusable_row;

      /* Give up if there is no row to reuse.  */
      if (MATRIX_ROW_BOTTOM_Y (first_reusable_row) >= yb
	  || !first_reusable_row->enabled_p
	  || (MATRIX_ROW_START_CHARPOS (first_reusable_row)
	      != CHARPOS (new_start)))
	return false;

      /* We can reuse fully visible rows beginning with
         first_reusable_row to the end of the window.  Set
         first_row_to_display to the first row that cannot be reused.
         Set pt_row to the row containing point, if there is any.  */
      pt_row = NULL;
      for (first_row_to_display = first_reusable_row;
	   MATRIX_ROW_BOTTOM_Y (first_row_to_display) < yb;
	   ++first_row_to_display)
	{
	  if (PT >= MATRIX_ROW_START_CHARPOS (first_row_to_display)
	      && (PT < MATRIX_ROW_END_CHARPOS (first_row_to_display)
		  || (PT == MATRIX_ROW_END_CHARPOS (first_row_to_display)
		      && first_row_to_display->ends_at_zv_p
		      && pt_row == NULL)))
	    pt_row = first_row_to_display;
	}

      /* Start displaying at the start of first_row_to_display.  */
      eassert (first_row_to_display->y < yb);
      init_to_row_start (&it, w, first_row_to_display);

      nrows_scrolled = (MATRIX_ROW_VPOS (first_reusable_row, w->current_matrix)
			- start_vpos);
      it.vpos = (MATRIX_ROW_VPOS (first_row_to_display, w->current_matrix)
		 - nrows_scrolled);
      it.current_y = (first_row_to_display->y - first_reusable_row->y
		      + WINDOW_TAB_LINE_HEIGHT (w) + WINDOW_HEADER_LINE_HEIGHT (w));

      /* Display lines beginning with first_row_to_display in the
         desired matrix.  Set last_text_row to the last row displayed
         that displays text.  */
      it.glyph_row = MATRIX_ROW (w->desired_matrix, it.vpos);
      if (pt_row == NULL)
	w->cursor.vpos = -1;
      last_text_row = NULL;
      while (it.current_y < it.last_visible_y && !f->fonts_changed)
	if (display_line (&it, w->cursor.vpos))
	  last_text_row = it.glyph_row - 1;

      /* If point is in a reused row, adjust y and vpos of the cursor
	 position.  */
      if (pt_row)
	{
	  w->cursor.vpos -= nrows_scrolled;
	  w->cursor.y -= first_reusable_row->y - start_row->y;
	}

      /* Give up if point isn't in a row displayed or reused.  (This
	 also handles the case where w->cursor.vpos < nrows_scrolled
	 after the calls to display_line, which can happen with scroll
	 margins.  See bug#1295.)  */
      if (w->cursor.vpos < 0)
	{
	  clear_glyph_matrix (w->desired_matrix);
	  return false;
	}

      /* Scroll the display.  */
      run.current_y = first_reusable_row->y;
      run.desired_y = WINDOW_TAB_LINE_HEIGHT (w) + WINDOW_HEADER_LINE_HEIGHT (w);
      run.height = it.last_visible_y - run.current_y;
      dy = run.current_y - run.desired_y;

      if (run.height)
	{
#ifdef HAVE_WINDOW_SYSTEM
	  update_begin (f);
	  gui_update_window_begin (w);
	  FRAME_RIF (f)->clear_window_mouse_face (w);
	  FRAME_RIF (f)->scroll_run_hook (w, &run);
	  gui_update_window_end (w, false, false);
	  update_end (f);
#endif
	}

      /* Adjust Y positions of reused rows.  */
      bottom_row = MATRIX_BOTTOM_TEXT_ROW (w->current_matrix, w);
      min_y = WINDOW_TAB_LINE_HEIGHT (w) + WINDOW_HEADER_LINE_HEIGHT (w);
      max_y = it.last_visible_y;
      for (row = first_reusable_row; row < first_row_to_display; ++row)
	{
	  row->y -= dy;
	  row->visible_height = row->height;
	  if (row->y < min_y)
	    row->visible_height -= min_y - row->y;
	  if (row->y + row->height > max_y)
	    row->visible_height -= row->y + row->height - max_y;
	  if (row->fringe_bitmap_periodic_p)
	    row->redraw_fringe_bitmaps_p = true;
	}

      /* Scroll the current matrix.  */
      eassert (nrows_scrolled > 0);
      rotate_matrix (w->current_matrix,
		     start_vpos,
		     MATRIX_ROW_VPOS (bottom_row, w->current_matrix),
		     -nrows_scrolled);

      /* Disable rows not reused.  */
      for (row -= nrows_scrolled; row < bottom_row; ++row)
	row->enabled_p = false;

      /* Point may have moved to a different line, so we cannot assume that
	 the previous cursor position is valid; locate the correct row.  */
      if (pt_row)
	{
	  for (row = MATRIX_ROW (w->current_matrix, w->cursor.vpos);
	       row < bottom_row
		 && PT >= MATRIX_ROW_END_CHARPOS (row)
		 && !row->ends_at_zv_p;
	       row++)
	    {
	      w->cursor.vpos++;
	      w->cursor.y = row->y;
	    }
	  if (row < bottom_row)
	    {
	      /* Can't simply scan the row for point with
		 bidi-reordered glyph rows.  Let set_cursor_from_row
		 figure out where to put the cursor, and if it fails,
		 give up.  */
	      if (!NILP (BVAR (XBUFFER (w->contents), bidi_display_reordering)))
		{
		  if (!set_cursor_from_row (w, row, w->current_matrix,
					    0, 0, 0, 0))
		    {
		      clear_glyph_matrix (w->desired_matrix);
		      return false;
		    }
		}
	      else
		{
		  struct glyph *glyph = row->glyphs[TEXT_AREA] + w->cursor.hpos;
		  struct glyph *end = row->glyphs[TEXT_AREA] + row->used[TEXT_AREA];

		  for (; glyph < end
			 && (!BUFFERP (glyph->object)
			     || glyph->charpos < PT);
		       glyph++)
		    {
		      w->cursor.hpos++;
		      w->cursor.x += glyph->pixel_width;
		    }
		}
	    }
	}

      /* Adjust window end.  A null value of last_text_row means that
	 the window end is in reused rows which in turn means that
	 only its vpos can have changed.  */
      if (last_text_row)
	adjust_window_ends (w, last_text_row, false);
      else
	w->window_end_vpos -= nrows_scrolled;

      w->window_end_valid = false;
      w->desired_matrix->no_scrolling_p = true;

#ifdef GLYPH_DEBUG
      debug_method_add (w, "try_window_reusing_current_matrix 2");
#endif
      return true;
    }

  return false;
}



/************************************************************************
   Window redisplay reusing current matrix when buffer has changed
 ************************************************************************/

static struct glyph_row *find_last_unchanged_at_beg_row (struct window *);
static struct glyph_row *find_first_unchanged_at_end_row (struct window *,
                                                          ptrdiff_t *, ptrdiff_t *);
static struct glyph_row *
find_last_row_displaying_text (struct glyph_matrix *, struct it *,
                               struct glyph_row *);


/* Return the last row in MATRIX displaying text.  If row START is
   non-null, start searching with that row.  IT gives the dimensions
   of the display.  Value is null if matrix is empty; otherwise it is
   a pointer to the row found.  */

static struct glyph_row *
find_last_row_displaying_text (struct glyph_matrix *matrix, struct it *it,
			       struct glyph_row *start)
{
  struct glyph_row *row, *row_found;

  /* Set row_found to the last row in IT->w's current matrix
     displaying text.  The loop looks funny but think of partially
     visible lines.  */
  row_found = NULL;
  row = start ? start : MATRIX_FIRST_TEXT_ROW (matrix);
  while (MATRIX_ROW_DISPLAYS_TEXT_P (row))
    {
      eassert (row->enabled_p);
      row_found = row;
      if (MATRIX_ROW_BOTTOM_Y (row) >= it->last_visible_y)
	break;
      ++row;
    }

  return row_found;
}


/* Return the last row in the current matrix of W that is not affected
   by changes at the start of current_buffer that occurred since W's
   current matrix was built.  Value is null if no such row exists.

   BEG_UNCHANGED is the number of characters unchanged at the start of
   current_buffer.  BEG + BEG_UNCHANGED is the buffer position of the
   first changed character in current_buffer.  Characters at positions <
   BEG + BEG_UNCHANGED are at the same buffer positions as they were
   when the current matrix was built.  */

static struct glyph_row *
find_last_unchanged_at_beg_row (struct window *w)
{
  ptrdiff_t first_changed_pos = BEG + BEG_UNCHANGED;
  struct glyph_row *row;
  struct glyph_row *row_found = NULL;
  int yb = window_text_bottom_y (w);

  /* Find the last row displaying unchanged text.  */
  for (row = MATRIX_FIRST_TEXT_ROW (w->current_matrix);
       MATRIX_ROW_DISPLAYS_TEXT_P (row)
	 && MATRIX_ROW_START_CHARPOS (row) < first_changed_pos;
       ++row)
    {
      if (/* If row ends before first_changed_pos, it is unchanged,
	     except in some case.  */
	  MATRIX_ROW_END_CHARPOS (row) <= first_changed_pos
	  /* When row ends in ZV and we write at ZV it is not
             unchanged.  */
	  && !row->ends_at_zv_p
	  /* When first_changed_pos is the end of a continued line,
	     row is not unchanged because it may be no longer
	     continued.  */
	  && !(MATRIX_ROW_END_CHARPOS (row) == first_changed_pos
	       && (row->continued_p
		   || row->exact_window_width_line_p))
	  /* If ROW->end is beyond ZV, then ROW->end is outdated and
	     needs to be recomputed, so don't consider this row as
	     unchanged.  This happens when the last line was
	     bidi-reordered and was killed immediately before this
	     redisplay cycle.  In that case, ROW->end stores the
	     buffer position of the first visual-order character of
	     the killed text, which is now beyond ZV.  */
	  && CHARPOS (row->end.pos) <= ZV)
	row_found = row;

      /* Stop if last visible row.  */
      if (MATRIX_ROW_BOTTOM_Y (row) >= yb)
	break;
    }

  return row_found;
}


/* Find the first glyph row in the current matrix of W that is not
   affected by changes at the end of current_buffer since the
   time W's current matrix was built.

   Return in *DELTA the number of chars by which buffer positions in
   unchanged text at the end of current_buffer must be adjusted.

   Return in *DELTA_BYTES the corresponding number of bytes.

   Value is null if no such row exists, i.e. all rows are affected by
   changes.  */

static struct glyph_row *
find_first_unchanged_at_end_row (struct window *w,
				 ptrdiff_t *delta, ptrdiff_t *delta_bytes)
{
  struct glyph_row *row;
  struct glyph_row *row_found = NULL;

  *delta = *delta_bytes = 0;

  /* Display must not have been paused, otherwise the current matrix
     is not up to date.  */
  eassert (w->window_end_valid);

  /* A value of window_end_pos >= END_UNCHANGED means that the window
     end is in the range of changed text.  If so, there is no
     unchanged row at the end of W's current matrix.  */
  if (w->window_end_pos >= END_UNCHANGED)
    return NULL;

  /* Set row to the last row in W's current matrix displaying text.  */
  row = MATRIX_ROW (w->current_matrix, w->window_end_vpos);

  /* If matrix is entirely empty, no unchanged row exists.  */
  if (MATRIX_ROW_DISPLAYS_TEXT_P (row))
    {
      /* The value of row is the last glyph row in the matrix having a
	 meaningful buffer position in it.  The end position of row
	 corresponds to window_end_pos.  This allows us to translate
	 buffer positions in the current matrix to current buffer
	 positions for characters not in changed text.  */
      ptrdiff_t Z_old =
	MATRIX_ROW_END_CHARPOS (row) + w->window_end_pos;
      ptrdiff_t Z_BYTE_old =
	MATRIX_ROW_END_BYTEPOS (row) + w->window_end_bytepos;
      ptrdiff_t last_unchanged_pos, last_unchanged_pos_old;
      struct glyph_row *first_text_row
	= MATRIX_FIRST_TEXT_ROW (w->current_matrix);

      *delta = Z - Z_old;
      *delta_bytes = Z_BYTE - Z_BYTE_old;

      /* Set last_unchanged_pos to the buffer position of the last
	 character in the buffer that has not been changed.  Z is the
	 index + 1 of the last character in current_buffer, i.e. by
	 subtracting END_UNCHANGED we get the index of the last
	 unchanged character, and we have to add BEG to get its buffer
	 position.  */
      last_unchanged_pos = Z - END_UNCHANGED + BEG;
      last_unchanged_pos_old = last_unchanged_pos - *delta;

      /* Search backward from ROW for a row displaying a line that
	 starts at a minimum position >= last_unchanged_pos_old.  */
      for (; row > first_text_row; --row)
	{
	  /* This used to abort, but it can happen.
	     It is ok to just stop the search instead here.  KFS.  */
	  if (!row->enabled_p || !MATRIX_ROW_DISPLAYS_TEXT_P (row))
	    break;

	  if (MATRIX_ROW_START_CHARPOS (row) >= last_unchanged_pos_old)
	    row_found = row;
	}
    }

  eassert (!row_found || MATRIX_ROW_DISPLAYS_TEXT_P (row_found));

  return row_found;
}


/* Make sure that glyph rows in the current matrix of window W
   reference the same glyph memory as corresponding rows in the
   frame's frame matrix.  This function is called after scrolling W's
   current matrix on a terminal frame in try_window_id and
   try_window_reusing_current_matrix.  */

static void
sync_frame_with_window_matrix_rows (struct window *w)
{
  struct frame *f = XFRAME (w->frame);
  struct glyph_row *window_row, *window_row_end, *frame_row;

  /* Preconditions: W must be a leaf window and full-width.  Its frame
     must have a frame matrix.  */
  eassert (BUFFERP (w->contents));
  eassert (WINDOW_FULL_WIDTH_P (w));
  eassert (!FRAME_WINDOW_P (f));

  /* If W is a full-width window, glyph pointers in W's current matrix
     have, by definition, to be the same as glyph pointers in the
     corresponding frame matrix.  Note that frame matrices have no
     marginal areas (see build_frame_matrix).  */
  window_row = w->current_matrix->rows;
  window_row_end = window_row + w->current_matrix->nrows;
  frame_row = f->current_matrix->rows + WINDOW_TOP_EDGE_LINE (w);
  while (window_row < window_row_end)
    {
      struct glyph *start = window_row->glyphs[LEFT_MARGIN_AREA];
      struct glyph *end = window_row->glyphs[LAST_AREA];

      frame_row->glyphs[LEFT_MARGIN_AREA] = start;
      frame_row->glyphs[TEXT_AREA] = start;
      frame_row->glyphs[RIGHT_MARGIN_AREA] = end;
      frame_row->glyphs[LAST_AREA] = end;

      /* Disable frame rows whose corresponding window rows have
	 been disabled in try_window_id.  */
      if (!window_row->enabled_p)
	frame_row->enabled_p = false;

      ++window_row, ++frame_row;
    }
}


/* Find the glyph row in window W containing CHARPOS.  Consider all
   rows between START and END (not inclusive).  END null means search
   all rows to the end of the display area of W.  Value is the row
   containing CHARPOS or null.  */

struct glyph_row *
row_containing_pos (struct window *w, ptrdiff_t charpos,
		    struct glyph_row *start, struct glyph_row *end, int dy)
{
  struct glyph_row *row = start;
  struct glyph_row *best_row = NULL;
  ptrdiff_t mindif = BUF_ZV (XBUFFER (w->contents)) + 1;
  int last_y;

  /* If we happen to start on a header-line or a tab-line, skip that.  */
  if (row->tab_line_p)
    ++row;
  if (row->mode_line_p)
    ++row;

  if ((end && row >= end) || !row->enabled_p)
    return NULL;

  last_y = window_text_bottom_y (w) - dy;

  while (true)
    {
      /* Give up if we have gone too far.  */
      if ((end && row >= end) || !row->enabled_p)
	return NULL;
      /* This formerly returned if they were equal.
	 I think that both quantities are of a "last plus one" type;
	 if so, when they are equal, the row is within the screen. -- rms.  */
      if (MATRIX_ROW_BOTTOM_Y (row) > last_y)
	return NULL;

      /* If it is in this row, return this row.  */
      if (! (MATRIX_ROW_END_CHARPOS (row) < charpos
	     || (MATRIX_ROW_END_CHARPOS (row) == charpos
		 /* The end position of a row equals the start
		    position of the next row.  If CHARPOS is there, we
		    would rather consider it displayed in the next
		    line, except when this line ends in ZV.  */
		 && !row_for_charpos_p (row, charpos)))
	  && charpos >= MATRIX_ROW_START_CHARPOS (row))
	{
	  struct glyph *g;

	  if (NILP (BVAR (XBUFFER (w->contents), bidi_display_reordering))
	      || (!best_row && !row->continued_p))
	    return row;
	  /* In bidi-reordered rows, there could be several rows whose
	     edges surround CHARPOS, all of these rows belonging to
	     the same continued line.  We need to find the row which
	     fits CHARPOS the best.  */
	  for (g = row->glyphs[TEXT_AREA];
	       g < row->glyphs[TEXT_AREA] + row->used[TEXT_AREA];
	       g++)
	    {
	      if (!STRINGP (g->object))
		{
		  if (g->charpos > 0 && eabs (g->charpos - charpos) < mindif)
		    {
		      mindif = eabs (g->charpos - charpos);
		      best_row = row;
		      /* Exact match always wins.  */
		      if (mindif == 0)
			return best_row;
		    }
		}
	    }
	}
      else if (best_row && !row->continued_p)
	return best_row;
      ++row;
    }
}


/* Try to redisplay window W by reusing its existing display.  W's
   current matrix must be up to date when this function is called,
   i.e., window_end_valid must be true.

   Value is

   >= 1	if successful, i.e. display has been updated
         specifically:
         1 means the changes were in front of a newline that precedes
           the window start, and the whole current matrix was reused
         2 means the changes were after the last position displayed
           in the window, and the whole current matrix was reused
         3 means portions of the current matrix were reused, while
           some of the screen lines were redrawn
   -1	if redisplay with same window start is known not to succeed
   0	if otherwise unsuccessful

   The following steps are performed:

   1. Find the last row in the current matrix of W that is not
   affected by changes at the start of current_buffer.  If no such row
   is found, give up.

   2. Find the first row in W's current matrix that is not affected by
   changes at the end of current_buffer.  Maybe there is no such row.

   3. Display lines beginning with the row + 1 found in step 1 to the
   row found in step 2 or, if step 2 didn't find a row, to the end of
   the window.

   4. If cursor is not known to appear on the window, give up.

   5. If display stopped at the row found in step 2, scroll the
   display and current matrix as needed.

   6. Maybe display some lines at the end of W, if we must.  This can
   happen under various circumstances, like a partially visible line
   becoming fully visible, or because newly displayed lines are displayed
   in smaller font sizes.

   7. Update W's window end information.  */

static int
try_window_id (struct window *w)
{
  struct frame *f = XFRAME (w->frame);
  struct glyph_matrix *current_matrix = w->current_matrix;
  struct glyph_matrix *desired_matrix = w->desired_matrix;
  struct glyph_row *last_unchanged_at_beg_row;
  struct glyph_row *first_unchanged_at_end_row;
  struct glyph_row *row;
  struct glyph_row *bottom_row;
  int bottom_vpos;
  struct it it;
  ptrdiff_t delta = 0, delta_bytes = 0, stop_pos;
  int dvpos, dy;
  struct text_pos start_pos;
  struct run run;
  int first_unchanged_at_end_vpos = 0;
  struct glyph_row *last_text_row, *last_text_row_at_end;
  struct text_pos start;
  ptrdiff_t first_changed_charpos, last_changed_charpos;

#ifdef GLYPH_DEBUG
  if (inhibit_try_window_id)
    return 0;
#endif

  /* This is handy for debugging.  */
#if false
#define GIVE_UP(X)						\
  do {								\
    redisplay_trace ("try_window_id give up %d\n", X);		\
    return 0;							\
  } while (false)
#else
#define GIVE_UP(X) return 0
#endif

  SET_TEXT_POS_FROM_MARKER (start, w->start);

  /* Don't use this for mini-windows because these can show
     messages and mini-buffers, and we don't handle that here.  */
  if (MINI_WINDOW_P (w))
    GIVE_UP (1);

  /* This flag is used to prevent redisplay optimizations.  */
  if (windows_or_buffers_changed || f->cursor_type_changed)
    GIVE_UP (2);

  /* This function's optimizations cannot be used if overlays have
     changed in the buffer displayed by the window, so give up if they
     have.  */
  if (w->last_overlay_modified != OVERLAY_MODIFF)
    GIVE_UP (200);

  /* Verify that narrowing has not changed.
     Also verify that we were not told to prevent redisplay optimizations.
     It would be nice to further
     reduce the number of cases where this prevents try_window_id.  */
  if (current_buffer->clip_changed
      || current_buffer->prevent_redisplay_optimizations_p)
    GIVE_UP (3);

  /* Window must either use window-based redisplay or be full width.  */
  if (!FRAME_WINDOW_P (f)
      && (!FRAME_LINE_INS_DEL_OK (f)
	  || !WINDOW_FULL_WIDTH_P (w)))
    GIVE_UP (4);

  /* Give up if point is known NOT to appear in W.  */
  if (PT < CHARPOS (start))
    GIVE_UP (5);

  /* Another way to prevent redisplay optimizations.  */
  if (w->last_modified == 0)
    GIVE_UP (6);

  /* Verify that window is not hscrolled.  */
  if (w->hscroll != 0)
    GIVE_UP (7);

  /* Verify that display wasn't paused.  */
  if (!w->window_end_valid)
    GIVE_UP (8);

  /* Likewise if highlighting trailing whitespace.  */
  if (!NILP (Vshow_trailing_whitespace))
    GIVE_UP (11);

  /* Can't use this if overlay arrow position and/or string have
     changed.  */
  if (overlay_arrows_changed_p (false))
    GIVE_UP (12);

  /* When word-wrap is on, adding a space to the first word of a
     wrapped line can change the wrap position, altering the line
     above it.  It might be worthwhile to handle this more
     intelligently, but for now just redisplay from scratch.  */
  if (!NILP (BVAR (XBUFFER (w->contents), word_wrap)))
    GIVE_UP (21);

  /* Under bidi reordering, adding or deleting a character in the
     beginning of a paragraph, before the first strong directional
     character, can change the base direction of the paragraph (unless
     the buffer specifies a fixed paragraph direction), which will
     require redisplaying the whole paragraph.  It might be worthwhile
     to find the paragraph limits and widen the range of redisplayed
     lines to that, but for now just give up this optimization and
     redisplay from scratch.  */
  if (!NILP (BVAR (XBUFFER (w->contents), bidi_display_reordering))
      && NILP (BVAR (XBUFFER (w->contents), bidi_paragraph_direction)))
    GIVE_UP (22);

  /* Give up if the buffer has line-spacing set, as Lisp-level changes
     to that variable require thorough redisplay.  */
  if (!NILP (BVAR (XBUFFER (w->contents), extra_line_spacing)))
    GIVE_UP (23);

  /* Give up if display-line-numbers is in relative mode, or when the
     current line's number needs to be displayed in a distinct face.  */
  if (EQ (Vdisplay_line_numbers, Qrelative)
      || EQ (Vdisplay_line_numbers, Qvisual)
      || (!NILP (Vdisplay_line_numbers)
	  && NILP (Finternal_lisp_face_equal_p (Qline_number,
						Qline_number_current_line,
						w->frame))))
    GIVE_UP (24);

  /* composition-break-at-point is incompatible with the optimizations
     in this function, because we need to recompose characters when
     point moves off their positions.  */
  if (composition_break_at_point)
    GIVE_UP (27);

  /* Make sure beg_unchanged and end_unchanged are up to date.  Do it
     only if buffer has really changed.  The reason is that the gap is
     initially at Z for freshly visited files.  The code below would
     set end_unchanged to 0 in that case.  */
  if (MODIFF > SAVE_MODIFF
      /* This seems to happen sometimes after saving a buffer.  */
      || BEG_UNCHANGED + END_UNCHANGED > Z_BYTE)
    {
      if (GPT - BEG < BEG_UNCHANGED)
	BEG_UNCHANGED = GPT - BEG;
      if (Z - GPT < END_UNCHANGED)
	END_UNCHANGED = Z - GPT;
    }

  /* The position of the first and last character that has been changed.  */
  first_changed_charpos = BEG + BEG_UNCHANGED;
  last_changed_charpos  = Z - END_UNCHANGED;

  /* If window starts after a line end, and the last change is in
     front of that newline, then changes don't affect the display.
     This case happens with stealth-fontification.  Note that although
     the display is unchanged, glyph positions in the matrix have to
     be adjusted, of course.  */
  row = MATRIX_ROW (w->current_matrix, w->window_end_vpos);
  if (MATRIX_ROW_DISPLAYS_TEXT_P (row)
      && ((last_changed_charpos < CHARPOS (start)
	   && CHARPOS (start) == BEGV)
	  || (last_changed_charpos < CHARPOS (start) - 1
	      && FETCH_BYTE (BYTEPOS (start) - 1) == '\n')))
    {
      ptrdiff_t Z_old, Z_delta, Z_BYTE_old, Z_delta_bytes;
      struct glyph_row *r0;

      /* Compute how many chars/bytes have been added to or removed
	 from the buffer.  */
      Z_old = MATRIX_ROW_END_CHARPOS (row) + w->window_end_pos;
      Z_BYTE_old = MATRIX_ROW_END_BYTEPOS (row) + w->window_end_bytepos;
      Z_delta = Z - Z_old;
      Z_delta_bytes = Z_BYTE - Z_BYTE_old;

      /* Give up if PT is not in the window.  Note that it already has
	 been checked at the start of try_window_id that PT is not in
	 front of the window start.  */
      if (PT >= MATRIX_ROW_END_CHARPOS (row) + Z_delta)
	GIVE_UP (13);

      /* If window start is unchanged, we can reuse the whole matrix
	 as is, after adjusting glyph positions.  No need to compute
	 the window end again, since its offset from Z hasn't changed.  */
      r0 = MATRIX_FIRST_TEXT_ROW (current_matrix);
      if (CHARPOS (start) == MATRIX_ROW_START_CHARPOS (r0) + Z_delta
	  && BYTEPOS (start) == MATRIX_ROW_START_BYTEPOS (r0) + Z_delta_bytes
	  /* PT must not be in a partially visible line.  */
	  && !(PT >= MATRIX_ROW_START_CHARPOS (row) + Z_delta
	       && MATRIX_ROW_BOTTOM_Y (row) > window_text_bottom_y (w)))
	{
	  /* Adjust positions in the glyph matrix.  */
	  if (Z_delta || Z_delta_bytes)
	    {
	      struct glyph_row *r1
		= MATRIX_BOTTOM_TEXT_ROW (current_matrix, w);
	      increment_matrix_positions (w->current_matrix,
					  MATRIX_ROW_VPOS (r0, current_matrix),
					  MATRIX_ROW_VPOS (r1, current_matrix),
					  Z_delta, Z_delta_bytes);
	    }

	  /* Set the cursor.  */
	  row = row_containing_pos (w, PT, r0, NULL, 0);
	  if (row)
	    set_cursor_from_row (w, row, current_matrix, 0, 0, 0, 0);
	  return 1;
	}
    }

  /* Handle the case that changes are all below what is displayed in
     the window, and that PT is in the window.  This shortcut cannot
     be taken if ZV is visible in the window, and text has been added
     there that is visible in the window.  */
  if (first_changed_charpos >= MATRIX_ROW_END_CHARPOS (row)
      /* ZV is not visible in the window, or there are no
	 changes at ZV, actually.  */
      && (current_matrix->zv > MATRIX_ROW_END_CHARPOS (row)
	  || first_changed_charpos == last_changed_charpos))
    {
      struct glyph_row *r0;

      /* Give up if PT is not in the window.  Note that it already has
	 been checked at the start of try_window_id that PT is not in
	 front of the window start.  */
      if (PT >= MATRIX_ROW_END_CHARPOS (row))
	GIVE_UP (14);

      /* If window start is unchanged, we can reuse the whole matrix
	 as is, without changing glyph positions since no text has
	 been added/removed in front of the window end.  */
      r0 = MATRIX_FIRST_TEXT_ROW (current_matrix);
      if (TEXT_POS_EQUAL_P (start, r0->minpos)
	  /* PT must not be in a partially visible line.  */
	  && !(PT >= MATRIX_ROW_START_CHARPOS (row)
	       && MATRIX_ROW_BOTTOM_Y (row) > window_text_bottom_y (w)))
	{
	  /* We have to compute the window end anew since text
	     could have been added/removed after it.  */
	  w->window_end_pos = Z - MATRIX_ROW_END_CHARPOS (row);
	  w->window_end_bytepos = Z_BYTE - MATRIX_ROW_END_BYTEPOS (row);

	  /* Set the cursor.  */
	  row = row_containing_pos (w, PT, r0, NULL, 0);
	  if (row)
	    set_cursor_from_row (w, row, current_matrix, 0, 0, 0, 0);
	  return 2;
	}
    }

  /* Give up if window start is in the changed area.

     The condition used to read

     (BEG_UNCHANGED + END_UNCHANGED != Z - BEG && ...)

     but why that was tested escapes me at the moment.  */
  if (CHARPOS (start) >= first_changed_charpos
      && CHARPOS (start) <= last_changed_charpos)
    GIVE_UP (15);

  /* Check that window start agrees with the start of the first glyph
     row in its current matrix.  Check this after we know the window
     start is not in changed text, otherwise positions would not be
     comparable.  */
  row = MATRIX_FIRST_TEXT_ROW (current_matrix);
  if (!TEXT_POS_EQUAL_P (start, row->minpos))
    GIVE_UP (16);

  /* Give up if the window ends in strings.  Overlay strings
     at the end are difficult to handle, so don't try.  */
  row = MATRIX_ROW (current_matrix, w->window_end_vpos);
  if (MATRIX_ROW_START_CHARPOS (row) == MATRIX_ROW_END_CHARPOS (row))
    GIVE_UP (20);

  /* Can't let scroll_run_hook below run on w32 GUI frames when
     position of point is indicated by the system caret, because
     scrolling the display will then "copy" the pixels used by the
     caret.  */
#ifdef HAVE_NTGUI
  if (FRAME_W32_P (f) && w32_use_visible_system_caret)
    GIVE_UP (25);
#endif

  /* Compute the position at which we have to start displaying new
     lines.  Some of the lines at the top of the window might be
     reusable because they are not displaying changed text.  Find the
     last row in W's current matrix not affected by changes at the
     start of current_buffer.  Value is null if changes start in the
     first line of window.  */
  last_unchanged_at_beg_row = find_last_unchanged_at_beg_row (w);
  if (last_unchanged_at_beg_row)
    {
      /* Avoid starting to display in the middle of a character, a TAB
	 for instance.  This is easier than to set up the iterator
	 exactly, and it's not a frequent case, so the additional
	 effort wouldn't really pay off.  */
      while ((MATRIX_ROW_ENDS_IN_MIDDLE_OF_CHAR_P (last_unchanged_at_beg_row)
	      || last_unchanged_at_beg_row->ends_in_newline_from_string_p)
	     && last_unchanged_at_beg_row > w->current_matrix->rows)
	--last_unchanged_at_beg_row;

      if (MATRIX_ROW_ENDS_IN_MIDDLE_OF_CHAR_P (last_unchanged_at_beg_row))
	GIVE_UP (17);

      if (! init_to_row_end (&it, w, last_unchanged_at_beg_row))
	GIVE_UP (18);
      /* Give up if the row starts with a display property that draws
	 on the fringes, since that could prevent correct display of
	 line-prefix and wrap-prefix.  */
      if (it.sp > 1
	  && it.method == GET_FROM_IMAGE && it.image_id == -1)
	GIVE_UP (26);
      start_pos = it.current.pos;

      /* Start displaying new lines in the desired matrix at the same
	 vpos we would use in the current matrix, i.e. below
	 last_unchanged_at_beg_row.  */
      it.vpos = 1 + MATRIX_ROW_VPOS (last_unchanged_at_beg_row,
				     current_matrix);
      it.glyph_row = MATRIX_ROW (desired_matrix, it.vpos);
      it.current_y = MATRIX_ROW_BOTTOM_Y (last_unchanged_at_beg_row);

      eassert (it.hpos == 0 && it.current_x == 0);
    }
  else
    {
      /* There are no reusable lines at the start of the window.
	 Start displaying in the first text line.  */
      start_display (&it, w, start);
      it.vpos = it.first_vpos;
      start_pos = it.current.pos;
    }

  /* Find the first row that is not affected by changes at the end of
     the buffer.  Value will be null if there is no unchanged row, in
     which case we must redisplay to the end of the window.  delta
     will be set to the value by which buffer positions beginning with
     first_unchanged_at_end_row have to be adjusted due to text
     changes.  */
  first_unchanged_at_end_row
    = find_first_unchanged_at_end_row (w, &delta, &delta_bytes);
  IF_DEBUG (debug_delta = delta);
  IF_DEBUG (debug_delta_bytes = delta_bytes);

  /* Set stop_pos to the buffer position up to which we will have to
     display new lines.  If first_unchanged_at_end_row != NULL, this
     is the buffer position of the start of the line displayed in that
     row.  For first_unchanged_at_end_row == NULL, use 0 to indicate
     that we don't stop at a buffer position.  */
  stop_pos = 0;
  if (first_unchanged_at_end_row)
    {
      eassert (last_unchanged_at_beg_row == NULL
	       || first_unchanged_at_end_row >= last_unchanged_at_beg_row);

      /* If this is a continuation line, move forward to the next one
	 that isn't.  Changes in lines above affect this line.
	 Caution: this may move first_unchanged_at_end_row to a row
	 not displaying text.  */
      while (MATRIX_ROW_CONTINUATION_LINE_P (first_unchanged_at_end_row)
	     && MATRIX_ROW_DISPLAYS_TEXT_P (first_unchanged_at_end_row)
	     && (MATRIX_ROW_BOTTOM_Y (first_unchanged_at_end_row)
		 < it.last_visible_y))
	++first_unchanged_at_end_row;

      if (!MATRIX_ROW_DISPLAYS_TEXT_P (first_unchanged_at_end_row)
	  || (MATRIX_ROW_BOTTOM_Y (first_unchanged_at_end_row)
	      >= it.last_visible_y))
	first_unchanged_at_end_row = NULL;
      else
	{
	  stop_pos = (MATRIX_ROW_START_CHARPOS (first_unchanged_at_end_row)
		      + delta);
	  first_unchanged_at_end_vpos
	    = MATRIX_ROW_VPOS (first_unchanged_at_end_row, current_matrix);
	  eassert (stop_pos >= Z - END_UNCHANGED);
	}
    }
  else if (last_unchanged_at_beg_row == NULL)
    GIVE_UP (19);


#ifdef GLYPH_DEBUG

  /* Either there is no unchanged row at the end, or the one we have
     now displays text.  This is a necessary condition for the window
     end pos calculation at the end of this function.  */
  eassert (first_unchanged_at_end_row == NULL
	   || MATRIX_ROW_DISPLAYS_TEXT_P (first_unchanged_at_end_row));

  debug_last_unchanged_at_beg_vpos
    = (last_unchanged_at_beg_row
       ? MATRIX_ROW_VPOS (last_unchanged_at_beg_row, current_matrix)
       : -1);
  debug_first_unchanged_at_end_vpos = first_unchanged_at_end_vpos;

#endif /* GLYPH_DEBUG */


  /* Display new lines.  Set last_text_row to the last new line
     displayed which has text on it, i.e. might end up as being the
     line where the window_end_vpos is.  */
  w->cursor.vpos = -1;
  last_text_row = NULL;
  overlay_arrow_seen = false;
  if (it.current_y < it.last_visible_y
      && !f->fonts_changed
      && (first_unchanged_at_end_row == NULL
	  || IT_CHARPOS (it) < stop_pos))
    it.glyph_row->reversed_p = false;
  while (it.current_y < it.last_visible_y
	 && !f->fonts_changed
	 && (first_unchanged_at_end_row == NULL
	     || IT_CHARPOS (it) < stop_pos))
    {
      if (display_line (&it, -1))
	last_text_row = it.glyph_row - 1;
    }

  if (f->fonts_changed)
    return -1;

  /* The redisplay iterations in display_line above could have
     triggered font-lock, which could have done something that
     invalidates IT->w window's end-point information, on which we
     rely below.  E.g., one package, which will remain unnamed, used
     to install a font-lock-fontify-region-function that called
     bury-buffer, whose side effect is to switch the buffer displayed
     by IT->w, and that predictably resets IT->w's window_end_valid
     flag, which we already tested at the entry to this function.
     Amply punish such packages/modes by giving up on this
     optimization in those cases.  */
  if (!w->window_end_valid)
    {
      clear_glyph_matrix (w->desired_matrix);
      return -1;
    }

  /* Compute differences in buffer positions, y-positions etc.  for
     lines reused at the bottom of the window.  Compute what we can
     scroll.  */
  if (first_unchanged_at_end_row
      /* No lines reused because we displayed everything up to the
         bottom of the window.  */
      && it.current_y < it.last_visible_y)
    {
      dvpos = (it.vpos
	       - MATRIX_ROW_VPOS (first_unchanged_at_end_row,
				  current_matrix));
      dy = it.current_y - first_unchanged_at_end_row->y;
      run.current_y = first_unchanged_at_end_row->y;
      run.desired_y = run.current_y + dy;
      run.height = it.last_visible_y - max (run.current_y, run.desired_y);
    }
  else
    {
      delta = delta_bytes = dvpos = dy
	= run.current_y = run.desired_y = run.height = 0;
      first_unchanged_at_end_row = NULL;
    }
  IF_DEBUG ((debug_dvpos = dvpos, debug_dy = dy));


  /* Find the cursor if not already found.  We have to decide whether
     PT will appear on this window (it sometimes doesn't, but this is
     not a very frequent case.)  This decision has to be made before
     the current matrix is altered.  A value of cursor.vpos < 0 means
     that PT is either in one of the lines beginning at
     first_unchanged_at_end_row or below the window.  Don't care for
     lines that might be displayed later at the window end; as
     mentioned, this is not a frequent case.  */
  if (w->cursor.vpos < 0)
    {
      /* Cursor in unchanged rows at the top?  */
      if (PT < CHARPOS (start_pos)
	  && last_unchanged_at_beg_row)
	{
	  row = row_containing_pos (w, PT,
				    MATRIX_FIRST_TEXT_ROW (w->current_matrix),
				    last_unchanged_at_beg_row + 1, 0);
	  if (row)
	    set_cursor_from_row (w, row, w->current_matrix, 0, 0, 0, 0);
	}

      /* Start from first_unchanged_at_end_row looking for PT.  */
      else if (first_unchanged_at_end_row)
	{
	  row = row_containing_pos (w, PT - delta,
				    first_unchanged_at_end_row, NULL, 0);
	  if (row)
	    set_cursor_from_row (w, row, w->current_matrix, delta,
				 delta_bytes, dy, dvpos);
	}

      /* Give up if cursor was not found.  */
      if (w->cursor.vpos < 0)
	{
	  clear_glyph_matrix (w->desired_matrix);
	  return -1;
	}
    }

  /* Don't let the cursor end in the scroll margins.  */
  {
    int this_scroll_margin = window_scroll_margin (w, MARGIN_IN_PIXELS);
    int cursor_height = MATRIX_ROW (w->desired_matrix, w->cursor.vpos)->height;

    if ((w->cursor.y < this_scroll_margin
	 && CHARPOS (start) > BEGV)
	/* Old redisplay didn't take scroll margin into account at the bottom,
	   but then global-hl-line-mode doesn't scroll.  KFS 2004-06-14 */
	|| (w->cursor.y
	    + (cursor_row_fully_visible_p (w, false, true, true)
	       ? 1
	       : cursor_height + this_scroll_margin)) > it.last_visible_y)
      {
	w->cursor.vpos = -1;
	clear_glyph_matrix (w->desired_matrix);
	return -1;
      }
  }

  /* Scroll the display.  Do it before changing the current matrix so
     that xterm.c doesn't get confused about where the cursor glyph is
     found.  */
  if (dy && run.height)
    {
      update_begin (f);

      if (FRAME_WINDOW_P (f))
	{
#ifdef HAVE_WINDOW_SYSTEM
	  gui_update_window_begin (w);
	  FRAME_RIF (f)->clear_window_mouse_face (w);
	  FRAME_RIF (f)->scroll_run_hook (w, &run);
	  gui_update_window_end (w, false, false);
#endif
	}
      else
	{
	  /* Terminal frame.  In this case, dvpos gives the number of
	     lines to scroll by; dvpos < 0 means scroll up.  */
	  int from_vpos
	    = MATRIX_ROW_VPOS (first_unchanged_at_end_row, w->current_matrix);
	  int from = WINDOW_TOP_EDGE_LINE (w) + from_vpos;
	  int end = (WINDOW_TOP_EDGE_LINE (w)
		     + window_wants_tab_line (w)
		     + window_wants_header_line (w)
		     + window_internal_height (w));

	  gui_clear_window_mouse_face (w);

	  /* Perform the operation on the screen.  */
	  if (dvpos > 0)
	    {
	      /* Scroll last_unchanged_at_beg_row to the end of the
		 window down dvpos lines.  */
	      set_terminal_window (f, end);

	      /* On dumb terminals delete dvpos lines at the end
		 before inserting dvpos empty lines.  */
	      if (!FRAME_SCROLL_REGION_OK (f))
		ins_del_lines (f, end - dvpos, -dvpos);

	      /* Insert dvpos empty lines in front of
                 last_unchanged_at_beg_row.  */
	      ins_del_lines (f, from, dvpos);
	    }
	  else if (dvpos < 0)
	    {
	      /* Scroll up last_unchanged_at_beg_vpos to the end of
		 the window to last_unchanged_at_beg_vpos - |dvpos|.  */
	      set_terminal_window (f, end);

	      /* Delete dvpos lines in front of
		 last_unchanged_at_beg_vpos.  ins_del_lines will set
		 the cursor to the given vpos and emit |dvpos| delete
		 line sequences.  */
	      ins_del_lines (f, from + dvpos, dvpos);

	      /* On a dumb terminal insert dvpos empty lines at the
                 end.  */
	      if (!FRAME_SCROLL_REGION_OK (f))
		ins_del_lines (f, end + dvpos, -dvpos);
	    }

	  set_terminal_window (f, 0);
	}

      update_end (f);
    }

  /* Shift reused rows of the current matrix to the right position.
     BOTTOM_ROW is the last + 1 row in the current matrix reserved for
     text.  */
  bottom_row = MATRIX_BOTTOM_TEXT_ROW (current_matrix, w);
  bottom_vpos = MATRIX_ROW_VPOS (bottom_row, current_matrix);
  if (dvpos < 0)
    {
      rotate_matrix (current_matrix, first_unchanged_at_end_vpos + dvpos,
		     bottom_vpos, dvpos);
      clear_glyph_matrix_rows (current_matrix, bottom_vpos + dvpos,
			       bottom_vpos);
    }
  else if (dvpos > 0)
    {
      rotate_matrix (current_matrix, first_unchanged_at_end_vpos,
		     bottom_vpos, dvpos);
      clear_glyph_matrix_rows (current_matrix, first_unchanged_at_end_vpos,
			       first_unchanged_at_end_vpos + dvpos);
    }

  /* For frame-based redisplay, make sure that current frame and window
     matrix are in sync with respect to glyph memory.  */
  if (!FRAME_WINDOW_P (f))
    sync_frame_with_window_matrix_rows (w);

  /* Adjust buffer positions in reused rows.  */
  if (delta || delta_bytes)
    increment_matrix_positions (current_matrix,
				first_unchanged_at_end_vpos + dvpos,
				bottom_vpos, delta, delta_bytes);

  /* Adjust Y positions.  */
  if (dy)
    shift_glyph_matrix (w, current_matrix,
			first_unchanged_at_end_vpos + dvpos,
			bottom_vpos, dy);

  if (first_unchanged_at_end_row)
    {
      first_unchanged_at_end_row += dvpos;
      if (first_unchanged_at_end_row->y >= it.last_visible_y
	  || !MATRIX_ROW_DISPLAYS_TEXT_P (first_unchanged_at_end_row))
	first_unchanged_at_end_row = NULL;
    }

  /* If scrolling up, there may be some lines to display at the end of
     the window.  */
  last_text_row_at_end = NULL;
  if (dy < 0)
    {
      /* Scrolling up can leave for example a partially visible line
	 at the end of the window to be redisplayed.  */
      /* Set last_row to the glyph row in the current matrix where the
	 window end line is found.  It has been moved up or down in
	 the matrix by dvpos.  */
      int last_vpos = w->window_end_vpos + dvpos;
      struct glyph_row *last_row = MATRIX_ROW (current_matrix, last_vpos);

      /* If last_row is the window end line, it should display text.  */
      eassert (MATRIX_ROW_DISPLAYS_TEXT_P (last_row));

      /* If window end line was partially visible before, begin
	 displaying at that line.  Otherwise begin displaying with the
	 line following it.  */
      if (MATRIX_ROW_BOTTOM_Y (last_row) - dy >= it.last_visible_y)
	{
	  init_to_row_start (&it, w, last_row);
	  it.vpos = last_vpos;
	  it.current_y = last_row->y;
	}
      else
	{
	  init_to_row_end (&it, w, last_row);
	  it.vpos = 1 + last_vpos;
	  it.current_y = MATRIX_ROW_BOTTOM_Y (last_row);
	  ++last_row;
	}

      /* We may start in a continuation line.  If so, we have to
	 get the right continuation_lines_width and current_x.  */
      it.continuation_lines_width = last_row->continuation_lines_width;
      it.hpos = it.current_x = 0;

      /* Display the rest of the lines at the window end.  */
      it.glyph_row = MATRIX_ROW (desired_matrix, it.vpos);
      while (it.current_y < it.last_visible_y && !f->fonts_changed)
	{
	  /* Is it always sure that the display agrees with lines in
	     the current matrix?  I don't think so, so we mark rows
	     displayed invalid in the current matrix by setting their
	     enabled_p flag to false.  */
	  SET_MATRIX_ROW_ENABLED_P (w->current_matrix, it.vpos, false);
	  if (display_line (&it, w->cursor.vpos))
	    last_text_row_at_end = it.glyph_row - 1;
	}
    }

  /* Update window_end_pos and window_end_vpos.  */
  if (first_unchanged_at_end_row && !last_text_row_at_end)
    {
      /* Window end line if one of the preserved rows from the current
	 matrix.  Set row to the last row displaying text in current
	 matrix starting at first_unchanged_at_end_row, after
	 scrolling.  */
      eassert (MATRIX_ROW_DISPLAYS_TEXT_P (first_unchanged_at_end_row));
      row = find_last_row_displaying_text (w->current_matrix, &it,
					   first_unchanged_at_end_row);
      eassume (row && MATRIX_ROW_DISPLAYS_TEXT_P (row));
      adjust_window_ends (w, row, true);
      eassert (w->window_end_bytepos >= 0);
      IF_DEBUG (debug_method_add (w, "A"));
    }
  else if (last_text_row_at_end)
    {
      adjust_window_ends (w, last_text_row_at_end, false);
      eassert (w->window_end_bytepos >= 0);
      IF_DEBUG (debug_method_add (w, "B"));
    }
  else if (last_text_row)
    {
      /* We have displayed either to the end of the window or at the
	 end of the window, i.e. the last row with text is to be found
	 in the desired matrix.  */
      adjust_window_ends (w, last_text_row, false);
      eassert (w->window_end_bytepos >= 0);
    }
  else if (first_unchanged_at_end_row == NULL
	   && last_text_row == NULL
	   && last_text_row_at_end == NULL)
    {
      /* Displayed to end of window, but no line containing text was
	 displayed.  Lines were deleted at the end of the window.  */
      int first_vpos = window_wants_tab_line (w) + window_wants_header_line (w);
      int vpos = w->window_end_vpos;
      struct glyph_row *current_row = current_matrix->rows + vpos;
      struct glyph_row *desired_row = desired_matrix->rows + vpos;

      for (row = NULL; !row; --vpos, --current_row, --desired_row)
	{
	  eassert (first_vpos <= vpos);
	  if (desired_row->enabled_p)
	    {
	      if (MATRIX_ROW_DISPLAYS_TEXT_P (desired_row))
		row = desired_row;
	    }
	  else if (MATRIX_ROW_DISPLAYS_TEXT_P (current_row))
	    row  = current_row;
	}

      w->window_end_vpos = vpos + 1;
      w->window_end_pos = Z - MATRIX_ROW_END_CHARPOS (row);
      w->window_end_bytepos = Z_BYTE - MATRIX_ROW_END_BYTEPOS (row);
      eassert (w->window_end_bytepos >= 0);
      IF_DEBUG (debug_method_add (w, "C"));
    }
  else
    emacs_abort ();

  IF_DEBUG ((debug_end_pos = w->window_end_pos,
	     debug_end_vpos = w->window_end_vpos));

  /* Record that display has not been completed.  */
  w->window_end_valid = false;
  w->desired_matrix->no_scrolling_p = true;
  return 3;

#undef GIVE_UP
}



/***********************************************************************
			More debugging support
 ***********************************************************************/

#ifdef GLYPH_DEBUG

void dump_glyph_row (struct glyph_row *, int, int) EXTERNALLY_VISIBLE;
void dump_glyph_matrix (struct glyph_matrix *, int) EXTERNALLY_VISIBLE;
void dump_glyph (struct glyph_row *, struct glyph *, int) EXTERNALLY_VISIBLE;


/* Dump the contents of glyph matrix MATRIX on stderr.

   GLYPHS 0 means don't show glyph contents.
   GLYPHS 1 means show glyphs in short form
   GLYPHS > 1 means show glyphs in long form.  */

void
dump_glyph_matrix (struct glyph_matrix *matrix, int glyphs)
{
  int i;
  for (i = 0; i < matrix->nrows; ++i)
    dump_glyph_row (MATRIX_ROW (matrix, i), i, glyphs);
}


/* Dump contents of glyph GLYPH to stderr.  ROW and AREA are
   the glyph row and area where the glyph comes from.  */

void
dump_glyph (struct glyph_row *row, struct glyph *glyph, int area)
{
  if (glyph->type == CHAR_GLYPH
      || glyph->type == GLYPHLESS_GLYPH)
    {
      fprintf (stderr,
	       "  %5"pD"d     %c %9"pD"d   %c %3d 0x%06x      %c %4d %1.1d%1.1d\n",
	       glyph - row->glyphs[TEXT_AREA],
	       (glyph->type == CHAR_GLYPH
		? 'C'
		: 'G'),
	       glyph->charpos,
	       (BUFFERP (glyph->object)
		? 'B'
		: (STRINGP (glyph->object)
		   ? 'S'
		   : (NILP (glyph->object)
		      ? '0'
		      : '-'))),
	       glyph->pixel_width,
	       glyph->u.ch,
	       (glyph->u.ch < 0x80 && glyph->u.ch >= ' '
		? (int) glyph->u.ch
		: '.'),
	       glyph->face_id,
	       glyph->left_box_line_p,
	       glyph->right_box_line_p);
    }
  else if (glyph->type == STRETCH_GLYPH)
    {
      fprintf (stderr,
	       "  %5"pD"d     %c %9"pD"d   %c %3d 0x%06x      %c %4d %1.1d%1.1d\n",
	       glyph - row->glyphs[TEXT_AREA],
	       'S',
	       glyph->charpos,
	       (BUFFERP (glyph->object)
		? 'B'
		: (STRINGP (glyph->object)
		   ? 'S'
		   : (NILP (glyph->object)
		      ? '0'
		      : '-'))),
	       glyph->pixel_width,
	       0u,
	       ' ',
	       glyph->face_id,
	       glyph->left_box_line_p,
	       glyph->right_box_line_p);
    }
  else if (glyph->type == IMAGE_GLYPH)
    {
      fprintf (stderr,
	       "  %5"pD"d     %c %9"pD"d   %c %3d 0x%06x      %c %4d %1.1d%1.1d\n",
	       glyph - row->glyphs[TEXT_AREA],
	       'I',
	       glyph->charpos,
	       (BUFFERP (glyph->object)
		? 'B'
		: (STRINGP (glyph->object)
		   ? 'S'
		   : (NILP (glyph->object)
		      ? '0'
		      : '-'))),
	       glyph->pixel_width,
	       (unsigned int) glyph->u.img_id,
	       '.',
	       glyph->face_id,
	       glyph->left_box_line_p,
	       glyph->right_box_line_p);
    }
  else if (glyph->type == COMPOSITE_GLYPH)
    {
      fprintf (stderr,
	       "  %5"pD"d     %c %9"pD"d   %c %3d 0x%06x",
	       glyph - row->glyphs[TEXT_AREA],
	       '+',
	       glyph->charpos,
	       (BUFFERP (glyph->object)
		? 'B'
		: (STRINGP (glyph->object)
		   ? 'S'
		   : (NILP (glyph->object)
		      ? '0'
		      : '-'))),
	       glyph->pixel_width,
	       (unsigned int) glyph->u.cmp.id);
      if (glyph->u.cmp.automatic)
	fprintf (stderr,
		 "[%d-%d]",
		 glyph->slice.cmp.from, glyph->slice.cmp.to);
      fprintf (stderr, " . %4d %1.1d%1.1d\n",
	       glyph->face_id,
	       glyph->left_box_line_p,
	       glyph->right_box_line_p);
    }
  else if (glyph->type == XWIDGET_GLYPH)
    {
#ifndef HAVE_XWIDGETS
      eassume (false);
#else
      fprintf (stderr,
	       "  %5td %4c %6td %c %3d %7p %c %4d %1.1d%1.1d\n",
	       glyph - row->glyphs[TEXT_AREA],
	       'X',
	       glyph->charpos,
	       (BUFFERP (glyph->object)
		? 'B'
		: (STRINGP (glyph->object)
		   ? 'S'
		   : '-')),
	       glyph->pixel_width,
	       glyph->u.xwidget,
	       '.',
	       glyph->face_id,
	       glyph->left_box_line_p,
	       glyph->right_box_line_p);
#endif
    }
}


/* Dump the contents of glyph row at VPOS in MATRIX to stderr.
   GLYPHS 0 means don't show glyph contents.
   GLYPHS 1 means show glyphs in short form
   GLYPHS > 1 means show glyphs in long form.  */

void
dump_glyph_row (struct glyph_row *row, int vpos, int glyphs)
{
  if (glyphs != 1)
    {
      fputs (("Row     Start       End Used oE><\\CTZFesm     X    Y    W    H    V    A    P\n"
	      "==============================================================================\n"),
	     stderr);
      fprintf (stderr, "%3d %9"pD"d %9"pD"d %4d %1.1d%1.1d%1.1d%1.1d\
%1.1d%1.1d%1.1d%1.1d%1.1d%1.1d%1.1d%1.1d  %4d %4d %4d %4d %4d %4d %4d\n",
	       vpos,
	       MATRIX_ROW_START_CHARPOS (row),
	       MATRIX_ROW_END_CHARPOS (row),
	       row->used[TEXT_AREA],
	       row->contains_overlapping_glyphs_p,
	       row->enabled_p,
	       row->truncated_on_left_p,
	       row->truncated_on_right_p,
	       row->continued_p,
	       MATRIX_ROW_CONTINUATION_LINE_P (row),
	       MATRIX_ROW_DISPLAYS_TEXT_P (row),
	       row->ends_at_zv_p,
	       row->fill_line_p,
	       row->ends_in_middle_of_char_p,
	       row->starts_in_middle_of_char_p,
	       row->mouse_face_p,
	       row->x,
	       row->y,
	       row->pixel_width,
	       row->height,
	       row->visible_height,
	       row->ascent,
	       row->phys_ascent);
      /* The next 3 lines should align to "Start" in the header.  */
      fprintf (stderr, "    %9"pD"d %9"pD"d\t%5d\n", row->start.overlay_string_index,
	       row->end.overlay_string_index,
	       row->continuation_lines_width);
      fprintf (stderr, "    %9"pD"d %9"pD"d\n",
	       CHARPOS (row->start.string_pos),
	       CHARPOS (row->end.string_pos));
      fprintf (stderr, "    %9d %9d\n", row->start.dpvec_index,
	       row->end.dpvec_index);
    }

  if (glyphs > 1)
    {
      int area;

      for (area = LEFT_MARGIN_AREA; area < LAST_AREA; ++area)
	{
	  struct glyph *glyph = row->glyphs[area];
	  struct glyph *glyph_end = glyph + row->used[area];

	  /* Glyph for a line end in text.  */
	  if (area == TEXT_AREA && glyph == glyph_end && glyph->charpos > 0)
	    ++glyph_end;

	  if (glyph < glyph_end)
	    fputs (" Glyph#  Type       Pos   O   W     Code      C Face LR\n",
		   stderr);

	  for (; glyph < glyph_end; ++glyph)
	    dump_glyph (row, glyph, area);
	}
    }
  else if (glyphs == 1)
    {
      int area;
      char s[SHRT_MAX + 4];

      for (area = LEFT_MARGIN_AREA; area < LAST_AREA; ++area)
	{
	  int i;

	  for (i = 0; i < row->used[area]; ++i)
	    {
	      struct glyph *glyph = row->glyphs[area] + i;
	      if (i == row->used[area] - 1
		  && area == TEXT_AREA
		  && NILP (glyph->object)
		  && glyph->type == CHAR_GLYPH
		  && glyph->u.ch == ' ')
		{
		  strcpy (&s[i], "[\\n]");
		  i += 4;
		}
	      else if (glyph->type == CHAR_GLYPH
		       && glyph->u.ch < 0x80
		       && glyph->u.ch >= ' ')
		s[i] = glyph->u.ch;
	      else
		s[i] = '.';
	    }

	  s[i] = '\0';
	  fprintf (stderr, "%3d: (%d) '%s'\n", vpos, row->enabled_p, s);
	}
    }
}


DEFUN ("dump-glyph-matrix", Fdump_glyph_matrix,
       Sdump_glyph_matrix, 0, 1, "p",
       doc: /* Dump the current matrix of the selected window to stderr.
Shows contents of glyph row structures.  With non-nil
parameter GLYPHS, dump glyphs as well.  If GLYPHS is 1 show
glyphs in short form, otherwise show glyphs in long form.

Interactively, no argument means show glyphs in short form;
with numeric argument, its value is passed as the GLYPHS flag.  */)
  (Lisp_Object glyphs)
{
  struct window *w = XWINDOW (selected_window);
  struct buffer *buffer = XBUFFER (w->contents);

  fprintf (stderr, "PT = %"pD"d, BEGV = %"pD"d. ZV = %"pD"d\n",
	   BUF_PT (buffer), BUF_BEGV (buffer), BUF_ZV (buffer));
  fprintf (stderr, "Cursor x = %d, y = %d, hpos = %d, vpos = %d\n",
	   w->cursor.x, w->cursor.y, w->cursor.hpos, w->cursor.vpos);
  fputs ("=============================================\n", stderr);
  dump_glyph_matrix (w->current_matrix,
		     TYPE_RANGED_FIXNUMP (int, glyphs) ? XFIXNUM (glyphs) : 0);
  return Qnil;
}


DEFUN ("dump-frame-glyph-matrix", Fdump_frame_glyph_matrix,
       Sdump_frame_glyph_matrix, 0, 0, "", doc: /* Dump the current glyph matrix of the selected frame to stderr.
Only text-mode frames have frame glyph matrices.  */)
  (void)
{
  struct frame *f = XFRAME (selected_frame);

  if (f->current_matrix)
    dump_glyph_matrix (f->current_matrix, 1);
  else
    fputs ("*** This frame doesn't have a frame glyph matrix ***\n", stderr);
  return Qnil;
}


DEFUN ("dump-glyph-row", Fdump_glyph_row, Sdump_glyph_row, 1, 2, "P",
       doc: /* Dump glyph row ROW to stderr.
Interactively, ROW is the prefix numeric argument and defaults to
the row which displays point.
Optional argument GLYPHS 0 means don't dump glyphs.
GLYPHS 1 means dump glyphs in short form.
GLYPHS > 1 or omitted means dump glyphs in long form.  */)
  (Lisp_Object row, Lisp_Object glyphs)
{
  struct glyph_matrix *matrix;
  EMACS_INT vpos;

  if (NILP (row))
    {
      int d1, d2, d3, d4, d5, ypos;
      bool visible_p = pos_visible_p (XWINDOW (selected_window), PT,
				      &d1, &d2, &d3, &d4, &d5, &ypos);
      if (visible_p)
	vpos = ypos;
      else
	vpos = 0;
    }
  else
    {
      CHECK_FIXNUM (row);
      vpos = XFIXNUM (row);
    }
  matrix = XWINDOW (selected_window)->current_matrix;
  if (vpos >= 0 && vpos < matrix->nrows)
    dump_glyph_row (MATRIX_ROW (matrix, vpos),
		    vpos,
		    TYPE_RANGED_FIXNUMP (int, glyphs) ? XFIXNUM (glyphs) : 2);
  return Qnil;
}


DEFUN ("dump-tab-bar-row", Fdump_tab_bar_row, Sdump_tab_bar_row, 1, 2, "P",
       doc: /* Dump glyph row ROW of the tab-bar of the current frame to stderr.
Interactively, ROW is the prefix numeric argument and defaults to zero.
GLYPHS 0 means don't dump glyphs.
GLYPHS 1 means dump glyphs in short form.
GLYPHS > 1 or omitted means dump glyphs in long form.

If there's no tab-bar, or if the tab-bar is not drawn by Emacs,
do nothing.  */)
  (Lisp_Object row, Lisp_Object glyphs)
{
#if defined (HAVE_WINDOW_SYSTEM)
  struct frame *sf = SELECTED_FRAME ();
  struct glyph_matrix *m = WINDOWP (sf->tab_bar_window)
    ? XWINDOW (sf->tab_bar_window)->current_matrix
    : sf->current_matrix;
  EMACS_INT vpos;

  if (NILP (row))
    vpos = WINDOWP (sf->tab_bar_window) ? 0 :
      FRAME_MENU_BAR_LINES (sf) > 0 ? 1 : 0;
  else
    {
      CHECK_FIXNUM (row);
      vpos = XFIXNUM (row);
    }
  if (vpos >= 0 && vpos < m->nrows)
    dump_glyph_row (MATRIX_ROW (m, vpos), vpos,
		    TYPE_RANGED_FIXNUMP (int, glyphs) ? XFIXNUM (glyphs) : 2);
#endif
  return Qnil;
}

DEFUN ("dump-tool-bar-row", Fdump_tool_bar_row, Sdump_tool_bar_row, 1, 2, "P",
       doc: /* Dump glyph row ROW of the tool-bar of the current frame to stderr.
Interactively, ROW is the prefix numeric argument and defaults to zero.
GLYPHS 0 means don't dump glyphs.
GLYPHS 1 means dump glyphs in short form.
GLYPHS > 1 or omitted means dump glyphs in long form.

If there's no tool-bar, or if the tool-bar is not drawn by Emacs,
do nothing.  */)
  (Lisp_Object row, Lisp_Object glyphs)
{
#if defined (HAVE_WINDOW_SYSTEM) && ! defined (HAVE_EXT_TOOL_BAR)
  struct frame *sf = SELECTED_FRAME ();
  struct glyph_matrix *m = XWINDOW (sf->tool_bar_window)->current_matrix;
  EMACS_INT vpos;

  if (NILP (row))
    vpos = 0;
  else
    {
      CHECK_FIXNUM (row);
      vpos = XFIXNUM (row);
    }
  if (vpos >= 0 && vpos < m->nrows)
    dump_glyph_row (MATRIX_ROW (m, vpos), vpos,
		    TYPE_RANGED_FIXNUMP (int, glyphs) ? XFIXNUM (glyphs) : 2);
#endif
  return Qnil;
}


DEFUN ("trace-redisplay", Ftrace_redisplay, Strace_redisplay, 0, 1, "P",
       doc: /* Toggle tracing of redisplay.
With ARG, turn tracing on if and only if ARG is positive.  */)
  (Lisp_Object arg)
{
  if (NILP (arg))
    trace_redisplay_p = !trace_redisplay_p;
  else
    {
      arg = Fprefix_numeric_value (arg);
      trace_redisplay_p = XFIXNUM (arg) > 0;
    }

  return Qnil;
}


DEFUN ("trace-to-stderr", Ftrace_to_stderr, Strace_to_stderr, 1, MANY, "",
       doc: /* Like `format', but print result to stderr.
usage: (trace-to-stderr STRING &rest OBJECTS)  */)
  (ptrdiff_t nargs, Lisp_Object *args)
{
  Lisp_Object s = Fformat (nargs, args);
  fwrite (SDATA (s), 1, SBYTES (s), stderr);
  return Qnil;
}

#endif /* GLYPH_DEBUG */



/***********************************************************************
		     Building Desired Matrix Rows
 ***********************************************************************/

/* Return a temporary glyph row holding the glyphs of an overlay arrow.
   Used for non-window-redisplay windows, and for windows without left
   fringe.  */

static struct glyph_row *
get_overlay_arrow_glyph_row (struct window *w, Lisp_Object overlay_arrow_string)
{
  struct frame *f = XFRAME (WINDOW_FRAME (w));
  struct buffer *buffer = XBUFFER (w->contents);
  struct buffer *old = current_buffer;
  const unsigned char *arrow_string = SDATA (overlay_arrow_string);
  ptrdiff_t arrow_len = SBYTES (overlay_arrow_string), char_num = 0;
  const unsigned char *arrow_end = arrow_string + arrow_len;
  const unsigned char *p;
  struct it it;
  bool multibyte_p;
  int n_glyphs_before;

  set_buffer_temp (buffer);
  init_iterator (&it, w, -1, -1, &scratch_glyph_row, DEFAULT_FACE_ID);
  scratch_glyph_row.reversed_p = false;
  it.glyph_row->used[TEXT_AREA] = 0;
  SET_TEXT_POS (it.position, 0, 0);

  multibyte_p = !NILP (BVAR (buffer, enable_multibyte_characters));
  p = arrow_string;
  while (p < arrow_end)
    {
      Lisp_Object face, ilisp;

      /* Get the next character.  */
      if (multibyte_p)
	it.c = it.char_to_display = check_char_and_length (p, &it.len);
      else
	{
	  it.c = it.char_to_display = *p, it.len = 1;
	  if (! ASCII_CHAR_P (it.c))
	    it.char_to_display = BYTE8_TO_CHAR (it.c);
	}
      p += it.len;

      /* Get its face.  */
      ilisp = make_fixnum (char_num++);
      face = Fget_text_property (ilisp, Qface, overlay_arrow_string);
      it.face_id = compute_char_face (f, it.char_to_display, face);

      /* Compute its width, get its glyphs.  */
      n_glyphs_before = it.glyph_row->used[TEXT_AREA];
      SET_TEXT_POS (it.position, -1, -1);
      PRODUCE_GLYPHS (&it);

      /* If this character doesn't fit any more in the line, we have
	 to remove some glyphs.  */
      if (it.current_x > it.last_visible_x)
	{
	  it.glyph_row->used[TEXT_AREA] = n_glyphs_before;
	  break;
	}
    }

  set_buffer_temp (old);
  return it.glyph_row;
}


/* Insert truncation glyphs at the start of IT->glyph_row.  Which
   glyphs to insert is determined by produce_special_glyphs.  */

static void
insert_left_trunc_glyphs (struct it *it)
{
  struct it truncate_it;
  struct glyph *from, *end, *to, *toend;

  eassert (!FRAME_WINDOW_P (it->f)
	   || (!it->glyph_row->reversed_p
	       && WINDOW_LEFT_FRINGE_WIDTH (it->w) == 0)
	   || (it->glyph_row->reversed_p
	       && WINDOW_RIGHT_FRINGE_WIDTH (it->w) == 0));

  /* Get the truncation glyphs.  */
  truncate_it = *it;
  truncate_it.current_x = 0;
  truncate_it.face_id = DEFAULT_FACE_ID;
  truncate_it.glyph_row = &scratch_glyph_row;
  truncate_it.area = TEXT_AREA;
  truncate_it.glyph_row->used[TEXT_AREA] = 0;
  CHARPOS (truncate_it.position) = BYTEPOS (truncate_it.position) = -1;
  truncate_it.object = Qnil;
  produce_special_glyphs (&truncate_it, IT_TRUNCATION);

  /* Overwrite glyphs from IT with truncation glyphs.  */
  if (!it->glyph_row->reversed_p)
    {
      short tused = truncate_it.glyph_row->used[TEXT_AREA];

      from = truncate_it.glyph_row->glyphs[TEXT_AREA];
      end = from + tused;
      to = it->glyph_row->glyphs[TEXT_AREA];
      toend = to + it->glyph_row->used[TEXT_AREA];
      if (FRAME_WINDOW_P (it->f))
	{
	  /* On GUI frames, when variable-size fonts are displayed,
	     the truncation glyphs may need more pixels than the row's
	     glyphs they overwrite.  We overwrite more glyphs to free
	     enough screen real estate, and enlarge the stretch glyph
	     on the right (see display_line), if there is one, to
	     preserve the screen position of the truncation glyphs on
	     the right.  */
	  int w = 0;
	  struct glyph *g = to;
	  short used;

	  /* The first glyph could be partially visible, in which case
	     it->glyph_row->x will be negative.  But we want the left
	     truncation glyphs to be aligned at the left margin of the
	     window, so we override the x coordinate at which the row
	     will begin.  */
	  it->glyph_row->x = 0;
	  while (g < toend && w < it->truncation_pixel_width)
	    {
	      w += g->pixel_width;
	      ++g;
	    }
	  if (g - to - tused > 0)
	    {
	      memmove (to + tused, g, (toend - g) * sizeof(*g));
	      it->glyph_row->used[TEXT_AREA] -= g - to - tused;
	    }
	  used = it->glyph_row->used[TEXT_AREA];
	  if (it->glyph_row->truncated_on_right_p
	      && WINDOW_RIGHT_FRINGE_WIDTH (it->w) == 0
	      && it->glyph_row->glyphs[TEXT_AREA][used - 2].type
	      == STRETCH_GLYPH)
	    {
	      int extra = w - it->truncation_pixel_width;

	      it->glyph_row->glyphs[TEXT_AREA][used - 2].pixel_width += extra;
	    }
	}

      while (from < end)
	*to++ = *from++;

      /* There may be padding glyphs left over.  Overwrite them too.  */
      if (!FRAME_WINDOW_P (it->f))
	{
	  while (to < toend && CHAR_GLYPH_PADDING_P (*to))
	    {
	      from = truncate_it.glyph_row->glyphs[TEXT_AREA];
	      while (from < end)
		*to++ = *from++;
	    }
	}

      if (to > toend)
	it->glyph_row->used[TEXT_AREA] = to - it->glyph_row->glyphs[TEXT_AREA];
    }
  else
    {
      short tused = truncate_it.glyph_row->used[TEXT_AREA];

      /* In R2L rows, overwrite the last (rightmost) glyphs, and do
	 that back to front.  */
      end = truncate_it.glyph_row->glyphs[TEXT_AREA];
      from = end + truncate_it.glyph_row->used[TEXT_AREA] - 1;
      toend = it->glyph_row->glyphs[TEXT_AREA];
      to = toend + it->glyph_row->used[TEXT_AREA] - 1;
      if (FRAME_WINDOW_P (it->f))
	{
	  int w = 0;
	  struct glyph *g = to;

	  while (g >= toend && w < it->truncation_pixel_width)
	    {
	      w += g->pixel_width;
	      --g;
	    }
	  if (to - g - tused > 0)
	    to = g + tused;
	  if (it->glyph_row->truncated_on_right_p
	      && WINDOW_LEFT_FRINGE_WIDTH (it->w) == 0
	      && it->glyph_row->glyphs[TEXT_AREA][1].type == STRETCH_GLYPH)
	    {
	      int extra = w - it->truncation_pixel_width;

	      it->glyph_row->glyphs[TEXT_AREA][1].pixel_width += extra;
	    }
	}

      while (from >= end && to >= toend)
	*to-- = *from--;
      if (!FRAME_WINDOW_P (it->f))
	{
	  while (to >= toend && CHAR_GLYPH_PADDING_P (*to))
	    {
	      from =
		truncate_it.glyph_row->glyphs[TEXT_AREA]
		+ truncate_it.glyph_row->used[TEXT_AREA] - 1;
	      while (from >= end && to >= toend)
		*to-- = *from--;
	    }
	}
      if (from >= end)
	{
	  /* Need to free some room before prepending additional
	     glyphs.  */
	  int move_by = from - end + 1;
	  struct glyph *g0 = it->glyph_row->glyphs[TEXT_AREA];
	  struct glyph *g = g0 + it->glyph_row->used[TEXT_AREA] - 1;

	  for ( ; g >= g0; g--)
	    g[move_by] = *g;
	  while (from >= end)
	    *to-- = *from--;
	  it->glyph_row->used[TEXT_AREA] += move_by;
	}
    }
}

/* Compute the hash code for ROW.  */
unsigned
row_hash (struct glyph_row *row)
{
  int area, k;
  unsigned hashval = 0;

  for (area = LEFT_MARGIN_AREA; area < LAST_AREA; ++area)
    for (k = 0; k < row->used[area]; ++k)
      hashval = ((((hashval << 4) + (hashval >> 24)) & 0x0fffffff)
		  + row->glyphs[area][k].u.val
		  + row->glyphs[area][k].face_id
		  + row->glyphs[area][k].padding_p
		  + (row->glyphs[area][k].type << 2));

  return hashval;
}

/* Compute the pixel height and width of IT->glyph_row.

   Most of the time, ascent and height of a display line will be equal
   to the max_ascent and max_height values of the display iterator
   structure.  This is not the case if

   1. We hit ZV without displaying anything.  In this case, max_ascent
   and max_height will be zero.

   2. We have some glyphs that don't contribute to the line height.
   (The glyph row flag contributes_to_line_height_p is for future
   pixmap extensions).

   The first case is easily covered by using default values because in
   these cases, the line height does not really matter, except that it
   must not be zero.  */

static void
compute_line_metrics (struct it *it)
{
  struct glyph_row *row = it->glyph_row;

  if (FRAME_WINDOW_P (it->f))
    {
      int i, min_y, max_y;

      /* The line may consist of one space only, that was added to
	 place the cursor on it.  If so, the row's height hasn't been
	 computed yet.  */
      if (row->height == 0)
	{
	  if (it->max_ascent + it->max_descent == 0)
	    it->max_descent = it->max_phys_descent = FRAME_LINE_HEIGHT (it->f);
	  row->ascent = it->max_ascent;
	  row->height = it->max_ascent + it->max_descent;
	  row->phys_ascent = it->max_phys_ascent;
	  row->phys_height = it->max_phys_ascent + it->max_phys_descent;
	  row->extra_line_spacing = it->max_extra_line_spacing;
	}

      /* Compute the width of this line.  */
      row->pixel_width = row->x;
      for (i = 0; i < row->used[TEXT_AREA]; ++i)
	row->pixel_width += row->glyphs[TEXT_AREA][i].pixel_width;

      eassert (row->pixel_width >= 0);
      eassert (row->ascent >= 0 && row->height > 0);

      row->overlapping_p = (MATRIX_ROW_OVERLAPS_SUCC_P (row)
			    || MATRIX_ROW_OVERLAPS_PRED_P (row));

      /* If first line's physical ascent is larger than its logical
         ascent, use the physical ascent, and make the row taller.
         This makes accented characters fully visible.  */
      if (row == MATRIX_FIRST_TEXT_ROW (it->w->desired_matrix)
	  && row->phys_ascent > row->ascent)
	{
	  row->height += row->phys_ascent - row->ascent;
	  row->ascent = row->phys_ascent;
	}

      /* Compute how much of the line is visible.  */
      row->visible_height = row->height;

      min_y = WINDOW_TAB_LINE_HEIGHT (it->w) + WINDOW_HEADER_LINE_HEIGHT (it->w);
      max_y = WINDOW_BOX_HEIGHT_NO_MODE_LINE (it->w);

      if (row->y < min_y)
	row->visible_height -= min_y - row->y;
      if (row->y + row->height > max_y)
	row->visible_height -= row->y + row->height - max_y;
    }
  else
    {
      row->pixel_width = row->used[TEXT_AREA];
      if (row->continued_p)
	row->pixel_width -= it->continuation_pixel_width;
      else if (row->truncated_on_right_p)
	row->pixel_width -= it->truncation_pixel_width;
      row->ascent = row->phys_ascent = 0;
      row->height = row->phys_height = row->visible_height = 1;
      row->extra_line_spacing = 0;
    }

  /* Compute a hash code for this row.  */
  row->hash = row_hash (row);

  it->max_ascent = it->max_descent = 0;
  it->max_phys_ascent = it->max_phys_descent = 0;
}


static void
clear_position (struct it *it)
{
  it->position.charpos = 0;
  it->position.bytepos = 0;
}

/* Append one space to the glyph row of iterator IT if doing a
   window-based redisplay.  The space has the same face as
   IT->face_id.  Value is true if a space was added.

   This function is called to make sure that there is always one glyph
   at the end of a glyph row that the cursor can be set on under
   window-systems.  (If there weren't such a glyph we would not know
   how wide and tall a box cursor should be displayed).

   At the same time this space let's a nicely handle clearing to the
   end of the line if the row ends in italic text.  */

static bool
append_space_for_newline (struct it *it, bool default_face_p)
{
  int n = it->glyph_row->used[TEXT_AREA];

  if (it->glyph_row->glyphs[TEXT_AREA] + n
      < it->glyph_row->glyphs[1 + TEXT_AREA])
    {
      /* Save some values that must not be changed.
	 Must save IT->c and IT->len because otherwise
	 ITERATOR_AT_END_P wouldn't work anymore after
	 append_space_for_newline has been called.  */
      enum display_element_type saved_what = it->what;
      int saved_c = it->c, saved_len = it->len;
      int saved_char_to_display = it->char_to_display;
      int saved_x = it->current_x;
      const int saved_face_id = it->face_id;
      bool saved_box_end = it->end_of_box_run_p;
      struct text_pos saved_pos = it->position;
      Lisp_Object saved_object = it->object;
      struct face *face;

      it->what = IT_CHARACTER;
      clear_position (it);
      it->object = Qnil;
      it->len = 1;

      int char_width = 1;

      if (default_face_p
#ifdef HAVE_WINDOW_SYSTEM
          || FRAME_WINDOW_P (it->f)
#endif
	  )
	{
	  const int local_default_face_id =
	    lookup_basic_face (it->w, it->f, DEFAULT_FACE_ID);

#ifdef HAVE_WINDOW_SYSTEM
	  if (FRAME_WINDOW_P (it->f))
	    {
	      struct face *default_face
		= FACE_FROM_ID (it->f, local_default_face_id);
	      struct font *font = (default_face->font
	                           ? default_face->font
	                           : FRAME_FONT (it->f));
	      char_width = (font->average_width
	                    ? font->average_width
	                    : font->space_width);
	    }
#endif
	  if (default_face_p)
	    it->face_id = local_default_face_id;
	}
      /* Corner case for when display-fill-column-indicator-mode
	 is active and the extra character should be added in the
	 same place than the line.  */

      const int indicator_column =
	fill_column_indicator_column (it, char_width);
      int saved_end_of_box_run = it->end_of_box_run_p;
      bool should_keep_end_of_box_run = false;

      if (it->current_x == indicator_column)
	{
	  it->c = it->char_to_display
	    = XFIXNAT (Vdisplay_fill_column_indicator_character);
	  it->face_id
	    = merge_faces (it->w, Qfill_column_indicator,
	                   0, saved_face_id);
	  face = FACE_FROM_ID (it->f, it->face_id);
	}
      else
	{
	  it->c = it->char_to_display = ' ';
	  /* If the default face was remapped, be sure to use the
	     remapped face for the appended newline.  */

	  face = FACE_FROM_ID (it->f, it->face_id);
	  it->face_id = FACE_FOR_CHAR (it->f, face, 0, -1, Qnil);
	  /* In R2L rows, we will prepend a stretch glyph that will
	     have the end_of_box_run_p flag set for it, so there's no
	     need for the appended newline glyph to have that flag
	     set.  */
	  if (!(it->glyph_row->reversed_p
		/* But if the appended newline glyph goes all the way to
		   the end of the row, there will be no stretch glyph,
		   so leave the box flag set.  */
		&& saved_x + FRAME_COLUMN_WIDTH (it->f) < it->last_visible_x))
	    should_keep_end_of_box_run = true;
	}
      PRODUCE_GLYPHS (it);
      /* Restore the end_of_box_run_p flag which was reset by
	 PRODUCE_GLYPHS.  */
      if (should_keep_end_of_box_run)
	it->end_of_box_run_p = saved_end_of_box_run;
#ifdef HAVE_WINDOW_SYSTEM
      if (FRAME_WINDOW_P (it->f))
	{
	  /* Make sure this space glyph has the right ascent and
	     descent values, or else cursor at end of line will look
	     funny, and height of empty lines will be incorrect.  */
	  struct glyph *g = it->glyph_row->glyphs[TEXT_AREA] + n;
	  struct font *font = face->font ? face->font : FRAME_FONT (it->f);
	  if (n == 0)
	    {
	      Lisp_Object height, total_height;
	      int extra_line_spacing = it->extra_line_spacing;
	      int boff = font->baseline_offset;

	      if (font->vertical_centering)
		boff = VCENTER_BASELINE_OFFSET (font, it->f) - boff;

	      it->object = saved_object; /* get_it_property needs this */
	      normal_char_ascent_descent (font, -1, &it->ascent, &it->descent);
	      /* Must do a subset of line height processing from
		 gui_produce_glyph for newline characters.  */
	      height = get_it_property (it, Qline_height);
	      if (CONSP (height)
	          && CONSP (XCDR (height))
	          && NILP (XCDR (XCDR (height))))
		{
		  total_height = XCAR (XCDR (height));
		  height = XCAR (height);
		}
	      else
		total_height = Qnil;
	      height = calc_line_height_property (it, height, font, boff, true);

	      if (it->override_ascent >= 0)
		{
		  it->ascent = it->override_ascent;
		  it->descent = it->override_descent;
		  boff = it->override_boff;
		}
	      if (EQ (height, Qt))
		extra_line_spacing = 0;
	      else
		{
		  Lisp_Object spacing;

		  it->phys_ascent = it->ascent;
		  it->phys_descent = it->descent;
		  if (!NILP (height)
		      && XFIXNUM (height) > it->ascent + it->descent)
		    it->ascent = XFIXNUM (height) - it->descent;

		  if (!NILP (total_height))
		    spacing = calc_line_height_property (it, total_height, font,
		                                         boff, false);
		  else
		    {
		      spacing = get_it_property (it, Qline_spacing);
		      spacing = calc_line_height_property (it, spacing, font,
		                                           boff, false);
		    }
		  if (FIXNUMP (spacing))
		    {
		      extra_line_spacing = XFIXNUM (spacing);
		      if (!NILP (total_height))
			extra_line_spacing -= (it->phys_ascent + it->phys_descent);
		    }
		}
	      if (extra_line_spacing > 0)
		{
		  it->descent += extra_line_spacing;
		  if (extra_line_spacing > it->max_extra_line_spacing)
		    it->max_extra_line_spacing = extra_line_spacing;
		}
	      it->max_ascent = it->ascent;
	      it->max_descent = it->descent;
	      /* Make sure compute_line_metrics recomputes the row height.  */
	      it->glyph_row->height = 0;
	    }

	  g->ascent = it->max_ascent;
	  g->descent = it->max_descent;
	}
#endif /* HAVE_WINDOW_SYSTEM  */
      it->override_ascent = -1;
      it->constrain_row_ascent_descent_p = false;
      it->current_x = saved_x;
      it->object = saved_object;
      it->position = saved_pos;
      it->what = saved_what;
      it->face_id = saved_face_id;
      it->len = saved_len;
      it->c = saved_c;
      it->char_to_display = saved_char_to_display;
      it->end_of_box_run_p = saved_box_end;
      return true;
    }

  return false;
}


/* Extend the face of the last glyph in the text area of IT->glyph_row
   to the end of the display line.  Called from display_line.  If the
   glyph row is empty, add a space glyph to it so that we know the
   face to draw.  Set the glyph row flag fill_line_p.  If the glyph
   row is R2L, prepend a stretch glyph to cover the empty space to the
   left of the leftmost glyph.  */

static void
extend_face_to_end_of_line (struct it *it)
{
  struct frame *f = it->f;

  /* If line is already filled, do nothing.  Non window-system frames
     get a grace of one more ``pixel'' because their characters are
     1-``pixel'' wide, so they hit the equality too early.  This grace
     is needed only for R2L rows that are not continued, to produce
     one extra blank where we could display the cursor.  */
  if ((it->current_x >= it->last_visible_x
       + (!FRAME_WINDOW_P (f)
	  && it->glyph_row->reversed_p
	  && !it->glyph_row->continued_p))
      /* If the window has display margins, we will need to extend
	 their face even if the text area is filled.  */
      && !(WINDOW_LEFT_MARGIN_WIDTH (it->w) > 0
	   || WINDOW_RIGHT_MARGIN_WIDTH (it->w) > 0))
    return;

  specpdl_ref count = SPECPDL_INDEX ();

  /* Don't allow the user to quit out of face-merging code, in case
     this is called when redisplaying a non-selected window, with
     point temporarily moved to window-point.  */
  specbind (Qinhibit_quit, Qt);
  /* The default face, possibly remapped. */
  struct face *default_face =
    FACE_FROM_ID_OR_NULL (f, lookup_basic_face (it->w, f, DEFAULT_FACE_ID));
  if (!default_face)
    return;

  const int extend_face_id =
    (it->face_id == default_face->id || it->s != NULL)
    ? it->face_id
    : (it->glyph_row->ends_at_zv_p
       ? default_face->id
       : face_at_pos (it, LFACE_EXTEND_INDEX));
  unbind_to (count, Qnil);

  /* Face extension extends the background and box of IT->extend_face_id
     to the end of the line.  If the background equals the background
     of the frame, we don't have to do anything.  */
  struct face *face = FACE_FROM_ID (f, (it->face_before_selective_p
                                        ? it->saved_face_id
                                        : extend_face_id));

  if (FRAME_WINDOW_P (f)
      && MATRIX_ROW_DISPLAYS_TEXT_P (it->glyph_row)
      && face->box == FACE_NO_BOX
      && face->underline == FACE_NO_UNDERLINE
      && !face->overline_p
      && !face->strike_through_p
      && face->background == FRAME_BACKGROUND_PIXEL (f)
#ifdef HAVE_WINDOW_SYSTEM
      && !face->stipple
#endif
      && !it->glyph_row->reversed_p
      && !display_fill_column_indicator)
    return;

  /* Set the glyph row flag indicating that the face of the last glyph
     in the text area has to be drawn to the end of the text area.  */
  it->glyph_row->fill_line_p = true;

  const int orig_face_id = it->face_id;
  /* If current character of IT is not ASCII, make sure we have the
     ASCII face.  This will be automatically undone the next time
     get_next_display_element returns a multibyte character.  Note
     that the character will always be single byte in unibyte
     text.  */
  if (!ASCII_CHAR_P (it->c))
    it->face_id = FACE_FOR_CHAR (f, face, 0, -1, Qnil);


#ifdef HAVE_WINDOW_SYSTEM
  if (FRAME_WINDOW_P (f))
    {
      /* If the row is empty, add a space with the current face of IT,
	 so that we know which face to draw.  */
      if (it->glyph_row->used[TEXT_AREA] == 0)
	{
	  it->glyph_row->glyphs[TEXT_AREA][0] = space_glyph;
	  it->glyph_row->glyphs[TEXT_AREA][0].face_id = face->id;
	  it->glyph_row->used[TEXT_AREA] = 1;
	}
      /* Mode line and the header line don't have margins, and
	 likewise the frame's tool-bar window, if there is any.  */
      if (!(it->glyph_row->mode_line_p
	    || (WINDOWP (f->tab_bar_window)
		&& it->w == XWINDOW (f->tab_bar_window))
#ifndef HAVE_EXT_TOOL_BAR
	    || (WINDOWP (f->tool_bar_window)
		&& it->w == XWINDOW (f->tool_bar_window))
#endif
	    ))
	{
	  if (WINDOW_LEFT_MARGIN_WIDTH (it->w) > 0
	      && it->glyph_row->used[LEFT_MARGIN_AREA] == 0)
	    {
	      it->glyph_row->glyphs[LEFT_MARGIN_AREA][0] = space_glyph;
	      it->glyph_row->glyphs[LEFT_MARGIN_AREA][0].face_id =
		default_face->id;
	      it->glyph_row->used[LEFT_MARGIN_AREA] = 1;
	    }
	  if (WINDOW_RIGHT_MARGIN_WIDTH (it->w) > 0
	      && it->glyph_row->used[RIGHT_MARGIN_AREA] == 0)
	    {
	      it->glyph_row->glyphs[RIGHT_MARGIN_AREA][0] = space_glyph;
	      it->glyph_row->glyphs[RIGHT_MARGIN_AREA][0].face_id =
		default_face->id;
	      it->glyph_row->used[RIGHT_MARGIN_AREA] = 1;
	    }

	  struct font *font = (default_face->font
	                       ? default_face->font
	                       : FRAME_FONT (f));

	  const int char_width = (font->average_width
	                          ? font->average_width
	                          : font->space_width);

	  const int indicator_column =
	    fill_column_indicator_column (it, char_width);

	  const char saved_char = it->char_to_display;
	  const struct text_pos saved_pos = it->position;
	  const bool saved_avoid_cursor = it->avoid_cursor_p;
	  const bool saved_box_start = it->start_of_box_run_p;
	  Lisp_Object save_object = it->object;
	  const int saved_face_id = it->face_id;

	  it->face_id = extend_face_id;
	  it->avoid_cursor_p = true;
	  it->object = Qnil;

	  const int stretch_height = it->ascent + it->descent;
	  const int stretch_ascent =
	    (stretch_height * FONT_BASE (font)) / FONT_HEIGHT (font);

	  if (indicator_column >= 0
	      && indicator_column > it->current_x
	      && indicator_column < it->last_visible_x)
	    {

	      /* Here we subtract char_width because we want the
		 column indicator in the column INDICATOR_COLUMN,
		 not after it.  */
	      const int stretch_width =
		indicator_column - it->current_x - char_width;

	      clear_position (it);

	      /* Only generate a stretch glyph if there is distance
		 between current_x and the indicator position.  */
	      if (stretch_width > 0)
		{
		  append_stretch_glyph (it, Qnil, stretch_width,
		                        stretch_height, stretch_ascent);
		}

	      /* Generate the glyph indicator only if
		 append_space_for_newline didn't already.  */
	      if (it->current_x < indicator_column)
		{
		  const int save_face_id = it->face_id;
		  const int save_ascent = it->ascent;
		  const int save_descent = it->descent;
		  it->char_to_display
		    = XFIXNAT (Vdisplay_fill_column_indicator_character);
		  it->face_id
		    = merge_faces (it->w, Qfill_column_indicator,
		                   0, extend_face_id);
		  PRODUCE_GLYPHS (it);
		  it->face_id = save_face_id;
		  it->ascent = save_ascent;
		  it->descent = save_descent;
		}
	    }

	  /* Fill space until window edge with the merged face.  Do that
	     only for L2R rows, as R2L rows are handled specially below.  */
	  if (!it->glyph_row->reversed_p)
	    {
	      const int stretch_width = it->last_visible_x - it->current_x;

	      if (stretch_width > 0)
		{
		  clear_position (it);
		  append_stretch_glyph (it, Qnil, stretch_width,
					stretch_height, stretch_ascent);
		}
	    }

	  it->char_to_display = saved_char;
	  it->position = saved_pos;
	  it->avoid_cursor_p = saved_avoid_cursor;
	  it->start_of_box_run_p = saved_box_start;
	  it->object = save_object;
	  it->face_id = saved_face_id;
	}
      if (it->glyph_row->reversed_p)
	{
	  /* Prepend a stretch glyph to the row, such that the
	     rightmost glyph will be drawn flushed all the way to the
	     right margin of the window.  The stretch glyph that will
	     occupy the empty space, if any, to the left of the
	     glyph.  */
	  struct font *font = face->font ? face->font : FRAME_FONT (f);
	  struct glyph *row_start = it->glyph_row->glyphs[TEXT_AREA];
	  struct glyph *row_end = row_start + it->glyph_row->used[TEXT_AREA];
	  struct glyph *g;
	  int row_width, stretch_ascent, stretch_width;
	  struct text_pos saved_pos;
	  int saved_face_id;
	  bool saved_avoid_cursor, saved_box_start;

	  for (row_width = 0, g = row_start; g < row_end; g++)
	    row_width += g->pixel_width;

	  /* FIXME: There are various minor display glitches in R2L
	     rows when only one of the fringes is missing.  The
	     strange condition below produces the least bad effect.  */
	  if ((WINDOW_LEFT_FRINGE_WIDTH (it->w) == 0)
	      == (WINDOW_RIGHT_FRINGE_WIDTH (it->w) == 0)
	      || WINDOW_RIGHT_FRINGE_WIDTH (it->w) != 0)
	    stretch_width = window_box_width (it->w, TEXT_AREA);
	  else
	    stretch_width = it->last_visible_x - it->first_visible_x;
	  stretch_width -= row_width;

	  if (stretch_width > 0)
	    {
	      stretch_ascent =
		(((it->ascent + it->descent)
		  * FONT_BASE (font)) / FONT_HEIGHT (font));
	      saved_pos = it->position;
	      clear_position (it);
	      saved_avoid_cursor = it->avoid_cursor_p;
	      it->avoid_cursor_p = true;
	      saved_face_id = it->face_id;
	      saved_box_start = it->start_of_box_run_p;
	      /* The last row's stretch glyph should get the default
		 face, to avoid painting the rest of the window with
		 the region face, if the region ends at ZV.  */
	      it->face_id = (it->glyph_row->ends_at_zv_p ?
	                     default_face->id : face->id);

	      it->start_of_box_run_p = false;
	      append_stretch_glyph (it, Qnil, stretch_width,
				    it->ascent + it->descent, stretch_ascent);
	      it->position = saved_pos;
	      it->avoid_cursor_p = saved_avoid_cursor;
	      it->face_id = saved_face_id;
	      it->start_of_box_run_p = saved_box_start;
	    }
	  /* If stretch_width comes out negative, it means that the
	     last glyph is only partially visible.  In R2L rows, we
	     want the leftmost glyph to be partially visible, so we
	     need to give the row the corresponding left offset.  */
	  if (stretch_width < 0)
	    it->glyph_row->x = stretch_width;
	}
      it->face_id = orig_face_id;
    }
  else
#endif	/* HAVE_WINDOW_SYSTEM */
    {
      /* Save some values that must not be changed.  */
      int saved_x = it->current_x;
      struct text_pos saved_pos = it->position;
      Lisp_Object saved_object = it->object;;
      enum display_element_type saved_what = it->what;

      it->what = IT_CHARACTER;
      clear_position (it);
      it->object = Qnil;
      it->c = it->char_to_display = ' ';
      it->len = 1;

      if (WINDOW_LEFT_MARGIN_WIDTH (it->w) > 0
	  && (it->glyph_row->used[LEFT_MARGIN_AREA]
	      < WINDOW_LEFT_MARGIN_WIDTH (it->w))
	  && !it->glyph_row->mode_line_p
	  && face->background != FRAME_BACKGROUND_PIXEL (f))
	{
	  struct glyph *g = it->glyph_row->glyphs[LEFT_MARGIN_AREA];
	  struct glyph *e = g + it->glyph_row->used[LEFT_MARGIN_AREA];

	  for (it->current_x = 0; g < e; g++)
	    it->current_x += g->pixel_width;

	  it->area = LEFT_MARGIN_AREA;
	  it->face_id = default_face->id;
	  while (it->glyph_row->used[LEFT_MARGIN_AREA]
		 < WINDOW_LEFT_MARGIN_WIDTH (it->w)
		 && g < it->glyph_row->glyphs[TEXT_AREA])
	    {
	      PRODUCE_GLYPHS (it);
	      /* term.c:produce_glyphs advances it->current_x only for
		 TEXT_AREA.  */
	      it->current_x += it->pixel_width;
	      g++;
	    }

	  it->current_x = saved_x;
	  it->area = TEXT_AREA;
	}

      /* The last row's blank glyphs should get the default face, to
	 avoid painting the rest of the window with the region face,
	 if the region ends at ZV.  */
      it->face_id = (it->glyph_row->ends_at_zv_p ?
                     default_face->id : face->id);

      /* Make sure our idea of current_x is in sync with the glyphs
	 actually in the glyph row.  They might differ because
	 append_space_for_newline can insert one glyph without
	 updating current_x.  */
      it->current_x = it->glyph_row->used[TEXT_AREA];

      /* The above assignment causes the code below to use a
	 non-standard semantics of it->current_x: it is measured
	 relative to the beginning of the text-area, thus disregarding
	 the window's hscroll.  That is why we need to correct the
	 indicator column for the hscroll, otherwise the indicator
	 will not move together with the text as result of horizontal
	 scrolling.  */
      const int indicator_column =
	fill_column_indicator_column (it, 1) - it->first_visible_x;

      /* Display fill-column indicator if needed.  */
      while (it->current_x <= it->last_visible_x)
	{
	  if (it->current_x != indicator_column)
	    PRODUCE_GLYPHS (it);
	  else
	    {
	      int saved_face_id = it->face_id;
	      it->face_id
		= merge_faces (it->w, Qfill_column_indicator, 0, extend_face_id);
	      it->c = it->char_to_display
		= XFIXNAT (Vdisplay_fill_column_indicator_character);

	      PRODUCE_GLYPHS (it);

	      it->face_id = saved_face_id;
	      it->c = it->char_to_display = ' ';
	    }
	}

      if (WINDOW_RIGHT_MARGIN_WIDTH (it->w) > 0
	  && (it->glyph_row->used[RIGHT_MARGIN_AREA]
	      < WINDOW_RIGHT_MARGIN_WIDTH (it->w))
	  && !it->glyph_row->mode_line_p
	  && face->background != FRAME_BACKGROUND_PIXEL (f))
	{
	  struct glyph *g = it->glyph_row->glyphs[RIGHT_MARGIN_AREA];
	  struct glyph *e = g + it->glyph_row->used[RIGHT_MARGIN_AREA];

	  for ( ; g < e; g++)
	    it->current_x += g->pixel_width;

	  it->area = RIGHT_MARGIN_AREA;
	  it->face_id = default_face->id;
	  while (it->glyph_row->used[RIGHT_MARGIN_AREA]
		 < WINDOW_RIGHT_MARGIN_WIDTH (it->w)
		 && g < it->glyph_row->glyphs[LAST_AREA])
	    {
	      PRODUCE_GLYPHS (it);
	      it->current_x += it->pixel_width;
	      g++;
	    }

	  it->area = TEXT_AREA;
	}

      /* Don't count these blanks really.  It would let us insert a left
	 truncation glyph below and make us set the cursor on them, maybe.  */
      it->current_x = saved_x;
      it->object = saved_object;
      it->position = saved_pos;
      it->what = saved_what;
      it->face_id = orig_face_id;
    }
}


/* Value is true if text starting at CHARPOS in current_buffer is
   trailing whitespace.  */

static bool
trailing_whitespace_p (ptrdiff_t charpos)
{
  ptrdiff_t bytepos = CHAR_TO_BYTE (charpos);
  int c = 0;

  while (bytepos < ZV_BYTE
	 && (c = FETCH_BYTE (bytepos),
	     c == ' ' || c == '\t'))
    ++bytepos;

  if (bytepos >= ZV_BYTE || c == '\n' || c == '\r')
    {
      if (bytepos != PT_BYTE)
	return true;
    }
  return false;
}


/* Highlight trailing whitespace, if any, in row at IT.  */

static void
highlight_trailing_whitespace (struct it *it)
{
  struct glyph_row *row = it->glyph_row;
  int used = row->used[TEXT_AREA];

  if (used)
    {
      struct glyph *start = row->glyphs[TEXT_AREA];
      struct glyph *glyph = start + used - 1;

      if (row->reversed_p)
	{
	  /* Right-to-left rows need to be processed in the opposite
	     direction, so swap the edge pointers. */
	  glyph = start;
	  start = row->glyphs[TEXT_AREA] + used - 1;
	}

      /* Skip over glyphs inserted to display the cursor at the
	 end of a line, for extending the face of the last glyph
	 to the end of the line on terminals, and for truncation
	 and continuation glyphs.  */
      if (!row->reversed_p)
	{
	  while (glyph >= start
	         && (glyph->type == CHAR_GLYPH
	             || glyph->type == STRETCH_GLYPH)
		 && NILP (glyph->object))
	    --glyph;
	}
      else
	{
	  while (glyph <= start
		 && (glyph->type == CHAR_GLYPH || glyph->type == STRETCH_GLYPH)
		 && NILP (glyph->object))
	    ++glyph;
	}

      /* If last glyph is a space or stretch, and it's trailing
	 whitespace, set the face of all trailing whitespace glyphs in
	 IT->glyph_row to `trailing-whitespace'.  */
      if ((row->reversed_p ? glyph <= start : glyph >= start)
	  && BUFFERP (glyph->object)
	  && (glyph->type == STRETCH_GLYPH
	      || (glyph->type == CHAR_GLYPH
		  && glyph->u.ch == ' '))
	  && trailing_whitespace_p (glyph->charpos))
	{
	  int face_id = lookup_named_face (it->w, it->f, Qtrailing_whitespace, false);
	  if (face_id < 0)
	    return;

	  if (!row->reversed_p)
	    {
	      while (glyph >= start
		     && BUFFERP (glyph->object)
		     && (glyph->type == STRETCH_GLYPH
			 || (glyph->type == CHAR_GLYPH
			     && glyph->u.ch == ' ')))
		(glyph--)->face_id = face_id;
	    }
	  else
	    {
	      while (glyph <= start
		     && BUFFERP (glyph->object)
		     && (glyph->type == STRETCH_GLYPH
			 || (glyph->type == CHAR_GLYPH
			     && glyph->u.ch == ' ')))
		(glyph++)->face_id = face_id;
	    }
	}
    }
}


/* Value is true if glyph row ROW should be
   considered to hold the buffer position CHARPOS.  */

static bool
row_for_charpos_p (struct glyph_row *row, ptrdiff_t charpos)
{
  bool result = true;

  if (charpos == CHARPOS (row->end.pos)
      || charpos == MATRIX_ROW_END_CHARPOS (row))
    {
      /* Suppose the row ends on a string.
	 Unless the row is continued, that means it ends on a newline
	 in the string.  If it's anything other than a display string
	 (e.g., a before-string from an overlay), we don't want the
	 cursor there.  (This heuristic seems to give the optimal
	 behavior for the various types of multi-line strings.)
	 One exception: if the string has `cursor' property on one of
	 its characters, we _do_ want the cursor there.  */
      if (CHARPOS (row->end.string_pos) >= 0)
	{
	  if (row->continued_p)
	    result = true;
	  else
	    {
	      /* Check for `display' property.  */
	      struct glyph *beg = row->glyphs[TEXT_AREA];
	      struct glyph *end = beg + row->used[TEXT_AREA] - 1;
	      struct glyph *glyph;

	      result = false;
	      for (glyph = end; glyph >= beg; --glyph)
		if (STRINGP (glyph->object))
		  {
		    Lisp_Object prop
		      = Fget_char_property (make_fixnum (charpos),
					    Qdisplay, Qnil);
		    result =
		      (!NILP (prop)
		       && display_prop_string_p (prop, glyph->object));
		    /* If there's a `cursor' property on one of the
		       string's characters, this row is a cursor row,
		       even though this is not a display string.  */
		    if (!result)
		      {
			Lisp_Object s = glyph->object;

			for ( ; glyph >= beg && EQ (glyph->object, s); --glyph)
			  {
			    ptrdiff_t gpos = glyph->charpos;

			    if (!NILP (Fget_char_property (make_fixnum (gpos),
							   Qcursor, s)))
			      {
				result = true;
				break;
			      }
			  }
		      }
		    break;
		  }
	    }
	}
      else if (MATRIX_ROW_ENDS_IN_MIDDLE_OF_CHAR_P (row))
	{
	  /* If the row ends in middle of a real character,
	     and the line is continued, we want the cursor here.
	     That's because CHARPOS (ROW->end.pos) would equal
	     PT if PT is before the character.  */
	  if (!row->ends_in_ellipsis_p)
	    result = row->continued_p;
	  else
	  /* If the row ends in an ellipsis, then
	     CHARPOS (ROW->end.pos) will equal point after the
	     invisible text.  We want that position to be displayed
	     after the ellipsis.  */
	    result = false;
	}
      /* If the row ends at ZV, display the cursor at the end of that
	 row instead of at the start of the row below.  */
      else
	result = row->ends_at_zv_p;
    }

  return result;
}

/* Value is true if glyph row ROW should be
   used to hold the cursor.  */

static bool
cursor_row_p (struct glyph_row *row)
{
  return row_for_charpos_p (row, PT);
}



/* Push the property PROP so that it will be rendered at the current
   position in IT.  Return true if PROP was successfully pushed, false
   otherwise.  Called from handle_line_prefix to handle the
   `line-prefix' and `wrap-prefix' properties.  */

static bool
push_prefix_prop (struct it *it, Lisp_Object prop)
{
  struct text_pos pos =
    STRINGP (it->string) ? it->current.string_pos : it->current.pos;

  eassert (it->method == GET_FROM_BUFFER
	   || it->method == GET_FROM_DISPLAY_VECTOR
	   || it->method == GET_FROM_STRING
	   || it->method == GET_FROM_IMAGE);

  /* We need to save the current buffer/string position, so it will be
     restored by pop_it, because iterate_out_of_display_property
     depends on that being set correctly, but some situations leave
     it->position not yet set when this function is called.  */
  push_it (it, &pos);

  if (STRINGP (prop))
    {
      if (SCHARS (prop) == 0)
	{
	  pop_it (it);
	  return false;
	}

      it->string = prop;
      it->string_from_prefix_prop_p = true;
      it->multibyte_p = STRING_MULTIBYTE (it->string);
      it->current.overlay_string_index = -1;
      IT_STRING_CHARPOS (*it) = IT_STRING_BYTEPOS (*it) = 0;
      it->end_charpos = it->string_nchars = SCHARS (it->string);
      it->method = GET_FROM_STRING;
      it->stop_charpos = 0;
      it->prev_stop = 0;
      it->base_level_stop = 0;
      it->cmp_it.id = -1;

      /* Force paragraph direction to be that of the parent
	 buffer/string.  */
      if (it->bidi_p && it->bidi_it.paragraph_dir == R2L)
	it->paragraph_embedding = it->bidi_it.paragraph_dir;
      else
	it->paragraph_embedding = L2R;

      /* Set up the bidi iterator for this display string.  */
      if (it->bidi_p)
	{
	  it->bidi_it.string.lstring = it->string;
	  it->bidi_it.string.s = NULL;
	  it->bidi_it.string.schars = it->end_charpos;
	  it->bidi_it.string.bufpos = IT_CHARPOS (*it);
	  it->bidi_it.string.from_disp_str = it->string_from_display_prop_p;
	  it->bidi_it.string.unibyte = !it->multibyte_p;
	  it->bidi_it.w = it->w;
	  bidi_init_it (0, 0, FRAME_WINDOW_P (it->f), &it->bidi_it);
	}
    }
  else if (CONSP (prop) && EQ (XCAR (prop), Qspace))
    {
      it->method = GET_FROM_STRETCH;
      it->object = prop;
    }
#ifdef HAVE_WINDOW_SYSTEM
  else if (IMAGEP (prop))
    {
      it->what = IT_IMAGE;
      it->image_id = lookup_image (it->f, prop, it->face_id);
      it->method = GET_FROM_IMAGE;
    }
#endif /* HAVE_WINDOW_SYSTEM */
  else
    {
      pop_it (it);		/* bogus display property, give up */
      return false;
    }

  return true;
}

/* Return the character-property PROP at the current position in IT.  */

static Lisp_Object
get_it_property (struct it *it, Lisp_Object prop)
{
  Lisp_Object position, object = it->object;

  if (STRINGP (object))
    position = make_fixnum (IT_STRING_CHARPOS (*it));
  else if (BUFFERP (object))
    {
      position = make_fixnum (IT_CHARPOS (*it));
      object = it->window;
    }
  else
    return Qnil;

  return Fget_char_property (position, prop, object);
}

/* Return the line-prefix/wrap-prefix property, checking both the
   current IT->OBJECT and the underlying buffer text.  */

static Lisp_Object
get_line_prefix_it_property (struct it *it, Lisp_Object prop)
{
  Lisp_Object prefix = get_it_property (it, prop);

  /* If we are looking at a display or overlay string, check also the
     underlying buffer text.  */
  if (NILP (prefix) && it->sp > 0 && STRINGP (it->object))
    return Fget_char_property (make_fixnum (IT_CHARPOS (*it)), prop,
			       it->w->contents);
  return prefix;
}

/* See if there's a line- or wrap-prefix, and if so, push it on IT.  */

static void
handle_line_prefix (struct it *it)
{
  Lisp_Object prefix;

  if (it->continuation_lines_width > 0)
    {
      prefix = get_line_prefix_it_property (it, Qwrap_prefix);
      if (NILP (prefix))
	prefix = Vwrap_prefix;
    }
  else
    {
      prefix = get_line_prefix_it_property (it, Qline_prefix);
      if (NILP (prefix))
	prefix = Vline_prefix;
    }
  if (! NILP (prefix) && push_prefix_prop (it, prefix))
    {
      /* If the prefix is wider than the window, and we try to wrap
	 it, it would acquire its own wrap prefix, and so on till the
	 iterator stack overflows.  So, don't wrap the prefix.  */
      it->line_wrap = TRUNCATE;
      it->avoid_cursor_p = true;
    }
}



/* Remove N glyphs at the start of a reversed IT->glyph_row.  Called
   only for R2L lines from display_line and display_string, when they
   decide that too many glyphs were produced by PRODUCE_GLYPHS, and
   the line/string needs to be continued on the next glyph row.  */
static void
unproduce_glyphs (struct it *it, int n)
{
  struct glyph *glyph, *end;

  eassert (it->glyph_row);
  eassert (it->glyph_row->reversed_p);
  eassert (it->area == TEXT_AREA);
  eassert (n <= it->glyph_row->used[TEXT_AREA]);

  if (n > it->glyph_row->used[TEXT_AREA])
    n = it->glyph_row->used[TEXT_AREA];
  glyph = it->glyph_row->glyphs[TEXT_AREA] + n;
  end = it->glyph_row->glyphs[TEXT_AREA] + it->glyph_row->used[TEXT_AREA];
  for ( ; glyph < end; glyph++)
    glyph[-n] = *glyph;
}

/* Find the positions in a bidi-reordered ROW to serve as ROW->minpos
   and ROW->maxpos.  */
static void
find_row_edges (struct it *it, struct glyph_row *row,
		ptrdiff_t min_pos, ptrdiff_t min_bpos,
		ptrdiff_t max_pos, ptrdiff_t max_bpos)
{
  /* FIXME: Revisit this when glyph ``spilling'' in continuation
     lines' rows is implemented for bidi-reordered rows.  */

  /* ROW->minpos is the value of min_pos, the minimal buffer position
     we have in ROW, or ROW->start.pos if that is smaller.  */
  if (min_pos <= ZV && min_pos < row->start.pos.charpos)
    SET_TEXT_POS (row->minpos, min_pos, min_bpos);
  else
    /* We didn't find buffer positions smaller than ROW->start, or
       didn't find _any_ valid buffer positions in any of the glyphs,
       so we must trust the iterator's computed positions.  */
      row->minpos = row->start.pos;
  if (max_pos <= 0)
    {
      max_pos = CHARPOS (it->current.pos);
      max_bpos = BYTEPOS (it->current.pos);
    }

  /* Here are the various use-cases for ending the row, and the
     corresponding values for ROW->maxpos:

     Line ends in a newline from buffer       eol_pos + 1
     Line is continued from buffer            max_pos + 1
     Line is truncated on right               it->current.pos
     Line ends in a newline from string       max_pos + 1(*)
      (*) + 1 only when line ends in a forward scan
     Line is continued from string            max_pos
     Line is continued from display vector    max_pos
     Line is entirely from a string           min_pos == max_pos
     Line is entirely from a display vector   min_pos == max_pos
     Line that ends at ZV                     ZV

     If you discover other use-cases, please add them here as
     appropriate.  */
  if (row->ends_at_zv_p)
    row->maxpos = it->current.pos;
  else if (row->used[TEXT_AREA])
    {
      bool seen_this_string = false;
      struct glyph_row *r1 = row - 1;

      /* Did we see the same display string on the previous row?  */
      if (STRINGP (it->object)
	  /* this is not the first row */
	  && row > it->w->desired_matrix->rows
	  /* previous row is not the header line or tab-line */
	  && !r1->mode_line_p
	  /* previous row also ends in a newline from a string */
	  && r1->ends_in_newline_from_string_p)
	{
	  struct glyph *start, *end;

	  /* Search for the last glyph of the previous row that came
	     from buffer or string.  Depending on whether the row is
	     L2R or R2L, we need to process it front to back or the
	     other way round.  */
	  if (!r1->reversed_p)
	    {
	      start = r1->glyphs[TEXT_AREA];
	      end = start + r1->used[TEXT_AREA];
	      /* Glyphs inserted by redisplay have nil as their object.  */
	      while (end > start
		     && NILP ((end - 1)->object)
		     && (end - 1)->charpos <= 0)
		--end;
	      if (end > start)
		{
		  if (EQ ((end - 1)->object, it->object))
		    seen_this_string = true;
		}
	      else
		/* If all the glyphs of the previous row were inserted
		   by redisplay, it means the previous row was
		   produced from a single newline, which is only
		   possible if that newline came from the same string
		   as the one which produced this ROW.  */
		seen_this_string = true;
	    }
	  else
	    {
	      end = r1->glyphs[TEXT_AREA] - 1;
	      start = end + r1->used[TEXT_AREA];
	      while (end < start
		     && NILP ((end + 1)->object)
		     && (end + 1)->charpos <= 0)
		++end;
	      if (end < start)
		{
		  if (EQ ((end + 1)->object, it->object))
		    seen_this_string = true;
		}
	      else
		seen_this_string = true;
	    }
	}
      /* Take note of each display string that covers a newline only
	 once, the first time we see it.  This is for when a display
	 string includes more than one newline in it.  */
      if (row->ends_in_newline_from_string_p && !seen_this_string)
	{
	  /* If we were scanning the buffer forward when we displayed
	     the string, we want to account for at least one buffer
	     position that belongs to this row (position covered by
	     the display string), so that cursor positioning will
	     consider this row as a candidate when point is at the end
	     of the visual line represented by this row.  This is not
	     required when scanning back, because max_pos will already
	     have a much larger value.  */
	  if (CHARPOS (row->end.pos) > max_pos)
	    inc_both (&max_pos, &max_bpos);
	  SET_TEXT_POS (row->maxpos, max_pos, max_bpos);
	}
      else if (CHARPOS (it->eol_pos) > 0)
	SET_TEXT_POS (row->maxpos,
		      CHARPOS (it->eol_pos) + 1, BYTEPOS (it->eol_pos) + 1);
      else if (row->continued_p)
	{
	  /* If max_pos is different from IT's current position, it
	     means IT->method does not belong to the display element
	     at max_pos.  However, it also means that the display
	     element at max_pos was displayed in its entirety on this
	     line, which is equivalent to saying that the next line
	     starts at the next buffer position.  */
	  if (IT_CHARPOS (*it) == max_pos && it->method != GET_FROM_BUFFER)
	    SET_TEXT_POS (row->maxpos, max_pos, max_bpos);
	  else
	    {
	      inc_both (&max_pos, &max_bpos);
	      SET_TEXT_POS (row->maxpos, max_pos, max_bpos);
	    }
	}
      else if (row->truncated_on_right_p)
	/* display_line already called reseat_at_next_visible_line_start,
	   which puts the iterator at the beginning of the next line, in
	   the logical order. */
	row->maxpos = it->current.pos;
      else if (max_pos == min_pos && it->method != GET_FROM_BUFFER)
	/* A line that is entirely from a string/image/stretch...  */
	row->maxpos = row->minpos;
      else
	emacs_abort ();
    }
  else
    row->maxpos = it->current.pos;
}

/* Like display_count_lines, but capable of counting outside of the
   current narrowed region.  */
static ptrdiff_t
display_count_lines_logically (ptrdiff_t start_byte, ptrdiff_t limit_byte,
			       ptrdiff_t count, ptrdiff_t *byte_pos_ptr)
{
  if (!display_line_numbers_widen || (BEGV == BEG && ZV == Z))
    return display_count_lines (start_byte, limit_byte, count, byte_pos_ptr);

  ptrdiff_t val;
  specpdl_ref pdl_count = SPECPDL_INDEX ();
  record_unwind_protect (save_restriction_restore, save_restriction_save ());
  Fwiden ();
  val = display_count_lines (start_byte, limit_byte, count, byte_pos_ptr);
  unbind_to (pdl_count, Qnil);
  return val;
}

/* Count the number of screen lines in window IT->w between character
   position IT_CHARPOS(*IT) and the line showing that window's point.  */
static ptrdiff_t
display_count_lines_visually (struct it *it)
{
  struct it tem_it;
  ptrdiff_t to;
  struct text_pos from;

  /* If we already calculated a relative line number, use that.  This
     trick relies on the fact that visual lines (a.k.a. "glyph rows")
     are laid out sequentially, one by one, for each sequence of calls
     to display_line or other similar function that follows a call to
     init_iterator.  */
  if (it->lnum_bytepos > 0)
    return it->lnum + 1;
  else
    {
      specpdl_ref count = SPECPDL_INDEX ();

      if (IT_CHARPOS (*it) <= PT)
	{
	  from = it->current.pos;
	  to = PT;
	}
      else
	{
	  SET_TEXT_POS (from, PT, PT_BYTE);
	  to = IT_CHARPOS (*it);
	}
      /* Need to disable visual mode temporarily, since otherwise the
	 call to move_it_to below and inside start_display will cause
	 infinite recursion.  */
      specbind (Qdisplay_line_numbers, Qrelative);
      start_display (&tem_it, it->w, from);
      /* Some redisplay optimizations could invoke us very far from
	 PT, which will make the caller painfully slow.  There should
	 be no need to go too far beyond the window's bottom, as any
	 such optimization will fail to show point anyway.  */
      move_it_to (&tem_it, to, -1,
		  tem_it.last_visible_y
		  + (SCROLL_LIMIT + 10) * FRAME_LINE_HEIGHT (tem_it.f),
		  -1, MOVE_TO_POS | MOVE_TO_Y);
      unbind_to (count, Qnil);
      return IT_CHARPOS (*it) <= PT ? -tem_it.vpos : tem_it.vpos;
    }
}

/* Produce the line-number glyphs for the current glyph_row.  If
   IT->glyph_row is non-NULL, populate the row with the produced
   glyphs.  */
static void
maybe_produce_line_number (struct it *it)
{
  ptrdiff_t last_line = it->lnum;
  ptrdiff_t start_from, bytepos;
  ptrdiff_t this_line;
  bool first_time = false;
  ptrdiff_t beg_byte;
  ptrdiff_t z_byte;
  bool line_numbers_wide;
  void *itdata = bidi_shelve_cache ();

  if (display_line_numbers_offset
      && !display_line_numbers_widen
      && !EQ (Vdisplay_line_numbers, Qvisual)
      && !EQ (Vdisplay_line_numbers, Qrelative))
    line_numbers_wide = true;
  else
    line_numbers_wide = display_line_numbers_widen;

  beg_byte = line_numbers_wide ? BEG_BYTE : BEGV_BYTE;
  z_byte = line_numbers_wide ? Z_BYTE : ZV_BYTE;

  if (EQ (Vdisplay_line_numbers, Qvisual))
    this_line = display_count_lines_visually (it);
  else
    {
      if (!last_line)
	{
	  /* If possible, reuse data cached by line-number-mode.  */
	  if (it->w->base_line_number > 0
	      && it->w->base_line_pos > 0
	      && it->w->base_line_pos <= IT_CHARPOS (*it)
	      /* line-number-mode always displays narrowed line
		 numbers, so we cannot use its data if the user wants
		 line numbers that disregard narrowing, or if the
		 buffer's narrowing has just changed.  */
	      && !(line_numbers_wide
		   && (BEG_BYTE != BEGV_BYTE || Z_BYTE != ZV_BYTE))
	      && !current_buffer->clip_changed)
	    {
	      start_from = CHAR_TO_BYTE (it->w->base_line_pos);
	      last_line = it->w->base_line_number - 1;
	    }
	  else
	    start_from = beg_byte;
	  if (!it->lnum_bytepos)
	    first_time = true;
	}
      else
	start_from = it->lnum_bytepos;

      /* Paranoia: what if someone changes the narrowing since the
	 last time display_line was called?  Shouldn't really happen,
	 but who knows what some crazy Lisp invoked by :eval could do?  */
      if (!(beg_byte <= start_from && start_from <= z_byte))
	{
	  last_line = 0;
	  start_from = beg_byte;
	}

      this_line =
	last_line + display_count_lines_logically (start_from,
						   IT_BYTEPOS (*it),
						   IT_CHARPOS (*it), &bytepos);
      eassert (this_line > 0 || (this_line == 0 && start_from == beg_byte));
      eassert (bytepos == IT_BYTEPOS (*it));
    }

  /* Record the line number information.  */
  if (this_line != last_line || !it->lnum_bytepos)
    {
      it->lnum = this_line;
      it->lnum_bytepos = IT_BYTEPOS (*it);
    }

  /* Produce the glyphs for the line number.  */
  struct it tem_it;
  char lnum_buf[INT_STRLEN_BOUND (ptrdiff_t) + 1];
  bool beyond_zv = IT_BYTEPOS (*it) >= ZV_BYTE ? true : false;
  ptrdiff_t lnum_offset = -1; /* to produce 1-based line numbers */
  int lnum_face_id = merge_faces (it->w, Qline_number, 0, DEFAULT_FACE_ID);
  int current_lnum_face_id
    = merge_faces (it->w, Qline_number_current_line, 0, DEFAULT_FACE_ID);
  /* From here onwards, we must prevent freeing realized faces, because
     we are using the above 2 face IDs for the glyphs we produce.  */
  bool save_free_realized_faces = inhibit_free_realized_faces;
  inhibit_free_realized_faces = true;
  /* Compute point's line number if needed.  */
  if ((EQ (Vdisplay_line_numbers, Qrelative)
       || EQ (Vdisplay_line_numbers, Qvisual)
       || lnum_face_id != current_lnum_face_id)
      && !it->pt_lnum)
    {
      ptrdiff_t ignored;
      if (PT_BYTE > it->lnum_bytepos && !EQ (Vdisplay_line_numbers, Qvisual))
	it->pt_lnum =
	  this_line + display_count_lines_logically (it->lnum_bytepos, PT_BYTE,
						     PT, &ignored);
      else
	it->pt_lnum = display_count_lines_logically (beg_byte, PT_BYTE, PT,
						     &ignored);
    }
  /* Compute the required width if needed.  */
  if (!it->lnum_width)
    {
      if (FIXNATP (Vdisplay_line_numbers_width))
	it->lnum_width = XFIXNAT (Vdisplay_line_numbers_width);

      /* Max line number to be displayed cannot be more than the one
	 corresponding to the last row of the desired matrix.  */
      ptrdiff_t max_lnum;

      if (NILP (Vdisplay_line_numbers_current_absolute)
	  && (EQ (Vdisplay_line_numbers, Qrelative)
	      || EQ (Vdisplay_line_numbers, Qvisual)))
	/* We subtract one more because the current line is always
	   zero in this mode.  */
	max_lnum = it->w->desired_matrix->nrows - 2;
      else if (EQ (Vdisplay_line_numbers, Qvisual))
	max_lnum = it->pt_lnum + it->w->desired_matrix->nrows - 1;
      else
	max_lnum = this_line + it->w->desired_matrix->nrows - 1 - it->vpos;
      max_lnum = max (1, max_lnum);
      it->lnum_width = max (it->lnum_width, log10 (max_lnum) + 1);
      eassert (it->lnum_width > 0);
    }
  if (EQ (Vdisplay_line_numbers, Qrelative))
    lnum_offset = it->pt_lnum;
  else if (EQ (Vdisplay_line_numbers, Qvisual))
    lnum_offset = 0;
  else if (display_line_numbers_offset)
    lnum_offset -= display_line_numbers_offset;

  /* Under 'relative', display the absolute line number for the
     current line, unless the user requests otherwise.  */
  ptrdiff_t lnum_to_display = eabs (this_line - lnum_offset);
  if ((EQ (Vdisplay_line_numbers, Qrelative)
       || EQ (Vdisplay_line_numbers, Qvisual))
      && lnum_to_display == 0
      && !NILP (Vdisplay_line_numbers_current_absolute))
    lnum_to_display = it->pt_lnum + 1;
  /* In L2R rows we need to append the blank separator, in R2L
     rows we need to prepend it.  But this function is usually
     called when no display elements were produced from the
     following line, so the paragraph direction might be unknown.
     Therefore we cheat and add 2 blanks, one on either side.  */
  pint2str (lnum_buf, it->lnum_width + 1, lnum_to_display);
  strcat (lnum_buf, " ");

  /* Setup for producing the glyphs.  */
  init_iterator (&tem_it, it->w, -1, -1, &scratch_glyph_row,
		 /* FIXME: Use specialized face.  */
		 DEFAULT_FACE_ID);
  scratch_glyph_row.reversed_p = false;
  scratch_glyph_row.used[TEXT_AREA] = 0;
  SET_TEXT_POS (tem_it.position, 0, 0);
  tem_it.avoid_cursor_p = true;
  tem_it.bidi_p = true;
  tem_it.bidi_it.type = WEAK_EN;
  /* According to UAX#9, EN goes up 2 levels in L2R paragraph and
     1 level in R2L paragraphs.  Emulate that, assuming we are in
     an L2R paragraph.  */
  tem_it.bidi_it.resolved_level = 2;

  /* We must leave space for 2 glyphs for continuation and truncation,
     and at least one glyph for buffer text.  */
  int width_limit =
    tem_it.last_visible_x - tem_it.first_visible_x
    - 3 * FRAME_COLUMN_WIDTH (it->f);

  tem_it.face_id = lnum_face_id;
  /* Avoid displaying any face other than line-number on
     empty lines beyond EOB.  */
  if (lnum_face_id != current_lnum_face_id
      && (EQ (Vdisplay_line_numbers, Qvisual)
	  ? this_line == 0
	  : this_line == it->pt_lnum)
      && it->what != IT_EOB)
    tem_it.face_id = current_lnum_face_id;
  else if (!beyond_zv)
    {
      if (display_line_numbers_major_tick > 0
	  && (lnum_to_display % display_line_numbers_major_tick == 0))
	tem_it.face_id = merge_faces (it->w, Qline_number_major_tick,
				      0, DEFAULT_FACE_ID);
      else if (display_line_numbers_minor_tick > 0
	       && (lnum_to_display % display_line_numbers_minor_tick == 0))
	tem_it.face_id = merge_faces (it->w, Qline_number_minor_tick,
				      0, DEFAULT_FACE_ID);
    }

  /* Produce glyphs for the line number in a scratch glyph_row.  */
  for (const char *p = lnum_buf; *p; p++)
    {
      /* For continuation lines and lines after ZV, instead of a line
	 number, produce a blank prefix of the same width.  */
      if (beyond_zv
	  /* Don't display the same line number more than once.  */
	  || (!EQ (Vdisplay_line_numbers, Qvisual)
	      && (it->continuation_lines_width > 0
		  || (this_line == last_line && !first_time))))
	tem_it.c = tem_it.char_to_display = ' ';
      else
	tem_it.c = tem_it.char_to_display = *p;
      tem_it.len = 1;
      /* Make sure these glyphs will have a "position" of -1.  */
      SET_TEXT_POS (tem_it.position, -1, -1);
      PRODUCE_GLYPHS (&tem_it);

      /* Stop producing glyphs, and refrain from producing the line
	 number, if we don't have enough space on this line.  */
      if (tem_it.current_x >= width_limit)
	{
	  it->lnum_width = 0;
	  it->lnum_pixel_width = 0;
	  bidi_unshelve_cache (itdata, false);
	  inhibit_free_realized_faces = save_free_realized_faces;
	  return;
	}
    }

  inhibit_free_realized_faces = save_free_realized_faces;

  /* Record the width in pixels we need for the line number display.  */
  it->lnum_pixel_width = tem_it.current_x;
  /* Copy the produced glyphs into IT's glyph_row.  */
  struct glyph *g = scratch_glyph_row.glyphs[TEXT_AREA];
  struct glyph *e = g + scratch_glyph_row.used[TEXT_AREA];
  struct glyph *p = it->glyph_row ? it->glyph_row->glyphs[TEXT_AREA] : NULL;
  short *u = it->glyph_row ? &it->glyph_row->used[TEXT_AREA] : NULL;

  eassert (it->glyph_row == NULL || it->glyph_row->used[TEXT_AREA] == 0);

  for ( ; g < e; g++)
    {
      it->current_x += g->pixel_width;
      /* The following is important when this function is called
	 from move_it_in_display_line_to: HPOS is incremented only
	 when we are in the visible portion of the glyph row.  */
      if (it->current_x > it->first_visible_x)
	it->hpos++;
      if (p)
	{
	  *p++ = *g;
	  (*u)++;
	}
    }

  /* Update IT's metrics due to glyphs produced for line numbers.
     Don't do that for rows beyond ZV, to avoid displaying a cursor of
     different dimensions there.  */
  if (!beyond_zv)
    {
      if (it->glyph_row)
	{
	  struct glyph_row *row = it->glyph_row;

	  it->max_ascent = max (row->ascent, tem_it.max_ascent);
	  it->max_descent = max (row->height - row->ascent, tem_it.max_descent);
	  it->max_phys_ascent = max (row->phys_ascent, tem_it.max_phys_ascent);
	  it->max_phys_descent = max (row->phys_height - row->phys_ascent,
				      tem_it.max_phys_descent);
	}
      else
	{
	  it->max_ascent = max (it->max_ascent, tem_it.max_ascent);
	  it->max_descent = max (it->max_descent, tem_it.max_descent);
	  it->max_phys_ascent = max (it->max_phys_ascent,
				     tem_it.max_phys_ascent);
	  it->max_phys_descent = max (it->max_phys_descent,
				      tem_it.max_phys_descent);
	}
    }

  it->line_number_produced_p = true;

  bidi_unshelve_cache (itdata, false);
}

/* Return true if this glyph row needs a line number to be produced
   for it.  */
static bool
should_produce_line_number (struct it *it)
{
  if (NILP (Vdisplay_line_numbers))
    return false;

  /* Don't display line numbers in minibuffer windows.  */
  if (MINI_WINDOW_P (it->w))
    return false;

#ifdef HAVE_WINDOW_SYSTEM
  /* Don't display line number in tooltip frames.  */
  if (FRAME_TOOLTIP_P (XFRAME (WINDOW_FRAME (it->w))))
    return false;
#endif

  /* If the character at current position has a non-nil special
     property, disable line numbers for this row.  This is for
     packages such as company-mode, which need this for their tricky
     layout, where line numbers get in the way.  */
  Lisp_Object val = Fget_char_property (make_fixnum (IT_CHARPOS (*it)),
					Qdisplay_line_numbers_disable,
					it->window);
  /* For ZV, we need to also look in empty overlays at that point,
     because get-char-property always returns nil for ZV, except if
     the property is in 'default-text-properties'.  */
  if (NILP (val) && IT_CHARPOS (*it) >= ZV)
    val = disable_line_numbers_overlay_at_eob ();
  return NILP (val) ? true : false;
}

/* Return true if ROW has no glyphs except those inserted by the
   display engine.  This is needed for indicate-empty-lines and
   similar features when the glyph row starts with glyphs which didn't
   come from buffer or string.  */
static bool
row_text_area_empty (struct glyph_row *row)
{
  if (!row->reversed_p)
    {
      for (struct glyph *g = row->glyphs[TEXT_AREA];
	   g < row->glyphs[TEXT_AREA] + row->used[TEXT_AREA];
	   g++)
	if (!NILP (g->object) || g->charpos > 0)
	  return false;
    }
  else
    {
      for (struct glyph *g = row->glyphs[TEXT_AREA] + row->used[TEXT_AREA] - 1;
	   g > row->glyphs[TEXT_AREA];
	   g--)
	if (!NILP ((g - 1)->object) || (g - 1)->charpos > 0)
	  return false;
    }

  return true;
}

/* Construct the glyph row IT->glyph_row in the desired matrix of
   IT->w from text at the current position of IT.  See dispextern.h
   for an overview of struct it.  Value is true if
   IT->glyph_row displays text, as opposed to a line displaying ZV
   only.  CURSOR_VPOS is the window-relative vertical position of
   the glyph row displaying the cursor, or -1 if unknown.  */

static bool
display_line (struct it *it, int cursor_vpos)
{
  struct glyph_row *row = it->glyph_row;
  Lisp_Object overlay_arrow_string;
  struct it wrap_it;
  void *wrap_data = NULL;
  bool may_wrap = false;
  int wrap_x UNINIT;
  int wrap_row_used = -1;
  int wrap_row_ascent UNINIT, wrap_row_height UNINIT;
  int wrap_row_phys_ascent UNINIT, wrap_row_phys_height UNINIT;
  int wrap_row_extra_line_spacing UNINIT;
  ptrdiff_t wrap_row_min_pos UNINIT, wrap_row_min_bpos UNINIT;
  ptrdiff_t wrap_row_max_pos UNINIT, wrap_row_max_bpos UNINIT;
  int cvpos;
  ptrdiff_t min_pos = ZV + 1, max_pos = 0;
  ptrdiff_t min_bpos UNINIT, max_bpos UNINIT;
  bool pending_handle_line_prefix = false;
  int tab_line = window_wants_tab_line (it->w);
  int header_line = window_wants_header_line (it->w);
  bool hscroll_this_line = (cursor_vpos >= 0
			    && it->vpos == cursor_vpos - tab_line - header_line
			    && hscrolling_current_line_p (it->w));
  int first_visible_x = it->first_visible_x;
  int last_visible_x = it->last_visible_x;
  int x_incr = 0;

  /* We always start displaying at hpos zero even if hscrolled.  */
  eassert (it->hpos == 0 && it->current_x == 0);

  if (MATRIX_ROW_VPOS (row, it->w->desired_matrix)
      >= it->w->desired_matrix->nrows)
    {
      it->w->nrows_scale_factor++;
      it->f->fonts_changed = true;
      return false;
    }

  /* Clear the result glyph row and enable it.  */
  prepare_desired_row (it->w, row, false);

  row->y = it->current_y;
  row->start = it->start;
  row->continuation_lines_width = it->continuation_lines_width;
  row->displays_text_p = true;
  row->starts_in_middle_of_char_p = it->starts_in_middle_of_char_p;
  it->starts_in_middle_of_char_p = false;
  it->stretch_adjust = 0;
  it->line_number_produced_p = false;

  /* If we are going to display the cursor's line, account for the
     hscroll of that line.  We subtract the window's min_hscroll,
     because that was already accounted for in init_iterator.  */
  if (hscroll_this_line)
    x_incr =
      (window_hscroll_limited (it->w, it->f) - it->w->min_hscroll)
      * FRAME_COLUMN_WIDTH (it->f);

  bool line_number_needed = should_produce_line_number (it);

  /* Move over display elements that are not visible because we are
     hscrolled.  This may stop at an x-position < first_visible_x
     if the first glyph is partially visible or if we hit a line end.  */
  if (it->current_x < it->first_visible_x + x_incr)
    {
      enum move_it_result move_result;

      this_line_min_pos = row->start.pos;
      if (hscroll_this_line)
	{
	  it->first_visible_x += x_incr;
	  it->last_visible_x  += x_incr;
	}
      if (current_buffer->long_line_optimizations_p
	  && it->line_wrap == TRUNCATE
	  && window_hscroll_limited (it->w, it->f) > large_hscroll_threshold)
	{
	  /* Special optimization for very long and truncated lines
	     which are hscrolled far to the left: jump directly to the
	     (approximate) position that is visible, instead of slowly
	     walking there.  */
	  ptrdiff_t chars_to_skip =
	    it->first_visible_x / FRAME_COLUMN_WIDTH (it->f);
	  move_result = fast_move_it_horizontally (it, chars_to_skip);

	  if (move_result == MOVE_X_REACHED)
	    it->current_x = it->first_visible_x;
	  else	/* use arbitrary value < first_visible_x */
	    it->current_x = it->first_visible_x - FRAME_COLUMN_WIDTH (it->f);
	}
      else
	move_result = move_it_in_display_line_to (it, ZV, it->first_visible_x,
						  MOVE_TO_POS | MOVE_TO_X);
      /* If we are under a large hscroll, move_it_in_display_line_to
	 could hit the end of the line without reaching
	 first_visible_x.  Pretend that we did reach it.  This is
	 especially important on a TTY, where we will call
	 extend_face_to_end_of_line, which needs to know how many
	 blank glyphs to produce.  */
      if (it->current_x < it->first_visible_x
	  && (move_result == MOVE_NEWLINE_OR_CR
	      || move_result == MOVE_POS_MATCH_OR_ZV))
	it->current_x = it->first_visible_x;

      /* In case move_it_in_display_line_to above "produced" the line
	 number.  */
      it->line_number_produced_p = false;

      /* Record the smallest positions seen while we moved over
	 display elements that are not visible.  This is needed by
	 redisplay_internal for optimizing the case where the cursor
	 stays inside the same line.  The rest of this function only
	 considers positions that are actually displayed, so
	 RECORD_MAX_MIN_POS will not otherwise record positions that
	 are hscrolled to the left of the left edge of the window.  */
      min_pos = CHARPOS (this_line_min_pos);
      min_bpos = BYTEPOS (this_line_min_pos);

      /* Produce line number, if needed.  */
      if (line_number_needed)
	maybe_produce_line_number (it);
    }
  else if (it->area == TEXT_AREA)
    {
      /* Line numbers should precede the line-prefix or wrap-prefix.  */
      if (line_number_needed)
	maybe_produce_line_number (it);

      /* We only do this when not calling move_it_in_display_line_to
	 above, because that function calls itself handle_line_prefix.  */
      handle_line_prefix (it);
    }
  else
    {
      /* Line-prefix and wrap-prefix are always displayed in the text
	 area.  But if this is the first call to display_line after
	 init_iterator, the iterator might have been set up to write
	 into a marginal area, e.g. if the line begins with some
	 display property that writes to the margins.  So we need to
	 wait with the call to handle_line_prefix until whatever
	 writes to the margin has done its job.  */
      pending_handle_line_prefix = true;
    }

  /* Get the initial row height.  This is either the height of the
     text hscrolled, if there is any, or zero.  */
  row->ascent = it->max_ascent;
  row->height = it->max_ascent + it->max_descent;
  row->phys_ascent = it->max_phys_ascent;
  row->phys_height = it->max_phys_ascent + it->max_phys_descent;
  row->extra_line_spacing = it->max_extra_line_spacing;

/* Utility macro to record max and min buffer positions seen until now.  */
#define RECORD_MAX_MIN_POS(IT)					\
  do								\
    {								\
      bool composition_p					\
	= !STRINGP ((IT)->string) && ((IT)->what == IT_COMPOSITION); \
      ptrdiff_t current_pos =					\
	composition_p ? (IT)->cmp_it.charpos			\
		      : IT_CHARPOS (*(IT));			\
      ptrdiff_t current_bpos =					\
	composition_p ? CHAR_TO_BYTE (current_pos)		\
		      : IT_BYTEPOS (*(IT));			\
      if (current_pos < min_pos)				\
	{							\
	  min_pos = current_pos;				\
	  min_bpos = current_bpos;				\
	}							\
      if (IT_CHARPOS (*it) > max_pos)				\
	{							\
	  max_pos = IT_CHARPOS (*it);				\
	  max_bpos = IT_BYTEPOS (*it);				\
	}							\
    }								\
  while (false)

  /* Loop generating characters.  The loop is left with IT on the next
     character to display.  */
  while (true)
    {
      int n_glyphs_before, hpos_before, x_before;
      int x, nglyphs;
      int ascent = 0, descent = 0, phys_ascent = 0, phys_descent = 0;

      /* Retrieve the next thing to display.  Value is false if end of
	 buffer reached.  */
      if (!get_next_display_element (it))
	{
	  bool row_has_glyphs = false;
	  /* Maybe add a space at the end of this line that is used to
	     display the cursor there under X.  Set the charpos of the
	     first glyph of blank lines not corresponding to any text
	     to -1.  */
	  if (IT_OVERFLOW_NEWLINE_INTO_FRINGE (it))
	    row->exact_window_width_line_p = true;
	  else if ((append_space_for_newline (it, true)
		    && row->used[TEXT_AREA] == 1)
		   || row->used[TEXT_AREA] == 0
		   || (row_has_glyphs = row_text_area_empty (row)))
	    {
	      row->glyphs[TEXT_AREA]->charpos = -1;
	      /* Don't reset the displays_text_p flag if we are
		 displaying line numbers or line-prefix.  */
	      if (!row_has_glyphs)
		row->displays_text_p = false;

	      if (!NILP (BVAR (XBUFFER (it->w->contents), indicate_empty_lines))
		  && (!MINI_WINDOW_P (it->w)))
		row->indicate_empty_line_p = true;
	    }

	  it->continuation_lines_width = 0;
	  /* Reset those iterator values set from display property
	     values.  This is for the case when the display property
	     ends at ZV, and is not a replacing property, so pop_it is
	     not called.  */
	  it->font_height = Qnil;
	  it->voffset = 0;
	  row->ends_at_zv_p = true;
	  /* A row that displays right-to-left text must always have
	     its last face extended all the way to the end of line,
	     even if this row ends in ZV, because we still write to
	     the screen left to right.  We also need to extend the
	     last face if the default face is remapped to some
	     different face, otherwise the functions that clear
	     portions of the screen will clear with the default face's
	     background color.  */
	  if (row->reversed_p
	      || lookup_basic_face (it->w, it->f, DEFAULT_FACE_ID)
              != DEFAULT_FACE_ID)
	    extend_face_to_end_of_line (it);
	  break;
	}

      /* Now, get the metrics of what we want to display.  This also
	 generates glyphs in `row' (which is IT->glyph_row).  */
      n_glyphs_before = row->used[TEXT_AREA];
      x = it->current_x;

      /* Remember the line height so far in case the next element doesn't
	 fit on the line.  */
      if (it->line_wrap != TRUNCATE)
	{
	  ascent = it->max_ascent;
	  descent = it->max_descent;
	  phys_ascent = it->max_phys_ascent;
	  phys_descent = it->max_phys_descent;

	  if (it->line_wrap == WORD_WRAP && it->area == TEXT_AREA)
	    {
              bool next_may_wrap = may_wrap;
              /* Can we wrap after this character?  */
              if (char_can_wrap_after (it))
		next_may_wrap = true;
              else
                next_may_wrap = false;
	      /* Can we wrap here? */
	      if (may_wrap && char_can_wrap_before (it))
		{
		  SAVE_IT (wrap_it, *it, wrap_data);
		  wrap_x = x;
		  wrap_row_used = row->used[TEXT_AREA];
		  wrap_row_ascent = row->ascent;
		  wrap_row_height = row->height;
		  wrap_row_phys_ascent = row->phys_ascent;
		  wrap_row_phys_height = row->phys_height;
		  wrap_row_extra_line_spacing = row->extra_line_spacing;
		  wrap_row_min_pos = min_pos;
		  wrap_row_min_bpos = min_bpos;
		  wrap_row_max_pos = max_pos;
		  wrap_row_max_bpos = max_bpos;
		}
	      /* Update may_wrap for the next iteration.  */
              may_wrap = next_may_wrap;
	    }
	}

      PRODUCE_GLYPHS (it);

      /* If this display element was in marginal areas, continue with
	 the next one.  */
      if (it->area != TEXT_AREA)
	{
	  row->ascent = max (row->ascent, it->max_ascent);
	  row->height = max (row->height, it->max_ascent + it->max_descent);
	  row->phys_ascent = max (row->phys_ascent, it->max_phys_ascent);
	  row->phys_height = max (row->phys_height,
				  it->max_phys_ascent + it->max_phys_descent);
	  row->extra_line_spacing = max (row->extra_line_spacing,
					 it->max_extra_line_spacing);
	  set_iterator_to_next (it, true);
	  /* If we didn't handle the line/wrap prefix above, and the
	     call to set_iterator_to_next just switched to TEXT_AREA,
	     process the prefix now.  */
	  if (it->area == TEXT_AREA && pending_handle_line_prefix)
	    {
	      /* Line numbers should precede the line-prefix or wrap-prefix.  */
	      if (line_number_needed)
		maybe_produce_line_number (it);

	      pending_handle_line_prefix = false;
	      handle_line_prefix (it);
	    }
	  continue;
	}

      /* Does the display element fit on the line?  If we truncate
	 lines, we should draw past the right edge of the window.  If
	 we don't truncate, we want to stop so that we can display the
	 continuation glyph before the right margin.  If lines are
	 continued, there are two possible strategies for characters
	 resulting in more than 1 glyph (e.g. tabs): Display as many
	 glyphs as possible in this line and leave the rest for the
	 continuation line, or display the whole element in the next
	 line.  Original redisplay did the former, so we do it also.  */
      nglyphs = row->used[TEXT_AREA] - n_glyphs_before;
      hpos_before = it->hpos;
      x_before = x;

      if (/* Not a newline.  */
	  nglyphs > 0
	  /* Glyphs produced fit entirely in the line.  */
	  && it->current_x < it->last_visible_x)
	{
	  it->hpos += nglyphs;
	  row->ascent = max (row->ascent, it->max_ascent);
	  row->height = max (row->height, it->max_ascent + it->max_descent);
	  row->phys_ascent = max (row->phys_ascent, it->max_phys_ascent);
	  row->phys_height = max (row->phys_height,
				  it->max_phys_ascent + it->max_phys_descent);
	  row->extra_line_spacing = max (row->extra_line_spacing,
					 it->max_extra_line_spacing);
	  if (it->current_x - it->pixel_width < it->first_visible_x
	      /* When line numbers are displayed, row->x should not be
		 offset, as the first glyph after the line number can
		 never be partially visible.  */
	      && !line_number_needed
	      /* In R2L rows, we arrange in extend_face_to_end_of_line
		 to add a right offset to the line, by a suitable
		 change to the stretch glyph that is the leftmost
		 glyph of the line.  */
	      && !row->reversed_p)
	    row->x = x - it->first_visible_x;
	  /* Record the maximum and minimum buffer positions seen so
	     far in glyphs that will be displayed by this row.  */
	  if (it->bidi_p)
	    RECORD_MAX_MIN_POS (it);
	}
      else
	{
	  int i, new_x;
	  struct glyph *glyph;

	  for (i = 0; i < nglyphs; ++i, x = new_x)
	    {
	      /* Identify the glyphs added by the last call to
		 PRODUCE_GLYPHS.  In R2L rows, they are prepended to
		 the previous glyphs.  */
	      if (!row->reversed_p)
		glyph = row->glyphs[TEXT_AREA] + n_glyphs_before + i;
	      else
		glyph = row->glyphs[TEXT_AREA] + nglyphs - 1 - i;
	      new_x = x + glyph->pixel_width;

	      if (/* Lines are continued.  */
		  it->line_wrap != TRUNCATE
		  && (/* Glyph doesn't fit on the line.  */
		      new_x > it->last_visible_x
		      /* Or it fits exactly on a window system frame.  */
		      || (new_x == it->last_visible_x
			  && FRAME_WINDOW_P (it->f)
			  && (row->reversed_p
			      ? WINDOW_LEFT_FRINGE_WIDTH (it->w)
			      : WINDOW_RIGHT_FRINGE_WIDTH (it->w)))))
		{
		  /* End of a continued line.  */

		  if (it->hpos == 0
		      || (new_x == it->last_visible_x
			  && FRAME_WINDOW_P (it->f)
			  && (row->reversed_p
			      ? WINDOW_LEFT_FRINGE_WIDTH (it->w)
			      : WINDOW_RIGHT_FRINGE_WIDTH (it->w))))
		    {
		      /* Current glyph is the only one on the line or
			 fits exactly on the line.  We must continue
			 the line because we can't draw the cursor
			 after the glyph.  */
		      row->continued_p = true;
		      it->current_x = new_x;
		      it->continuation_lines_width += new_x;
		      ++it->hpos;
		      if (i == nglyphs - 1)
			{
			  /* If line-wrap is on, check if a previous
			     wrap point was found.  */
			  if (!IT_OVERFLOW_NEWLINE_INTO_FRINGE (it)
			      && wrap_row_used > 0 /* Found.  */
			      /* Even if there is a previous wrap
				 point, continue the line here as
				 usual, if (i) the previous character
				 allows wrapping after it, AND (ii)
				 the current character allows wrapping
				 before it.  Because this is a valid
				 break point, we can just continue to
				 the next line at here, there is no
				 need to wrap early at the previous
				 wrap point.  */
			      && (!may_wrap || !char_can_wrap_before (it)))
			    goto back_to_wrap;

			  /* Record the maximum and minimum buffer
			     positions seen so far in glyphs that will be
			     displayed by this row.  */
			  if (it->bidi_p)
			    RECORD_MAX_MIN_POS (it);
			  set_iterator_to_next (it, true);
			  if (IT_OVERFLOW_NEWLINE_INTO_FRINGE (it))
			    {
			      if (!get_next_display_element (it))
				{
				  row->exact_window_width_line_p = true;
				  it->continuation_lines_width = 0;
				  it->font_height = Qnil;
				  it->voffset = 0;
				  row->continued_p = false;
				  row->ends_at_zv_p = true;
				}
			      else if (ITERATOR_AT_END_OF_LINE_P (it))
				{
				  row->continued_p = false;
				  row->exact_window_width_line_p = true;
				}
			      /* If line-wrap is on, check if a
				 previous wrap point was found.  */
			      else if (wrap_row_used > 0
				       /* Even if there is a previous
					  wrap point, continue the
					  line here as usual, if (i)
					  the previous character was a
					  space or tab AND (ii) the
					  current character is not,
					  AND (iii) the current
					  character allows wrapping
					  before it.  */
				       && (!may_wrap || !char_can_wrap_before (it)))
				goto back_to_wrap;

			    }
			}
		      else if (it->bidi_p)
			RECORD_MAX_MIN_POS (it);
		      if (WINDOW_LEFT_MARGIN_WIDTH (it->w) > 0
			  || WINDOW_RIGHT_MARGIN_WIDTH (it->w) > 0)
			extend_face_to_end_of_line (it);
		    }
		  else if (CHAR_GLYPH_PADDING_P (*glyph)
			   && !FRAME_WINDOW_P (it->f))
		    {
		      /* A padding glyph that doesn't fit on this line.
			 This means the whole character doesn't fit
			 on the line.  */
		      if (row->reversed_p)
			unproduce_glyphs (it, row->used[TEXT_AREA]
					       - n_glyphs_before);
		      row->used[TEXT_AREA] = n_glyphs_before;

		      /* Fill the rest of the row with continuation
			 glyphs like in 20.x.  */
		      while (row->glyphs[TEXT_AREA] + row->used[TEXT_AREA]
			     < row->glyphs[1 + TEXT_AREA])
			produce_special_glyphs (it, IT_CONTINUATION);

		      row->continued_p = true;
		      it->current_x = x_before;
		      it->continuation_lines_width += x_before;

		      /* Restore the height to what it was before the
			 element not fitting on the line.  */
		      it->max_ascent = ascent;
		      it->max_descent = descent;
		      it->max_phys_ascent = phys_ascent;
		      it->max_phys_descent = phys_descent;
		      if (WINDOW_LEFT_MARGIN_WIDTH (it->w) > 0
			  || WINDOW_RIGHT_MARGIN_WIDTH (it->w) > 0)
			extend_face_to_end_of_line (it);
		    }
		  else if (wrap_row_used > 0)
		    {
		    back_to_wrap:
		      if (row->reversed_p)
			unproduce_glyphs (it,
					  row->used[TEXT_AREA] - wrap_row_used);
		      RESTORE_IT (it, &wrap_it, wrap_data);
		      it->continuation_lines_width += wrap_x;
		      row->used[TEXT_AREA] = wrap_row_used;
		      row->ascent = wrap_row_ascent;
		      row->height = wrap_row_height;
		      row->phys_ascent = wrap_row_phys_ascent;
		      row->phys_height = wrap_row_phys_height;
		      row->extra_line_spacing = wrap_row_extra_line_spacing;
		      min_pos = wrap_row_min_pos;
		      min_bpos = wrap_row_min_bpos;
		      max_pos = wrap_row_max_pos;
		      max_bpos = wrap_row_max_bpos;
		      row->continued_p = true;
		      row->ends_at_zv_p = false;
		      row->exact_window_width_line_p = false;

		      /* Make sure that a non-default face is extended
			 up to the right margin of the window.  */
		      extend_face_to_end_of_line (it);
		    }
		  else if ((it->what == IT_CHARACTER
			    || it->what == IT_STRETCH
			    || it->what == IT_COMPOSITION)
			   && it->c == '\t' && FRAME_WINDOW_P (it->f))
		    {
		      /* A TAB that extends past the right edge of the
			 window.  This produces a single glyph on
			 window system frames.  We leave the glyph in
			 this row and let it fill the row, but don't
			 consume the TAB.  */
		      if ((row->reversed_p
			   ? WINDOW_LEFT_FRINGE_WIDTH (it->w)
			   : WINDOW_RIGHT_FRINGE_WIDTH (it->w)) == 0)
			produce_special_glyphs (it, IT_CONTINUATION);
		      it->continuation_lines_width += it->last_visible_x;
		      row->ends_in_middle_of_char_p = true;
		      row->continued_p = true;
		      glyph->pixel_width = it->last_visible_x - x;
		      it->starts_in_middle_of_char_p = true;
		      if (WINDOW_LEFT_MARGIN_WIDTH (it->w) > 0
			  || WINDOW_RIGHT_MARGIN_WIDTH (it->w) > 0)
			extend_face_to_end_of_line (it);
		    }
		  else
		    {
		      /* Something other than a TAB that draws past
			 the right edge of the window.  Restore
			 positions to values before the element.  */
		      if (row->reversed_p)
			unproduce_glyphs (it, row->used[TEXT_AREA]
					       - (n_glyphs_before + i));
		      row->used[TEXT_AREA] = n_glyphs_before + i;

		      /* Display continuation glyphs.  */
		      it->current_x = x_before;
		      it->continuation_lines_width += x;
		      if (!FRAME_WINDOW_P (it->f)
			  || (row->reversed_p
			      ? WINDOW_LEFT_FRINGE_WIDTH (it->w)
			      : WINDOW_RIGHT_FRINGE_WIDTH (it->w)) == 0)
			produce_special_glyphs (it, IT_CONTINUATION);
		      row->continued_p = true;

		      extend_face_to_end_of_line (it);

		      if (nglyphs > 1 && i > 0)
			{
			  row->ends_in_middle_of_char_p = true;
			  it->starts_in_middle_of_char_p = true;
			}

		      /* Restore the height to what it was before the
			 element not fitting on the line.  */
		      it->max_ascent = ascent;
		      it->max_descent = descent;
		      it->max_phys_ascent = phys_ascent;
		      it->max_phys_descent = phys_descent;
		    }

		  break;
		}
	      else if (new_x > it->first_visible_x)
		{
		  /* Increment number of glyphs actually displayed.  */
		  ++it->hpos;

		  /* Record the maximum and minimum buffer positions
		     seen so far in glyphs that will be displayed by
		     this row.  */
		  if (it->bidi_p)
		    RECORD_MAX_MIN_POS (it);

		  if (x < it->first_visible_x && !row->reversed_p
		      && !line_number_needed)
		    /* Glyph is partially visible, i.e. row starts at
		       negative X position.  Don't do that in R2L
		       rows, where we arrange to add a right offset to
		       the line in extend_face_to_end_of_line, by a
		       suitable change to the stretch glyph that is
		       the leftmost glyph of the line.  */
		    row->x = x - it->first_visible_x;
		  /* When the last glyph of an R2L row only fits
		     partially on the line, we need to set row->x to a
		     negative offset, so that the leftmost glyph is
		     the one that is partially visible.  But if we are
		     going to produce the truncation glyph, this will
		     be taken care of in produce_special_glyphs.  */
		  if (row->reversed_p
		      && new_x > it->last_visible_x
		      && !line_number_needed
		      && !(it->line_wrap == TRUNCATE
			   && WINDOW_LEFT_FRINGE_WIDTH (it->w) == 0))
		    {
		      eassert (FRAME_WINDOW_P (it->f));
		      row->x = it->last_visible_x - new_x;
		    }
		}
	      else
		{
		  /* Glyph is completely off the left margin of the
		     window.  This should not happen because of the
		     move_it_in_display_line at the start of this
		     function, unless the text display area of the
		     window is empty.  */
		  eassert (it->first_visible_x <= it->last_visible_x);
		}
	    }
	  /* Even if this display element produced no glyphs at all,
	     we want to record its position.  */
	  if (it->bidi_p && nglyphs == 0)
	    RECORD_MAX_MIN_POS (it);

	  row->ascent = max (row->ascent, it->max_ascent);
	  row->height = max (row->height, it->max_ascent + it->max_descent);
	  row->phys_ascent = max (row->phys_ascent, it->max_phys_ascent);
	  row->phys_height = max (row->phys_height,
				  it->max_phys_ascent + it->max_phys_descent);
	  row->extra_line_spacing = max (row->extra_line_spacing,
					 it->max_extra_line_spacing);

	  /* End of this display line if row is continued.  */
	  if (row->continued_p || row->ends_at_zv_p)
	    break;
	}

    at_end_of_line:
      /* Is this a line end?  If yes, we're also done, after making
	 sure that a non-default face is extended up to the right
	 margin of the window.  */
      if (ITERATOR_AT_END_OF_LINE_P (it))
	{
	  int used_before = row->used[TEXT_AREA];

	  row->ends_in_newline_from_string_p = STRINGP (it->object);

	  /* Add a space at the end of the line that is used to
	     display the cursor there.  */
	  if (!IT_OVERFLOW_NEWLINE_INTO_FRINGE (it))
	    append_space_for_newline (it, false);

	  /* Extend the face to the end of the line.  */
	  extend_face_to_end_of_line (it);

	  /* Make sure we have the position.  */
	  if (used_before == 0)
	    row->glyphs[TEXT_AREA]->charpos = CHARPOS (it->position);

	  /* Record the position of the newline, for use in
	     find_row_edges.  */
	  it->eol_pos = it->current.pos;

	  /* Consume the line end.  This skips over invisible lines.  */
	  set_iterator_to_next (it, true);
	  it->continuation_lines_width = 0;
	  break;
	}

      /* Detect overly-wide wrap-prefixes made of (space ...) display
	 properties.  When such a wrap prefix reaches past the right
	 margin of the window, we need to avoid the call to
	 set_iterator_to_next below, so that it->line_wrap is left at
	 its TRUNCATE value wisely set by handle_line_prefix.
	 Otherwise, set_iterator_to_next will pop the iterator stack,
	 restore it->line_wrap, and redisplay might infloop.  */
      bool overwide_wrap_prefix =
	CONSP (it->object) && EQ (XCAR (it->object), Qspace)
	&& it->sp > 0 && it->method == GET_FROM_STRETCH
	&& it->current_x >= it->last_visible_x
	&& it->continuation_lines_width > 0
	&& it->line_wrap == TRUNCATE && it->stack[0].line_wrap != TRUNCATE;

      /* Proceed with next display element.  Note that this skips
	 over lines invisible because of selective display.  */
      if (!overwide_wrap_prefix)
	set_iterator_to_next (it, true);

      /* If we truncate lines, we are done when the last displayed
	 glyphs reach past the right margin of the window.  */
      if (it->line_wrap == TRUNCATE
	  && ((FRAME_WINDOW_P (it->f)
	       /* Images are preprocessed in produce_image_glyph such
		  that they are cropped at the right edge of the
		  window, so an image glyph will always end exactly at
		  last_visible_x, even if there's no right fringe.  */
	       && ((row->reversed_p
		    ? WINDOW_LEFT_FRINGE_WIDTH (it->w)
		    : WINDOW_RIGHT_FRINGE_WIDTH (it->w))
		   || it->what == IT_IMAGE))
	      ? (it->current_x >= it->last_visible_x)
	      : (it->current_x > it->last_visible_x)))
	{
	  /* Maybe add truncation glyphs.  */
	  if (!FRAME_WINDOW_P (it->f)
	      || (row->reversed_p
		  ? WINDOW_LEFT_FRINGE_WIDTH (it->w)
		  : WINDOW_RIGHT_FRINGE_WIDTH (it->w)) == 0)
	    {
	      int i, n;

	      if (!row->reversed_p)
		{
		  for (i = row->used[TEXT_AREA] - 1; i > 0; --i)
		    if (!CHAR_GLYPH_PADDING_P (row->glyphs[TEXT_AREA][i]))
		      break;
		}
	      else
		{
		  for (i = 0; i < row->used[TEXT_AREA]; i++)
		    if (!CHAR_GLYPH_PADDING_P (row->glyphs[TEXT_AREA][i]))
		      break;
		  /* Remove any padding glyphs at the front of ROW, to
		     make room for the truncation glyphs we will be
		     adding below.  The loop below always inserts at
		     least one truncation glyph, so also remove the
		     last glyph added to ROW.  */
		  unproduce_glyphs (it, i + 1);
		  /* Adjust i for the loop below.  */
		  i = row->used[TEXT_AREA] - (i + 1);
		}

	      /* produce_special_glyphs overwrites the last glyph, so
		 we don't want that if we want to keep that last
		 glyph, which means it's an image.  */
	      if (it->current_x > it->last_visible_x)
		{
		  it->current_x = x_before;
		  if (!FRAME_WINDOW_P (it->f))
		    {
		      for (n = row->used[TEXT_AREA]; i < n; ++i)
			{
			  row->used[TEXT_AREA] = i;
			  produce_special_glyphs (it, IT_TRUNCATION);
			}
		    }
		  else
		    {
		      row->used[TEXT_AREA] = i;
		      produce_special_glyphs (it, IT_TRUNCATION);
		    }
		  it->hpos = hpos_before;
		}
	    }
	  else if (IT_OVERFLOW_NEWLINE_INTO_FRINGE (it))
	    {
	      /* Don't truncate if we can overflow newline into fringe.  */
	      if (!get_next_display_element (it))
		{
		  it->continuation_lines_width = 0;
		  it->font_height = Qnil;
		  it->voffset = 0;
		  row->ends_at_zv_p = true;
		  row->exact_window_width_line_p = true;
		  break;
		}
	      if (ITERATOR_AT_END_OF_LINE_P (it))
		{
		  row->exact_window_width_line_p = true;
		  goto at_end_of_line;
		}
	      it->current_x = x_before;
	      it->hpos = hpos_before;
	    }

	  row->truncated_on_right_p = true;
	  it->continuation_lines_width = 0;
	  reseat_at_next_visible_line_start (it, false);
	  /* We insist below that IT's position be at ZV because in
	     bidi-reordered lines the character at visible line start
	     might not be the character that follows the newline in
	     the logical order.  */
	  if (IT_BYTEPOS (*it) > BEG_BYTE)
	    row->ends_at_zv_p =
	      IT_BYTEPOS (*it) >= ZV_BYTE
	      && (ZV_BYTE <= 1 || FETCH_BYTE (ZV_BYTE - 1) != '\n');
	  else
	    row->ends_at_zv_p = false;
	  break;
	}
    }

  if (wrap_data)
    bidi_unshelve_cache (wrap_data, true);

  /* If line is not empty and hscrolled, maybe insert truncation glyphs
     at the left window margin.  */
  if (it->first_visible_x
      && IT_CHARPOS (*it) != CHARPOS (row->start.pos))
    {
      if (!FRAME_WINDOW_P (it->f)
	  || (((row->reversed_p
		? WINDOW_RIGHT_FRINGE_WIDTH (it->w)
		: WINDOW_LEFT_FRINGE_WIDTH (it->w)) == 0)
	      /* Don't let insert_left_trunc_glyphs overwrite the
		 first glyph of the row if it is an image.  */
	      && row->glyphs[TEXT_AREA]->type != IMAGE_GLYPH))
	insert_left_trunc_glyphs (it);
      row->truncated_on_left_p = true;
    }

  /* Remember the position at which this line ends.

     BIDI Note: any code that needs MATRIX_ROW_START/END_CHARPOS
     cannot be before the call to find_row_edges below, since that is
     where these positions are determined. */
  row->end = it->current;
  if (!it->bidi_p)
    {
      row->minpos = row->start.pos;
      row->maxpos = row->end.pos;
    }
  else
    {
      /* ROW->minpos and ROW->maxpos must be the smallest and
	 `1 + the largest' buffer positions in ROW.  But if ROW was
	 bidi-reordered, these two positions can be anywhere in the
	 row, so we must determine them now.  */
      find_row_edges (it, row, min_pos, min_bpos, max_pos, max_bpos);
    }

  /* If the start of this line is the overlay arrow-position, then
     mark this glyph row as the one containing the overlay arrow.
     This is clearly a mess with variable size fonts.  It would be
     better to let it be displayed like cursors under X.  */
  if ((MATRIX_ROW_DISPLAYS_TEXT_P (row) || !overlay_arrow_seen)
      && (overlay_arrow_string = overlay_arrow_at_row (it, row),
	  !NILP (overlay_arrow_string)))
    {
      /* Overlay arrow in window redisplay is a fringe bitmap.  */
      if (STRINGP (overlay_arrow_string))
	{
	  struct glyph_row *arrow_row
	    = get_overlay_arrow_glyph_row (it->w, overlay_arrow_string);
	  struct glyph *glyph = arrow_row->glyphs[TEXT_AREA];
	  struct glyph *arrow_end = glyph + arrow_row->used[TEXT_AREA];
	  struct glyph *p = row->glyphs[TEXT_AREA];
	  struct glyph *p2, *end;

	  /* Copy the arrow glyphs.  */
	  while (glyph < arrow_end)
	    *p++ = *glyph++;

	  /* Throw away padding glyphs.  */
	  p2 = p;
	  end = row->glyphs[TEXT_AREA] + row->used[TEXT_AREA];
	  while (p2 < end && CHAR_GLYPH_PADDING_P (*p2))
	    ++p2;
	  if (p2 > p)
	    {
	      while (p2 < end)
		*p++ = *p2++;
	      row->used[TEXT_AREA] = p2 - row->glyphs[TEXT_AREA];
	    }
	}
      else
	{
	  eassert (FIXNUMP (overlay_arrow_string));
	  row->overlay_arrow_bitmap = XFIXNUM (overlay_arrow_string);
	}
      overlay_arrow_seen = true;
    }

  /* Highlight trailing whitespace.  */
  if (!NILP (Vshow_trailing_whitespace))
    highlight_trailing_whitespace (it);

  /* Compute pixel dimensions of this line.  */
  compute_line_metrics (it);

  /* Implementation note: No changes in the glyphs of ROW or in their
     faces can be done past this point, because compute_line_metrics
     computes ROW's hash value and stores it within the glyph_row
     structure.  */

  /* Record whether this row ends inside an ellipsis.  */
  row->ends_in_ellipsis_p
    = (it->method == GET_FROM_DISPLAY_VECTOR
       && it->ellipsis_p);

  /* Save fringe bitmaps in this row.  */
  row->left_user_fringe_bitmap = it->left_user_fringe_bitmap;
  row->left_user_fringe_face_id = it->left_user_fringe_face_id;
  row->right_user_fringe_bitmap = it->right_user_fringe_bitmap;
  row->right_user_fringe_face_id = it->right_user_fringe_face_id;

  it->left_user_fringe_bitmap = 0;
  it->left_user_fringe_face_id = 0;
  it->right_user_fringe_bitmap = 0;
  it->right_user_fringe_face_id = 0;

  /* When they turn off tooltip-mode on a GUI frame, we call 'message'
     with message-truncate-lines bound to non-nil, which produces
     truncation bitmaps on the fringe.  Force redrawing of the fringes
     in that case, to make sure the fringe bitmaps are removed when a
     shorter message is displayed.  */
  if (MINI_WINDOW_P (it->w) && it->line_wrap == TRUNCATE
      && FRAME_WINDOW_P (it->f) && !cursor_in_echo_area)
    row->redraw_fringe_bitmaps_p = true;

  /* Maybe set the cursor.  */
  cvpos = it->w->cursor.vpos;
  if ((cvpos < 0
       /* In bidi-reordered rows, keep checking for proper cursor
	  position even if one has been found already, because buffer
	  positions in such rows change non-linearly with ROW->VPOS,
	  when a line is continued.  One exception: when we are at ZV,
	  display cursor on the first suitable glyph row, since all
	  the empty rows after that also have their position set to ZV.  */
       /* FIXME: Revisit this when glyph ``spilling'' in continuation
	  lines' rows is implemented for bidi-reordered rows.  */
       || (it->bidi_p
	   && !MATRIX_ROW (it->w->desired_matrix, cvpos)->ends_at_zv_p))
      && PT >= MATRIX_ROW_START_CHARPOS (row)
      && PT <= MATRIX_ROW_END_CHARPOS (row)
      && cursor_row_p (row))
    set_cursor_from_row (it->w, row, it->w->desired_matrix, 0, 0, 0, 0);

  /* Prepare for the next line.  This line starts horizontally at (X
     HPOS) = (0 0).  Vertical positions are incremented.  As a
     convenience for the caller, IT->glyph_row is set to the next
     row to be used.  */
  it->current_x = it->hpos = 0;
  it->current_y += row->height;
  /* Restore the first and last visible X if we adjusted them for
     current-line hscrolling.  */
  if (hscroll_this_line)
    {
      it->first_visible_x = first_visible_x;
      it->last_visible_x  = last_visible_x;
    }
  SET_TEXT_POS (it->eol_pos, 0, 0);
  ++it->vpos;
  ++it->glyph_row;
  /* The next row should by default use the same value of the
     reversed_p flag as this one.  set_iterator_to_next decides when
     it's a new paragraph, and PRODUCE_GLYPHS recomputes the value of
     the flag accordingly.  */
  if (it->glyph_row < MATRIX_BOTTOM_TEXT_ROW (it->w->desired_matrix, it->w))
    it->glyph_row->reversed_p = row->reversed_p;
  it->start = row->end;
  return MATRIX_ROW_DISPLAYS_TEXT_P (row);

#undef RECORD_MAX_MIN_POS
}

DEFUN ("current-bidi-paragraph-direction", Fcurrent_bidi_paragraph_direction,
       Scurrent_bidi_paragraph_direction, 0, 1, 0,
       doc: /* Return paragraph direction at point in BUFFER.
Value is either `left-to-right' or `right-to-left'.
If BUFFER is omitted or nil, it defaults to the current buffer.

Paragraph direction determines how the text in the paragraph is displayed.
In left-to-right paragraphs, text begins at the left margin of the window
and the reading direction is generally left to right.  In right-to-left
paragraphs, text begins at the right margin and is read from right to left.

See also `bidi-paragraph-direction'.  */)
  (Lisp_Object buffer)
{
  struct buffer *buf = current_buffer;
  struct buffer *old = buf;

  if (! NILP (buffer))
    {
      CHECK_BUFFER (buffer);
      buf = XBUFFER (buffer);
    }

  if (NILP (BVAR (buf, bidi_display_reordering))
      || NILP (BVAR (buf, enable_multibyte_characters))
      /* When we are loading loadup.el, the character property tables
	 needed for bidi iteration are not yet available.  */
      || redisplay__inhibit_bidi)
    return Qleft_to_right;
  else if (!NILP (BVAR (buf, bidi_paragraph_direction)))
    return BVAR (buf, bidi_paragraph_direction);
  else
    {
      /* Determine the direction from buffer text.  We could try to
	 use current_matrix if it is up to date, but this seems fast
	 enough as it is.  */
      struct bidi_it itb;
      ptrdiff_t pos = BUF_PT (buf);
      ptrdiff_t bytepos = BUF_PT_BYTE (buf);
      int c;
      void *itb_data = bidi_shelve_cache ();

      set_buffer_temp (buf);
      /* bidi_paragraph_init finds the base direction of the paragraph
	 by searching forward from paragraph start.  We need the base
	 direction of the current or _previous_ paragraph, so we need
	 to make sure we are within that paragraph.  To that end, find
	 the previous non-empty line.  */
      if (pos >= ZV && pos > BEGV)
	dec_both (&pos, &bytepos);
      AUTO_STRING (trailing_white_space, "[\f\t ]*\n");
      if (fast_looking_at (trailing_white_space,
			   pos, bytepos, ZV, ZV_BYTE, Qnil) > 0)
	{
	  while ((c = FETCH_BYTE (bytepos)) == '\n'
		 || c == ' ' || c == '\t' || c == '\f')
	    {
	      if (bytepos <= BEGV_BYTE)
		break;
	      bytepos--;
	      pos--;
	    }
	  while (!CHAR_HEAD_P (FETCH_BYTE (bytepos)))
	    bytepos--;
	}
      bidi_init_it (pos, bytepos, FRAME_WINDOW_P (SELECTED_FRAME ()), &itb);
      itb.paragraph_dir = NEUTRAL_DIR;
      itb.string.s = NULL;
      itb.string.lstring = Qnil;
      itb.string.bufpos = 0;
      itb.string.from_disp_str = false;
      itb.string.unibyte = false;
      /* We have no window to use here for ignoring window-specific
	 overlays.  Using NULL for window pointer will cause
	 compute_display_string_pos to use the current buffer.  */
      itb.w = NULL;
      bidi_paragraph_init (NEUTRAL_DIR, &itb, true);
      bidi_unshelve_cache (itb_data, false);
      set_buffer_temp (old);
      switch (itb.paragraph_dir)
	{
	case L2R:
	  return Qleft_to_right;
	  break;
	case R2L:
	  return Qright_to_left;
	  break;
	default:
	  emacs_abort ();
	}
    }
}

DEFUN ("bidi-find-overridden-directionality",
       Fbidi_find_overridden_directionality,
       Sbidi_find_overridden_directionality, 3, 4, 0,
       doc: /* Return position between FROM and TO where directionality was overridden.

This function returns the first character position in the specified
region of OBJECT where characters have their bidirectional
properties affected in a way that might make its text look confusingly
on display.  For example, characters whose `bidi-class' property is `L',
could be forced to display as `R' by a directional override, and
likewise characters whose `bidi-class' is `R' or `AL' that are
forced to display as `L'.

If no such character is found, the function returns nil.

OBJECT is a Lisp string or buffer to search for overridden
directionality, and defaults to the current buffer if nil.
OBJECT can also be a window, in which case the function will search
the buffer displayed in that window.  Passing the window instead of
a buffer is preferable when the buffer is displayed in some window,
because this function will then be able to correctly account for
window-specific overlays, which can affect the results.

Optional argument BASE-DIR specifies the base paragraph directory
of the text.  It should be a symbol, either `left-to-right'
or `right-to-left', and defaults to `left-to-right'.

Strong directional characters `L', `R', and `AL' can have their
intrinsic directionality overridden by directional override control
characters RLO (u+202E) and LRO (u+202D).  They can also have their
directionality affected by other formatting control characters: LRE
(u+202A), RLE (u+202B), LRI (u+2066), and RLI (u+2067).  See the
function `get-char-code-property' for a way to inquire about the
`bidi-class' property of a character.  Characters whose intrinsic
directionality is weak or neutral, such as numbers or punctuation
characters, can be forced to display in a very different place with
respect of its surrounding characters, so as to make the surrounding
text confuse the user regarding what the text says.

Also see the `highlight-confusing-reorderings' function, which can be
useful in similar circumstances as this function.  */)
  (Lisp_Object from, Lisp_Object to, Lisp_Object object, Lisp_Object base_dir)
{
  struct buffer *buf = current_buffer;
  struct buffer *old = buf;
  struct window *w = NULL;
  bool frame_window_p = FRAME_WINDOW_P (SELECTED_FRAME ());
  struct bidi_it itb;
  ptrdiff_t from_pos, to_pos, from_bpos;
  void *itb_data;

  if (!NILP (object))
    {
      if (BUFFERP (object))
	buf = XBUFFER (object);
      else if (WINDOWP (object))
	{
	  w = decode_live_window (object);
	  buf = XBUFFER (w->contents);
	  frame_window_p = FRAME_WINDOW_P (XFRAME (w->frame));
	}
      else
	CHECK_STRING (object);
    }

  if (STRINGP (object))
    {
      /* Characters in unibyte strings are always treated by bidi.c as
	 strong LTR.  */
      if (!STRING_MULTIBYTE (object)
	  /* When we are loading loadup.el, the character property
	     tables needed for bidi iteration are not yet
	     available.  */
	  || redisplay__inhibit_bidi)
	return Qnil;

      validate_subarray (object, from, to, SCHARS (object), &from_pos, &to_pos);
      if (from_pos >= SCHARS (object))
	return Qnil;

      /* Set up the bidi iterator.  */
      itb_data = bidi_shelve_cache ();
      itb.paragraph_dir = NEUTRAL_DIR;
      itb.string.lstring = object;
      itb.string.s = NULL;
      itb.string.schars = SCHARS (object);
      itb.string.bufpos = 0;
      itb.string.from_disp_str = false;
      itb.string.unibyte = false;
      itb.w = w;
      bidi_init_it (0, 0, frame_window_p, &itb);
    }
  else
    {
      /* Nothing this fancy can happen in unibyte buffers, or in a
	 buffer that disabled reordering, or if FROM is at EOB.  */
      if (NILP (BVAR (buf, bidi_display_reordering))
	  || NILP (BVAR (buf, enable_multibyte_characters))
	  /* When we are loading loadup.el, the character property
	     tables needed for bidi iteration are not yet
	     available.  */
	  || redisplay__inhibit_bidi)
	return Qnil;

      set_buffer_temp (buf);
      validate_region (&from, &to);
      from_pos = XFIXNUM (from);
      to_pos = XFIXNUM (to);
      if (from_pos >= ZV)
	return Qnil;

      /* Set up the bidi iterator.  */
      itb_data = bidi_shelve_cache ();
      from_bpos = CHAR_TO_BYTE (from_pos);
      if (from_pos == BEGV)
	{
	  itb.charpos = BEGV;
	  itb.bytepos = BEGV_BYTE;
	}
      else if (FETCH_BYTE (from_bpos - 1) == '\n')
	{
	  itb.charpos = from_pos;
	  itb.bytepos = from_bpos;
	}
      else
	itb.charpos = find_newline_no_quit (from_pos, CHAR_TO_BYTE (from_pos),
					    -1, &itb.bytepos);
      itb.paragraph_dir = NEUTRAL_DIR;
      itb.string.s = NULL;
      itb.string.lstring = Qnil;
      itb.string.bufpos = 0;
      itb.string.from_disp_str = false;
      itb.string.unibyte = false;
      itb.w = w;
      bidi_init_it (itb.charpos, itb.bytepos, frame_window_p, &itb);
    }

  ptrdiff_t found;
  bidi_dir_t bdir = EQ (base_dir, Qright_to_left) ? R2L : L2R;
  do {
    bidi_paragraph_init (bdir, &itb, false);
    while ((found = bidi_find_first_overridden (&itb)) < from_pos)
      ;
  } while (found == ZV && itb.ch == '\n' && itb.charpos < to_pos);

  bidi_unshelve_cache (itb_data, false);
  set_buffer_temp (old);

  return (from_pos <= found && found < to_pos) ? make_fixnum (found) : Qnil;
}

DEFUN ("move-point-visually", Fmove_point_visually,
       Smove_point_visually, 1, 1, 0,
       doc: /* Move point in the visual order in the specified DIRECTION.
DIRECTION can be 1, meaning move to the right, or -1, which moves to the
left.

Value is the new character position of point.  */)
  (Lisp_Object direction)
{
  struct window *w = XWINDOW (selected_window);
  struct buffer *b = XBUFFER (w->contents);
  struct glyph_row *row;
  int dir;
  Lisp_Object paragraph_dir;

#define ROW_GLYPH_NEWLINE_P(ROW,GLYPH)		\
  (!(ROW)->continued_p				\
   && NILP ((GLYPH)->object)			\
   && (GLYPH)->type == CHAR_GLYPH		\
   && (GLYPH)->u.ch == ' '			\
   && (GLYPH)->charpos >= 0			\
   && !(GLYPH)->avoid_cursor_p)

  CHECK_FIXNUM (direction);
  dir = XFIXNUM (direction);
  if (dir > 0)
    dir = 1;
  else
    dir = -1;

  /* If current matrix is up-to-date, we can use the information
     recorded in the glyphs, at least as long as the goal is on the
     screen.  */
  if (w->window_end_valid
      && !windows_or_buffers_changed
      && b
      && !b->clip_changed
      && !b->prevent_redisplay_optimizations_p
      && !window_outdated (w)
      /* We rely below on the cursor coordinates to be up to date, but
	 we cannot trust them if some command moved point since the
	 last complete redisplay.  */
      && w->last_point == BUF_PT (b)
      && w->cursor.vpos >= 0
      && w->cursor.vpos < w->current_matrix->nrows
      && (row = MATRIX_ROW (w->current_matrix, w->cursor.vpos))->enabled_p)
    {
      struct glyph *g = row->glyphs[TEXT_AREA];
      struct glyph *e = dir > 0 ? g + row->used[TEXT_AREA] : g - 1;
      struct glyph *gpt = g + w->cursor.hpos;

      for (g = gpt + dir; (dir > 0 ? g < e : g > e); g += dir)
	{
	  if (BUFFERP (g->object) && g->charpos != PT)
	    {
	      SET_PT (g->charpos);
	      w->cursor.vpos = -1;
	      return make_fixnum (PT);
	    }
	  else if (!NILP (g->object) && !EQ (g->object, gpt->object))
	    {
	      ptrdiff_t new_pos;

	      if (BUFFERP (gpt->object))
		{
		  new_pos = PT;
		  if ((gpt->resolved_level - row->reversed_p) % 2 == 0)
		    new_pos += (row->reversed_p ? -dir : dir);
		  else
		    new_pos -= (row->reversed_p ? -dir : dir);
		  new_pos = clip_to_bounds (BEGV, new_pos, ZV);
		  /* If we didn't move, we've hit BEGV or ZV, so we
		     need to signal a suitable error.  */
		  if (new_pos == PT)
		    break;
		}
	      else if (BUFFERP (g->object))
		new_pos = g->charpos;
	      else
		break;
	      SET_PT (new_pos);
	      w->cursor.vpos = -1;
	      return make_fixnum (PT);
	    }
	  else if (ROW_GLYPH_NEWLINE_P (row, g))
	    {
	      /* Glyphs inserted at the end of a non-empty line for
		 positioning the cursor have zero charpos, so we must
		 deduce the value of point by other means.  */
	      if (g->charpos > 0)
		SET_PT (g->charpos);
	      else if (row->ends_at_zv_p && PT != ZV)
		SET_PT (ZV);
	      else if (PT != MATRIX_ROW_END_CHARPOS (row) - 1)
		SET_PT (MATRIX_ROW_END_CHARPOS (row) - 1);
	      else
		break;
	      w->cursor.vpos = -1;
	      return make_fixnum (PT);
	    }
	}
      if (g == e || NILP (g->object))
	{
	  if (row->truncated_on_left_p || row->truncated_on_right_p)
	    goto simulate_display;
	  if (!row->reversed_p)
	    row += dir;
	  else
	    row -= dir;
	  if (!(MATRIX_FIRST_TEXT_ROW (w->current_matrix) <= row
		&& row < MATRIX_BOTTOM_TEXT_ROW (w->current_matrix, w)))
	    goto simulate_display;

	  if (dir > 0)
	    {
	      if (row->reversed_p && !row->continued_p)
		{
		  SET_PT (MATRIX_ROW_END_CHARPOS (row) - 1);
		  w->cursor.vpos = -1;
		  return make_fixnum (PT);
		}
	      g = row->glyphs[TEXT_AREA];
	      e = g + row->used[TEXT_AREA];
	      for ( ; g < e; g++)
		{
		  if (BUFFERP (g->object)
		      /* Empty lines have only one glyph, which stands
			 for the newline, and whose charpos is the
			 buffer position of the newline.  */
		      || ROW_GLYPH_NEWLINE_P (row, g)
		      /* When the buffer ends in a newline, the line at
			 EOB also has one glyph, but its charpos is -1.  */
		      || (row->ends_at_zv_p
			  && !row->reversed_p
			  && NILP (g->object)
			  && g->type == CHAR_GLYPH
			  && g->u.ch == ' '))
		    {
		      if (g->charpos > 0)
			SET_PT (g->charpos);
		      else if (!row->reversed_p
			       && row->ends_at_zv_p
			       && PT != ZV)
			SET_PT (ZV);
		      else
			continue;
		      w->cursor.vpos = -1;
		      return make_fixnum (PT);
		    }
		}
	    }
	  else
	    {
	      if (!row->reversed_p && !row->continued_p)
		{
		  SET_PT (MATRIX_ROW_END_CHARPOS (row) - 1);
		  w->cursor.vpos = -1;
		  return make_fixnum (PT);
		}
	      e = row->glyphs[TEXT_AREA];
	      g = e + row->used[TEXT_AREA] - 1;
	      for ( ; g >= e; g--)
		{
		  if (BUFFERP (g->object)
		      || (ROW_GLYPH_NEWLINE_P (row, g)
			  && g->charpos > 0)
		      /* Empty R2L lines on GUI frames have the buffer
			 position of the newline stored in the stretch
			 glyph.  */
		      || g->type == STRETCH_GLYPH
		      || (row->ends_at_zv_p
			  && row->reversed_p
			  && NILP (g->object)
			  && g->type == CHAR_GLYPH
			  && g->u.ch == ' '))
		    {
		      if (g->charpos > 0)
			SET_PT (g->charpos);
		      else if (row->reversed_p
			       && row->ends_at_zv_p
			       && PT != ZV)
			SET_PT (ZV);
		      else
			continue;
		      w->cursor.vpos = -1;
		      return make_fixnum (PT);
		    }
		}
	    }
	}
    }

 simulate_display:

  /* If we wind up here, we failed to move by using the glyphs, so we
     need to simulate display instead.  */

  if (b)
    paragraph_dir = Fcurrent_bidi_paragraph_direction (w->contents);
  else
    paragraph_dir = Qleft_to_right;
  if (EQ (paragraph_dir, Qright_to_left))
    dir = -dir;
  if (PT <= BEGV && dir < 0)
    xsignal0 (Qbeginning_of_buffer);
  else if (PT >= ZV && dir > 0)
    xsignal0 (Qend_of_buffer);
  else
    {
      struct text_pos pt;
      struct it it;
      int pt_x, target_x, pixel_width, pt_vpos;
      bool at_eol_p;
      bool overshoot_expected = false;
      bool target_is_eol_p = false;
      void *itdata = bidi_shelve_cache ();

      /* Setup the arena.  */
      SET_TEXT_POS (pt, PT, PT_BYTE);
      start_display (&it, w, pt);
      /* When lines are truncated, we could be called with point
	 outside of the windows edges, in which case move_it_*
	 functions either prematurely stop at window's edge or jump to
	 the next screen line, whereas we rely below on our ability to
	 reach point, in order to start from its X coordinate.  So we
	 need to disregard the window's horizontal extent in that case.  */
      if (it.line_wrap == TRUNCATE)
	it.last_visible_x = DISP_INFINITY;

      if (it.cmp_it.id < 0
	  && it.method == GET_FROM_STRING
	  && it.area == TEXT_AREA
	  && it.string_from_display_prop_p
	  && (it.sp > 0 && it.stack[it.sp - 1].method == GET_FROM_BUFFER))
	overshoot_expected = true;

      /* Find the X coordinate of point.  We start from the beginning
	 of this or previous line to make sure we are before point in
	 the logical order (since the move_it_* functions can only
	 move forward).  */
    reseat:
      reseat_at_previous_visible_line_start (&it);
      it.current_x = it.hpos = it.current_y = it.vpos = 0;
      if (IT_CHARPOS (it) != PT)
	{
	  move_it_to (&it, overshoot_expected ? PT - 1 : PT,
		      -1, -1, -1, MOVE_TO_POS);
	  /* If we missed point because the character there is
	     displayed out of a display vector that has more than one
	     glyph, retry expecting overshoot.  */
	  if (it.method == GET_FROM_DISPLAY_VECTOR
	      && it.current.dpvec_index > 0
	      && !overshoot_expected)
	    {
	      overshoot_expected = true;
	      goto reseat;
	    }
	  else if (IT_CHARPOS (it) != PT && !overshoot_expected)
	    move_it_in_display_line (&it, PT, -1, MOVE_TO_POS);
	}
      pt_x = it.current_x;
      pt_vpos = it.vpos;
      if (dir > 0 || overshoot_expected)
	{
	  struct glyph_row *row = it.glyph_row;

	  /* When point is at beginning of line, we don't have
	     information about the glyph there loaded into struct
	     it.  Calling get_next_display_element fixes that.  */
	  if (pt_x == 0)
	    get_next_display_element (&it);
	  at_eol_p = ITERATOR_AT_END_OF_LINE_P (&it);
	  it.glyph_row = NULL;
	  PRODUCE_GLYPHS (&it);	/* compute it.pixel_width */
	  it.glyph_row = row;
	  /* PRODUCE_GLYPHS advances it.current_x, so we must restore
	     it, lest it will become out of sync with it's buffer
	     position.  */
	  it.current_x = pt_x;
	}
      else
	at_eol_p = ITERATOR_AT_END_OF_LINE_P (&it);
      pixel_width = it.pixel_width;
      if (overshoot_expected && at_eol_p)
	pixel_width = 0;
      else if (pixel_width <= 0)
	pixel_width = 1;

      /* If there's a display string (or something similar) at point,
	 we are actually at the glyph to the left of point, so we need
	 to correct the X coordinate.  */
      if (overshoot_expected)
	{
	  if (it.bidi_p)
	    pt_x += pixel_width * it.bidi_it.scan_dir;
	  else
	    pt_x += pixel_width;
	}

      /* Compute target X coordinate, either to the left or to the
	 right of point.  On TTY frames, all characters have the same
	 pixel width of 1, so we can use that.  On GUI frames we don't
	 have an easy way of getting at the pixel width of the
	 character to the left of point, so we use a different method
	 of getting to that place.  */
      if (dir > 0)
	target_x = pt_x + pixel_width;
      else
	target_x = pt_x - (!FRAME_WINDOW_P (it.f)) * pixel_width;

      /* Target X coordinate could be one line above or below the line
	 of point, in which case we need to adjust the target X
	 coordinate.  Also, if moving to the left, we need to begin at
	 the left edge of the point's screen line.  */
      if (dir < 0)
	{
	  if (pt_x > 0)
	    {
	      start_display (&it, w, pt);
	      if (it.line_wrap == TRUNCATE)
		it.last_visible_x = DISP_INFINITY;
	      reseat_at_previous_visible_line_start (&it);
	      it.current_x = it.current_y = it.hpos = 0;
	      if (pt_vpos != 0)
		move_it_by_lines (&it, pt_vpos);
	    }
	  else
	    {
	      move_it_by_lines (&it, -1);
	      target_x = it.last_visible_x - !FRAME_WINDOW_P (it.f);
	      target_is_eol_p = true;
	      /* Under word-wrap, we don't know the x coordinate of
		 the last character displayed on the previous line,
		 which immediately precedes the wrap point.  To find
		 out its x coordinate, we try moving to the right
		 margin of the window, which will stop at the wrap
		 point, and then reset target_x to point at the
		 character that precedes the wrap point.  This is not
		 needed on GUI frames, because (see below) there we
		 move from the left margin one grapheme cluster at a
		 time, and stop when we hit the wrap point.  */
	      if (!FRAME_WINDOW_P (it.f) && it.line_wrap == WORD_WRAP)
		{
		  void *it_data = NULL;
		  struct it it2;

		  SAVE_IT (it2, it, it_data);
		  move_it_in_display_line_to (&it, ZV, target_x,
					      MOVE_TO_POS | MOVE_TO_X);
		  /* If we arrived at target_x, that _is_ the last
		     character on the previous line.  */
		  if (it.current_x != target_x)
		    target_x = it.current_x - 1;
		  RESTORE_IT (&it, &it2, it_data);
		}
	    }
	}
      else
	{
	  if (at_eol_p
	      || (target_x >= it.last_visible_x
		  && it.line_wrap != TRUNCATE))
	    {
	      if (pt_x > 0)
		move_it_by_lines (&it, 0);
	      move_it_by_lines (&it, 1);
	      target_x = 0;
	    }
	}

      /* Move to the target X coordinate.  */
      /* On GUI frames, as we don't know the X coordinate of the
	 character to the left of point, moving point to the left
	 requires walking, one grapheme cluster at a time, until we
	 find ourself at a place immediately to the left of the
	 character at point.  */
      if (FRAME_WINDOW_P (it.f) && dir < 0)
	{
	  struct text_pos new_pos;
	  enum move_it_result rc = MOVE_X_REACHED;

	  if (it.current_x == 0)
	    get_next_display_element (&it);
	  if (it.what == IT_COMPOSITION)
	    {
	      new_pos.charpos = it.cmp_it.charpos;
	      new_pos.bytepos = -1;
	    }
	  else
	    new_pos = it.current.pos;

	  while (it.current_x + it.pixel_width <= target_x
		 && (rc == MOVE_X_REACHED
		     /* Under word-wrap, move_it_in_display_line_to
			stops at correct coordinates, but sometimes
			returns MOVE_POS_MATCH_OR_ZV.  */
		     || (it.line_wrap == WORD_WRAP
			 && rc == MOVE_POS_MATCH_OR_ZV)))
	    {
	      int new_x = it.current_x + it.pixel_width;

	      /* For composed characters, we want the position of the
		 first character in the grapheme cluster (usually, the
		 composition's base character), whereas it.current
		 might give us the position of the _last_ one, e.g. if
		 the composition is rendered in reverse due to bidi
		 reordering.  */
	      if (it.what == IT_COMPOSITION)
		{
		  new_pos.charpos = it.cmp_it.charpos;
		  new_pos.bytepos = -1;
		}
	      else
		new_pos = it.current.pos;
	      if (new_x == it.current_x)
		new_x++;
	      rc = move_it_in_display_line_to (&it, ZV, new_x,
					       MOVE_TO_POS | MOVE_TO_X);
	      if (ITERATOR_AT_END_OF_LINE_P (&it) && !target_is_eol_p)
		break;
	    }
	  /* The previous position we saw in the loop is the one we
	     want.  */
	  if (new_pos.bytepos == -1)
	    new_pos.bytepos = CHAR_TO_BYTE (new_pos.charpos);
	  it.current.pos = new_pos;
	}
      else if (it.current_x != target_x)
	move_it_in_display_line_to (&it, ZV, target_x, MOVE_TO_POS | MOVE_TO_X);

      /* If we ended up in a display string that covers point, move to
	 buffer position to the right in the visual order.  */
      if (dir > 0)
	{
	  while (IT_CHARPOS (it) == PT)
	    {
	      set_iterator_to_next (&it, false);
	      if (!get_next_display_element (&it))
		break;
	    }
	}

      /* Move point to that position.  */
      SET_PT_BOTH (IT_CHARPOS (it), IT_BYTEPOS (it));
      bidi_unshelve_cache (itdata, false);
    }

  return make_fixnum (PT);

#undef ROW_GLYPH_NEWLINE_P
}

DEFUN ("bidi-resolved-levels", Fbidi_resolved_levels,
       Sbidi_resolved_levels, 0, 1, 0,
       doc: /* Return the resolved bidirectional levels of characters at VPOS.

The resolved levels are produced by the Emacs bidi reordering engine
that implements the UBA, the Unicode Bidirectional Algorithm.  Please
read the Unicode Standard Annex 9 (UAX#9) for background information
about these levels.

VPOS is the zero-based number of the current window's screen line
for which to produce the resolved levels.  If VPOS is nil or omitted,
it defaults to the screen line of point.  If the window displays a
header line, VPOS of zero will report on the header line, and first
line of text in the window will have VPOS of 1.

Value is an array of resolved levels, indexed by glyph number.
Glyphs are numbered from zero starting from the beginning of the
screen line, i.e. the left edge of the window for left-to-right lines
and from the right edge for right-to-left lines.  The resolved levels
are produced only for the window's text area; text in display margins
is not included.

If the selected window's display is not up-to-date, or if the specified
screen line does not display text, this function returns nil.  It is
highly recommended to bind this function to some simple key, like F8,
in order to avoid these problems.

This function exists mainly for testing the correctness of the
Emacs UBA implementation, in particular with the test suite.  */)
  (Lisp_Object vpos)
{
  struct window *w = XWINDOW (selected_window);
  struct buffer *b = XBUFFER (w->contents);
  int nrow;
  struct glyph_row *row;

  if (NILP (vpos))
    {
      int d1, d2, d3, d4, d5;

      pos_visible_p (w, PT, &d1, &d2, &d3, &d4, &d5, &nrow);
    }
  else
    {
      CHECK_FIXNUM (vpos);
      nrow = XFIXNUM (vpos);
    }

  /* We require up-to-date glyph matrix for this window.  */
  if (w->window_end_valid
      && !windows_or_buffers_changed
      && b
      && !b->clip_changed
      && !b->prevent_redisplay_optimizations_p
      && !window_outdated (w)
      && nrow >= 0
      && nrow < w->current_matrix->nrows
      && (row = MATRIX_ROW (w->current_matrix, nrow))->enabled_p
      && MATRIX_ROW_DISPLAYS_TEXT_P (row))
    {
      struct glyph *g, *e, *g1;
      int nglyphs, i;
      Lisp_Object levels;

      if (!row->reversed_p)	/* Left-to-right glyph row.  */
	{
	  g = g1 = row->glyphs[TEXT_AREA];
	  e = g + row->used[TEXT_AREA];

	  /* Skip over glyphs at the start of the row that was
	     generated by redisplay for its own needs.  */
	  while (g < e
		 && NILP (g->object)
		 && g->charpos < 0)
	    g++;
	  g1 = g;

	  /* Count the "interesting" glyphs in this row.  */
	  for (nglyphs = 0; g < e && !NILP (g->object); g++)
	    nglyphs++;

	  /* Create and fill the array.  */
	  levels = make_uninit_vector (nglyphs);
	  for (i = 0; g1 < g; i++, g1++)
	    ASET (levels, i, make_fixnum (g1->resolved_level));
	}
      else			/* Right-to-left glyph row.  */
	{
	  g = row->glyphs[TEXT_AREA] + row->used[TEXT_AREA] - 1;
	  e = row->glyphs[TEXT_AREA] - 1;
	  while (g > e
		 && NILP (g->object)
		 && g->charpos < 0)
	    g--;
	  g1 = g;
	  for (nglyphs = 0; g > e && !NILP (g->object); g--)
	    nglyphs++;
	  levels = make_uninit_vector (nglyphs);
	  for (i = 0; g1 > g; i++, g1--)
	    ASET (levels, i, make_fixnum (g1->resolved_level));
	}
      return levels;
    }
  else
    return Qnil;
}



/***********************************************************************
			       Menu Bar
 ***********************************************************************/

/* Redisplay the menu bar in the frame for window W.

   The menu bar of X frames that don't have X toolkit support is
   displayed in a special window W->frame->menu_bar_window.

   The menu bar of terminal frames is treated specially as far as
   glyph matrices are concerned.  Menu bar lines are not part of
   windows, so the update is done directly on the frame matrix rows
   for the menu bar.  */

static void
display_menu_bar (struct window *w)
{
  struct frame *f = XFRAME (WINDOW_FRAME (w));
  struct it it;
  Lisp_Object items;
  int i;

  /* Don't do all this for graphical frames.  */
#ifdef HAVE_NTGUI
  if (FRAME_W32_P (f))
    return;
#endif
#if defined (HAVE_PGTK)
  if (FRAME_PGTK_P (f))
    return;
#endif

#if defined (USE_X_TOOLKIT) || defined (USE_GTK)
  if (FRAME_X_P (f))
    return;
#endif

#ifdef HAVE_NS
  if (FRAME_NS_P (f))
    return;
#endif /* HAVE_NS */

#ifdef HAVE_HAIKU
  if (FRAME_HAIKU_P (f))
    return;
#endif /* HAVE_HAIKU */

#if defined (USE_X_TOOLKIT) || defined (USE_GTK)
  eassert (!FRAME_WINDOW_P (f));
  init_iterator (&it, w, -1, -1, f->desired_matrix->rows, MENU_FACE_ID);
  it.first_visible_x = 0;
  it.last_visible_x = FRAME_PIXEL_WIDTH (f);
#elif defined (HAVE_X_WINDOWS) /* X without toolkit.  */
  if (FRAME_WINDOW_P (f))
    {
      /* Menu bar lines are displayed in the desired matrix of the
	 dummy window menu_bar_window.  */
      struct window *menu_w;
      menu_w = XWINDOW (f->menu_bar_window);
      init_iterator (&it, menu_w, -1, -1, menu_w->desired_matrix->rows,
		     MENU_FACE_ID);
      it.first_visible_x = 0;
      it.last_visible_x = FRAME_PIXEL_WIDTH (f);
    }
  else
#endif /* not USE_X_TOOLKIT and not USE_GTK */
    {
      /* This is a TTY frame, i.e. character hpos/vpos are used as
	 pixel x/y.  */
      init_iterator (&it, w, -1, -1, f->desired_matrix->rows,
		     MENU_FACE_ID);
      it.first_visible_x = 0;
      it.last_visible_x = FRAME_COLS (f);
    }

  /* FIXME: This should be controlled by a user option.  See the
     comments in redisplay_tool_bar and display_mode_line about
     this.  */
  it.paragraph_embedding = L2R;

  /* Clear all rows of the menu bar.  */
  for (i = 0; i < FRAME_MENU_BAR_LINES (f); ++i)
    {
      struct glyph_row *row = it.glyph_row + i;
      clear_glyph_row (row);
      row->enabled_p = true;
      row->full_width_p = true;
      row->reversed_p = false;
    }

  /* Display all items of the menu bar.  */
  items = FRAME_MENU_BAR_ITEMS (it.f);
  for (i = 0; i < ASIZE (items); i += 4)
    {
      Lisp_Object string;

      /* Stop at nil string.  */
      string = AREF (items, i + 1);
      if (NILP (string))
	break;

      /* Remember where item was displayed.  */
      ASET (items, i + 3, make_fixnum (it.hpos));

      /* Display the item, pad with one space.  */
      if (it.current_x < it.last_visible_x)
	display_string (NULL, string, Qnil, 0, 0, &it,
			SCHARS (string) + 1, 0, 0, STRING_MULTIBYTE (string));
    }

  /* Fill out the line with spaces.  */
  if (it.current_x < it.last_visible_x)
    display_string ("", Qnil, Qnil, 0, 0, &it, -1, 0, 0, -1);

  /* Compute the total height of the lines.  */
  compute_line_metrics (&it);
}

/* Deep copy of a glyph row, including the glyphs.  */
static void
deep_copy_glyph_row (struct glyph_row *to, struct glyph_row *from)
{
  struct glyph *pointers[1 + LAST_AREA];
  int to_used = to->used[TEXT_AREA];

  /* Save glyph pointers of TO.  */
  memcpy (pointers, to->glyphs, sizeof to->glyphs);

  /* Do a structure assignment.  */
  *to = *from;

  /* Restore original glyph pointers of TO.  */
  memcpy (to->glyphs, pointers, sizeof to->glyphs);

  /* Copy the glyphs.  */
  memcpy (to->glyphs[TEXT_AREA], from->glyphs[TEXT_AREA],
	  min (from->used[TEXT_AREA], to_used) * sizeof (struct glyph));

  /* If we filled only part of the TO row, fill the rest with
     space_glyph (which will display as empty space).  */
  if (to_used > from->used[TEXT_AREA])
    fill_up_frame_row_with_spaces (to, to_used);
}

/* Display one menu item on a TTY, by overwriting the glyphs in the
   frame F's desired glyph matrix with glyphs produced from the menu
   item text.  Called from term.c to display TTY drop-down menus one
   item at a time.

   ITEM_TEXT is the menu item text as a C string.

   FACE_ID is the face ID to be used for this menu item.  FACE_ID
   could specify one of 3 faces: a face for an enabled item, a face
   for a disabled item, or a face for a selected item.

   X and Y are coordinates of the first glyph in the frame's desired
   matrix to be overwritten by the menu item.  Since this is a TTY, Y
   is the zero-based number of the glyph row and X is the zero-based
   glyph number in the row, starting from left, where to start
   displaying the item.

   SUBMENU means this menu item drops down a submenu, which
   should be indicated by displaying a proper visual cue after the
   item text.  */

void
display_tty_menu_item (const char *item_text, int width, int face_id,
		       int x, int y, bool submenu)
{
  struct it it;
  struct frame *f = SELECTED_FRAME ();
  struct window *w = XWINDOW (f->selected_window);
  struct glyph_row *row;
  size_t item_len = strlen (item_text);

  eassert (FRAME_TERMCAP_P (f));

  /* Don't write beyond the matrix's last row.  This can happen for
     TTY screens that are not high enough to show the entire menu.
     (This is actually a bit of defensive programming, as
     tty_menu_display already limits the number of menu items to one
     less than the number of screen lines.)  */
  if (y >= f->desired_matrix->nrows)
    return;

  init_iterator (&it, w, -1, -1, f->desired_matrix->rows + y, MENU_FACE_ID);
  it.first_visible_x = 0;
  it.last_visible_x = FRAME_COLS (f) - 1;
  row = it.glyph_row;
  /* Start with the row contents from the current matrix.  */
  deep_copy_glyph_row (row, f->current_matrix->rows + y);
  bool saved_width = row->full_width_p;
  row->full_width_p = true;
  bool saved_reversed = row->reversed_p;
  row->reversed_p = false;
  row->enabled_p = true;

  /* Arrange for the menu item glyphs to start at (X,Y) and have the
     desired face.  */
  eassert (x < f->desired_matrix->matrix_w);
  it.current_x = it.hpos = x;
  it.current_y = it.vpos = y;
  int saved_used = row->used[TEXT_AREA];
  bool saved_truncated = row->truncated_on_right_p;
  row->used[TEXT_AREA] = x;
  it.face_id = face_id;
  it.line_wrap = TRUNCATE;

  /* FIXME: This should be controlled by a user option.  See the
     comments in redisplay_tool_bar and display_mode_line about this.
     Also, if paragraph_embedding could ever be R2L, changes will be
     needed to avoid shifting to the right the row characters in
     term.c:append_glyph.  */
  it.paragraph_embedding = L2R;

  /* Pad with a space on the left.  */
  display_string (" ", Qnil, Qnil, 0, 0, &it, 1, 0, FRAME_COLS (f) - 1, -1);
  width--;
  /* Display the menu item, pad with spaces to WIDTH.  */
  if (submenu)
    {
      display_string (item_text, Qnil, Qnil, 0, 0, &it,
		      item_len, 0, FRAME_COLS (f) - 1, -1);
      width -= item_len;
      /* Indicate with " >" that there's a submenu.  */
      display_string (" >", Qnil, Qnil, 0, 0, &it, width, 0,
		      FRAME_COLS (f) - 1, -1);
    }
  else
    display_string (item_text, Qnil, Qnil, 0, 0, &it,
		    width, 0, FRAME_COLS (f) - 1, -1);

  row->used[TEXT_AREA] = max (saved_used, row->used[TEXT_AREA]);
  row->truncated_on_right_p = saved_truncated;
  row->hash = row_hash (row);
  row->full_width_p = saved_width;
  row->reversed_p = saved_reversed;
}

/***********************************************************************
			      Mode Line
 ***********************************************************************/

/* Redisplay mode lines in the window tree whose root is WINDOW.
   If FORCE, redisplay mode lines unconditionally.
   Otherwise, redisplay only mode lines that are garbaged.  Value is
   the number of windows whose mode lines were redisplayed.  */

static int
redisplay_mode_lines (Lisp_Object window, bool force)
{
  int nwindows = 0;

  while (!NILP (window))
    {
      struct window *w = XWINDOW (window);

      if (WINDOWP (w->contents))
	nwindows += redisplay_mode_lines (w->contents, force);
      else if (force
	       || FRAME_GARBAGED_P (XFRAME (w->frame))
	       || !MATRIX_MODE_LINE_ROW (w->current_matrix)->enabled_p)
	{
	  struct text_pos lpoint;
	  struct buffer *old = current_buffer;

	  /* Set the window's buffer for the mode line display.  */
	  SET_TEXT_POS (lpoint, PT, PT_BYTE);
	  set_buffer_internal_1 (XBUFFER (w->contents));

	  /* Point refers normally to the selected window.  For any
	     other window, set up appropriate value.  */
	  if (!EQ (window, selected_window))
	    {
	      struct text_pos pt;

	      CLIP_TEXT_POS_FROM_MARKER (pt, w->pointm);
	      TEMP_SET_PT_BOTH (CHARPOS (pt), BYTEPOS (pt));
	    }

	  /* Display mode lines.  */
	  clear_glyph_matrix (w->desired_matrix);
	  if (display_mode_lines (w))
	    ++nwindows;

	  /* Restore old settings.  */
	  set_buffer_internal_1 (old);
	  TEMP_SET_PT_BOTH (CHARPOS (lpoint), BYTEPOS (lpoint));
	}

      window = w->next;
    }

  return nwindows;
}


/* Display the mode line, the header line, and the tab-line of window
   W.  Value is the sum number of mode lines, header lines, and tab
   lines actually displayed.  */

static int
display_mode_lines (struct window *w)
{
  Lisp_Object old_selected_window = selected_window;
  Lisp_Object new_frame = w->frame;
  specpdl_ref count = SPECPDL_INDEX ();
  int n = 0;

  record_unwind_protect (restore_selected_window, selected_window);
  record_unwind_protect
    (restore_frame_selected_window, XFRAME (new_frame)->selected_window);

  if (window_wants_mode_line (w))
    {
      Lisp_Object window;
      Lisp_Object default_help
	= buffer_local_value (Qmode_line_default_help_echo, w->contents);

      /* Set up mode line help echo.  Do this before selecting w so it
	 can reasonably tell whether a mouse click will select w.  */
      XSETWINDOW (window, w);
      if (FUNCTIONP (default_help))
	wset_mode_line_help_echo (w, safe_call1 (default_help, window));
      else if (STRINGP (default_help))
	wset_mode_line_help_echo (w, default_help);
      else
	wset_mode_line_help_echo (w, Qnil);
    }

  selected_frame = new_frame;
  /* FIXME: If we were to allow the mode-line's computation changing the buffer
     or window's point, then we'd need select_window_1 here as well.  */
  XSETWINDOW (selected_window, w);
  XFRAME (new_frame)->selected_window = selected_window;

  /* These will be set while the mode line specs are processed.  */
  line_number_displayed = false;
  w->column_number_displayed = -1;

  if (window_wants_mode_line (w))
    {
      Lisp_Object window_mode_line_format
	= window_parameter (w, Qmode_line_format);
      struct window *sel_w = XWINDOW (old_selected_window);

      /* Select mode line face based on the real selected window.  */
      display_mode_line (w,
			 CURRENT_MODE_LINE_ACTIVE_FACE_ID_3 (sel_w, sel_w, w),
			 NILP (window_mode_line_format)
			 ? BVAR (current_buffer, mode_line_format)
			 : window_mode_line_format);
      ++n;
    }

  if (window_wants_tab_line (w))
    {
      Lisp_Object window_tab_line_format
	= window_parameter (w, Qtab_line_format);

      display_mode_line (w, TAB_LINE_FACE_ID,
			 NILP (window_tab_line_format)
			 ? BVAR (current_buffer, tab_line_format)
			 : window_tab_line_format);
      ++n;
    }

  if (window_wants_header_line (w))
    {
      Lisp_Object window_header_line_format
	= window_parameter (w, Qheader_line_format);

      display_mode_line (w, HEADER_LINE_FACE_ID,
			 NILP (window_header_line_format)
			 ? BVAR (current_buffer, header_line_format)
			 : window_header_line_format);
      ++n;
    }

  unbind_to (count, Qnil);

  if (n > 0)
    w->must_be_updated_p = true;
  return n;
}


/* Display mode or header/tab line of window W.  FACE_ID specifies
   which line to display; it is either MODE_LINE_ACTIVE_FACE_ID,
   HEADER_LINE_FACE_ID or TAB_LINE_FACE_ID.  FORMAT is the
   mode/header/tab line format to display.  Value is the pixel height
   of the mode/header/tab line displayed.  */

static int
display_mode_line (struct window *w, enum face_id face_id, Lisp_Object format)
{
  struct it it;
  struct face *face;
  specpdl_ref count = SPECPDL_INDEX ();

  init_iterator (&it, w, -1, -1, NULL, face_id);
  /* Don't extend on a previously drawn mode-line.
     This may happen if called from pos_visible_p.  */
  it.glyph_row->enabled_p = false;
  prepare_desired_row (w, it.glyph_row, true);

  it.glyph_row->mode_line_p = true;
  if (face_id == TAB_LINE_FACE_ID)
    {
      it.glyph_row->tab_line_p = true;
      w->desired_matrix->tab_line_p = true;
    }
  else if (face_id == HEADER_LINE_FACE_ID)
    w->desired_matrix->header_line_p = true;

  /* FIXME: This should be controlled by a user option.  But
     supporting such an option is not trivial, since the mode line is
     made up of many separate strings.  */
  it.paragraph_embedding = L2R;

  record_unwind_protect (unwind_format_mode_line,
			 format_mode_line_unwind_data (NULL, NULL,
						       Qnil, false));

  /* Temporarily make frame's keyboard the current kboard so that
     kboard-local variables in the mode_line_format will get the right
     values.  */
  push_kboard (FRAME_KBOARD (it.f));
  record_unwind_save_match_data ();

  if (NILP (Vmode_line_compact)
      || face_id == HEADER_LINE_FACE_ID || face_id == TAB_LINE_FACE_ID)
    {
      mode_line_target = MODE_LINE_DISPLAY;
      display_mode_element (&it, 0, 0, 0, format, Qnil, false);
    }
  else
    {
      Lisp_Object mode_string = Fformat_mode_line (format, Qnil, Qnil, Qnil);
      if (EQ (Vmode_line_compact, Qlong)
	  && WINDOW_TOTAL_COLS (w) >= SCHARS (mode_string))
	{
	  /* The window is wide enough; just display the mode line we
	     just computed. */
	  display_string (NULL, mode_string, Qnil,
			  0, 0, &it, 0, 0, 0,
			  STRING_MULTIBYTE (mode_string));
	}
      else
	{
	  /* Compress the mode line. */
	  ptrdiff_t i = 0, i_byte = 0, start = 0;
	  int prev = 0;

	  while (i < SCHARS (mode_string))
	    {
	      int c = fetch_string_char_advance (mode_string, &i, &i_byte);
	      if (c == ' ' && prev == ' ')
		{
		  display_string (NULL,
				  Fsubstring (mode_string, make_fixnum (start),
					      make_fixnum (i - 1)),
				  Qnil, 0, 0, &it, 0, 0, 0,
				  STRING_MULTIBYTE (mode_string));
		  /* Skip past the rest of the space characters. */
		  while (c == ' ' && i < SCHARS (mode_string))
		      c = fetch_string_char_advance (mode_string, &i, &i_byte);
		  start = i - 1;
		}
	      prev = c;
	    }

	  /* Display the final bit. */
	  if (start < i)
	    display_string (NULL,
			    Fsubstring (mode_string, make_fixnum (start),
					make_fixnum (i)),
			    Qnil, 0, 0, &it, 0, 0, 0,
			    STRING_MULTIBYTE (mode_string));
	}
    }
  pop_kboard ();

  unbind_to (count, Qnil);

  /* Fill up with spaces.  */
  display_string (" ", Qnil, Qnil, 0, 0, &it, 10000, -1, -1, 0);

  compute_line_metrics (&it);
  it.glyph_row->full_width_p = true;
  it.glyph_row->continued_p = false;
  it.glyph_row->truncated_on_left_p = false;
  it.glyph_row->truncated_on_right_p = false;

  /* Make a 3D mode-line have a shadow at its right end.  */
  face = FACE_FROM_ID (it.f, face_id);
  extend_face_to_end_of_line (&it);
  if (face->box != FACE_NO_BOX)
    {
      struct glyph *last = (it.glyph_row->glyphs[TEXT_AREA]
			    + it.glyph_row->used[TEXT_AREA] - 1);
      int box_thickness = face->box_vertical_line_width;
      last->right_box_line_p = true;
      /* Add back the space for the right box line we subtracted in
	 init_iterator, since the right_box_line_p flag will make the
	 glyph wider.  We actually add only as much space as is
	 available for the last glyph of the modeline and whatever
	 space is left beyond it, since that glyph could be only
	 partially visible */
      if (box_thickness > 0)
	last->pixel_width += max (0, (box_thickness
				      - (it.current_x - it.last_visible_x)));
    }

  return it.glyph_row->height;
}

/* Move element ELT in LIST to the front of LIST.
   Return the updated list.  */

static Lisp_Object
move_elt_to_front (Lisp_Object elt, Lisp_Object list)
{
  register Lisp_Object tail, prev;
  register Lisp_Object tem;

  tail = list;
  prev = Qnil;
  while (CONSP (tail))
    {
      tem = XCAR (tail);

      if (EQ (elt, tem))
	{
	  /* Splice out the link TAIL.  */
	  if (NILP (prev))
	    list = XCDR (tail);
	  else
	    Fsetcdr (prev, XCDR (tail));

	  /* Now make it the first.  */
	  Fsetcdr (tail, list);
	  return tail;
	}
      else
	prev = tail;
      tail = XCDR (tail);
      maybe_quit ();
    }

  /* Not found--return unchanged LIST.  */
  return list;
}

/* Subroutine to call Fset_text_properties through
   internal_condition_case_n.  ARGS are the arguments of
   Fset_text_properties, in order.  */

static Lisp_Object
safe_set_text_properties (ptrdiff_t nargs, Lisp_Object *args)
{
  eassert (nargs == 4);
  return Fset_text_properties (args[0], args[1], args[2], args[3]);
}

/* Contribute ELT to the mode line for window IT->w.  How it
   translates into text depends on its data type.

   IT describes the display environment in which we display, as usual.

   DEPTH is the depth in recursion.  It is used to prevent
   infinite recursion here.

   FIELD_WIDTH is the number of characters the display of ELT should
   occupy in the mode line, and PRECISION is the maximum number of
   characters to display from ELT's representation.  See
   display_string for details.

   Returns the hpos of the end of the text generated by ELT.

   PROPS is a property list to add to any string we encounter.

   If RISKY, remove (disregard) any properties in any string
   we encounter, and ignore :eval and :propertize.

   The global variable `mode_line_target' determines whether the
   output is passed to `store_mode_line_noprop',
   `store_mode_line_string', or `display_string'.  */

static int
display_mode_element (struct it *it, int depth, int field_width, int precision,
		      Lisp_Object elt, Lisp_Object props, bool risky)
{
  int n = 0, field, prec;
  bool literal = false;

 tail_recurse:
  if (depth > 100)
    elt = build_string ("*too-deep*");

  depth++;

  switch (XTYPE (elt))
    {
    case Lisp_String:
      {
	/* A string: output it and check for %-constructs within it.  */
	unsigned char c;
	ptrdiff_t offset = 0;

	if (SCHARS (elt) > 0
	    && (!NILP (props) || risky))
	  {
	    Lisp_Object oprops, aelt;
	    oprops = Ftext_properties_at (make_fixnum (0), elt);

	    /* If the starting string's properties are not what
	       we want, translate the string.  Also, if the string
	       is risky, do that anyway.  */

	    if (NILP (Fequal (props, oprops)) || risky)
	      {
		/* If the starting string has properties,
		   merge the specified ones onto the existing ones.  */
		if (! NILP (oprops) && !risky)
		  {
		    Lisp_Object tem;

		    oprops = Fcopy_sequence (oprops);
		    tem = props;
		    while (CONSP (tem))
		      {
			oprops = plist_put (oprops, XCAR (tem),
					    XCAR (XCDR (tem)));
			tem = XCDR (XCDR (tem));
		      }
		    props = oprops;
		  }

		aelt = Fassoc (elt, mode_line_proptrans_alist, Qnil);
		if (! NILP (aelt) && !NILP (Fequal (props, XCDR (aelt))))
		  {
		    /* AELT is what we want.  Move it to the front
		       without consing.  */
		    elt = XCAR (aelt);
		    mode_line_proptrans_alist
		      = move_elt_to_front (aelt, mode_line_proptrans_alist);
		  }
		else
		  {
		    Lisp_Object tem;

		    /* If AELT has the wrong props, it is useless.
		       so get rid of it.  */
		    if (! NILP (aelt))
		      mode_line_proptrans_alist
			= Fdelq (aelt, mode_line_proptrans_alist);

		    elt = Fcopy_sequence (elt);
		    /* PROPS might cause set-text-properties to signal
		       an error, so we call it via internal_condition_case_n,
		       to avoid an infloop in redisplay due to the error.  */
		    internal_condition_case_n (safe_set_text_properties,
					       4,
					       ((Lisp_Object [])
					       {make_fixnum (0),
						   Flength (elt),
						   props,
						   elt}),
					       Qt, safe_eval_handler);
		    /* Add this item to mode_line_proptrans_alist.  */
		    mode_line_proptrans_alist
		      = Fcons (Fcons (elt, props),
			       mode_line_proptrans_alist);
		    /* Truncate mode_line_proptrans_alist
		       to at most 50 elements.  */
		    tem = Fnthcdr (make_fixnum (50),
				   mode_line_proptrans_alist);
		    if (! NILP (tem))
		      XSETCDR (tem, Qnil);
		  }
	      }
	  }

	offset = 0;

	if (literal)
	  {
	    prec = precision - n;
	    switch (mode_line_target)
	      {
	      case MODE_LINE_NOPROP:
	      case MODE_LINE_TITLE:
		n += store_mode_line_noprop (SSDATA (elt), -1, prec);
		break;
	      case MODE_LINE_STRING:
		n += store_mode_line_string (NULL, elt, true, 0, prec, Qnil);
		break;
	      case MODE_LINE_DISPLAY:
		n += display_string (NULL, elt, Qnil, 0, 0, it,
				     0, prec, 0, STRING_MULTIBYTE (elt));
		break;
	      }

	    break;
	  }

	/* Handle the non-literal case.  */

	while ((precision <= 0 || n < precision)
	       && SREF (elt, offset) != 0
	       && (mode_line_target != MODE_LINE_DISPLAY
		   || it->current_x < it->last_visible_x))
	  {
	    ptrdiff_t last_offset = offset;

	    /* Advance to end of string or next format specifier.  */
	    while ((c = SREF (elt, offset++)) != '\0' && c != '%')
	      ;

	    if (offset - 1 != last_offset)
	      {
		ptrdiff_t nchars, nbytes;

		/* Output to end of string or up to '%'.  Field width
		   is length of string.  Don't output more than
		   PRECISION allows us.  */
		offset--;

		prec = c_string_width (SDATA (elt) + last_offset,
				       offset - last_offset, precision - n,
				       &nchars, &nbytes);

		switch (mode_line_target)
		  {
		  case MODE_LINE_NOPROP:
		  case MODE_LINE_TITLE:
		    n += store_mode_line_noprop (SSDATA (elt) + last_offset, 0, prec);
		    break;
		  case MODE_LINE_STRING:
		    {
		      ptrdiff_t bytepos = last_offset;
		      ptrdiff_t charpos = string_byte_to_char (elt, bytepos);
		      ptrdiff_t endpos = (precision <= 0
					  ? string_byte_to_char (elt, offset)
					  : charpos + nchars);
		      Lisp_Object mode_string
			= Fsubstring (elt, make_fixnum (charpos),
				      make_fixnum (endpos));
		      n += store_mode_line_string (NULL, mode_string, false,
						   0, 0, Qnil);
		    }
		    break;
		  case MODE_LINE_DISPLAY:
		    {
		      ptrdiff_t bytepos = last_offset;
		      ptrdiff_t charpos = string_byte_to_char (elt, bytepos);

		      if (precision <= 0)
			nchars = string_byte_to_char (elt, offset) - charpos;
		      n += display_string (NULL, elt, Qnil, 0, charpos,
					   it, 0, nchars, 0,
					   STRING_MULTIBYTE (elt));
		    }
		    break;
		  }
	      }
	    else /* c == '%' */
	      {
		ptrdiff_t percent_position = offset;

		/* Get the specified minimum width.  Zero means
		   don't pad.  */
		field = 0;
		while ((c = SREF (elt, offset++)) >= '0' && c <= '9')
		  field = field * 10 + c - '0';

		/* Don't pad beyond the total padding allowed.  */
		if (field_width - n > 0 && field > field_width - n)
		  field = field_width - n;

		/* Note that either PRECISION <= 0 or N < PRECISION.  */
		prec = precision - n;

		if (c == 'M')
		  n += display_mode_element (it, depth, field, prec,
					     Vglobal_mode_string, props,
					     risky);
		else if (c != 0)
		  {
		    bool multibyte;
		    ptrdiff_t bytepos, charpos;
		    const char *spec;
		    Lisp_Object string;

		    bytepos = percent_position;
		    charpos = (STRING_MULTIBYTE (elt)
			       ? string_byte_to_char (elt, bytepos)
			       : bytepos);
		    spec = decode_mode_spec (it->w, c, field, &string);
		    eassert (NILP (string) || STRINGP (string));
		    multibyte = !NILP (string) && STRING_MULTIBYTE (string);
		    /* Non-ASCII characters in SPEC should cause mode-line
		       element be displayed as a multibyte string.  */
		    ptrdiff_t nbytes = strlen (spec);
		    ptrdiff_t nchars, mb_nbytes;
		    parse_str_as_multibyte ((const unsigned char *)spec, nbytes,
					    &nchars, &mb_nbytes);
		    if (!(nbytes == nchars || nbytes != mb_nbytes))
		      multibyte = true;

		    switch (mode_line_target)
		      {
		      case MODE_LINE_NOPROP:
		      case MODE_LINE_TITLE:
			n += store_mode_line_noprop (spec, field, prec);
			break;
		      case MODE_LINE_STRING:
			{
			  Lisp_Object tem = build_string (spec);
			  props = Ftext_properties_at (make_fixnum (charpos), elt);
			  /* Should only keep face property in props */
			  n += store_mode_line_string (NULL, tem, false,
						       field, prec, props);
			}
			break;
		      case MODE_LINE_DISPLAY:
			{
			  int nglyphs_before, nwritten;

			  nglyphs_before = it->glyph_row->used[TEXT_AREA];
			  nwritten = display_string (spec, string, elt,
						     charpos, 0, it,
						     field, prec, 0,
						     multibyte);

			  /* Assign to the glyphs written above the
			     string where the `%x' came from, position
			     of the `%'.  */
			  if (nwritten > 0)
			    {
			      struct glyph *glyph
				= (it->glyph_row->glyphs[TEXT_AREA]
				   + nglyphs_before);
			      int i;

			      for (i = 0; i < nwritten; ++i)
				{
				  glyph[i].object = elt;
				  glyph[i].charpos = charpos;
				}

			      n += nwritten;
			    }
			}
			break;
		      }
		  }
		else /* c == 0 */
		  break;
	      }
	  }
      }
      break;

    case Lisp_Symbol:
      /* A symbol: process the value of the symbol recursively
	 as if it appeared here directly.  Avoid error if symbol void.
	 Special case: if value of symbol is a string, output the string
	 literally.  */
      {
	register Lisp_Object tem;

	/* If the variable is not marked as risky to set
	   then its contents are risky to use.  */
	if (NILP (Fget (elt, Qrisky_local_variable)))
	  risky = true;

	tem = Fboundp (elt);
	if (!NILP (tem))
	  {
	    tem = Fsymbol_value (elt);
	    /* If value is a string, output that string literally:
	       don't check for % within it.  */
	    if (STRINGP (tem))
	      literal = true;

	    if (!EQ (tem, elt))
	      {
		/* Give up right away for nil or t.  */
		elt = tem;
		goto tail_recurse;
	      }
	  }
      }
      break;

    case Lisp_Cons:
      {
	register Lisp_Object car, tem;

	/* A cons cell: five distinct cases.
	   If first element is :eval or :propertize, do something special.
	   If first element is a string or a cons, process all the elements
	   and effectively concatenate them.
	   If first element is a negative number, truncate displaying cdr to
	   at most that many characters.  If positive, pad (with spaces)
	   to at least that many characters.
	   If first element is a symbol, process the cadr or caddr recursively
	   according to whether the symbol's value is non-nil or nil.  */
	car = XCAR (elt);
	if (EQ (car, QCeval))
	  {
	    /* An element of the form (:eval FORM) means evaluate FORM
	       and use the result as mode line elements.  */

	    if (risky)
	      break;

	    if (CONSP (XCDR (elt)))
	      {
		Lisp_Object spec;
		spec = safe__eval (true, XCAR (XCDR (elt)));
		/* The :eval form could delete the frame stored in the
		   iterator, which will cause a crash if we try to
		   access faces and other fields (e.g., FRAME_KBOARD)
		   on that frame.  This is a nonsensical thing to do,
		   and signaling an error from redisplay might be
		   dangerous, but we cannot continue with an invalid frame.  */
		if (!FRAME_LIVE_P (it->f))
		  signal_error (":eval deleted the frame being displayed", elt);
		n += display_mode_element (it, depth, field_width - n,
					   precision - n, spec, props,
					   risky);
	      }
	  }
	else if (EQ (car, QCpropertize))
	  {
	    /* An element of the form (:propertize ELT PROPS...)
	       means display ELT but applying properties PROPS.  */

	    if (risky)
	      break;

	    if (CONSP (XCDR (elt)))
	      n += display_mode_element (it, depth, field_width - n,
					 precision - n, XCAR (XCDR (elt)),
					 XCDR (XCDR (elt)), risky);
	  }
	else if (SYMBOLP (car))
	  {
	    tem = Fboundp (car);
	    elt = XCDR (elt);
	    if (!CONSP (elt))
	      goto invalid;
	    /* elt is now the cdr, and we know it is a cons cell.
	       Use its car if CAR has a non-nil value.  */
	    if (!NILP (tem))
	      {
		tem = Fsymbol_value (car);
		if (!NILP (tem))
		  {
		    elt = XCAR (elt);
		    goto tail_recurse;
		  }
	      }
	    /* Symbol's value is nil (or symbol is unbound)
	       Get the cddr of the original list
	       and if possible find the caddr and use that.  */
	    elt = XCDR (elt);
	    if (NILP (elt))
	      break;
	    else if (!CONSP (elt))
	      goto invalid;
	    elt = XCAR (elt);
	    goto tail_recurse;
	  }
	else if (FIXNUMP (car))
	  {
	    register int lim = XFIXNUM (car);
	    elt = XCDR (elt);
	    if (lim < 0)
	      {
		/* Negative int means reduce maximum width.  */
		if (precision <= 0)
		  precision = -lim;
		else
		  precision = min (precision, -lim);
	      }
	    else if (lim > 0)
	      {
		/* Padding specified.  Don't let it be more than
		   current maximum.  */
		if (precision > 0)
		  lim = min (precision, lim);

		/* If that's more padding than already wanted, queue it.
		   But don't reduce padding already specified even if
		   that is beyond the current truncation point.  */
		field_width = max (lim, field_width);
	      }
	    goto tail_recurse;
	  }
	else if (STRINGP (car) || CONSP (car))
	  FOR_EACH_TAIL_SAFE (elt)
	    {
	      if (0 < precision && precision <= n)
		break;
	      n += display_mode_element (it, depth,
					 /* Pad after only the last
					    list element.  */
					 (! CONSP (XCDR (elt))
					  ? field_width - n
					  : 0),
					 precision - n, XCAR (elt),
					 props, risky);
	    }
      }
      break;

    default:
    invalid:
      elt = build_string ("*invalid*");
      goto tail_recurse;
    }

  /* Pad to FIELD_WIDTH.  */
  if (field_width > 0 && n < field_width)
    {
      switch (mode_line_target)
	{
	case MODE_LINE_NOPROP:
	case MODE_LINE_TITLE:
	  n += store_mode_line_noprop ("", field_width - n, 0);
	  break;
	case MODE_LINE_STRING:
	  n += store_mode_line_string ("", Qnil, false, field_width - n, 0,
				       Qnil);
	  break;
	case MODE_LINE_DISPLAY:
	  n += display_string ("", Qnil, Qnil, 0, 0, it, field_width - n,
			       0, 0, 0);
	  break;
	}
    }

  return n;
}

/* Store a mode-line string element in mode_line_string_list.

   If STRING is non-null, display that C string.  Otherwise, the Lisp
   string LISP_STRING is displayed.

   FIELD_WIDTH is the minimum number of output glyphs to produce.
   If STRING has fewer characters than FIELD_WIDTH, pad to the right
   with spaces.  FIELD_WIDTH <= 0 means don't pad.

   PRECISION is the maximum number of characters to output from
   STRING.  PRECISION <= 0  means don't truncate the string.

   If COPY_STRING, make a copy of LISP_STRING before adding
   properties to the string.

   PROPS are the properties to add to the string.
   The mode_line_string_face face property is always added to the string.
 */

static int
store_mode_line_string (const char *string, Lisp_Object lisp_string,
			bool copy_string,
			int field_width, int precision, Lisp_Object props)
{
  ptrdiff_t len;
  int n = 0;

  if (string != NULL)
    {
      len = strnlen (string, precision <= 0 ? SIZE_MAX : precision);
      lisp_string = make_string (string, len);
      if (NILP (props))
	props = mode_line_string_face_prop;
      else if (!NILP (mode_line_string_face))
	{
	  Lisp_Object face = plist_get (props, Qface);
	  props = Fcopy_sequence (props);
	  if (NILP (face))
	    face = mode_line_string_face;
	  else
	    face = list2 (face, mode_line_string_face);
	  props = plist_put (props, Qface, face);
	}
      Fadd_text_properties (make_fixnum (0), make_fixnum (len),
			    props, lisp_string);
    }
  else
    {
      len = SCHARS (lisp_string);
      if (precision > 0 && len > precision)
	{
	  len = precision;
	  lisp_string = Fsubstring (lisp_string, make_fixnum (0), make_fixnum (len));
	  precision = -1;
	}
      if (!NILP (mode_line_string_face))
	{
	  Lisp_Object face;
	  if (NILP (props))
	    props = Ftext_properties_at (make_fixnum (0), lisp_string);
	  face = plist_get (props, Qface);
	  if (NILP (face))
	    face = mode_line_string_face;
	  else
	    face = list2 (face, mode_line_string_face);
	  props = list2 (Qface, face);
	  if (copy_string)
	    lisp_string = Fcopy_sequence (lisp_string);
	}
      if (!NILP (props))
	Fadd_text_properties (make_fixnum (0), make_fixnum (len),
			      props, lisp_string);
    }

  if (len > 0)
    {
      mode_line_string_list = Fcons (lisp_string, mode_line_string_list);
      n += len;
    }

  if (field_width > len)
    {
      field_width -= len;
      lisp_string = Fmake_string (make_fixnum (field_width), make_fixnum (' '),
				  Qnil);
      if (!NILP (props))
	Fadd_text_properties (make_fixnum (0), make_fixnum (field_width),
			      props, lisp_string);
      mode_line_string_list = Fcons (lisp_string, mode_line_string_list);
      n += field_width;
    }

  return n;
}


DEFUN ("format-mode-line", Fformat_mode_line, Sformat_mode_line,
       1, 4, 0,
       doc: /* Format a string out of a mode line format specification.
First arg FORMAT specifies the mode line format (see `mode-line-format'
for details) to use.

By default, the format is evaluated for the currently selected window.

Optional second arg FACE specifies the face property to put on all
characters for which no face is specified.  The value nil means the
default face.  The value t means whatever face the window's mode line
currently uses (either `mode-line' or `mode-line-inactive',
depending on whether the window is the selected window or not).
An integer value means the value string has no text
properties.

Optional third and fourth args WINDOW and BUFFER specify the window
and buffer to use as the context for the formatting (defaults
are the selected window and the WINDOW's buffer).  */)
     (Lisp_Object format, Lisp_Object face,
      Lisp_Object window, Lisp_Object buffer)
{
  struct it it;
  int len;
  struct window *w;
  struct buffer *old_buffer = NULL;
  int face_id;
  bool no_props = FIXNUMP (face);
  specpdl_ref count = SPECPDL_INDEX ();
  Lisp_Object str;
  int string_start = 0;

  w = decode_any_window (window);
  XSETWINDOW (window, w);

  if (NILP (buffer))
    buffer = w->contents;
  CHECK_BUFFER (buffer);

  /* Make formatting the modeline a non-op when noninteractive, otherwise
     there will be problems later caused by a partially initialized frame.  */
  if (NILP (format) || noninteractive)
    return empty_unibyte_string;

  if (no_props)
    face = Qnil;

  face_id = (NILP (face) || EQ (face, Qdefault)) ? DEFAULT_FACE_ID
    : EQ (face, Qt) ? (EQ (window, selected_window)
		       ? MODE_LINE_ACTIVE_FACE_ID : MODE_LINE_INACTIVE_FACE_ID)
    : EQ (face, Qmode_line_active) ? MODE_LINE_ACTIVE_FACE_ID
    : EQ (face, Qmode_line_inactive) ? MODE_LINE_INACTIVE_FACE_ID
    : EQ (face, Qheader_line) ? HEADER_LINE_FACE_ID
    : EQ (face, Qtab_line) ? TAB_LINE_FACE_ID
    : EQ (face, Qtab_bar) ? TAB_BAR_FACE_ID
    : EQ (face, Qtool_bar) ? TOOL_BAR_FACE_ID
    : DEFAULT_FACE_ID;

  old_buffer = current_buffer;

  /* Save things including mode_line_proptrans_alist,
     and set that to nil so that we don't alter the outer value.  */
  record_unwind_protect (unwind_format_mode_line,
			 format_mode_line_unwind_data
			   (XFRAME (WINDOW_FRAME (w)),
			    old_buffer, selected_window, true));
  mode_line_proptrans_alist = Qnil;

  Fselect_window (window, Qt);
  set_buffer_internal_1 (XBUFFER (buffer));

  init_iterator (&it, w, -1, -1, NULL, face_id);

  if (no_props)
    {
      mode_line_target = MODE_LINE_NOPROP;
      mode_line_string_face_prop = Qnil;
      mode_line_string_list = Qnil;
      string_start = MODE_LINE_NOPROP_LEN (0);
    }
  else
    {
      mode_line_target = MODE_LINE_STRING;
      mode_line_string_list = Qnil;
      mode_line_string_face = face;
      mode_line_string_face_prop
	= NILP (face) ? Qnil : list2 (Qface, face);
    }

  push_kboard (FRAME_KBOARD (it.f));
  display_mode_element (&it, 0, 0, 0, format, Qnil, false);
  pop_kboard ();

  if (no_props)
    {
      len = MODE_LINE_NOPROP_LEN (string_start);
      str = make_string (mode_line_noprop_buf + string_start, len);
    }
  else
    {
      mode_line_string_list = Fnreverse (mode_line_string_list);
      str = Fmapconcat (Qidentity, mode_line_string_list,
			empty_unibyte_string);
    }

  return unbind_to (count, str);
}

/* Write a null-terminated, right justified decimal representation of
   the positive integer D to BUF using a minimal field width WIDTH.  */

static void
pint2str (register char *buf, register int width, register ptrdiff_t d)
{
  register char *p = buf;

  if (d <= 0)
    *p++ = '0';
  else
    {
      while (d > 0)
	{
	  *p++ = d % 10 + '0';
	  d /= 10;
	}
    }

  for (width -= (int) (p - buf); width > 0; --width)
    *p++ = ' ';
  *p-- = '\0';
  while (p > buf)
    {
      d = *buf;
      *buf++ = *p;
      *p-- = d;
    }
}

/* Write a null-terminated, right justified decimal and "human
   readable" representation of the nonnegative integer D to BUF using
   a minimal field width WIDTH.  D should be smaller than 999.5e24. */

static const char power_letter[] =
  {
    0,	 /* no letter */
    'k', /* kilo */
    'M', /* mega */
    'G', /* giga */
    'T', /* tera */
    'P', /* peta */
    'E', /* exa */
    'Z', /* zetta */
    'Y'	 /* yotta */
  };

static void
pint2hrstr (char *buf, int width, ptrdiff_t d)
{
  /* We aim to represent the nonnegative integer D as
     QUOTIENT.TENTHS * 10 ^ (3 * EXPONENT). */
  ptrdiff_t quotient = d;
  int remainder = 0;
  /* -1 means: do not use TENTHS. */
  int tenths = -1;
  int exponent = 0;

  /* Length of QUOTIENT.TENTHS as a string. */
  int length;

  char * psuffix;
  char * p;

  if (quotient >= 1000)
    {
      /* Scale to the appropriate EXPONENT. */
      do
	{
	  remainder = quotient % 1000;
	  quotient /= 1000;
	  exponent++;
	}
      while (quotient >= 1000);

      /* Round to nearest and decide whether to use TENTHS or not. */
      if (quotient <= 9)
	{
	  tenths = remainder / 100;
	  if (remainder % 100 >= 50)
	    {
	      if (tenths < 9)
		tenths++;
	      else
		{
		  quotient++;
		  if (quotient == 10)
		    tenths = -1;
		  else
		    tenths = 0;
		}
	    }
	}
      else
	if (remainder >= 500)
	  {
	    if (quotient < 999)
	      quotient++;
	    else
	      {
		quotient = 1;
		exponent++;
		tenths = 0;
	      }
	  }
    }

  /* Calculate the LENGTH of QUOTIENT.TENTHS as a string. */
  if (tenths == -1 && quotient <= 99)
    if (quotient <= 9)
      length = 1;
    else
      length = 2;
  else
    length = 3;
  p = psuffix = buf + max (width, length);

  /* Print EXPONENT. */
  *psuffix++ = power_letter[exponent];
  *psuffix = '\0';

  /* Print TENTHS. */
  if (tenths >= 0)
    {
      *--p = '0' + tenths;
      *--p = '.';
    }

  /* Print QUOTIENT. */
  do
    {
      int digit = quotient % 10;
      *--p =  '0' + digit;
    }
  while ((quotient /= 10) != 0);

  /* Print leading spaces. */
  while (buf < p)
    *--p = ' ';
}

/* Set a mnemonic character for coding_system (Lisp symbol) in BUF.
   If EOL_FLAG, set also a mnemonic character for end-of-line
   type of CODING_SYSTEM.  Return updated pointer into BUF.  */

static unsigned char invalid_eol_type[] = "(*invalid*)";

static char *
decode_mode_spec_coding (Lisp_Object coding_system, char *buf, bool eol_flag)
{
  Lisp_Object val;
  bool multibyte = !NILP (BVAR (current_buffer, enable_multibyte_characters));
  const unsigned char *eol_str;
  int eol_str_len;
  /* The EOL conversion we are using.  */
  Lisp_Object eoltype;

  val = CODING_SYSTEM_SPEC (coding_system);
  eoltype = Qnil;

  if (!VECTORP (val))		/* Not yet decided.  */
    {
      *buf++ = multibyte ? '-' : ' ';
      if (eol_flag)
	eoltype = eol_mnemonic_undecided;
      /* Don't mention EOL conversion if it isn't decided.  */
    }
  else
    {
      Lisp_Object attrs;
      Lisp_Object eolvalue;

      attrs = AREF (val, 0);
      eolvalue = AREF (val, 2);

      if (multibyte)
	buf += CHAR_STRING (XFIXNAT (CODING_ATTR_MNEMONIC (attrs)),
			    (unsigned char *) buf);
      else
	*buf++ = ' ';

      if (eol_flag)
	{
	  /* The EOL conversion that is normal on this system.  */

	  if (NILP (eolvalue))	/* Not yet decided.  */
	    eoltype = eol_mnemonic_undecided;
	  else if (VECTORP (eolvalue)) /* Not yet decided.  */
	    eoltype = eol_mnemonic_undecided;
	  else			/* eolvalue is Qunix, Qdos, or Qmac.  */
	    eoltype = (EQ (eolvalue, Qunix)
		       ? eol_mnemonic_unix
		       : EQ (eolvalue, Qdos)
		       ? eol_mnemonic_dos : eol_mnemonic_mac);
	}
    }

  if (eol_flag)
    {
      /* Mention the EOL conversion if it is not the usual one.  */
      if (STRINGP (eoltype))
	{
	  eol_str = SDATA (eoltype);
	  eol_str_len = SBYTES (eoltype);
	}
      else if (CHARACTERP (eoltype))
	{
	  int c = XFIXNAT (eoltype);
	  return buf + CHAR_STRING (c, (unsigned char *) buf);
	}
      else
	{
	  eol_str = invalid_eol_type;
	  eol_str_len = sizeof (invalid_eol_type) - 1;
	}
      memcpy (buf, eol_str, eol_str_len);
      buf += eol_str_len;
    }

  return buf;
}

/* Return the approximate percentage N is of D (rounding upward), or 99,
   whichever is less.  Assume 0 < D and 0 <= N <= D * INT_MAX / 100.  */

static int
percent99 (ptrdiff_t n, ptrdiff_t d)
{
  int percent = (d - 1 + 100.0 * n) / d;
  return min (percent, 99);
}

/* Return a string for the output of a mode line %-spec for window W,
   generated by character C.  FIELD_WIDTH > 0 means pad the string
   returned with spaces to that value.  Set *STRING to be a Lisp
   string if the resulting string is taken from that Lisp string;
   otherwise, set *STRING to Qnil.

   Note we operate on the current buffer for most purposes.  */

static char lots_of_dashes[] = "--------------------------------------------------------------------------------------------------------------------------------------------";

static const char *
decode_mode_spec (struct window *w, register int c, int field_width,
		  Lisp_Object *string)
{
  Lisp_Object obj;
  struct frame *f = XFRAME (WINDOW_FRAME (w));
  char *decode_mode_spec_buf = f->decode_mode_spec_buffer;
  /* We are going to use f->decode_mode_spec_buffer as the buffer to
     produce strings from numerical values, so limit preposterously
     large values of FIELD_WIDTH to avoid overrunning the buffer's
     end.  The size of the buffer is enough for FRAME_MESSAGE_BUF_SIZE
     bytes plus the terminating null.  */
  int width = min (field_width, FRAME_MESSAGE_BUF_SIZE (f));
  struct buffer *b = current_buffer;

  obj = Qnil;
  *string = Qnil;

  switch (c)
    {
    case '*':
      if (!NILP (BVAR (b, read_only)))
	return "%";
      if (BUF_MODIFF (b) > BUF_SAVE_MODIFF (b))
	return "*";
      return "-";

    case '+':
      /* This differs from %* only for a modified read-only buffer.  */
      if (BUF_MODIFF (b) > BUF_SAVE_MODIFF (b))
	return "*";
      if (!NILP (BVAR (b, read_only)))
	return "%";
      return "-";

    case '&':
      /* This differs from %* in ignoring read-only-ness.  */
      if (BUF_MODIFF (b) > BUF_SAVE_MODIFF (b))
	return "*";
      return "-";

    case '%':
      return "%";

    case '[':
      {
	int i;
	char *p;

	if (command_loop_level > 5)
	  return "[[[... ";
	p = decode_mode_spec_buf;
	for (i = 0; i < command_loop_level; i++)
	  *p++ = '[';
	*p = 0;
	return decode_mode_spec_buf;
      }

    case ']':
      {
	int i;
	char *p;

	if (command_loop_level > 5)
	  return " ...]]]";
	p = decode_mode_spec_buf;
	for (i = 0; i < command_loop_level; i++)
	  *p++ = ']';
	*p = 0;
	return decode_mode_spec_buf;
      }

    case '-':
      {
	register int i;

	/* Let lots_of_dashes be a string of infinite length.  */
	if (mode_line_target == MODE_LINE_NOPROP
	    || mode_line_target == MODE_LINE_STRING)
	  return "--";
	if (field_width <= 0
	    || field_width > sizeof (lots_of_dashes))
	  {
	    for (i = 0; i < FRAME_MESSAGE_BUF_SIZE (f) - 1; ++i)
	      decode_mode_spec_buf[i] = '-';
	    decode_mode_spec_buf[i] = '\0';
	    return decode_mode_spec_buf;
	  }
	else
	  return lots_of_dashes;
      }

    case 'b':
      obj = BVAR (b, name);
      break;

    case 'c':
    case 'C':
      /* %c, %C, and %l are ignored in `frame-title-format'.
         (In redisplay_internal, the frame title is drawn _before_ the
         windows are updated, so the stuff which depends on actual
         window contents (such as %l) may fail to render properly, or
         even crash emacs.)  */
      if (mode_line_target == MODE_LINE_TITLE)
	return "";
      else
	{
	  ptrdiff_t col = current_column ();
	  int disp_col = (c == 'C') ? col + 1 : col;
	  w->column_number_displayed = col;
	  pint2str (decode_mode_spec_buf, width, disp_col);
	  return decode_mode_spec_buf;
	}

    case 'e':
#if !defined SYSTEM_MALLOC && !defined HYBRID_MALLOC
      {
	if (NILP (Vmemory_full))
	  return "";
	else
	  return "!MEM FULL! ";
      }
#else
      return "";
#endif

    case 'F':
      /* %F displays the frame name.  */
      if (!NILP (f->title))
	return SSDATA (f->title);
      if (f->explicit_name || ! FRAME_WINDOW_P (f))
	return SSDATA (f->name);
      return "Emacs";

    case 'f':
      obj = BVAR (b, filename);
      break;

    case 'i':
      {
	ptrdiff_t size = ZV - BEGV;
	pint2str (decode_mode_spec_buf, width, size);
	return decode_mode_spec_buf;
      }

    case 'I':
      {
	ptrdiff_t size = ZV - BEGV;
	pint2hrstr (decode_mode_spec_buf, width, size);
	return decode_mode_spec_buf;
      }

    case 'l':
      {
	ptrdiff_t startpos, startpos_byte, line, linepos, linepos_byte;
	ptrdiff_t topline, nlines, height;
	ptrdiff_t junk;

	/* %c, %C, and %l are ignored in `frame-title-format'.  */
	if (mode_line_target == MODE_LINE_TITLE)
	  return "";

	startpos = marker_position (w->start);
	startpos_byte = marker_byte_position (w->start);
	height = WINDOW_TOTAL_LINES (w);
	/* We cannot cope with w->start being outside of the
	   accessible portion of the buffer; in particular,
	   display_count_lines call below might infloop if called with
	   startpos_byte outside of the [BEGV_BYTE..ZV_BYTE] region.
	   Such w->start means we were called in some "creative" way
	   when the buffer's restriction was changed, but the window
	   wasn't yet redisplayed after that.  If that happens, we
	   need to determine a new base line.  */
	if (!(BUF_BEGV_BYTE (b) <= startpos_byte
	      && startpos_byte <= BUF_ZV_BYTE (b)))
	  {
	    startpos = BUF_BEGV (b);
	    startpos_byte = BUF_BEGV_BYTE (b);
	    w->base_line_pos = 0;
	    w->base_line_number = 0;
	  }

	/* If we decided that this buffer isn't suitable for line numbers,
	   don't forget that too fast.  */
	if (w->base_line_pos == -1)
	  goto no_value;

	/* If the buffer is very big, don't waste time.  */
	if (FIXNUMP (Vline_number_display_limit)
	    && BUF_ZV (b) - BUF_BEGV (b) > XFIXNUM (Vline_number_display_limit))
	  {
	    w->base_line_pos = 0;
	    w->base_line_number = 0;
	    goto no_value;
	  }

	if (w->base_line_number > 0
	    && w->base_line_pos > 0
	    && w->base_line_pos <= startpos)
	  {
	    line = w->base_line_number;
	    linepos = w->base_line_pos;
	    linepos_byte = buf_charpos_to_bytepos (b, linepos);
	  }
	else
	  {
	    line = 1;
	    linepos = BUF_BEGV (b);
	    linepos_byte = BUF_BEGV_BYTE (b);
	  }

	/* Count lines from base line to window start position.  */
	nlines = display_count_lines (linepos_byte,
				      startpos_byte,
				      startpos, &junk);

	topline = nlines + line;

	/* Determine a new base line, if the old one is too close
	   or too far away, or if we did not have one.
	   "Too close" means it's plausible a scroll-down would
	   go back past it.  */
	if (startpos == BUF_BEGV (b))
	  {
	    w->base_line_number = topline;
	    w->base_line_pos = BUF_BEGV (b);
	  }
	else if (nlines < height + 25 || nlines > height * 3 + 50
		 || linepos == BUF_BEGV (b))
	  {
	    ptrdiff_t limit = BUF_BEGV (b);
	    ptrdiff_t limit_byte = BUF_BEGV_BYTE (b);
	    ptrdiff_t position;
	    ptrdiff_t distance
	      = (line_number_display_limit_width < 0 ? 0
		 : INT_MULTIPLY_WRAPV (line_number_display_limit_width,
				       height * 2 + 30,
				       &distance)
		 ? PTRDIFF_MAX : distance);

	    if (startpos - distance > limit)
	      {
		limit = startpos - distance;
		limit_byte = CHAR_TO_BYTE (limit);
	      }

	    nlines = display_count_lines (startpos_byte,
					  limit_byte,
					  - (height * 2 + 30),
					  &position);
	    /* If we couldn't find the lines we wanted within
	       line_number_display_limit_width chars per line,
	       give up on line numbers for this window.  */
	    if (position == limit_byte && limit == startpos - distance)
	      {
		w->base_line_pos = -1;
		w->base_line_number = 0;
		goto no_value;
	      }

	    w->base_line_number = topline - nlines;
	    w->base_line_pos = BYTE_TO_CHAR (position);
	  }

	/* Now count lines from the start pos to point.  */
	nlines = display_count_lines (startpos_byte,
				      PT_BYTE, PT, &junk);

	/* Record that we did display the line number.  */
	line_number_displayed = true;

	/* Make the string to show.  */
	pint2str (decode_mode_spec_buf, width, topline + nlines);
	return decode_mode_spec_buf;
    no_value:
        {
	  char *p = decode_mode_spec_buf;
	  int pad = width - 2;
	  while (pad-- > 0)
	    *p++ = ' ';
	  *p++ = '?';
	  *p++ = '?';
	  *p = '\0';
	  return decode_mode_spec_buf;
	}
      }
      break;

    case 'm':
      obj = BVAR (b, mode_name);
      break;

    case 'n':
      if (BUF_BEGV (b) > BUF_BEG (b) || BUF_ZV (b) < BUF_Z (b))
	return " Narrow";
      break;

      /* Display the "degree of travel" of the window through the buffer.  */
    case 'o':
      {
        ptrdiff_t toppos = marker_position (w->start);
        ptrdiff_t botpos = BUF_Z (b) - w->window_end_pos;
        ptrdiff_t begv = BUF_BEGV (b);
        ptrdiff_t zv = BUF_ZV (b);

        if (zv <= botpos)
          return toppos <= begv ? "All" : "Bottom";
        else if (toppos <= begv)
          return "Top";
        else
          {
          sprintf (decode_mode_spec_buf, "%2d%%",
                   percent99 (toppos - begv, (toppos - begv) + (zv - botpos)));
          return decode_mode_spec_buf;
          }
      }

      /* Display percentage of buffer above the top of the screen.  */
    case 'p':
      {
	ptrdiff_t pos = marker_position (w->start);
	ptrdiff_t begv = BUF_BEGV (b);
	ptrdiff_t zv = BUF_ZV (b);

	if (w->window_end_pos <= BUF_Z (b) - zv)
	  return pos <= begv ? "All" : "Bottom";
	else if (pos <= begv)
	  return "Top";
	else
	  {
	    sprintf (decode_mode_spec_buf, "%2d%%",
		     percent99 (pos - begv, zv - begv));
	    return decode_mode_spec_buf;
	  }
      }

      /* Display percentage of size above the bottom of the screen.  */
    case 'P':
      {
	ptrdiff_t toppos = marker_position (w->start);
	ptrdiff_t botpos = BUF_Z (b) - w->window_end_pos;
	ptrdiff_t begv = BUF_BEGV (b);
	ptrdiff_t zv = BUF_ZV (b);

	if (zv <= botpos)
	  return toppos <= begv ? "All" : "Bottom";
	else
	  {
	    sprintf (decode_mode_spec_buf,
		     &"Top%2d%%"[begv < toppos ? sizeof "Top" - 1 : 0],
		     percent99 (botpos - begv, zv - begv));
	    return decode_mode_spec_buf;
	  }
      }

      /* Display percentage offsets of top and bottom of the window,
         using "All" (but not "Top" or "Bottom") where appropriate.  */
    case 'q':
      {
        ptrdiff_t toppos = marker_position (w->start);
        ptrdiff_t botpos = BUF_Z (b) - w->window_end_pos;
        ptrdiff_t begv = BUF_BEGV (b);
        ptrdiff_t zv = BUF_ZV (b);
        int top_perc, bot_perc;

        if ((toppos <= begv) && (zv <= botpos))
          return "All   ";

        top_perc = toppos <= begv ? 0 : percent99 (toppos - begv, zv - begv);
        bot_perc = zv <= botpos ? 100 : percent99 (botpos - begv, zv - begv);

        if (top_perc == bot_perc)
          sprintf (decode_mode_spec_buf, "%d%%", top_perc);
        else
          sprintf (decode_mode_spec_buf, "%d-%d%%", top_perc, bot_perc);

        return decode_mode_spec_buf;
      }

    case 's':
      /* status of process */
      obj = Fget_buffer_process (Fcurrent_buffer ());
      if (NILP (obj))
	return "no process";
#ifndef MSDOS
      obj = Fsymbol_name (Fprocess_status (obj));
#endif
      break;

    case '@':
      {
	specpdl_ref count = inhibit_garbage_collection ();
	Lisp_Object curdir = BVAR (current_buffer, directory);
	Lisp_Object val = Qnil;

	if (STRINGP (curdir))
	  val = safe_call1 (intern ("file-remote-p"), curdir);

	val = unbind_to (count, val);

	if (NILP (val))
	  return "-";
	else
	  return "@";
      }

    case 'z':
      /* coding-system (not including end-of-line format) */
    case 'Z':
      /* coding-system (including end-of-line type) */
      {
	bool eol_flag = (c == 'Z');
	char *p = decode_mode_spec_buf;

	if (! FRAME_WINDOW_P (f))
	  {
	    /* No need to mention EOL here--the terminal never needs
	       to do EOL conversion.  */
	    p = decode_mode_spec_coding (CODING_ID_NAME
					 (FRAME_KEYBOARD_CODING (f)->id),
					 p, false);
	    p = decode_mode_spec_coding (CODING_ID_NAME
					 (FRAME_TERMINAL_CODING (f)->id),
					 p, false);
	  }
	p = decode_mode_spec_coding (BVAR (b, buffer_file_coding_system),
				     p, eol_flag);

#if false /* This proves to be annoying; I think we can do without. -- rms.  */
#ifdef subprocesses
	obj = Fget_buffer_process (Fcurrent_buffer ());
	if (PROCESSP (obj))
	  {
	    p = decode_mode_spec_coding
	      (XPROCESS (obj)->decode_coding_system, p, eol_flag);
	    p = decode_mode_spec_coding
	      (XPROCESS (obj)->encode_coding_system, p, eol_flag);
	  }
#endif /* subprocesses */
#endif /* false */
	*p = 0;
	return decode_mode_spec_buf;
      }
    }

  if (STRINGP (obj))
    {
      *string = obj;
      return SSDATA (obj);
    }
  else
    return "";
}

/* Return the number of lines between start_byte and end_byte in the
   current buffer. */

ptrdiff_t
count_lines (ptrdiff_t start_byte, ptrdiff_t end_byte)
{
  ptrdiff_t ignored;
  return display_count_lines (start_byte, end_byte, ZV, &ignored);
}

/* Count up to COUNT lines starting from START_BYTE.  COUNT negative
   means count lines back from START_BYTE.  But don't go beyond
   LIMIT_BYTE.  Return the number of lines thus found (always
   nonnegative).

   Set *BYTE_POS_PTR to the byte position where we stopped.  This is
   either the position COUNT lines after/before START_BYTE, if we
   found COUNT lines, or LIMIT_BYTE if we hit the limit before finding
   COUNT lines.  */

static ptrdiff_t
display_count_lines (ptrdiff_t start_byte,
		     ptrdiff_t limit_byte, ptrdiff_t count,
		     ptrdiff_t *byte_pos_ptr)
{
  register unsigned char *cursor;
  unsigned char *base;

  register ptrdiff_t ceiling;
  register unsigned char *ceiling_addr;
  ptrdiff_t orig_count = count;

  /* If we are not in selective display mode,
     check only for newlines.  */
  bool selective_display
    = (!NILP (BVAR (current_buffer, selective_display))
       && !FIXNUMP (BVAR (current_buffer, selective_display)));

  if (count > 0)
    {
      while (start_byte < limit_byte)
	{
	  ceiling =  BUFFER_CEILING_OF (start_byte);
	  ceiling = min (limit_byte - 1, ceiling);
	  ceiling_addr = BYTE_POS_ADDR (ceiling) + 1;
	  base = (cursor = BYTE_POS_ADDR (start_byte));

	  do
	    {
	      if (selective_display)
		{
		  while (*cursor != '\n' && *cursor != 015
			 && ++cursor != ceiling_addr)
		    continue;
		  if (cursor == ceiling_addr)
		    break;
		}
	      else
		{
		  cursor = memchr (cursor, '\n', ceiling_addr - cursor);
		  if (! cursor)
		    break;
		}

	      cursor++;

	      if (--count == 0)
		{
		  start_byte += cursor - base;
		  *byte_pos_ptr = start_byte;
		  return orig_count;
		}
	    }
	  while (cursor < ceiling_addr);

	  start_byte += ceiling_addr - base;
	}
    }
  else
    {
      while (start_byte > limit_byte)
	{
	  ceiling = BUFFER_FLOOR_OF (start_byte - 1);
	  ceiling = max (limit_byte, ceiling);
	  ceiling_addr = BYTE_POS_ADDR (ceiling);
	  base = (cursor = BYTE_POS_ADDR (start_byte - 1) + 1);
	  while (true)
	    {
	      if (selective_display)
		{
		  while (--cursor >= ceiling_addr
			 && *cursor != '\n' && *cursor != 015)
		    continue;
		  if (cursor < ceiling_addr)
		    break;
		}
	      else
		{
		  cursor = memrchr (ceiling_addr, '\n', cursor - ceiling_addr);
		  if (! cursor)
		    break;
		}

	      if (++count == 0)
		{
		  start_byte += cursor - base + 1;
		  *byte_pos_ptr = start_byte;
		  /* When scanning backwards, we should
		     not count the newline posterior to which we stop.  */
		  return - orig_count - 1;
		}
	    }
	  start_byte += ceiling_addr - base;
	}
    }

  *byte_pos_ptr = limit_byte;

  if (count < 0)
    return - orig_count + count;
  return orig_count - count;

}



/***********************************************************************
			 Displaying strings
 ***********************************************************************/

/* Display a NUL-terminated string, starting with index START.

   If STRING is non-null, display that C string.  Otherwise, the Lisp
   string LISP_STRING is displayed.  There's a case that STRING is
   non-null and LISP_STRING is not nil.  It means STRING is a string
   data of LISP_STRING.  In that case, we display LISP_STRING while
   ignoring its text properties.

   If FACE_STRING is not nil, FACE_STRING_POS is a position in
   FACE_STRING.  Display STRING or LISP_STRING with the face at
   FACE_STRING_POS in FACE_STRING:

   Display the string in the environment given by IT, but use the
   standard display table, temporarily.

   FIELD_WIDTH is the minimum number of output glyphs to produce.
   If STRING has fewer characters than FIELD_WIDTH, pad to the right
   with spaces.  If STRING has more characters, more than FIELD_WIDTH
   glyphs will be produced.  FIELD_WIDTH <= 0 means don't pad.

   PRECISION is the maximum number of characters to output from
   STRING.  PRECISION < 0  means don't truncate the string.

   This is roughly equivalent to printf format specifiers:

   FIELD_WIDTH	PRECISION	PRINTF
   ----------------------------------------
   -1		-1		%s
   -1		10		%.10s
   10		-1		%10s
   20		10		%20.10s

   MULTIBYTE zero means do not display multibyte chars, > 0 means do
   display them, and < 0 means obey the current buffer's value of
   enable_multibyte_characters.

   Value is the number of columns displayed.  */

static int
display_string (const char *string, Lisp_Object lisp_string, Lisp_Object face_string,
		ptrdiff_t face_string_pos, ptrdiff_t start, struct it *it,
		int field_width, int precision, int max_x, int multibyte)
{
  int hpos_at_start = it->hpos;
  int saved_face_id = it->face_id;
  struct glyph_row *row = it->glyph_row;
  ptrdiff_t it_charpos;

  /* Initialize the iterator IT for iteration over STRING beginning
     with index START.  */
  reseat_to_string (it, NILP (lisp_string) ? string : NULL, lisp_string,
                    start, precision, field_width, multibyte);

  if (string && STRINGP (lisp_string))
    /* LISP_STRING is the one returned by decode_mode_spec.  We should
       ignore its text properties.  */
    it->stop_charpos = it->end_charpos;

  /* If displaying STRING, set up the face of the iterator from
     FACE_STRING, if that's given.  */
  if (STRINGP (face_string))
    {
      ptrdiff_t endptr;
      struct face *face;

      it->face_id
	= face_at_string_position (it->w, face_string, face_string_pos,
	                           0, &endptr, it->base_face_id, false, 0);
      face = FACE_FROM_ID (it->f, it->face_id);
      it->face_box_p = face->box != FACE_NO_BOX;

      /* If we have a display spec, but there's no Lisp string being
	 displayed, then check whether we've got one from the
	 :propertize being passed in and use that.  */
      if (NILP (lisp_string))
	{
	  Lisp_Object display = Fget_text_property (make_fixnum (0), Qdisplay,
						    face_string);
	  if (!NILP (display))
	    {
	      Lisp_Object min_width = plist_get (display, Qmin_width);
	      if (!NILP (min_width))
		display_min_width (it, 0, face_string, min_width);
	    }
	}
    }

  /* Set max_x to the maximum allowed X position.  Don't let it go
     beyond the right edge of the window.  */
  if (max_x <= 0)
    max_x = it->last_visible_x;
  else
    max_x = min (max_x, it->last_visible_x);

  /* Skip over display elements that are not visible because IT->w is
     hscrolled.  */
  if (it->current_x < it->first_visible_x)
    move_it_in_display_line_to (it, 100000, it->first_visible_x,
				MOVE_TO_POS | MOVE_TO_X);

  row->ascent = it->max_ascent;
  row->height = it->max_ascent + it->max_descent;
  row->phys_ascent = it->max_phys_ascent;
  row->phys_height = it->max_phys_ascent + it->max_phys_descent;
  row->extra_line_spacing = it->max_extra_line_spacing;

  if (STRINGP (it->string))
    it_charpos = IT_STRING_CHARPOS (*it);
  else
    it_charpos = IT_CHARPOS (*it);

  /* This condition is for the case that we are called with current_x
     past last_visible_x.  */
  while (it->current_x < max_x)
    {
      int x_before, x, n_glyphs_before, i, nglyphs;

      /* Get the next display element.  */
      if (!get_next_display_element (it))
	break;

      /* Produce glyphs.  */
      x_before = it->current_x;
      n_glyphs_before = row->used[TEXT_AREA];
      PRODUCE_GLYPHS (it);

      nglyphs = row->used[TEXT_AREA] - n_glyphs_before;
      i = 0;
      x = x_before;
      while (i < nglyphs)
	{
	  struct glyph *glyph = row->glyphs[TEXT_AREA] + n_glyphs_before + i;

	  if (it->line_wrap != TRUNCATE
	      && x + glyph->pixel_width > max_x)
	    {
	      /* End of continued line or max_x reached.  */
	      if (CHAR_GLYPH_PADDING_P (*glyph))
		{
		  /* A wide character is unbreakable.  */
		  if (row->reversed_p)
		    unproduce_glyphs (it, row->used[TEXT_AREA]
				      - n_glyphs_before);
		  row->used[TEXT_AREA] = n_glyphs_before;
		  it->current_x = x_before;
		}
	      else
		{
		  if (row->reversed_p)
		    unproduce_glyphs (it, row->used[TEXT_AREA]
				      - (n_glyphs_before + i));
		  row->used[TEXT_AREA] = n_glyphs_before + i;
		  it->current_x = x;
		}
	      break;
	    }
	  else if (x + glyph->pixel_width >= it->first_visible_x)
	    {
	      /* Glyph is at least partially visible.  */
	      ++it->hpos;
	      if (x < it->first_visible_x)
		row->x = x - it->first_visible_x;
	    }
	  else
	    {
	      /* Glyph is off the left margin of the display area.
		 Should not happen.  */
	      emacs_abort ();
	    }

	  row->ascent = max (row->ascent, it->max_ascent);
	  row->height = max (row->height, it->max_ascent + it->max_descent);
	  row->phys_ascent = max (row->phys_ascent, it->max_phys_ascent);
	  row->phys_height = max (row->phys_height,
				  it->max_phys_ascent + it->max_phys_descent);
	  row->extra_line_spacing = max (row->extra_line_spacing,
					 it->max_extra_line_spacing);
	  x += glyph->pixel_width;
	  ++i;
	}

      /* Stop if max_x reached.  */
      if (i < nglyphs)
	break;

      /* Stop at line ends.  */
      if (ITERATOR_AT_END_OF_LINE_P (it))
	{
	  it->continuation_lines_width = 0;
	  break;
	}

      set_iterator_to_next (it, true);
      if (STRINGP (it->string))
	it_charpos = IT_STRING_CHARPOS (*it);
      else
	it_charpos = IT_CHARPOS (*it);

      /* Stop if truncating at the right edge.  */
      if (it->line_wrap == TRUNCATE
	  && it->current_x >= it->last_visible_x)
	{
	  /* Add truncation mark, but don't do it if the line is
	     truncated at a padding space.  */
	  if (it_charpos < it->string_nchars)
	    {
	      if (!FRAME_WINDOW_P (it->f))
		{
		  int ii, n;

		  if (it->current_x > it->last_visible_x)
		    {
		      if (!row->reversed_p)
			{
			  for (ii = row->used[TEXT_AREA] - 1; ii > 0; --ii)
			    if (!CHAR_GLYPH_PADDING_P (row->glyphs[TEXT_AREA][ii]))
			      break;
			}
		      else
			{
			  for (ii = 0; ii < row->used[TEXT_AREA]; ii++)
			    if (!CHAR_GLYPH_PADDING_P (row->glyphs[TEXT_AREA][ii]))
			      break;
			  unproduce_glyphs (it, ii + 1);
			  ii = row->used[TEXT_AREA] - (ii + 1);
			}
		      for (n = row->used[TEXT_AREA]; ii < n; ++ii)
			{
			  row->used[TEXT_AREA] = ii;
			  produce_special_glyphs (it, IT_TRUNCATION);
			}
		    }
		  produce_special_glyphs (it, IT_TRUNCATION);
		}
	      row->truncated_on_right_p = true;
	    }
	  break;
	}
    }

  /* Maybe insert a truncation at the left.  */
  if (it->first_visible_x
      && it_charpos > 0)
    {
      if (!FRAME_WINDOW_P (it->f)
	  || (row->reversed_p
	      ? WINDOW_RIGHT_FRINGE_WIDTH (it->w)
	      : WINDOW_LEFT_FRINGE_WIDTH (it->w)) == 0)
	insert_left_trunc_glyphs (it);
      row->truncated_on_left_p = true;
    }

  it->face_id = saved_face_id;

  /* Value is number of columns displayed.  */
  return it->hpos - hpos_at_start;
}



/* This is like a combination of memq and assq.  Return 1/2 if PROPVAL
   appears as an element of LIST or as the car of an element of LIST.
   If PROPVAL is a list, compare each element against LIST in that
   way, and return 1/2 if any element of PROPVAL is found in LIST.
   Otherwise return 0.  This function cannot quit.
   The return value is 2 if the text is invisible but with an ellipsis
   and 1 if it's invisible and without an ellipsis.  */

int
invisible_prop (Lisp_Object propval, Lisp_Object list)
{
  Lisp_Object tail, proptail;

  for (tail = list; CONSP (tail); tail = XCDR (tail))
    {
      register Lisp_Object tem;
      tem = XCAR (tail);
      if (EQ (propval, tem))
	return 1;
      if (CONSP (tem) && EQ (propval, XCAR (tem)))
	return NILP (XCDR (tem)) ? 1 : 2;
    }

  if (CONSP (propval))
    {
      for (proptail = propval; CONSP (proptail); proptail = XCDR (proptail))
	{
	  Lisp_Object propelt;
	  propelt = XCAR (proptail);
	  for (tail = list; CONSP (tail); tail = XCDR (tail))
	    {
	      register Lisp_Object tem;
	      tem = XCAR (tail);
	      if (EQ (propelt, tem))
		return 1;
	      if (CONSP (tem) && EQ (propelt, XCAR (tem)))
		return NILP (XCDR (tem)) ? 1 : 2;
	    }
	}
    }

  return 0;
}

DEFUN ("invisible-p", Finvisible_p, Sinvisible_p, 1, 1, 0,
       doc: /* Non-nil if text properties at POS cause text there to be currently invisible.
POS should be a marker or a buffer position; the value of the `invisible'
property at that position in the current buffer is examined.
POS can also be the actual value of the `invisible' text or overlay
property of the text of interest, in which case the value itself is
examined.

The non-nil value returned can be t for currently invisible text that is
entirely hidden on display, or some other non-nil, non-t value if the
text is replaced by an ellipsis.

Note that whether text with `invisible' property is actually hidden on
display may depend on `buffer-invisibility-spec', which see.  */)
  (Lisp_Object pos)
{
  Lisp_Object prop
    = (FIXNATP (pos) || MARKERP (pos)
       ? Fget_char_property (pos, Qinvisible, Qnil)
       : pos);
  int invis = TEXT_PROP_MEANS_INVISIBLE (prop);
  return (invis == 0 ? Qnil
	  : invis == 1 ? Qt
	  : make_fixnum (invis));
}

/* Calculate a width or height in pixels from a specification using
   the following elements:

   SPEC ::=
     NUM      - a (fractional) multiple of the default font width/height
     (NUM)    - specifies exactly NUM pixels
     UNIT     - a fixed number of pixels, see below.
     ELEMENT  - size of a display element in pixels, see below.
     (NUM . SPEC) - equals NUM * SPEC
     (+ SPEC SPEC ...)  - add pixel values
     (- SPEC SPEC ...)  - subtract pixel values
     (- SPEC)           - negate pixel value

   NUM ::=
     INT or FLOAT   - a number constant
     SYMBOL         - use symbol's (buffer local) variable binding.

   UNIT ::=
     in       - pixels per inch  *)
     mm       - pixels per 1/1000 meter  *)
     cm       - pixels per 1/100 meter   *)
     width    - width of current font in pixels.
     height   - height of current font in pixels.

     *) using the ratio(s) defined in display-pixels-per-inch.

   ELEMENT ::=

     left-fringe          - left fringe width in pixels
     right-fringe         - right fringe width in pixels

     left-margin          - left margin width in pixels
     right-margin         - right margin width in pixels

     scroll-bar           - scroll-bar area width in pixels

   Examples:

   Pixels corresponding to 5 inches:
     (5 . in)

   Total width of non-text areas on left side of window (if scroll-bar is on left):
     '(space :width (+ left-fringe left-margin scroll-bar))

   Align to first text column (in header line):
     '(space :align-to 0)

   Align to middle of text area minus half the width of variable `my-image'
   containing a loaded image:
     '(space :align-to (0.5 . (- text my-image)))

   Width of left margin minus width of 1 character in the default font:
     '(space :width (- left-margin 1))

   Width of left margin minus width of 2 characters in the current font:
     '(space :width (- left-margin (2 . width)))

   Center 1 character over left-margin (in header line):
     '(space :align-to (+ left-margin (0.5 . left-margin) -0.5))

   Different ways to express width of left fringe plus left margin minus one pixel:
     '(space :width (- (+ left-fringe left-margin) (1)))
     '(space :width (+ left-fringe left-margin (- (1))))
     '(space :width (+ left-fringe left-margin (-1)))

   If ALIGN_TO is NULL, returns the result in *RES.  If ALIGN_TO is
   non-NULL, the value of *ALIGN_TO is a window-relative pixel
   coordinate, and *RES is the additional pixel width from that point
   till the end of the stretch glyph.

   WIDTH_P non-zero means take the width dimension or X coordinate of
   the object specified by PROP, WIDTH_P zero means take the height
   dimension or the Y coordinate.  (Therefore, if ALIGN_TO is
   non-NULL, WIDTH_P should be non-zero.)

   FONT is the font of the face of the surrounding text.

   The return value is non-zero if width or height were successfully
   calculated, i.e. if PROP is a valid spec.  */

static bool
calc_pixel_width_or_height (double *res, struct it *it, Lisp_Object prop,
			    struct font *font, bool width_p, int *align_to)
{
  /* Don't adjust for line number if we didn't yet produce it for this
     screen line.  This is for when this function is called from
     move_it_in_display_line_to that was called by display_line to get
     past the glyphs hscrolled off the left side of the window.  */
  int lnum_pixel_width = it->line_number_produced_p ? it->lnum_pixel_width : 0;
  double pixels;

# define OK_PIXELS(val) (*res = (val), true)
# define OK_ALIGN_TO(val) (*align_to = (val), true)

  if (NILP (prop))
    return OK_PIXELS (0);

  eassert (FRAME_LIVE_P (it->f));

  if (SYMBOLP (prop))
    {
      if (SCHARS (SYMBOL_NAME (prop)) == 2)
	{
	  char *unit = SSDATA (SYMBOL_NAME (prop));

	  /* The UNIT expression, e.g. as part of (NUM . UNIT).  */
	  if (unit[0] == 'i' && unit[1] == 'n')
	    pixels = 1.0;
	  else if (unit[0] == 'm' && unit[1] == 'm')
	    pixels = 25.4;
	  else if (unit[0] == 'c' && unit[1] == 'm')
	    pixels = 2.54;
	  else
	    pixels = 0;
	  if (pixels > 0)
	    {
	      double ppi = (width_p ? FRAME_RES_X (it->f)
			    : FRAME_RES_Y (it->f));

	      if (ppi > 0)
		return OK_PIXELS (ppi / pixels);
	      return false;
	    }
	}

#ifdef HAVE_WINDOW_SYSTEM
      /* 'height': the height of FONT.  */
      if (EQ (prop, Qheight))
	return OK_PIXELS (font
			  ? normal_char_height (font, -1)
			  : FRAME_LINE_HEIGHT (it->f));
      /* 'width': the width of FONT.  */
      if (EQ (prop, Qwidth))
	return OK_PIXELS (font
			  ? FONT_WIDTH (font)
			  : FRAME_COLUMN_WIDTH (it->f));
#else
      if (EQ (prop, Qheight) || EQ (prop, Qwidth))
	return OK_PIXELS (1);
#endif

      /* 'text': the width or height of the text area.  */
      if (EQ (prop, Qtext))
	  return OK_PIXELS (width_p
			    ? (window_box_width (it->w, TEXT_AREA)
			       - lnum_pixel_width)
			    : WINDOW_BOX_HEIGHT_NO_MODE_LINE (it->w));

      /* ':align_to'.  First time we compute the value, window
	 elements are interpreted as the position of the element's
	 left edge.  */
      if (align_to && *align_to < 0)
	{
	  *res = 0;
	  /* 'left': left edge of the text area.  */
	  if (EQ (prop, Qleft))
	    return OK_ALIGN_TO (window_box_left_offset (it->w, TEXT_AREA)
				+ lnum_pixel_width);
	  /* 'right': right edge of the text area.  */
	  if (EQ (prop, Qright))
	    return OK_ALIGN_TO (window_box_right_offset (it->w, TEXT_AREA));
	  /* 'center': the center of the text area.  */
	  if (EQ (prop, Qcenter))
	    return OK_ALIGN_TO (window_box_left_offset (it->w, TEXT_AREA)
				+ lnum_pixel_width
				+ window_box_width (it->w, TEXT_AREA) / 2);
	  /* 'left-fringe': left edge of the left fringe.  */
	  if (EQ (prop, Qleft_fringe))
	    return OK_ALIGN_TO (WINDOW_HAS_FRINGES_OUTSIDE_MARGINS (it->w)
				? WINDOW_LEFT_SCROLL_BAR_AREA_WIDTH (it->w)
				: window_box_right_offset (it->w, LEFT_MARGIN_AREA));
	  /* 'right-fringe': left edge of the right fringe.  */
	  if (EQ (prop, Qright_fringe))
	    return OK_ALIGN_TO (WINDOW_HAS_FRINGES_OUTSIDE_MARGINS (it->w)
				? window_box_right_offset (it->w, RIGHT_MARGIN_AREA)
				: window_box_right_offset (it->w, TEXT_AREA));
	  /* 'left-margin': left edge of the left display margin.  */
	  if (EQ (prop, Qleft_margin))
	    return OK_ALIGN_TO (window_box_left_offset (it->w, LEFT_MARGIN_AREA));
	  /* 'right-margin': left edge of the right display margin.  */
	  if (EQ (prop, Qright_margin))
	    return OK_ALIGN_TO (window_box_left_offset (it->w, RIGHT_MARGIN_AREA));
	  /* 'scroll-bar': left edge of the vertical scroll bar.  */
	  if (EQ (prop, Qscroll_bar))
	    return OK_ALIGN_TO (WINDOW_HAS_VERTICAL_SCROLL_BAR_ON_LEFT (it->w)
				? 0
				: (window_box_right_offset (it->w, RIGHT_MARGIN_AREA)
				   + (WINDOW_HAS_FRINGES_OUTSIDE_MARGINS (it->w)
				      ? WINDOW_RIGHT_FRINGE_WIDTH (it->w)
				      : 0)));
	}
      else
	{
	  /* Otherwise, the elements stand for their width.  */
	  if (EQ (prop, Qleft_fringe))
	    return OK_PIXELS (WINDOW_LEFT_FRINGE_WIDTH (it->w));
	  if (EQ (prop, Qright_fringe))
	    return OK_PIXELS (WINDOW_RIGHT_FRINGE_WIDTH (it->w));
	  if (EQ (prop, Qleft_margin))
	    return OK_PIXELS (WINDOW_LEFT_MARGIN_WIDTH (it->w));
	  if (EQ (prop, Qright_margin))
	    return OK_PIXELS (WINDOW_RIGHT_MARGIN_WIDTH (it->w));
	  if (EQ (prop, Qscroll_bar))
	    return OK_PIXELS (WINDOW_SCROLL_BAR_AREA_WIDTH (it->w));
	}

      prop = buffer_local_value (prop, it->w->contents);
      if (BASE_EQ (prop, Qunbound))
	prop = Qnil;
    }

  if (NUMBERP (prop))
    {
      int base_unit = (width_p
		       ? FRAME_COLUMN_WIDTH (it->f)
		       : FRAME_LINE_HEIGHT (it->f));
      if (width_p && align_to && *align_to < 0)
	return OK_PIXELS (XFLOATINT (prop) * base_unit + lnum_pixel_width);
      return OK_PIXELS (XFLOATINT (prop) * base_unit);
    }

  if (CONSP (prop))
    {
      Lisp_Object car = XCAR (prop);
      Lisp_Object cdr = XCDR (prop);

      if (SYMBOLP (car))
	{
#ifdef HAVE_WINDOW_SYSTEM
	  /* '(image PROPS...)': width or height of the specified image.  */
	  if (FRAME_WINDOW_P (it->f)
	      && valid_image_p (prop))
	    {
	      ptrdiff_t id = lookup_image (it->f, prop, it->face_id);
	      struct image *img = IMAGE_FROM_ID (it->f, id);

	      return OK_PIXELS (width_p ? img->width : img->height);
	    }
	  /* '(xwidget PROPS...)': dimensions of the specified xwidget.  */
	  if (FRAME_WINDOW_P (it->f) && valid_xwidget_spec_p (prop))
	    {
              /* TODO: Don't return dummy size.  */
              return OK_PIXELS (100);
            }
#endif
	  /* '(+ EXPR...)' or '(- EXPR...)' add or subtract
	     recursively calculated values.  */
	  if (EQ (car, Qplus) || EQ (car, Qminus))
	    {
	      bool first = true;
	      double px;

	      pixels = 0;
	      while (CONSP (cdr))
		{
		  if (!calc_pixel_width_or_height (&px, it, XCAR (cdr),
						   font, width_p, align_to))
		    return false;
		  if (first)
		    pixels = (EQ (car, Qplus) ? px : -px), first = false;
		  else
		    pixels += px;
		  cdr = XCDR (cdr);
		}
	      if (EQ (car, Qminus))
		pixels = -pixels;
	      return OK_PIXELS (pixels);
	    }

	  car = buffer_local_value (car, it->w->contents);
	  if (BASE_EQ (car, Qunbound))
	    car = Qnil;
	}

      /* '(NUM)': absolute number of pixels.  */
      if (NUMBERP (car))
	{
	  double fact;
	  int offset =
	    width_p && align_to && *align_to < 0 ? lnum_pixel_width : 0;
	  pixels = XFLOATINT (car);
	  if (NILP (cdr))
	    return OK_PIXELS (pixels + offset);
	  if (calc_pixel_width_or_height (&fact, it, cdr,
					  font, width_p, align_to))
	    return OK_PIXELS (pixels * fact + offset);
	  return false;
	}

      return false;
    }

  return false;
}

void
get_font_ascent_descent (struct font *font, int *ascent, int *descent)
{
#ifdef HAVE_WINDOW_SYSTEM
  normal_char_ascent_descent (font, -1, ascent, descent);
#else
  *ascent = 1;
  *descent = 0;
#endif
}


/***********************************************************************
			     Glyph Display
 ***********************************************************************/

#ifdef HAVE_WINDOW_SYSTEM

#ifdef GLYPH_DEBUG

extern void dump_glyph_string (struct glyph_string *) EXTERNALLY_VISIBLE;
void
dump_glyph_string (struct glyph_string *s)
{
  fputs ("glyph string\n", stderr);
  fprintf (stderr, "  x, y, w, h = %d, %d, %d, %d\n",
	   s->x, s->y, s->width, s->height);
  fprintf (stderr, "  ybase = %d\n", s->ybase);
  fprintf (stderr, "  hl = %u\n", s->hl);
  fprintf (stderr, "  left overhang = %d, right = %d\n",
	   s->left_overhang, s->right_overhang);
  fprintf (stderr, "  nchars = %d\n", s->nchars);
  fprintf (stderr, "  extends to end of line = %d\n",
	   s->extends_to_end_of_line_p);
  fprintf (stderr, "  font height = %d\n", FONT_HEIGHT (s->font));
  fprintf (stderr, "  bg width = %d\n", s->background_width);
}

#endif /* GLYPH_DEBUG */

/* Initialize glyph string S.  CHAR2B is a suitably allocated vector
   of 2-byte unsigned integers for S; it can't be allocated in
   init_glyph_string because it must be allocated via `alloca'.  W
   is the window on which S is drawn.  ROW and AREA are the glyph row
   and area within the row from which S is constructed.  START is the
   index of the first glyph structure covered by S.  HL is a
   face-override for drawing S.  */

#ifdef HAVE_NTGUI
/* We set inhibit-quit here due to paranoia: get_frame_dc acquires the
   critical section, and we cannot QUIT while we hold the critical
   section.  If any of the code run by callers of ALLOCATE_HDC happens
   to call Lisp (might be possible due to all the hooks lying around),
   we must prevent it from quitting.  */
# define ALLOCATE_HDC(hdc, f)			\
  Lisp_Object prev_quit = Vinhibit_quit;	\
  Vinhibit_quit = Qt;				\
  HDC hdc = get_frame_dc ((f))
# define RELEASE_HDC(hdc, f)			\
  release_frame_dc ((f), (hdc));		\
  Vinhibit_quit = prev_quit
#else
# define ALLOCATE_HDC(hdc, f)
# define RELEASE_HDC(hdc, f)
#endif

static void
init_glyph_string (struct glyph_string *s,
#ifdef HAVE_NTGUI
		   HDC hdc,
#endif
		   unsigned *char2b, struct window *w, struct glyph_row *row,
		   enum glyph_row_area area, int start, enum draw_glyphs_face hl)
{
  memset (s, 0, sizeof *s);
  s->w = w;
  s->f = XFRAME (w->frame);
#ifdef HAVE_NTGUI
  s->hdc = hdc;
#endif
  s->char2b = char2b;
  s->hl = hl;
  s->row = row;
  s->area = area;
  s->first_glyph = row->glyphs[area] + start;
  s->height = row->height;
  s->y = WINDOW_TO_FRAME_PIXEL_Y (w, row->y);
  s->ybase = s->y + row->ascent;
}


/* Append the list of glyph strings with head H and tail T to the list
   with head *HEAD and tail *TAIL.  Set *HEAD and *TAIL to the result.  */

static void
append_glyph_string_lists (struct glyph_string **head, struct glyph_string **tail,
			   struct glyph_string *h, struct glyph_string *t)
{
  if (h)
    {
      if (*head)
	(*tail)->next = h;
      else
	*head = h;
      h->prev = *tail;
      *tail = t;
    }
}


/* Prepend the list of glyph strings with head H and tail T to the
   list with head *HEAD and tail *TAIL.  Set *HEAD and *TAIL to the
   result.  */

static void
prepend_glyph_string_lists (struct glyph_string **head, struct glyph_string **tail,
			    struct glyph_string *h, struct glyph_string *t)
{
  if (h)
    {
      if (*head)
	(*head)->prev = t;
      else
	*tail = t;
      t->next = *head;
      *head = h;
    }
}


/* Append glyph string S to the list with head *HEAD and tail *TAIL.
   Set *HEAD and *TAIL to the resulting list.  */

static void
append_glyph_string (struct glyph_string **head, struct glyph_string **tail,
		     struct glyph_string *s)
{
  s->next = s->prev = NULL;
  append_glyph_string_lists (head, tail, s, s);
}


/* Get face and two-byte form of character C in face FACE_ID on frame F.
   The encoding of C is returned in *CHAR2B.  DISPLAY_P means
   make sure that X resources for the face returned are allocated.
   Value is a pointer to a realized face that is ready for display if
   DISPLAY_P.  */

static struct face *
get_char_face_and_encoding (struct frame *f, int c, int face_id,
			    unsigned *char2b, bool display_p)
{
  struct face *face = FACE_FROM_ID (f, face_id);
  unsigned code = 0;

  if (face->font)
    {
      code = face->font->driver->encode_char (face->font, c);

      if (code == FONT_INVALID_CODE)
	code = 0;
    }
  /* Ensure that the code is only 2 bytes wide.  */
  *char2b = code & 0xFFFF;

  /* Make sure X resources of the face are allocated.  */
#ifdef HAVE_X_WINDOWS
  if (display_p)
#endif
    {
      eassert (face != NULL);
      prepare_face_for_display (f, face);
    }

  return face;
}


/* Get face and two-byte form of character glyph GLYPH on frame F.
   The encoding of GLYPH->u.ch is returned in *CHAR2B.  Value is
   a pointer to a realized face that is ready for display.  */

static struct face *
get_glyph_face_and_encoding (struct frame *f, struct glyph *glyph,
			     unsigned *char2b)
{
  struct face *face;
  unsigned code = 0;

  eassert (glyph->type == CHAR_GLYPH);
  face = FACE_FROM_ID (f, glyph->face_id);

  /* Make sure X resources of the face are allocated.  */
  prepare_face_for_display (f, face);

  if (face->font)
    {
      if (CHAR_BYTE8_P (glyph->u.ch))
	code = CHAR_TO_BYTE8 (glyph->u.ch);
      else
	code = face->font->driver->encode_char (face->font, glyph->u.ch);

      if (code == FONT_INVALID_CODE)
	code = 0;
    }

  /* Ensure that the code is only 2 bytes wide.  */
  *char2b = code & 0xFFFF;
  return face;
}


/* Get glyph code of character C in FONT in the two-byte form CHAR2B.
   Return true iff FONT has a glyph for C.  */

static bool
get_char_glyph_code (int c, struct font *font, unsigned *char2b)
{
  unsigned code;

  if (CHAR_BYTE8_P (c))
    code = CHAR_TO_BYTE8 (c);
  else
    code = font->driver->encode_char (font, c);

  if (code == FONT_INVALID_CODE)
    return false;

  /* Ensure that the code is only 2 bytes wide.  */
  *char2b = code & 0xFFFF;
  return true;
}


/* Fill glyph string S with composition components specified by S->cmp.

   BASE_FACE is the base face of the composition.
   S->cmp_from is the index of the first component for S.

   OVERLAPS non-zero means S should draw the foreground only, and use
   its physical height for clipping.  See also draw_glyphs.

   Value is the index of a component not in S.  */

static int
fill_composite_glyph_string (struct glyph_string *s, struct face *base_face,
			     int overlaps)
{
  int i;
  /* For all glyphs of this composition, starting at the offset
     S->cmp_from, until we reach the end of the definition or encounter a
     glyph that requires the different face, add it to S.  */
  struct face *face;

  eassert (s);

  s->for_overlaps = overlaps;
  s->face = NULL;
  s->font = NULL;
  for (i = s->cmp_from; i < s->cmp->glyph_len; i++)
    {
      int c = COMPOSITION_GLYPH (s->cmp, i);

      /* TAB in a composition means display glyphs with padding space
	 on the left or right.  */
      if (c != '\t')
	{
	  int face_id = FACE_FOR_CHAR (s->f, base_face->ascii_face, c,
				       -1, Qnil);

	  face = get_char_face_and_encoding (s->f, c, face_id,
					     s->char2b + i, true);
	  if (face)
	    {
	      if (! s->face)
		{
		  s->face = face;
		  s->font = s->face->font;
		}
	      else if (s->face != face)
		break;
	    }
	}
      ++s->nchars;
    }
  s->cmp_to = i;

  if (s->face == NULL)
    {
      s->face = base_face->ascii_face;
      s->font = s->face->font;
    }

  if (s->hl == DRAW_MOUSE_FACE
      || (s->hl == DRAW_CURSOR
	  && MATRIX_ROW (s->w->current_matrix,
			 s->w->phys_cursor.vpos)->mouse_face_p
	  && cursor_in_mouse_face_p (s->w)))
    {
      int c = COMPOSITION_GLYPH (s->cmp, 0);
      Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (s->f);
      s->face = FACE_FROM_ID_OR_NULL (s->f, hlinfo->mouse_face_face_id);
      if (!s->face)
	s->face = FACE_FROM_ID (s->f, MOUSE_FACE_ID);

      s->face = FACE_FROM_ID (s->f, FACE_FOR_CHAR (s->f, s->face, c, -1, Qnil));
      prepare_face_for_display (s->f, s->face);
    }

  /* All glyph strings for the same composition has the same width,
     i.e. the width set for the first component of the composition.  */
  s->width = s->first_glyph->pixel_width;

  /* If the specified font could not be loaded, use the frame's
     default font, but record the fact that we couldn't load it in
     the glyph string so that we can draw rectangles for the
     characters of the glyph string.  */
  if (s->font == NULL)
    {
      s->font_not_found_p = true;
      s->font = FRAME_FONT (s->f);
    }

  /* Adjust base line for subscript/superscript text.  */
  s->ybase += s->first_glyph->voffset;

  return s->cmp_to;
}

static int
fill_gstring_glyph_string (struct glyph_string *s, int face_id,
			   int start, int end, int overlaps)
{
  struct glyph *glyph, *last;
  int voffset;
  Lisp_Object lgstring;
  int i;
  bool glyph_not_available_p;

  s->for_overlaps = overlaps;
  glyph = s->row->glyphs[s->area] + start;
  last = s->row->glyphs[s->area] + end;
  voffset = glyph->voffset;
  glyph_not_available_p = glyph->glyph_not_available_p;
  s->cmp_id = glyph->u.cmp.id;
  s->cmp_from = glyph->slice.cmp.from;
  s->cmp_to = glyph->slice.cmp.to + 1;
  if (s->hl == DRAW_MOUSE_FACE
      || (s->hl == DRAW_CURSOR
	  && MATRIX_ROW (s->w->current_matrix,
			 s->w->phys_cursor.vpos)->mouse_face_p
	  && cursor_in_mouse_face_p (s->w)))
    {
      Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (s->f);
      s->face = FACE_FROM_ID_OR_NULL (s->f, hlinfo->mouse_face_face_id);
      if (!s->face)
	s->face = FACE_FROM_ID (s->f, MOUSE_FACE_ID);
      prepare_face_for_display (s->f, s->face);
    }
  else
    s->face = FACE_FROM_ID (s->f, face_id);
  lgstring = composition_gstring_from_id (s->cmp_id);
  s->font = XFONT_OBJECT (LGSTRING_FONT (lgstring));
  /* The width of a composition glyph string is the sum of the
     composition's glyph widths.  */
  s->width = s->first_glyph->pixel_width;
  glyph++;
  while (glyph < last
	 && glyph->u.cmp.automatic
	 && glyph->u.cmp.id == s->cmp_id
	 && glyph->face_id == face_id
	 && s->cmp_to == glyph->slice.cmp.from
	 && glyph->glyph_not_available_p == glyph_not_available_p)
    {
      s->width += glyph->pixel_width;
      s->cmp_to = (glyph++)->slice.cmp.to + 1;
    }

  for (i = s->cmp_from; i < s->cmp_to; i++)
    {
      Lisp_Object lglyph = LGSTRING_GLYPH (lgstring, i);
      unsigned code = LGLYPH_CODE (lglyph);

      /* Ensure that the code is only 2 bytes wide.  */
      s->char2b[i] = code & 0xFFFF;
    }

  /* If the specified font could not be loaded, record that fact in
     S->font_not_found_p so that we can draw rectangles for the
     characters of the glyph string.  */
  if (glyph_not_available_p)
    s->font_not_found_p = true;

  /* Adjust base line for subscript/superscript text.  */
  s->ybase += voffset;

  return glyph - s->row->glyphs[s->area];
}


/* Fill glyph string S from a sequence glyphs for glyphless characters.
   See the comment of fill_glyph_string for arguments.
   Value is the index of the first glyph not in S.  */


static int
fill_glyphless_glyph_string (struct glyph_string *s, int face_id,
			     int start, int end, int overlaps)
{
  struct glyph *glyph, *last;
  int voffset;

  eassert (s->first_glyph->type == GLYPHLESS_GLYPH);
  s->for_overlaps = overlaps;
  glyph = s->row->glyphs[s->area] + start;
  last = s->row->glyphs[s->area] + end;
  voffset = glyph->voffset;
  s->face = FACE_FROM_ID (s->f, face_id);
  s->font = s->face->font ? s->face->font : FRAME_FONT (s->f);
  if (s->hl == DRAW_MOUSE_FACE
      || (s->hl == DRAW_CURSOR
	  && MATRIX_ROW (s->w->current_matrix,
			 s->w->phys_cursor.vpos)->mouse_face_p
	  && cursor_in_mouse_face_p (s->w)))
    {
      Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (s->f);
      s->face = FACE_FROM_ID_OR_NULL (s->f, hlinfo->mouse_face_face_id);
      if (!s->face)
	s->face = FACE_FROM_ID (s->f, MOUSE_FACE_ID);
      prepare_face_for_display (s->f, s->face);
    }
  s->nchars = 1;
  s->width = glyph->pixel_width;
  glyph++;
  while (glyph < last
	 && glyph->type == GLYPHLESS_GLYPH
	 && glyph->voffset == voffset
	 && glyph->face_id == face_id)
    {
      s->nchars++;
      s->width += glyph->pixel_width;
      glyph++;
    }
  s->ybase += voffset;
  return glyph - s->row->glyphs[s->area];
}


/* Fill glyph string S from a sequence of character glyphs.

   FACE_ID is the face id of the string.  START is the index of the
   first glyph to consider, END is the index of the last + 1.
   OVERLAPS non-zero means S should draw the foreground only, and use
   its physical height for clipping.  See also draw_glyphs.

   Value is the index of the first glyph not in S.  */

static int
fill_glyph_string (struct glyph_string *s, int face_id,
		   int start, int end, int overlaps)
{
  struct glyph *glyph, *last;
  int voffset;
  bool glyph_not_available_p;

  eassert (s->f == XFRAME (s->w->frame));
  eassert (s->nchars == 0);
  eassert (start >= 0 && end > start);

  s->for_overlaps = overlaps;
  glyph = s->row->glyphs[s->area] + start;
  last = s->row->glyphs[s->area] + end;
  voffset = glyph->voffset;
  s->padding_p = glyph->padding_p;
  glyph_not_available_p = glyph->glyph_not_available_p;

  while (glyph < last
	 && glyph->type == CHAR_GLYPH
	 && glyph->voffset == voffset
	 /* Same face id implies same font, nowadays.  */
	 && glyph->face_id == face_id
	 && glyph->glyph_not_available_p == glyph_not_available_p)
    {
      s->face = get_glyph_face_and_encoding (s->f, glyph,
					     s->char2b + s->nchars);
      ++s->nchars;
      eassert (s->nchars <= end - start);
      s->width += glyph->pixel_width;
      if (glyph++->padding_p != s->padding_p)
	break;
    }

  s->font = s->face->font;

  if (s->hl == DRAW_MOUSE_FACE
      || (s->hl == DRAW_CURSOR
	  && MATRIX_ROW (s->w->current_matrix,
			 s->w->phys_cursor.vpos)->mouse_face_p
	  && cursor_in_mouse_face_p (s->w)))
    {
      Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (s->f);
      s->face = FACE_FROM_ID_OR_NULL (s->f, hlinfo->mouse_face_face_id);
      if (!s->face)
	s->face = FACE_FROM_ID (s->f, MOUSE_FACE_ID);
      s->face
        = FACE_FROM_ID (s->f, FACE_FOR_CHAR (s->f, s->face,
					     s->first_glyph->u.ch, -1, Qnil));
      prepare_face_for_display (s->f, s->face);
    }

  /* If the specified font could not be loaded, use the frame's font,
     but record the fact that we couldn't load it in
     S->font_not_found_p so that we can draw rectangles for the
     characters of the glyph string.  */
  if (s->font == NULL || glyph_not_available_p)
    {
      s->font_not_found_p = true;
      s->font = FRAME_FONT (s->f);
    }

  /* Adjust base line for subscript/superscript text.  */
  s->ybase += voffset;

  eassert (s->face && s->face->gc);
  return glyph - s->row->glyphs[s->area];
}


/* Fill glyph string S from image glyph S->first_glyph.  */

static void
fill_image_glyph_string (struct glyph_string *s)
{
  eassert (s->first_glyph->type == IMAGE_GLYPH);
  s->img = IMAGE_FROM_ID (s->f, s->first_glyph->u.img_id);
  eassert (s->img);
  s->slice = s->first_glyph->slice.img;
  s->face = FACE_FROM_ID (s->f, s->first_glyph->face_id);
  s->font = s->face->font;
  if (s->hl == DRAW_MOUSE_FACE
      || (s->hl == DRAW_CURSOR
	  && MATRIX_ROW (s->w->current_matrix,
			 s->w->phys_cursor.vpos)->mouse_face_p
	  && cursor_in_mouse_face_p (s->w)))
    {
      Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (s->f);
      s->face = FACE_FROM_ID_OR_NULL (s->f, hlinfo->mouse_face_face_id);
      if (!s->face)
	s->face = FACE_FROM_ID (s->f, MOUSE_FACE_ID);
      prepare_face_for_display (s->f, s->face);
    }
  s->width = s->first_glyph->pixel_width;

  /* Adjust base line for subscript/superscript text.  */
  s->ybase += s->first_glyph->voffset;
}


#ifdef HAVE_XWIDGETS
static void
fill_xwidget_glyph_string (struct glyph_string *s)
{
  eassert (s->first_glyph->type == XWIDGET_GLYPH);
  s->face = FACE_FROM_ID (s->f, s->first_glyph->face_id);
  s->font = s->face->font;
  if (s->hl == DRAW_MOUSE_FACE
      || (s->hl == DRAW_CURSOR
	  && MATRIX_ROW (s->w->current_matrix,
			 s->w->phys_cursor.vpos)->mouse_face_p
	  && cursor_in_mouse_face_p (s->w)))
    {
      Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (s->f);
      s->face = FACE_FROM_ID_OR_NULL (s->f, hlinfo->mouse_face_face_id);
      if (!s->face)
	s->face = FACE_FROM_ID (s->f, MOUSE_FACE_ID);
      prepare_face_for_display (s->f, s->face);
    }
  s->width = s->first_glyph->pixel_width;
  s->ybase += s->first_glyph->voffset;
  s->xwidget = xwidget_from_id (s->first_glyph->u.xwidget);
}
#endif
/* Fill glyph string S from a sequence of stretch glyphs.

   START is the index of the first glyph to consider,
   END is the index of the last + 1.

   Value is the index of the first glyph not in S.  */

static int
fill_stretch_glyph_string (struct glyph_string *s, int start, int end)
{
  struct glyph *glyph, *last;
  int voffset, face_id;

  eassert (s->first_glyph->type == STRETCH_GLYPH);

  glyph = s->row->glyphs[s->area] + start;
  last = s->row->glyphs[s->area] + end;
  face_id = glyph->face_id;
  s->face = FACE_FROM_ID (s->f, face_id);
  s->font = s->face->font;
  if (s->hl == DRAW_MOUSE_FACE
      || (s->hl == DRAW_CURSOR
	  && MATRIX_ROW (s->w->current_matrix,
			 s->w->phys_cursor.vpos)->mouse_face_p
	  && cursor_in_mouse_face_p (s->w)))
    {
      Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (s->f);
      s->face = FACE_FROM_ID_OR_NULL (s->f, hlinfo->mouse_face_face_id);
      if (!s->face)
	s->face = FACE_FROM_ID (s->f, MOUSE_FACE_ID);
      prepare_face_for_display (s->f, s->face);
    }
  s->width = glyph->pixel_width;
  s->nchars = 1;
  voffset = glyph->voffset;

  for (++glyph;
       (glyph < last
	&& glyph->type == STRETCH_GLYPH
	&& glyph->voffset == voffset
	&& glyph->face_id == face_id);
       ++glyph)
    s->width += glyph->pixel_width;

  /* Adjust base line for subscript/superscript text.  */
  s->ybase += voffset;

  /* The case that face->gc == 0 is handled when drawing the glyph
     string by calling prepare_face_for_display.  */
  eassert (s->face);
  return glyph - s->row->glyphs[s->area];
}

static struct font_metrics *
get_per_char_metric (struct font *font, const unsigned *char2b)
{
  static struct font_metrics metrics;

  if (! font)
    return NULL;
  if (*char2b == FONT_INVALID_CODE)
    return NULL;

  font->driver->text_extents (font, char2b, 1, &metrics);
  return &metrics;
}

/* A subroutine that computes "normal" values of ASCENT and DESCENT
   for FONT.  Values are taken from font-global ones, except for fonts
   that claim preposterously large values, but whose glyphs actually
   have reasonable dimensions.  C is the character to use for metrics
   if the font-global values are too large; if C is negative, the
   function selects a default character.  */
static void
normal_char_ascent_descent (struct font *font, int c, int *ascent, int *descent)
{
  *ascent = FONT_BASE (font);
  *descent = FONT_DESCENT (font);

  if (FONT_TOO_HIGH (font))
    {
      unsigned char2b;

      /* Get metrics of C, defaulting to a reasonably sized ASCII
	 character.  */
      if (get_char_glyph_code (c >= 0 ? c : '{', font, &char2b))
	{
	  struct font_metrics *pcm = get_per_char_metric (font, &char2b);
	  eassume (pcm);

	  if (!(pcm->width == 0 && pcm->rbearing == 0 && pcm->lbearing == 0))
	    {
	      /* We add 1 pixel to character dimensions as heuristics
		 that produces nicer display, e.g. when the face has
		 the box attribute.  */
	      *ascent = pcm->ascent + 1;
	      *descent = pcm->descent + 1;
	    }
	}
    }
}

/* A subroutine that computes a reasonable "normal character height"
   for fonts that claim preposterously large vertical dimensions, but
   whose glyphs are actually reasonably sized.  C is the character
   whose metrics to use for those fonts, or -1 for default
   character.  */
static int
normal_char_height (struct font *font, int c)
{
  int ascent, descent;

  normal_char_ascent_descent (font, c, &ascent, &descent);

  return ascent + descent;
}

/* EXPORT for RIF:
   Set *LEFT and *RIGHT to the left and right overhang of GLYPH on
   frame F.  Overhangs of glyphs other than type CHAR_GLYPH are
   assumed to be zero.  */

void
gui_get_glyph_overhangs (struct glyph *glyph, struct frame *f, int *left, int *right)
{
  *left = *right = 0;

  if (glyph->type == CHAR_GLYPH)
    {
      unsigned char2b;
      struct face *face = get_glyph_face_and_encoding (f, glyph, &char2b);
      if (face->font)
	{
	  struct font_metrics *pcm = get_per_char_metric (face->font, &char2b);
	  if (pcm)
	    {
	      if (pcm->rbearing > pcm->width)
		*right = pcm->rbearing - pcm->width;
	      if (pcm->lbearing < 0)
		*left = -pcm->lbearing;
	    }
	}
    }
  else if (glyph->type == COMPOSITE_GLYPH)
    {
      if (! glyph->u.cmp.automatic)
	{
	  struct composition *cmp = composition_table[glyph->u.cmp.id];

	  if (cmp->rbearing > cmp->pixel_width)
	    *right = cmp->rbearing - cmp->pixel_width;
	  if (cmp->lbearing < 0)
	    *left = - cmp->lbearing;
	}
      else
	{
	  Lisp_Object gstring = composition_gstring_from_id (glyph->u.cmp.id);
	  struct font_metrics metrics;

	  composition_gstring_width (gstring, glyph->slice.cmp.from,
				     glyph->slice.cmp.to + 1, &metrics);
	  if (metrics.rbearing > metrics.width)
	    *right = metrics.rbearing - metrics.width;
	  if (metrics.lbearing < 0)
	    *left = - metrics.lbearing;
	}
    }
}


/* Return the index of the first glyph preceding glyph string S that
   is overwritten by S because of S's left overhang.  Value is -1
   if no glyphs are overwritten.  */

static int
left_overwritten (struct glyph_string *s)
{
  int k;

  if (s->left_overhang)
    {
      int x = 0, i;
      struct glyph *glyphs = s->row->glyphs[s->area];
      int first = s->first_glyph - glyphs;

      for (i = first - 1; i >= 0 && x > -s->left_overhang; --i)
	x -= glyphs[i].pixel_width;

      k = i + 1;
    }
  else
    k = -1;

  return k;
}


/* Return the index of the first glyph preceding glyph string S that
   is overwriting S because of its right overhang.  Value is -1 if no
   glyph in front of S overwrites S.  */

static int
left_overwriting (struct glyph_string *s)
{
  int i, k, x;
  struct glyph *glyphs = s->row->glyphs[s->area];
  int first = s->first_glyph - glyphs;

  k = -1;
  x = 0;
  for (i = first - 1; i >= 0; --i)
    {
      int left, right;
      gui_get_glyph_overhangs (glyphs + i, s->f, &left, &right);
      if (x + right > 0)
	k = i;
      x -= glyphs[i].pixel_width;
    }

  return k;
}


/* Return the index of the last glyph following glyph string S that is
   overwritten by S because of S's right overhang.  Value is -1 if
   no such glyph is found.  */

static int
right_overwritten (struct glyph_string *s)
{
  int k = -1;

  if (s->right_overhang)
    {
      int x = 0, i;
      struct glyph *glyphs = s->row->glyphs[s->area];
      int first = (s->first_glyph - glyphs
		   + (s->first_glyph->type == COMPOSITE_GLYPH ? 1 : s->nchars));
      int end = s->row->used[s->area];

      for (i = first; i < end && s->right_overhang > x; ++i)
	x += glyphs[i].pixel_width;

      k = i;
    }

  return k;
}


/* Return the index of the last glyph following glyph string S that
   overwrites S because of its left overhang.  Value is negative
   if no such glyph is found.  */

static int
right_overwriting (struct glyph_string *s)
{
  int i, k, x;
  int end = s->row->used[s->area];
  struct glyph *glyphs = s->row->glyphs[s->area];
  int first = (s->first_glyph - glyphs
	       + (s->first_glyph->type == COMPOSITE_GLYPH ? 1 : s->nchars));

  k = -1;
  x = 0;
  for (i = first; i < end; ++i)
    {
      int left, right;
      gui_get_glyph_overhangs (glyphs + i, s->f, &left, &right);
      if (x - left < 0)
	k = i;
      x += glyphs[i].pixel_width;
    }

  return k;
}


/* Set background width of glyph string S.  START is the index of the
   first glyph following S.  LAST_X is the right-most x-position + 1
   in the drawing area.

   If S->hl is DRAW_CURSOR, S->f is a window system frame, and the
   cursor in S's window is currently inside mouse face, also update
   S->width to take into account potentially differing :box
   properties between the original face and the mouse face.  */

static void
set_glyph_string_background_width (struct glyph_string *s, int start, int last_x)
{
  /* If the face of this glyph string has to be drawn to the end of
     the drawing area, set S->extends_to_end_of_line_p.  */

  if (start == s->row->used[s->area]
      && ((s->row->fill_line_p
	   && (s->hl == DRAW_NORMAL_TEXT
	       || s->hl == DRAW_IMAGE_RAISED
	       || s->hl == DRAW_IMAGE_SUNKEN))
	  || s->hl == DRAW_MOUSE_FACE))
    s->extends_to_end_of_line_p = true;

  /* If S extends its face to the end of the line, set its
     background_width to the distance to the right edge of the drawing
     area.  */
  if (s->extends_to_end_of_line_p)
    s->background_width = last_x - s->x + 1;
  else
    {
      s->background_width = s->width;
#ifdef HAVE_WINDOW_SYSTEM
      if (FRAME_WINDOW_P (s->f)
	  && s->hl == DRAW_CURSOR
	  && MATRIX_ROW (s->w->current_matrix,
			 s->w->phys_cursor.vpos)->mouse_face_p
	  && cursor_in_mouse_face_p (s->w))
	{
	  /* Adjust the background width of the glyph string, because
	     if the glyph's face has the :box attribute, its
	     pixel_width might be different when it's displayed in the
	     mouse-face, if that also has the :box attribute.  */
	  struct glyph *g = s->first_glyph;
	  struct face *regular_face = FACE_FROM_ID (s->f, g->face_id);
	  s->background_width +=
	    adjust_glyph_width_for_mouse_face (g, s->row, s->w,
					       regular_face, s->face);
	  /* S->width is probably worth adjusting here as well.  */
	  s->width = s->background_width;
        }
#endif
    }
}


/* Return glyph string that shares background with glyph string S and
   whose `background_width' member has been set.  */

static struct glyph_string *
glyph_string_containing_background_width (struct glyph_string *s)
{
  if (s->cmp)
    while (s->cmp_from)
      s = s->prev;

  return s;
}


/* Compute overhangs and x-positions for glyph string S and its
   predecessors, or successors.  X is the starting x-position for S.
   BACKWARD_P means process predecessors.  */

static void
compute_overhangs_and_x (struct glyph_string *s, int x, bool backward_p)
{
  if (backward_p)
    {
      while (s)
	{
	  if (FRAME_RIF (s->f)->compute_glyph_string_overhangs)
	    FRAME_RIF (s->f)->compute_glyph_string_overhangs (s);
	  if (!s->cmp || s->cmp_to == s->cmp->glyph_len)
	    x -= s->width;
	  s->x = x;
	  s = s->prev;
	}
    }
  else
    {
      while (s)
	{
	  if (FRAME_RIF (s->f)->compute_glyph_string_overhangs)
	    FRAME_RIF (s->f)->compute_glyph_string_overhangs (s);
	  s->x = x;
	  if (!s->cmp || s->cmp_to == s->cmp->glyph_len)
	    x += s->width;
	  s = s->next;
	}
    }
}



/* The following macros are only called from draw_glyphs below.
   They reference the following parameters of that function directly:
     `w', `row', `area', and `overlap_p'
   as well as the following local variables:
     `s', `f', and `hdc' (in W32)  */

#ifdef HAVE_NTGUI
/* On W32, silently add local `hdc' variable to argument list of
   init_glyph_string.  */
#define INIT_GLYPH_STRING(s, char2b, w, row, area, start, hl) \
  init_glyph_string (s, hdc, char2b, w, row, area, start, hl)
#else
#define INIT_GLYPH_STRING(s, char2b, w, row, area, start, hl) \
  init_glyph_string (s, char2b, w, row, area, start, hl)
#endif

/* Add a glyph string for a stretch glyph to the list of strings
   between HEAD and TAIL.  START is the index of the stretch glyph in
   row area AREA of glyph row ROW.  END is the index of the last glyph
   in that glyph row area.  X is the current output position assigned
   to the new glyph string constructed.  HL overrides that face of the
   glyph; e.g. it is DRAW_CURSOR if a cursor has to be drawn.  LAST_X
   is the right-most x-position of the drawing area.  */

/* SunOS 4 bundled cc, barfed on continuations in the arg lists here
   and below -- keep them on one line.  */
#define BUILD_STRETCH_GLYPH_STRING(START, END, HEAD, TAIL, HL, X, LAST_X)   \
     do									    \
       {								    \
	 s = alloca (sizeof *s);					    \
	 INIT_GLYPH_STRING (s, NULL, w, row, area, START, HL);		    \
	 START = fill_stretch_glyph_string (s, START, END);                 \
	 append_glyph_string (&HEAD, &TAIL, s);				    \
         s->x = (X);							    \
       }								    \
     while (false)


/* Add a glyph string for an image glyph to the list of strings
   between HEAD and TAIL.  START is the index of the image glyph in
   row area AREA of glyph row ROW.  END is the index of the last glyph
   in that glyph row area.  X is the current output position assigned
   to the new glyph string constructed.  HL overrides that face of the
   glyph; e.g. it is DRAW_CURSOR if a cursor has to be drawn.  LAST_X
   is the right-most x-position of the drawing area.  */

#define BUILD_IMAGE_GLYPH_STRING(START, END, HEAD, TAIL, HL, X, LAST_X) \
     do									\
       {								\
	 s = alloca (sizeof *s);					\
	 INIT_GLYPH_STRING (s, NULL, w, row, area, START, HL);		\
	 fill_image_glyph_string (s);					\
	 append_glyph_string (&HEAD, &TAIL, s);				\
	 ++START;							\
         s->x = (X);							\
       }								\
     while (false)

#ifndef HAVE_XWIDGETS
# define BUILD_XWIDGET_GLYPH_STRING(START, END, HEAD, TAIL, HL, X, LAST_X) \
     eassume (false)
#else
# define BUILD_XWIDGET_GLYPH_STRING(START, END, HEAD, TAIL, HL, X, LAST_X) \
     do									\
       {								\
	 s = alloca (sizeof *s);					\
	 INIT_GLYPH_STRING (s, NULL, w, row, area, START, HL);		\
	 fill_xwidget_glyph_string (s);					\
	 append_glyph_string (&(HEAD), &(TAIL), s);			\
	 ++(START);							\
         s->x = (X);							\
       }								\
     while (false)
#endif

/* Add a glyph string for a sequence of character glyphs to the list
   of strings between HEAD and TAIL.  START is the index of the first
   glyph in row area AREA of glyph row ROW that is part of the new
   glyph string.  END is the index of the last glyph in that glyph row
   area.  X is the current output position assigned to the new glyph
   string constructed.  HL overrides that face of the glyph; e.g. it
   is DRAW_CURSOR if a cursor has to be drawn.  LAST_X is the
   right-most x-position of the drawing area.  */

#define BUILD_CHAR_GLYPH_STRINGS(START, END, HEAD, TAIL, HL, X, LAST_X)	   \
     do									   \
       {								   \
	 int face_id;							   \
	 unsigned *char2b;					   \
									   \
	 face_id = (row)->glyphs[area][START].face_id;			   \
									   \
	 s = alloca (sizeof *s);					   \
	 SAFE_NALLOCA (char2b, 1, (END) - (START));			   \
	 INIT_GLYPH_STRING (s, char2b, w, row, area, START, HL);	   \
	 append_glyph_string (&HEAD, &TAIL, s);				   \
	 s->x = (X);							   \
	 START = fill_glyph_string (s, face_id, START, END, overlaps);	   \
       }								   \
     while (false)


/* Add a glyph string for a composite sequence to the list of strings
   between HEAD and TAIL.  START is the index of the first glyph in
   row area AREA of glyph row ROW that is part of the new glyph
   string.  END is the index of the last glyph in that glyph row area.
   X is the current output position assigned to the new glyph string
   constructed.  HL overrides that face of the glyph; e.g. it is
   DRAW_CURSOR if a cursor has to be drawn.  LAST_X is the right-most
   x-position of the drawing area.  */

#define BUILD_COMPOSITE_GLYPH_STRING(START, END, HEAD, TAIL, HL, X, LAST_X) \
  do {									    \
    int face_id = (row)->glyphs[area][START].face_id;			    \
    struct face *base_face = FACE_FROM_ID (f, face_id);		    \
    ptrdiff_t cmp_id = (row)->glyphs[area][START].u.cmp.id;		    \
    struct composition *cmp = composition_table[cmp_id];		    \
    unsigned *char2b;						            \
    struct glyph_string *first_s = NULL;				    \
    int n;								    \
    									    \
    SAFE_NALLOCA (char2b, 1, cmp->glyph_len);				    \
    									    \
    /* Make glyph_strings for each glyph sequence that is drawable by	    \
       the same face, and append them to HEAD/TAIL.  */			    \
    for (n = 0; n < cmp->glyph_len;)					    \
      {									    \
	s = alloca (sizeof *s);						    \
	INIT_GLYPH_STRING (s, char2b, w, row, area, START, HL);		    \
	append_glyph_string (&(HEAD), &(TAIL), s);			    \
	s->cmp = cmp;							    \
	s->cmp_from = n;						    \
	s->x = (X);							    \
	if (n == 0)							    \
	  first_s = s;							    \
	n = fill_composite_glyph_string (s, base_face, overlaps);	    \
      }									    \
    									    \
    ++START;								    \
    s = first_s;							    \
  } while (false)


/* Add a glyph string for a glyph-string sequence to the list of strings
   between HEAD and TAIL.  */

#define BUILD_GSTRING_GLYPH_STRING(START, END, HEAD, TAIL, HL, X, LAST_X) \
  do {									  \
    int face_id;							  \
    unsigned *char2b;						          \
    Lisp_Object gstring;						  \
    									  \
    face_id = (row)->glyphs[area][START].face_id;			  \
    gstring = (composition_gstring_from_id				  \
	       ((row)->glyphs[area][START].u.cmp.id));			  \
    s = alloca (sizeof *s);						  \
    SAFE_NALLOCA (char2b, 1, LGSTRING_GLYPH_LEN (gstring));		  \
    INIT_GLYPH_STRING (s, char2b, w, row, area, START, HL);		  \
    append_glyph_string (&(HEAD), &(TAIL), s);				  \
    s->x = (X);								  \
    START = fill_gstring_glyph_string (s, face_id, START, END, overlaps); \
  } while (false)


/* Add a glyph string for a sequence of glyphless character's glyphs
   to the list of strings between HEAD and TAIL.  The meanings of
   arguments are the same as those of BUILD_CHAR_GLYPH_STRINGS.  */

#define BUILD_GLYPHLESS_GLYPH_STRING(START, END, HEAD, TAIL, HL, X, LAST_X) \
  do									    \
    {									    \
      int face_id;							    \
									    \
      face_id = (row)->glyphs[area][START].face_id;			    \
									    \
      s = alloca (sizeof *s);						    \
      INIT_GLYPH_STRING (s, NULL, w, row, area, START, HL);		    \
      append_glyph_string (&HEAD, &TAIL, s);				    \
      s->x = (X);							    \
      START = fill_glyphless_glyph_string (s, face_id, START, END,	    \
					   overlaps);			    \
    }									    \
  while (false)


/* Build a list of glyph strings between HEAD and TAIL for the glyphs
   of AREA of glyph row ROW on window W between indices START and END.
   HL overrides the face for drawing glyph strings, e.g. it is
   DRAW_CURSOR to draw a cursor.  X and LAST_X are start and end
   x-positions of the drawing area.

   This is an ugly monster macro construct because we must use alloca
   to allocate glyph strings (because draw_glyphs can be called
   asynchronously).  */

#define BUILD_GLYPH_STRINGS_1(START, END, HEAD, TAIL, HL, X, LAST_X)	\
  do									\
    {									\
      HEAD = TAIL = NULL;						\
      while (START < END)						\
	{								\
	  struct glyph *first_glyph = (row)->glyphs[area] + START;	\
	  switch (first_glyph->type)					\
	    {								\
	    case CHAR_GLYPH:						\
	      BUILD_CHAR_GLYPH_STRINGS (START, END, HEAD, TAIL,		\
					HL, X, LAST_X);			\
	      break;							\
									\
	    case COMPOSITE_GLYPH:					\
	      if (first_glyph->u.cmp.automatic)				\
		BUILD_GSTRING_GLYPH_STRING (START, END, HEAD, TAIL,	\
					    HL, X, LAST_X);		\
	      else							\
		BUILD_COMPOSITE_GLYPH_STRING (START, END, HEAD, TAIL,	\
					      HL, X, LAST_X);		\
	      break;							\
									\
	    case STRETCH_GLYPH:						\
	      BUILD_STRETCH_GLYPH_STRING (START, END, HEAD, TAIL,	\
					  HL, X, LAST_X);		\
	      break;							\
									\
	    case IMAGE_GLYPH:						\
	      BUILD_IMAGE_GLYPH_STRING (START, END, HEAD, TAIL,		\
					HL, X, LAST_X);			\
	      break;

#define BUILD_GLYPH_STRINGS_XW(START, END, HEAD, TAIL, HL, X, LAST_X)	\
            case XWIDGET_GLYPH:                                         \
              BUILD_XWIDGET_GLYPH_STRING (START, END, HEAD, TAIL,       \
                                          HL, X, LAST_X);               \
              break;

#define BUILD_GLYPH_STRINGS_2(START, END, HEAD, TAIL, HL, X, LAST_X)	\
	    case GLYPHLESS_GLYPH:					\
	      BUILD_GLYPHLESS_GLYPH_STRING (START, END, HEAD, TAIL,	\
					    HL, X, LAST_X);		\
	      break;							\
									\
	    default:							\
	      emacs_abort ();						\
	    }								\
									\
	  if (s)							\
	    {								\
	      set_glyph_string_background_width (s, START, LAST_X);	\
	      (X) += s->width;						\
	    }								\
	}								\
    } while (false)


#define BUILD_GLYPH_STRINGS(START, END, HEAD, TAIL, HL, X, LAST_X)	\
    BUILD_GLYPH_STRINGS_1(START, END, HEAD, TAIL, HL, X, LAST_X)	\
    BUILD_GLYPH_STRINGS_XW(START, END, HEAD, TAIL, HL, X, LAST_X)	\
    BUILD_GLYPH_STRINGS_2(START, END, HEAD, TAIL, HL, X, LAST_X)


/* Draw glyphs between START and END in AREA of ROW on window W,
   starting at x-position X.  X is relative to AREA in W.  HL is a
   face-override with the following meaning:

   DRAW_NORMAL_TEXT	draw normally
   DRAW_CURSOR		draw in cursor face
   DRAW_MOUSE_FACE	draw in mouse face.
   DRAW_INVERSE_VIDEO	draw in mode line face
   DRAW_IMAGE_SUNKEN	draw an image with a sunken relief around it
   DRAW_IMAGE_RAISED	draw an image with a raised relief around it

   If OVERLAPS is non-zero, draw only the foreground of characters and
   clip to the physical height of ROW.  Non-zero value also defines
   the overlapping part to be drawn:

   OVERLAPS_PRED		overlap with preceding rows
   OVERLAPS_SUCC		overlap with succeeding rows
   OVERLAPS_BOTH		overlap with both preceding/succeeding rows
   OVERLAPS_ERASED_CURSOR	overlap with erased cursor area

   Value is the x-position reached, relative to AREA of W.  */

static int
draw_glyphs (struct window *w, int x, struct glyph_row *row,
	     enum glyph_row_area area, ptrdiff_t start, ptrdiff_t end,
	     enum draw_glyphs_face hl, int overlaps)
{
  struct glyph_string *head, *tail;
  struct glyph_string *s;
  struct glyph_string *clip_head = NULL, *clip_tail = NULL;
  int i, j, x_reached, last_x, area_left = 0;
  struct frame *f = XFRAME (WINDOW_FRAME (w));

  ALLOCATE_HDC (hdc, f);

  /* Let's rather be paranoid than getting a SEGV.  */
  end = min (end, row->used[area]);
  start = clip_to_bounds (0, start, end);

  /* Translate X to frame coordinates.  Set last_x to the right
     end of the drawing area.  */
  if (row->full_width_p)
    {
      /* X is relative to the left edge of W, without scroll bars
	 or fringes.  */
      area_left = WINDOW_LEFT_EDGE_X (w);
      last_x = (WINDOW_LEFT_EDGE_X (w) + WINDOW_PIXEL_WIDTH (w)
		- (row->mode_line_p ? WINDOW_RIGHT_DIVIDER_WIDTH (w) : 0));
    }
  else
    {
      area_left = window_box_left (w, area);
      last_x = area_left + window_box_width (w, area);
    }
  x += area_left;

  /* Build a doubly-linked list of glyph_string structures between
     head and tail from what we have to draw.  Note that the macro
     BUILD_GLYPH_STRINGS will modify its start parameter.  That's
     the reason we use a separate variable `i'.  */
  i = start;
  USE_SAFE_ALLOCA;
  BUILD_GLYPH_STRINGS (i, end, head, tail, hl, x, last_x);
  if (tail)
    {
      s = glyph_string_containing_background_width (tail);
      x_reached = s->x + s->background_width;
    }
  else
    x_reached = x;

  /* If there are any glyphs with lbearing < 0 or rbearing > width in
     the row, redraw some glyphs in front or following the glyph
     strings built above.  */
  if (head && !overlaps && row->contains_overlapping_glyphs_p)
    {
      struct glyph_string *h, *t;
      Mouse_HLInfo *hlinfo = MOUSE_HL_INFO (f);
      int mouse_beg_col UNINIT, mouse_end_col UNINIT;
      bool check_mouse_face = false;
      int dummy_x = 0;

      /* If mouse highlighting is on, we may need to draw adjacent
	 glyphs using mouse-face highlighting.  */
      if (area == TEXT_AREA && row->mouse_face_p
	  && hlinfo->mouse_face_beg_row >= 0
	  && hlinfo->mouse_face_end_row >= 0)
	{
	  ptrdiff_t row_vpos = MATRIX_ROW_VPOS (row, w->current_matrix);

	  if (row_vpos >= hlinfo->mouse_face_beg_row
	      && row_vpos <= hlinfo->mouse_face_end_row)
	    {
	      check_mouse_face = true;
	      mouse_beg_col = (row_vpos == hlinfo->mouse_face_beg_row)
		? hlinfo->mouse_face_beg_col : 0;
	      mouse_end_col = (row_vpos == hlinfo->mouse_face_end_row)
		? hlinfo->mouse_face_end_col
		: row->used[TEXT_AREA];
	    }
	}

      /* Compute overhangs for all glyph strings.  */
      if (FRAME_RIF (f)->compute_glyph_string_overhangs)
	for (s = head; s; s = s->next)
	  FRAME_RIF (f)->compute_glyph_string_overhangs (s);

      /* Prepend glyph strings for glyphs in front of the first glyph
	 string that are overwritten because of the first glyph
	 string's left overhang.  The background of all strings
	 prepended must be drawn because the first glyph string
	 draws over it.  */
      i = left_overwritten (head);
      if (i >= 0)
	{
	  enum draw_glyphs_face overlap_hl;

	  /* If this row contains mouse highlighting, attempt to draw
	     the overlapped glyphs with the correct highlight.  This
	     code fails if the overlap encompasses more than one glyph
	     and mouse-highlight spans only some of these glyphs.
	     However, making it work perfectly involves a lot more
	     code, and I don't know if the pathological case occurs in
	     practice, so we'll stick to this for now.  --- cyd  */
	  if (check_mouse_face
	      && mouse_beg_col < start && mouse_end_col > i)
	    overlap_hl = DRAW_MOUSE_FACE;
	  else
	    overlap_hl = DRAW_NORMAL_TEXT;

	  if (hl != overlap_hl)
	    clip_head = head;
	  j = i;
	  BUILD_GLYPH_STRINGS (j, start, h, t,
			       overlap_hl, dummy_x, last_x);
	  start = i;
	  compute_overhangs_and_x (t, head->x, true);
	  prepend_glyph_string_lists (&head, &tail, h, t);
	  if (clip_head == NULL)
	    clip_head = head;
	}

      /* Prepend glyph strings for glyphs in front of the first glyph
	 string that overwrite that glyph string because of their
	 right overhang.  For these strings, only the foreground must
	 be drawn, because it draws over the glyph string at `head'.
	 The background must not be drawn because this would overwrite
	 right overhangs of preceding glyphs for which no glyph
	 strings exist.  */
      i = left_overwriting (head);
      if (i >= 0)
	{
	  enum draw_glyphs_face overlap_hl;

	  if (check_mouse_face
	      && mouse_beg_col < start && mouse_end_col > i)
	    overlap_hl = DRAW_MOUSE_FACE;
	  else
	    overlap_hl = DRAW_NORMAL_TEXT;

	  if (hl == overlap_hl || clip_head == NULL)
	    clip_head = head;
	  BUILD_GLYPH_STRINGS (i, start, h, t,
			       overlap_hl, dummy_x, last_x);
	  for (s = h; s; s = s->next)
	    s->background_filled_p = true;
	  compute_overhangs_and_x (t, head->x, true);
	  prepend_glyph_string_lists (&head, &tail, h, t);
	}

      /* Append glyphs strings for glyphs following the last glyph
	 string tail that are overwritten by tail.  The background of
	 these strings has to be drawn because tail's foreground draws
	 over it.  */
      i = right_overwritten (tail);
      if (i >= 0)
	{
	  enum draw_glyphs_face overlap_hl;

	  if (check_mouse_face
	      && mouse_beg_col < i && mouse_end_col > end)
	    overlap_hl = DRAW_MOUSE_FACE;
	  else
	    overlap_hl = DRAW_NORMAL_TEXT;

	  if (hl != overlap_hl)
	    clip_tail = tail;
	  BUILD_GLYPH_STRINGS (end, i, h, t,
			       overlap_hl, x, last_x);
	  /* Because BUILD_GLYPH_STRINGS updates the first argument,
	     we don't have `end = i;' here.  */
	  compute_overhangs_and_x (h, tail->x + tail->width, false);
	  append_glyph_string_lists (&head, &tail, h, t);
	  if (clip_tail == NULL)
	    clip_tail = tail;
	}

      /* Append glyph strings for glyphs following the last glyph
	 string tail that overwrite tail.  The foreground of such
	 glyphs has to be drawn because it writes into the background
	 of tail.  The background must not be drawn because it could
	 paint over the foreground of following glyphs.  */
      i = right_overwriting (tail);
      if (i >= 0)
	{
	  enum draw_glyphs_face overlap_hl;
	  if (check_mouse_face
	      && mouse_beg_col < i && mouse_end_col > end)
	    overlap_hl = DRAW_MOUSE_FACE;
	  else
	    overlap_hl = DRAW_NORMAL_TEXT;

	  if (hl == overlap_hl || clip_tail == NULL)
	    clip_tail = tail;
	  i++;			/* We must include the Ith glyph.  */
	  BUILD_GLYPH_STRINGS (end, i, h, t,
			       overlap_hl, x, last_x);
	  for (s = h; s; s = s->next)
	    s->background_filled_p = true;
	  compute_overhangs_and_x (h, tail->x + tail->width, false);
	  append_glyph_string_lists (&head, &tail, h, t);
	}
      tail = glyph_string_containing_background_width (tail);
      if (clip_tail)
	clip_tail = glyph_string_containing_background_width (clip_tail);
      if (clip_head || clip_tail)
	for (s = head; s; s = s->next)
	  {
	    s->clip_head = clip_head;
	    s->clip_tail = clip_tail;
	  }
    }

  /* Draw all strings.  */
  for (s = head; s; s = s->next)
    FRAME_RIF (f)->draw_glyph_string (s);

  /* When focus a sole frame and move horizontally, this clears on_p
     causing a failure to erase prev cursor position. */
  if (area == TEXT_AREA
      && !row->full_width_p
      /* When drawing overlapping rows, only the glyph strings'
	 foreground is drawn, which doesn't erase a cursor
	 completely. */
      && !overlaps)
    {
      int x0 = clip_head ? clip_head->x : (head ? head->x : x);
      int x1 = (clip_tail ? clip_tail->x + clip_tail->background_width
		: (tail ? tail->x + tail->background_width : x));
      x0 -= area_left;
      x1 -= area_left;

      notice_overwritten_cursor (w, TEXT_AREA, x0, x1,
				 row->y, MATRIX_ROW_BOTTOM_Y (row));
    }

  /* Value is the x-position up to which drawn, relative to AREA of W.
     This doesn't include parts drawn because of overhangs.  */
  if (row->full_width_p)
    x_reached = FRAME_TO_WINDOW_PIXEL_X (w, x_reached);
  else
    x_reached -= area_left;

  RELEASE_HDC (hdc, f);

  SAFE_FREE ();
  return x_reached;
}

/* Find the first glyph in the run of underlined glyphs preceding the
   beginning of glyph string S, and return its font (which could be
   NULL).  This is needed because that font determines the underline
   position and thickness for the entire run of the underlined glyphs.
   This function is called from the draw_glyph_string method of GUI
   frame's redisplay interface (RIF) when it needs to draw in an
   underlined face.  */
struct font *
font_for_underline_metrics (struct glyph_string *s)
{
  struct glyph *g0 = s->row->glyphs[s->area], *g;

  for (g = s->first_glyph - 1; g >= g0; g--)
    {
      struct face *prev_face = FACE_FROM_ID (s->f, g->face_id);
      if (!(prev_face && prev_face->underline != FACE_NO_UNDERLINE))
	break;
    }

  /* If preceding glyphs are not underlined, use the font of S.  */
  if (g == s->first_glyph - 1)
    return s->font;
  else
    {
      /* Otherwise use the font of the last glyph we saw in the above
	 loop whose face had the underline_p flag set.  */
      return FACE_FROM_ID (s->f, g[1].face_id)->font;
    }
}

/* Expand row matrix if too narrow.  Don't expand if area
   is not present.  */

#define IT_EXPAND_MATRIX_WIDTH(it, area)		\
  {							\
    if (!it->f->fonts_changed				\
	&& (it->glyph_row->glyphs[area]			\
	    < it->glyph_row->glyphs[area + 1]))		\
      {							\
	it->w->ncols_scale_factor++;			\
	it->f->fonts_changed = true;			\
      }							\
  }

/* Store one glyph for IT->char_to_display in IT->glyph_row.
   Called from gui_produce_glyphs when IT->glyph_row is non-null.  */

static void
append_glyph (struct it *it)
{
  struct glyph *glyph;
  enum glyph_row_area area = it->area;

  eassert (it->glyph_row);
  eassert (it->char_to_display != '\n' && it->char_to_display != '\t');

  glyph = it->glyph_row->glyphs[area] + it->glyph_row->used[area];
  if (glyph < it->glyph_row->glyphs[area + 1])
    {
      /* If the glyph row is reversed, we need to prepend the glyph
	 rather than append it.  */
      if (it->glyph_row->reversed_p && area == TEXT_AREA)
	{
	  struct glyph *g;

	  /* Make room for the additional glyph.  */
	  for (g = glyph - 1; g >= it->glyph_row->glyphs[area]; g--)
	    g[1] = *g;
	  glyph = it->glyph_row->glyphs[area];
	}
      glyph->charpos = CHARPOS (it->position);
      glyph->object = it->object;
      if (it->pixel_width > 0)
	{
	  eassert (it->pixel_width <= SHRT_MAX);
	  glyph->pixel_width = it->pixel_width;
	  glyph->padding_p = false;
	}
      else
	{
	  /* Assure at least 1-pixel width.  Otherwise, cursor can't
	     be displayed correctly.  */
	  glyph->pixel_width = 1;
	  glyph->padding_p = true;
	}
      glyph->ascent = it->ascent;
      glyph->descent = it->descent;
      glyph->voffset = it->voffset;
      glyph->type = CHAR_GLYPH;
      glyph->avoid_cursor_p = it->avoid_cursor_p;
      glyph->multibyte_p = it->multibyte_p;
      if (it->glyph_row->reversed_p && area == TEXT_AREA)
	{
	  /* In R2L rows, the left and the right box edges need to be
	     drawn in reverse direction.  */
	  glyph->right_box_line_p = it->start_of_box_run_p;
	  glyph->left_box_line_p = it->end_of_box_run_p;
	}
      else
	{
	  glyph->left_box_line_p = it->start_of_box_run_p;
	  glyph->right_box_line_p = it->end_of_box_run_p;
	}
      glyph->overlaps_vertically_p = (it->phys_ascent > it->ascent
				      || it->phys_descent > it->descent);
      glyph->glyph_not_available_p = it->glyph_not_available_p;
      glyph->face_id = it->face_id;
      glyph->u.ch = it->char_to_display;
      glyph->slice.img = null_glyph_slice;
      glyph->font_type = FONT_TYPE_UNKNOWN;
      if (it->bidi_p)
	{
	  glyph->resolved_level = it->bidi_it.resolved_level;
	  eassert ((it->bidi_it.type & 7) == it->bidi_it.type);
	  glyph->bidi_type = it->bidi_it.type;
	}
      else
	{
	  glyph->resolved_level = 0;
	  glyph->bidi_type = UNKNOWN_BT;
	}
      ++it->glyph_row->used[area];
    }
  else
    IT_EXPAND_MATRIX_WIDTH (it, area);
}

/* Store one glyph for the composition IT->cmp_it.id in IT->glyph_row.
   Called from gui_produce_glyphs when IT->glyph_row is non-null.  */

static void
append_composite_glyph (struct it *it)
{
  struct glyph *glyph;
  enum glyph_row_area area = it->area;

  eassert (it->glyph_row);

  glyph = it->glyph_row->glyphs[area] + it->glyph_row->used[area];
  if (glyph < it->glyph_row->glyphs[area + 1])
    {
      /* If the glyph row is reversed, we need to prepend the glyph
	 rather than append it.  */
      if (it->glyph_row->reversed_p && it->area == TEXT_AREA)
	{
	  struct glyph *g;

	  /* Make room for the new glyph.  */
	  for (g = glyph - 1; g >= it->glyph_row->glyphs[it->area]; g--)
	    g[1] = *g;
	  glyph = it->glyph_row->glyphs[it->area];
	}
      glyph->charpos = it->cmp_it.charpos;
      glyph->object = it->object;
      eassert (it->pixel_width <= SHRT_MAX);
      glyph->pixel_width = it->pixel_width;
      glyph->ascent = it->ascent;
      glyph->descent = it->descent;
      glyph->voffset = it->voffset;
      glyph->type = COMPOSITE_GLYPH;
      if (it->cmp_it.ch < 0)
	{
	  glyph->u.cmp.automatic = false;
	  glyph->u.cmp.id = it->cmp_it.id;
	  glyph->slice.cmp.from = glyph->slice.cmp.to = 0;
	}
      else
	{
	  glyph->u.cmp.automatic = true;
	  glyph->u.cmp.id = it->cmp_it.id;
	  glyph->slice.cmp.from = it->cmp_it.from;
	  glyph->slice.cmp.to = it->cmp_it.to - 1;
	}
      glyph->avoid_cursor_p = it->avoid_cursor_p;
      glyph->multibyte_p = it->multibyte_p;
      if (it->glyph_row->reversed_p && area == TEXT_AREA)
	{
	  /* In R2L rows, the left and the right box edges need to be
	     drawn in reverse direction.  */
	  glyph->right_box_line_p = it->start_of_box_run_p;
	  glyph->left_box_line_p = it->end_of_box_run_p;
	}
      else
	{
	  glyph->left_box_line_p = it->start_of_box_run_p;
	  glyph->right_box_line_p = it->end_of_box_run_p;
	}
      glyph->overlaps_vertically_p = (it->phys_ascent > it->ascent
				      || it->phys_descent > it->descent);
      glyph->padding_p = false;
      glyph->glyph_not_available_p = it->glyph_not_available_p;
      glyph->face_id = it->face_id;
      glyph->font_type = FONT_TYPE_UNKNOWN;
      if (it->bidi_p)
	{
	  glyph->resolved_level = it->bidi_it.resolved_level;
	  eassert ((it->bidi_it.type & 7) == it->bidi_it.type);
	  glyph->bidi_type = it->bidi_it.type;
	}
      ++it->glyph_row->used[area];
    }
  else
    IT_EXPAND_MATRIX_WIDTH (it, area);
}


/* Change IT->ascent and IT->height according to the setting of
   IT->voffset.  */

static void
take_vertical_position_into_account (struct it *it)
{
  if (it->voffset)
    {
      if (it->voffset < 0)
	/* Increase the ascent so that we can display the text higher
	   in the line.  */
	it->ascent -= it->voffset;
      else
	/* Increase the descent so that we can display the text lower
	   in the line.  */
	it->descent += it->voffset;
    }
}


/* Produce glyphs/get display metrics for the image IT is loaded with.
   See the description of struct display_iterator in dispextern.h for
   an overview of struct display_iterator.  */

static void
produce_image_glyph (struct it *it)
{
  struct image *img;
  struct face *face;
  int glyph_ascent, crop;
  struct glyph_slice slice;

  eassert (it->what == IT_IMAGE);

  face = FACE_FROM_ID (it->f, it->face_id);
  /* Make sure X resources of the face is loaded.  */
  prepare_face_for_display (it->f, face);

  if (it->image_id < 0)
    {
      /* Fringe bitmap.  */
      it->ascent = it->phys_ascent = 0;
      it->descent = it->phys_descent = 0;
      it->pixel_width = 0;
      it->nglyphs = 0;
      return;
    }

  img = IMAGE_FROM_ID (it->f, it->image_id);
  /* Make sure X resources of the image is loaded.  */
  prepare_image_for_display (it->f, img);

  slice.x = slice.y = 0;
  slice.width = img->width;
  slice.height = img->height;

  if (FIXNUMP (it->slice.x))
    slice.x = XFIXNUM (it->slice.x);
  else if (FLOATP (it->slice.x))
    slice.x = XFLOAT_DATA (it->slice.x) * img->width;

  if (FIXNUMP (it->slice.y))
    slice.y = XFIXNUM (it->slice.y);
  else if (FLOATP (it->slice.y))
    slice.y = XFLOAT_DATA (it->slice.y) * img->height;

  if (FIXNUMP (it->slice.width))
    slice.width = XFIXNUM (it->slice.width);
  else if (FLOATP (it->slice.width))
    slice.width = XFLOAT_DATA (it->slice.width) * img->width;

  if (FIXNUMP (it->slice.height))
    slice.height = XFIXNUM (it->slice.height);
  else if (FLOATP (it->slice.height))
    slice.height = XFLOAT_DATA (it->slice.height) * img->height;

  if (slice.x >= img->width)
    slice.x = img->width;
  if (slice.y >= img->height)
    slice.y = img->height;
  if (slice.x + slice.width >= img->width)
    slice.width = img->width - slice.x;
  if (slice.y + slice.height > img->height)
    slice.height = img->height - slice.y;

  if (slice.width == 0 || slice.height == 0)
    return;

  it->ascent = it->phys_ascent = glyph_ascent = image_ascent (img, face, &slice);

  it->descent = slice.height - glyph_ascent;
  if (slice.y == 0)
    it->descent += img->vmargin;
  if (slice.y + slice.height == img->height)
    it->descent += img->vmargin;
  it->phys_descent = it->descent;

  it->pixel_width = slice.width;
  if (slice.x == 0)
    it->pixel_width += img->hmargin;
  if (slice.x + slice.width == img->width)
    it->pixel_width += img->hmargin;

  /* It's quite possible for images to have an ascent greater than
     their height, so don't get confused in that case.  */
  if (it->descent < 0)
    it->descent = 0;

  it->nglyphs = 1;

  if (face->box != FACE_NO_BOX)
    {
      /* If you change the logic here, please change it in
	 get_cursor_offset_for_mouse_face as well. */
      if (face->box_horizontal_line_width > 0)
	{
	  if (slice.y == 0)
	    it->ascent += face->box_horizontal_line_width;
	  if (slice.y + slice.height == img->height)
	    it->descent += face->box_horizontal_line_width;
	}

      if (face->box_vertical_line_width > 0)
	{
	  if (it->start_of_box_run_p && slice.x == 0)
	    it->pixel_width += face->box_vertical_line_width;
	  if (it->end_of_box_run_p && slice.x + slice.width == img->width)
	    it->pixel_width += face->box_vertical_line_width;
	}
    }

  take_vertical_position_into_account (it);

  /* Automatically crop wide image glyphs at right edge so we can
     draw the cursor on same display row.  */
  if ((crop = it->pixel_width - (it->last_visible_x - it->current_x), crop > 0)
      && (it->hpos == 0 || it->pixel_width > it->last_visible_x / 4))
    {
      it->pixel_width -= crop;
      slice.width -= crop;
    }

  if (it->glyph_row)
    {
      struct glyph *glyph;
      enum glyph_row_area area = it->area;

      glyph = it->glyph_row->glyphs[area] + it->glyph_row->used[area];
      if (it->glyph_row->reversed_p)
	{
	  struct glyph *g;

	  /* Make room for the new glyph.  */
	  for (g = glyph - 1; g >= it->glyph_row->glyphs[it->area]; g--)
	    g[1] = *g;
	  glyph = it->glyph_row->glyphs[it->area];
	}
      if (glyph < it->glyph_row->glyphs[area + 1])
	{
	  glyph->charpos = CHARPOS (it->position);
	  glyph->object = it->object;
	  glyph->pixel_width = clip_to_bounds (-1, it->pixel_width, SHRT_MAX);
	  glyph->ascent = glyph_ascent;
	  glyph->descent = it->descent;
	  glyph->voffset = it->voffset;
	  glyph->type = IMAGE_GLYPH;
	  glyph->avoid_cursor_p = it->avoid_cursor_p;
	  glyph->multibyte_p = it->multibyte_p;
	  if (it->glyph_row->reversed_p && area == TEXT_AREA)
	    {
	      /* In R2L rows, the left and the right box edges need to be
		 drawn in reverse direction.  */
	      glyph->right_box_line_p = it->start_of_box_run_p;
	      glyph->left_box_line_p = it->end_of_box_run_p;
	    }
	  else
	    {
	      glyph->left_box_line_p = it->start_of_box_run_p;
	      glyph->right_box_line_p = it->end_of_box_run_p;
	    }
	  glyph->overlaps_vertically_p = false;
          glyph->padding_p = false;
	  glyph->glyph_not_available_p = false;
	  glyph->face_id = it->face_id;
	  glyph->u.img_id = img->id;
	  glyph->slice.img = slice;
	  glyph->font_type = FONT_TYPE_UNKNOWN;
	  if (it->bidi_p)
	    {
	      glyph->resolved_level = it->bidi_it.resolved_level;
	      eassert ((it->bidi_it.type & 7) == it->bidi_it.type);
	      glyph->bidi_type = it->bidi_it.type;
	    }
	  ++it->glyph_row->used[area];
	}
      else
	IT_EXPAND_MATRIX_WIDTH (it, area);
    }
}

static void
produce_xwidget_glyph (struct it *it)
{
#ifdef HAVE_XWIDGETS
  struct xwidget *xw;
  int glyph_ascent, crop;
  eassert (it->what == IT_XWIDGET);

  struct face *face = FACE_FROM_ID (it->f, it->face_id);
  /* Make sure X resources of the face is loaded.  */
  prepare_face_for_display (it->f, face);

  xw = it->xwidget;
  it->ascent = it->phys_ascent = glyph_ascent = xw->height/2;
  it->descent = xw->height/2;
  it->phys_descent = it->descent;
  it->pixel_width = xw->width;
  /* It's quite possible for images to have an ascent greater than
     their height, so don't get confused in that case.  */
  if (it->descent < 0)
    it->descent = 0;

  it->nglyphs = 1;

  if (face->box != FACE_NO_BOX)
    {
      if (face->box_horizontal_line_width > 0)
	{
	  it->ascent += face->box_horizontal_line_width;
	  it->descent += face->box_horizontal_line_width;
	}

      if (face->box_vertical_line_width > 0)
	{
	  if (it->start_of_box_run_p)
	    it->pixel_width += face->box_vertical_line_width;
	  it->pixel_width += face->box_vertical_line_width;
	}
    }

  take_vertical_position_into_account (it);

  /* Automatically crop wide image glyphs at right edge so we can
     draw the cursor on same display row.  */
  crop = it->pixel_width - (it->last_visible_x - it->current_x);
  if (crop > 0 && (it->hpos == 0 || it->pixel_width > it->last_visible_x / 4))
    it->pixel_width -= crop;

  if (it->glyph_row)
    {
      enum glyph_row_area area = it->area;
      struct glyph *glyph
	= it->glyph_row->glyphs[area] + it->glyph_row->used[area];

      if (it->glyph_row->reversed_p)
	{
	  struct glyph *g;

	  /* Make room for the new glyph.  */
	  for (g = glyph - 1; g >= it->glyph_row->glyphs[it->area]; g--)
	    g[1] = *g;
	  glyph = it->glyph_row->glyphs[it->area];
	}
      if (glyph < it->glyph_row->glyphs[area + 1])
	{
	  glyph->charpos = CHARPOS (it->position);
	  glyph->object = it->object;
	  glyph->pixel_width = clip_to_bounds (-1, it->pixel_width, SHRT_MAX);
	  glyph->ascent = glyph_ascent;
	  glyph->descent = it->descent;
	  glyph->voffset = it->voffset;
	  glyph->type = XWIDGET_GLYPH;
	  glyph->avoid_cursor_p = it->avoid_cursor_p;
	  glyph->multibyte_p = it->multibyte_p;
	  if (it->glyph_row->reversed_p && area == TEXT_AREA)
	    {
	      /* In R2L rows, the left and the right box edges need to be
		 drawn in reverse direction.  */
	      glyph->right_box_line_p = it->start_of_box_run_p;
	      glyph->left_box_line_p = it->end_of_box_run_p;
	    }
	  else
	    {
	      glyph->left_box_line_p = it->start_of_box_run_p;
	      glyph->right_box_line_p = it->end_of_box_run_p;
	    }
          glyph->overlaps_vertically_p = 0;
          glyph->padding_p = 0;
	  glyph->glyph_not_available_p = 0;
	  glyph->face_id = it->face_id;
          glyph->u.xwidget = it->xwidget->xwidget_id;
	  glyph->font_type = FONT_TYPE_UNKNOWN;
	  if (it->bidi_p)
	    {
	      glyph->resolved_level = it->bidi_it.resolved_level;
	      eassert ((it->bidi_it.type & 7) == it->bidi_it.type);
	      glyph->bidi_type = it->bidi_it.type;
	    }
	  ++it->glyph_row->used[area];
	}
      else
	IT_EXPAND_MATRIX_WIDTH (it, area);
    }
#endif
}

/* Append a stretch glyph to IT->glyph_row.  OBJECT is the source
   of the glyph, WIDTH and HEIGHT are the width and height of the
   stretch.  ASCENT is the ascent of the glyph (0 <= ASCENT <= HEIGHT).  */

static void
append_stretch_glyph (struct it *it, Lisp_Object object,
		      int width, int height, int ascent)
{
  struct glyph *glyph;
  enum glyph_row_area area = it->area;

  eassert (ascent >= 0 && ascent <= height);

  glyph = it->glyph_row->glyphs[area] + it->glyph_row->used[area];
  if (glyph < it->glyph_row->glyphs[area + 1])
    {
      /* If the glyph row is reversed, we need to prepend the glyph
	 rather than append it.  */
      if (it->glyph_row->reversed_p && area == TEXT_AREA)
	{
	  struct glyph *g;

	  /* Make room for the additional glyph.  */
	  for (g = glyph - 1; g >= it->glyph_row->glyphs[area]; g--)
	    g