--
-- This file is part of TALER
-- Copyright (C) 2014--2023 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--


CREATE FUNCTION create_table_purse_decision(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_decision';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(purse_decision_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',action_timestamp INT8 NOT NULL'
      ',refunded BOOL NOT NULL'
      ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Purses that were decided upon (refund or merge)'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
END
$$;

CREATE FUNCTION constrain_table_purse_decision(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_decision';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_purse_action_serial_id_key'
    ' UNIQUE (purse_decision_serial_id) '
  );
END
$$;


CREATE OR REPLACE FUNCTION purse_decision_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  UPDATE purse_requests
     SET was_decided=TRUE
   WHERE purse_pub=NEW.purse_pub;
  IF NEW.refunded
  THEN
    INSERT INTO coin_history
      (coin_pub
      ,table_name
      ,serial_id)
    SELECT
      pd.coin_pub
     ,'purse_decision'
     ,NEW.purse_decision_serial_id
    FROM purse_deposits pd
    WHERE purse_pub = NEW.purse_pub;
  ELSE
    INSERT INTO reserve_history
      (reserve_pub
      ,table_name
      ,serial_id)
    SELECT
      reserve_pub
     ,'purse_decision'
     ,NEW.purse_decision_serial_id
    FROM purse_merges
    WHERE purse_pub=NEW.purse_pub;
  END IF;
  RETURN NEW;
END $$;
COMMENT ON FUNCTION purse_decision_insert_trigger()
  IS 'Automatically generate coin history entry and update decision status for the purse.';


CREATE FUNCTION master_table_purse_decision()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER purse_decision_on_insert
    AFTER INSERT
     ON purse_decision
     FOR EACH ROW EXECUTE FUNCTION purse_decision_insert_trigger();
END $$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_decision'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_decision'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('purse_decision'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
