package Crypt::Sodium::XS::onetimeauth;
use strict;
use warnings;

use Crypt::Sodium::XS;
use Exporter 'import';

_define_constants();

my @constant_bases = qw(
  BYTES
  KEYBYTES
);

my @bases = qw(
  init
  keygen
  verify
);

my $default = [
  "onetimeauth",
  (map { "onetimeauth_$_" } @bases),
  (map { "onetimeauth_$_" } @constant_bases, "PRIMITIVE"),
];
my $poly1305 = [
  "onetimeauth_poly1305",
  (map { "onetimeauth_poly1305_$_" } @bases),
  (map { "onetimeauth_poly1305_$_" } @constant_bases),
];

our %EXPORT_TAGS = (
  all => [ @$default, @$poly1305 ],
  default => $default,
  poly1305 => $poly1305,
);

our @EXPORT_OK = @{$EXPORT_TAGS{all}};

1;

__END__

=encoding utf8

=head1 NAME

Crypt::Sodium::XS::onetimeauth - Single-use secret key message authentication

=head1 SYNOPSIS

  use Crypt::Sodium::XS::onetimeauth ":default";

  # NOTE: use a new key for every message
  my $key = onetimeauth_keygen();
  my $msg = "authenticate me";

  my $mac = onetimeauth($msg, $key);
  die "message tampered!" unless onetimeauth_verify($mac, $msg, $key);

=head1 DESCRIPTION

L<Crypt::Sodium::XS::onetimeauth> uses Poly1305, a Wegman-Carter authenticator
designed by D. J. Bernstein.  Poly1305 takes a 32-byte, one-time key and a
message and produces a 16-byte tag that authenticates the message such that an
attacker has a negligible chance of producing a valid tag for a inauthentic
message.

Poly1305 keys have to be:

=over 4

=item * secret

An attacker can compute a valid authentication tag for any message, for any
given key. The security of Poly1305 relies on the fact that attackers don't
know the key being used to compute the tag. This implies that they have to be:

=item * unpredictable

Do not use timestamps or counters.

=item * unique

Never reuse a key. A new key is required for every single message. The key can
be recovered if two messages are authenticated with the same key.

=back

The standard way to use Poly1305's is to derive a dedicated subkey from a (key,
nonce) tuple, for example by taking the first bytes generated by a stream
cipher.

Due to its output size, Poly1305 is recommended for online protocols,
exchanging many small messages, rather than for authenticating very large
files.

Finally, Poly1305 is not a replacement for a hash function.

=head1 FUNCTIONS

Nothing is exported by default. A C<:default> tag imports the functions and
constants as documented below. A separate import tag is provided for each of
the primitives listed in L</PRIMITIVES>. For example, C<:poly1305> imports
C<onetimeauth_poly1305_verify>. You should use at least one import tag.

=head2 onetimeauth

  my $mac = onetimeauth($message, $key);

=head2 onetimeauth_init

  my $multipart = onetimeauth_init();

Returns a multipart onetimeauth object. See L</MULTI-PART INTERFACE>.

=head2 onetimeauth_keygen

  my $key = onetimeauth_keygen();

=head2 onetimeauth_verify

  my $is_valid = onetimeauth_verify($mac, $message, $key);

=head1 MULTI-PART INTERFACE

A multipart onetimeauth object is created by calling the L</onetimeauth_init>
function. Data to be authenticated is added by calling the L</update> method of
that object as many times as desired. An output mac is generated by calling its
L</final> method. Do not use the object after calling L</final>.

The multipart onetimeauth object is an opaque object which provides the
following methods:

=head2 clone

  my $multipart_copy = $multipart->clone;

=head2 final

  my $mac = $multipart->final;

=head2 update

  $multipart->update($message);
  $multipart->update(@messages);

=head1 CONSTANTS

=head2 onetimeauth_PRIMITIVE

  my $default_primitive = onetimeauth_PRIMITIVE();

=head2 onetimeauth_BYTES

  my $mac_length = onetimeauth_BYTES();

=head2 onetimeauth_KEYBYTES

  my $key_length = onetimeauth_KEYBYTES();

=head1 PRIMITIVES

All constants (except _PRIMITIVE) and functions have
C<onetimeauth_E<lt>primitiveE<gt>>-prefixed couterparts (e.g.,
onetimeauth_poly1305_keypair, onetimeauth_poly1305_KEYBYTES).

=over 4

=item * poly1305

=back

=head1 SEE ALSO

=over 4

=item L<Crypt::Sodium::XS>

=item L<Crypt::Sodium::XS::OO::onetimeauth>

=item L<https://doc.libsodium.org/advanced/poly1305>

=back

=head1 FEEDBACK

For reporting bugs, giving feedback, submitting patches, etc. please use the
following:

=over 4

=item *

RT queue at L<https://rt.cpan.org/Dist/Display.html?Name=Crypt-Sodium-XS>

=item *

IRC channel C<#sodium> on C<irc.perl.org>.

=item *

Email the author directly.

=back

=head1 AUTHOR

Brad Barden E<lt>perlmodules@5c30.orgE<gt>

=head1 COPYRIGHT & LICENSE

Copyright (c) 2022 Brad Barden. All rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut
