static char rcsid[] = "800Query.c,v 1.12 1995/11/01 20:33:50 duane Exp";
/*
 *  800Query.c - httpd-based CGI program that processes a user query
 *  from the Mosaic forms interface, sends the query to the appropriate
 *  broker, and then sends the result set back to the user.
 *
 *  This program is used with the 1-800 Phone Numbers Broker:
 *
 *      http://harvest.cs.colorado.edu/Harvest/brokers/800/
 * 
 *  Note that NCSA httpd 1.3 waits for the CGI program to completely 
 *  exit before it sends any data to the client.  Oh well...
 *
 *  Usage:  Run from cgi-bin directory under httpd server
 *
 *  Supports the following CGI tags:
 *      host            If defined, specifies the broker host
 *      port            If defined, specifies the broker port number
 *      query           The query string
 *      class           Will AND the keywords to the end of "query"
 *      caseflag        If defined, adds '#index case insenstive' to flags
 *      wordflag        If defined, adds '#index matchword' to flags
 *      errorflag       If defined, adds '#index error number' to flags
 *      maxresultflag   If defined, adds '#index maxresult number' to flags
 *      opaqueflag      If defined, adds '#opaque' to flags
 *      descflag        If defined, adds '#desc' to flags
 *      verbose         If defined, prints URLs concisely
 *
 *  Bill Camargo & Darren Hardy, July 1994  
 *
 *  ----------------------------------------------------------------------
 *  Copyright (c) 1994, 1995.  All rights reserved.
 *  
 *    The Harvest software was developed by the Internet Research Task
 *    Force Research Group on Resource Discovery (IRTF-RD):
 *  
 *          Mic Bowman of Transarc Corporation.
 *          Peter Danzig of the University of Southern California.
 *          Darren R. Hardy of the University of Colorado at Boulder.
 *          Udi Manber of the University of Arizona.
 *          Michael F. Schwartz of the University of Colorado at Boulder.
 *          Duane Wessels of the University of Colorado at Boulder.
 *  
 *    This copyright notice applies to software in the Harvest
 *    ``src/'' directory only.  Users should consult the individual
 *    copyright notices in the ``components/'' subdirectories for
 *    copyright information about other software bundled with the
 *    Harvest source code distribution.
 *  
 *  TERMS OF USE
 *    
 *    The Harvest software may be used and re-distributed without
 *    charge, provided that the software origin and research team are
 *    cited in any use of the system.  Most commonly this is
 *    accomplished by including a link to the Harvest Home Page
 *    (http://harvest.cs.colorado.edu/) from the query page of any
 *    Broker you deploy, as well as in the query result pages.  These
 *    links are generated automatically by the standard Broker
 *    software distribution.
 *    
 *    The Harvest software is provided ``as is'', without express or
 *    implied warranty, and with no support nor obligation to assist
 *    in its use, correction, modification or enhancement.  We assume
 *    no liability with respect to the infringement of copyrights,
 *    trade secrets, or any patents, and are not responsible for
 *    consequential damages.  Proper use of the Harvest software is
 *    entirely the responsibility of the user.
 *  
 *  DERIVATIVE WORKS
 *  
 *    Users may make derivative works from the Harvest software, subject 
 *    to the following constraints:
 *  
 *      - You must include the above copyright notice and these 
 *        accompanying paragraphs in all forms of derivative works, 
 *        and any documentation and other materials related to such 
 *        distribution and use acknowledge that the software was 
 *        developed at the above institutions.
 *  
 *      - You must notify IRTF-RD regarding your distribution of 
 *        the derivative work.
 *  
 *      - You must clearly notify users that your are distributing 
 *        a modified version and not the original Harvest software.
 *  
 *      - Any derivative product is also subject to these copyright 
 *        and use restrictions.
 *  
 *    Note that the Harvest software is NOT in the public domain.  We
 *    retain copyright, as specified above.
 *  
 *  HISTORY OF FREE SOFTWARE STATUS
 *  
 *    Originally we required sites to license the software in cases
 *    where they were going to build commercial products/services
 *    around Harvest.  In June 1995 we changed this policy.  We now
 *    allow people to use the core Harvest software (the code found in
 *    the Harvest ``src/'' directory) for free.  We made this change
 *    in the interest of encouraging the widest possible deployment of
 *    the technology.  The Harvest software is really a reference
 *    implementation of a set of protocols and formats, some of which
 *    we intend to standardize.  We encourage commercial
 *    re-implementations of code complying to this set of standards.  
 *  
 */
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <memory.h>

/* External Functions from util.c */
extern void getword(), unescape_url(), plustospace(), nltospace();
extern char x2c();

/* Local functions */
static void sigdie();
static void cgi_fatal();
static void do_indent();
static int htmlize_brk_obj();
static int htmlize_opaque();
static int htmlize_url();
static void htmlize_desc();
static int broker_comm();
static int do_read();
static char *xstrerror();
static char *get_time();

/* Local variables */
static int read_timeout = 0;

#ifndef BUFSIZ
#define BUFSIZ	8192
#endif

#if BUFSIZ < 8192
#undef BUFSIZ
#define BUFSIZ	8192		/* Need large buffers */
#endif

#define XFER_TIMEOUT 	300	/* 5 minutes */
#define ENTSIZ 		32
#define HARVEST_URL	"http://harvest.cs.colorado.edu/"

typedef struct {
	char name[BUFSIZ];
	char val[BUFSIZ];
} entry;

static entry entries[ENTSIZ];
static char querystr[BUFSIZ];	/* Holds the users Query */
static char userquery[BUFSIZ];	/* Holds the form specified query */
static char userclass[BUFSIZ];	/* Holds the form specified class */
static char commandstr[BUFSIZ];	/* Holds the command to the Broker */
static char errmsg[BUFSIZ];	/* Buffer for error messages */
static char tfile[L_tmpnam + 1];	/* tmpfile */

static int maxresultflag = 0;
static int descflag = 0;
static int opaqueflag = 0;
static int verbose = 0;
static int nwierdmatch = 0;	/* Number of other matches */
static int nobjects = 0;	/* Number of phones in result set */
static int ncategories = 0;	/* Number of categories in result set */
static int nreturned = 0;	/* Number of matched lines returned */

/* 
 *  All of the fluffy error messages to users go here 
 */

#define NO_REPLICA_MSG	\
"</pre><p>\n\
Sorry, there are no known replicas for this broker.\n\
Please, change the \"Broker Host\" value on the query page.\n"

#define BROKER_DOWN_MSG \
"</pre><p>\n\
Sorry, the Broker at %s on port %d is unavailable.  Try again later.\n"

#define BROKER_LOAD_MSG \
"</pre><p>\n\
Sorry, the Broker at %s on port %d is currently too heavily loaded\n\
to process your request.  Try again later or try your query at one\n\
of the replica Brokers by selecting a different \"Broker Host\" on\n\
the query page.\n</pre>"

#define PARSE_ERROR_MSG \
"</pre><p>\n\
Sorry, your query:<p><hr><pre>\t%s\n</pre><p><hr>\n\
does not have the proper syntax.  Or the BrokerQuery program is out-of-date.\n\
<p>\n\
Common syntax mistakes include:\n\
<ul>\n\
<li> Listing two words without an AND or OR joining them.\n\
For example, <p><pre>\tresource discovery</pre><p>should be \n\
<p><pre>\tresource AND discovery</pre>\n\
<li> Missing punctuation.  Structured queries need a colon and optional\n\
parentheses.  For example,\n\
<p><pre>\tType:PostScript</pre>\n\
<p>or,\n\
<p><pre>\t( Type : PostScript )</pre>\n\
</ul>\n"

#define MISCONFIG_MSG \
"</pre><p>\n\
Sorry, but that query page or link generated an incorrectly configured query\n\
to the broker.  Please report this error to your systems administrator.\n\
<p><b>If you are using Lynx, try using Mosaic instead.  We've found a bug in\n\
Lynx that is breaking our Broker interface.  We're working on the fix.\n\
Thanks.</b>\n"


int main(argc, argv)
int argc;
char *argv[];
{
	int x, m = 0, errors = 0, wordflag = 0, caseflag = 0, olderr = 0,
	 goterrflag = 0;
	char *s;
	char *cl = getenv("QUERY_STRING");
	char *gl = getenv("REQUEST_METHOD");
	int sport = 0;
	char hhost[BUFSIZ];

	tfile[0] = '\0';
	(void) signal(SIGHUP, sigdie);
	(void) signal(SIGINT, sigdie);
	(void) signal(SIGQUIT, sigdie);
	(void) signal(SIGTERM, sigdie);
	(void) signal(SIGABRT, sigdie);

	/* Print a simple MIME header here.  MIME hdrs end with \n\n */
	printf("Content-type: text/html\n\n");

	if (!gl || (memcmp(gl, "GET", 3))) {
		cgi_fatal("This script must be used with a METHOD of GET.\n");
	}
	if (cl == NULL) {
		cgi_fatal("No query information to decode.\n");
	}
#ifdef DEBUG
	fprintf(stderr, "[%s] BrokerQuery: Received URL: %s\n", get_time(), cl);
#endif

	/* Parse the URL string for the parameters */
	for (x = 0; (cl[0] != '\0') && (x < ENTSIZ); x++) {
		m = x;
		getword(entries[x].val, cl, '&');
		plustospace(entries[x].val);
		unescape_url(entries[x].val);
		nltospace(entries[x].val);	/* strip \n from query */
		getword(entries[x].name, entries[x].val, '=');
	}

	memset(querystr, '\0', BUFSIZ);
	memset(userquery, '\0', BUFSIZ);
	memset(userclass, '\0', BUFSIZ);
	memset(commandstr, '\0', BUFSIZ);
	for (x = 0; x <= m; x++) {
		if (memcmp(entries[x].name, "query", 5) == 0) {
			strncpy(userquery, entries[x].val, BUFSIZ);
		} else if (memcmp(entries[x].name, "class", 5) == 0) {
			strncpy(userclass, entries[x].val, BUFSIZ);
		} else if (memcmp(entries[x].name, "caseflag", 8) == 0) {
			caseflag = 1;
		} else if (memcmp(entries[x].name, "wordflag", 8) == 0) {
			wordflag = 1;
		} else if (memcmp(entries[x].name, "opaqueflag", 10) == 0) {
			opaqueflag = 1;
		} else if (memcmp(entries[x].name, "descflag", 8) == 0) {
			descflag = 1;
		} else if (memcmp(entries[x].name, "maxresultflag", 13) == 0) {
			maxresultflag = atoi(entries[x].val);
			if (maxresultflag < 0)
				maxresultflag = 0;
		} else if (memcmp(entries[x].name, "errorflag", 9) == 0) {
			goterrflag = 1;
			if (entries[x].val[0] == 'B') {
				errors = -1;
			} else {
				errors = atoi(entries[x].val);
			}
			/* 
			 *  We can encode host:port in the host field; used to
			 *  hide the port number for the replica servers from users
			 */
		} else if (memcmp(entries[x].name, "host", 4) == 0) {
			strncpy(hhost, entries[x].val, BUFSIZ);
			if ((s = strchr(hhost, ':')) != NULL) {
				*s = '\0';
				sport = atoi(++s);
			}
		} else if (memcmp(entries[x].name, "port", 4) == 0) {
			sport = atoi(entries[x].val);
		} else if (memcmp(entries[x].name, "verbose", 7) == 0) {
			verbose = 1;
		}
	}

	/* Sanity checks */
	if (hhost != NULL && !strcmp(hhost, "No Replicas")) {
		sprintf(errmsg, NO_REPLICA_MSG);
		cgi_fatal(errmsg);
	}
	if ((sport == 0) || (hhost == NULL)) {
		sprintf(errmsg, MISCONFIG_MSG);
		cgi_fatal(errmsg);
	}
	/* Append the userclass to the query string */
	if (userclass[0]) {
		sprintf(querystr, "%s AND %s", userclass, userquery);
	} else {
		strcpy(querystr, userquery);
	}

	/* Build the Broker query */
	strcpy(commandstr, "#USER ");
	opaqueflag = descflag = 1;
	if (opaqueflag) {
		strcat(commandstr, "#opaque ");
	}
	if (descflag) {
		strcat(commandstr, "#desc ");
	}
	if (goterrflag) {
		char tmpbuf[BUFSIZ];
		sprintf(tmpbuf, "#index error %d ", errors);
		strcat(commandstr, tmpbuf);
	}
	if (maxresultflag) {
		char tmpbuf[BUFSIZ];
		sprintf(tmpbuf, "#index maxresult %d ", maxresultflag);
		strcat(commandstr, tmpbuf);
	}
	if (caseflag) {
		strcat(commandstr, "#index case insenstive ");
	}
	if (wordflag) {
		strcat(commandstr, "#index matchword ");
	}
	strcat(commandstr, "#END ");
	strcat(commandstr, querystr);


	/* Print the HTML header for the query result set */
	printf("<title>Broker Query Results for: %s</title>\n", querystr);
	printf("<h1>Broker Query Results for:</h1>\n");
	printf("<p><pre>\n");

	printf("\t%s\n", querystr);
	if (olderr == 1) {
		printf("<P>\n<B>Hey! You're using an old version of Mosaic.</B><P>");
	}
	printf("</pre><hr><p><OL>\n");

	broker_comm(hhost, sport, commandstr);
	fflush(stdout);
	exit(0);
}

/*
 *  broker_comm() - Communicate with the Broker at host serverstr on port
 *  port.  Issue the query Qstr and return the results to the user in HTML.
 */
static int broker_comm(serverstr, port, Qstr)
char *serverstr;
int port;
char *Qstr;
{
	FILE *sfp;
	int sock, n, done = 0, sv;
	char hpptr[BUFSIZ], ret[BUFSIZ], *retptr = NULL;
	struct sockaddr_in me;
	struct hostent *hp;

	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		sprintf(errmsg, "Cannot open a socket: %s\n", xstrerror(errno));
		cgi_fatal(errmsg);
	}
	me.sin_family = AF_INET;
	if ((hp = gethostbyname(serverstr)) == NULL) {
		sprintf(errmsg, "%s: Unknown host.\n", serverstr);
		cgi_fatal(errmsg);
	}
	memcpy((char *) &me.sin_addr, (char *) hp->h_addr, hp->h_length);
	me.sin_port = htons(port);

	if (connect(sock, (struct sockaddr *) &me, sizeof me) < 0) {
		if (errno == ECONNREFUSED) {
			printf(BROKER_DOWN_MSG, serverstr, port);
		} else if (errno == ETIMEDOUT) {
			printf(BROKER_LOAD_MSG, serverstr, port);
		}
		sprintf(errmsg, "Cannot connect to %s on port %d: %s\n",
			serverstr, port, xstrerror(errno));
		cgi_fatal(errmsg);
	}
#ifdef DEBUG
	fprintf(stderr, "[%s] BrokerQuery: Making connection %s:%d\n",
		get_time(), serverstr, port);
#endif
	tmpnam(tfile);
	if ((sfp = fopen(tfile, "w+")) == NULL) {
		sprintf(errmsg, "%s: %s\n", tfile, xstrerror(errno));
		cgi_fatal(errmsg);
	}
	if (write(sock, Qstr, strlen(Qstr)) < 0) {
		sprintf(errmsg, "Cannot write data to %s: %s\n",
			hp->h_name, xstrerror(errno));
		cgi_fatal(errmsg);
	}
	/* Talk to broker, and save the results in a tfile */
	/* Use select so that if the broker goes down, we'll timeout & exit */
	while (1) {
		read_timeout = 0;
		n = do_read(sock, ret, BUFSIZ);
		if (n == 0)
			break;	/* nothing to do */
		if (n < 0) {
			if (read_timeout == 1) {
				printf(BROKER_LOAD_MSG, serverstr, port);
				sprintf(errmsg,
					"Cannot read data from %s: Timeout after %d seconds.\n",
					hp->h_name, XFER_TIMEOUT);
			} else {
				sprintf(errmsg, "Cannot read data from %s: %s\n",
					hp->h_name, xstrerror(errno));
			}
			cgi_fatal(errmsg);
		}
		fwrite(ret, 1, n, sfp);
	}
	close(sock);
	rewind(sfp);

	/* Now process the broker's results */
	hpptr[0] = '\0';
	done = 0;
	while (!done) {
		retptr = (char *) malloc(BUFSIZ);
		retptr[0] = '\0';
		if (fgets(retptr, BUFSIZ, sfp) == NULL) {
			done = 1;
			break;
		}
		sv = atoi(retptr);
		switch (sv) {
		case 101:	/* Message to the user */
		case 102:	/* ?????, opaque data size? */
			/*      fprintf(stdout, "<P>\n<B>%s</B><P>", retptr + 6); */
			break;
		case 120:	/* The URL of the match */
			htmlize_url(retptr + 6);
			break;
		case 122:	/* Opaque data */
			htmlize_opaque(retptr + 6);
			break;
		case 124:	/* The description line of the match */
			htmlize_desc(atoi(retptr + 6), sfp);
			break;
		case 125:	/* URL to the content summary */
			htmlize_brk_obj(retptr + 6);
			break;
		case 126:	/* URL to the Broker Home Page */
			strncpy(hpptr, retptr + 6, BUFSIZ);
			break;
		case 130:	/* Object Ends, start a new one */
			/* printf("<p>\n"); */
			break;
		case 103:	/* End of the Broker Results */
		case 111:	/* Error Message that ends Broker Results */
			if (memcmp(retptr + 6, "PARSE ERROR", 10) == 0)
				printf(PARSE_ERROR_MSG, querystr);
			else if (memcmp(retptr + 6, "Broker is too heavily loaded", 27) == 0)
				printf(BROKER_LOAD_MSG);
			else
				printf("<P>\n<B>%s</B><P>", retptr + 6);
			done = 1;
			break;
		default:
			break;
		}
		free(retptr);
	}
	/* Clean up */
	fclose(sfp);
	(void) unlink(tfile);

	/* Print the trailing portion of the result set */
	printf("</OL><HR><p>\n<B>Found ");
	if (nobjects > 0) {
		printf("%d phone numbers, ", nobjects);
	}
	if (nwierdmatch > 0) {
		printf("%d other matched lines, ", nwierdmatch);
	}
	printf("%d categories.</b>\n", ncategories);
	if (nreturned == maxresultflag && maxresultflag > 0) {
		printf("<p><i>NOTE: the results were truncated at %d phone numbers", maxresultflag);
		if (nwierdmatch > 0)
			printf(" plus other matched lines");
		printf(".\n</i>");
	}
	if (hpptr[0] != '\0') {
		printf("<p><b>Return to <a href=\"%s\">Broker Home Page</a></b>\n",
		       hpptr);
	}
	printf("<p><hr><p><address>\n");
	printf("This Broker was built using the ");
	printf("<a href=\"%s\">Harvest</a> system.\n", HARVEST_URL);
	printf("</address>\n");
	return 0;
}

static void do_indent()
{
	printf("     ");
}

static char saveurl[BUFSIZ];

static int htmlize_url(url)
char *url;
{
	char *fn, *host, *path;
	char tmp[BUFSIZ], nbuf[BUFSIZ];
	int n;

	++ncategories;
	strcpy(saveurl, url);
	saveurl[strlen(saveurl) - 1] = '\0';
	return 1;
	sprintf(nbuf, "%d.", ++nobjects);
	n = strlen(url);
	url[n - 1] = '\0';
	if (verbose == 0) {
		printf("<b>%-5s</b><a href=\"%s\">%s</a>\n", nbuf, url, url);
		return (1);
	}
	fn = strrchr(url, '/');
	fn++;
	if (strlen(fn) == 0)
		printf("<b>%-5s</b><I>filename:</I> <a href=\"%s\">%s</a>\n",
		       nbuf, url, url);
	else
		printf("<b>%-5s</b><I>filename:</I> <a href=\"%s\">%s</a>\n",
		       nbuf, url, fn);
	host = strstr(url, "//");
	host += 2;
	path = strchr(host, '/');
	n = path - host;
	memset(tmp, '\0', BUFSIZ);
	memcpy(tmp, host, n);
	do_indent();
	printf("<I>host:</I> %s\n", tmp);
	n = fn - path;
	memset(tmp, '\0', BUFSIZ);
	memcpy(tmp, path, n);
	do_indent();
	printf("<I>path:</I> %s\n", tmp);

	return (1);
}

static int htmlize_opaque(data)
char *data;
{
	int n;
	char *play;

	++nreturned;
	if ((play = strstr(data, "800")) == NULL) {
		if ((play = strchr(data, '#')) == NULL)
			return 1;
		printf("<i>Other matched line:</i>%s<BR>", ++play);
		++nwierdmatch;
		return 1;
	}
	++nobjects;
	printf("%s<BR>", play);
	return 1;
}

static int htmlize_brk_obj(obj_url)
char *obj_url;
{
	char *urlstart, *s;

	/* We need to save the start of the URL for the object */
	if ((urlstart = malloc(strlen(obj_url) + 1)) == NULL)
		return (0);
	strcpy(urlstart, obj_url);
	s = strchr(urlstart, '/');
	if (s == NULL)
		return (0);
	s = strchr(s + 1, '/');
	if (s == NULL)
		return (0);
	s = strchr(s + 1, '/');
	if (s == NULL)
		return (0);
	*s = '\0';

	/* We only want the pathname, so chop it off */
	s = strchr(obj_url, '/');
	if (s == NULL)
		return (0);
	s = strchr(s + 1, '/');
	if (s == NULL)
		return (0);
	s = strchr(s + 1, '/');
	if (s == NULL)
		return (0);
	s++;

	printf("<a href=\"%s/Harvest/cgi-bin/DisplayObject.cgi?object=%s\">[content summary]</a><BR>\n", urlstart, s);
	free(urlstart);
	return (1);
}

static void htmlize_desc(size, fp)
int size;
FILE *fp;
{
	char *descbuf;

	if (size < 1)
		return;
	if ((descbuf = (char *) malloc(size + 1)) == NULL) {
		cgi_fatal("Internal Error: malloc failed.");
	}
	memset(descbuf, '\0', size + 1);
	if (fread(descbuf, 1, size, fp) == 0) {
		sprintf(errmsg, "Cannot read description: %s\n",
			xstrerror(errno));
		cgi_fatal(errmsg);
	}
	do_indent();
	printf("<P><LI><B>Category: <A HREF=\"%s\">%s</A></B>\n",
	       saveurl, descbuf);
	return;
}

/*
 * do_read() - read that performs timeout.
 */
static int do_read(s, buf, sz)
int s;
char *buf;
int sz;
{
	fd_set readDetect;
	struct timeval timeout;
	int err, readBytes = 0;
	extern int select();

	memset(&timeout, '\0', sizeof(struct timeval));

	/* read until timeout or bytes are read */
	while (1) {
		FD_ZERO(&readDetect);
		FD_SET(s, &readDetect);
		timeout.tv_sec = XFER_TIMEOUT;
		timeout.tv_usec = 0;

		/* wait for data for seconds */
		err = select(s + 1, &readDetect, NULL, NULL, &timeout);
		if (err < 0) {
			if (errno == EINTR)
				continue;
			if (errno == ECONNRESET)
				return(0);
			return (-1);
		}
		/* timeout on the read */
		if (err == 0) {
			read_timeout = 1;	/* so other procs know we timed out */
			return (-1);
		}
		if (FD_ISSET(s, &readDetect)) {
			if ((readBytes = read(s, buf, sz)) < 0)
				if (errno == ECONNRESET)
					return(0);
			break;
		}
	}
	return (readBytes);
}


/*
 *  cgi_fatal() - Print error message to user via stdout; and nicely log
 *  the error to stderr where httpd will save it in error_log.
 */
static void cgi_fatal(s)
char *s;
{
	fprintf(stdout, "%s\n", s);
	fprintf(stderr, "[%s] BrokerQuery: %s", get_time(), s);
	fflush(stdout);
	fflush(stderr);
	if (tfile[0] != '\0')
		(void) unlink(tfile);
	exit(1);
}

/*  sigdie() - Exit gracefully */
static void sigdie(sig)
int sig;
{
	sprintf(errmsg, "Killed by signal %d...", sig);
	cgi_fatal(errmsg);
}


/*
 *   xstrerror() - same as strerror(3)
 */
static char *xstrerror(n)
int n;
{
	extern int sys_nerr;
	extern char *sys_errlist[];

	if (n < 0 || n >= sys_nerr)
		return (NULL);
	return (sys_errlist[n]);
}

/* get_time from NCSA httpd */
static char *get_time()
{
	time_t t;
	char *time_string;

	t = time(NULL);
	time_string = ctime(&t);
	time_string[strlen(time_string) - 1] = '\0';
	return (time_string);
}
