/* $Id: io-handler.c,v 1.1 1995/04/19 00:24:10 dante Exp $ */
#include <stdio.h>
#include <unistd.h>
#include <xmalloc.h>
#include <sys/types.h>
#if !defined(HAVE_FCNTL_H)
#include <fcntl.h>
#endif
#include <sys/time.h>
#include <queue.h>
#include <io-handler.h>

extern int debug;

IODescriptor **io_list = NULL;

int io_list_count = 0;
int io_list_max = 0;

static void
IO_list_add (IODescriptor *descriptor);

IORequest *
IORequest_new (IOFunction *func, IOCleanup *cleanup, void *state)
{
  IORequest *request;

  request = xmalloc (sizeof (IORequest));
  
  request->func = func;
  request->cleanup = cleanup;
  request->state = state;
  return (request);
}

IORequest_free (IORequest *request)
{
  if (request->cleanup)
    request->cleanup (request->state);
  xfree (request);
}

/* Setup an IO descriptor for a file descriptor.  This function simply 
 * creates an place to stuff IO functions.
 * Use IODescriptor_add_reader() and  IODescriptor_add_writer() to 
 * actually initiate IO operations.
 *
 * Malloc an io descriptor and fill it up with good stuff.
 */

IODescriptor *
IODescriptor_new (int fd)
{
  IODescriptor *descriptor;

  descriptor = xmalloc (sizeof (IODescriptor));

  descriptor->fd = fd;
  descriptor->please_close = 0;
  descriptor->read_requests = queue_new ();
  descriptor->read = NULL;


  descriptor->write_requests = queue_new ();
  descriptor->write = NULL;

  return (descriptor);
}

void
IODescriptor_free (IODescriptor *descriptor)
{
  QUEUE_FREE (descriptor->read_requests, IORequest_free);
  QUEUE_FREE (descriptor->write_requests, IORequest_free);

  if (descriptor->read)
    IORequest_free (descriptor->read);

  if (descriptor->write)
    IORequest_free (descriptor->write);

  xfree (descriptor);

}
/* Add a read function to the io descriptor */
void
IODescriptor_add_reader (IODescriptor *descriptor, 
			 IOFunction   *reader,
			 IOCleanup    *cleanup,
			 void         *state)
{
  IORequest *request;

  request = IORequest_new (reader, cleanup, state);
  
  if (queue_empty (descriptor->read_requests) && 
      descriptor->read == NULL)
    descriptor-> read = request;
  else 
    queue_insert (descriptor->read_requests, request);
  
  if (descriptor->read == NULL)
    descriptor->read = queue_remove (descriptor->read_requests);
}

/* Add a write function to the io descriptor */
void
IODescriptor_add_writer (IODescriptor *descriptor, 
			 IOFunction   *writer, 
			 IOCleanup    *cleanup,
			 void         *state)
{
  IORequest *request;

  request = IORequest_new (writer, cleanup, state);
  
  if (queue_empty (descriptor->write_requests) && 
      descriptor->write == NULL)
    descriptor-> write = request;
  else
    queue_insert (descriptor->write_requests, request);

  if (descriptor->write == NULL)
    descriptor->write = queue_remove (descriptor->write_requests);
}

/*
 * Execute the current read IORequest associated with an IODescriptor.
 * if the read function returns non-zero, then we have read an object and
 * can be freed and the next function instated.
 * If it returns a -1 then and EOF has ocurred and we can terminate the 
 * IO for the file descriptor.
 * 
 */
int
IODescriptor_read (IODescriptor *io) 
{
  int n;

  if (io->fd == -1)
    return (-1);

  n  = io->read->func (io->fd, io->read->state);
  if (n > 0)
    {
      IORequest_free (io->read);
      io->read = queue_remove (io->read_requests);
    }
  else if (n == -1)
    IODescriptor_terminate (io);
  return (n);
}

int
IODescriptor_write (IODescriptor *io) 
{
  int n;

  if (io->fd == -1)
    return (-1);

  n  = io->write->func (io->fd, io->write->state);
  if (n > 0)
    {
      IORequest_free (io->write);
      io->write = queue_remove (io->write_requests);
    } 
  else if (n == -1)
    IODescriptor_terminate (io);

  return (n);
}

void
IODescriptor_terminate (IODescriptor *io)
{
  if (io->read)
    IORequest_free (io->read);
  io->read = NULL;

  if (io->write)
    IORequest_free (io->write);
  io->write = NULL;

  close (io->fd);
  io->fd = -1;
}

/*
 *  Maintain a list of ongoing io operations 
 */

static void
IO_list_add (IODescriptor *descriptor)
{
  if (io_list_count >= io_list_max)
    {
      io_list_max += 5;
      io_list =
	xrealloc (io_list, (io_list_max + 1) * sizeof (IODescriptor *));
    }
  io_list[io_list_count++] = descriptor;
  io_list[io_list_count] = NULL;
}

/* compact the io list, deleting those whose io->fd field is -1 */
static void 
IO_list_compact ()
{
  int i;
  int skip = 0;

  if (io_list_count == 0)
    return;

  for (i = 0; io_list[i] != NULL; i++)
    {
      if (io_list[i] && io_list[i]->fd == -1)
	{
	  io_list_count--;
	  skip++;
	  IODescriptor_free (io_list[i]);
	}
      else 
	{
	  if (skip > 0) 
	    io_list[i - skip] = io_list[i];
	}
    }
  io_list[i - skip] = NULL;
}

int
IO_process (struct timeval *timeout)
{
  int i;
  int n;
  fd_set readset;
  fd_set writeset;
  fd_set exceptset;

  /* Get a list of connections we need to worry about */
  
  FD_ZERO (&readset);
  FD_ZERO (&writeset);
  FD_ZERO (&exceptset);

  for (i = 0; i < io_list_count; i++)
    {
      /* Wait for write functions to terminate */
      if (io_list[i]->please_close != 0 && io_list[i]->write == NULL)
	IODescriptor_terminate (io_list[i]);
      else
	{
	  if (io_list[i]->fd >= 0)
	    {
	      FD_SET (io_list[i]->fd, &exceptset);
	      if (io_list[i]->read)
		FD_SET (io_list[i]->fd, &readset);
	      if (io_list[i]->write)
		FD_SET (io_list[i]->fd, &writeset);
	    }
	}
    }

#ifdef INSIDE_CONTROL
  {
    extern int sw_data_connect;
    extern int data_socket;

    if (sw_data_connect) FD_CLR (data_socket, &readset);
  }
#endif /* INSIDE_CONTROL */

  n = select (FD_SETSIZE, &readset, &writeset, &exceptset, timeout);
  
  /* Something has happened */
  if (n > 0)
    {
      for (i = 0; i < io_list_count; i++)
	{

	  if (io_list[i]->fd != -1 &&
	      FD_ISSET (io_list[i]->fd, &exceptset))
	    {
	      IODescriptor_terminate (io_list[i]);
	      continue;
	    }

	  if (io_list[i]->fd != -1 &&
	      FD_ISSET (io_list[i]->fd, &readset))
	    IODescriptor_read (io_list[i]);
	  
	  if (io_list[i]->fd != -1 &&
	      FD_ISSET (io_list[i]->fd, &writeset))
	    IODescriptor_write (io_list[i]);
	}
    }
  
  /* Now we have to cleanup a io descriptor's
   * If the file descriptor is set to -1, we have to do the clean up.
   */

  IO_list_compact ();

  /* Start the export queue, if nothing is queued, nothing will happen.
   * This should be done in a more generic way.
   */
  export_start ();

  return (n);
}

IODescriptor *
IODescriptor_by_fd (int fd)
{
  int i;

  for (i = 0; i < io_list_count; i++)
    if (io_list[i]->fd == fd)
      return (io_list[i]);
  return (NULL);
}

/* 
 * Fire up an io handler.  If they read and write functions are provided,
 * they are installed as the initial handlers.
 */

IODescriptor *
IO_start (int fd, 
	  IOFunction *read,  IOCleanup *read_cleanup,  void *read_state,
	  IOFunction *write, IOCleanup *write_cleanup, void *write_state,
	  IOCleanup *cleanup)
{
  IODescriptor *descriptor;
  descriptor = IODescriptor_new (fd);

  descriptor->cleanup = cleanup;

  if (read != NULL)
    IODescriptor_add_reader (descriptor, read, read_cleanup, read_state);

  if (write != NULL)
    IODescriptor_add_writer (descriptor, write, write_cleanup, write_state);

  IO_list_add (descriptor);
  return (descriptor);
}

/* 
 * Output data on file descriptor.
 * It actually gets stuffed in a queue for later transmission.
 */
int
IO_output (int fd,  IOFunction *writer, IOCleanup *cleanup, void *state)
{
  IODescriptor *descriptor;

  /* Lookup the iodescriptor for this fd */
  descriptor = IODescriptor_by_fd (fd);
  if (descriptor == NULL)
    return (-1);
  
  IODescriptor_add_writer (descriptor, writer, cleanup, state);
}

/* User close function */

void 
IO_close (int fd)
{
  IODescriptor *descriptor;

  descriptor = IODescriptor_by_fd (fd);
  if (descriptor != NULL)
    descriptor->please_close = 1;
}

void *
IO_get_reader (int fd)
{
  IODescriptor *descriptor;

  descriptor = IODescriptor_by_fd (fd);
  if (descriptor != NULL)
    if (descriptor->read)
      return (descriptor->read->func);
}

void
IO_set_reader (int fd, IOFunction *reader)
{
  IODescriptor *descriptor;

  descriptor = IODescriptor_by_fd (fd);
  if (descriptor != NULL)
    if (descriptor->read)
      descriptor->read->func = reader;
}

void *
IO_get_writer (int fd)
{
  IODescriptor *descriptor;

  descriptor = IODescriptor_by_fd (fd);
  if (descriptor != NULL)
    if (descriptor->read)
      return (descriptor->read->func);
}

void
IO_set_writer (int fd, IOFunction *writer)
{
  IODescriptor *descriptor;

  descriptor = IODescriptor_by_fd (fd);
  if (descriptor != NULL)
    if (descriptor->write)
      descriptor->write->func = writer;
}

/* return the io state */
void *
IO_read_state (int fd)
{
  IODescriptor *descriptor;

  descriptor = IODescriptor_by_fd (fd);
  if (descriptor != NULL)
    if (descriptor->read)
      return (descriptor->read->state);
  return (NULL);
}


#ifdef NEVER
IO_sleep_read (int fd, Queue *queue)
{
  IODescriptor *descriptor;
  IOFunction *reader;

  descriptor = IODescriptor_by_fd (fd);

  if (descriptor != NULL)
    if (descriptor->read)
      {
	descriptor->sleeping_on_read = 1;
	descriptor->wait = queue;
	queue_insert (queue, descriptor);
	return (reader);
      }
  return (NULL);
}

IO_wakeup_read (Queue *queue)
{
  IODescriptor *descriptor;

  descriptor = queue_remove (queue);
  decriptor->wait = NULL;
  descriptor->sleeping_on_read = 0;
}

#endif
