/* $Id: fd-config.c,v 1.3 1995/07/25 20:05:20 dante Exp $ */
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <command.h>
#include <list.h>
#include <xmalloc.h>
#include <string.h>
#include <properties.h>
#include <dynamic_string.h>
#include <getopt.h>
#define NOTSET -1

AnonymousObject *defaults_list = NULL;
AnonymousObject *replica_list = NULL;
AnonymousObject *site_list = NULL;
AnonymousObject *group_list = NULL;
AnonymousObject *local_list = NULL;
AnonymousObject *mappings;

char *(*symbol_filter) ();

char *local_host = NULL;

char *
expand_filename (char *name);

char *
expand_string (char *string);

void
fd_error (char *fmt, ...);

typedef struct
{
  Property **properties;
  char *name;
} Site;

typedef struct
{
  Property **properties;
  char *name;
} Replica;

typedef struct
{
  Property **properties;
  AnonymousObject sites;
  char *name;
} Group;

typedef struct _context
{
  char *name;
  void *data;
  struct _context *previous;
} Context;

/* Some global variables */
int fd_lineno = 0;
int fd_error_count = 0;
char *fd_filename = NULL;
FILE *fd_output = stdout;
Context *current_context = NULL;

/* Some methods for data types */
Site *
site_new (char *name)
{
  Site *site;

  site = xmalloc (sizeof (Site));
  site->properties = NULL;
  site->name = strdup (name);
  return (site);
}


/* Some methods for data types */
Replica *
replica_new (char *name)
{
  Replica *replica;

  replica = xmalloc (sizeof (Replica));
  replica->properties = NULL;
  replica->name = strdup (name);
  return (replica);
}

Group *
group_new (char *name)
{
  Group *group;

  group = xmalloc (sizeof (Group));
  group->properties = NULL;
  group->name = strdup (name);
  group->sites.properties = NULL;
  return (group);
}


Context *
context_new (char *tag, void *data, void *previous)
{
  Context *context;

  context = xmalloc (sizeof (Context));
  context->name = strdup (tag);
  context->data = data;
  context->previous = previous;
  return (context);
}

void
context_free (Context *context)
{
  if (context == NULL)
    return;
  xfree (context->name);
  xfree (context);
}

void
context_push (char *tag, void *value)
{
  current_context = context_new (tag, value, current_context);
}

void *
context_get_value (char *tag)
{
  Context *context;

  /* Make sure the named context exists */
  for (context = current_context; context != NULL; context = context->previous)
    {
      if (strcmp (context->name, tag) == 0)
	return (context->data);
    }
  return (NULL);
}

/* Pop context until we have popd the one that matches the tag.  If the
 * tag is NULL, just do a simple pop.
 * Return current context, or null if no tag matches.
 */
Context *
context_pop_til (char *tag);

Context *
context_pop (char *tag)
{
  Context *context;

  if (tag != NULL)
    context_pop_til (tag);
    
  if (current_context !=  NULL)
    {
      context = current_context;
      current_context = context->previous;
      context_free (context);
    }
  return (current_context);
}

Context *
context_pop_til (char *tag)
{
  Context *target;
  Context *context;

  if (current_context == NULL)
    return (NULL);

  if (tag == NULL)
    {
      while (current_context != NULL)
	context_pop (NULL);
      return (NULL);
    }

  while (current_context != NULL && strcmp (current_context->name, tag) != 0)
    context_pop (NULL);

  return (current_context);
}

/* Add to a property list containg mappings */
void
mapping_add (AnonymousObject *mappings, char *tag, char *alias)
{
  set_property (mappings, strdup (tag), strdup (alias), Prop_String, 0);
}

char *
convert_to_floodd (char *tag)
{
  char *name;

  name = get_property_value (mappings, tag);
  if (name != NULL)
    return (name);
  else
    return (tag);
}

int 
is_integer (char *string)
{
  if (string == NULL || *string == '\0')
    return (0);

  if (*string == '-' || *string == '+')
    string++;

  while (*string && isdigit (*string))
    string++;

  if (*string != '\0')
    return (0);

  return (1);
}

/* This command assumes there is only one property with the
 * given name in the replica/site/local lists.
 */
Property *
get_parameter (char *tag, char *host)
{
  Replica *replica;
  Site *site;
  AnonymousObject *local;
  Property  *property;

  if (host == NULL)
    goto just_default;

  replica = get_property_value (replica_list, host);
  if (replica == NULL)
    {
      fd_error ("No replica `%s'\n", host);
      exit (1);
    }

  /* Check the replica property list */
  property = get_property ((AnonymousObject *) replica, tag);
  if (property != NULL)
    return (property);

  /* Check the site property list */
  site = get_property_value ((AnonymousObject *) replica, "Site");

  if (site != NULL)
    {
      property =
	get_property ((AnonymousObject *) site, tag);

      if (property != NULL)
	  return (property);
    }
  /* Check the local property list */
  local = get_property_value ((AnonymousObject *) replica, "Local");
  if (local != NULL)
    {
      property = 
	get_property (local, tag);
      
      if (property != NULL)
	return (property);
    }

 just_default:

  property = get_property (defaults_list, tag);
  if (property != NULL)
    return (property);

  return (NULL);
}

void
set_default (AnonymousObject *object, char *tag, Proptype type)
{
  Property *deflt;
  
  deflt = get_property (defaults_list, tag);
 
  if (deflt == NULL)
    return;

  if (deflt->type != type)
    {
      fd_error ("Property type mismatch\n");
      return;
    }
  
  switch (type)
    {
    case Prop_Integer:
      set_property (object, tag, deflt->value, 
		    Prop_Integer, 0);
      break;

    case Prop_String:
      set_property (object, tag, strdup (deflt->value), 
		    Prop_String, 0);
      break;
    case Prop_Filename:
      set_property (object, tag, strdup (deflt->value), 
		    Prop_Filename, 0);
      break;
    default:
      fd_error ("set_default: Unknown property type `%d'\n", type);
      break;
    }
}

void
set_host (char **args)
{
  if (args[1] == NULL)
    {
      fd_error ("%s requires an argument\n", args[0]);
      return;
    }

  if (local_host == NULL)
    local_host = strdup (args[1]);
}

Replica *
get_replica (char **args)
{
  Replica *replica;

  if (strcmp  (current_context->name, "Replica") == 0)
    return (current_context->data);

  return (NULL);
}

/* Prepened the value of the "RootDirectory" parameter if the 
 * value of `name' does not begin with a '/'
 */
char *
expand_filename (char *name)
{
  Property  *property;
  Context *context;
  char *filename;
  char *prefix;

  if (*name != '/' && current_context != NULL)
    {
      for (context = current_context; 
	   context->previous != NULL; 
	   context = current_context->previous);

      /* If the toplevel is a "Replica" context, then chew on the replica
       * othewise, just gok for defaults, as at this time I don't see how 
       * a Group would have a directory context.
       */
      if (strcmp (context->name, "Replica") == 0)
	property =
	  get_parameter ("RootDirectory", ((Replica *)context->data)->name);
      else
	property =
	  get_parameter ("RootDirectory", NULL);
	

      prefix = property->value;

      if (prefix != NULL)
	{
	  filename = xmalloc (strlen (prefix) + strlen (name) + 2);
	  strcpy (filename, prefix);
	  strcat (filename, "/");
	  strcat (filename, name);
	}
    }
  else
    filename = strdup (name);
  return (filename);
}


void
set_replica (char **args)
{
  Replica *replica;
  int n;

  if (args[1] == NULL)
    {
      fd_error ("%s requires an argument\n", args[0]);
      return;
    }

  /* Clean up any junk on the stack */
  
  context_pop_til (NULL);
  
  replica =  get_property_value (replica_list, args[1]);

  if (replica != NULL)
    {
      fd_error ("%s: %s already defined\n", args[0], args[1]);
      return;
    }

  replica = replica_new (args[1]);

  set_property (replica_list, strdup (args[1]), replica, Prop_List, 0);
  
  set_default ((AnonymousObject *) replica, "BinDirectory", Prop_Filename);
  set_default ((AnonymousObject *) replica, "LibDirectory", Prop_Filename);
  set_default ((AnonymousObject *) replica, "DataDirectory", Prop_Filename);

  context_push ("Replica", replica);
}

void
set_replica_integer (char **args)
{
  Replica *replica;

  if (args[1] == NULL)
    {
      fd_error ("Integer argument required\n");
      return;
    }
  /* If we have no context, then we must be setting defaults */
  if (current_context == NULL)
    replica = (Replica *) defaults_list;
  else
    {
      replica = get_replica (args);

      if (replica == NULL)
	{
	  fd_error ("Cannot set defaults inside context\n");
	  return;
	}
    }
  
  set_property (replica, args[0], atoi (args[1]), Prop_Integer, 0);
}

void
set_replica_string (char **args)
{
  Replica *replica;
  char *tag;

  if (args[1] == NULL)
    {
      fd_error ("String argument required\n");
      return;
    }

  /* If we have no context, then we must be setting defaults */
  if (current_context == NULL)
    replica = (Replica *) defaults_list;
  else
    {
      replica = get_replica (args);

      if (replica == NULL)
	return;
    }

  tag = get_property_value (mappings, args[0]);
  if (tag  == NULL)
    tag = args[0];

  set_property (replica, tag, expand_string (args[1]), Prop_String, 0);
} 


void
set_replica_filename (char **args)
{
  Replica *replica;
  char *tag;

  if (args[1] == NULL)
    {
      fd_error ("String argument required\n");
      return;
    }

  /* If we have no context, then we must be setting defaults */
  if (current_context == NULL)
    replica = (Replica *) defaults_list;
  else
    {
      replica = get_replica (args);

      if (replica == NULL)
	return;
    }

  tag = get_property_value (mappings, args[0]);
  if (tag  == NULL)
    tag = args[0];

  set_property (replica, tag, expand_string (args[1]), Prop_Filename, 0);
} 
void
set_site (char **args)
{
  Property *site;
  Replica *replica;
  AnonymousObject *obj;
  Context *context;
  int n;

#ifdef  NEVER
  if (args[1] == NULL)
    {
      fd_error ("%s requires an argument\n", args[0]);
      return;
    }
#endif
  /* Clean up any junk on the stack */
  context = context_pop_til ("Replica");
  if (context == NULL)
    {
      fd_error ("Not in Replica context\n");
      return;
    }

  /* Get the site name from the replica name */
  replica = context->data;
  site =  get_property_value (site_list, replica->name);

  /* don't think we need this any more */
  if (site != NULL)
    {
      fd_error ("%s: %s already defined\n", args[0], replica->name);
      return;
    }

  obj = xmalloc (sizeof (AnonymousObject));
  memset (obj, 0, sizeof (AnonymousObject));

  /* Now, put the site on the replica property list and the site list.
   * On the site list is goes it's tag will be the site name, while on
   * the replica list it's tag will be "Site".
   */
  set_property (replica, "Site", obj, Prop_List, 0);
  set_property (site_list, replica->name, obj, Prop_List, 0);
  
  context_push ("Site", obj);
  
  site = get_property (site_list, replica->name);

  set_property (site->value, 
		"Hostname", strdup(replica->name), Prop_String, 0);
  set_property (site->value, 
		"SiteName", strdup(replica->name), Prop_String, 0);

  set_default (site->value, "DataPort", Prop_Integer);
  set_default (site->value, "ClientPort", Prop_Integer);
}


Site *
get_site (char **args)
{
  Site *site;

  if (strcmp  (current_context->name, "Site") == 0)
    return (current_context->data);

  return (NULL);
}

void
set_site_integer (char **args)
{
  Site *site;

  /* If we have no context, then we must be setting defaults */
  if (current_context == NULL)
    site = (Site *) defaults_list;
  else
    {
      site = get_site (args);

      if (site == NULL)
	{
	  fd_error ("Cannot set defaults inside context\n");
	  return;
	}
    }
  set_property (site, args[0], atoi (args[1]), Prop_Integer, 0);
}

void
set_site_string (char **args)
{
  Site *site;
  char *tag;

  site = get_site (args);
  if (site == NULL)
    return;

  tag = get_property_value (mappings, args[0]);
  if (tag  == NULL)
    tag = args[0];

  set_property (site, tag, strdup (args[1]), Prop_String, 0);
} 


Group *
get_group (char **args)
{
  Group *group;

  if (strcmp  (current_context->name, "Group") == 0)
    return (current_context->data);

  return (NULL);
}

void
set_group (char **args)
{
  Property *group;
  AnonymousObject *obj;

  if (args[1] == NULL)
    {
      fd_error ("%s requires an argument\n", args[1]);
      return;
    }

  group = (Property *) get_property (group_list, args[1]);

  if (group != NULL)
    {
      fd_error ("%s: %s already defined\n", args[0], args[1]);
      return;
    }

  /* Clean up any junk on the stack */
  context_pop_til (NULL);

  obj = xmalloc (sizeof (Group));
  memset (obj, 0, sizeof (Group));
  set_property (group_list, strdup (args[1]), obj, Prop_List, 0);

  context_push ("Group", obj);

  group = get_property (group_list, args[1]);

  set_property (group->value, "GroupName", strdup (args[1]), Prop_String, 0);
  
  /* Set some defaults */

  set_default (group->value, "TopologyCommand",   Prop_String);
  set_default (group->value, "BandwidthPeriod",   Prop_Integer);
  set_default (group->value, "EstimatesPeriod",   Prop_Integer);
  set_default (group->value, "JoinPeriod",        Prop_Integer);
  set_default (group->value, "PingPeriod",        Prop_Integer);
  set_default (group->value, "UpdatePeriod",      Prop_Integer);
  set_default (group->value, "TopologyGenerator", Prop_Filename);

}

void
set_group_members (char **args)
{
  Group *group;
  Property *site;
  int i;

  if (args[1] == NULL)
    {
      fd_error ("%s requires arguments\n", args[0]);
      return;
    }

  group = get_group (args);

  if (group == NULL)
    {
      fd_error ("%s: %s not defined\n", args[0], args[1]);
      return;
    }

  for (i = 1; args[i] != NULL; i++)
    {
      site = get_property_by_key (site_list,
				  "SiteName", Prop_String, args[i]);

      if (site == NULL)
	fd_error ("%s: No site `%s' defined.\n", args[0], args[i]);
      else
	{
	  set_property (&group->sites, args[i],
			strdup (args[1]), Prop_String, 0);
	}
    }
}

void
set_group_master (char **args)
{
  Group *group;
  Property *site;
  int i;

  group = get_group (args);

  site = get_property_value (site_list, args[1]);

  if (site == NULL)
    {
      fd_error ("%s: Site `%s' not defined\n", args[0], args[1]);
      return;
    }

  set_property (group, "GroupMaster", strdup (args[1]), Prop_String, 0);
}

void
set_group_integer (char **args)
{
  Group *group;

  /* If we have no context, then we must be setting defaults */
  if (current_context == NULL)
    group = (Group *) defaults_list;
  else
    {
      group = get_group (args);

      if (group == NULL)
	{
	  fd_error ("Cannot set defaults inside context\n");
	  return;
	}
    }
  set_property (group, args[0], atoi (args[1]), Prop_Integer, 0);
}

void
set_group_string (char **args)
{
  Group *group;

  /* If we have no context, then we must be setting defaults */
  if (current_context == NULL)
    group = (Group *) defaults_list;
  else
    {
      group = get_group (args);

      if (group == NULL)
	{
	  fd_error ("Cannot set defaults inside context\n");
	  return;
	}
    }
  set_property (group, args[0], strdup (args[1]), Prop_String, 0);
}

void
set_group_filename (char **args)
{
  Group *group;

  /* If we have no context, then we must be setting defaults */
  if (current_context == NULL)
    group = (Group *) defaults_list;
  else
    {
      group = get_group (args);

      if (group == NULL)
	{
	  fd_error ("Cannot set defaults inside context\n");
	  return;
	}
    }
  set_property (group, args[0], expand_string (args[1]), Prop_Filename, 0);
}

AnonymousObject *
get_local (char **args)
{
  Site *site;

  if (current_context == NULL)
    return (NULL);

  if (strcmp  (current_context->name, "Local") == 0)
    return (current_context->data);
  else
    {
      fd_error ("%s: \n", args[0]);
      return;
    }

  return (NULL);
}

set_local (char **args)
{
  Context *context;
  Replica *replica;
  AnonymousObject *local;
  int n;

  context = context_pop_til ("Replica");
  if (context == NULL)
    {
      fd_error ("%s: Not in Replica context.\n", args[0]);
      return;
    }

  replica = context->data;


  local = get_property_value ((AnonymousObject *)replica, "local");

  if (local != NULL)
    {
      fd_error ("%s: %s already defined\n", args[0], args[1]);
      return;
    }

  local = xmalloc (sizeof (AnonymousObject));
  memset (local, 0, sizeof (AnonymousObject));
  set_property (replica, strdup("local"), local, Prop_List, 0);

  set_default (local, "DataStorageMax", Prop_Integer);
  set_default (local, "ExportCommand",  Prop_Filename);
  set_default (local, "LogPurgePeriod", Prop_Integer);
  set_default (local, "AccessFile",     Prop_Filename);

  context_push ("Local", local);
}

void
set_local_integer (char **args)
{
  AnonymousObject *local;

  if (args[1] == NULL)
    {
      fd_error ("%s: Requires and argument.\n", args[0]);
      return;
    }

  if (current_context == NULL)
    local = defaults_list;
  else
    {
      local = get_local (args);
      if (local == NULL)
	{
	  fd_error ("Cannot set local outside context\n");
	  return;
	}
    }
  set_property (local, args[0], atoi (args[1]), Prop_Integer, 0);
}

void
set_local_string (char **args)
{
  AnonymousObject *local;
  if (args[1] == NULL)
    {
      fd_error ("%s: Requires and argument.\n", args[0]);
      return;
    }
  
  if (current_context == NULL)
    local = defaults_list;
  else
    {
      local = get_local (args);

      if (local == NULL)
	{
	  fd_error ("Cannot set local outside context\n");
	  return;
	}
    }
  set_property (local, args[0], strdup (args[1]), Prop_String, 0);
}

void
set_local_filename (char **args)
{
  AnonymousObject *local;
  if (args[1] == NULL)
    {
      fd_error ("%s: Requires and argument.\n", args[0]);
      return;
    }
  
  if (current_context == NULL)
    local = defaults_list;
  else
    {
      local = get_local (args);

      if (local == NULL)
	{
	  fd_error ("Cannot set local outside context\n");
	  return;
	}
    }
  set_property (local, args[0], expand_string (args[1]), Prop_Filename, 0);
}

void
set_context (char **args)
{
  if (strcmp (args[1], "Replica") == 0)
    set_replica (&args[1]);
  if (strcmp (args[1], "Site") == 0)
    set_site (&args[1]);
  if (strcmp (args[1], "Group") == 0)
    set_group (&args[1]);
  if (strcmp (args[1], "Local") == 0)
    set_local (&args[1]);

  if (*args[1] == '/')
    {
      Context *context;
      context = context_pop_til (&args[1][1]);
      if (context == NULL)
	fd_error ("No previous `%s' context\n");
      else
	context_pop (NULL);
    }
}

void
fd_config_initialize () 
{ 
  mappings = xmalloc (sizeof (AnonymousObject));
  memset (mappings, 0, sizeof (AnonymousObject));

  defaults_list = xmalloc (sizeof (AnonymousObject));
  memset (defaults_list, 0, sizeof (AnonymousObject));

  replica_list = xmalloc (sizeof (AnonymousObject));
  memset (replica_list, 0, sizeof (AnonymousObject));

  site_list = xmalloc (sizeof (Site));
  memset (site_list, 0, sizeof (Site));

  group_list = xmalloc (sizeof (Group));
  memset (group_list, 0, sizeof (Group));

  local_list = xmalloc (sizeof (AnonymousObject));
  memset (local_list, 0, sizeof (AnonymousObject));

  /*
   * Generic context handling command 
   */
  cmd_add_command ("<", set_context,
		   "Begin a new context.", NULL);

  cmd_add_command ("Host", set_host,
                   "Generate config for this site.", NULL);

  /*
   * Replica commands
   */
  cmd_add_command ("RootDirectory", set_replica_string,
                   "The root directory for a replica.", NULL);
  cmd_add_command ("BinDirectory", set_replica_filename,
                   "The binary directory for a replica.", NULL);
  cmd_add_command ("DataDirectory", set_replica_filename,
                   "The data  directory for a replica.", NULL);
  cmd_add_command ("LibDirectory", set_replica_filename,
                   "The library directory for a replica.", NULL);
  cmd_add_command ("Username", set_replica_string,
                   "The username to use for a replica.", NULL);
  cmd_add_command ("DebugLevel", set_replica_integer,
                   "Set the debug level for a replica.", NULL);
  cmd_add_command ("DebugFlag", set_replica_string,
                   "Set the debug flag for a replica.", NULL);

  /*
   * Site commands 
   */

  cmd_add_command ("DataPort", set_site_integer,
                   "The data port for a site.", NULL);
  cmd_add_command ("ClientPort", set_site_integer,
                   "The client port for a site.", NULL);
  cmd_add_command ("InternetAddress", set_site_string,
                   "The internet address for a site.", NULL);
  cmd_add_command ("Hostname", set_site_integer,
                   "The hostname to use for a site.", NULL);
  cmd_add_command ("SiteLongitude", set_site_string,
                   "The longitude for a site.", NULL);
  cmd_add_command ("SiteLattitude", set_site_string,
                   "The lattitude for a site.", NULL);
  cmd_add_command ("Name", set_site_string,
                   "The name for a site.", NULL);

  /*
   * Group commands 
   */
#ifdef NEVER
  cmd_add_command ("<Group", set_group, "Name a group", NULL);
  cmd_add_command ("</Group>", pop_group,
                   "Name a group", NULL);
#endif
  cmd_add_command ("Members", set_group_members,
		   "The members of a group", NULL);
  cmd_add_command ("Master", set_group_master,
		  "The group master.", NULL);
  cmd_add_command ("MaxSize", set_group_integer,
		  "The maximum size of the group", NULL);
  cmd_add_command ("BandwidthPeriod", set_group_integer,
		  "How often to compute bandwidth to a randomly chosen site.", NULL);
  cmd_add_command ("Connectivity", set_group_integer,
		  "How often to compute bandwidth to a randomly chosen site.", NULL);
  cmd_add_command ("PingPeriod", set_group_integer,
		  "How often to compute RTT to a randomly chosen site.", NULL);
  cmd_add_command ("EstimatesPeriod", set_group_integer,
		  "How often to distrbute RTT and BW estimates.", NULL);
  cmd_add_command ("JoinPeriod", set_group_integer,
		  "How often to arbitrarily rejoin the group.", NULL);
  cmd_add_command ("UpdatePeriod", set_group_integer,
		  "How often to send out topology updates.", NULL);
  cmd_add_command ("TopologyGenerator", set_group_filename,
		  "Command to use to generate topology", NULL);

  /*
   * Local Parameters
   */
#ifdef NEVER
  cmd_add_command ("<Local>", set_local,
		   "Set local parameters for a replica", NULL);
#endif
  cmd_add_command ("AuxillaryInfo", set_local_string,
                   "Specify auxillary info for floodd html interface.", NULL);

  cmd_add_command ("Maintainer", set_local_string,
                   "Specify the maintainer of the site.", NULL);

  cmd_add_command ("DataStorageMax", set_local_integer,
                   "Maximum size of internal data storage.", NULL);

  cmd_add_command ("ExportCommand", set_local_filename,
                   "Command used to export data to application.", NULL);

  cmd_add_command ("Password", set_local_string,
                   "Administrative password..", NULL);

  cmd_add_command ("LogPurgePeriod", set_local_integer,
                   "How often to purge logs.", NULL);

  cmd_add_command ("AccessFile", set_local_filename,
                   "File containing access permissions.", NULL);

  /* 
   * Set up some mappings between our config format and other output formats
   */
  mapping_add (mappings, "ClientPort", 		"client-port");
  mapping_add (mappings, "DataPort", 		"data-port");
  mapping_add (mappings, "InternetAddress",	"inet-address");
  mapping_add (mappings, "Hostname", 		"hostname");
  mapping_add (mappings, "SiteName", 		"site-name");

  mapping_add (mappings, "GroupName", 		"group-name");
  mapping_add (mappings, "Connectivity", 	"connectivity");
  mapping_add (mappings, "EstimatesPeriod", 	"estimates-period");
  mapping_add (mappings, "JoinPeriod", 		"join-period");
  mapping_add (mappings, "BandwidthPeriod", 	"bandwidth-period");
  mapping_add (mappings, "PingPeriod", 		"ping-period");
  mapping_add (mappings, "UpdatePeriod", 	"update-period");
  mapping_add (mappings, "GroupMaster", 	"master-site");
  mapping_add (mappings, "TopologyGenerator", 	"topology-generator");
  mapping_add (mappings, "AllowSite", 		"allow-site");
  mapping_add (mappings, "DenySite", 		"deny-site");

  mapping_add (mappings, "AuxillaryInfo", 	"auxillary-info");
  mapping_add (mappings, "DataStorageMax", 	"data-storage-max");
  mapping_add (mappings, "ExportCommand",  	"export-command");
  mapping_add (mappings, "AccessFile",  	"access-file");
  mapping_add (mappings, "LogPurgePeriod",	"log-purge-period");
  mapping_add (mappings, "SiteLongitude",       "longitude");
  mapping_add (mappings, "SiteLattitude",       "lattitude");
  mapping_add (mappings, "Maintainer",          "maintainer");
  mapping_add (mappings, "Password",            "password");

}

void
output_sites (Property *property)
{
  fprintf (fd_output, "(:site-define\n");
  
  symbol_filter = convert_to_floodd;

  describe_property_list (((AnonymousObject *)property->value)->properties, 
			  1, fd_output, NULL);
  fprintf (fd_output, ")\n");
}


output_site_name (Property *property)
{
  fprintf (fd_output, "\n (:site \"%s\")", property->tag);
}


output_tags (Property *property)
{
  fprintf (fd_output, "%s ", property->tag);
}

void
output_groups (Property *property)
{
  symbol_filter = convert_to_floodd;

  /* Only output the group if the local_host is defined and the local_host
   * is in the group
   */
  if (local_host != NULL)
    {
      Property *site;
      site = get_property (&((Group *)property->value)->sites,
			   local_host);
      if (site == NULL)
	return;

    }
  fprintf (fd_output, "(:group-define\n");

  describe_property_list (((AnonymousObject *)property->value)->properties, 
			  1, fd_output, NULL);
  /* output the sites */
  apply_property_list (((Group *)property->value)->sites.properties, output_site_name);
  fprintf (fd_output, ")\n");
}

void
output_local ()
{
  Replica *replica;
  AnonymousObject *local;

  if (local_host == NULL)
    return;
  
  replica = get_property_value (replica_list, local_host);
  if (replica == NULL)
    {
      fd_error ("No know replica `%s'\n", local_host);
      return;
    }
  local = (AnonymousObject *) 
    get_property_value ((AnonymousObject *) replica, "Local");
  
  if (local != NULL)
    {
      fprintf (fd_output, "(:local\n ");
      describe_property_list (local->properties, 1, fd_output, NULL);
      fprintf (fd_output, ")\n");
    }
}

int list_sites = 0;
int list_groups = 0;

struct option longopts[] =
{
  /* { name  has_arg  *flag  val } */
  {"configuration", 0, 0, 'c'}, /* output a configuration for a site */
  {"help",          0, 0, 'h'}, /* Help message */
  {"list-groups",   1, 0, 'g'}, /* list groups */
  {"list-sites",    1, 0, 'l'}, /* list site */
  {"parameter",     1, 0, 'p'}, /* output the value for a parameter  */
  {"site",          1, 0, 's'}, /* override the local_host variable*/
  {NULL,            0, 0,   0}, /* terminate the array */
};

main (int argc, char **argv)
{
  FILE *file;
  char *line;
  int optc;           /* current option */
  int output_configuration = 0;
  char *output_parameter = NULL;
  int file_count;
  char *readline ();

  while ((optc = getopt_long (argc, argv, "cghlp:s:",
			      longopts, (int *)0)) != EOF) {
	switch (optc) {
	case 'c':
	  output_configuration = 1;
	  break;
	case 'g':
	  list_groups = 1;
	  break;
	case 'h':
	  usage ();
	  break;
	case 'l':
	  list_sites = 1;
	  break;
	case 'p':
	  output_parameter = strdup (optarg);
	  break;

	case 's':
	  local_host = strdup (optarg);
	  break;
	}
      }

  file_count = argc - optind;

  if (file_count > 2)
    {
      fprintf (stderr, "Usage: %s [filename]", argv[0]);
      exit (1);
    }

  if (file_count == 1)
    {
      fd_filename = argv[optind];
      file = fopen (argv[optind], "r");
      if (file == NULL)
	{
	  fprintf (stderr, "Could not open `%s'\n", fd_filename);
	  exit (1);
	}
    }
  else
    {
      fd_filename = "stdin";
      file = stdin;
    }

  fd_config_initialize ();

  while (line = (char *) readline (NULL, file))
    {
      fd_lineno++;
      if (*line != '#')
	cmd_execute_line (line);
      xfree (line);
    }

  if (list_sites)
    {
      apply_property_list (replica_list->properties, output_tags);
    }
  
  if (list_groups)
    {
      apply_property_list (group_list->properties, output_tags);
    }

  if (local_host && output_configuration)
    {
      apply_property_list (site_list->properties, output_sites);
      apply_property_list (group_list->properties, output_groups);
      output_local ();
    }

  /* When we search for a parameter, we look for a replica named 
   * for the local_host and then we search it's property list
   * and the the property list of the replica's site.
   */

  if (local_host && output_parameter != NULL)
    {
      Property *property;
      
      property = get_parameter (output_parameter, local_host);
      if (property != NULL)
	print_property_value (property, stdout);
    }
  else if (output_parameter != NULL)
    {
      Property *property;

      property = get_property (defaults_list, NULL);
      if (property != NULL)
	print_property_value (property, stdout);
    }

  if (fd_error_count)
    exit (1);
  exit (0);
}

usage ()
{

  printf ("Usage: %s [options] [config-file]\n");
  printf ("-c --configuration        output floodd configuration file\n");
  printf ("-h --help                 display this message\n");
  printf ("-g --list-groups          output list of groups\n");
  printf ("-l --list-sites           output list of sites\n");
  printf ("-p name --parameter name  output parameter value\n");
  printf ("-s site --site site       output for this site, if not specified, output defaults\n");

exit (0);
}
/* Expand environment variables of the form ${foo} in strings */
#define SEPARATORS "/{}()[]|;.,<>:"

char *
expand_string (char *string)
{
  string_declare (result);
  string_declare (variable);
  
  string_init (result, 32);
  string_init (variable, 32);

  while (*string)
    {
      if (*string == '$')
	{
	  char *begin_variable;
	  char *end_variable;
	  char *value;
	  char *pos;

	  string++;
	  if (*string == '$')	/* escaped $ */
	    {
	      string_append_character (result, '$');
	      string++;
	    }
	  else			/* we have an environment variable */
	    {
	      string_reset (variable);
	      if (*string == '{')
		{
		  begin_variable = ++string;
		  end_variable = strchr (string, '}');
		  string = end_variable + 1;

		  if (end_variable == NULL)
		    goto done;
		}
	      else
		{
		  begin_variable = string++;
		  end_variable = strpbrk (string, SEPARATORS);
		  if (end_variable == NULL)
		    {
		      end_variable = begin_variable;
		      while (*end_variable)
			end_variable++;
		    }
		  string = end_variable;
		}
	      for (pos = begin_variable; pos != end_variable; pos++)
		string_append_character (variable, *pos);
	      if (variable != NULL)
		{
		  Property *property;
#ifdef NEVER		  
		  value = (char *) getenv (variable);
#endif
		  property = get_parameter (variable, NULL);
		  if (property != NULL)
		    string_append_string (result, property->value);
		}
	    }
	}
      else
	string_append_character (result, *string++);
    }
 done:
  string_free (variable);
  return (result);
}

/* error handler */
void
fd_error (char *fmt, ...)
{ 
  va_list ap;
  va_start (ap, fmt);
  fprintf (stderr, "%s %d:", fd_filename, fd_lineno);
  vfprintf (stderr, fmt, ap);
  va_end (ap);
  fd_error_count++;
}
