/* $Id: properties.c,v 1.3 1995/07/25 20:08:02 dante Exp $ */
/* properties.c - handle properties that we know about */
#include <stdio.h>
#include <ctype.h>
#include <properties.h>
#include <dynamic_string.h>
#include <xmalloc.h>
static int indentation = 0;
static int indentation_increment = 1;

/* Make a property */
Property *
make_property (tag, value, type, size) 
     char *tag;
     void *value;
     Proptype type;
     int size;
{
  Property *prop;

  prop = (Property *)xmalloc (sizeof (Property));
  /*
  prop->tag = unique_string (tag);
  */
  prop->tag = (char *)strdup (tag);
  prop->type = type;
  prop->size = size;
  prop->value = value;
  prop->describer = NULL;
  return (prop);
}

/* Get and return a pointer to the property structure in OBJECT of TAG,
   or return a NULL property if OBJECT doesn't have a TAG property. */
Property *
get_property (object, tag)
     AnonymousObject *object;
     char *tag;
{
  register int i;
  Property **props;

  props = object->properties;

  for (i = 0; (props != (Property **)NULL) && (props[i]); i++)
    if (strcasecmp (props[i]->tag, tag) == 0)
      return (props[i]);

  return ((Property *)NULL);
}

/* Search a list of property lists for a list containing a property with
 * the given key.
 */

Property *
get_property_by_key (object, tag, type, key)
     AnonymousObject  *object;
     char *tag;
     Proptype type;
     void *key;
{
  register int i;

  Property **props = object->properties;


  for (i = 0; (props != (Property **)NULL) && (props[i]); i++)
    {
      if (props[i]->type == Prop_List)
	{
	  Property *prop;

	  prop = 
	    get_property ((AnonymousObject *) 
			  (((AnonymousObject *)&props[i]->value)->properties),
			  tag);
	  if (prop != NULL && prop->type == type &&
	      (compare_property (prop, key) == 0))
	    return (props[i]);
	}
  }
  return ((Property *)NULL);
}
 
void *
get_property_value (object, tag)
     AnonymousObject *object;
     char *tag;
{
  Property *prop;

  prop = get_property (object, tag);
  if (prop == NULL)
    return (NULL);
  
  return (prop->value);
}


/* Give OBJECT a property of TAG with a VALUE of type TYPE.  Size is only
   used when TYPE is Prop_Pointer.  Returns the old value. */
Property *
set_property (object, tag, value, type, size)
     AnonymousObject *object;
     char *tag;
     void *value;
     Proptype type;
     int size;
{
  register int i;
  Property *new_prop;
  Property **props;

#ifdef NEVER
  new_prop = (Property *)xmalloc (sizeof (Property));
  /*
  new_prop->tag = unique_string (tag);
  */
  new_prop->tag = (char *)strdup (tag);
  new_prop->type = type;
  new_prop->size = size;
  new_prop->value = value;
  new_prop->describer = NULL;
#endif
  new_prop = make_property (tag, value, type , size);

  /* See if this object has a TAG property. */
  props = object->properties;

  for (i = 0;
       (props != (Property **)NULL) && (props[i] != (Property *)NULL);
       i++)
    if (strcasecmp (props[i]->tag, tag) == 0)
      {
	Property *retval;

	retval = props[i];
	props[i] = new_prop;
	return (retval);
      }

  /* The property wasn't found on the list.  Increase the size of the list
     to allow for the new property. */
  object->properties = (Property **)
    xrealloc (object->properties, (2 + i) * sizeof (Property *));
  object->properties[i] = new_prop;
  object->properties[i + 1] = (Property *)NULL;
  return ((Property *)NULL);
}

/* Free the memory that set_property () specifically allocated.  Other memory
   that may have been allocated to PROP must be freed beforehand by the
   caller. */
void
free_property (prop)
     Property *prop;
{
  if (prop)
    free (prop);
}

/* Free a property array. */
void
free_properties (props)
     Property **props;
{
  if (props)
    {
      register int i;
      register Property *prop;

      for (i = 0; prop = props[i]; i++)
	free_property (prop);

      free (props);
    }
}

/* Merge PROPERTIES into OBJECT. */
/* The merge is dependent on mode */
/*   OVERRIDE - replaces current properties if name conflict
 *   NO_OVERRIDE - don't replace properties if names conflict
 */
void
merge_properties (properties, object, mode)
     Property **properties;
     AnonymousObject *object;
     int mode;
{
  register int i;
  register Property *prop;

  for (i = 0; properties && (prop = properties[i]); i++)
    {
      if (mode == OVERRIDE || 
	  (get_property (object, prop->tag) == NULL && mode == NO_OVERRIDE))
	{
	  if (prop->type == Prop_String || prop->type == Prop_Filename)
	    set_property (object, prop->tag, strdup(prop->value), 
			  prop->type, prop->size);
	  else
	    set_property (object, prop->tag, prop->value, 
			  prop->type, prop->size);
	  
	}
    }
}

/* Remove from OBJECT the property with tag TAG.  Returns the removed property.
   The caller is responsible for calling free_property () on the results. */
Property *
del_property (object, tag)
     AnonymousObject *object;
     char *tag;
{
  register int i, j;
  Property **props;
  Property *prop;

  props = object->properties;

  if (!props)
    return ((Property *)NULL);

  for (i = 0; props[i]; i++)
    if (strcasecmp (props[i]->tag, tag) == 0)
      {
	prop = props[i];
	for (j = i + 1; props[j]; j++)
	  props[i++] = props[j];
	props[i] = (Property *)NULL;
	return (prop);
      }

  return ((Property *)NULL);
}

static char symbol_buffer[1000];

static char *
string_to_symbol (string)
     char *string;
{
  register int i;

  for (i = 0; symbol_buffer[i] = string[i]; i++)
    {
      if (isspace (string[i]))
	symbol_buffer[i] = '-';
    }
  symbol_buffer[i] = '\0';
  return (symbol_buffer);
}

char *(*symbol_filter)();

char *
symbol_output (string)
     char *string;
{
  if (symbol_filter != NULL)
    return (symbol_filter (string));
  else
    return (string_to_symbol (string));
}

static int
indent (stream, indentation)
     FILE *stream;
     int indentation;
{
  int i;

  for (i = 0; i < indentation; i++)
    if (putc (' ', stream) == EOF)
      return (EOF);
  return (indentation);
}

static int
newline (stream, indentation)
     FILE *stream;
{
  register int i;
  int n;
  int total = 0;

  /* Print the newline, and then indent INDENTATION. */
  n = fprintf (stream, "\n");

  if (n == EOF)
    return (EOF);
  else
    total += n;
  
  n = indent (stream, indentation);

  if (n == EOF)
    return (EOF);

  return (total + indentation);
}

/* Describe the property PROP on STREAM.
   If PROP has a `describer' function, then that function is used to get
   a printed representation of the property.  This is generally only used
   when the property type is Prop_Pointer.
   Note that this function actually prints out property lists as if they
   were association lists. This seems to be the easiest thing to read and
   parse in Lisp. */
void
describe_property (prop, indentation, stream)
     Property *prop;
     FILE *stream;
{

  fprintf (stream, "(:%s ", symbol_output (prop->tag));

  switch (prop->type)
    {
    case Prop_Filename:
    case Prop_String:
      {
	extern char *expand_string ();
	char *tmp = expand_string ((char *) prop->value);
	fprintf (stream, "\"%s\"", tmp);
	xfree (tmp);
      }
      break;

    case Prop_Integer:
      fprintf (stream, "%d", (int)prop->value);
      break;

    case Prop_Float:
      {
	double val;

	val = *((double *)prop->value);
	fprintf (stream, "%lg", val);
      }
      break;

    case Prop_Properties:
      indentation += indentation_increment;
      newline (stream, indentation);
      describe_property_list ((Property **)prop->value, indentation, 
			      stream, (char *)NULL);
      indentation -= indentation_increment;
      break;

    case Prop_Pointer:
      if (prop->describer)
	(*prop->describer) (prop, stream);
      else
	fprintf (stream, "#<pointer %0X>", prop->value);
      break;

    case Prop_List:
      indentation += indentation_increment;
      describe_property_list (((AnonymousObject *)prop->value)->properties, 
			      indentation, stream, NULL);
      indentation -= indentation_increment;
      break;

    default:
      fprintf (stream, "#<Bad Property Type: %d", prop->type);
    }
  fprintf (stream, ")");
}

int
compare_property (prop, key)
     Property *prop;
     void *key;
{

  switch (prop->type)
    {
    case Prop_Filename:
    case Prop_String:
      return (strcmp (prop->value, key));
      break;

    case Prop_Integer:
      if (((int) prop->value) == (int) key)
	return (0);
      return (-1);
      break;


    case Prop_Float:
    default:
      return (-1);
    }
}

print_property_value (prop, stream)
     Property *prop;
     FILE *stream;
{
  switch (prop->type)
    {
    case Prop_Filename:
    case Prop_String:
      {
	extern char *expand_string ();
	char *tmp = expand_string ((char *) prop->value);
	fprintf (stream, "%s", tmp);
	xfree (tmp);
      }
      break;

    case Prop_Integer:
      fprintf (stream, "%d", (int)prop->value);
      break;

    case Prop_Float:
      {
	double val;

	val = *((double *)prop->value);
	fprintf (stream, "%lg", val);
      }
      break;
    }
}

delete_property (Property *prop)
{
  if (prop->type == Prop_String || prop->type == Prop_Filename)
    xfree (prop->value);

  xfree (prop->tag);
  xfree (prop);
}

void
delete_property_list (Property **properties)
{
  int i;

  for (i = 0; properties[i] != NULL; i++)
    delete_property (properties[i]);
  xfree (properties);
}

int
describe_property_list (props, indentation, stream, tag)
     Property **props;
     FILE *stream;
     char *tag;
{
  Property *prop;
  int total = 0;
  register int i;
  int n;

#ifdef NEVER
  n = fprintf (stream, "(");

  if (n == EOF)
    return (EOF);
  else
    total += n;

  if (tag)
    {
      n = fprintf (stream, "%s", tag);
      if (n == EOF)
	return (EOF);
      else
	total += n;

      indentation += indentation_increment;

      n = newline (stream, indentation);

      if (n == EOF)
	return (EOF);
      else
	total += n;

    }
  else
    indentation += 2;
#endif
  indent (stream, indentation);

  for (i = 0; props && (prop = props[i]); i++)
    {
      describe_property (prop, indentation, stream);
      if (props[i + 1])
	{
	n = newline (stream, indentation);

	if (n == EOF)
	  return (EOF);
	else
	  total += n;

      }
    }

#ifdef NEVER
  n = fprintf (stream, ")");

  if (n == EOF)
    return (EOF);
  else
    total += n;
#endif
  indentation -= (tag ? indentation_increment : 2);
  return (n);
}

int
apply_property_list (props, function)
     Property **props;
     CPFunction *function;
{
  int i;

  for (i = 0; props && props[i]; i++)
    function (props[i]);

}

#ifdef NEVER
/* Versions that decribe properties into strings. */
/* This code relies on dynamic_string.h */

describe_property_string (prop, buffer, indentation)
     Property *prop;
     char *buffer;
     int indentation;
{
  strcat (buffer, "(:");
  strcat (buffer, symbol_output (prop->tag));
  strcat (buffer, " ");
  switch (prop->type)
    {
    case Prop_Filename:
    case Prop_String:
      strcat (buffer, "\"");
      strcat (buffer, (char *)prop->value);
      strcat (buffer, "\"");
      break;

    case Prop_Integer:
      {
	char buf[32];
	sprintf (buf, "%d",  (int)prop->value);
	strcat (buffer, buf);
      }
      break;

    case Prop_Float:
      {
	double val;
	char buf[32];

	val = *((double *)prop->value);
	sprintf (buf, "%lg",  val);
	strcat (buffer, buf);
      }
      break;

#ifdef NEVER
    case Prop_Properties:
      indentation += indentation_increment;
      newline (stream, indentation);
      describe_property_list ((Property **)prop->value, indentation, 
			      stream, (char *)NULL);
      indentation -= indentation_increment;
      break;
    case Prop_Pointer:
      if (prop->describer)
	(*prop->describer) (prop, stream);
      else
	fprintf (stream, "#<pointer %0X>", prop->value);
      break;
#endif

    default:
      {
	char buf[32];
	strcat (buffer, "#<Bad Property Type: ");
	sprintf (buf, "%d",  (int)prop->type);
	strcat (buffer, buf);
      }
      break;
    }
  strcat (buffer, ")");
}

char *
describe_property_list_string (props, indentation, tag)
     Property **props;
     int indentation;
     char *tag;
{
  Property *prop;
  register int i;
  string_declare (result);
  int n;
  char buffer[1024];

  string_init (result, 128);
  string_append_string (result, "(");

  if (tag)
    {
      string_append_string (result, tag);

      indentation += indentation_increment;

      string_append_string (result, "\n");
      {
	int i;
	for (i = 0; i < indentation; i++)
	  string_append_string (result, " ");
      }
    }
  else
    indentation += 2;

  for (i = 0; props && (prop = props[i]); i++)
    {
      buffer[0] = '\0';
      describe_property_string (prop, buffer, indentation);
      string_append_string (result, buffer);

      if (props[i + 1])
	{
	  string_append_string (result, "\n");
	  {
	    int i;
	    for (i = 0; i < indentation; i++)
	      string_append_string (result, " ");
	  }
	}
    }

  string_append_string (result, ")");

  indentation -= (tag ? indentation_increment : 2);
  return (result);
}

#endif
