/* $Id: network.c,v 1.5 1995/07/25 20:07:24 dante Exp $ */
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#if defined (HAVE_SYSLOG_H)
#  include <syslog.h>
#endif /* !SYS_SYSLOG */
#include <floodd.h>
#include <dynamic_string.h>
#if defined (HAVE_RANDOM)
extern long random ();
#else
#define srandom srand
#define random rand
#endif /* !HAVE_RANDOM */

/* Threads for handling datablocks */
/* Note:  All threads are responsible for freeing data blocks passed
 *        to them.  Before any pthread_create, the blocks must be referenced
 *        to so that the thread the block is passed to can deallocate 
 *        it.
 */

extern int debug;
extern int export_fd;

/* Possible threads to fire up to handle a datablock */
void receive_topology (DataBlock *datablock);
void receive_estimates (DataBlock *datablock);
void receive_data (DataBlock *datablock);
void receive_join (DataBlock *datablock);
void receive_local (DataBlock *datablock);

void
flood_datablock (DataBlock *datablock, SiteInfo *except);

void
flood_datablock_to_group (Group *group, DataBlock *datablock, SiteInfo *except);

struct
{
  int type;
  void (*thread) ();
} datablock_types[] =
{
  { TYPE_BANDWIDTH, NULL},
  { TYPE_TOPOLOGY,  receive_topology},
  { TYPE_ESTIMATES, receive_estimates},
  { TYPE_DATA,      receive_data},
  { TYPE_FEDEX,     receive_local},
  { TYPE_JOIN,	    receive_join},
  { 0,              NULL},
};

/* Startup a thread to deal with a datablock depending on what type of
 * data we have received.  The possible datablock types are in the 
 * datablock_types array.
 */

int
dispatch_datablock (DataBlock *datablock)
{
  int i;

  for (i = 0; datablock_types[i].type != 0; i++)
    if (datablock_types[i].type == datablock->header.type)
      {
	if (datablock_types[i].thread)
	  {
	    datablock_types[i].thread (datablock);
	  }
	break;
      }
  if (datablock_types[i].type == 0 && debug)
    fprintf (stderr, "Unknown datablock type `%d'\n", datablock->header.type);

  /* Make sure we ack everyone */
  if (datablock->header.type != TYPE_FEDEX)
    {
      siteinfo_ack (&datablock->header.id);

      /* Log that we have received the message */
      log_message (&datablock->header.id);
    }
}


/* Process a topology update */
void
receive_topology (DataBlock *datablock)
{
  Group *group;
  char *data;
  char *token;
  char *site_definitions;
  char *group_defines;
  char *group_definition;
  char *topology_definition;
  char *group_name;
  char *site_list;
  struct timeval time;
  int version;

  /* Read group name */

  data = datablock->data;

  memset (&version, 0, sizeof(version));

  token = (char *)strstr (data, "(:group-name");
  group_name = return_token (next_token (next_token (token)));
  group = group_by_name (group_name);

  if (group == NULL)
    {
      if (debug)
	printf ("receive_topology():Bad group name `%s'\n", group_name);
      xfree (group_name);
      goto receive_topology_finish;
    }
  xfree (group_name);

  token = (char *)strstr (data, "(:version");
  token = next_word (token);
  version = atoi (token);

  if (version <= group->topology_version)
    {
      printf ("Discarding topology update %d in favor of current %d\n",
		version, group->topology_version);
      goto receive_topology_finish;
    }

  /* Check to see that the topology is reasonable, i.e. it should be with 15 
   * minutes of local time.
   */
  time = timeval_current ();

  if (abs (time.tv_sec - version) > 15 * 60)
    {
      if (debug)
	syslog (LOG_ERR, 
		"Discarding topology update %d.  It seems to way out of date (%ld)\n",
		version, time.tv_sec);

      goto receive_topology_finish;
    }
	
      
  if (debug)
    printf ("Received topology version %d for `%s'\n", version, group->name);

  /* Flood the stuff out again according to the old topology*/

  flood_datablock_to_group (group, datablock, NULL);

  /* Update our sitelist */
  site_list = strstr (data, "(:site-list");
  site_definitions = next_token (next_token (next_token (site_list)));


  /* Update our group parameters */
  group_defines = strstr (data, "(:group-define");
  if (group_defines != NULL)
    {
      group_definition = (return_list (group_defines));
      parse_config_string(group_definition, 2);
      xfree (group_definition);
    }

#ifdef NEVER
  /* Skip to next token in the site list (to avoid opening '(' 
   * And pass the result to the parser
   */
  site_definitions = return_list (skip_token (token));
#endif
  parse_config_string(site_definitions, 2);

#ifdef NEVER
  xfree (site_definitions);
#endif

  /* Now update our own topology */
  
  token = strstr (data, "(:topology ");
  topology_definition = return_list (token);

  group_topology_update (group, version, topology_definition);

  /* Save the topology */
  xfree (group->topology);
  group->topology = topology_definition;

 receive_topology_finish:

  /* Flood the datablock out again according to the new topology */
  /* This is required to ensure that when a new site joins, it is not
   * left out of a topology round.
   * we should probably keep a log of the datablocks sent to another site
   * so that we don't send duplicates.  If we are replicating large
   * things, when this could be big lose.
   */
  flood_datablock_to_group (group, datablock, NULL);
  
  if (group != NULL && group->neighbors != NULL && group->neighbors[0] != NULL)
    flush_hold_queue ();
}

/* Take the statistics datablock and update the stat tables */
void
receive_estimates (DataBlock *datablock)
{
  SiteInfo *site;
  Group *group;
  char *line;
  char group_name[512];

  site = site_by_id (&datablock->header.id.site);
  if (site == NULL)
    goto receive_statistics_done;

  if (debug)
    {
      printf ("Receiving statistics from %s `", site->name);
      line = datablock->data;
      while (*line)
	{
	  if (*line == '\n')
	    putchar (' ');
	  else 
	    putchar (*line);
	  line++;
	}
      printf ("'\n");
    }

  LIST_FREE_ELEMENTS (site->estimates, xfree);
  
  line = datablock->data;
  sscanf (line, "%s", group_name);
  group = group_by_name (group_name);
  if (group == NULL)
    goto receive_statistics_done;

  line = (char *) next_line (line);
  while (*line != '\0')
    {
      Statistics *stat = xmalloc (sizeof (*stat));
      
      scan_siteid (line, &(stat->id));
      /* skip site id */
      line = skip_word (line);
      sscanf (line, "%d %d", 
	      &stat->round_trip_time, &stat->bandwidth);

      list_insert (site->estimates, stat);
      /* Skip to next line */
      line = (char *)next_line (line);
    }

  flood_datablock_to_group (group, datablock, NULL);

 receive_statistics_done:
;
/*  datablock_free (datablock);*/
}


/* Receive a datablock conntaining application data to be passed
 * to the local application
 */

void
receive_data (DataBlock *datablock) 
{
  SiteInfo *from;

  /* put the block on the export queue - don't free it here since it will
   * be free by the export queue handler.
   */
  datablock_reference (datablock);
  queue_insert (export_queue, datablock);

  if (export_fd == NOTSET)
    export_start ();
  
  if (debug)
      printf ("Received Datablock (%d bytes): %s:%u:%u\n",
	      datablock->length,
	      siteid_to_string (&datablock->header.id.site), 
	      ntohl (datablock->header.id.time.tv_sec),
	      ntohl (datablock->header.id.time.tv_usec));

  from = site_by_id (&datablock->header.sender);
  /* Now flood the block to the groups */  
  flood_datablock (datablock, from);
}

/* Receive a join message.
 * Install the site in our sitelist so we can start collecting statistics
 * and then floodd it back out to other sites.
 * The master site will eventually receive it and incorportate it into the
 * next topology update.
 */

void
receive_join (DataBlock *datablock)
{
  SiteInfo *from;

  parse_list (datablock->data, 2);

  /* Now flood the block */  
  from = site_by_id (&datablock->header.sender);

  flood_datablock (datablock, from);
}

void
receive_local (DataBlock *datablock) 
{
  SiteInfo *from;

  /* put the block on the export queue - don't free it here since it will
   * be free by the export queue handler.
   */
  if (debug)
      printf ("FEDEX: Received (%d bytes): %s:%u:%u\n",
	      datablock->length,
	      siteid_to_string (&datablock->header.id.site), 
	      ntohl (datablock->header.id.time.tv_sec),
	      ntohl (datablock->header.id.time.tv_usec));

  datablock_reference (datablock);
  queue_insert (export_queue, datablock);
}

/* Flush hold queue */
flush_hold_queue ()
{
  DataBlock *datablock;
  
  datablock = queue_remove (whoami->datablocks_to_send);
  while (datablock != NULL)
    {
      flood_datablock (datablock, NULL);
      datablock = queue_remove (whoami->datablocks_to_send);
    }
}

/* Send a data block to a site, except if it is DOWN and the datablock is
 * a topology update or an estimates update.
 * If the site is not DOWN, fire up a thread if necessary to send the
 * datablocks.  It is necessary to fire up a thread if the 
 */
send_queue_insert (SiteInfo *site, DataBlock *datablock)
{
  if (site->status == STATUS_DOWN) 
    {
      if (datablock->header.type == TYPE_TOPOLOGY)
	return;

      if (datablock->header.type == TYPE_ESTIMATES)
	return;
    }
  /* Make sure the datablock has not already been acked */
  if (messagelog_find_messageid (site->acks, &datablock->header.id) != NULL)
    {
      if (debug)
	{
	  printf ("FLOODD: %s:%u:%u acked.",
		  siteid_to_string (&datablock->header.id.site), 
		  ntohl (datablock->header.id.time.tv_sec),
		  ntohl (datablock->header.id.time.tv_usec));
	  printf (" Not sent to %s\n", siteid_to_string (&site->id));
	}
      return;
    }

  /* By default we put it here */
  datablock_reference (datablock);

  queue_insert (site->datablocks_to_send, datablock);

  /* See if we need to fire up a sender thingy */
  if (site->sender_fd == NOTSET && site != whoami)
    connection_write_start (site);
}

/* Send the given data block to all neighbors, except `except' */
void
flood_datablock (DataBlock *datablock, SiteInfo *except)
{
  SiteInfo *site;
  int i;
  
  for (i = 0; groups[i] != NULL; i++)
    {
      flood_datablock_to_group (groups[i], datablock, except);
    }
}

void
flood_datablock_to_group (Group *group, DataBlock *datablock, SiteInfo *except)
{ 
  SiteInfo *site;
  int i;

  if (group->neighbors == NULL || group->neighbors[0] == NULL)
    {
      /* Send it manually to every site */
      /* we might end up sending duplicate, but who really cares? */
      for (i = 0; group->sites[i] != NULL; i++)
	if (group->sites[i] != whoami)
	  send_queue_insert (group->sites[i], datablock);
    }
  else
    for (i = 0; group->neighbors[i] != NULL; i++)
      {
	site = site_by_id (group->neighbors[i]);
	
	if (site && site != except && 
	    site->datablocks_to_send->tail->value != datablock)
	  {
	    if (site != whoami)
	      send_queue_insert (site, datablock);
	  }
      }
}

/* Send a string the the client interace.  This is primarily intended
 * for use by the join operation.
 */
int
send_string_to_client_interface (SiteInfo *site, char *string)
{
  int fd;
  int devnull (int fd, void *state);

  /* Just copy the address for the data port and change the portnumber to
   * the client address 
   */

  /* Connect to the client interface */

  fd = connect_site (site, site->client_port);

  if (fd < 0)
    return (-1);

  IO_start (fd,
	    devnull, NULL, NULL,
	    NULL, NULL, NULL,
	    NULL);

  IO_set_reader (fd, devnull);
  client_output_string (fd, string);

  IO_close (fd);		/* closes when done writing */
}

#ifdef STATISTICS
/* start of Adding by erhyuan */

/* initailize counters to zero */
int 
initialize_counters ()
{
  count_purge_block (0, -1);
  count_purge_byte (0, -1);
  count_sendout_block (-1);
  count_sendout_byte (-1);
  count_received_block (-1);
  count_received_byte (-1);
  count_duplicate_block (-1);
  count_duplicate_byte (-1);
  count_estimates_bytes_sent (-1);
  count_estimates_bytes_received (-1);
  count_bandwidth_bytes_sent (-1);
  count_bandwidth_bytes_received (-1);
  count_export(-1);
}

/* return the number of purged data-blocks according to its type.
 * oper: negative for reset, otherwise increase by the value of oper.
 *  specify oper=0 to return current value of counter.
 */
#define MAX_DATA_TYPES	10
int 
count_purge_block (int data_type, int oper)
{
  static int counter[MAX_DATA_TYPES];
  int i;

  /* return 0 if over range */
  if (data_type >= MAX_DATA_TYPES)
    return (0);

  /* reset counter values */
  if (oper < 0)
    {
      for (i = 0; i < MAX_DATA_TYPES; i++)
	counter[i] = 0;
    }
  else 
    {  	/* increased by value of oper */
      counter[data_type] += oper;
    }
  return (counter[data_type]);
}

int 
count_purge_byte (int data_type, int oper)
{
  static int counter[MAX_DATA_TYPES];
  int i;

  /* return 0 if over range */
  if (data_type >= MAX_DATA_TYPES)
    return (0);

  /* reset counter values */
  if (oper < 0)
    {
      for (i = 0; i < MAX_DATA_TYPES; i++)
	counter[i] = 0;
    }
  else 
    {  	/* increased by value of oper */
      counter[data_type] += oper;
    }
  return (counter[data_type]);
}

/* Need a queue to remember purged data-blocks,
   We count duplicate blocks as one block.
*/
Queue *
init_count_purge()
{
  Queue *log_queue;
	
  /* create queue for logging purged blocks */
  log_queue = queue_new();
  return (log_queue);
}

void 
free_msg (Message *msg);

/* When counting is done,
   remove the queue and free the memory.
*/
void 
end_count_purge (Queue *queue)
{
  /* free each element on the queue */
  QUEUE_RESET (queue, free_msg );
  /* free queue structure */
  xfree (queue );
}

/* Allocate memory and copy data from given one */
/* Duplicate a message object */
Message *
dup_message (Message *msg)
{
  Message *dup;
  int size;

  size = sizeof( Message);
  dup = xmalloc( size);
  memcpy (dup, msg, size);
  
  return (dup);
}

/* free the message object */
void 
free_msg (Message *msg)
{
  xfree (msg);
}

/* Check if msg is already on the queue,
   if yes ,just return 0.
   if not, log to the queue and return 1.
*/
int 
log_purged (Queue *queue, Message *message)
{
  Message *log;
  QueueElement *ptr;

  ptr = queue->queue.next;
  while (ptr != NULL)
    {
      log = (Message *)ptr->value;
      if (messageid_compare (&log->id, &message->id) == 0 ) 
	break;
      ptr = ptr->next;
    }
  if (ptr == NULL)
    {
      /* msg is not on the queue, log into and return 1 */
      queue_insert (queue, (void *)dup_message (message));
      return (1);
    }
  else 
    {
      return (0);
    }
}

/* handle counting when giving a block wanna purged */
void 
count_purge_a_block (Queue *p_log, DataBlock *block)
{
  Message *msg;

  /* We only check header field to tell difference */
  msg = &(block->header);
  if (log_purged (p_log, msg))
    {
      switch( msg->type ) 
	{
	case 0:	/* TYPE_BANDWIDTH */
	case 1: 	/* TYPE_TOPOLOGY */
	case 2:	/* TYPE_ESTIMATES */
	case 3:	/* TYPE_DATA */
	case 4:	/* TYPE_FEDEX */
	case 5:	/* TYPE_JOIN */
	  /* increase the block counter by 1 */
	  count_purge_block (msg->type, 1 );
	  /* increase the byte counter by size of data */
	  count_purge_byte (msg->type, block->length );
	  break;
	}
    }
}

/* handle counting while giving a whole queue wanna purged */
void 
count_purge_queue (Queue *p_log, Queue *qu)
{
  DataBlock *block;
  QueueElement *ptr;

  ptr = qu->queue.next;
  while (ptr != NULL)
    {
      block = (DataBlock *)ptr->value;
      count_purge_a_block( p_log, block );
      ptr = ptr->next;
    }
}

/* end   of Adding by erhyuan */
#endif /* STATISTICS */

/* Do our best to purge datablock.
 * realclean indicates the severity of the purge
 * 0 - delete things on the down datasites, and hold queues.
 * 1 - delete every datablock on every queue that is not currently
 *     being sent.
 */

purge_datablocks (int realclean)
{
  int i;
#ifdef STATISTICS
  Queue* log_purged;

  log_purged = init_count_purge();
#endif /* STATISTICS */
  /* Go through the site list */
  for (i = 0; sites && sites[i]; i++)
    {
      /* clean out the hold queue */
      if (sites[i] == whoami)
	{
#ifdef STATISTICS
	count_purge_queue (log_purged, sites[i]->datablocks_to_send);
#endif /* STATISTICS */
	QUEUE_RESET (sites[i]->datablocks_to_send, datablock_free);
      }
      /* clean out down sites */
      else if (sites[i]->status == STATUS_DOWN) 
	{
#ifdef STATISTICS
	count_purge_queue (log_purged, sites[i]->datablocks_to_send);
#endif /* STATISTICS */
	QUEUE_RESET (sites[i]->datablocks_to_send, datablock_free);
      }
      else if (realclean)
	{
	  /* if the site is not sending, we can remove
	   * everything from the send queue.
	   * Otherwise, remove everthing but the first
	   * datablock.
	   */
	  if (sites[i]->sender_fd == NOTSET)
	    {
#ifdef STATISTICS
	      count_purge_queue (log_purged, sites[i]->datablocks_to_send);
#endif /* STATISTICS */
	      QUEUE_RESET (sites[i]->datablocks_to_send, datablock_free);
	    }
	  else
	    {
	      DataBlock *head;
	      if (!queue_empty (sites[i]->datablocks_to_send))
		{
		  head = queue_remove (sites[i]->datablocks_to_send);
#ifdef STATISTICS
		  count_purge_queue (log_purged, sites[i]->datablocks_to_send);
#endif /* STATISTICS */
		  QUEUE_RESET (sites[i]->datablocks_to_send, datablock_free);
		  queue_insert (sites[i]->datablocks_to_send, head);
		}
	    }
	}
    }
#ifdef STATISTICS
    end_count_purge (log_purged);
#endif /* STATISTICS */
}

/* Purge any sites that have been down too long */
purge_sites ()
{
  SiteInfo *site;
  struct timeval current;
  int purge_time;
  int i, j;
  int dont_delete;

  current = timeval_current ();

  purge_time = current.tv_sec - site_purge_period;

  for (i = 0; sites[i]; i++)
    {
      if (sites[i] != whoami && sites[i]->last_heard_from < purge_time)
	{
	  /* Make sure the down site is not a group master */
	  dont_delete = 0;
	  for (j = 0; groups[j] != NULL; j++)
	    if (groups[j]->master_site == sites[i])
	      {
		dont_delete = 1;
		break;
	      }

	  if  (dont_delete)
	    continue;

	  /* We should only purge this site if it is down and no active
	   * sender exists 
	   */
	  if (sites[i]->status == STATUS_DOWN && sites[i]->sender_fd == NOTSET)
	    {
	      Statistics *stat;

	      /* First, delete the site from any groups thay may exist */
	      for (j = 0; groups[j] != NULL; j++)
		group_site_remove (groups[j], sites[i]);

	      /* Remove sites from estimates list */

	      stat = 
		LIST_FIND (whoami->estimates, &sites[i]->id, statistics_by_id);

	      if (stat != NULL)
		{
		  list_delete (whoami->estimates, stat);
		  statistics_free (stat);
		}

	      site = list_delete (site_list, sites[i]);
	      siteinfo_free (site);
	      /* We know we deleted it from the list so we know the list
	       * is one shorter, so...
	       */
	      i--;
	    }
	}
    }
}


void
purge_logs ()
{
  int i;
  extern MessageLog *received_log;

  if (received_log == NULL)
    received_log = messagelog_new ();
  log_purge (received_log);
  for (i = 0; sites[i] != NULL; i++)
    log_purge (sites[i]->acks);
}


/* Send a datablock to the site given by the siteid. 
 * If the siteid is null, it should be 
 * be a neighbor.
 */
int
send_to_siteid (SiteID *siteid, DataBlock *datablock)
{
  Group *group;
  SiteInfo *site = NULL;

  if (siteid != (SiteID *) -1 && siteid != (SiteID *)NULL)
    site = site_by_id (siteid);
  else
    {
      /* Pick A neighbor at random - it would really be more interesting to 
       * pick a neighbor that is guaranteed to be uniq in the group (insofar
       * as the last topology update is concerned.
       * The next question is which group to use.  Lets pick a group at 
       * random also. 
       * 
       */
      SiteID *id;
      group = groups[random () % group_list->count];
      if (group->neighbor_list->count > 0)
	{
	  id = group->neighbors[random () % group->neighbor_list->count];
	  site = site_by_id (id);
	}
    }
  
  if (site == NULL)
    return (-1);
  send_queue_insert (site, datablock);
  return (0);
}

