/* $Id: messagelog.c,v 1.1 1995/04/22 22:20:27 dante Exp $ */
/* Manipulate the message logs */
#include <stdio.h>
#include <floodd.h>

/* Some comments:
 *    We are logging incomming messageID's from various senders.  Each log
 *    corresponds to a particular sender.  We expect the the message id's 
 *    from a particular site to be roughly ordered hence we can use a linked
 *    list to represent the logs.  The time for a sequential search
 *    of a log should be relatively short since we either find the message
 *    id at the head of the list, or we find a message id with an order id.
 *    I would expect the usual case to be to find a message at the head of
 *    the log, or find an older message at the head.
 *    
 */

int log_size_in_bytes;	    	/* keep track of how much space we consume */
int log_size_in_messages;	/* keep track of the number of messages */

extern int log_purge_period;

/* A stupid little message log structure for the moment.  This should
 * problably be made into a hash table or something.
 */
MessageLog *received_log = NULL;

LogEntry *
logentry_create (MessageID *id)
{
  LogEntry *entry;

  entry = xmalloc (sizeof (LogEntry));

  entry->id = *id;
  entry->time_received = timeval_current ();
  entry->next = NULL;

  log_size_in_bytes += sizeof (*entry);
  log_size_in_messages++;

  return (entry);
}

void
logentry_delete (LogEntry *entry)
{
  log_size_in_bytes -= sizeof (*entry);
  log_size_in_messages--;

  xfree (entry);
}

MessageLog *
messagelog_new ()
{
  MessageLog *log;

  log = xmalloc (sizeof (MessageLog));
  log->logentries = NULL;
  return (log);
}

void
messagelog_insert_messageid (MessageLog *log, MessageID *id)
{
  LogEntry *newentry;
  LogEntry *entry;

  newentry = logentry_create (id);

  if (log->logentries == NULL)
    {
      log->logentries = newentry;
      return;
    }
  
  /* Check first to see if we need to be stuffed into to the head of the 
   * log
   */
  if (log->logentries == NULL || 
      timeval_compare (&id->time, &log->logentries->id.time) > 0)
    {
      newentry->next = log->logentries;
      log->logentries = newentry;
      return;
    }
  
  for (entry = log->logentries; entry->next != NULL; entry = entry->next)
    {
      if (timeval_compare (&id->time, &entry->next->id.time) > 0)
	break;
    }
  /* We have found where to insert so now do it */
  newentry->next = entry->next;
  entry->next = newentry;

  return;
}

LogEntry *
messagelog_find_messageid (MessageLog *log, MessageID *id)
{
  LogEntry *entry;
  int result;

  if (log->logentries == NULL)
    return (NULL);

  for (entry = log->logentries; entry != NULL; entry = entry->next)
    {
      result = timeval_compare (&id->time, &entry->id.time);
      if (result < 0)
	continue;
      if (result == 0)
	return (entry);

      /* Since the list is ordered, if we get this far we have failed in
       * in our search.
       */
      break;
    }
  return (NULL);
}


void
messagelog_log (MessageLog *log, MessageID *id)
{
  if (messagelog_find_messageid (log, id) == NULL)
    messagelog_insert_messageid (log, id);
}

void
log_message (MessageID *id)
{
  if (received_log == NULL)
    received_log = messagelog_new ();
  messagelog_log (received_log, id);
}

LogEntry *
logged_message (MessageID *id)
{
  int i;

  if (received_log == NULL)
    received_log = messagelog_new ();

  return (messagelog_find_messageid (received_log, id));
}

void
messagelog_delete_rest (LogEntry *entry)
{
  LogEntry *next;
  LogEntry *current;

  next = entry->next;
  entry->next = NULL;

  while (next != NULL)
    {
      current = next;
      next = current->next;
      logentry_delete (current);
    }
}

/* Remove old entries from the log.  Anything older than log_purge_period is 
 * purged
 */
void
log_purge (MessageLog *log)
{
  LogEntry *entry;
  struct timeval purge_time;

  purge_time = timeval_current ();
  purge_time.tv_sec = purge_time.tv_sec - log_purge_period;

  if (log->logentries == NULL)
    return;

  for (entry = log->logentries; entry != NULL && entry->next != NULL; 
       entry = entry->next)
    {
      if (timeval_compare (&purge_time, &entry->next->time_received) > 0)
	{
	  messagelog_delete_rest (entry);
	  break;
	}
    }

  if (timeval_compare (&purge_time, &log->logentries->time_received) > 0)
    {
      logentry_delete (log->logentries);
      log->logentries = NULL;
    }
}
