/* $Id: io.c,v 1.4 1995/07/25 20:06:41 dante Exp dante $ */
#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <fcntl.h>
#if defined (HAVE_SYSLOG_H)
#  include <syslog.h>
#endif /* !SYS_SYSLOG */
#include <floodd.h>
#include <io-handler.h>

extern int debug;
extern SiteInfo *whoami;
extern int data_socket;
extern int client_socket;
extern int ping_socket;
extern Queue *export_queue;
extern int export_fd;

int export_pid = NOTSET;	/* keep track of the export pid so we 
				 * know when we are done
				 */


/* A place for io to wait until memory is free. */
Queue *datablock_memory_queue = NULL;


void
compute_bandwidth (SiteID *siteid, struct timeval *start_time, int length);

/* Various IO handlers */

typedef struct
{
  
  SiteID siteid;
  DataBlock *datablock;
  int bytes;
  struct timeval start_time;
} ConnectState;

ConnectState *
connect_state_new (SiteID siteid, DataBlock *datablock)
{
  ConnectState *state;

  state = xmalloc (sizeof (ConnectState));

  state->siteid = siteid;
  state->datablock = datablock;
  state->bytes = 0;
  
  state->start_time.tv_sec  = -1;
  state->start_time.tv_usec = -1;
  return (state);
}

void
connect_state_free (ConnectState *state)
{
  if (state->datablock != NULL)
    datablock_free (state->datablock);
  xfree (state);
}

/* Start sending datablocks on a connection.
 * We return non-zero only when we have no more datablock to read or 
 * an error occurs.
 * We have to be careful when we return so that we don't leave the
 * state->datablock assigned to the datablock value that we pull off of 
 * the datablocks_to_send_queue.  If we return an error the 
 * data_connection_cleanup function weill be called and free the datablock
 * even though it is still on the send queue.  The next time we try and 
 * free it, it might have already been freed.
 */

int
data_connection_write (int fd, ConnectState *state)
{
  SiteInfo *site = NULL;
  int result = 0;
  int n = -1;

  site = site_by_id (&state->siteid);
  if (site == NULL)
    {
      result = -1;
      goto error;
    }

  /* See if we need to grab a datablock to send */
  if (state->datablock == NULL)
    {
      if (queue_empty (site->datablocks_to_send))
	return (-1);

      state->datablock = queue_head (site->datablocks_to_send);
      datablock_reference (state->datablock);
      state->bytes = 0;

      if (debug)
	syslog (LOG_DEBUG, 
		"DATA CONNECTION WRITE(%d): sending datablock 0x%x of %d bytes \n", 
		fd, state->datablock, state->datablock->length);
    }

  /* make sure we haven't just received an ack for this datablock */
  if (messagelog_find_messageid (site->acks,
				 &state->datablock->header.id) != NULL)
    {
      if (queue_head (site->datablocks_to_send) == state->datablock)
	{
	  queue_remove (site->datablocks_to_send);
	  /* deref w.r.t. the queue */
	  datablock_free (state->datablock);
	}
      result = -1;
      goto error;
    }

  if (state->start_time.tv_sec <= 0)
    state->start_time = timeval_current ();

  n = datablock_write (state->datablock, fd, &state->bytes, 0);

  /* Return if we are not done */
  if (n == 0)
    return (0);

  if (n == -1)
    {
      if (debug)
	syslog (LOG_DEBUG, "DATA CONNECTION WRITE(%d): ERROR\n", fd);
      
      if (site == NULL)
	site = site_by_id (&state->siteid);

      if (site != NULL)
	site->status = STATUS_DOWN;

      result = -1;
      goto error;
    }

  if (debug)
    syslog (LOG_DEBUG, "DATA CONNECTION WRITE(%d): Done. %d bytes\n", fd,
	     state->bytes);

  /* Collect bandwidth statistics on outgoing datablock */
  compute_bandwidth (&state->siteid, &state->start_time, state->bytes);

  /* We have successfully sent a datablock.  Remove it from the queue
   * and "free" it.
   */

  site = site_by_id (&state->siteid);
  if (site == NULL)
    {
      result = -1;
      goto error;
    }

#ifdef STATISTICS
  /* count total blocks sent out */
  count_sendout_block (1);
  count_sendout_byte (state->bytes);
  switch (state->datablock->header.type)
    {
    case TYPE_ESTIMATES:
      count_estimates_bytes_sent (state->bytes);
      break;
    case TYPE_BANDWIDTH:
      count_bandwidth_bytes_sent (state->bytes);
      break;
    }

#endif /* STATISTICS */

  /* make sure the datablock we are sending is at the head of the queue */
  {
    DataBlock *block;
    block = queue_head (site->datablocks_to_send);
    if (block == state->datablock)
      {
	queue_remove (site->datablocks_to_send);
	/* This removes the reference for the datablock wrt the queue */
	datablock_free (block);
      }
  }

  result = state->bytes;

 error:
  if (state->datablock != NULL)
    {
      datablock_free (state->datablock);
      state->datablock = NULL;
    }

  state->bytes = 0;
  if (result == -1)
    return (-1);
  else
    return (0);
}

/* Just read an incomming message and toss it */
int
devnull (int fd, void *state)
{
  char buffer[1024];
  int n;

  n = read (fd, buffer, sizeof (buffer));
  if (n == -1 && (errno != 0 && errno != EWOULDBLOCK))
    {
      if (debug)
	syslog (LOG_INFO, "devnull().read(): %m");
      return (-1);
    }
  return (0);
}

#ifdef STATISTICS
/* Start of adding by erhyuan */
/* oper : < 0 for reset, 0 for return counter, 1 for increased by 1 */
/* counting total blocks sent out */
int
count_export (int oper)
{
  static int counter;
  
  if (oper < 0 ) 
    counter = 0;
  else 
    counter += oper;

  return (counter);
}

int
count_sendout_byte (int oper)
{
  static int counter;
  
  if (oper < 0 ) 
    counter = 0;
  else 
    counter += oper;

  return (counter);
}

int
count_sendout_block (int oper)
{
  static int counter;
  
  if (oper < 0 ) 
    counter = 0;
  else 
    counter += oper;

  return (counter);
}

/* oper : < 0 for reset, 0 for return counter, 1 for increased by 1 */
/* counting duplicate blocks received */
int 
count_duplicate_byte (int oper)
{
  static int counter;

  if (oper < 0) 
    counter = 0;
  else 
    counter += oper;

  return( counter );	
}

int 
count_duplicate_block (int oper)
{
  static int counter;

  if (oper < 0) 
    counter = 0;
  else 
    counter += oper;

  return( counter );	
}

/* oper : < 0 for reset, 0 for return counter, 1 for increased by 1 */
/* counting total blocks received */
int 
count_received_byte( int oper )
{
  static int counter;

  if (oper < 0)
    counter = 0;
  else 
    counter += oper;
  return (counter);
}

int 
count_received_block( int oper )
{
  static int counter;

  if (oper < 0)
    counter = 0;
  else 
    counter += oper;
  return (counter);
}

int 
count_estimates_bytes_received( int oper )
{
  static int counter;

  if (oper < 0)
    counter = 0;
  else 
    counter += oper;
  return (counter);
}

int 
count_estimates_bytes_sent( int oper )
{
  static int counter;

  if (oper < 0)
    counter = 0;
  else 
    counter += oper;
  return (counter);
}

int 
count_bandwidth_bytes_sent( int oper )
{
  static int counter;

  if (oper < 0)
    counter = 0;
  else 
    counter += oper;
  return (counter);
}

int 
count_bandwidth_bytes_received( int oper )
{
  static int counter;

  if (oper < 0)
    counter = 0;
  else 
    counter += oper;
  return (counter);
}

int 
count_ping_bytes_sent (int oper)
{
  static int counter;

  if (oper < 0)
    counter = 0;
  else 
    counter += oper;
  return (counter);
}

/* End   of adding by erhyuan */
/* End   of adding by erhyuan */
#endif /* STATISTICS */

/* Receive incomming data.  We place the data on the floodd export queue
 * to be passed to the local application and then flood the data back
 * out to other sites.
 */

int 
data_connection_read (int fd, ConnectState *state)
{
  int n;

  if (state->bytes <= 0)
    state->start_time = timeval_current ();

  n = datablock_read (&state->datablock, fd, &state->bytes);
  
  if (n == 0)
    return (0);

  if (n == -1)
    {
      if (debug)
	{
	  if (state->datablock == NULL)
	    syslog (LOG_DEBUG, "Duplicate datablock cancelled?\n");
	  else if (state->datablock != NULL)
	    syslog (LOG_DEBUG, 
		    "DATA CONNECTION READ(%d): Error receiving datablock.\n", 
		    fd);
	}
      if (state->datablock != NULL)
	{
	  datablock_free (state->datablock);
	  state->datablock = NULL;
	}
      return(-1);
    }
  
  if (debug)
    syslog (LOG_DEBUG, 
	     "DATA CONNECTION READ(%d): Done. %d bytes read\n", fd,
	     state->bytes);
  
  /* Collect bandwidth statistics on incomming datablock */
  
  compute_bandwidth (&state->datablock->header.sender, &state->start_time, 
		     state->bytes);

#ifdef STATISTICS
  count_received_block ( 1 );
  count_received_byte ( state->bytes );

  switch (state->datablock->header.type)
    {
    case TYPE_ESTIMATES:
      count_estimates_bytes_received (state->datablock->header.data.length);
      break;
    case TYPE_BANDWIDTH:
      count_bandwidth_bytes_received (state->datablock->header.data.length);
      break;
    }

#endif /* STATISTICS */
  /* If we haven't received this datablock before, then deal with it */
  if (!logged_message (&state->datablock->header.id)) 
    {
      dispatch_datablock (state->datablock);
    }
  else
    {
#ifdef STATISTICS
      count_duplicate_block ( 1 );
      count_duplicate_byte ( state->bytes );
#endif /* STATISTICS */
      if (debug)
	printf ("Duplicate Datablock: (%s:%d:%d)\n",
		siteid_to_string (&state->datablock->header.id.site),
		ntohl (state->datablock->header.id.time.tv_sec),
		ntohl (state->datablock->header.id.time.tv_usec));
    }
  datablock_free (state->datablock);
  state->datablock = NULL;
  state->bytes = 0;
  state->start_time.tv_sec = -1;
  state->start_time.tv_usec = -1;
  return (0);
}

void
data_connection_cleanup (ConnectState *state)
{
  SiteInfo *site;

  site = site_by_id (&state->siteid);

  if (site != NULL)
    site->sender_fd = NOTSET;
  
  connect_state_free (state);
}

int
connect_site (SiteInfo *site, int port)
{
  int sock;
  struct linger linger;
  struct sockaddr_in address;

  linger.l_onoff = 1;
  linger.l_linger = 60;

  sock = socket (AF_INET, SOCK_STREAM, 0);
  setsockopt (sock, SOL_SOCKET, SO_LINGER, (char *) &linger, sizeof (linger));

  /* Set the file descriptor to be non-blocking ... */

#if defined (O_NONBLOCK)
  fcntl (sock, F_SETFL, O_NONBLOCK);
#else
#if defined (O_NDELAY)
  fcntl (sock, F_SETFL, O_NDELAY);
#endif
#endif
  memcpy (&address, site->address, sizeof (address));

  if (port > 0)
    address.sin_port = htons(port);
  
  if (connect (sock, (struct sockaddr *)&address, sizeof (address)) < 0)
    if (errno != EINPROGRESS)
      {
	close (sock);
	return (-1);
      }

  return (sock);
}

/* Establish a connection - open a connection to a given site.
 * return a file descriptor for the socket or -1 on error.
 * If the port is zero, connectto the address given in the 
 * site->address, otherwise, connect the same internet address, but
 * a different port.
 */

/* Start up a connection to do stuff */
void
connection_write_start (SiteInfo *site)
{
  int sock;

  sock = connect_site (site, site->data_port);

  if (sock == -1)
    {
      syslog (LOG_ERR, "connection_write_start(): %m");
      return;
    }

  if (debug)
    syslog (LOG_DEBUG, "DATA CONNECTION WRITE(%d): Starting...\n", sock);

  site->sender_fd = sock;
  /* Now fire up an io function on this socket */
  IO_start (sock,
	    devnull, NULL, NULL, /* read stuff */
	    data_connection_write, 
	    data_connection_cleanup,
	    connect_state_new (site->id, NULL),
	    NULL);
}

/*
 * Implement the export interface 
 */

/* Things are a little weird here.
 * Since we are forking and execing our export routine, we need to be
 * careful when we start up a new one.  We should not start up a new export
 * routine, until the old one has terminated.
 * 
 * To ensure this, we need to keep track of the export routine, and 
 * wait for a sigchild to come in.  When the sigchild comes in, we 
 * Stuff a dummy export_terminate routine in the export_io handler.
 * The export_terminate routine need to check to if another export
 * routine needs to be started up.
 */

/* Handle the exporting of datablock to a client */
export_write (int fd, ConnectState *state)
{
  int n;

  if (state->datablock == NULL)
    {
      return (-1);
    }

  if (state->bytes == 0)
    {
      if (debug)
	printf ("EXPORT Start. %d bytes. Datablock: (%s:d:%d)\n",
		state->datablock->length,
		siteid_to_string (&state->datablock->header.id.site),
		ntohl (state->datablock->header.id.time.tv_sec),
		ntohl (state->datablock->header.id.time.tv_usec));
    }
  /* write a datablock with just the data (the last argument is a flag 
   * that tell it to just send the data) 
   */
  n = datablock_write (state->datablock, fd, &state->bytes, 1);

  if (debug && n == -1)
    syslog (LOG_DEBUG, "EXPORT WRITE(%d): ERROR\n", fd);

#ifdef NEVER  
  if (debug && n >= 0)
    syslog (LOG_DEBUG, "EXPORT WRITE(%d): %d bytes sent\n",
	     fd, state->bytes);
#endif

  if (n == -1)
    {
      state->datablock = NULL;
      return (-1);
    }
  
  /* Return if we are not done */
  if (n == 0) 
    return (0);
  
  if (debug)
    printf ("EXPORT WRITE(%d): Done. %d bytes. Datablock: (%s:%d:%d)\n", fd,
	    state->bytes,
	    siteid_to_string (&state->datablock->header.id.site),
	    ntohl (state->datablock->header.id.time.tv_sec),
	    ntohl (state->datablock->header.id.time.tv_usec));
  
  /* Remove the datablock from the export queue and "free" it*/
  queue_remove (export_queue);

  /* This removes our reference to the datablock */
  datablock_free (state->datablock);

  state->bytes = 0;
  state->datablock = queue_head (export_queue);
  if (state->datablock == NULL)
    return (-1);
  
  return (0);
}

export_cleanup (ConnectState *state)
{
  /* We don't want to free the datablock here */
  state->datablock = NULL;
  data_connection_cleanup (state);
}

#if !defined (LOCAL)

export_start ()
{
  int sockets[2];
  extern char *export_command;
  SiteID nullid;
  int result;
  struct linger linger;

  /* First make sure that we are still really waiting for a child */
  sig_child ();

  if (export_fd != NOTSET)
    return;

  /* if the export command is not set, just remove everthing from the
   * export queue.
   */
  if (export_command == NULL)
    {
      QUEUE_RESET (export_queue, datablock_free);
      return;
    }

  if (queue_empty (export_queue))
    return;
      
  result = socketpair (AF_UNIX, SOCK_STREAM, 0, sockets);

  if (result == -1)
    {
      syslog (LOG_ERR, "export_start: %m");
      return;
    }

  memset (&nullid, 0, sizeof (nullid));
  export_fd = sockets[0];

#ifdef STATISTICS
      count_export(1);
#endif /* STATISTICS */

  export_pid = fork ();
  
  if (export_pid == 0)
    {
      int i;
      int max;
      /* child */

      /* Close every file descriptor */
      max = getdtablesize ();
      for (i = 0; i < max; i++)
	if (i != sockets[1])
	  close (i);

      dup2(sockets[1], 0);
      if (!debug)
	{
	  int fd;
	  fd = open ("/dev/null", O_WRONLY);
	  /* Set stdout/stderr to be /dev/null */
	  dup2 (fd, 1);
	  dup2 (fd, 2);
	}
      else
	{
	  int fd;
	  fd = open ("/dev/tty", O_WRONLY);
	  dup2 (fd, 1);
	  dup2 (fd, 2);
	}

      /* Fire up the export command */
      execl ("/bin/sh", "sh", "-c", export_command, NULL);
      /* If we return here, an error must have occurred */
      
      syslog (LOG_ERR, "Export command failed: %m");
      exit (1);
    }

  if (export_pid == -1)
    {
      export_pid = NOTSET;
      export_fd = NOTSET;
      syslog (LOG_ERR, "Could not fork: %m\n");
      close (sockets[0]);
      close (sockets[1]);
      return;
    }

  linger.l_onoff = 1;
  linger.l_linger = 1000;

  setsockopt (export_fd, SOL_SOCKET, SO_LINGER, (char *) &linger, sizeof (linger));

  /* close the other end of the socket pair */
  close (sockets[1]);

  /* parent */
  fcntl (sockets[0], F_SETFL, O_NONBLOCK);
  
  /* Now fire up an io function on this socket */
  IO_start (sockets[0],
	    devnull, NULL, NULL, /* read stuff */
	    export_write, 
	    export_cleanup,
	    connect_state_new (nullid, queue_head (export_queue)),
	    NULL);

  if (debug)
    syslog (LOG_DEBUG, "Export (%d): Starting...\n", sockets[0]);
}
#endif

#ifdef LOCAL
  
export_start ()
{
  extern char *export_command;
  SiteID nullid;
  int result;

  memset (&nullid, 0, sizeof (nullid));

  /* First make sure that we are still really waiting for a child */
  sig_child ();

  if (export_fd != NOTSET)
    return;

  /* if the export command is not set, just remove everthing from the
   * export queue.
   */
  if (export_command == NULL)
    {
      QUEUE_RESET (export_queue, datablock_free);
      return;
    }

  if (queue_empty (export_queue))
    return;

  fd = child_start (command, &export_pid);

  if (fd == -1)
    return;
}

/* returns file descriptor */
int 
child_start (char *command, &child_pid)
{
  int sockets[2];
  int i;
  int max;
  int result;
  int child_fd;
  struct linger linger;

  result = socketpair (AF_UNIX, SOCK_STREAM, 0, sockets);

  if (result == -1)
    {
      syslog (LOG_ERR, "child_start `%s': %m", command);
      return (-1);
    }
  
  child_fd = sockets[0];

  *child_pid = fork ();

  if (*child_pid == 0)
    {
      /* child */
      
      /* Close every file descriptor */
      max = getdtablesize ();
      for (i = 0; i < max; i++)
	if (i != sockets[1])
	  close (i);
      
      /* Setup up input and and output */
      dup2(sockets[1], 0);
      dup2(sockets[1], 1);
      if (!debug)
	{
	  int fd;
	  fd = open ("/dev/null", O_WRONLY);
	  /* Set stderr to be /dev/null */
	  dup2 (fd, 2);
	}
      else
	{
	  int fd;
	  fd = open ("/dev/tty", O_WRONLY);
	  dup2 (fd, 2);
	}

      /* Fire up the export command */
      execl ("/bin/sh", "sh", "-c", command, NULL);
      /* If we return here, an error must have occurred */
    }

  if (child_pid == -1)
    {
      close (sockets[0]);
      close (sockets[1]);
      return (-1);
    }

  linger.l_onoff = 1;
  linger.l_linger = 1000;

  setsockopt (child_fd, SOL_SOCKET, SO_LINGER, &linger, sizeof (linger));

  /* close the other end of the socket pair */
  close (sockets[1]);

  /* parent */
  fcntl (child_fd, F_SETFL, O_NONBLOCK);
  return (child_fd);
}
#endif

/* Handle ping requests */
int
ping_accept (int fd, void *state)
{  
  struct sockaddr_in from;
  int from_length;
  struct timeval time, current_time;
  Message message;
  Message raw;
  SiteInfo *who;
  Statistics *stat;
  int length;
  
  from_length = sizeof (struct sockaddr_in);

  length = 
    recvfrom (fd, (char *) &raw, sizeof (Message), 0, 
	      (struct sockaddr *) &from, &from_length);
  
  message = raw;
  message.type = ntohl (raw.type);

  if (length <= 0)
    {
      if (debug > 2)
	syslog (LOG_DEBUG, "ping_request().recvfrom: %m");
      return (0);
    }
  
  who = site_by_id (&message.sender);

  if (who == NULL)
    {
      if (debug)
	syslog (LOG_DEBUG, "Received data from Unknown site\n");
      
      send_unknown_reply (&message.sender);
      return (0);
    }
  
  if (length != sizeof (Message))
    {
      if (debug)
	syslog (LOG_DEBUG, "Receive some unkown type of datagram message\n");
      return (0);
    }

  siteinfo_up (who);

  switch (message.type)
    {
    case TYPE_PING:	      /* Received a ping message from someone */
      message.type = htonl (TYPE_PING_REPLY);
      message.sender = whoami->id;
      site_send_message (who, &message, sizeof (Message));
      break;

    case TYPE_PING_REPLY:
      current_time = timeval_current ();
      time = timeval_subtract (current_time, message.id.time);
      
      stat = LIST_FIND (whoami->estimates, &who->id, statistics_by_id);
      if (stat == NULL)
	{
	  stat = statistics_new (who->id);
	  list_insert (whoami->estimates, stat);
	}
      /* Ignore really weird numbers */
      if (time.tv_sec < 5)
	{
	  statistics_update_rtt (stat, time);
	  if (debug)
	    printf ("Ping Result from `%s': %d.%d seconds\n",
		    who->name, time.tv_sec, time.tv_usec);
	}
      break;

    case TYPE_ACK:
      siteinfo_receive_ack (who, &message.id);
      break;

      /* The site did not know who we were, hence we probably want to 
       * issue a join.
       */
    case TYPE_UNKNOWN:
      {
	int i, j;
	/* Go through all groups and find the site */
	for (i = 0; groups[i] != NULL; i++)
	  for (j = 0; groups[i]->sites[j] != NULL; j++)
	    if (who == groups[i]->sites[j])
	      {
		if (groups[i]->join_event != NULL)
		  reschedule_now (groups[i]->join_event);
		break;
	      }
      }
      break;

    default:
      if (debug)
	syslog (LOG_DEBUG, "Unknown message type %d\n", message.type);
      break;
    }
  return (0);
}

/* Accept new data connections */
int
data_connect (int fd, void *state)
{
  int new_fd;
  struct sockaddr_in from;
  int from_length;
  SiteID nullsiteid;  
  from_length = sizeof (from);

  memset (&nullsiteid, 0, sizeof (nullsiteid));
  new_fd = accept (fd, (struct sockaddr *) &from, &from_length);

  if (new_fd == -1)
    {
      if (debug)
	syslog (LOG_DEBUG, "data_connect().accept(): %m");
      return (0);
    }

  fcntl (new_fd, F_SETFL, O_NONBLOCK);

  if (debug)
    syslog (LOG_DEBUG, "DATA CONNECTION READ(%d): Starting...", new_fd);


  /* Start IO */
  IO_start (new_fd, 
	    data_connection_read, connect_state_free,
	    connect_state_new (nullsiteid, NULL),
	    NULL, NULL,
	    NULL,
	    NULL);
  return (0);
}

/* Accept a new client connection */
int
client_connect (int fd, void *state)
{  
  int new_fd;
  struct sockaddr_in from;
  int from_length;
  extern int client_connection_read ();
  extern int client_state_free ();

  from_length = sizeof (from);
  new_fd = accept (client_socket, (struct sockaddr *) &from, &from_length);

  if (new_fd == -1)
    return (0);


#if defined (O_NONBLOCK)
  fcntl (new_fd, F_SETFL, O_NONBLOCK);
#else
#if defined (O_NDELAY)
  fcntl (new_fd, F_SETFL, O_NDELAY);
#endif
#endif

  IO_start (new_fd, 
	    client_connection_read,
	    client_state_free,
	    client_state_new (&from, from_length),
	    NULL, NULL, NULL,
	    NULL);
  return (0);
}

sig_child ()
{
  int result;

  result = waitpid (export_pid, NULL, WNOHANG);
  
  if (result == 0)
    return;

  if (result == -1 && errno == EINTR)
    return;
    
  export_pid = NOTSET;
  export_fd = NOTSET;
}

/* set up the basic io mechanism
 * This consists of handlers for data connections, ping connections,
 * and client connections.
 */

io_initialize ()
{
  datablock_memory_queue = queue_new ();

  /* Start up blocking IO on stdout, stderr */
  IO_start (1,
	    NULL, NULL, NULL,
	    NULL, NULL, NULL,
	    NULL);

  IO_start (2,
	    NULL, NULL, NULL,
	    NULL, NULL, NULL,
	    NULL);
  
  IO_start (data_socket, 
	    data_connect, NULL, NULL,
	    NULL, NULL, NULL,
	    NULL);

  IO_start (client_socket,
	    client_connect, NULL, NULL, 
	    NULL, NULL, NULL,
	    NULL);

  IO_start (ping_socket, 
	    ping_accept, NULL, NULL,
	    NULL, NULL, NULL,
	    NULL);

  signal (SIGCHLD, (void *) sig_child);
}

/*
 * Compute bandwidth for a datablock exchange (either incomming or outgoing)
 */

void
compute_bandwidth (SiteID *siteid, struct timeval *start_time, int length)
{
  SiteInfo *site;
  /* Determine bandwidth only if datablock was larger than  16 K
   * This is an arbitrary number.  Should be more intelligent about
   * this.
   */

  if (length >= bandwidth_size)
    {
      Statistics *stat;
      struct timeval time;

      site = site_by_id (siteid);

      if (site == NULL)
	return;

      time = timeval_subtract (timeval_current (), *start_time);
      
      stat = LIST_FIND (whoami->estimates, &site->id, statistics_by_id);
      if (stat == NULL)
	{
	  stat = statistics_new (site->id);
	  list_insert (whoami->estimates, stat);
	}
    
      statistics_update_bandwidth (stat, length, time);
    }
}

/*
 * Return the total number of available file descriptors.
 *
 * On some systems, like 4.2BSD and its descendents, there is a system call
 * that returns the size of the descriptor table: getdtablesize().  There are
 * lots of ways to emulate this on non-BSD systems.
 *
 * On System V.3, this can be obtained via a call to ulimit:
 *	return (ulimit(4, 0L));
 *
 * On other System V systems, NOFILE is defined in /usr/include/sys/param.h
 * (this is what we assume below), so we can simply use it:
 *	return (NOFILE);
 *
 * On POSIX systems, there are specific functions for retrieving various
 * configuration parameters:
 *	return (sysconf(_SC_OPEN_MAX));
 *
 */

#if !defined (HAVE_GETDTABLESIZE)
int
getdtablesize ()
{
#  if defined (_POSIX_VERSION) && defined (_SC_OPEN_MAX)
  return (sysconf(_SC_OPEN_MAX));	/* Posix systems use sysconf */
#  else /* ! (_POSIX_VERSION && _SC_OPEN_MAX) */
#    if defined (USGr3)
  return (ulimit (4, 0L));	/* System V.3 systems use ulimit(4, 0L) */
#    else /* !USGr3 */
#      if defined (NOFILE)	/* Other systems use NOFILE */
  return (NOFILE);
#      else /* !NOFILE */
  return (20);			/* XXX - traditional value is 20 */
#      endif /* !NOFILE */
#    endif /* !USGr3 */
#  endif /* ! (_POSIX_VERSION && _SC_OPEN_MAX) */
}
#endif /* !HAVE_GETDTABLESIZE */
