/* $Id: config.c,v 1.4 1995/07/25 20:03:19 dante Exp $ */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netdb.h>
#include <floodd.h>
#include <queue.h>
#include <siteinfo.h>
#include <group.h>
#include <utility.h>
#include <dynamic_string.h>

#include <config.h>

/* Definitions for parsing options lines */



/* Functions to set siteinfo */

void
set_inetaddr (SiteInfo *site, char *value, int errordescriptor)
{
  if (isaddress (value))
    {
      if (site->inetaddr != NULL && strcmp (value, site->inetaddr) == 0)
	return;
      
      if (site->inetaddr)
	xfree (site->inetaddr);
      site->inetaddr = (char *) strdup (value);

      if (site->data_port != -1)
	siteinfo_set_address (site);
    }
  else
    {
      char buffer[1024];
      if (debug)
	{
	  sprintf (buffer, "Invalid inet address `%s' for %s:%s\n", value,
		   site->name, site->host);
	  client_output_string (errordescriptor, buffer);
	}
    }

}

void
set_host (SiteInfo *site, char *value, int errordescriptor)
{
  if (site->host != NULL && strcmp (site->host, value) == 0)
    return;

  if (site->host)
    xfree (site->host);

  site->host = (char *) strdup (value);

  if (gethostbyname (value) == NULL)
    {
      {
	char buffer[1024];
	if (debug)
	  {
	    sprintf (buffer, "Could not find host `%s' by name\n", value);
	    client_output_string (errordescriptor, buffer);
	  }
      }
      return;
    }

  /* Set up the site address if the data_port has been set and the 
   * inet_address field has not been set
   */
  
  if (site->data_port != -1 && site->inetaddr != NULL)
    siteinfo_set_address (site);

}

void
set_data_port (SiteInfo *site, char *value, int errordescriptor)
{
  int port;

  port = atoi (value);

  if (port == site->data_port)
    return;

  site->data_port = port;

  /* Set up the site address if the hostname has been set */
  if (site->host != NULL)
    siteinfo_set_address (site);
}

void
set_client_port (SiteInfo *site, char *value, int errordescriptor)
{
  site->client_port = atoi (value);
}

void
set_coordinates (SiteInfo *site, char *value, int errordescriptor)
{
  char *longitude;

  longitude = (char *) strchr (value, ',');
  if (longitude == NULL)
    {
      char buffer[512];
      sprintf (buffer, "Invalid coordinate `%s'\n", value);
      client_output_string (errordescriptor, buffer);
      return;
    }
  *longitude = '\0';
  longitude++;
  site->lattitude = atof (value);
  site->longitude = atof (longitude);
}

void
set_longitude (SiteInfo *site, char *value, int errordescriptor)
{
  float x;
  
  x = atof (value);
  if (fabs (x) > 180.0)
    {
      char buffer[128];
      sprintf (buffer, "Bad longitude value: `%s'\n", value);
      client_output_string (errordescriptor, buffer);
    }
  else
    site->longitude = x;
}

void
set_lattitude (SiteInfo *site, char *value, int errordescriptor)
{
  float x;
  
  x  = atof (value);
  if (fabs (x) > 90.0)
    {
      char buffer[128];
      sprintf (buffer, "Bad lattitude value: `%s'\n", value);
      client_output_string (errordescriptor, buffer);
    }
  else
    site->lattitude = x;
}

/* Group parameters */
void
group_site (Group *group, char *value, int errordescriptor)
{
  SiteInfo *site;

  site = site_by_name (value);
  
  if (site == NULL)
    {
      char buffer[128];
      sprintf (buffer, "Bad site `%s' name for group `%s'\n", 
	       value, group->name);
      client_output_string (errordescriptor, buffer);
    }
  else
    group_site_add (group, site);
}

void
set_topology_generator (Group *group, char *value, int errordescriptor)
{
  if (group->topology_generator)
    xfree (group->topology_generator);
 
  group->topology_generator = strdup (value);
}

void
set_connectivity (Group *group, char *value, int errordescriptor)
{
  int n;
  n = atoi (value);
  if (n < 2)
    {
      char buffer[128];
      sprintf (buffer, "Group `%s': connectivity must be > 1\n",
	       group->name);
      client_output_string (errordescriptor, buffer);
      return;
    }
  group->topology_connectivity = n;

}

void
set_ping_period (Group *group, char *value, int errordescriptor)
{
  int period;

  period = atof (value);

  if (period <= 0)
    {
      char buffer[128];
      sprintf (buffer, "Bad ping period `%s' for group `%s'\n",
	       value, group->name);
      client_output_string (errordescriptor, buffer);
      return;
    }
  group->ping_period = period;

}

void
set_bandwidth_period (Group *group, char *value, int errordescriptor)
{
  int period;

  period = atof (value);

  if (period <= 0)
    {
      char buffer[128];
      sprintf (buffer, "Bad bandwidth period `%s' for group `%s'\n",
	       value, group->name);
      client_output_string (errordescriptor, buffer);
      return;
    }

  group->bandwidth_period = period;
  if (group->join_period == NOTSET)
    group->join_period = group->bandwidth_period;
}

void
set_join_period (Group *group, char *value, int errordescriptor)
{
  int period;

  period = atoi (value);
  
  if (period <= 0)
    {
      char buffer[128];
      sprintf (buffer, "Bad join period `%s'\n",  value);
      client_output_string (errordescriptor, buffer);
      return;
    }
  group->join_period = period;
}

void
set_master (Group *group, char *value, int errordescriptor)
{
  SiteInfo *site;
  char buffer[128];

  site = site_by_name (value);

  if (site)
    {
      /* Add the site to the group even though it might have already
       * been added by a (:site ...) command.
       */
      group_site (group, value, errordescriptor);
      if (group->master_site != NULL)
	{
	  group->master_site = site;
	  group_initialize (group);
	}
      else
	group->master_site = site;
    }
  else
    {
      sprintf (buffer, "Cannot set master site. Bad site name `%s'\n", value);
      client_output_string (errordescriptor, buffer);
    }
}

void
set_update_period (Group *group, char *value, int errordescriptor)
{
  int period;
  period = atof (value);

  if (period <= 0)
    {
      char buffer[128];
      sprintf (buffer, "Bad update period `%s' for group `%s'\n",
	       value, group->name);
      client_output_string (errordescriptor, buffer);
      return;
    }
  group->update_period = period;
}

void
set_estimates_period (Group *group, char *value, int errordescriptor)
{
  int period;

  period = atof (value);

  if (period <= 0)
    {
      char buffer[128];
      sprintf (buffer, "Bad estimates period `%s' for group `%s'\n",
	       value, group->name);
      client_output_string (errordescriptor, buffer);
      return;
    }

  group->estimates_period = period;
}

void
set_allow_site (Group *group, char *value, int errordescriptor)
{
  extern int mask_equal (void *, void *);
  extern void *mask_new (char *value);
  void *mask;
  mask = mask_new (value);

  if (mask != NULL)
    if (list_find (group->allow_list, mask, mask_equal) == NULL)
      list_insert (group->allow_list, mask);
}

void
set_deny_site (Group *group, char *value, int errordescriptor)
{
  extern int mask_equal (void *, void *);
  extern void *mask_new (char *value);
  void *mask;

  mask = mask_new (value);

  if (mask != NULL)
    if (list_find (group->deny_list, mask, mask_equal) == NULL)
      list_insert (group->deny_list, mask);
}

/* set maximum group size */
void
set_max_size (Group *group, char *value, int errordescriptor)
{
  int size;

  size = atoi (value);

  if (size < 1)
    {
      char buffer[128];
      sprintf (buffer, "Bad maximum group size `%s' for group `%s'\n",
	       value, group->name);
      client_output_string (errordescriptor, buffer);
    }
  else
    group->max_size = size;
}

void
set_export_command (void *dummy, char *value, int errordescriptor)
{
  if (export_command)
    xfree (export_command);
  export_command = strdup (value);
}

void
set_access_file (void *dummy, char *value, int errordescriptor)
{

  if (access_file)
    xfree (access_file);
  access_file = strdup (value);
}

void
set_auxillary_info (void *dummy, char *value, int errordescriptor)
{

  if (auxillary_info)
    xfree (auxillary_info);
  auxillary_info = strdup (value);
}

void
set_data_storage_max (void *dummy, char *value, int errordescriptor)
{
  int size;

  size = atoi (value);
  if (size < 1024)
    {
      char buffer[128];
      sprintf (buffer, "data-storage-size too small (%d), should be > %d\n",
	       size, 32 * 1024);
      client_output_string (errordescriptor, buffer);
      return;
    }
  datablock_set_max_bytes (size);
}

void
set_maintainer (void *dummy, char *value, int errordescriptor)
{
  if (maintainer)
    xfree (maintainer);
  maintainer = strdup (value);
}

/* We can only set the password once.*/
void
set_password (void *dummy, char *value, int errordescriptor)
{
  if (password == NULL)
    password = strdup (value);
}

void
join_group (Group *group, char *siteid, int errordescriptor)
{
  SiteID id;
  SiteInfo *site;

  scan_siteid (siteid, &id);

  site = site_by_id (&id);

  if (site == NULL)
    {
      char buffer[128];
      sprintf (buffer, "No such site: `%s'\n", siteid);
      client_output_string (errordescriptor, buffer);
      return;
    }
  group_site_add (group, site);
}

void
set_site_purge_period (void *null, char *value, int errordescriptor)
{
  int period;

  period = atoi (value);
  
  if (period <= 0)
    {
      char buffer[128];
      sprintf (buffer,
	       "set_site_purge_period too small (%d), should be > %d\n",
	       period, 0);
      client_output_string (errordescriptor, buffer);
      return;
    }
  site_purge_period = period;
}

void
set_log_purge_period (void *null, char *value, int errordescriptor)
{
  int period;

  period = atoi (value);
  
  if (period <= 0)
    {
      char buffer[128];
      sprintf (buffer,
	       "set_log_purge_period too small (%d), should be > %d\n",
	       period, 0);
      client_output_string (errordescriptor, buffer);
      return;
    }
  log_purge_period = period;
}

void
set_bandwidth_size (void *dummy, char *value, int errordescriptor)
{
  int size;

  size = atoi (value);
  if (size <= 0)
    {
      char buffer[128];
      sprintf (buffer,
	       "set_bandwidth_size too small (%d), should be > %d\n",
	       size, 0);
      client_output_string (errordescriptor, buffer);
      return;
    }
  bandwidth_size = size;
}

/* Field name should be unambigious */
Field *
field_find (char *name, Field *fields)
{
  Field *result = NULL;
  int i;
  int offset;

  for (i = 0; fields[i].field != NULL; i++)
    {
      if (strncmp (name, fields[i].field, strlen (name)) == 0)
	{
	  if (result != NULL)
	    return ((Field *) -1);	/* ambigious field  */
	  else
	    result = &fields[i];
	}
    }
  return (result);
}

Field site_fields[] =
{
  {"client-port",	set_client_port,	INTEGER},
  {"coordinates",	set_coordinates,	INTEGER},
  {"data-port",		set_data_port,		INTEGER},
  {"inet-address",	set_inetaddr,		STRING},
  {"hostname",		set_host,		STRING},
  {"longitude",		set_longitude,		INTEGER},
  {"lattitude",		set_lattitude,		INTEGER},
  {"site-name",		NULL, 			STRING},
  {NULL, NULL},
};

Field group_fields[] =
{
  {"bandwidth-period",	set_bandwidth_period,	INTEGER},
  {"connectivity", 	set_connectivity,	INTEGER},
  {"estimates-period",	set_estimates_period,	INTEGER},
  {"group-name", 	NULL, 			STRING},
  {"join-period", 	set_join_period,	INTEGER},
  {"master-site", 	set_master,		STRING},
  {"max-size", 		set_max_size,		INTEGER},
  {"ping-period", 	set_ping_period,	INTEGER},
  {"site", 		group_site, 		STRING},
  {"topology-generator", set_topology_generator, STRING},
  {"update-period", 	set_update_period,	INTEGER},
  {"allow-site", 	set_allow_site,		STRING},
  {"deny-site", 	set_deny_site,		STRING},
  {NULL, NULL},
};

Field join_fields[] =
{
  {"site-id",		join_group,		STRING},
  {"group-name",	NULL,			STRING},
  {NULL, NULL},
};

Field local_fields[] = 
{
  {"data-storage-max",	set_data_storage_max,	INTEGER},
  {"export-command",	set_export_command,	STRING},
  {"log-purge-period",	set_log_purge_period,	INTEGER},
  {"site-purge-period",	set_site_purge_period,	INTEGER},
  {"access-file",	set_access_file,	STRING},
  {"bandwidth-size",	set_bandwidth_size,	INTEGER},
  {"auxillary-info",	set_auxillary_info,	STRING},
  {"maintainer",	set_maintainer,		STRING},
  {"password",		set_password,		STRING},
  {NULL, NULL},
};

ListType list_types[] =
{
  {":group-define", "group-name", 
     group_fields, 
     (GenericFunction *) group_by_name, 
     (GenericFunction *) group_new, 
     (void **) &group_list, 
     (void **) &groups},
  {":group-join", "group-name", 
     join_fields, 
     (GenericFunction *) group_by_name,
     NULL, 
     (void **) &group_list, 
     (void **) &groups},
  {":site-define", "site-name", 
     site_fields, 
     (GenericFunction *) site_by_name, 
     (GenericFunction *) siteinfo_new, 
     (void **) &site_list, 
     (void **) &sites},
  {":local", NULL, 
     local_fields, 
     NULL, 
     NULL, 
     NULL, 
     NULL},
  {NULL, NULL, NULL, NULL, NULL, NULL, NULL},
};

parse_list_by_type (ListType *type, char *list, int errordescriptor)
{
  char *field_string;
  char *field_name;
  char *field_value;
  char *key = NULL;
  void *object = NULL;
  int object_is_new = 0;
  char buffer[128];

  /* Find the identifying field first */
  if (type->key != NULL)
    {
      key = return_token (next_token (strstr (list, type->key)));

      if (key == NULL)
	{
	  sprintf (buffer, "Could not find key field `%s'\n", type->key);
	  client_output_string (errordescriptor, buffer);
	  return;
	}

      if (type->lookup_object != NULL)
	object = type->lookup_object (key);
      else 
	object = NULL;
    }
  
  /* Create a new object if one does not already exist 
  * and there is function with which to do so.
   */
  if (object == NULL && type->new_object != NULL)
    {
      object = type->new_object (key);
      object_is_new = 1;
    }
  else
    xfree (key);

  if ((object_is_new || type->lookup_object != NULL) && object == NULL)
    return (0);

  field_string = list;

  while (field_string != NULL && *field_string != '\0' && *field_string != ')')
    {
      Field *field;

      field_name = return_token (next_token(field_string));
      field_value = return_token (next_token (next_token (field_string)));

      if (field_name == NULL || *field_name == '\0' || field_value == NULL)
	{
	  sprintf (buffer, "Parse error\n");
	  client_output_string (errordescriptor, buffer);
	  return;
	}

      /* Skip the leading colon when doing the lookup */
      field = field_find (&field_name[1], type->fields);
      if (field == NULL)
	{
	  sprintf (buffer, "Bad field name: `%s'\n", field_name);
	  client_output_string (errordescriptor, buffer);
	  return;
	}
      if (field == (Field *) -1)
	{
	  sprintf (buffer, "Ambigious field name: `%s'\n", field_name);
	  client_output_string (errordescriptor, buffer);
	  return;
	}
      else 
	if (field->set != NULL)
	  field->set (object, field_value, errordescriptor);
      
      xfree (field_name);
      xfree (field_value);
      field_string = next_list (field_string);
    }

  /* Insert the new object into a table if necessary */
  if (object_is_new && type->object_list)
    {
      void *array;

      array = list_insert (*type->object_list, object);
      
      if (type->object_array)
	*type->object_array = array;
    }
}

parse_list (char *list, int errordescriptor)
{
  char *listtype_name;
  ListType *listtype;
  char buffer[128];

  if (*list != '(')
    {
      sprintf (buffer, "Bad list syntax: `%s'", list);
      client_output_string (errordescriptor, buffer);
      return (1);
    }
  
  list = next_token (list);

  listtype_name = return_token (list);

  listtype = listtype_find (listtype_name);
  
  if (listtype == NULL || listtype == (ListType *)-1)
    {
      sprintf (buffer, "Invalid list type `%s'\n", listtype_name);
      client_output_string (errordescriptor, buffer);
      xfree (listtype_name);
      return (-1);
    }

  parse_list_by_type (listtype, next_token (list), errordescriptor);

  xfree (listtype_name);
  return (0);
}

/* The config string should consist of a number of lisp lists */
void
parse_config_string (char *string, int errordescriptor)
{
  char *list;

  string = skip_space (string);
  list = return_list (string);
  while (list != NULL && *list != '\0' && *list != ')')
    {
      parse_list (list, errordescriptor);
      xfree (list);
      string = next_list (string);
      list = return_list (string);
    }
  xfree (list);
}

ListType *
listtype_find (char *type)
{
  ListType *result = NULL;
  int i;

  for (i = 0; list_types[i].type != NULL; i++)
    {
      if (strncmp (type, list_types[i].type, strlen (type)) == 0)
	{
	  if (result != NULL)
	    return ((ListType *) -1);	/* ambigious field  */
	  else
	    result = &list_types[i];
	}
    }
  return (result);
}


/* 
 * Read and update the access lists if necessary 
 */

void
access_list_update (void)
{
  static int last_access_mod_time = 0;	/* last time access file was modifi */
  struct stat statbuf;
  int i;
  extern int mask_free (void *);

  if (access_file == NULL)
    return;
    
  if (stat (access_file, &statbuf) < 0)
    return;

  if (statbuf.st_mtime != last_access_mod_time)
    {
      last_access_mod_time = statbuf.st_mtime;

      for (i = 0; groups[i] != NULL; i++)
	{
	  list_free_elements (groups[i]->allow_list, mask_free);
	  list_free_elements (groups[i]->deny_list, mask_free);
	}
      process_config_file (access_file);
      return;
    }
}
