
/* Copyright (c) 1992 Vincent Cate
 * All Rights Reserved.
 *
 * Permission to use and modify this software and its documentation
 * is hereby granted, provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative works
 * or modified versions, and any portions thereof, and that both notices
 * appear in supporting documentation.  This software or any derivate works
 * may not be sold or distributed without prior written approval from
 * Vincent Cate.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND VINCENT CATE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL
 * VINCENT CATE BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Users of this software agree to return to Vincent Cate any improvements
 * or extensions that they make and grant Vincent Cate the rights to
 * redistribute these changes.
 *
 */

#define REPORTINTERVAL 10000

/*  
 *  Do check:
 *       that no inode number is used more than once.
 *
 *       that every directory has a ".", "..", and ".alex.info"
 *
 *       that the "." and ".." are right (have scripts for this but should do it here)
 *
 *
 *  Could check:
 *       that the only things in a directory were things listed in .alex.info
 *            not that important as unlisted files become old and are flushed by evictor.c
 * 
 *       that each name is unique
 *
 *  Problems:
 *       memory usage is a full path for every entry in any directory.
 *       there may be 200,000 to 1,000,000 files now
 *       maybe 50 bytes each.
 *       For now just do an "unlimit DATASIZE"
 *
 *  Special feature:
 *       output files with "readme" in them when give "-r" flag.
 */


#define TableSize 50000
#define NOINODE -1

#include "alexincs.h"
#include "alex.h"

FILE *debugLog = NULL;

int ReadMeFlag=0;
int HistoFlag=0;
int QuickFlag=0;

char   UidStr[]="Vincent.Cate@cmu.edu";

int  TotalVirtualBytes=0;

struct FsckEntry {
    int Inode;
    struct FsckEntry *Next;
    char *Name;
};

struct FsckEntry Table[TableSize];


fh_fdclose()
{
    printf("fh_fdclose should not be called");
}

char LastUidStr[500];

int CheckFileCache()
{
    printf("CheckFileCache should not be called");
}


/* Collecting histograms for size and age */
#define MAXSTATAGE 3650
#define MAXSTATKB  1000000
int AgeInDays[MAXSTATAGE+1];
int SizeInKB[MAXSTATKB+1];
int TypeTotals[100];

CollectHistos(CurrentPtr)
struct ParsedDir *CurrentPtr;
{
    static int Now=0;
    int Age, AgeD, Size, i;

    if (Now==0) {
        Now = (int) TimeInSeconds();
        printf("Now = %d\n", Now);
        printf("ADAY = %d\n", ADAY);
        printf("Date = %d\n", CurrentPtr->Date);
        for (i=0; i<= UNKNOWN; i++) 
            TypeTotals[i]=0;

        for (i=0; i <= MAXSTATKB; i++) 
            SizeInKB[i]=0;

        for (i=0; i <= MAXSTATAGE; i++) 
            AgeInDays[i]=0;
    }


    Age  = Now - CurrentPtr->Date;
    AgeD = Age / ADAY;   
    if (AgeD < 0)          AgeD=0;
    if (AgeD > MAXSTATAGE) AgeD=MAXSTATAGE;
    AgeInDays[AgeD]++;

    Size= CurrentPtr->Size / 1024;
    if (Size < 0)         Size=0;
    if (Size > MAXSTATKB) Size=MAXSTATKB;
    SizeInKB[Size]++;

    TypeTotals[CurrentPtr->Type]++;
}


PrintStats()
{
    int i;

    printf("PrintStats\n");

    for (i=0; i<UNKNOWN; i++) {
        printf("Type %s %d\n", ATypeToString(i), TypeTotals[i]);
    }

    printf("\n\n Ages\n");

    for (i=0; i <= MAXSTATAGE; i++) {
        if (AgeInDays[i]>0) {
            printf("%d %d\n", i, AgeInDays[i]); 
        }
    }

    printf("\n\n Sizes\n");

    for (i=0; i <= MAXSTATKB; i++) {
        if (SizeInKB[i] > 0) {
            printf("%d %d\n", i, SizeInKB[i]); 
        }
    }

}

int NumberOfFiles;

int CheckOneEntry(CurrentPtr, AlexInfoPath)
struct ParsedDir *CurrentPtr;
char *AlexInfoPath;
{
    int Inode;
    struct FsckEntry *Head;
    char TmpStr[500];

    Inode=CurrentPtr->Inode;

    if (Inode == 0) {
        printf("Zero inode number %s\n", AlexInfoPath);
        fflush(stdout);
        return(AFAIL);
    }

    (void) strcpy(TmpStr, AlexInfoPath);
    (void) strcpy(rindex(TmpStr, '/')+1, CurrentPtr->Name);
    SimplifyPath(TmpStr);

    if (ReadMeFlag) {
        if ((CurrentPtr->Type==AFILE) && (HasStringNoCase(TmpStr, "README"))) {
            printf("%s\n", TmpStr);
        }
        return(AOK);
    }


    Head= &Table[Inode % TableSize];       /* hash to table location and make pointer to that */

    while ((Head->Inode != NOINODE) && (Head->Inode != Inode) && (Head->Next != NULL)) {
        Head=Head->Next;
    }

    if (Head->Inode == Inode) {                            /* have we seen this inode ? */
        if (Head->Name == NULL) {
            printf("ERROR NULL Name at inode %d", Inode);
            fflush(stdout);
            abort();
        }
        if (!streql(Head->Name, TmpStr)) {
            printf("Duplicate of %d   Old=%s   New=%s\n", Inode, Head->Name, TmpStr);
            fflush(stdout);
            return(AFAIL);
        } else {
            return(AOK);
        }
    }

                                                          /* have not seen Inode before */

    /* CollectHistos(CurrentPtr); */

    TotalVirtualBytes += CurrentPtr->Size;

    if (streql(CurrentPtr->Name, ".")) {
        printf("ERROR The . was new in %s\n", TmpStr);
    }

    if (streql(CurrentPtr->Name, "..")) {
        printf("ERROR The .. did not point so something known in %s\n", AlexInfoPath);
    }

    if (Head->Inode != NOINODE) {                         /* if not already empty make another */
        NumberOfFiles++;
        Head->Next = (struct FsckEntry *) malloc(sizeof(struct FsckEntry));
        if (Head->Next == NULL) {
            printf("CheckOneEntry malloc failed");
            fflush(stdout);
            abort();
        }
        Head=Head->Next;                                       /* pointing to new empty one */

    }

    Head->Inode=Inode;                                         /* set new values */
    Head->Name = strsave(TmpStr);
    if (Head->Name == NULL) {
        MallocDeath("Ug");
    }
    Head->Next=NULL;

    return(AOK);
}


CheckOneInfo(AlexInfoPath)
char *AlexInfoPath;
{
    int Status, Result;
    struct ActiveAlexInfo AAI;
    struct ParsedDir Current;
    int NumChecked;
    int HasDot, HasDotDot, HasAlexInfo, ReadSomething;

    NumChecked=0;
    Result=AOK;
    Status=ALEXINFOINOPEN(AlexInfoPath, &AAI, UidStr, 0);
    if (Status != AOK) {
        printf("File may be gone - could not open %s\n", AlexInfoPath);
        fflush(stdout);
        AlexInfoInClose(&AAI);
        return(AFAIL);
    }
 
    HasDot=0;
    HasDotDot=0;
    HasAlexInfo=0; 
    ReadSomething=0;
    while (Status==AOK) { 
        Status=AlexInfoInNext(&AAI, &Current);
        if (Status==AOK) {
            ReadSomething=1;

            if (streql(Current.Name, ".")) {
                HasDot++;
            } else if (streql(Current.Name, "..")) {
                HasDotDot++;
            } else if (streql(Current.Name, ALEXINFO)) {
                HasAlexInfo++;
            } else if (HistoFlag) {
                CollectHistos(&Current); 
            }

            if (!QuickFlag) {
                Status=CheckOneEntry(&Current, AlexInfoPath);
                if (Status != AOK) {
                    Result=AFAIL;
                }
            }
            NumChecked++;
            Status=AOK;                  /* only no input is reason to stop */
        }
    }

    if (!ReadMeFlag) {
      if (ReadSomething) {
          if (HasDot != 1)     { printf("No/Bad . in %s\n",          AlexInfoPath); Result=AFAIL; }
          if (HasDotDot != 1)  { printf("No/Bad .. in %s\n",         AlexInfoPath); Result=AFAIL; }
#if DBDOTALEX < DEBUGLEVEL
          if (HasAlexInfo != 1){ printf("No/Bad .alex.info in %s\n", AlexInfoPath); Result=AFAIL; }
#endif
      } else {
          printf("CheckOneInfo could not open %s\n", AlexInfoPath);
      }
    }

    LogN("CheckOneInfo looked at ", NumChecked);

    AlexInfoInClose(&AAI);
    return(Result);
}

Init()
{

    int i;

    InitReadOnlyVariables();

    for (i=0; i<TableSize; i++) {
        Table[i].Inode= NOINODE;
        Table[i].Name=NULL;
        Table[i].Next=NULL;
    }

    NumberOfFiles=0;
}


inputerror()
{
    fprintf(stderr, "usage: alexfsck [-r] \n");
    exit(1);
}
 
main(argc, argv)
int argc;
char *argv[];
{
    char   AlexInfoPath[MAXPATH];
    int LastReport, Status;
    int NumberOfInfos;

    char *cp;

    argc--, argv++;
    while (argc > 0 && argv[0][0] == '-') {
        for (cp = &argv[0][1]; *cp; cp++) {
            switch (*cp) {
                case 'r': ReadMeFlag++; break;
                case 'h': HistoFlag++; break;
                case 'q': QuickFlag++; break;      /* no checking */
                default:  inputerror();
            }
        }
        argc--, argv++;
    }


    Init();
    LastReport=0;
    NumberOfInfos=0;

    while (gets(AlexInfoPath) != NULL) {
        Status=CheckOneInfo(AlexInfoPath);
        if (Status != AOK) {
            printf("CheckOneInfo found problem with %s\n", AlexInfoPath);
        }
        NumberOfInfos++;
        if ((NumberOfFiles - LastReport) > REPORTINTERVAL) {
            printf("After %d .alex.infos we have checked %d files\n", NumberOfInfos, NumberOfFiles);
            LastReport=NumberOfFiles;
        }
        fflush(stdout);
    }

    printf("Total files checked is %d\n", NumberOfFiles);
    printf("TotalVirtualBytes is %d\n", TotalVirtualBytes);

    if (HistoFlag) {
        PrintStats();
    }
    fflush(stdout);
}



