
/*
 * refile.c -- move or link message(s) from a source folder
 *          -- into one or more destination folders
 *
 * $Id$
 */

#include <h/mh.h>
#include <errno.h>

static struct swit switches[] = {
#define	DRAFTSW          0
    { "draft", 0 },
#define	LINKSW           1
    { "link", 0 },
#define	NLINKSW          2
    { "nolink", 0 },
#define	PRESSW           3
    { "preserve", 0 },
#define	NPRESSW          4
    { "nopreserve", 0 },
#define	SRCSW            5
    { "src +folder", 0 },
#define	FILESW           6
    { "file file", 0 },
#define	RPROCSW          7
    { "rmmproc program", 0 },
#define	NRPRCSW          8
    { "normmproc", 0 },
#define VERSIONSW        9
    { "version", 0 },
#define	HELPSW          10
    { "help", 4 },
    { NULL, 0 }
};

extern int  errno;

static char maildir[BUFSIZ];

struct st_fold {
    char *f_name;
    struct msgs *f_mp;
};

/*
 * static prototypes
 */
static void opnfolds (struct st_fold *, int);
static void clsfolds (struct st_fold *, int);
static void removeit (struct msgs *, int, char **);
static int m_file (char *, struct st_fold *, int, int);


main (int argc, char **argv)
{
    int	linkf = 0, prsrvf = 0, filep = 0;
    int foldp = 0, msgp = 0, isdf = 0;
    int i, msgnum;
    char *cp, *folder = NULL, buf[100];
    char **ap, **argp, *arguments[MAXARGS];
    char *filevec[NFOLDERS + 2];
    char **files = &filevec[1];		/* leave room for removeit:vec[0] */
    char *msgs[MAXARGS];
    struct st_fold folders[NFOLDERS + 1];
    struct msgs *mp;

#ifdef LOCALE
    setlocale(LC_ALL, "");
#endif
    invo_name = r1bindex (argv[0], '/');
    if ((cp = m_find (invo_name)) != NULL) {
	ap = brkstring (cp = getcpy (cp), " ", "\n");
	ap = copyip (ap, arguments);
    } else {
	ap = arguments;
    }
    copyip (argv + 1, ap);
    argp = arguments;

    while ((cp = *argp++)) {
	if (*cp == '-')
	    switch (smatch (++cp, switches)) {
		case AMBIGSW: 
		    ambigsw (cp, switches);
		    done (1);
		case UNKWNSW: 
		    adios (NULL, "-%s unknown\n", cp);

		case HELPSW: 
		    sprintf (buf, "%s [msgs] [switches] +folder ...", invo_name);
		    print_help (buf, switches);
		    done (1);
		case VERSIONSW:
		    print_version(invo_name);
		    done (1);

		case LINKSW: 
		    linkf++;
		    continue;
		case NLINKSW: 
		    linkf = 0;
		    continue;

		case PRESSW: 
		    prsrvf++;
		    continue;
		case NPRESSW: 
		    prsrvf = 0;
		    continue;

		case SRCSW: 
		    if (folder)
			adios (NULL, "only one source folder at a time!");
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    folder = path (*cp == '+' || *cp == '@' ? cp + 1 : cp,
				   *cp != '@' ? TFOLDER : TSUBCWF);
		    continue;
		case DRAFTSW:
		    if (filep > NFOLDERS)
			adios (NULL, "only %d files allowed!", NFOLDERS);
		    isdf = 0;
		    files[filep++] = getcpy (m_draft (NULL, NULL, 1, &isdf));
		    continue;
		case FILESW: 
		    if (filep > NFOLDERS)
			adios (NULL, "only %d files allowed!", NFOLDERS);
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    files[filep++] = path (cp, TFILE);
		    continue;

		case RPROCSW: 
		    if (!(rmmproc = *argp++) || *rmmproc == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    continue;
		case NRPRCSW: 
		    rmmproc = (char *)0;
		    continue;
	    }
	if (*cp == '+' || *cp == '@') {
	    if (foldp > NFOLDERS)
		adios (NULL, "only %d folders allowed!", NFOLDERS);
	    folders[foldp++].f_name =
		    path (cp + 1, *cp == '+' ? TFOLDER : TSUBCWF);
	}
	else
	    msgs[msgp++] = cp;
    }

    if (!m_find ("path"))
	free (path ("./", TFOLDER));
    if (foldp == 0)
	adios (NULL, "no folder specified");

#ifdef WHATNOW
    if (!msgp && !foldp && !filep && (cp = getenv ("mhdraft")) && *cp)
	files[filep++] = cp;
#endif /* WHATNOW */

    if (filep > 0) {
	if (folder || msgp)
	    adios (NULL, "use -file or some messages, not both");
	opnfolds (folders, foldp);
	for (i = 0; i < filep; i++)
	    if (m_file (files[i], folders, foldp, prsrvf))
		done (1);
	if (!linkf)
	    removeit (NULLMP, filep, filevec);
	done (0);
    }

    if (!msgp)
	msgs[msgp++] = "cur";
    if (!folder)
	folder = m_getfolder ();
    strcpy (maildir, m_maildir (folder));

    if (chdir (maildir) == NOTOK)
	adios (maildir, "unable to change directory to");
    if (!(mp = m_gmsg (folder)))
	adios (NULL, "unable to read folder %s", folder);
    if (mp->hghmsg == 0)
	adios (NULL, "no messages in %s", folder);

    /* parse the message range/sequence/name and set SELECTED */
    for (msgnum = 0; msgnum < msgp; msgnum++)
	if (!m_convert (mp, msgs[msgnum]))
	    done (1);
    m_setseq (mp);	/* set the previous-sequence */

    opnfolds (folders, foldp);
    for (msgnum = mp->lowsel; msgnum <= mp->hghsel; msgnum++)
	if (is_selected (mp, msgnum)) {
	    cp = getcpy (m_name (msgnum));
	    if (m_file (cp, folders, foldp, prsrvf))
		done (1);
	    free (cp);
	    if (!linkf) {
#ifdef notdef
		mp->msgstats[msgnum] |= DELETED;
#endif /* notdef */
		mp->msgstats[msgnum] &= ~EXISTS;
	    }
	}
    if (!linkf)
	mp->msgflags |= SEQMOD;
    clsfolds (folders, foldp);

    m_replace (pfolder, folder);	/* update current folder */
    if (mp->hghsel != mp->curmsg
	    && (mp->numsel != mp->nummsg || linkf))
	m_setcur (mp, mp->hghsel);
    m_sync (mp);
    m_update ();	/* update the context file */

    if (!linkf)
	removeit (mp, filep, filevec);

    done (0);
}


static void
opnfolds (struct st_fold *folders, int nfolders)
{
    register char *cp;
    char nmaildir[BUFSIZ];
    register struct st_fold *fp, *ep;
    register struct msgs *mp;
    struct stat st;

    for (ep = (fp = folders) + nfolders; fp < ep; fp++) {
	chdir (m_maildir (""));
	strcpy (nmaildir, m_maildir (fp->f_name));

	if (stat (nmaildir, &st) == NOTOK) {
	    if (errno != ENOENT)
		adios (nmaildir, "error on folder");
	    cp = concat ("Create folder \"", nmaildir, "\"? ", NULL);
	    if (!getanswer (cp))
		done (1);
	    free (cp);
	    if (!makedir (nmaildir))
		adios (NULL, "unable to create folder %s", nmaildir);
	}

	if (chdir (nmaildir) == NOTOK)
	    adios (nmaildir, "unable to change directory to");
	if (!(mp = m_gmsg (fp->f_name)))
	    adios (NULL, "unable to read folder %s", fp->f_name);
	mp->curmsg = 0;

	fp->f_mp = mp;

	chdir (maildir);
    }
}


static void
clsfolds (struct st_fold *folders, int nfolders)
{
    register struct st_fold *fp, *ep;
    register struct msgs *mp;

    for (ep = (fp = folders) + nfolders; fp < ep; fp++) {
	mp = fp->f_mp;
	m_setseq (mp);
	m_sync (mp);
    }
}


static void
removeit (struct msgs *mp, int filep, char **files)
{
    register int i, vecp;
    register char *cp, **vec;

    if (rmmproc) {
	if (filep > 0) {
	    vec = files++;	/* filevec[1] */
	    files[filep] = NULL;
	} else {
	    if (mp->numsel > MAXARGS - 2)
		adios (NULL, "more than %d messages for %s exec",
			MAXARGS - 2, rmmproc);
	    vec = (char **) calloc ((size_t) (mp->numsel + 2), sizeof *vec);
	    if (vec == NULL)
		adios (NULL, "unable to allocate exec vector");
	    vecp = 1;
	    for (i = mp->lowsel; i <= mp->hghsel; i++)
		if (is_selected (mp, i))
		    vec[vecp++] = getcpy (m_name (i));
	    vec[vecp] = NULL;
	}

	fflush (stdout);
	vec[0] = r1bindex (rmmproc, '/');
	execvp (rmmproc, vec);
	adios (rmmproc, "unable to exec");
    }

    if (filep > 0) {
	files++;	/* filevec[1] */
	for (i = 0; i < filep; i++)
	    if (unlink (files[i]) == NOTOK)
		admonish (files[i], "unable to unlink");
    } else {
	for (i = mp->lowsel; i <= mp->hghsel; i++)
	    if (is_selected (mp, i))
		if (unlink (cp = m_name (i)) == NOTOK)
		    admonish (cp, "unable to unlink");
    }
}


static int
m_file (char *msg, struct st_fold *folders, int nfolders, int prsrvf)
{
    int in, out, linkerr, msgnum;
    register char *nmsg;
    char newmsg[BUFSIZ];
    register struct st_fold *fp, *ep;
    register struct msgs *mp;
    struct stat st, s1;

    for (ep = (fp = folders) + nfolders; fp < ep; fp++) {
	mp = fp->f_mp;
	if (prsrvf && (msgnum = m_atoi (nmsg = msg)) > 0) {
	    if (msgnum >= mp->hghoff)
		if ((mp = m_remsg (mp, 0, msgnum + MAXFOLDER)))
		    fp->f_mp = mp;
		else
		    adios (NULL, "unable to allocate folder storage");
	    if (!(mp->msgstats[msgnum] & EXISTS)) {
		mp->msgstats[msgnum] |= EXISTS;
#ifdef notdef
		mp->msgstats[msgnum] &= ~DELETED;
#endif /* notdef */
		mp->nummsg++;
	    }
	    set_selected (mp, msgnum);
	    if (msgnum > mp->hghmsg)
		mp->hghmsg = msgnum;
	}
	else {
	    if (mp->hghmsg >= mp->hghoff)
		if ((mp = m_remsg (mp, 0, mp->hghoff + MAXFOLDER)))
		    fp->f_mp = mp;
		else
		    adios (NULL, "unable to allocate folder storage");

	    nmsg = m_name (msgnum = ++mp->hghmsg);
	    mp->nummsg++;
	    mp->msgstats[msgnum] |= EXISTS;
	    set_selected (mp, msgnum);
	}
	if (mp->lowmsg == 0)
	    mp->lowmsg = msgnum;
	if (mp->lowsel == 0 || msgnum < mp->lowsel)
	    mp->lowsel = msgnum;
	if (msgnum > mp->hghsel)
	    mp->hghsel = msgnum;

	sprintf (newmsg, "%s/%s", mp->foldpath, nmsg);
	if (link (msg, newmsg) == NOTOK) {
#ifndef	EISREMOTE
	    linkerr = errno;
#else /* EISREMOTE */
	    if ((linkerr = errno) == EISREMOTE)
		linkerr = EXDEV;
#endif /* EISREMOTE */
	    if (linkerr == EEXIST
		    || (linkerr == EXDEV && stat (newmsg, &st) != NOTOK)) {
		if (linkerr != EEXIST
			|| stat (msg, &s1) == NOTOK
			|| stat (newmsg, &st) == NOTOK
			|| s1.st_ino != st.st_ino) {
		    advise (NULL, "message %s:%s already exists",
			    fp->f_name, newmsg);
		    return 1;
		}
		continue;
	    }
	    if (linkerr == EXDEV) {
		if ((in = open (msg, 0)) == NOTOK) {
		    advise (msg, "unable to open message %s");
		    return 1;
		}
		fstat (in, &st);
		if ((out = creat (newmsg, (int) st.st_mode & 0777))
			== NOTOK) {
		    advise (newmsg, "unable to create");
		    close (in);
		    return 1;
		}
		cpydata (in, out, msg, newmsg);
		close (in);
		close (out);
	    }
	    else {
		advise (newmsg, "error linking %s to", msg);
		return 1;
	    }
	}
    }

    return 0;
}
