
/*
 * dropsbr.c -- write to a mailbox
 *
 * $Id$
 */

#include <h/nmh.h>

#ifndef	MMDFONLY
# include <h/mh.h>
# include <h/dropsbr.h>
# include <zotnet/mts/mts.h>
#else
# include "dropsbr.h"
# include "strings.h"
# include "mmdfonly.h"
#endif

#include <errno.h>

#ifdef NTOHLSWAP
# include <netinet/in.h>
#else
# define ntohl(n) (n)
#endif

#define	MMDF 1
#define	UUCP 2

static int mbx_style = MMDF;

extern int errno;

/*
 * static prototypes
 */
static int mbx_create (char *, uid_t, gid_t, mode_t);
static int mbx_chk (int);
static int map_open (char *, int *, int);


int
mbx_mmdf(void)
{
    int style = mbx_style;

    mbx_style = MMDF;
    return style;
}

int
mbx_uucp(void)
{
    int style = mbx_style;

    mbx_style = UUCP;
    return style;
}


int
mbx_open (char *file, uid_t uid, gid_t gid, mode_t mode)
{
    int clear, fd;

    if ((fd = mbx_Xopen (file, uid, gid, mode, &clear)) == NOTOK)
	return fd;

    if (!clear)
	switch (mbx_style) {
	    case MMDF: 
	    default: 
		if (mbx_chk (fd) == NOTOK) {
		    close (fd);
		    return NOTOK;
		}
		break;

	    case UUCP: 
		if (lseek (fd, (off_t) 0, SEEK_END) == (off_t) NOTOK) {
		    close (fd);
		    return NOTOK;
		}
		break;
	}

    return fd;
}


int
mbx_Xopen (char *file, uid_t uid, gid_t gid, mode_t mode, int *clear)
{
    register int j;
    int	count, fd;
    struct stat st;

    for (*clear = 0, count = 4, j = 0; count > 0; count--)
	if ((fd = lkopen (file, 6)) == NOTOK)
	    switch (errno) {
		case ENOENT: 
		    if (mbx_create (file, uid, gid, mode) == NOTOK)
			return NOTOK;
		    (*clear)++;
		    break;

#ifdef	BSD42
		case EWOULDBLOCK:
#endif	/* BSD42 */
		case ETXTBSY: 
		    j = errno;
		    sleep (5);
		    break;

		default: 
		    return NOTOK;
	    }
	else {
	    *clear = fstat (fd, &st) != NOTOK && st.st_size == (off_t)0;
	    break;
	}

    errno = j;
    return fd;
}


static int
mbx_create (char *file, uid_t uid, gid_t gid, mode_t mode)
{
    int fd;

    if ((fd = creat (file, 0600)) == NOTOK)
	return NOTOK;

    close (fd);
    chown (file, uid, gid);
    chmod (file, mode);

    return OK;
}


static int
mbx_chk (int fd)
{
    size_t count;
    char ldelim[BUFSIZ];

    count = strlen (mmdlm2);

    if (lseek (fd, (off_t) (-count), SEEK_END) == (off_t) NOTOK
	    || read (fd, ldelim, count) != count)
	return NOTOK;
    ldelim[count] = 0;

    if (strcmp (ldelim, mmdlm2)
	    && write (fd, "\n", 1) != 1
	    && write (fd, mmdlm2, count) != count)
	return NOTOK;

    return OK;
}


int
mbx_read (FILE *fp, long pos, struct drop **drops, int noisy)
{
    register int len, size;
    register long ld1, ld2;
    register char *bp;
    char buffer[BUFSIZ];
    register struct drop *cp, *dp, *ep, *pp;

    pp = (struct drop *) calloc ((size_t) (len = MAXFOLDER), sizeof *dp);
    if (pp == NULL) {
	if (noisy)
	    admonish (NULL, "unable to allocate drop storage");
	return NOTOK;
    }

    ld1 = (long) strlen (mmdlm1);
    ld2 = (long) strlen (mmdlm2);

    fseek (fp, pos, 0);
    for (ep = (dp = pp) + len - 1; fgets (buffer, sizeof buffer, fp);) {
	size = 0;
	if (strcmp (buffer, mmdlm1) == 0)
	    pos += ld1, dp->d_start = (long) pos;
	else {
	    dp->d_start = (long)pos , pos += (long) strlen (buffer);
	    for (bp = buffer; *bp; bp++, size++)
		if (*bp == '\n')
		    size++;
	}

	while (fgets (buffer, sizeof buffer, fp) != NULL)
	    if (strcmp (buffer, mmdlm2) == 0)
		break;
	    else {
		pos += (long) strlen (buffer);
		for (bp = buffer; *bp; bp++, size++)
		    if (*bp == '\n')
			size++;
	    }

	if (dp->d_start != (long) pos) {
	    dp->d_id = 0;
	    dp->d_size = (long) size;
	    dp->d_stop = pos;
	    dp++;
	}
	pos += ld2;

	if (dp >= ep) {
	    register int    curlen = dp - pp;

	    cp = (struct drop *) realloc ((char *) pp,
		                    (size_t) (len += MAXFOLDER) * sizeof *pp);
	    if (cp == NULL) {
		if (noisy)
		    admonish (NULL, "unable to allocate drop storage");
		free ((char *) pp);
		return 0;
	    }
	    dp = cp + curlen, ep = (pp = cp) + len - 1;
	}
    }

    if (dp == pp)
	free ((char *) pp);
    else
	*drops = pp;
    return (dp - pp);
}


int
mbx_write(char *mailbox, int md, FILE *fp, int id, long last,
           long pos, off_t stop, int mapping, int noisy)
{
    register int i, j, size;
    off_t start;
    long off;
    register char *cp;
    char buffer[BUFSIZ];

    off = (long) lseek (md, (off_t) 0, SEEK_CUR);
    j = strlen (mmdlm1);
    if (write (md, mmdlm1, j) != j)
	return NOTOK;
    start = lseek (md, (off_t) 0, SEEK_CUR);
    size = 0;

    fseek (fp, pos, 0);
    while (fgets (buffer, sizeof buffer, fp) && (pos < stop)) {
	i = strlen (buffer);
	for (j = 0; (j = stringdex (mmdlm1, buffer)) >= 0; buffer[j]++)
	    continue;
	for (j = 0; (j = stringdex (mmdlm2, buffer)) >= 0; buffer[j]++)
	    continue;
	if (write (md, buffer, i) != i)
	    return NOTOK;
	pos += (long) i;
	if (mapping)
	    for (cp = buffer; i-- > 0; size++)
		if (*cp++ == '\n')
		    size++;
    }

    stop = lseek (md, (off_t) 0, SEEK_CUR);
    j = strlen (mmdlm2);
    if (write (md, mmdlm2, j) != j)
	return NOTOK;
    if (mapping)
	map_write (mailbox, md, id, last, start, stop, off, size, noisy);

    return OK;
}


int
mbx_copy (char *mailbox, int md, int fd, int mapping, char *text, int noisy)
{
    register int i, j, size;
    off_t start, stop;
    long pos;
    register char *cp;
    char buffer[BUFSIZ];
    register FILE *fp;

    pos = (long) lseek (md, (off_t) 0, SEEK_CUR);
    size = 0;

    switch (mbx_style) {
	case MMDF: 
	default: 
	    j = strlen (mmdlm1);
	    if (write (md, mmdlm1, j) != j)
		return NOTOK;
	    start = lseek (md, (off_t) 0, SEEK_CUR);

	    if (text) {
		i = strlen (text);
		if (write (md, text, i) != i)
		    return NOTOK;
		for (cp = text; *cp++; size++)
		    if (*cp == '\n')
			size++;
	    }
		    
	    while ((i = read (fd, buffer, sizeof buffer)) > 0) {
		for (j = 0;
			(j = stringdex (mmdlm1, buffer)) >= 0;
			buffer[j]++)
		    continue;
		for (j = 0;
			(j = stringdex (mmdlm2, buffer)) >= 0;
			buffer[j]++)
		    continue;
		if (write (md, buffer, i) != i)
		    return NOTOK;
		if (mapping)
		    for (cp = buffer; i-- > 0; size++)
			if (*cp++ == '\n')
			    size++;
	    }

	    stop = lseek (md, (off_t) 0, SEEK_CUR);
	    j = strlen (mmdlm2);
	    if (write (md, mmdlm2, j) != j)
		return NOTOK;
	    if (mapping)
		map_write (mailbox, md, 0, (long)0, start, stop, pos, size, noisy);

	    return (i != NOTOK ? OK : NOTOK);

	case UUCP: 		/* I hate this... */
	    if ((j = dup (fd)) == NOTOK)
		return NOTOK;
	    if ((fp = fdopen (j, "r")) == NULL) {
		close (j);
		return NOTOK;
	    }
	    start = lseek (md, (off_t) 0, SEEK_CUR);

	    if (text) {
		i = strlen (text);
		if (write (md, text, i) != i)
		    return NOTOK;
		for (cp = text; *cp++; size++)
		    if (*cp == '\n')
			size++;
	    }
		    
	    for (j = 0; fgets (buffer, sizeof buffer, fp) != NULL; j++) {
		if (j != 0 && strncmp (buffer, "From ", 5) == 0) {
		    write (md, ">", 1);
		    size++;
		}
		i = strlen (buffer);
		if (write (md, buffer, i) != i) {
		    fclose (fp);
		    return NOTOK;
		}
		if (mapping)
		    for (cp = buffer; i-- > 0; size++)
			if (*cp++ == '\n')
			    size++;
	    }
	    if (write (md, "\n", 1) != 1) {
		fclose (fp);
		return NOTOK;
	    }
	    if (mapping) size += 2;

	    fclose (fp);
	    lseek (fd, (off_t) 0, SEEK_END);
	    stop = lseek (md, (off_t)0, SEEK_CUR);
	    if (mapping)
		map_write (mailbox, md, 0, (long)0, start, stop, pos, size, noisy);

	    return OK;
    }
}


int
mbx_size (int md, off_t start, off_t stop)
{
    register int i, fd;
    register long pos;
    register FILE *fp;

    if ((fd = dup (md)) == NOTOK || (fp = fdopen (fd, "r")) == NULL) {
	if (fd != NOTOK)
	    close (fd);
	return NOTOK;
    }

    fseek (fp, start, 0);
    for (i = 0, pos = stop - start; pos-- > 0; i++)
	if (fgetc (fp) == '\n')
	    i++;

    fclose (fp);

    return i;
}


int
mbx_close (char *mailbox, int md)
{
    lkclose (md, mailbox);

    return OK;
}


/* This function is performed implicitly by getbbent.c:

		bb->bb_map = map_name (bb->bb_file);
*/

char *
map_name (char *file)
{
    register char *cp, *dp;
    static char buffer[BUFSIZ];

    if ((dp = strchr(cp = r1bindex (file, '/'), '.')) == NULL)
	dp = cp + strlen (cp);
    if (cp == file)
	sprintf (buffer, ".%.*s%s", dp - cp, cp, ".map");
    else
	sprintf (buffer, "%.*s.%.*s%s", cp - file, file, dp - cp, cp, ".map");

    return buffer;
}


int
map_read (char *file, long pos, struct drop **drops, int noisy)
{
    register int i, md, msgp;
    register char *cp;
    struct drop d;
    register struct drop *mp, *dp;

    if ((md = open (cp = map_name (file), 0)) == NOTOK
	    || map_chk (cp, md, mp = &d, pos, noisy)) {
	if (md != NOTOK)
	    close (md);
	return 0;
    }

    msgp = mp->d_id;
    dp = (struct drop *) calloc ((size_t) (msgp + 1), sizeof *dp);
    if (dp == NULL) {
	close (md);
	return 0;
    }

    memcpy((char *) dp, (char *) mp, sizeof *dp);

    lseek (md, (off_t) sizeof(*mp), SEEK_SET);
    if ((i = read (md, (char *) (dp + 1), msgp * sizeof *dp)) < sizeof *dp) {
	i = 0;
	free ((char *) dp);
    }
    else {
#ifdef	NTOHLSWAP
	register struct drop *tdp;
	int j;

	for (j = 0, tdp = dp; j < i / sizeof(*dp); j++, tdp++) {
	    tdp->d_id = ntohl(tdp->d_id);
	    tdp->d_size = ntohl(tdp->d_size);
	    tdp->d_start = ntohl(tdp->d_start);
	    tdp->d_stop = ntohl(tdp->d_stop);
	}
#endif
	*drops = dp;
    }

    close (md);

    return (i / sizeof *dp);
}


int
map_write (char *mailbox, int md, int id, long last, off_t start,
           off_t stop, long pos, int size, int noisy)
{
    register int i;
    int clear, fd, td;
    char *file;
    register struct drop *dp;
    struct drop d1, d2, *rp;
    register FILE *fp;

    if ((fd = map_open (file = map_name (mailbox), &clear, md)) == NOTOK)
	return NOTOK;

    if (!clear && map_chk (file, fd, &d1, pos, noisy)) {
	unlink (file);
	mbx_close (file, fd);
	if ((fd = map_open (file, &clear, md)) == NOTOK)
	    return NOTOK;
	clear++;
    }

    if (clear) {
	if ((td = dup (md)) == NOTOK || (fp = fdopen (td, "r")) == NULL) {
	    if (noisy)
		admonish (file, "unable to %s", td != NOTOK ? "fdopen" : "dup");
	    if (td != NOTOK)
		close (td);
	    mbx_close (file, fd);
	    return NOTOK;
	}

	switch (i = mbx_read (fp, (off_t)0, &rp, noisy)) {
	    case NOTOK:
		fclose (fp);
		mbx_close (file, fd);
		return NOTOK;

	    case OK:
		break;

	    default:
		d1.d_id = 0;
		for (dp = rp; i-- >0; dp++) {
		    if (dp->d_start == start)
			dp->d_id = id;
		    lseek (fd, (off_t) (++d1.d_id * sizeof(*dp)), SEEK_SET);
		    if (write (fd, (char *) dp, sizeof *dp) != sizeof *dp) {
			if (noisy)
			    admonish (file, "write error");
			mbx_close (file, fd);
			fclose (fp);
			return NOTOK;
		    }
		}
		free ((char *) rp);
		break;
	}
    }
    else {
	if (last == 0)
	    last = d1.d_start;
	dp = &d2;
	dp->d_id = id;
	dp->d_size = (long) (size ? size : mbx_size (fd, start, stop));
	dp->d_start = start;
	dp->d_stop = stop;
	lseek (fd, (off_t) (++d1.d_id * sizeof(*dp)), SEEK_SET);
	if (write (fd, (char *) dp, sizeof *dp) != sizeof *dp) {
	    if (noisy)
		admonish (file, "write error");
	    mbx_close (file, fd);
	    return NOTOK;
	}
    }

    dp = &d1;
    dp->d_size = DRVRSN;
    dp->d_start = (long) last;
    dp->d_stop = lseek (md, (off_t) 0, SEEK_CUR);

    lseek (fd, (off_t) 0, SEEK_SET);
    if (write (fd, (char *) dp, sizeof *dp) != sizeof *dp) {
	if (noisy)
	    admonish (file, "write error");
	mbx_close (file, fd);
	return NOTOK;
    }

    mbx_close (file, fd);

    return OK;
}


static int
map_open (char *file, int *clear, int md)
{
    mode_t mode;
    struct stat st;

    mode = fstat (md, &st) != NOTOK ? (mode_t) (st.st_mode & 0777) : m_gmprot ();
    return mbx_Xopen (file, st.st_uid, st.st_gid, mode, clear);
}


int
map_chk (char *file, int fd, struct drop *dp, long pos, int noisy)
{
    long count;
    struct drop d, tmpd;
    register struct drop *dl;

    if (read (fd, (char *) &tmpd, sizeof *dp) != sizeof *dp) {
#ifdef	notdef
	admonish (NULL, "%s: missing or partial index", file);
#endif	/* notdef */
	return NOTOK;
    }
#ifndef	NTOHLSWAP
    *dp = tmpd;		/* if ntohl(n)=(n), can use struct assign */
#else
    dp->d_id = ntohl(tmpd.d_id);
    dp->d_size = ntohl(tmpd.d_size);
    dp->d_start = ntohl(tmpd.d_start);
    dp->d_stop = ntohl(tmpd.d_stop);
#endif
    
    if (dp->d_size != DRVRSN) {
	if (noisy)
	    admonish (NULL, "%s: version mismatch (%d != %d)", file,
				dp->d_size, DRVRSN);
	return NOTOK;
    }

    if (dp->d_stop != pos) {
	if (noisy && pos != (long) 0)
	    admonish (NULL,
		    "%s: pointer mismatch or incomplete index (%ld!=%ld)", 
		    file, dp->d_stop, (long) pos);
	return NOTOK;
    }

    if ((long) ((dp->d_id + 1) * sizeof(*dp)) != (long) lseek (fd, (off_t) 0, SEEK_END)) {
	if (noisy)
	    admonish (NULL, "%s: corrupt index(1)", file);
	return NOTOK;
    }

    dl = &d;
    count = (long) strlen (mmdlm2);
    lseek (fd, (off_t) (dp->d_id * sizeof(*dp)), SEEK_SET);
    if (read (fd, (char *) dl, sizeof *dl) != sizeof *dl
	    || (ntohl(dl->d_stop) != dp->d_stop
		&& ntohl(dl->d_stop) + count != dp->d_stop)) {
	if (noisy)
	    admonish (NULL, "%s: corrupt index(2)", file);
	return NOTOK;
    }

    return OK;
}
