/*
  Get-file-ownership

  Program to get file  ownership for you if it can match
  designated file(s) into your priviledges. (in 'database')

  Used at NIC.FUNET.FI to give moderators access to parts of archive so
  that they can moderate incoming file areas, and sometimes much more.

  "omi-file"  Finnish for "get-file-ownership" (`omi' - take ownership of...)

  Args:  [ -m mode ] [ -g group ] [ -o owner ] filenames.

  By Matti Aarnio <mea@nic.funet.fi> Oct-1990

*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <re_comp.h>

/*#define OMIFILEDAT	"/usr/local/etc/omi-file.dat"
#define OMIFILELOG	"/usr/local/etc/omi-file.log"
*/
#define OMIFILEDAT	"/l/etc/omi-file.dat"
#define OMIFILELOG	"/l/log/omi-file.log"

#ifndef  __
#ifndef	__STDC__
#define	volatile
#define const
#define __(x) ()
#else
#define __(x) x
#endif
#endif

#if 0
extern char *re_comp __((const char *s)); /* Null=ok compilation, otherwise errmsg str */
extern int   re_exec __((const char *s)); /* 1=match, 0=no match, -1=illegal regexpr   */

extern char *strchr __((const char *str, char key));
extern char *strrchr __((const char *str, char key));
extern void *malloc __(( const int size ));
extern void *realloc __(( void *ptr, const int size ));
extern int getuid __(( void ));
extern int getgid __(( void ));
extern int getgroups __(( int gidsetlen, int *gidset ));
extern struct group *getgrnam __(( const char *name ));
#endif

#define NGroups 64 /* 32 is OSF/1 max, but play safe.. */

gid_t Groups[NGroups];
int GroupsLen = 0;

struct DBstr {
	uid_t	uid;
	gid_t	gid;
	char	regexp[1];
      } DBstr;


static 	void usage	   __(( void ));
static	void read_database __(( void ));
extern	void omi_file      __(( const char *cwd, const char *fname ));

static struct DBstr **DataBase = NULL;	/* Base of database */
static char *pname;		/* Program invication name */
static int  NewGroup = -1;	/* chown args - magic values these... */
static int  NewUid   = -1;	/*   default is not to touch */
static int  NewMode  = -1;	/* New protection.  Default is not to touch */
static int  debug    =  0;

extern int getopt();
extern int optind, opterr;
extern char *optarg;

char *logname = NULL;
FILE *logfp = NULL;

main(argc,argv)
     int argc;
     char *argv[];
{
	char *fname;
	char cwd[MAXPATHLEN];
	struct group *groupp;
	struct passwd *pwent;
	int c;

	setbuf(stdout,NULL);
	setbuf(stderr,NULL);

	logname = getenv("LOGNAME");
	if (!logname) logname = "??";

	pname = *argv;

/*printf("%s: getuid()=%d, geteuid=%d, getgid=%d\n",pname,getuid(),geteuid(),getgid());*/

	if(argc < 2) usage();

	logfp = fopen(OMIFILELOG,"a");
	if (!logfp) {
	  fprintf(stderr,"%s: Can't open logfile '%s'\n",pname);
	  exit(99);
	}

	*cwd = 0;
	getcwd( cwd, sizeof cwd );
	if(*cwd == 0) {
	  fprintf(stderr,"%s: Can't getcwd() !  FAULT!\n",pname);
	  exit(99);
	}
	strcat(cwd,"/");

	while ((c = getopt(argc,argv,"do:g:m:")) != -1) {
	  switch (c) {
	    case 'd':
	        debug = 1;
		break;
	    case 'o':
		NewUid = atoi(optarg);
		if (!isdigit(*optarg)) {
		  if (pwent = getpwnam(optarg)) {
		    NewUid = pwent->pw_uid;
		  } else {
		    fprintf(stderr,"%s: Bad argument for  -o  option.  Numeric uid or user name\n",pname);
		    exit(5);
		  }
		}
		break;
	    case 'g':
		NewGroup = atoi(optarg);
		if (!isdigit(*optarg)) {
		  if (groupp = getgrnam(optarg)) {
		    NewGroup = groupp->gr_gid;
		  } else {
		    fprintf(stderr,"%s: Bad argument for  -g  option.  Numeric group or group name\n",pname);
		    exit(5);
		  }
		}
		break;
	    case 'm':
		NewMode = strtol(optarg,NULL,8); /* Octal numeric */
		if (!isdigit(*optarg)) {
		  fprintf(stderr,"%s: Bad argument for  -m  option.  Numeric mask value expected. (octal)\n",pname);
		  exit(5);
		}
		break;
	    default:
		break;
	  }
	}
	/* ----  getopt() done   ---- */

	if( optind == argc ){
	  fprintf(stderr,"%s: No arguments as files to operate.  Won't do a thing!\n",pname);
	  exit(7);
	}

	GroupsLen = getgroups( NGroups, Groups );
	read_database();

	for(; optind < argc; ++optind)
	  omi_file( cwd, argv[optind] );

	return 0;
}


static void usage()
{
	fprintf(stderr,"\
%s: [ -d] [ -g group ] [ -m mode ] [ -o owner ] filenames\n\
%s: Defaults:     mode: no change, owner: yourself, group: no change\n\
%s:               owner: no change   If mode or group set!\n\
%s: Restrictions: ABSOLUTE file paths from '%s',\n",
		pname,pname,pname,pname,OMIFILEDAT);
	fprintf(stderr,"\
%s:               Mode: regular file: value &  0777 -- eg rwx bits only.\n\
%s:                     directory:    value & 03777 -- rwx+sgid+sticky\n",
		pname,pname);
  exit(2);
}


/* Read in the database file */
static void read_database()
{
	FILE *dbf = fopen(OMIFILEDAT,"r");
	struct DBstr **db = NULL, **Dp, *Ds;
	char *s,*p;
	int  cnt = 0, dbcnt = 0, inpcnt = 0;
	char buf[1024],gids[20],uids[20];
	uid_t uid;
	gid_t gid, gidOurs = 0, *GG, GGlen;
	int len;
	struct passwd *pwent;
	struct group *grpp;

	if( dbf == NULL ) {
	  fprintf(stderr,"%s: Failed to open authorization database file '%s'\n",pname,OMIFILEDAT);
	  exit(2);
	}
	while( !feof(dbf) ) {
	  *buf = 0;
	  if( fgets( buf, sizeof buf, dbf ) == NULL ) break;
	  /* EOF propably... */
	  /* Format:  '#' in col 1 is for comment lines,
	     Empty line is considered as comment,
	     Line starting with space is comment.
	     
	     uuu:ggg:regexpr
	     
	     uuu = numerical/symbolic UID for this regexpr. '*' for wild
	     ggg = same for GID.
	     */

	  if( s = strrchr(buf,'\n') ) *s = 0; /* Zap \n */
	  if( *buf == '#' ) continue; /* Comment line */
	  if( *buf == ' ' ) continue; /* Comment line */
	  if( *buf == 0 ) continue; /* Comment line (empty line) */
	  /* Impose size limits and better error handling! */
	  *gids = 0; *uids = 0;

	  if (debug)
	    fprintf(stderr,"%s: DBase line: '%s'\n",pname,buf);
    
	  if ((s = strchr(buf,':')) == NULL) exit(9); /* FATAL! */
	  *s++ = 0;
	  strcpy(uids,buf);
	  if ((p = strchr(s,':')) == NULL) exit(10); /* FATAL too! */
	  *p = 0;
	  strcpy(gids,s);
	  *p = '^';		/* regexp start of line */
	  /* p  points to begin of asciz regexpr. */
	  len = strlen(p);
	  if( *uids == '*' ) uid = -1;
	  else {
	    if( isdigit(*uids) ) uid = atoi(uids);
	    else {
	      pwent = getpwnam(uids);
	      if( pwent == NULL ) {
		fprintf(stderr,
			"%s: Database has bad user name (\"%s\") in it!\n",
			pname, uids);
		exit(11);	/* FATAL! */
	      }
	      uid = pwent->pw_uid;
	    }
	  }
	  if( *gids == '*' ) gid = -1;
	  else {
	    if( isdigit(*gids) ) gid = atoi(gids);
	    else {
	      grpp = getgrnam(gids);
	      if( grpp == NULL ) {
		fprintf(stderr,
			"%s: Database has bad group name (\"%s\") in it!\n",
			pname, gids);
		exit(12);	/* FATAL! */
	      }
	      gid = grpp->gr_gid;
	    }
	  }
	  /* All fields are gotten: uid, gid, p+len */
	  Ds = malloc( sizeof(struct DBstr)+len );
	  Ds->uid = uid;
	  Ds->gid = gid;
	  ++inpcnt;
	  if( (uid != -1) && (uid != getuid()) ) continue; /* Not us, forget. */
	  if( (gid != -1) ) {	/* Test if GID is one of ours */
	    GG = Groups;
	    GGlen = GroupsLen;
	    gidOurs = 0;
	    while( GGlen-- > 0 ) {
	      if( gidOurs |= (*GG == gid ) ) break;
	      ++GG;
	    }			/* while */
	    if( !gidOurs ) continue; /* Wasn't ours. Forget. */
	  }			/* GID test */
    

	  strcpy(Ds->regexp,p);
	  if ( (s = re_comp(p)) != NULL) {
	    fprintf(stderr,"%s: Given regexpr ('%s') didn't satisfy re_comp() routine: %s",pname,p,s);
	    exit(15);
	  }			/* if */
	  if (debug)
	    fprintf(stderr,"%s: Dbase scan: uid=%d, gid=%d, regexp='%s'\n",
		    pname,uid,gid,p);

	  if( cnt >= dbcnt ) {
	    if( db == NULL ) db = malloc(8);
	    dbcnt += 8;
	    Dp = realloc( db, sizeof( struct DBstr *) * (dbcnt +1));
	    if (Dp == NULL) exit(13); /* FATAL! */
	    db = Dp;
	  }
	  db[cnt++] = Ds;
	  db[cnt]   = NULL;
	}
	fclose(dbf);
	if( (db == NULL) || (cnt == 0) ) {
	  if (inpcnt != 0)
	    fprintf(stderr,"%s: None of authorizations match you!      ABORT!\n",pname);
	  else
	    fprintf(stderr,"%s: Didn't read in authorization database! ABORT!\n",pname);
	  exit(2);
	}
	DataBase = db;
}


int Match_file( fname )
     const char *fname;  /* Canonical form */
{
	struct DBstr **db = DataBase, *Dp;
	char *s, *s2;
	int rc;
	struct stat64 stat1, stat2;

	int myuid = getuid();
	
/*printf("test1: lstat(%s)\n",fname);*/
	if (lstat64(fname,&stat1)) return 0;
	s = strrchr(fname,'/');
	if (!s) return 0;
	*s = 0;
	if (stat64(fname,&stat2)) return 0;
/*printf("test2: stat(%s), st_uid=%d, myuid=%d\n",fname,stat2.st_uid,myuid);*/
	*s = '/';

	if (stat2.st_uid == myuid) return 1;

	while ( *db != NULL ) {
	  Dp = *db++;
	  if (debug)
	    fprintf(stderr,"%s: Attempting match to: uid=%d, gid=%d, regexp='%s'\n",
		    pname,Dp->uid,Dp->gid,Dp->regexp);
	  /* If wild or not, UID and GID have been matched WHILE parsing ! */
	  /* Only one left is REGEXPR ! */

	  if (debug)
	    fprintf(stderr,"     Matching regexp part!\n");
	  if ( (s = re_comp(Dp->regexp)) != NULL) {
	    fprintf(stderr,"%s: Given regexpr ('%s') didn't satisfy re_comp() routine: %s",pname,Dp->regexp,s);
	    exit(15);
	  }			/* if */
	  rc = re_exec( fname );
	  if( rc < 0 ) {
	    fprintf(stderr,"%s: re_exec() failed! Last regexpr ('%s') was faulty!\n",pname,Dp->regexp);
	    exit(16);
	  }
	  if( rc > 0 ) {
	    if (debug)
	      fprintf(stderr,"   MATCHED!\n");
	    return 1;
	  }
	  if (debug)
	    fprintf(stderr,"   DIDN'T!\n");
	}			/* while */
	return 0;		/* No match */
}

/* canonalize_fname() -- at invocation, buf contains CWD. */

void canonalize_fname(buf,fname)
     char *buf;
     const char *fname;
{
	char *s, *p;
	struct stat64 stats;
	int rc;
	char curdir[MAXPATHLEN];

	s = strrchr(fname,'/');
	if (s == NULL) {
	  p = buf + strlen(buf);
	  if (*(p-1) != '/')
	    *p++ = '/';
	  strcpy(p,fname);
	  rc = lstat64(buf,&stats);
	  if (rc != 0 || (!S_ISREG(stats.st_mode) &&
			  !S_ISDIR(stats.st_mode))) {
	    fprintf(stderr,"File `%s' is not regular file or directory!",buf);
	    if (S_ISLNK(stats.st_mode))
	      fprintf(stderr,"  Symlink... :-)\n");
	    else {
	      fprintf(stderr,"  Rejected!\n");
	      *buf = 0;
	    }
	  }
	  if (stats.st_nlink != 1 &&
	      !S_ISDIR(stats.st_mode)) {
	    fprintf(stderr,"Foo! File `%s' has multiple (%d) hard links to it, and is not directory!   Rejected!\n",
		    buf,stats.st_nlink);
	    *buf = 0;
	  }
	  return;
	}
	*s = 0;
	rc = chdir(fname);
	*s = '/';
	if (rc != 0) {
	  fprintf(stderr,"Can't refer to directory this file is in: `%s'\n",
		  fname);
	  *buf = 0;
	  return;
	}
	*curdir = 0;
	getcwd(curdir,MAXPATHLEN);
	if (*curdir == 0) {
	  *buf = 0;
	  fprintf(stderr,"getcwd() failed! ABORT!\n");
	  exit(3);
	}
	chdir(buf);
	strcat(curdir,s);	/* Canonic name! */
	strcpy(buf,curdir);
	rc = lstat64(buf,&stats);
	if (rc != 0 ||
	    !(S_ISREG(stats.st_mode) ||
	      S_ISDIR(stats.st_mode))) {
	  if (S_ISLNK(stats.st_mode))
	    fprintf(stderr,"File `%s' is a SYMLINK.  Are you sure ?\n",buf);
	  else {
	    fprintf(stderr,"File `%s' is not regular file or directory!  Rejected!\n",buf);
	    *buf = 0;
	  }
	}
	if (stats.st_nlink != 1 &&
	      !S_ISDIR(stats.st_mode)) {
	  fprintf(stderr,"Foo! File `%s' has multiple (%d) hard links to it!   Rejected!\n",
		  buf,stats.st_nlink);
	  *buf = 0;
	}
	return;
}


void omi_file( cwd,fname )
     const char *cwd;
     const char *fname;
{
	char buf[MAXPATHLEN];
	int rc;
	struct stat64 stats;

	if (debug)
	  fprintf(stderr,"%s: cwd='%s', fname='%s'\n",pname,cwd,fname);
	strcpy(buf,cwd);
	canonalize_fname(buf,fname);
	if (debug)
	  fprintf(stderr,"%s: Canonic fname='%s'\n",pname,buf);
	if (*buf == 0) return;

	if (Match_file(buf) == 0) {
	  fprintf(stderr,"%s: Didn't match path '%s' into allowed files.\n",
		  pname,buf);
	  return;
	} else {
	  rc = stat64(buf,&stats); /* Can't fail actually... */

	  if (NewUid == -1 && NewGroup == -1 && NewMode == -1)
	    NewUid = getuid();
	  rc = chown(buf,NewUid,NewGroup);
	  if (rc != 0) perror("chown()");

	  fprintf(logfp,"%d\t%s\tuser=%s\tuid=%d\tNewUid=%d\tNewGroup=%d",
		  time(NULL),buf,logname,getuid(),NewUid,NewGroup);

	  /* NewMode is very restricted -- Only rwx bits are allowed to be
	     modified, SETUID, SETGID is too dangerous... */
      
	  if (NewMode != -1) {
	    /* Fine, new mode!
	       Won't accept anything aside of normal protection bits!
	       --- oops, for a directory*/

	    fprintf(logfp,"\tNewMode=0%o",NewMode);

	    if (S_ISREG(stats.st_mode)) { /* Regular file ! */
	      rc = chmod(buf, NewMode & 0777);
	      if (rc != 0) perror("chmod()");
	    } else if (S_ISDIR(stats.st_mode)) { /* Directory! */
	      rc = chmod(buf, NewMode & 03777); /* Sgid,sticky */
	      if (rc != 0) perror("chmod()");
	    } else
	      printf("Oops, \"%s\" is not a regular file, or directory!\n",buf);
	  }
	  if (NewMode == -1 && (NewGroup != -1 || NewUid != -1)) {
	    /* Lets see: UID or GID has changed, but didn't set new mode!
	       Lets make sure SUIDs don't follow!*/
	    if (S_ISREG(stats.st_mode)) { /* Regular file ! */
	      rc = chmod(buf, stats.st_mode & 0777);
	      if (rc != 0) perror("chmod()");
	    } else if (S_ISDIR(stats.st_mode)) { /* Directory ! */
	      rc = chmod(buf, stats.st_mode & 03777);
	      if (rc != 0) perror("chmod()");
	    } else
	      printf("Oops, \"%s\" is not a regular file, or directory!\n",buf);
	  }
	  fprintf(logfp,"\n");
	}
}
