/*
  File:	wff3x1.c
  Authors: K.R. Sloan
  Last Modified: 8 December 1987
  Purpose: convolves a wff image with a 3x1 template,
           plus a constant (DC) value.
           only those pixels which fall under the center pixel of the 
           template are changed (i.e., a 1 pixel border of the window is 
           left alone.)

           The template is applied horizontally, or vertically, depending
           on the boolean parameter "HORIZONTAL".

           Template (and DC) values are scaled integers
           (positive or negative.)  The scale factor is 2048.  So,
            2048 == 1.0, 1024 == 0.5, -512 == -0.25, etc.
           
           It is strongly recommended that the individual magnitudes not
           be too large, and that the sum of the values in the template
           (plus the DC value) lie in the range [0,2048] (i.e., [0.0,1.0])
           
 */

#include <stdio.h>
#include <wff.h>
#include <sys/file.h>
#include <sys/errno.h>

#define ShiftFactor 11
#define ScaleFactor 2048

#define ScanLineLength (wffMaxCoordinate+1)

static char RoutineName[] = "dq3x1";

int
wff3x1(fdIn, fdOut, Template, DC, HORIZONTAL)
 FILE *fdIn, *fdOut;
 int Template[3]; 
 int DC;
 int HORIZONTAL;
 {
  FrameBufferType *FBin, *FBout;
  int Bottom, Left, Top, Right, BandsPerPixel, passed;
  char WhatBands[10];
  int BitsPerBand;
  char Name[NameLength], Value[ValueLength];
  int x, y, j, k, l, b, n;
  static unsigned short A[ScanLineLength],
                        B[ScanLineLength], 
                        C[ScanLineLength],
                        NewLine[ScanLineLength];
  unsigned short *LastLine, *ThisLine, *NextLine, *t, *p;
  int dc,t0,t1,t2,value;
  int MaxPixelValue;

  FBin  = (FrameBufferType *)0;
  FBout = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBin))           {                        return 1; }
  if (FAILURE == PassImageIn(fdIn, FBin)) { (void)CloseFB(&FBin);  return 1; }

  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   { (void)CloseFB(&FBin); return 1; }
  if (FAILURE == GetColorSystem(FBin, WhatBands, &BitsPerBand))
   { (void)CloseFB(&FBin); return 1; }
  if (FAILURE == GetDescriptor(FBin,"BitsPerBand",Value))
   { (void)CloseFB(&FBin); return 1; }
  BitsPerBand = atoi(Value);
  BandsPerPixel = strlen(WhatBands);

  if (16 < BitsPerBand )
   {
    fprintf(stderr,"wff3x1: can't handle %d BitsPerBand!",BitsPerBand);
    (void)CloseFB(&FBin);  
    return 1; 
   }
  if (FAILURE == OpenFB(&FBout)) { (void)CloseFB(&FBin);  return 1; }

  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    SetDescriptor(FBout, Name, Value);
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  /* Finally, pass the pixels */

  MaxPixelValue = (1 << BitsPerBand) - 1;
  dc = DC*MaxPixelValue;
  t0 = Template[0];
  t1 = Template[1];
  t2 = Template[2];

  if (0 == HORIZONTAL)
   {
    LastLine = &A[0]; ThisLine = &B[0]; NextLine = &C[0];
    /* load up two lines of buffer, and write out one unchanged line */
    p = ThisLine;
    for (x=Left;x<=Right;x++,p += BandsPerPixel) 
     {
      if (FAILURE == NextPixelIn(FBin,p)) return 1;
      if (FAILURE == NextPixelOut(FBout,p)) return 1;
     }

    p = NextLine;
    for (x=Left;x<=Right;x++,p += BandsPerPixel) 
     if (FAILURE == NextPixelIn(FBin,p)) return 1;

    /* step through destination, one pixel at a time */
    for (y = Bottom+1; y < Top; y++)
     {
      /* move forward one line */
      t = LastLine; LastLine = ThisLine; ThisLine = NextLine; NextLine = t;
      p = NextLine;
      for (x=Left;x<=Right;x++,p += BandsPerPixel) 
       if (FAILURE == NextPixelIn(FBin,p)) return 1;
      k = 0;
      for (x = Left; x <= Right; x++)
       for (b=0;b<BandsPerPixel; b++,k++)
        {
         value  = dc + t0*LastLine[k] 
                     + t1*ThisLine[k] 
                     + t2*NextLine[k];
         value =  value >> ShiftFactor;

        if      (value < 0)             value = 0;
        else if (value > MaxPixelValue) value = MaxPixelValue;
        NewLine[k] = (unsigned short) value;
       }
      p = NewLine;
      for (x=Left;x<=Right;x++,p += BandsPerPixel) 
      if (FAILURE == NextPixelOut(FBout,p)) return 1;
     }
    /* write out last line unchanged */
    p = NextLine;
    for (x=Left;x<=Right;x++,p += BandsPerPixel) 
     if (FAILURE == NextPixelOut(FBout,p)) return 1;
   }
  else
   {
    ThisLine = &A[0];
    /* step through destination, one pixel at a time */
    for (y = Bottom; y <= Top; y++)
     {
      /* move forward one line */
      p = ThisLine;
      for (x=Left;x<=Right;x++,p += BandsPerPixel) 
       if (FAILURE == NextPixelIn(FBin,p)) return 1;

      j = -BandsPerPixel; k = 0; l = BandsPerPixel;
      for (b=0;b<BandsPerPixel; b++,j++,k++,l++)
       NewLine[k] = ThisLine[k];
   
      for (x = Left+1; x < Right; x++)
       for (b=0;b<BandsPerPixel; b++,j++,k++,l++)
        {
         value = dc; 
         value += t0*ThisLine[j];
         value += t1*ThisLine[k];
         value += t2*ThisLine[l];
         value =  value >> ShiftFactor;

        if      (value < 0)             value = 0;
        else if (value > MaxPixelValue) value = MaxPixelValue;
        NewLine[k] = (unsigned short) value;
       }
      for (b=0;b<BandsPerPixel; b++,k++)
       NewLine[k] = ThisLine[k];

      p = NewLine;
      for (x=Left;x<=Right;x++,p += BandsPerPixel) 
       if (FAILURE == NextPixelOut(FBout,p)) return 1;
     }
   }

  (void)CloseFB(&FBin);
  if (FAILURE == CloseFB(&FBout)) return 1;

  return 0;
 }
