/*
  File: wff.c
  Authors: K.R. Sloan,
           Daniel O'Donnell,
           James Painter,
           Mike Schmidt
  Last Modified: 9 January 1990
  Purpose: the Washington File Format 
           this version has (zero-based) Run Length Encoding
                        and a few petty optimizations
 */
#include <stdio.h>
#include <strings.h>
#include <sys/time.h>
#include <wff.h>

#define TRUE (-1)
#define FALSE (0)
#define AIScode (0)
#define RLEcode (1)
#define SequentialIn  (0)
#define SequentialOut (1)
#define Random        (2)
#define MaxCount    (255)

extern char *calloc();

/*
               PRIVATE PROCEDURES
 */

static int
FindByName(FrameBuffer, ThisName)
 FrameBufferType *FrameBuffer;
 char *ThisName;
 {
  char s[NameLength];
  int i;

  for (i=0; i<MaxNV; i++) /* look for match */
   if (0 == strcmp(FrameBuffer->NV[i].Name, ThisName)) return(i);
  for (i=0; i<MaxNV; i++) /* else return empty slot */
   if ('\0' == FrameBuffer->NV[i].Name[0]) return(i);
 }

static void
DestroyAux(FrameBuffer)
 FrameBufferType *FrameBuffer;
 {
  if ((unsigned char *)0  != FrameBuffer->AIS)   cfree(FrameBuffer->AIS);
  if ((unsigned short *)0 != FrameBuffer->Pixel) cfree(FrameBuffer->Pixel);
  FrameBuffer->AIS   = (unsigned char *)0; 
  FrameBuffer->Pixel = (unsigned short *)0; 
  FrameBuffer->Count = -1;
  FrameBuffer->BitsPerPixel  = 0; 
  FrameBuffer->BytesPerPixel = 0;
  FrameBuffer->VALID = FALSE;
 }

static int
CreateAux(FrameBuffer)
 FrameBufferType *FrameBuffer;
 {
  unsigned int Size;
  
  if (TRUE == FrameBuffer->VALID) return(SUCCESS);

  FrameBuffer->BitsPerPixel = (  (FrameBuffer->BandsPerPixel)
                               * (FrameBuffer->BitsPerBand  ) );
  FrameBuffer->BytesPerPixel = FrameBuffer->BitsPerPixel / 8;
  if ((FrameBuffer->BitsPerPixel % 8) > 0) (FrameBuffer->BytesPerPixel)++;

  Size =  (FrameBuffer->BytesPerPixel)
        * ((FrameBuffer->Right)-(FrameBuffer->Left)+1)
        * ((FrameBuffer->Top) -(FrameBuffer->Bottom)+1); 

  if (Random == FrameBuffer->AccessType)
   {
    FrameBuffer->AIS = (unsigned char *)calloc(Size, sizeof(unsigned char));
    if ((unsigned char *)0 == FrameBuffer->AIS) return(FAILURE);
   }
  if (RLEcode == FrameBuffer->EncodingType)
   {
    FrameBuffer->Pixel = (unsigned short *)calloc(FrameBuffer->BandsPerPixel,
                                                 sizeof(unsigned short));
    FrameBuffer->Count = -1;
    if ((unsigned short *)0 == FrameBuffer->Pixel) return(FAILURE);
   }
  FrameBuffer->VALID = TRUE;
  return(SUCCESS);
 }

static void
SetFBValues(FrameBuffer,Name,Value)
 FrameBufferType *FrameBuffer;
 char *Name, *Value;
 {
  if (0 == strcmp(Name,"Left"))
   {
    FrameBuffer->Left = atoi(Value);            DestroyAux(FrameBuffer);
   }
  if (0 == strcmp(Name,"Bottom"))
   {
    FrameBuffer->Bottom = atoi(Value);          DestroyAux(FrameBuffer);
   }
  if (0 == strcmp(Name,"Right"))
   {
    FrameBuffer->Right = atoi(Value);           DestroyAux(FrameBuffer);
   }
  if (0 == strcmp(Name,"Top"))
   {
    FrameBuffer->Top = atoi(Value);             DestroyAux(FrameBuffer);
   }
  if (0 == strcmp(Name,"BitsPerBand"))
   {
    FrameBuffer->BitsPerBand = atoi(Value);     DestroyAux(FrameBuffer);
   }
  if (0 == strcmp(Name,"ColorSystem")) 
   {
    FrameBuffer->BandsPerPixel = strlen(Value); DestroyAux(FrameBuffer);    
   }
  if (0 == strcmp(Name,"Encoding"))
   {
    if      (0 == strcmp(Value,"AIS"))
     FrameBuffer->EncodingType = AIScode;
    else if (0 == strcmp(Value,"RLE"))
     FrameBuffer->EncodingType = RLEcode;
    else
     fprintf(stderr,"SetFBValues: unknown encoding type %s\n",Value);
    DestroyAux(FrameBuffer);
   }
 }

static int 
WriteHeader(FileDesc,FrameBuffer)
 FILE *FileDesc;
 FrameBufferType *FrameBuffer;
 {
  int N;
  char Name[NameLength], Value[ValueLength];
  struct timeval tp;
  struct timezone tzp;
  char *ASCIItime;
  char hostname[128];
  int  hostnamelen = 128;

  gethostname(hostname,hostnamelen); hostname[hostnamelen-1] = '\0';

  gettimeofday(&tp,&tzp);
  ASCIItime = ctime(&tp);

  fprintf(FileDesc,
  "* Washington Image File Format 2.4 - K.R. Sloan - 2 November 1988\n");
  fprintf(FileDesc,"* Host: %s\n",hostname);
  fprintf(FileDesc,"* Time: %s",ASCIItime);   /* \n is in ASCIItime */
  fprintf(FileDesc,"* Name Value\n");

  for(N=0;N<MaxNV;N++)
   {
    GetDescriptorN(FrameBuffer, N, Name, Value);
    if ('\0' == Name[0]) break;
    fprintf(FileDesc,"%s %s\n",Name,Value);
    if (ferror(FileDesc))
     {
      fprintf(stderr,"WriteHeader: error writing %s = %s\n",Name,Value);
      return(FAILURE);
     }
   }
  fprintf(FileDesc,"* and now, the Array of Image Samples ...\n");
  fprintf(FileDesc,"\f");

  return(SUCCESS);
 }

static int 
ReadHeader(FileDesc, FrameBuffer)
 FILE *FileDesc;
 FrameBufferType *FrameBuffer;
 {
  int N;
  char Name[NameLength], Value[ValueLength], c;

  for(N=0;N<MaxNV;N++)
   {
    c = getc(FileDesc);
    if ( ferror(FileDesc) || feof(FileDesc) )
     {
      fprintf(stderr,"ReadHeader: error?\n");
      return(FAILURE);
     }
    if ('\f' == c) break;
    if ('*'  == c)
     { /* ignore comments */
      fscanf(FileDesc,"%*[^\n]%c",&c);
      N--; 
     }
    else
     {
      Name[0] = c;
      if (1 != fscanf(FileDesc,"%s ",&Name[1]))
       {
         fprintf(stderr,"ReadHeader: error!\n");
         return(FAILURE);
       }
      fscanf(FileDesc,"%[^\n]%c",Value,&c);
      SetDescriptor(FrameBuffer,Name,Value);
     }
   }
  for(;'\f' != c;)
   {
    c = getc(FileDesc);
    if (ferror(FileDesc) || feof(FileDesc))
     {
      fprintf(stderr,"ReadHeader: error?\n");
      return(FAILURE);
     }
   }

  for (;N<MaxNV;N++)
   { 
    FrameBuffer->NV[N].Name[0] = '\0';
    FrameBuffer->NV[N].Value[0] = '\0';
   }
  return(SUCCESS);
 }

static int
WriteRLEPair (FileDesc, FrameBuffer, Count, Value, x, y)
 FILE *FileDesc;
 FrameBufferType *FrameBuffer;
 int Count;
 unsigned char *Value;
 int x, y;
 {
  register unsigned char *p, *pend;

  (void) putc(Count, FileDesc);
  if (ferror(FileDesc))
   {
    fprintf(stderr, "WriteRLEPair: Error x=%d, y=%d\n",x,y);
    return(FAILURE);
   }

  p = Value; pend = Value+(FrameBuffer->BytesPerPixel);
  for(;p<pend;p++) (void)putc(*p, FileDesc);

  if (ferror(FileDesc))
   {
    fprintf(stderr, "WriteRLEPair: Error x=%d, y=%d\n",x,y);
    return(FAILURE);
   }
  return(SUCCESS);
 }

static int 
WriteArray(FileDesc,FrameBuffer)
 FILE *FileDesc;
 FrameBufferType *FrameBuffer;
 {
  int x, y;
  unsigned char *p, *prevalue;
  int count, bpp, B, T, L, R;
  unsigned char *p1, *p2;
  int b, match;

  if (   (TRUE != FrameBuffer->VALID)
      && (FAILURE == CreateAux(FrameBuffer)) )
   {
    fprintf(stderr,"WriteArray: can't allocate array\n");
    return(FAILURE);
   }
  p = FrameBuffer->AIS;
  bpp = FrameBuffer->BytesPerPixel;
  B = FrameBuffer->Bottom; T = FrameBuffer->Top;
  L = FrameBuffer->Left;   R = FrameBuffer->Right;
  if (RLEcode == FrameBuffer->EncodingType)
   { 
     /* initialize cleverly to save a test in the inner loop       */
     /* prevalue points at the value of the existing run           */
     /* p points at the pixel to be tested for equality            */
     /* count of ZERO means ONE in the current run                 */
     /* L is temporarily incremented to shorten the first scanline */ 
    prevalue = p;  p += bpp; count = 0; L++;  /* skip over first value */
    for (y=B;y<=T;y++)
     {
      for (x=L; x<=R; x++,p += bpp)
       {
        match = TRUE; p1 = prevalue; p2 = p;
        for (b=0; b<bpp; b++)
         if (*(p1++) != *(p2++)) { match=FALSE; break; }
       
        if ( (TRUE == match) && (MaxCount > count) ) count++; 
        else
         {
          if (FAILURE == WriteRLEPair (FileDesc,FrameBuffer,count,
                                       prevalue, x, y))
           return(FAILURE);
          prevalue = p;  count = 0;  /* start of a new run */
         }
       }
      if (y == B) L--;    /* adjust Left to true value for remaining lines */
     }
    /* write final run  (there will always be one) */
    return(WriteRLEPair (FileDesc,FrameBuffer,count, prevalue, x, y));
   }
  else if (AIScode == FrameBuffer->EncodingType)
   {
    for (y=B;y<=T;y++)
     for (x=L; x<=R; x++,p += bpp)
      {
       fwrite(p,
              sizeof(unsigned char),
              bpp,
              FileDesc);
       if ( ferror(FileDesc) )
        {
         fprintf(stderr,"WriteArray: Error x=%d, y=%d\n",x,y);
         return(FAILURE);
        }
      }
   }
  else
   {
    fprintf(stderr,"WriteArray: Unknown encoding!");
    return(FAILURE);
   }
  fflush(FileDesc);
  return(SUCCESS);
 }

static int
ReadArray(FileDesc, FrameBuffer)
 FILE *FileDesc;
 FrameBufferType *FrameBuffer;
 {
  int BytesRead;
  int x, y;
  int count;
  unsigned char *p, *prevalue, *pend;
  int b, bpp;
  int B, T, L, R;

  if (   (TRUE != FrameBuffer->VALID)
      && (FAILURE == CreateAux(FrameBuffer)) )
   {
    fprintf(stderr,"ReadArray: can't allocate array\n");
    return(FAILURE);
   }

  p = FrameBuffer->AIS;
  bpp = FrameBuffer->BytesPerPixel;
  B = FrameBuffer->Bottom;  T = FrameBuffer->Top;
  L = FrameBuffer->Left;    R = FrameBuffer->Right;
  if (RLEcode == FrameBuffer->EncodingType)
   {
    count = -1;   /* not in a run */
    for (y=B;y<=T;y++)
     for (x=L; x<=R; x++)
      {
       pend = p + bpp;
       if (0 > count)
        {
         count = getc(FileDesc);
         if ( ferror(FileDesc) || feof(FileDesc) )
          {
           fprintf(stderr,"ReadArray: error x=%d, y=%d, BytesRead=%d\n",
                              x,y,BytesRead);
           return(FAILURE);
          }
         prevalue = p;
         for(;p<pend;) *(p++) = getc(FileDesc);
         if (ferror(FileDesc) || feof(FileDesc))
          {
           fprintf(stderr,"ReadArray: error x=%d, y=%d, BytesRead=%d\n",
                               x,y,BytesRead);
           return(FAILURE);
          }
        }
       else 
        for (;p<pend;) *(p++) = *(prevalue++);
       count--;
      }
   }
  else if (AIScode == FrameBuffer->EncodingType)
   {
    for (y=B;y<=T;y++)
     for (x=L; x<=R; x++)
      {
       pend = p + bpp;
       for(;p<pend;) *(p++) = getc(FileDesc);
       if ( ferror(FileDesc) || feof(FileDesc) )
        {
         fprintf(stderr,"ReadArray: error x=%d, y=%d, BytesRead=%d\n",
                           x,y,BytesRead);
         return(FAILURE);
        }
      }
   }
  else 
   {
    fprintf(stderr, "ReadArray: Unknown encoding!");
    return(FAILURE);
   }
  return(SUCCESS);
 }


/*
            EXPORTED PROCEDURES
 */

int GetBounds(FrameBuffer, Bottom, Left, Top, Right)
 FrameBufferType *FrameBuffer;
 int *Bottom, *Left, *Top,*Right;
 {
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"GetBounds: no FrameBuffer!\n");
    return(FAILURE);
   }
  *Bottom = FrameBuffer->Bottom;
  *Left   = FrameBuffer->Left;
  *Top    = FrameBuffer->Top;
  *Right  = FrameBuffer->Right;
  return(SUCCESS);
 }

int SetBounds(FrameBuffer, Bottom, Left, Top, Right)
 FrameBufferType *FrameBuffer;
 int Bottom,Left,Top,Right;
 {
  char s[10];
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"SetBounds: no FrameBuffer!\n");
    return(FAILURE);
   }
  if (   (Top  <0) || (wffMaxCoordinate < Top) 
      || (Right<0) || (wffMaxCoordinate < Right)
      || (Top < Bottom) || (Right < Left)  )
   {
    fprintf(stderr,"SetBounds: cannot set BLTR to %d,%d,%d,%d!\n",
                    Bottom,Left,Top,Right);
    fprintf(stderr,"SetBounds: coordinates must lie in [0, %d]\n",
                    wffMaxCoordinate);
    return(FAILURE);
   }
  sprintf(s,"%4d\0",Bottom);
  if (FAILURE == SetDescriptor(FrameBuffer,"Bottom",s)) return(FAILURE);
  sprintf(s,"%4d\0",Left);
  if (FAILURE == SetDescriptor(FrameBuffer,"Left",s))   return(FAILURE);
  sprintf(s,"%4d\0",Top);
  if (FAILURE == SetDescriptor(FrameBuffer,"Top",s))    return(FAILURE);
  sprintf(s,"%4d\0",Right);
  if (FAILURE == SetDescriptor(FrameBuffer,"Right",s))  return(FAILURE);
  sprintf(s,"%4d\0",Top-Bottom+1);
  if (FAILURE == SetDescriptor(FrameBuffer,"height",s)) return(FAILURE);
  sprintf(s,"%4d\0",Right-Left+1);
  if (FAILURE == SetDescriptor(FrameBuffer,"width",s))  return(FAILURE);

  return (SUCCESS);
 }

int GetColorSystem(FrameBuffer, WhatBands, BitsPerBand)
 FrameBufferType *FrameBuffer;
 char *WhatBands;
 int *BitsPerBand;
 {
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"GetColorSystem: no FrameBuffer!\n");
    return(FAILURE);
   }
  if (FAILURE == GetDescriptor(FrameBuffer,"ColorSystem",WhatBands))
   return(FAILURE);
  *BitsPerBand = FrameBuffer->BitsPerBand;
  return(SUCCESS);
 }

int SetColorSystem(FrameBuffer, WhatBands, BitsPerBand)
 FrameBufferType *FrameBuffer;
 char *WhatBands;
 int BitsPerBand;
 {
  char s[40];
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"SetColorSystem: no FrameBuffer!\n");
    return(FAILURE);
   }
  if (FAILURE == SetDescriptor(FrameBuffer,"ColorSystem",WhatBands))
   return(FAILURE);
  sprintf(s,"%5d\0",BitsPerBand);
  if (FAILURE == SetDescriptor(FrameBuffer,"BitsPerBand",s))
   return(FAILURE);
  return(SUCCESS);
 }

int SetDescriptor(FrameBuffer, Name, Value)
 FrameBufferType *FrameBuffer;
 char *Name;
 char *Value;
 {
  int i;
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"SetDescriptor: no FrameBuffer!\n");
    return(FAILURE);
   }
  i = FindByName(FrameBuffer, Name);
  strcpy(FrameBuffer->NV[i].Name, Name);
  strcpy(FrameBuffer->NV[i].Value, Value);
  SetFBValues(FrameBuffer,Name,Value);
  return(SUCCESS);
 }

int GetDescriptor(FrameBuffer, Name, Value)
 FrameBufferType *FrameBuffer;
 char *Name;
 char *Value;
 {
  int i;
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"GetDescriptor: no FrameBuffer!\n");
    return(FAILURE);
   }
  i = FindByName(FrameBuffer, Name);
  strcpy(Value, FrameBuffer->NV[i].Value);
  return(SUCCESS);
 }

int GetDescriptorN(FrameBuffer, N, Name, Value)
 FrameBufferType *FrameBuffer;
 int N; 
 char *Name;
 char *Value;
 {
  int i;

  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"GetDescriptorN: no FrameBuffer!\n");
    return(FAILURE);
   }
  strcpy(Name, FrameBuffer->NV[N].Name);
  strcpy(Value, FrameBuffer->NV[N].Value);
  return(SUCCESS);
 }

int OpenFB(FrameBufferPtr)
 FrameBufferType **FrameBufferPtr;
 {
  int i;
  FrameBufferType *FrameBuffer;
  int OK;

  FrameBuffer = *FrameBufferPtr;
  if ((FrameBufferType *)0 != FrameBuffer)
   {
    fprintf(stderr,"OpenFB: already open!\n");
    return(FAILURE);
   }
  FrameBuffer = (FrameBufferType *)calloc(1, sizeof (FrameBufferType));
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"OpenFB: can't allocate FrameBuffer!\n");
    return(FAILURE);
   }

  for (i=0;i<MaxNV;i++)
   { 
    FrameBuffer->NV[i].Name[0] = '\0';
    FrameBuffer->NV[i].Value[0] = '\0';
   }
                     FrameBuffer->Top = 511;
  FrameBuffer->Left = 0;                 FrameBuffer->Right = 511;
                     FrameBuffer->Bottom = 0;   
  OK = SetColorSystem(FrameBuffer,"RGB",8);
  if (SUCCESS == OK)
   OK = SetBounds(FrameBuffer,FrameBuffer->Bottom,FrameBuffer->Left,
                               FrameBuffer->Right,FrameBuffer->Top);
  if (SUCCESS == OK)
   OK = SetDescriptor(FrameBuffer,"Encoding","AIS");
  if (SUCCESS == OK)
   OK = SetDescriptor(FrameBuffer,"AspectRatio","1.0");
  if (SUCCESS == OK)
   OK = SetDescriptor(FrameBuffer,"Title","none");

  if (FAILURE == OK) 
   {
    cfree(FrameBuffer); FrameBuffer = (FrameBufferType *)0;
   }
  FrameBuffer->AccessType = Random;
  FrameBuffer->FileDescriptor = (FILE *)0;
  *FrameBufferPtr = FrameBuffer;
  return(OK);
 }

int CloseFB(FrameBufferPtr)
 FrameBufferType **FrameBufferPtr;
 {
  if ((FrameBufferType *)0 == *FrameBufferPtr)
   {
    fprintf(stderr,"CloseFB: no FrameBuffer!\n");
    return(FAILURE);
   }
  if (SequentialOut == (*FrameBufferPtr)->AccessType)
   wffFlush(*FrameBufferPtr);
  DestroyAux(*FrameBufferPtr);
  cfree((char *)*FrameBufferPtr);
  *FrameBufferPtr = (FrameBufferType *)0;
  return(SUCCESS);
 }

int ReadImage(FileDescriptor, FrameBuffer)
 FILE *FileDescriptor;
 FrameBufferType *FrameBuffer;
 {
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"ReadImage: no FrameBuffer!\n");
    return(FAILURE);
   }
  if (Random != FrameBuffer->AccessType)
   {
    fprintf(stderr,"ReadImage: not open for random access!\n");
    return(FAILURE);
   }
  if (FAILURE == ReadHeader(FileDescriptor,FrameBuffer)) return(FAILURE);
  if (FAILURE == ReadArray (FileDescriptor,FrameBuffer)) return(FAILURE);
  return(SUCCESS);
 }

int WriteImage(FileDescriptor, FrameBuffer)
 FILE *FileDescriptor;
 FrameBufferType *FrameBuffer;
 {
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"WriteImage: no FrameBuffer!\n");
    return(FAILURE);
   }

  if (Random != FrameBuffer->AccessType)
   {
    fprintf(stderr,"WriteImage: not open for random access!\n");
    return(FAILURE);
   }

  if (FAILURE == WriteHeader(FileDescriptor,FrameBuffer)) return(FAILURE);
  if (FAILURE == WriteArray(FileDescriptor,FrameBuffer))  return(FAILURE);
  return(SUCCESS);
 }

int
GetPixel(FrameBuffer, x, y, Pixel)
 FrameBufferType *FrameBuffer;
 int  x, y;
 unsigned short  *Pixel;
 {
  int u,v,w,i,bp,bs,bb,band,bands,sample;
  unsigned char *p,*q;
  unsigned char ThisByte;

  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"GetPixel: no FrameBuffer!\n");
    return(FAILURE);
   }

  if (Random != FrameBuffer->AccessType)
   {
    fprintf(stderr,"GetPixel: not open for random access!\n");
    return(FAILURE);
   }

  if (   (TRUE != FrameBuffer->VALID)
      && (FAILURE == CreateAux(FrameBuffer)) )
   {
    fprintf(stderr,"GetPixel: can't allocate array\n");
    return(FAILURE);
   }

  u = x-(FrameBuffer->Left);
  v = y-(FrameBuffer->Bottom);
  w = (FrameBuffer->Right)-(FrameBuffer->Left)+1;
  i = u + v*w;
  p = &(FrameBuffer->AIS[i*(FrameBuffer->BytesPerPixel)]);

  bands = FrameBuffer->BandsPerPixel;
  bb = FrameBuffer->BitsPerBand;
  bp = 0;
  
  for(band=0;band<bands;band++)
   {
    sample = 0; bs = 0;
    if (0 == bp) 
     { 
      if (0 != band) ++p; 
      ThisByte = *p; bp = 8; 
     }
    while ((bb-bs)>=bp)
     {
      sample = (sample << bp) | (ThisByte >> (8-bp));
      bs += bp; 
      if (bs<bb) { ThisByte = *(++p); bp = 8; }
      else       { bp = 0; break;}
     }
    if (bb>bs)
     {
      sample = (sample << (bb-bs)) | (ThisByte >> (8-bb+bs));
      ThisByte = ThisByte << (bb-bs);
      bp -= (bb-bs);
     }
    *(Pixel++) = sample;
   }
  return(SUCCESS);
 }

int
PutPixel(FrameBuffer, x, y, Pixel)
 FrameBufferType *FrameBuffer;
 int  x, y;
 unsigned short  *Pixel;
 {
  int u,v,w,i,bp,bs,bb,band,bands,sample;
  unsigned char *p,*q;
  unsigned char ThisByte;

  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"PutPixel: no FrameBuffer!\n");
    return(FAILURE);
   }

  if (Random != FrameBuffer->AccessType)
   {
    fprintf(stderr,"PutPixel: not open for random access!\n");
    return(FAILURE);
   }

  if (   (TRUE != FrameBuffer->VALID)
      && (FAILURE == CreateAux(FrameBuffer)) )
   {
    fprintf(stderr,"PutPixel: can't allocate array\n");
    return(FAILURE);
   }

  u = x-(FrameBuffer->Left);
  v = y-(FrameBuffer->Bottom);
  w = (FrameBuffer->Right)-(FrameBuffer->Left)+1;
  i = u + v*w;
  p = &(FrameBuffer->AIS[i*(FrameBuffer->BytesPerPixel)]);

  bands = FrameBuffer->BandsPerPixel;
  bb = FrameBuffer->BitsPerBand;
  bp = 0;

  for(band=0;band<bands;band++)
   {
    sample = (*(Pixel++) << (16 - bb)) & 0xffff;
    bs = bb;
    if ((0 != band) && (0 == bp)) p++;
    while ((8-bp)<=bs)
     {
      ThisByte = (ThisByte << (8-bp)) | (sample >> (16-8+bp));
      sample = sample << (8-bp);  bs -=  (8-bp);
      *p = ThisByte; bp = 0;
      if (bs>0) p++; 
     }
    if (bs>0)
     {
      ThisByte = (ThisByte << bs) | (sample >> (16-bs));
      bp += bs;
     }
   }
  if (bp>0) *p = ThisByte << (8-bp);
  return(SUCCESS);
 }

int
GetBlock(FrameBuffer,bottom,left,top,right, Block)
 FrameBufferType *FrameBuffer;
 int bottom,left,top,right;
 unsigned short *Block;
 {
  register int x,y;
  register unsigned short *b;
  int bands;

  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"GetBlock: no FrameBuffer!\n");
    return(FAILURE);
   }

  if (Random != FrameBuffer->AccessType)
   {
    fprintf(stderr,"GetBlock: not open for random access!\n");
    return(FAILURE);
   }

  if (   (bottom < FrameBuffer->Bottom)
      || (left   < FrameBuffer->Left)
      || (right  > FrameBuffer->Right)
      || (top    > FrameBuffer->Top)
      || (bottom > top)
      || (left   > right) )
   {
    fprintf(stderr,"GetBlock: bad window\n");
    return(FAILURE);
   }

  bands = FrameBuffer->BandsPerPixel;
  b = Block;
  for (y=bottom;y<=top;y++)
   for(x=left; x<=right; x++,b += bands)
    if (FAILURE == GetPixel(FrameBuffer,x,y,b)) return(FAILURE);
  return(SUCCESS);
 }

int 
PutBlock(FrameBuffer,bottom,left,top,right,Block)
 FrameBufferType *FrameBuffer;
 int bottom,left,top,right;
 unsigned short *Block;
 {
  register int x,y;
  register unsigned short *b;
  int bands;

  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"PutBlock: no FrameBuffer!\n");
    return(FAILURE);
   }

  if (Random != FrameBuffer->AccessType)
   {
    fprintf(stderr,"PutBlock: not open for random access!\n");
    return(FAILURE);
   }

  if (   (bottom < FrameBuffer->Bottom)
      || (left   < FrameBuffer->Left)
      || (right  > FrameBuffer->Right)
      || (top    > FrameBuffer->Top)
      || (bottom > top)
      || (left   > right) )
   {
    fprintf(stderr,"PutBlock: bad window\n");
    return(FAILURE);
   }

  b = Block;
  bands = FrameBuffer->BandsPerPixel;
  for (y=bottom;y<=top;y++)
   for (x=left; x<=right; x++,b += bands)
    if (FAILURE == PutPixel(FrameBuffer,x,y,b)) return(FAILURE);
  return(SUCCESS);
 }

int
PutConstant(FrameBuffer,bottom,left,top,right,pixel)
 FrameBufferType *FrameBuffer;
 int bottom,left,top,right;
 unsigned short *pixel;
 {
  register int x,y;

  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"PutConstant: no FrameBuffer!\n");
    return(FAILURE);
   }

  if (Random != FrameBuffer->AccessType)
   {
    fprintf(stderr,"PutConstant: not open for random access!\n");
    return(FAILURE);
   }

  if (   (bottom < FrameBuffer->Bottom)
      || (left   < FrameBuffer->Left)
      || (right  > FrameBuffer->Right)
      || (top    > FrameBuffer->Top)
      || (bottom > top)
      || (left   > right) )
   {
    fprintf(stderr,"PutConstant: bad window\n");
    return(FAILURE);
   }

  for (y=bottom;y<=top;y++)
   for (x=left;x<=right;x++)
    if (FAILURE == PutPixel(FrameBuffer,x,y,pixel)) return(FAILURE);
  return(SUCCESS);
 }

int
PassImageIn(FileDescriptor, FrameBuffer)
 FILE *FileDescriptor;
 FrameBufferType *FrameBuffer;
 {
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"PassImageIn: no FrameBuffer!\n");
    return(FAILURE);
   }
  DestroyAux(FrameBuffer);
  FrameBuffer->AccessType = SequentialIn;
  FrameBuffer->FileDescriptor = FileDescriptor;
  if (FAILURE == CreateAux(FrameBuffer))                 return(FAILURE);
  if (FAILURE == ReadHeader(FileDescriptor,FrameBuffer)) return(FAILURE);
  return(SUCCESS);
 }

int
PassImageOut(FileDescriptor, FrameBuffer)
 FILE *FileDescriptor;
 FrameBufferType *FrameBuffer;
 {
  if ((FrameBufferType *)0 == FrameBuffer)
   {
    fprintf(stderr,"PassImageOut: no FrameBuffer!\n");
    return(FAILURE);
   }
  DestroyAux(FrameBuffer);
  FrameBuffer->AccessType = SequentialOut;
  FrameBuffer->FileDescriptor = FileDescriptor;
  if (FAILURE == CreateAux(FrameBuffer))                  return(FAILURE);
  if (FAILURE == WriteHeader(FileDescriptor,FrameBuffer)) return(FAILURE);
  return(SUCCESS);
 }

int
NextNPixelsIn(FrameBuffer, N, pixels)
 FrameBufferType *FrameBuffer;     
 int N;
 unsigned short *pixels;
 {
  register FrameBufferType *FB;
  register unsigned short *p, *pend;
  register int  i, count;
  register int bp, bs, sample;
  int n, bb, band, Bands;
  unsigned char ThisByte;
  FILE *fd;
  int RunLengthEncoded;
  unsigned short *pixel;

  if ((FrameBufferType *)0 == (FB = FrameBuffer))
    {
      fprintf(stderr,"NextNPixelsIn: no FrameBuffer!\n");
      return(FAILURE);
    }
  
  if (SequentialIn != FB->AccessType)
   {
    fprintf(stderr,"NextNPixelsIn: not open for Sequential Input!\n");
    return(FAILURE);
   }

  if ((FILE *)0 == (fd = (FB->FileDescriptor)))
   {
    fprintf(stderr,"NextNPixelsIn: no File!\n");
    return(FAILURE);
   }

  if (   (TRUE != FB->VALID)
      && (FAILURE == CreateAux(FrameBuffer)) )
   {
    fprintf(stderr,"NextNPixelsIn: no AIS?\n");
    return(FAILURE);
   }

  RunLengthEncoded = (RLEcode == FB->EncodingType);
  Bands = FB->BandsPerPixel;
  bb = FB->BitsPerBand;

  /*
     Handle the common case of 8 bits per band quickly.
   */
  if (8 == bb) 
   {
    if (!RunLengthEncoded)	/* AIS encoding */
     {
      count = N*Bands;
      p = pixels; pend = p+count;
      while (p < pend) *p++ = (unsigned char) getc(fd);
      if (ferror(fd) || feof(fd))
       {
        fprintf(stderr, "NextNPixelsIn: read error?\n");
        return(FAILURE);
       }
      return(SUCCESS);
     }
    else			/* Run length encoded */
     {
      for(n=0, pixel=pixels; n<N; )
       {
        if (0 > FB->Count) /* Get new run */
         {
          FB->Count = getc(fd);
          p = FB->Pixel; pend = p+Bands;
          while (p < pend) *p++ = (unsigned char) getc(fd);
          if (ferror(fd) || feof(fd))
           {
            fprintf(stderr,"NextPixelIn: read error?\n");
            return(FAILURE);
           }
         }
        while(n<N && FB->Count >=0)
         {
          p=FrameBuffer->Pixel; pend=p+Bands;
          while( p < pend) *pixel++ = *p++;
          FB->Count--; n++;
         }
       }
      return(SUCCESS);
     }
    }

  /* 
    This code handles the general case 
   */
  for(n=0,pixel=pixels; n<N; n++, pixel+=Bands)
   {	/* For each requested pixel... */
    if (RunLengthEncoded)
     {
      if (0 > FB->Count)
       {
        FB->Count = getc(fd);
        if (ferror(fd) || feof(fd))
         {
          fprintf(stderr,"NextNPixelsIn: read error?\n");
          return(FAILURE);
         }
       }  
      else
       {
        while (n<N && FB->Count >= 0)
         {
          p=FrameBuffer->Pixel; pend=p+Bands;
          while (p < pend) *pixel++ = *p++;
          FB->Count--; n++;
         }
        n--; pixel -= Bands; /* Back off one */
        continue;            /* Next pixel */
       }
     }
      
    ThisByte = (unsigned char) getc(fd);
    if (ferror(fd) || feof(fd))
     {
      fprintf(stderr,"NextNPixelsIn: read error?\n");
      return(FAILURE);
     }
      
    bp = 0;
    for(band=0;band<Bands;band++)
     {
      sample = 0; bs = 0;
      if (0 == bp) 
       {
        if (0 != band)
         {
          ThisByte = (unsigned char) getc(fd);
          if (ferror(fd) || feof(fd))
           {
            fprintf(stderr,"NextNPixelsIn: read error?\n");
            return(FAILURE);
           }
         } 
        bp = 8; 
       }
      while ((bb-bs)>=bp)
       {
        sample = (sample << bp) | (ThisByte >> (8-bp));
        bs += bp; 
        if (bs<bb) 
         {
          ThisByte = (unsigned char) getc(fd);
          if (ferror(fd) || feof(fd))
           {
            fprintf(stderr,"NextNPixelsIn: read error?\n");
            return(FAILURE);
           }
          bp = 8;
         }
        else       { bp = 0; break; }
       }
      if (bb>bs)
       {
        sample = (sample << (bb-bs)) | (ThisByte >> (8-bb+bs));
        ThisByte = ThisByte << (bb-bs);
        bp -= (bb-bs);
       }
      pixel[band] = sample;
     }
      
    if (RunLengthEncoded)
     {
      p = FrameBuffer->Pixel; pend = p+Bands;
      while (p < pend) *p++ = *pixel++;
      FB->Count--; 
      pixel -= Bands;
     }
   }
  return(SUCCESS);
 }


static int
PixelBytesOut(FrameBuffer, pixel)
 FrameBufferType *FrameBuffer;            
 unsigned short *pixel;
 {
  register FrameBufferType *FB;
  int bp,bs,bb,band,bands,sample;
  unsigned int p;
  unsigned char ThisByte;
  FILE *fd;

  fd = (FB = FrameBuffer)->FileDescriptor;
  bands = FB->BandsPerPixel;
  bb = FB->BitsPerBand;
  bp = 0;
  ThisByte = 0;

  for(band=0;band<bands;band++)
   {
    sample = (*(pixel++) << (16 - bb)) & 0xffff;
    bs = bb;

    while ((8-bp)<=bs)
     {
      ThisByte = (ThisByte << (8-bp)) | (sample >> (16-8+bp));
      sample = sample << (8-bp);  bs -=  (8-bp);
      p =  ThisByte;
      (void) putc (p, fd);
      if (ferror(fd))
       {
        fprintf(stderr,"PixelBytesOut: write error?\n");
        return(FAILURE);
       }
      bp = 0;
     }
    if (bs>0)
     {
      ThisByte = (ThisByte << bs) | (sample >> (16-bs));
      bp += bs;
     }
   }

  if (bp>0)
   {
    p = (ThisByte << (8-bp));
    (void) putc (p, fd);
    if (ferror(fd))
     {
      fprintf(stderr,"PixelBytesOut: write error?\n");
      return(FAILURE);
     }
   }
  return(SUCCESS);
 }

int
NextNPixelsOut(FrameBuffer, N, pixels)
 FrameBufferType *FrameBuffer;            
 int N;
 unsigned short *pixels;
 {
  register FrameBufferType *FB;
  register unsigned short *pixel;
  FILE *fd;
  int match, band, bands, bb;
  unsigned short *b1, *b2, *bend;
  int n;

  if ((FrameBufferType *)0 == (FB = FrameBuffer))
   {
    fprintf(stderr,"NextNPixelsOut: no FrameBuffer!\n");
    return(FAILURE);
   }

  if (SequentialOut != FB->AccessType)
   {
    fprintf(stderr,"NextNPixelsOut: not open for Sequential Output!\n");
    return(FAILURE);
   }

  if ((FILE *)0 == (fd = (FB->FileDescriptor)))
   {
    fprintf(stderr,"NextNPixelsOut: no File!\n");
    return(FAILURE);
   }

  if (   (TRUE != FB->VALID)
      && (FAILURE == CreateAux(FrameBuffer)) )
   {
    fprintf(stderr,"NextNPixelsOut: no AIS?\n");
    return(FAILURE);
   }

  bands = FB->BandsPerPixel;
  bb    = FB->BitsPerBand;
  if (RLEcode == FB->EncodingType)
   {
    for(n=0, pixel=pixels; n<N; n++, pixel+=bands)
     {
      if (0 <=  FB->Count)
       {
        match = TRUE; b1 = FB->Pixel; bend = b1 + bands; b2 = pixel;
        for(;b1<bend;)
         if (*(b1++) != *(b2++)) { match = FALSE; break; }
        if ((TRUE == match) && (MaxCount > FB->Count))
         FB->Count++;
        else 
         {
          (void) putc (FB->Count, fd);
          if (ferror(fd))
           {
            fprintf(stderr,"NextNPixelsOut: write error?\n");
            return(FAILURE);
           }
          if (8 != bb) 
           {
            if (FAILURE == PixelBytesOut(FrameBuffer,FrameBuffer->Pixel))
             { 
              fprintf(stderr,"NextNPixelsOut: write error?\n");
              return(FAILURE);
             }
            b1=FB->Pixel; bend = b1 + bands; b2 = pixel; 
            for(;b1<bend;) *(b1++) = *(b2++);
            FB->Count = 0; 
           }
          else 
           { /* 8 bit case, faster than PixelBytesOut */
            b1=FB->Pixel; bend = b1 + bands; b2 = pixel; 
            for(;b1<bend;)  
             {
              (void) putc((unsigned char) *b1, fd);
              *(b1++) = *(b2++);
             }
            if (ferror(fd))
             {
              fprintf(stderr,"NextNPixelsOut: write error?\n");
              return(FAILURE);
             }
            FB->Count = 0; 
           }
         }
       }
      else
       {
        b1=FB->Pixel; bend = b1 + bands; b2 = pixel; 
        for(;b1<bend;) *(b1++) = *(b2++);
        FB->Count = 0; 
       }
     }
   }
  else 
   {
    if (8 != bb) 
     {
      pixel = pixels;
      for(n=0; n<N; n++)
       {
        if (FAILURE == PixelBytesOut(FrameBuffer, pixel))
         return(FAILURE);
        pixel += bands;
       }
     }
    else 
     {	/* 8 bit case */
      b1 = pixels;
      bend = b1 + bands*N;
      for(;b1<bend;b1++)
      (void) putc((unsigned char) *b1, fd);
      if (ferror(fd))
       {
        fprintf(stderr,"NextNPixelsOut: write error?\n");
        return(FAILURE);
       }
     }
   }
  return(SUCCESS);
 }

int wffFlush (FrameBuffer)
 FrameBufferType *FrameBuffer;
 {
  register FrameBufferType *FB;
  FILE *fd;

  if ((FrameBufferType *)0 == (FB = FrameBuffer))
   {
    fprintf (stderr,"wffFlush: No FrameBuffer!\n");
    return(FAILURE);
   }

/*
  if (SequentialOut != FB->AccessType)
   {
    fprintf(stderr,"wffFlush: not open for Sequential Output!\n");
    return(FAILURE);
   }
 */

  if ((FILE *)0 == (fd = (FB->FileDescriptor)))
   {
    fprintf(stderr,"wffFlush: No File!\n");
    return(FAILURE);
   }

  if (RLEcode == FB->EncodingType)
   {
    if (0 <= FB->Count)
     {
      (void) putc (FB->Count, fd);
      if (ferror(fd))
       {
        fprintf(stderr,"wffFlush: write error?\n");
        return(FAILURE);
       }
      if (FAILURE == PixelBytesOut(FrameBuffer,FB->Pixel))
       {
        return(FAILURE);
       }
      FB->Count = -1;
     }
   }
  fflush(fd);
  return(SUCCESS);
 }
