/*
 File: vcr2wff.c
 Author: K.R. Sloan
 Last Modified: 17 February 1992
 Purpose: covert from VICAR format to .wff

 NOTE: this program is based on bare minimum specifications for VICAR
       images.  It works on the images that I have seen, and on the images
       produced by wff2vcr (of course?).  It is not to be taken as a 
       specification of VICAR images.  If you spot an obvious error, or
       know enough to provide guidance on further extensions, please
       contact <sloan@cis.uab.edu>

 */
#include <assert.h>
#include <stdio.h>
#include <strings.h>
#include "wff.h"
char *calloc();

static int VERBOSE = 0;
static char *RoutineName;
static usage()
 {
  fprintf(stderr, "Usage is:\n\t%s [-h][-v] -f vcr > wff\n", RoutineName);
 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s); exit(-1);
 }

/* VICAR stuff */
 /* tags, and guesses as to meaning... */
static int  LBLSIZE;       /* size of header, must be int mult of NS */
static char FORMAT[80];    /* 'BYTE' is OK */
static char TYPE[80];      /* 'IMAGE' is OK */
static int  BUFSIZe;       /* integer multiple of NS ? */
static int  DIM;           /* == 3? */
static int  EOL;           /* == 0? */
static int  RECSIZE;       /* == LBLSIZE? */
static char ORG[80];       /* `BSQ` is OK */
static int  NL;            /* height */
static int  NS;            /* width */ 
static int  NB;            /* samples per pixel? */
static int  N1;            /* == NL? */ 
static int  N2;            /* == NS? */
static int  N3;            /* == NB? */
static int  N4;            /* 0 is OK */
static int  NBB;           /* 0 is OK */ 
static int  NLB;           /* 0 is OK */
static char HOST[80];      /* machine type? */
static char INTFMT[80];    /* integer format? */
static char REALFMT[80];   /* real format? */
static char TASK[80];      /* processing applied? */
static char USER[80];      /* who was responsible? */
static char DAT_TIM[80];   /* when? */
static char COMMENT[80];   /* comment! */

static void ParseVICARHeader(fd)
 FILE *fd;
 {
  char Name[81],Value[1024];

  fscanf(fd,"%80[^=]=%d",Name,&LBLSIZE);
  if (VERBOSE) fprintf(stderr,"[%s = %d]\n",Name,LBLSIZE);
  if (0 != strcmp("LBLSIZE",Name)) FatalError("This is not a VICAR file");

  while(ftell(fd) < LBLSIZE)
   {
    register char *c;
    int done;

    fscanf(fd," %80[^=]=",Name);
    if ('\0' == Name[0]) break;
    Value[0] = fgetc(fd);
    c = &Value[1];
    if ('\'' == Value[0]) 
     for(;;c++)
      {
       *c = fgetc(fd);
       if ('\'' == *c) { *++c = '\0'; break; }
      }
    else 
     for(;;c++)
      {
       *c = fgetc(fd);
       if (' ' == *c)  { *c = '\0'; done =  0; break;}
       if ('\0' == *c) {            done = -1; break;}
      }

    if (VERBOSE) fprintf(stderr,"[%s = %s]\n",Name,Value);
    Value[80] = '\0';    /* for our own protection... */

    if (0 == strcmp("FORMAT" ,Name)) {strcpy(FORMAT ,Value); continue;} 
    if (0 == strcmp("TYPE"   ,Name)) {strcpy(TYPE   ,Value); continue;} 
    if (0 == strcmp("BUFSIZ" ,Name)) {BUFSIZe = atoi(Value); continue;}
    if (0 == strcmp("DIM"    ,Name)) {DIM     = atoi(Value); continue;}
    if (0 == strcmp("EOL"    ,Name)) {EOL     = atoi(Value); continue;}
    if (0 == strcmp("RECSIZE",Name)) {RECSIZE = atoi(Value); continue;}
    if (0 == strcmp("ORG"    ,Name)) {strcpy(ORG    ,Value); continue;} 
    if (0 == strcmp("NL"     ,Name)) {NL      = atoi(Value); continue;}
    if (0 == strcmp("NS"     ,Name)) {NS      = atoi(Value); continue;}
    if (0 == strcmp("NB"     ,Name)) {NB      = atoi(Value); continue;}
    if (0 == strcmp("N1"     ,Name)) {N1      = atoi(Value); continue;}
    if (0 == strcmp("N2"     ,Name)) {N2      = atoi(Value); continue;}
    if (0 == strcmp("N3"     ,Name)) {N3      = atoi(Value); continue;}
    if (0 == strcmp("N4"     ,Name)) {N4      = atoi(Value); continue;}
    if (0 == strcmp("NBB"    ,Name)) {NBB     = atoi(Value); continue;}
    if (0 == strcmp("NLB"    ,Name)) {NLB     = atoi(Value); continue;}
    if (0 == strcmp("HOST"   ,Name)) {strcpy(HOST   ,Value); continue;} 
    if (0 == strcmp("INTFMT" ,Name)) {strcpy(INTFMT ,Value); continue;} 
    if (0 == strcmp("REALFMT",Name)) {strcpy(REALFMT,Value); continue;} 
    if (0 == strcmp("TASK"   ,Name)) {strcpy(TASK   ,Value); continue;} 
    if (0 == strcmp("USER"   ,Name)) {strcpy(USER   ,Value); continue;} 
    if (0 == strcmp("DAT_TIM",Name)) {strcpy(DAT_TIM,Value); continue;} 
    if (0 == strcmp("COMMENT",Name)) {strcpy(DAT_TIM,Value); continue;} 
    if (done) break;
   }  
 }

static void ReadVICARScanLine(fd,y,VICARScanLine,VICARScanLineLength)
 FILE *fd;
 int y;
 unsigned char *VICARScanLine;
 int VICARScanLineLength;
 {
  if (fseek(fd,LBLSIZE+(y*VICARScanLineLength),0))
   FatalError("fseek failed");
  if (VICARScanLineLength != fread(VICARScanLine,1,VICARScanLineLength,fd))
   FatalError("fread failed");
 }

/* WFF stuff */
static FrameBufferType *SetUpWFFFile(stream, xsize, ysize, zsize)
 FILE *stream;
 int xsize, ysize, zsize;
 {
  FrameBufferType *FrameBuffer;
  int Bottom, Left, Top, Right;
  int BitsPerBand;
  char WhatBands[10], Name[NameLength], Value[ValueLength];

  /* we only handle the simple cases...*/
  BitsPerBand = 8;
  if      (1 == zsize) strcpy(WhatBands,"I");
  else if (3 == zsize) strcpy(WhatBands,"RGB");
  else 
   {
    fprintf(stderr,"SetUpWFFFile: NB = %d ???\n",NB);
    FatalError("strange NB");
   }

  FrameBuffer = (FrameBufferType *)0;

  OpenFB(&FrameBuffer);
  assert(FrameBuffer);

  Bottom = 0; Left = 0; Top = xsize-1; Right = ysize-1;
  SetBounds(FrameBuffer, Bottom, Left, Top, Right);

  SetColorSystem(FrameBuffer, WhatBands,  BitsPerBand);


  strcpy(Name,"X-CreatedBy");
  strcpy(Value,RoutineName);
  SetDescriptor(FrameBuffer, Name, Value);

  strcpy(Name,"Encoding");
  strcpy(Value,"AIS");
  SetDescriptor(FrameBuffer, Name, Value);

  /* Header operations over, now we can start the output stream */
  PassImageOut(stream, FrameBuffer);
  return (FrameBuffer);
 }  

static void WriteWFFScanLine(FB, xsize, ScanLine)
 FrameBufferType *FB;
 int xsize;
 unsigned short *ScanLine;
 {
  assert (SUCCESS == NextNPixelsOut(FB, xsize, ScanLine));
 }


int main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  char *VICARFileName;
  FILE *fd;
  unsigned char *VICARScanLine;
  unsigned char *Iptr;
  int VICARScanLineLength;
  FrameBufferType *FB;
  int x,y;

  RoutineName = argv[ArgsParsed++];
  while(ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit(-1); }
    switch (argv[ArgsParsed++][1])
     {
      case 'f': VICARFileName = argv[ArgsParsed++]; break;
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': usage(); exit(-1);
     }
   }
  
  fd = fopen(VICARFileName,"r");
  assert(fd);

  if (VERBOSE) fprintf(stderr,"%s: Parsing VICAR header\n",RoutineName);
  /* Parse VICAR Header, and do sanity check */
  NL = 0; NS = 0;  NB = 0; 
  ParseVICARHeader(fd);
  switch(NB)
   {
    case 1: break;

    default:  fprintf(stderr,"%s: NB = %d ???\n",RoutineName,NB);
              FatalError("strange NB");
              break;
   }

  VICARScanLineLength = NS*NB;
  VICARScanLine = (unsigned char *)calloc(sizeof (unsigned char),
                                          VICARScanLineLength);
  assert(VICARScanLine);

  if (VERBOSE) fprintf(stderr,"%s: VICAR image is %d x %d x %d\n",
                          RoutineName,NS,NL,NB);

  FB = SetUpWFFFile(stdout, NS, NL, NB);
  assert(FB);

  if (VERBOSE) fprintf(stderr,"%s: reading VICAR image",RoutineName);
  for(y=(NL-1);y>=0;y--)  /* flip in y */
   {
    unsigned short Pixel[1];

    ReadVICARScanLine(fd,y,VICARScanLine,VICARScanLineLength);
    Iptr = VICARScanLine;

    /* this is slow, but you'll find it easier to extend...*/  
    for(x=0;x<NS;x++)
     {
      Pixel[0] = (unsigned short) *(Iptr++);;
      NextPixelOut(FB, (unsigned short *)Pixel);
     }
    wffFlush(FB);
    if (VERBOSE) fprintf(stderr,".");
   }

  CloseFB(&FB);

  if (VERBOSE) fprintf(stderr,"\n");
  exit(0);
 }
