/* ************************************************************************
*   File: battle.field.c                                                  *
*  Usage: functions and ACMD for battle-field                             *
*                                                                         *
*  Copyright (C) 1993, 94 by Rat von Krysa (rat@mail.kolej.mff.cuni.cz)   *
*  Written for Circle MUD by Rat.                                                    *
************************************************************************ */

#define _BATTLE_FIELD_C

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "db.h"
#include "interpreter.h"
#include "handler.h"

extern struct room_data *world;
extern struct char_data *character_list;
extern struct descriptor_data *descriptor_list;
extern struct command_info cmd_info[];
extern sh_int r_mortal_start_room;

int battle_status = BATTLE_NOTRUN;
int battle_minlev;
int battle_maxlev;
int battle_cost;
int battle_money;
extern room_num battle_start_room;
extern room_num battle_recall_room;
extern room_num battle_min_room;
extern room_num battle_max_room;
extern room_num r_battle_start_room;
extern room_num r_battle_recall_room;
extern room_num r_battle_min_room;
extern room_num r_battle_max_room;

void check_battle_rooms(void) {
  if ((r_battle_start_room = real_room(battle_start_room)) == NOWHERE ||
      (r_battle_recall_room = real_room(battle_recall_room)) == NOWHERE ||
      (r_battle_min_room = real_room(battle_min_room)) == NOWHERE ||
      (r_battle_max_room = real_room(battle_max_room)) == NOWHERE) {
    battle_status = BATTLE_ERROR;
    log("SYSERR: Warning: One of battle-rooms doesn't exists.");
  }
  sprintf(buf, "Battle-rooms: %d (%d) %d (%d) %d (%d) %d (%d)",
	    battle_start_room, r_battle_start_room, battle_recall_room, r_battle_recall_room,
	    battle_min_room, r_battle_min_room, battle_max_room, r_battle_max_room);
  log(buf);
}

void cancel_battle(void) {
  struct char_data *ch;
  for(ch = character_list; ch; ch = ch->next) {
    if (PLR_FLAGGED(ch, PLR_IN_BATTLE)) {
      REMOVE_BIT(PLR_FLAGS(ch), PLR_IN_BATTLE);
      if (ch->in_room != NOWHERE) char_from_room(ch);
      char_to_room(ch, ch->was_battle);
      look_at_room(ch, 1);
      if (FIGHTING(ch)) stop_fighting(FIGHTING(ch)); stop_fighting(ch);
      GET_POS(ch) = POS_STANDING;
      GET_GOLD(ch) += GET_LEVEL(ch) * battle_cost;
    }
  }
  battle_status = BATTLE_NOTRUN;
  send_to_prf("Battle closed...\r\n", NULL, PRF_BATTLE);
}

ACMD(do_battle) {
  argument = one_argument(argument, buf);
  if (!strcmp(buf, "canal")) {
    send_to_char(((PRF_FLAGS(ch) ^= PRF_BATTLE) & PRF_BATTLE) ?
		 "You can now hear battle-messages.\r\n" :
		 "You are now deaf to the battle-messages.\r\n", ch);
    return;
  }
  if (battle_status == BATTLE_ERROR) {
    send_to_char("There is error in battle-field... Wait for shutdown :-(\r\n", ch);
    return;
  }
  if (!strcmp(buf, "info")) {
    if (battle_status == BATTLE_NOTRUN) send_to_char("Battle not running.\r\n", ch);
    else {
      if (battle_status == BATTLE_RUNNING) send_to_char("Running.\r\n", ch);
      else send_to_char("Waiting for players...\r\n", ch);
      sprintf(buf, "Levels: %d-%d\r\n"
                   "Cost per level: %d\r\n",
	      battle_minlev, battle_maxlev, battle_cost);
      send_to_char(buf, ch);
    }
  }
  else if (!strcmp(buf, "open") && GET_LEVEL(ch) >= BATTLE_OPEN_LEV) {
    if (battle_status != BATTLE_NOTRUN) send_to_char("Already running...\r\n", ch);
    else {
      battle_minlev = strtol(argument, &argument, 0);
      battle_maxlev = strtol(argument, &argument, 0);
      skip_spaces(&argument);
      if (*argument && (battle_minlev > battle_maxlev ||
	  battle_maxlev > LVL_IMPL || battle_minlev < 1)) {
        sprintf(buf, "Only for levels %d-%d\r\n", BATTLE_MIN_LEV, BATTLE_MAX_LEV);
	send_to_char(buf, ch);
	return;
      }
      if (*argument) {
	battle_cost = strtol(argument, &argument, 0);
	if (isspace(*argument) || !*argument) {
	  battle_status = BATTLE_WAITING;
	  battle_money = 0;
	  send_to_char(OK, ch);
          sprintf(buf, "%s has opened battle-field for levels %d-%d.\r\n"
                       "Costs per level: %d.\r\n",
                  GET_NAME(ch), battle_minlev, battle_maxlev, battle_cost);
          send_to_prf(buf, ch, PRF_BATTLE);
	  return;
	}
      }
      send_to_char("battle open minlevel maxlevel cost_per_level\r\n", ch);
    }
  }
  else if (!strcmp(buf, "close")  && GET_LEVEL(ch) >= BATTLE_CLOSE_LEV) {
    if (battle_status == BATTLE_NOTRUN) send_to_char("Not running\r\n", ch);
    else cancel_battle();
  }
  else if (!strcmp(buf, "go") &&
           GET_LEVEL(ch) >= BATTLE_MIN_LEV && GET_LEVEL(ch) <= BATTLE_MAX_LEV) {
    int costs;
    if (battle_status == BATTLE_NOTRUN) send_to_char("Not running\r\n", ch);
    else if (battle_status == BATTLE_RUNNING) send_to_char("Running... Try later\r\n", ch);
    else if (PLR_FLAGGED(ch, PLR_IN_BATTLE)) send_to_char("But you are always in battle\r\n", ch);
    else if (GET_LEVEL(ch) > battle_maxlev || GET_LEVEL(ch) < battle_minlev) {
      sprintf(buf, "Only for levels %d-%d\r\n", battle_minlev, battle_maxlev);
      send_to_char(buf, ch);
    }
    else if (GET_GOLD(ch) < (costs = battle_cost * GET_LEVEL(ch))) {
      sprintf(buf, "You must pay %d coins to enter battle-field.\r\n",
	      battle_cost * GET_LEVEL(ch));
      send_to_char(buf, ch);
    }
    else {
      SET_BIT(PLR_FLAGS(ch), PLR_IN_BATTLE);
      battle_money += costs; GET_GOLD(ch) -= costs;
      ch->was_battle = ch->in_room;
      char_from_room(ch);
      char_to_room(ch, r_battle_start_room);
      send_to_char(OK, ch);
      sprintf(buf, "%s (level %d) has entered battle-field.\r\n",
              GET_NAME(ch), GET_LEVEL(ch));
      send_to_prf(buf, ch, PRF_BATTLE);
      look_at_room(ch, 1);
    }
  }
  else {
    strcpy(buf, "battle { canal | info ");
    if (GET_LEVEL(ch) >= BATTLE_MIN_LEV && GET_LEVEL(ch) <= BATTLE_MAX_LEV)
      strcat(buf, "| go ");
    if (GET_LEVEL(ch) >= BATTLE_OPEN_LEV) strcat(buf, "| open ");
    if (GET_LEVEL(ch) >= BATTLE_CLOSE_LEV) strcat(buf, "| close ");
    strcat(buf, "}\r\n");
    send_to_char(buf, ch);
  }
}

void battle_activity(void) {
  struct char_data *ch;
  if (battle_status == BATTLE_ERROR || battle_status == BATTLE_NOTRUN) return;
  if (battle_status == BATTLE_RUNNING) {
    int num = 0;
    struct char_data *winner;
    struct descriptor_data *dch;
    for(dch = descriptor_list; dch; dch = dch->next) {
      if (dch->connected) continue;
      ch = (dch->original ? dch->original : dch->character);
      if (PLR_FLAGGED(ch, PLR_IN_BATTLE)) {
	if ((ch->in_room >= r_battle_min_room) && (ch->in_room <= r_battle_max_room)) {
	  num++;
	  winner = ch;
	}
	else {
	  REMOVE_BIT(PLR_FLAGS(ch), PLR_IN_BATTLE);
	  if (FIGHTING(ch)) stop_fighting(FIGHTING(ch)); stop_fighting(ch);
	  GET_POS(ch) = POS_STANDING; GET_HIT(ch) = 1; 
          if (GET_CLASS(ch) != CLASS_ALCHEMIST) GET_MANA(ch) = 1;
	  if (ch->in_room != NOWHERE)
	    char_from_room(ch);
	  char_to_room(ch, r_mortal_start_room);
	  look_at_room(ch, 1);
	}
      }
    }
    if (num <= 1) {
      battle_status = BATTLE_NOTRUN;
      send_to_prf("The battle is over...\r\n", NULL, PRF_BATTLE);
      if (num) { 
	if (GET_LEVEL(winner) < LVL_IMMORT -1) {
	  GET_EXP(winner) += 10000; gain_exp(winner, 1);
	  GET_GOLD(winner) += battle_money / 2;
	}
	REMOVE_BIT(PLR_FLAGS(winner), PLR_IN_BATTLE);
	sprintf(buf, "The winner is %s!\r\n", GET_NAME(winner));
	send_to_prf(buf, winner, PRF_BATTLE);
        send_to_char("You are the winner!\r\n", winner);
	if (ch->in_room != NOWHERE)
	  char_from_room(winner);
	char_to_room(winner, r_mortal_start_room); look_at_room(winner, 1);
	GET_HIT(winner) = GET_MAX_HIT(winner);
        if (GET_CLASS(winner) != CLASS_ALCHEMIST) 
          GET_MANA(winner) = (GET_LEVEL(winner)>=LVL_IMMORT) ? BATTLE_IMM_MANA : GET_MAX_MANA(winner);
      }
    }
  }
  else if (++battle_status == BATTLE_RUNNING) {
    struct char_data *next;
    int num = 0;
    for(ch = world[r_battle_start_room].people; ch; ch = ch->next_in_room) 
      if (!IS_NPC(ch) && PLR_FLAGGED(ch, PLR_IN_BATTLE)) num++;
    if (num >= BATTLE_MIN_PLR) { 
      for(ch = world[r_battle_start_room].people; ch; ch = next) {
	next = ch->next_in_room;
	if (!IS_NPC(ch) && PLR_FLAGGED(ch, PLR_IN_BATTLE)) {
	  if (ch->in_room != NOWHERE)
	    char_from_room(ch);
	  char_to_room(ch, number(r_battle_min_room, r_battle_max_room)); look_at_room(ch, 1);
	  if (GET_CLASS(ch) != CLASS_ALCHEMIST)
            GET_MANA(ch) = (GET_LEVEL(ch)>=LVL_IMMORT) ? BATTLE_IMM_MANA : GET_MAX_MANA(ch);
	  GET_HIT(ch) = (GET_LEVEL(ch)>=LVL_IMMORT) ? BATTLE_IMM_HIT: GET_MAX_HIT(ch);
	  GET_MOVE(ch) = GET_MAX_MOVE(ch);
	  while (ch->affected)
	    affect_remove(ch, ch->affected);
	  send_to_char("There is a brief flash of light!\r\n"
		       "You feel slightly different.\r\n", ch);
	}
      }
      send_to_prf("Battle was started... FIGHT!\r\n", NULL, PRF_BATTLE);
    }
    else {
      send_to_prf("Battle wasn't started - not enough players.\r\n", NULL, PRF_BATTLE);
      cancel_battle();
    }
  }
  else {
    sprintf(buf, "The battle-gate will be opened yet %d seconds\r\n", 
	    (BATTLE_RUNNING - battle_status) * PULSE_BATTLE / PASSES_PER_SEC);
    send_to_prf(buf, NULL, PRF_BATTLE);
  }
}
