/*
 *	Copyright (c) 1993-1996 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	BPFT $Id: show.c,v 2.0 1996/01/15 20:02:10 bob Exp $
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 * Redistribution in binary form may occur without any restrictions.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/*	show.c - show network traffic	*/

static char copyright[] = "* Copyright (c) 1993-1996 CAD lab, Institute of Telecommunication, Novosibirsk *";

#include <curses.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/udp.h>
#include <netinet/udp_var.h>
#include <netinet/tcp.h>
#include <netinet/tcpip.h>

#include "addrtoname.h"
#include "trafshow.h"

/*
 * Single record's format description
 */
struct t_entry {
	struct	in_addr	src;	/* source ip address */
	u_short	sport;		/* source port */
	struct	in_addr	dst;	/* destination ip address */
	u_short	dport;		/* destination port */
	u_short	proto;		/* ip protocol */
	u_long	bytes;		/* bytes in ip datagram */
	u_long	obytes;		/* old bytes */
	u_int	cps;		/* chars per second :-) */
};

typedef struct t_entry *p_entry;
struct t_entry *entries;
static u_short iplen, dlen;
int n_entry = 0;
u_long tcp_pdata = 0;
u_long udp_pdata = 0;
u_long all_bytes = 0;
int page = 0;
int page_size;
int l_nflag;

extern void onterm();

traf_init(fflag)
	int fflag;
{
	register i;

	entries = (struct t_entry *)calloc(LINES * MAX_PAGES, sizeof(struct t_entry));
	if (!entries) onterm();
	init_addrtoname(device_name, fflag);
	l_nflag = nflag;
	alarm(scr_interval);
	return;
}

/*
 * Pretty print an Internet address (net address + port).
 */
char *
inet_print(in, port, proto)
	struct in_addr in;
	u_short port;
	u_char proto;
{
	register char *p;
	static char aline[80];

	if (l_nflag) p = (char *)intoa(in);
	else p = ipaddr_string(&in);
	sprintf(aline, "%-28.28s", p);

	if (port) {
		char pline[11], *cp;
		register alen = 28, plen;

		plen = sprintf(pline, "..%.8s", proto == IPPROTO_TCP ?
			       tcpport_string(port) : udpport_string(port));

		if ((cp = index(aline, ' ')) != NULL)
			alen = cp - aline;
		if ((alen + plen) >= 28)
			bcopy(pline, &aline[28 - plen], plen);
		else	bcopy(pline, &aline[alen], plen);
	}
	return aline;
}

void
main_print(i)
	register i;
{
	char *proto;

	addstr(inet_print(entries[i].src, entries[i].sport, entries[i].proto));
	addch(' ');
	addstr(inet_print(entries[i].dst, entries[i].dport, entries[i].proto));

	switch(entries[i].proto) {
		case IPPROTO_TCP:
			proto = "tcp";
			break;
		case IPPROTO_UDP:
			proto = "udp";
			break;
		case IPPROTO_ICMP:
			proto = "icmp";
			break;
		case IPPROTO_EGP:
			proto = "egp";
			break;
		case IPPROTO_OSPF:
			proto = "ospf";
			break;
		case IPPROTO_IGMP:
			proto = "igmp";
			break;
		case IPPROTO_RAW:
			proto = "raw";
			break;
		default:
			proto = "unkn";
	}
	printw(" %-4.4s %10ld", proto, entries[i].bytes);
	if (entries[i].cps) printw(" %-6d", entries[i].cps);
	clrtoeol();
}

static int
sortbybytes(e1, e2)
	register p_entry e1, e2;
{
	if (e1->bytes > e2->bytes) return -1;
	if (e1->bytes < e2->bytes) return 1;
	return 0;
}

void
statusline()
{
	move(LINES-1, 0);
	printw("(%.5s) TCP pkts: %-9ld UDP pkts: %-9ld KBytes: %-9ld  Page %d of %d",
	       device_name, tcp_pdata, udp_pdata, all_bytes/1024,
	       page + 1, n_entry/page_size + 1);
	clrtoeol();
}

/*
 * If scr_interval time any entry wasn't changed then purge it and redraw screen
 */
deleteold(delete)
	int delete;
{
	register i, j;

	if (delete) {
		alarm(0);
		for (i = j = 0; i < n_entry; i++) {
			if (entries[i].bytes != entries[i].obytes)
				entries[i].cps = (entries[i].bytes - entries[i].obytes)/scr_interval;
			else	entries[i].bytes = 0, j++;
			entries[i].obytes = entries[i].bytes;
		}
		qsort(entries, n_entry, sizeof(struct t_entry), sortbybytes);
		n_entry -= j;
	}
	j = n_entry/page_size;
	if (page > j) page = j;

	move(SCR_OFFS, 0);
	if (n_entry) {
		for (i = 0, j = page * page_size;
		     i < page_size && j < n_entry; i++, j++) {
			move(i + SCR_OFFS, 0);
			main_print(j);
		}
	} else {
		switch (alarm_flag) {
		case 1:
			addstr("Where is the tcp or udp packets contained data? It is exactly what I want! Now!");
			break;
		case 2:
			addstr("Nothing to show, this interface sleeping or broken. blah-blah-gluk-gluk-wait...");
			break;
		default:
			addstr(copyright);
			alarm_flag = 1;
		}
	}
	clrtobot();
	statusline();
	if (delete) alarm(scr_interval);
}

/*
 * Add new entry or add bytes to existed record.
 */
int
traf_add(e)
	register p_entry e;
{
	register i, j;

	all_bytes += e->bytes;
	j = page * page_size;
	for (i = 0; i < n_entry; i++) {
		if (e->src.s_addr == entries[i].src.s_addr &&
		    e->sport == entries[i].sport &&
		    e->dst.s_addr == entries[i].dst.s_addr &&
		    e->dport == entries[i].dport &&
		    e->proto == entries[i].proto) {
			entries[i].bytes += e->bytes;
			if (i >= j && i < j + page_size) {
				move(i - j + SCR_OFFS, 63);
				printw("%10ld", entries[i].bytes);
			}
			return TRUE;
		}
	}
	if (i == LINES * MAX_PAGES) {
		mvaddstr(LINES-1, 70, "overflow");
		clrtoeol();
		return FALSE;
	}
	/* add new entry */
	entries[i] = *e;
	entries[i].obytes = entries[i].cps = 0;
	if (i >= j && i < j + page_size) {
		move(i - j + SCR_OFFS, 0);
		main_print(i);
	}
	n_entry++;
	statusline();
	return TRUE;
}

void
traf_tcp(tp, ip)
	register struct tcphdr *tp;
	register struct ip *ip;
{
	struct t_entry t;

	if ((u_char *)(tp + 1) > snapend || dlen < sizeof(struct tcphdr))
		return;

	t.src.s_addr = ip->ip_src.s_addr;
	t.sport = ntohs(tp->th_sport);
	t.dst.s_addr = ip->ip_dst.s_addr;
	t.dport = ntohs(tp->th_dport);
	t.proto = IPPROTO_TCP;
	t.bytes = iplen;
	if (traf_add(&t)) tcp_pdata++;
}

void
traf_udp(up, ip)
	register struct udphdr *up;
	register struct ip *ip;
{
	register u_char *cp = (u_char *)(up + 1);
	struct t_entry t;

	if (cp > snapend || dlen < sizeof(struct udphdr))
		return;

	t.src.s_addr = ip->ip_src.s_addr;
	t.sport = ntohs(up->uh_sport);
	t.dst.s_addr = ip->ip_dst.s_addr;
	t.dport = ntohs(up->uh_dport);
	t.proto = IPPROTO_UDP;
	t.bytes = iplen;
	if (traf_add(&t)) udp_pdata++;
}

void
traf_ip(ip)
	register struct ip *ip;
{
	struct t_entry t;

	t.src.s_addr = ip->ip_src.s_addr;
	t.sport = 0;
	t.dst.s_addr = ip->ip_dst.s_addr;
	t.dport = 0;
	t.proto = ip->ip_p;
	t.bytes = iplen;
	traf_add(&t);
}

void
processing_ip(ip, length)
	register struct ip *ip;
	register int length;
{
	register int hlen;

	if ((u_char *)(ip + 1) > snapend) return;

	if (length < sizeof(struct ip)) {
		move(LINES-1, strlen(device_name) + 3);
		printw("truncated-ip: discard %d bytes", length);
		clrtoeol();
		goto refresh_screen;
	}
	iplen = ntohs(ip->ip_len);
	if (iplen < 1) return;

	if (length < iplen) {
		move(LINES-1, strlen(device_name) + 3);
		printw("truncated-ip: %d bytes missing", iplen - length);
		clrtoeol();
		goto refresh_screen;
	}
	hlen = ip->ip_hl * 4;
	dlen = iplen - hlen;
	/*
	 * If this is fragment zero, hand it to the next higher level protocol,
	 * else it is fragmented datagram.
	 */
	if (!(ntohs(ip->ip_off) & 0x1fff)) {
		register unsigned char *cp = (unsigned char *)ip + hlen;
		switch (ip->ip_p) {
		case IPPROTO_TCP:
			traf_tcp((struct tcphdr *)cp, ip);
			break;
		case IPPROTO_UDP:
			traf_udp((struct udphdr *)cp, ip);
			break;
		default:
			traf_ip(ip);
		}
	} else	traf_ip(ip);

	if (alarm_flag) {
		alarm_flag = 0;
		deleteold(TRUE);
	}

refresh_screen:
	if (kflag && kbhit(0) && processing_keyb()) deleteold(FALSE);
	move(0,79);
	refresh();

	return;
}

int
inputchar()
{
	return getch();
}

processing_keyb()
{
	int ch;

	if ((ch = getch()) == ERR) onterm();
	switch(ch) {
	case ESC:
		switch(get_arrow_key(inputchar)) {
		case KEYMAP_UP:
		case KEYMAP_PAGE_UP:
			goto page_up;
		case KEYMAP_DOWN:
		case KEYMAP_PAGE_DOWN:
			goto page_down;
		case KEYMAP_LEFT:
		case KEYMAP_HOME:
			goto home_list;
		case KEYMAP_RIGHT:
		case KEYMAP_END:
			goto end_list;
		default:
			mvaddstr(LINES-1, COLS-15, "Bad command\007");
			return 0;
		}
		break;

	case 'k':		/* line up */
	case ctrl('P'):
	case '\b':		/* page up */
	case 'b':
	case ctrl('U'):
	case ctrl('B'):
page_up:	
		if (page > 0) page--;
		break;

	case 'j':
	case ctrl('N'):		/* line down */
	case ' ':		/* page down */
	case ctrl('D'):
	case ctrl('F'):
page_down:
		if (page < n_entry/page_size) page++;
		break;

	case ctrl('A'):		/* home */
home_list:
		page = 0;
		break;

	case ctrl('E'):		/* end */
end_list:
		page = n_entry/page_size;
		break;

	case '\r':		/* enter */
	case '\n':
	case '\t':		/* tab */
		l_nflag ^= 1;
		break;

	case 'q':		/* quit */
		onterm();

	case ctrl('L'):		/* refresh screen */
		wrefresh(curscr);
		break;

	default:
		mvaddstr(LINES-1, COLS-15, "Bad command\007");
		return 0;
	}
	return 1;
}
