/*
 *	Copyright (c) 1993-1996 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	BPFT $Id: main.c,v 2.0 1996/01/15 20:01:36 bob Exp $
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 * Redistribution in binary form may occur without any restrictions.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <curses.h>
#include <stdlib.h>
#include <signal.h>

#include "interface.h"
#include "addrtoname.h"
#include "trafshow.h"

/*
 * Command line switches.
 * Left - default value, right - action, if switch has been used.
 */
int dflag = 0;	/* dump to compiled packet-matching code to stdout and stop */
int Oflag = 1;	/* don't run the packet-matching code optimizer if have bug */
int pflag = 0;	/* don't put the interface into promiscuous mode */
int kflag = 1;	/* disable keyboard input checking */
int nflag = 0;	/* don't convert addresses to host names */
int Nflag = 0;	/* output only host names without domain */

/* Global variables */
char *program_name;		/* myself */
char *device_name = 0;		/* interface name */
int alarm_flag = 0;		/* flag -- screen refresh requested */
int scr_interval = DEFAULT_SCR;	/* screen refresh interval in seconds */
int snaplen = DEFAULT_SNAPLEN;	/* length of saved portion of packet */

void
main(argc, argv)
	int argc;
	char **argv;
{
	struct bpf_program *parse();
	void bpf_dump(), usage(), onterm(), onalarm();

	int op, cnt = -1, fflag = 0, if_fd = -1;
	struct bpf_program *fcode;
	char *infile = 0;
	char *cmdbuf;
	extern char *optarg;
	extern int optind, opterr;

	program_name = stripdir(argv[0]);

	opterr = 0;
	while ((op = getopt(argc, argv, "c:dfF:i:knNOpr:t:")) != EOF)
		switch (op) {
		case 'c':
			cnt = atoi(optarg);
			if (cnt < 1) usage();
			break;
		case 'd':
			++dflag;
			break;
		case 'f':
			++fflag;
			break;
		case 'F':
			infile = optarg;
			break;
		case 'i':
			device_name = optarg;
			break;
		case 'k':
			kflag = 0;
			break;
		case 'n':
			++nflag;
			break;
		case 'N':
			++Nflag;
			break;
		case 'O':
			Oflag = 0;
			break;
		case 'p':
			++pflag;
			break;
		case 'r':
			scr_interval = atoi(optarg);
			if (scr_interval < 1) usage();
			break;
		case 't':
			dns_timeout = atoi(optarg);
			if (dns_timeout < 1) usage();
			break;
		default:
			usage();
		}

	/* Find network interface */
	if (device_name == 0)
		if ((device_name = getenv("IFF_LISTEN")) == NULL)
			if ((device_name = lookup_device()) == 0)
				error("can't find any interfaces");

	/* Attach bpf interface to the network interface */
	if_fd = bpf_init(device_name, pflag);

	if (infile) 
		cmdbuf = read_infile(infile);
	else
		cmdbuf = copy_argv(&argv[optind]);

	fcode = parse(cmdbuf, Oflag);
	if (dflag) {
		bpf_dump(fcode, dflag);
		exit(0);
	}

	initterm();
	signal(SIGHUP, onterm);
	signal(SIGINT, onterm);
	signal(SIGQUIT, onterm);
	signal(SIGTERM, onterm);
	signal(SIGTSTP, SIG_IGN);
	signal(SIGALRM, onalarm);
	traf_init(fflag);
	bpf_readloop(cnt, if_fd, fcode);
	onterm();
}

/*
 * SIGALRM interrupt handler, should be ran each scr_interval seconds.
 */
void
onalarm()
{
	if (alarm_flag) {
		deleteold(TRUE);
		move(0,79);
		refresh();
	} else alarm(scr_interval);
	alarm_flag++;
}

void
onterm()
{
	signal(SIGINT, SIG_IGN);
	signal(SIGUSR1, SIG_IGN);
	mvcur(0, COLS - 1, LINES - 1, 0);
	endwin();
	printf(DEFCOLOR);
	exit(0);
}

initterm()
{
	puts(NICECOLOR);
	COLS = 80;
	if (initscr() == ERR) exit(1);
	if (LINES < 20) {
		addstr("Must more LINES on term");
		onterm();
	}
	if (COLS < 80) {
		addstr("Must more COLS on term");
		onterm();
	}
	page_size = LINES - SCR_OFFS - 2;

	cbreak();
	noecho();
	nonl();
	leaveok(stdscr, FALSE);
	scrollok(stdscr, FALSE);

	mvaddstr(0, 0, "\
From Address                 To Address                   Proto    Bytes  CPS");
	mvaddstr(1, 0, "\
================================================================================");
	move(0, 79);
	refresh();
}

void
usage()
{
	extern char version[];

	fprintf(stderr, "trafshow v%s - full screen show network traffic\n", version);
	fprintf(stderr,
"Usage: %s [-dfknNOp -c num -i name -r sec -t sec] [-F file | expr]\n\
\t-c number\tcount number of packets and exit\n\
\t-d\t\tprint compiled packet-matching code and exit\n\
\t-f\t\tconvert addresses to name only for local hosts\n\
\t-F file\t\tuse file as input for the filter expression\n\
\t-i name\t\tnetwork interface name; ne0, sl0, ppp0, lo0, etc\n\
\t-k\t\tdisable keyboard input checking\n\
\t-n\t\tdon't convert addresses to host names\n\
\t-N\t\toutput only host names without domain\n\
\t-O\t\tdon't run the packet-matching code optimizer\n\
\t-p\t\tdon't put the interface into promiscuous mode\n\
\t-r seconds\tscreen refresh interval, default %d sec\n\
\t-t seconds\tmax timeout in DNS query, default %d sec\n\
\texpr\t\tfilter expression like tcpdump's\n",
		program_name, DEFAULT_SCR, DEFAULT_TIMEOUT);

	exit(-1);
}
