/*
 *	Copyright (c) 1993 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	BPFT $Id: main.c,v 1.4 1994/01/06 15:29:53 bob Exp $
 *
 *	$Log: main.c,v $
 * Revision 1.4  1994/01/06  15:29:53  bob
 * New keywords for user defined output format
 *
 * Revision 1.3  1993/11/06  12:55:30  bob
 * Add '-a' flag, modify usage() display
 *
 * Revision 1.2  1993/11/01  15:10:06  bob
 * Added '-r' flags, fixed return codes
 *
 * Revision 1.1  1993/10/20  16:05:18  bob
 * Initial revision
 *
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 * Redistribution in binary form may occur without any restrictions.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/*	main.c - tcp/udp data traffic log manager	*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <paths.h>
#include <fcntl.h>
#include <signal.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/param.h>

#include "../include/interface.h"
#include "../include/addrtoname.h"
#include "../include/pathnames.h"
#include "traflog.h"

int aflag;		/* display all records, default last */
int fflag;		/* convert addresses to names only for local hosts */
int lflag;		/* output list of log file records */
int nflag;		/* don't convert addresses to host names */
int Nflag;		/* output host names without domain */
int rflag;		/* only return number of records or KB */
int sflag;		/* summary traffic */

#define	LASTOFFSET	999
int begin = 0, endin = LASTOFFSET;
int tbflag = 0, teflag = 0;
struct timeval tb, te;

char *device_name = NULL;
char *program_name, *f_end;

static time_t tval;

int fvnum = 0;
struct fmt_var *fv = NULL;

int
main(argc, argv)
	int argc;
	char **argv;
{
	register FILE *fd;
	FILE *fdw = NULL;
	register int op;
	int dflag = 0;		/* dump pattern */
	char buf[80], *cmdbuf, *infile = NULL;
	extern char *optarg;
	extern int optind, opterr;
	void usage(), onpipe();

	program_name = stripdir(argv[0]);
	if (time(&tval) == -1)
		error("error in function 'time'");

	opterr = 0;
	while ((op = getopt(argc, argv, "ab:de:fF:i:lnNo:rsw:")) != EOF)
		switch (op) {
		case 'a':
			++aflag;
			break;
		case 'b':
			if (!(begin = atoi(optarg)))
				usage();
			if (begin > LASTOFFSET) {
				setthetime(optarg, &tb);
				++tbflag;
			}
			break;
		case 'd':
			++dflag;
			break;
		case 'e':
			if (!(endin = atoi(optarg)))
				usage();
			if (endin > LASTOFFSET) {
				setthetime(optarg, &te);
				++teflag;
			}
			break;
		case 'f':
			++fflag;
			break;
		case 'F':
			infile = optarg;
			break;
		case 'i':
			device_name = optarg;
			break;
		case 'l':
			if (sflag)
				usage();
			++lflag;
			break;
		case 'n':
			++nflag;
			break;
		case 'N':
			++Nflag;
			break;
		case 'o':
			if (!get_format(optarg))
				error("output format '%s' not found", optarg);
			break;
		case 'r':
			++rflag;
			break;
		case 's':
			if (lflag)
				usage();
			++sflag;
			break;
		case 'w':
			if ((fdw = fopen(optarg, "w")) == NULL)
				error("can't write to %s", optarg);
			break;
		default:
			usage();
		}

	signal(SIGPIPE, onpipe);

	if (infile) {
		if ((cmdbuf = getenv("PATTERNPATH")) != NULL) {
			infile = stripdir(infile);
			sprintf(buf, "%s/%s", cmdbuf, infile);
			infile = buf;
		}
		cmdbuf = read_infile(infile);
	} else
		cmdbuf = copy_argv(&argv[optind]);

	pattern(cmdbuf);
	if (dflag)
		exit(pat_print());

	if (device_name == 0)
		if ((device_name = getenv("IFF_LISTEN")) == NULL)
			if ((device_name = lookup_device()) == 0)
				error("can't find any network interfaces");

	init_addrtoname(device_name, fflag);

	if (!localnet && !netmask)
		strcpy(buf, device_name);
	else
		sprintf(buf, "%strafd.%s", PATH_TOSAVE, device_name);

	if ((fd = fopen(buf, "r")) == NULL)
		error("can't open %s", buf);

	log_loop(fd, fdw);
	exit(0);
}

int
get_format(f_name)
	char *f_name;
{
	int cmnt, f_len = strlen(f_name);
	static char *dflt = PATH_TRAFLOG_FMT;
	char *f_fmt, *f_tmp;
	register char *ptr;
	struct stat st;

	if ((f_fmt = getenv("FORMATPATH")) == NULL)
		f_fmt = dflt;
	if ((cmnt = open(f_fmt, O_RDONLY)) < 0)
		error("can't open %s", f_fmt);
	if (fstat(cmnt, &st) < 0)
		error("can't fstat %s", f_fmt);
	if ((f_fmt = malloc(st.st_size)) == NULL)
		error("can't allocate memory");
	if (read(cmnt, f_fmt, st.st_size) != st.st_size)
		error("can't read format file");
	close(cmnt);
	cmnt = 0;
	f_end = f_fmt + st.st_size;
	for (ptr = f_fmt; ptr < f_end; ptr++) {
		if (*ptr == '\n') {
			cmnt = 0;
			continue;
		}
		if (*ptr == '#') {
			cmnt++;
			continue;
		}
		if (cmnt)
			continue;
		if (f_len) {
			if (!strncmp(ptr, f_name, f_len))
				f_len = 0;
			continue;
		}
		if (*ptr == '}' && (*(ptr+1) == ';' || *(ptr+2) == ';'))
			break;
		ptr += iskey(ptr);
	}
	return fvnum;
}

int
iskey(token)
	char *token;
{
	int i;

	if (!strncmp(token, "from", 4)) {
		if (i = isformat(token + 4))
			fv[fvnum-1].cmd = FROM;
		return i+4;
	}
	if (!strncmp(token, "to", 2)) {
		if (i = isformat(token + 2))
			fv[fvnum-1].cmd = TO;
		return i+2;
	}
	if (!strncmp(token, "sport", 5)) {
		if (i = isformat(token + 5))
			fv[fvnum-1].cmd = SPORT;
		return i+5;
	}
	if (!strncmp(token, "dport", 5)) {
		if (i = isformat(token + 5))
			fv[fvnum-1].cmd = DPORT;
		return i+5;
	}
	if (!strncmp(token, "proto", 5)) {
		if (i = isformat(token + 5))
			fv[fvnum-1].cmd = PROTO;
		return i+5;
	}
	if (!strncmp(token, "bytes", 5)) {
		if (i = isformat(token + 5))
			fv[fvnum-1].cmd = BYTES;
		return i+5;
	}
	if (!strncmp(token, "psize", 5)) {
		if (i = isformat(token + 5))
			fv[fvnum-1].cmd = PSIZE;
		return i+5;
	}
	if (!strncmp(token, "ftime", 5)) {
		if (i = isformat(token + 5))
			fv[fvnum-1].cmd = FTIME;
		return i+5;
	}
	if (!strncmp(token, "ltime", 5)) {
		if (i = isformat(token + 5))
			fv[fvnum-1].cmd = LTIME;
		return i+5;
	}
	return 0;
}

int
isformat(fmt)
	char *fmt;
{
	int i;
	char *ptr;
	char *str[2];

	ptr = fmt;
	for (i = 0; i < 2; i++)
		if ((ptr = memchr(ptr, '"', f_end - ptr)) != NULL) {
			str[i] = ptr;
			ptr += 3;
		} else
			return 0;
	++str[0];
	*str[1] = 0;
	if (strchr(str[0], '%') == NULL)
		return 0;
	while ((ptr = strrchr(str[0], '\\')) != NULL) {
		switch (*(ptr+1)) {
			case 'a':
				*ptr = 0x07;	/* BEL */
				break;
			case 'b':
				*ptr = 0x08;	/* BS */
				break;
			case 'f':
				*ptr = 0x0c;	/* FF */
				break;
			case 'n':
				*ptr = 0x0a;	/* LF */
				break;
			case 'r':
				*ptr = 0x0d;	/* CR */
				break;
			case 't':
				*ptr = 0x09;	/* HT */
				break;
			case 'v':
				*ptr = 0x0b;	/* VT */
				break;
			default:
				return 0;
		}
		strncpy(ptr+1, ptr+2, strlen(ptr+2) + 1);
	}
	if ((fv = realloc(fv, ++fvnum * sizeof(struct fmt_var))) == NULL)
		error("can't reallocate memory");
	fv[fvnum-1].format = str[0];

	return (str[1] - fmt);
}

static days_in_month[12] =
	{31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
	
#define	ATOI2(ar)	((ar)[0] - '0') * 10 + ((ar)[1] - '0'); (ar) += 2;
setthetime(p, tv)
	register char *p;
	struct timeval *tv;
{
	register struct tm *lt;
	int dot;
	char *t;

	for (t = p, dot = 0; *t; ++t)
		if (!isdigit(*t) && (*t != '.' || dot++))
			badformat();

	lt = localtime(&tval);

	if (t = index(p, '.')) {		/* .ss */
		*t++ = '\0';
		lt->tm_sec = ATOI2(t);
		if (lt->tm_sec >= 60)
			badformat();
	} else
		lt->tm_sec = 0;

	for (t = p; *t; ++t)
		if (!isdigit(*t))
			badformat();

	switch (strlen(p)) {
	case 10:				/* yy */
		lt->tm_year = ATOI2(p);
		if (lt->tm_year <= 69)		/* hack for 2000 ;-} */
			lt->tm_year += 100;
		/* FALLTHROUGH */
	case 8:					/* mm */
		lt->tm_mon = ATOI2(p);
		if (lt->tm_mon > 12)
			badformat();
		--lt->tm_mon;			/* time struct is 0 - 11 */
		/* FALLTHROUGH */
	case 6:					/* dd */
		lt->tm_mday = ATOI2(p);
		if (lt->tm_mday > days_in_month[lt->tm_mon] 
		    && !(((lt->tm_year % 4 == 0 && lt->tm_year % 100 != 0)
		    || lt->tm_year % 400 == 0) && lt->tm_mon == 1
		    && lt->tm_mday == 29))
			badformat();
		/* FALLTHROUGH */
	case 4:					/* hh */
		lt->tm_hour = ATOI2(p);
		if (lt->tm_hour > 23)
			badformat();
		/* FALLTHROUGH */
	case 2:					/* mm */
		lt->tm_min = ATOI2(p);
		if (lt->tm_min > 59)
			badformat();
		break;
	default:
		badformat();
	}

	/* convert broken-down time to GMT clock time */
	if ((tval = mktime(lt)) == -1)
		badformat();

	tv->tv_sec = tval;
	tv->tv_usec = 0;
}

badformat()
{
	error("illegal time format");
}

void
onpipe()
{
	exit(1);
}

void
usage()
{
	extern char version[];

	fprintf(stderr, "traflog v%s - tcp/udp data traffic log manager\n", version);
	fprintf(stderr,
"Usage: %s -l [-i iface] [-b #] [-e #] [-r]\n\
-d [-fnN] [-F file | pattern]\n\
[-i iface] [-b #] [-e #] [-afnNrs] [-o format] [-w file] [-F file | pattern]\n",
		program_name);
	exit(-1);
}
