/*
 *	Copyright (c) 1993 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	BPFT $Id: interfaces.c,v 1.1 1993/10/20 16:03:56 bob Exp $
 *
 *	$Log: interfaces.c,v $
 * Revision 1.1  1993/10/20  16:03:56  bob
 * Initial revision
 *
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 * Redistribution in binary form may occur without any restrictions.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/mbuf.h>
#include <sys/socket.h>

#include <net/if.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/udp.h>
#include <netinet/udp_var.h>
#include <netinet/tcp.h>
#include <netinet/tcpip.h>
#include <net/bpf.h>
#include <net/slip.h>
#include <net/slcompress.h>

#include "interface.h"

u_char *packetp;
u_char *snapend;

/*
 * Here defined hardware depended funtions table.
 */
static struct if_func {
	void (*f)();
	int type;
} if_funcs[] = {
	{	bpf_ether,	DLT_EN10MB	},	/* Ethernet */
	{	bpf_slip,	DLT_SLIP	},	/* SLIP */
	{	bpf_ppp,	DLT_PPP		},	/* PPP */
#ifdef	DLT_C_HDLC
	{	bpf_chdlc,	DLT_C_HDLC	},	/* Cisco HDLC */
#endif
	{	bpf_null,	DLT_NULL	},	/* loopback */
	{ 0, 0 },
};

/*
 * Assign data link type to interface function.
 */
void
(*link_func())()
{
	struct if_func *p;

	for (p = if_funcs; p->f; ++p)
		if (link_type == p->type)
			return p->f;
	error("unknown data link type 0x%x", link_type);
}

/* Link-level header length */
#ifndef	ETHER_HDRLEN
#define	ETHER_HDRLEN	sizeof(struct ether_header)
#endif

#ifndef	SLIP_HDRLEN
#ifdef	SLC_BPFHDR
#define	SLIP_HDRLEN	SLC_BPFHDR
#else
#define	SLIP_HDRLEN	16	/* bpf slip header length */
#endif
#endif

#ifndef	PPP_HDRLEN
#ifdef	SLC_BPFHDR
#define	PPP_HDRLEN	SLC_BPFHDR
#else
#define	PPP_HDRLEN	4	/* sizeof(struct ppp_header) */
#endif
#endif

#ifdef	DLT_C_HDLC
#ifndef	CHDLC_HDRLEN
#define	CHDLC_HDRLEN	4	/* sizeof(struct cisco_hdr) */
#endif
#endif

#ifndef	NULL_HDRLEN
#define	NULL_HDRLEN	4	/* loopback header length */
#endif

/*
 * This is the interface depended routines for ethernet, slip, ppp and lo.
 * Use as (*usage_func)();
 *
 *	'p' is the points to the packet,
 *	'length' is the length of the packet,
 *	'caplen' is the number of bytes actually captured.
 *
 * WARNING:
 * processing_ip() is the user's function, see interface.h for more details.
 */
void
bpf_ether(p, length, caplen)
	register u_char *p;
	int length;
	int caplen;
{
	if (caplen < ETHER_HDRLEN)
		return;
	packetp = p;
	snapend = p + caplen;
	if (ntohs(((struct ether_header *)p)->ether_type) == ETHERTYPE_IP)
		processing_ip((struct ip *)(p + ETHER_HDRLEN),
			      length - ETHER_HDRLEN);
}

void
bpf_slip(p, length, caplen)
	u_char *p;
	int length;
	int caplen;
{
	if (caplen < SLIP_HDRLEN)
		return;
	packetp = p;
	snapend = p + caplen;
	processing_ip((struct ip *)(p + SLIP_HDRLEN), length - SLIP_HDRLEN);
}

void
bpf_ppp(p, length, caplen)
	register u_char *p;
	int length;
	int caplen;
{
	register hdrlen = 0;
	u_short type;

	if (caplen < PPP_HDRLEN)
		return;
	packetp = p;
	snapend = p + caplen;

#ifdef	SLC_BPFHDRLEN
	p += SLC_BPFHDRLEN;	/* pointer to link level header */
#endif
	/* PPP address and PPP control fields may be present (-acfc) */
	if (p[0] == 0xff && p[1] == 0x03) {
		p += 2;
		hdrlen += 2;
	}
	/* Retrive the protocol type */
	if (*p & 01) {	/* Compressed protocol field (pfc) */
		type = *p++;
		hdrlen++;
	} else {	/* Un-compressed protocol field (-pfc) */
		type = ntohs(*(u_short *)p);
		p += 2;
		hdrlen += 2;
	}
	if (type == 0x21) {	/* IP protocol */
#ifdef	SLC_BPFHDR
		p = packetp + SLC_BPFHDR;	/* skip bpf pseudo header */
		hdrlen = SLC_BPFHDR;
#endif
		processing_ip((struct ip *)p, length - hdrlen);
	}
}

#ifdef	DLT_C_HDLC
void
bpf_chdlc(p, length, caplen)
	u_char *p;
	int length;
	int caplen;
{
	if (caplen < CHDLC_HDRLEN)
		return;
	packetp = p;
	snapend = p + caplen;
	if (ntohs(*(u_short *)(p + 2)) == 0x0800) /* IP protocol */
		processing_ip((struct ip *)(p + CHDLC_HDRLEN),
			      length - CHDLC_HDRLEN);
}
#endif

void
bpf_null(p, length, caplen)
	u_char *p;
	int length;
	int caplen;
{
  	u_int family;

	bcopy(p, &family, sizeof(family));
	packetp = p;
	snapend = p + caplen;
	if (family == AF_INET)
		processing_ip((struct ip *)(p + NULL_HDRLEN),
			      length - NULL_HDRLEN);
}
