/*
 *	Copyright (c) 1993 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	BPFT $Id: bpf.c,v 1.1 1993/10/20 16:03:56 bob Exp $
 *
 *	$Log: bpf.c,v $
 * Revision 1.1  1993/10/20  16:03:56  bob
 * Initial revision
 *
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 * Redistribution in binary form may occur without any restrictions.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <stdio.h>
#include <netdb.h>
#include <ctype.h>
#include <errno.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <net/bpf.h>
#include <net/if.h>
#include <string.h>

#include "interface.h"

extern int errno;
extern void (*link_func())();
extern struct bpf_program *parse();

int link_type;
void (*usage_func)();

/*
 * Find and open bpf interface. For internal usage, not for users!
 */
static inline int
bpf_open()
{
	int fd;
	int n = 0;
	char device[sizeof "/dev/bpf000"];

	/*
	 * Go through all the minors and find one that isn't in use.
	 */
	do {
		(void)sprintf(device, "/dev/bpf%d", n++);
		fd = open(device, O_RDONLY);
	} while (fd < 0 && errno == EBUSY);

	if (fd < 0) {
		(void)fprintf(stderr, "%s: ", program_name);
		perror(device);
		exit(-1);
	}
	return fd;
}

/*
 * Open and initialize bpf interface, return file descriptor.
 */
int
bpf_init(device, pflag)
	char *device;
	int pflag;
{
	int if_fd;
	struct timeval timeout;
	struct ifreq ifr;
	struct bpf_version bv;
	char buf_err[80];

	if_fd = bpf_open();

	if (ioctl(if_fd, BIOCVERSION, (caddr_t)&bv) < 0)
		warning("kernel bpf interpreter may be out of date");
	else if (bv.bv_major != BPF_MAJOR_VERSION ||
		 bv.bv_minor < BPF_MINOR_VERSION)
		error("requires bpf language %d.%d or higher; kernel is %d.%d",
		      BPF_MAJOR_VERSION, BPF_MINOR_VERSION,
		      bv.bv_major, bv.bv_minor);

	strncpy(ifr.ifr_name, device, sizeof(ifr.ifr_name));
	if (ioctl(if_fd, BIOCSETIF, (caddr_t)&ifr) < 0) {
		fprintf(stderr, "%s: BIOCSETIF: ", program_name);
		perror(device);
		exit(-1);
	}
	/* Get the data link layer type. */
	if (ioctl(if_fd, BIOCGDLT, (caddr_t)&link_type) < 0) {
		sprintf(buf_err, "%s: BIOCGDLT", program_name);
		perror(buf_err);
		exit(-1);
	}
	/* set timeout */
	timeout.tv_sec = 1;
	timeout.tv_usec = 0;
	if (ioctl(if_fd, BIOCSRTIMEOUT, (caddr_t)&timeout) < 0) {
		sprintf(buf_err, "%s: BIOCSRTIMEOUT", program_name);
		perror(buf_err);
		exit(-1);
	}
	/* set promiscuous mode if requested, but only for broadcast nets */
	if (pflag == 0) {
		switch (link_type) {

		case DLT_SLIP:
		case DLT_PPP:
		case DLT_NULL:
			break;

		default:
			if (ioctl(if_fd, BIOCPROMISC, (void *)0) < 0) {
				sprintf(buf_err, "%s: BIOCPROMISC",
					program_name);
				perror(buf_err);
				exit(-1);
			}
		}
	}
	lookup_net(device);
	usage_func = link_func();
	return(if_fd);
}

/*
 * Get bpf status and Put it to &recv and &drop, return 0 if fail.
 */
int
bpf_stats(fd, recv, drop)
	int fd;
	unsigned int *recv;
	unsigned int *drop;
{
	struct bpf_stat s;
	
	if (ioctl(fd, BIOCGSTATS, &s) < 0)
		return 0;
	*recv = s.bs_recv;
	*drop = s.bs_drop;
	return 1;
}

/*
 * Main loop for reading ip packets from assigned bpf interface,
 * will be called (*usage_func)() per each packet. / see also interfaces.c /
 *
 * Must exit only if set up packets counter (cnt) and return 0,
 * else return 1 and set errno when come across error reading from bpf.
 */
int
bpf_readloop(cnt, if_fd, fp)
	int cnt;
	int if_fd;
	struct bpf_program *fp;
{
	u_char *buf;
	u_int bufsize;
	char buf_err[80];
	int cc;

	if (ioctl(if_fd, BIOCGBLEN, (caddr_t)&bufsize) < 0) {
		sprintf(buf_err, "%s: BIOCGBLEN", program_name);
		perror(buf_err);
		exit(1);
	}
	buf = (u_char *)malloc(bufsize + 4);

	if (fp == NULL)
		fp = parse(NULL, 1);
	if (ioctl(if_fd, BIOCSETF, (caddr_t)fp) < 0) {
		sprintf(buf_err, "%s: BIOCSETF", program_name);
		perror(buf_err);
		exit(1);
	}
	while (1) {
		register u_char *bp, *btp;
		register int hdrlen, caplen, bplen;

		if ((cc = read(if_fd, (char *)buf, (int)bufsize)) < 0) {
			/* Don't choke when we get ptraced */
			if (errno == EINTR)
				continue;
#if defined(sun) && !defined(BSD)
			/*
			 * Due to a SunOS bug, after 2^31 bytes, the kernel
			 * file offset overflows and read fails with EINVAL.
			 * The lseek() to 0 will fix things.
			 */
			if (errno == EINVAL &&
			    (long)(tell(if_fd) + bufsize) < 0) {
				(void)lseek(if_fd, 0, 0);
				continue;
			}
#endif
			return 0; /* signaling Error, use errno */
		}
		/*
		 * Loop through each packet.
		 */
#define bhp ((struct bpf_hdr *)bp)
		bp = btp = buf;
		bplen = 0;

		/*
		 * This is original loop are potentialy dangerous,
		 * function (*usage_func)() may crash to core!
		 *
		 * while (bp < ep) {
		 *	register int caplen, hdrlen;
		 *	if (cnt >= 0 && --cnt < 0)
		 *		goto out;
		 *	(*usage_func)(bp + (hdrlen = bhp->bh_hdrlen),
		 *		   bhp->bh_datalen, caplen = bhp->bh_caplen);
		 *	bp += BPF_WORDALIGN(caplen + hdrlen);
		 * }
		 */

		/* Use it instead above, all works fine! */
/*
* Serge P. Kovalyov <kovalyov@math.nsk.su> has been traced it and found:
* if a buffer contains n IP packets then exactly n-1 of them will be processed.
*
*		while ((btp += BPF_WORDALIGN((hdrlen = bhp->bh_hdrlen) +
*					     (caplen = bhp->bh_caplen))) < ep) {
*			if (cnt >= 0 && --cnt < 0)
*				return 1;
*			(*usage_func)(bp + hdrlen, bhp->bh_datalen, caplen);
*			bp = btp;
*		}
*/
		/* Thanks to Serge Kovalyov for the fix of this */
		if (cc) do {
			if (cnt >= 0 && --cnt < 0)
				return 1; /* left packets counter */
			bp = buf + bplen;
			hdrlen = (int)bhp->bh_hdrlen;
			caplen = bhp->bh_caplen;
			bplen += BPF_WORDALIGN(caplen + hdrlen);
			if (bplen > cc) break; /* bplen == cc - processed! */
			(*usage_func)(bp + hdrlen, bhp->bh_datalen, caplen);
		} while (bplen < cc);
#undef bhp
	}
	return 1; /* ???, don't work, only for compiler */
}
