/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

/*
 * PDP11
 *
 * this is an implementation of a memory that caches virtual
 * to physical address translations with a lot of memory overhead.
 */
# include "proc.h"

typedef struct Space	Space;
typedef struct SpaceReg	SpaceReg;

struct SpaceReg {
	unsigned	par;		/* page address << 6 shifted	*/
	ushort		pdr;		/* orig pdr			*/
	unsigned	plf;		/* plf in right place		*/
	unsigned	tag;		/* translation tag		*/
	ushort		*ppdr;		/* pointer to real pdr		*/
};
	
struct Space {
	SpaceReg	r[8];		/* registers			*/
	unsigned	c[0x10000];	/* adress translation cache	*/
	unsigned	t[0x10000];	/* translation tags		*/
};

struct MMG {
	Space		s[4][2];	/* address spaces		*/
	Space		*a[4][2];	/* access spaces		*/
	SpaceReg	*s_last;	/* last accessed space		*/
	int		hits;		/* cache hits			*/
	int		misses;		/* cache misses			*/
	int		off;		/* disables if true		*/
	int		mmr0;		/* copy of proc.mmr0		*/
	int		mmr3;		/* copy of proc.mmr3		*/
	ushort		mem[Off(PHYSMEM)];
};

static void	flush_all(MMG *);
static void	flush_space(MMG *, int, int);
static void 	flush_page(MMG *, int, int, int);


void		cm_reset(MMG *);		/* bus reset		*/
void		cm_mmr0(MMG *);			/* mmr0 written		*/
void		cm_mmr3(MMG *);			/* mmr3 written		*/
void		cm_par(MMG *, int, int, int);	/* par written		*/
void		cm_pdr(MMG *, int, int, int);	/* pdr written		*/
void		cm_ccr(MMG *);			/* ccr written		*/
void		cm_info(MMG *);			/* odt info		*/
void		cm_store16(unsigned, int, int, int, ushort);
void		cm_store18(unsigned, int, int, int, ushort);
void		cm_store22(unsigned, int, int, int, ushort);
unsigned	cm_fetch16(unsigned, int, int);
unsigned	cm_fetch18(unsigned, int, int);
unsigned	cm_fetch22(unsigned, int, int);
ushort  	*cm_addrphys(MMG *, unsigned);	/* for dma		*/

MMG		mmg;		/* can have only one		*/
Memops		memops = {
	cm_reset,		/* reset	*/
	cm_mmr0,		/* mmr0		*/
	cm_mmr3,		/* mmr3		*/
	cm_par,			/* par		*/
	cm_pdr,			/* pdr		*/
	cm_ccr,			/* ccr		*/
	cm_info,		/* info		*/
	cm_store16,		/* store16	*/
	cm_store18,		/* store18	*/
	cm_store22,		/* store22	*/
	cm_fetch16,		/* fetch16	*/
	cm_fetch18,		/* fetch18	*/
	cm_fetch22,		/* fetch22	*/
	cm_addrphys,		/* addrphys	*/
};


/*
 * 16-bit mapping.
 */
unsigned
cm_fetch16(unsigned a, int dspace, int mode)
{
	if(a >= 0160000)
		return iofetch(a | 017760000);
	if(a >= proc.physmem)
		Trap4(040);
	return mmg.mem[Off(a)];
}

void
cm_store16(unsigned a, int type, int dspace, int mode, ushort v)
{
	if(a >= 0160000) {
		iostore(a | 017760000, type, v);
		return;
	}
	if(a >= proc.physmem)
		Trap4(040);

	if(type & M_Low)
		SETLB(mmg.mem[Off(a)], v);
	else if(type & M_High)
		SETHB(mmg.mem[Off(a)], v);
	else
		mmg.mem[Off(a)] = v;
}


/* 
 * 18/22 bit mapping
 */

/*
 * Translate logical to physical address with check.
 * The 18/22 bit routines are essential the same.
 */
# define GEN(NAME,MASK,IOP,OR)							\
static inline unsigned 							\
physaddr##NAME(unsigned a, int mode, int dspace, int store)		\
{									\
	int 		page = a >> 13;					\
	ushort 		pdr;						\
	unsigned 	plf;						\
	unsigned 	pa, bno;					\
	Space		*s;						\
									\
	s = mmg.a[mode][dspace];					\
	mmg.s_last = &s->r[page];					\
	if(!mmg.off && mmg.s_last->tag == s->t[a]) {			\
		mmg.hits++;						\
		return s->c[a];						\
	}								\
	mmg.misses++;							\
									\
	if(mode == 2)							\
		MemTrap(page, mode, dspace, 0100000);			\
									\
	pdr = mmg.s_last->pdr;						\
	plf = mmg.s_last->plf;						\
									\
	if(!(pdr & 02))							\
		MemTrap(page, mode, dspace, 0100000);			\
	if(store && !(pdr & 04))					\
		MemTrap(page, mode, dspace, 020000);			\
									\
	bno = a & 017700;						\
									\
	if(pdr & 010) {							\
		if(bno < plf)						\
			MemTrap(page, mode, dspace, 040000);		\
	} else {							\
		if(bno > plf)						\
			MemTrap(page, mode, dspace, 040000);		\
	}								\
	s->t[a] = mmg.s_last->tag;					\
	return s->c[a] = (mmg.s_last->par + (a & 017777)) & MASK;	\
}									\
									\
									\
									\
/*									\
 * Fetch from logical address a, in space 				\
 */									\
unsigned								\
cm_fetch##NAME(unsigned a, int dspace, int mode)			\
{									\
	unsigned pa = physaddr##NAME(a, mode, dspace, 0);		\
									\
	if(pa >= IOP)							\
		return iofetch(pa | OR);				\
									\
	if(pa >= proc.physmem)						\
		Trap4(040);						\
									\
	return mmg.mem[Off(pa)];					\
}									\
									\
									\
void									\
cm_store##NAME(unsigned a, int type, int dspace, int mode, ushort v)	\
{									\
	unsigned pa;							\
									\
	pa = physaddr##NAME(a, mode, dspace, 1);			\
									\
	if(pa >= IOP) {							\
		iostore(pa | OR, type, v);				\
		mmg.s_last->pdr |= 0100;				\
		return;							\
	}								\
	if(pa >= proc.physmem)						\
		Trap4(040);						\
									\
	if(type & M_Low)						\
		SETLB(mmg.mem[Off(pa)], v);				\
	else if(type & M_High)						\
		SETHB(mmg.mem[Off(pa)], v);				\
	else {								\
		mmg.mem[Off(pa)] = v;					\
	}								\
	mmg.s_last->pdr |= 0100;					\
}
GEN(18,      0777777,   0760000, 017000000)
GEN(22, 037777777777, 017760000,         0)

/*
 * routines called from generic interface
 */
void
mem_init(void)
{
	int mode, space, reg;

	for(mode = 0; mode < 4; mode++) {
		for(space = 0; space < 2; space++) {
			mmg.a[mode][space] = &mmg.s[mode][0];
			for(reg = 0; reg < 8; reg++)
				mmg.s[mode][space].r[reg].ppdr = &proc.pdr[mode][space][reg];
		}
	}
	flush_all(&mmg);
	proc.mmg = &mmg;
	proc.memops = &memops;
	memsys.data = &mmg;
}

void	
cm_reset(MMG *m)
{
	cm_mmr0(m);
	cm_mmr3(m);
	flush_all(m);
}

void
cm_mmr0(MMG *m)
{
	/*
	 * must flush cache, if mapping is switched on/off
 	 */
	if(m->mmr0 != (proc.mmr0 & 1))
		flush_all(m);
	m->mmr0 = proc.mmr0 & 1;
}

void
cm_mmr3(MMG *m)
{
	/*
	 * flush cache if mapping is switched to/from 22-bit
	 */
	if((proc.mmr0 & 1) && (m->mmr3 & 020) != (proc.mmr3 & 020))
		flush_all(m);
	/*
	 * recompute pointers to accessible address spaces 
	 * If the change enables a d-space, it must be flushed.
	 */
	if(proc.mmr3 & 4) {
		m->a[0][1] = &m->s[0][1];
		flush_space(m, 0, 1);
	} else
		m->a[0][1] = &m->s[0][0];

	if(proc.mmr3 & 2) {
		m->a[1][1] = &m->s[1][1];
		flush_space(m, 1, 1);
	} else
		m->a[1][1] = &m->s[1][0];

	if(proc.mmr3 & 1) {
		m->a[3][1] = &m->s[3][1];
		flush_space(m, 3, 1);
	} else
		m->a[3][1] = &m->s[3][0];
	m->mmr3 = proc.mmr3;
}


void
cm_par(MMG *m, int mode, int space, int reg)
{
	unsigned newpar = proc.par[mode][space][reg] << 6;

	if(m->s[mode][space].r[reg].par != newpar) {
		m->s[mode][space].r[reg].par = newpar;
		flush_page(m, mode, space, reg);
	}
}


void
cm_pdr(MMG *m, int mode, int space, int reg)
{
	ushort newpdr = proc.pdr[mode][space][reg] & 077416;

	if(m->s[mode][space].r[reg].pdr != newpdr) {
		m->s[mode][space].r[reg].pdr = newpdr;
		m->s[mode][space].r[reg].plf = (newpdr & 077400) >> 2;
		flush_page(m, mode, space, reg);
	}
}

void
cm_ccr(MMG *m)
{
	if(proc.ccr & 0400)
		flush_all(m);
	m->off = proc.ccr & 060;
}

/*
 * flush translation cache of one page in a specific address space
 */
static void
flush_page(MMG *m, int mode, int space, int page)
{
	m->s[mode][space].r[page].tag++;
}

/* 
 * flush an entire address space
 */
static void
flush_space(MMG *m, int mode, int space)
{
	int i;

	for(i = 0; i < 8; i++)
		flush_page(m, mode, space, i);
}

/*
 * flush the entire cache
 */
static void
flush_all(MMG *m)
{
	flush_space(m, 0, 0);
	flush_space(m, 0, 1);
	flush_space(m, 1, 0);
	flush_space(m, 1, 1);
	flush_space(m, 3, 0);
	flush_space(m, 3, 1);
}

void
cm_info(MMG *m)
{
	printf("Translation cache memory\n");
	printf("Hits/Misses %d/%d = %f %% hits\n", m->hits, m->misses,
		100.0 * m->hits / ((double)m->hits + m->misses));
	printf("Cache is %s\n", mmg.off ? "off" : "on");
}

/*************************************************************
 *
 * physical memory access (for dma and monitor)
 */
ushort *
cm_addrphys(MMG *m, unsigned a)
{
	return &m->mem[Off(a)];
}
