IMPLEMENTATION MODULE BBCVDU;

(*
   Author:         Brian Williams
   Implementation: University of Hamburg Modula-2 under VAX/VMS version 4
   Date Started:   30/6/86
                 - Use graphic routines from TEK4010VDU.

   Description:
   Implements the InitBBC procedure that initializes the terminal dependent
   routines and parameters used in DVItoVDU.
   The BBC VDU assumes the use of TERMULATOR, which allows free movement
   between VT100, Tek4010, and BBC Vdu modes of operation, via private
   escape sequences.
   The Tektronix 4010 terminal emulation has a screen 
   1024 pixels wide by 780 pixels high.
   (The actual resolution of the BBC screen is 640 by 256 pixels;
   the appropriate scaling is done automatically.)

   The bottom left pixel is the point (x=0,y=0). Note that x coordinates
   increase to the right and y coordinates increase up the screen.
   DVItoVDU uses a coordinate scheme in which horizontal (=h) coordinates
   also increase to the right but vertical (=v) coordinates increase DOWN the
   screen, i.e. the top left pixel on the screen is the point (h=0,v=0).
   This means that the Tektronix 4010 routines will have to do a
   simple translation of the vertical coordinates passed by DVItoVDU.
*)

FROM VDUInterface IMPORT
   (* PROCEDURE variables: *)
   MoveToTextLine,
   ClearTextLine,
   ClearScreen,
   StartText,
   StartGraphics,
   LoadFont,
   ShowChar,
   ShowRectangle,
   ResetVDU,
   (* INTEGER variables: *)
   DVIstatusl,
   windowstatusl,
   messagel,
   commandl,
   bottoml,
   windowv,
   windowh,
   windowht,
   windowwd;

FROM TEK4010VDU IMPORT
   TEK4010ClearScreen,
   TEK4010StartText,
   TEK4010StartGraphics,
   TEK4010LoadFont,
   TEK4010ShowChar,
   TEK4010ShowRectangle;

FROM ScreenIO IMPORT
   Write, WriteCard,
   WriteString;


CONST
   CAN = 30C;
   ESC = 33C;
   GS  = 35C;
   FS  = 34C;
   US  = 37C;


(**********************************************************************)

PROCEDURE MoveAbs (row, col : CARDINAL);

(* We move into BBC mode and move the cursor to the
   given position.  BBC numbers text lines from zero.
*)

BEGIN
Write(ESC); WriteString('[*v');
Write(US);
Write(CHR(col-1));
Write(CHR(row-1));
Write(ESC); WriteString('[%v');
END MoveAbs;

(**********************************************************************)

PROCEDURE BBCMoveToTextLine (line : CARDINAL);

BEGIN
Write(US);   (* switch to alpha mode *)
Write(CAN);
MoveAbs(line,1);
END BBCMoveToTextLine;

(**********************************************************************)

PROCEDURE BBCClearTextLine (line : CARDINAL);

BEGIN
Write(US);   (* switch to alpha mode *)
Write(CAN);
MoveAbs(line,1);
WriteString('                                                     ');
WriteString('                           ');   (* erase to end of line *)
MoveAbs(line,1);
END BBCClearTextLine;

(**********************************************************************)

PROCEDURE BBCClearScreen;

BEGIN
Write(ESC);   (* switch to alpha mode *)
Write(CHR(12));
END BBCClearScreen;

(**********************************************************************)

PROCEDURE BBCResetVDU;

BEGIN
Write(ESC); WriteString('[%t');
END BBCResetVDU;

(**********************************************************************)

PROCEDURE InitBBC;

(* The dialog region will be the top 4 text lines in VT100 mode:
      Line 1 = DVI status line,
      Line 2 = window status line,
      Line 3 = message line,
      Line 4 = command line.
   The window region will be text lines 5 to 24 in VT100 mode.
*)

BEGIN
DVIstatusl := 1;      (* DVItoVDU assumes top text line = 1 *)
windowstatusl := 2;
messagel := 3;
commandl := 4;
bottoml := 24;        (* also number of text lines on screen *)
(* The above values assume the BBC is in BBC mode;
   the following values assume it is emulating a Tektronix 4010.
   Note that windowv must be given a value using DVItoVDU's coordinate scheme
   where top left pixel is (0,0).
*)
windowv := 130;       (* approx. height in TEK4010 pixels of 4 text lines
                         i.e. 4 * 780/24 *)
windowh := 0;
windowht := 780-windowv;
windowwd := 1024;

MoveToTextLine := BBCMoveToTextLine;
ClearTextLine  := BBCClearTextLine;
ClearScreen    := BBCClearScreen;
StartText      := TEK4010StartText;
StartGraphics  := TEK4010StartGraphics;
LoadFont       := TEK4010LoadFont;
ShowChar       := TEK4010ShowChar;
ShowRectangle  := TEK4010ShowRectangle;
ResetVDU       := BBCResetVDU;
Write(ESC); WriteString('[*t');
END InitBBC;

(**********************************************************************)

BEGIN
END BBCVDU.
