/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/* All Kerning and Ligature stuff goes here. */

#include <stdio.h>
#include "defs.h"
#include "pfd2tfm.h"
#include "char.h"
#include "defenc.h"
#include "extfil.h"

extern int Verbose;
extern char * Malloc();
extern char * StrcpyAlloc();
extern int LigatureDefs;
extern CAFM AfmChar[MAX_CHAR_PS];
extern int CharCode;
extern char * CharacterNameHashTable;
extern char * ps_def_encodings_dir;

extern struct kpx KerningTable [KERNS];
extern int KerningTableIndex;
extern struct lig LigatureTable [LIGATURES];
extern int LigatureTableIndex;

void li();

/*
 * SaveLigatureInfo
 * ****************
 * Save ligature information.
 *
 * name1: the first character's name.
 * name2: the second character's name.
 * lign:  the resulting ligature to be used.
 */
void
SaveLigatureInfo (name1, name2, lign)
     char *name1;
     char *name2;
     char *lign;
{
  if (Verbose > V_SOME)
    fprintf (stderr, "SaveLigatureInfo(): \"%s\" and \"%s\" give \"%s\"\n",
	     name1, name2, lign);
  LigatureTable[LigatureTableIndex].l_cleft = StrcpyAlloc(name1);
  LigatureTable[LigatureTableIndex].l_cright = StrcpyAlloc(name2);
  LigatureTable[LigatureTableIndex].l_lig_name = StrcpyAlloc(lign);
  if (++LigatureTableIndex == LIGATURES)
    Fatal("SaveLigatureInfo(): Too many ligature pairs in afm file");
}

/*
 * KernAndLigInit
 * **************
 * Inititialisation of the kerning and ligature table.
 */
void
KernAndLigInit()
{
  KerningTableIndex = 0;
  LigatureTableIndex = 0;
}

/*
 * BuildKernStructure
 * ******************
 * Build the kerning table by going back to the two sources of
 * kerning information (from the pfd file, and from the afm file).
 * Now everything is known, and the kerning pairs of characters
 * can be made hanging of AfmChar, so the data can be accessed, when
 * the propertly list for the generation of the tfm file is later
 * written.
 * This procedure builds the links between the AfmChar table
 * and the kerning information either collected from the PFD file or
 * the AFM file. Each character has a pointer to a linked list,
 * which identifies any other character, with which there is some
 * kerning. This organization is needed later, when the .pl file is
 * written to generate the tfm file with the help of pltotf.
 * table: the table containing all the kerning information.
 * num_els: how many kerning pairs it contains.
 */
void
BuildKernStructure ()
{
  int i;
  int leftchar_index;
  struct kpx * kptr;
  struct kpx * oldkptr;

  /* Loop through all table elements: This way we establish
   * information from the left character to all right characters
   * with which there is some kerning going on. The insertion
   * procedure below has been done in such a way, that later
   * insertions will be simply rejected, preventing a kerning instruction
   * from the AFM file being inserted.
   */
  for (i=0; i<KerningTableIndex; i++) {
    /* Get the indices of the two kerned characters */
    leftchar_index = CharNameToIndex(KerningTable[i].k_cleft);
    if ((kptr=AfmChar[leftchar_index].c_kern) == NULL) {
      AfmChar[leftchar_index].c_kern = &KerningTable[i];
    } else {
      while (kptr != NULL &&
	     strcmp (kptr->k_cright, KerningTable[i].k_cright) != 0) {
	oldkptr = kptr;
	kptr = kptr->k_next;
      }
      if (kptr == NULL) {
	oldkptr->k_next = &KerningTable[i];
      } else {
	fprintf (stderr, "Warning: Duplicate kern entry: \"%s\"-\"%s\"\
 ignored\n", KerningTable[leftchar_index].k_cleft, KerningTable[i].k_cright);
      } /* fi */
    } /* fi */
  } /* for */
}

/*
 * BuildLigatureStructure
 * **********************
 * See, whether there are certain characters available,
 * and if so add ligatures.
 * Then add the ligature structure to AfmChar[].
 */
void
BuildLigatureStructure()
{
  int i, left_i;
  LIG_P p;
  EX_FILES ex_f;
  char buffer[256];
  char ch1[256];
  char ch2[256];
  char ch_lig[256];

  /* Read in ligature information from an external file. The
     file's number is what we need. */
  if (LigatureDefs != -1) {
    sprintf (buffer, "%s/%s%d.lig", ps_def_encodings_dir, DEF_LIG, LigatureDefs);
    FExOpen(&ex_f, EFT_READ, 0, buffer, NULL);
    while (fscanf(EX_FP(ex_f), "%s %s %s", ch1, ch2, ch_lig) == 3) {
      SaveLigatureInfo(ch1, ch2, ch_lig);
    }
    FExClose(&ex_f);
  }

  /* Now build the linked list of ligature information, so this info
     can be written to the pl file. */
  for (i=0; i<LigatureTableIndex; i++) {
    left_i = CharNameToIndex(LigatureTable[i].l_cleft);
    p = AfmChar[left_i].c_lig;
    AfmChar[left_i].c_lig = &LigatureTable[i];
    LigatureTable[i].l_next = p;
  } /* for */
}

#ifdef ABC
/*
 * li
 * **
 * If three characters are in the hash table, add a ligature
 * def.
 * n1, n2: two names
 * li: the resulting ligature
 */
void
li(n1, n2, lig)
     char *n1, *n2, *lig;
{
  if (LookUpKeyInHashTable(CharacterNameHashTable, n1) != NULL &&
      LookUpKeyInHashTable(CharacterNameHashTable, n2) != NULL &&
      LookUpKeyInHashTable(CharacterNameHashTable, lig) != NULL)
    SaveLigatureInfo(n1, n2, lig);
}
#endif

/*
 * SaveKerningInfo
 * ***************
 * Save kerning information (from .afm or .pfd file).
 *
 * name1, name2: the two names of characters, between which the kerning
 *          is done;
 * amount: how much kerning
 */
void
SaveKerningInfo (name1, name2, amount)
     char * name1;
     char * name2;
     int amount;
{
  if (Verbose > V_SOME)
    fprintf (stderr, "SaveKerningInfo(): [%3d]: \"%s\"-\"%s\": %d\n",
	     KerningTableIndex, name1, name2, amount);
  KerningTable[KerningTableIndex].k_cleft = name1;  
  KerningTable[KerningTableIndex].k_cright = name2;  
  KerningTable[KerningTableIndex].k_dist = amount;
  if (++KerningTableIndex == KERNS)
    Fatal("KerningFromAfmFile(): Too many kern pairs in AFM file.");
}
