/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/* Default character code vector business. */

#include <stdio.h>
#include "defs.h"
#include "pfd2tfm.h"
#include "char.h"
#include "defenc.h"
#include "extfil.h"

extern CAFM AfmChar[MAX_CHAR_PS];
extern char * SetUpHashTable();
extern char * GetNextKeyHash();
extern char * LookUpKeyInHashTableE();
extern char * ps_def_encodings_dir;
extern int Verbose;

/* Structure to maintain information about default encoding vectors */
typedef struct fh {
  int fh_code; /* What code by default? */
  int fh_used; /* Was this default ever used? */
} FE, *FE_P;

typedef struct evd { /* Encoding vector default */
  int evd_class; /* Default encoding class */
  int evd_ligat; /* Default ligature information */
} EVD, *EVD_P;

char * DefEncHash; /* Maps font names to encoding class numbers. */
char * DefaultCharHashTable[NEVS]; /* For each encoding class map character
				      name to code. */

/* Results from running lex go here! */
int DefaultInt;
char * DefaultString;

int EncodingClass; /* [0..ENVS] or -1 (= no defaults) */
int LigatureDefs; /* Default ligature number, = -1: none */

/*
 * LookUpDefCode
 * *************
 * Look up default code.
 * name: character for which the code is looked up.
 *
 * RET: which character code was found, -1 if none.
 */
int
LookUpDefCode(name)
     char * name;
{
  FE_P p;
  int code;

  /* Is encoding class ok. */
  if (EncodingClass != -1 && (EncodingClass < 0 || EncodingClass >= NEVS))
    Fatal ("LookUpDefCode(): illegal encoding class");
  
  /* EncodingClass -1: no default encoding. */
  if (EncodingClass == -1)
    return (-1);

  /* If not in hash table return -1. */
  if ((p=(FE_P)LookUpKeyInHashTable(DefaultCharHashTable[EncodingClass], name))==NULL)
    return (-1);

  /* "code" is the character code found in the hash table. */
  code = p->fh_code;
  p->fh_used = TRUE;

  if (AfmChar[code].c_used) {
    fprintf (stderr, "LookUpDefCode(\"%s\"): '%o already used [\"%s\"]\n",
	     name, code, AfmChar[code].c_string);
    return (-1);
  }
  if (AfmChar[code].c_map &&
      Strcmp(AfmChar[code].c_string, name) != NULL) {
    fprintf (stderr, "LookUpDefCode(\"%s\"): '%o reserved for map\n",
	     name, code);
    return (-1);
  }
  return (code);
}

/*
 * LoadDefEVectors
 * ***************
 * Load all the default encoding vectors. Also load the map which
 * maps the name of the font to the encoding class which has to
 * be selected.
 */
void
LoadDefEVectors()
{
  int i; /* Loop index through encoding vectors */
  char buffer[256];
  EX_FILES ex_f; /* Default encoding vector file */
  int token; /* Token read from encoding vector file. */
  int token_2; /* Token read from encoding vector file. */
  int i1, i2;
  FE_P fe_p;
  EVD_P evd_p;

  /* Load file DEF_CLASSES which contains a map (from font name) to find
     the default encoding vector and the default ligature information. */
  DefEncHash = SetUpHashTable (15, sizeof(EVD), "Encoding Vector Defaults");
  sprintf (buffer, "%s/%s", ps_def_encodings_dir, DEF_CLASSES);
  
  FExOpen(&ex_f, EFT_READ, EFQ_NO_STDIN, buffer, NULL);
  SetupLex (&ex_f, DEFENC_LEX, FALSE);

  while ((token=Mylex()) != 0) {
    if (token != TOC_ID)
      Fatal ("LoadDefVectors(): id expected");
    strcpy (buffer, DefaultString);
    if ((token=Mylex()) != TOC_INT)
      Fatal ("LoadDefVectors(): integer expected");
    i1 = DefaultInt;
    if ((token=Mylex()) != TOC_INT)
      Fatal ("LoadDefVectors(): integer expected");
    i2 = DefaultInt;
#ifdef DEBUG
    fprintf (stderr, "Font \"%s\" -> ev: %d, lig: %d\n",
	     buffer, i1, i2);
#endif
    evd_p = (EVD_P) InsertKeyIntoHashTable (DefEncHash, DefaultString);
    evd_p->evd_class = i1;
    evd_p->evd_ligat = i2;
  }
  
  FExClose(&ex_f);

  /* Load default encoding vectors. */
  for (i=0; i<NEVS; i++) {
    sprintf (buffer, "Default encoding vectors: hash table #%d", i);
    DefaultCharHashTable[i] = SetUpHashTable (150, sizeof(FE), buffer);
    sprintf (buffer, "%s/%s%d.%s", ps_def_encodings_dir, DEF_EV, i, DEF_EV_EXT);
    FExOpen (&ex_f, EFT_READ, 0, buffer, NULL);
    SetupLex (&ex_f, DEFENC_LEX, FALSE);

    while ((token=Mylex()) != 0) {
      token_2 = Mylex();
#ifdef DEBUG
      fprintf (stderr, "LoadDefEVectors(): %s->'%o\n",
	       DefaultString, DefaultInt);
#endif
      if (token != TOC_ID || token_2 != TOC_INT)
	Fatal2 ("LoadDefEVectors(): Error in file %s", buffer);
      fe_p = (FE_P) InsertKeyIntoHashTable (DefaultCharHashTable[i], DefaultString);
      fe_p->fh_code = DefaultInt;
      fe_p->fh_used = FALSE;
    }
    FExClose(&ex_f);
  }
}

/*
 * InitEncodingClass
 * *****************
 * It is here where we compute the default encoding class and the
 * default ligatures for a particular font.
 *
 * name: of the font (PostScript Name).
 */
void
InitEncodingClass (name)
     char * name;
{
  EVD_P evd_p;

  EncodingClass = 0; /* Class 0 default encoding */
  LigatureDefs = 0; /* Default ligatures */

  if ((evd_p=(EVD_P)LookUpKeyInHashTable(DefEncHash, name)) != NULL) {
    EncodingClass = evd_p->evd_class;
    LigatureDefs =  evd_p->evd_ligat;
  }
  if ((EncodingClass < 0 || EncodingClass >= NEVS) && EncodingClass != -1)
    Fatal2 ("Illegal encoding class %d", EncodingClass);
  if (Verbose > V_SOME) {
    fprintf (stderr, "InitEncodingClass(): Encoding class = %d\n",
	     EncodingClass);
    fprintf (stderr, "InitEncodingClass(): Ligature class = %d\n",
	     LigatureDefs);
  }

}

/*
 * ReportUsageDefCodes
 * *******************
 * Report those default codes which were never used.
 */
void
ReportUsageDefCodes()
{
  FE_P ptr;
  char * key;

  if (EncodingClass == -1)
    return;
  if (Verbose == V_QUIET)
    return;
  InitNextKeyHash (DefaultCharHashTable[EncodingClass]);
  while((key = GetNextKeyHash(DefaultCharHashTable[EncodingClass]))!=NULL){
    ptr = (FE_P) LookUpKeyInHashTableE (DefaultCharHashTable[EncodingClass], key,
					"ReportUsageDefCodes()");
    if (ptr->fh_used)
      continue;
    fprintf (stderr, "ReportUsageDefCodes(): not used: \"%s\"\n", key);
  }
}
