/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


#include <stdio.h>
#include "defs.h"
#include "units.h"
#include "dvitps.h"
#include "pk-com.h"
#include "extfil.h"
#include "fontd.h"

extern char *GetBytes();
extern void HandleCheckSum();
extern void WarningTfmVersusPxlWidth();
extern void PsDownLoad();
extern void OpenFontFileCached();
extern int HConv;

extern FE_P CurFontPointer;

/*
 * ReadPkFileInfo
 * **************
 * Read the basic information from a pk file.
 */
void
ReadPkFileInfo()
{
  char *buffer;
  int command;

  int paint_switch;
  int i;
  int size_par;
  int dyn_f;

  int flag;
  int pl;
  int cc;
  int tfm; /* Tfm width read from .pk file. */
  int dm;
  int dx, dy;
  int w;
  int h;
  int hoff;
  int voff;
  int length_pre;
  int preamble_code;
  int k;
  int done;
  CE_P ce;
  int check_sum; /* The check sum as found in the .pk file. */
  int fseek_amount; /* Number of bytes in each of the three character formats
		       from tfm[*] to voff[*] */

  PsDownLoad ("pixel-fonts.pro", 0);

  OpenFontFileCached();

  if (NoSignExtend(CurFontPointer->f_ex_file.ef_filep, 1) != PK_PRE)
    Fatal ("ReadPkFileInfo(): illegal preamble");
  if (NoSignExtend(CurFontPointer->f_ex_file.ef_filep, 1) != PK_ID)
    Fatal ("ReadPkFileInfo(): illegal pk file id");
  buffer = GetBytes (CurFontPointer->f_ex_file.ef_filep,
		     NoSignExtend(CurFontPointer->f_ex_file.ef_filep, 1));
  NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4); /* skip that. */

  /* Checksum business. */
  check_sum = NoSignExtend(CurFontPointer->f_ex_file.ef_filep, 4);
  switch (CurFontPointer->f_class) {
    case FOCLASS_AS_REQUESTED:
      HandleCheckSum (check_sum);
      break;
    case FOCLASS_REPLACEMENT_FONT:
      /* Note that when a replacement font is used the checksum as found in
	 the dvi file has no relevancy. */
      CurFontPointer->f_c = check_sum;
      break;
    default:
      Fatal ("ReadPkFileInfo(): FOCLASS_ illegal.");
    }

  NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4); /* hppp */
  NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4); /* vppp */

  /* No characters downloaded. */
  for (i=0; i<MAX_CHARS; i++)
    CurFontPointer->f_ch[i].c_dl = FALSE;
#ifdef DEBUG
  fprintf (stderr, "%% PK font file name: %s\n", CurFontPointer->f_ex_file.ef_fn);
#endif

  done = FALSE;
  while (! done) {
    command = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 1);
#ifdef DEBUG
    fprintf (stderr, "%% PK_code: '%o\n", command);
#endif
    switch (command) {
      case PK_XXX1:
      case PK_XXX2:
      case PK_XXX3:
      case PK_XXX4:
        k = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, command-PK_XXX1+1);
	FExSeek (&(CurFontPointer->f_ex_file), k, FSEEK_REL);
	break;
      case PK_YYY:
	NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4);
        break;
      case PK_POST:
	done = TRUE;
        break;
      case PK_NO_OP:
        break;
      default:
	flag = command;
        dyn_f = (flag & 0xF0) >> 4;
	if (dyn_f == 15)
	  Fatal ("ReadPkFileInfo(): dny_f = 15");
        paint_switch = (flag & 0x8) >> 3;
        size_par = (flag & 0x4) >> 2;
        length_pre = (flag & 0x3);
	preamble_code = flag & 0x7;

	/* Now read in the preamble for a character. */
	if (preamble_code <= 3) { /* short format */
	  pl = (flag&0x3)*256 + NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 1);
	  cc = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 1);
#ifdef DEBUG
	  fprintf (stderr, "%% PK (short form): '%o\n", cc);
#endif
	  tfm = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 3);
	  dm = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 1);
	  dx = dm;
	  dy = 0;
	  w = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 1);
	  h = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 1);
	  hoff = SignExtend (CurFontPointer->f_ex_file.ef_filep, 1);
	  voff = SignExtend (CurFontPointer->f_ex_file.ef_filep, 1);
	  fseek_amount = 8;
	} else
	if (preamble_code <= 6) { /* extended short form */
	  pl = (flag&0x3)*65536 + NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 2);
	  cc = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 1);
#ifdef DEBUG
	  fprintf (stderr, "%% PK (extended short form): '%o\n", cc);
#endif
	  tfm = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 3);
	  dm = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 2);
	  dx = dm;
	  dy = 0;
	  w = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 2);
	  h = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 2);
	  hoff = SignExtend (CurFontPointer->f_ex_file.ef_filep, 2);
	  voff = SignExtend (CurFontPointer->f_ex_file.ef_filep, 2);
	  fseek_amount = 13;
	} else {                /* long format of chars */
	  pl = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4);
	  cc = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4);
#ifdef DEBUG
	  fprintf (stderr, "%% PK (long form): '%o\n", cc);
#endif
	  tfm = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4);
	  dm = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4);
	  dx = dm;
	  dy = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4);
	  w = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4);
	  h = NoSignExtend (CurFontPointer->f_ex_file.ef_filep, 4);
	  hoff = SignExtend (CurFontPointer->f_ex_file.ef_filep, 4);
	  voff = SignExtend (CurFontPointer->f_ex_file.ef_filep, 4);
	  fseek_amount = 28;
	} /* long format */

	/* Code common to all three forms of the preamble starts here. */
	if (cc >= MAX_CHARS)
	  Fatal ("ReadPkFileInfo(): char code >= MAX_CHARS");
	ce = &(CurFontPointer->f_ch[cc]);
	if (ce->c_dl)
	  Fatal ("ReadPkFileInfo(): ce->c_dl already set.");
	ce->c_flag = flag;

	/* tfm contains the tfm width from the pk file which is being
	   read in. */
	switch (CurFontPointer->f_class) {
	  case FOCLASS_AS_REQUESTED:
	    ce->c_w_tfm = (DVIU) (CurFontPointer->f_s * FIXES_TO_FLOAT(tfm));
	    WarningTfmVersusPxlWidth(cc, ce);
	    break;
	  case FOCLASS_REPLACEMENT_FONT:
	    /* In case of a replacement font the tfm width to be used is already
	       loaded (from the font which was supposed to be used originally.)
	       Now we convert from tfm units to DVIUs. */
	    ce->c_w_tfm = (DVIU) (CurFontPointer->f_s * FIXES_TO_FLOAT(ce->c_w_tfm));
	    break;
	  }
	ce->c_rw = HConv * PixRound (ce->c_w_tfm, HConv);
	ce->c_pw = w;
	ce->c_ph = h;
	ce->c_xoff = hoff;
	ce->c_yoff = voff;
	ce->c_offset = FExTell (&(CurFontPointer->f_ex_file));
#ifdef DEBUG
	fprintf (stderr, "%% PK: code: '%o, tfmw: %d, rounded: %d\n",
		 cc, ce->c_w_tfm, ce->c_rw);
	fprintf (stderr,"%%\t width: %d, height: %d, hoff: %d, voff: %d,\
 file offset: %d\n",
		 w, h, hoff, voff, ce->c_offset);
#endif
	FExSeek (&(CurFontPointer->f_ex_file), pl-fseek_amount, FSEEK_REL);
      } /* switch */
  } /* while */
}
