// Copyright (c) 1991 by Parag Patel.  All Rights Reserved.
static const char rcsid[] = "$Header: font.C,v 1.42 91/04/02 17:59:01 hmgr Exp $";

// font manipulation and basic typesetting functions
//
// by Parag Patel

#include "defs.h"


// local globals
static int fontsonpage;			// # of fonts on current page
static long devv;			// current device Vertical coord
static long devh;			// Horizontal coord (both in pixels)
static font *devfont;			// current font in device


font::font()
{
    chr = new fontchar[MAXCHARS];
    // this initialization is redundant, but safer
    fp = NULL;
    use = 0;
    path = basename = NULL;
    onpage = toomany = downloaded = setup = NULL;
}

font::~font()
{
    if (fp != NULL)
	fclose(fp);
    delete[MAXCHARS] chr;
}

// return the fontlist location for the font number specified
// - returns one more than the current size if there is no such font
static long getfontloc(long fnum)
{
    for (long f = 0; f < fontlist.size(); f++)
	if (fontlist[f] != NULL && fontlist[f]->num == fnum)
	    break;
    return f;
}

// clear font info for a new page - there are currently no fonts on
// this page - reset the device coordinates to "unknown" - there is no
// current font selected, nor a current font in the device
// 
void clearfonts()
{
    for (int i = 0; i < fontlist.size(); i++)
	if (fontlist[i] != NULL)
	    fontlist[i]->toomany = fontlist[i]->onpage = FALSE;
    fontsonpage = 0;
    devh = devv = -MAXLONG;
    currfont = devfont = NULL;
}


static void setupbits(font &f)
{
    static long minm = MAXLONG;
    static long maxm = -MAXLONG;
    static long minn = MAXLONG;
    static long maxn = -MAXLONG;

    if (f.minm >= minm && f.maxm <= maxm
	    && f.minn >= minn && f.maxn <= maxn)
	return;

    if (fontbits != NULL)
    {
	for (long i = maxn - minn; i >= 0; i--)
	    delete fontbits[i];
	delete fontbits;
    }

    if (f.maxm > maxm)
	maxm = f.maxm + 10;
    if (f.minm < minm)
	minm = f.minm - 10;
    if (f.maxn > maxn)
	maxn = f.maxn + 10;
    if (f.minn < minn)
	minn = f.minn - 10;

    debug(4, "newbits maxm=%ld minm=%ld  maxn=%ld minn=%ld",
	    maxm, minm, maxn, minn);

    // allocate the memory that we will need
    fontbits = new Bitvec *[maxn - minn + 2];
    for (long i = maxn - minn; i >= 0; i--)
	fontbits[i] = new Bitvec(maxm - minm + 2);
}


// define a font "fnum" from the file "fp" at the magnification "mag"
// - "fp" is a pointer into a DVI file right after the FNTDEF byte
// - this function will be called twice for each font - once in the
// postamble and once just before it is used (if it is used)
// 
void definefont(FILE *fp, long fnum, long mag)
{
    // find the font with this number in our fontlist
    long floc = getfontloc(fnum);

    // if already loaded, then we are not in the postamble anymore
    boolean loaded = (fontlist[floc] != NULL);

    debug(2, "fontnum = %ld(%ld)%s", fnum, floc, loaded ? " loaded" : "");

    // allocate space for this font, if necessary
    if (!loaded)
	fontlist[floc] = new font;

    font &f = *fontlist[floc].ptr;

    if (loaded)
    {
	// ignore the values in the file - we already have them
	(void)getsval(4, fp);
	(void)getsval(4, fp);
	(void)getsval(4, fp);
	skipbytes(getuval(1, fp) + getuval(1, fp), fp);
	return;
    }

    // initialize the font and start reading the values out of the file
    f.fp = NULL;
    f.use = 0;
    f.num = fnum;
    f.checksum = getsval(4, fp);
    f.scaledsize = getsval(4, fp);
    f.designsize = getsval(4, fp);
    debug(3, "checksum=%ld scaled=%ld design=%ld",
	    f.checksum, f.scaledsize, f.designsize);

    f.mag = (long)((double)mag * (double)f.scaledsize /
	                                (double)f.designsize + 0.5);

    // get the name of the font file - if the library name length
    // (liblen) is 0, then use the default library directory instead

    int liblen = (int)getuval(1, fp);
    int namelen = (int)getuval(1, fp);

    f.path = NULL;
    f.basename = new char[namelen + 1];
    int i;

    // read the optional font directory name from the DVI file
    if (liblen != 0)
    {
	// just use what the DVI file has
	char *s = f.path = new char[liblen + 1];
	for (i = 0; i < liblen; i++)
	    *s++ = (unsigned char)getuval(1, fp);
	*s = '\0';
    }

    // get the name of the font file to load
    char *s = f.basename;
    for (i = 0; i < namelen; i++)
	*s++ = (unsigned char)getuval(1, fp);
    *s = '\0';
    debug(4, "libname=%s  basename=%s", 
	    f.path == NULL ? "" : f.path, f.basename);

    // just defined - not yet downloaded - not setup
    f.setup = f.downloaded = f.onpage = f.toomany = FALSE;

    // no characters are downloaded either
    for (i = 0; i < MAXCHARS; i++)
	f.chr[i].downloaded = f.chr[i].charbig = FALSE;
}


// change to a new font "fnum" - this is the FNT DVI command
// 
void newfont(long fnum)
{
    long f = getfontloc(fnum);
    debug(5, "Change to font %ld(%ld)", fnum, f);

    currfont = fontlist[f];
    if (currfont == NULL)
	quit("Font %ld(%ld) has not been defined yet", fnum, f);
}


// move the device's cursor to the current H and V values
// 
void makecurrent(boolean force)
{
    long h = dev_sp2dev(H);
    long v = dev_sp2dev(V);

    // use the most efficient move that we can
    if (force || (devh != h && devv != v))
	dev_movehv(h, v);
    else if (devh != h)
	dev_moveh(h);
    else if (devv != v)
	dev_movev(v);

    debug(8, "lastH=%ld lastV=%ld  H=%ld V=%ld", devh, devv, h, v);

    // update the current cursor values
    devh = h;
    devv = v;
}


static void dumpescape(const char *s, long len)
{
    for (long i = 0; i < len; s++, i++)
    {
	if (*s != '\\' && *s != '`')
	{
	    putchar(*s);
	    continue;
	}
	s++;
	i++;
	if (i >= len)
	    break;

	if (isdigit(*s))
	{
	    int chr = *s - '0';
	    int base = 8;
	    if (*s == '0')
	    {
		s++;
		i++;
		switch (isupper(*s) ? tolower(*s) : *s)
		{
		Case 'b': 
		    base = 2;
		Case 'o': 
		    base = 8;
		Case 'd': 
		    base = 10;
		Case 'x': 
		    base = 16;
		Default: 
		    s--;
		    i--;
		}
	    }
	    s++;
	    i++;
	    for (; isxdigit(*s) && i < len; s++, i++)
	    {
		int d = (islower(*s) ? toupper(*s) : *s)
			- (isdigit(*s) ? '0' : 'A' - 10);
		if (d >= base)
		    break;
		chr = (chr * base) + d;
	    }
	    s--;
	    i--;
	    putchar(chr);
	    continue;
	}

	switch (isupper(*s) ? tolower(*s) : *s)
	{
	Case 'b': 
	    putchar('\b');
	Case 'f': 
	    putchar('\f');
	Case 'n': 
	    putchar('\n');
	Case 'r': 
	    putchar('\r');
	Case 't': 
	    putchar('\t');
	Case 'v': 
	    putchar('\v');
	Case '\\': 
	    putchar('\\');
	Case '`': 
	    putchar('`');
	Case 'e': 
	    putchar('\033');
	Case '?': 
	    putchar('\177');
	Case '^': 
	    s++;
	    i++;
	    putchar(*s == '?' ? '\177' : (*s & 037));
	Default: 
	    // putchar('\\');
	    putchar(*s);
	}
    }
}

// DVI XXX special command - "num" is the number of bytes in the DVI
//      file that are for the special command
// take the special string as a file name of a raw device-specific
// image file and dump it straight to the device
// 
void special(FILE *fp, long num)
{
    debug(2, "special length = %ld", num);

    // allocate a buffer that is large enough for this length
    static char *buf = NULL;
    static long blen = 0;
    if (num >= blen)
    {
	if (buf != NULL)
	    delete buf;
	buf = new char[blen = num + 1];
    }

    // read in the data from the DVI file into our buffer
    char *s = buf;
    for (long i = num; i-- > 0;)
	*s++ = (unsigned char)getuval(1, fp);
    *s = '\0';

    // we use the first word (up to a space) for identifying different
    // kinds of special files
    enum Specials
    {
	S_NONE = 0,
	S_RASTERFILE,
	S_ESCAPESEQ,
	S_RAWSTRING,
    } type;

    // handle special escape-sequences
    if (strncmp("escapeseq ", buf, 10) == 0)
    {
	s = buf + 10;
	num -= 10;
	type = S_ESCAPESEQ;
	debug(4, "Special == escapeseq = %s", s);
	mesg(" <<escapeseq");
	makecurrent();
	dev_push();
	dumpescape(s, num);
	dev_pop();
	mesg(">>");
	return;
    }

    // handle special raw-strings
    if (strncmp("rawstring ", buf, 10) == 0)
    {
	s = buf + 10;
	num -= 10;
	type = S_RAWSTRING;
	debug(4, "Special == rawstring = %s", s);
	mesg(" <<rawstring");
	makecurrent();
	dev_push();
	while (num-- > 0)
	    putchar(*s++);
	dev_pop();
	mesg(">>");
	return;
    }

    // handle raw file dumps - usually graphics data
    if (strncmp("rasterfile ", buf, 11) == 0)
    {
	s = buf + 11;
	type = S_RASTERFILE;
    }
    else if (strncmp("pcl:", buf, 4) == 0)
    {
	s = buf + 4;
	type = S_RASTERFILE;
    }
    else
	s = buf;

    debug(4, "Special == rasterfile = %s", s);
    mesg(" <%s", s);

    // now open this file and dump it to the device

    FILE *sp = fopenp(dviinput, s, F_READ);
    if (sp == NULL)
	quit("Cannot open \"%s\" for reading", s);

    // move the cursor, and save the current location in the device
    makecurrent();
    dev_push();

    // dump the file
    int l;
    char dat[1024];
    while ((l = fread(dat, sizeof *dat, sizeof dat, sp)) > 0)
	fwrite(dat, sizeof *dat, l, stdout);

    fclose(sp);

    // restore the cursor
    dev_pop();
    mesg(">");
}


// change the current font that is downloaded to the device if we have
// to - this is completely demand-driven - a font is never even read into
// memory much less downloaded unless a character from it is actually
// going to be printed
// 
static void changefont()
{
    // do we need to download this font?
    if (!currfont->downloaded)
    {
	mesg(" (");

	if (!currfont->setup)
	{
	    // load in the font info - we're really going to use it
	    setupfont(*currfont);
	    setupbits(*currfont);
	    currfont->setup = TRUE;
	}
	mesg("%s", currfont->basename);

	static int fontsdown = 0;
	currfont->use++;

	if (fontsdown < MAXLOADED)
	    fontsdown++;
	else
	{
	    // we need to delete a font from the device
	    register int i;
	    int fn = -1;
	    long u = MAXLONG;

	    // find the least recently used font
	    for (i = 0; i < fontlist.size(); i++)
		if (fontlist[i] != NULL && fontlist[i]->downloaded
			&& !fontlist[i]->onpage && fontlist[i]->use < u)
		{
		    fn = i;
		    u = fontlist[i]->use;
		}

	    // we need too many fonts - dump currfont in graphics mode
	    if (fn < 0)
	    {
		currfont->toomany = TRUE;
		mesg("#)");
		return;
	    }

	    // mark it as not downloaded, then delete it
	    font &f = *fontlist[fn].ptr;
	    for (i = 0; i < MAXCHARS; i++)
		f.chr[i].downloaded = FALSE;
	    f.toomany = f.downloaded = FALSE;
	    dev_delfont(f);
	}

	// downloaded this font
	dev_newfont(*currfont);
	currfont->downloaded = TRUE;

	mesg(")");
    }

    // select this font in the device
    dev_usefont(*currfont);
    devfont = currfont;

    // if this font is already being used on page, we are done
    if (currfont->onpage || currfont->toomany)
	return;
    currfont->onpage = TRUE;

    // are there too many fonts on this page?  if so, mark
    // the font as toomany to print it in graphics mode
    if (++fontsonpage > MAXONPAGE)
    {
	debug(3, "too many fonts on page - currfont marked");
	currfont->toomany = TRUE;
	mesg("#");
    }
}


// typeset a character - this is the DVI SET or PUT command - if "move"
// is TRUE, then the cursor will be moved right by the width of the
// character
//
void typeset(long ch, boolean move, double magval)
{
    fontchar & gch = currfont->chr[ch];

    // change the font if we have to
    if (currfont != devfont)
	changefont();

    // download the character if we have to - this also gets bigchars
    if (!gch.downloaded || gch.charbig || currfont->toomany)
	downchar((int)ch, currfont->toomany);

    // print the character if it is not too big
    if (!gch.charbig && !currfont->toomany)
    {
	makecurrent();
	int len;
	const char *dat = dev_char2dev((int)ch, len);
	fwrite(dat, sizeof *dat, len, stdout);
	devh += gch.dx >> 16;
    }

    if (move)
	moveright(gch.width * magval);
}


// DVI SETRULE or PUTRULE command - again, if "move" is TRUE, then the
// cursor is moved to the right, else it isn't
// 
void typerule(FILE *fp, boolean move, double magval)
{
    double a = Getsval(4, fp) * magval;
    double b = Getsval(4, fp) * magval;

    // let the device do the work (if it can)
    if (a > 0 && b > 0)
	dev_rule(a, b);

    if (move)
	moveright(b);
}
