------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                  S Y S T E M . S C A N _ I N T E G E R                   --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.1 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

function System.Scan_Integer
  (Str  : String;
   Ptr  : access Natural;
   Max  : Natural)
   return Integer
is
   type Utype is mod 2 ** Integer'Size;
   --  Type used to collect unsigned integer value. We use the corresponding
   --  unsigned type to ease overflow checking (we don't want to depend on
   --  constraint error for this checking).

   Maxint : constant := Integer'Last + 1;
   --  Largest unsigned Integer value, set to a value one greater than the
   --  largest positive number so that the largest negative number is handled.

   Max_10 : constant Utype := Maxint / 10;
   --  Largest integer that can be multiplied by 10 without exceeding Maxint

   Base   : Utype := 10;
   --  Reset for based number case

   Uval : Utype;
   --  Accumulated integer result

   Minus : Boolean := False;
   --  Set to True if result is negative

   function Scan_Int return Utype;
   --  Scans unsigned integer value starting at current character which is
   --  known to be a digit. The longest possible syntactically valid numeral
   --  is scanned out, and on return Ptr.all points past the last character.
   --  The scanned value is returned. Any value greater than Maxint is
   --  treated as an overflow value by the caller.

   function Scan_Int return Utype is
      Result : Utype;

   begin
      Result := Character'Pos (Str (Ptr.all)) - 48;

      loop
         Ptr.all := Ptr.all + 1;

         --  Return result if end of field

         if Ptr.all > Max then
            return Result;

         --  Non-digit encountered

         elsif Str (Ptr.all) not in '0' .. '9' then

            --  If syntactically valid underline, just skip it

            if Str (Ptr.all) = '_'
              and then Ptr.all < Max
              and then Str (Ptr.all + 1) in '0' .. '9'
            then
               null;

            --  If any other non-digit, return result

            else
               return Result;
            end if;

         --  Accumulate result unless we have overflow

         elsif Result <= Max_10 then
            Result := 10 * Result + Character'Pos (Str (Ptr.all)) - 48;

         --  If we have overflow, make sure result is out of range

         else
            Result := Maxint + 1;
         end if;
      end loop;
   end Scan_Int;

--  Start of processing for System.Scan_Integer

begin
   --  Scan past initial blanks

   while Str (Ptr.all) = ' ' loop
      Ptr.all := Ptr.all + 1;

      if Ptr.all > Max then
         raise Constraint_Error;
      end if;
   end loop;

   --  Remember an initial minus sign

   if Str (Ptr.all) = '-' then
      Minus := True;
      Ptr.all := Ptr.all + 1;

      if Ptr.all > Max then
         raise Constraint_Error;
      end if;
   end if;

   --  Scan initial unsigned integer

   if Str (Ptr.all) not in '0' .. '9' then
      raise Constraint_Error;
   end if;

   Uval := Scan_Int;

   --  Deal with based case

   if Ptr.all < Max
     and then (Str (Ptr.all) = ':'
                or else Str (Ptr.all) = '#')
     and then (Str (Ptr.all + 1) in '0' .. '9'
                or else Str (Ptr.all + 1) in 'A' .. 'F'
                or else Str (Ptr.all + 1) in 'a' .. 'f')
   then
      Base := Uval;

      declare
         Base_Char : constant Character := Str (Ptr.all);
         Base_Loc  : constant Natural   := Ptr.all;

         Max_B     : constant Utype := Maxint / Base;
         --  Maximum value that can be multiplied by Base without exceeding
         --  Maxint, used to check for approaching overflow in conversion.

         Digit     : Utype;

      begin
         Ptr.all := Ptr.all + 1;

         --  Initialize value to collect, checking base. In the following
         --  conversion circuit, we simply set Uval out of range if we
         --  encounter a bad digit or bad base value.

         if Base in 2 .. 16 then
            Uval := 0;
         else
            Uval := Maxint + 1;
         end if;

         --  Loop to scan out based integer value. If Error gets set to
         --  True, we continue the scan but don't collect any more digits

         loop
            --  If at end of string with no base character encountered then
            --  this isn't a based number at all, so reset to continue scan
            --  of original integer (the one we thought was a base).

            if Ptr.all > Max then
               Ptr.all := Base_Loc;
               Uval := Base;
               Base := 10;
               goto Exponent;
            end if;

            --  Process next character

            if Str (Ptr.all) in '0' .. '9' then
               Digit := Character'Pos (Str (Ptr.all)) - 48;

            elsif Str (Ptr.all) in 'A' .. 'F' then
               Digit := Character'Pos (Str (Ptr.all)) - 55;

            elsif Str (Ptr.all) in 'a' .. 'f' then
               Digit := Character'Pos (Str (Ptr.all)) - 87;

            --  Non-digit character

            else
               --  If terminating base character, we are done with loop

               if Str (Ptr.all) = Base_Char then
                  exit;

               --  Just ignore a valid underline

               elsif Str (Ptr.all) = '_'
                 and then Ptr.all < Max
                 and then (Str (Ptr.all) in '0' .. '9'
                            or else Str (Ptr.all) in 'A' .. 'F'
                            or else Str (Ptr.all) in 'a' .. 'f')
               then
                  goto Continue;

               --  Here is case where we have a non-valid based digit, so this
               --  isn't a based number after all, and we simply return the
               --  original value scanned as the base, resetting the pointer.
   
               else
                  Ptr.all := Base_Loc;
                  Uval := Base;
                  Base := 10;
                  goto Exponent;
               end if;
            end if;

            if Digit >= Base then
               Uval := Maxint + 1;
            end if;

            --  Here we accumulate the value, checking overflow

            if Uval < Max_B then
               Uval := Uval * Base + Digit;
            else
               Uval := Maxint + 1;
            end if;

            <<Continue>>
               Ptr.all := Ptr.all + 1;
         end loop;

         --  Based number successfully scanned out

         Ptr.all := Ptr.all + 1;
      end;
   end if;

   --  Come here with Base set and scanned integer value in Uval. Only
   --  remaining step is to deal with exponent if one is present.

   <<Exponent>>

      if Ptr.all < Max
        and then (Str (Ptr.all) = 'E' or else Str (Ptr.all) = 'e')
        and then Str (Ptr.all + 1) in '0' .. '9'
      then
         declare
            Expon  : Utype;
            Exploc : Natural := Ptr.all;

            Max_B  : constant Utype := Maxint / Base;
            --  Maximum value that can be multiplied by Base without exceeding
            --  Maxint, used to check for approaching overflow in conversion.

         begin
            Ptr.all := Ptr.all + 1;
            Expon := Scan_Int;

            if Expon > 64 then
               if Uval /= 0 then
                  Uval := Maxint + 1;
               end if;

            else
               for J in 1 .. Expon loop
                  if Uval <= Max_B then
                     Uval := Uval * Base;
                  else
                     Uval := Maxint + 1;
                  end if;
               end loop;
            end if;
         end;
      end if;

   --  Return result, dealing with sign

   if Uval < Maxint then
      if Minus then
         return -(Integer (Uval));
      else
         return Integer (Uval);
      end if;

   elsif Uval = Maxint and then Minus then
      return Integer'First;

   else
      raise Constraint_Error;
   end if;

end System.Scan_Integer;
