/* $Id: util.c,v 30000.23 1993/05/18 09:09:47 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/***********************************
 * Fugue utilities.                *
 *                                 *
 * Written by Greg Hudson and Ken  *
 * Keys.                           *
 *                                 *
 * Uppercase/lowercase table       *
 * Memory allocation routines      *
 * String handling routines        *
 * Mail checker                    *
 * Cleanup routine                 *
 ***********************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "output.h"
#include "macro.h"
#include "keyboard.h"
#include "socket.h"
#include "signal.h"

static char *mailfile = NULL;
static int mailerror = FALSE;
static int mail_mtime = 0;      /* mail modification time */

static int FDECL(cmatch,(char *s1, int c1));
static int FDECL(wmatch,(char *wlist, char **s2));

int    FDECL(smatch,(char *s1, char *s2));
int    FDECL(smatch_check,(CONST char *s1));
void   NDECL(init_table);
#ifndef HAVE_STRTOL
char   FDECL(strtochr,(char **sp));
#endif
#ifndef DMALLOC
Aline *FDECL(new_aline,(char *str, int attrs));
#else
Aline *FDECL(dnew_aline,(char *str, int attrs, char *file, int line));
#endif
void   FDECL(free_aline,(Aline *aline));
#ifndef HAVE_STRSTR
char  *FDECL(STRSTR,(CONST char *s1, CONST char *s2));
#endif
char  *FDECL(cstrchr,(CONST char *s, int c));
char  *FDECL(estrchr,(CONST char *s, int c, int e));
int    FDECL(cstrcmp,(CONST char *s, CONST char *t));
int    FDECL(cstrncmp,(CONST char *s, CONST char *t, int n));
int    FDECL(numarg,(char **str));
char  *FDECL(stripstr,(char *s));
void   FDECL(startopt,(char *args, char *opts));
char   FDECL(nextopt,(char **arg, int *num));
TOGGLER(change_maildelay);
TOGGLER(change_mailfile);
void   NDECL(check_mail);
void   NDECL(cleanup);
void   FDECL(die,(CONST char *why));

int mail_flag = 0;
char lowercase_values[128], uppercase_values[128];    /* for lcase(), ucase() */

/* A number of non-standard compilers don't properly handle tolower()
 * or toupper() called on a character that wouldn't ordinarily be
 * converted.  Fugue uses its own table (referred to by the ucase()
 * and lcase() macros in util.h) to ensure correct conversions.
 */
void init_table()
{
    int i;

    for (i = 0; i < 128; i++) {
        lowercase_values[i] = isupper(i) ? tolower(i) : i;
        uppercase_values[i] = islower(i) ? toupper(i) : i;
    }
}

#ifndef HAVE_STRTOL
char strtochr(sp)
    char **sp;
{
    int c;

    if (**sp != '0') {
        c = atoi(*sp);
        while (isdigit(*++(*sp)));
    } else if (lcase(*++(*sp)) == 'x') {
        for ((*sp)++, c = 0; isxdigit(**sp); (*sp)++)
            c = c * 16 + lcase(**sp) - (isdigit(**sp) ? '0' : ('a' - 10));
    } else {
        for (c = 0; isdigit(**sp); (*sp)++)
            c = c * 8 + **sp - '0';
    }
    return (char)(c % 128);
}
#endif


/* String handlers
 * Some of these are already present in most C libraries, but go by
 * different names or are not always there.  Since they're small, TF
 * simply uses its own routines with non-standard but consistant naming.
 * These are heavily used functions, so speed is favored over simplicity.
 */

/* case-insensitive strchr() */
char *cstrchr(s, c)
    register CONST char *s;
    register char c;
{
    for (c = lcase(c); *s; s++) if (lcase(*s) == c) return (char *)s;
    return (c) ? NULL : (char *)s;
}

/* c may be escaped by preceeding it with e */
char *estrchr(s, c, e)
    register CONST char *s;
    register char c, e;
{
    while (*s) {
        if (*s == c) return (char *)s;
        if (*s == e) {
            if (*++s) s++;
        } else s++;
    }
    return NULL;
}

#ifndef HAVE_STRSTR
char *STRSTR(s1, s2) 
    register CONST char *s1, *s2;
{
    int len;

    for (len = strlen(s2); s1 = strchr(s1, *s2); s1++) {
        if (strncmp(s1 + 1, s2 + 1, len - 1) == 0) return (char *)s1;
    }
    return NULL;
}
#endif

/* case-insensitive strcmp() */
int cstrcmp(s, t)
    register CONST char *s, *t;
{
    while (*s && lcase(*s) == lcase(*t)) s++, t++;
    return lcase(*s) - lcase(*t);
}

/* case-insensitive strncmp() */
int cstrncmp(s, t, n)
    register CONST char *s, *t;
    int n;
{
    while (n && *s && lcase(*s) == lcase(*t)) s++, t++, n--;
    return (n == 0) ? 0 : lcase(*s) - lcase(*t);
}

/* convert numeric argument.  *str will advance to beginning of next */
/* argument, or be NULL for failure. */
int numarg(str)
    char **str;
{
    char *start, *temp;
    for (start = temp = *str; isdigit(*temp); temp++);
    if (temp == *str) {
        *str = NULL;
        return 0;
    }
    for (*str = temp; isspace(**str); (*str)++);
    return atoi(start);
}

static int cmatch(s1, c1)
    char *s1;
    char c1;
{
    int result = FALSE;

    c1 = lcase(c1);
    if (*s1 == '^') {
        s1++;
        result = TRUE;
    }
    while (*s1) {
        if (*s1 == '\\') s1++;                   /* note that *(s1+1) != '\0' */
        if (*(s1 + 1) == '-' && *(s1 + 2)) {
            char c = *s1;
            char *end = s1 + 2;

            if (*end == '\\') end++;
            if (c > *end) {
                s1++;
                if (lcase(c) == c1) return result;
            } else {
                while (c <= *end) if (lcase(c++) == c1) return result;
                s1 = end + 1;
            }
        } else if (lcase(*s1++) == c1) return result;
    }
    return !result;
}

static int wmatch(wlist, s2)
    char *wlist;       /* word list                      */
    char **s2;         /* buffer to match from           */
{
    char *matchstr,    /* which word to find             */
         *strend,      /* end of current word from wlist */
         *matchbuf,    /* where to find from             */
         *bufend;      /* end of match buffer            */
    int  result = 1;   /* intermediate result            */

    if (!wlist || !*s2) return 1;
    matchbuf = *s2;
    matchstr = wlist;
    if ((bufend = strchr(matchbuf, ' ')) == NULL)
        *s2 += strlen(*s2);
    else
        *(*s2 = bufend) = '\0';
    do {
        if ((strend = estrchr(matchstr, '|', '\\')) != NULL)
            *strend = '\0';
        result = smatch(matchstr, matchbuf);
        if (strend != NULL) *strend++ = '|';
    } while (result && (matchstr = strend) != NULL);
    if (bufend != NULL) *bufend = ' ';
    return result;
}

/* smatch_check() should be used on s1 to check pattern syntax before
 * calling smatch().
 */

int smatch(s1, s2)
    char *s1, *s2;
{
    char ch;
    char *start = s2;

    while (*s1) {
        switch (*s1) {
        case '\\':
            s1++;
            if (lcase(*s1++) != lcase(*s2++)) return 1;
            break;
        case '?':
            if (!*s2++) return 1;
            s1++;
            break;
        case '*':
            while (*s1 == '*' || (*s1 == '?' && *s2++)) s1++;
            if (*s1 == '?') return 1;
            if (*s1 == '{') {
                if (s2 == start || *(s2 - 1) == ' ')
                    if (!smatch(s1, s2)) return 0;
                while ((s2 = strchr(s2, ' ')) != NULL)
                    if (!smatch(s1, ++s2)) return 0;
                return 1;
            } else if (*s1 == '[') {
                while (*s2) if (!smatch(s1, s2++)) return 0;
                return 1;
            }
            ch = (*s1 == '\\' && *(s1 + 1)) ? *(s1 + 1) : *s1;
            while ((s2 = cstrchr(s2, ch)) != NULL) {
                if (!smatch(s1, s2++)) return 0;
            }
            return 1;
        case '[':
            {
                char *end;

                if (!(end = estrchr(s1, ']', '\\'))) {     /* can't happen if */
                    tfputs("% smatch: unmatched '['", tferr); /* smatch_check */
                    return 1;                                /* is used first */
                }
                *end = '\0';
                if (cmatch(s1 + 1, *s2++)) {
                    *end = ']';
                    return 1;
                }
                *end = ']';
                s1 = end + 1;
            }
            break;
        case '{':
            if (s2 != start && !isspace(*(s2 - 1))) return 1;
            {
                char *end;

                if (!(end = estrchr(s1, '}', '\\'))) {      /* can't happen if*/
                    tfputs("% smatch: unmatched '{'", tferr); /* smatch_check */
                    return 1;                                /* is used first */
                }
                *end = '\0';
                if (wmatch(s1 + 1, &s2)) {
                    *end = '}';
                    return 1;
                }
                *end = '}';
                s1 = end + 1;
            }
            break;
        default:
            if(lcase(*s1++) != lcase(*s2++)) return 1;
            break;
        }
    }
    return lcase(*s1) - lcase(*s2);
}

/* verify syntax of smatch pattern */
int smatch_check(s1)
    CONST char *s1;
{
    int inword = FALSE;

    while (*s1) {
        switch (*s1) {
        case '\\':
            if (*++s1) s1++;
            break;
        case '[':
            if (!(s1 = estrchr(s1, ']', '\\'))) {
                tfputs("% pattern error: unmatched '['", tferr);
                return 0;
            }
            s1++;
            break;
        case '{':
            if (inword) {
                tfputs("% pattern error: nested '{'", tferr);
                return 0;
            }
            inword = TRUE;
            s1++;
            break;
        case '}':
            inword = FALSE;
            s1++;
            break;
        case '?':
        case '*':
        default:
            s1++;
            break;
        }
    }
    if (inword) tfputs("% pattern error: unmatched '{'", tferr);
    return !inword;
}

/* remove leading and trailing spaces */
char *stripstr(s)
    char *s;
{
    char *start, *end;

    if (!*s) return s;
    for (start = s; isspace(*start); start++);
    if (*start) {
        for (end = start + strlen(start) - 1; isspace(*end); end--);
        *++end = '\0';
    } else end = start;
    if (start != s)
        while ((*s++ = *start++));    /* strcpy may not be safe */
    return s;
}


/* General command option parser

   startopt should be called before nextopt.  args is the argument list
   to be parsed, opts is a string containing valid options.  Options which
   take string arguments should be followed by a ':'; options which take
   numeric argumens should be followed by a '#'.  String arguments may be
   omitted.

   nextopt returns the next option character.  If option takes a string
   argument, a pointer to it is returned in *arg; an integer argument
   is returned in *num.  If end of options is reached, nextopt returns
   '\0', and *arg points to remainder of argument list.  End of options
   is marked by '\0', '=', '-' by itself, or a word not beggining with
   '-'.  If an invalid option is encountered, an error message is
   printed and '?' is returned.

   Option Syntax Rules:
      All options must be preceded by '-'.
      Options may be grouped after a single '-'.
      There must be no space between an option and its argument.
      String option-arguments may be quoted.  Quotes in the arg must be escaped.
      All options must precede operands.
      A '--' or '-' with no option may be used to mark the end of the options.
      The relative order of the options should not matter (not enforced).
*/

static char *argp, *options;
static int inword;

void startopt(args, opts)
    char *args, *opts;
{
    argp = args;
    options = opts;
    inword = 0;
}

char nextopt(arg, num)
    char **arg;
    int *num;
{
    short error = FALSE;
    char *q, opt;
    STATIC_BUFFER(buffer)

    if (!inword) {
        while (isspace(*argp)) argp++;
        if (strcmp(argp, "--") == 0 || strncmp(argp, "-- ", 3) == 0) {
            for (*arg = argp + 2; isspace(**arg); ++*arg);
            return '\0';
        } else if (*argp != '-' || !*++argp || isspace(*argp)) {
            for (*arg = argp; isspace(**arg); ++*arg);
            return '\0';
        }
    } else if (*argp == '=') {        /* '=' marks end, & is part of parms */
        *arg = argp;                  /*... for stuff like  /def -t"foo"=bar */
        return '\0';
    }
    opt = *argp;
    if (opt == ':' || opt == '#') error = TRUE;
    else if ((q = strchr(options, opt)) != NULL) ;
    else if (isdigit(opt) && (q = strchr(options, '0'))) ;
    else error = TRUE;
    if (error) {
        tfprintf(tferr, "%% invalid option: %c", opt);
        return '?';
    }
    if (*q == '0') {
        *num = atoi(argp);
        while (isdigit(*++argp));
        return '0';
    } else if (*++q == ':') {
        Stringterm(buffer, 0);
        if (*++argp == '"') {
            for (argp++; *argp && *argp != '"'; Stringadd(buffer, *argp++))
                if (*argp == '\\' && (argp[1] == '"' || argp[1] == '\\'))
                    argp++;
            if (!*argp) {
                tfprintf(tferr, "%% unmatched \" in %c option", opt);
                return '?';
            } else argp++;
        } else while (*argp && !isspace(*argp)) Stringadd(buffer, *argp++);
        *arg = buffer->s;
    } else if (*q == '#') {
        argp++;
        if (!isdigit(*argp)) {
            tfprintf(tferr, "%% %c option requires numeric argument", opt);
            return '?';
        }
        *num = atoi(argp);
        while (isdigit(*++argp));
    } else argp++;
    inword = (*argp && !isspace(*argp));
    return opt;
}

#ifndef DMALLOC
Aline *new_aline(str, attrs)
    char *str;
    int attrs;
#else
Aline *dnew_aline(str, attrs, file, line)
    char *str;
    char *file;
    int attrs, line;
#endif
{
    Aline *aline;

#ifndef DMALLOC
    aline = (Aline *)MALLOC(sizeof(Aline));
#else
    aline = (Aline *)dmalloc(sizeof(Aline), file, line);
#endif
    aline->str = str ? STRDUP(str) : NULL;
    aline->attrs = attrs;
    aline->links = 0;
    aline->time = (TIME_T)time(0);
    return aline;
}

void free_aline(aline)
    Aline *aline;
{
    if (!--(aline->links)) {
        FREE(aline->str);
        FREE(aline);
    }
}

void change_maildelay()
{
    extern int mailtime;
    mailtime = 0;
}

void change_mailfile()
{
    mailfile = getvar("MAIL");
    mailerror = FALSE;
    mail_mtime = 0;
    change_maildelay();
}

void check_mail()
{
    struct stat buf;
    STATIC_BUFFER(path);

    if (mailerror || maildelay <= 0) return;
    if (!mailfile && !(mailfile = getvar("MAIL"))) {
        if ((mailfile = getvar("LOGNAME")) || (mailfile = getvar("USER"))) {
            Stringcat(Stringcpy(path, MAILDIR), mailfile);
            setvar("MAIL", path->s, FALSE);
            mailfile = getvar("MAIL");
        } else {
            tfputs("% Warning:  Can't figure out name of mail file.", tferr);
            mailerror = TRUE;
            return;
        }
    }
    if (!*mailfile) return;

    if (stat(mailfile, &buf) < 0) {
        if (mail_flag) put_mail(mail_flag = 0);
        mail_mtime = 0;
    } else {
        if (buf.st_size == 0 || buf.st_mtime <= buf.st_atime) {
            if (mail_flag) put_mail(mail_flag = 0);
        } else {
            if (!mail_flag) put_mail(mail_flag = 1);
            if (buf.st_mtime > mail_mtime)
                do_hook(H_MAIL, "%% You have new mail in %s", "%s", mailfile);
        }
        mail_mtime = buf.st_mtime;
    }
}

/* Cleanup and error routines. */
void cleanup()
{
    extern int screen_setup;

    reset_tty();
    disconnect_all();
    if (screen_setup) fix_screen();
}

void die(why)
    CONST char *why;
{
    cleanup();
    puts(why);
    exit(1);
}

