/* attrib.c */

#include <ctype.h>
#include <string.h>

#include "config.h"
#include "db.h"
#include "externs.h"
#include "interface.h"
#include "oldattrib.h"
	
#ifdef MEM_CHECK
#include "mem_check.h"
#endif

extern dbref match_thing();	/* for LATTR */

extern ATTR *aname_hash_lookup(); /* for attrib hash stuff, in atr_tab.c */

/* hash stuff. Based on K&R */
struct atr_entry {
  struct atr_entry *next;
  char *name;
};

static struct atr_entry *atr_hashtab[ATR_HASH_SIZE];

/*------------------------------------------------------------------------
 * Various attribute stuff
 */


char *clean_atr_name(s)
    char *s;
{
    static char buf[BUFFER_LEN];
    char *q = buf;
    char *a;

    if(!*s || !s) {
      sprintf(buf, "NULL");
      return buf;
    }

    if(!strcasecmp("KILL", s)) {
      sprintf(buf, "DEATH");
      return buf;
    }

    if(!strcasecmp("KILL", s+1) &&
       (*s == 'o' || *s == 'O' || *s == 'a' || *s == 'A')) {
      sprintf(buf, "%c%s", *s, s+1);
      return buf;
    }

    for(a = s; *a; a++)
      if(isprint(*a) && !isspace(*a))
	*q++ = *a;
    *q = '\0';
    return buf;
}

ATTR *atr_str(s)
    char *s;
{
  ATTR *result;
  ATTR *a;
  char *q;

  result = (ATTR *)malloc(sizeof (ATTR));
  q = clean_atr_name(s);
  result->name = (char *) malloc(strlen(q)+1);
#ifdef MEM_CHECK
  add_check("attribute");
  add_check("attribute_name");
#endif
  strcpy(result->name, strupper(q));
  result->flags = AF_ODARK;

  a = atr_match(result->name);
  if (a)
    result->flags = a->flags;

  return result;
}

struct boolatr *alloc_atr(name, s)
    char *name;
    char *s;
{
  struct boolatr *a;
  const char *p;

  a = (struct boolatr *)malloc(sizeof (struct boolatr));
  a->name = (char *)malloc(strlen(name)+1);
  strcpy(a->name,name);
  p = compress(s);
  a->text = (char *)malloc(strlen(p)+1);
  strcpy(a->text, p);
#ifdef MEM_CHECK
  add_check("bool_atr");
  add_check("bool_atr_name");
  add_check("bool_atr_val");
#endif
  return a;
}

void atr_clr(thing, atr)
    dbref thing;
    char *atr;
{
  ALIST *ptr = db[thing].list;

  while(ptr) {
    if(!strcasecmp(atr, AL_NAME(ptr))) {
      AL_DISPOSE(ptr);
      return;
    }
    ptr = AL_NEXT(ptr);
  }
}

ALIST *AL_MAKE(type, next, string, owner, flags)
    char *type;
    ALIST *next;
    char *string;
    dbref owner;
    dbref flags;
{
    ALIST *ptr;
    const char *p;

    ptr = (ALIST *)malloc(sizeof(ALIST));
    AL_ATTR(ptr) = atr_str(type);
    AL_CREATOR(ptr) = owner;

    p = compress(string);
    AL_STR(ptr) = (char *)malloc(strlen(p) + 1);
#ifdef MEM_CHECK
    add_check("ALIST");
    add_check("attribute_value");
#endif
    strcpy(AL_STR(ptr), p);

    if (flags != NOTHING)
	AL_FLAGS(ptr) |= flags;

    AL_NEXT(ptr) = next;
    return ptr;
}

void atr_new_add(thing, atr, s, player, flags)
    dbref thing;
    char *atr;
    char *s;
    dbref player;
    dbref flags;
{
    ALIST *ptr;

    /* s = (char *) compress(s); */        /* This is redundant. */

    ptr = AL_MAKE(atr, db[thing].list, s, player, flags);

    /* Now we have to hack around the auto-ODARK'ing of attributes. */
    if (!(flags & AF_ODARK))
	AL_FLAGS(ptr) &= ~AF_ODARK;

    db[thing].list = ptr;
}

int atr_add(thing, atr, s, player, flags)
    dbref thing;
    char *atr;
    char *s;
    dbref player;
    dbref flags;
{
  ALIST *ptr;
  dbref privs;

  privs = Owner(player);

  if (!s) 
    s = (char *)"";

  /* walk attribute list until we find a match */
  for (ptr = db[thing].list;
       (ptr && strcmp(atr, AL_NAME(ptr)));
       ptr = AL_NEXT(ptr))
    ;

  if (!*s) {
    /* attribute deletion */

    if (ptr) {
      /* found it. Wipe it out, unless it's locked. */
      if (!Can_Write_Attr(privs, thing, AL_ATTR(ptr))) {
	return -1;
      } else {
        AL_DISPOSE(ptr);
        return 1;
      }
    } else {
      /* doesn't exist, so just forget it. */
      return 0;
    }
  }

  /* if we reach this point, we are changing the text of an attribute */

  if (!ptr) {
    /* add a brand new attribute */
    db[thing].list = AL_MAKE(atr, db[thing].list, s, privs, flags);
    return 1;
  }

  /* modify an existing attribute */

  if (!Can_Write_Attr(privs, thing, AL_ATTR(ptr))) {
    /* it's locked. Too bad. */
    return -1;
  } 

  /* we can do it. Free up old string and replace it with new one. */

  free((char *) AL_STR(ptr));
#ifdef MEM_CHECK
  del_check("attribute_value");
#endif

  AL_STR(ptr) = (char *) strdup(compress(s));
#ifdef MEM_CHECK
  add_check("attribute_value");
#endif

  AL_CREATOR(ptr) = privs;
  if (flags != NOTHING)
    AL_FLAGS(ptr) = flags;
  if (AL_BAD(ptr))
    AL_FLAGS(ptr) &= ~AF_NUKED;
  return 1;
}


ATTR *atr_get_partname(thing, atr)
     dbref thing;
     char *atr;
{
  /* check prefix matches */

  ALIST *ptr;

  if(thing == NOTHING || !atr) return NULL;

  for(ptr = db[thing].list; ptr; ptr = AL_NEXT(ptr)) {
    if(!AL_BAD(ptr) && string_prefix(AL_NAME(ptr), atr)) {
      return(AL_ATTR(ptr));
    }
  }
  return (ATTR *)NULL;
}

ATTR *atr_get_aliasname(thing, atr)
     dbref thing;
     char *atr;
{
  /* check partial matches by alias */

  ALIST *ptr;
  ATTR *ap;

  if(thing == NOTHING || !atr) return NULL;

  /* find the real name */
  ap = aname_hash_lookup(strupper(atr));
  if (!ap)
    return (ATTR *)NULL;

  for (ptr = db[thing].list; ptr; ptr = AL_NEXT(ptr)) {
    if (!AL_BAD(ptr) && !strcmp(AL_NAME(ptr), ap->name)) {
      return AL_ATTR(ptr);
    }
  }
  return (ATTR *)NULL;
}

ATTR *atr_get_fullname(thing, atr)
     dbref thing;
     char *atr;
{
  /* check exact matches only */

  ALIST *ptr;

  if(thing == NOTHING || !atr) return NULL;

  for(ptr = db[thing].list; ptr; ptr = AL_NEXT(ptr)) {
    if(!AL_BAD(ptr) && !strcmp(AL_NAME(ptr), atr)) {
      return AL_ATTR(ptr);
    }
  }
  return (ATTR *)NULL;
}


ATTR *atr_get(thing, atr)
     dbref thing;
     char *atr;
{
    ATTR *attrib;
    int parent_depth = 0;	        /* excessive recursion prevention */
    dbref temp;

    if ((thing == NOTHING) || !atr) 
	return NULL;

    /* Check the full name first */
    for (parent_depth = 0, temp = thing;
	 (temp != NOTHING) && (parent_depth < MAX_PARENTS);
	 parent_depth++, temp = Parent(temp)) {
	if ((attrib = atr_get_fullname(temp, atr)) != NULL) {
	    if (!(attrib->flags & AF_PRIVATE) || (temp == thing))
		return (ATTR *)attrib;          /* Got it. */
	}
    }

    /* If we reach here, we haven't got anything. Try an alias match. */
    for (parent_depth = 0, temp = thing;
	 (temp != NOTHING) && (parent_depth < MAX_PARENTS);
	 parent_depth++, temp = Parent(temp)) {
	if ((attrib = atr_get_aliasname(temp, atr)) != NULL) {
	    if (!(attrib->flags & AF_PRIVATE) || (temp == thing))
		return (ATTR *)attrib;
	}
    }

    return (ATTR *) NULL;		/* Didn't find anything. */
}

ATTR *atr_get_noparent(thing, atr)
    dbref thing;
    char *atr;
{
  ATTR *attrib;

  if(thing == NOTHING || !atr) return NULL;

  attrib = atr_get_fullname(thing, atr);
  if (!attrib)
    attrib = atr_get_aliasname(thing, atr);
  return (ATTR *)attrib;
}

void free_attrib(thisattr)
    ATTR *thisattr;
{
  if (thisattr) {
    if (thisattr->name)
      free((char *) thisattr->name);
    if (thisattr->value)
      free((char *) thisattr->value);
    free((char *) thisattr);
  }

#ifdef MEM_CHECK
  del_check("attribute_name");
  del_check("attribute_value");
  del_check("attribute");
#endif
}

void atr_free(thing)
    dbref thing;
{
  ALIST *ptr, *next;

  for(ptr = db[thing].list; ptr; ptr = next) {
    next = AL_NEXT(ptr);
    free_attrib(AL_ATTR(ptr));
    free((char *) ptr);
#ifdef MEM_CHECK
    del_check("ALIST");
#endif
  }
  db[thing].list = (ALIST *)NULL;
}

/* reconstruct an attribute list */
void atr_collect(thing)
    dbref thing;
{
  ALIST *ptr, *next;

  ptr = db[thing].list;
  db[thing].list = NULL;

  while(ptr) {
    if(!AL_BAD(ptr)) {
      db[thing].list = AL_MAKE(AL_NAME(ptr), db[thing].list,
			       AL_STR(ptr), AL_CREATOR(ptr), AL_FLAGS(ptr));
    }
    next = AL_NEXT(ptr);
    free_attrib(AL_ATTR(ptr));
    free((char *) ptr);
#ifdef MEM_CHECK
    del_check("ALIST");
#endif
    ptr = next;
  }
}

void atr_cpy(dest, source)
    dbref dest, source;
{
  ALIST *ptr;
  ptr = db[source].list;

  db[dest].list = NULL;
  while (ptr) {
    if (!AL_BAD(ptr) && !(AL_FLAGS(ptr) & AF_NOCOPY)) {
      db[dest].list = AL_MAKE(AL_NAME(ptr), db[dest].list,
			      AL_STR(ptr), AL_CREATOR(ptr), AL_FLAGS(ptr));
    }
    ptr = AL_NEXT(ptr);
  }
}

const char *convert_atr(oldatr)
    dbref oldatr;
{
  static const char result[MAX_COMMAND_LEN];
  int factor = 0;

  switch(oldatr) {
    case A_OSUCC:
      return "OSUCCESS";
    case A_OFAIL:
      return "OFAILURE";
    case A_FAIL:
      return "FAILURE";
    case A_SUCC:
      return "SUCCESS";
    case A_PASS:
      return "XYXXY";
    case A_DESC:
      return "DESCRIBE";
    case A_SEX:
      return "SEX";
    case A_ODROP:
      return "ODROP";
    case A_DROP:
      return "DROP";
    case A_OKILL:
      return "OKILL";
    case A_KILL:
      return "KILL";
    case A_ASUCC:
      return "ASUCCESS";
    case A_AFAIL:
      return "AFAILURE";
    case A_ADROP:
      return "ADROP";
    case A_AKILL:
      return "AKILL";
    case A_USE:
      return "DOES";
    case A_CHARGES:
      return "CHARGES";
    case A_RUNOUT:
      return "RUNOUT";
    case A_STARTUP:
      return "STARTUP";
    case A_ACLONE:
      return "ACLONE";
    case A_APAY:
      return "APAYMENT";
    case A_OPAY:
      return "OPAYMENT";
    case A_PAY:
      return "PAYMENT";
    case A_COST:
      return "COST";
    case A_RAND:
      return "RAND";
    case A_LISTEN:
      return "LISTEN";
    case A_AAHEAR:
      return "AAHEAR";
    case A_AMHEAR:
      return "AMHEAR";
    case A_AHEAR:
      return "AHEAR";
    case A_LAST:
      return "LAST";
    case A_QUEUE:
      return "QUEUE";
    case A_IDESC:
      return "IDESCRIBE";
    case A_ENTER:
      return "ENTER";
    case A_OXENTER:
      return "OXENTER";
    case A_AENTER:
      return "AENTER";
    case A_ADESC:
      return "ADESCRIBE";
    case A_ODESC:
      return "ODESCRIBE";
    case A_RQUOTA:
      return "RQUOTA";
    case A_ACONNECT:
      return "ACONNECT";
    case A_ADISCONNECT:
      return "ADISCONNECT";
    case A_LEAVE:
      return "LEAVE";
    case A_ALEAVE:
      return "ALEAVE";
    case A_OLEAVE:
      return "OLEAVE";
    case A_OENTER:
      return "OENTER";
    case A_OXLEAVE:
      return "OXLEAVE";
    default:
      if(oldatr >= 100 && oldatr < 126)
	factor = 0;
      else if(oldatr >= 126 && oldatr < 152)
	factor = 1;
      else if(oldatr >= 152 && oldatr < 178)
	factor = 2;
      else {
	fprintf(stderr,
		"ERROR: Invalid attribute number in convert_atr. aborting.\n");
	fflush(stderr);
	abort();
      }
      sprintf((char *)result, "%c%c",
	      'V'+factor, oldatr - (100 + (factor * 26)) + 'A');
      return result;
  }
  /*NOTREACHED*/
  return "";
}

ATTR *atr_match(string)
  char *string;
{
    /* this function expects an upper-cased string! */

    return ((ATTR *) aname_hash_lookup(string));
}

/*---------------------------------------------------------------------------
 * Command checking for $ and ^, and hash stuff for attrib inheritance checks
 */

struct atr_entry *atr_hash_lookup(s)
     char *s;
{
  struct atr_entry *ap;

  for (ap = atr_hashtab[hash_fn(s, ATR_HASH_MASK)];
       ap != NULL; ap = ap->next)
    if (!strcmp(s, ap->name))
      return ap;
  return NULL;			/* not found */
}

struct atr_entry *atr_hash_insert(name)
     char *name;
{
  struct atr_entry *ap;
  unsigned hashval;

  /* can assume that it's not going to be inserted unless it doesn't
   * exist, since atr_hash_lookup is called before.
   */

  ap = (struct atr_entry *) malloc(sizeof(struct atr_entry));
  if ((ap == NULL) || ((ap->name = (char *) strdup(name)) == NULL))
    return NULL;
  hashval = hash_fn(name, ATR_HASH_MASK);
  ap->next = atr_hashtab[hashval];
  atr_hashtab[hashval] = ap;
  return ap;
}

void atr_hash_free()
{
  struct atr_entry *ap;
  struct atr_entry *temp;
  int i;

  temp = NULL;

  for (i = 0; i < ATR_HASH_SIZE; i++) {
    for (ap = atr_hashtab[i]; ap != NULL; ap = temp) {
      temp = ap->next;
      free(ap->name);
      free(ap);
    }
    atr_hashtab[i] = NULL;
  }
}
      
int atr_comm_match(thing, player, type, end, str, just_match)
    dbref thing, player;
    char type, end;
    char *str;
   int just_match;		/* if 1, just give match */
{
  ALIST *ptr;
  int match = 0;
  char tbuf1[BUFFER_LEN];
  char *s;
  dbref parent;
  int rec = 0;			/* prevent recursion */
  int ok = 1;			/* uselock check */

  if (thing < 0 || thing >= db_top)
    return 0;

  /* if the NO_COMMAND flag is set, we can skip this object if we're
   * looking for $ (there's no reason for NO_COMMAND to block ^-patterns,
   * since setting an object !LISTENER already does that). Also,
   * if it's HALT, it can't execute commands anyway, so don't bother
   * checking. 
   */
  if (Halted(thing) || (NoCommand(thing) && (type == '$')))
    return 0;

  /* only $ checks should look at the parent and do a uselock check */
  if (type == '$') {
    parent = Parent(thing);
    ok = eval_boolexp(player, db[thing].usekey, thing, 0, USELOCK);
  } else
    parent = NOTHING;
  
  /* the uselock on the child is the only thing which matters.
   * if it's locked, we don't need to bother checking for commands;
   * silently fail.
   */
  if (!eval_boolexp(player, db[thing].usekey, thing, 0, USELOCK))
    return 0;

  /* check object itself */
  for(ptr = db[thing].list; ptr; ptr = AL_NEXT(ptr)) {
    if (!AL_BAD(ptr) && !(AL_FLAGS(ptr) & AF_NOPROG)) {

      /* first record that the attribute has been read */
      if ((type == '$') && (parent != NOTHING))
	atr_hash_insert(AL_NAME(ptr));

      /* we move the check for '$' as first char to here because we should
       * insert into the hash table no matter what's in the attribute.
       */
      if (*AL_STR(ptr) == type) {
	strcpy(tbuf1, uncompress(AL_STR(ptr)));
	for(s = tbuf1 + 1; *s && (*s != end); s++);
	if(!*s)
	  continue;
	*s++ = '\0';
	if(wild_match(tbuf1 + 1, str)) {
	  match++;
	  if (!just_match)
	    parse_que(thing, s, player);
	}
      }
    }
  }

  /* now check parent */
  if (parent == NOTHING)
    return match;

  for (rec = 0; (parent != NOTHING) && (rec < MAX_PARENTS);
       rec++, parent = Parent(parent)) {
    for(ptr = db[parent].list; ptr; ptr = AL_NEXT(ptr)) {
      if (!AL_BAD(ptr) && !(AL_FLAGS(ptr) & AF_NOPROG) && 
	  !(AL_FLAGS(ptr) & AF_PRIVATE)) {
	/* first check to see if attribute has been entered before. If
	 * not, enter it into the hash table and check it for a command.
	 */
	if (atr_hash_lookup(AL_NAME(ptr)) == NULL) {

	  if (Parent(parent) != NOTHING)
	    atr_hash_insert(AL_NAME(ptr));

	  if (*AL_STR(ptr) == type) {
	    strcpy(tbuf1, uncompress(AL_STR(ptr)));
	    for(s = tbuf1 + 1; *s && (*s != end); s++)
	      ;
	    if(!*s)
	      continue;
	    *s++ = '\0';
	    if(wild_match(tbuf1 + 1, str)) {
	      match++;
	      if (!just_match)
		parse_que(thing, s, player);
	    }
	  }

	}
      }
    }
  }
  if (rec > MAX_PARENTS)
      do_log(LT_ERR, 0, 0, "atr_comm_match recursion on parented object #%d",
	     thing);
  atr_hash_free();
  return match;
}

/*--------------------------------------------------------------------------
 * Other stuff
 */

void do_atrlock(player, arg1, arg2)
    dbref player;
    const char *arg1, *arg2;
{
  dbref thing;
  char *p;
  ALIST *ptr;
  int status;

  if(!arg2 || !*arg2)
    status = 0;
  else {
    if(!strcasecmp(arg2, "on")) {
      status = 1;
    } else if(!strcasecmp(arg2, "off")) {
      status = 2;
    } else
     status = 0;
  }

  if(!arg1 || !*arg1) {
    notify(player, "You need to give an object/attribute pair.");
    return;
  }

  if(!(p = (char *) index(arg1, '/')) || !(*(p+1))) {
    notify(player, "You need to give an object/attribute pair.");
    return;
  }
  *p++ = '\0';

  init_match(player, arg1, NOTYPE);
  match_everything();
  if((thing = noisy_match_result()) == NOTHING)
    return;

  for(ptr = db[thing].list; ptr; ptr = AL_NEXT(ptr))
    if(!AL_BAD(ptr) && !strcasecmp(AL_NAME(ptr), p))
      break;

  if(!ptr)
    for(ptr = db[thing].list; ptr; ptr = AL_NEXT(ptr))
      if(!AL_BAD(ptr) && string_prefix(AL_NAME(ptr), p))
	break;

  if(ptr) {
    if(!status) {
      notify(player, tprintf("That attribute is %slocked.",
			     (AL_FLAGS(ptr) & AF_LOCKED) ? "" : "un"));
      return;
    } else if (!Wizard(player) &&
	      (Owner(AL_CREATOR(ptr)) != Owner(player))) {
      notify(player, "You need to own the attribute to change its lock.");
      return;
    } else {
      if(status == 1) {
        AL_FLAGS(ptr) |= AF_LOCKED;
        notify(player, "Attribute locked.");
	return;
      } else if(status == 2) {
	AL_FLAGS(ptr) &= ~AF_LOCKED;
	notify(player, "Attribute unlocked.");
	return;
      } else {
	notify(player, "Invalid status on atrlock.. Notify god.");
	return;
      }
    }
  } else
    notify(player, "No such attribute.");
  return;
}

void do_atrchown(player, arg1, arg2)
    dbref player;
    const char *arg1, *arg2;
{
  dbref thing, new_owner;
  char *p;
  ALIST *ptr;

  if(!arg1 || !*arg1) {
    notify(player, "You need to give an object/attribute pair.");
    return; 
  }

  if(!(p = (char *) index(arg1, '/')) || !(*(p+1))) {
    notify(player, "You need to give an object/attribute pair.");
    return;
  }
  *p++ = '\0';

  init_match(player, arg1, NOTYPE);
  match_everything();
  if((thing = noisy_match_result()) == NOTHING)
    return;

  if((!arg2 && !*arg2) || !strcasecmp(arg2, "me"))
    new_owner = player;
  else
    new_owner = lookup_player(arg2);

  if(new_owner == NOTHING) {
    notify(player, "I can't find that player");
    return;
  }

  for(ptr = db[thing].list; ptr; ptr = AL_NEXT(ptr))
    if(!AL_BAD(ptr) && !strcasecmp(AL_NAME(ptr), p))
      break;

  if(!ptr)
    for(ptr = db[thing].list; ptr; ptr = AL_NEXT(ptr))
      if(!AL_BAD(ptr) && string_prefix(AL_NAME(ptr), p))
	break;

  if(ptr) {
    if ((controls(player, thing) && !(AL_FLAGS(ptr) & AF_LOCKED)) ||
	(Owner(player) == Owner(AL_CREATOR(ptr)))) {
      if(new_owner != Owner(thing) && !Wizard(player)) {
	notify(player,
         "You can only chown an attribute to the current owner of the object.");
	return;
      }
      AL_CREATOR(ptr) = Owner(new_owner);
      notify(player, "Attribute owner changed.");
      return;
    } else {
      notify(player, "You don't have the permission to chown that.");
      return;
    }
  } else
    notify(player, "No such attribute.");
}

ATTR *atr_complete_match(player, atr, privs)
    dbref player;
    char *atr;
    dbref privs;
{
  /* return a pointer to attribute with name atr on player, provided
   * that privs can see it. If not, then we check every object that
   * player is carrying, for the same thing.
   */

  ATTR *a;
  dbref thing;
  char *name;

  name = strupper(atr);

  if ((a = atr_get(player, name)) != NULL) {
    if (Can_Read_Attr(privs, player, a))
      return a;
    else {
      DOLIST(thing, Contents(player)) {
	if ((a = atr_get(thing, name)) != NULL) {
	  if (Can_Read_Attr(privs, thing, a))
	    return a;
	}
      }
    }
  }
  return NULL;
}

/* this function produces a space-separated list of attributes that are 
 * on an object. It really belongs in eval.c, but it's here so it can
 * use some of the attribute stuff needed.
 */
XFUNCTION(fun_lattr)
{
  ALIST *list;
  dbref thing;
  char *pattern, *bp;

  pattern = (char *) index(args[0], '/');
  if (pattern)
    *pattern++ = '\0';
  else
    pattern = (char *)"*";	/* match anything */

  thing = match_thing(privs, args[0]);
  if (thing == NOTHING) {
    strcpy(buff, "#-1 NO MATCH");
    return;
  }
  if (!Can_Examine(privs, thing)) {
    strcpy(buff, "#-1 PERMISSION DENIED");
    return;
  }

  bp = buff;
  for (list = db[thing].list; list; list = AL_NEXT(list)) {
    if (AL_BAD(list)) continue;
    if (AL_FLAGS(list) & AF_DARK) continue;
    if ((AL_FLAGS(list) & AF_MDARK) && !See_All(privs)) continue;
    if (local_wild_match(pattern, AL_NAME(list))) {
      if (bp != buff)
	safe_chr(' ', buff, &bp);
      safe_str(AL_NAME(list), buff, &bp);
    }
  }
  *bp = '\0';
}
