/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 1993, 1994, Elan Feingold (feingold@zko.dec.com)           *
 *                                                                           *
 *     PERMISSION TO USE, COPY, MODIFY, AND TO DISTRIBUTE THIS SOFTWARE      *
 *     AND ITS DOCUMENTATION FOR ANY PURPOSE IS HEREBY GRANTED WITHOUT       *
 *     FEE, PROVIDED THAT THE ABOVE COPYRIGHT NOTICE APPEAR IN ALL           *
 *     COPIES AND MODIFIED COPIES AND THAT BOTH THAT COPYRIGHT NOTICE AND    *
 *     THIS PERMISSION NOTICE APPEAR IN SUPPORTING DOCUMENTATION.  THERE     *
 *     IS NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR      *
 *     ANY PURPOSE.  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS       *
 *     OR IMPLIED WARRANTY.                                                  *
 *                                                                           *
 *****************************************************************************/

#include <X11/X.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Form.h>

#include <stdio.h>
#include <stdlib.h>

#include "utils.h"
#include "riskgame.h"
#include "gui.h"
#include "network.h"
#include "client.h"
#include "colormap.h"
#include "callbacks.h"
#include "game.h"
#include "dice.h"
#include "debug.h"


#define MAX_LINES 5

#define MAX_DIGITS 3
#define MAX_STRING "888"

Int   iQueryResult;

/************************************************************************ 
 *  FUNCTION: UTIL_PrintArmies
 *  HISTORY: 
 *     01.28.94  ESF  Created.
 *     02.05.94  ESF  Added number centering (finally!)
 *     03.30.94  ESF  Changed EXTRA_Y to be smaller.
 *     04.02.94  ESF  Fixed font problem, not XSetFont'ing.
 *     05.05.94  ESF  Fixed so that 0 armies doesn't print anything.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_PrintArmies(Int iCountry, Int iNumArmies, Int iColor)
{
  Int          x, y, iWidth, iHeight;
  Char         strNumber[4];

  /* Is number of armies right? */
  if (iNumArmies>999)
    {
      UTIL_PopupDialog("Warning", "UTIL: Too many armies", 
		       1, "Ok", NULL, NULL);
      return;
    }
  
  /* Setup what we want to print (could be empty) */
  if (iNumArmies>0)
    sprintf(strNumber, "%d", iNumArmies);
  else
    sprintf(strNumber, " ");

  /* Erase the old number, assume three digits */
  x       = RISK_GetTextXOfCountry(iCountry);
  y       = RISK_GetTextYOfCountry(iCountry);
  iHeight = (pFont->max_bounds.ascent + pFont->max_bounds.descent);
  iWidth  = XTextWidth(pFont, MAX_STRING, MAX_DIGITS);

  /* Draw erasing rectangle on screen and pixmap */
  XSetForeground(hDisplay, hGC, COLOR_CountryToColor(iCountry));
  XFillRectangle(hDisplay, hWindow, hGC, x, y-iHeight, 
		 iWidth, iHeight+1);
  XFillRectangle(hDisplay, pixMapImage, hGC, x, y-iHeight, 
		 iWidth, iHeight+1);

  /* Center the number, based on the number of digits */
  if (strlen(strNumber)==1)
    x += iWidth/3;
  else if (strlen(strNumber)==2)
    x += iWidth/4;

  /* Finally draw the text in the very first color before the countries */
  XSetForeground(hDisplay, hGC, iColor);
  XSetFont(hDisplay, hGC, pFont->fid);

  if (iNumArmies>=0)
    {
      XDrawString(hDisplay, hWindow, hGC, x, y, strNumber, 
		  strlen(strNumber));
      XDrawString(hDisplay, pixMapImage, hGC, x, y, strNumber, 
		  strlen(strNumber));
    }

  /* Flush the queue of requests */
  XFlush(hDisplay);
}


/************************************************************************ 
 *  FUNCTION: UTIL_DisplayMessage
 *  HISTORY: 
 *     02.04.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_DisplayMessage(String strFrom, String strMessage)
{
  static Int  iLength[MAX_LINES], iLines=0, iPosition=0;
  String      strOldMessage, strNewMessage;
  Arg         pArgs[8];
  Int         iCount;
  
  /* Get the old text */
  iCount = 0;
  XtSetArg(pArgs[iCount], XtNstring, &strOldMessage); iCount++;
  XtGetValues(wMsgText, pArgs, iCount);
  
  /* Bookeeping for correct scrolling */
  iLength[iLines % MAX_LINES] = strlen(strMessage) + 
    strlen(strFrom) + 3;
  iLines++;
  
  /* Build up the new message */
  strNewMessage = (String)MEM_Alloc(strlen(strOldMessage) + 
				 strlen(strMessage) + 
				 strlen(strFrom) + 4);
  strcpy(strNewMessage, strOldMessage);
  strcat(strNewMessage, strFrom);
  strcat(strNewMessage, ": ");
  strcat(strNewMessage, strMessage);
  strcat(strNewMessage, "\n");
  
  iCount = 0;
  XtSetArg(pArgs[iCount], XtNstring, strNewMessage); iCount++;
  XtSetValues(wMsgText, pArgs, iCount);
  
  if (iLines >= MAX_LINES)
    {
      iPosition += iLength[iLines % MAX_LINES];
      iCount = 0;
      XtSetArg(pArgs[iCount], XtNdisplayPosition, iPosition); iCount++;
      XtSetValues(wMsgText, pArgs, iCount);
    }
}


/************************************************************************ 
 *  FUNCTION: UTIL_DisplayComment
 *  HISTORY: 
 *     02.07.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_DisplayComment(String strComment)
{
  Arg   pArgs[1];
  Int   iCount;

  iCount=0;
  XtSetArg(pArgs[iCount], XtNlabel, strComment); iCount++;
  XtSetValues(wCommentLabel, pArgs, iCount);
}



/************************************************************************ 
 *  FUNCTION: UTIL_SetPlayerTurnIndicator
 *  HISTORY: 
 *     03.03.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_SetPlayerTurnIndicator(Int iPlayer)
{
  
}


/************************************************************************ 
 *  FUNCTION: UTIL_DisplayActionString
 *  HISTORY: 
 *     03.03.94  ESF  Created.
 *     03.29.94  ESF  Added player attack mode history.
 *     03.30.94  ESF  Fixed bug, not erasing iAttackSrc.
 *     04.02.94  ESF  Added clearing of error.
 *     05.07.94  ESF  Removed clearing of error.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_DisplayActionString(Int iState, Int iPlayer)
{
  switch (iState)
    {
    case STATE_FORTIFY:
      sprintf(strScratch, "%s to place an army (%d remaining)...", 
	      RISK_GetNameOfPlayer(iPlayer),
	      RISK_GetNumArmiesOfPlayer(iPlayer));
      XawListHighlight(wActionList, iActionState = ACTION_PLACE);
      XawListHighlight(wAttackList,
		       RISK_GetDiceModeOfPlayer(iCurrentPlayer));
      XawListHighlight(wMsgDestList,
		       RISK_GetMsgDstModeOfPlayer(iCurrentPlayer));
      break;

      
    case STATE_PLACE:
      sprintf(strScratch, "%s placing armies (%d remaining)...", 
	      RISK_GetNameOfPlayer(iPlayer),
	      RISK_GetNumArmiesOfPlayer(iPlayer));
      XawListHighlight(wActionList, iActionState = ACTION_PLACE);
      XawListHighlight(wAttackList, 
		       RISK_GetDiceModeOfPlayer(iCurrentPlayer));
      XawListHighlight(wMsgDestList,
		       RISK_GetMsgDstModeOfPlayer(iCurrentPlayer));
      break;

    case STATE_ATTACK:
      sprintf(strScratch, "%s attacking...", 
	      RISK_GetNameOfPlayer(iPlayer));
      iActionState = RISK_GetAttackModeOfPlayer(iCurrentPlayer);
      XawListHighlight(wActionList, 
		       RISK_GetAttackModeOfPlayer(iCurrentPlayer));
      XawListHighlight(wAttackList,
		       RISK_GetDiceModeOfPlayer(iCurrentPlayer));
      XawListHighlight(wMsgDestList,
		       RISK_GetMsgDstModeOfPlayer(iCurrentPlayer));
      break;

    case STATE_MOVE:
      if (iAttackSrc >= 0)
	{
	  UTIL_DarkenCountry(iAttackSrc);
	  iAttackSrc = -1;
	}

      sprintf(strScratch, "%s executing a free move...", 
	      RISK_GetNameOfPlayer(iPlayer));
      XawListHighlight(wActionList, iActionState = ACTION_MOVE);
      XawListHighlight(wAttackList,
		       RISK_GetDiceModeOfPlayer(iCurrentPlayer));
      XawListHighlight(wMsgDestList,
		       RISK_GetMsgDstModeOfPlayer(iCurrentPlayer));
      break;
    }

  UTIL_DisplayComment(strScratch);
}


/************************************************************************ 
 *  FUNCTION: UTIL_DisplayError
 *  HISTORY: 
 *     03.04.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_DisplayError(String strError)
{
  Arg   pArgs[1];
  Int   iCount;
  
  iCount=0;
  XtSetArg(pArgs[iCount], XtNlabel, strError); iCount++;
  XtSetValues(wErrorLabel, pArgs, iCount);
}


/************************************************************************ 
 *  FUNCTION: UTIL_ServerEnterState
 *  HISTORY: 
 *     03.05.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_ServerEnterState(Int iNewState)
{
  MsgEnterState m;

  /* Change states and let the server know about it */
  m.iState = iNewState;
  NET_SendMessage(iWriteSocket, MSG_ENTERSTATE, &m);
}


/************************************************************************ 
 *  FUNCTION: UTIL_GetArmyNumber
 *  HISTORY: 
 *     03.16.94  ESF  Created.
 *     03.28.94  ESF  Fixed minor printing bug.
 *     04.01.94  ESF  Fixed bug in centering shell, had XtNy with an x.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int UTIL_GetArmyNumber(Int iMinArmies, Int iMaxArmies, Boolean fLetCancel)
{
  Int        x, y, iNumArmies;
  XEvent     xEvent;
  String     strBuffer;

  UTIL_CenterShell(wArmiesShell, wToplevel, &x, &y);
  XtVaSetValues(wArmiesShell, 
		XtNallowShellResize, False,
		XtNx, x, 
		XtNy, y, 
		XtNborderWidth, 1,
		XtNtitle, "Frisk",
		NULL);

  /* Set the default number of armies as the maximum */
  sprintf(strScratch, "%d", iMaxArmies);
  XtVaSetValues(wArmiesText, XtNstring, strScratch, NULL);

  /* Don't display the cancel button if there is no chance of this */
  if (!fLetCancel)
    XtVaSetValues(wCancelArmiesButton, XtNsensitive, False, NULL);
  else
    XtVaSetValues(wCancelArmiesButton, XtNsensitive, True, NULL);

  XtPopup(wArmiesShell, XtGrabExclusive);
  while(XtGrabKeyboard(wArmiesText, True, GrabModeAsync, GrabModeAsync, 
		       CurrentTime) == GrabNotViewable)
    ; /* TwiddleThumbs() */

 keep_going:
  iQueryResult = QUERY_INPROGRESS;
  while (iQueryResult == QUERY_INPROGRESS) 
    {
      /* pass events */
      XNextEvent(hDisplay, &xEvent);
      XtDispatchEvent(&xEvent);
    }

  /* User must have selected one of the buttons */
  if (!fLetCancel && iQueryResult==QUERY_NO)
    goto keep_going;
  else if (iQueryResult==QUERY_NO)
    {
      UTIL_DisplayError("");
      XtUngrabKeyboard(wArmiesText, CurrentTime);
      XtPopdown(wArmiesShell);
      return(0);
    }
  
  /* Get number of armies */
  XtVaGetValues(wArmiesText, XtNstring, &strBuffer, NULL);
  iNumArmies = atoi(strBuffer);

  if (iNumArmies<iMinArmies || iNumArmies>iMaxArmies)
    {
      if (iNumArmies < iMinArmies)
	sprintf(strScratch, "You must move at least %d armies.", iMinArmies);
      else
	sprintf(strScratch, "You can't move more than %d armies.", iMaxArmies);
      UTIL_DisplayError(strScratch);
      
      goto keep_going;
    }

  UTIL_DisplayError("");
  XtPopdown(wArmiesShell);
  XtUngrabKeyboard(wArmiesText, CurrentTime);
  return(iNumArmies);
}


void UTIL_QueryYes(Widget w, XtPointer pData, XtPointer pCalldata)    
{ iQueryResult = QUERY_YES; }
void UTIL_QueryNo(Widget w, XtPointer pData, XtPointer pCalldata)     
{ iQueryResult = QUERY_NO; }
void UTIL_QueryCancel(Widget w, XtPointer pData, XtPointer pCalldata) 
{ iQueryResult = QUERY_CANCEL; }


/************************************************************************ 
 *  FUNCTION: UTIL_CenterShell
 *  HISTORY: 
 *     03.16.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_CenterShell(Widget wCenter, Widget wBase, Int *x, Int *y)
{
  Dimension  dimWidth, dimHeight, dimTopWidth, dimTopHeight;
  Window     hWindow;

  /* Make sure the two widgets are realized */
  if (!XtIsRealized(wCenter))
    XtRealizeWidget(wCenter);
  if (!XtIsRealized(wBase))
    XtRealizeWidget(wBase);

  /* Get the dimensions of the shells and center the popup. */
  XtVaGetValues(wCenter, 
		XtNwidth, &dimWidth, 
		XtNheight, &dimHeight, NULL);
  XtVaGetValues(wBase, 
		XtNwidth, &dimTopWidth, 
		XtNheight, &dimTopHeight, NULL);
  
  XTranslateCoordinates(hDisplay, XtWindow(wBase), 
			XDefaultRootWindow(hDisplay),
			(dimTopWidth-dimWidth)/2, 
			(dimTopHeight-dimHeight)/2, 
			x, y, &hWindow);
}


/************************************************************************ 
 *  FUNCTION: UTIL_ServerEndTurn
 *  HISTORY: 
 *     03.17.94  ESF  Created.
 *     03.28.94  ESF  Added card code, do we need Sync on the call?
 *     03.28.94  ESF  Moved iAttack/iDefend stuff here.
 *     03.29.94  ESF  Added iMove stuff.
 *     04.11.94  ESF  Added clear dice call.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_ServerEndTurn(void)
{
  /* Get the player a card if he or she needs one */
  if (fGetsCard)
    {
      MsgRequestCard msg;
      
      msg.iPlayer = iCurrentPlayer;
      NET_SendMessage(iWriteSocket, MSG_REQUESTCARD, &msg);
    }
  
  /* Notify the server of the end-of-turn condition and wait for a response */
  NET_SendSyncMessage(iWriteSocket, MSG_ENDTURN, NULL,
		      iReadSocket, MSG_TURNNOTIFY, CBK_IncomingMessage);

  /* Darken the country if the player was attacking or moving */
  if (iAttackSrc >= 0)
    UTIL_DarkenCountry(iAttackSrc);
  if (iMoveSrc >= 0)
    UTIL_DarkenCountry(iMoveSrc);    
  
  /* Clear the dice box */
  DICE_Hide();

  /* Init. variables for next turn */
  fGetsCard  = FALSE;
  fCanExchange = TRUE;
  iAttackSrc = iAttackDst = iLastAttackSrc = iLastAttackDst = -1;
  iMoveSrc = iMoveDst = -1;
}


/************************************************************************ 
 *  FUNCTION: UTIL_LightCountry
 *  HISTORY: 
 *     03.28.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_LightCountry(Int iCountry)
{
  UTIL_PrintArmies(iCountry,
		   RISK_GetNumArmiesOfCountry(iCountry),
		   WhitePixel(hDisplay, 0));
}


/************************************************************************ 
 *  FUNCTION: UTIL_DarkenCountry
 *  HISTORY: 
 *     03.28.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_DarkenCountry(Int iCountry)
{
  UTIL_PrintArmies(iCountry, 
		   RISK_GetNumArmiesOfCountry(iCountry),
		   BlackPixel(hDisplay, 0));
}


/************************************************************************ 
 *  FUNCTION: UTIL_PopupDialog
 *  HISTORY: 
 *     04.02.94  ESF  Created.
 *     05.12.94  ESF  Revamped.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int UTIL_PopupDialog(String strTitle, String strLabel, Int iNumOptions, 
		     String strOption1, String strOption2, String strOption3)
{
  String               pstrOptions[3];
  Int                  i, x, y, iWidth;
  XEvent               xEvent;
  static XFontStruct  *pLabelFont=NULL;

  /* Initialize these for later */
  pstrOptions[0] = strOption1;  
  pstrOptions[1] = strOption2;  
  pstrOptions[2] = strOption3;

  /* Get the font if necessary */
  if (pLabelFont==NULL)
    if ((pLabelFont=XLoadQueryFont(hDisplay, "*helvetica-b*-o-*14*"))==NULL)
      {
	printf("Can't find the font \"*helvetica-b*-o-*14*\"!!\n");
	UTIL_ExitProgram(-1);
      }

  /* Sanity check */
  if (iNumOptions>3 || iNumOptions<1)
    {
      UTIL_PopupDialog("Warning", "UTIL: Wacked # of options", 
		       1, "Ok", NULL, NULL);
      return(-1);
    }

  /* Make sure the dialog is realized */
  if(!XtIsRealized(wDialogShell))
     XtRealizeWidget(wDialogShell);

  /* Set the title */
  XtVaSetValues(wDialogShell, 
		XtNtitle, strTitle == NULL ? "Frisk Dialog" : strTitle,
		XtNallowShellResize, True,
		NULL);

  /* Make sure that the label is big enough to hold the string, and
   * if the length of the button strings are larger, make it bigger.
   * Let there be a minimum width of 200...  There are a few magic 
   * numbers here, excuse them :)
   */
  
  iWidth = MAX(200,
	       MAX(XTextWidth(pLabelFont, strLabel, strlen(strLabel))+50,
		   XTextWidth(pLabelFont, strOption1, 
			      strOption1 ? strlen(strOption1) : 0)+
		   XTextWidth(pLabelFont, strOption2, 
			      strOption2 ? strlen(strOption2) : 0)+
		   XTextWidth(pLabelFont, strOption3, 
			      strOption3 ? strlen(strOption3) : 0)+
		   140));

  XtVaSetValues(wDialogLabel, 
		XtNlabel, strLabel, 
		XtNwidth, iWidth,
		NULL);

  /* Set the button resources */
  for (i=0; i!=3; i++)
    if (iNumOptions>i)
      {
	XtMapWidget(wDialogButton[i]);
	XtVaSetValues(wDialogButton[i], 
		      XtNlabel, pstrOptions[i]==NULL ? "Null": pstrOptions[i], 
		      XtNfromHoriz, i>0 ? wDialogButton[i-1] : NULL,
		      NULL);
      }
  else
    {
      XtUnmapWidget(wDialogButton[i]);
      XtVaSetValues(wDialogButton[i],
		    XtNfromHoriz, 0, 
		    NULL); 
    }

  /* We need to unrealize it so that the changes will take place (?) */
  XtUnrealizeWidget(wDialogShell);

  /* Center and resize the shell, and then actually pop the dialog up */
  XtVaSetValues(wDialogShell, XtNwidth, iWidth, NULL);
  UTIL_CenterShell(wDialogShell, wToplevel, &x, &y);
  XtVaSetValues(wDialogShell, 
		XtNx, x, 
		XtNy, y,
		XtNborderWidth, 1,
		NULL);
  XtPopup(wDialogShell, XtGrabExclusive);
  
  iQueryResult = QUERY_INPROGRESS;
  while (iQueryResult == QUERY_INPROGRESS) 
    {
      /* pass events */
      XNextEvent(hDisplay, &xEvent);
      XtDispatchEvent(&xEvent);
    }

  /* User must have selected one of the buttons */
  XtPopdown(wDialogShell);
  return(iQueryResult);  
}


/************************************************************************ 
 *  FUNCTION: UTIL_PlayerIsLocal
 *  HISTORY: 
 *     04.11.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Boolean UTIL_PlayerIsLocal(Int iPlayer)
{
  return (RISK_GetClientOfPlayer(iPlayer) == iThisClient ? TRUE : FALSE);
}


/************************************************************************ 
 *  FUNCTION: UTIL_SendMessage
 *  HISTORY: 
 *     05.03.94  ESF  Created.
 *     05.04.94  ESF  Enhanced.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_SendMessage(void)
{
  String                strMess;
  XawListReturnStruct  *pItem;
  MsgMessagePacket      mess;
  char                  strScratch2[128];

  /* Get the message and destination from widgets */
  XtVaGetValues(wSendMsgText, XtNstring, &strMess, NULL);
  pItem = XawListShowCurrent(wMsgDestList);

  /* Who sent the message? */
  if (RISK_GetClientOfPlayer(iCurrentPlayer) == iThisClient)
    mess.strFrom = RISK_GetNameOfPlayer(iCurrentPlayer);
  else if (UTIL_NumPlayersAtClient(iThisClient) == 1)
    mess.strFrom = UTIL_NameOfNthPlayerAtClient(iThisClient, 0); /* !!! */
  else
    {
      sprintf(strScratch2, "Someone at %s", strClientName);
      mess.strFrom = strScratch2;
    }

  /* Who do we send the message to? */
  if (pItem->list_index == -1 || pItem->list_index == 0)
    mess.iTo = -1;
  else
    mess.iTo = RISK_GetClientOfPlayer(pItem->list_index-1);

  /* Build the actual message */
  if (pItem->list_index == -1 || pItem->list_index == 0)
    sprintf(strScratch, "(To Everybody) \"%s\"", strMess);
  else
    sprintf(strScratch, "(To %s) \"%s\"", pItem->string, strMess);    
  
  mess.strMessage = strScratch;

  /* Display the message locally if necessary */
  if (mess.iTo != iThisClient && mess.iTo != -1)
    UTIL_DisplayMessage(mess.strFrom, mess.strMessage);
  
  /* Send the message */
  NET_SendMessage(iWriteSocket, MSG_MESSAGEPACKET, &mess);
  
  /* Erase the old message */
  strMess = (String)MEM_Alloc(16); 
  strcpy(strMess, "");
  XtVaSetValues(wSendMsgText, XtNstring, strMess, NULL);
}


/************************************************************************ 
 *  FUNCTION: UTIL_NameOfNthPlayerAtClient
 *  HISTORY: 
 *     05.04.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
String UTIL_NameOfNthPlayerAtClient(Int iClient, Int iIndex)
{
  Int i, iFound;

  for (i=0, iFound=-1; i!=MAX_PLAYERS && iFound!=iIndex ; i++)
    if (RISK_GetClientOfPlayer(i) == iClient)
      iFound++;

  if (iFound==iIndex)
    return RISK_GetNameOfPlayer(i-1);
  else
    return (String)NULL;
}


/************************************************************************ 
 *  FUNCTION: UTIL_GetNthPlayerAtClient
 *  HISTORY: 
 *     05.04.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int UTIL_GetNthPlayerAtClient(Int iClient, Int iIndex)
{
  Int i, iFound;

  for (i=0, iFound=-1; i!=MAX_PLAYERS && iFound!=iIndex ; i++)
    if (RISK_GetClientOfPlayer(i) == iClient)
      iFound++;

  if (iFound==iIndex)
    return (i-1);
  else
    return -1;
}


/************************************************************************ 
 *  FUNCTION: 
 *  HISTORY: 
 *     05.04.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int UTIL_NumPlayersAtClient(Int iClient)
{
  Int i, iCount;

  for (i=iCount=0; i!=MAX_PLAYERS; i++)
    if (RISK_GetClientOfPlayer(i) == iClient)
      iCount++;

  return iCount;
}


/************************************************************************ 
 *  FUNCTION: 
 *  HISTORY: 
 *     05.06.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_RefreshMsgDest(Int iNumStrings)
{
  /* Update the listbox */
  XtVaSetValues(wMsgDestList, 
		XtNlist, pstrMsgDstString,
		XtNnumberStrings, iNumStrings,
		NULL);
}


/************************************************************************ 
 *  FUNCTION: 
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *     05.15.94  ESF  Fixed bug, was deleting iThisClient instead of iClient.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_DeleteMsgDst(Int iClient)
{
  Int       i, j, k;
  Boolean   fDone;

  /* Delete from the listbox */
  for (i=0; i!=MAX_PLAYERS+1; i++)
    if (RISK_GetClientOfPlayer(i) == iClient)
      {
	/* Get rid of player i */
	for (j=1, fDone=FALSE; j!=iIndexMD && !fDone; j++)
	  if (i == piMsgDstPlayerID[j])
	    {
	      fDone = TRUE;
	      MEM_Free(pstrMsgDstString[j]);
	      iIndexMD--;

	      /* Copy the other players over */
	      for (k=j; k!=iIndexMD; k++)
		{
		  pstrMsgDstString[k] = pstrMsgDstString[k+1];
		  piMsgDstPlayerID[k] = piMsgDstPlayerID[k+1];
		}
	    }
      }
  
  UTIL_RefreshMsgDest(iIndexMD);		
}



/************************************************************************ 
 *  FUNCTION: UTIL_ExitProgram
 *  HISTORY: 
 *     06.16.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_ExitProgram(Int iExitValue)
{
  MEM_TheEnd();
  exit(iExitValue);
}







