/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 1993, 1994, Elan Feingold (feingold@zko.dec.com)           *
 *                                                                           *
 *     PERMISSION TO USE, COPY, MODIFY, AND TO DISTRIBUTE THIS SOFTWARE      *
 *     AND ITS DOCUMENTATION FOR ANY PURPOSE IS HEREBY GRANTED WITHOUT       *
 *     FEE, PROVIDED THAT THE ABOVE COPYRIGHT NOTICE APPEAR IN ALL           *
 *     COPIES AND MODIFIED COPIES AND THAT BOTH THAT COPYRIGHT NOTICE AND    *
 *     THIS PERMISSION NOTICE APPEAR IN SUPPORTING DOCUMENTATION.  THERE     *
 *     IS NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR      *
 *     ANY PURPOSE.  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS       *
 *     OR IMPLIED WARRANTY.                                                  *
 *                                                                           *
 *****************************************************************************/

#include <X11/Xlib.h>
#include <X11/X.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <stdio.h>
#include <stdlib.h>

#include "gui.h"
#include "riskgame.h"
#include "network.h"
#include "client.h"
#include "types.h"
#include "utils.h"
#include "cards.h"
#include "colormap.h"
#include "callbacks.h"
#include "debug.h"

/* We need a structure to hold the directory information */
struct Directory
{
  int   iWidth, iHeight, iLength;
  long  lOffset;
} pDirectory[NUM_COUNTRIES];

/* Here's where we store the cached pixmaps */
Pixmap pixCards[NUM_COUNTRIES];

Int          iNumCountries;
Boolean      fDirectoryRead = FALSE;
FILE        *hFile;
XFontStruct *pCardFont;   

#define PICTURE_FRACTION 0.6

/************************************************************************ 
 *  FUNCTION: CARD_RenderCard
 *  HISTORY: 
 *     02.21.94  ESF  Created.
 *     02.22.94  ESF  Fixed positioning bug, made prettier.
 *     03.02.94  ESF  Added the printing of the country name.
 *     03.29.94  ESF  Fixed the setup for printing the jokers.
 *     04.02.94  ESF  Fixed name centering, wasn't XSetFont'ing.
 *     05.06.94  ESF  Made it more beautiful, took out hacks.
 *     06.25.94  ESF  Optimized card decompression.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CARD_RenderCard(Int iCard, Int iPosition)
{
  Byte  *pbBogus;
  Int    i;

  /* Range check */
  if (iCard<0 || iCard>=NUM_CARDS || iPosition>=MAX_CARDS || iPosition<0)
    {
      UTIL_PopupDialog("Warning!", "Bogus request to CARD_RenderCard()!\n", 1,
		       "Ok", NULL, NULL);
      return;
    }

  /* If we need to init things, do it */
  if (!fDirectoryRead)
    {
      fDirectoryRead = TRUE;

      /* Load the font */
      if ((pCardFont=XLoadQueryFont(hDisplay, "*helvetica-b*-r-*12*")) == NULL)
	{
	  UTIL_PopupDialog("Warning", 
			   "Could not open card font (using fixed)\n", 1,
			   "Ok", NULL, NULL);
	  
	  /* Assume 'fixed' is always there -- not good. */
	  pCardFont = XLoadQueryFont(hDisplay, "fixed");
	}


      /* Open the file */
      if ((hFile=fopen(COUNTRYFILE, "r"))==NULL)
	{
	  sprintf(strScratch, "CARDS: Cannot open %s!\n", COUNTRYFILE);
	  UTIL_PopupDialog("Fatal Error", strScratch, 1, "Ok", NULL, NULL);
	  NET_SendMessage(iWriteSocket, MSG_EXIT, NULL);
	  UTIL_ExitProgram(-1);
	}
      
      /* Read the directory */
      fscanf(hFile, "%d%c", &iNumCountries, (char *)&pbBogus);
      if (iNumCountries > NUM_COUNTRIES) /* One is for ocean */
	{
	  UTIL_PopupDialog("Fatal Error", 
			   "CARDS: Too many countries!",
			   1, "Ok", NULL, NULL);
	  NET_SendMessage(iWriteSocket, MSG_EXIT, NULL);
	  UTIL_ExitProgram(-1);
	}
      fread(pDirectory, iNumCountries, sizeof(pDirectory[0]), hFile);

      /* Init the pixmap cache */
      for (i=0; i!=NUM_COUNTRIES; i++)
	pixCards[i] = 0L;

      /* Hack for now */
      iNumCountries = NUM_COUNTRIES;
    }


  /* Check to see if pixmap is in cache.  If not, build it */
  if(pixCards[iCard] == 0L)
    {
      XImage  *pimageCountry;
      Byte    *pbImage, *pbCompressed, *pTemp, *pTemp2;
      Byte    bLength, bColor;
      Int     iPictureWidth, iPictureHeight, iPictureOffset;
      Int     iFontHeight, iFontWidth;
      Int     iCardColor, iCardBlack;
	      
      /* Create the pixmap */
      pixCards[iCard] = XCreatePixmap(hDisplay, pixMapImage, 
				      CARD_WIDTH, CARD_HEIGHT, 8);

      /* Makes life easier */
      iPictureWidth  = CARD_WIDTH;
      iPictureHeight = CARD_HEIGHT * PICTURE_FRACTION; 
      iPictureOffset = CARD_HEIGHT - iPictureHeight;

      /* Set the font */
      XSetFont(hDisplay, hGC, pCardFont->fid);

      /* Fill the card up with white and black */
      XSetForeground(hDisplay, hGC, BlackPixel(hDisplay, 0));
      XFillRectangle(hDisplay, pixCards[iCard], hGC, 
		     0, 0, 
		     CARD_WIDTH, CARD_HEIGHT);
      XSetForeground(hDisplay, hGC, WhitePixel(hDisplay, 0));
      XFillRectangle(hDisplay, pixCards[iCard], hGC, 
		     0, 0, 
		     CARD_WIDTH, iPictureOffset);
      
      /* Find out which country goes with the card */
      if (iCard < iNumCountries)
	{
	  /* Allocate the memory for the card */
	  pbImage = (Byte *)XtMalloc(pDirectory[iCard].iWidth * 
				     pDirectory[iCard].iHeight);
	  pbCompressed = (Byte *)MEM_Alloc(pDirectory[iCard].iLength);

	  /* Go seek the card */
	  fseek(hFile, pDirectory[iCard].lOffset, SEEK_SET);
	  
	  /* Read it in from the data file */
	  fread(pbCompressed, pDirectory[iCard].iLength, 1, hFile);

	  /* Get the colors of the card.  This is an optimization, 
	   * since we should get the color from the compressed card
	   * bitmap.  However, we know what color the card should
	   * be because of our nice organization of the card ->
	   * country mapping.
	   */
	  iCardColor = COLOR_CountryToColor(iCard);
	  iCardBlack = BlackPixel(hDisplay, 0);

	  /* Uncompress the data */
	  for (i=0,pTemp2=pbCompressed,pTemp=pbImage; 
	       i<pDirectory[iCard].iLength; i+=2)
	    {
	      /* Get a color segment */
	      bLength = *pTemp2++;
	      bColor  = *pTemp2++;
	      
	      /* Put it in the country image */
	      if (bColor==BLACK)
		memset(pTemp, iCardBlack, bLength);
	      else
		memset(pTemp, iCardColor, bLength);
	      pTemp += bLength; 
	    }

	  /* Create the image, and use a special scaling algorithm to
	   * shrink it down to the necessary size to fit in the card.
	   */
	  
	  pimageCountry =
	    XCreateImage(hDisplay, 
			 DefaultVisual(XtDisplay(wCardToggle[iPosition]),
				       iScreen),
			 8, ZPixmap, 0, (char *)pbImage, 
			 pDirectory[iCard].iWidth, 
			 pDirectory[iCard].iHeight, 
	 		 8, pDirectory[iCard].iWidth);

	  /* Compress the card image, being careful about the pointer */
	  pimageCountry = CARD_ScaleImage(pimageCountry, 
					  4/5.0 * (float)iPictureWidth, 
					  4/5.0 * (float)iPictureHeight);
	  
	  /* Dump image, centered, on lower 2/3 of card */
	  XPutImage(hDisplay, pixCards[iCard], hGC, pimageCountry, 0, 0, 
		    (iPictureWidth - pimageCountry->width)/2,
		    iPictureOffset + 
		    (iPictureHeight - pimageCountry->height)/2,
		    pimageCountry->width,
		    pimageCountry->height);
	  
	  /* Write the name of the country on the card */
	  iFontHeight = (pCardFont->max_bounds.ascent 
			 + pFont->max_bounds.descent);
	  iFontWidth  = XTextWidth(pCardFont,
				   RISK_GetNameOfCountry(iCard),
				   strlen(RISK_GetNameOfCountry(iCard)));

	  XDrawString(hDisplay, pixCards[iCard], hGC, 
		      (iPictureWidth - iFontWidth)/2, 
		      iPictureOffset+iFontHeight, 
		      RISK_GetNameOfCountry(iCard),
		      strlen(RISK_GetNameOfCountry(iCard)));

	  /* Lose all of the used memory */
	  XDestroyImage(pimageCountry);
	  MEM_Free(pbCompressed);

	  /* Now put the appropriate bitmap there */
	  sprintf(strScratch, "%s", (iCard%3==0) ? "Cavalry" : 
		  (iCard%3==1) ? "Infantry" : "Artillery");
	  iFontWidth = XTextWidth(pCardFont, strScratch, strlen(strScratch));
	  XSetForeground(hDisplay, hGC, BlackPixel(hDisplay, 0));
	  XDrawString(hDisplay, pixCards[iCard], hGC, 
		      (CARD_WIDTH - iFontWidth)/2,
		      iFontHeight,
		      strScratch, strlen(strScratch));
	}
      else
	{
	  /* Jokers are all white */
	  XSetForeground(hDisplay, hGC, WhitePixel(hDisplay, 0));
	  XFillRectangle(hDisplay, pixCards[iCard], hGC, 
			 0, 0, 
			 CARD_WIDTH, CARD_HEIGHT);

	  /* It's a joker, put all of the bitmaps there */
	  sprintf(strScratch, "%s", "Joker");

	  iFontWidth = XTextWidth(pCardFont, strScratch, strlen(strScratch));
	  iFontHeight = (pCardFont->max_bounds.ascent + 
			 pCardFont->max_bounds.descent);

	  XSetForeground(hDisplay, hGC, BlackPixel(hDisplay, 0));
	  XDrawString(hDisplay, pixCards[iCard], hGC, 
		      (CARD_WIDTH - iFontWidth)/2,
		      iFontHeight,
		      strScratch, strlen(strScratch));
	}
    }

  /* Now it's built, put it in the card's bitmap resource, manage the card */
  XtVaSetValues(wCardToggle[iPosition], 
		XtNbitmap, pixCards[iCard], 
		NULL);
  XtManageChild(wCardToggle[iPosition]);
}


/************************************************************************ 
 *  FUNCTION: CARD_ScaleImage
 *  HISTORY: 
 *     ??.??.93  ESF  Created for the XDissolver.
 *     02.22.94  ESF  Made pretty, changed to work with 8 bit images.
 *  PURPOSE: 
 *     Using a Bresenham like algorithm, squish the image passed in to
 *   a box [x, y].  Return the compressed image.  Preserve aspect
 *   ratio of the image, so find out the MAX of the compression in
 *   the x and y directions.
 *  NOTES: 
 ************************************************************************/
XImage *CARD_ScaleImage(XImage *pimageInput, Int iMaxWidth, Int iMaxHeight)
{
  XImage   *pimageCompressed;
  Byte     *pbData;
  Int       i, j, iImageX, iImageY;
  float     rWidthCompression, rHeightCompression, rCompression;
  float     rWidthThreshold, rHeightThreshold;
  Int       iOutputWidth, iOutputHeight;

  /* Find out what Compression ratio we need */
  rWidthCompression = MAX(pimageInput->width / (float)iMaxWidth, 1.0); 
  rHeightCompression = MAX(pimageInput->height / (float)iMaxHeight, 1.0);
  rCompression = MAX(rWidthCompression, rHeightCompression);
  
  /* if close enough, optimize by simply returning the input image */
  if (rCompression<1.001)
    return pimageInput;

  /* Find out the size of the output image */
  iOutputWidth = (float)pimageInput->width/rCompression;
  iOutputHeight = (float)pimageInput->height/rCompression;
  
  /* This will hold the compressed input data as we're compressing it */
  pbData = (Byte *)XtMalloc(sizeof(Byte)*iOutputWidth*iOutputHeight);
  pimageCompressed = XCreateImage(hDisplay, 
				  DefaultVisual(hDisplay, iScreen), 
				  8, ZPixmap, 0, (char *)pbData, 
				  iOutputWidth, iOutputHeight, 8, 
				  iOutputWidth);
  
   /* Now compress the data in the incoming image structure */
  for (j=0, rHeightThreshold=rCompression, iImageY=0; 
       j!=pimageInput->height; 
       j++, rHeightThreshold++)
    {
      for (i=0, iImageX=0, rWidthThreshold=rCompression; 
	   i!=pimageInput->width; 
	   i++, rWidthThreshold++)
	{
	  if (rWidthThreshold>rCompression)
	    {
	      XPutPixel(pimageCompressed, iImageX++, iImageY, 
			XGetPixel(pimageInput, i, j));
	      rWidthThreshold -= rCompression;
	    }
	}
      if (rHeightThreshold>rCompression)
	{
	  iImageY++;
	  rHeightThreshold -= rCompression;
	}
    } 

  /* Destroy the old image */
  XDestroyImage(pimageInput);
  
  return(pimageCompressed);
}
