/****************************************************************************
 * This module is based on Twm, but has been siginificantly modified 
 * by Rob Nation (nation@rocket.sanders.lockheed.com 
 ****************************************************************************/
/*****************************************************************************/
/**       Copyright 1988 by Evans & Sutherland Computer Corporation,        **/
/**                          Salt Lake City, Utah                           **/
/**  Portions Copyright 1989 by the Massachusetts Institute of Technology   **/
/**                        Cambridge, Massachusetts                         **/
/**                                                                         **/
/**                           All Rights Reserved                           **/
/**                                                                         **/
/**    Permission to use, copy, modify, and distribute this software and    **/
/**    its documentation  for  any  purpose  and  without  fee is hereby    **/
/**    granted, provided that the above copyright notice appear  in  all    **/
/**    copies and that both  that  copyright  notice  and  this  permis-    **/
/**    sion  notice appear in supporting  documentation,  and  that  the    **/
/**    names of Evans & Sutherland and M.I.T. not be used in advertising    **/
/**    in publicity pertaining to distribution of the  software  without    **/
/**    specific, written prior permission.                                  **/
/**                                                                         **/
/**    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD    **/
/**    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-    **/
/**    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR    **/
/**    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-    **/
/**    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA    **/
/**    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER    **/
/**    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE    **/
/**    OR PERFORMANCE OF THIS SOFTWARE.                                     **/
/*****************************************************************************/


/***********************************************************************
 *
 * fvwm menu code
 *
 ***********************************************************************/

#include <stdio.h>
#include <signal.h>
#include <strings.h>
#include <X11/Xos.h>
#include "fvwm.h"
#include "menus.h"
#include "misc.h"
#include "parse.h"
#include "screen.h"

extern XEvent Event;
extern int BlockEnterLeave;

int RootFunction = (int)NULL;
MenuRoot *ActiveMenu = NULL;		/* the active menu */
MenuItem *ActiveItem = NULL;		/* the active menu item */
int WindowMoved = FALSE;
int menuFromFrameOrWindowOrTitlebar = FALSE;

extern int menu_on,move_on;

/* Globals used to keep track of whether the mouse has moved during
   a resize function. */
int ResizeOrigX;
int ResizeOrigY;

Cursor LastCursor;

extern char *Action;
extern int Context;
extern FvwmWindow *ButtonWindow, *Tmp_win;
extern XEvent Event, ButtonEvent;
void flush_expose();

/***********************************************************************
 *
 *  Procedure:
 *	PaintEntry - draws a single entry in a poped up menu
 *
 ***********************************************************************/
void PaintEntry(MenuRoot *mr, MenuItem *mi)
{
  int y_offset,text_y,y;
  
  y_offset = mi->item_num * Scr.EntryHeight;
  text_y = y_offset + Scr.StdFont.y;

  if ((mi->state)&&(mi->func != F_TITLE))
    {
      XSetForeground(dpy, Scr.NormalGC, Scr.StdColors.fore);
      XFillRectangle(dpy, mr->w, Scr.NormalGC, 0, y_offset,
		     mr->width, Scr.EntryHeight);
      FB(Scr.StdColors.back, Scr.StdColors.fore); 
      XDrawString(dpy, mr->w, Scr.NormalGC, mi->x,
		  text_y, mi->item, mi->strlen);
    }
  else
    {
      XSetForeground(dpy, Scr.NormalGC, Scr.StdColors.back);
      XFillRectangle(dpy, mr->w, Scr.NormalGC, 0, y_offset,
		     mr->width, Scr.EntryHeight);
      FB(Scr.StdColors.fore, Scr.StdColors.back);
      XDrawString(dpy, mr->w, Scr.NormalGC, mi->x,
		  text_y, mi->item, mi->strlen);
      if(mi->func == F_TITLE)
	{
	  /* now draw the dividing line */
	  y = ((mi->item_num+1) * Scr.EntryHeight)-1;
	  XDrawLine(dpy, mr->w, Scr.NormalGC, 0, y, mr->width, y);
	}
    }
  return;
}
    

/***********************************************************************
 *
 *  Procedure:
 *	PaintMenu - draws the entire menu
 *
 ***********************************************************************/
void PaintMenu(MenuRoot *mr, XEvent *e)
{
  MenuItem *mi;
  
  for (mi = mr->first; mi != NULL; mi = mi->next)
    {
      int y_offset = mi->item_num * Scr.EntryHeight;
      /* be smart about handling the expose, redraw only the entries
       * that we need to
       */
      if (e->xexpose.y < (y_offset + Scr.EntryHeight) &&
	  (e->xexpose.y + e->xexpose.height) > y_offset)
	{
	  PaintEntry(mr, mi);
	}
    }
  XSync(dpy, 0);
  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	Updates menu display to reflect the highlighted item
 *
 ***********************************************************************/
void UpdateMenu()
{
  MenuItem *mi;
  int i, x, y, x_root, y_root, entry;
  
  while (TRUE)
    {
      /* block until there is an event */
      if (!menuFromFrameOrWindowOrTitlebar) 
	{
	  XMaskEvent(dpy,
		     ButtonPressMask | ButtonReleaseMask |
		     EnterWindowMask | ExposureMask |
		     VisibilityChangeMask | LeaveWindowMask |
		     ButtonMotionMask, &Event);
	}
      if (Event.type == MotionNotify) 
	{
	  /* discard any extra motion events before a release */
	  while((XCheckMaskEvent(dpy,ButtonMotionMask|ButtonReleaseMask,
				 &Event))&&(Event.type != ButtonRelease));

	}
      DispatchEvent (1);
      
      /* A button release means we're done */
      if (Event.type == ButtonRelease || Cancel) 
	{
	  menuFromFrameOrWindowOrTitlebar = FALSE;
	  return;
	}
      if (Event.type == MotionNotify)
	{
	  XQueryPointer( dpy, ActiveMenu->w, &JunkRoot, &JunkChild,
			&x_root, &y_root, &x, &y, &JunkMask);
	  /* look for the entry that the mouse is in */
	  entry = y / Scr.EntryHeight;
	  mi = ActiveMenu->first;
	  i=0;
	  while((i!=entry)&&(mi!=NULL))
	    {
	      i++;
	      mi=mi->next;
	    }
	  /* if we weren't on the active entry, let's turn the old
	   * active one off */
	  if ((ActiveItem)&&(ActiveItem->func != F_TITLE)&&(mi!=ActiveItem))
	    {
	      ActiveItem->state = 0;
	      PaintEntry(ActiveMenu, ActiveItem);
	    }
	  
	  /* if we weren't on the active item, change the active item
	   * and turn it on */
	  if ((mi!=ActiveItem)&&(mi != NULL))
	    {
	      if (mi->func != F_TITLE && !mi->state)
		{
		  mi->state = 1;
		  PaintEntry(ActiveMenu, mi);
		}
	    }
	  ActiveItem = mi;
	}
      XFlush(dpy);
    }
  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	PopUpMenu - pop up a pull down menu
 *
 *  Inputs:
 *	menu	- the root pointer of the menu to pop up
 *	x, y	- location of upper left of menu
 *      center	- whether or not to center horizontally over position
 *
 ***********************************************************************/
Bool PopUpMenu (MenuRoot *menu, int x, int y)
{
  if ((!menu)||(menu->w == None)||(menu->items == 0))
    return False;
  
  menu_on = 1;
  InstallRootColormap();

  
  ActiveMenu = menu;
  
  x -= (menu->width / 2);
  y -= (Scr.EntryHeight / 2);
  
  /* clip to screen */
  if (x + menu->width > Scr.MyDisplayWidth) 
    x = Scr.MyDisplayWidth - menu->width;
  if (x < 0) x = 0;

  if (y + menu->height > Scr.MyDisplayHeight) 
    y = Scr.MyDisplayHeight - menu->height;
  if (y < 0) y = 0;

  XMoveWindow(dpy, menu->w, x, y);
  XMapRaised(dpy, menu->w);
  XGrabPointer(dpy, Scr.Root, True,
	       ButtonPressMask | ButtonReleaseMask |
	       ButtonMotionMask | PointerMotionHintMask,
	       GrabModeAsync, GrabModeAsync,
	       menu->w, Scr.MenuCursor, CurrentTime);
  XSync(dpy, 0);
  return True;
}


/***********************************************************************
 *
 *  Procedure:
 *	PopDownMenu - unhighlight the current menu selection and
 *		take down the menus
 *
 ***********************************************************************/
void PopDownMenu()
{
  if (ActiveMenu == NULL)
    return;
  
  menu_on = 0;
  if (ActiveItem)
    {
      ActiveItem->state = 0;
      PaintEntry(ActiveMenu, ActiveItem);
    }
  
  XUnmapWindow(dpy, ActiveMenu->w);
  UninstallRootColormap();
  
  XFlush(dpy);
  ActiveMenu = NULL;
  ActiveItem = NULL;
  if (Context & (C_WINDOW | C_FRAME | C_TITLE | C_SIDEBAR))
    menuFromFrameOrWindowOrTitlebar = TRUE;
}

/***********************************************************************
 *
 *  Procedure:
 *	Decide whether or not a window is part of an Fvwm-controlled window
 *
 ***********************************************************************/
static Bool belongs_to_fvwm_window (FvwmWindow *t, Window w)
{
  if (!t) return False;
  
  if (w == t->frame || w == t->title_w || w == t->icon_w ||
      w==t->left_side_w || w == t->right_side_w || w== t->bottom_w ||
      w == t->w)
    return True;
  
  return False;
}


/***********************************************************************
 *
 *  Procedure:
 *	ExecuteFunction - execute a fvwm root function
 *
 *  Inputs:
 *	func	- the function to execute
 *	action	- the menu action to execute 
 *	w	- the window to execute this function on
 *	tmp_win	- the fvwm window structure
 *	event	- the event that caused the function
 *	context - the context in which the button was pressed
 *
 *  Returns:
 *	TRUE if should continue with remaining actions else FALSE to abort
 *
 ***********************************************************************/
void ExecuteFunction(int func,char *action, Window w, FvwmWindow *tmp_win, 
		    XEvent *eventp, int context,int val1, int val2)
{
  FvwmWindow *t;

  RootFunction = (int)NULL;

  if (Cancel)
    return;
  
  switch (func)
    {
    case F_NOP:
    case F_TITLE:
      break;
    default:
      XGrabPointer(dpy, Scr.Root, True,
		   ButtonPressMask | ButtonReleaseMask,
		   GrabModeAsync, GrabModeAsync,
		   Scr.Root, Scr.WaitCursor, CurrentTime);
      break;
    }
  
  switch (func)
    {
    case F_NOP:
    case F_TITLE:
      break;
      
    case F_BEEP:
      XBell(dpy, 0);
      break;
      
    case F_RESIZE:
      if (DeferExecution(context, func, Scr.MoveCursor))
	return;
      PopDownMenu();
      if(resize_window(eventp,w,tmp_win))
	return;
      break;
      
    case F_MOVE:
      if (DeferExecution(context, func, Scr.MoveCursor))
	return;
      move_on = 1;
      PopDownMenu();
      if(move_window(eventp,w,tmp_win,context))
	{
	  move_on = 0;
	  return;
	}
      move_on = 0;
      break;
      
    case F_SCROLL:
      MoveViewport(Scr.Vx + val1, Scr.Vy+val2);
      break;
    case F_PANNER:
      show_panner();
      break;
    case F_ICONIFY:
      if (DeferExecution(context, func, Scr.SelectCursor))
	return;
      
      if (tmp_win->icon)
	DeIconify(tmp_win);
      else
	Iconify(tmp_win, eventp->xbutton.x_root-5,eventp->xbutton.y_root-5);
      break;
      
    case F_RAISE:
      if (DeferExecution(context, func, Scr.SelectCursor))
	return;
      
      if (w == tmp_win->icon_w )
	XRaiseWindow(dpy, tmp_win->icon_w);
      else if(tmp_win->title_height != 0)
	XRaiseWindow(dpy, tmp_win->frame);
      else
	XRaiseWindow(dpy, tmp_win->w);
      if (LookInList(Scr.OnTop,tmp_win->name, &tmp_win->class))
	tmp_win->flags |= ONTOP;

      KeepOnTop();
      break;
      
    case F_LOWER:
      if (DeferExecution(context, func, Scr.SelectCursor))
	return;
      
      if (w == tmp_win->icon_w)
	XLowerWindow(dpy, tmp_win->icon_w);
      else if(tmp_win->title_height != 0)
	XLowerWindow(dpy, tmp_win->frame);
      else
	XLowerWindow(dpy, tmp_win->w);

      tmp_win->flags &= ~ONTOP;
      break;
      
    case F_DESTROY:
      if (DeferExecution(context, func, Scr.DestroyCursor))
	return;
      
      XKillClient(dpy, tmp_win->w);
      break;
      
    case F_DELETE:
      if (DeferExecution(context, func, Scr.DestroyCursor))
	return;
      
      if (tmp_win->protocols & DoesWmDeleteWindow)
	send_clientmessage (tmp_win->w, _XA_WM_DELETE_WINDOW, lastTimestamp);
      else
	XBell (dpy, 0);
      break;
      
    case F_EXEC:
      PopDownMenu();
      XUngrabServer (dpy);
      XSync (dpy, 0);
      system(action);
      break;
      
    case F_REFRESH:
      {
	XSetWindowAttributes attributes;
	unsigned long valuemask;
	
	valuemask = (CWBackPixel);
	attributes.background_pixel = Scr.StdColors.fore;
	attributes.backing_store = NotUseful;
	w = XCreateWindow (dpy, Scr.Root, 0, 0,
			   (unsigned int) Scr.MyDisplayWidth,
			   (unsigned int) Scr.MyDisplayHeight,
			   (unsigned int) 0,
			   CopyFromParent, (unsigned int) CopyFromParent,
			   (Visual *) CopyFromParent, valuemask,
			   &attributes);
	XMapWindow (dpy, w);
	XDestroyWindow (dpy, w);
	XFlush (dpy);
      }
      break;
      
    case F_QUIT:
      Done();
      break;
    }
  
  if (ButtonPressed == -1) XUngrabPointer(dpy, CurrentTime);
  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	DeferExecution - defer the execution of a function to the
 *	    next button press if the context is C_ROOT
 *
 *  Inputs:
 *	context	- the context in which the mouse button was pressed
 *	func	- the function to defer
 *	cursor	- the cursor to display while waiting
 *
 ***********************************************************************/
int DeferExecution(int context,int func, Cursor cursor)
{
  if (context & C_ROOT)
    {
      LastCursor = cursor;
      XGrabPointer(dpy, Scr.Root, True,
		   ButtonPressMask | ButtonReleaseMask,
		   GrabModeAsync, GrabModeAsync,
		   Scr.Root, cursor, CurrentTime);
      
      RootFunction = func;
      
      return (TRUE);
    }
  
  return (FALSE);
}

/***********************************************************************
 *
 *  Procedure:
 *	FocusOnRoot - put input focus on the root window
 *
 ***********************************************************************/
void FocusOnRoot()
{
  XSetInputFocus (dpy, PointerRoot, RevertToPointerRoot, lastTimestamp);
  if (Scr.Focus != NULL)
    SetBorder (Scr.Focus, False,False);
  InstallWindowColormaps(0, &Scr.FvwmRoot);
  Scr.Focus = NULL;
}


/***********************************************************************
 *
 *  Procedure:
 *	DeIconify a window
 *
 ***********************************************************************/
void DeIconify(FvwmWindow *tmp_win)
{
  FvwmWindow *t;
  
  tmp_win->mapped = TRUE;
  tmp_win->flags |= MAPPED;
  if(tmp_win->title_height)
    {
      XMapWindow(dpy, tmp_win->w);
      XMapRaised(dpy, tmp_win->frame);
    }
  else
    XMapRaised(dpy, tmp_win->w);
  SetMapStateProp(tmp_win, NormalState);
  
  if (tmp_win->icon_w) 
    XUnmapWindow(dpy, tmp_win->icon_w);
  tmp_win->icon = FALSE;
  tmp_win->flags &= ~ICON;
  
  
  /* now de-iconify transients */
  for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
    {
      if (t->transient && t->transientfor == tmp_win->w)
	{
	  t->mapped = TRUE;
	  t->flags |= MAPPED;
	  if(t->title_height != 0)
	    {
	      XMapRaised(dpy, t->frame);
	      XMapWindow(dpy, t->w);
	    }
	  else
	    XMapRaised(dpy, t->w);
	  SetMapStateProp(t, NormalState);
	  
	  if (t->icon_w) 
	    XUnmapWindow(dpy, t->icon_w);
	  t->icon = FALSE;
	  t->flags &= ~ICON;
	}
    }
  KeepOnTop();
  XSync (dpy, 0);
  return;
}


void KeepOnTop()
{
  FvwmWindow *t;

  /* re-raise on ONTOP windows */
  for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
    {
      if(t->flags & ONTOP)
	{
	  if(t->icon_w)
	    XRaiseWindow(dpy,t->icon_w);
	  if(t->title_height)
	    XRaiseWindow(dpy,t->frame);
	  else
	    XRaiseWindow(dpy,t->w);
	}
    }
}

void Iconify(FvwmWindow *tmp_win, int def_x, int def_y)
{
  FvwmWindow *t;
  XWindowAttributes winattrs;
  unsigned long eventMask;
  
  if (tmp_win->icon_w == (int)NULL)
    {
      CreateIconWindow(tmp_win, def_x, def_y);
    }
  
  XMapRaised(dpy, tmp_win->icon_w);
  
  XGetWindowAttributes(dpy, tmp_win->w, &winattrs);
  eventMask = winattrs.your_event_mask;
  
  /* iconify transients first */
  for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
    {
      if (t->transient && t->transientfor == tmp_win->w)
	{
	  /*
	   * Prevent the receipt of an UnmapNotify, since that would
	   * cause a transition to the Withdrawn state.
	   */
	  t->mapped = FALSE;
	  t->flags &= ~MAPPED;
	  XSelectInput(dpy, t->w, eventMask & ~StructureNotifyMask);
	  XUnmapWindow(dpy, t->w);
	  XSelectInput(dpy, t->w, eventMask);
	  if(t->title_height)
	    XUnmapWindow(dpy, t->frame);
	  if (t->icon_w)
	    XUnmapWindow(dpy, t->icon_w);
	  SetMapStateProp(t, IconicState);
	  SetBorder (t, False,False);
	  if (t == Scr.Focus)
	    {
	      XSetInputFocus (dpy, PointerRoot, RevertToPointerRoot, 
			      lastTimestamp);
	      Scr.Focus = NULL;
	    }
	  t->icon = TRUE;
	  t->flags |= ICON;
	}
    } 
  
  /*
   * Prevent the receipt of an UnmapNotify, since that would
   * cause a transition to the Withdrawn state.
   */
  tmp_win->mapped = FALSE;
  tmp_win->flags &= ~MAPPED;
  XSelectInput(dpy, tmp_win->w, eventMask & ~StructureNotifyMask);
  XUnmapWindow(dpy, tmp_win->w);
  XSelectInput(dpy, tmp_win->w, eventMask);
  if(tmp_win->title_height)
    XUnmapWindow(dpy, tmp_win->frame);
  SetMapStateProp(tmp_win, IconicState);
  
  SetBorder (tmp_win, False,False);
  if (tmp_win == Scr.Focus)
    {
      XSetInputFocus (dpy, PointerRoot, RevertToPointerRoot, 
		      lastTimestamp);
      Scr.Focus = NULL;
    }
  tmp_win->icon = TRUE;
  tmp_win->flags |= ICON;

  KeepOnTop();
  XSync (dpy, 0);
  return;
}


void SetMapStateProp(FvwmWindow *tmp_win, int state)
{
  unsigned long data[2];		/* "suggested" by ICCCM version 1 */
  
  data[0] = (unsigned long) state;
  data[1] = (unsigned long) tmp_win->icon_w;
  
  XChangeProperty (dpy, tmp_win->w, _XA_WM_STATE, _XA_WM_STATE, 32, 
		   PropModeReplace, (unsigned char *) data, 2);
  return;
}


void SetBorder (FvwmWindow *t, Bool onoroff,Bool force)
{
  int y;
  XEvent dummy;
  static FvwmWindow *last_window;
  extern FvwmWindow *UnHighLight_win;

  if((XCheckTypedWindowEvent (dpy, t->frame, DestroyNotify, &dummy))||
      (XCheckTypedWindowEvent (dpy, t->title_w, DestroyNotify, &dummy))||
      (XCheckTypedWindowEvent (dpy, t->left_side_w, DestroyNotify, &dummy))||
      (XCheckTypedWindowEvent (dpy, t->right_side_w, DestroyNotify, &dummy))||
      (XCheckTypedWindowEvent (dpy, t->bottom_w, DestroyNotify, &dummy))||
      (XCheckTypedWindowEvent (dpy, t->w, DestroyNotify, &dummy)))
    {
      Destroy(t);
      return;
    }
  
  if(!onoroff)
    if(UnHighLight_win == t)
      UnHighLight_win = NULL;

  if(t->title_w)
    {
      if (onoroff) 
	{
	  /* don't re-draw just for kicks */
	  if((!force)&&(last_window == t))
	     return;
	  last_window = t;

	  Gcv.stipple = Scr.gray;
	  FB(Scr.HiColors.fore,Scr.HiColors.back);
	  XSetWindowBackground(dpy,t->frame,Scr.HiColors.back);
	  XSetWindowBackground(dpy,t->title_w,Scr.HiColors.back);
	  XSetWindowBackground(dpy,t->left_side_w,Scr.HiColors.back);
	  XSetWindowBackground(dpy,t->right_side_w,Scr.HiColors.back);
	  XSetWindowBackground(dpy,t->bottom_w,Scr.HiColors.back);
	}
      else
	{
	  /* don't re-draw just for kicks */
	  if((!force)&&(last_window != t))
	     return;
	  if(last_window == t) last_window = (FvwmWindow *)0;
	  Gcv.stipple = Scr.light_gray;	  
	  FB(Scr.StdColors.fore,Scr.StdColors.back);
	  XSetWindowBackground(dpy,t->frame,Scr.StdColors.back);
	  XSetWindowBackground(dpy,t->title_w,Scr.StdColors.back);
	  XSetWindowBackground(dpy,t->left_side_w,Scr.StdColors.back);
	  XSetWindowBackground(dpy,t->right_side_w,Scr.StdColors.back);
	  XSetWindowBackground(dpy,t->bottom_w,Scr.StdColors.back);
	}
      XClearWindow(dpy,t->frame);
      XClearWindow(dpy,t->title_w);
      XClearWindow(dpy,t->left_side_w);
      XClearWindow(dpy,t->right_side_w);
      XClearWindow(dpy,t->bottom_w);
      if(Scr.d_depth < 2)
	{
	  Gcv.fill_style = FillOpaqueStippled;
	  XChangeGC(dpy,Scr.NormalGC,GCFillStyle|GCStipple,&Gcv);
	  XFillRectangle(dpy,t->bottom_w,Scr.NormalGC,1,1,
			 t->frame_width ,t->frame_height);
	  XFillRectangle(dpy,t->left_side_w,Scr.NormalGC,1,1,
			 t->frame_width ,t->frame_height);
	  XFillRectangle(dpy,t->right_side_w,Scr.NormalGC,1,1,
			 t->frame_width ,t->frame_height);
	  XFillRectangle(dpy,t->title_w,Scr.NormalGC,1,1,
			 t->frame_width ,t->frame_height);
	  XFillRectangle(dpy,t->frame,Scr.NormalGC,1,1,
			 t->frame_width ,t->frame_height);
	 }
      Gcv.fill_style = FillSolid;
      XChangeGC(dpy,Scr.NormalGC,GCFillStyle,&Gcv);
      FB(Scr.BWColors.fore,Scr.BWColors.back);
      XDrawLine(dpy,t->bottom_w,Scr.NormalGC,1,t->boundary_width-2,
		t->title_width,t->boundary_width-2);
      XDrawLine(dpy,t->bottom_w,Scr.NormalGC,t->title_width-1,
		1, t->title_width-1,t->boundary_width);

      XDrawLine(dpy,t->title_w,Scr.NormalGC,1,t->title_height-2,
		t->title_width,t->title_height-1);
      XDrawLine(dpy,t->title_w,Scr.NormalGC,t->title_width-1,
		1, t->title_width-1,t->title_height);
      
      y= t->frame_height - t->title_height - 2*t->corner_width +
	t->boundary_width - 2*t->frame_bw-1;
      
      XDrawLine(dpy,t->left_side_w,Scr.NormalGC,1,y,t->boundary_width-1,y);
      XDrawLine(dpy,t->left_side_w,Scr.NormalGC,t->boundary_width-2,
		1, t->boundary_width-2,y);
      
      XDrawLine(dpy,t->right_side_w,Scr.NormalGC,1,y,t->boundary_width-1,y);
      XDrawLine(dpy,t->right_side_w,Scr.NormalGC,t->boundary_width-2,
		1, t->boundary_width-2,y);
      
      XDrawLine(dpy,t->frame,Scr.NormalGC,1,t->frame_height-1,
		t->frame_width-1,t->frame_height-1);
      XDrawLine(dpy,t->frame,Scr.NormalGC,t->frame_width-1,1,
		t->frame_width-1,t->frame_height-1);

      y= t->corner_width - t->boundary_width + t->title_height-1;
      XDrawLine(dpy,t->frame,Scr.NormalGC,1,y,t->frame_width-1,y);
      XDrawLine(dpy,t->frame,Scr.NormalGC,t->boundary_width-1,
		t->title_height-2,t->frame_width - t->boundary_width+1,
		t->title_height-2);
      XDrawLine(dpy,t->frame,Scr.NormalGC,t->boundary_width-2,
		t->title_height-2,t->boundary_width-2,
		t->frame_height - t->boundary_width+1);
      XDrawLine(dpy,t->frame,Scr.NormalGC,t->corner_width-2,
		0,t->corner_width-2,t->frame_height-1);
      FB(Scr.BWColors.back,Scr.BWColors.back);

      XDrawLine(dpy,t->frame,Scr.NormalGC,t->boundary_width-1,
		t->frame_height - t->boundary_width+1,
		t->frame_width - t->boundary_width+1,
		t->frame_height-t->boundary_width+1);
      XDrawLine(dpy,t->frame,Scr.NormalGC,0,t->frame_height-t->corner_width,
		t->frame_width,t->frame_height - t->corner_width);
      XDrawLine(dpy,t->frame,Scr.NormalGC,t->frame_width-t->corner_width+1,
		0,t->frame_width - t->corner_width+1,t->frame_height);
      XDrawLine(dpy,t->frame,Scr.NormalGC,
		t->frame_width - t->boundary_width + 1,
		t->title_height-1,t->frame_width - t->boundary_width + 1,
		t->frame_height-t->boundary_width);
      if(Scr.d_depth >=2)
	{
	  XDrawLine(dpy,t->frame,Scr.NormalGC,0,0,t->frame_width,0);
	  XDrawLine(dpy,t->frame,Scr.NormalGC,0,0,0,t->frame_height);

	  XDrawLine(dpy,t->left_side_w,Scr.NormalGC,0,0,t->frame_width,0);
	  XDrawLine(dpy,t->left_side_w,Scr.NormalGC,0,0,0,t->frame_height);

	  XDrawLine(dpy,t->right_side_w,Scr.NormalGC,0,0,t->frame_width,0);
	  XDrawLine(dpy,t->right_side_w,Scr.NormalGC,0,0,0,t->frame_height);

	  XDrawLine(dpy,t->bottom_w,Scr.NormalGC,0,0,t->frame_width,0);
	  XDrawLine(dpy,t->bottom_w,Scr.NormalGC,0,0,0,t->frame_height);
	}

      SetTitleBar(t,onoroff);
      FB(Scr.StdColors.fore,Scr.StdColors.back);
      flush_expose (t->frame);
      flush_expose (t->right_side_w);
      flush_expose (t->left_side_w);
      flush_expose (t->bottom_w);  
    }
  else      /* no decorative border */
    {
      if(onoroff)
	{
	  if(Scr.d_depth<2)
	    XSetWindowBorder (dpy, t->w, Scr.HiColors.fore);
	  else
	    XSetWindowBorder (dpy, t->w, Scr.HiColors.back);
	  XSync(dpy,0);
	}
      else
	{
	  XSync(dpy,0);
	  if(Scr.d_depth<2)
	    XSetWindowBorderPixmap (dpy, t->w, Scr.border_gray);
	  else
	    XSetWindowBorder (dpy, t->w, Scr.StdColors.back);	    
	  XSync(dpy,0);
	}
    }
}

void SetTitleBar (FvwmWindow *t,Bool onoroff)
{
  int hor_off;
  int w;
  XEvent dummy;

  if(t->title_height == 0)
    return;
  if((XCheckTypedWindowEvent (dpy, t->frame, DestroyNotify, &dummy))||
      (XCheckTypedWindowEvent (dpy, t->title_w, DestroyNotify, &dummy))||
      (XCheckTypedWindowEvent (dpy, t->left_side_w, DestroyNotify, &dummy))||
      (XCheckTypedWindowEvent (dpy, t->right_side_w, DestroyNotify, &dummy))||
      (XCheckTypedWindowEvent (dpy, t->bottom_w, DestroyNotify, &dummy))||
      (XCheckTypedWindowEvent (dpy, t->w, DestroyNotify, &dummy)))
    {
      Destroy(t);
      return;
    }

  if (onoroff) 
    {
      Gcv.stipple = Scr.gray;
      FB(Scr.BWColors.fore,Scr.HiColors.back);
      XSetWindowBackground(dpy,t->title_w,Scr.HiColors.back);
    }
  else
    {
      Gcv.stipple = Scr.light_gray;	  
      FB(Scr.BWColors.fore,Scr.StdColors.back);
      XSetWindowBackground(dpy,t->title_w,Scr.StdColors.back);
    }
  XClearWindow(dpy,t->title_w);
  if(Scr.d_depth < 2)
    {
        Gcv.fill_style = FillOpaqueStippled;
	XChangeGC(dpy,Scr.NormalGC,GCFillStyle|GCStipple,&Gcv);

	XFillRectangle(dpy,t->title_w,Scr.NormalGC,1,1,
		       t->frame_width ,t->frame_height);
      }

  Gcv.fill_style = FillSolid;
  XChangeGC(dpy,Scr.NormalGC,GCFillStyle,&Gcv);
  
  XDrawLine(dpy,t->title_w,Scr.NormalGC,1,t->title_height-2,
	    t->title_width,t->title_height-2);
  XDrawLine(dpy,t->title_w,Scr.NormalGC,t->title_width-1,
	    1, t->title_width-1,t->title_height);
  
  w=XTextWidth(Scr.StdFont.font,t->name,strlen(t->name));
  hor_off = (t->title_width - w)/2;

  if(Scr.d_depth<2)
    {
      XClearArea(dpy,t->title_w,hor_off - 4, 0, w+8,t->title_height,False);
  
      XDrawLine(dpy,t->title_w,Scr.NormalGC,hor_off-4,0,hor_off-4,
		t->title_height);
      XDrawLine(dpy,t->title_w,Scr.NormalGC,hor_off-5,1,hor_off-5,
		t->title_height);
      XDrawLine(dpy,t->title_w,Scr.NormalGC,hor_off+w+2,0,hor_off+w+2,
		t->title_height);
    }
  if (onoroff)
    {
      FB(Scr.HiColors.fore,Scr.HiColors.back);  
    }
  else
    {
      FB(Scr.StdColors.fore,Scr.StdColors.back);
    }
  XDrawString (dpy, t->title_w, Scr.NormalGC,hor_off, Scr.StdFont.y, 
	       t->name, strlen(t->name));

  if(Scr.d_depth >=2)
    {
      FB(Scr.BWColors.back,Scr.BWColors.fore); 
      XDrawLine(dpy,t->title_w,Scr.NormalGC,0,0,t->frame_width,0);
      XDrawLine(dpy,t->title_w,Scr.NormalGC,0,0,0,t->frame_height);
    }
  FB(Scr.StdColors.fore,Scr.StdColors.back);
  flush_expose (t->title_w);
}

/*
 * ICCCM Client Messages - Section 4.2.8 of the ICCCM dictates that all
 * client messages will have the following form:
 *
 *     event type	ClientMessage
 *     message type	_XA_WM_PROTOCOLS
 *     window		tmp->w
 *     format		32
 *     data[0]		message atom
 *     data[1]		time stamp
 */
void send_clientmessage (Window w, Atom a, Time timestamp)
{
  XClientMessageEvent ev;
  
  ev.type = ClientMessage;
  ev.window = w;
  ev.message_type = _XA_WM_PROTOCOLS;
  ev.format = 32;
  ev.data.l[0] = a;
  ev.data.l[1] = timestamp;
  XSendEvent (dpy, w, False, 0L, (XEvent *) &ev);
}


void CreateIconWindow(FvwmWindow *tmp_win, int def_x, int def_y)
{
  unsigned long event_mask;
  int final_x, final_y;
  
  FB(Scr.StdColors.fore, Scr.StdColors.back);
  
  tmp_win->icon_w_width = XTextWidth(Scr.StdFont.font,tmp_win->icon_name, 
				     strlen(tmp_win->icon_name));
  
  tmp_win->icon_w_width += 6;
  
  event_mask = 0;
  tmp_win->icon_w = None;
  
  tmp_win->icon_w = 
    XCreateSimpleWindow(dpy, Scr.Root, 0,0,tmp_win->icon_w_width, 
			ICON_HEIGHT, Scr.IconBorderWidth, 
			Scr.StdColors.fore, Scr.StdColors.back);
  event_mask = ExposureMask;
  
  XSelectInput (dpy, tmp_win->icon_w,
		KeyPressMask | ButtonPressMask | ButtonReleaseMask |
		event_mask);
  
  /* I need to figure out where to put the icon window now, because 
   * getting here means that I am going to make the icon visible
   */
  if (tmp_win->wmhints &&
      tmp_win->wmhints->flags & IconPositionHint)
    {
      final_x = tmp_win->wmhints->icon_x;
      final_y = tmp_win->wmhints->icon_y;
    }
  else
    {
      final_x = def_x;
      final_y = def_y;
    }
  
  if (final_x > Scr.MyDisplayWidth)
    final_x = Scr.MyDisplayWidth - tmp_win->icon_w_width -
      (2 * Scr.IconBorderWidth);
  
  if (final_y > Scr.MyDisplayHeight)
    final_y = Scr.MyDisplayHeight - 
      Scr.StdFont.height - 4 - (2 * Scr.IconBorderWidth);
  
  XMoveWindow(dpy, tmp_win->icon_w, final_x, final_y);
  tmp_win->icon_x_loc = final_x;
  tmp_win->icon_y_loc = final_y;
  XMapSubwindows(dpy, tmp_win->icon_w);
  XSaveContext(dpy, tmp_win->icon_w, FvwmContext, (caddr_t)tmp_win);
  XDefineCursor(dpy, tmp_win->icon_w, Scr.IconCursor);
  return;
}

Bool move_window(XEvent *eventp,Window w,FvwmWindow *tmp_win,int context)
{
  Window rootw;
  int origX, origY;
  Bool fromtitlebar = False;
  
  rootw = eventp->xbutton.root;
  
  BlockEnterLeave = 1;
  
  XGrabServer(dpy);
  XGrabPointer(dpy, eventp->xbutton.root, True,
	       ButtonPressMask | ButtonReleaseMask |
	       ButtonMotionMask | PointerMotionMask, 
	       GrabModeAsync, GrabModeAsync,
	       Scr.Root, Scr.MoveCursor, CurrentTime);
  
  if ((context & C_ICON) && tmp_win->icon_w)
    {
      w = tmp_win->icon_w;
      DragX = eventp->xbutton.x;
      DragY = eventp->xbutton.y;
    }
  else if (w != tmp_win->icon_w)
    {
      if(Tmp_win->title_height)
	{
	  XTranslateCoordinates(dpy, w, tmp_win->frame,
				eventp->xbutton.x, eventp->xbutton.y, 
				&DragX, &DragY, &JunkChild);
	  w = tmp_win->frame;
	}
      else
	{
	  DragX = eventp->xbutton.x;
	  DragY = eventp->xbutton.y;
	  w = tmp_win->w;
	}
      
    }
  
  DragWindow = None;
  
  XGetGeometry(dpy, w, &JunkRoot, &origDragX, &origDragY,
	       (unsigned int *)&DragWidth, (unsigned int *)&DragHeight, 
	       &JunkBW,  &JunkDepth);
  
  origX = eventp->xbutton.x_root;
  origY = eventp->xbutton.y_root;
  CurrentDragX = origDragX;
  CurrentDragY = origDragY;
  
  InstallRootColormap();
  DragWindow = w;
  
  /*
   * see if this is being done from the titlebar
   */
  fromtitlebar = belongs_to_fvwm_window (tmp_win, eventp->xbutton.window);
  
  if (menuFromFrameOrWindowOrTitlebar) 
    {
      /* warp the pointer to the middle of the window */
      XWarpPointer(dpy, None, Scr.Root, 0, 0, 0, 0, 
		   origDragX + DragWidth / 2, 
		   origDragY + DragHeight / 2);
      XFlush(dpy);
    }
  
  while (TRUE)
    {
      long releaseEvent = menuFromFrameOrWindowOrTitlebar ? 
	ButtonPress : ButtonRelease;
      long movementMask = menuFromFrameOrWindowOrTitlebar ?
	PointerMotionMask : ButtonMotionMask;
      
      /* block until there is an interesting event */
      XMaskEvent(dpy, ButtonPressMask | ButtonReleaseMask |
		 EnterWindowMask | LeaveWindowMask |
		 ExposureMask | movementMask |
		 VisibilityChangeMask, &Event);
      
      /* throw away enter and leave events until release */
      if (Event.xany.type == EnterNotify ||
	  Event.xany.type == LeaveNotify) continue; 
      
      if (Event.type == MotionNotify) 
	{
	  /* discard any extra motion events before a logical release */
	  while(XCheckMaskEvent(dpy, movementMask | releaseEvent, &Event))
	    if (Event.type == releaseEvent)
	      break;
	}
      
      /* test to see if we have a second button press to abort move */
      if (!menuFromFrameOrWindowOrTitlebar)
	if (Event.type == ButtonPress && DragWindow != None) 
	  {
	    MoveOutline(Scr.Root, 0, 0, 0, 0);
	    DragWindow = None;
	  }
      
      if (fromtitlebar && Event.type == ButtonPress) 
	{
	  fromtitlebar = False;
	  CurrentDragX = origX = Event.xbutton.x_root;
	  CurrentDragY = origY = Event.xbutton.y_root;
	  if(tmp_win->title_height)
	    XTranslateCoordinates (dpy, rootw, tmp_win->frame,
				   origX, origY,
				   &DragX, &DragY, &JunkChild);
	  else
	    XTranslateCoordinates (dpy, rootw, tmp_win->w,
				   origX, origY,
				   &DragX, &DragY, &JunkChild);
	  continue;
	}
      
      DispatchEvent (2);
      
      if (Cancel)
	{
	  WindowMoved = FALSE;
	  UninstallRootColormap();
	  move_on = 0;
	  return True;	/* XXX should this be FALSE? */
	}
      if (Event.type == releaseEvent)
	{
	  MoveOutline(rootw, 0, 0, 0, 0);
	  if (menuFromFrameOrWindowOrTitlebar)
	    XMoveWindow(dpy, DragWindow, 
			Event.xbutton.x_root - DragWidth / 2,
			Event.xbutton.y_root - DragHeight / 2);
	  break;
	}
      
      /* something left to do only if the pointer moved */
      if (Event.type != MotionNotify)
	continue;
      
      XQueryPointer(dpy, rootw, &(eventp->xmotion.root), &JunkChild,
		    &(eventp->xmotion.x_root), &(eventp->xmotion.y_root),
		    &JunkX, &JunkY, &JunkMask);
      
      WindowMoved = TRUE;
      DragWindow = w;
      
      if (DragWindow != None)
	{
	  int xl, yt, w, h;
	  if (!menuFromFrameOrWindowOrTitlebar) 
	    {
	      xl = eventp->xmotion.x_root - DragX - JunkBW;
	      yt = eventp->xmotion.y_root - DragY - JunkBW;
	    }
	  else 
	    {
	      xl = eventp->xmotion.x_root - (DragWidth / 2);
	      yt = eventp->xmotion.y_root - (DragHeight / 2);
	    }		  
	  w = DragWidth + 2 * JunkBW;
	  h = DragHeight + 2 * JunkBW;
	  
	  CurrentDragX = xl;
	  CurrentDragY = yt;
	  MoveOutline(eventp->xmotion.root, xl, yt, w, h);
	}
    }
  UninstallRootColormap();
  return False;
}


Bool resize_window(XEvent *eventp,Window w,FvwmWindow *tmp_win)
{
  int origX, origY;

  BlockEnterLeave = 1;
  /* can't resize icons */
  if (w != tmp_win->icon_w) 
    {
      if(tmp_win->title_height)
	w = tmp_win->frame;
      else
	w = tmp_win->w;      

      origX = eventp->xbutton.x_root;
      origY = eventp->xbutton.y_root;
      
      /* Save pointer position so we can tell if it was moved or
	 not during the resize. */
      ResizeOrigX = origX;
      ResizeOrigY = origY;
      
      StartResize (tmp_win);
      
      do {
	XMaskEvent(dpy,
		   ButtonPressMask | ButtonReleaseMask |
		   EnterWindowMask | LeaveWindowMask |
		   ButtonMotionMask | PointerMotionMask,  &Event);
	
	if (Event.type == MotionNotify) 
	  {
	    /* discard any extra motion events before a release */
	    while(XCheckMaskEvent(dpy, ButtonMotionMask |
				  ButtonReleaseMask | PointerMotionMask,
				  &Event))
	      if (Event.type == ButtonRelease)
		break;
	  }
	DispatchEvent(1);
	
      } while (!(Event.type == ButtonRelease || Cancel));
    }
  return True;
}


void show_panner()
{
  unsigned long valuemask;
  unsigned int width,height,ww,wh;
  int window_x,window_y,wx,wy;
  XSetWindowAttributes attributes;
  Window Panner_w;
  FvwmWindow *t;
  int x,y;

  width = (Scr.VxMax + Scr.MyDisplayWidth)/32;
  height = (Scr.VyMax + Scr.MyDisplayHeight)/32;

  XSync(dpy,0);

  XQueryPointer(dpy, Scr.Root,&JunkRoot, &JunkChild,
		&x,&y,&JunkX, &JunkY, &JunkMask);

  window_x = x - Scr.Vx*width/(Scr.VxMax+Scr.MyDisplayWidth);
  if ((window_x + width) > Scr.MyDisplayWidth) 
    window_x = Scr.MyDisplayWidth - width;
  if (window_x < 0) window_x = 0;
  x = window_x + Scr.Vx*width/(Scr.VxMax + Scr.MyDisplayWidth);

  window_y = y - Scr.Vy*height/(Scr.VyMax + Scr.MyDisplayHeight);
  if ((window_y + height) > Scr.MyDisplayHeight) 
    y = Scr.MyDisplayHeight - height;
  if (window_y < 0) window_y = 0;
  y = window_y + Scr.Vy*height/(Scr.VyMax + Scr.MyDisplayHeight);
  
  InstallRootColormap();

  FB(Scr.StdColors.fore,Scr.StdColors.back);
  valuemask = (CWBackPixel | CWBorderPixel | CWEventMask);
  attributes.background_pixel = Scr.StdColors.back;
  attributes.border_pixel = Scr.StdColors.fore;
  attributes.event_mask = (ExposureMask | EnterWindowMask);
  Panner_w = XCreateWindow (dpy, Scr.Root, window_x, window_y, width,
			    height, (unsigned int) 1,
			    CopyFromParent, (unsigned int) CopyFromParent,
			    (Visual *) CopyFromParent,
			    valuemask, &attributes);
  XMapRaised(dpy, Panner_w);

  
  XSync(dpy, 0);

  XGrabPointer(dpy, Scr.Root, True,
	       ButtonPressMask | ButtonReleaseMask |
	       ButtonMotionMask | PointerMotionHintMask,
	       GrabModeAsync, GrabModeAsync,
	       Panner_w, Scr.PositionCursor, CurrentTime);
  XWarpPointer(dpy, None, Scr.Root, 0, 0, 0, 0, x, y);
  while(XCheckTypedEvent(dpy,MotionNotify,&Event));


  Event.type = 0;
  while(Event.type != ButtonRelease)
    {
      XMaskEvent(dpy,
		 ButtonPressMask | ButtonReleaseMask |
		 EnterWindowMask | ExposureMask |
		 LeaveWindowMask | ButtonMotionMask, &Event);

      DispatchEvent(1);
      switch(Event.type)
	{
	case Expose:
	  if(Event.xany.window == Panner_w)
	    {
	      XClearWindow(dpy,Panner_w);
	      for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
		{
		  if(!(t->flags & STICKY))
		     {
		       if(t->flags & ICON)
			 {
			   /* show the icon loc */
			   wx = (t->icon_x_loc + Scr.Vx)/32;
			   wy = (t->icon_y_loc + Scr.Vy)/32;
			   ww = t->icon_w_width/32;
			   wh = (ICON_HEIGHT)/32;
			 }
		       else if (t->flags & BORDER)
			 {
			   /* show the actual window */
			   wx = (t->frame_x + Scr.Vx)/32;
			   wy = (t->frame_y + Scr.Vy)/32;
			   ww = t->frame_width/32;
			   wh = t->frame_height/32;
			 }
		       else
			 {
			   /* show undecorated window loc */
			   wx = (t->attr.x + Scr.Vx)/32;
			   wy = (t->attr.y + Scr.Vy)/32;
			   ww = t->attr.width/32;
			   wh = t->attr.height/32;
			 }
		       if(ww<2)ww=2;
		       if(wh<2)wh=2;
		       XFillRectangle(dpy,Panner_w,Scr.NormalGC,wx,wy,ww ,wh);
		     }
		}
	    }
	  break;
	case MotionNotify:
	  while(XCheckTypedEvent(dpy,MotionNotify,&Event));
	  MoveViewport((Event.xmotion.x_root - window_x)*32,
		       (Event.xmotion.y_root - window_y)*32);
	  break;
	default:
	  break;
	}
    }
  XDestroyWindow(dpy,Panner_w);
  XUngrabPointer(dpy,CurrentTime);
  ButtonPressed = -1;
}
