/****************************************************************************
 * This module is based on Twm, but has been siginificantly modified 
 * by Rob Nation (nation@rocket.sanders.lockheed.com)
 ****************************************************************************/
/*****************************************************************************/
/**       Copyright 1988 by Evans & Sutherland Computer Corporation,        **/
/**                          Salt Lake City, Utah                           **/
/**  Portions Copyright 1989 by the Massachusetts Institute of Technology   **/
/**                        Cambridge, Massachusetts                         **/
/**                                                                         **/
/**                           All Rights Reserved                           **/
/**                                                                         **/
/**    Permission to use, copy, modify, and distribute this software and    **/
/**    its documentation  for  any  purpose  and  without  fee is hereby    **/
/**    granted, provided that the above copyright notice appear  in  all    **/
/**    copies and that both  that  copyright  notice  and  this  permis-    **/
/**    sion  notice appear in supporting  documentation,  and  that  the    **/
/**    names of Evans & Sutherland and M.I.T. not be used in advertising    **/
/**    in publicity pertaining to distribution of the  software  without    **/
/**    specific, written prior permission.                                  **/
/**                                                                         **/
/**    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD    **/
/**    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-    **/
/**    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR    **/
/**    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-    **/
/**    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA    **/
/**    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER    **/
/**    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE    **/
/**    OR PERFORMANCE OF THIS SOFTWARE.                                     **/
/*****************************************************************************/


/***********************************************************************
 *
 * fvwm event handling
 *
 ***********************************************************************/

#include <stdio.h>
#include <strings.h>
#include "fvwm.h"
#include <X11/Xatom.h>
#include "menus.h"
#include "misc.h"
#include "parse.h"
#include "screen.h"

void RedoIconName();
unsigned int mods_used = (ShiftMask | ControlMask | Mod1Mask);
extern int menuFromFrameOrWindowOrTitlebar;
int BlockEnterLeave = 0;

char *Action;
int Context = C_NO_CONTEXT;	/* current button press context */
FvwmWindow *ButtonWindow;	/* button press window structure */
XEvent ButtonEvent;		/* button press event */
XEvent Event;			/* the current event */
FvwmWindow *Tmp_win;		/* the current fvwm window */

/* Used in HandleEnterNotify to remove border highlight from a window 
 * that has not recieved a LeaveNotify event because of a pointer grab 
 */
FvwmWindow *UnHighLight_win = NULL;

Window DragWindow;		/* variables used in moving windows */
int origDragX;
int origDragY;
int DragX;
int DragY;
int DragWidth;
int DragHeight;
int CurrentDragX;
int CurrentDragY;

/* Vars to tell if the resize has moved. */
extern int ResizeOrigX;
extern int ResizeOrigY;

int ButtonPressed = -1;
int Cancel = FALSE;

void flush_expose();

int menu_on=0 ,resize_on=0,move_on=0, unhighlight = 0;
int last_event_type=0;
FvwmWindow *last_event_Fvwmwindow=0;
Window last_event_window=0;

/***********************************************************************
 *
 *  Procedure:
 *	InitEvents - initialize the event jump table
 *
 ***********************************************************************
 */

void InitEvents()
{
  ResizeWindow = (int)NULL;
  DragWindow = (int)NULL;
}



Time lastTimestamp = CurrentTime;	/* until Xlib does this for us */

void StashEventTime (XEvent *ev)
{
    switch (ev->type) {
      case KeyPress:
      case KeyRelease:
	lastTimestamp = ev->xkey.time;
	break;
      case ButtonPress:
      case ButtonRelease:
	lastTimestamp = ev->xbutton.time;
	break;
      case MotionNotify:
	lastTimestamp = ev->xmotion.time;
	break;
      case EnterNotify:
      case LeaveNotify:
	lastTimestamp = ev->xcrossing.time;
	break;
      case PropertyNotify:
	lastTimestamp = ev->xproperty.time;
	break;
      case SelectionClear:
	lastTimestamp = ev->xselectionclear.time;
	break;
      case SelectionRequest:
	lastTimestamp = ev->xselectionrequest.time;
	break;
      case SelectionNotify:
	lastTimestamp = ev->xselection.time;
	break;
    }
    return;
}

/***********************************************************************
 *
 *  Procedure:
 *	DispatchEvent - handle a single X event stored in global var Event
 *      type 2 is is for a call during an f.move
 *
 ***********************************************************************
 */
void DispatchEvent (int type)
{
    Window w = Event.xany.window;
    StashEventTime (&Event);

    if (XFindContext (dpy, w, FvwmContext, (caddr_t *) &Tmp_win) == XCNOENT)
      Tmp_win = NULL;
    last_event_type = Event.type;
    last_event_Fvwmwindow = Tmp_win;
    last_event_window = w;

    if (((type==2)&&((!menuFromFrameOrWindowOrTitlebar)||
		     ((menuFromFrameOrWindowOrTitlebar)&&
		      (Event.type == Expose))))||
	(type==1))
      {
	switch(Event.type)
	  {
	  case Expose:
	    HandleExpose();
	    break;
	  case DestroyNotify:
	    HandleDestroyNotify();
	    break;
	  case MapRequest:
	    HandleMapRequest();
	    break;
	  case MapNotify:
	    HandleMapNotify();
	    break;
	  case UnmapNotify:
	    HandleUnmapNotify();
	    break;
	  case MotionNotify:
	    HandleMotionNotify();
	    break;
	  case ButtonRelease:
	    HandleButtonRelease();
	    break;
	  case ButtonPress:
	    HandleButtonPress();
	    break;
	  case EnterNotify:
	    HandleEnterNotify();
	    break;
	  case LeaveNotify:
	    HandleLeaveNotify();
	    break;
	  case ConfigureRequest:
	    HandleConfigureRequest();
	    break;
	  case ClientMessage:
	    HandleClientMessage();
	    break;
	  case PropertyNotify:
	    HandlePropertyNotify();
	    break;
	  case KeyPress:
	    HandleKeyPress();
	    break;
	  default:
	    break;
	  }
	last_event_type =0;
      }
    return;
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleEvents - handle X events
 *
 ***********************************************************************
 */

void HandleEvents()
{
  while (TRUE)
    {
      WindowMoved = FALSE;
      
      XNextEvent(dpy, &Event);
      (void) DispatchEvent (1);
    }
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleKeyPress - key press event handler
 *
 ***********************************************************************
 */

void HandleKeyPress()
{
  FuncKey *key;
  unsigned int modifier;
  
  Context = C_NO_CONTEXT;
  
  if (Event.xany.window == Scr.Root)
    Context = C_ROOT;
  if (Tmp_win)
    {
      if (Event.xany.window == Tmp_win->title_w)
	Context = C_TITLE;
      if (Event.xany.window == Tmp_win->w)
	Context = C_WINDOW;
      if (Event.xany.window == Tmp_win->icon_w)
	Context = C_ICON;
      if (Event.xany.window == Tmp_win->frame)
	Context = C_FRAME;
      if ((Event.xany.window == Tmp_win->left_side_w)||
	  (Event.xany.window == Tmp_win->right_side_w)||
	  (Event.xany.window == Tmp_win->bottom_w))
	Context = C_SIDEBAR;

    }

  modifier = (Event.xkey.state & mods_used);
  for (key = Scr.FuncKeyRoot.next; key != NULL; key = key->next)
    {
      if (key->keycode == Event.xkey.keycode &&
	  key->mods == modifier &&
	  (key->cont & Context))
	{
	  /* weed out the functions that don't make sense to execute
	   * from a key press 
	   */
	  if (key->func == F_MOVE || key->func == F_RESIZE)
	    return;
	  
	  ExecuteFunction(key->func, key->action, Event.xany.window,
			  Tmp_win, &Event,Context,key->val1,key->val2);
	  XUngrabPointer(dpy, CurrentTime);
	  return;
	  
	}
    }
  
  /* if we get here, no function key was bound to the key.  Send it
   * to the client if it was in a window we know about.
   */
  if (Tmp_win)
    {
      if (Event.xany.window == Tmp_win->icon_w ||
	  Event.xany.window == Tmp_win->frame ||
	  Event.xany.window == Tmp_win->title_w ||
	  Event.xany.window == Tmp_win->right_side_w ||
	  Event.xany.window == Tmp_win->left_side_w ||
	  Event.xany.window == Tmp_win->bottom_w)
        {
	  Event.xkey.window = Tmp_win->w;
	  XSendEvent(dpy, Tmp_win->w, False, KeyPressMask, &Event);
        }
    }
}


static void free_window_names (tmp, nukename, nukeicon)
     FvwmWindow *tmp;
     Bool nukename, nukeicon;
{
  /*
   * XXX - are we sure that nobody ever sets these to another constant (check
   * fvwm windows)?
   */
  if (tmp->icon_name == tmp->name) nukename = False;
  
#define isokay(v) ((v) && (v) != NoName)
  if (nukename && isokay(tmp->name)) XFree (tmp->name);
  if (nukeicon && isokay(tmp->icon_name)) XFree (tmp->icon_name);
#undef isokay
  return;
}




/***********************************************************************
 *
 *  Procedure:
 *	HandlePropertyNotify - property notify event handler
 *
 ***********************************************************************
 */

void HandlePropertyNotify()
{
  char *prop = NULL;
  Atom actual = None;
  int actual_format;
  unsigned long nitems, bytesafter;
#define MAX_NAME_LEN 200L		/* truncate to this many */
#define MAX_ICON_NAME_LEN 200L		/* ditto */
  
  switch (Event.xproperty.atom) {
  case XA_WM_NAME:
    if (XGetWindowProperty (dpy, Tmp_win->w, Event.xproperty.atom, 0L, 
			    MAX_NAME_LEN, False, XA_STRING, &actual,
			    &actual_format, &nitems, &bytesafter,
			    (unsigned char **) &prop) != Success ||
	actual == None)
      return;
    if (!prop) prop = NoName;
    free_window_names (Tmp_win, True, False);
    
    Tmp_win->name = prop;
    
    if(!Tmp_win->icon)
      {
	if(Scr.Focus == Tmp_win)
	  SetTitleBar(Tmp_win,True);
	else
	  SetTitleBar(Tmp_win,False);
      }

    /*
     * if the icon name is NoName, set the name of the icon to be
     * the same as the window 
     */
    if (Tmp_win->icon_name == NoName) 
      {
	Tmp_win->icon_name = Tmp_win->name;
	RedoIconName();
      }
    break;
    
  case XA_WM_ICON_NAME:
    if (XGetWindowProperty (dpy, Tmp_win->w, Event.xproperty.atom, 0, 
			    MAX_ICON_NAME_LEN, False, XA_STRING, &actual,
			    &actual_format, &nitems, &bytesafter,
			    (unsigned char **) &prop) != Success ||
	actual == None)
      return;
    if (!prop) prop = NoName;
    free_window_names (Tmp_win, False, True);
    Tmp_win->icon_name = prop;
    
    RedoIconName();
    break;
    
  case XA_WM_HINTS:
    if (Tmp_win->wmhints) XFree ((char *) Tmp_win->wmhints);
    Tmp_win->wmhints = XGetWMHints(dpy, Event.xany.window);
    break;
    
  case XA_WM_NORMAL_HINTS:
    GetWindowSizeHints (Tmp_win);
    break;
    
  default:
    if (Event.xproperty.atom == _XA_WM_PROTOCOLS) 
      {
	FetchWmProtocols (Tmp_win);
	break;
      }
    break;
  }
}


/***********************************************************************
 *
 *  Procedure:
 *	RedoIconName - procedure to re-position the icon window and name
 *
 ***********************************************************************
 */

void RedoIconName()
{
  if (Tmp_win->icon_w == (int)NULL)
    return;
  
  Tmp_win->icon_w_width = XTextWidth(Scr.StdFont.font, Tmp_win->icon_name, 
				     strlen(Tmp_win->icon_name));
  
  Tmp_win->icon_w_width += 6;
  
  XResizeWindow(dpy, Tmp_win->icon_w, Tmp_win->icon_w_width,ICON_HEIGHT);
  if (Tmp_win->icon)
    XClearArea(dpy, Tmp_win->icon_w, 0, 0, 0, 0, True);
  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleClientMessage - client message event handler
 *
 ***********************************************************************
 */

void HandleClientMessage()
{
  if (Event.xclient.message_type == _XA_WM_CHANGE_STATE)
    {
      if (Tmp_win != NULL)
	{
	  if (Event.xclient.data.l[0] == IconicState && !Tmp_win->icon)
	    {
	      XEvent button;
	      
	      XQueryPointer( dpy, Scr.Root, &JunkRoot, &JunkChild,
			    &(button.xmotion.x_root),
			    &(button.xmotion.y_root),
			    &JunkX, &JunkY, &JunkMask);
	      
	      ExecuteFunction(F_ICONIFY, NULLSTR, Event.xany.window,
			      Tmp_win, &button, C_FRAME,0,0);
	      XUngrabPointer(dpy, CurrentTime);
	    }
	}
    }
}

/***********************************************************************
 *
 *  Procedure:
 *	HandleExpose - expose event handler
 *
 ***********************************************************************
 */

void HandleExpose()
{
  MenuRoot *tmp;

  if (XFindContext(dpy, Event.xany.window, MenuContext, (caddr_t *)&tmp) == 0)
    {
      PaintMenu(tmp, &Event);
      return;
    }
  if (Event.xexpose.count != 0)
    {
      return;
    }
  if (Tmp_win != NULL)
    {
      if (Event.xany.window == Tmp_win->title_w)
	{
	  if(Scr.Focus == Tmp_win)
	    SetTitleBar(Tmp_win,True);
	  else
	    SetTitleBar(Tmp_win,False);

	}
      else if((Event.xany.window == Tmp_win->right_side_w)||
	       (Event.xany.window == Tmp_win->left_side_w)||
	       (Event.xany.window == Tmp_win->bottom_w)||
	       (Event.xany.window == Tmp_win->frame))
	{
	  if(Scr.Focus == Tmp_win)
	    SetBorder(Tmp_win,True,True);
	  else
	    SetBorder(Tmp_win,False,True);
	}
      else if((Event.xany.window == Tmp_win->w)&&
	      (Tmp_win->title_height==0))
	{
	  if(Scr.Focus == Tmp_win)
	    SetBorder(Tmp_win,True,True);
	  else
	    SetBorder(Tmp_win,False,True);	
	}
      else if (Event.xany.window == Tmp_win->icon_w)
	{
	  XDrawString (dpy, Tmp_win->icon_w,
		       Scr.NormalGC,
		       ICON_X_TEXT, ICON_Y_TEXT,
		       Tmp_win->icon_name, strlen(Tmp_win->icon_name));
	  flush_expose (Event.xany.window);
	  return;
	}
    }
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleDestroyNotify - DestroyNotify event handler
 *
 ***********************************************************************
 */

void Destroy(FvwmWindow *Tmp_win)
{ 
  XEvent dummy;
  /*
   * Warning, this is also called by HandleUnmapNotify; if it ever needs to
   * look at the event, HandleUnmapNotify will have to mash the UnmapNotify
   * into a DestroyNotify.
   */
  if(Tmp_win->title_height)
    XDestroyWindow(dpy, Tmp_win->frame);

  if (Tmp_win->icon_w) 
    XDestroyWindow(dpy, Tmp_win->icon_w);

  if (Tmp_win == NULL)
    return;

  /* purge any remaining events for this window */
  while(XCheckWindowEvent(dpy,Tmp_win->w,~0L,&dummy));
  if(Tmp_win->icon_w)
    while(XCheckWindowEvent(dpy,Tmp_win->icon_w,~0L,&dummy));    
  if(Tmp_win->title_height)
    {
      while(XCheckWindowEvent(dpy,Tmp_win->frame,~0L,&dummy));
      while(XCheckWindowEvent(dpy,Tmp_win->title_w,~0L,&dummy));
      while(XCheckWindowEvent(dpy,Tmp_win->left_side_w,~0L,&dummy));
      while(XCheckWindowEvent(dpy,Tmp_win->right_side_w,~0L,&dummy));
      while(XCheckWindowEvent(dpy,Tmp_win->bottom_w,~0L,&dummy));
    }
  
  if (Tmp_win == Scr.Focus)
    {
      Scr.Focus = NULL;
      FocusOnRoot();
    }
  XDeleteContext(dpy, Tmp_win->w, FvwmContext);
  if (Tmp_win->icon_w)
    XDeleteContext(dpy, Tmp_win->icon_w, FvwmContext);
  if (Tmp_win->title_height)
    {
      XDeleteContext(dpy, Tmp_win->title_w, FvwmContext);
      XDeleteContext(dpy, Tmp_win->left_side_w, FvwmContext);
      XDeleteContext(dpy, Tmp_win->right_side_w, FvwmContext);
      XDeleteContext(dpy, Tmp_win->bottom_w, FvwmContext);
      XDeleteContext(dpy, Tmp_win->frame, FvwmContext);
    }
  Tmp_win->prev->next = Tmp_win->next;
  if (Tmp_win->next != NULL)
    Tmp_win->next->prev = Tmp_win->prev;
  free_window_names (Tmp_win, True, True);		/* 1, 2, 3 */
  if (Tmp_win->wmhints)					/* 4 */
    XFree ((char *)Tmp_win->wmhints);
  if (Tmp_win->class.res_name && Tmp_win->class.res_name != NoName)  /* 5 */
    XFree ((char *)Tmp_win->class.res_name);
  if (Tmp_win->class.res_class && Tmp_win->class.res_class != NoName) /* 6 */
    XFree ((char *)Tmp_win->class.res_class);

  if(UnHighLight_win == Tmp_win)
    UnHighLight_win= NULL;

  free((char *)Tmp_win);
  return;
}


void HandleDestroyNotify()
{
  Destroy(Tmp_win);
}




/***********************************************************************
 *
 *  Procedure:
 *	HandleMapRequest - MapRequest event handler
 *
 ***********************************************************************
 */

void
HandleMapRequest()
{
    int stat;

    Event.xany.window = Event.xmaprequest.window;
    stat = XFindContext(dpy, Event.xany.window, FvwmContext, (caddr_t *)&Tmp_win);
    if (stat == XCNOENT)
	Tmp_win = NULL;

    /* If the window has never been mapped before ... */
    if (Tmp_win == NULL)
    {
	/* Add decorations. */
	Tmp_win = AddWindow(Event.xany.window);
	if (Tmp_win == NULL)
	    return;
    }

    /* If it's not merely iconified, and we have hints, use them. */
    if ((! Tmp_win->icon) &&
	Tmp_win->wmhints && (Tmp_win->wmhints->flags & StateHint))
    {
	int state;

	state = Tmp_win->wmhints->initial_state;
	switch (state) 
	{
	    case DontCareState:
	    case NormalState:
	    case InactiveState:
		XMapWindow(dpy, Tmp_win->w);
		if(Tmp_win->title_height)
		  XMapWindow(dpy, Tmp_win->frame);
		SetMapStateProp(Tmp_win, NormalState);
		break;

	    case IconicState:
		Iconify(Tmp_win, 0, 0);
		break;
	}
    }
    /* If no hints, or currently an icon, just "deiconify" */
    else
    {
	DeIconify(Tmp_win);
    }
    KeepOnTop();
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleMapNotify - MapNotify event handler
 *
 ***********************************************************************
 */

void HandleMapNotify()
{
    if (Tmp_win == NULL)
	return;

    /*
     * Need to do the grab to avoid race condition of having server send
     * MapNotify to client before the frame gets mapped; this is bad because
     * the client would think that the window has a chance of being viewable
     * when it really isn't.
     */
    XGrabServer (dpy);
    if (Tmp_win->icon_w)
	XUnmapWindow(dpy, Tmp_win->icon_w);
    if(Tmp_win->title_height)
      {
	XMapSubwindows(dpy, Tmp_win->frame);
	XMapWindow(dpy, Tmp_win->frame);
      }
    else
      {
	XMapSubwindows(dpy, Tmp_win->w);
	XMapWindow(dpy, Tmp_win->w);    
      }

    XUngrabServer (dpy);
    XFlush (dpy);
    Tmp_win->mapped = TRUE;
    Tmp_win->icon = FALSE;
    Tmp_win->flags |= MAPPED;
    Tmp_win->flags &= ~ICON;
    KeepOnTop();
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleUnmapNotify - UnmapNotify event handler
 *
 ***********************************************************************
 */

void HandleUnmapNotify()
{
  int dstx, dsty;
  Window dumwin;
  XEvent dummy;

  /*
   * The July 27, 1988 ICCCM spec states that a client wishing to switch
   * to WithdrawnState should send a synthetic UnmapNotify with the
   * event field set to (pseudo-)root, in case the window is already
   * unmapped (which is the case for fvwm for IconicState).  Unfortunately,
   * we looked for the FvwmContext using that field, so try the window
   * field also.
   */
  if (Tmp_win == NULL)
    {
      Event.xany.window = Event.xunmap.window;
      if (XFindContext(dpy, Event.xany.window,
		       FvwmContext, (caddr_t *)&Tmp_win) == XCNOENT)
	Tmp_win = NULL;
    }
  
  if (Tmp_win == NULL || (!Tmp_win->mapped && !Tmp_win->icon))
    return;

  XGrabServer (dpy);
  if(XCheckTypedWindowEvent (dpy, Event.xunmap.window, DestroyNotify,&dummy)) 
    {
      Destroy(Tmp_win);
      XUngrabServer(dpy);
      return;
    }
      
  /*
   * The program may have unmapped the client window, from either
   * NormalState or IconicState.  Handle the transition to WithdrawnState.
   *
   * We need to reparent the window back to the root (so that fvwm exiting 
   * won't cause it to get mapped) and then throw away all state (pretend 
   * that we've received a DestroyNotify).
   */
  if (XTranslateCoordinates (dpy, Event.xunmap.window, Tmp_win->attr.root,
			     0, 0, &dstx, &dsty, &dumwin)) 
    {
      XEvent ev;
      Bool reparented;
      reparented = XCheckTypedWindowEvent (dpy, Event.xunmap.window, 
						ReparentNotify, &ev);
      SetMapStateProp (Tmp_win, WithdrawnState);
      if (reparented) 
	{
	  if (Tmp_win->old_bw) XSetWindowBorderWidth (dpy,
						      Event.xunmap.window, 
						      Tmp_win->old_bw);
	  if (Tmp_win->wmhints && (Tmp_win->wmhints->flags & IconWindowHint))
	    XUnmapWindow (dpy, Tmp_win->wmhints->icon_window);
	} 
      else
	{
	  XReparentWindow (dpy, Event.xunmap.window, Tmp_win->attr.root,
			   dstx, dsty);
	  RestoreWithdrawnLocation (Tmp_win);
	}
      XRemoveFromSaveSet (dpy, Event.xunmap.window);
      XSelectInput (dpy, Event.xunmap.window, NoEventMask);
      HandleDestroyNotify ();		/* do not need to mash event before */
    } /* else window no longer exists and we'll get a destroy notify */
  XUngrabServer (dpy);
  XFlush (dpy);
}

void MoveViewport(int newx, int newy)
{
  FvwmWindow *t;
  int deltax,deltay;

  if(newx > Scr.VxMax)
    newx = Scr.VxMax;
  if(newy > Scr.VyMax)
    newy = Scr.VyMax;
  if(newx <0)
    newx = 0;
  if(newy <0)
    newy = 0;

  deltay = Scr.Vy - newy;
  deltax = Scr.Vx - newx;
  if((deltax!=0)||(deltay!=0))
    {
      for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
	{
	  if(!(t->flags & STICKY))
	    {
	      if(t->icon_w)
		{
		  t->icon_x_loc += deltax;
		  t->icon_y_loc += deltay;
		  XMoveWindow(dpy,t->icon_w,t->icon_x_loc,t->icon_y_loc);
		}
	      if(t->title_height)
		{
		  t->frame_x += deltax;
		  t->frame_y += deltay;
		  XMoveWindow(dpy,t->frame,t->frame_x,t->frame_y);
		}
	      else
		{
		  t->frame_x += deltax;
		  t->frame_y += deltay;
		  XMoveWindow(dpy,t->w,t->frame_x,t->frame_y);
		}
	    }
	}
    }
  Scr.Vx = newx;
  Scr.Vy = newy;
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleMotionNotify - MotionNotify event handler
 *
 **********************************************************************/
void HandleMotionNotify()
{
  int delta_x, delta_y;

  if ((ResizeWindow == (Window)0)&&(DragWindow == (Window)0))
    {
      while(XCheckTypedEvent(dpy,MotionNotify,&Event));
      delta_x = 0;
      delta_y = 0;
      if(Event.xmotion.x_root >= Scr.MyDisplayWidth -3)
	delta_x = 32;
      if(Event.xmotion.y_root >= Scr.MyDisplayHeight -3)
	delta_y = 32;
      if(Event.xmotion.x_root < 3)
	delta_x = -32;
      if(Event.xmotion.y_root < 3)
	delta_y = -32;
      if((delta_x !=0)||(delta_y != 0))
	{
	  XWarpPointer(dpy, None, Scr.Root, 0, 0, 0, 0, 
		       Event.xmotion.x_root - delta_x/16,
		       Event.xmotion.y_root - delta_y/16);
	  MoveViewport(Scr.Vx + delta_x,Scr.Vy+delta_y);
	}
    }

  if (ResizeWindow != (int)NULL)
    {
	XQueryPointer( dpy, Event.xany.window,
	    &(Event.xmotion.root), &JunkChild,
	    &(Event.xmotion.x_root), &(Event.xmotion.y_root),
	    &(Event.xmotion.x), &(Event.xmotion.y),
	    &JunkMask);

	  WindowMoved = TRUE;

	XFindContext(dpy, ResizeWindow, FvwmContext, (caddr_t *)&Tmp_win);
	DoResize(Event.xmotion.x_root, Event.xmotion.y_root, Tmp_win);
    }
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleButtonRelease - ButtonRelease event handler
 *
 ***********************************************************************/
void HandleButtonRelease()
{
    int xl, yt;
    unsigned mask;

    if (DragWindow != None)
    {
	MoveOutline(Scr.Root, 0, 0, 0, 0);

	XFindContext(dpy, DragWindow, FvwmContext, (caddr_t *)&Tmp_win);
	if ((DragWindow == Tmp_win->frame)||(DragWindow == Tmp_win->w))
	  {
	    xl = Event.xbutton.x_root - DragX - Tmp_win->frame_bw;
	    yt = Event.xbutton.y_root - DragY - Tmp_win->frame_bw;
	  }
	else
	  {
	    xl = Event.xbutton.x_root - DragX - Scr.IconBorderWidth;
	    yt = Event.xbutton.y_root - DragY - Scr.IconBorderWidth;
	  }

	CurrentDragX = xl;
	CurrentDragY = yt;
	if ((DragWindow == Tmp_win->frame)||(DragWindow == Tmp_win->w))
	  SetupFrame (Tmp_win, xl, yt,
		       Tmp_win->frame_width, Tmp_win->frame_height,FALSE);
	else
	  {
	    XMoveWindow (dpy, DragWindow, xl, yt);

	    if(DragWindow == Tmp_win->icon_w)
	      {
		Tmp_win->icon_x_loc = xl;
		Tmp_win->icon_y_loc = yt;
	      }
	  }
	    

	XRaiseWindow(dpy, DragWindow);
	KeepOnTop();
	UninstallRootColormap();

	DragWindow = (int)NULL;
    }

    if (ResizeWindow != (int)NULL)
      {
	EndResize();
      }

    if (ActiveMenu != (int)NULL && RootFunction == (int)NULL)
      {
	if (ActiveItem != NULL)
	  {
	    int func = ActiveItem->func;
	    Action = ActiveItem->action;
	    switch (func) 
	      {
	      case F_MOVE:
	      case F_RESIZE:
		ButtonPressed = -1;
		break;
	      case F_REFRESH:
	      default:
		break;
	      }
	    if(ButtonWindow)
	      {
		if(ButtonWindow->title_height)
		  ExecuteFunction(func, Action,ButtonWindow->frame,
				  ButtonWindow, &Event, Context,
				  ActiveItem->val1,ActiveItem->val2);
		else
		  ExecuteFunction(func, Action,ButtonWindow->w,
				  ButtonWindow, &Event, Context,
				  ActiveItem->val1,ActiveItem->val2);
	      }
	    else
	      ExecuteFunction(func,Action,None,ButtonWindow, &Event, Context,
				  ActiveItem->val1,ActiveItem->val2);
	    Context = C_NO_CONTEXT;
	    ButtonWindow = NULL;
	    
	    /* if we are not executing a defered command, then take down the
	     * menu
	     */
	    if (RootFunction == (int)NULL)
	    {
		PopDownMenu();
	    }
	}
	else
	    PopDownMenu();
    }

    mask = (Button1Mask|Button2Mask|Button3Mask|Button4Mask|Button5Mask);
    switch (Event.xbutton.button)
    {
	case Button1: mask &= ~Button1Mask; break;
	case Button2: mask &= ~Button2Mask; break;
	case Button3: mask &= ~Button3Mask; break;
	case Button4: mask &= ~Button4Mask; break;
	case Button5: mask &= ~Button5Mask; break;
    }

    if (RootFunction != (int)NULL ||
	ResizeWindow != None ||
	DragWindow != None)
	ButtonPressed = -1;

    if (RootFunction == (int)NULL &&
	(Event.xbutton.state & mask) == 0 &&
	DragWindow == None &&
	ResizeWindow == None)
    {
	XUngrabPointer(dpy, CurrentTime);
	XUngrabServer(dpy);
	XFlush(dpy);
	BlockEnterLeave = 0;
	ButtonPressed = -1;
	Cancel = FALSE;
    }
}


static void do_menu (menu)
    MenuRoot *menu;			/* menu to pop up */
{
  int x = Event.xbutton.x_root;
  int y = Event.xbutton.y_root;
  
  XGrabServer(dpy);
  if (PopUpMenu (menu, x, y)) 
    UpdateMenu();
  else
    XBell (dpy, 0);
  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleButtonPress - ButtonPress event handler
 *
 ***********************************************************************
 */
void HandleButtonPress()
{
  unsigned int modifier;
  Cursor cur;
  MouseButton *MouseEntry;

  /* pop down the menu, if any */
  if (ActiveMenu != NULL)
    PopDownMenu();
  
  XSync(dpy, 0);			/* XXX - remove? */
  
  if (ButtonPressed != -1 )
    {
      /* we got another butt press in addition to one still held
       * down, we need to cancel the operation we were doing
       */
      Cancel = TRUE;
      CurrentDragX = origDragX;
      CurrentDragY = origDragY;
      if (!menuFromFrameOrWindowOrTitlebar)
	MoveOutline(Scr.Root, 0, 0, 0, 0);
      XUnmapWindow(dpy, Scr.SizeWindow);
      ResizeWindow = None;
      DragWindow = None;
      cur = Scr.LeftButt;
      if (Event.xbutton.button == Button2)
	cur = Scr.MiddleButt;
      else if (Event.xbutton.button >= Button3)
	cur = Scr.RightButt;
      
      XGrabPointer(dpy, Scr.Root, True,
		   ButtonReleaseMask | ButtonPressMask,
		   GrabModeAsync, GrabModeAsync,
		   Scr.Root, cur, CurrentTime);
      
      return;
    }
  else
    ButtonPressed = Event.xbutton.button;
  
  if (ResizeWindow != None || DragWindow != None  || ActiveMenu != NULL)
    return;
  
  Context = C_NO_CONTEXT;
  if (Event.xany.window == Scr.Root)
    Context = C_ROOT;
  if (Tmp_win)
    {
      if (Event.xany.window == Tmp_win->title_w)
	Context = C_TITLE;
      if ((Event.xany.window == Tmp_win->left_side_w)||
	  (Event.xany.window == Tmp_win->right_side_w)||
	  (Event.xany.window == Tmp_win->bottom_w))
	Context = C_SIDEBAR;
      else if (Event.xany.window == Tmp_win->w) 
	{
	  fprintf(stderr,"ERROR! ERROR! ERROR! YOU SHOULD NOT BE HERE!!!\n");
	  Context = C_WINDOW;
	}
      else if (Event.xany.window == Tmp_win->icon_w)
	Context = C_ICON;
      else if (Event.xany.window == Tmp_win->frame) 
	{
	  /* since we now place a button grab on the frame instead
	   * of the window, (see GrabButtons() in add_window.c), we
	   * need to figure out where the pointer exactly is before
	   * assigning Context.  If the pointer is on the application
	   * window we will change the event structure to look as if
	   * it came from the application window.
	   */
	  if (Event.xbutton.subwindow == Tmp_win->w) 
	    {
	      Event.xbutton.window = Tmp_win->w;
              Event.xbutton.y -= Tmp_win->title_height;
	      Context = C_WINDOW;
	    }
	  else Context = C_FRAME;
	}
    }
  /* this section of code checks to see if we were in the middle of
   * a command executed from a menu
   */
  if (RootFunction != (int)NULL)
    {
      if (Event.xany.window == Scr.Root)
	{
	  /* if the window was the Root, we don't know for sure it
	   * it was the root.  We must check to see if it happened to be
	   * inside of a client that was getting button press events.
	   */
	  XTranslateCoordinates(dpy, Scr.Root, Scr.Root,
				Event.xbutton.x, 
				Event.xbutton.y, 
				&JunkX, &JunkY, &Event.xany.window);
	  
	  if (Event.xany.window == 0 ||
	      (XFindContext(dpy, Event.xany.window, FvwmContext,
			    (caddr_t *)&Tmp_win) == XCNOENT))
	    {
	      RootFunction = (int)NULL;
	      XBell(dpy, 0);
	      return;
	    }
	  
	  XTranslateCoordinates(dpy, Scr.Root, Event.xany.window,
				Event.xbutton.x, Event.xbutton.y, 
				&JunkX, &JunkY, &JunkChild);

	  Event.xbutton.x = JunkX;
	  Event.xbutton.y = JunkY;
	  Context = C_WINDOW;
	}
      ExecuteFunction(RootFunction, Action, Event.xany.window,
		      Tmp_win, &Event, Context,
		      ActiveItem->val1,ActiveItem->val2);
      
      RootFunction = (int)NULL;
      return;
    }
  
  ButtonEvent = Event;
  ButtonWindow = Tmp_win;
  
  /* if we get to here, we have to execute a function or pop up a 
   * menu
   */
  modifier = (Event.xbutton.state & mods_used);
  if (Context == C_NO_CONTEXT)
    return;
  
  RootFunction = (int)NULL;
  /* need to search for an appropriate mouse binding */
  MouseEntry = Scr.MouseButtonRoot;
  while(MouseEntry != (MouseButton *)0)
    {
      if((MouseEntry->Button == Event.xbutton.button)&&
	 (MouseEntry->Context & Context)&&
	 (MouseEntry->Modifier == modifier))
	{
	  /* got a match, now process it */
	  if(MouseEntry->func == F_MENU)
	    {
	      do_menu (MouseEntry->menu);
	      break;
	    }
	  else if (MouseEntry->func != (int)NULL)
	    {
	      Action = MouseEntry->item ? MouseEntry->item->action : NULL;
	      ExecuteFunction(MouseEntry->func, Action, Event.xany.window, 
			      Tmp_win, &Event, Context,MouseEntry->val1,
			      MouseEntry->val2);
	      break;
	    }
	}
      MouseEntry = MouseEntry->NextButton;
    }
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleEnterNotify - EnterNotify event handler
 *
 ***********************************************************************
 */
int level = 0;
void HandleEnterNotify()
{
  XEnterWindowEvent *ewp = &Event.xcrossing;
  XEvent d;

  level = 1;
  if(BlockEnterLeave)return;
  
  level ++;
  /* look for a matching leaveNotify which would nullify this enterNotify */
  if(XCheckTypedWindowEvent (dpy, ewp->window, LeaveNotify, &d))
    {
      if((d.xcrossing.mode == NotifyNormal)&&
	 (d.xcrossing.detail != NotifyInferior) )
	{
	  return;
	}
    }

  level ++;
  /*
   * Save the id of the window entered.  This will be used to remove
   * border highlight on entering the next application window.
   */
  if (UnHighLight_win && (Tmp_win!= UnHighLight_win))
    {
      unhighlight = 1;
      SetBorder (UnHighLight_win, False,False);	/* application window */
    }

  if (ewp->window == Scr.Root)
    UnHighLight_win = NULL;
  else if (Tmp_win)
    UnHighLight_win = Tmp_win;
  unhighlight=0;

  level ++;
  /*
   * if we aren't in the middle of menu processing
   */
  if (!ActiveMenu) 
    {
      level=100;
      /*
       * We're not interested in pseudo Enter/Leave events generated
       * from grab initiations.
       */
      if (ewp->mode == NotifyGrab)
	{
	  return;
	}
      /*
       * if entering root window, restore fvwm default colormap so that 
       * titlebars are legible
       */
      if (ewp->window == Scr.Root) 
	{
	  if((XCheckTypedWindowEvent(dpy,ewp->window, DestroyNotify, &d)))
	    {
	      Destroy(Tmp_win);
	      return;
	    }
	  InstallWindowColormaps(EnterNotify, &Scr.FvwmRoot);
	  return;
	}

	/*
	 * if we have an event for a specific one of our windows
	 */
      if (Tmp_win) 
	{
	  XGrabServer(dpy);
	  if((XCheckTypedWindowEvent(dpy,Tmp_win->w, DestroyNotify, &Event))||
	     (XCheckTypedWindowEvent(dpy,Tmp_win->frame,DestroyNotify,&Event))||
	     (XCheckTypedWindowEvent(dpy,Tmp_win->title_w,DestroyNotify, &Event))||
	     (XCheckTypedWindowEvent(dpy,Tmp_win->left_side_w,DestroyNotify,&Event))||
	     (XCheckTypedWindowEvent(dpy,Tmp_win->right_side_w,DestroyNotify,&Event))||
	     (XCheckTypedWindowEvent(dpy,Tmp_win->bottom_w,DestroyNotify, &Event)))
	    {
	      XUngrabServer(dpy);
	      Destroy(Tmp_win);
	      return;
	    }
	  if((XCheckTypedWindowEvent(dpy,Tmp_win->w, UnmapNotify, &Event))||
	     (XCheckTypedWindowEvent(dpy,Tmp_win->frame,UnmapNotify,&Event))||
	     (XCheckTypedWindowEvent(dpy,Tmp_win->title_w,UnmapNotify, &Event))||
	     (XCheckTypedWindowEvent(dpy,Tmp_win->left_side_w,UnmapNotify,&Event))||
	     (XCheckTypedWindowEvent(dpy,Tmp_win->right_side_w,UnmapNotify,&Event))||
	     (XCheckTypedWindowEvent(dpy,Tmp_win->bottom_w,UnmapNotify, &Event)))
	    {
	      XUngrabServer(dpy);
	      HandleUnmapNotify();
	      return;
	    }
	  level ++;
	  if (Tmp_win->mapped) 
	    {
	      level ++;
	      /*
	       * If entering the frame or the icon manager, then do 
	       * "window activation things":
	       *
	       *     1.  turn on highlight window (if any)
	       *     2.  install frame colormap
	       *     3.  set frame and highlight window (if any) border
	       *     4.  focus on client window to forward typing
	       *     5.  send WM_TAKE_FOCUS if requested
	       */
	      if (ewp->window == Tmp_win->frame)
		{
		  level=200;
		  Scr.Focus = Tmp_win;
  		  InstallWindowColormaps (EnterNotify,	/* 2 */
					  &Scr.FvwmRoot);
		  SetBorder (Tmp_win, True,True); 	        /* 3 */
		  if (Tmp_win->title_w &&                 	/* 4 */
		      Tmp_win->wmhints && Tmp_win->wmhints->input)
		    XSetInputFocus (dpy, Tmp_win->w,RevertToPointerRoot, 
				    ewp->time);
		  if (Tmp_win->protocols & DoesWmTakeFocus)	/* 5 */
		    send_clientmessage (Tmp_win->w,
					_XA_WM_TAKE_FOCUS, ewp->time);
		} 
	      else if (ewp->window == Tmp_win->w) 
		{
		  level =300;
		  /*
		   * If we are entering the application window, install
		   * its colormap(s).
		   */
		  InstallWindowColormaps(EnterNotify, Tmp_win);
		  level =310;
		  if(Scr.Focus != Tmp_win)
		    {
		      level =350;
		      Scr.Focus = Tmp_win;
		      SetBorder (Tmp_win, True,True); 	        /* 3 */
		      level =360;
		      if(Tmp_win->wmhints && Tmp_win->wmhints->input)
			XSetInputFocus (dpy, Tmp_win->w,RevertToPointerRoot, 
					ewp->time);
		      level =370;
		      if (Tmp_win->protocols & DoesWmTakeFocus)	/* 5 */
			send_clientmessage (Tmp_win->w,
					    _XA_WM_TAKE_FOCUS, ewp->time);
		      level =380;
		    }
		}
	    }			/* end if Tmp_win->mapped */
	  level =400;
	  if (Tmp_win->wmhints != NULL &&
	      ewp->window == Tmp_win->wmhints->icon_window)
	    InstallWindowColormaps(EnterNotify, Tmp_win);
	  XUngrabServer(dpy);
	  return;
	}				/* end if Tmp_win */
      XUngrabServer(dpy);
    }					/* end if !ActiveMenu */

  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	HandleLeaveNotify - LeaveNotify event handler
 *
 ***********************************************************************
 */

void HandleLeaveNotify()
{
  XEvent dummy;

  if(BlockEnterLeave)return;  

  /* look for a matching EnterNotify which would nullify this LeaveNotify */
  if(XCheckTypedWindowEvent (dpy, Event.xcrossing.window, EnterNotify, &dummy))
    {
      if (dummy.xcrossing.mode != NotifyGrab)
	return;
    }

  if (Tmp_win != NULL)
    {
      /*
       * We're not interested in pseudo Enter/Leave events generated
       * from grab initiations and terminations.
       */
      if (Event.xcrossing.mode != NotifyNormal)
	return;
      
      if (Event.xcrossing.detail != NotifyInferior) 
	{
	  if((Event.xcrossing.window==Tmp_win->frame)||
	     ((Event.xcrossing.window==Tmp_win->w)&&
	      (Tmp_win->title_height == 0)))
	     {
	      SetBorder (Tmp_win, False,False);
	      XSetInputFocus (dpy, PointerRoot, RevertToPointerRoot, 
			      Event.xcrossing.time);
	      Scr.Focus = NULL;
	    } 
	  else if(Event.xcrossing.window == Tmp_win->w)
	    {
	      InstallWindowColormaps (LeaveNotify, &Scr.FvwmRoot);
	    }
	}
      XSync (dpy, 0);
      return;
    }
}

/***********************************************************************
 *
 *  Procedure:
 *	HandleConfigureRequest - ConfigureRequest event handler
 *
 ***********************************************************************
 */

void HandleConfigureRequest()
{
    XWindowChanges xwc;
    unsigned long xwcm;
    int x, y, width, height, bw;
    int gravx, gravy;
    XConfigureRequestEvent *cre = &Event.xconfigurerequest;

    /*
     * Event.xany.window is Event.xconfigurerequest.parent, so Tmp_win will
     * be wrong
     */
    Event.xany.window = cre->window;	/* mash parent field */
    if (XFindContext (dpy, cre->window, FvwmContext, (caddr_t *) &Tmp_win) ==
	XCNOENT)
      Tmp_win = NULL;


    /*
     * According to the July 27, 1988 ICCCM draft, we should ignore size and
     * position fields in the WM_NORMAL_HINTS property when we map a window.
     * Instead, we'll read the current geometry.  Therefore, we should respond
     * to configuration requests for windows which have never been mapped.
     */
    if (!Tmp_win || Tmp_win->icon_w == cre->window) {
	xwcm = cre->value_mask & 
	    (CWX | CWY | CWWidth | CWHeight | CWBorderWidth);
	xwc.x = cre->x;
	xwc.y = cre->y;
	if((Tmp_win)&&(Tmp_win->icon_w == cre->window))
	  {
	    Tmp_win->icon_x_loc = cre->x;
	    Tmp_win->icon_y_loc = cre->y;
	  }

	xwc.width = cre->width;
	xwc.height = cre->height;
	xwc.border_width = cre->border_width;
	XConfigureWindow(dpy, Event.xany.window, xwcm, &xwc);
	return;

    }

    if (cre->value_mask & CWStackMode) {
	FvwmWindow *otherwin;

	xwc.sibling = (((cre->value_mask & CWSibling) &&
			(XFindContext (dpy, cre->above, FvwmContext,
				       (caddr_t *) &otherwin) == XCSUCCESS))
		       ? otherwin->frame : cre->above);
	xwc.stack_mode = cre->detail;
	if(Tmp_win->title_height)
	  XConfigureWindow (dpy, Tmp_win->frame, 
			    cre->value_mask & (CWSibling | CWStackMode), &xwc);
	else
	  XConfigureWindow (dpy, Tmp_win->w, 
			    cre->value_mask & (CWSibling | CWStackMode), &xwc);
    }


    /* Don't modify frame_XXX fields before calling SetupWindow! */
    x = Tmp_win->frame_x;
    y = Tmp_win->frame_y;
    width = Tmp_win->frame_width;
    height = Tmp_win->frame_height;
    bw = Tmp_win->frame_bw;

    /*
     * Section 4.1.5 of the ICCCM states that the (x,y) coordinates in the
     * configure request are for the upper-left outer corner of the window.
     * This means that we need to adjust for the additional title height as
     * well as for any border width changes that we decide to allow.  The
     * current window gravity is to be used in computing the adjustments, just
     * as when initially locating the window.  Note that if we do decide to 
     * allow border width changes, we will need to send the synthetic 
     * ConfigureNotify event.
     */
    GetGravityOffsets (Tmp_win, &gravx, &gravy);

    if (cre->value_mask & CWBorderWidth) 
      {
	Tmp_win->old_bw = cre->border_width;  /* for restoring */
      }

    if (cre->value_mask & CWX) {	/* override even if border change */
	x = cre->x - bw;
    }
    if (cre->value_mask & CWY) {
	y = cre->y - ((gravy < 0) ? 0 : Tmp_win->title_height) - bw;
    }

    if (cre->value_mask & CWWidth) {
	width = cre->width;
    }
    if (cre->value_mask & CWHeight) {
	height = cre->height + Tmp_win->title_height;
    }

    /*
     * SetupWindow (x,y) are the location of the upper-left outer corner and
     * are passed directly to XMoveResizeWindow (frame).  The (width,height)
     * are the inner size of the frame.  The inner width is the same as the 
     * requested client window width; the inner height is the same as the
     * requested client window height plus any title bar slop.
     */
    SetupFrame (Tmp_win, x, y, width, height,FALSE);
}


void flush_expose (w)
    Window w;
{
    XEvent dummy;

    while (XCheckTypedWindowEvent (dpy, w, Expose, &dummy)) ;
}


/***********************************************************************
 *
 *  Procedure:
 *	InstallWindowColormaps - install the colormaps for one fvwm window
 *
 *  Inputs:
 *	type	- type of event that caused the installation
 *	tmp	- for a subset of event types, the address of the
 *		  window structure, whose colormaps are to be installed.
 *
 ***********************************************************************
 */

void InstallWindowColormaps (type, tmp)
    int type;
    FvwmWindow *tmp;
{
  XWindowAttributes attributes;
  static Colormap last_cmap;

  switch (type) 
    {
    case PropertyNotify:
    default:
      /* Save the colormap to be loaded for when force loading of
       * root colormap(s) ends.
       */
      Scr.pushed_window = tmp;
      /* Don't load any new colormap if root colormap(s) has been
       * force loaded.
       */
      if (Scr.root_pushes)
	return;
      break;
    }
  if(tmp == &Scr.FvwmRoot)
    XGetWindowAttributes(dpy,Scr.Root,&attributes);
  else
    XGetWindowAttributes(dpy,tmp->w,&attributes);
  
  if(last_cmap != attributes.colormap)
    {
      last_cmap = attributes.colormap;
      XInstallColormap(dpy,attributes.colormap);    
    }
}


/***********************************************************************
 *
 *  Procedures:
 *	<Uni/I>nstallRootColormap - Force (un)loads root colormap(s)
 *
 *	   These matching routines provide a mechanism to insure that
 *	   the root colormap(s) is installed during operations like
 *	   rubber banding or menu display that require colors from
 *	   that colormap.  Calls may be nested arbitrarily deeply,
 *	   as long as there is one UninstallRootColormap call per
 *	   InstallRootColormap call.
 *
 *	   The final UninstallRootColormap will cause the colormap list
 *	   which would otherwise have be loaded to be loaded, unless
 *	   Enter or Leave Notify events are queued, indicating some
 *	   other colormap list would potentially be loaded anyway.
 ***********************************************************************
 */

void InstallRootColormap()
{
    FvwmWindow *tmp;
    if (Scr.root_pushes == 0) 
      {
	/*
	 * The saving and restoring of cmapInfo.pushed_window here
	 * is a slimy way to remember the actual pushed list and
	 * not that of the root window.
	 */
	tmp = Scr.pushed_window;
	InstallWindowColormaps(0, &Scr.FvwmRoot);
	Scr.pushed_window = tmp;
      }
    Scr.root_pushes++;
    return;
}


void UninstallRootColormap()
{
  if (Scr.root_pushes)
    Scr.root_pushes--;
  
  if (!Scr.root_pushes) 
    {
      XSync (dpy, 0);
      InstallWindowColormaps(0, Scr.pushed_window);
    }
  return;
}




  
/***********************************************************************
 *
 *  Procedure:
 *	RestoreWithdrawnLocation
 * 
 *  Puts windows back where they were before fvwm took over 
 *
 ***********************************************************************
 */

void RestoreWithdrawnLocation (FvwmWindow *tmp)
{
  int gravx, gravy;
  unsigned int bw, mask;
  XWindowChanges xwc;

  if (XGetGeometry (dpy, tmp->w, &JunkRoot, &xwc.x, &xwc.y, 
		    &JunkWidth, &JunkHeight, &bw, &JunkDepth)) 
    {
      GetGravityOffsets (tmp, &gravx, &gravy);
      if (gravy < 0) xwc.y -= tmp->title_height;
      else if (gravy > 0)xwc.y += tmp->boundary_width;
      
      mask = (CWX | CWY);
      if (bw != tmp->old_bw) 
	{
	  xwc.x -= (gravx + 1) * tmp->old_bw;
	  xwc.y -= (gravy + 1) * tmp->old_bw;
	  xwc.border_width = tmp->old_bw;
	  mask |= CWBorderWidth;
	}
      xwc.x += gravx * (tmp->frame_bw+tmp->boundary_width);
      xwc.y += gravy * tmp->frame_bw;
      XConfigureWindow (dpy, tmp->w, mask, &xwc);
      if (tmp->wmhints && (tmp->wmhints->flags & IconWindowHint)) 
	XUnmapWindow (dpy, tmp->wmhints->icon_window);
    }
}

