/* pam_umotd module */

/*
 * Modified for pam_umotd by Isaac Clerencia <isaac@sindominio.net>
 *
 * Modified for pam_motd by Ben Collins <bcollins@debian.org>
 *
 * Based off of:
 * $Id: pam_motd.c,v 1.1 1999/11/08 05:46:15 morgan Exp $
 * 
 * Written by Michael K. Johnson <johnsonm@redhat.com> 1996/10/24
 *
 */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>

#include <security/_pam_macros.h>

static void _log_err(int err, const char *format, ...)
{
	va_list args;

	va_start(args, format);
	openlog("PAM-umotd", LOG_CONS|LOG_PID, LOG_AUTH);
	vsyslog(err, format, args);
	va_end(args);
	closelog();
}

/*
 * here, we make a definition for the externally accessible function
 * in this file (this definition is required for static a module
 * but strongly encouraged generally) it is used to instruct the
 * modules include file to define the function prototypes.
 */

#define PAM_SM_SESSION
#define DEFAULT_UMOTD "/etc/umotd"
#include <security/pam_modules.h>

/* --- session management functions (only) --- */

PAM_EXTERN
int pam_sm_close_session(pam_handle_t *pamh, int flags, int argc,
                        const char **argv)
{
     return PAM_IGNORE;
}

PAM_EXTERN
int pam_sm_open_session(pam_handle_t *pamh, int flags, int argc,
                   const char **argv)
{
     int retval = PAM_IGNORE;
     int fd;
     char *mtmp=NULL, *umotd_basepath=NULL, *umotd_path;
	  const char *user;
     struct pam_conv *conversation;
     struct pam_message message;
     struct pam_message *pmessage = &message;
     struct pam_response *resp = NULL;
     struct stat st;

     if (flags & PAM_SILENT) {
	return retval;
     }

    for (; argc-- > 0; ++argv) {
        if (!strncmp(*argv,"umotd=",5)) {
            umotd_basepath = (char *) strdup(5+*argv);
            if (umotd_basepath != NULL) {
                D(("set umotd path: %s", umotd_basepath));
            } else {
                D(("failed to duplicate umotd path - ignored"));
            }
	}
     }

     if (umotd_basepath == NULL)
		umotd_basepath = DEFAULT_UMOTD;

		if(pam_get_user(pamh, &user, NULL) != PAM_SUCCESS || user == NULL
				|| *user == '\0'){
			return PAM_USER_UNKNOWN;
		}
		umotd_path=malloc(strlen(umotd_basepath) + strlen(user) + 2);
		sprintf(umotd_path,"%s/%s",umotd_basepath,user);

     message.msg_style = PAM_TEXT_INFO;

     if ((fd = open(umotd_path, O_RDONLY, 0)) >= 0) {
       /* fill in message buffer with contents of motd */
       if ((fstat(fd, &st) < 0) || !st.st_size)
         return retval;
       message.msg = mtmp = malloc(st.st_size+1);
       /* if malloc failed... */
       if (!message.msg) return retval;
       read(fd, mtmp, st.st_size);
       if (mtmp[st.st_size-1] == '\n')
	  mtmp[st.st_size-1] = '\0';
       else
	  mtmp[st.st_size] = '\0';
       close(fd);
       /* Use conversation function to give user contents of umotd */
       pam_get_item(pamh, PAM_CONV, (const void **)&conversation);
       conversation->conv(1, (const struct pam_message **)&pmessage,
			  &resp, conversation->appdata_ptr);
       free(mtmp);
       if (resp)
	   _pam_drop_reply(resp, 1);
     }

     return retval;
}


#ifdef PAM_STATIC

/* static module data */

struct pam_module _pam_umotd_modstruct = {
     "pam_umotd",
     NULL,
     NULL,
     NULL,
     pam_sm_open_session,
     pam_sm_close_session,
     NULL,
};

#endif

/* end of module definition */
