/*
 *  linux/mm/bigpages.c
 *
 *  Copyright (C) 2002  Ingo Molnar
 */

#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/spinlock.h>
#include <linux/highmem.h>
#include <linux/smp_lock.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>

static spinlock_t bigpages_lock = SPIN_LOCK_UNLOCKED;
unsigned long nr_bigpages;
static LIST_HEAD(bigpages_list);

#define ORDER_BIGPAGE (PMD_SHIFT - PAGE_SHIFT)

struct page *alloc_bigpage(void)
{
	list_t *head = &bigpages_list;
	struct page *page = NULL;

	spin_lock(&bigpages_lock);
	if (nr_bigpages) {
		page = list_entry(head->next, struct page, list);
		list_del_init(head->next);
		nr_bigpages--;
	}
	spin_unlock(&bigpages_lock);

	return page;
}

void free_bigpage(struct page *page)
{
	struct page *p;
	int i;

#ifndef CONFIG_DISCONTIGMEM
	BUG_ON((page - mem_map) % BIGPAGE_PAGES);
#endif
	for (i = 0 ; i < (1 << ORDER_BIGPAGE); i++) {
		p = page + i;
		set_page_count(p, 2);
		set_bit(PG_bigpage, &p->flags);
		clear_highpage(p);
	}
	spin_lock(&bigpages_lock);
	nr_bigpages++;
	list_add(&page->list, &bigpages_list);
	spin_unlock(&bigpages_lock);
}

static int grow_bigpages_pool(int pages)
{
	struct page *page;
	int allocated = 0;

	while (pages) {
		page = alloc_pages(__GFP_HIGHMEM, ORDER_BIGPAGE);
		if (!page)
			break;
		free_bigpage(page);
		pages--;
		allocated++;
	}
	printk("bigpage subsystem: allocated %ld bigpages (=%ldMB).\n",
		nr_bigpages, nr_bigpages << (BIGPAGE_SHIFT - 20));
	return allocated;
}

static __initdata int boot_bigpages;

static __init int reserve_bigpages(char *str)
{
        unsigned long pages = memparse(str, &str) >> PAGE_SHIFT;

	pages >>= ORDER_BIGPAGE;
	boot_bigpages = pages;

	return 0;
}

static __init int init_bigpage_pool(void)
{
	grow_bigpages_pool(boot_bigpages);
	return 0;
}

__setup("bigpages=", reserve_bigpages);
__initcall(init_bigpage_pool);

