/*
 * fan.c -- fan status/control
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Sos Pter <sp@osb.hu>, 2002-2003
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/pm.h>
#include <linux/proc_fs.h>

#include <asm/system.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#ifdef OMNIBOOK_STANDALONE
#include "omnibook.h"
#else
#include <linux/omnibook.h>
#endif

#include "ec.h"

static struct proc_dir_entry *proc_fan;

int omnibook_get_fan(void)
{
	u8 fan;
	int retval;
	
	switch (omnibook_ectype) {
	case XE3GF:
		retval = omnibook_ec_read(XE3GF_FSRD, &fan);
		if (retval)
			return retval;
		retval = fan;
		break;
	case OB500:
		retval = omnibook_io_read(OB500_GPO1, &fan);
		if (retval)
			return retval;
		retval = (fan & OB500_FAN_OFF_MASK) ? 0 : 1;
		break;
	case OB510:
		retval = omnibook_io_read(OB510_GPIO, &fan);
		if (retval)
			return retval;
		retval = (fan & OB510_FAN_OFF_MASK) ? 0 : 1;
		break;
	case OB6000:
	case OB6100:
		retval = omnibook_ec_read(OB6000_STA1, &fan);
		if (retval)
			return retval;
		retval = (fan & OB6000_FAN_MASK) ? 1 : 0;
		break;
	case OB4150:
		retval = omnibook_ec_read(OB4150_STA1, &fan);
		if (retval)
			return retval;
		retval = (fan & OB4150_FAN_MASK) ? 1 : 0;
		break;
	default:
		printk(KERN_INFO "%s: Fan status monitoring is unsupported on this machie.\n", MODULE_NAME);
		retval = -ENODEV;
	}

	return retval;
}
	
int omnibook_fan_on(void)
{
	u8 fan;
	int retval;

	switch (omnibook_ectype) {
	case XE3GF:
		retval = omnibook_ec_read(XE3GF_FSRD, &fan);
		if (retval)
			return retval;
		retval = omnibook_ec_write(XE3GF_FSRD, fan | XE3GF_FAN_ON_MASK);
		if (retval)
			return retval;
		break;
	case OB500:
		retval = omnibook_io_read(OB500_GPO1, &fan);
		if (retval)
			return retval;
		retval = omnibook_io_write(OB500_GPO1, fan & ~OB500_FAN_ON_MASK);
		if (retval)
			return retval;
		break;
	case OB510:
		retval = omnibook_io_read(OB510_GPIO, &fan);
		if (retval)
			return retval;
		retval = omnibook_io_write(OB510_GPIO, fan & ~OB510_FAN_ON_MASK);
		if (retval)
			return retval;
		break;
	case OB6000:
	case OB6100:
		retval = omnibook_ec_read(OB6000_STA1, &fan);
		if (retval)
			return retval;
		retval = omnibook_ec_write(OB6000_STA1, fan | OB6000_FAN_MASK);
		if (retval)
			return retval;
		break;
	case OB4150:
		retval = omnibook_ec_read(OB4150_STA1, &fan);
		if (retval)
			return retval;
		retval = omnibook_ec_write(OB4150_STA1, fan | OB4150_FAN_MASK);
		if (retval)
			return retval;
		break;
	default:
		printk(KERN_INFO "%s: Direct fan control is unsupported on this machie.\n", MODULE_NAME);
		retval = -ENODEV;
	}

	return retval;
}

int omnibook_fan_off(void)
{
	u8 fan;
	int retval;

	switch (omnibook_ectype) {
	case OB500:
		retval = omnibook_io_read(OB500_GPO1, &fan);
		if (retval)
			return retval;
		retval = omnibook_io_write(OB500_GPO1, fan | OB500_FAN_OFF_MASK);
		if (retval)
			return retval;
		break;
	case OB510:
		retval = omnibook_io_read(OB510_GPIO, &fan);
		if (retval)
			return retval;
		retval = omnibook_io_write(OB510_GPIO, fan | OB510_FAN_OFF_MASK);
		if (retval)
			return retval;
		break;
	case OB6000:
	case OB6100:
		retval = omnibook_ec_read(OB6000_STA1, &fan);
		if (retval)
			return retval;
		retval = omnibook_ec_write(OB6000_STA1, fan & ~OB6000_FAN_MASK);
		if (retval)
			return retval;
		break;
	case OB4150:
		retval = omnibook_ec_read(OB4150_STA1, &fan);
		if (retval)
			return retval;
		retval = omnibook_ec_write(OB4150_STA1, fan & ~OB4150_FAN_MASK);
		if (retval)
			return retval;
		break;
	default:
		printk(KERN_INFO "%s: Direct fan control is unsupported on this machie.\n", MODULE_NAME);
		retval = -ENODEV;
	}

	return retval;
}

static int omnibook_fan_status(char *buffer, char **start, off_t off, int count, int *eof, void *data)
{
	int len;
	int fan;
	char *b = buffer;
	char *str;

	fan = omnibook_get_fan();
	str = (fan) ? "on" : "off";

	if (fan > 1)
		b += sprintf(b, "Fan is %s (level %d)\n", str, fan);
	else
		b += sprintf(b, "Fan is %s\n", str);

	len = b - buffer;
	if (len < off + count)
	*eof = 1;
	*start = buffer + off;
	len -= off;
	if (len > count)
	len = count;
	if (len < 0)
	len = 0;
	return len;
}

static int omnibook_fan_set(struct file *file, const char *buffer, unsigned long count, void *data)
{
	char status[1] = {'\0'};

	if (copy_from_user(status, buffer, 1))
		return -EFAULT;
	switch (*status) {
	case '0':
		omnibook_fan_off();
		break;
	case '1':
		omnibook_fan_on();
		break;
	default:
		count = -EINVAL;
	}
	
	return count;
}

int __init omnibook_fan_init(void)
{
	mode_t pmode;

	switch (omnibook_ectype) {
	case XE3GF:
	case OB500:
	case OB510:
	case OB6000:
	case OB6100:
	case OB4150:
		pmode = S_IFREG | S_IWUSR | S_IRUGO;
		if (omnibook_userset)
			pmode = pmode | S_IWUGO;
		proc_fan = create_proc_entry("fan", pmode, omnibook_proc_root);
		break;
	default:
		printk(KERN_INFO "%s: Fan status monitor and control is unsupported on this machine.\n", MODULE_NAME);
		return 0;
	}
	
	if (proc_fan) {
		proc_fan->read_proc = omnibook_fan_status;
		proc_fan->write_proc = omnibook_fan_set;
	} else {
		printk(KERN_ERR "%s: Unable to create /proc/%s/fan.\n", MODULE_NAME, MODULE_NAME);
		return -ENOENT;
	}
	printk(KERN_INFO "%s: Fan status monitor and control is enabled.\n", MODULE_NAME);
	return 0;
}

void __exit omnibook_fan_cleanup(void)
{
	if (proc_fan)
		remove_proc_entry("fan", omnibook_proc_root);
}

EXPORT_SYMBOL(omnibook_get_fan);
EXPORT_SYMBOL(omnibook_fan_on);
EXPORT_SYMBOL(omnibook_fan_off);

/* End of file */
