/*
 * ps.c		- show process status
 *
 * Copyright (c) 1992 Branko Lankester
 *
 */

#include <sys/types.h>
#include <termios.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <pwd.h>
#include <errno.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include "ps.h"
#include "psdata.h"


#define	PS_D	0	/* default format (short) */
#define	PS_L	1	/* long format */
#define	PS_U	2	/* user format */
#define	PS_J	3	/* jobs format */
#define	PS_S	4	/* signal format */
#define	PS_V	5	/* vm format */
#define	PS_M	6	/* mem. stuff */
#define	PS_X	7	/* regs etc., for testing */
#define	PS_Y	8	/* system call */

char *hdrs[] = {
"  PID TT STAT  TIME COMMAND",
"  UID   PID  PPID PRI NI SIZE  RSS WCHAN      STAT  TT   TIME COMMAND",
"USER       PID %CPU %MEM SIZE  RSS TT STAT  START   TIME COMMAND",
" PPID   PID  PGID   SID TT TPGID  STAT   UID   TIME COMMAND",
"  UID   PID SIGNAL   BLOCKED  IGNORED  CATCHED  STAT  TT   TIME COMMAND",
"  PID TT STAT   TIME  PAGEIN TSIZ DSIZ  RSS   LIM %MEM COMMAND",
"  PID TT MAJFLT MINFLT  TRS  DRS SIZE SWAP  RSS SHRD  LIB  DT COMMAND",
" F   PID    STACK      ESP      EIP TMOUT ALARM STAT  TT   TIME COMMAND",
"  PID TT ST     EIP SYSTEM CALL                           TIME COMMAND",
};

#define	KSTK(reg,stack)	(((unsigned long *)(stack))[1007+(reg)])

extern (*fmt_fnc[])();	/* forward declaration */

long time_now;

/*
 * command line options
 */
int fmt;
int all;
int kern_comm;
int no_ctty;
int run_only;
char *ctty;
int npid;
int *pids;
int show_env;
int num_outp;		/* numeric fields for user or wchan */
int first_task = 1;	/* don't show task 0 */
int pg_shift = 2;	/* default: show k instead of pages */
int Sum;


main(argc, argv)
char **argv;
{
    char *p;
    int no_header = 0;
    int fopt = 0;
    int width = 0;
    int Update = 0;


repeat:
    if (argc > 1) {
	for (p = argv[1]; *p; ++p) {
	    switch (*p) {
		case 'l': fmt = PS_L; ++fopt; break;
		case 'u': fmt = PS_U; ++fopt; break;
		case 'j': fmt = PS_J; ++fopt; break;
		case 's': fmt = PS_S; ++fopt; break;
		case 'v': fmt = PS_V; ++fopt; break;
		case 'm': fmt = PS_M; ++fopt; break;
		case 'X': fmt = PS_X; ++fopt; break; /* regs */
		case 'y': fmt = PS_Y; ++fopt; break; /* regs */
		case 'a': all = 1; break;
		case 'c': kern_comm = 1; break;
		case '0': first_task = 0; /*fallthrough*/
		case 'x': no_ctty = 1; break;
		case 't': ctty = p + 1; break;
		case 'r': run_only = 1; break;
		case 'e': show_env = 1; break;
		case 'w': ++width; break;
		case 'h': no_header = 1; break;
		case 'n': num_outp = 1; break;
		case 'S': Sum = 1; break;
		case 'p': pg_shift = 0; break;
		case 'U': Update = 1; break;
#ifdef DEBUG
		case 'd': ++Debug; break;
#endif
		case 'g':	/* old flag, ignore */
		case 'f':	/* sysv, ignore */
		case ',':
		case '-': break;
		default:
		    if (npid)
			usage();
		    /* expect comma separated list of pids */
		    pids = (int *) xmalloc(strlen(p)*2+2);
		    while (isdigit(*p)) {
			pids[npid++] = atoi(p);
			while (isdigit(*p))
			    ++p;
			if (*p != ',')
			    break;
			++p;
		    }
		    if (*p)
			usage();
	    }
	    if (ctty || npid)
		break;		/* pid and tty always last */
	}
	if (fopt > 1) {
	    fprintf(stderr, "ps: specify only one of j,l,s,u,v,X,y\n");
	    exit(1);
	}
    }
    if (argc > 2 && argv[2][0] != '/') {
	++argv;
	--argc;
	goto repeat;
    }

    /*
     * only allow different namelist if already read access to /dev/kmem
     */
    if (argc > 2 && access(kmem_path, 4 /*R_OK*/)) {
	perror(kmem_path);
	exit(1);
    }

    if (argc > 3)
	swappath[0] = argv[3];

    if (open_sys(argc > 2 ? argv[2] : NULL, Update) == -1) {
	perror(argc > 2 ? argv[2] : "cannot open psdatabase");
	exit(1);
    }

    set_maxcmd(width);
    read_globals();
    time_now = time(0L);
    if (!no_header)
	puts(hdrs[fmt]);
    show_procs();
    exit(0);
}


usage()
{
    fprintf(stderr, "usage:  ps acehjlnrsSuUvwx{t<tty>,#} [system-path] [swap-path]\n");
    exit(1);
}


/*
 * set maximum chars displayed on a line
 */
set_maxcmd(w_opts)
{
    struct winsize win;
    int cols = 80;

    if (ioctl(1, TIOCGWINSZ, &win) != -1 && win.ws_col > 0)
	cols = win.ws_col;

    switch (w_opts) {
	case 0: break;
	case 1: cols += 52; break;	/* 80 -> 132 */
	case 2: cols *= 2; break;
	default: cols = MAXCMD;
    }
    maxcmd = cols - strlen(hdrs[fmt]) + 6;
}



show_procs()
{
    struct task_struct *taskp;
    struct task_struct task_buf;
    char stack_buf[PAGE_SIZE];
    int tty, i, uid, j;
    off_t _task = k_addr("_task");

    uid = getuid();

    if (ctty)
	tty = tty_to_dev(ctty);

    for (i = first_task; i < NR_TASKS; ++i) {
	kmemread(&taskp, _task + 4*i, 4);
	if (taskp) {
	    kmemread(&task_buf, taskp, sizeof(task_buf));
			/* check if valid, proc may have exited */
	    if ((unsigned) task_buf.state > 5 ||
		    (task_buf.pid <= 0 && i != 0) ||
		    !task_buf.kernel_stack_page)
		continue;

	    if (npid) {
		for (j = 0; j < npid; ++j)
		    if (task_buf.pid == pids[j])
			break;
		if (j == npid)
		    continue;
	    } else if (ctty) {
		if (task_buf.tty != tty)
		    continue;
	    } else
		if (!all && task_buf.uid != uid ||
		    !no_ctty && task_buf.tty == -1 ||
		    run_only && task_buf.state != TASK_RUNNING &&
			    task_buf.state != TASK_UNINTERRUPTIBLE)
			continue;

	    kmemread(&stack_buf, task_buf.kernel_stack_page, PAGE_SIZE);
	    (fmt_fnc[fmt])(&task_buf,&stack_buf);
	    if (fmt != PS_V && fmt != PS_M)
		show_time(&task_buf, &stack_buf);
	    printf("%s\n", cmd_args(&task_buf, &stack_buf));
	}
    }
}



show_short(task, stack)
struct task_struct *task;
reg_t * stack;
{
    printf("%5d %-2s %-4s",
	task->pid,
	dev_to_tty(task->tty),
	status(task));
}

show_long(task, stack)
struct task_struct *task;
reg_t * stack;
{
    printf("%5d %5d %5d %3d %2d %4d %4d %-10.10s %-5s %-2s ",
	task->euid,
	task->pid,
	get_kword(&task->p_pptr->pid),
	2 * PZERO - task->counter,	/* sort of priority */
	PZERO - task->priority,		/* nice value */
	VSIZE(task,stack),
	task->rss * 4,
	(task->state == TASK_INTERRUPTIBLE ||
	 task->state == TASK_UNINTERRUPTIBLE ||
	 Debug > 1 && task->state == TASK_STOPPED ? 
	    wchan(task->tss.ebp, stack, num_outp) : ""),
	status(task),
	dev_to_tty(task->tty));
}

show_jobs(task, stack)
struct task_struct *task;
reg_t * stack;
{
    printf("%5d %5d %5d %5d %-2s %5d  %-4s %5d ",
	get_kword(&task->p_pptr->pid),
	task->pid,
	task->pgrp,
	task->session,
	dev_to_tty(task->tty),
	tty_pgrp(task->tty),
	status(task),
	task->euid);
}

show_user(task, stack)
struct task_struct *task;
reg_t * stack;
{
    time_t start;
    int pcpu, pmem;
    unsigned long proc_jiffies;

    if (num_outp)
	printf("%5d    ", task->euid);
    else
	printf("%-8s ", user_from_uid(task->euid));

    jiffies = get_kword(k_addr("_jiffies"));
    proc_jiffies = jiffies - task->start_time;
    start = time_now - proc_jiffies / HZ;
    if (proc_jiffies == 0)
	pcpu = 500;
    else
	pcpu = (double) (task->utime + task->stime) * 1000 / proc_jiffies;
    if (pcpu >= 1000)
	pcpu = 999;
    pmem = task->rss * 1000 / nr_pages;

    printf("%5d %2d.%d %2d.%d %4d %4d %-2s %-5s%.6s ",
	task->pid,
	pcpu / 10, pcpu % 10,
	pmem / 10, pmem % 10,
	VSIZE(task,stack),
	task->rss * 4,
	dev_to_tty(task->tty),
	status(task),
 	ctime(&start) + (time_now - start > 3600*24 ? 4 : 10));
}

show_sig(task, stack)
struct task_struct *task;
reg_t * stack;
{
    unsigned long sigignore=0, sigcatch=0, bit=1;
    int i;

    for (i=0; i<32; ++i) {
	switch((int) task->sigaction[i].sa_handler) {
	    case 1: sigignore |= bit; break;
	    case 0: break;
	    default: sigcatch |= bit;
	}
	bit <<= 1;
    }
    printf("%5d %5d %08x %08x %08x %08x %-5s %-2s ",
	task->euid,
	task->pid,
	task->signal,
	task->blocked,
	sigignore,
	sigcatch,
	status(task),
	dev_to_tty(task->tty));
}

show_vm(task, stack)
struct task_struct *task;
reg_t * stack;
{
    int pmem;

    printf("%5d %-2s %-5s",
	task->pid,
	dev_to_tty(task->tty),
	status(task));
    show_time(task);
    printf(" %6d %4d %4d %4d ",
	task->maj_flt + (Sum ? task->cmaj_flt : 0),
	task->end_code / 1024,
	SIZE(task, stack), task->rss*4);
    if (task->rlim[RLIMIT_RSS].rlim_cur == RLIM_INFINITY)
	printf("   xx ");
    else
	printf("%5d ", task->rlim[RLIMIT_RSS].rlim_cur / 1024);
    pmem = task->rss * 1000 / nr_pages;
    printf("%2d.%d ", pmem / 10, pmem % 10);
}


show_m(task, stack)
struct task_struct *task;
reg_t * stack;
{
    struct mem_info *mi;

    mi = get_mem_info(task);
    printf("%5d %-2s %6d %6d %4d %4d %4d %4d %4d %4d %4d %3d ", 
	task->pid,
	dev_to_tty(task->tty),
	task->maj_flt + (Sum ? task->cmaj_flt : 0),
	task->min_flt + (Sum ? task->cmin_flt : 0),
	/*task->end_code / 4096 << pg_shift,*/
	mi->trs << pg_shift,
	mi->drs << pg_shift,
	mi->size << pg_shift, 
	mi->swap << pg_shift,
	mi->rss << pg_shift,
	mi->share << pg_shift,
	mi->lrs << pg_shift, 
	mi->dt << pg_shift);
}

show_regs(task, stack)
struct task_struct *task;
reg_t * stack;
{
    printf("%2x %5d %8x %8x %8x ",
	task->flags,
	task->pid,
	/**
	task->start_code >> 16,
	**/
	task->start_stack,
	KSTK_ESP(stack),
	KSTK_EIP(stack));

    prtime(task->timeout, jiffies);
    prtime(task->it_real_value, 0);

    if (Debug > 1)
	printf("%5x ", task->exit_code);

    printf("%-5s %-2s ",
	status(task),
	dev_to_tty(task->tty));
}

struct {
    int n;
    int fmt_mask;	/* bit mask for output fmt, bit set means decimal */
    char *name;
} syscall_ent[] = {
#include "syscallent.h"
};

char *
syscall_str(nr, args, ret_code)
long *args;
{
    static char buf[1024];
    int i, m;
    char *p;

    if (nr >= sizeof(syscall_ent)/sizeof(syscall_ent[0])) {
	sprintf(buf, "call%d", nr);
	i = 5;
	m = 0;
    } else {
	strcpy(buf, syscall_ent[nr].name);
	i = syscall_ent[nr].n;
	m = syscall_ent[nr].fmt_mask;
    }
    p = buf + strlen(buf);
    *p++ = '(';
    while (--i >= 0) {
	sprintf(p, (m&1) ? "%d" : "%#x", *args++);
	p += strlen(p);
	if (i) {
	    *p++ = ',';
	    *p++ = ' ';
	}
	m >>= 1;
    }
    if (ret_code == -ENOSYS)
	sprintf(p, ")");	/* in systemcall */
    else
	if (nr == ret_code)
	    sprintf(p, ") (= %d)", ret_code);	/* can be restart */
	else
	    sprintf(p, ") = %d", ret_code);	/* leaving syscall */
    return buf;
}

show_syscall(task, stack)
struct task_struct *task;
reg_t * stack;
{
    int orig_eax;
    char *s = NULL;

    printf("%5d ", task->pid);
    printf("%-2s %-1.1s %8x ", dev_to_tty(task->tty), status(task), KSTK(EIP, stack));
    orig_eax = KSTK(ORIG_EAX, stack);

    if (task->state != TASK_ZOMBIE && orig_eax > 0)
	printf("%-35s ", syscall_str(orig_eax, &KSTK(EBX, stack), KSTK(EAX, stack)));
    else if (task->state == TASK_ZOMBIE || Debug <= 1)
	printf("%-35s ", "");
    else
	printf("%d  ax:%x, bx:%x, cx:%x, dx:%x, si:%x, di:%x, bp:%x ",
	    KSTK(ORIG_EAX, stack), KSTK(EAX, stack), KSTK(EBX, stack),
	    KSTK(ECX, stack), KSTK(EDX, stack), KSTK(ESI, stack),
	    KSTK(EDI, stack), KSTK(EBP, stack));
}

prtime(t, rel)
unsigned long t, rel;
{
    if (t == 0) {
	printf("      ");
	return;
    }
    if ((long) t == -1) {
	printf("   xx ");
	return;
    }
    if ((long) (t -= rel) < 0)
	t = 0;
    
    if (t > 9999)
	printf("%5d ", t / 100);
    else
	printf("%2d.%02d ", t / 100, t % 100);
}

int (*fmt_fnc[])() = {
    show_short,
    show_long,
    show_user,
    show_jobs,
    show_sig,
    show_vm,
    show_m,
    show_regs,
    show_syscall,
};


show_time(task, stack)
struct task_struct *task;
reg_t * stack;
{
    unsigned t;

    t = (task->utime + task->stime) / HZ;
    if (Sum)
	t += (task->cutime + task->cstime) / HZ;

    printf("%3d:%02d ", t / 60, t % 60);
}
