/* text2atm.c - Converts textual representation of ATM address to binary
		encoding */

/* Written 1995 by Werner Almesberger, EPFL-LRC */


#include <ctype.h>
#include <string.h>
#include <limits.h>
#include <linux/atm.h>
#include <linux/atmsap.h>

#include "atm.h"


static int try_pvc(const char *text,struct sockaddr_atmpvc *addr,int flags)
{
    int part[3];
    int i;

    part[0] = part[1] = part[2] = 0;
    i = 0;
    while (1) {
	if (!*text) return -1; /* empty or ends with a dot */
	if (i == 3) return 1; /* too long */
	if (isdigit(*text)) {
	    if (*text == '0' && isdigit(text[1])) return 1;
		/* no leading zeroes */
	    do {
		if (part[i] > INT_MAX/10) return 1; /* number too big */
		part[i] = part[i]*10+*text++-'0';
	    }
	    while (isdigit(*text));
	    i++;
	    if (!*text) break;
	    if (*text++ != '.') return 1; /* non-PVC character */
	    continue;
	}
	if (*text == '*') {
	    if (!(flags & T2A_WILDCARD)) return -1; /* not allowed */
	    part[i++] = ATM_ITF_ANY; /* all *_ANY have the same value */
	}
	else {
	    if (*text != '?') return 1; /* invalid character */
	    if (!(flags & T2A_UNSPEC)) return -1; /* not allowed */
	    part[i++] = ATM_VPI_UNSPEC; /* all *_UNSPEC have the same
					   value */
	}
	if (!*++text) break;
	if (*text++ != '.') return -1; /* dot required */
    }
    if (i < 2) return 1; /* no dots */
    if (i == 2) {
	part[2] = part[1];
	part[1] = part[0];
	part[0] = 0; /* default interface */
    }
    if (part[0] > SHRT_MAX || part[2] > ATM_MAX_VCI) return 1; /* too big */
    if (part[1] > (flags & T2A_NNI ? ATM_MAX_VPI_NNI : ATM_MAX_VPI))
	return 1; /* too big */
    if (part[0] == ATM_VPI_UNSPEC) return -1; /* bad */
    addr->sap_family = AF_ATMPVC;
    addr->sap_addr.itf = part[0];
    addr->sap_addr.vpi = part[1];
    addr->sap_addr.vci = part[2];
    memset(&addr->sap_txtp,0,sizeof(struct atm_trafprm)); /* paranoia */
    memset(&addr->sap_rxtp,0,sizeof(struct atm_trafprm));
    return 0;
}


static int do_try_nsap(const char *text,struct sockaddr_atmsvc *addr,int flags)
{
    const char *walk;
    int count,pos,dot;
    char value;

    count = dot = 0;
    for (walk = text; *walk; walk++)
	if (isdigit(*walk)) {
	    if (count++ == 15) break;
	    dot = 1;
	}
	else if (*text != '.') break;
	    else if (!dot) return -1; /* two dots in a row */
		else dot = 0;
    if (*walk != ':') pos = 0;
    else {
	if (!dot || *text == '0') return -1;
	addr->sas_addr.prv[0] = ATM_AFI_E164;
	addr->sas_addr.prv[1] = 0;
	memset(addr->sas_addr.prv+1,0,8);
	for (pos = 18-count-1; *text; text++)
	    if (*text != '.')
		if (*text == ':') break;
		else {
		    if (pos & 1) addr->sas_addr.prv[pos >> 1] |= *text-'0';
		    else addr->sas_addr.prv[pos >> 1] = (*text-'0') << 4;
		    pos++;
		}
	addr->sas_addr.prv[8] |= 0xf;
	text++;
	pos++;
    }
    for (dot = 0; *text; text++)
	if (isxdigit(*text)) {
	    if (pos == ATM_ESA_LEN*2) return 1; /* too long */
	    value = isdigit(*text) ? *text-'0' : (islower(*text) ?
	      toupper(*text) : *text)-'A'+10;
	    if (pos & 1) addr->sas_addr.prv[pos >> 1] |= value;
	    else addr->sas_addr.prv[pos >> 1] = value << 4;
	    pos++;
	    dot = 1;
	}
	else if (*text != '.') return 1;
	    else {
		if (!dot) return -1; /* two dots in a row */
		dot = 0;
	    }
    if (!dot) return -1;
    if (!*addr->sas_addr.prv) return 1; /* no leading zeroes */
    return pos != ATM_ESA_LEN*2;
}


static int try_nsap(const char *text,struct sockaddr_atmsvc *addr,int flags)
{
    int result;

    result = do_try_nsap(text,addr,flags);
    if (result) return result;
    addr->sas_family = AF_ATMSVC;
    addr->sas_addr.bhli.hl_type = ATM_HL_NONE;
    addr->sas_addr.blli = NULL;
    *addr->sas_addr.pub = 0;
    return 0;
}


static int try_e164(const char *text,struct sockaddr_atmsvc *addr,int flags)
{
    int i,dot;

    if (*text == ':') text++;
    for (i = dot = 0; *text; text++)
	if (isdigit(*text)) {
	    if (i == ATM_E164_LEN) return 1; /* too long */
	    addr->sas_addr.pub[i++] = *text;
	    dot = 1;
	}
	else if (*text != '.') break;
	    else {
		if (!dot) return -1; /* two dots in a row */
		dot = 0;
	    }
    if (!dot) return -1;
    *addr->sas_addr.prv = 0;
    if (*text)
	if (*text++ != '+') return 1;
	else if (do_try_nsap(text,addr,flags)) return -1;
    addr->sas_family = AF_ATMSVC;
    addr->sas_addr.bhli.hl_type = ATM_HL_NONE;
    addr->sas_addr.blli = NULL;
    return 0;
}


int text2atm(const char *text,struct sockaddr *addr,int length,int flags)
{
    int result;

    if (!*text) return -1;
    if (!(flags & (T2A_PVC | T2A_SVC))) flags |= T2A_PVC | T2A_SVC;
    if (length < sizeof(struct sockaddr_atmpvc)) return -1;
    if (flags & T2A_PVC) {
	result = try_pvc(text,(struct sockaddr_atmpvc *) addr,flags);
	if (result <= 0) return result;
    }
    if (!(flags & T2A_SVC)) return -1;
    if (length < sizeof(struct sockaddr_atmsvc)) return -1;
    result = try_nsap(text,(struct sockaddr_atmsvc *) addr,flags);
    if (result <= 0) return result;
    result = try_e164(text,(struct sockaddr_atmsvc *) addr,flags);
    if (result <= 0) return result;
    if (!(flags & T2A_NAME)) return -1;
    /* @@@ name resolution */
    return -1;
}
