/* io.c - I/O operations */
 
/* Written 1995 by Werner Almesberger, EPFL-LRC */
 

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/atm.h>
#include <linux/atmarp.h>

#include "atmd.h"

#include "table.h"
#include "arp.h"
#include "itf.h"
#include "io.h"


#define COMPONENT "IO"


struct timeval now;


static int kernel,incoming;


/* ----- kernel interface -------------------------------------------------- */


void open_all(void)
{
    struct sockaddr_atmsvc addr;
    struct atm_blli blli;

    if ((kernel = socket(PF_ATMSVC,SOCK_DGRAM,ATM_ATMARP)) < 0)
	diag(COMPONENT,DIAG_FATAL,"socket: %s",strerror(errno));
    if ((incoming = socket(PF_ATMSVC,SOCK_DGRAM,ATM_AAL5)) < 0)
	diag(COMPONENT,DIAG_FATAL,"socket: %s",strerror(errno));
    memset(&addr,0,sizeof(addr));
    addr.sas_family = AF_ATMSVC;
    addr.sas_addr.bhli.hl_type = ATM_HL_NONE;
    addr.sas_addr.blli = &blli;
    blli.l2_proto = ATM_L2_ISO8802;
    blli.l3_proto = ATM_L3_NONE;
    blli.next = NULL;
    if (bind(incoming,(struct sockaddr *) &addr,sizeof(addr)) < 0)
	diag(COMPONENT,DIAG_FATAL,"bind: %s",strerror(errno));
    if (listen(incoming,5) < 0)
	diag(COMPONENT,DIAG_FATAL,"listen: %s",strerror(errno));
}


#if 1
#define KERNEL_BUFFER_SIZE 500
#else
#define KERNEL_BASE_LEN (sizeof(struct atmsvc_msg)-sizeof(struct atm_blli))
#define KERNEL_BUFFER_SIZE (sizeof(struct atmsvc_msg)+(ATM_MAX_BLLI-1)* \
  sizeof(struct atm_blli)+1)
#endif


static void recv_kernel(void)
{
    struct atmarp_ctrl *ctrl;
    unsigned char buffer[KERNEL_BUFFER_SIZE];
    int size;

    size = read(kernel,buffer,KERNEL_BUFFER_SIZE);
    if (size < 0) {
	diag(COMPONENT,DIAG_ERROR,"read kernel: %s",strerror(errno));
	return;
    }
    ctrl = (struct atmarp_ctrl *) buffer;
    if (ctrl->magic != ATMARP_CTRL_MAGIC) {
	diag(COMPONENT,DIAG_ERROR,"invalid magic number (0x%x) in kernel msg",
	  ctrl->magic);
	return;
    }
    switch (ctrl->type) {
	case act_need:
	    need_ip(ctrl->itf_num,ctrl->arg);
	    break;
	case act_up:
	    ctrl->type = act_complete;
	    ctrl->arg = itf_up(ctrl->itf_num);
	    if (write(kernel,ctrl,sizeof(*ctrl)) < 0)
		diag(COMPONENT,DIAG_ERROR,"write reply: %s",strerror(errno));
	    break;
	case act_down:
	    itf_down(ctrl->itf_num);
	    break;
	case act_ioctl:
	    ctrl->type = act_complete;
	    ctrl->arg = arp_ioctl(ctrl->itf_num,ctrl->arg,(struct atmarpreq *)
	      ctrl->data);
	    if (write(kernel,ctrl,sizeof(*ctrl)) < 0)
		diag(COMPONENT,DIAG_ERROR,"write reply: %s",strerror(errno));
	    break;
	default:
	    diag(COMPONENT,DIAG_ERROR,"invalid control msg type 0x%x",
	      ctrl->type);
    }
}


void close_all(void)
{
    (void) close(incoming);
    (void) close(kernel); /* may get major complaints from the kernel ... */
}


/* ----- common part ------------------------------------------------------- */


#define MAX_BUFFER 1024


static fd_set rset,cset;


int do_close(int fd)
{
    int result;

    result = close(fd);
    FD_CLR(fd,&rset); /* we might open a new fd with the same number, so ... */
    FD_CLR(fd,&cset);
    return result;
}


static void recv_vcc(VCC *vcc)
{
    unsigned char buffer[MAX_BUFFER];
    int size;

    size = read(vcc->fd,buffer,KERNEL_BUFFER_SIZE);
    if (!size) {
	disconnect_vcc(vcc);
	return;
    }
    if (size < 0) {
	diag(COMPONENT,DIAG_ERROR,"read vcc: %s",strerror(errno));
	return;
    }
    if (debug) {
	int i;
	for (i = 0; i < size; i++) printf("%02X ",buffer[i]);
	printf("\n");
    }
    incoming_arp(vcc,(struct atmarphdr *) buffer,size);
}


static void accept_new(void)
{
    struct sockaddr_atmsvc addr;
    ITF *itf;
    ENTRY *entry;
    VCC *vcc;
    int fd,len;

    len = sizeof(addr);
    if ((fd = accept(incoming,(struct sockaddr *) &addr,&len)) < 0) {
	diag(COMPONENT,DIAG_ERROR,"accept: %s",strerror(errno));
	return;
    }
    if (ioctl(fd,ATMARP_MKIP,0) < 0) {
        diag(COMPONENT,DIAG_ERROR,"ioctl ATMARP_MKIP: %s",strerror(errno));
        (void) do_close(fd);
        return;
    }
    vcc = alloc_t(VCC);
    vcc->state = vs_svc;
    vcc->fd = fd;
    for (itf = itfs; itf; itf = itf->next) {
	entry = lookup_addr(itf,&addr);
	if (entry) {
	    vcc->flags = entry->flags;
	    vcc->entry = entry;
	    Q_INSERT_HEAD(entry->vccs,vcc);
	    return;
	}
    }
    entry = alloc_t(ENTRY);
    entry->state = as_unknown;
    entry->ip = 0;
    entry->addr = alloc_t(struct sockaddr_atmsvc);
    *entry->addr = addr;
    entry->flags = 0;
    entry->vccs = NULL;
    entry->itf = itfs; /* grr, yet another routing decision @@@ */
    vcc->flags = 0;
    vcc->entry = entry;
    Q_INSERT_HEAD(entry->vccs,vcc);
    Q_INSERT_HEAD(itfs->table,entry);
    incoming_call(vcc);
}


int connect_vcc(struct sockaddr *local,struct sockaddr *remote)
{
    int fd,flags;

    if ((fd = socket(remote->sa_family,SOCK_DGRAM,ATM_AAL5)) < 0) {
	diag(COMPONENT,DIAG_ERROR,"socket: %s",strerror(errno));
	return -errno;
    }
    if (local) {
    if (debug) {
	int i;

	for (i = 0; i < sizeof(struct sockaddr_atmsvc); i++)
	    printf("%02X ",((unsigned char *) local)[i]);
	printf("\n");
    }
   ((struct sockaddr_atmsvc *) local)->sas_addr.bhli.hl_type = 1; /* uuuh @@@ */
	if (bind(fd,local,sizeof(struct sockaddr_atmsvc)) < 0) {
	    diag(COMPONENT,DIAG_ERROR,"bind: %s",strerror(errno));
	    return -errno;
	}
}
    if ((flags = fcntl(fd,F_GETFL)) < 0) {
	diag(COMPONENT,DIAG_ERROR,"fcntl F_GETFL: %s",strerror(errno));
	return -errno;
    }
    flags |= O_NONBLOCK;
    if (fcntl(fd,F_SETFL,flags) < 0) {
	diag(COMPONENT,DIAG_ERROR,"fcntl F_GETFL: %s",strerror(errno));
	return -errno;
    }
    if (remote->sa_family == AF_ATMSVC) { /* @@@ that's cheating */
	static struct atm_blli blli;

	((struct sockaddr_atmsvc *) remote)->sas_addr.blli = &blli;
	blli.l2_proto = ATM_L2_ISO8802;
	blli.l3_proto = ATM_L3_NONE;
	blli.next = NULL;
    }
    if (connect(fd,remote,remote->sa_family == AF_ATMPVC ?
      sizeof(struct sockaddr_atmpvc) : sizeof(struct sockaddr_atmsvc)) < 0) {
	diag(COMPONENT,DIAG_ERROR,"connect: %s",strerror(errno));
	if (errno != EINPROGRESS) return -errno;
    }
    if (ioctl(fd,ATMARP_MKIP,0) < 0) {
        diag(COMPONENT,DIAG_ERROR,"ioctl ATMARP_MKIP: %s",strerror(errno));
        (void) do_close(fd);
        return -errno;
    }
    return fd;
}


int set_ip(int fd,int ip)
{
    if (ioctl(fd,ATMARP_SETENTRY,ip) >= 0) return 0;
    diag(COMPONENT,DIAG_ERROR,"ioctl ATMARP_SETENTRY: %s",strerror(errno));
    (void) do_close(fd);
    return -errno;
}


static void complete_connect(VCC *vcc)
{
    struct sockaddr_atmsvc dummy;

    if (vcc->state != vs_connecting)
	diag(COMPONENT,DIAG_FATAL,"conecting non-connecting VCC 0x%08x",
	  (unsigned long) vcc);
    memset(&dummy,0,sizeof(dummy));
    if (!connect(vcc->fd,(struct sockaddr *) &dummy,sizeof(dummy)))
	vcc_connected(vcc);
    else {
	diag(COMPONENT,DIAG_INFO,"connect: %s",strerror(errno));
	(void) do_close(vcc->fd);
	vcc_failed(vcc);
    }
}


void poll_loop(void)
{
    ITF *itf;
    ENTRY *entry;
    VCC *vcc;
    int fds,ret;

    gettimeofday(&now,NULL);
    while (1) {
	FD_ZERO(&rset);
	FD_ZERO(&cset);
	FD_SET(kernel,&rset);
	FD_SET(incoming,&rset);
	fds = (kernel > incoming ? kernel : incoming)+1;
	for (itf = itfs; itf; itf = itf->next)
	    for (entry = itf->table; entry; entry = entry->next)
		for (vcc = entry->vccs; vcc; vcc = vcc->next) {
		    if (vcc->state != vs_connecting) FD_SET(vcc->fd,&rset);
		    else FD_SET(vcc->fd,&cset);
		    if (vcc->fd >= fds) fds = vcc->fd+1;
		}
	for (vcc = pending; vcc; vcc = vcc->next) {
	    FD_SET(vcc->fd,&rset);
	    if (vcc->fd >= fds) fds = vcc->fd+1;
	}
	ret = select(fds,&rset,&cset,&cset,next_timer());
	if (ret < 0) {
	    if (errno != EINTR) perror("select");
	}
	else {
	    diag(COMPONENT,DIAG_DEBUG,"----------");
	    gettimeofday(&now,NULL);
	    if (FD_ISSET(kernel,&rset)) recv_kernel();
	    if (FD_ISSET(incoming,&rset)) accept_new();
	    for (itf = itfs; itf; itf = itf->next)
		for (entry = itf->table; entry; entry = entry->next)
		    for (vcc = entry->vccs; vcc; vcc = vcc->next)
			if (FD_ISSET(vcc->fd,&rset)) recv_vcc(vcc);
			else if (FD_ISSET(vcc->fd,&cset))
				complete_connect(vcc);
	    for (vcc = pending; vcc; vcc = vcc->next)
		if (FD_ISSET(vcc->fd,&rset)) recv_vcc(vcc);
	    expire_timers();
	      /* expire timers after handling messges to make sure we don't
		 time out unnecessarily because of scheduling delays */
	}
	dump_all();
    }
}


void send_packet(int fd,void *data,int length)
{
    int wrote;

    if (debug) {
	int i;
	for (i = 0; i < length; i++)
	    printf("%02X ",((unsigned char *) data)[i]);
	printf("\n");
    }
    if ((wrote = write(fd,data,length)) == length) return;
    if (wrote < 0)
	diag(COMPONENT,DIAG_ERROR,"write: %s",strerror(errno));
    else diag(COMPONENT,DIAG_ERROR,"short write: %d < %d",wrote,length);
}
