/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/

/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */


/*++
* Module Name:
*   addr_reg.h
*
* Abstract:
*   This file is the interface to the ATM address registration.
*
* Authors:
*   DMW - Douglas M. Washabaugh
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date       Name  Description
*   06-Oct-94  DMW   Created.
*   07-Nov-94  DMW   Updated with review comments, reversed order of routines.
*   29-Mar-95  TLR   Changed callback typedef to a pointer.
*   16-May-95  TLR   Added support for event reporting.
*   06-Sep-95  TLR   Added special address allocation call.
*
* Description:   
*
*   MANAGEMENT INTEFACE:
*   
*   The management interface does not perform any work, but enables applications
*   to have some visibility into this module.  Currently the management
*   interface exports functions to determine the state of address registration
*   and to get a list of the ATM addresses that have been registered.  Generally
*   these routines can be called at any time.
*
* Call Sequence:
*   addr_reg_create              - Creates an instance, inits the MAC addresses.
*   addr_reg_client_register     - Client registers
*   addr_reg_atm_addr_alloc      - Gets and registers an ATM address.
*   addr_reg_atm_addr_dealloc    - Deregister an ATM address.
*   addr_reg_destroy             - Returns any allocated resources.
*
* Issues:
*   - Do the clients need to be able to deregister themselves?  Currently this
*     interface assumes that they do not.
*
*   - Does the management interface require the state of each client?  Does
*     it require a list of clients?
--*/

/* Define the types of events for address status change callbacks. */

typedef enum
   {
   ADDR_REG_EVENT_ATM_ADDR_ALLOCATED,   /* An ATM address was registered.   */
   ADDR_REG_EVENT_ATM_ADDR_INVALIDATED  /* Registered addr was invalidated. */
   } ADDR_REG_ADDR_EVENT;

/* Define values for the addr_reg_special_alloc call. */

#define ADDR_REG_ESI_NEW  0x8001
#define ADDR_REG_ESI_ANY  0x8002
#define ADDR_REG_SEL_ANY  0x8003

/*++
* ---------------------
* - ADDR_REG_CALLBACK -
* ---------------------
*
* Overview:
*   This is the callback made whenever an address is invalidated or registered.
*   In the event an address is invalidated, the selector of the address is not
*   significant.  All addresses with the provided prefix and ESI are to be
*   considered invalidated by this single callback.
*
* Arguments:
*   context    - (IN)  The caller context (handle) that was specified when
*                      the address was registered.
*
*   p_atm_addr - (IN)  Pointer to the ATM address that was registered.  This
*                      pointer is only valid in the callback routine.  It 
*                      may not be saved away.  Instead, the ATM address must
*                      be copied.
*
*   event      - (IN)  The event associated with the ATM address.  The event 
*                      is one of: ADDR_REG_ADDR_EVENT_REGISTERED, 
*                      ADDR_REG_ADDR_EVENT_INVALIDATED.
*
* Returns:
*   none
*
* Preconditions:
*   The registration of an address was started and or completed. 
*
* Postconditions:
*   none
*
* Issues:
*   none
--*/                                              

typedef void (*ADDR_REG_CALLBACK) (HANDLE               context,
                                   ADDR_ATM            *p_atm_addr,
                                   ADDR_REG_ADDR_EVENT  event);   

/*++
* ============================
* = addr_reg_client_register =
* ============================
*
* Overview:
*   Registers a client.  After a client is registered, it may register and 
*   deregister ATM addresses.  A client must
*   register so that it can be told about changes in ATM addresses.
*
* Arguments:
*   addr_reg_handle - (IN)  The handle that was returned from addr_reg_create.
*
*   p_callback      - (IN)  A pointer to a routine to call when an ATM address
*                           registration or deregistration has completed.
* 
*   callback_handle - (IN)  A value that is passed on a callback to the client.
*
*   p_text          - (IN)  A text string to describe this client.
*
*   p_client_handle - (OUT) Handle that may be required for subsequent calls
*                           to this module.
* Returns:
*   STATUS_K_SUCCESS   - Client is registered.
*   STATUS_K_RESOURCES - Client is NOT registered.
*
* Preconditions:
*   addr_reg_create returned a valid handle.
*
* Postconditions:
*   On success, the client handle is valid.
*
* Issues:
*   none
--*/                                              

STATUS addr_reg_client_register (HANDLE             addr_reg_handle,
                                 ADDR_REG_CALLBACK  callback,
                                 HANDLE             callback_handle,
                                 const char         *p_text,
                                 HANDLE            *p_client_handle); 
/*
 *   Deregisters a client.  
 *
 * Arguments:
 *   client_handle - Handle to this client.
 *                   
 * Returns:
 *   0   - Client is deregistered.
 *   -1  - Client is NOT deregistered.
 *
 */                                              

int addr_reg_client_deregister(HANDLE client_handle);

/*++
* ===========================
* = addr_reg_atm_addr_alloc =
* ===========================
*
* Overview:
*   Directs this module to start the process of registering an ATM address
*   with the switch.  When the registration completes, the client is called
*   back.
*  
*   This routine can be anytime after the client has registered.  Note the 
*   client can only have one outstanding registration request at a time.
*
* Arguments:
*   client_handle - (IN) The handle that was returned from
*                        addr_reg_client_register.
*
* Returns:
*   STATUS_K_PENDING - Operation successfully queued.
*   STATUS_K_BUSY    - Client already has a pending operation.
*
* Preconditions:
*   - The routine addr_reg_create was called. 
*   - The routine addr_reg_client_register was called.
*
* Postconditions:
*   The registration of the address will start at the next opportunity.  An
*   opportunity occurs when there is not another pending address registration
*   and there is at least one network prefix with an available MAC address.
*
* Issues:
*   none
--*/                                              

STATUS addr_reg_atm_addr_alloc (HANDLE client_handle);
 
/*++
* =============================
* = addr_reg_atm_addr_dealloc =
* =============================
*
* Overview:
*   The specified ATM address is deallocated and can be reused.  After this
*   call completes, no more packets destined to the ATM address should be sent.
*
* Arguments:
*   client_handle - (IN) The handle that was returned from
*                        addr_reg_client_register.
*
*   p_atm_addr    - (IN) Pointer to the ATM address to deregister.
*
* Returns:
*   STATUS_K_SUCCESS  - The the operation was successfully queued.
*   STATUS_K_ARG2_BAD - The MAC address is not registered.
*
* Preconditions:
*   - The routine addr_reg_create was called. 
*   - The routine addr_reg_client_register was called.
*
* Postconditions:
*   The specified ATM address available for reuse.
*
* Issues:
*   none
--*/                                              

STATUS addr_reg_atm_addr_dealloc (HANDLE    client_handle,
                                  ADDR_ATM *p_atm_addr); 

int address_convert(char *parsestring, ADDR_ATM *atm_addr);
int addr_getesi(unsigned char *mac_addr);
int addr_getouratmaddr(struct sockaddr_atmsvc *addr);
void addr_set_atm_addr(ADDR_ATM *set_to);



