/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/
/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */

/*
* Module Name:
*   map.h
*   
* Overview:
*   This describes the interface to the Mapping Module.  This module provides
*   a high-speed store and lookup function.  It contains only the details
*   needed to perform the storage and lookup functions.  It is not concerned
*   with the details of how they will be used.
*
* Authors:
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date       Name  Description 
*   17-Jan-95  TLR   Created.
*   23-Jan-95  TLR   Updated with inputs from team review.
*   22-Sep-95  TLR   Added get-first and get-next calls.
*
* Description:
*   This module provides a general purpose storage and lookup service.  The
*   user of this service must define a structure that contains the data fields
*   to be stored.  The structure must conform to two requirements:
*
*       1. The first member of the structure must be of type MAP_LIST_HDR,
*
*       2. The second member of the structure must be the search key that will
*          be used when searching the list.
*
*   The following is a template structure:
*
*       typedef struct
*          {
*          MAP_LIST_HDR     list_hdr;
*          ADDR_ATM         addr;
*          -- Additional storage elements go here.
*          } LIST_ELEMENT;
*
*   In this example, the 'addr' member is used as a search key.  If the list
*   is implemented as a hash table or an ordered list or tree, the search key
*   is also used for element insertion.
*
*   The length of the key is specified for each allocated list at the time the
*   list is allocated.
*
*   Note that in the function prototypes, a handle is used to refer to
*   an element in the list.  This element is of the type defined above.
*   The handle must be cast from a pointer to the element.  For example:
*
*       LIST_ELEMENT    *p_elt;
*
*       map_list_add (list1, (HANDLE) p_elt);
*
* Call Sequence Requirements:
*   The only requirement is that for a given list, the map_list_alloc be called
*   prior to any other call.
*
*   - map_list_alloc
*     - map_list_add
*     - map_list_delete
*     - map_list_lookup
*     - map_list_traverse
*     - map_list_dealloc
*
*/

/*************************************************************************
 * Types that are specific to this interface.                            *
 *************************************************************************/

/* MAINTENANCE TIP
 *
 *  The following structure contains members that are specific to the list
 *  implementation.  They have to appear here due to shortcomings of the C
 *  programming language.  If the list implementation is changed, this
 *  structure must also be changed.
 */
typedef struct _hdr
   {
   struct _hdr    *p_next;
   struct _hdr    *p_prev;
   } MAP_LIST_HDR;


/*************************************************************************
 * Function Prototypes for this interface.                               *
 *************************************************************************/

/*++
* ================
* = MAP_CALLBACK =
* ================
*
* Overview:
*   This is the function prototype for list callbacks.  There are cases in
*   which list operations must make a callback for each member of the list.
*
* Arguments:
*   element_handle    - (IN)  Handle that refers to a member of the list.
*   parameter         - (IN)  General purpose handle provided by the caller.
*
* Returns:
*   General purpose boolean flag.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
typedef BOOLEAN (*MAP_CALLBACK) (HANDLE element_handle,
                                 HANDLE parameter);


/*++
* ==================
* = map_list_alloc =
* ==================
*
* Overview:
*   Allocates an empty map list.
*
* Arguments:
*   os_handle        - (IN)  Handle required by the utl_os module.
*   key_length       - (IN)  Length in bytes of the search key for this list.
*   p_list_handle    - (OUT) Map handle describing new map list.
*
* Returns:
*   STATUS_K_SUCCESS    - The list was successfully created.
*   STATUS_K_RESOURCES  - There were insufficient resources to create the list.
*
* Preconditions:
*   None
*
* Postconditions:
*   p_list_handle refers to an empty list.
--*/
STATUS map_list_alloc (UINT8     key_length,
                       HANDLE   *p_list_handle);


/*++
* ====================
* = map_list_dealloc =
* ====================
*
* Overview:
*   Deallocates a map list.  For each member of the list, a callback is called
*   which must properly delete and deallocate the stored element.
*
*   The boolean return value from the callback is ignored by this function.
*
* Arguments:
*   list_handle         - (IN) Handle returned by map_list_alloc.
*   destroy_callback    - (IN) Address of callback routine to destroy members
*                              of the list.  If this callback address is NULL,
*                              no callback will be invoked.
*   parameter           - (IN) Generic parameter to be passed to the callback.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   The map_handle is invalid.
--*/
void map_list_dealloc (HANDLE          list_handle,
                       MAP_CALLBACK    destroy_callback,
                       HANDLE          parameter);


/*++
* ======================
* = map_element_insert =
* ======================
*
* Overview:
*   Adds an element to a map list.
*
* Arguments:
*   list_handle     - (IN)  Handle returned by map_list_alloc.
*   member_handle   - (IN)  Handle of element to add.
*
* Returns:
*   None
*
* Preconditions:
*   The new element is not a member of the list.
*
* Postconditions:
*   The new element is a member of the list.   
--*/
void map_element_insert (HANDLE    list_handle,
                         HANDLE    member_handle);


/*++
* ======================
* = map_element_delete =
* ======================
*
* Overview:
*   Deletes an element from a map list.  Note that this function does not
*   destroy or deallocate anything, it simply removes an entry from the list.
*
* Arguments:
*   list_handle     - (IN)  Handle returned by map_list_alloc.
*   member_handle   - (IN)  Handle of element to be deleted.
*
* Returns:
*   None
*
* Preconditions:
*   The element is a member of the list.
*
* Postconditions:
*   The element is not a member of the list.
--*/
void map_element_delete (HANDLE list_handle,
                         HANDLE member_handle);


/*++
* =====================
* = map_list_size_get =
* =====================
*
* Overview:
*   Returns the number of elements in the list.
*
* Arguments:
*   list_handle     - (IN)  Handle returned by map_list_alloc.
*
* Returns:
*   The list size in number of elements.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
UINT32 map_list_size_get (HANDLE list_handle);

/*++
* ==================
* = map_key_search =
* ==================
*
* Overview:
*   Searches the list for an element containing a key.  If a matching element
*   is found, the handle of that member is returned.
*
* Arguments:
*   list_handle     - (IN)  Handle returned by map_list_alloc.
*
*   p_key           - (IN)  Pointer to the key to be searched for.  The length
*                           of this key is the length that was
*                           specified when the list was allocated.
*
*   p_member_handle - (OUT) If the search was successful, this is the handle
*                           of the found member.
*
* Returns:
*   STATUS_K_SUCCESS    - Search was successful, member handle is valid.
*   STATUS_K_NOT_FOUND  - Search failed, member handle is not valid.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS map_key_search (HANDLE   list_handle,
							  void    *p_key,
							  HANDLE  *p_member_handle);

/*++
* =========================
* = map_key_search_offset =
* =========================
*
* Overview:
*   Searches the list for an element containing a key.  If a matching element
*   is found, the handle of that member is returned. This function is similar
*   to map_key_search, the only difference being that in this function the
*   offset and length into the structure to be searched is specified.
*
* Arguments:
*   list_handle     - (IN)  Handle returned by map_list_alloc.
*
*   p_key           - (IN)  Pointer to the key to be searched for.  The length
*                           of this key is the length that was
*                           specified when the list was allocated.
*	 offset          - (IN)  offset into the structure being searched.
*	 len             - (IN)  Length of the key.
*   p_member_handle - (OUT) If the search was successful, this is the handle
*                           of the found member.
*
* Returns:
*   STATUS_K_SUCCESS    - Search was successful, member handle is valid.
*   STATUS_K_NOT_FOUND  - Search failed, member handle is not valid.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/

STATUS map_key_search_offset (HANDLE   list_handle,
										void    *p_key,
										UINT32   offset,
										INT8     len,
										HANDLE  *p_member_handle);

/*++
* =====================
* = map_list_traverse =
* =====================
*
* Overview:
*   Traverses an entire list, calling the callback for every element in the
*   list.
*
*   For example, this function might be used to update an age timer in each
*   member of a list.
*
*   The boolean return value from the callback is used to continue or stop the
*   traversal.  If the return value is TRUE, the traversal continues.  If it is
*   FALSE, the traversal stops.  This feature allows for a slow but general
*   list search.  This might be useful if it is necessary to search with a key
*   other than the specified search key.
*
*   Note that this routine provides a fast and efficient traversal of the list.
*   The get-first/get-next traversal method may not be as efficient for some
*   map list implementations.
*
* Arguments:
*   list_handle         - (IN)  Handle returned by map_list_alloc.
*   traverse_callback   - (IN)  Callback to be invoked for each member.
*   parameter           - (IN)  Generic parameter to be passed to the callback.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
void map_list_traverse (HANDLE         list_handle,
                        MAP_CALLBACK   traverse_callback,
                        HANDLE         parameter);

/*++
* =================
* = map_get_first =
* =================
*
* Overview:
*   Locate and return the handle of the "first" member of the map list.  The
*   "first" element is defined to be the one with the numerically lowest key.
*
* Arguments:
*   list_handle         - (IN)  Handle returned by map_list_alloc.
*   p_member_handle     - (OUT) Handle of first member if found.
*
* Returns:
*   STATUS_K_SUCCESS    - Successful completion, *p_member_handle is valid.
*   STATUS_K_NOT_FOUND  - The list is empty and has no first member.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS map_get_first (HANDLE   list_handle,
                      HANDLE  *p_member_handle);

/*++
* ================
* = map_get_next =
* ================
*
* Overview:
*   Locate and return the handle of the "next" member of the map list.  The
*   "next" element is defined to be the one with the key that is numerically
*   closest to (and greater than) the provided key.
*
* Arguments:
*   list_handle         - (IN)  Handle returned by map_list_alloc.
*   p_member_handle     - (OUT) Handle of next member if found.
*
* Returns:
*   STATUS_K_SUCCESS    - Successful completion, *p_member_handle is valid.
*   STATUS_K_NOT_FOUND  - There is no "next" entry.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS map_get_next (HANDLE  list_handle,
                     void   *p_key,
                     HANDLE *p_member_handle);


