/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/
/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */

/*
* Module Name:
*   map.c
*   
* Overview:
*   This file implements the mapping module.  A map list is implemented as a
*   doubly linked ring.  The search algorithm is linear search with reordering.
*
* Authors:
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date       Name  Description 
*   18-Jan-95  TLR   Created.
*   23-Jan-95  TLR   Updated with inputs from team review.
*    6-Mar-95  TLR   Fixed a bug in the key_search routine.
*/

#ifdef _DEBUG
#define STATIC
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#else
#define STATIC static
#define assert(expr)
#endif

/* Include General Headers. */

#include "g_types.h"      /* General Purpose Types      */
#include "codes.h"      /* Return Status Codes        */
#include "utl.h"        /* General Purpose Utilities  */
#include "utl_os.h"     /* Operating System Utilities */

/* Include Headers for Mapping Module Interfaces. */

#include "map.h"

#define MAX_KEY   100

/****************************************************************************
 *  Internal Data Structures
 ****************************************************************************/

/* MAP_LIST
 *
 *  This structure describes a map list.
 *
 *  p_head......Pointer to the head element in the list.  This is the first
 *              element in the linear search order.
 *
 *  p_tail......Pointer to the tail element in the list.  This is the last
 *              element in the linear search order.
 *
 *  size........Number of elements in the list.
 *
 *  key_length..The length of the search key field in bytes.
 *
 */
typedef struct _map_list
   {
   MAP_LIST_HDR  *p_head;
   MAP_LIST_HDR  *p_tail;
   UINT32         size;
   UINT8          key_length;
   UINT8          thread_count;
   } MAP_LIST;


/****************************************************************************
 *  Doubly linked list macros
 ****************************************************************************/

/*++
* ========================
* = link_to_head (MACRO) =
* ========================
*
* Overview:
*   Places an element onto the head of a doubly linked list.
*
* Arguments:
*   list        - List to be modified.
*   p_element   - Pointer to element to be added to the list.
*              
* Returns:
*   Not Applicable
*
* Preconditions:
*   None
*
* Postconditions:
*   The element pointed to by p_element is at the head of the list.
*
*   The size of the list is one greater than before the macro call.
*
* Description:
*   Place the new element at the head of the list.  The old head element
*   (if there was one) becomes the second element.  The list structure must
*   remain well-formed (i.e. all of the pointers are correct and the size
*   mmeber of the list structure is equal to the number of elements in the
*   list).
*
*   The first step is to initialize the links of the element to insert. The
*   next step depends upon whether the list was previously empty or not.  If
*   the list was previously empty, then the tail of the list is set to point
*   to the new element.  Otherwise, the element that was the old head must
*   have its previous link set to point to the new element.
*
*   The final step is to set the head of the list to point to the new element,
*   and to increment the count of elements in the list.
*
* Issues:
*   None
*
--*/
#define link_to_head(list,p_element)             \
   {                                             \
   (p_element)->p_prev = NULL;                   \
   (p_element)->p_next = (list)->p_head;         \
                                                 \
   if ((list)->p_head == NULL)                   \
      {                                          \
      (list)->p_tail = (p_element);              \
      assert ((list)->size == 0);                \
      }                                          \
   else                                          \
      (list)->p_head->p_prev = (p_element);      \
                                                 \
   (list)->p_head      = (p_element);            \
   (list)->size       += 1;                      \
   }

/*++
* ==================
* = unlink (MACRO) =
* ==================
*
* Overview:
*   Unlinks an element from a list, leaving the list intact.
*
* Arguments:
*   list        (IN) List to be modified.
*   p_element   (IN) Pointer to the element to be removed from the list.
*              
* Returns:
*   Not Applicable
*
* Preconditions:
*   The element pointed to by p_element is a member of the list.
*
* Postconditions:
*   The element pointed to by p_element is not a member of the list.
*
*   The size of the list is one less than before the macro call.
*
* Description:
*   Wherever the element is in the list, it is removed from the list and
*   the elements neighbors (previous and next) are list are "reconnected".
*   To reconnect a list, the "next" link of the previous element must be set
*   to the next element.  Similiarly, the "prev" link of the next element must
*   be set to point to the previous element.
*
*   The list must remain well-formed.  Note that nothing is deleted, an element
*   is simply dis-associated with the list.
*
*   The first steps are to update the head and tail of the list.  The head 
*   must be updated only if the element was the head, and the tail must be
*   updated if the element was the tail (note that for a list of size 1, the 
*   element can be both the head and the tail).
*
*   The final step is to reconnect the list, as described above.
*   The assertion states that either the list is has at least one element,
*   or the list is empty and that the head and tail pointers are NULL.
*
* Issues:
*   None
*
--*/
#define unlink(list,p_element)                                      \
   {                                                                \
   if ((list)->p_head == (p_element))                               \
      (list)->p_head = (p_element)->p_next;                         \
                                                                    \
   if ((list)->p_tail == (p_element))                               \
      (list)->p_tail = (p_element)->p_prev;                         \
                                                                    \
   if ((p_element)->p_prev != NULL)                                 \
      (p_element)->p_prev->p_next = (p_element)->p_next;            \
                                                                    \
   if ((p_element)->p_next != NULL)                                 \
      (p_element)->p_next->p_prev = (p_element)->p_prev;            \
                                                                    \
   (list)->size -= 1;                                               \
   assert (((list)->size > 0) ||                                    \
           (((list)->size == 0) &&                                  \
            ((list)->p_head == NULL) && ((list)->p_tail == NULL))); \
   }


/****************************************************************************
 *  Function Implementations
 ****************************************************************************/

/*++
* ==================
* = map_list_alloc =
* ==================
*
* Description:
*   Allocate a MAP_LIST descriptor and initialize its fields to represent
*   an empty list.
*
* Issues:
*   None
*
--*/
STATUS map_list_alloc (UINT8     key_length,
                       HANDLE   *p_list_handle)
   {
   MAP_LIST    *p_map;

   /* Allocate a MAP_LIST descriptor for the new list.  If memory allocation
    * fails, return error status.
    */
   p_map = (MAP_LIST*) os_mem_alloc (sizeof (MAP_LIST));
   if (p_map == NULL)
      return STATUS_K_RESOURCES;

   /* Initialize the list elements to make an empty list. */

   p_map->p_head     = NULL;
   p_map->p_tail     = NULL;
   p_map->size       = 0;

   /* Store the key length in the descriptor block. */

   p_map->key_length   = key_length;
   p_map->thread_count = 0;

   /* Convert the address of the descriptor block to a handle and return it
    * to the caller with success status.
    */
   *p_list_handle = (HANDLE) p_map;
   return STATUS_K_SUCCESS;
   }

/*++
* ====================
* = map_list_dealloc =
* ====================
*
* Description:
*   Iteratively request the deletion of each element in the list then
*   deallocate the list descriptor.
*
* Issues:
*   None
*
--*/
void map_list_dealloc (HANDLE          list_handle,
                       MAP_CALLBACK    destroy_callback,
                       HANDLE          parameter)
   {
   MAP_LIST       *p_map;
   MAP_LIST_HDR   *p_ptr, *p_save;

   /* Obtain a pointer to the MAP_LIST descriptor block. */

   p_map = (MAP_LIST*) list_handle;
   p_map->thread_count += 1;

   /* Traverse the list in its entirety calling the destroy callback for each
    * entry in the list.
    */
   p_ptr = p_map->p_head;
   while (p_ptr != NULL)
      {
      /* Save the p_next pointer before destroying the entry. */

      p_save = p_ptr->p_next;
      (void) destroy_callback ((HANDLE) p_ptr, parameter);
      p_ptr  = p_save;
      }

   /* Deallocate the memory used for the descriptor block. */

   os_mem_dealloc (p_map);
   }

/*++
* ======================
* = map_element_insert =
* ======================
*
* Description:
*   Link the element represented by the element_handle onto the head of the
*   list.
*
* Issues:
*   None
*
--*/
void map_element_insert (HANDLE    list_handle,
                         HANDLE    element_handle)
   {
   MAP_LIST       *p_map;
   MAP_LIST_HDR   *p_element;

   /* Convert the list handle to a pointer to a MAP_LIST descriptor. */

   p_map = (MAP_LIST*) list_handle;
   p_map->thread_count += 1;

   /* Convert the element handle to a pointer to a MAP_LIST_HDR.  Note that
    * the actual element structure is larger than a MAP_LIST_HDR structure but
    * this function doesn't care about what's beyond the header.
    */
   p_element = (MAP_LIST_HDR*) element_handle;

   /* Link the new element to the head of the list. */

   link_to_head (p_map, p_element);
   p_map->thread_count -= 1;
   }

/*++
* ======================
* = map_element_delete =
* ======================
*
* Description:
*   Unlink the element represented by element_handle from the list.  No memory
*   is deallocated by this function.
*
* Issues:
*   None
*
--*/
void map_element_delete (HANDLE list_handle,
                         HANDLE element_handle)
   {
   MAP_LIST       *p_map;
   MAP_LIST_HDR   *p_element;

   /* Convert the list handle to a pointer to a MAP_LIST descriptor. */

   p_map = (MAP_LIST*) list_handle;
   p_map->thread_count += 1;

   /* Convert the element handle to a pointer to a MAP_LIST_HDR.  Note that
    * the actual element structure is larger than a MAP_LIST_HDR structure but
    * this function doesn't care about what's beyond the header.
    */
   p_element = (MAP_LIST_HDR*) element_handle;

   /* Unlink the element from the list. */

   unlink (p_map, p_element);
   p_map->thread_count -= 1;
   }

/*++
* =====================
* = map_list_size_get =
* =====================
*
* Description:
*   Return the size parameter of the list.
*
* Issues:
*   None
*
--*/
UINT32 map_list_size_get (HANDLE list_handle)
   {
   MAP_LIST *p_map;

   p_map = (MAP_LIST *) list_handle;

   return p_map->size;
   }

/*++
* ==================
* = map_key_search =
* ==================
*
* Description:
*   Linearly search the list for an element that has a key field that is
*   identical to the provided search key.  If a matching element is found
*   ensure that it is left on the head of the list.
*
* Issues:
*   None
*
--*/
STATUS map_key_search (HANDLE   list_handle,
                       void    *p_key,
                       HANDLE  *p_element_handle)
   {
   MAP_LIST       *p_map;
   MAP_LIST_HDR   *p_element;
   INT8            index;
	UINT8          *search_key;
   UINT8          *test_key;
   BOOLEAN         found;

   /* Obtain a pointer to a MAP_LIST descriptor from the list handle. */

   p_map = (MAP_LIST*) list_handle;
   p_map->thread_count += 1;

   /* Cast the address of the search key to a pointer to a byte. */

   search_key = (UINT8*) p_key;

   /* Iterate linearly through the list searching for a matching key. */

   p_element = p_map->p_head;
   found     = FALSE;
   while (p_element != NULL)
      {
      /* Cast the address of the key in the list element to a pointer to a
       * byte.
       */
      test_key = (UINT8*) ((unsigned long) p_element + sizeof (MAP_LIST_HDR));

      /* Search the key byte by byte from the rear forward.  This is done
       * because it is assumed that the keys (network addresses) will have
       * their unique octets right justified.  This is to make mismatches
       * fail more quickly and therefore more efficiently.
       */
		index = p_map->key_length - 1;
      found = TRUE;
      while ((found) && (index >= 0))
         {
         if (search_key[index] != test_key[index])
            found = FALSE;
         index -= 1;
         }

      /* If a matching element has been found, stop the search. */

      if (found)
         break;

      /* Go to the next element in the list. */

      p_element = p_element->p_next;
      }

   if (found)
      {
      /* If a matching element was found, return the pointer to that element
       * in the form of a handle.
       */
      *p_element_handle = (HANDLE) p_element;

      /* If the found element is not on the head of the list, unlink it from
       * its current location and place it at the head of the list.
       */
		if ((p_element != p_map->p_head) && (p_map->thread_count == 1))
         {
         unlink       (p_map, p_element);
         link_to_head (p_map, p_element);
         }

      /* Return success status. */

      p_map->thread_count -= 1;
      return STATUS_K_SUCCESS;
      }

	/* If a matching element was not found, return error status. */

   p_map->thread_count -= 1;
   return STATUS_K_NOT_FOUND;
   }


/*++
* =========================
* = map_key_search_offset =
* =========================
*
* Description:
*   Linearly search the list for an element that has a key field that is
*   identical to the provided search key.  If a matching element is found
*   ensure that it is left on the head of the list.
*
* Issues:
*   None
*
--*/
STATUS map_key_search_offset (HANDLE   list_handle,
										void    *p_key,
										UINT32   offset,
										INT8     len,
										HANDLE  *p_element_handle)
	{
	MAP_LIST       *p_map;
	MAP_LIST_HDR   *p_element;
	INT8            index;
	UINT8          *search_key;
	UINT8          *test_key;
	BOOLEAN         found;

	/* Obtain a pointer to a MAP_LIST descriptor from the list handle. */

	p_map = (MAP_LIST*) list_handle;
	p_map->thread_count += 1;

	/* Cast the address of the search key to a pointer to a byte. */

	search_key = (UINT8*) p_key;

	/* Iterate linearly through the list searching for a matching key. */

	p_element = p_map->p_head;
	found     = FALSE;
	while (p_element != NULL)
		{
		/* Cast the address of the key in the list element to a pointer to a
		 * byte.
		 */
		test_key = (UINT8*) ((unsigned long) p_element +
									 sizeof (MAP_LIST_HDR) +
									 offset);

		/* Search the key byte by byte from the rear forward.  This is done
		 * because it is assumed that the keys (network addresses) will have
		 * their unique octets right justified.  This is to make mismatches
		 * fail more quickly and therefore more efficiently.
		 */
		index = len - 1;
		found = TRUE;
		while ((found) && (index >= 0))
			{
			if (search_key[index] != test_key[index])
				found = FALSE;
			index -= 1;
			}

		/* If a matching element has been found, stop the search. */

		if (found)
			break;

		/* Go to the next element in the list. */

		p_element = p_element->p_next;
		}

	if (found)
		{
		/* If a matching element was found, return the pointer to that element
		 * in the form of a handle.
		 */
		*p_element_handle = (HANDLE) p_element;

		/* If the found element is not on the head of the list, unlink it from
		 * its current location and place it at the head of the list.
		 */
		if ((p_element != p_map->p_head) && (p_map->thread_count == 1))
			{
			unlink       (p_map, p_element);
			link_to_head (p_map, p_element);
			}

		/* Return success status. */

		p_map->thread_count -= 1;
		return STATUS_K_SUCCESS;
		}

	/* If a matching element was not found, return error status. */

	p_map->thread_count -= 1;
	return STATUS_K_NOT_FOUND;
	}

/*++
* =====================
* = map_list_traverse =
* =====================
*
* Description:
*   Traverse the list and invoke a callback for each element of the list.
*   If the callback returns BOOLEAN FALSE, stop traversing the list.
*
* Issues:
*   The callback must not do a lookup on this list.  The reordering of the
*   elements might lead to unpredictable behavior during traversal.
*
--*/
void map_list_traverse (HANDLE         list_handle,
                        MAP_CALLBACK   traverse_callback,
                        HANDLE         parameter)
   {
   MAP_LIST       *p_map;
   MAP_LIST_HDR   *p_element, *p_save;
   BOOLEAN         cont;

   /* Obtain a pointer to a MAP_LIST descriptor from the list handle. */

   p_map = (MAP_LIST*) list_handle;
   p_map->thread_count += 1;

   /* Traverse the list, calling the traverse_callback for each element. */

   p_element = p_map->p_head;
   while (p_element != NULL)
      {
      /* Save the next pointer before invoking the callback.  This is in
       * case the callback deletes the element.
       */
      p_save = p_element->p_next;

      /* If the callback returns FALSE, stop iterating through the list. */

      cont = traverse_callback ((HANDLE) p_element, parameter);
      if (!cont) break;

      /* Go to the next element in the list. */

      p_element = p_save;
      }

   p_map->thread_count -= 1;
   }

#define COMPARE_LT   1
#define COMPARE_EQ   2
#define COMPARE_GT   4
#define COMPARE_LE   (COMPARE_LT | COMPARE_EQ)
#define COMPARE_GE   (COMPARE_GT | COMPARE_EQ)

static UINT32 compare_multiple (UINT8  *p_arg1,
                                UINT8  *p_arg2,
                                UINT8   key_length)
   {
   int      i;

   for (i = 0; i < key_length; i++)
      {
      if (p_arg1[i] > p_arg2[i])
         return COMPARE_GT;
      if (p_arg1[i] < p_arg2[i])
         return COMPARE_LT;
      }
   return COMPARE_EQ;
   }


/*++
* =================
* = map_get_first =
* =================
*
* Description:
*   If the list is empty, return NOT_FOUND.  Otherwise, find the entry with
*   the lowest numerical key.
*
* Issues:
*
--*/
STATUS map_get_first (HANDLE   list_handle,
                      HANDLE  *p_member_handle)
   {
   MAP_LIST       *p_map;
   MAP_LIST_HDR   *p_element, *p_lowest;
   UINT8           temp[MAX_KEY], *p_key;
   int             i;

   /* Obtain a pointer to a MAP_LIST descriptor from the list handle. */

   p_map = (MAP_LIST*) list_handle;

   /* Handle the case in which the list is empty. */

   if (p_map->size == 0)
      return STATUS_K_NOT_FOUND;

   /* Set temporary key to greatest possible value. */

   for (i = 0; i < MAX_KEY; i++)
      temp[i] = 0xFF;

   /* Initialize pointers for list traversal. */

   p_element = p_map->p_head;
   p_lowest  = p_element;

   while (p_element != NULL)
      {
      p_key = (UINT8 *) ((unsigned long) p_element +
                         sizeof (MAP_LIST_HDR));
      if (compare_multiple (temp,
                            p_key,
                            p_map->key_length) & COMPARE_GT)
         {
         p_lowest = p_element;
         utl_mem_copy (p_key, temp, p_map->key_length);
         }

      p_element = p_element->p_next;
      }

   *p_member_handle = (HANDLE) p_lowest;
   return STATUS_K_SUCCESS;
   }

/*++
* ================
* = map_get_next =
* ================
*
* Description:
*   If the list is empty, return NOT_FOUND.  Otherwise, find the entry with
*   the lowest numerical key which is greater than the supplied key.
*
* Issues:
*
--*/
STATUS map_get_next (HANDLE  list_handle,
                     void   *p_key,
                     HANDLE *p_member_handle)
   {
   MAP_LIST       *p_map;
   MAP_LIST_HDR   *p_element, *p_lowest;
   UINT8          *p_member_key;

   /* Obtain a pointer to a MAP_LIST descriptor from the list handle. */

   p_map = (MAP_LIST*) list_handle;

   /* Handle the case in which the list is empty. */

   if (p_map->size == 0)
      return STATUS_K_NOT_FOUND;

   /* Initialize pointers for list traversal. */

   p_element = p_map->p_head;
   p_lowest  = NULL;

   /* Traverse the list looking for the next greater element. */

   while (p_element != NULL)
      {
      p_member_key = (UINT8 *) ((unsigned long) p_element +
                                 sizeof (MAP_LIST_HDR));
      if (compare_multiple (p_member_key,
                            p_key,
                            p_map->key_length) & COMPARE_GT)
         {
         /* If we get here, we have an "eligible" entry which is greater
          * than the supplied key.  Check to see if it's either the first
          * eligible entry or less than the previous least.
          */
         if ((p_lowest == NULL) ||
             (compare_multiple (p_member_key,
                                (UINT8 *) ((unsigned long) p_lowest +
                                           sizeof (MAP_LIST_HDR)),
                                p_map->key_length) & COMPARE_LT))
            {
            p_lowest = p_element;
            }
         }

      p_element = p_element->p_next;
      }

   if (p_lowest == NULL)
      {
      return STATUS_K_NOT_FOUND;
      }
   else
      {
      *p_member_handle = (HANDLE) p_lowest;
      return STATUS_K_SUCCESS;
      }
   }

