/* io.c - I/O operations */
 
/* Written 1995,1996 by Werner Almesberger, EPFL-LRC */
 

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/atm.h>
#include <linux/atmdev.h>
#include <linux/atmsvc.h>

#include "atm.h"
#include "atmd.h"
#include "q2931.h"

#include "proto.h"
#include "io.h"


#define COMPONENT "IO"


struct timeval now;
struct sockaddr_atmpvc signaling_pvc;


static int kernel,signaling;


/* ----- kernel interface -------------------------------------------------- */


static int open_kernel(void)
{
    int s;

    if ((s = socket(PF_ATMSVC,SOCK_DGRAM,0)) < 0) {
	perror("socket");
	return -1;
    }
    if (ioctl(s,ATMSIGD_CTRL,0) < 0) {
	perror("ioctl ATMSIGD_CTRL");
	return -1;
    }
    return s;
}


#define KERNEL_BASE_LEN (sizeof(struct atmsvc_msg)-sizeof(struct atm_blli))
#define KERNEL_BUFFER_SIZE (sizeof(struct atmsvc_msg)+(ATM_MAX_BLLI-1)* \
  sizeof(struct atm_blli)+1)


static void recv_kernel(void)
{
    static unsigned char buffer[KERNEL_BUFFER_SIZE];
    int size,bllis;

    size = read(kernel,buffer,KERNEL_BUFFER_SIZE);
    if (size < 0) {
	perror("read kernel");
	return;
    }
    if (size < KERNEL_BASE_LEN) {
	diag(COMPONENT,DIAG_FATAL,"kernel message too short (%d < %d)",size,
	  KERNEL_BASE_LEN);
	return;
    }
    if (size == KERNEL_BUFFER_SIZE) { /* gotcha ! */
	diag(COMPONENT,DIAG_FATAL,"kernel message too big (>= %d)",
	  KERNEL_BUFFER_SIZE);
	return;
    }
    bllis = (size-KERNEL_BASE_LEN)/sizeof(struct atm_blli);
    if (size != KERNEL_BASE_LEN+bllis*sizeof(struct atm_blli)) {
	diag(COMPONENT,DIAG_FATAL,
	  "kernel message has bad length (got %d, base is %d, incr is %d)",
	  size,KERNEL_BASE_LEN,sizeof(struct atm_blli));
	return;
    }
    from_kernel((struct atmsvc_msg *) buffer,size);
}


void to_kernel(struct atmsvc_msg *msg)
{
    struct atm_blli *walk;
    int size,wrote;

    size = bllis2msg(0);
    for (walk = msg->svc.sas_addr.blli; walk; walk = walk->next)
	size += sizeof(struct atm_blli);
    diag("KERNEL",DIAG_DEBUG,"TO KERNEL: %s (%d) for 0x%lx/0x%lx <%d>",
      as_name[msg->type],msg->reply,msg->vcc,msg->listen_vcc,size);
	/* should be "IO" ... */
    wrote = write(kernel,msg,size);
    if (wrote == size) return;
    if (wrote < 0) {
	perror("kernel write");
	return;
    }
    diag(COMPONENT,DIAG_ERROR,"bad kernel write: wanted %d, wrote %d",size,
      wrote);
}


static void close_kernel(void)
{
    (void) close(kernel); /* may get major complaints from the kernel ... */
}


/* ----- signaling interface ----------------------------------------------- */


static int open_signaling(void)
{
    int s;

    if ((s = socket(PF_ATMPVC,SOCK_DGRAM,ATM_AAL5)) < 0) {
	perror("socket");
	return -1;
    }
    signaling_pvc.sap_family = AF_ATMPVC;
    signaling_pvc.sap_rxtp.class = signaling_pvc.sap_txtp.class = ATM_UBR;
    signaling_pvc.sap_rxtp.max_sdu = signaling_pvc.sap_txtp.max_sdu = MAX_Q_MSG;
    if (bind(s,(struct sockaddr *) &signaling_pvc,sizeof(signaling_pvc)) < 0) {
	perror("bind");
	return -1;
    }
    return s;
}


static void recv_signaling(void)
{
    static unsigned char buffer[MAX_Q_MSG];
    int size;

    size = read(signaling,buffer,MAX_Q_MSG);
    if (size < 1) {
	perror("read signaling");
	return;
    }
#if 0
    {
	int i;

	for (i = 0; i < size; i++) printf("%02X ",buffer[i]);
	if (size) putchar('\n');
    }
#endif
    from_net(buffer,size);
}


void to_net(void *msg,int size)
{
    int wrote,i;

    diag(COMPONENT,DIAG_DEBUG,"TO NET: %s (0x%02x) CR 0x%06x (%d bytes)",
      mid2name(((unsigned char *) msg)[5]),((unsigned char *) msg)[5],
      (((unsigned char *) msg)[2] << 16) |
      (((unsigned char *) msg)[3] << 8) | ((unsigned char *) msg)[4],
      size);
    if (debug) {
	for (i = 0; i < size; i++) printf("%02X ",((unsigned char *) msg)[i]);
	if (size) putchar('\n');
    }
    wrote = write(signaling,msg,size);
    if (wrote == size) return;
    if (wrote < 0) {
	perror("signaling write");
	return;
    }
    diag(COMPONENT,DIAG_WARN,"bad signaling write: wanted %d, wrote %d",size,
      wrote);
}


static void close_signaling(void)
{
    (void) close(signaling);
}


/* ----- addresses --------------------------------------------------------- */


LOCAL_ADDR local_addr[MAX_ADDRS+1];


int get_addr(int itf)
{
    struct atmif_sioc req;
    struct sockaddr_atmsvc buffer[MAX_ADDRS];
    LOCAL_ADDR *from,*to;
    int addrs,i;

    for (from = to = local_addr; from->state != ls_unused; from++)
	if (from->state != ls_removed) {
	    from->state = itf == from->itf ? ls_removed : ls_same;
	    *to++ = *from;
	}
    req.number = itf;
    req.arg = buffer;
    req.length = sizeof(buffer);
    if (ioctl(kernel,ATM_GETADDR,&req) < 0)
	diag(COMPONENT,DIAG_FATAL,"ioctl ATM_GETADDR yields \"%s\"",
	  strerror(errno));
    addrs = req.length/sizeof(struct sockaddr_atmsvc);
    for (i = 0; i < addrs; i++) {
	for (from = local_addr; from->state != ls_unused; from++)
	    if (from->itf == itf && atm_equal(buffer+i,&from->addr,0,0)) break;
	if (from->state != ls_unused) from->state = ls_same;
	else if (to == local_addr+MAX_ADDRS-1)
		diag(COMPONENT,DIAG_WARN,"local address table overflow");
	    else {
		to->state = ls_added;
		to->itf = itf;
		to->addr = buffer[i];
		to++;
	    }
    }
    to->state = ls_unused;
    return addrs;
}


/* ----- common part ------------------------------------------------------- */


int open_all(void)
{
    kernel = open_kernel();
    if (kernel < 0) return -1;
    signaling = open_signaling();
    if (signaling < 0) {
	close_kernel();
	return -1;
    }
    local_addr[0].state = ls_unused;
    return 0;
}


void close_all(void)
{
    close_kernel();
    close_signaling();
}


void poll_loop(void)
{
    fd_set perm,set;
    int fds,ret;

    FD_ZERO(&perm);
    FD_SET(kernel,&perm);
    FD_SET(signaling,&perm);
    fds = kernel > signaling ? kernel+1 : signaling+1;
    gettimeofday(&now,NULL);
    while (1) {
	set = perm;
	ret = select(fds,&set,NULL,NULL,next_timer());
	if (ret < 0) {
	    if (errno != EINTR) perror("select");
	}
	else {
	    diag(COMPONENT,DIAG_DEBUG,"----------");
	    gettimeofday(&now,NULL);
	    if (FD_ISSET(kernel,&set)) recv_kernel();
	    if (FD_ISSET(signaling,&set)) recv_signaling();
	    expire_timers();
	      /* expire timers after handling messges to make sure we don't
		 time out unnecessarily because of scheduling delays */
	}
    }
}
