/*	Copyright (c) 1982 Michael Landy, Yoav Cohen, and George Sperling

Disclaimer:  No guarantees of performance accompany this software,
nor is any responsibility assumed on the part of the authors.  All the
software has been tested extensively and every effort has been made to
insure its reliability.   */

/*
 * reduce - reduce a sequence by pixel averaging
 *
 * usage:	reduce [-h hfactor] [-v vfactor] [-t tfactor]
 * 		reduce [spacefactor]
 * 		reduce [-u]
 *
 * 
 * Reduce reduces the input sequence vertically by vfactor, horizontally by 
 * hfactor and temporally by tfactor.  The spatial factors default to 1, and 
 * tfactor defaults to 1.  In the second calling sequence form, the user
 * specifies spacefactor, which is applied in both spatial dimensions, and
 * defaults to 2.  The -u switch displays a usage summary.  This filter works
 * on byte, short, integer, float, and complex images.
 *
 * to load:	cc -o reduce reduce.c -lhipl
 *
 * Rewritten by Michael Landy - 11/5/87
 * short added by Bill Johnston - 3 Apr 89
 */

#include <stdio.h>
#include <ctype.h>
#include <hipl_format.h>
char Progname[]="reduce";

char usage[] =
"usage:	reduce [-h hfactor] [-v vfactor] [-t tfactor]\n\
	reduce [spacefactor]\n\
	reduce [-u]\n\
\n\
Defaults: hfactor = vfactor = tfactor = 1.  -u displays this message.\n\
If present in the command line, spacefactor must be the first argument;\n\
in this case hfactor and vfactor are both set to spacefactor, and tfactor\n\
defaults to 1.  No arguments defaults to spacefactor = 2.";

int sspec = 0;

main(argc,argv)

int argc;
char **argv;

{
	struct header hd;
	register int off,i,j,k,x,y;
	int t,tfactor = 1,yfactor = 1,xfactor = 1;
	int iframes,irows,icols,iframepix,iframebytes,oframes,orows,ocols;
	int oframepix,oframebytes,scale,start,excess;

	start = 1;
	if (argc >= 2 && isdigit(argv[1][0])) {
		xfactor = yfactor = atoi(argv[1]);
		start = 2;
		sspec++;
	}
	for (i=start;i<argc;i++)
		if (argv[i][0] == '-')
		    switch (argv[i][1]) {
		    case 'u': perr(usage);
		    case 't': sspec++; tfactor = atoi(argv[++i]); continue;
		    case 'v': sspec++; yfactor = atoi(argv[++i]); continue;
		    case 'h': sspec++; xfactor = atoi(argv[++i]); continue;
		    default: perr("unrecognized option: %s",argv[i]);
		    }
	if (!sspec)
		xfactor = yfactor = 2;
	read_header(&hd);
	hd.num_frame = oframes = (iframes = hd.num_frame) / tfactor;
	hd.rows = orows = (irows = hd.rows) / yfactor;
	hd.cols = ocols = (icols = hd.cols) / xfactor;
	excess = icols - (ocols*xfactor);
	oframepix = orows*ocols;
	iframepix = irows*icols;
	scale = tfactor*yfactor*xfactor;
	update_header(&hd,argc,argv);
	write_header(&hd);

	if (hd.pixel_format == PFCOMPLEX) {
		register float *ip,*op;
		float *ofr,*ifr;

		iframebytes = iframepix * 2*sizeof(float);
		oframebytes = oframepix * 2*sizeof(float);

		if ((ifr = (float *) malloc(iframebytes)) == NULL)
			perr("reduce: can't get core.");
		if ((ofr = (float *) malloc(oframebytes)) == NULL)
			perr("reduce: can't get core.");

		for (t=oframes;t;t--) {
		    for (op=ofr,i=2*oframepix;i;i--)
			*op++ = 0;
		    for (k=tfactor;k;k--) {
			if (pread(0,ifr,iframebytes) != iframebytes)
			    perr("reduce: read error");
			for (op=ofr,ip=ifr,y=orows;y;y--) {
			    for (i=yfactor;i;i--) {
				for (x=ocols;x;x--) {
				    for (j=xfactor;j;j--) {
					*op += *ip++;
					*(op+1) += *ip++;
				    }
				    op += 2;
				}
				op -= ocols*2;
				ip += excess*2;
			    }
			    op += ocols*2;
			}
		    }
		    for (op=ofr,i=oframepix;i;i--) {
			*op++ /= scale;
			*op++ /= scale;
		    }
		    if (write(1,ofr,oframebytes) != oframebytes)
			perr("reduce: write error");
		}
	}
	else if (hd.pixel_format == PFFLOAT) {
		register float *ip,*op;
		float *ofr,*ifr;

		iframebytes = iframepix * sizeof(float);
		oframebytes = oframepix * sizeof(float);

		if ((ifr = (float *) malloc(iframebytes)) == NULL)
			perr("reduce: can't get core.");
		if ((ofr = (float *) malloc(oframebytes)) == NULL)
			perr("reduce: can't get core.");

		for (t=oframes;t;t--) {
		    for (op=ofr,i=oframepix;i;i--)
			*op++ = 0;
		    for (k=tfactor;k;k--) {
			if (pread(0,ifr,iframebytes) != iframebytes)
			    perr("reduce: read error");
			for (op=ofr,ip=ifr,y=orows;y;y--) {
			    for (i=yfactor;i;i--) {
				for (x=ocols;x;x--) {
				    for (j=xfactor;j;j--) {
					*op += *ip++;
				    }
				    op++;
				}
				op -= ocols;
				ip += excess;
			    }
			    op += ocols;
			}
		    }
		    for (op=ofr,i=oframepix;i;i--)
			*op++ /= scale;
		    if (write(1,ofr,oframebytes) != oframebytes)
			perr("reduce: write error");
		}
	}
	else if (hd.pixel_format == PFINT) {
		register int *ip,*op;
		int *ofr,*ifr;

		iframebytes = iframepix * sizeof(int);
		oframebytes = oframepix * sizeof(int);

		if ((ifr = (int *) malloc(iframebytes)) == NULL)
			perr("reduce: can't get core.");
		if ((ofr = (int *) malloc(oframebytes)) == NULL)
			perr("reduce: can't get core.");

		for (t=oframes;t;t--) {
		    for (op=ofr,i=oframepix;i;i--)
			*op++ = 0;
		    for (k=tfactor;k;k--) {
			if (pread(0,ifr,iframebytes) != iframebytes)
			    perr("reduce: read error");
			for (op=ofr,ip=ifr,y=orows;y;y--) {
			    for (i=yfactor;i;i--) {
				for (x=ocols;x;x--) {
				    for (j=xfactor;j;j--) {
					*op += *ip++;
				    }
				    op++;
				}
				op -= ocols;
				ip += excess;
			    }
			    op += ocols;
			}
		    }
		    for (op=ofr,i=oframepix;i;i--)
			*op++ /= scale;
		    if (write(1,ofr,oframebytes) != oframebytes)
			perr("reduce: write error");
		}
	}
	else if (hd.pixel_format == PFSHORT) {
		register short *ip,*op;
		short *ofr,*ifr;
		register int *sop;
		int *sofr,soframebytes;

		iframebytes = iframepix * sizeof(short);
		oframebytes = oframepix * sizeof(short);
		soframebytes = oframepix * sizeof(int);

		if ((ifr = (short *) malloc(iframebytes)) == NULL)
			perr("reduce: can't get core.");
		if ((ofr = (short *) malloc(oframebytes)) == NULL)
			perr("reduce: can't get core.");
		if ((sofr = (int *) malloc(soframebytes)) == NULL)
			perr("reduce: can't get core.");

		for (t=oframes;t;t--) {
		    for (sop=sofr,i=oframepix;i;i--)
			*sop++ = 0;
		    for (k=tfactor;k;k--) {
			if (pread(0,ifr,iframebytes) != iframebytes)
			    perr("reduce: read error");
			for (sop=sofr,ip=ifr,y=orows;y;y--) {
			    for (i=yfactor;i;i--) {
				for (x=ocols;x;x--) {
				    for (j=xfactor;j;j--) {
					*sop += *ip++;
				    }
				    sop++;
				}
				sop -= ocols;
				ip += excess;
			    }
			    sop += ocols;
			}
		    }
		    for (op=ofr,sop=sofr,i=oframepix;i;i--)
			*op++ = *sop++ / scale;
		    if (write(1,ofr,oframebytes) != oframebytes)
			perr("reduce: write error");
		}
	}
	else if (hd.pixel_format == PFBYTE) {
		register unsigned char *ip,*op;
		unsigned char *ofr,*ifr;
		register int *sop;
		int *sofr,soframebytes;

		iframebytes = iframepix * sizeof(unsigned char);
		oframebytes = oframepix * sizeof(unsigned char);
		soframebytes = oframepix * sizeof(int);

		if ((ifr = (unsigned char *) malloc(iframebytes)) == NULL)
			perr("reduce: can't get core.");
		if ((ofr = (unsigned char *) malloc(oframebytes)) == NULL)
			perr("reduce: can't get core.");
		if ((sofr = (int *) malloc(soframebytes)) == NULL)
			perr("reduce: can't get core.");

		for (t=oframes;t;t--) {
		    for (sop=sofr,i=oframepix;i;i--)
			*sop++ = 0;
		    for (k=tfactor;k;k--) {
			if (pread(0,ifr,iframebytes) != iframebytes)
			    perr("reduce: read error");
			for (sop=sofr,ip=ifr,y=orows;y;y--) {
			    for (i=yfactor;i;i--) {
				for (x=ocols;x;x--) {
				    for (j=xfactor;j;j--) {
					*sop += *ip++;
				    }
				    sop++;
				}
				sop -= ocols;
				ip += excess;
			    }
			    sop += ocols;
			}
		    }
		    for (op=ofr,sop=sofr,i=oframepix;i;i--)
			*op++ = *sop++ / scale;
		    if (write(1,ofr,oframebytes) != oframebytes)
			perr("reduce: write error");
		}
	}
	else
		perr("reduce: inseq must be int, short, byte, real, or complex");
	exit(0);
}
