/*	Copyright (c) 1982 Michael Landy, Yoav Cohen, and George Sperling

Disclaimer:  No guarantees of performance accompany this software,
nor is any responsibility assumed on the part of the authors.  All the
software has been tested extensively and every effort has been made to
insure its reliability.   */

/*
 * mask.c - filter an image by applying one or more masks and then
 *          applying another function to the various mask outputs.
 *	    The input is in byte format, and the output is floating point.
 *
 * usage:	mask [-f filter-number] <input_sequence >output_sequence
 *			or
 *		mask -m filter-descriptor-file <in >out
 *
 * where filter-number specifies:
 *
 *	filter number   mask type	function
 *		1	3-level (3x3)	maxabs
 *		2	3-level (5x5)	maxabs
 *		3	3-level (7x7)	maxabs
 *		4	3-level (9x9)	maxabs
 *		5	5-level (3x3)	maxabs
 *		6	Prewitt (3x3)	mean-square
 *		7	Prewitt (3x3)	sumabs
 *		8	Prewitt (5x5)	mean-square
 *		9	Prewitt (5x5)	sumabs
 *		10	Prewitt (7x7)	mean-square
 *		11	Prewitt	(7x7)	sumabs
 *		12	Prewitt (9x9)	mean-square
 *		13	Prewitt (9x9)	sumabs
 *		14	Roberts (2x2)	mean-square
 *		15	Roberts (2x2)	sumabs
 *		16	Sobel (3x3)	mean-square
 *		17	Sobel (3x3)	sumabs
 *		18	compass (3x3)	max
 *		19	Kirsch (3x3)	max
 *		20	pyramid (5x5)	maxabs
 *		21	pyramid (7x7)	maxabs
 *		22	pyramid (9x9)	maxabs
 *		23	4x4		mean-square
 *		24	Laplacian - a	Identity
 *		25	Laplacian - b	Identity
 *		26	Laplacian - c	Identity
 *		27	Laplacian - d	Identity
 *		28	Kasvand-Lapl	Identity
 *		29	Kasvand-line	max-floor
 *		30	Eberlein	max-abs-sub-floor
 *		...	etc. (see /usr/hipl/masks)
 *
 * The default filter is 1.  The definition for each of these filters
 * is to be found in /usr/hipl/masks/maskn, where n is the filter number.
 * The -m switch allows a new filter to be supplied by the user.  The format
 * of the filter definition file is as follows:
 *
 *	"filter name"
 *	masksize number-of-masks function-number
 *	mask-1
 *	  .
 *	  .
 *	  .
 *	mask-(masksize)
 *
 * where the masksize is the length of a side of all masks (which must be
 * square), masks are given as a sequence of integers in column-fastest order,
 * and the function applied to the output of the masks is chosen from:
 *
 *	1	MAXABS	- the maximum absolute value of all mask outputs
 *	2	MEANSQ  - the square root of the sum of the squares of all masks
 *	3	SUMABS  - the sum of the absolute value of all mask outputs
 *	4	MAX	- the maximum mask output
 *	5	MAXFLR	- the maximum mask output, floored at zero.
 *	6	MXASFLR	- the larger of |mask-1| and |mask-2|, minus |mask-3|,
 *			  floored at zero.
 *	7	MUL	- the product of the mask outputs, each floored at zero.
 *	8	NORM	- the first mask output normalized by the sum of the
 *			  mask entries.
 *	9	DIFF	- the value of the pixel minus the normalized mask
 *			  output.
 *	10	ORIENT	- compute orientation: 360*atan(mask2/mask1)/2*PI
 *
 * to load:	cc -o mask mask.c -lm -lhipl
 *
 * Michael Landy - 4/21/82
 *
 *  made maskfile size bigger, from 40 to 120: char maskfile[120] = MSKLIB; -LBL
 */

#include <hipl_format.h>
#include <stdio.h>
#include <math.h>
char *Progname;

#define	MAXABS		1
#define	MEANSQ		2
#define	SUMABS		3
#define MAXX		4
#define	MAXFLR		5
#define	MXASFLR		6
#define	MUL		7
#define	NORM		8
#define	DIFF		9
#define	ORIENT		10
#define	MAXMASKS	10

#define tcol(c)	(c<0? 0 : (c>=col? col-1 : c))
#define trow(c) (c<0? 0 : (c>=row? row-1 : c))

char maskfile[120] = MSKLIB;

main(argc,argv)

int argc;
char **argv;

{
	struct header hd;
	int row,col,nmask,masksz,**mlist,*mval,mfunc,frameno,r,c,dr,dc;
	int minusd,plusd,m,*p,mr,mc;
	int filter,**maskar,i,j,k;
	int flag,boundl,boundr,boundt,boundb,pinc,norm;
	float val,*ofr,*op;
	char *s,*pic,*mfile,name[100],*ppic;
	FILE *fp;
	double d2r = (360./(2.*3.1415926));

	Progname = strsave(*argv);
	read_header(&hd);
	if (hd.pixel_format != PFBYTE)
		perr("image pixel format must be bytes");
	row = hd.rows;
	col = hd.cols;
	pic = (char *) halloc(row*col,sizeof (char));
	ofr = (float *) halloc(row*col,sizeof (float));
	hd.pixel_format = PFFLOAT;
	hd.bits_per_pixel = 32;
	update_header(&hd,argc,argv);
	write_header(&hd);

	if (argv[argc-1][0]=='-' && argv[argc-1][1]=='D') argc--;
	if (argc<=1) {
		strcat(maskfile,"/mask1");
		mfile = maskfile;
	}
	else if (argc==2)
		perr("insufficient number of arguments");
	else if (argv[1][0]=='-') {
		if (argv[1][1]=='f') {
			strcat(maskfile,"/mask");
			strcat(maskfile,argv[2]);
			mfile = maskfile;
		}
		else if (argv[1][1]=='m') {
			mfile = argv[2];
		}
		else
			perr("unknown switch");
	}
	else
		perr("first argument should be -f or -m");
	if ((fp = fopen(mfile,"r"))==NULL)
		perr("can't open mask file %s",mfile);
	s = name;
	while((*s++ = getc(fp)) != '\n');
	*s++ = '\0';
	fscanf(fp,"%d %d %d",&masksz,&nmask,&mfunc);
	if (masksz < 1 || masksz > 50 || nmask < 1 || nmask > 10 ||
		mfunc < 1 || mfunc > MAXMASKS) {
		fprintf(stderr,"mask:	bad filter descriptor value:	");
		fprintf(stderr,"mask size = %d, number of masks = %d,",
			masksz,nmask);
		perr(" function number = %d\n",mfunc);
	}
	mlist = (int **) halloc(nmask,sizeof (int **));
	mval = (int *) halloc(nmask,sizeof (int));

	norm = 0;
	for (i=0;i<nmask;i++) {
		p = (int *) halloc(masksz*masksz,sizeof (int));
		mlist[i] = p;
		for (mr=0;mr<masksz;mr++) {
		    for (mc=0;mc<masksz;mc++) {
			if (fscanf(fp,"%d",&j) == EOF)
				perr("mask:	unexpected \
end-of-file in the filter descriptor file\n");
			norm += j;
			*p++ = j;
		    }
		}
	}

	fprintf(stderr,"mask: applying filter:	%s\n",name);
	plusd = masksz/2;
	minusd = plusd - masksz + 1;
	boundl = boundt = -minusd;	/* compute boundaries of where the */
	boundr = col - plusd - 1;	/* mask overlaps the image completely */
	boundb = row - plusd - 1;	/* for more efficient convolution */
	pinc = col - masksz;		/* calculation. */
	for (frameno=0;frameno<hd.num_frame;frameno++) {
		op = ofr;
		fprintf(stderr,"mask: starting frame #%d\n",frameno);
		if (pread(0,pic,row*col) != row*col)
			perr("unexpected end-of-file");
		for (r=0;r<row;r++) {
		    for (c=0;c<col;c++) {
			flag = (r < boundt || r > boundb ||
				c < boundl || c > boundr) ? 1 : 0;
			for (m=0;m<nmask;m++) {
			    k = 0;
			    p = mlist[m];
			    if (flag) {
				for (dr=minusd;dr<=plusd;dr++) {
				    for (dc=minusd;dc<=plusd;dc++) {
					k += *p++ * (pic[trow(r+dr)*col +
						tcol(c+dc)] & 0377);
				    }
				}
			    }
			    else {
				ppic = pic + (r + minusd)*col + c + minusd;
				for (dr=minusd;dr<=plusd;dr++) {
				    for (dc=minusd;dc<=plusd;dc++) {
					k += *p++ * (*ppic++ & 0377);
				    }
				    ppic += pinc;
				}
			    }
			    mval[m] = k;
			}
			k = 0;
			switch(mfunc) {
			case MAXABS:
				k = abs(mval[0]);
				for (i=1;i<nmask;i++)
					if(abs(mval[i])>k) k = abs(mval[i]);
				val = k;
				break;
			case MEANSQ:
				for (i=0;i<nmask;i++)
					k += mval[i]*mval[i];
				val = k;
				val = sqrt((double) val);
				break;
			case SUMABS:
				for (i=0;i<nmask;i++)
					k += abs(mval[i]);
				val = k;
				break;
			case MAXX:
				k = mval[0];
				for (i=1;i<nmask;i++)
					if (mval[i]>k) k=mval[i];
				val = k;
				break;
			case MAXFLR:
				k = mval[0];
				for (i=1;i<nmask;i++)
					if (mval[i]>k) k=mval[i];
				val = k>0 ? k : 0;
				break;
			case MXASFLR:
				k = abs(mval[0])>abs(mval[1])
					? abs(mval[0]) : abs(mval[1]);
				k -= abs(mval[2]);
				val = k>0 ? k : 0;
				break;
			case MUL:
				k = 1;
				for (i=0;i<nmask;i++)
					k *= mval[i]>0 ? mval[i] : 0;
				val = k;
				break;
			case NORM:
				val = ((float) mval[0])/norm;
				break;
			case DIFF:
				val = (pic[r*col+c]&0377) - 
					((float) mval[0])/norm;
				break;
			case ORIENT:
				if (mval[0] == 0 && mval[1] == 0)
					val = 0;
				else
					val = d2r*atan2((double) (mval[1]),
						(double) (mval[2]));
				break;
			default:
				perr("bad function type!?");
			}
			*op++ = val;
		    }
		}
		if (write(1,ofr,row*col*sizeof(float))!=row*col*sizeof(float))
			perr("write error");
	}
	return(0);
}
